"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.AuthenticationType = void 0;

var _opensearch_security_client = require("../../backend/opensearch_security_client");

var _tenant_resolver = require("../../multitenancy/tenant_resolver");

var _errors = require("../../errors");

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

class AuthenticationType {
  constructor(config, sessionStorageFactory, router, esClient, coreSetup, logger) {
    this.config = config;
    this.sessionStorageFactory = sessionStorageFactory;
    this.router = router;
    this.esClient = esClient;
    this.coreSetup = coreSetup;
    this.logger = logger;

    _defineProperty(this, "type", void 0);

    _defineProperty(this, "securityClient", void 0);

    _defineProperty(this, "authHandler", async (request, response, toolkit) => {
      var _this$config$multiten;

      // skip auth for APIs that do not require auth
      if (this.authNotRequired(request)) {
        return toolkit.authenticated();
      } // if browser request, auth logic is:
      //   1. check if request includes auth header or paramter(e.g. jwt in url params) is present, if so, authenticate with auth header.
      //   2. if auth header not present, check if auth cookie is present, if no cookie, send to authentication workflow
      //   3. verify whether auth cookie is valid, if not valid, send to authentication workflow
      //   4. if cookie is valid, pass to route handlers


      const authHeaders = {};
      let cookie;
      let authInfo; // if this is an REST API call, suppose the request includes necessary auth header
      // see https://www.elastic.co/guide/en/opensearch-dashboards/master/using-api.html

      if (this.requestIncludesAuthInfo(request)) {
        try {
          const additonalAuthHeader = this.getAdditionalAuthHeader(request);
          Object.assign(authHeaders, additonalAuthHeader);
          authInfo = await this.securityClient.authinfo(request, additonalAuthHeader);
          cookie = await this.getCookie(request, authInfo); // set tenant from cookie if exist

          const browserCookie = await this.sessionStorageFactory.asScoped(request).get();

          if (browserCookie && (0, _tenant_resolver.isValidTenant)(browserCookie.tenant)) {
            cookie.tenant = browserCookie.tenant;
          }

          this.sessionStorageFactory.asScoped(request).set(cookie);
        } catch (error) {
          return response.unauthorized({
            body: error.message
          });
        }
      } else {
        // no auth header in request, try cookie
        try {
          cookie = await this.sessionStorageFactory.asScoped(request).get();
        } catch (error) {
          this.logger.error(`Error parsing cookie: ${error.message}`);
          cookie = undefined;
        }

        if (!cookie || !(await this.isValidCookie(cookie))) {
          // clear cookie
          this.sessionStorageFactory.asScoped(request).clear(); // for assets, we can still pass it to resource handler as notHandled.
          // marking it as authenticated may result in login pop up when auth challenge
          // is enabled.

          if (request.url.pathname && request.url.pathname.startsWith('/bundles/')) {
            return toolkit.notHandled();
          } // send to auth workflow


          return this.handleUnauthedRequest(request, response, toolkit);
        } // extend session expiration time


        if (this.config.session.keepalive) {
          cookie.expiryTime = Date.now() + this.config.session.ttl;
          this.sessionStorageFactory.asScoped(request).set(cookie);
        } // cookie is valid
        // build auth header


        const authHeadersFromCookie = this.buildAuthHeaderFromCookie(cookie);
        Object.assign(authHeaders, authHeadersFromCookie);
        const additonalAuthHeader = this.getAdditionalAuthHeader(request);
        Object.assign(authHeaders, additonalAuthHeader);
      } // resolve tenant if necessary


      if ((_this$config$multiten = this.config.multitenancy) !== null && _this$config$multiten !== void 0 && _this$config$multiten.enabled && (0, _tenant_resolver.isMultitenantPath)(request)) {
        try {
          const tenant = await this.resolveTenant(request, cookie, authHeaders, authInfo); // return 401 if no tenant available

          if (!(0, _tenant_resolver.isValidTenant)(tenant)) {
            return response.badRequest({
              body: 'No available tenant for current user, please reach out to your system administrator'
            });
          } // set tenant in header


          Object.assign(authHeaders, {
            securitytenant: tenant
          }); // set tenant to cookie

          if (tenant !== cookie.tenant) {
            cookie.tenant = tenant;
            this.sessionStorageFactory.asScoped(request).set(cookie);
          }
        } catch (error) {
          this.logger.error(`Failed to resolve user tenant: ${error}`);

          if (error instanceof _errors.UnauthenticatedError) {
            if (request.url.pathname && request.url.pathname.startsWith('/bundles/')) {
              return toolkit.notHandled();
            }

            this.sessionStorageFactory.asScoped(request).clear();
            return this.handleUnauthedRequest(request, response, toolkit);
          }

          throw error;
        }
      }

      return toolkit.authenticated({
        requestHeaders: authHeaders
      });
    });

    this.securityClient = new _opensearch_security_client.SecurityClient(esClient);
    this.type = '';
  }

  authNotRequired(request) {
    const pathname = request.url.pathname;

    if (!pathname) {
      return false;
    } // allow requests to ignored routes


    if (AuthenticationType.ROUTES_TO_IGNORE.includes(pathname)) {
      return true;
    } // allow requests to routes that doesn't require authentication


    if (this.config.auth.unauthenticated_routes.indexOf(pathname) > -1) {
      // TODO: use opensearch-dashboards server user
      return true;
    }

    return false;
  }

  async resolveTenant(request, cookie, authHeader, authInfo) {
    if (!authInfo) {
      try {
        authInfo = await this.securityClient.authinfo(request, authHeader);
      } catch (error) {
        throw new _errors.UnauthenticatedError(error);
      }
    }

    const selectedTenant = (0, _tenant_resolver.resolveTenant)(request, authInfo.user_name, authInfo.tenants, this.config, cookie);
    return selectedTenant;
  }

  isPageRequest(request) {
    const path = request.url.pathname || '/';
    return path.startsWith('/app/') || path === '/' || path.startsWith('/goto/');
  } // abstract functions for concrete auth types to implement


}

exports.AuthenticationType = AuthenticationType;

_defineProperty(AuthenticationType, "ROUTES_TO_IGNORE", ['/api/core/capabilities', // FIXME: need to figureout how to bypass this API call
'/app/login']);

_defineProperty(AuthenticationType, "REST_API_CALL_HEADER", 'osd-xsrf');
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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