"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.PRIVATE_TENANTS = exports.GLOBAL_TENANTS = void 0;
exports.isMultitenantPath = isMultitenantPath;
exports.isValidTenant = isValidTenant;
exports.resolveTenant = resolveTenant;

var _lodash = require("lodash");

/*
 *   Copyright OpenSearch Contributors
 *
 *   Licensed under the Apache License, Version 2.0 (the "License").
 *   You may not use this file except in compliance with the License.
 *   A copy of the License is located at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   or in the "license" file accompanying this file. This file is distributed
 *   on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 *   express or implied. See the License for the specific language governing
 *   permissions and limitations under the License.
 */
const PRIVATE_TENANT_SYMBOL = '__user__';
const GLOBAL_TENANT_SYMBOL = '';
const PRIVATE_TENANTS = [PRIVATE_TENANT_SYMBOL, 'private'];
exports.PRIVATE_TENANTS = PRIVATE_TENANTS;
const GLOBAL_TENANTS = ['global', GLOBAL_TENANT_SYMBOL];
/**
 * Resovles the tenant the user is using.
 *
 * @param request OpenSearchDashboards request.
 * @param config security plugin config.
 * @param cookie cookie extracted from the request. The cookie should have been parsed by AuthenticationHandler.
 * pass it as parameter instead of extracting again.
 * @param authInfo authentication info, the Elasticsearch authinfo API response.
 *
 * @returns user preferred tenant of the request.
 */

exports.GLOBAL_TENANTS = GLOBAL_TENANTS;

function resolveTenant(request, username, availabeTenants, config, cookie) {
  var _request$url, _request$url$searchPa, _request$url2, _request$url2$searchP, _config$multitenancy, _config$multitenancy2, _config$multitenancy3;

  let selectedTenant;
  const securityTenant_ = request === null || request === void 0 ? void 0 : (_request$url = request.url) === null || _request$url === void 0 ? void 0 : (_request$url$searchPa = _request$url.searchParams) === null || _request$url$searchPa === void 0 ? void 0 : _request$url$searchPa.get('securityTenant_');
  const securitytenant = request === null || request === void 0 ? void 0 : (_request$url2 = request.url) === null || _request$url2 === void 0 ? void 0 : (_request$url2$searchP = _request$url2.searchParams) === null || _request$url2$searchP === void 0 ? void 0 : _request$url2$searchP.get('securitytenant');

  if (securityTenant_) {
    selectedTenant = securityTenant_;
  } else if (securitytenant) {
    selectedTenant = securitytenant;
  } else if (request.headers.securitytenant || request.headers.securityTenant_) {
    selectedTenant = request.headers.securitytenant ? request.headers.securitytenant : request.headers.securityTenant_;
  } else if (isValidTenant(cookie.tenant)) {
    selectedTenant = cookie.tenant;
  } else {
    selectedTenant = undefined;
  }

  const preferredTenants = (_config$multitenancy = config.multitenancy) === null || _config$multitenancy === void 0 ? void 0 : _config$multitenancy.tenants.preferred;
  const globalTenantEnabled = ((_config$multitenancy2 = config.multitenancy) === null || _config$multitenancy2 === void 0 ? void 0 : _config$multitenancy2.tenants.enable_global) || false;
  const privateTenantEnabled = ((_config$multitenancy3 = config.multitenancy) === null || _config$multitenancy3 === void 0 ? void 0 : _config$multitenancy3.tenants.enable_private) || false;
  return resolve(username, selectedTenant, preferredTenants, availabeTenants, globalTenantEnabled, privateTenantEnabled);
}
/**
 * Determines whether the request requires tenant info.
 * @param request opensearch-dashboards request.
 *
 * @returns true if the request requires tenant info, otherwise false.
 */


function isMultitenantPath(request) {
  var _request$url$pathname, _request$url$pathname2, _request$url$pathname3, _request$url$pathname4, _request$url$pathname5;

  return ((_request$url$pathname = request.url.pathname) === null || _request$url$pathname === void 0 ? void 0 : _request$url$pathname.startsWith('/opensearch')) || ((_request$url$pathname2 = request.url.pathname) === null || _request$url$pathname2 === void 0 ? void 0 : _request$url$pathname2.startsWith('/api')) || ((_request$url$pathname3 = request.url.pathname) === null || _request$url$pathname3 === void 0 ? void 0 : _request$url$pathname3.startsWith('/app')) || ( // short url path
  (_request$url$pathname4 = request.url.pathname) === null || _request$url$pathname4 === void 0 ? void 0 : _request$url$pathname4.startsWith('/goto')) || // bootstrap.js depends on tenant info to fetch opensearch-dashboards configs in tenant index
  (((_request$url$pathname5 = request.url.pathname) === null || _request$url$pathname5 === void 0 ? void 0 : _request$url$pathname5.indexOf('bootstrap.js')) || -1) > -1 || request.url.pathname === '/';
}

function resolve(username, requestedTenant, preferredTenants, availableTenants, // is an object like { tenant_name_1: true, tenant_name_2: false, ... }
globalTenantEnabled, privateTenantEnabled) {
  const availableTenantsClone = (0, _lodash.cloneDeep)(availableTenants);
  delete availableTenantsClone[username];

  if (!globalTenantEnabled && !privateTenantEnabled && (0, _lodash.isEmpty)(availableTenantsClone)) {
    return undefined;
  }

  if (isValidTenant(requestedTenant)) {
    requestedTenant = requestedTenant;

    if (requestedTenant in availableTenants) {
      return requestedTenant;
    }

    if (privateTenantEnabled && username in availableTenants && PRIVATE_TENANTS.indexOf(requestedTenant) > -1) {
      return PRIVATE_TENANT_SYMBOL;
    }

    if (globalTenantEnabled && GLOBAL_TENANTS.indexOf(requestedTenant) > -1) {
      return GLOBAL_TENANT_SYMBOL;
    }
  }

  if (preferredTenants && !(0, _lodash.isEmpty)(preferredTenants)) {
    for (const element of preferredTenants) {
      const tenant = element.toLowerCase();

      if (globalTenantEnabled && GLOBAL_TENANTS.indexOf(tenant) > -1) {
        return GLOBAL_TENANT_SYMBOL;
      }

      if (privateTenantEnabled && PRIVATE_TENANTS.indexOf(tenant) > -1 && username in availableTenants) {
        return PRIVATE_TENANT_SYMBOL;
      }

      if (tenant in availableTenants) {
        return tenant;
      }
    }
  }

  if (globalTenantEnabled) {
    return GLOBAL_TENANT_SYMBOL;
  }

  if (privateTenantEnabled) {
    return PRIVATE_TENANT_SYMBOL;
  } // fall back to the first tenant in the available tenants


  return (0, _lodash.findKey)(availableTenantsClone, () => true);
}
/**
 * Return true if tenant parameter is a valid tenent.
 *
 * Note: empty string '' is valid, which means global tenant.
 *
 * @param tenant
 */


function isValidTenant(tenant) {
  return tenant !== undefined && tenant !== null;
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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