"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerAppAnalyticsRouter = registerAppAnalyticsRouter;

var _configSchema = require("@osd/config-schema");

var _application_analytics = require("../../../common/constants/application_analytics");

var _app_analytics_adaptor = require("../../../server/adaptors/application_analytics/app_analytics_adaptor");

/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

/* eslint-disable no-console */
function registerAppAnalyticsRouter(router) {
  const appAnalyticsBackend = new _app_analytics_adaptor.AppAnalyticsAdaptor(); // Fetches all existing applications

  router.get({
    path: `${_application_analytics.APP_ANALYTICS_API_PREFIX}/`,
    validate: {}
  }, async (context, request, response) => {
    const opensearchClient = context.observability_plugin.observabilityClient.asScoped(request);
    let applicationsData = [];

    try {
      applicationsData = await appAnalyticsBackend.fetchApps(opensearchClient);
      return response.ok({
        body: {
          data: applicationsData
        }
      });
    } catch (err) {
      console.error('Error occurred while fetching applications', err);
      return response.custom({
        statusCode: err.statusCode || 500,
        body: err.message
      });
    }
  }); // Fetch application by id

  router.get({
    path: `${_application_analytics.APP_ANALYTICS_API_PREFIX}/{appId}`,
    validate: {
      params: _configSchema.schema.object({
        appId: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchClient = context.observability_plugin.observabilityClient.asScoped(request);

    try {
      const appObject = await appAnalyticsBackend.fetchAppById(opensearchClient, request.params.appId);
      return response.ok({
        body: appObject
      });
    } catch (err) {
      console.error('Error occurred while fetching application', err);
      return response.custom({
        statusCode: err.statusCode || 500,
        body: err.message
      });
    }
  }); // Create a new application

  router.post({
    path: `${_application_analytics.APP_ANALYTICS_API_PREFIX}/`,
    validate: {
      body: _configSchema.schema.object({
        name: _configSchema.schema.string(),
        description: _configSchema.schema.maybe(_configSchema.schema.string()),
        baseQuery: _configSchema.schema.string(),
        servicesEntities: _configSchema.schema.arrayOf(_configSchema.schema.string()),
        traceGroups: _configSchema.schema.arrayOf(_configSchema.schema.string()),
        availabilityVisId: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    const opensearchClient = context.observability_plugin.observabilityClient.asScoped(request);

    try {
      const newAppId = await appAnalyticsBackend.createNewApp(opensearchClient, request.body.name, request.body.description || '', request.body.baseQuery, request.body.servicesEntities, request.body.traceGroups, request.body.availabilityVisId || '');
      return response.ok({
        body: {
          message: 'Application Created',
          newAppId
        }
      });
    } catch (err) {
      console.error('Error occurred while creating a new application', err);
      return response.custom({
        statusCode: err.statusCode || 500,
        body: err.message
      });
    }
  }); // Renames an existing application

  router.put({
    path: `${_application_analytics.APP_ANALYTICS_API_PREFIX}/rename`,
    validate: {
      body: _configSchema.schema.object({
        appId: _configSchema.schema.string(),
        name: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchClient = context.observability_plugin.observabilityClient.asScoped(request);

    try {
      await appAnalyticsBackend.renameApp(opensearchClient, request.body.appId, request.body.name);
      return response.ok({
        body: {
          message: 'Application Renamed'
        }
      });
    } catch (err) {
      console.error('Error occurred while renaming an existing application', err);
      return response.custom({
        statusCode: err.statusCode || 500,
        body: err.message
      });
    }
  }); // Updates an existing application

  router.put({
    path: `${_application_analytics.APP_ANALYTICS_API_PREFIX}/`,
    validate: {
      body: _configSchema.schema.object({
        appId: _configSchema.schema.string(),
        updateBody: _configSchema.schema.object({
          name: _configSchema.schema.maybe(_configSchema.schema.string()),
          description: _configSchema.schema.maybe(_configSchema.schema.string()),
          servicesEntities: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.string())),
          traceGroups: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.string())),
          panelId: _configSchema.schema.maybe(_configSchema.schema.string()),
          availabilityVisId: _configSchema.schema.maybe(_configSchema.schema.string())
        })
      })
    }
  }, async (context, request, response) => {
    const opensearchClient = context.observability_plugin.observabilityClient.asScoped(request);

    try {
      const updatedAppId = await appAnalyticsBackend.updateApp(opensearchClient, request.body.appId, request.body.updateBody);
      return response.ok({
        body: {
          message: 'Application Updated',
          updatedAppId
        }
      });
    } catch (err) {
      console.error('Error occurred while updating an existing application', err);
      return response.custom({
        statusCode: err.statusCode || 500,
        body: err.message
      });
    }
  }); // Delete applications

  router.delete({
    path: `${_application_analytics.APP_ANALYTICS_API_PREFIX}/{appList}`,
    validate: {
      params: _configSchema.schema.object({
        appList: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchClient = context.observability_plugin.observabilityClient.asScoped(request);

    try {
      const delResponse = await appAnalyticsBackend.deleteApp(opensearchClient, request.params.appList);
      return response.ok({
        body: delResponse
      });
    } catch (err) {
      return response.custom({
        statusCode: err.statusCode || 500,
        body: err.message
      });
    }
  });
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbImFwcF9hbmFseXRpY3Nfcm91dGVyLnRzIl0sIm5hbWVzIjpbInJlZ2lzdGVyQXBwQW5hbHl0aWNzUm91dGVyIiwicm91dGVyIiwiYXBwQW5hbHl0aWNzQmFja2VuZCIsIkFwcEFuYWx5dGljc0FkYXB0b3IiLCJnZXQiLCJwYXRoIiwiQVBJX1BSRUZJWCIsInZhbGlkYXRlIiwiY29udGV4dCIsInJlcXVlc3QiLCJyZXNwb25zZSIsIm9wZW5zZWFyY2hDbGllbnQiLCJvYnNlcnZhYmlsaXR5X3BsdWdpbiIsIm9ic2VydmFiaWxpdHlDbGllbnQiLCJhc1Njb3BlZCIsImFwcGxpY2F0aW9uc0RhdGEiLCJmZXRjaEFwcHMiLCJvayIsImJvZHkiLCJkYXRhIiwiZXJyIiwiY29uc29sZSIsImVycm9yIiwiY3VzdG9tIiwic3RhdHVzQ29kZSIsIm1lc3NhZ2UiLCJwYXJhbXMiLCJzY2hlbWEiLCJvYmplY3QiLCJhcHBJZCIsInN0cmluZyIsImFwcE9iamVjdCIsImZldGNoQXBwQnlJZCIsInBvc3QiLCJuYW1lIiwiZGVzY3JpcHRpb24iLCJtYXliZSIsImJhc2VRdWVyeSIsInNlcnZpY2VzRW50aXRpZXMiLCJhcnJheU9mIiwidHJhY2VHcm91cHMiLCJhdmFpbGFiaWxpdHlWaXNJZCIsIm5ld0FwcElkIiwiY3JlYXRlTmV3QXBwIiwicHV0IiwicmVuYW1lQXBwIiwidXBkYXRlQm9keSIsInBhbmVsSWQiLCJ1cGRhdGVkQXBwSWQiLCJ1cGRhdGVBcHAiLCJkZWxldGUiLCJhcHBMaXN0IiwiZGVsUmVzcG9uc2UiLCJkZWxldGVBcHAiXSwibWFwcGluZ3MiOiI7Ozs7Ozs7QUFNQTs7QUFPQTs7QUFFQTs7QUFmQTtBQUNBO0FBQ0E7QUFDQTs7QUFDQTtBQWFPLFNBQVNBLDBCQUFULENBQW9DQyxNQUFwQyxFQUFxRDtBQUMxRCxRQUFNQyxtQkFBbUIsR0FBRyxJQUFJQywwQ0FBSixFQUE1QixDQUQwRCxDQUcxRDs7QUFDQUYsRUFBQUEsTUFBTSxDQUFDRyxHQUFQLENBQ0U7QUFDRUMsSUFBQUEsSUFBSSxFQUFHLEdBQUVDLCtDQUFXLEdBRHRCO0FBRUVDLElBQUFBLFFBQVEsRUFBRTtBQUZaLEdBREYsRUFLRSxPQUNFQyxPQURGLEVBRUVDLE9BRkYsRUFHRUMsUUFIRixLQUlrRTtBQUNoRSxVQUFNQyxnQkFBNEMsR0FBR0gsT0FBTyxDQUFDSSxvQkFBUixDQUE2QkMsbUJBQTdCLENBQWlEQyxRQUFqRCxDQUNuREwsT0FEbUQsQ0FBckQ7QUFHQSxRQUFJTSxnQkFBdUMsR0FBRyxFQUE5Qzs7QUFDQSxRQUFJO0FBQ0ZBLE1BQUFBLGdCQUFnQixHQUFHLE1BQU1iLG1CQUFtQixDQUFDYyxTQUFwQixDQUE4QkwsZ0JBQTlCLENBQXpCO0FBQ0EsYUFBT0QsUUFBUSxDQUFDTyxFQUFULENBQVk7QUFDakJDLFFBQUFBLElBQUksRUFBRTtBQUNKQyxVQUFBQSxJQUFJLEVBQUVKO0FBREY7QUFEVyxPQUFaLENBQVA7QUFLRCxLQVBELENBT0UsT0FBT0ssR0FBUCxFQUFpQjtBQUNqQkMsTUFBQUEsT0FBTyxDQUFDQyxLQUFSLENBQWMsNENBQWQsRUFBNERGLEdBQTVEO0FBQ0EsYUFBT1YsUUFBUSxDQUFDYSxNQUFULENBQWdCO0FBQ3JCQyxRQUFBQSxVQUFVLEVBQUVKLEdBQUcsQ0FBQ0ksVUFBSixJQUFrQixHQURUO0FBRXJCTixRQUFBQSxJQUFJLEVBQUVFLEdBQUcsQ0FBQ0s7QUFGVyxPQUFoQixDQUFQO0FBSUQ7QUFDRixHQTVCSCxFQUowRCxDQW1DMUQ7O0FBQ0F4QixFQUFBQSxNQUFNLENBQUNHLEdBQVAsQ0FDRTtBQUNFQyxJQUFBQSxJQUFJLEVBQUcsR0FBRUMsK0NBQVcsVUFEdEI7QUFFRUMsSUFBQUEsUUFBUSxFQUFFO0FBQ1JtQixNQUFBQSxNQUFNLEVBQUVDLHFCQUFPQyxNQUFQLENBQWM7QUFDcEJDLFFBQUFBLEtBQUssRUFBRUYscUJBQU9HLE1BQVA7QUFEYSxPQUFkO0FBREE7QUFGWixHQURGLEVBU0UsT0FDRXRCLE9BREYsRUFFRUMsT0FGRixFQUdFQyxRQUhGLEtBSWtFO0FBQ2hFLFVBQU1DLGdCQUE0QyxHQUFHSCxPQUFPLENBQUNJLG9CQUFSLENBQTZCQyxtQkFBN0IsQ0FBaURDLFFBQWpELENBQ25ETCxPQURtRCxDQUFyRDs7QUFHQSxRQUFJO0FBQ0YsWUFBTXNCLFNBQVMsR0FBRyxNQUFNN0IsbUJBQW1CLENBQUM4QixZQUFwQixDQUN0QnJCLGdCQURzQixFQUV0QkYsT0FBTyxDQUFDaUIsTUFBUixDQUFlRyxLQUZPLENBQXhCO0FBSUEsYUFBT25CLFFBQVEsQ0FBQ08sRUFBVCxDQUFZO0FBQ2pCQyxRQUFBQSxJQUFJLEVBQUVhO0FBRFcsT0FBWixDQUFQO0FBR0QsS0FSRCxDQVFFLE9BQU9YLEdBQVAsRUFBaUI7QUFDakJDLE1BQUFBLE9BQU8sQ0FBQ0MsS0FBUixDQUFjLDJDQUFkLEVBQTJERixHQUEzRDtBQUNBLGFBQU9WLFFBQVEsQ0FBQ2EsTUFBVCxDQUFnQjtBQUNyQkMsUUFBQUEsVUFBVSxFQUFFSixHQUFHLENBQUNJLFVBQUosSUFBa0IsR0FEVDtBQUVyQk4sUUFBQUEsSUFBSSxFQUFFRSxHQUFHLENBQUNLO0FBRlcsT0FBaEIsQ0FBUDtBQUlEO0FBQ0YsR0FoQ0gsRUFwQzBELENBdUUxRDs7QUFDQXhCLEVBQUFBLE1BQU0sQ0FBQ2dDLElBQVAsQ0FDRTtBQUNFNUIsSUFBQUEsSUFBSSxFQUFHLEdBQUVDLCtDQUFXLEdBRHRCO0FBRUVDLElBQUFBLFFBQVEsRUFBRTtBQUNSVyxNQUFBQSxJQUFJLEVBQUVTLHFCQUFPQyxNQUFQLENBQWM7QUFDbEJNLFFBQUFBLElBQUksRUFBRVAscUJBQU9HLE1BQVAsRUFEWTtBQUVsQkssUUFBQUEsV0FBVyxFQUFFUixxQkFBT1MsS0FBUCxDQUFhVCxxQkFBT0csTUFBUCxFQUFiLENBRks7QUFHbEJPLFFBQUFBLFNBQVMsRUFBRVYscUJBQU9HLE1BQVAsRUFITztBQUlsQlEsUUFBQUEsZ0JBQWdCLEVBQUVYLHFCQUFPWSxPQUFQLENBQWVaLHFCQUFPRyxNQUFQLEVBQWYsQ0FKQTtBQUtsQlUsUUFBQUEsV0FBVyxFQUFFYixxQkFBT1ksT0FBUCxDQUFlWixxQkFBT0csTUFBUCxFQUFmLENBTEs7QUFNbEJXLFFBQUFBLGlCQUFpQixFQUFFZCxxQkFBT1MsS0FBUCxDQUFhVCxxQkFBT0csTUFBUCxFQUFiO0FBTkQsT0FBZDtBQURFO0FBRlosR0FERixFQWNFLE9BQ0V0QixPQURGLEVBRUVDLE9BRkYsRUFHRUMsUUFIRixLQUlrRTtBQUNoRSxVQUFNQyxnQkFBNEMsR0FBR0gsT0FBTyxDQUFDSSxvQkFBUixDQUE2QkMsbUJBQTdCLENBQWlEQyxRQUFqRCxDQUNuREwsT0FEbUQsQ0FBckQ7O0FBSUEsUUFBSTtBQUNGLFlBQU1pQyxRQUFRLEdBQUcsTUFBTXhDLG1CQUFtQixDQUFDeUMsWUFBcEIsQ0FDckJoQyxnQkFEcUIsRUFFckJGLE9BQU8sQ0FBQ1MsSUFBUixDQUFhZ0IsSUFGUSxFQUdyQnpCLE9BQU8sQ0FBQ1MsSUFBUixDQUFhaUIsV0FBYixJQUE0QixFQUhQLEVBSXJCMUIsT0FBTyxDQUFDUyxJQUFSLENBQWFtQixTQUpRLEVBS3JCNUIsT0FBTyxDQUFDUyxJQUFSLENBQWFvQixnQkFMUSxFQU1yQjdCLE9BQU8sQ0FBQ1MsSUFBUixDQUFhc0IsV0FOUSxFQU9yQi9CLE9BQU8sQ0FBQ1MsSUFBUixDQUFhdUIsaUJBQWIsSUFBa0MsRUFQYixDQUF2QjtBQVNBLGFBQU8vQixRQUFRLENBQUNPLEVBQVQsQ0FBWTtBQUNqQkMsUUFBQUEsSUFBSSxFQUFFO0FBQ0pPLFVBQUFBLE9BQU8sRUFBRSxxQkFETDtBQUVKaUIsVUFBQUE7QUFGSTtBQURXLE9BQVosQ0FBUDtBQU1ELEtBaEJELENBZ0JFLE9BQU90QixHQUFQLEVBQWlCO0FBQ2pCQyxNQUFBQSxPQUFPLENBQUNDLEtBQVIsQ0FBYyxpREFBZCxFQUFpRUYsR0FBakU7QUFDQSxhQUFPVixRQUFRLENBQUNhLE1BQVQsQ0FBZ0I7QUFDckJDLFFBQUFBLFVBQVUsRUFBRUosR0FBRyxDQUFDSSxVQUFKLElBQWtCLEdBRFQ7QUFFckJOLFFBQUFBLElBQUksRUFBRUUsR0FBRyxDQUFDSztBQUZXLE9BQWhCLENBQVA7QUFJRDtBQUNGLEdBOUNILEVBeEUwRCxDQXlIMUQ7O0FBQ0F4QixFQUFBQSxNQUFNLENBQUMyQyxHQUFQLENBQ0U7QUFDRXZDLElBQUFBLElBQUksRUFBRyxHQUFFQywrQ0FBVyxTQUR0QjtBQUVFQyxJQUFBQSxRQUFRLEVBQUU7QUFDUlcsTUFBQUEsSUFBSSxFQUFFUyxxQkFBT0MsTUFBUCxDQUFjO0FBQ2xCQyxRQUFBQSxLQUFLLEVBQUVGLHFCQUFPRyxNQUFQLEVBRFc7QUFFbEJJLFFBQUFBLElBQUksRUFBRVAscUJBQU9HLE1BQVA7QUFGWSxPQUFkO0FBREU7QUFGWixHQURGLEVBVUUsT0FDRXRCLE9BREYsRUFFRUMsT0FGRixFQUdFQyxRQUhGLEtBSWtFO0FBQ2hFLFVBQU1DLGdCQUE0QyxHQUFHSCxPQUFPLENBQUNJLG9CQUFSLENBQTZCQyxtQkFBN0IsQ0FBaURDLFFBQWpELENBQ25ETCxPQURtRCxDQUFyRDs7QUFJQSxRQUFJO0FBQ0YsWUFBTVAsbUJBQW1CLENBQUMyQyxTQUFwQixDQUNKbEMsZ0JBREksRUFFSkYsT0FBTyxDQUFDUyxJQUFSLENBQWFXLEtBRlQsRUFHSnBCLE9BQU8sQ0FBQ1MsSUFBUixDQUFhZ0IsSUFIVCxDQUFOO0FBS0EsYUFBT3hCLFFBQVEsQ0FBQ08sRUFBVCxDQUFZO0FBQ2pCQyxRQUFBQSxJQUFJLEVBQUU7QUFDSk8sVUFBQUEsT0FBTyxFQUFFO0FBREw7QUFEVyxPQUFaLENBQVA7QUFLRCxLQVhELENBV0UsT0FBT0wsR0FBUCxFQUFpQjtBQUNqQkMsTUFBQUEsT0FBTyxDQUFDQyxLQUFSLENBQWMsdURBQWQsRUFBdUVGLEdBQXZFO0FBQ0EsYUFBT1YsUUFBUSxDQUFDYSxNQUFULENBQWdCO0FBQ3JCQyxRQUFBQSxVQUFVLEVBQUVKLEdBQUcsQ0FBQ0ksVUFBSixJQUFrQixHQURUO0FBRXJCTixRQUFBQSxJQUFJLEVBQUVFLEdBQUcsQ0FBQ0s7QUFGVyxPQUFoQixDQUFQO0FBSUQ7QUFDRixHQXJDSCxFQTFIMEQsQ0FrSzFEOztBQUNBeEIsRUFBQUEsTUFBTSxDQUFDMkMsR0FBUCxDQUNFO0FBQ0V2QyxJQUFBQSxJQUFJLEVBQUcsR0FBRUMsK0NBQVcsR0FEdEI7QUFFRUMsSUFBQUEsUUFBUSxFQUFFO0FBQ1JXLE1BQUFBLElBQUksRUFBRVMscUJBQU9DLE1BQVAsQ0FBYztBQUNsQkMsUUFBQUEsS0FBSyxFQUFFRixxQkFBT0csTUFBUCxFQURXO0FBRWxCZ0IsUUFBQUEsVUFBVSxFQUFFbkIscUJBQU9DLE1BQVAsQ0FBYztBQUN4Qk0sVUFBQUEsSUFBSSxFQUFFUCxxQkFBT1MsS0FBUCxDQUFhVCxxQkFBT0csTUFBUCxFQUFiLENBRGtCO0FBRXhCSyxVQUFBQSxXQUFXLEVBQUVSLHFCQUFPUyxLQUFQLENBQWFULHFCQUFPRyxNQUFQLEVBQWIsQ0FGVztBQUd4QlEsVUFBQUEsZ0JBQWdCLEVBQUVYLHFCQUFPUyxLQUFQLENBQWFULHFCQUFPWSxPQUFQLENBQWVaLHFCQUFPRyxNQUFQLEVBQWYsQ0FBYixDQUhNO0FBSXhCVSxVQUFBQSxXQUFXLEVBQUViLHFCQUFPUyxLQUFQLENBQWFULHFCQUFPWSxPQUFQLENBQWVaLHFCQUFPRyxNQUFQLEVBQWYsQ0FBYixDQUpXO0FBS3hCaUIsVUFBQUEsT0FBTyxFQUFFcEIscUJBQU9TLEtBQVAsQ0FBYVQscUJBQU9HLE1BQVAsRUFBYixDQUxlO0FBTXhCVyxVQUFBQSxpQkFBaUIsRUFBRWQscUJBQU9TLEtBQVAsQ0FBYVQscUJBQU9HLE1BQVAsRUFBYjtBQU5LLFNBQWQ7QUFGTSxPQUFkO0FBREU7QUFGWixHQURGLEVBaUJFLE9BQ0V0QixPQURGLEVBRUVDLE9BRkYsRUFHRUMsUUFIRixLQUlrRTtBQUNoRSxVQUFNQyxnQkFBNEMsR0FBR0gsT0FBTyxDQUFDSSxvQkFBUixDQUE2QkMsbUJBQTdCLENBQWlEQyxRQUFqRCxDQUNuREwsT0FEbUQsQ0FBckQ7O0FBSUEsUUFBSTtBQUNGLFlBQU11QyxZQUFZLEdBQUcsTUFBTTlDLG1CQUFtQixDQUFDK0MsU0FBcEIsQ0FDekJ0QyxnQkFEeUIsRUFFekJGLE9BQU8sQ0FBQ1MsSUFBUixDQUFhVyxLQUZZLEVBR3pCcEIsT0FBTyxDQUFDUyxJQUFSLENBQWE0QixVQUhZLENBQTNCO0FBS0EsYUFBT3BDLFFBQVEsQ0FBQ08sRUFBVCxDQUFZO0FBQ2pCQyxRQUFBQSxJQUFJLEVBQUU7QUFDSk8sVUFBQUEsT0FBTyxFQUFFLHFCQURMO0FBRUp1QixVQUFBQTtBQUZJO0FBRFcsT0FBWixDQUFQO0FBTUQsS0FaRCxDQVlFLE9BQU81QixHQUFQLEVBQWlCO0FBQ2pCQyxNQUFBQSxPQUFPLENBQUNDLEtBQVIsQ0FBYyx1REFBZCxFQUF1RUYsR0FBdkU7QUFDQSxhQUFPVixRQUFRLENBQUNhLE1BQVQsQ0FBZ0I7QUFDckJDLFFBQUFBLFVBQVUsRUFBRUosR0FBRyxDQUFDSSxVQUFKLElBQWtCLEdBRFQ7QUFFckJOLFFBQUFBLElBQUksRUFBRUUsR0FBRyxDQUFDSztBQUZXLE9BQWhCLENBQVA7QUFJRDtBQUNGLEdBN0NILEVBbkswRCxDQW1OMUQ7O0FBQ0F4QixFQUFBQSxNQUFNLENBQUNpRCxNQUFQLENBQ0U7QUFDRTdDLElBQUFBLElBQUksRUFBRyxHQUFFQywrQ0FBVyxZQUR0QjtBQUVFQyxJQUFBQSxRQUFRLEVBQUU7QUFDUm1CLE1BQUFBLE1BQU0sRUFBRUMscUJBQU9DLE1BQVAsQ0FBYztBQUNwQnVCLFFBQUFBLE9BQU8sRUFBRXhCLHFCQUFPRyxNQUFQO0FBRFcsT0FBZDtBQURBO0FBRlosR0FERixFQVNFLE9BQ0V0QixPQURGLEVBRUVDLE9BRkYsRUFHRUMsUUFIRixLQUlrRTtBQUNoRSxVQUFNQyxnQkFBNEMsR0FBR0gsT0FBTyxDQUFDSSxvQkFBUixDQUE2QkMsbUJBQTdCLENBQWlEQyxRQUFqRCxDQUNuREwsT0FEbUQsQ0FBckQ7O0FBR0EsUUFBSTtBQUNGLFlBQU0yQyxXQUFXLEdBQUcsTUFBTWxELG1CQUFtQixDQUFDbUQsU0FBcEIsQ0FDeEIxQyxnQkFEd0IsRUFFeEJGLE9BQU8sQ0FBQ2lCLE1BQVIsQ0FBZXlCLE9BRlMsQ0FBMUI7QUFJQSxhQUFPekMsUUFBUSxDQUFDTyxFQUFULENBQVk7QUFDakJDLFFBQUFBLElBQUksRUFBRWtDO0FBRFcsT0FBWixDQUFQO0FBR0QsS0FSRCxDQVFFLE9BQU9oQyxHQUFQLEVBQWlCO0FBQ2pCLGFBQU9WLFFBQVEsQ0FBQ2EsTUFBVCxDQUFnQjtBQUNyQkMsUUFBQUEsVUFBVSxFQUFFSixHQUFHLENBQUNJLFVBQUosSUFBa0IsR0FEVDtBQUVyQk4sUUFBQUEsSUFBSSxFQUFFRSxHQUFHLENBQUNLO0FBRlcsT0FBaEIsQ0FBUDtBQUlEO0FBQ0YsR0EvQkg7QUFpQ0QiLCJzb3VyY2VzQ29udGVudCI6WyIvKlxuICogQ29weXJpZ2h0IE9wZW5TZWFyY2ggQ29udHJpYnV0b3JzXG4gKiBTUERYLUxpY2Vuc2UtSWRlbnRpZmllcjogQXBhY2hlLTIuMFxuICovXG4vKiBlc2xpbnQtZGlzYWJsZSBuby1jb25zb2xlICovXG5cbmltcG9ydCB7IHNjaGVtYSB9IGZyb20gJ0Bvc2QvY29uZmlnLXNjaGVtYSc7XG5pbXBvcnQge1xuICBJUm91dGVyLFxuICBJT3BlblNlYXJjaERhc2hib2FyZHNSZXNwb25zZSxcbiAgUmVzcG9uc2VFcnJvcixcbiAgSUxlZ2FjeVNjb3BlZENsdXN0ZXJDbGllbnQsXG59IGZyb20gJy4uLy4uLy4uLy4uLy4uL3NyYy9jb3JlL3NlcnZlcic7XG5pbXBvcnQgeyBBUFBfQU5BTFlUSUNTX0FQSV9QUkVGSVggYXMgQVBJX1BSRUZJWCB9IGZyb20gJy4uLy4uLy4uL2NvbW1vbi9jb25zdGFudHMvYXBwbGljYXRpb25fYW5hbHl0aWNzJztcbmltcG9ydCB7IEFwcGxpY2F0aW9uTGlzdFR5cGUgfSBmcm9tICcuLi8uLi8uLi9jb21tb24vdHlwZXMvYXBwX2FuYWx5dGljcyc7XG5pbXBvcnQgeyBBcHBBbmFseXRpY3NBZGFwdG9yIH0gZnJvbSAnLi4vLi4vLi4vc2VydmVyL2FkYXB0b3JzL2FwcGxpY2F0aW9uX2FuYWx5dGljcy9hcHBfYW5hbHl0aWNzX2FkYXB0b3InO1xuXG5leHBvcnQgZnVuY3Rpb24gcmVnaXN0ZXJBcHBBbmFseXRpY3NSb3V0ZXIocm91dGVyOiBJUm91dGVyKSB7XG4gIGNvbnN0IGFwcEFuYWx5dGljc0JhY2tlbmQgPSBuZXcgQXBwQW5hbHl0aWNzQWRhcHRvcigpO1xuXG4gIC8vIEZldGNoZXMgYWxsIGV4aXN0aW5nIGFwcGxpY2F0aW9uc1xuICByb3V0ZXIuZ2V0KFxuICAgIHtcbiAgICAgIHBhdGg6IGAke0FQSV9QUkVGSVh9L2AsXG4gICAgICB2YWxpZGF0ZToge30sXG4gICAgfSxcbiAgICBhc3luYyAoXG4gICAgICBjb250ZXh0LFxuICAgICAgcmVxdWVzdCxcbiAgICAgIHJlc3BvbnNlXG4gICAgKTogUHJvbWlzZTxJT3BlblNlYXJjaERhc2hib2FyZHNSZXNwb25zZTxhbnkgfCBSZXNwb25zZUVycm9yPj4gPT4ge1xuICAgICAgY29uc3Qgb3BlbnNlYXJjaENsaWVudDogSUxlZ2FjeVNjb3BlZENsdXN0ZXJDbGllbnQgPSBjb250ZXh0Lm9ic2VydmFiaWxpdHlfcGx1Z2luLm9ic2VydmFiaWxpdHlDbGllbnQuYXNTY29wZWQoXG4gICAgICAgIHJlcXVlc3RcbiAgICAgICk7XG4gICAgICBsZXQgYXBwbGljYXRpb25zRGF0YTogQXBwbGljYXRpb25MaXN0VHlwZVtdID0gW107XG4gICAgICB0cnkge1xuICAgICAgICBhcHBsaWNhdGlvbnNEYXRhID0gYXdhaXQgYXBwQW5hbHl0aWNzQmFja2VuZC5mZXRjaEFwcHMob3BlbnNlYXJjaENsaWVudCk7XG4gICAgICAgIHJldHVybiByZXNwb25zZS5vayh7XG4gICAgICAgICAgYm9keToge1xuICAgICAgICAgICAgZGF0YTogYXBwbGljYXRpb25zRGF0YSxcbiAgICAgICAgICB9LFxuICAgICAgICB9KTtcbiAgICAgIH0gY2F0Y2ggKGVycjogYW55KSB7XG4gICAgICAgIGNvbnNvbGUuZXJyb3IoJ0Vycm9yIG9jY3VycmVkIHdoaWxlIGZldGNoaW5nIGFwcGxpY2F0aW9ucycsIGVycik7XG4gICAgICAgIHJldHVybiByZXNwb25zZS5jdXN0b20oe1xuICAgICAgICAgIHN0YXR1c0NvZGU6IGVyci5zdGF0dXNDb2RlIHx8IDUwMCxcbiAgICAgICAgICBib2R5OiBlcnIubWVzc2FnZSxcbiAgICAgICAgfSk7XG4gICAgICB9XG4gICAgfVxuICApO1xuXG4gIC8vIEZldGNoIGFwcGxpY2F0aW9uIGJ5IGlkXG4gIHJvdXRlci5nZXQoXG4gICAge1xuICAgICAgcGF0aDogYCR7QVBJX1BSRUZJWH0ve2FwcElkfWAsXG4gICAgICB2YWxpZGF0ZToge1xuICAgICAgICBwYXJhbXM6IHNjaGVtYS5vYmplY3Qoe1xuICAgICAgICAgIGFwcElkOiBzY2hlbWEuc3RyaW5nKCksXG4gICAgICAgIH0pLFxuICAgICAgfSxcbiAgICB9LFxuICAgIGFzeW5jIChcbiAgICAgIGNvbnRleHQsXG4gICAgICByZXF1ZXN0LFxuICAgICAgcmVzcG9uc2VcbiAgICApOiBQcm9taXNlPElPcGVuU2VhcmNoRGFzaGJvYXJkc1Jlc3BvbnNlPGFueSB8IFJlc3BvbnNlRXJyb3I+PiA9PiB7XG4gICAgICBjb25zdCBvcGVuc2VhcmNoQ2xpZW50OiBJTGVnYWN5U2NvcGVkQ2x1c3RlckNsaWVudCA9IGNvbnRleHQub2JzZXJ2YWJpbGl0eV9wbHVnaW4ub2JzZXJ2YWJpbGl0eUNsaWVudC5hc1Njb3BlZChcbiAgICAgICAgcmVxdWVzdFxuICAgICAgKTtcbiAgICAgIHRyeSB7XG4gICAgICAgIGNvbnN0IGFwcE9iamVjdCA9IGF3YWl0IGFwcEFuYWx5dGljc0JhY2tlbmQuZmV0Y2hBcHBCeUlkKFxuICAgICAgICAgIG9wZW5zZWFyY2hDbGllbnQsXG4gICAgICAgICAgcmVxdWVzdC5wYXJhbXMuYXBwSWRcbiAgICAgICAgKTtcbiAgICAgICAgcmV0dXJuIHJlc3BvbnNlLm9rKHtcbiAgICAgICAgICBib2R5OiBhcHBPYmplY3QsXG4gICAgICAgIH0pO1xuICAgICAgfSBjYXRjaCAoZXJyOiBhbnkpIHtcbiAgICAgICAgY29uc29sZS5lcnJvcignRXJyb3Igb2NjdXJyZWQgd2hpbGUgZmV0Y2hpbmcgYXBwbGljYXRpb24nLCBlcnIpO1xuICAgICAgICByZXR1cm4gcmVzcG9uc2UuY3VzdG9tKHtcbiAgICAgICAgICBzdGF0dXNDb2RlOiBlcnIuc3RhdHVzQ29kZSB8fCA1MDAsXG4gICAgICAgICAgYm9keTogZXJyLm1lc3NhZ2UsXG4gICAgICAgIH0pO1xuICAgICAgfVxuICAgIH1cbiAgKTtcblxuICAvLyBDcmVhdGUgYSBuZXcgYXBwbGljYXRpb25cbiAgcm91dGVyLnBvc3QoXG4gICAge1xuICAgICAgcGF0aDogYCR7QVBJX1BSRUZJWH0vYCxcbiAgICAgIHZhbGlkYXRlOiB7XG4gICAgICAgIGJvZHk6IHNjaGVtYS5vYmplY3Qoe1xuICAgICAgICAgIG5hbWU6IHNjaGVtYS5zdHJpbmcoKSxcbiAgICAgICAgICBkZXNjcmlwdGlvbjogc2NoZW1hLm1heWJlKHNjaGVtYS5zdHJpbmcoKSksXG4gICAgICAgICAgYmFzZVF1ZXJ5OiBzY2hlbWEuc3RyaW5nKCksXG4gICAgICAgICAgc2VydmljZXNFbnRpdGllczogc2NoZW1hLmFycmF5T2Yoc2NoZW1hLnN0cmluZygpKSxcbiAgICAgICAgICB0cmFjZUdyb3Vwczogc2NoZW1hLmFycmF5T2Yoc2NoZW1hLnN0cmluZygpKSxcbiAgICAgICAgICBhdmFpbGFiaWxpdHlWaXNJZDogc2NoZW1hLm1heWJlKHNjaGVtYS5zdHJpbmcoKSksXG4gICAgICAgIH0pLFxuICAgICAgfSxcbiAgICB9LFxuICAgIGFzeW5jIChcbiAgICAgIGNvbnRleHQsXG4gICAgICByZXF1ZXN0LFxuICAgICAgcmVzcG9uc2VcbiAgICApOiBQcm9taXNlPElPcGVuU2VhcmNoRGFzaGJvYXJkc1Jlc3BvbnNlPGFueSB8IFJlc3BvbnNlRXJyb3I+PiA9PiB7XG4gICAgICBjb25zdCBvcGVuc2VhcmNoQ2xpZW50OiBJTGVnYWN5U2NvcGVkQ2x1c3RlckNsaWVudCA9IGNvbnRleHQub2JzZXJ2YWJpbGl0eV9wbHVnaW4ub2JzZXJ2YWJpbGl0eUNsaWVudC5hc1Njb3BlZChcbiAgICAgICAgcmVxdWVzdFxuICAgICAgKTtcblxuICAgICAgdHJ5IHtcbiAgICAgICAgY29uc3QgbmV3QXBwSWQgPSBhd2FpdCBhcHBBbmFseXRpY3NCYWNrZW5kLmNyZWF0ZU5ld0FwcChcbiAgICAgICAgICBvcGVuc2VhcmNoQ2xpZW50LFxuICAgICAgICAgIHJlcXVlc3QuYm9keS5uYW1lLFxuICAgICAgICAgIHJlcXVlc3QuYm9keS5kZXNjcmlwdGlvbiB8fCAnJyxcbiAgICAgICAgICByZXF1ZXN0LmJvZHkuYmFzZVF1ZXJ5LFxuICAgICAgICAgIHJlcXVlc3QuYm9keS5zZXJ2aWNlc0VudGl0aWVzLFxuICAgICAgICAgIHJlcXVlc3QuYm9keS50cmFjZUdyb3VwcyxcbiAgICAgICAgICByZXF1ZXN0LmJvZHkuYXZhaWxhYmlsaXR5VmlzSWQgfHwgJydcbiAgICAgICAgKTtcbiAgICAgICAgcmV0dXJuIHJlc3BvbnNlLm9rKHtcbiAgICAgICAgICBib2R5OiB7XG4gICAgICAgICAgICBtZXNzYWdlOiAnQXBwbGljYXRpb24gQ3JlYXRlZCcsXG4gICAgICAgICAgICBuZXdBcHBJZCxcbiAgICAgICAgICB9LFxuICAgICAgICB9KTtcbiAgICAgIH0gY2F0Y2ggKGVycjogYW55KSB7XG4gICAgICAgIGNvbnNvbGUuZXJyb3IoJ0Vycm9yIG9jY3VycmVkIHdoaWxlIGNyZWF0aW5nIGEgbmV3IGFwcGxpY2F0aW9uJywgZXJyKTtcbiAgICAgICAgcmV0dXJuIHJlc3BvbnNlLmN1c3RvbSh7XG4gICAgICAgICAgc3RhdHVzQ29kZTogZXJyLnN0YXR1c0NvZGUgfHwgNTAwLFxuICAgICAgICAgIGJvZHk6IGVyci5tZXNzYWdlLFxuICAgICAgICB9KTtcbiAgICAgIH1cbiAgICB9XG4gICk7XG5cbiAgLy8gUmVuYW1lcyBhbiBleGlzdGluZyBhcHBsaWNhdGlvblxuICByb3V0ZXIucHV0KFxuICAgIHtcbiAgICAgIHBhdGg6IGAke0FQSV9QUkVGSVh9L3JlbmFtZWAsXG4gICAgICB2YWxpZGF0ZToge1xuICAgICAgICBib2R5OiBzY2hlbWEub2JqZWN0KHtcbiAgICAgICAgICBhcHBJZDogc2NoZW1hLnN0cmluZygpLFxuICAgICAgICAgIG5hbWU6IHNjaGVtYS5zdHJpbmcoKSxcbiAgICAgICAgfSksXG4gICAgICB9LFxuICAgIH0sXG4gICAgYXN5bmMgKFxuICAgICAgY29udGV4dCxcbiAgICAgIHJlcXVlc3QsXG4gICAgICByZXNwb25zZVxuICAgICk6IFByb21pc2U8SU9wZW5TZWFyY2hEYXNoYm9hcmRzUmVzcG9uc2U8YW55IHwgUmVzcG9uc2VFcnJvcj4+ID0+IHtcbiAgICAgIGNvbnN0IG9wZW5zZWFyY2hDbGllbnQ6IElMZWdhY3lTY29wZWRDbHVzdGVyQ2xpZW50ID0gY29udGV4dC5vYnNlcnZhYmlsaXR5X3BsdWdpbi5vYnNlcnZhYmlsaXR5Q2xpZW50LmFzU2NvcGVkKFxuICAgICAgICByZXF1ZXN0XG4gICAgICApO1xuXG4gICAgICB0cnkge1xuICAgICAgICBhd2FpdCBhcHBBbmFseXRpY3NCYWNrZW5kLnJlbmFtZUFwcChcbiAgICAgICAgICBvcGVuc2VhcmNoQ2xpZW50LFxuICAgICAgICAgIHJlcXVlc3QuYm9keS5hcHBJZCxcbiAgICAgICAgICByZXF1ZXN0LmJvZHkubmFtZVxuICAgICAgICApO1xuICAgICAgICByZXR1cm4gcmVzcG9uc2Uub2soe1xuICAgICAgICAgIGJvZHk6IHtcbiAgICAgICAgICAgIG1lc3NhZ2U6ICdBcHBsaWNhdGlvbiBSZW5hbWVkJyxcbiAgICAgICAgICB9LFxuICAgICAgICB9KTtcbiAgICAgIH0gY2F0Y2ggKGVycjogYW55KSB7XG4gICAgICAgIGNvbnNvbGUuZXJyb3IoJ0Vycm9yIG9jY3VycmVkIHdoaWxlIHJlbmFtaW5nIGFuIGV4aXN0aW5nIGFwcGxpY2F0aW9uJywgZXJyKTtcbiAgICAgICAgcmV0dXJuIHJlc3BvbnNlLmN1c3RvbSh7XG4gICAgICAgICAgc3RhdHVzQ29kZTogZXJyLnN0YXR1c0NvZGUgfHwgNTAwLFxuICAgICAgICAgIGJvZHk6IGVyci5tZXNzYWdlLFxuICAgICAgICB9KTtcbiAgICAgIH1cbiAgICB9XG4gICk7XG5cbiAgLy8gVXBkYXRlcyBhbiBleGlzdGluZyBhcHBsaWNhdGlvblxuICByb3V0ZXIucHV0KFxuICAgIHtcbiAgICAgIHBhdGg6IGAke0FQSV9QUkVGSVh9L2AsXG4gICAgICB2YWxpZGF0ZToge1xuICAgICAgICBib2R5OiBzY2hlbWEub2JqZWN0KHtcbiAgICAgICAgICBhcHBJZDogc2NoZW1hLnN0cmluZygpLFxuICAgICAgICAgIHVwZGF0ZUJvZHk6IHNjaGVtYS5vYmplY3Qoe1xuICAgICAgICAgICAgbmFtZTogc2NoZW1hLm1heWJlKHNjaGVtYS5zdHJpbmcoKSksXG4gICAgICAgICAgICBkZXNjcmlwdGlvbjogc2NoZW1hLm1heWJlKHNjaGVtYS5zdHJpbmcoKSksXG4gICAgICAgICAgICBzZXJ2aWNlc0VudGl0aWVzOiBzY2hlbWEubWF5YmUoc2NoZW1hLmFycmF5T2Yoc2NoZW1hLnN0cmluZygpKSksXG4gICAgICAgICAgICB0cmFjZUdyb3Vwczogc2NoZW1hLm1heWJlKHNjaGVtYS5hcnJheU9mKHNjaGVtYS5zdHJpbmcoKSkpLFxuICAgICAgICAgICAgcGFuZWxJZDogc2NoZW1hLm1heWJlKHNjaGVtYS5zdHJpbmcoKSksXG4gICAgICAgICAgICBhdmFpbGFiaWxpdHlWaXNJZDogc2NoZW1hLm1heWJlKHNjaGVtYS5zdHJpbmcoKSksXG4gICAgICAgICAgfSksXG4gICAgICAgIH0pLFxuICAgICAgfSxcbiAgICB9LFxuICAgIGFzeW5jIChcbiAgICAgIGNvbnRleHQsXG4gICAgICByZXF1ZXN0LFxuICAgICAgcmVzcG9uc2VcbiAgICApOiBQcm9taXNlPElPcGVuU2VhcmNoRGFzaGJvYXJkc1Jlc3BvbnNlPGFueSB8IFJlc3BvbnNlRXJyb3I+PiA9PiB7XG4gICAgICBjb25zdCBvcGVuc2VhcmNoQ2xpZW50OiBJTGVnYWN5U2NvcGVkQ2x1c3RlckNsaWVudCA9IGNvbnRleHQub2JzZXJ2YWJpbGl0eV9wbHVnaW4ub2JzZXJ2YWJpbGl0eUNsaWVudC5hc1Njb3BlZChcbiAgICAgICAgcmVxdWVzdFxuICAgICAgKTtcblxuICAgICAgdHJ5IHtcbiAgICAgICAgY29uc3QgdXBkYXRlZEFwcElkID0gYXdhaXQgYXBwQW5hbHl0aWNzQmFja2VuZC51cGRhdGVBcHAoXG4gICAgICAgICAgb3BlbnNlYXJjaENsaWVudCxcbiAgICAgICAgICByZXF1ZXN0LmJvZHkuYXBwSWQsXG4gICAgICAgICAgcmVxdWVzdC5ib2R5LnVwZGF0ZUJvZHlcbiAgICAgICAgKTtcbiAgICAgICAgcmV0dXJuIHJlc3BvbnNlLm9rKHtcbiAgICAgICAgICBib2R5OiB7XG4gICAgICAgICAgICBtZXNzYWdlOiAnQXBwbGljYXRpb24gVXBkYXRlZCcsXG4gICAgICAgICAgICB1cGRhdGVkQXBwSWQsXG4gICAgICAgICAgfSxcbiAgICAgICAgfSk7XG4gICAgICB9IGNhdGNoIChlcnI6IGFueSkge1xuICAgICAgICBjb25zb2xlLmVycm9yKCdFcnJvciBvY2N1cnJlZCB3aGlsZSB1cGRhdGluZyBhbiBleGlzdGluZyBhcHBsaWNhdGlvbicsIGVycik7XG4gICAgICAgIHJldHVybiByZXNwb25zZS5jdXN0b20oe1xuICAgICAgICAgIHN0YXR1c0NvZGU6IGVyci5zdGF0dXNDb2RlIHx8IDUwMCxcbiAgICAgICAgICBib2R5OiBlcnIubWVzc2FnZSxcbiAgICAgICAgfSk7XG4gICAgICB9XG4gICAgfVxuICApO1xuXG4gIC8vIERlbGV0ZSBhcHBsaWNhdGlvbnNcbiAgcm91dGVyLmRlbGV0ZShcbiAgICB7XG4gICAgICBwYXRoOiBgJHtBUElfUFJFRklYfS97YXBwTGlzdH1gLFxuICAgICAgdmFsaWRhdGU6IHtcbiAgICAgICAgcGFyYW1zOiBzY2hlbWEub2JqZWN0KHtcbiAgICAgICAgICBhcHBMaXN0OiBzY2hlbWEuc3RyaW5nKCksXG4gICAgICAgIH0pLFxuICAgICAgfSxcbiAgICB9LFxuICAgIGFzeW5jIChcbiAgICAgIGNvbnRleHQsXG4gICAgICByZXF1ZXN0LFxuICAgICAgcmVzcG9uc2VcbiAgICApOiBQcm9taXNlPElPcGVuU2VhcmNoRGFzaGJvYXJkc1Jlc3BvbnNlPGFueSB8IFJlc3BvbnNlRXJyb3I+PiA9PiB7XG4gICAgICBjb25zdCBvcGVuc2VhcmNoQ2xpZW50OiBJTGVnYWN5U2NvcGVkQ2x1c3RlckNsaWVudCA9IGNvbnRleHQub2JzZXJ2YWJpbGl0eV9wbHVnaW4ub2JzZXJ2YWJpbGl0eUNsaWVudC5hc1Njb3BlZChcbiAgICAgICAgcmVxdWVzdFxuICAgICAgKTtcbiAgICAgIHRyeSB7XG4gICAgICAgIGNvbnN0IGRlbFJlc3BvbnNlID0gYXdhaXQgYXBwQW5hbHl0aWNzQmFja2VuZC5kZWxldGVBcHAoXG4gICAgICAgICAgb3BlbnNlYXJjaENsaWVudCxcbiAgICAgICAgICByZXF1ZXN0LnBhcmFtcy5hcHBMaXN0XG4gICAgICAgICk7XG4gICAgICAgIHJldHVybiByZXNwb25zZS5vayh7XG4gICAgICAgICAgYm9keTogZGVsUmVzcG9uc2UsXG4gICAgICAgIH0pO1xuICAgICAgfSBjYXRjaCAoZXJyOiBhbnkpIHtcbiAgICAgICAgcmV0dXJuIHJlc3BvbnNlLmN1c3RvbSh7XG4gICAgICAgICAgc3RhdHVzQ29kZTogZXJyLnN0YXR1c0NvZGUgfHwgNTAwLFxuICAgICAgICAgIGJvZHk6IGVyci5tZXNzYWdlLFxuICAgICAgICB9KTtcbiAgICAgIH1cbiAgICB9XG4gICk7XG59XG4iXX0=