"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createVisualReport = void 0;

var _puppeteerCore = _interopRequireDefault(require("puppeteer-core"));

var _dompurify = _interopRequireDefault(require("dompurify"));

var _jsdom = require("jsdom");

var _constants = require("../constants");

var _helpers = require("../helpers");

var _fs = _interopRequireDefault(require("fs"));

var _lodash = _interopRequireDefault(require("lodash"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */
const createVisualReport = async (reportParams, queryUrl, logger, extraHeaders, timezone) => {
  const {
    core_params,
    report_name: reportName,
    report_source: reportSource
  } = reportParams;
  const coreParams = core_params;
  const {
    header,
    footer,
    window_height: windowHeight,
    window_width: windowWidth,
    report_format: reportFormat
  } = coreParams;
  const window = new _jsdom.JSDOM('').window;
  const DOMPurify = (0, _dompurify.default)(window);
  const reportHeader = header ? DOMPurify.sanitize(header) : _constants.DEFAULT_REPORT_HEADER;
  const reportFooter = footer ? DOMPurify.sanitize(footer) : ''; // set up puppeteer

  const browser = await _puppeteerCore.default.launch({
    headless: true,

    /**
     * TODO: temp fix to disable sandbox when launching chromium on Linux instance
     * https://github.com/puppeteer/puppeteer/blob/main/docs/troubleshooting.md#setting-up-chrome-linux-sandbox
     */
    args: ['--no-sandbox', '--disable-setuid-sandbox', '--disable-gpu', '--no-zygote', '--single-process', '--font-render-hinting=none'],
    executablePath: _constants.CHROMIUM_PATH,
    ignoreHTTPSErrors: true,
    env: {
      TZ: timezone || 'UTC'
    }
  });
  const page = await browser.newPage();
  page.setDefaultNavigationTimeout(0);
  page.setDefaultTimeout(100000); // use 100s timeout instead of default 30s
  // Set extra headers that are needed

  if (!_lodash.default.isEmpty(extraHeaders)) {
    await page.setExtraHTTPHeaders(extraHeaders);
  }

  logger.info(`original queryUrl ${queryUrl}`);
  await page.goto(queryUrl, {
    waitUntil: 'networkidle0'
  }); // should add to local storage after page.goto, then access the page again - browser must have an url to register local storage item on it

  await page.evaluate(
  /* istanbul ignore next */
  key => {
    localStorage.setItem(key, 'false');
  }, _constants.SECURITY_CONSTANTS.TENANT_LOCAL_STORAGE_KEY);
  await page.goto(queryUrl, {
    waitUntil: 'networkidle0'
  });
  logger.info(`page url ${page.url()}`);
  await page.setViewport({
    width: windowWidth,
    height: windowHeight
  });
  let buffer; // remove unwanted elements

  await page.evaluate(
  /* istanbul ignore next */
  (reportSource, REPORT_TYPE) => {
    // remove buttons
    document.querySelectorAll("[class^='euiButton']").forEach(e => e.remove()); // remove top navBar

    document.querySelectorAll("[class^='euiHeader']").forEach(e => e.remove()); // remove visualization editor

    if (reportSource === REPORT_TYPE.visualization) {
      var _document$querySelect, _document$querySelect2;

      (_document$querySelect = document.querySelector('[data-test-subj="splitPanelResizer"]')) === null || _document$querySelect === void 0 ? void 0 : _document$querySelect.remove();
      (_document$querySelect2 = document.querySelector('.visEditor__collapsibleSidebar')) === null || _document$querySelect2 === void 0 ? void 0 : _document$querySelect2.remove();
    }

    document.body.style.paddingTop = '0px';
  }, reportSource, _constants.REPORT_TYPE); // force wait for any resize to load after the above DOM modification

  await page.waitFor(1000); // crop content

  switch (reportSource) {
    case _constants.REPORT_TYPE.dashboard:
      await page.waitForSelector(_constants.SELECTOR.dashboard, {
        visible: true
      });
      break;

    case _constants.REPORT_TYPE.visualization:
      await page.waitForSelector(_constants.SELECTOR.visualization, {
        visible: true
      });
      break;

    case _constants.REPORT_TYPE.notebook:
      await page.waitForSelector(_constants.SELECTOR.notebook, {
        visible: true
      });
      break;

    default:
      throw Error(`report source can only be one of [Dashboard, Visualization]`);
  } // wait for dynamic page content to render


  await waitForDynamicContent(page);
  await addReportStyle(page);
  await addReportHeader(page, reportHeader);
  await addReportFooter(page, reportFooter); // create pdf or png accordingly

  if (reportFormat === _constants.FORMAT.pdf) {
    const scrollHeight = await page.evaluate(
    /* istanbul ignore next */
    () => document.documentElement.scrollHeight);
    buffer = await page.pdf({
      margin: undefined,
      width: windowWidth,
      height: scrollHeight + 'px',
      printBackground: true,
      pageRanges: '1'
    });
  } else if (reportFormat === _constants.FORMAT.png) {
    buffer = await page.screenshot({
      fullPage: true
    });
  }

  const curTime = new Date();
  const timeCreated = curTime.valueOf();
  const fileName = `${(0, _helpers.getFileName)(reportName, curTime)}.${reportFormat}`;
  await browser.close();
  return {
    timeCreated,
    dataUrl: buffer.toString('base64'),
    fileName
  };
};

exports.createVisualReport = createVisualReport;

const addReportStyle = async page => {
  const css = _fs.default.readFileSync(`${__dirname}/style.css`).toString();

  await page.evaluate(
  /* istanbul ignore next */
  style => {
    const styleElement = document.createElement('style');
    styleElement.innerHTML = style;
    document.getElementsByTagName('head')[0].appendChild(styleElement);
  }, css);
};

const addReportHeader = async (page, header) => {
  const headerHtml = _fs.default.readFileSync(`${__dirname}/header_template.html`).toString().replace('<!--CONTENT-->', header);

  await page.evaluate(
  /* istanbul ignore next */
  headerHtml => {
    var _content$parentNode;

    const content = document.body.firstChild;
    const headerContainer = document.createElement('div');
    headerContainer.className = 'reportWrapper';
    headerContainer.innerHTML = headerHtml;
    content === null || content === void 0 ? void 0 : (_content$parentNode = content.parentNode) === null || _content$parentNode === void 0 ? void 0 : _content$parentNode.insertBefore(headerContainer, content);
  }, headerHtml);
};

const addReportFooter = async (page, footer) => {
  const headerHtml = _fs.default.readFileSync(`${__dirname}/footer_template.html`).toString().replace('<!--CONTENT-->', footer);

  await page.evaluate(
  /* istanbul ignore next */
  headerHtml => {
    var _content$parentNode2;

    const content = document.body.firstChild;
    const headerContainer = document.createElement('div');
    headerContainer.className = 'reportWrapper';
    headerContainer.innerHTML = headerHtml;
    content === null || content === void 0 ? void 0 : (_content$parentNode2 = content.parentNode) === null || _content$parentNode2 === void 0 ? void 0 : _content$parentNode2.insertBefore(headerContainer, null);
  }, headerHtml);
}; // add waitForDynamicContent function


const waitForDynamicContent = async (page, timeout = 30000, interval = 1000, checks = 5) => {
  const maxChecks = timeout / interval;
  let passedChecks = 0;
  let previousLength = 0;
  let i = 0;

  while (i++ <= maxChecks) {
    let pageContent = await page.content();
    let currentLength = pageContent.length;
    previousLength === 0 || previousLength != currentLength ? passedChecks = 0 : passedChecks++;

    if (passedChecks >= checks) {
      break;
    }

    previousLength = currentLength;
    await page.waitFor(interval);
  }
};
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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