"use strict";
/*
 * SPDX-License-Identifier: Apache-2.0
 *
 * The OpenSearch Contributors require contributions made to
 * this file be licensed under the Apache-2.0 license or a
 * compatible open source license.
 *
 * Any modifications Copyright OpenSearch Contributors. See
 * GitHub history for details.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.Env = void 0;
const tslib_1 = require("tslib");
/*
 * Licensed to Elasticsearch B.V. under one or more contributor
 * license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright
 * ownership. Elasticsearch B.V. licenses this file to you under
 * the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
const path_1 = require("path");
const load_json_file_1 = tslib_1.__importDefault(require("load-json-file"));
class Env {
    /**
     * @internal
     */
    constructor(homeDir, pkg, options) {
        this.homeDir = homeDir;
        this.configDir = path_1.resolve(this.homeDir, 'config');
        this.binDir = path_1.resolve(this.homeDir, 'bin');
        this.logDir = path_1.resolve(this.homeDir, 'log');
        /**
         * BEWARE: this needs to stay roughly synchronized with the @osd/optimizer
         * `packages/osd-optimizer/src/optimizer_config.ts` determines the paths
         * that should be searched for plugins to build
         */
        this.pluginSearchPaths = [
            path_1.resolve(this.homeDir, 'src', 'plugins'),
            path_1.resolve(this.homeDir, 'plugins'),
            ...(options.cliArgs.runExamples ? [path_1.resolve(this.homeDir, 'examples')] : []),
            path_1.resolve(this.homeDir, '..', 'opensearch-dashboards-extra'),
        ];
        this.cliArgs = Object.freeze(options.cliArgs);
        this.configs = Object.freeze(options.configs);
        this.isDevClusterMaster = options.isDevClusterMaster;
        const isDevMode = this.cliArgs.dev || this.cliArgs.envName === 'development';
        this.mode = Object.freeze({
            dev: isDevMode,
            name: isDevMode ? 'development' : 'production',
            prod: !isDevMode,
        });
        const isOpenSearchDashboardsDistributable = Boolean(pkg.build && pkg.build.distributable === true);
        this.packageInfo = Object.freeze({
            branch: pkg.branch,
            buildNum: isOpenSearchDashboardsDistributable ? pkg.build.number : Number.MAX_SAFE_INTEGER,
            buildSha: isOpenSearchDashboardsDistributable
                ? pkg.build.sha
                : 'XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX',
            version: pkg.version,
            dist: isOpenSearchDashboardsDistributable,
        });
    }
    /**
     * @internal
     */
    static createDefault(repoRoot, options, pkg) {
        if (!pkg) {
            pkg = load_json_file_1.default.sync(path_1.join(repoRoot, 'package.json'));
        }
        return new Env(repoRoot, pkg, options);
    }
}
exports.Env = Env;
