'use strict';

Object.defineProperty(exports, '__esModule', { value: true });

/* global performance */

// TODO: Node's built-in performance API has poor performance for getEntriesByName()
// so currently we avoid it: https://github.com/nolanlawson/marky/issues/29
var perf = typeof performance !== 'undefined' && performance;

var now = perf && perf.now ? function () { return perf.now(); } : function () { return Date.now(); }

function throwIfEmpty (name) {
  if (!name) {
    throw new Error('name must be non-empty')
  }
}

// simple binary sort insertion
function insertSorted (arr, item) {
  var low = 0;
  var high = arr.length;
  var mid;
  while (low < high) {
    mid = (low + high) >>> 1; // like (num / 2) but faster
    if (arr[mid].startTime < item.startTime) {
      low = mid + 1;
    } else {
      high = mid;
    }
  }
  arr.splice(low, 0, item);
}

exports.mark = void 0;
exports.stop = void 0;
exports.getEntries = void 0;
exports.clear = void 0;

if (
  perf &&
  perf.mark &&
  perf.getEntriesByName &&
  perf.getEntriesByType &&
  perf.clearMeasures
) {
  exports.mark = function (name) {
    throwIfEmpty(name);
    perf.mark(("start " + name));
  };
  exports.stop = function (name) {
    throwIfEmpty(name);
    perf.mark(("end " + name));
    var measure = perf.measure(name, ("start " + name), ("end " + name));
    if (measure) {
      // return value from performance.measure not supported in all browsers
      // https://developer.mozilla.org/en-US/docs/Web/API/Performance/measure#browser_compatibility
      return measure
    }
    var entries = perf.getEntriesByName(name);
    return entries[entries.length - 1]
  };
  exports.getEntries = function () { return perf.getEntriesByType('measure'); };
  exports.clear = function () {
    perf.clearMarks();
    perf.clearMeasures();
  };
} else {
  var marks = {};
  var entries = [];
  exports.mark = function (name) {
    throwIfEmpty(name);
    var startTime = now();
    marks['$' + name] = startTime;
  };
  exports.stop = function (name) {
    throwIfEmpty(name);
    var endTime = now();
    var startTime = marks['$' + name];
    if (!startTime) {
      throw new Error(("no known mark: " + name))
    }
    var entry = {
      startTime: startTime,
      name: name,
      duration: endTime - startTime,
      entryType: 'measure'
    };
    // per the spec this should be at least 150:
    // https://www.w3.org/TR/resource-timing-1/#extensions-performance-interface
    // we just have no limit, per Chrome and Edge's de-facto behavior
    insertSorted(entries, entry);
    return entry
  };
  exports.getEntries = function () { return entries; };
  exports.clear = function () {
    marks = {};
    entries = [];
  };
}
