"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.CallFeed = exports.CallFeedEvent = void 0;

var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));

var _events = _interopRequireDefault(require("events"));

/*
Copyright 2021 Šimon Brandner <simon.bra.ag@gmail.com>

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/
const POLLING_INTERVAL = 250; // ms

const SPEAKING_THRESHOLD = -60; // dB

let CallFeedEvent;
exports.CallFeedEvent = CallFeedEvent;

(function (CallFeedEvent) {
  CallFeedEvent["NewStream"] = "new_stream";
  CallFeedEvent["MuteStateChanged"] = "mute_state_changed";
  CallFeedEvent["VolumeChanged"] = "volume_changed";
  CallFeedEvent["Speaking"] = "speaking";
})(CallFeedEvent || (exports.CallFeedEvent = CallFeedEvent = {}));

class CallFeed extends _events.default {
  constructor(stream, userId, purpose, client, roomId, audioMuted, videoMuted) {
    super();
    this.stream = stream;
    this.userId = userId;
    this.purpose = purpose;
    this.client = client;
    this.roomId = roomId;
    this.audioMuted = audioMuted;
    this.videoMuted = videoMuted;
    (0, _defineProperty2.default)(this, "measuringVolumeActivity", false);
    (0, _defineProperty2.default)(this, "audioContext", void 0);
    (0, _defineProperty2.default)(this, "analyser", void 0);
    (0, _defineProperty2.default)(this, "frequencyBinCount", void 0);
    (0, _defineProperty2.default)(this, "speakingThreshold", SPEAKING_THRESHOLD);
    (0, _defineProperty2.default)(this, "speaking", false);

    if (this.hasAudioTrack) {
      this.initVolumeMeasuring();
    }
  }

  get hasAudioTrack() {
    return this.stream.getAudioTracks().length > 0;
  }

  initVolumeMeasuring() {
    const AudioContext = window.AudioContext || window.webkitAudioContext;
    if (!this.hasAudioTrack || !AudioContext) return;
    this.audioContext = new AudioContext();
    this.analyser = this.audioContext.createAnalyser();
    this.analyser.fftSize = 512;
    this.analyser.smoothingTimeConstant = 0.1;
    const mediaStreamAudioSourceNode = this.audioContext.createMediaStreamSource(this.stream);
    mediaStreamAudioSourceNode.connect(this.analyser);
    this.frequencyBinCount = new Float32Array(this.analyser.frequencyBinCount);
  }
  /**
   * Returns callRoom member
   * @returns member of the callRoom
   */


  getMember() {
    const callRoom = this.client.getRoom(this.roomId);
    return callRoom.getMember(this.userId);
  }
  /**
   * Returns true if CallFeed is local, otherwise returns false
   * @returns {boolean} is local?
   */


  isLocal() {
    return this.userId === this.client.getUserId();
  }
  /**
   * Returns true if audio is muted or if there are no audio
   * tracks, otherwise returns false
   * @returns {boolean} is audio muted?
   */


  isAudioMuted() {
    return this.stream.getAudioTracks().length === 0 || this.audioMuted;
  }
  /**
   * Returns true video is muted or if there are no video
   * tracks, otherwise returns false
   * @returns {boolean} is video muted?
   */


  isVideoMuted() {
    // We assume only one video track
    return this.stream.getVideoTracks().length === 0 || this.videoMuted;
  }
  /**
   * Replaces the current MediaStream with a new one.
   * This method should be only used by MatrixCall.
   * @param newStream new stream with which to replace the current one
   */


  setNewStream(newStream) {
    this.stream = newStream;
    this.emit(CallFeedEvent.NewStream, this.stream);

    if (this.hasAudioTrack) {
      this.initVolumeMeasuring();
    } else {
      this.measureVolumeActivity(false);
    }
  }
  /**
   * Set feed's internal audio mute state
   * @param muted is the feed's audio muted?
   */


  setAudioMuted(muted) {
    this.audioMuted = muted;
    this.emit(CallFeedEvent.MuteStateChanged, this.audioMuted, this.videoMuted);
  }
  /**
   * Set feed's internal video mute state
   * @param muted is the feed's video muted?
   */


  setVideoMuted(muted) {
    this.videoMuted = muted;
    this.emit(CallFeedEvent.MuteStateChanged, this.audioMuted, this.videoMuted);
  }
  /**
   * Starts emitting volume_changed events where the emitter value is in decibels
   * @param enabled emit volume changes
   */


  measureVolumeActivity(enabled) {
    if (enabled) {
      if (!this.audioContext || !this.analyser || !this.frequencyBinCount || !this.hasAudioTrack) return;
      this.measuringVolumeActivity = true;
      this.volumeLooper();
    } else {
      this.measuringVolumeActivity = false;
      this.emit(CallFeedEvent.VolumeChanged, -Infinity);
    }
  }

  setSpeakingThreshold(threshold) {
    this.speakingThreshold = threshold;
  }

  volumeLooper() {
    if (!this.analyser) return;
    setTimeout(() => {
      if (!this.measuringVolumeActivity) return;
      this.analyser.getFloatFrequencyData(this.frequencyBinCount);
      let maxVolume = -Infinity;

      for (let i = 0; i < this.frequencyBinCount.length; i++) {
        if (this.frequencyBinCount[i] > maxVolume) {
          maxVolume = this.frequencyBinCount[i];
        }
      }

      this.emit(CallFeedEvent.VolumeChanged, maxVolume);
      const newSpeaking = maxVolume > this.speakingThreshold;

      if (this.speaking !== newSpeaking) {
        this.speaking = newSpeaking;
        this.emit(CallFeedEvent.Speaking, this.speaking);
      }

      this.volumeLooper();
    }, POLLING_INTERVAL);
  }

}

exports.CallFeed = CallFeed;