/*
 * Copyright (C) 2021 Apple Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1.  Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 * 2.  Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE INC. AND ITS CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL APPLE INC. OR ITS CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#if ENABLE(SERVICE_WORKER)
#include "WebSWServerToContextConnection.h"

#include "ArgumentCoders.h"
#include "Decoder.h"
#include "HandleMessage.h"
#include "WebCoreArgumentCoders.h"
#include "WebSWServerToContextConnectionMessages.h"
#include <WebCore/ExceptionData.h>
#include <WebCore/MessageWithMessagePorts.h>
#include <WebCore/ProcessQualified.h>
#include <WebCore/ScriptExecutionContextIdentifier.h>
#include <WebCore/ServiceWorkerClientData.h>
#include <WebCore/ServiceWorkerClientQueryOptions.h>
#include <WebCore/ServiceWorkerContextData.h>
#include <WebCore/ServiceWorkerIdentifier.h>
#include <WebCore/ServiceWorkerJobDataIdentifier.h>
#include <optional>
#include <wtf/ObjectIdentifier.h>
#include <wtf/URLHash.h>
#include <wtf/text/WTFString.h>

namespace Messages {

namespace WebSWServerToContextConnection {

void SkipWaiting::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SkipWaiting::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SkipWaiting::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

void FindClientByVisibleIdentifier::callReply(IPC::Decoder& decoder, CompletionHandler<void(std::optional<WebCore::ServiceWorkerClientData>&&)>&& completionHandler)
{
    std::optional<std::optional<WebCore::ServiceWorkerClientData>> data;
    decoder >> data;
    if (!data) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*data));
}

void FindClientByVisibleIdentifier::cancelReply(CompletionHandler<void(std::optional<WebCore::ServiceWorkerClientData>&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<std::optional<WebCore::ServiceWorkerClientData>>::create());
}

void FindClientByVisibleIdentifier::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const std::optional<WebCore::ServiceWorkerClientData>& data)
{
    encoder.get() << data;
    connection.sendSyncReply(WTFMove(encoder));
}

void Claim::callReply(IPC::Decoder& decoder, CompletionHandler<void(std::optional<WebCore::ExceptionData>&&)>&& completionHandler)
{
    std::optional<std::optional<WebCore::ExceptionData>> result;
    decoder >> result;
    if (!result) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*result));
}

void Claim::cancelReply(CompletionHandler<void(std::optional<WebCore::ExceptionData>&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<std::optional<WebCore::ExceptionData>>::create());
}

void Claim::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const std::optional<WebCore::ExceptionData>& result)
{
    encoder.get() << result;
    connection.sendSyncReply(WTFMove(encoder));
}

} // namespace WebSWServerToContextConnection

} // namespace Messages

namespace WebKit {

void WebSWServerToContextConnection::didReceiveMessage(IPC::Connection& connection, IPC::Decoder& decoder)
{
    if (decoder.messageName() == Messages::WebSWServerToContextConnection::ScriptContextFailedToStart::name())
        return IPC::handleMessage<Messages::WebSWServerToContextConnection::ScriptContextFailedToStart>(connection, decoder, this, &WebSWServerToContextConnection::scriptContextFailedToStart);
    if (decoder.messageName() == Messages::WebSWServerToContextConnection::ScriptContextStarted::name())
        return IPC::handleMessage<Messages::WebSWServerToContextConnection::ScriptContextStarted>(connection, decoder, this, &WebSWServerToContextConnection::scriptContextStarted);
    if (decoder.messageName() == Messages::WebSWServerToContextConnection::DidFinishInstall::name())
        return IPC::handleMessage<Messages::WebSWServerToContextConnection::DidFinishInstall>(connection, decoder, this, &WebSWServerToContextConnection::didFinishInstall);
    if (decoder.messageName() == Messages::WebSWServerToContextConnection::DidFinishActivation::name())
        return IPC::handleMessage<Messages::WebSWServerToContextConnection::DidFinishActivation>(connection, decoder, this, &WebSWServerToContextConnection::didFinishActivation);
    if (decoder.messageName() == Messages::WebSWServerToContextConnection::SetServiceWorkerHasPendingEvents::name())
        return IPC::handleMessage<Messages::WebSWServerToContextConnection::SetServiceWorkerHasPendingEvents>(connection, decoder, this, &WebSWServerToContextConnection::setServiceWorkerHasPendingEvents);
    if (decoder.messageName() == Messages::WebSWServerToContextConnection::SkipWaiting::name())
        return IPC::handleMessageAsync<Messages::WebSWServerToContextConnection::SkipWaiting>(connection, decoder, this, &WebSWServerToContextConnection::skipWaiting);
    if (decoder.messageName() == Messages::WebSWServerToContextConnection::WorkerTerminated::name())
        return IPC::handleMessage<Messages::WebSWServerToContextConnection::WorkerTerminated>(connection, decoder, this, &WebSWServerToContextConnection::workerTerminated);
    if (decoder.messageName() == Messages::WebSWServerToContextConnection::FindClientByVisibleIdentifier::name())
        return IPC::handleMessageAsync<Messages::WebSWServerToContextConnection::FindClientByVisibleIdentifier>(connection, decoder, this, &WebSWServerToContextConnection::findClientByVisibleIdentifier);
    if (decoder.messageName() == Messages::WebSWServerToContextConnection::MatchAll::name())
        return IPC::handleMessage<Messages::WebSWServerToContextConnection::MatchAll>(connection, decoder, this, &WebSWServerToContextConnection::matchAll);
    if (decoder.messageName() == Messages::WebSWServerToContextConnection::Claim::name())
        return IPC::handleMessageAsync<Messages::WebSWServerToContextConnection::Claim>(connection, decoder, this, &WebSWServerToContextConnection::claim);
    if (decoder.messageName() == Messages::WebSWServerToContextConnection::SetScriptResource::name())
        return IPC::handleMessage<Messages::WebSWServerToContextConnection::SetScriptResource>(connection, decoder, this, &WebSWServerToContextConnection::setScriptResource);
    if (decoder.messageName() == Messages::WebSWServerToContextConnection::PostMessageToServiceWorkerClient::name())
        return IPC::handleMessage<Messages::WebSWServerToContextConnection::PostMessageToServiceWorkerClient>(connection, decoder, this, &WebSWServerToContextConnection::postMessageToServiceWorkerClient);
    if (decoder.messageName() == Messages::WebSWServerToContextConnection::DidFailHeartBeatCheck::name())
        return IPC::handleMessage<Messages::WebSWServerToContextConnection::DidFailHeartBeatCheck>(connection, decoder, this, &WebSWServerToContextConnection::didFailHeartBeatCheck);
    UNUSED_PARAM(connection);
    UNUSED_PARAM(decoder);
#if ENABLE(IPC_TESTING_API)
    if (connection.ignoreInvalidMessageForTesting())
        return;
#endif // ENABLE(IPC_TESTING_API)
    ASSERT_NOT_REACHED_WITH_MESSAGE("Unhandled message %s to %" PRIu64, IPC::description(decoder.messageName()), decoder.destinationID());
}

} // namespace WebKit

#endif // ENABLE(SERVICE_WORKER)
