# -*- coding: utf-8 -*-
# Copyright: (c) 2022, Felix Fontein (@felixfontein) <felix@fontein.de>
# GNU General Public License v3.0+ (see COPYING or https://www.gnu.org/licenses/gpl-3.0.txt)

# The data inside here is private to this collection. If you use this from outside the collection,
# you are on your own. There can be random changes to its format even in bugfix releases!

from __future__ import absolute_import, division, print_function
__metaclass__ = type


class APIData(object):
    def __init__(self, primary_keys=None,
                 stratify_keys=None,
                 has_identifier=False,
                 single_value=False,
                 unknown_mechanism=False,
                 fully_understood=False,
                 fixed_entries=False,
                 fields=None):
        if sum([primary_keys is not None, stratify_keys is not None, has_identifier, single_value, unknown_mechanism]) > 1:
            raise ValueError('primary_keys, stratify_keys, has_identifier, single_value, and unknown_mechanism are mutually exclusive')
        if unknown_mechanism and fully_understood:
            raise ValueError('unknown_mechanism and fully_understood cannot be combined')
        self.primary_keys = primary_keys
        self.stratify_keys = stratify_keys
        self.has_identifier = has_identifier
        self.single_value = single_value
        self.unknown_mechanism = unknown_mechanism
        self.fully_understood = fully_understood
        self.fixed_entries = fixed_entries
        if fixed_entries and primary_keys is None:
            raise ValueError('fixed_entries can only be used with primary_keys')
        if fields is None:
            raise ValueError('fields must be provided')
        self.fields = fields
        if primary_keys:
            for pk in primary_keys:
                if pk not in fields:
                    raise ValueError('Primary key {pk} must be in fields!'.format(pk=pk))
        if stratify_keys:
            for sk in stratify_keys:
                if sk not in fields:
                    raise ValueError('Stratify key {sk} must be in fields!'.format(sk=sk))


class KeyInfo(object):
    def __init__(self, _dummy=None, can_disable=False, remove_value=None, default=None, required=False, automatically_computed_from=None):
        if _dummy is not None:
            raise ValueError('KeyInfo() does not have positional arguments')
        if sum([required, default is not None, automatically_computed_from is not None, can_disable]) > 1:
            raise ValueError('required, default, automatically_computed_from, and can_disable are mutually exclusive')
        if not can_disable and remove_value is not None:
            raise ValueError('remove_value can only be specified if can_disable=True')
        self.can_disable = can_disable
        self.remove_value = remove_value
        self.automatically_computed_from = automatically_computed_from
        self.default = default
        self.required = required


def split_path(path):
    parts = path.split()
    if len(parts) == 1 and parts[0] == '':
        parts = []
    return parts


def join_path(path):
    return ' '.join(path)


# How to obtain this information:
# 1. Run `/export verbose` in the CLI;
# 2. All attributes listed there go into the `fields` list;
#    attributes which can have a `!` ahead should have `canDisable=True`
# 3. All bold attributes go into the `primary_keys` list -- this is not always true!

PATHS = {
    ('interface', 'bridge'): APIData(
        # fully_understood=True,
        primary_keys=('name', ),
        fields={
            'admin-mac': KeyInfo(),
            'ageing-time': KeyInfo(default='5m'),
            'arp': KeyInfo(default='enabled'),
            'arp-timeout': KeyInfo(default='auto'),
            'auto-mac': KeyInfo(default=False),
            'comment': KeyInfo(can_disable=True, remove_value=''),
            'dhcp-snooping': KeyInfo(default=False),
            'disabled': KeyInfo(default=False),
            'fast-forward': KeyInfo(default=True),
            'forward-delay': KeyInfo(default='15s'),
            'igmp-snooping': KeyInfo(default=False),
            'max-message-age': KeyInfo(default='20s'),
            'mtu': KeyInfo(default='auto'),
            'name': KeyInfo(),
            'priority': KeyInfo(default='0x8000'),
            'protocol-mode': KeyInfo(default='rstp'),
            'transmit-hold-count': KeyInfo(default=6),
            'vlan-filtering': KeyInfo(default=False),
        },
    ),
    ('interface', 'ethernet'): APIData(
        fixed_entries=True,
        fully_understood=True,
        primary_keys=('default-name', ),
        fields={
            'default-name': KeyInfo(),
            'advertise': KeyInfo(),
            'arp': KeyInfo(default='enabled'),
            'arp-timeout': KeyInfo(default='auto'),
            'auto-negotiation': KeyInfo(default=True),
            'bandwidth': KeyInfo(default='unlimited/unlimited'),
            'disabled': KeyInfo(default=False),
            'full-duplex': KeyInfo(default=True),
            'l2mtu': KeyInfo(default=1598),
            'loop-protect': KeyInfo(default='default'),
            'loop-protect-disable-time': KeyInfo(default='5m'),
            'loop-protect-send-interval': KeyInfo(default='5s'),
            'mac-address': KeyInfo(),
            'mtu': KeyInfo(default=1500),
            'name': KeyInfo(),
            'orig-mac-address': KeyInfo(),
            'rx-flow-control': KeyInfo(default='off'),
            'sfp-rate-select': KeyInfo(default='high'),
            'sfp-shutdown-temperature': KeyInfo(default='95C'),
            'speed': KeyInfo(),
            'tx-flow-control': KeyInfo(default='off'),
        },
    ),
    ('interface', 'list'): APIData(
        primary_keys=('name', ),
        fields={
            'comment': KeyInfo(can_disable=True, remove_value=''),
            'exclude': KeyInfo(),
            'include': KeyInfo(),
            'name': KeyInfo(),
        },
    ),
    ('interface', 'lte', 'apn'): APIData(
        unknown_mechanism=True,
        # primary_keys=('default', ),
        fields={
            'default': KeyInfo(),
            'add-default-route': KeyInfo(),
            'apn': KeyInfo(),
            'default-route-distance': KeyInfo(),
            'name': KeyInfo(),
            'use-peer-dns': KeyInfo(),
        },
    ),
    ('interface', 'wireless', 'security-profiles'): APIData(
        unknown_mechanism=True,
        # primary_keys=('default', ),
        fields={
            'default': KeyInfo(),
            'authentication-types': KeyInfo(),
            'disable-pmkid': KeyInfo(),
            'eap-methods': KeyInfo(),
            'group-ciphers': KeyInfo(),
            'group-key-update': KeyInfo(),
            'interim-update': KeyInfo(),
            'management-protection': KeyInfo(),
            'management-protection-key': KeyInfo(),
            'mode': KeyInfo(),
            'mschapv2-password': KeyInfo(),
            'mschapv2-username': KeyInfo(),
            'name': KeyInfo(),
            'radius-called-format': KeyInfo(),
            'radius-eap-accounting': KeyInfo(),
            'radius-mac-accounting': KeyInfo(),
            'radius-mac-authentication': KeyInfo(),
            'radius-mac-caching': KeyInfo(),
            'radius-mac-format': KeyInfo(),
            'radius-mac-mode': KeyInfo(),
            'static-algo-0': KeyInfo(),
            'static-algo-1': KeyInfo(),
            'static-algo-2': KeyInfo(),
            'static-algo-3': KeyInfo(),
            'static-key-0': KeyInfo(),
            'static-key-1': KeyInfo(),
            'static-key-2': KeyInfo(),
            'static-key-3': KeyInfo(),
            'static-sta-private-algo': KeyInfo(),
            'static-sta-private-key': KeyInfo(),
            'static-transmit-key': KeyInfo(),
            'supplicant-identity': KeyInfo(),
            'tls-certificate': KeyInfo(),
            'tls-mode': KeyInfo(),
            'unicast-ciphers': KeyInfo(),
            'wpa-pre-shared-key': KeyInfo(),
            'wpa2-pre-shared-key': KeyInfo(),
        },
    ),
    ('ip', 'hotspot', 'profile'): APIData(
        unknown_mechanism=True,
        # primary_keys=('default', ),
        fields={
            'default': KeyInfo(),
            'dns-name': KeyInfo(),
            'hotspot-address': KeyInfo(),
            'html-directory': KeyInfo(),
            'html-directory-override': KeyInfo(),
            'http-cookie-lifetime': KeyInfo(),
            'http-proxy': KeyInfo(),
            'login-by': KeyInfo(),
            'name': KeyInfo(),
            'rate-limit': KeyInfo(),
            'smtp-server': KeyInfo(),
            'split-user-domain': KeyInfo(),
            'use-radius': KeyInfo(),
        },
    ),
    ('ip', 'hotspot', 'user', 'profile'): APIData(
        unknown_mechanism=True,
        # primary_keys=('default', ),
        fields={
            'default': KeyInfo(),
            'add-mac-cookie': KeyInfo(),
            'address-list': KeyInfo(),
            'idle-timeout': KeyInfo(),
            'insert-queue-before': KeyInfo(can_disable=True),
            'keepalive-timeout': KeyInfo(),
            'mac-cookie-timeout': KeyInfo(),
            'name': KeyInfo(),
            'parent-queue': KeyInfo(can_disable=True),
            'queue-type': KeyInfo(can_disable=True),
            'shared-users': KeyInfo(),
            'status-autorefresh': KeyInfo(),
            'transparent-proxy': KeyInfo(),
        },
    ),
    ('ip', 'ipsec', 'mode-config'): APIData(
        unknown_mechanism=True,
        # primary_keys=('default', ),
        fields={
            'default': KeyInfo(),
            'name': KeyInfo(),
            'responder': KeyInfo(),
            'use-responder-dns': KeyInfo(),
        },
    ),
    ('ip', 'ipsec', 'policy', 'group'): APIData(
        unknown_mechanism=True,
        # primary_keys=('default', ),
        fields={
            'default': KeyInfo(),
            'name': KeyInfo(),
        },
    ),
    ('ip', 'ipsec', 'profile'): APIData(
        unknown_mechanism=True,
        # primary_keys=('default', ),
        fields={
            'default': KeyInfo(),
            'dh-group': KeyInfo(),
            'dpd-interval': KeyInfo(),
            'dpd-maximum-failures': KeyInfo(),
            'enc-algorithm': KeyInfo(),
            'hash-algorithm': KeyInfo(),
            'lifetime': KeyInfo(),
            'name': KeyInfo(),
            'nat-traversal': KeyInfo(),
            'proposal-check': KeyInfo(),
        },
    ),
    ('ip', 'ipsec', 'proposal'): APIData(
        unknown_mechanism=True,
        # primary_keys=('default', ),
        fields={
            'default': KeyInfo(),
            'auth-algorithms': KeyInfo(),
            'disabled': KeyInfo(),
            'enc-algorithms': KeyInfo(),
            'lifetime': KeyInfo(),
            'name': KeyInfo(),
            'pfs-group': KeyInfo(),
        },
    ),
    ('ip', 'pool'): APIData(
        fully_understood=True,
        primary_keys=('name', ),
        fields={
            'name': KeyInfo(),
            'ranges': KeyInfo(),
        },
    ),
    ('ip', 'dhcp-server'): APIData(
        fully_understood=True,
        primary_keys=('name', ),
        fields={
            'address-pool': KeyInfo(required=True),
            'authoritative': KeyInfo(default=True),
            'disabled': KeyInfo(default=False),
            'interface': KeyInfo(required=True),
            'lease-script': KeyInfo(default=''),
            'lease-time': KeyInfo(default='10m'),
            'name': KeyInfo(),
            'use-radius': KeyInfo(default=False),
        },
    ),
    ('routing', 'ospf', 'instance'): APIData(
        unknown_mechanism=True,
        # primary_keys=('default', ),
        fields={
            'default': KeyInfo(),
            'disabled': KeyInfo(),
            'distribute-default': KeyInfo(),
            'domain-id': KeyInfo(can_disable=True),
            'domain-tag': KeyInfo(can_disable=True),
            'in-filter': KeyInfo(),
            'metric-bgp': KeyInfo(),
            'metric-connected': KeyInfo(),
            'metric-default': KeyInfo(),
            'metric-other-ospf': KeyInfo(),
            'metric-rip': KeyInfo(),
            'metric-static': KeyInfo(),
            'mpls-te-area': KeyInfo(can_disable=True),
            'mpls-te-router-id': KeyInfo(can_disable=True),
            'name': KeyInfo(),
            'out-filter': KeyInfo(),
            'redistribute-bgp': KeyInfo(),
            'redistribute-connected': KeyInfo(),
            'redistribute-other-ospf': KeyInfo(),
            'redistribute-rip': KeyInfo(),
            'redistribute-static': KeyInfo(),
            'router-id': KeyInfo(),
            'routing-table': KeyInfo(can_disable=True),
            'use-dn': KeyInfo(can_disable=True),
        },
    ),
    ('routing', 'ospf', 'area'): APIData(
        unknown_mechanism=True,
        # primary_keys=('default', ),
        fields={
            'default': KeyInfo(),
            'area-id': KeyInfo(),
            'disabled': KeyInfo(),
            'instance': KeyInfo(),
            'name': KeyInfo(),
            'type': KeyInfo(),
        },
    ),
    ('routing', 'ospf-v3', 'instance'): APIData(
        unknown_mechanism=True,
        # primary_keys=('default', ),
        fields={
            'default': KeyInfo(),
            'disabled': KeyInfo(),
            'distribute-default': KeyInfo(),
            'metric-bgp': KeyInfo(),
            'metric-connected': KeyInfo(),
            'metric-default': KeyInfo(),
            'metric-other-ospf': KeyInfo(),
            'metric-rip': KeyInfo(),
            'metric-static': KeyInfo(),
            'name': KeyInfo(),
            'redistribute-bgp': KeyInfo(),
            'redistribute-connected': KeyInfo(),
            'redistribute-other-ospf': KeyInfo(),
            'redistribute-rip': KeyInfo(),
            'redistribute-static': KeyInfo(),
            'router-id': KeyInfo(),
        },
    ),
    ('routing', 'ospf-v3', 'area'): APIData(
        unknown_mechanism=True,
        # primary_keys=('default', ),
        fields={
            'default': KeyInfo(),
            'area-id': KeyInfo(),
            'disabled': KeyInfo(),
            'instance': KeyInfo(),
            'name': KeyInfo(),
            'type': KeyInfo(),
        },
    ),
    ('snmp', 'community'): APIData(
        unknown_mechanism=True,
        # primary_keys=('default', ),
        fields={
            'default': KeyInfo(),
            'addresses': KeyInfo(),
            'authentication-password': KeyInfo(),
            'authentication-protocol': KeyInfo(),
            'disabled': KeyInfo(),
            'encryption-password': KeyInfo(),
            'encryption-protocol': KeyInfo(),
            'name': KeyInfo(),
            'read-access': KeyInfo(),
            'security': KeyInfo(),
            'write-access': KeyInfo(),
        },
    ),
    ('caps-man', 'aaa'): APIData(
        single_value=True,
        fully_understood=True,
        fields={
            'called-format': KeyInfo(default='mac:ssid'),
            'interim-update': KeyInfo(default='disabled'),
            'mac-caching': KeyInfo(default='disabled'),
            'mac-format': KeyInfo(default='XX:XX:XX:XX:XX:XX'),
            'mac-mode': KeyInfo(default='as-username'),
        },
    ),
    ('caps-man', 'manager', 'interface'): APIData(
        unknown_mechanism=True,
        # primary_keys=('default', ),
        fields={
            'default': KeyInfo(),
            'disabled': KeyInfo(),
            'forbid': KeyInfo(),
            'interface': KeyInfo(),
        },
    ),
    ('certificate', 'settings'): APIData(
        single_value=True,
        fully_understood=True,
        fields={
            'crl-download': KeyInfo(default=False),
            'crl-store': KeyInfo(default='ram'),
            'crl-use': KeyInfo(default=False),
        },
    ),
    ('interface', 'bridge', 'port'): APIData(
        fully_understood=True,
        primary_keys=('interface', ),
        fields={
            'auto-isolate': KeyInfo(default=False),
            'bpdu-guard': KeyInfo(default=False),
            'bridge': KeyInfo(required=True),
            'broadcast-flood': KeyInfo(default=True),
            'comment': KeyInfo(can_disable=True, remove_value=''),
            'disabled': KeyInfo(default=False),
            'edge': KeyInfo(default='auto'),
            'fast-leave': KeyInfo(default=False),
            'frame-types': KeyInfo(default='admit-all'),
            'horizon': KeyInfo(default='none'),
            'hw': KeyInfo(default=True),
            'ingress-filtering': KeyInfo(default=False),
            'interface': KeyInfo(),
            'internal-path-cost': KeyInfo(default=10),
            'learn': KeyInfo(default='auto'),
            'multicast-router': KeyInfo(default='temporary-query'),
            'path-cost': KeyInfo(default=10),
            'point-to-point': KeyInfo(default='auto'),
            'priority': KeyInfo(default='0x80'),
            'pvid': KeyInfo(default=1),
            'restricted-role': KeyInfo(default=False),
            'restricted-tcn': KeyInfo(default=False),
            'tag-stacking': KeyInfo(default=False),
            'trusted': KeyInfo(default=False),
            'unknown-multicast-flood': KeyInfo(default=True),
            'unknown-unicast-flood': KeyInfo(default=True),
        },
    ),
    ('interface', 'bridge', 'port-controller'): APIData(
        single_value=True,
        fully_understood=True,
        fields={
            'bridge': KeyInfo(default='none'),
            'cascade-ports': KeyInfo(default=''),
            'switch': KeyInfo(default='none'),
        },
    ),
    ('interface', 'bridge', 'port-extender'): APIData(
        single_value=True,
        fully_understood=True,
        fields={
            'control-ports': KeyInfo(default=''),
            'excluded-ports': KeyInfo(default=''),
            'switch': KeyInfo(default='none'),
        },
    ),
    ('interface', 'bridge', 'settings'): APIData(
        single_value=True,
        fully_understood=True,
        fields={
            'allow-fast-path': KeyInfo(default=True),
            'use-ip-firewall': KeyInfo(default=False),
            'use-ip-firewall-for-pppoe': KeyInfo(default=False),
            'use-ip-firewall-for-vlan': KeyInfo(default=False),
        },
    ),
    ('ip', 'firewall', 'connection', 'tracking'): APIData(
        single_value=True,
        fully_understood=True,
        fields={
            'enabled': KeyInfo(default='auto'),
            'generic-timeout': KeyInfo(default='10m'),
            'icmp-timeout': KeyInfo(default='10s'),
            'loose-tcp-tracking': KeyInfo(default=True),
            'tcp-close-timeout': KeyInfo(default='10s'),
            'tcp-close-wait-timeout': KeyInfo(default='10s'),
            'tcp-established-timeout': KeyInfo(default='1d'),
            'tcp-fin-wait-timeout': KeyInfo(default='10s'),
            'tcp-last-ack-timeout': KeyInfo(default='10s'),
            'tcp-max-retrans-timeout': KeyInfo(default='5m'),
            'tcp-syn-received-timeout': KeyInfo(default='5s'),
            'tcp-syn-sent-timeout': KeyInfo(default='5s'),
            'tcp-time-wait-timeout': KeyInfo(default='10s'),
            'tcp-unacked-timeout': KeyInfo(default='5m'),
            'udp-stream-timeout': KeyInfo(default='3m'),
            'udp-timeout': KeyInfo(default='10s'),
        },
    ),
    ('ip', 'neighbor', 'discovery-settings'): APIData(
        single_value=True,
        fully_understood=True,
        fields={
            'discover-interface-list': KeyInfo(),
            'lldp-med-net-policy-vlan': KeyInfo(default='disabled'),
            'protocol': KeyInfo(default='cdp,lldp,mndp'),
        },
    ),
    ('ip', 'settings'): APIData(
        single_value=True,
        fully_understood=True,
        fields={
            'accept-redirects': KeyInfo(default=False),
            'accept-source-route': KeyInfo(default=False),
            'allow-fast-path': KeyInfo(default=True),
            'arp-timeout': KeyInfo(default='30s'),
            'icmp-rate-limit': KeyInfo(default=10),
            'icmp-rate-mask': KeyInfo(default='0x1818'),
            'ip-forward': KeyInfo(default=True),
            'max-neighbor-entries': KeyInfo(default=8192),
            'route-cache': KeyInfo(default=True),
            'rp-filter': KeyInfo(default=False),
            'secure-redirects': KeyInfo(default=True),
            'send-redirects': KeyInfo(default=True),
            'tcp-syncookies': KeyInfo(default=False),
        },
    ),
    ('ipv6', 'settings'): APIData(
        single_value=True,
        fully_understood=True,
        fields={
            'accept-redirects': KeyInfo(default='yes-if-forwarding-disabled'),
            'accept-router-advertisements': KeyInfo(default='yes-if-forwarding-disabled'),
            'forward': KeyInfo(default=True),
            'max-neighbor-entries': KeyInfo(default=8192),
        },
    ),
    ('interface', 'detect-internet'): APIData(
        single_value=True,
        fully_understood=True,
        fields={
            'detect-interface-list': KeyInfo(default='none'),
            'internet-interface-list': KeyInfo(default='none'),
            'lan-interface-list': KeyInfo(default='none'),
            'wan-interface-list': KeyInfo(default='none'),
        },
    ),
    ('interface', 'l2tp-server', 'server'): APIData(
        single_value=True,
        fully_understood=True,
        fields={
            'allow-fast-path': KeyInfo(default=False),
            'authentication': KeyInfo(default='pap,chap,mschap1,mschap2'),
            'caller-id-type': KeyInfo(default='ip-address'),
            'default-profile': KeyInfo(default='default-encryption'),
            'enabled': KeyInfo(default=False),
            'ipsec-secret': KeyInfo(default=''),
            'keepalive-timeout': KeyInfo(default=30),
            'max-mru': KeyInfo(default=1450),
            'max-mtu': KeyInfo(default=1450),
            'max-sessions': KeyInfo(default='unlimited'),
            'mrru': KeyInfo(default='disabled'),
            'one-session-per-host': KeyInfo(default=False),
            'use-ipsec': KeyInfo(default=False),
        },
    ),
    ('interface', 'ovpn-server', 'server'): APIData(
        single_value=True,
        fully_understood=True,
        fields={
            'auth': KeyInfo(),
            'cipher': KeyInfo(),
            'default-profile': KeyInfo(default='default'),
            'enabled': KeyInfo(default=False),
            'keepalive-timeout': KeyInfo(default=60),
            'mac-address': KeyInfo(),
            'max-mtu': KeyInfo(default=1500),
            'mode': KeyInfo(default='ip'),
            'netmask': KeyInfo(default=24),
            'port': KeyInfo(default=1194),
            'require-client-certificate': KeyInfo(default=False),
        },
    ),
    ('interface', 'pptp-server', 'server'): APIData(
        single_value=True,
        fully_understood=True,
        fields={
            'authentication': KeyInfo(default='mschap1,mschap2'),
            'default-profile': KeyInfo(default='default-encryption'),
            'enabled': KeyInfo(default=False),
            'keepalive-timeout': KeyInfo(default=30),
            'max-mru': KeyInfo(default=1450),
            'max-mtu': KeyInfo(default=1450),
            'mrru': KeyInfo(default='disabled'),
        },
    ),
    ('interface', 'sstp-server', 'server'): APIData(
        single_value=True,
        fully_understood=True,
        fields={
            'authentication': KeyInfo(default='pap,chap,mschap1,mschap2'),
            'certificate': KeyInfo(default='none'),
            'default-profile': KeyInfo(default='default'),
            'enabled': KeyInfo(default=False),
            'force-aes': KeyInfo(default=False),
            'keepalive-timeout': KeyInfo(default=60),
            'max-mru': KeyInfo(default=1500),
            'max-mtu': KeyInfo(default=1500),
            'mrru': KeyInfo(default='disabled'),
            'pfs': KeyInfo(default=False),
            'port': KeyInfo(default=443),
            'tls-version': KeyInfo(default='any'),
            'verify-client-certificate': KeyInfo(default='no'),
        },
    ),
    ('interface', 'wireless', 'align'): APIData(
        single_value=True,
        fully_understood=True,
        fields={
            'active-mode': KeyInfo(default=True),
            'audio-max': KeyInfo(default=-20),
            'audio-min': KeyInfo(default=-100),
            'audio-monitor': KeyInfo(default='00:00:00:00:00:00'),
            'filter-mac': KeyInfo(default='00:00:00:00:00:00'),
            'frame-size': KeyInfo(default=300),
            'frames-per-second': KeyInfo(default=25),
            'receive-all': KeyInfo(default=False),
            'ssid-all': KeyInfo(default=False),
        },
    ),
    ('interface', 'wireless', 'cap'): APIData(
        single_value=True,
        fully_understood=True,
        fields={
            'bridge': KeyInfo(default='none'),
            'caps-man-addresses': KeyInfo(default=''),
            'caps-man-certificate-common-names': KeyInfo(default=''),
            'caps-man-names': KeyInfo(default=''),
            'certificate': KeyInfo(default='none'),
            'discovery-interfaces': KeyInfo(default=''),
            'enabled': KeyInfo(default=False),
            'interfaces': KeyInfo(default=''),
            'lock-to-caps-man': KeyInfo(default=False),
            'static-virtual': KeyInfo(default=False),
        },
    ),
    ('interface', 'wireless', 'sniffer'): APIData(
        single_value=True,
        fully_understood=True,
        fields={
            'channel-time': KeyInfo(default='200ms'),
            'file-limit': KeyInfo(default=10),
            'file-name': KeyInfo(default=''),
            'memory-limit': KeyInfo(default=10),
            'multiple-channels': KeyInfo(default=False),
            'only-headers': KeyInfo(default=False),
            'receive-errors': KeyInfo(default=False),
            'streaming-enabled': KeyInfo(default=False),
            'streaming-max-rate': KeyInfo(default=0),
            'streaming-server': KeyInfo(default='0.0.0.0'),
        },
    ),
    ('interface', 'wireless', 'snooper'): APIData(
        single_value=True,
        fully_understood=True,
        fields={
            'channel-time': KeyInfo(default='200ms'),
            'multiple-channels': KeyInfo(default=True),
            'receive-errors': KeyInfo(default=False),
        },
    ),
    ('ip', 'accounting'): APIData(
        single_value=True,
        fully_understood=True,
        fields={
            'account-local-traffic': KeyInfo(default=False),
            'enabled': KeyInfo(default=False),
            'threshold': KeyInfo(default=256),
        },
    ),
    ('ip', 'accounting', 'web-access'): APIData(
        single_value=True,
        fully_understood=True,
        fields={
            'accessible-via-web': KeyInfo(default=False),
            'address': KeyInfo(default='0.0.0.0/0'),
        },
    ),
    ('ip', 'address'): APIData(
        fully_understood=True,
        primary_keys=('address', 'interface', ),
        fields={
            'address': KeyInfo(),
            'comment': KeyInfo(can_disable=True, remove_value=''),
            'disabled': KeyInfo(default=False),
            'interface': KeyInfo(),
            'network': KeyInfo(automatically_computed_from=('address', )),
        },
    ),
    ('ip', 'cloud'): APIData(
        single_value=True,
        fully_understood=True,
        fields={
            'ddns-enabled': KeyInfo(default=False),
            'ddns-update-interval': KeyInfo(default='none'),
            'update-time': KeyInfo(default=True),
        },
    ),
    ('ip', 'cloud', 'advanced'): APIData(
        single_value=True,
        fully_understood=True,
        fields={
            'use-local-address': KeyInfo(default=False),
        },
    ),
    ('ip', 'dhcp-client'): APIData(
        fully_understood=True,
        primary_keys=('interface', ),
        fields={
            'add-default-route': KeyInfo(default=True),
            'comment': KeyInfo(can_disable=True, remove_value=''),
            'default-route-distance': KeyInfo(default=1),
            'dhcp-options': KeyInfo(default='hostname,clientid'),
            'disabled': KeyInfo(default=False),
            'interface': KeyInfo(),
            'use-peer-dns': KeyInfo(default=True),
            'use-peer-ntp': KeyInfo(default=True),
        },
    ),
    ('ip', 'dhcp-server', 'config'): APIData(
        single_value=True,
        fully_understood=True,
        fields={
            'accounting': KeyInfo(default=True),
            'interim-update': KeyInfo(default='0s'),
            'store-leases-disk': KeyInfo(default='5m'),
        },
    ),
    ('ip', 'dhcp-server', 'lease'): APIData(
        fully_understood=True,
        primary_keys=('server', 'address', ),
        fields={
            'address': KeyInfo(),
            'address-lists': KeyInfo(default=''),
            'always-broadcast': KeyInfo(),
            'client-id': KeyInfo(can_disable=True, remove_value=''),
            'comment': KeyInfo(can_disable=True, remove_value=''),
            'dhcp-option': KeyInfo(default=''),
            'disabled': KeyInfo(default=False),
            'insert-queue-before': KeyInfo(can_disable=True),
            'mac-address': KeyInfo(can_disable=True, remove_value=''),
            'server': KeyInfo(),
        },
    ),
    ('ip', 'dhcp-server', 'network'): APIData(
        fully_understood=True,
        primary_keys=('address', ),
        fields={
            'address': KeyInfo(),
            'caps-manager': KeyInfo(default=''),
            'comment': KeyInfo(can_disable=True, remove_value=''),
            'dhcp-option': KeyInfo(default=''),
            'dns-server': KeyInfo(default=''),
            'gateway': KeyInfo(automatically_computed_from=('address', )),
            'netmask': KeyInfo(automatically_computed_from=('address', )),
            'ntp-server': KeyInfo(default=''),
            'wins-server': KeyInfo(default=''),
        },
    ),
    ('ip', 'dns'): APIData(
        single_value=True,
        fully_understood=True,
        fields={
            'allow-remote-requests': KeyInfo(),
            'cache-max-ttl': KeyInfo(default='1w'),
            'cache-size': KeyInfo(default='2048KiB'),
            'max-concurrent-queries': KeyInfo(default=100),
            'max-concurrent-tcp-sessions': KeyInfo(default=20),
            'max-udp-packet-size': KeyInfo(default=4096),
            'query-server-timeout': KeyInfo(default='2s'),
            'query-total-timeout': KeyInfo(default='10s'),
            'servers': KeyInfo(default=''),
            'use-doh-server': KeyInfo(default=''),
            'verify-doh-cert': KeyInfo(default=False),
        },
    ),
    ('ip', 'dns', 'static'): APIData(
        fully_understood=True,
        stratify_keys=('name', ),
        fields={
            'address': KeyInfo(),
            'cname': KeyInfo(),
            'comment': KeyInfo(can_disable=True, remove_value=''),
            'disabled': KeyInfo(default=False),
            'forward-to': KeyInfo(),
            'mx-exchange': KeyInfo(),
            'mx-preference': KeyInfo(),
            'name': KeyInfo(required=True),
            'ns': KeyInfo(),
            'srv-port': KeyInfo(),
            'srv-priority': KeyInfo(),
            'srv-target': KeyInfo(),
            'srv-weight': KeyInfo(),
            'text': KeyInfo(),
            'ttl': KeyInfo(default='1d'),
            'type': KeyInfo(),
        },
    ),
    ('ip', 'firewall', 'address-list'): APIData(
        fully_understood=True,
        primary_keys=('address', 'list', ),
        fields={
            'address': KeyInfo(),
            'disabled': KeyInfo(default=False),
            'list': KeyInfo(),
        },
    ),
    ('ip', 'firewall', 'filter'): APIData(
        fully_understood=True,
        stratify_keys=('chain', ),
        fields={
            'action': KeyInfo(),
            'chain': KeyInfo(),
            'comment': KeyInfo(can_disable=True, remove_value=''),
            'connection-bytes': KeyInfo(can_disable=True),
            'connection-limit': KeyInfo(can_disable=True),
            'connection-mark': KeyInfo(can_disable=True),
            'connection-nat-state': KeyInfo(can_disable=True),
            'connection-rate': KeyInfo(can_disable=True),
            'connection-state': KeyInfo(can_disable=True),
            'connection-type': KeyInfo(can_disable=True),
            'content': KeyInfo(can_disable=True),
            'disabled': KeyInfo(),
            'dscp': KeyInfo(can_disable=True),
            'dst-address': KeyInfo(can_disable=True),
            'dst-address-list': KeyInfo(can_disable=True),
            'dst-address-type': KeyInfo(can_disable=True),
            'dst-limit': KeyInfo(can_disable=True),
            'dst-port': KeyInfo(can_disable=True),
            'fragment': KeyInfo(can_disable=True),
            'hotspot': KeyInfo(can_disable=True),
            'icmp-options': KeyInfo(can_disable=True),
            'in-bridge-port': KeyInfo(can_disable=True),
            'in-bridge-port-list': KeyInfo(can_disable=True),
            'in-interface': KeyInfo(can_disable=True),
            'in-interface-list': KeyInfo(can_disable=True),
            'ingress-priority': KeyInfo(can_disable=True),
            'ipsec-policy': KeyInfo(can_disable=True),
            'ipv4-options': KeyInfo(can_disable=True),
            'layer7-protocol': KeyInfo(can_disable=True),
            'limit': KeyInfo(can_disable=True),
            'log': KeyInfo(),
            'log-prefix': KeyInfo(),
            'nth': KeyInfo(can_disable=True),
            'out-bridge-port': KeyInfo(can_disable=True),
            'out-bridge-port-list': KeyInfo(can_disable=True),
            'out-interface': KeyInfo(can_disable=True),
            'out-interface-list': KeyInfo(can_disable=True),
            'p2p': KeyInfo(can_disable=True),
            'packet-mark': KeyInfo(can_disable=True),
            'packet-size': KeyInfo(can_disable=True),
            'per-connection-classifier': KeyInfo(can_disable=True),
            'port': KeyInfo(can_disable=True),
            'priority': KeyInfo(can_disable=True),
            'protocol': KeyInfo(can_disable=True),
            'psd': KeyInfo(can_disable=True),
            'random': KeyInfo(can_disable=True),
            'routing-mark': KeyInfo(can_disable=True),
            'routing-table': KeyInfo(can_disable=True),
            'src-address': KeyInfo(can_disable=True),
            'src-address-list': KeyInfo(can_disable=True),
            'src-address-type': KeyInfo(can_disable=True),
            'src-mac-address': KeyInfo(can_disable=True),
            'src-port': KeyInfo(can_disable=True),
            'tcp-flags': KeyInfo(can_disable=True),
            'tcp-mss': KeyInfo(can_disable=True),
            'time': KeyInfo(can_disable=True),
            'tls-host': KeyInfo(can_disable=True),
            'ttl': KeyInfo(can_disable=True),
        },
    ),
    ('ip', 'firewall', 'nat'): APIData(
        fully_understood=True,
        stratify_keys=('chain', ),
        fields={
            'action': KeyInfo(),
            'chain': KeyInfo(),
            'comment': KeyInfo(can_disable=True, remove_value=''),
            'dst-address': KeyInfo(can_disable=True),
            'dst-port': KeyInfo(can_disable=True),
            'out-interface': KeyInfo(can_disable=True),
            'out-interface-list': KeyInfo(can_disable=True),
            'protocol': KeyInfo(can_disable=True),
            'to-addresses': KeyInfo(can_disable=True),
            'to-ports': KeyInfo(can_disable=True),
        },
    ),
    ('ip', 'hotspot', 'user'): APIData(
        unknown_mechanism=True,
        # primary_keys=('default', ),
        fields={
            'default': KeyInfo(),
            'comment': KeyInfo(can_disable=True, remove_value=''),
            'disabled': KeyInfo(),
            'name': KeyInfo(),
        },
    ),
    ('ip', 'ipsec', 'settings'): APIData(
        single_value=True,
        fully_understood=True,
        fields={
            'accounting': KeyInfo(default=True),
            'interim-update': KeyInfo(default='0s'),
            'xauth-use-radius': KeyInfo(default=False),
        },
    ),
    ('ip', 'proxy'): APIData(
        single_value=True,
        fully_understood=True,
        fields={
            'always-from-cache': KeyInfo(default=False),
            'anonymous': KeyInfo(default=False),
            'cache-administrator': KeyInfo(default='webmaster'),
            'cache-hit-dscp': KeyInfo(default=4),
            'cache-on-disk': KeyInfo(default=False),
            'cache-path': KeyInfo(default='web-proxy'),
            'enabled': KeyInfo(default=False),
            'max-cache-object-size': KeyInfo(default='2048KiB'),
            'max-cache-size': KeyInfo(default='unlimited'),
            'max-client-connections': KeyInfo(default=600),
            'max-fresh-time': KeyInfo(default='3d'),
            'max-server-connections': KeyInfo(default=600),
            'parent-proxy': KeyInfo(default='::'),
            'parent-proxy-port': KeyInfo(default=0),
            'port': KeyInfo(default=8080),
            'serialize-connections': KeyInfo(default=False),
            'src-address': KeyInfo(default='::'),
        },
    ),
    ('ip', 'smb'): APIData(
        single_value=True,
        fully_understood=True,
        fields={
            'allow-guests': KeyInfo(default=True),
            'comment': KeyInfo(default='MikrotikSMB'),
            'domain': KeyInfo(default='MSHOME'),
            'enabled': KeyInfo(default=False),
            'interfaces': KeyInfo(default='all'),
        },
    ),
    ('ip', 'smb', 'shares'): APIData(
        unknown_mechanism=True,
        # primary_keys=('default', ),
        fields={
            'default': KeyInfo(),
            'comment': KeyInfo(can_disable=True, remove_value=''),
            'directory': KeyInfo(),
            'disabled': KeyInfo(),
            'max-sessions': KeyInfo(),
            'name': KeyInfo(),
        },
    ),
    ('ip', 'smb', 'users'): APIData(
        unknown_mechanism=True,
        # primary_keys=('default', ),
        fields={
            'default': KeyInfo(),
            'disabled': KeyInfo(),
            'name': KeyInfo(),
            'password': KeyInfo(),
            'read-only': KeyInfo(),
        },
    ),
    ('ip', 'socks'): APIData(
        single_value=True,
        fully_understood=True,
        fields={
            'auth-method': KeyInfo(default='none'),
            'connection-idle-timeout': KeyInfo(default='2m'),
            'enabled': KeyInfo(default=False),
            'max-connections': KeyInfo(default=200),
            'port': KeyInfo(default=1080),
            'version': KeyInfo(default=4),
        },
    ),
    ('ip', 'ssh'): APIData(
        single_value=True,
        fully_understood=True,
        fields={
            'allow-none-crypto': KeyInfo(default=False),
            'always-allow-password-login': KeyInfo(default=False),
            'forwarding-enabled': KeyInfo(default=False),
            'host-key-size': KeyInfo(default=2048),
            'strong-crypto': KeyInfo(default=False),
        },
    ),
    ('ip', 'tftp', 'settings'): APIData(
        single_value=True,
        fully_understood=True,
        fields={
            'max-block-size': KeyInfo(default=4096),
        },
    ),
    ('ip', 'traffic-flow'): APIData(
        single_value=True,
        fully_understood=True,
        fields={
            'active-flow-timeout': KeyInfo(default='30m'),
            'cache-entries': KeyInfo(default='32k'),
            'enabled': KeyInfo(default=False),
            'inactive-flow-timeout': KeyInfo(default='15s'),
            'interfaces': KeyInfo(default='all'),
            'packet-sampling': KeyInfo(default=False),
            'sampling-interval': KeyInfo(default=0),
            'sampling-space': KeyInfo(default=0),
        },
    ),
    ('ip', 'traffic-flow', 'ipfix'): APIData(
        single_value=True,
        fully_understood=True,
        fields={
            'bytes': KeyInfo(default=True),
            'dst-address': KeyInfo(default=True),
            'dst-address-mask': KeyInfo(default=True),
            'dst-mac-address': KeyInfo(default=True),
            'dst-port': KeyInfo(default=True),
            'first-forwarded': KeyInfo(default=True),
            'gateway': KeyInfo(default=True),
            'icmp-code': KeyInfo(default=True),
            'icmp-type': KeyInfo(default=True),
            'igmp-type': KeyInfo(default=True),
            'in-interface': KeyInfo(default=True),
            'ip-header-length': KeyInfo(default=True),
            'ip-total-length': KeyInfo(default=True),
            'ipv6-flow-label': KeyInfo(default=True),
            'is-multicast': KeyInfo(default=True),
            'last-forwarded': KeyInfo(default=True),
            'nat-dst-address': KeyInfo(default=True),
            'nat-dst-port': KeyInfo(default=True),
            'nat-events': KeyInfo(default=False),
            'nat-src-address': KeyInfo(default=True),
            'nat-src-port': KeyInfo(default=True),
            'out-interface': KeyInfo(default=True),
            'packets': KeyInfo(default=True),
            'protocol': KeyInfo(default=True),
            'src-address': KeyInfo(default=True),
            'src-address-mask': KeyInfo(default=True),
            'src-mac-address': KeyInfo(default=True),
            'src-port': KeyInfo(default=True),
            'sys-init-time': KeyInfo(default=True),
            'tcp-ack-num': KeyInfo(default=True),
            'tcp-flags': KeyInfo(default=True),
            'tcp-seq-num': KeyInfo(default=True),
            'tcp-window-size': KeyInfo(default=True),
            'tos': KeyInfo(default=True),
            'ttl': KeyInfo(default=True),
            'udp-length': KeyInfo(default=True),
        },
    ),
    ('ip', 'upnp'): APIData(
        single_value=True,
        fully_understood=True,
        fields={
            'allow-disable-external-interface': KeyInfo(default=False),
            'enabled': KeyInfo(default=False),
            'show-dummy-rule': KeyInfo(default=True),
        },
    ),
    ('ipv6', 'dhcp-client'): APIData(
        fully_understood=True,
        primary_keys=('interface', 'request'),
        fields={
            'add-default-route': KeyInfo(default=False),
            'comment': KeyInfo(can_disable=True, remove_value=''),
            'default-route-distance': KeyInfo(default=1),
            'dhcp-options': KeyInfo(default=''),
            'disabled': KeyInfo(default=False),
            'interface': KeyInfo(),
            'pool-name': KeyInfo(required=True),
            'pool-prefix-length': KeyInfo(default=64),
            'prefix-hint': KeyInfo(default='::/0'),
            'request': KeyInfo(),
            'use-peer-dns': KeyInfo(default=True),
        },
    ),
    ('ipv6', 'firewall', 'address-list'): APIData(
        fully_understood=True,
        primary_keys=('address', 'list', ),
        fields={
            'address': KeyInfo(),
            'disabled': KeyInfo(default=False),
            'dynamic': KeyInfo(default=False),
            'list': KeyInfo(),
        },
    ),
    ('ipv6', 'firewall', 'filter'): APIData(
        fully_understood=True,
        stratify_keys=('chain', ),
        fields={
            'action': KeyInfo(),
            'chain': KeyInfo(),
            'comment': KeyInfo(can_disable=True, remove_value=''),
            'connection-bytes': KeyInfo(can_disable=True),
            'connection-limit': KeyInfo(can_disable=True),
            'connection-mark': KeyInfo(can_disable=True),
            'connection-rate': KeyInfo(can_disable=True),
            'connection-state': KeyInfo(can_disable=True),
            'connection-type': KeyInfo(can_disable=True),
            'content': KeyInfo(can_disable=True),
            'disabled': KeyInfo(),
            'dscp': KeyInfo(can_disable=True),
            'dst-address': KeyInfo(can_disable=True),
            'dst-address-list': KeyInfo(can_disable=True),
            'dst-address-type': KeyInfo(can_disable=True),
            'dst-limit': KeyInfo(can_disable=True),
            'dst-port': KeyInfo(can_disable=True),
            'headers': KeyInfo(can_disable=True),
            'hop-limit': KeyInfo(can_disable=True),
            'icmp-options': KeyInfo(can_disable=True),
            'in-bridge-port': KeyInfo(can_disable=True),
            'in-bridge-port-list': KeyInfo(can_disable=True),
            'in-interface': KeyInfo(can_disable=True),
            'in-interface-list': KeyInfo(can_disable=True),
            'ingress-priority': KeyInfo(can_disable=True),
            'ipsec-policy': KeyInfo(can_disable=True),
            'limit': KeyInfo(can_disable=True),
            'log': KeyInfo(),
            'log-prefix': KeyInfo(),
            'nth': KeyInfo(can_disable=True),
            'out-bridge-port': KeyInfo(can_disable=True),
            'out-bridge-port-list': KeyInfo(can_disable=True),
            'out-interface': KeyInfo(can_disable=True),
            'out-interface-list': KeyInfo(can_disable=True),
            'packet-mark': KeyInfo(can_disable=True),
            'packet-size': KeyInfo(can_disable=True),
            'per-connection-classifier': KeyInfo(can_disable=True),
            'port': KeyInfo(can_disable=True),
            'priority': KeyInfo(can_disable=True),
            'protocol': KeyInfo(can_disable=True),
            'random': KeyInfo(can_disable=True),
            'src-address': KeyInfo(can_disable=True),
            'src-address-list': KeyInfo(can_disable=True),
            'src-address-type': KeyInfo(can_disable=True),
            'src-mac-address': KeyInfo(can_disable=True),
            'src-port': KeyInfo(can_disable=True),
            'tcp-flags': KeyInfo(can_disable=True),
            'tcp-mss': KeyInfo(can_disable=True),
            'time': KeyInfo(can_disable=True),
        },
    ),
    ('ipv6', 'nd'): APIData(
        unknown_mechanism=True,
        # primary_keys=('default', ),
        fields={
            'default': KeyInfo(),
            'advertise-dns': KeyInfo(),
            'advertise-mac-address': KeyInfo(),
            'disabled': KeyInfo(),
            'hop-limit': KeyInfo(),
            'interface': KeyInfo(),
            'managed-address-configuration': KeyInfo(),
            'mtu': KeyInfo(),
            'other-configuration': KeyInfo(),
            'ra-delay': KeyInfo(),
            'ra-interval': KeyInfo(),
            'ra-lifetime': KeyInfo(),
            'reachable-time': KeyInfo(),
            'retransmit-interval': KeyInfo(),
        },
    ),
    ('ipv6', 'nd', 'prefix', 'default'): APIData(
        single_value=True,
        fully_understood=True,
        fields={
            'autonomous': KeyInfo(default=True),
            'preferred-lifetime': KeyInfo(default='1w'),
            'valid-lifetime': KeyInfo(default='4w2d'),
        },
    ),
    ('ipv6', 'route'): APIData(
        fields={
            'bgp-as-path': KeyInfo(can_disable=True),
            'bgp-atomic-aggregate': KeyInfo(can_disable=True),
            'bgp-communities': KeyInfo(can_disable=True),
            'bgp-local-pref': KeyInfo(can_disable=True),
            'bgp-med': KeyInfo(can_disable=True),
            'bgp-origin': KeyInfo(can_disable=True),
            'bgp-prepend': KeyInfo(can_disable=True),
            'check-gateway': KeyInfo(can_disable=True),
            'disabled': KeyInfo(),
            'distance': KeyInfo(),
            'dst-address': KeyInfo(),
            'gateway': KeyInfo(),
            'route-tag': KeyInfo(can_disable=True),
            'scope': KeyInfo(),
            'target-scope': KeyInfo(),
        },
    ),
    ('mpls', ): APIData(
        single_value=True,
        fully_understood=True,
        fields={
            'allow-fast-path': KeyInfo(default=True),
            'dynamic-label-range': KeyInfo(default='16-1048575'),
            'propagate-ttl': KeyInfo(default=True),
        },
    ),
    ('mpls', 'interface'): APIData(
        unknown_mechanism=True,
        # primary_keys=('default', ),
        fields={
            'default': KeyInfo(),
            'disabled': KeyInfo(),
            'interface': KeyInfo(),
            'mpls-mtu': KeyInfo(),
        },
    ),
    ('mpls', 'ldp'): APIData(
        single_value=True,
        fully_understood=True,
        fields={
            'distribute-for-default-route': KeyInfo(default=False),
            'enabled': KeyInfo(default=False),
            'hop-limit': KeyInfo(default=255),
            'loop-detect': KeyInfo(default=False),
            'lsr-id': KeyInfo(default='0.0.0.0'),
            'path-vector-limit': KeyInfo(default=255),
            'transport-address': KeyInfo(default='0.0.0.0'),
            'use-explicit-null': KeyInfo(default=False),
        },
    ),
    ('port', 'firmware'): APIData(
        single_value=True,
        fully_understood=True,
        fields={
            'directory': KeyInfo(default='firmware'),
            'ignore-directip-modem': KeyInfo(default=False),
        },
    ),
    ('ppp', 'aaa'): APIData(
        single_value=True,
        fully_understood=True,
        fields={
            'accounting': KeyInfo(default=True),
            'interim-update': KeyInfo(default='0s'),
            'use-circuit-id-in-nas-port-id': KeyInfo(default=False),
            'use-radius': KeyInfo(default=False),
        },
    ),
    ('radius', 'incoming'): APIData(
        single_value=True,
        fully_understood=True,
        fields={
            'accept': KeyInfo(default=False),
            'port': KeyInfo(default=3799),
        },
    ),
    ('routing', 'bfd', 'interface'): APIData(
        unknown_mechanism=True,
        # primary_keys=('default', ),
        fields={
            'default': KeyInfo(),
            'disabled': KeyInfo(),
            'interface': KeyInfo(),
            'interval': KeyInfo(),
            'min-rx': KeyInfo(),
            'multiplier': KeyInfo(),
        },
    ),
    ('routing', 'mme'): APIData(
        single_value=True,
        fully_understood=True,
        fields={
            'bidirectional-timeout': KeyInfo(default=2),
            'gateway-class': KeyInfo(default='none'),
            'gateway-keepalive': KeyInfo(default='1m'),
            'gateway-selection': KeyInfo(default='no-gateway'),
            'origination-interval': KeyInfo(default='5s'),
            'preferred-gateway': KeyInfo(default='0.0.0.0'),
            'timeout': KeyInfo(default='1m'),
            'ttl': KeyInfo(default=50),
        },
    ),
    ('routing', 'rip'): APIData(
        single_value=True,
        fully_understood=True,
        fields={
            'distribute-default': KeyInfo(default='never'),
            'garbage-timer': KeyInfo(default='2m'),
            'metric-bgp': KeyInfo(default=1),
            'metric-connected': KeyInfo(default=1),
            'metric-default': KeyInfo(default=1),
            'metric-ospf': KeyInfo(default=1),
            'metric-static': KeyInfo(default=1),
            'redistribute-bgp': KeyInfo(default=False),
            'redistribute-connected': KeyInfo(default=False),
            'redistribute-ospf': KeyInfo(default=False),
            'redistribute-static': KeyInfo(default=False),
            'routing-table': KeyInfo(default='main'),
            'timeout-timer': KeyInfo(default='3m'),
            'update-timer': KeyInfo(default='30s'),
        },
    ),
    ('routing', 'ripng'): APIData(
        single_value=True,
        fully_understood=True,
        fields={
            'distribute-default': KeyInfo(default='never'),
            'garbage-timer': KeyInfo(default='2m'),
            'metric-bgp': KeyInfo(default=1),
            'metric-connected': KeyInfo(default=1),
            'metric-default': KeyInfo(default=1),
            'metric-ospf': KeyInfo(default=1),
            'metric-static': KeyInfo(default=1),
            'redistribute-bgp': KeyInfo(default=False),
            'redistribute-connected': KeyInfo(default=False),
            'redistribute-ospf': KeyInfo(default=False),
            'redistribute-static': KeyInfo(default=False),
            'timeout-timer': KeyInfo(default='3m'),
            'update-timer': KeyInfo(default='30s'),
        },
    ),
    ('snmp', ): APIData(
        single_value=True,
        fully_understood=True,
        fields={
            'contact': KeyInfo(default=''),
            'enabled': KeyInfo(default=False),
            'engine-id': KeyInfo(default=''),
            'location': KeyInfo(default=''),
            'src-address': KeyInfo(default='::'),
            'trap-community': KeyInfo(default='public'),
            'trap-generators': KeyInfo(default='temp-exception'),
            'trap-target': KeyInfo(default=''),
            'trap-version': KeyInfo(default=1),
        },
    ),
    ('system', 'clock'): APIData(
        single_value=True,
        fully_understood=True,
        fields={
            'time-zone-autodetect': KeyInfo(default=True),
            'time-zone-name': KeyInfo(default='manual'),
        },
    ),
    ('system', 'clock', 'manual'): APIData(
        single_value=True,
        fully_understood=True,
        fields={
            'dst-delta': KeyInfo(default='00:00'),
            'dst-end': KeyInfo(default='jan/01/1970 00:00:00'),
            'dst-start': KeyInfo(default='jan/01/1970 00:00:00'),
            'time-zone': KeyInfo(default='+00:00'),
        },
    ),
    ('system', 'identity'): APIData(
        single_value=True,
        fully_understood=True,
        fields={
            'name': KeyInfo(default='Mikrotik'),
        },
    ),
    ('system', 'leds', 'settings'): APIData(
        single_value=True,
        fully_understood=True,
        fields={
            'all-leds-off': KeyInfo(default='never'),
        },
    ),
    ('system', 'note'): APIData(
        single_value=True,
        fully_understood=True,
        fields={
            'note': KeyInfo(default=''),
            'show-at-login': KeyInfo(default=True),
        },
    ),
    ('system', 'ntp', 'client'): APIData(
        single_value=True,
        fully_understood=True,
        fields={
            'enabled': KeyInfo(default=False),
            'primary-ntp': KeyInfo(default='0.0.0.0'),
            'secondary-ntp': KeyInfo(default='0.0.0.0'),
            'server-dns-names': KeyInfo(default=''),
        },
    ),
    ('system', 'package', 'update'): APIData(
        single_value=True,
        fully_understood=True,
        fields={
            'channel': KeyInfo(default='stable'),
        },
    ),
    ('system', 'routerboard', 'settings'): APIData(
        single_value=True,
        fully_understood=True,
        fields={
            'auto-upgrade': KeyInfo(default=False),
            'baud-rate': KeyInfo(default=115200),
            'boot-delay': KeyInfo(default='2s'),
            'boot-device': KeyInfo(default='nand-if-fail-then-ethernet'),
            'boot-protocol': KeyInfo(default='bootp'),
            'enable-jumper-reset': KeyInfo(default=True),
            'enter-setup-on': KeyInfo(default='any-key'),
            'force-backup-booter': KeyInfo(default=False),
            'protected-routerboot': KeyInfo(default='disabled'),
            'reformat-hold-button': KeyInfo(default='20s'),
            'reformat-hold-button-max': KeyInfo(default='10m'),
            'silent-boot': KeyInfo(default=False),
        },
    ),
    ('system', 'upgrade', 'mirror'): APIData(
        single_value=True,
        fully_understood=True,
        fields={
            'check-interval': KeyInfo(default='1d'),
            'enabled': KeyInfo(default=False),
            'primary-server': KeyInfo(default='0.0.0.0'),
            'secondary-server': KeyInfo(default='0.0.0.0'),
            'user': KeyInfo(default=''),
        },
    ),
    ('system', 'watchdog'): APIData(
        single_value=True,
        fully_understood=True,
        fields={
            'auto-send-supout': KeyInfo(default=False),
            'automatic-supout': KeyInfo(default=True),
            'ping-start-after-boot': KeyInfo(default='5m'),
            'ping-timeout': KeyInfo(default='1m'),
            'watch-address': KeyInfo(default='none'),
            'watchdog-timer': KeyInfo(default=True),
        },
    ),
    ('tool', 'bandwidth-server'): APIData(
        single_value=True,
        fully_understood=True,
        fields={
            'allocate-udp-ports-from': KeyInfo(default=2000),
            'authenticate': KeyInfo(default=True),
            'enabled': KeyInfo(default=True),
            'max-sessions': KeyInfo(default=100),
        },
    ),
    ('tool', 'e-mail'): APIData(
        single_value=True,
        fully_understood=True,
        fields={
            'address': KeyInfo(default='0.0.0.0'),
            'from': KeyInfo(default='<>'),
            'password': KeyInfo(default=''),
            'port': KeyInfo(default=25),
            'start-tls': KeyInfo(default=False),
            'user': KeyInfo(default=''),
        },
    ),
    ('tool', 'graphing'): APIData(
        single_value=True,
        fully_understood=True,
        fields={
            'page-refresh': KeyInfo(default=300),
            'store-every': KeyInfo(default='5min'),
        },
    ),
    ('tool', 'mac-server'): APIData(
        single_value=True,
        fully_understood=True,
        fields={
            'allowed-interface-list': KeyInfo(),
        },
    ),
    ('tool', 'mac-server', 'mac-winbox'): APIData(
        single_value=True,
        fully_understood=True,
        fields={
            'allowed-interface-list': KeyInfo(),
        },
    ),
    ('tool', 'mac-server', 'ping'): APIData(
        single_value=True,
        fully_understood=True,
        fields={
            'enabled': KeyInfo(default=True),
        },
    ),
    ('tool', 'romon'): APIData(
        single_value=True,
        fully_understood=True,
        fields={
            'enabled': KeyInfo(default=False),
            'id': KeyInfo(default='00:00:00:00:00:00'),
            'secrets': KeyInfo(default=''),
        },
    ),
    ('tool', 'romon', 'port'): APIData(
        fields={
            'cost': KeyInfo(),
            'disabled': KeyInfo(),
            'forbid': KeyInfo(),
            'interface': KeyInfo(),
            'secrets': KeyInfo(),
        },
    ),
    ('tool', 'sms'): APIData(
        single_value=True,
        fully_understood=True,
        fields={
            'allowed-number': KeyInfo(default=''),
            'auto-erase': KeyInfo(default=False),
            'channel': KeyInfo(default=0),
            'port': KeyInfo(default='none'),
            'receive-enabled': KeyInfo(default=False),
            'secret': KeyInfo(default=''),
            'sim-pin': KeyInfo(default=''),
        },
    ),
    ('tool', 'sniffer'): APIData(
        single_value=True,
        fully_understood=True,
        fields={
            'file-limit': KeyInfo(default='1000KiB'),
            'file-name': KeyInfo(default=''),
            'filter-cpu': KeyInfo(default=''),
            'filter-direction': KeyInfo(default='any'),
            'filter-interface': KeyInfo(default=''),
            'filter-ip-address': KeyInfo(default=''),
            'filter-ip-protocol': KeyInfo(default=''),
            'filter-ipv6-address': KeyInfo(default=''),
            'filter-mac-address': KeyInfo(default=''),
            'filter-mac-protocol': KeyInfo(default=''),
            'filter-operator-between-entries': KeyInfo(default='or'),
            'filter-port': KeyInfo(default=''),
            'filter-size': KeyInfo(default=''),
            'filter-stream': KeyInfo(default=False),
            'memory-limit': KeyInfo(default='100KiB'),
            'memory-scroll': KeyInfo(default=True),
            'only-headers': KeyInfo(default=False),
            'streaming-enabled': KeyInfo(default=False),
            'streaming-server': KeyInfo(default='0.0.0.0:37008'),
        },
    ),
    ('tool', 'traffic-generator'): APIData(
        single_value=True,
        fully_understood=True,
        fields={
            'latency-distribution-max': KeyInfo(default='100us'),
            'measure-out-of-order': KeyInfo(default=True),
            'stats-samples-to-keep': KeyInfo(default=100),
            'test-id': KeyInfo(default=0),
        },
    ),
    ('user', 'aaa'): APIData(
        single_value=True,
        fully_understood=True,
        fields={
            'accounting': KeyInfo(default=True),
            'default-group': KeyInfo(default='read'),
            'exclude-groups': KeyInfo(default=''),
            'interim-update': KeyInfo(default='0s'),
            'use-radius': KeyInfo(default=False),
        },
    ),
    ('queue', 'interface'): APIData(
        primary_keys=('name', ),
        fully_understood=True,
        fields={
            'name': KeyInfo(required=True),
            'queue': KeyInfo(required=True),
        },
    ),
    ('interface', 'ethernet', 'switch'): APIData(
        fixed_entries=True,
        primary_keys=('name', ),
        fully_understood=True,
        fields={
            'cpu-flow-control': KeyInfo(default=True),
            'mirror-source': KeyInfo(default='none'),
            'mirror-target': KeyInfo(default='none'),
            'name': KeyInfo(),
        },
    ),
    ('interface', 'ethernet', 'switch', 'port'): APIData(
        fixed_entries=True,
        primary_keys=('name', ),
        fully_understood=True,
        fields={
            'default-vlan-id': KeyInfo(),
            'name': KeyInfo(),
            'vlan-header': KeyInfo(default='leave-as-is'),
            'vlan-mode': KeyInfo(default='disabled'),
        },
    ),
    ('ip', 'dhcp-client', 'option'): APIData(
        fixed_entries=True,
        primary_keys=('name', ),
        fully_understood=True,
        fields={
            'code': KeyInfo(),
            'name': KeyInfo(),
            'value': KeyInfo(),
        },
    ),
    ('ppp', 'profile'): APIData(
        has_identifier=True,
        fields={
            'address-list': KeyInfo(),
            'bridge': KeyInfo(can_disable=True),
            'bridge-horizon': KeyInfo(can_disable=True),
            'bridge-learning': KeyInfo(),
            'bridge-path-cost': KeyInfo(can_disable=True),
            'bridge-port-priority': KeyInfo(can_disable=True),
            'change-tcp-mss': KeyInfo(),
            'dns-server': KeyInfo(can_disable=True),
            'idle-timeout': KeyInfo(can_disable=True),
            'incoming-filter': KeyInfo(can_disable=True),
            'insert-queue-before': KeyInfo(can_disable=True),
            'interface-list': KeyInfo(can_disable=True),
            'local-address': KeyInfo(can_disable=True),
            'name': KeyInfo(),
            'on-down': KeyInfo(),
            'on-up': KeyInfo(),
            'only-one': KeyInfo(),
            'outgoing-filter': KeyInfo(can_disable=True),
            'parent-queue': KeyInfo(can_disable=True),
            'queue-type': KeyInfo(can_disable=True),
            'rate-limit': KeyInfo(can_disable=True),
            'remote-address': KeyInfo(can_disable=True),
            'session-timeout': KeyInfo(can_disable=True),
            'use-compression': KeyInfo(),
            'use-encryption': KeyInfo(),
            'use-ipv6': KeyInfo(),
            'use-mpls': KeyInfo(),
            'use-upnp': KeyInfo(),
            'wins-server': KeyInfo(can_disable=True),
        },
    ),
    ('queue', 'type'): APIData(
        has_identifier=True,
        fields={
            'kind': KeyInfo(),
            'mq-pfifo-limit': KeyInfo(),
            'name': KeyInfo(),
            'pcq-burst-rate': KeyInfo(),
            'pcq-burst-threshold': KeyInfo(),
            'pcq-burst-time': KeyInfo(),
            'pcq-classifier': KeyInfo(),
            'pcq-dst-address-mask': KeyInfo(),
            'pcq-dst-address6-mask': KeyInfo(),
            'pcq-limit': KeyInfo(),
            'pcq-rate': KeyInfo(),
            'pcq-src-address-mask': KeyInfo(),
            'pcq-src-address6-mask': KeyInfo(),
            'pcq-total-limit': KeyInfo(),
            'pfifo-limit': KeyInfo(),
            'red-avg-packet': KeyInfo(),
            'red-burst': KeyInfo(),
            'red-limit': KeyInfo(),
            'red-max-threshold': KeyInfo(),
            'red-min-threshold': KeyInfo(),
            'sfq-allot': KeyInfo(),
            'sfq-perturb': KeyInfo(),
        },
    ),
    ('routing', 'bgp', 'instance'): APIData(
        fixed_entries=True,
        primary_keys=('name', ),
        fully_understood=True,
        fields={
            'as': KeyInfo(),
            'client-to-client-reflection': KeyInfo(),
            'cluster-id': KeyInfo(can_disable=True),
            'confederation': KeyInfo(can_disable=True),
            'disabled': KeyInfo(),
            'ignore-as-path-len': KeyInfo(),
            'name': KeyInfo(),
            'out-filter': KeyInfo(),
            'redistribute-connected': KeyInfo(),
            'redistribute-ospf': KeyInfo(),
            'redistribute-other-bgp': KeyInfo(),
            'redistribute-rip': KeyInfo(),
            'redistribute-static': KeyInfo(),
            'router-id': KeyInfo(),
            'routing-table': KeyInfo(),
        },
    ),
    ('system', 'logging', 'action'): APIData(
        has_identifier=True,
        fields={
            'bsd-syslog': KeyInfo(),
            'disk-file-count': KeyInfo(),
            'disk-file-name': KeyInfo(),
            'disk-lines-per-file': KeyInfo(),
            'disk-stop-on-full': KeyInfo(),
            'memory-lines': KeyInfo(),
            'memory-stop-on-full': KeyInfo(),
            'name': KeyInfo(),
            'remember': KeyInfo(),
            'remote': KeyInfo(),
            'remote-port': KeyInfo(),
            'src-address': KeyInfo(),
            'syslog-facility': KeyInfo(),
            'syslog-severity': KeyInfo(),
            'syslog-time-format': KeyInfo(),
            'target': KeyInfo(),
        },
    ),
    ('user', 'group'): APIData(
        fixed_entries=True,
        primary_keys=('name', ),
        fully_understood=True,
        fields={
            'name': KeyInfo(),
            'policy': KeyInfo(),
            'skin': KeyInfo(default='default'),
        },
    ),
    ('caps-man', 'manager'): APIData(
        single_value=True,
        fields={
            'ca-certificate': KeyInfo(default='none'),
            'certificate': KeyInfo(default='none'),
            'enabled': KeyInfo(default=False),
            'package-path': KeyInfo(default=''),
            'require-peer-certificate': KeyInfo(default=False),
            'upgrade-policy': KeyInfo(default='none'),
        },
    ),
    ('ip', 'firewall', 'service-port'): APIData(
        primary_keys=('name', ),
        fully_understood=True,
        fields={
            'disabled': KeyInfo(default=False),
            'name': KeyInfo(),
            'ports': KeyInfo(),
            'sip-direct-media': KeyInfo(),
            'sip-timeout': KeyInfo(),
        },
    ),
    ('ip', 'hotspot', 'service-port'): APIData(
        fixed_entries=True,
        primary_keys=('name', ),
        fully_understood=True,
        fields={
            'disabled': KeyInfo(default=False),
            'name': KeyInfo(),
            'ports': KeyInfo(),
        },
    ),
    ('ip', 'ipsec', 'policy'): APIData(
        has_identifier=True,
        fields={
            'disabled': KeyInfo(),
            'dst-address': KeyInfo(),
            'group': KeyInfo(),
            'proposal': KeyInfo(),
            'protocol': KeyInfo(),
            'src-address': KeyInfo(),
            'template': KeyInfo(),
        },
    ),
    ('ip', 'service'): APIData(
        fixed_entries=True,
        primary_keys=('name', ),
        fully_understood=True,
        fields={
            'address': KeyInfo(),
            'certificate': KeyInfo(),
            'disabled': KeyInfo(default=False),
            'name': KeyInfo(),
            'port': KeyInfo(),
            'tls-version': KeyInfo(),
        },
    ),
    ('system', 'logging'): APIData(
        has_identifier=True,
        fields={
            'action': KeyInfo(),
            'disabled': KeyInfo(),
            'prefix': KeyInfo(),
            'topics': KeyInfo(),
        },
    ),
    ('system', 'resource', 'irq'): APIData(
        has_identifier=True,
        fields={
            'cpu': KeyInfo(),
        },
    ),
}
