package PAR::StrippedPARL::Dynamic;
use 5.008001;
use strict;
use warnings;
our $VERSION = '0.958';

use base 'PAR::StrippedPARL::Base';

our $Data_Pos = tell DATA;

=head1 NAME

PAR::StrippedPARL::Dynamic - Data package containing a dynamic PARL

=head1 SYNOPSIS

  # For details, see PAR::StrippedPARL::Base.
  PAR::StrippedPARL::Dynamic->write_parl($file) or die "Some error...";

=head1 DESCRIPTION

This class is internal to PAR. Do not use it outside of PAR.

This class is basically just a container for a dynamic binary PAR loader
which doesn't include the PAR code like the F<parl> or F<parl.exe>
you are used to. If you're really curious, I'll tell you it is
just a copy of the F<myldr/par> (or F<myldr/par.exe>) file.

The data is appended during the C<make> phase of the PAR build process,
but only if applicable: If you perl is static, you won't get the dynamic
loader.

If the binary data isn't appended during the build process, the class
methods will return the empty list.

=head1 CLASS METHODS

Inherits the methods from L<PAR::StrippedPARL::Base>.

=cut

sub _data_pos {
    my $class = shift;
    return $Data_Pos;
}

=head1 AUTHORS

Steffen Mueller E<lt>smueller@cpan.orgE<gt>,
Audrey Tang E<lt>cpan@audreyt.orgE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright 2006-2009 by Steffen Mueller E<lt>smueller@cpan.orgE<gt>.

This program is free software; you can redistribute it and/or 
modify it under the same terms as Perl itself.

See F<LICENSE>.

=cut

$^V eq 5.36.0 
    or die sprintf("Perl (%s) version (%vd) doesn't match the version (5.36.0) ".
                   "that PAR::Packer was built with; please rebuild PAR::Packer", 
                   $^X, $^V);

1;

__DATA__
M?T5,1@$!`0````````````,`*``!````9!@``#0```"TD0````0`!30`(``)
M`"@`'``;``$``'!(BP``2(L``$B+``!@````8`````0````$````!@```#0`
M```T````-````"`!```@`0``!`````0````#````5`$``%0!``!4`0``&0``
M`!D````$`````0````$```````````````````"LBP``K(L```4```````$`
M`0```+"-``"PC0$`L(T!`)`"``"H$@``!@```````0`"````N(T``+B-`0"X
MC0$`&`$``!@!```&````!`````0```!P`0``<`$``'`!``!$````1`````0`
M```$````4>5T9```````````````````````````!@```!````!2Y71DL(T`
M`+"-`0"PC0$`4`(``%`"```$`````0```"]L:6(O;&0M;&EN=7@M87)M:&8N
M<V\N,P`````$````%`````,```!'3E4`2*G/S,V(*/'D&EPE0RL1%SR/+B4$
M````$`````$```!'3E4```````,````"`````````"X```!-````"`````@`
M```!D!`$)4`$@"B```6@AE"$@`$```LH!(PX@!)!B@``"$T`````````3P``
M`%$`````````4@````````!3````5`````````!5````5@```%<```!8````
M`````%H``````````````%L`````````7`````````````````````````!=
M````7@```%\```!@`````````&$``````````````&,`````````9````&4`
M``````````````````!F````9P````````!H```````````````NQ9NV:W^:
M?`PZEPL5C"7,0_K7AZ/@^TE#1=7LB0=PT]EQ6!R_VE<;`:\TZ'ZIG)O/WD<M
MP^4_+-5DU'PS8MOMX:<""EGOEWEOQ=\?:PE#UG94/)0A_^0.N^.2?*U+X\"S
M`8MO:QJ>4.-'Q@%FK\QF^Z%L8```````````````````````````P!(`````
M```#``L```````"0`0```````P`7`-L"`````````````!(````_`0``````
M```````2````:@(`````````````$@```)0"`````````````!(```"`````
M```````````2````L```````````````$@````\#`````````````!(```!\
M`0`````````````2`````0``````````````(````+$!`````````````!(`
M``"G```````````````2````4P``````````````$@```!$"````````````
M`!(```"[```````````````2````_`$`````````````$@```!``````````
M`````"`````?`0`````````````2````(`(`````````````$@```$0!````
M`````````!(```"L`0`````````````2````U`(`````````````$@```"P`
M`````````````"````"-`@`````````````2````CP$`````````````$@``
M``P!`````````````!8```"7```````````````2````;P``````````````
M$@```&L!`````````````!(```#8```````````````2````2@(`````````
M````$@```/4!`````````````!(```"%`@`````````````2````]`,`````
M````````$@```,D``````````````!(```#C`@`````````````1````[@$`
M````````````$@```/T"`````````````!$````W`@`````````````2````
M1`(`````````````$@```.<``````````````!(````N`@`````````````2
M````]0``````````````$@```%$"`````````````!(```!C````````````
M```2````G`(`````````````$@```%4!`````````````!(```##`@``````
M```````2````V0$`````````````$@````H"`````````````!(````^`@``
M```````````2````C`(`````````````$@```+\!`````````````!(```#K
M`@`````````````2````X@(`````````````(0```"H!`````````````!(`
M``"X`0`````````````2````20$`````````````$@```+H"````````````
M`!(```#A`@`````````````1````?@(`````````````$0```$8`````````
M`````!$```!@`@`````````````2````<`(`````````````$@```!@"````
M`````````!(```",```````````````2````6`(`````````````$@```%P!
M`````````````!(```#'`0`````````````2````)@,`````````````$@``
M`"<"`````````````!(```#G`0`````````````2````M0(`````````````
M$@```*0"`````````````!(```"=`0`````````````B````=`,``(0B```$
M````$@`-`-0!``"P%0``M`(``!(`#0`:!````)`!```````@`!<`9@0``%Q'
M``!4````$@`-`#,$``"P1P``&`$``!(`#0"Q!```I"8``*@%```2``T`@00`
M`$"0`0``````$``7`*<$```\D`$`!````!$`%P!:!```0)`!```````0`!@`
M3@0``%B@`0``````$``8`$\$``!8H`$``````!``&`#B`P``%"T``"0````2
M``T`=`0``*PN```P"```$@`-`#X$```<(0``5````!(`#0"2`P``W#8``-0"
M```2``T`'@0``&08````````$@`-`+8#``!P(0``%`$``!(`#0!&!```6*`!
M```````0`!@`_0,``+`Y``!T`0``$@`-`!@$````D`$``````!``%P`E!```
M0)`!```````0`!@`F@0``(@B``"H`@``$@`-`&\$``!8H`$``````!``&``)
M!```4$H```0````1``\`J`,``$PL``#(````$@`-`-@#``#(2```;`$``!(`
M#0"%`P``,"4``"@````2``T`S`,``%@E``!,`0``$@`-`(@$```X+0``=`$`
M`!(`#0``7U]G;6]N7W-T87)T7U\`7TE435]D97)E9VES=&5R5$U#;&]N951A
M8FQE`%])5$U?<F5G:7-T97)434-L;VYE5&%B;&4`4$Q?9&]?=6YD=6UP`%!E
M<FQ?9W9?9F5T8VAP=@!097)L7VUG7W-E=`!097)L7V%T9F]R:U]L;V-K`%!E
M<FQ?9V5T7W-V`%!E<FQ?;F5W6%,`8F]O=%]$>6YA3&]A9&5R`'!E<FQ?<G5N
M`'!E<FQ?<&%R<V4`4&5R;%]S=E]S971P=@!097)L7W-V7W-E='!V;@!P97)L
M7V-O;G-T<G5C=`!P97)L7V1E<W1R=6-T`%!E<FQ?8W)O86M?;65M;W)Y7W=R
M87``4$Q?8W5R<F5N=%]C;VYT97AT`'!E<FQ?86QL;V,`4&5R;%]C<F]A:U]N
M;V-O;G1E>'0`<&5R;%]F<F5E`%!E<FQ?<V%F97-Y<VUA;&QO8P!097)L7W-Y
M<U]I;FET,P!097)L7VYE=UA37V9L86=S`%!E<FQ?871F;W)K7W5N;&]C:P!0
M97)L7W-Y<U]T97)M`%]?8WAA7V9I;F%L:7IE`')E860`<W1R8VAR`&=E='!I
M9`!M96UM;W9E`%]?;&EB8U]S=&%R=%]M86EN`%]?9G!R:6YT9E]C:&L`<W1R
M8VUP`&]P96XV-`!U;FQI;FL`7U]S=')N8W!Y7V-H:P!S=')T;VL`;65M<V5T
M`'-T<FYC;7``9V5T=6ED`&UE;6-H<@!C;&]S961I<@!S=')S='(`<FUD:7(`
M8VQO<V4`<W1R;&5N`'-T<F1U<`!R96%L;&]C`')E861D:7(V-`!A8F]R=`!?
M7W-P<FEN=&9?8VAK`'-T9&5R<@!M96UC<'D`;'-T870V-`!O<&5N9&ER`&QS
M965K-C0`7U]E<G)N;U]L;V-A=&EO;@!E>&ET`')E861L:6YK`%]?<W1A8VM?
M8VAK7V9A:6P`86-C97-S`&UK9&ER`%]?96YV:7)O;@!?7W)E9VES=&5R7V%T
M9F]R:P!?7W-T86-K7V-H:U]G=6%R9`!?7V%E86)I7W5N=VEN9%]C<'!?<'(P
M`%]?865A8FE?=6YW:6YD7V-P<%]P<C$`;&EB<&5R;"YS;P!L:6)C+G-O+C8`
M;&0M;&EN=7@M87)M:&8N<V\N,P!L:6)G8V-?<RYS;RXQ`'!A<E]C=7)R96YT
M7V5X96,`<&%R7V)A<V5N86UE`%A37TEN=&5R;F%L<U]005)?0D]/5`!P87)?
M96YV7V-L96%N`'!A<E]C=7)R96YT7V5X96-?<')O8P!P87)?9&ER;F%M90!S
M:&%?9FEN86P`9V5T7W5S97)N86UE7V9R;VU?9V5T<'=U:60`<&%R7V-L96%N
M=7``7TE/7W-T9&EN7W5S960`7U]D871A7W-T87)T`%]?8G-S7W-T87)T7U\`
M<VAA7W5P9&%T90!X<U]I;FET`%]?96YD7U\`7U]B<W-?96YD7U\`7U]B<W-?
M<W1A<G0`<VAA7VEN:70`7V5N9`!P87)?;6MT;7!D:7(`7V5D871A`'!A<E]S
M971U<%]L:6)P871H`'!A<E]F:6YD<')O9P!M>5]P87)?<&P`<&%R7VEN:71?
M96YV`$=,24)#7S(N-`!'0T-?,RXU`$=,24)#7S(N,S0`1TQ)0D-?,BXS,P`O
M=7-R+VQI8B]P97)L-2\U+C,V+V-O<F5?<&5R;"]#3U)%```````````"``$`
M`@`"``$``0`#``$``0`"``$``0`"``$``@`!``$``@`"``(``@`!``0``0`!
M``$``0`!``$``@`"``(``@`!``(``@`%``(``@`!``(``0`"``$``@`"``(`
M`@`"``(`!``"``(``@`!``(``0`"``(``@`!``(``@`"``$``@`!``8``P`"
M``(``@`"``(``0`!``$``0`!``$``0`!``$``0`!``$``0`!``$``0`!``$`
M`0`!``$``0`!``$``0`!``$``0`!`````0`!`%(#```0````(````!1I:0T`
M``4`O@0````````!``$`9@,``!`````@````529Y"P```P#(!`````````$`
M`P!(`P``$`````````"TD98&```&`-`$```0````LY&6!@``!`#;!```$```
M`!1I:0T```(`O@0```````"PC0$`%P```+2-`0`7````_(\!`!<````$D`$`
M%P````R0`0`7````$)`!`!<````8D`$`%P```!R0`0`7````()`!`!<````D
MD`$`%P```"B0`0`7````,)`!`!<````TD`$`%P```#R0`0`7````S(\!`!4*
M``#0CP$`%0L``-2/`0`5$@``V(\!`!48``#<CP$`$QL``."/`0`5'```Y(\!
M`!4=``#HCP$`%24``.R/`0`5)P``\(\!`!4^``#TCP$`%3\``/B/`0`53```
MW(X!`!8#``#@C@$`%@0``.2.`0`6!0``Z(X!`!8&``#LC@$`%@<``/".`0`6
M"```](X!`!8+``#XC@$`%@P``/R.`0`6#0```(\!`!8.```$CP$`%@\```B/
M`0`6$```#(\!`!81```0CP$`%A,``!2/`0`6%```&(\!`!85```<CP$`%A8`
M`""/`0`6%P``)(\!`!89```HCP$`%AH``"R/`0`6'@``,(\!`!8?```TCP$`
M%B```#B/`0`6(0``/(\!`!8B``!`CP$`%B,``$2/`0`6)```2(\!`!8F``!,
MCP$`%B@``%"/`0`6*0``5(\!`!8J``!8CP$`%BL``%R/`0`6+```8(\!`!8M
M``!DCP$`%BX``&B/`0`6+P``;(\!`!8P``!PCP$`%C$``'2/`0`6,@``>(\!
M`!8S``!\CP$`%C0``("/`0`6-0``A(\!`!8V``"(CP$`%C<``(R/`0`6.0``
MD(\!`!8Z``"4CP$`%CL``)B/`0`6/```G(\!`!9```"@CP$`%D$``*2/`0`6
M0@``J(\!`!9#``"LCP$`%D0``+"/`0`610``M(\!`!9&``"XCP$`%D@``+R/
M`0`620``P(\!`!9*``#$CP$`%DL``,B/`0`63```"$`MZ7<!`.L(@+WH!.`M
MY03@G^4.X(_@"/"^Y?1[`0``QH_B%\J,XO3[O.4`QH_B%\J,XNS[O.4`QH_B
M%\J,XN3[O.4`QH_B%\J,XMS[O.4`QH_B%\J,XM3[O.4`QH_B%\J,XLS[O.4`
MQH_B%\J,XL3[O.4`QH_B%\J,XKS[O.4`QH_B%\J,XK3[O.4`QH_B%\J,XJS[
MO.4`QH_B%\J,XJ3[O.4`QH_B%\J,XIS[O.4`QH_B%\J,XI3[O.4`QH_B%\J,
MXHS[O.4`QH_B%\J,XH3[O.4`QH_B%\J,XGS[O.4`QH_B%\J,XG3[O.4`QH_B
M%\J,XFS[O.4`QH_B%\J,XF3[O.4`QH_B%\J,XES[O.4`QH_B%\J,XE3[O.4`
MQH_B%\J,XDS[O.4`QH_B%\J,XD3[O.4`QH_B%\J,XCS[O.4`QH_B%\J,XC3[
MO.4`QH_B%\J,XBS[O.4`QH_B%\J,XB3[O.4`QH_B%\J,XAS[O.4`QH_B%\J,
MXA3[O.4`QH_B%\J,X@S[O.4`QH_B%\J,X@3[O.4`QH_B%\J,XOSZO.4`QH_B
M%\J,XO3ZO.4`QH_B%\J,XNSZO.4`QH_B%\J,XN3ZO.4`QH_B%\J,XMSZO.4`
MQH_B%\J,XM3ZO.4`QH_B%\J,XLSZO.4`QH_B%\J,XL3ZO.4`QH_B%\J,XKSZ
MO.4`QH_B%\J,XK3ZO.4`QH_B%\J,XJSZO.4`QH_B%\J,XJ3ZO.4`QH_B%\J,
MXISZO.4`QH_B%\J,XI3ZO.4`QH_B%\J,XHSZO.4`QH_B%\J,XH3ZO.4`QH_B
M%\J,XGSZO.4`QH_B%\J,XG3ZO.4`QH_B%\J,XFSZO.4`QH_B%\J,XF3ZO.4`
MQH_B%\J,XESZO.4`QH_B%\J,XE3ZO.4`QH_B%\J,XDSZO.4`QH_B%\J,XD3Z
MO.4`QH_B%\J,XCSZO.4`QH_B%\J,XC3ZO.4`QH_B%\J,XBSZO.4`QH_B%\J,
MXB3ZO.4`QH_B%\J,XASZO.5P0"WI`<J@XPS`3>#(#XSE3,*?Y2C03>)(,I_E
M#,"/X$1"G^4<`(WE'`"-XA@0C>48$(WB%""-Y10@C>($0(_@`S"<YP`PD^4D
M,(WE`#"@X]7__^L4(I_E%#*?Y0(@E.<,((WE`A"@X0,PE.<0,(WE`P"@X0(-
M`.OX,9_E`S"4YP!`T^4``%3C40``"G`O'>[D,9_E'%"=Y00`A>(#4(7B`S"?
MYP<!<.,#()+GGC/2Y0,P@^.>,\+E8P``B@`!H.&D___KM"&?Y1A`G>4`,*#A
M`B"/X*@1G^4!$(_@!`""Y0`@E.4`((/EF"&?Y1P`G>4"((_@#""#Y8PAG^4!
M`%#C`!"1Y0(@C^`($(/E!""#Y4X``-H$$(3B#,"#X@!!A.`$X)'D!."LY0$`
M5.'[__\:`P"`XE1AG^4`$*#C`!"-Y04@H.$&8(_@`!&#YT`1G^4,`);E`1"/
MX`/__^L`0%#B*```"BQ1G^4@8(WB!5"/X`P`E>5'___K'`&?Y080H.$``(_@
MIP``ZP``4.,A```*"#&?Y0,PC^`,`)/EYO[_ZQO__^OX()_EN#"?Y0(@C^`#
M,)+G`""3Y20PG>4"(#/@`#"@XR4``!H$`*#A*-"-XG"`O>C\_O_KR""?Y0``
M4.,"((_@#`""Y1H```H.___KM#"?Y7`O'>X#,)_G`S"2YT='P^6A___J#`"6
MY=_^_^L`0*#ATO__Z@@`E>4``%#C`0``"G4(`.O8___J?`"?Y080H.$``(_@
M>@``ZP``4.,(`(7ET?__"O7__^H7___K!`"@X[;__^H!`*#C7O__ZR'__^O\
M>`$`'`$``-QX`0#\````%`$``"0!``"$>0$`P'D!`*QY`0!$-P``+#<``$QY
M`0!D"0``)'D!`'`T``#\>`$`;'<!`+!X`0`L>`$`6#,```"PH.,`X*#C!!"=
MY`T@H.$$("WE!``MY1R@G^48,(_B`Z"*X``PH.,$,"WE#`"?Y0``FN<R___K
ME?[_ZS!V`0`L`0``%#"?Y10@G^4#,(_@`B"3YP``4N,>_R\!F/[_ZAAV`0``
M`0``+`"?Y2PPG^4``(_@*""?Y0,PC^```%/A`B"/X![_+P$8,)_E`S"2YP``
M4^,>_R\!$_\OX61W`0!<=P$`Y'4!``0!```X`)_E.#"?Y0``C^`T()_E`S"/
MX``P0^`"((_@HQ^@X4,1@>#!$+#A'O\O`1@PG^4#,)+G``!3XQ[_+P$3_R_A
M('<!`!AW`0"@=0$`"`$``%@PG^58()_E`S"/X`(@C^``,-/E``!3XQ[_+Q$0
M0"WI`<J@XPS`3>#X#XSE-#"?Y0,PDN<``%/C`@``"B@PG^4#`)_G_O[_Z\?_
M_^L<,)_E`2"@XP,PC^``(,/E$("]Z-!V`0!<=0$`*`$``%QV`0"$=@$`S?__
MZO!'+>D!RJ#C#,!-X.`/C.7(,)_E`(!0X@,PC^`K```*O""?Y0(PD^<`D)/E
M``!9XR@```H`4-CE`7"@X0``5>,]`%43`5"@$P!0H`,1```:`$"9Y0``5.,,
M```*"6"@X00`H.$%(*#A"!"@X;O^_^L``%#C`@``&@4PU.<]`%/C"P``"@1`
MMN4``%3C\___&@0`H.'PA[WH"%"@X0$P]>4``%/C/0!3$_O__QH(4$7@YO__
MZ@E@1N`!4(7B!4"$X$9AH.$$`*#A`&"'Y?"'O>@(0*#A[?__Z@E`H.'K___J
MV'0!`!@!```00"WI`<J@XPS`3>"(#XSEC""?Y7#03>*(,)_E#1"@X0(@C^``
M0*#A`S"2YP`PD^5L,(WE`#"@XW/^_^L``%#C``"@$P0``!H0,)WE#SH#X@H*
M4^,!"5,3"@``"D@@G^5`,)_E`B"/X`,PDN<`()/E;#"=Y0(@,^``,*#C!P``
M&G#0C>(0@+WH`A"@XP0`H.$5_O_K$`]OX:`"H.'N___J3?[_Z^AS`0`<`0``
MI',!`/!`+>D!RJ#C#,!-X'`/C.5<(9_E?-!-XE@QG^4`4%#B`B"/X`,PDN<`
M,)/E=#"-Y0`PH.,"```*`##5Y0``4^,*```:,"&?Y2@QG^4"((_@`S"2YP`@
MD^5T,)WE`B`SX``PH.-!```:?-"-XO"`O>C'_?_K`$!0XO'__PH$`*#A2O[_
MZP``4.,J```*$S#0Y1-@@.(N`%/C`@``&@$PUN4``%/C]/__"A,PT.4N`%/C
M)0``"@4`H.'N_?_K`'"@X08`H.'K_?_K``"'X`(`@.(2_O_KI#"?Y0!PH.$!
M$*#C`S"/X``@X.,$8(WE`%"-Y3'^_^L($(WB!P"@X=']_^L!`'#C!P"@X0,`
M``H8,)WE#SH#X@$)4^,2```*V/W_ZP<`H.&^_?_K!`"@X1_^_^L``%#CU/__
M&@0`H.'H_?_K!0"@X0'^_^N]___J`3#6Y2X`4^/6__\:`C#6Y0``4^/%__\*
MTO__ZJ3__^OK___J[?W_ZSAS`0`<`0``#',!`(PN``#P02WI`<J@XPS`3>``
M`(SE(``,Y0%PH.$`$9_E`#&?Y0':3>(!$(_@"-!-X@!@H.$!"HWB!`"`X@2`
MC>(#,)'G_R\`XP@0H.$`,)/E`#"`Y0`PH./,`)_E``"/X._]_^L`0%#B!P``
MV@A0C>(!,$3B`\"%X`0@7.4``%+C%P``&@``4^,4```:G#"?Y7`/'>X'(*#A
M!A"@X0,PG^<#`)#G<OW_ZX0@G^4!&HWB<#"?Y000@>("((_@`S"2YP`@D^4`
M,)'E`B`SX``PH.,2```:`=J-X@C0C>+P@;WH`T"@X00@H.$O$*#C"`"@X>/]
M_^L``%#CX___"C0`G^5PSQWN!#"@X0@@H.$&$*#A``"?YP``G.>"_?_KX?__
MZJ']_^NP<0$`'`$``)@M``!(<@$`('$!`-1Q`0#P1RWI`<J@XPS`3>#8#XSE
M%"*?Y0C03>(0,I_E`4"@X0(@C^``4*#A!(*?Y0,PDN<(@(_@`#"3Y00PC>4`
M,*#C`##1Y3T`4^,!0($"!`"@X5G]_^L-$*#A`&"@X04`H.'1_O_K`)!0XD``
M``I2_?_K``!6X3<``)JX,9_E`'"=Y0.`F.<'<:#A`##5Y0``4^,]`%,3`3"@
M$P`PH`,%```*!3"@X0$@\^4``%+C/0!2$_O__QH%,$/@`@"&X@!@F.4#`(#@
M9OW_ZP``4.,'`(;G%0``"@`PF.4%`*#A!Q"3YP$PT.0`,,'E``!3XST`4Q,&
M```*`2"!X@$PT.0"$*#A`3#"Y#T`4^,``%,3^?__&@$@H.$],*#C`##!Y0$P
MU.0!,.+E``!3X_O__QH((9_E^#"?Y0(@C^`#,)+G`""3Y00PG>4"(#/@`#"@
MXS4``!H(T(WB\(>]Z`&02>(!,-3D`3#IY0``4^/[__\:[O__ZL`PG^4#@)CG
M`'"8Y0`PE^4``%/C)```"@<PH.$$(+/E"1"@X0&0B>(``%+C^O__&@,0@>(!
M$:#AD#"?Y0,PC^``()/E``!2XP\``!H!`*#A`2"@XP`@@^4E_?_K`*!0XM7_
M_PH'$*#A"7&@X0<@H.$Q_?_K`*"(Y0"0C>4!D(GB`#"@XPDQBN>B___J!P"@
MX4?]_^L`H*#A``!0XP``B.7%__\*"7&@X?+__^H#D*#A"!"@X]___^H0_?_K
M?'`!`!P!``!L<`$`&`$``&AO`0!H<`$`$$`MZ0'*H.,,P$W@Z`^,Y7`0G^5P
M3QWN;""?Y1#03>)H`)_E`B"/X&0PG^4!$)_G`S"/X``@DN<!`)3G5!"?Y0P@
MC>4!$(_@(?W_ZT@0G^5(,)_E`,"@XT0@G^4#,(_@`1"?YP(@C^`!`)3G-!"?
MY03`C>4!$(_@`#"-Y;?\_^L0T(WB$("]Z#AO`0`T;@$`$`$``,AI``!`*@``
M!&\!`!`M````%@``+"H``#@`G^4(T$WB-""?Y7#/'>XP$)_E`B"/X"PPG^4`
M`)_G`S"/X`$@DN<@$)_E``"<YP$0C^`$((WE"-"-XOG\_^J<;@$`F&T!`!`!
M```L:0``J"D``'!`+>D!RJ#C#,!-X```C.4!RDSB``",Y2@`#.7<()_E`MI-
MXM@PG^48T$WB`B"/X`(:C>(4$('B#$"-X@%JC>(#,)+G$&"&X@`PD^4`,('E
M`#"@X\S\_^NH$)_E`2`!XZ0PG^4!$(_@\`#-X0,PC^`!$*#C!`"@X<_\_^L$
M`*#A`2J@XP80H.'%_/_K`$!0XA@``+H!`(3BH/S_ZP!04.(4```*!""@X080
MH.%W_/_K`#"@XP0PQ>=0()_E`AJ-XCPPG^44$('B`B"/X`,PDN<`()/E`#"1
MY0(@,^``,*#C`P``&@4`H.$"VHWB&-"-XG"`O>B,_/_K`%"@X^W__^HL;0$`
M'`$``&`I``!,*0``D&P!`+G__^KP3RWI`<J@XPS`3>```(SEL``,Y5@BG^5"
MW4WB5#*?Y0S03>("((_@`&"@X4@"G^5"S8WB&("-X@3`C.(#,)+G`5"@X0``
MC^`($*#A`#"3Y0`PC.4`,*#CO/W_ZR\0H.,`0*#A!@"@X0_\_^L``%#C$@``
M"@0"G^4&$*#A``"/X,?^_^L&`*#A]"&?Y4(=C>+@,9_E!!"!X@(@C^`#,)+G
M`""3Y0`PD>4"(#/@`#"@XVX``!I"W8WB#-"-XO"/O>@&`*#A)?S_ZP"0H.$%
M`*#A0_S_ZZP1G^4!$(_@4OS_ZP"P4.+B__\*G#&?Y9RAG^4#,(_@%#"-Y90Q
MG^4*H(_@`S"/X!`PC>4*,*#A"Z"@X0.PH.$`,-KE``!3XS(```H``%3C!```
M"@H0H.$$`*#A</S_ZP``4.,W```*"@"@X0;\_^L!($#B`C"*X`(@VN<O`%+C
M!0``&@#`H.,`P,/E`0!`X@$@<^4O`%+C^O__"@$PB>(`,(/@`0I3X[S__ZJ$
M4(WB$#"=Y0$JH.,!$*#C!0"@X0A@C>7PH,WA0/S_ZP@0H.$%`*#AX/O_ZP``
M4.,#```:*#"=Y0\Z`^(""5/C&P``"LP0G^4``*#C`1"/X!7\_^L`H*#A``!:
MXZ3__PH`,-KE``!3X\S__QH``%3C&P``"@`PU.4N`%/C`P``"I2@G^4!`*#C
M"J"/X-?__^H!,-3E``!3X_C__QI\$)_E``"@XP$0C^#_^__K`*"@X>C__^H!
M$*#C!0"@X;C[_^L``%#CWO__&E0`G^4%$*#A``"/X%+^_^L%`*#AX/O_ZXC_
M_^H4H)WE`0"@X[___^KG^__K'&P!`!P!``!P*```1"@``*AK`0#X)P``P"<`
M`.`G``#4)P``!"<``*@F``"L)@``<"8````PT.4``%/C'O\O`0$@@.(O`%/C
M`@"@`0$PTN0``%/C^O__&A[_+^$00"WI`<J@XPS`3>#X#XSE`$!0XC<```H`
M,-3E``!3XS0```J9^__K`3!`X@,PA.`#`%3A$0``*@,@H.$!``#J!`!2X30`
M``H",*#A`1!2Y"\`4>/Y__\*`P!4X0<``"H#(*#A`0``Z@0`4N$E```*`C"@
MX0$04N0O`%'C^?__&@,`5.$!,$,2"@``&@`PU.4O`%/C)```"IP`G^400+WH
M``"/X)K[_^H`$-+E`3!#XB\`4>,"```:`R"@X0,`5.'X__\Z!"!"X`(P@N(!
M"E/C$P``BF0`G^4$$*#A`3J@XP$@@N(``(_@`$"@X4G[_^L$`*#A$("]Z$0`
MG^400+WH``"/X(+[_^H$,*#A`P!4X0$P0Q+G__\:V___Z@0PH.',___J``"@
MXQ"`O>@4`)_E$$"]Z```C^!U^__J;"4``!1J`0#@)```K"0``'!`+>D!RJ#C
M#,!-X,`/C.4<)9_E,-!-XAC%G^4(X(WB%#6?Y0(@C^`,P(_@#&6?Y0S`C.($
M0(WB!F"/X`!5G^4#,)+G!5"/X``PD^4L,(WE`#"@XP\`O.@/`*[H#P"\Z`\`
MKN@`,)SEL##.X000H.$&`*#AKOS_ZP``4.,.```*P#2?Y00`G>4#,)7G`!"3
MY0`Q@>`$()/E`"&!YP``4N/Q__\*!#"#X@0@D^4$((/D``!2X_O__QKK___J
MB&2?Y09@C^`$$*#A!@"@X9C\_^L``%#C#@``"F@TG^4$`)WE`S"5YP`0D^4`
M,8'@!""3Y0`A@><``%+C\?__"@0P@^($()/E!""#Y```4N/[__\:Z___ZC1D
MG^4&8(_@!!"@X08`H.&"_/_K``!0XPX```H0-)_E!`"=Y0,PE><`$)/E`#&!
MX`0@D^4`(8'G``!2X_'__PH$,(/B!""3Y00@@^0``%+C^___&NO__^K@8Y_E
M!F"/X`00H.$&`*#A;/S_ZP``4.,.```*N#.?Y00`G>4#,)7G`!"3Y0`Q@>`$
M()/E`"&!YP``4N/Q__\*!#"#X@0@D^4$((/D``!2X_O__QKK___JC&.?Y09@
MC^`$$*#A!@"@X5;\_^L``%#C#@``"F`SG^4$`)WE`S"5YP`0D^4`,8'@!""3
MY0`A@><``%+C\?__"@0P@^($()/E!""#Y```4N/[__\:Z___ZCACG^4&8(_@
M!!"@X08`H.%`_/_K``!0XPX```H(,Y_E!`"=Y0,PE><`$)/E`#&!X`0@D^4`
M(8'G``!2X_'__PH$,(/B!""3Y00@@^0``%+C^___&NO__^KD8I_E!F"/X`00
MH.$&`*#A*OS_ZP``4.,.```*L#*?Y00`G>4#,)7G`!"3Y0`Q@>`$()/E`"&!
MYP``4N/Q__\*!#"#X@0@D^4$((/D``!2X_O__QKK___JD&*?Y09@C^`$$*#A
M!@"@X13\_^L``%#C#@``"E@RG^4$`)WE`S"5YP`0D^4`,8'@!""3Y0`A@><`
M`%+C\?__"@0P@^($()/E!""#Y```4N/[__\:Z___ZCQBG^4&8(_@!!"@X08`
MH.'^^__K``!0XPX```H`,I_E!`"=Y0,PE><`$)/E`#&!X`0@D^4`(8'G``!2
MX_'__PH$,(/B!""3Y00@@^0``%+C^___&NO__^KH89_E!F"/X`00H.$&`*#A
MZ/O_ZP``4.,.```*J#&?Y00`G>4#,)7G`!"3Y0`Q@>`$()/E`"&!YP``4N/Q
M__\*!#"#X@0@D^4$((/D``!2X_O__QKK___JE&&?Y09@C^`$$*#A!@"@X=+[
M_^L``%#C#@``"E`QG^4$`)WE`S"5YP`0D^4`,8'@!""3Y0`A@><``%+C\?__
M"@0P@^($()/E!""#Y```4N/[__\:Z___ZD`!G^4$$*#A``"/X+W[_^L`$%#B
M`@``"BP!G^4``(_@S?S_ZR0!G^4$$*#A``"/X+3[_^L`$%#B`@``"A`!G^4`
M`(_@Q/S_ZP@!G^4$$*#A``"/X*O[_^L`$%#B$@``"O0`G^4``(_@N_S_Z^P0
MG^7L`)_E`1"/X```C^"V_/_KX""?Y80PG^4"((_@`S"2YP`@D^4L,)WE`B`S
MX``PH.,7```:,-"-XG"`O>BX`)_E!!"@X0``C^"2^__K`!!0X@,```JD`)_E
M``"/X*+\_^OE___JF!"?Y0H@H.,9`(WB`1"/X%_Z_^L``%#CWO__&H``G^4C
M$(WB``"/X);\_^O9___J,/K_ZP!H`0"<8P``'`$``(PD``#@9P$`&`$``!`D
M``#`(P``=",``"0C``#<(@``("(``#@B``#L(0``H"$``,P@``#X(```S"``
M`.@@``#H(```Y"````@@``#P(```2"```'!C`0"0(```%"```'@@``#D'P``
M!.`MY0'*H.,,P$W@\`^,Y9@PG^4,T$WBE!"?Y90@G^4#,(_@`1"/X```D^4"
M()'G`0!PXP`@DN4$((WE`""@XPH```IP()_E:#"?Y0(@C^`#,)+G`""3Y00P
MG>4"(#/@`#"@XQ```!H,T(WB!/"=Y$@`G^4-$*#A``"/X$+[_^L``%#C!```
M"@``T.4P`%#C``!0$P$`H!,``*`#(#"?Y0,PC^```(/EY?__ZN?Y_^N48P$`
M6&(!`!P!```P8@$`Z!X``!1C`0`00"WI`<J@XPS`3>#X#XSEE_G_Z[?Y_^L`
M`%#C``"0%1"`O>CP0"WI`<J@XPS`3>#0#XSE."&?Y1S03>(T,9_E$!"-X@(@
MC^`L89_E`%"@X09@C^`#,)+G!@"@X0`PD^44,(WE`#"@XQ7[_^L`0%#B`@``
M"@`PU.4``%/C#@``&O@@G^7L,)_E`B"/X`,PDN<`()/E%#"=Y0(@,^``,*#C
M,@``&M@`G^4%$*#A``"/X!S0C>+P0+WH%OS_Z@40H.&5^?_K``!0XPH```JT
M()_EH#"?Y0(@C^`#,)+G`""3Y10PG>4"(#/@`#"@XQ\``!H<T(WB\("]Z`4`
MH.%T^?_K`'"@X00`H.%Q^?_K``"'X`(`@.*8^?_K;#"?Y6P@G^4!$*#C`S"/
MX`A`C>4"((_@`%"-Y00@C>4`(.#C`$"@X;3Y_^M(()_E*#"?Y0(@C^`#,)+G
M`""3Y10PG>4"(#/@`#"@XP00H`$&`*`!S___"H3Y_^MP80$`'`$``-0>```L
M80$`>!X``.!@`0`D'0``&!T``&A@`0#P3RWI`,"@XP+IH.,,P$W@#N!,X`'*
M3.(``(SE#@!<X?O__QH@`0[E;!>?Y6PGG^4"V4WB`1"/X&0WG^7\T$WB`'"@
MX0,PC^#X8(WB`B"1YP()C>)TH$;B$."#X@`@DN7T((#E`""@X_0`@.('`)/I
M*)"#X@<`BNA<P$;B:$!&X@\`ON@/`*SH&(>?Y>!01N((@(_@`P">Z`,`C.@'
M`)GH!P"$Z``'G^4%$*#A``"/X)WZ_^L`0%#B`@``"@`PU.4``%/C"@$`&@/Y
M_^LC^?_K``!0X^D```H`H)#E``!:X^8```H*`*#A$_G_ZX``H.$!`(#B.OG_
MZP`PVN4`0*#A`)"@X0``4^,+```*G+:?Y0NPC^`$`*#A`#"-Y0$0H.,+,*#A
M`"#@XP)`A.)3^?_K`3#ZY0``4^/U__\:<`:?Y5R@1N(``(_@`@``Z@0`NN4`
M`%#C#0``"@40H.%R^O_K`$!0XOC__PH`,-3E``!3X_7__PJF^O_K``!0X_+_
M_PH$`*#A#?G_ZP!`4.(.```:=*!&X@1`FN0``%3C"@``"@`PU.4``%/CL@``
M"@0`H.&7^O_K``!0X_7__PH$`*#A_OC_ZP!`4.+Q__\*!`"@X=GX_^L`H*#A
M"0"@X=;X_^L`H(K@`:N*X@2@BN(*`*#A^_C_Z[PUG^4`L*#A`1"@XP,PC^`(
M,(WEK#6?Y0`@X.,,D(WE`S"/X`0PC>6<-9_E`$"-Y0,PC^`4^?_K!QV@XPL`
MH.%^^/_K`0!PXP,``!HI^?_K`!"0Y1$`4>.=```:!1"@X0L`H.'P^/_K`)!0
MXA@```I8%9_E`#"7Y50EG^4!`)CG`B"/X`"PC>4!$*#C``"0Y=SX_^L`@*#C
M."6?Y0(9C>($-9_E]!"!X@(@C^`#,)+G`""3Y0`PD>4"(#/@`#"@XS<!`!H(
M`*#A`MF-XOS0C>+PC[WHT#`6Y0\Z`^(!"5/CXO__&LA`%N5^^/_K``!4X=[_
M_QK0,!;E4S#HYP<-4^/:__\:"@"@X;CX_^L`@*#AQ`2?Y040H.$``(_@!?K_
MZP`04.*.```*``"7Y2W\_^L`H%#BB@``"@H`H.&!^/_K`$"@X0,`4.,,``":
MC!2?Y00`0.(``(K@`1"/X-_X_^L``%#C!0``&@0`5./9```*!$"*X`4P5.4O
M`%/CU0``"EQ$G^4$0(_@`#"4Y0$`<^/"```*``!3XW,```I$%)_E1`2?Y0$0
MC^!`=)_E``"/X#QDG^7T^O_K.%2?Y:;X_^LT-)_E"`"-Y0$0H.,#,(_@##"-
MY20TG^4(`*#A`+"-Y0`@X.,#,(_@!#"-Y1`TG^4'<(_@!F"/X`50C^`#,(_@
MH?C_ZP\``.JZ^/_K`#"0Y1$`4^,1```:C_C_ZP&0B>(`P*#A!S"@X00@H.$!
M$*#C"`"@X1!@C>4,D(WE(!"-Z0"PC>60^/_K!QV@XP@`H.'Z]__K`$"@X0$`
M<./I__\*"P"@X2+X_^N8`Y_E"!"@X0``C^#$^O_K"`"@X87^_^N$___J`T"@
MX5+__^IH0$;B!`"4Y```4.,I```*!1"@X:3Y_^L`,%#B^/__"@`PT^4``%/C
M]?__"D/X_^L`H%#B\O__"A_X_^N``*#A`0"`X@K__^KT(I_E`#"7Y0(`F.<$
M$(WE`1"@XP"PC>48(Y_E``"0Y0(@C^!"^/_K9/__ZF3^_^L$(Y_EG#*?Y0(9
MC>("((_@]!"!X@,PDN<`()/E`#"1Y0(@,^``,*#CG```&@0`H.$"V8WB_-"-
MXO!/O>@A^/_JR**?Y0T`H.,G^/_K4S"@XP!`H.$*H(_@`)"@X>S^_^H`H)?E
M<O__Z@H`H.$`$*#C`OC_ZP!`4.*&__\*`##@XQ$@X.,"<*#C!%!&X@!PC>42
M^/_K!B"@XP40H.$$`*#AU??_ZP0P5N4``%/C`@``&@,P5N5#`%/C5```"@`P
MH.,`(*#C`#"-Y00`H.$`,*#C`OC_Z[<$`.L`<*#A`@``Z@40H.$'`*#AQP0`
MZP(IH.,%$*#A!`"@X;_W_^L`(%#B]O__R@B2G^4$`*#A,*!&XN#W_^M%4$;B
M!Q"@X0F0C^`*0*#A,7!&XD0`1N+\!`#K`2#UY00`H.$`((WE"3"@X0$0H.,`
M(.#C#OC_ZP)`A.('`%7A]?__&K@AG^4!$*#C"*"-Y0@`H.$"((_@`+"-Y00@
MC>4`(*#CG#&?Y0@@1N68(9_E`S"/X`(@C^`,((WE`"#@X_OW_^MO___J@`&?
MY040H.$``(_@(?G_ZP``4.,``(0%J___"@`PT.4``%/C,`!3$P$PH!,`,*`#
M`#"$Y2[__^H$0)?E``!4XR;__PH$`*#AEO?_ZP,`4.,B__^:,!&?Y00`0.(`
M`(3@`1"/X/7W_^L``%#C!*"@`1K__^H",%;E00!3XZ?__QH!,%;E0P!3XZ3_
M_QH`,-;E2`!3XZ'__QH!,-;E10!3XY[__QH`,.#C!`"@X0!PC>4Y(.#CHO?_
MZP40H.$I(*#C!`"@X67W_^O`,)_EP""?Y0$0H.,#,(_@##"-Y;0PG^4"((_@
M"`"@X00@C>4#,(_@"%"-Y0"PC>4`(.#CNO?_ZR[__^J4]__KZ%\!`!P!```0
M80$`C%\!`.0;``"4'```"!P``+@;``"D&@``B!L``"`!``"(&P``@%T!`$0;
M```,&P``W%T!`/`9``!@&0``I!H``$@;``#,&```<!L``.`8``"(&@``+!@`
M`-@8```86P$`4!@``)`7``!8%@``[!<``+`8``!D%@``;!<``,@7``!$%0``
MZ!8``/!`+>D!RJ#C#,!-X.`/C.5<$I_E#-!-XE@BG^5P3QWN`1"/X%`RG^50
M`I_E#6"@X0(@D><``(_@!A"@X0`@DN4$((WE`""@XP,PG^<#4)3G9##5Y64P
MQ>6G^/_K``!0XP(```H`,-#E,0!3XV8```K4^__K"#*?Y0,PG^<#4)3G`!*?
MY04`H.$#,*#C`2"@XP$0C^#U]O_K`%!0X@P```KD,9_E#""5Y0,PG^<`$)+E
MV"&?Y0,`E.<"((_@\?;_ZPPPE>4`$)/E"#"1Y0$%$^-9```:N!&?Y0,PH.,!
M(*#C`1"?YP$`E.>H$9_E`1"/X-[V_^L`4%#B"P``"I@!G^4&$*#A``"/X'OX
M_^L`(%#B00``"H0QG^4,$)7E`S"?YP`0D>4#`)3GUO;_ZW`QG^4!(*#C;!&?
MY6QQG^4!$(_@`S"?YP=PC^`#4)3G!0"@X;?V_^M4(9_E`!"@X04`H.$"((_@
MQ_;_ZT0QG^5$`9_E`""@XP80H.$``(_@`S"?YP,PE.=E(,/E6_C_ZP!`H.$`
M`%#C"`"'Y1(```H'':#C!`"@X9;V_^L!((#B`#"@X1$`4.,!`%(3)0``BO@@
MG^6P,)_E`B"/X`,PDN<`()/E!#"=Y0(@,^``,*#C&P``&@S0C>+P@+WH!`"7
MY7?]_^L`0*#A``!0XP@`A^7F__\:N`"?Y0``C^`!]__K`3#0Y0``4^.9__\*
ME/__ZJ`PG^4,()7E`S"/X```DN4$,)/E`!"3Y??X_^N[___JA#"?Y0,PG^<#
M`)3GT_;_ZZ#__^K:]O_K<!"?Y00@H.%L`)_E`1"/X```C^`$$)'E`!"1Y>?V
M_^O,5P$`'`$``+!8`0"`%```@%@!`-P5``!06`$`N!4``!A8`0",%0``:!,`
M`-Q7`0"X5P$`0!4``"!8`0`T%0``?%<!`.@2```H5@$`F!0``$!7`0"X5@$`
M!%<!`%P4``#P0"WI`<J@XPS`3>#@#XSE0"&?Y0S03>(\,9_E`%"@X0(@C^`#
M,)+G`#"3Y00PC>4`,*#CV_K_ZP`PT.4`0*#A``!3XP4```H!((#B+P!3XP)`
MH`$!,-+D``!3X_K__QKX<)_E`&!5X@%@H!,'<(_@`#"7Y0$`<^,F```*``!3
MXP!@H`,!8`82``!6XPH``!K,()_EP#"?Y0(@C^`#,)+G`""3Y00PG>4"(#/@
M`#"@XR<``!H,T(WB\("]Z``PU>4``%/C\?__"I@0G^4$(*#C!`"@X0$0C^"J
M]O_K``!0X^K__QJ`()_E;#"?Y0(@C^`#,)+G`""3Y00PG>4"(#/@`#"@XQ(`
M`!H%`*#A#-"-XO!`O>@I^/_J4`"?Y0T0H.$``(_@O_?_ZP``4.,``(<%UO__
M"@`PT.4``%/C,`!3$P$PH!,`,*`#`V`&X``PA^7,___J9/;_Z_A4`0`<`0``
MY%4!`(!4`0#@$0``+%0!`-P0````````\$\MZ0'*H.,,P$W@@`Z,Y1!,G^57
MWTWB#.R?Y1`PC>($0(_@`Q"@X1P@@.)<P(#B#N"4YP#@GN54X8WE`."@XXT*
M8O3@`/#S`@!<X=T*0?3Z__\:!DO=[0@[W>T1'HWB"AO=[0```.JR$6+R""O3
M[30@@^(`"]/M"#"#X@$`4^&/5V+TL`%"\[4!0/.T`4#SLT%C\K$Q8?*@&&#R
M,`#A\[`!8?(."\/M[O__&B!`D.F9.0?C@CI%XPQ`D.40P)WE`&"0Y000+N`0
M<)#E`\",X`40`>!ED:#A!\",X`00(>`4()WEYLV,X`'`C.`)$"[@`R""X`80
M`>!F@:#A!"""X!B@G>4.$"'@`A"!X`@@*>`#H(K@#"`"X`Z@BN`)("+@"B""
MX!R@G>7L'8'@;,&@X0.@BN`)H(K@#)`HX`&0">#A+8+@")`IX&$1H.$*D(G@
M(*"=Y>*=B>`#H(K@"*"*X`&`+.`"@`C@8B&@X0R`*.`"L"'@"H"(X"2@G>4)
ML`O@Z8V(X`.@BN`,H(K@:<&@X0&0*^`,L"+@"I")X"B@G>4(L`O@Z)V)X`.@
MBN`!H(K@:!&@X0*`*^`!L"S@"H"(X"R@G>4)L`O@Z8V(X`.@BN`"H(K@:2&@
MX0R0*^`"L"'@"I")X#"@G>4(L`O@Z)V)X`.@BN`,H(K@:,&@X0&`*^`,L"+@
M"H"(X#2@G>4)L`O@Z8V(X`.@BN`!H(K@:1&@X0*0*^`!L"S@"I")X#B@G>4(
ML`O@Z)V)X`.@BN`"H(K@:"&@X0R`*^`"L"'@"H"(X#R@G>4)L`O@Z8V(X`.@
MBN`,H(K@:<&@X0&0*^`,L"+@"I")X$"@G>4(L`O@Z)V)X`.@BN`!H(K@:!&@
MX0*`*^`!L"S@"H"(X$2@G>4)L`O@Z8V(X`.@BN`"H(K@:2&@X0R0*^`"L"'@
M"I")X$B@G>4(L`O@Z)V)X`.@BN`,H(K@:,&@X0&`*^`,L"+@"H"(X$R@G>4)
ML`O@Z8V(X`.@BN`!H(K@:1&@X0*0*^`!L"S@"I")X%"@G>4(L`O@Z)V)X`.@
MBN`"H(K@:"&@X0R`*^`*@(C@5*"=Y>F-B.`#H(K@#,"*X`*@(>`)H`K@:9&@
MX0&@*N`,H(K@6,"=Y>BMBN`#P(S@`<",X`D0(N`($`'@:(&@X0(0(>`,$('@
M7,"=Y>H=@>`#,(S@",`IX`K`#.`",(/@"<`LX&`@G>4#P(S@H3L.X]D^1N-J
MH:#A`R""X.'-C.`)D(+@9""=Y0,@@N`(((+@"H`HX`&`*.`(@(G@89&@X6@0
MG>7LC8C@`Q"!X`H0@>`)H"K@#*`JX&S!H.$*((+@Z*V"X&P@G>4#((+@"2""
MX`R0*>`(D"G@:(&@X0D0@>#JG8'@<!"=Y0,0@>`,$('@",`LX`K`+.!JH:#A
M#"""X.G-@N!T()WE`R""X`@@@N`*@"C@"8`HX&F1H.$($('@[(V!X'@0G>4#
M$('@"A"!X`F@*N`,H"K@;,&@X0H@@N#HK8+@?""=Y0,@@N`)((+@#)`IX`B0
M*>!H@:#A"1"!X.J=@>"`$)WE`Q"!X`P0@>`(P"S@"L`LX&JAH.$,((+@Z<V"
MX(0@G>4#((+@""""X`J`*.`)@"C@:9&@X0@0@>#LC8'@B!"=Y0,0@>`*$('@
M":`JX`R@*N!LP:#A"B""X.BM@N",()WE`R""X&JQH.$)((+@#)`IX`B0*>!H
M@:#A"1"!X.J=@>"0$)WE`Q"!X`P0@>`(P"S@"L`LX`P@@N#IK8+@E""=Y0,@
M@N`(((+@"X`HX`G`*.!ID:#A#!"!X)C`G>7JC8'@"1`KX`/`C.`*$"'@"\",
MX)RPG>5JH:#A`2""X`.PB^#H'8+@";"+X`J0*>`(("G@:(&@X0+`C."@()WE
MX9V,X`,@@N`*((+@"*`JX`'`*N"DH)WE81&@X0S`B^`#H(K@Z<V,X`B@BN`!
M@"C@"8`HX&F1H.$(((+@L+"=Y>R-@N"H()WE`R""X`$@@N`)$"'@#!`AX&S!
MH.$!H(K@#!`IX.BMBN`($"'@`2""X&B!H.'J'8+@K""=Y0,@@N`(,"S@"C`C
MX`D@@N`#((+@W#P+XQL_2.-JD:#A`["+X`F@B.$,L(O@M,"=Y>$M@N`!H`K@
M`\",X&$1H.$(P(S@"8`(X`B@BN$!@(GA"Z"*X`*`".`!L`G@XJV*X`N`B.&\
ML)WE#("(X+C`G>5B(:#A`["+X`/`C.`!L(O@"9",X`+`@>$*P`S@`A`!X`'`
MC.'JC8C@"<",X,"0G>5JH:#AZ,V,X`H0@N$#D(G@`I")X`@0`>`*(`+@:(&@
MX0(0@>$((`K@"["!X`@0BN$,$`'@[+V+X`(0@>%LP:#A"1"!X,20G>4,((CA
MZQV!X`.0B>`+(`+@"I")X,B@G>5KL:#A`Z"*X`B@BN`,@`C@""""X6&!H.$)
MD(+@"R",X>&=B>`!(`+@S!"=Y0,0@>`,$('@"\`,X`S`@N$(((OA"L",X`D@
M`N`(H`O@Z<V,X`H@@N'0H)WE:9&@X0$0@N`#H(K@"2"(X0N@BN#4L)WE#"`"
MX.P=@>`#L(O@;,&@X0BPB^`)@`C@""""X6&!H.$*H(+@#"")X>&MBN`!(`+@
MV!"=Y0,0@>`)$('@#)`)X`F0@N$(((SA"Y")X`H@`N`(L`S@ZIV)X`L@@N'@
ML)WE`1""X-P@G>5JH:#A`["+X`,@@N`(L(O@#,""X`H@B.$)(`+@"H`(X.D=
M@>`(@(+A:9&@X0R`B.`)((KA8<&@X>&-B.`!(`+@Y!"=Y0,0@>`*$('@":`*
MX`J@@N$,((GA"Z"*X`@@`N`,L`G@Z*V*X`L@@N'LL)WE`1""X.@@G>5H@:#A
M`["+X`,@@N`,L(O@"9""X`@@C.$*(`+@",`,X.H=@>`,((+A:J&@X0F0@N`*
M((CA8<&@X>&=B>`!(`+@\!"=Y0,0@>`($('@"H`(X`B`@N$,((KA"X"(X`D@
M`N`,L`K@Z8V(X`L@@N'XL)WE`2""X/00G>5ID:#A`["+X`,0@>`,L(O@"J"!
MX`D0C.$($`'@"<`,X`S`@>'H+8+@"L",X/R@G>5H@:#AXLV,X`@0B>$#,(K@
M"3"#X`(0`>`(D`G@8B&@X0*@B.$)$('A"Q"!X`R@"N`"L`C@[!V!X&R1H.$+
MH(KA`,&=Y0.@BN#6,0SC8CI,XP/`C.#AK8K@"(",X`3!G>5AL:#A`\",X`+`
MC.`)("+@`2`BX`@1G>4"@(C@"R`IX.J-B.`*("+@`Q"!X&JAH.$)$('@`L",
MX`J0*^#HS8S@")`IX`PAG>4)$('@:(&@X>R=@>`0$9WE`R""X`,0@>`+((+@
M"A"!X`B@*N`,H"K@;,&@X0H@@N#IK8+@%"&=Y0,@@N`(((+@#(`HX`F`*.!I
MD:#A"!"!X.J-@>`8$9WE`Q"!X`P0@>`)P"S@"L`LX&JAH.$,((+@Z,V"X!PA
MG>4#((+@"2""X`J0*>`(D"G@:(&@X0D0@>#LG8'@(!&=Y0,0@>`*$('@"*`J
MX`R@*N!LP:#A"B""X.FM@N`D(9WE`R""X`@@@N`,@"C@"8`HX&F1H.$($('@
MZHV!X"@1G>4#$('@#!"!X`G`+.`*P"S@:J&@X0P@@N#HS8+@+"&=Y0,@@N`)
M((+@"I`IX`B0*>!H@:#A"1"!X.R=@>`P$9WE`Q"!X`H0@>`(H"K@#*`JX&S!
MH.$*((+@Z:V"X#0AG>4#((+@""""X`R`*.`)@"C@:9&@X0@0@>#JC8'@.!&=
MY0,0@>`,$('@"<`LX`K`+.!JH:#A#"""X.C-@N`\(9WE`R""X`D@@N`*D"G@
M")`IX&B!H.$)$('@0)&=Y>P=@>`#D(G@"I")X`B@*N`,H"K@;,&@X0H@@N!$
MH9WEX2V"X`.@BN`(H(K@#(`HX`&`*.!A$:#A")")X$B!G>7BG8G@`X"(X`R`
MB.`!P"S@`L`LX&(AH.$,P(K@`G"'X.G-C.`0<(#E;.&.X`C@C>5,X9WE`S".
MX`8P@^`!,(/@`A`AX`D0(>!ID:#A`1"(X`D@(N#L'8'@#"`BX`(P@^`)0(3@
MX3V#X`Q`C>4`,(WE`5"%X#P@G^4$4(WEWPIM]`(@C^`H,)_ECPI`]`,PDN<`
M()/E5#&=Y0(@,^``,*#C`0``&E??C>+PC[WH4O/_ZX!3`0`<`0``J$<!`!!`
M+>D!RJ#C#,!-X/@/C.5@`*#C1O/_ZP<+W^T(*]_M"3O?[0`@H.,$"\#M&""`
MY5P@@.7?*D#T$("]Z/#ATL,``````2-%9XFKS>_^W+J8=E0R$/!'+>D!RJ#C
M#,!-X.`/C.44,)#E`F"@X8(AH.$`4*#A`C"3X!@@D.44,(#E`4"@X5P`D.4!
M(((BIBZ"X!R`A>(``%#C&""%Y2```!H_`%;C+@``VD"01N(I8Z#A`7"&X@=S
MA.`$$*#A"#"@X4!`A.(`X)'E$!"!X@S`$>40,(/B"``1Y00@$>4$`%'A$.`#
MY0S``^4(``/E!"`#Y?/__QH!0*#A!0"@X;+\_^L'`%3AZ___&@9C2>`&(*#A
M!Q"@X0@`H.'A\O_K7&"%Y?"'O>A`<&#B``"(X`8`5^$&<*"A!R"@X=GR_^M<
M,)7E`S"'X%PPA>5``%/C\(>]&`4`H.$'8$;@!T"$X)K\_^O.___J!'"@X>?_
M_^IP0"WI`<J@XP*++>T,P$W@Z`^,Y0%`H.$4$)'E`%"@X1Q@A.(Q/[_F%!"4
MY1@`E.71$>7G,"^_YA@K0^P!(('B?S#@XS@`4N,"`(;@`3#&YT$``-I`(&+B
M`!"@XXOR_^L$`*#A??S_ZRPPA.)0`,#R#PI&]`\*0_0\,(3B#PI#]$PPA.(/
M!T/T5#"$X@0`H.$/AP/T<?S_ZP,PU.4`,,7E!`"@X;(PU.$!,,7E`#"4Y2,T
MH.$",,7E`#"4Y0,PQ>4',-3E!##%Y;8PU.$%,,7E!#"4Y2,TH.$&,,7E!#"4
MY0<PQ>4+,-3E"##%Y;HPU.$),,7E"#"4Y2,TH.$*,,7E"#"4Y0LPQ>4/,-3E
M###%Y;XPU.$-,,7E##"4Y2,TH.$.,,7E##"4Y0\PQ>43,-3E$##%Y;(QU.$1
M,,7E$#"4Y2,TH.$2,,7E$#"4Y1,PQ>4"B[WL<$"]Z%OR_^HX(&+B`!"@XTGR
M_^O&___J"#"?Y0,PC^``,)/EI_+_ZL1%`0`(0"WI"("]Z`$``@`]````0```
M``L```!`````$````$`````F````0````#4```!``````P```$`````&````
M0````!0```!`````#@```$`````-````0````"4```!`````$@```$`````H
M````0`````(```!`````&````$`````)````0`````H```!`````````````
M```E<R\E<P```"]P<F]C+W-E;&8O97AE``!$>6YA3&]A9&5R.CIB;V]T7T1Y
M;F%,;V%D97(`26YT97)N86QS.CI005(Z.D)/3U0`````+W!R;V,O)6DO)7,`
M97AE`"X```!005)?5$5-4`````!005)?4%)/1TY!344`````.@```"5S)7,E
M<P``+P```%!%4DPU3$E"`````%!%4DQ,24(`4$523#5/4%0`````4$523$E/
M``!005)?24Y)5$E!3$E:140`4$%27U-005=.140`4$%27T-,14%.````4$%2
M7T1%0E5'````4$%27T-!0TA%````4$%27T=,3T)!3%]$14)51P````!005)?
M1TQ/0D%,7U1-4$1)4@```%!!4E]435!$25(``%!!4E]'3$]"04Q?5$5-4`!0
M05)?1TQ/0D%,7T-,14%.`````%!!4E]#3$5!3CT``#$```!,1%],24)205)9
M7U!!5$@`4UE35$5-```E,#)X`````"5S)7,E<R5S`````'!A<BT`````)7,Z
M(&-R96%T:6]N(&]F('!R:79A=&4@<W5B9&ER96-T;W)Y("5S(&9A:6QE9"`H
M97)R;F\])6DI"@``)7,Z('!R:79A=&4@<W5B9&ER96-T;W)Y("5S(&ES('5N
M<V%F92`H<&QE87-E(')E;6]V92!I="!A;F0@<F5T<GD@>6]U<B!O<&5R871I
M;VXI"@``4$%42`````!P87)L`````"YP87(`````)7,E<V-A8VAE+25S)7,`
M`"5S)7-T96UP+25U)7,````E<R5S=&5M<"TE=2TE=25S`````!@```!P97)L
M`````#````!005(Z.E!A8VME<CHZ5D524TE/3@`````Q+C`U-````%5N86)L
M92!T;R!C<F5A=&4@8V%C:&4@9&ER96-T;W)Y`````"5S.B!C<F5A=&EO;B!O
M9B!P<FEV871E(&-A8VAE('-U8F1I<F5C=&]R>2`E<R!F86EL960@*&5R<FYO
M/25I*0H`````+64``"TM``!54T52`````%5315).04U%`````%1%35!$25(`
M5$U0`"]T;7``````979A;"`G97AE8R`O=7-R+V)I;B]P97)L("`M4R`D,"`D
M>S$K(B1`(GTG"FEF(#`[(",@;F]T(')U;FYI;F<@=6YD97(@<V]M92!S:&5L
M;`IP86-K86=E(%]?<&%R7W!L.PIM>2`H)%!!4E]-04=)0RP@)'!A<E]T96UP
M+"`D<')O9VYA;64L($!T;7!F:6QE+"`E36]D=6QE0V%C:&4I.PI%3D0@>R!I
M9B`H)$5.5GM005)?0TQ%04Y]*2!["G)E<75I<F4@1FEL93HZ5&5M<#L*<F5Q
M=6ER92!&:6QE.CI"87-E;F%M93L*<F5Q=6ER92!&:6QE.CI3<&5C.PIM>2`D
M=&]P9&ER(#T@1FEL93HZ0F%S96YA;64Z.F1I<FYA;64H)'!A<E]T96UP*3L*
M;W5T<RAQ<5M296UO=FEN9R!F:6QE<R!I;B`B)'!A<E]T96UP(ETI.PI&:6QE
M.CI&:6YD.CIF:6YD9&5P=&@H<W5B('L@*"`M9"`I(#\@<FUD:7(@.B!U;FQI
M;FL@?2P@)'!A<E]T96UP*3L*<FUD:7(@)'!A<E]T96UP.PII9B`H+60@)'!A
M<E]T96UP("8F("1>3R!N92`G35-7:6XS,B<I('L*;7D@)'1M<"`](&YE=R!&
M:6QE.CI496UP*`I414U03$%412`]/B`G=&UP6%A86%@G+`I$25(@/3X@1FEL
M93HZ0F%S96YA;64Z.F1I<FYA;64H)'1O<&1I<BDL"E-51D9)6"`]/B`G+F-M
M9"<L"E5.3$E.2R`]/B`P+`HI.PIM>2`D9FEL96YA;64@/2`D=&UP+3YF:6QE
M;F%M93L*<')I;G0@)'1M<"`\/"(N+BXB.PIX/3$[('=H:6QE(%L@7"1X("UL
M="`Q,"!=.R!D;PIR;2`M<F8@)R1P87)?=&5M<"<*:68@6R!<(2`M9"`G)'!A
M<E]T96UP)R!=.R!T:&5N"F)R96%K"F9I"G-L965P(#$*>#U@97AP<B!<)'@@
M*R`Q8`ID;VYE"G)M("<D9FEL96YA;64G"BXN+@IC;&]S92`D=&UP.PIC:&UO
M9"`P-S`P+"`D9FEL96YA;64["FUY("1C;60@/2`B)&9I;&5N86UE(#XO9&5V
M+VYU;&P@,CXF,2`F(CL*<WES=&5M*"1C;60I.PIO=71S*'%Q6U-P87=N960@
M8F%C:V=R;W5N9"!P<F]C97-S('1O('!E<F9O<FT@8VQE86YU<#H@)&9I;&5N
M86UE72D["GT*?2!]"D)%1TE.('L*)%!!4E]-04=)0R`](")<;E!!4BYP;5QN
M(CL*26YT97)N86QS.CI005(Z.D)/3U0H*2!I9B!D969I;F5D("9);G1E<FYA
M;',Z.E!!4CHZ0D]/5#L*979A;"!["E]P87)?:6YI=%]E;G8H*3L*;7D@)'%U
M:65T(#T@(21%3E9[4$%27T1%0E5'?3L*;7D@)7-Y<R`]("@*<&%T:%]S97`@
M("`@/3X@*"1>3R`]?B`O7DU35VEN+R`_("<[)R`Z("<Z)RDL"E]E>&4@("`@
M("`@(#T^("@D7D\@/7X@+UXH/SI-4U=I;GQ/4S)\8WEG=VEN*2\@/R`G+F5X
M92<@.B`G)RDL"E]D96QI;2`@("`@(#T^("@D7D\@/7X@+UY-4U=I;GQ/4S(O
M(#\@)UQ<)R`Z("<O)RDL"BD["E]S971?<')O9VYA;64H*3L*;W5T<RAQ<5M<
M)'!R;V=N86UE(#T@(B1P<F]G;F%M92)=*3L*7W-E=%]P87)?=&5M<"@I.PIO
M=71S*'%Q6UPD14Y6>U!!4E]414U0?2`]("(D14Y6>U!!4E]414U0?2)=*3L*
M;7D@*"1S=&%R=%]P;W,L("1D871A7W!O<RD["DU!1TE#.B!["FQO8V%L("13
M24=[7U]705).7U]](#T@<W5B('M].PIU;FQE<W,@*&]P96X@7T9(+"`G/#IR
M87<G+"`D<')O9VYA;64I('L*;W5T<RAQ<5M#86XG="!R96%D(&9R;VT@9FEL
M92`B)'!R;V=N86UE(ETI.R`@(R!D;VXG="!U<V4@)"$@:&5R92!A<R!I="!R
M97%U:7)E<R!%<G)N;RYP;0IL87-T($U!1TE#.PI]"FUY("1C:'5N:U]S:7IE
M(#T@-C0@*B`Q,#(T.PIM>2`D8G5F.PIM>2`D<VEZ92`]("US(%]&2#L*;7D@
M)&UA9VEC7W!O<R`]("1S:7IE("T@)'-I>F4@)2`D8VAU;FM?<VEZ93L@(R!.
M3U1%.B`D;6%G:6-?<&]S(&ES(&$@;75L=&EP;&4@;V8@)&-H=6YK7W-I>F4*
M=VAI;&4@*"1M86=I8U]P;W,@/CT@,"D@>PIS965K(%]&2"P@)&UA9VEC7W!O
M<RP@,#L*<F5A9"!?1D@L("1B=68L("1C:'5N:U]S:7IE("L@;&5N9W1H*"10
M05)?34%'24,I.PII9B`H*&UY("1I(#T@<FEN9&5X*"1B=68L("1005)?34%'
M24,I*2`^/2`P*2!["B1M86=I8U]P;W,@*ST@)&D["FQA<W0["GT*)&UA9VEC
M7W!O<R`M/2`D8VAU;FM?<VEZ93L*?0II9B`H)&UA9VEC7W!O<R`\(#`I('L*
M;W5T<RAQ<5M#86XG="!F:6YD(&UA9VEC('-T<FEN9R`B)%!!4E]-04=)0R(@
M:6X@9FEL92`B)'!R;V=N86UE(ETI.PIL87-T($U!1TE#.PI]"G-E96L@7T9(
M+"`D;6%G:6-?<&]S("T@-"P@,#L*<F5A9"!?1D@L("1B=68L(#0["G-E96L@
M7T9(+"`D;6%G:6-?<&]S("T@-"`M('5N<&%C:R@B3B(L("1B=68I+"`P.PHD
M9&%T85]P;W,@/2!T96QL(%]&2#L*;7D@)7)E<75I<F5?;&ES=#L*<F5A9"!?
M1D@L("1B=68L(#0[("`@("`@("`@("`@("`@("`@("`@("`@("`@(R!R96%D
M('1H92!F:7)S="`B1DE,12(*=VAI;&4@*"1B=68@97$@(D9)3$4B*2!["G)E
M860@7T9(+"`D8G5F+"`T.PIR96%D(%]&2"P@)&)U9BP@=6YP86-K*").(BP@
M)&)U9BD["FUY("1F=6QL;F%M92`]("1B=68["F]U=',H<7%;56YP86-K:6YG
M($9)3$4@(B1F=6QL;F%M92(N+BY=*3L*;7D@)&-R8R`]("@@)&9U;&QN86UE
M(#U^('-\7BA;82UF7&1=>SA]*2]\?"`I(#\@)#$@.B!U;F1E9CL*;7D@*"1B
M87-E;F%M92P@)&5X="D@/2`H)&)U9B`]?B!M?"@_.BXJ+RD_*"XJ*2A<+BXJ
M*7PI.PIR96%D(%]&2"P@)&)U9BP@-#L*<F5A9"!?1D@L("1B=68L('5N<&%C
M:R@B3B(L("1B=68I.PII9B`H9&5F:6YE9"@D97AT*2!A;F0@)&5X="`A?B`O
M7"XH/SIP;7QP;'QI>'QA;"DD+VDI('L*;7D@)&9I;&5N86UE(#T@7W-A=F5?
M87,H(B1C<F,D97AT(BP@)&)U9BP@,#<U-2D["B1005(Z.DAE879Y.CI&=6QL
M0V%C:&5[)&9U;&QN86UE?2`]("1F:6QE;F%M93L*)%!!4CHZ2&5A=GDZ.D9U
M;&Q#86-H97LD9FEL96YA;65](#T@)&9U;&QN86UE.PI]"F5L<VEF("@@)&9U
M;&QN86UE(#U^(&U\7B\_<VAL:6(O?"!A;F0@9&5F:6YE9"`D14Y6>U!!4E]4
M14U0?2`I('L*;7D@)&9I;&5N86UE(#T@7W-A=F5?87,H(B1B87-E;F%M921E
M>'0B+"`D8G5F+"`P-S4U*3L*;W5T<R@B4TA,24(Z("1F:6QE;F%M95QN(BD[
M"GT*96QS92!["B1-;V1U;&5#86-H97LD9G5L;&YA;65](#T@>PIB=68@/3X@
M)&)U9BP*8W)C(#T^("1C<F,L"FYA;64@/3X@)&9U;&QN86UE+`I].PHD<F5Q
M=6ER95]L:7-T>R1F=6QL;F%M97TK*SL*?0IR96%D(%]&2"P@)&)U9BP@-#L*
M?0IL;V-A;"!`24Y#(#T@*'-U8B!["FUY("@D<V5L9BP@)&UO9'5L92D@/2!`
M7SL*<F5T=7)N(&EF(')E9B`D;6]D=6QE(&]R("$D;6]D=6QE.PIM>2`D:6YF
M;R`]("1-;V1U;&5#86-H97LD;6]D=6QE?3L*<F5T=7)N('5N;&5S<R`D:6YF
M;SL*9&5L971E("1R97%U:7)E7VQI<W1[)&UO9'5L97T["B1)3D-[)&UO9'5L
M97T@/2`B+VQO861E<B\D:6YF;R\D;6]D=6QE(CL*:68@*"1%3E9[4$%27T-,
M14%.?2!A;F0@9&5F:6YE9"@F24\Z.D9I;&4Z.FYE=RDI('L*;7D@)&9H(#T@
M24\Z.D9I;&4M/FYE=U]T;7!F:6QE(&]R(&1I92`B0V%N)W0@8W)E871E('1E
M;7`@9FEL93H@)"$B.PHD9F@M/F)I;FUO9&4H*3L*)&9H+3YP<FEN="@D:6YF
M;RT^>V)U9GTI.PHD9F@M/G-E96LH,"P@,"D["G)E='5R;B`D9F@["GT*96QS
M92!["FUY("1F:6QE;F%M92`](%]S879E7V%S*"(D:6YF;RT^>V-R8WTN<&TB
M+"`D:6YF;RT^>V)U9GTI.PHD:6YF;RT^>V9I;&5](#T@)&9I;&5N86UE.PIO
M<&5N(&UY("1F:"P@)SPZ<F%W)RP@)&9I;&5N86UE(&]R(&1I92!Q<5M#86XG
M="!R96%D("(D9FEL96YA;64B.B`D(5T["G)E='5R;B`D9F@["GT*9&EE(")"
M;V]T<W1R87!P:6YG(&9A:6QE9#H@8V%N)W0@9FEN9"!M;V1U;&4@)&UO9'5L
M92$B.PI]+"!`24Y#*3L*<F5Q=6ER92!84TQO861E<CL*<F5Q=6ER92!005(Z
M.DAE879Y.PIR97%U:7)E($-A<G`Z.DAE879Y.PIR97%U:7)E($5X<&]R=&5R
M.CI(96%V>3L*4$%2.CI(96%V>3HZ7VEN:71?9'EN86QO861E<B@I.PIR97%U
M:7)E($E/.CI&:6QE.PIW:&EL92`H;7D@)&9I;&5N86UE(#T@*'-O<G0@:V5Y
M<R`E<F5Q=6ER95]L:7-T*5LP72D@>PIU;FQE<W,@*"1)3D-[)&9I;&5N86UE
M?2!O<B`D9FEL96YA;64@/7X@+T)31%!!3B\I('L*:68@*"1F:6QE;F%M92`]
M?B`O7"YP;6,_)"]I*2!["G)E<75I<F4@)&9I;&5N86UE.PI]"F5L<V4@>PID
M;R`D9FEL96YA;64@=6YL97-S("1F:6QE;F%M92`]?B`O<VET96-U<W1O;6EZ
M95PN<&PD+SL*?0I]"F1E;&5T92`D<F5Q=6ER95]L:7-T>R1F:6QE;F%M97T[
M"GT*=6YL97-S("@D8G5F(&5Q(")02UPP,#-<,#`T(BD@>PIO=71S*'%Q6TYO
M('II<"!F;W5N9"!A9G1E<B!&24Q%('-E8W1I;VX@:6X@9FEL92`B)'!R;V=N
M86UE(ETI.PIL87-T($U!1TE#(#L*?0HD<W1A<G1?<&]S(#T@*'1E;&P@7T9(
M*2`M(#0[("`@("`@("`@("`@("`@(",@<W1A<G0@;V8@>FEP"GT*;7D@0'!A
M<E]A<F=S.PIM>2`H)&]U="P@)&)U;F1L92P@)&QO9V9H+"`D8V%C:&5?;F%M
M92D["F1E;&5T92`D14Y6>U!!4E]!4%!?4D554T5].R`C('-A;FET:7IE("A2
M155312!M87D@8F4@82!S96-U<FET>2!P<F]B;&5M*0HD<75I970@/2`P('5N
M;&5S<R`D14Y6>U!!4E]$14)51WT["FEF("@A)'-T87)T7W!O<R!O<B`H)$%2
M1U9;,%T@97$@)RTM<&%R+6]P=&EO;G,G("8F('-H:69T*2D@>PIM>2`E9&ES
M=%]C;60@/2!Q=R@*<"`@(&)L:6)?=&]?<&%R"FD@("!I;G-T86QL7W!A<@IU
M("`@=6YI;G-T86QL7W!A<@IS("`@<VEG;E]P87(*=B`@('9E<FEF>5]P87(*
M*3L*:68@*$!!4D=6(&%N9"`D05)'5ELP72!E<2`G+2UR975S92<I('L*<VAI
M9G0@0$%21U8["B1%3E9[4$%27T%04%]2155317T@/2!S:&EF="!`05)'5CL*
M?0IE;'-E('L@(R!N;W)M86P@<&%R;"!B96AA=FEO=7(*;7D@0&%D9%]T;U]I
M;F,["G=H:6QE("A`05)'5BD@>PHD05)'5ELP72`]?B`O7BTH6T%)34]"3&)Q
M<&EU<U1V72DH+BHI+R!O<B!L87-T.PII9B`H)#$@97$@)TDG*2!["G!U<V@@
M0&%D9%]T;U]I;F,L("0R.PI]"F5L<VEF("@D,2!E<2`G32<I('L*979A;"`B
M=7-E("0R(CL*?0IE;'-I9B`H)#$@97$@)T$G*2!["G5N<VAI9G0@0'!A<E]A
M<F=S+"`D,CL*?0IE;'-I9B`H)#$@97$@)T\G*2!["B1O=70@/2`D,CL*?0IE
M;'-I9B`H)#$@97$@)V(G*2!["B1B=6YD;&4@/2`G<VET92<["GT*96QS:68@
M*"0Q(&5Q("=")RD@>PHD8G5N9&QE(#T@)V%L;"<["GT*96QS:68@*"0Q(&5Q
M("=Q)RD@>PHD<75I970@/2`Q.PI]"F5L<VEF("@D,2!E<2`G3"<I('L*;W!E
M;B`D;&]G9F@L("(^/B(L("0R(&]R(&1I92!Q<5M#86XG="!O<&5N(&QO9R!F
M:6QE("(D,B(Z("0A73L*?0IE;'-I9B`H)#$@97$@)U0G*2!["B1C86-H95]N
M86UE(#T@)#(["GT*<VAI9G0H0$%21U8I.PII9B`H;7D@)&-M9"`]("1D:7-T
M7V-M9'LD,7TI('L*9&5L971E("1%3E9[)U!!4E]414U0)WT["FEN:71?:6YC
M*"D["G)E<75I<F4@4$%2.CI$:7-T.PHF>R)005(Z.D1I<W0Z.B1C;60B?2@I
M('5N;&5S<R!`05)'5CL*)GLB4$%2.CI$:7-T.CHD8VUD(GTH)%\I(&9O<B!`
M05)'5CL*97AI=#L*?0I]"G5N<VAI9G0@0$E.0RP@0&%D9%]T;U]I;F,["GT*
M?0II9B`H)&]U="D@>PI["G)E<75I<F4@24\Z.D9I;&4["G)E<75I<F4@07)C
M:&EV93HZ6FEP.PIR97%U:7)E($1I9V5S=#HZ4TA!.PI]"FUY("1P87(@/2!S
M:&EF="A`05)'5BD["FUY("1Z:7`["FEF("AD969I;F5D("1P87(I('L*;W!E
M;B!M>2`D9F@L("<\.G)A=R<L("1P87(@;W(@9&EE('%Q6T-A;B=T(&9I;F0@
M<&%R(&9I;&4@(B1P87(B.B`D(5T["F)L97-S*"1F:"P@)TE/.CI&:6QE)RD[
M"B1Z:7`@/2!!<F-H:79E.CI::7`M/FYE=SL**"`D>FEP+3YR96%D1G)O;49I
M;&5(86YD;&4H)&9H+"`D<&%R*2`]/2!!<F-H:79E.CI::7`Z.D%:7T]+*"D@
M*0IO<B!D:64@<7%;17)R;W(@<F5A9&EN9R!Z:7`@87)C:&EV92`B)'!A<B)=
M.PI]"FUY("5E;G8@/2!D;R!["FEF("@D>FEP(&%N9"!M>2`D;65T82`]("1Z
M:7`M/F-O;G1E;G1S*"=-151!+GEM;"<I*2!["B1M971A(#U^(',O+BI><&%R
M.B0O+VUS.PHD;65T82`]?B!S+UY<4RXJ+R]M<SL*)&UE=&$@/7X@+UX@("A;
M7CI=*RDZ("@N*RDD+VUG.PI]"GT["FEF("AD969I;F5D("1P87(I('L*;W!E
M;B!M>2`D<&@L("<\.G)A=R<L("1P87(@;W(@9&EE('%Q6T-A;B=T(')E860@
M<&%R(&9I;&4@(B1P87(B.B`D(5T["FUY("1B=68["G)E860@)'!H+"`D8G5F
M+"`T.PID:64@<7%;(B1P87(B(&ES(&YO="!A('!A<B!F:6QE72!U;FQE<W,@
M)&)U9B!E<2`B4$M<,#`S7#`P-"(["F-L;W-E("1P:#L*?0I#<F5A=&50871H
M*"1O=70I(#L*;7D@)&9H(#T@24\Z.D9I;&4M/FYE=R@*)&]U="P*24\Z.D9I
M;&4Z.D]?0U)%050H*2!\($E/.CI&:6QE.CI/7U=23TY,62@I('P@24\Z.D9I
M;&4Z.D]?5%)53D,H*2P*,#<W-RP**2!O<B!D:64@<7%;0V%N)W0@8W)E871E
M(&9I;&4@(B1O=70B.B`D(5T["B1F:"T^8FEN;6]D92@I.PIS965K(%]&2"P@
M,"P@,#L*;7D@)&QO861E<CL*:68@*&1E9FEN960@)&1A=&%?<&]S*2!["G)E
M860@7T9(+"`D;&]A9&5R+"`D9&%T85]P;W,["GT@96QS92!["FQO8V%L("0O
M(#T@=6YD968["B1L;V%D97(@/2`\7T9(/CL*?0II9B`H(21%3E9[4$%27U9%
M4D)!5$E-?2!A;F0@)&QO861E<B`]?B`O7B@_.B,A?%Q`<F5M*2\I('L*<F5Q
M=6ER92!005(Z.D9I;'1E<CHZ4&]D4W1R:7`["E!!4CHZ1FEL=&5R.CI0;V13
M=')I<"T^87!P;'DH7"1L;V%D97(L("0P*3L*?0IF;W)E86-H(&UY("1K97D@
M*'-O<G0@:V5Y<R`E96YV*2!["FUY("1V86P@/2`D96YV>R1K97E](&]R(&YE
M>'0["B1V86P@/2!E=F%L("1V86P@:68@)'9A;"`]?B`O7ELG(ETO.PIM>2`D
M;6%G:6,@/2`B7U]%3E9?4$%27R(@+B!U8R@D:V5Y*2`N(")?7R(["FUY("1S
M970@/2`B4$%27R(@+B!U8R@D:V5Y*2`N("(])'9A;"(["B1L;V%D97(@/7X@
M<WLD;6%G:6,H("LI?7L*)&UA9VEC("X@)'-E="`N("@G("<@>"`H;&5N9W1H
M*"0Q*2`M(&QE;F=T:"@D<V5T*2DI"GUE9SL*?0HD9F@M/G!R:6YT*"1L;V%D
M97(I.PII9B`H)&)U;F1L92D@>PIR97%U:7)E(%!!4CHZ2&5A=GD["E!!4CHZ
M2&5A=GDZ.E]I;FET7V1Y;F%L;V%D97(H*3L*:6YI=%]I;F,H*3L*<F5Q=6ER
M95]M;V1U;&5S*"D["FUY($!I;F,@/2!G<F5P('L@(2]"4T1004XO('T*9W)E
M<"!["B@D8G5N9&QE(&YE("=S:71E)RD@;W(**"1?(&YE("1#;VYF:6<Z.D-O
M;F9I9WMA<F-H;&EB97AP?2!A;F0*)%\@;F4@)$-O;F9I9SHZ0V]N9FEG>W!R
M:79L:6)E>'!]*3L*?2!`24Y#.PIS?"\K?"]\9RP@<WPO)'Q\(&9O<F5A8V@@
M0&EN8SL*;7D@)69I;&5S.PHD9FEL97-[)%]]*RL@9F]R($!$>6YA3&]A9&5R
M.CID;%]S:&%R961?;V)J96-T<RP@=F%L=65S("5)3D,["FUY("1L:6)?97AT
M(#T@)$-O;F9I9SHZ0V]N9FEG>VQI8E]E>'1].R`@("`@("`@(",@6%A8(&QI
M8E]E>'0@=G,@9&QE>'0@/PIM>2`E=W)I='1E;CL*9F]R96%C:"!M>2`D:V5Y
M("AS;W)T(&ME>7,@)69I;&5S*2!["FUY("@D9FEL92P@)&YA;64I.PII9B`H
M9&5F:6YE9"AM>2`D9F,@/2`D4$%2.CI(96%V>3HZ1G5L;$-A8VAE>R1K97E]
M*2D@>PHH)&9I;&4L("1N86UE*2`]("@D:V5Y+"`D9F,I.PI]"F5L<V4@>PIF
M;W)E86-H(&UY("1D:7(@*$!I;F,I('L*:68@*"1K97D@/7X@;7Q>7%$D9&ER
M7$4O*"XJ*21\:2D@>PHH)&9I;&4L("1N86UE*2`]("@D:V5Y+"`D,2D["FQA
M<W0["GT*:68@*"1K97D@/7X@;7Q>+VQO861E<B];7B]=*R\H+BHI)'PI('L*
M:68@*&UY("1R968@/2`D36]D=6QE0V%C:&5[)#%]*2!["B@D9FEL92P@)&YA
M;64I(#T@*"1R968L("0Q*3L*;&%S=#L*?0II9B`H+68@(B1D:7(O)#$B*2![
M"B@D9FEL92P@)&YA;64I(#T@*"(D9&ER+R0Q(BP@)#$I.PIL87-T.PI]"GT*
M?0I]"FYE>'0@=6YL97-S(&1E9FEN960@)&YA;64["FYE>'0@:68@)'=R:71T
M96Y[)&YA;65]*RL["FYE>'0@:68@(7)E9B@D9FEL92D@86YD("1F:6QE(#U^
M("]<+EQ1)&QI8E]E>'1<120O:3L*;W5T<RAS<')I;G1F*'%Q6U!A8VMI;F<@
M1DE,12`B)7,B+BXN72P@<F5F("1F:6QE(#\@)&9I;&4M/GMN86UE?2`Z("1F
M:6QE*2D["FUY("1C;VYT96YT.PII9B`H<F5F*"1F:6QE*2D@>PHD8V]N=&5N
M="`]("1F:6QE+3Y[8G5F?3L*?0IE;'-E('L*;&]C86P@)"\@/2!U;F1E9CL*
M;W!E;B!M>2`D9F@L("<\.G)A=R<L("1F:6QE(&]R(&1I92!Q<5M#86XG="!R
M96%D("(D9FEL92(Z("0A73L*)&-O;G1E;G0@/2`\)&9H/CL*8VQO<V4@)&9H
M.PI005(Z.D9I;'1E<CHZ4&]D4W1R:7`M/F%P<&QY*%PD8V]N=&5N="P@(CQE
M;6)E9&1E9#XO)&YA;64B*0II9B`A)$5.5GM005)?5D520D%424U](&%N9"`D
M;F%M92`]?B`O7"XH/SIP;7QI>'QA;"DD+VD["E!!4CHZ1FEL=&5R.CI0871C
M:$-O;G1E;G0M/FYE=RT^87!P;'DH7"1C;VYT96YT+"`D9FEL92P@)&YA;64I
M.PI]"B1F:"T^<')I;G0H(D9)3$4B+`IP86-K*"=.)RP@;&5N9W1H*"1N86UE
M*2`K(#DI+`IS<')I;G1F*"(E,#AX+R5S(BP@07)C:&EV93HZ6FEP.CIC;VUP
M=71E0U)#,S(H)&-O;G1E;G0I+"`D;F%M92DL"G!A8VLH)TXG+"!L96YG=&@H
M)&-O;G1E;G0I*2P*)&-O;G1E;G0I.PIO=71S*'%Q6U=R:71T96X@87,@(B1N
M86UE(ETI.PI]"GT*)'II<"T^=W)I=&54;T9I;&5(86YD;&4H)&9H*2!I9B`D
M>FEP.PHD8V%C:&5?;F%M92`]('-U8G-T<B`D8V%C:&5?;F%M92P@,"P@-#`[
M"FEF("@A)&-A8VAE7VYA;64@86YD(&UY("1M=&EM92`]("AS=&%T*"1O=70I
M*5LY72D@>PIM>2`D8W1X(#T@1&EG97-T.CI32$$M/FYE=R@Q*3L*;W!E;B!M
M>2`D9F@L("(\.G)A=R(L("1O=70["B1C='@M/F%D9&9I;&4H)&9H*3L*8VQO
M<V4@)&9H.PHD8V%C:&5?;F%M92`]("1C='@M/FAE>&1I9V5S=#L*?0HD8V%C
M:&5?;F%M92`N/2`B7#`B('@@*#0Q("T@;&5N9W1H("1C86-H95]N86UE*3L*
M)&-A8VAE7VYA;64@+CT@(D-!0TA%(CL*)&9H+3YP<FEN="@D8V%C:&5?;F%M
M92D["B1F:"T^<')I;G0H<&%C:R@G3B<L("1F:"T^=&5L;"`M(&QE;F=T:"@D
M;&]A9&5R*2DI.PHD9F@M/G!R:6YT*"1005)?34%'24,I.PHD9F@M/F-L;W-E
M.PIC:&UO9"`P-S4U+"`D;W5T.PIE>&ET.PI]"GL*;&%S="!U;FQE<W,@9&5F
M:6YE9"`D<W1A<G1?<&]S.PI?9FEX7W!R;V=N86UE*"D["G)E<75I<F4@4$%2
M.PI005(Z.DAE879Y.CI?:6YI=%]D>6YA;&]A9&5R*"D["GL*<F5Q=6ER92!&
M:6QE.CI&:6YD.PIR97%U:7)E($%R8VAI=F4Z.EII<#L*?0IM>2`D9F@@/2!)
M3SHZ1FEL92T^;F5W.R`@("`@("`@("`@("`@("`@("`@("`@("`@("`@(R!!
M<F-H:79E.CI::7`@;W!E<F%T97,@;VX@86X@24\Z.DAA;F1L90HD9F@M/F9D
M;W!E;BAF:6QE;F\H7T9(*2P@)W(G*2!O<B!D:64@<7%;9F1O<&5N*"D@9F%I
M;&5D.B`D(5T["D%R8VAI=F4Z.EII<#HZ<V5T0VAU;FM3:7IE*"US(%]&2"D[
M"FUY("1Z:7`@/2!!<F-H:79E.CI::7`M/FYE=SL**"`D>FEP+3YR96%D1G)O
M;49I;&5(86YD;&4H)&9H+"`D<')O9VYA;64I(#T]($%R8VAI=F4Z.EII<#HZ
M05I?3TLH*2`I"F]R(&1I92!Q<5M%<G)O<B!R96%D:6YG('II<"!A<F-H:79E
M("(D<')O9VYA;64B73L*07)C:&EV93HZ6FEP.CIS971#:'5N:U-I>F4H-C0@
M*B`Q,#(T*3L*<'5S:"!`4$%2.CI,:6)#86-H92P@)'II<#L*)%!!4CHZ3&EB
M0V%C:&5[)'!R;V=N86UE?2`]("1Z:7`["B1Q=6EE="`]("$D14Y6>U!!4E]$
M14)51WT["FEF("AD969I;F5D("1%3E9[4$%27U1%35!]*2![(",@<VAO=6QD
M(&)E('-E="!A="!T:&ES('!O:6YT(0IF;W)E86-H(&UY("1M96UB97(@*"`D
M>FEP+3YM96UB97)S("D@>PIN97AT(&EF("1M96UB97(M/FES1&ER96-T;W)Y
M.PIM>2`D;65M8F5R7VYA;64@/2`D;65M8F5R+3YF:6QE3F%M93L*;F5X="!U
M;FQE<W,@)&UE;6)E<E]N86UE(#U^(&U["EX*+S]S:&QI8B\**#\Z)$-O;F9I
M9SHZ0V]N9FEG>W9E<G-I;VY]+RD_"B@_.B1#;VYF:6<Z.D-O;F9I9WMA<F-H
M;F%M97TO*3\**%M>+UTK*0HD"GUX.PIM>2`D97AT<F%C=%]N86UE(#T@)#$[
M"FUY("1D97-T7VYA;64@/2!&:6QE.CI3<&5C+3YC871F:6QE*"1%3E9[4$%2
M7U1%35!]+"`D97AT<F%C=%]N86UE*3L*:68@*"UF("1D97-T7VYA;64@)B8@
M+7,@7R`]/2`D;65M8F5R+3YU;F-O;7!R97-S9613:7IE*"DI('L*;W5T<RAQ
M<5M3:VEP<&EN9R`B)&UE;6)E<E]N86UE(B!S:6YC92!I="!A;')E861Y(&5X
M:7-T<R!A="`B)&1E<W1?;F%M92)=*3L*?2!E;'-E('L*;W5T<RAQ<5M%>'1R
M86-T:6YG("(D;65M8F5R7VYA;64B('1O("(D9&5S=%]N86UE(ETI.PHD;65M
M8F5R+3YE>'1R86-T5&]&:6QE3F%M960H)&1E<W1?;F%M92D["F-H;6]D*#`U
M-34L("1D97-T7VYA;64I(&EF("1>3R!E<2`B:'!U>"(["GT*?0I]"GT*=6YL
M97-S("@D4$%2.CI,:6)#86-H97LD<')O9VYA;65]*2!["F1I92`\/"`B+B(@
M=6YL97-S($!!4D=6.PI5<V%G93H@)#`@6R`M06QI8BYP87(@72!;("U)9&ER
M(%T@6R`M36UO9'5L92!=(%L@<W)C+G!A<B!=(%L@<')O9W)A;2YP;"!="B0P
M(%L@+4)\+6(@72!;+4]O=71F:6QE72!S<F,N<&%R"BX*)$5.5GM005)?4%)/
M1TY!345](#T@)'!R;V=N86UE(#T@)#`@/2!S:&EF="A`05)'5BD["GT*<W5B
M($-R96%T95!A=&@@>PIM>2`H)&YA;64I(#T@0%\["G)E<75I<F4@1FEL93HZ
M0F%S96YA;64["FUY("@D8F%S96YA;64L("1P871H+"`D97AT*2`]($9I;&4Z
M.D)A<V5N86UE.CIF:6QE<&%R<V4H)&YA;64L("@G7"XN*B<I*3L*<F5Q=6ER
M92!&:6QE.CI0871H.PI&:6QE.CI0871H.CIM:W!A=&@H)'!A=&@I('5N;&5S
M<R@M92`D<&%T:"D[(",@;6MP871H(&1I97,@=VET:"!E<G)O<@I]"G-U8B!R
M97%U:7)E7VUO9'5L97,@>PIR97%U:7)E(&QI8CL*<F5Q=6ER92!$>6YA3&]A
M9&5R.PIR97%U:7)E(&EN=&5G97(["G)E<75I<F4@<W1R:6-T.PIR97%U:7)E
M('=A<FYI;F=S.PIR97%U:7)E('9A<G,["G)E<75I<F4@0V%R<#L*<F5Q=6ER
M92!#87)P.CI(96%V>3L*<F5Q=6ER92!%<G)N;SL*<F5Q=6ER92!%>'!O<G1E
M<CHZ2&5A=GD["G)E<75I<F4@17AP;W)T97(["G)E<75I<F4@1F-N=&P["G)E
M<75I<F4@1FEL93HZ5&5M<#L*<F5Q=6ER92!&:6QE.CI3<&5C.PIR97%U:7)E
M(%A33&]A9&5R.PIR97%U:7)E($-O;F9I9SL*<F5Q=6ER92!)3SHZ2&%N9&QE
M.PIR97%U:7)E($E/.CI&:6QE.PIR97%U:7)E($-O;7!R97-S.CI:;&EB.PIR
M97%U:7)E($%R8VAI=F4Z.EII<#L*<F5Q=6ER92!$:6=E<W0Z.E-(03L*<F5Q
M=6ER92!005(["G)E<75I<F4@4$%2.CI(96%V>3L*<F5Q=6ER92!005(Z.D1I
M<W0["G)E<75I<F4@4$%2.CI&:6QT97(Z.E!O9%-T<FEP.PIR97%U:7)E(%!!
M4CHZ1FEL=&5R.CI0871C:$-O;G1E;G0["G)E<75I<F4@871T<FEB=71E<SL*
M979A;"![(')E<75I<F4@0W=D('T["F5V86P@>R!R97%U:7)E(%=I;C,R('T[
M"F5V86P@>R!R97%U:7)E(%-C86QA<CHZ571I;"!].PIE=F%L('L@<F5Q=6ER
M92!!<F-H:79E.CI5;GII<#HZ0G5R<W0@?3L*979A;"![(')E<75I<F4@5&EE
M.CI(87-H.CI.86UE9$-A<'1U<F4@?3L*979A;"![(')E<75I<F4@4&5R;$E/
M.R!R97%U:7)E(%!E<FQ)3SHZ<V-A;&%R('T["F5V86P@>R!R97%U:7)E('5T
M9C@@?3L*?0IS=6(@7W-E=%]P87)?=&5M<"!["FEF("AD969I;F5D("1%3E9[
M4$%27U1%35!](&%N9"`D14Y6>U!!4E]414U0?2`]?B`O*"XK*2\I('L*)'!A
M<E]T96UP(#T@)#$["G)E='5R;CL*?0IF;W)E86-H(&UY("1P871H("@**&UA
M<"`D14Y6>R1??2P@<7<H(%!!4E]435!$25(@5$U01$E2(%1%35!$25(@5$5-
M4"!435`@*2DL"G%W*"!#.EQ<5$5-4"`O=&UP("X@*0HI('L*;F5X="!U;FQE
M<W,@9&5F:6YE9"`D<&%T:"!A;F0@+60@)'!A=&@@86YD("UW("1P871H.PIM
M>2`D=7-E<FYA;64["FUY("1P=W5I9#L*979A;"![*"1P=W5I9"D@/2!G971P
M=W5I9"@D/BD@:68@9&5F:6YE9"`D/CM].PII9B`H(&1E9FEN960H)E=I;C,R
M.CI,;V=I;DYA;64I("D@>PHD=7-E<FYA;64@/2`F5VEN,S(Z.DQO9VEN3F%M
M93L*?0IE;'-I9B`H9&5F:6YE9"`D<'=U:60I('L*)'5S97)N86UE(#T@)'!W
M=6ED.PI]"F5L<V4@>PHD=7-E<FYA;64@/2`D14Y6>U5315).04U%?2!\?"`D
M14Y6>U5315)]('Q\("=365-414TG.PI]"B1U<V5R;F%M92`]?B!S+UQ7+U\O
M9SL*;7D@)'-T;7!D:7(@/2`B)'!A=&@D<WES>U]D96QI;7UP87(M(BYU;G!A
M8VLH(D@J(BP@)'5S97)N86UE*3L*;6MD:7(@)'-T;7!D:7(L(#`W-34["FEF
M("@A)$5.5GM005)?0TQ%04Y](&%N9"!M>2`D;71I;64@/2`H<W1A="@D<')O
M9VYA;64I*5LY72D@>PIO<&5N(&UY("1F:"P@(CPZ<F%W(BP@)'!R;V=N86UE
M(&]R(&1I92!Q<5M#86XG="!R96%D("(D<')O9VYA;64B.B`D(5T["G-E96L@
M)&9H+"`M,3@L(#(["FUY("1B=68["G)E860@)&9H+"`D8G5F+"`V.PII9B`H
M)&)U9B!E<2`B7#!#04-(12(I('L*<V5E:R`D9F@L("TU."P@,CL*<F5A9"`D
M9F@L("1B=68L(#0Q.PHD8G5F(#U^(',O7#`O+V<["B1S=&UP9&ER("X]("(D
M<WES>U]D96QI;7UC86-H92TD8G5F(CL*?0IE;'-E('L*;7D@)&1I9V5S="`]
M(&5V86P*>PIR97%U:7)E($1I9V5S=#HZ4TA!.PIM>2`D8W1X(#T@1&EG97-T
M.CI32$$M/FYE=R@Q*3L*;W!E;B!M>2`D9F@L("(\.G)A=R(L("1P<F]G;F%M
M92!O<B!D:64@<7%;0V%N)W0@<F5A9"`B)'!R;V=N86UE(CH@)"%=.PHD8W1X
M+3YA9&1F:6QE*"1F:"D["F-L;W-E*"1F:"D["B1C='@M/FAE>&1I9V5S=#L*
M?2!\?"`D;71I;64["B1S=&UP9&ER("X]("(D<WES>U]D96QI;7UC86-H92TD
M9&EG97-T(CL*?0IC;&]S92@D9F@I.PI]"F5L<V4@>PHD14Y6>U!!4E]#3$5!
M3GT@/2`Q.PHD<W1M<&1I<B`N/2`B)'-Y<WM?9&5L:6U]=&5M<"TD)"(["GT*
M)$5.5GM005)?5$5-4'T@/2`D<W1M<&1I<CL*;6MD:7(@)'-T;7!D:7(L(#`W
M-34["FQA<W0["GT*)'!A<E]T96UP(#T@)#$@:68@)$5.5GM005)?5$5-4'T@
M86YD("1%3E9[4$%27U1%35!](#U^("\H+BLI+SL*?0IS=6(@7W-A=F5?87,@
M>PIM>2`H)&YA;64L("1C;VYT96YT<RP@)&UO9&4I(#T@0%\["FUY("1F=6QL
M;F%M92`]("(D<&%R7W1E;7`O)&YA;64B.PIU;FQE<W,@*"UE("1F=6QL;F%M
M92D@>PIM>2`D=&5M<&YA;64@/2`B)&9U;&QN86UE+B0D(CL*;W!E;B!M>2`D
M9F@L("<^.G)A=R<L("1T96UP;F%M92!O<B!D:64@<7%;0V%N)W0@=W)I=&4@
M(B1T96UP;F%M92(Z("0A73L*<')I;G0@)&9H("1C;VYT96YT<SL*8VQO<V4@
M)&9H.PIC:&UO9"`D;6]D92P@)'1E;7!N86UE(&EF(&1E9FEN960@)&UO9&4[
M"G)E;F%M92@D=&5M<&YA;64L("1F=6QL;F%M92D@;W(@=6YL:6YK*"1T96UP
M;F%M92D["GT*<F5T=7)N("1F=6QL;F%M93L*?0IS=6(@7W-E=%]P<F]G;F%M
M92!["FEF("AD969I;F5D("1%3E9[4$%27U!23T=.04U%?2!A;F0@)$5.5GM0
M05)?4%)/1TY!345](#U^("\H+BLI+RD@>PHD<')O9VYA;64@/2`D,3L*?0HD
M<')O9VYA;64@?'P]("0P.PII9B`H)$5.5GM005)?5$5-4'T@86YD(&EN9&5X
M*"1P<F]G;F%M92P@)$5.5GM005)?5$5-4'TI(#X](#`I('L*)'!R;V=N86UE
M(#T@<W5B<W1R*"1P<F]G;F%M92P@<FEN9&5X*"1P<F]G;F%M92P@)'-Y<WM?
M9&5L:6U]*2`K(#$I.PI]"FEF("@A)$5.5GM005)?4%)/1TY!345](&]R(&EN
M9&5X*"1P<F]G;F%M92P@)'-Y<WM?9&5L:6U]*2`^/2`P*2!["FEF("AO<&5N
M(&UY("1F:"P@)SPG+"`D<')O9VYA;64I('L*<F5T=7)N(&EF("US("1F:#L*
M?0II9B`H+7,@(B1P<F]G;F%M921S>7-[7V5X97TB*2!["B1P<F]G;F%M92`N
M/2`D<WES>U]E>&5].PIR971U<FX["GT*?0IF;W)E86-H(&UY("1D:7(@*'-P
M;&ET("]<421S>7-[<&%T:%]S97!]7$4O+"`D14Y6>U!!5$A]*2!["FYE>'0@
M:68@97AI<W1S("1%3E9[4$%27U1%35!](&%N9"`D9&ER(&5Q("1%3E9[4$%2
M7U1%35!].PHD9&ER(#U^(',O7%$D<WES>U]D96QI;7U<120O+SL**"@D<')O
M9VYA;64@/2`B)&1I<B1S>7-[7V1E;&EM?21P<F]G;F%M921S>7-[7V5X97TB
M*2P@;&%S="D*:68@+7,@(B1D:7(D<WES>U]D96QI;7TD<')O9VYA;64D<WES
M>U]E>&5](CL**"@D<')O9VYA;64@/2`B)&1I<B1S>7-[7V1E;&EM?21P<F]G
M;F%M92(I+"!L87-T*0II9B`M<R`B)&1I<B1S>7-[7V1E;&EM?21P<F]G;F%M
M92(["GT*?0IS=6(@7V9I>%]P<F]G;F%M92!["B0P(#T@)'!R;V=N86UE('Q\
M/2`D14Y6>U!!4E]04D]'3D%-17T["FEF("AI;F1E>"@D<')O9VYA;64L("1S
M>7-[7V1E;&EM?2D@/"`P*2!["B1P<F]G;F%M92`]("(N)'-Y<WM?9&5L:6U]
M)'!R;V=N86UE(CL*?0IM>2`D<'=D(#T@*&1E9FEN960@)D-W9#HZ9V5T8W=D
M*2`_($-W9#HZ9V5T8W=D*"D*.B`H*&1E9FEN960@)E=I;C,R.CI'971#=V0I
M(#\@5VEN,S(Z.D=E=$-W9"@I(#H@8'!W9&`I.PIC:&]M<"@D<'=D*3L*)'!R
M;V=N86UE(#U^(',O7B@_/5PN7"X_7%$D<WES>U]D96QI;7U<12DO)'!W9"1S
M>7-[7V1E;&EM?2\["B1%3E9[4$%27U!23T=.04U%?2`]("1P<F]G;F%M93L*
M?0IS=6(@7W!A<E]I;FET7V5N=B!["FEF("@@)$5.5GM005)?24Y)5$E!3$E:
M141]*RL@/3T@,2`I('L*<F5T=7)N.PI](&5L<V4@>PHD14Y6>U!!4E])3DE4
M24%,25I%1'T@/2`R.PI]"F9O<B`H<7<H(%-005=.140@5$5-4"!#3$5!3B!$
M14)51R!#04-(12!04D]'3D%-12`I("D@>PID96QE=&4@)$5.5GLG4$%27R<N
M)%]].PI]"F9O<B`H<7<O(%1-4$1)4B!414U0($-,14%.($1%0E5'("\I('L*
M)$5.5GLG4$%27R<N)%]](#T@)$5.5GLG4$%27T=,3T)!3%\G+B1??2!I9B!E
M>&ES=',@)$5.5GLG4$%27T=,3T)!3%\G+B1??3L*?0IM>2`D<&%R7V-L96%N
M(#T@(E]?14Y67U!!4E]#3$5!3E]?("`@("`@("`@("`@("`@(CL*:68@*"1%
M3E9[4$%27U1%35!]*2!["F1E;&5T92`D14Y6>U!!4E]#3$5!3GT["GT*96QS
M:68@*"%E>&ES=',@)$5.5GM005)?1TQ/0D%,7T-,14%.?2D@>PIM>2`D=F%L
M=64@/2!S=6)S='(H)'!A<E]C;&5A;BP@,3(@*R!L96YG=&@H(D-,14%.(BDI
M.PHD14Y6>U!!4E]#3$5!3GT@/2`D,2!I9B`D=F%L=64@/7X@+UY005)?0TQ%
M04X]*%Q3*RDO.PI]"GT*<W5B(&]U=',@>PIR971U<FX@:68@)'%U:65T.PII
M9B`H)&QO9V9H*2!["G!R:6YT("1L;V=F:"`B0%]<;B(["GT*96QS92!["G!R
M:6YT(")`7UQN(CL*?0I]"G-U8B!I;FET7VEN8R!["G)E<75I<F4@0V]N9FEG
M.PIP=7-H($!)3D,L(&=R97`@9&5F:6YE9"P@;6%P("1#;VYF:6<Z.D-O;F9I
M9WLD7WTL('%W*`IA<F-H;&EB97AP('!R:79L:6)E>'`@<VET96%R8VAE>'`@
M<VET96QI8F5X<`IV96YD;W)A<F-H97AP('9E;F1O<FQI8F5X<`HI.PI]"G!A
M8VMA9V4@;6%I;CL*<F5Q=6ER92!005(["G5N<VAI9G0@0$E.0RP@7"9005(Z
M.F9I;F1?<&%R.PI005(M/FEM<&]R="A`<&%R7V%R9W,I.PID:64@<7%;<&%R
M+G!L.B!#86XG="!O<&5N('!E<FP@<V-R:7!T("(D<')O9VYA;64B.B!.;R!S
M=6-H(&9I;&4@;W(@9&ER96-T;W)Y7&Y="G5N;&5S<R`M92`D<')O9VYA;64[
M"F1O("1P<F]G;F%M93L*0T]213HZ97AI="@D,2D@:68@*"1`(#U^+UY?5$M?
M15A)5%]<*"A<9"LI7"DO*3L*9&EE("1`(&EF("1`.PI].PHD.CI?7T524D]2
M(#T@)$`@:68@)$`["GT*0T]213HZ97AI="@D,2D@:68@*"0Z.E]?15)23U(@
M/7XO7E]42U]%6$E47UPH*%QD*RE<*2\I.PID:64@)#HZ7U]%4E)/4B!I9B`D
M.CI?7T524D]2.PHQ.PI?7T5.1%]?"@```'!E<FQX<VDN8P```%]?14Y67U!!
M4E]#3$5!3E]?("`@("`@("`@("`@("`@`````#T```!`````"P```$`````0
M````0````"8```!`````-0```$`````#````0`````8```!`````%````$``
M```.````0`````T```!`````)0```$`````2````0````"@```!``````@``
M`$`````8````0`````D```!`````"@```$```````````````(&R`8&PL*P'
M`````+ZR`8&PL*\_`````&B*_W^PJ@F`%(W_?P$````8D/]_L*L>@)21_W_,
M__]_Q)+_?P$`````E?]_L*@#@*25_W\!````E*'_?["PJ("PH?]_`0```!RC
M_W^H__]_1*O_?["K`H`0KO]_`0``````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``#4&0``8!D```$````]`P```0```$@#```!````4@,```$```!F`P``#P``
M`.8$```,````P!(```T```!(2@``&0```+"-`0`;````!````!H```"TC0$`
M'`````0```#U_O]OM`$```4```"P"0``!@```!`#```*````"04```L````0
M````%0`````````#````T(X!``(```#@`0``%````!$````7````X!```!$`
M```0$```$@```-`````3````"````!@`````````^___;P$```C^__]OD`\`
M`/___V\#````\/__;[H.``#Z__]O#@``````````````````````````````
M``````````````````````````"XC0$```````````#,$@``S!(``,P2``#,
M$@``S!(``,P2``#,$@``S!(``,P2``#,$@``S!(``,P2``#,$@``S!(``,P2
M``#,$@``S!(``,P2``#,$@``S!(``,P2``#,$@``S!(``,P2``#,$@``S!(`
M`,P2``#,$@``S!(``,P2``#,$@``S!(``,P2``#,$@``S!(``,P2``#,$@``
MS!(``,P2``#,$@``S!(``,P2``#,$@``S!(``,P2``#,$@``S!(``,P2``#,
M$@``S!(``,P2``#,$@``S!(``,P2``#,$@``S!(``,P2``#,$@``S!(``,P2
M````````````````````````````````````````````````````````````
M``````"P%0````````20`0#_____$$X``$!+`````````$P```1,```$3@``
M2$L```Q.````````\$T``/A-````````&$X``$=#0SH@*$=.52D@,3(N,2XP
M`$=#0SH@*&-R;W-S=&]O;"U.1R`Q+C(U+C`N,CA?8C@X9#,S."D@,3(N,2XP
M`$$T````865A8FD``2H````%-RU!``8*!T$(`0D""@,,`1($$P$4`14!%P,8
M`1D!&@(<`2(!`"YS:'-T<G1A8@`N:6YT97)P`"YN;W1E+F=N=2YB=6EL9"UI
M9``N;F]T92Y!0DDM=&%G`"YG;G4N:&%S:``N9'EN<WEM`"YD>6YS='(`+F=N
M=2YV97)S:6]N`"YG;G4N=F5R<VEO;E]R`"YR96PN9'EN`"YR96PN<&QT`"YI
M;FET`"YT97AT`"YF:6YI`"YR;V1A=&$`+D%232YE>'1A8@`N05)-+F5X:61X
M`"YE:%]F<F%M90`N:6YI=%]A<G)A>0`N9FEN:5]A<G)A>0`N9'EN86UI8P`N
M9V]T`"YD871A`"YB<W,`+F-O;6UE;G0`+D%232YA='1R:6)U=&5S````````
M```````````````````````````````````````````````+`````0````(`
M``!4`0``5`$``!D```````````````$`````````$P````<````"````<`$`
M`'`!```D```````````````$`````````"8````'`````@```)0!``"4`0``
M(```````````````!``````````T````]O__;P(```"T`0``M`$``%P!```%
M``````````0````$````/@````L````"````$`,``!`#``"@!@``!@````,`
M```$````$````$8````#`````@```+`)``"P"0``"04``````````````0``
M``````!.````____;P(```"Z#@``N@X``-0````%``````````(````"````
M6P```/[__V\"````D`\``)`/``"`````!@````,````$`````````&H````)
M`````@```!`0```0$```T`````4`````````!`````@```!S````"0```$(`
M``#@$```X!```.`!```%````%@````0````(````?`````$````&````P!(`
M`,`2```,```````````````$`````````'<````!````!@```,P2``#,$@``
MY`(`````````````!`````0```""`````0````8```"P%0``L!4``)@T````
M``````````@`````````B`````$````&````2$H``$A*```(````````````
M```$`````````(X````!`````@```%!*``!02@``X$``````````````!```
M``````"6`````0````(````PBP``,(L``!@```````````````0`````````
MH0````$``'""````2(L``$B+``!@````#0`````````$`````````*P````!
M`````@```*B+``"HBP``!```````````````!`````````"V````#@````,`
M``"PC0$`L(T```0```````````````0````$````P@````\````#````M(T!
M`+2-```$```````````````$````!````,X````&`````P```+B-`0"XC0``
M&`$```8`````````!`````@```#7`````0````,```#0C@$`T(X``#`!````
M``````````0````$````W`````$````#`````)`!``"0``!`````````````
M```$`````````.(````(`````P```$"0`0!`D```&!``````````````"```
M``````#G`````0```#``````````0)```#\```````````````$````!````
M\`````,``'```````````'^0```U```````````````!``````````$````#
C``````````````"TD`````$``````````````0``````````
