# pyOCD debugger
# Copyright (c) 2020 NXP
# Copyright (c) 2018 Arm Limited
# SPDX-License-Identifier: Apache-2.0
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

import logging

from ...coresight.coresight_target import CoreSightTarget
from ...core.memory_map import (FlashRegion, RomRegion, RamRegion, MemoryMap)
from ...debug.svd.loader import SVDFile
from ...coresight.ap import AccessPort, APv1Address
from ...coresight.cortex_m import CortexM

LOG = logging.getLogger(__name__)
FCFB = 0x42464346
FCFB_ADDR = 0x30000400
RESET_HANDLE_ADDR = 0x30002004

FPB_CTRL = 0xE0002000
FPB_COMP0 = 0xE0002008

SRC_SCR = 0x40c04000
SRC_SBMR = 0x40c04004
SRC_SBMR2 = 0x40c0400c

IOMUX_LPSR_GPR26 = 0x40C0C068
IOMUX_LPSR_GPR0 = 0x40c0c000
IOMUX_LPSR_GPR1 = 0x40c0c004

FLASH_ALGO_CM4 = {
    'load_address' : 0x20200000,

    # Flash algorithm as a hex string
    'instructions': [
    0xE00ABE00, 0x062D780D, 0x24084068, 0xD3000040, 0x1E644058, 0x1C49D1FA, 0x2A001E52, 0x4770D1F2,
    0x28001e40, 0x4770d1fc, 0x4605b570, 0x4616460c, 0xcc0fe002, 0x3e10c50f, 0xd2fa2e10, 0xd3022e08,
    0xc503cc03, 0x2e043e08, 0xcc01d307, 0x1f36c501, 0x7821e003, 0x1c647029, 0x1e761c6d, 0xbd70d2f9,
    0x4770ba40, 0x4770ba40, 0x4770ba40, 0x4770ba40, 0x4770ba40, 0x4770ba40, 0x4770bac0, 0x4770bac0,
    0x4770bac0, 0x4770bac0, 0x4770bac0, 0x4770bac0, 0xb5304770, 0xb0894d6c, 0x6868444d, 0x1c40496c,
    0x486a6068, 0x68002400, 0xd1244288, 0x20004969, 0x185201c2, 0x1c406014, 0xd3f9284f, 0x02402001,
    0x49646008, 0x31801c40, 0xf0006008, 0x4962fe10, 0x44792214, 0xf7ff4668, 0x4668ffa7, 0xfa46f000,
    0x4478485d, 0x68003818, 0xa8079007, 0xfadaf000, 0x485a495b, 0x485b6048, 0x20216384, 0x0240495a,
    0xf0016288, 0x4859fa67, 0xaa059005, 0x68292001, 0xfa73f001, 0xd1052800, 0x68292001, 0xfa73f001,
    0xd0002800, 0xb0092001, 0x2000bd30, 0x48464770, 0x4448b510, 0x20016801, 0xfa93f001, 0xd0002800,
    0xbd102001, 0x06c9211f, 0x1842b510, 0xd1090500, 0x2301483d, 0x68014448, 0x2001031b, 0xfa6cf001,
    0xd0002800, 0xbd102001, 0xb0c3b5f0, 0x4604460e, 0x25004617, 0xa80321fc, 0xfbe3f001, 0x44784836,
    0x684038b4, 0x201f9002, 0x182406c0, 0x0e000620, 0x2001d024, 0x182106c0, 0xa802b2e2, 0xfb72f001,
    0x21ffb2e0, 0x1a0a3101, 0x9000a902, 0x92011840, 0xf0014639, 0x9800fb67, 0x1a22ab02, 0x44484822,
    0x20016801, 0xfa25f001, 0xd1382800, 0x21ff9801, 0x31011904, 0x1a08b2e0, 0x1a3619c7, 0x90000a30,
    0x0228e00c, 0x190219c3, 0x44484817, 0x20016801, 0xfa0ff001, 0xd1222800, 0x1c6d9800, 0xd8f042a8,
    0x42b5022d, 0xb2f2d01e, 0xa80219e9, 0xfb3af001, 0xb2f0192c, 0x18222101, 0x185106c9, 0x320122ff,
    0xab021a12, 0xf00118c0, 0x4807fb2d, 0x44484622, 0xab026801, 0xf0012001, 0x2800f9ec, 0x2001d002,
    0xbdf0b043, 0xe7fb2000, 0x00000004, 0x40c84800, 0x001170a0, 0x40cc0000, 0x000026c6, 0x00000503,
    0x40cc0a00, 0x40c04000, 0x40c0c040, 0xc0000007, 0x4604b5fe, 0xd0032804, 0xd0012c05, 0xfaf8f001,
    0xd03c2c04, 0x21002003, 0xf941f001, 0x06400101, 0x0f4e0e40, 0x2c049002, 0x2003d033, 0xf0012130,
    0xf002f936, 0x9101f81a, 0x2c049000, 0x2003d02b, 0xf0012120, 0xf002f92c, 0x4604f810, 0x460d2001,
    0xf00240b0, 0x4606f802, 0x4620460f, 0x9b014629, 0xf0019a00, 0x4604fd91, 0x9802460d, 0xfffdf001,
    0x462b4622, 0xfd58f001, 0x4bfe2200, 0xfff8f001, 0x463b4632, 0xfd80f001, 0xffa2f001, 0x2002bdfe,
    0x2002e7c1, 0x2002e7ca, 0xb510e7d2, 0xf0020003, 0x0406fa33, 0x28261513, 0x48f32f2a, 0x68006801,
    0x03c0b2cc, 0x21010f80, 0x40811c40, 0x48ef0049, 0xfb09f001, 0xe0194360, 0xbd1048ed, 0x318049ea,
    0x06446908, 0x0e646808, 0x6a090080, 0x00890880, 0xf0010889, 0x49e5faf8, 0x43481900, 0x48e5e006,
    0x2004bd10, 0x2005e000, 0xff82f7ff, 0xd1f72800, 0xfa7ef001, 0x2802b510, 0x2803d001, 0x223fd116,
    0x409a00cb, 0xd0022802, 0xd10f2803, 0x48d6e005, 0x6b013040, 0x401148d8, 0x48d3e003, 0x48d56b01,
    0x40d94011, 0x3a0c460a, 0xd9012a17, 0xfa60f001, 0xd0042900, 0xfac7f001, 0x43482112, 0x2000bd10,
    0x4601bd10, 0x48c9b510, 0xf002000b, 0x111ff9d5, 0x161b1619, 0x1f1d1d16, 0x2a25231f, 0x502e2e2c,
    0x36343230, 0x403b3939, 0x49494747, 0x00504b4b, 0x680048c2, 0xd5010740, 0xbd1048c1, 0xbd1048c1,
    0xbd1048c1, 0xbd1048c1, 0xe0002000, 0xf7ff2002, 0xe02aff7c, 0xe0002100, 0x20022101, 0xffaaf7ff,
    0x2102e023, 0x2103e7f9, 0x2003e7f7, 0x2100e7ef, 0x2101e004, 0x2102e002, 0x2103e000, 0xe7ed2003,
    0xe7e42001, 0xf7ff2001, 0x0840ff60, 0x2001e00d, 0xff5bf7ff, 0xf0012105, 0xe006fa76, 0xe7d62004,
    0xe7d42005, 0xf0002001, 0x2800f804, 0xf001d1cb, 0xb530f9ff, 0x49a6b0ff, 0x4604b0a0, 0x00d2224f,
    0x46684479, 0xfdd0f7ff, 0x01e148a2, 0x68081809, 0x05406809, 0xb2cd0f40, 0x466a00e1, 0x5c081889,
    0xf7ff1c6d, 0x4629ff8e, 0xfa4df001, 0xd1012800, 0xf9def001, 0xb020b07f, 0xb5f8bd30, 0x68404604,
    0x39684601, 0xd9012968, 0xf9d2f001, 0x68294d86, 0x4e924f91, 0xd50a0489, 0x43c96829, 0x4208b2c9,
    0x7820d105, 0x03c06829, 0x03c04388, 0xf000d029, 0x6829fdbd, 0x22014628, 0x43910752, 0x0f92044a,
    0x2203d004, 0x43910352, 0x60014331, 0x79254a84, 0x78224011, 0x07924604, 0x43150bd2, 0x4981430d,
    0x6005430d, 0x8f4ff3bf, 0x8f6ff3bf, 0x201e4970, 0xf835f001, 0x00806820, 0x433dd5fc, 0x602543b5,
    0x6828bdf8, 0xd4020440, 0x43386828, 0x68286028, 0xd5f50040, 0x43b06828, 0xbdf86028, 0x68084962,
    0x43904a71, 0x4310054a, 0x47706008, 0x4d5eb5f8, 0x35404604, 0x4f6d6828, 0x02004e68, 0x6838d515,
    0x42886861, 0x6938d111, 0x428868a1, 0x6828d10d, 0xd4030480, 0x14716828, 0x60284308, 0x00406828,
    0x6828d502, 0x602843b0, 0xf000bdf8, 0x6b28fd5f, 0x4308495f, 0x68286328, 0x4208495e, 0x43c9d003,
    0x43304008, 0x485c6028, 0x68606238, 0x68a06038, 0x48526138, 0x60283008, 0x201e4c45, 0xf0004621,
    0x4856ffde, 0x46216028, 0xf00020fa, 0x4954ffd8, 0x68286029, 0xd5fc0080, 0x03402001, 0x60294301,
    0x602943b1, 0x494a6b28, 0x400843c9, 0xbdf86328, 0x30404835, 0x4a466b01, 0x63014311, 0x06026801,
    0x60014311, 0x4a476801, 0x60014011, 0xb5f84770, 0x4611468c, 0x2917390c, 0x2802d803, 0x2803d003,
    0xf001d006, 0x4839f915, 0x38104c38, 0xe0033c30, 0x4c254825, 0x34203030, 0x46636801, 0x254000db,
    0x4029409d, 0x68079100, 0x409e2680, 0x60074337, 0x213f6807, 0x438f4099, 0x06916007, 0x68020e89,
    0x43114099, 0x68216001, 0x23024662, 0x40594093, 0x68016021, 0x600143b1, 0x9a006801, 0x42914029,
    0xbdf8d0fa, 0x4c10b5f8, 0x69204605, 0x02804e1b, 0x7828d504, 0x43886921, 0xd05a0740, 0xfcd6f000,
    0x491b6b20, 0x63204308, 0x49147828, 0x0f400740, 0x18453110, 0x4f066125, 0x4639201e, 0xff5ff000,
    0xe0334819, 0x4176e360, 0x40c84200, 0x016e3600, 0x3b9aca00, 0x1c9c3800, 0x1f78a400, 0x40c840c0,
    0x00f42400, 0x003d0900, 0x02dc6c00, 0x17d78400, 0x0000232c, 0x40cc0000, 0x00004000, 0x40000000,
    0xfffe7f00, 0x40002000, 0x20006000, 0x40c84280, 0x80808080, 0x00802000, 0x0fffffff, 0x40800808,
    0x40800008, 0xff7fdff7, 0x00200800, 0x61254305, 0x201e4639, 0xff23f000, 0x02c02001, 0x61254385,
    0x00806920, 0x48fed5fc, 0x61254305, 0x612543b5, 0x6920bdf8, 0xd4040480, 0x21016920, 0x43080349,
    0x69206120, 0xd5f30040, 0x43b06920, 0xbdf86120, 0xb5104770, 0xd0022900, 0x03522201, 0x2200e000,
    0x039b2301, 0x290018d2, 0x2104d001, 0x2108e000, 0xfea9f000, 0xb510bd10, 0xd0012900, 0xe0002104,
    0x22012108, 0xf0000412, 0xbd10fe9e, 0x2900b510, 0x2104d001, 0x2108e000, 0x05922201, 0xfe93f000,
    0xb510bd10, 0xd0012900, 0xe0002104, 0x22012108, 0xf0000352, 0xbd10fe88, 0x460db570, 0x21014604,
    0xffeff7ff, 0x462849d7, 0xfec9f000, 0x46202100, 0xffe7f7ff, 0xb510bd70, 0xd0012900, 0xe0002104,
    0x22012108, 0xf00003d2, 0xbd10fe6e, 0xb081b5ff, 0x461d9f0a, 0x4604460e, 0xd0022801, 0xf7ff2100,
    0x463affa8, 0x46202130, 0xfe5df000, 0x46202120, 0xf0009a03, 0x2000fe58, 0x90004601, 0x4632237f,
    0xf0004620, 0x2c01fe58, 0x2019d008, 0x90002307, 0x462a065b, 0x46202100, 0xfe4df000, 0x46202101,
    0xffa4f7ff, 0x206449b7, 0xfe89f000, 0x46202101, 0xff7ff7ff, 0xbdf0b005, 0x49b32201, 0x28000392,
    0xd0026808, 0x60084310, 0x43904770, 0x2201e7fb, 0x035249ad, 0x68082800, 0x4310d002, 0x47706008,
    0xe7fb4390, 0x4604b5f8, 0xf0002600, 0x4da6fbcf, 0x49a66828, 0xd1004381, 0x21012601, 0xf7ff2002,
    0x2e00ff6a, 0x2000d011, 0xffe1f7ff, 0xf7ff2001, 0x2100ffd3, 0xf7ff2002, 0x2100ff8e, 0xf7ff2002,
    0x2100ff48, 0xf7ff2002, 0x2001ff61, 0xffcff7ff, 0x900068a0, 0x78217863, 0x68622002, 0xff86f7ff,
    0x200221e1, 0xff68f7ff, 0x00806828, 0x2101d5fc, 0xf7ff2002, 0x2000ff70, 0xffaef7ff, 0x20022100,
    0xff39f7ff, 0xb510bdf8, 0x0004460a, 0x2051d005, 0x42820100, 0x2a14d801, 0xf000d801, 0x4984ff89,
    0x210060a1, 0xd3034282, 0x20367061, 0xe0427020, 0x00c02051, 0xd3074282, 0x21187061, 0xf0004610,
    0x7020ffe2, 0xe035487b, 0x304520ff, 0xd3084282, 0x70602001, 0x4610210c, 0xffd5f000, 0x48767020,
    0x2aa2e028, 0x2002d308, 0x21067060, 0xf0004610, 0x7020ffca, 0xe01d4871, 0xd3082a51, 0x70602003,
    0x46104601, 0xffbff000, 0x486d7020, 0x2a28e012, 0x2004d308, 0x00507060, 0xf0002103, 0x7020ffb4,
    0xe0074868, 0x70602005, 0x21030090, 0xffabf000, 0x48657020, 0x20004341, 0xbd106061, 0x2200b50e,
    0x92004601, 0x46689201, 0xf7ff9202, 0x2800ff9c, 0x2001d001, 0x4668bd0e, 0xff54f7ff, 0xbd0e2000,
    0x2000b510, 0xff43f7ff, 0xf7ff2001, 0x2100ff35, 0xf7ff2002, 0x2100fef0, 0xf7ff2002, 0x2100feaa,
    0xf7ff2002, 0xbd10fec3, 0x22014947, 0x03923140, 0x69082800, 0x4310d002, 0x47706108, 0xe7fb4390,
    0x22014941, 0x03523140, 0x69082800, 0x4310d002, 0x47706108, 0xe7fb4390, 0x460ab510, 0xd0050004,
    0x01802027, 0xd8014282, 0xd2012a9c, 0xfef0f000, 0x42822103, 0x7021d302, 0xe02120d0, 0x01402027,
    0xd3024282, 0x210c7021, 0x2027e006, 0x42820100, 0x2000d306, 0x21067020, 0xf0004610, 0xe00fff44,
    0x303920ff, 0xd3034282, 0x70202001, 0xe7f32103, 0x2a9c2002, 0xd9027020, 0x21030050, 0x2068e7ed,
    0x20006060, 0xb51cbd10, 0x46684601, 0xffc4f7ff, 0xd0012800, 0xbd1c2001, 0xf7ff4668, 0x2000fcde,
    0xb5f8bd1c, 0x26004604, 0xfab0f000, 0x35404d16, 0x49166928, 0xd1004381, 0x21012601, 0xf7ff2003,
    0x2e00fe4a, 0x2000d011, 0xff9af7ff, 0xf7ff2001, 0x2100ff8b, 0xf7ff2003, 0x2100fe6e, 0xf7ff2003,
    0x2100fe28, 0xf7ff2003, 0x2001fe41, 0xff88f7ff, 0x900068a0, 0x78217863, 0x68622003, 0x0000e016,
    0x00002008, 0x3b9aca00, 0x40c84300, 0x20002000, 0x0fffffff, 0x00aaaaaa, 0x01555555, 0x02aaaaaa,
    0x05555555, 0x0aaaaaaa, 0x15555554, 0xfe4ef7ff, 0x200321e1, 0xfe30f7ff, 0x00806928, 0x2101d5fc,
    0xf7ff2003, 0x2000fe38, 0xff4ef7ff, 0x20032100, 0xfe01f7ff, 0xb50ebdf8, 0x46012200, 0x92019200,
    0x92024668, 0xfebff7ff, 0xd0012800, 0xbd0e2001, 0xf7ff4668, 0x2000ff96, 0xb510bd0e, 0xf7ff2000,
    0x2001ff3f, 0xff30f7ff, 0x20032100, 0xfe13f7ff, 0x20032100, 0xfdcdf7ff, 0x20032100, 0xfde6f7ff,
    0x2201bd10, 0x039249e0, 0x68082800, 0x4310d002, 0x47706008, 0xe7fb4390, 0x49db2201, 0x28000652,
    0xd0026808, 0x60084310, 0x43904770, 0x2201e7fb, 0x069249d5, 0x68082800, 0x4310d002, 0x47706008,
    0xe7fb4390, 0x49d02201, 0x28000352, 0xd0026808, 0x60084310, 0x43904770, 0x49cbe7fb, 0x00806808,
    0x4770d5fc, 0x4604b538, 0xfa00f000, 0x46082101, 0xfda1f7ff, 0xf7ff2001, 0x48c4ffe5, 0x4ac49000,
    0x23002129, 0xf7ff2001, 0x21e1fdd1, 0xf7ff2001, 0xf7fffdb3, 0x2101ffe2, 0xf7ff4608, 0x2000fdbc,
    0xffaff7ff, 0xf7ff7820, 0x7860ffb7, 0xffbff7ff, 0x20012100, 0xfd7ff7ff, 0xb510bd38, 0xf7ff2000,
    0x2000ffc1, 0xffa8f7ff, 0xf7ff2000, 0x2001ffb0, 0xff97f7ff, 0x20012100, 0xfd9df7ff, 0x20012100,
    0xfd57f7ff, 0x20012100, 0xfd70f7ff, 0x48a9bd10, 0x06c96a01, 0x2114d404, 0x6a016201, 0xd5fc0049,
    0xb5104770, 0xd3012802, 0xfdb2f000, 0x790b780a, 0xd0012a00, 0x06122201, 0x884a4313, 0x0412794c,
    0x43220124, 0x798a4313, 0x43130212, 0x031279ca, 0x01c24313, 0x18104898, 0x7a4b6003, 0x011b7a0a,
    0x7a8b431a, 0x021b7acc, 0x43230324, 0x7b0b431a, 0x431a041b, 0x051b7b4b, 0x7b8b431a, 0x061b7bc9,
    0x0709431a, 0x6102430a, 0xb570bd10, 0x6900488b, 0x0fca0581, 0x0fc905c1, 0x2a00b2c0, 0x4c88d011,
    0x62222220, 0x06000289, 0x63214301, 0x00d56a20, 0x62204328, 0x20014983, 0xfc01f000, 0x43a86a20,
    0xbd706220, 0x3040487b, 0x08496901, 0x61010049, 0x22016a01, 0x62014311, 0x01cb4770, 0x497a2201,
    0x18590612, 0x1252600a, 0x10d2604a, 0x680a608a, 0x02520a52, 0x680a600a, 0x0dc005c0, 0x041b2303,
    0x43024318, 0x4871600a, 0x6c086088, 0xd0fc2800, 0x00806c08, 0x20014770, 0xb5f84770, 0x2400460f,
    0xf0002073, 0x2001f8b1, 0x07c04e69, 0x4d6863b0, 0x35804868, 0x46396268, 0xf0004867, 0x2900fd94,
    0x2107d114, 0x0589b280, 0xd01a2816, 0x280ddc08, 0x280fd021, 0x2810d00c, 0x2814d00d, 0xe00dd106,
    0xd0112819, 0xd012281e, 0xd01328f0, 0xbdf82000, 0x05a42401, 0x2401e00f, 0xe00c05e4, 0x05a42403,
    0x2401e009, 0xe0060624, 0x05a42405, 0x2403e003, 0xe00005e4, 0x6a28460c, 0x43204388, 0x20016228,
    0x62a80400, 0x62682040, 0x63b00600, 0x60702000, 0x28006a28, 0x2001dafc, 0x4945bdf8, 0x31802040,
    0x48436288, 0x04026b01, 0x63014311, 0x20014770, 0xb5f84770, 0x2400460f, 0xf0002073, 0x2001f855,
    0x07c04e3e, 0x4d3d63b0, 0x3580483a, 0x46396268, 0xf0004839, 0x2900fd38, 0x2107d114, 0x0589b280,
    0xd01a2816, 0x280ddc08, 0x280fd021, 0x2810d00c, 0x2814d00d, 0xe00dd106, 0xd0112819, 0xd012281e,
    0xd01328f0, 0xbdf82000, 0x05a42401, 0x2401e00f, 0xe00c05e4, 0x05a42403, 0x2401e009, 0xe0060624,
    0x05a42405, 0x2403e003, 0xe00005e4, 0x6a28460c, 0x43204388, 0x20016228, 0x62a80400, 0x62682040,
    0x63b00600, 0x60702000, 0x28006a28, 0x2001dafc, 0x491abdf8, 0x31802040, 0x48186288, 0x22016b01,
    0x43110792, 0x47706301, 0x48150142, 0x18102101, 0x400a6802, 0xd0054291, 0xf3bf6001, 0xf3bf8f4f,
    0x68008f6f, 0x00004770, 0x40c842c0, 0x0fffffff, 0x0aaaaaaa, 0x40c84000, 0x40cc4000, 0x40c84100,
    0x40c84900, 0x3b9aca00, 0x40150000, 0x01008000, 0x40434000, 0x00201000, 0x1c9c3800, 0x40438000,
    0x40cc6000, 0x6a114abc, 0x061b2301, 0x06004399, 0x62104308, 0x4ab84770, 0x23076a11, 0x0740029b,
    0x0cc04399, 0x62104308, 0x4ab34770, 0x04c06911, 0x03490b49, 0x43080cc0, 0x47706110, 0x69114aae,
    0x09090109, 0x43080700, 0x47706110, 0x69114aaa, 0x061b230f, 0x43990700, 0x43080900, 0x47706110,
    0x2800b500, 0x2001d004, 0xfff0f7ff, 0xe00a2001, 0xd0042900, 0xf7ff2005, 0x2005ffe9, 0x2000e003,
    0xffe4f7ff, 0xf7ff2000, 0xbd00ffd9, 0x2100b500, 0xf7ff2001, 0x20ffffe5, 0xf7ff30ef, 0x2001ffc6,
    0xffb9f7ff, 0xf7ff2001, 0xbd00ffad, 0x2100b510, 0xf0004608, 0x22fffa4c, 0x42903206, 0x2100d01b,
    0xf0004608, 0x2100fa40, 0xf0004608, 0x20fffa40, 0x30014989, 0x60083140, 0x6a014888, 0x04122201,
    0x62014391, 0x20014986, 0xfa79f000, 0x6b014882, 0x43112210, 0xbd106301, 0xb5102200, 0x46104611,
    0xfa21f000, 0x229dbd10, 0x0092b510, 0x1a802114, 0xfc39f000, 0x3a404a78, 0x231f6b11, 0x06c0051b,
    0x09c04399, 0x63104308, 0xb570bd10, 0x3c404c72, 0x08406920, 0x61200040, 0x267d4d71, 0x462900f6,
    0xf0004630, 0x6922fa4c, 0x04c02001, 0x61224302, 0x46304629, 0xfa43f000, 0x21206920, 0x61204308,
    0x46304629, 0xfa3bf000, 0x21046920, 0x61204308, 0xb570bd70, 0x3c404c60, 0x21056920, 0x61204388,
    0x267d4d5f, 0x462900f6, 0xf0004630, 0x6922fa28, 0x43822020, 0x46296122, 0xf0004630, 0x6920fa20,
    0x04c92101, 0x61204388, 0xb570bd70, 0x3c404c52, 0x21016b20, 0x63204308, 0x267d4d51, 0x462900f6,
    0xf0004630, 0x6b22fa0c, 0x04402001, 0x63224302, 0x46304629, 0xfa03f000, 0x21016b20, 0x43080489,
    0x46296320, 0xf0004630, 0x6b20f9fa, 0x43882104, 0xbd706320, 0x4c40b570, 0x6b203c40, 0x43082105,
    0x4d3f6320, 0x00f6267d, 0x46304629, 0xf9e7f000, 0x20016b22, 0x43820480, 0x46296322, 0xf0004630,
    0x6b20f9de, 0x04492101, 0x63204388, 0xb570bd70, 0x3c404c31, 0x08406920, 0x61200040, 0x21016b20,
    0x63204308, 0x267d4d2e, 0x462900f6, 0xf0004630, 0x6922f9c6, 0x04c02001, 0x61224302, 0x21016b20,
    0x43080449, 0x46296320, 0xf0004630, 0x6920f9b8, 0x43082120, 0x6b206120, 0x43080349, 0x46296320,
    0xf0004630, 0x6921f9ac, 0x43012004, 0x6b216121, 0x63214381, 0xb570bd70, 0x3c404c17, 0x20046921,
    0x61214381, 0x43016b21, 0x69206321, 0x00400840, 0x6b206120, 0x43082101, 0x4d116320, 0x00f6267d,
    0x46304629, 0xf98bf000, 0x20016b22, 0x43820480, 0x69206322, 0x43882120, 0x46296120, 0xf0004630,
    0x6b20f97e, 0x04492101, 0x63204388, 0x00896920, 0x61204388, 0x0000bd70, 0x40c84540, 0x40c84800,
    0x3b9aca00, 0x25ffb5f8, 0x461f3501, 0x11e6022c, 0xf0010003, 0x0507fa51, 0xa7784c23, 0x00f2f4d1,
    0x29004897, 0x6a014b97, 0x43a1d00a, 0x6a016201, 0x02090a09, 0x62014311, 0x68186307, 0xe0cc4060,
    0x432143a1, 0x6a016201, 0x02090a09, 0x62014311, 0x40606818, 0x488a6018, 0xe0f03040, 0x2900488a,
    0xd0116901, 0x40316903, 0x610343a3, 0x0a1b6903, 0x4313021b, 0x62076103, 0x406a6902, 0x69026102,
    0x428a4032, 0xe0ccd0fb, 0x40316903, 0x432343a3, 0x69036103, 0x021b0a1b, 0x61034313, 0x406a6902,
    0x69026102, 0x428a4032, 0x6b00d0fb, 0x4877bdf8, 0x68012900, 0x6803d014, 0x43a34031, 0x68036003,
    0x021b0a1b, 0x60034313, 0x68026107, 0x6002406a, 0x40326802, 0xd0fb428a, 0x43a16801, 0xe0a06001,
    0x40316803, 0x432343a3, 0x68036003, 0x021b0a1b, 0x60034313, 0x406a6802, 0x68026002, 0x428a4032,
    0x6a00d0fb, 0x4861bdf8, 0x6b012900, 0x6b03d016, 0x43a34031, 0x6b036303, 0x021b0a1b, 0x63034313,
    0x32404a5a, 0x6b026017, 0x6302406a, 0x40326b02, 0xd0fb428a, 0x43a16b01, 0xe0726301, 0x40316b03,
    0x432343a3, 0x6b036303, 0x021b0a1b, 0x63034313, 0x406a6b02, 0x6b026302, 0x428a4032, 0x484bd0fb,
    0xe0483040, 0x2900484a, 0xd0116a01, 0x40316a03, 0x620343a3, 0x0a1b6a03, 0x4313021b, 0x63076203,
    0x406a6a02, 0x6a026202, 0x428a4032, 0xe048d0fb, 0x40316a03, 0x432343a3, 0x6a036203, 0x021b0a1b,
    0x62034313, 0x406a6a02, 0x6a026202, 0x428a4032, 0x4837d0fb, 0xe0423040, 0x03ed2501, 0x29004836,
    0x6a014b34, 0x43a1d00b, 0x6a016201, 0x02090a09, 0x62014311, 0x68186307, 0x60184068, 0x43a1e021,
    0x62014321, 0x0a096a01, 0x43110209, 0x68186201, 0x60184068, 0x69004826, 0xe022bdf8, 0x4b21e7ff,
    0x33402501, 0x29004824, 0xd00c6a01, 0x620143a1, 0x0a096a01, 0x43110209, 0x63076201, 0x40686b18,
    0x20006318, 0x43a1e7e8, 0x62014321, 0x0a096a01, 0x43110209, 0x6b186201, 0x63184068, 0x68004814,
    0xf000e7da, 0x4613f9b5, 0x2101460a, 0x2300e6da, 0x4619460a, 0xb5ffe6d6, 0x461cb081, 0x460f4615,
    0x98019e0a, 0xfff3f7ff, 0x43a040b5, 0x43284025, 0x46394602, 0xf7ff9801, 0xb005ffe6, 0x0000bdf0,
    0x40c84800, 0x40c84500, 0x40c84840, 0x40c84880, 0x40c84900, 0x40c84400, 0x40c848c0, 0x460cb570,
    0x1e4d1900, 0x43a81e40, 0x1d001900, 0xf8a8f000, 0xd0071e02, 0x19101ce4, 0x1f0143a8, 0x800b4b13,
    0x804a1a82, 0xb510bd70, 0x1f004601, 0x4b0f8802, 0xd103429a, 0x1a088840, 0xf8c3f000, 0xb510bd10,
    0xd00b2800, 0xf8d0f001, 0x23004a09, 0xf8dff000, 0x43db2300, 0x1a1b2200, 0xd201418a, 0xf958f000,
    0x08800789, 0xf7fe4308, 0xbd10fd2b, 0x00003039, 0x000f4240, 0x68004836, 0x483606c1, 0xd5024448,
    0x04092121, 0x2101e001, 0x69c90549, 0x47706001, 0x44494930, 0x68096809, 0x4b2e4708, 0x681b444b,
    0x6a5b68db, 0x4a2b4718, 0x6812444a, 0x685268d2, 0xb4304710, 0x444c4c27, 0x68e46824, 0x46a468a4,
    0x4760bc30, 0x9c04b538, 0x4c229400, 0x6824444c, 0x696468e4, 0xbd3847a0, 0x4c1eb430, 0x6824444c,
    0x692468e4, 0xbc3046a4, 0x4b1a4760, 0x681b444b, 0x6a9b68db, 0x4b174718, 0x681b444b, 0x6adb68db,
    0x4a144718, 0x6812444a, 0x68d268d2, 0x4a114710, 0x6812444a, 0x69d268d2, 0xb4304710, 0x444c4c0d,
    0x68e46824, 0x46a46a24, 0x4760bc30, 0x68094908, 0xd50106c9, 0xe0004908, 0x47084908, 0x4c05b430,
    0x6824444c, 0x6b6468e4, 0xbc3046a4, 0x00004760, 0x40c84800, 0x0000000c, 0x0021a3af, 0x0020426b,
    0x4606b570, 0xff36f000, 0x46306805, 0x08c4300b, 0x42b400e4, 0x2000d801, 0x6868bd70, 0x2800462a,
    0x6801d018, 0xd31142a1, 0x4621460b, 0x428b3108, 0x6843d308, 0x604b1901, 0x1b1b6803, 0x6051600b,
    0xe0016004, 0x60516841, 0xbd701d00, 0x68404602, 0xd1e62800, 0x46284621, 0xff16f000, 0xd1dc2800,
    0xb570bd70, 0x1f054604, 0xff04f000, 0x2c006800, 0x6844d01a, 0x4620e001, 0x2c006864, 0x42acd001,
    0x6801d3f9, 0x42aa180a, 0x682ad104, 0x18894605, 0xe0006001, 0x68286045, 0x42a11941, 0x6861d105,
    0x68216069, 0x60281840, 0x606cbd70, 0xb5febd70, 0x46104605, 0x4318460c, 0x46aed073, 0x2000468c,
    0x46011aad, 0xd367419c, 0x27004666, 0x463d2401, 0x419d1ab6, 0x4613d302, 0x2421463a, 0x46764665,
    0x0c36042f, 0x0c2d433e, 0x419d1ab6, 0x0c15d304, 0x432b041b, 0x34100412, 0x46764665, 0x0a36062f,
    0x0a2d433e, 0x419d1ab6, 0x0e15d304, 0x432b021b, 0x34080212, 0x46764665, 0x0936072f, 0x092d433e,
    0x419d1ab6, 0x0f15d304, 0x432b011b, 0x1d240112, 0x46764665, 0x08b607af, 0x08ad433e, 0x419d1ab6,
    0x0f95d304, 0x432b009b, 0x1ca40092, 0x46764665, 0x087607ef, 0x086d433e, 0x419d1ab6, 0x1892d31a,
    0x1c64415b, 0x1800e016, 0x46654676, 0x1ab74149, 0x9101419d, 0xd3099000, 0x1ab14660, 0x468e4198,
    0x98004684, 0x25009901, 0x41691c40, 0x085207dd, 0x085b432a, 0xd5e61e64, 0x46634672, 0xbdf0b003,
    0x2000e7ff, 0x46c04601, 0x462a46c0, 0xe7f54623, 0x2000b510, 0xfeb1f000, 0x46c046c0, 0xf0002001,
    0xbd10fe53, 0x2a04b5f8, 0x0783d32c, 0x780bd012, 0x70031c49, 0x1e521c40, 0xd00b0783, 0x1c49780b,
    0x1c407003, 0x07831e52, 0x780bd004, 0x70031c49, 0x1e521c40, 0x0f9b078b, 0x1ac9d005, 0x232000df,
    0xc9081bde, 0xf7fee00a, 0xbdf8fbaf, 0xc908461d, 0x461c40fd, 0x432c40b4, 0x1f12c010, 0xd2f52a04,
    0x1ac908f3, 0xd4f01e52, 0x1c49780b, 0x1c407003, 0xd4ea1e52, 0x1c49780b, 0x1c407003, 0xd4e42a01,
    0x70017809, 0xe001bdf8, 0x1f09c004, 0xd2fb2904, 0xd501078b, 0x1c808002, 0xd00007c9, 0x47707002,
    0xd00b2900, 0xd00207c3, 0x1c407002, 0x29021e49, 0x0783d304, 0x8002d502, 0x1e891c80, 0x2200e7e3,
    0x2200e7ee, 0x2200e7df, 0x428b0903, 0x0a03d32c, 0xd311428b, 0x469c2300, 0x4603e04e, 0xd43c430b,
    0x08432200, 0xd331428b, 0x428b0903, 0x0a03d31c, 0xd301428b, 0xe03f4694, 0x428b09c3, 0x01cbd301,
    0x41521ac0, 0x428b0983, 0x018bd301, 0x41521ac0, 0x428b0943, 0x014bd301, 0x41521ac0, 0x428b0903,
    0x010bd301, 0x41521ac0, 0x428b08c3, 0x00cbd301, 0x41521ac0, 0x428b0883, 0x008bd301, 0x41521ac0,
    0x428b0843, 0x004bd301, 0x41521ac0, 0xd2001a41, 0x41524601, 0x47704610, 0x0fcae05d, 0x4249d000,
    0xd3001003, 0x40534240, 0x469c2200, 0x428b0903, 0x0a03d32d, 0xd312428b, 0x018922fc, 0x0a03ba12,
    0xd30c428b, 0x11920189, 0xd308428b, 0x11920189, 0xd304428b, 0xd03a0189, 0xe0001192, 0x09c30989,
    0xd301428b, 0x1ac001cb, 0x09834152, 0xd301428b, 0x1ac0018b, 0x09434152, 0xd301428b, 0x1ac0014b,
    0x09034152, 0xd301428b, 0x1ac0010b, 0x08c34152, 0xd301428b, 0x1ac000cb, 0x08834152, 0xd301428b,
    0x1ac0008b, 0xd2d94152, 0x428b0843, 0x004bd301, 0x41521ac0, 0xd2001a41, 0x46634601, 0x105b4152,
    0xd3014610, 0x2b004240, 0x4249d500, 0x46634770, 0xd300105b, 0xb5014240, 0x46c02000, 0xbd0246c0,
    0x460cb5f0, 0x419c1a87, 0x1a86460d, 0xd203419d, 0x41a11bc0, 0x416319d2, 0x0d1f0d0d, 0x1bec4ec9,
    0xd05c43ae, 0x0d7f057f, 0x2f002600, 0x052dd051, 0x43a946ac, 0x230102dd, 0x051b0aed, 0x2320431d,
    0xd4241b1b, 0x40e74617, 0x41711838, 0x40e7462f, 0x409d19cc, 0x46311828, 0x0d0c4161, 0x4461d127,
    0xd513409a, 0x1c402300, 0x00524159, 0x0840d10e, 0x22010040, 0x0552004b, 0xd30742d3, 0x07402005,
    0xd50142c1, 0xe00049b0, 0x463049b0, 0x006bbdf0, 0xd0002a00, 0x189a2201, 0x3c20231f, 0xd5011b1b,
    0xe0022300, 0x182840e5, 0x0d0c4171, 0x2401d0d7, 0x19090524, 0x07cd07c4, 0x08490840, 0x43280fe4,
    0x2c004461, 0x2400d0d5, 0x41611c40, 0xd1d0409a, 0x056ae7cd, 0xd1d90d52, 0x07c90fc9, 0x4d9ce7d5,
    0x2800004e, 0x2401d001, 0x2400e000, 0x005f1934, 0xd0012a00, 0xe0002601, 0x19be2600, 0x42ac4f95,
    0x42aed80c, 0x42b4d80a, 0x2501d006, 0x42ec056d, 0x4610d0bc, 0xbdf04619, 0xd0fc4299, 0x46392000,
    0xb5f0bdf0, 0x460cb41f, 0x1a81460f, 0x4625419f, 0x419d1a86, 0x2501d206, 0x406f07ed, 0x41bc1a40,
    0x417b1852, 0x0d1e0d21, 0x46ac1b8d, 0x438d4d7d, 0x0575d07e, 0xd07c0d6d, 0x46ae050d, 0x031b43ac,
    0x4d790b1b, 0x419d4252, 0x46662320, 0xd3141b9b, 0x40f74617, 0x26001838, 0x41749000, 0x462f4660,
    0x19e44107, 0x409d9800, 0x41661828, 0x4671d416, 0x409a1871, 0x2301d525, 0x1893e020, 0x416a462a,
    0x2b000052, 0x1c52d000, 0x3e20231e, 0x2b001b9b, 0x4135dd61, 0x182817ee, 0xd5e84166, 0x41761800,
    0x409a1c5b, 0x1c40d300, 0x18f3054b, 0x40e32415, 0x2b00d310, 0x4671d052, 0x0fd31871, 0xd30218c0,
    0xb0051c49, 0x2301bdf0, 0x429a07db, 0x0840d1f9, 0xe7f60040, 0x05522201, 0xd10118b2, 0xd0532800,
    0x0acc2301, 0x439902db, 0x2a001e89, 0x0c02d108, 0x0c02d003, 0x39100400, 0x4602e002, 0x39202000,
    0x0b552300, 0x0212d105, 0x0b552308, 0x0152d101, 0x0c55230d, 0x0112d101, 0x0cd51d1b, 0x0092d101,
    0x0d151c9b, 0x0052d101, 0x25201c5b, 0xe0011aee, 0xe022e02c, 0x40f54605, 0x40984315, 0x07e11aca,
    0x05131869, 0x2a0018c9, 0x2003dabb, 0x18080740, 0x07c90fc1, 0x4671e00f, 0xe7b21861, 0x084007f1,
    0x10714308, 0x004a4471, 0x2301d00c, 0x429a055b, 0x0fc9d2a7, 0x200007c9, 0x0549e7a3, 0xd1260d49,
    0xe0012000, 0xd1f42800, 0xe79a2100, 0x00664d28, 0xd0012800, 0xe0002101, 0x18712100, 0x2a00005f,
    0x2601d001, 0x2600e000, 0x4f2219be, 0xd81042a9, 0xd80e42ae, 0xd00842b1, 0x056d2501, 0xd00642e9,
    0x02a84619, 0x46104041, 0x429ce77b, 0x4621d001, 0x2000e777, 0xe7744639, 0x460cb510, 0xd505405c,
    0x07e42401, 0xf7ff4063, 0xbd10ff0c, 0xfe78f7ff, 0xb430bd10, 0x405c460c, 0x2401d504, 0x406307e4,
    0xe66dbc30, 0xe6fcbc30, 0x2401b570, 0x406107e4, 0x405d460d, 0xf7ffd402, 0xbd70fe63, 0xf7ff4063,
    0xbd70fef0, 0x000007ff, 0xfff00000, 0x7ff00000, 0xffe00000, 0x7ff80000, 0x4df9b5f0, 0x091e090c,
    0x402e402c, 0xb08b462f, 0xd07d43a7, 0xd0fc43b5, 0xd07a2e00, 0xd0fc2c00, 0x405d460d, 0x43250fed,
    0x25ff1bac, 0x196404ad, 0x02c59408, 0x02c90d44, 0x0d504321, 0x430402dc, 0x07c02001, 0x43044301,
    0x910102d2, 0x0c110c20, 0x468c9500, 0x46230401, 0x910a438b, 0x04094661, 0x438a4de2, 0x0a019109,
    0x1949447d, 0x78093980, 0x434d4605, 0x186d2601, 0x1b7505f6, 0x0cc94369, 0x1c890b64, 0x1864434c,
    0x1b2c01b5, 0x042e0c25, 0x434c43b4, 0x0c21434d, 0x09891869, 0x9d009105, 0xd00107e9, 0x07c92101,
    0x462c9104, 0x08679901, 0x432f07cd, 0x9d05084c, 0x434d0be1, 0x46290c2d, 0x43754666, 0x460d9502,
    0x462e4345, 0x1b7d9d02, 0x950041b4, 0x435d460d, 0x0c2c9401, 0x460d042f, 0x46ae4355, 0x433d0c2d,
    0x94039502, 0x9c044675, 0x1b64042d, 0xd2069404, 0x9d039c02, 0x26001c64, 0x95034175, 0xac019402,
    0x9f00cc70, 0xe0011b7d, 0xe162e13e, 0x040941b4, 0x91079500, 0x99059401, 0x436108a4, 0x460c0c09,
    0x0b654344, 0x460c04e7, 0x43744666, 0x0b6446a6, 0x9503433c, 0x46749402, 0x04e49d04, 0x94041b2c,
    0x9c02d206, 0x1c649d03, 0x41752600, 0x94029503, 0xcc70ac01, 0x1b7d9f00, 0x940141b4, 0x435c460c,
    0x0f659500, 0x460c00e7, 0x46a64354, 0x433c0f64, 0x94029503, 0x9d044674, 0x1b2c00e4, 0xd2069404,
    0x9d039c02, 0x26001c64, 0x95034175, 0xac019402, 0x9f00cc70, 0x41b41b7d, 0x06af09ae, 0x00c99d04,
    0x432f09ad, 0x06a49d04, 0x950406ad, 0x43349d07, 0x91071869, 0x0be19d05, 0x0c2d434d, 0x46664629,
    0x95024375, 0x4345460d, 0x9d02462e, 0x41b41b7d, 0x460d9500, 0x9401435d, 0x042f0c2c, 0x4355460d,
    0x0c2d46ae, 0x9502433d, 0x46759403, 0x042d9c04, 0x94041b64, 0x9c02d206, 0x1c649d03, 0x41752600,
    0x94029503, 0xcc70ac01, 0x1b7d9f00, 0x940141b4, 0x9406058c, 0x0a899c07, 0x95001861, 0x9c019107,
    0x08a49905, 0x0c094361, 0x4345460d, 0x04ef0b68, 0x4664460d, 0x0b6c4365, 0x9f04433c, 0x1b7d04ed,
    0xd20246ac, 0x1c642500, 0x9e004168, 0x1b379d01, 0x46084185, 0x95014358, 0x0f46460c, 0x435400c5,
    0x43280f60, 0x00e44665, 0xd2021b2c, 0x1c402500, 0x9d01416e, 0x41b51a3f, 0x09bd4628, 0x43280680,
    0x09a04684, 0x024e06bf, 0x06a49906, 0x43071876, 0x20009d07, 0x990a4168, 0x43199007, 0x9b092000,
    0x46609000, 0x431319ff, 0x0fe24140, 0x433a0064, 0x460446a4, 0x418c1ad7, 0xd90342a0, 0x90002001,
    0x463a4620, 0x18940fc7, 0x46624140, 0x43220fd2, 0x00644664, 0x9c009404, 0x006446be, 0x46a41ad7,
    0x97024604, 0x4677418c, 0xd1012f00, 0xd90442a0, 0x1c404660, 0x9a024684, 0x0fc44620, 0x189446a6,
    0x9a044140, 0x0fd24605, 0x46644322, 0x1ad30064, 0x4671418d, 0xd1012900, 0xd90242a8, 0x461a4628,
    0x43101c64, 0x28002201, 0x4316d000, 0x20000261, 0x19899b07, 0x415c4604, 0x2c002302, 0x1849db02,
    0x41644613, 0x0ace9d08, 0x195d142d, 0x0567054b, 0x052c0ae1, 0x9908190d, 0x07c9433e, 0x2b004069,
    0x4093d008, 0x2200d306, 0x41511c76, 0xd1012b00, 0x00760876, 0x2d004a1c, 0x4215db43, 0x43aad041,
    0x4630d03f, 0xbdf0b00b, 0x460d1806, 0x4c17414d, 0x41ac4276, 0x1895d31a, 0x415c461c, 0x426d4a13,
    0xd31341a2, 0x05522201, 0x42d4004c, 0x005cd10a, 0xd00b42d4, 0xd502404b, 0x43110292, 0x0049e7e2,
    0xe7df0849, 0x0fc94059, 0xe01807c9, 0x49082000, 0x4805e7d8, 0xd10d4201, 0xd0f74203, 0x0000e7f2,
    0x07ff0000, 0x00000bc4, 0x7ff00000, 0xffe00000, 0x7ff80000, 0x40594811, 0x12c04301, 0x20004041,
    0x2301e7c0, 0x069b9a08, 0xdc05429a, 0x07522203, 0x0fc91889, 0xe7b507c9, 0x07522205, 0x4a071889,
    0x12d24311, 0xe7ad4051, 0x4615b570, 0x4602461c, 0x4628460b, 0xf7ff4621, 0xbd70fddf, 0x7fffffff,
    0x02ca4603, 0xb5100d40, 0x150a4310, 0x2401d004, 0x432007e4, 0xdb052a00, 0x1aa24c0a, 0x40d0d404,
    0xdd002aff, 0xbd102000, 0x2b00004a, 0x2001d001, 0x2000e000, 0x22011810, 0x42d00552, 0x17c8d8f2,
    0xbd1043c0, 0x0000041e, 0x0c03221f, 0x220fd102, 0xd0020400, 0xd0030e03, 0x2000e004, 0x47704601,
    0x3a080200, 0xd1010f03, 0x1f120100, 0xd1010f83, 0x1e920080, 0xdb012800, 0x1e520040, 0x18510512,
    0x12c00542, 0x46101841, 0x0fc14770, 0xd00007c9, 0x22014240, 0x43110792, 0x2101e7d6, 0xe7d30789,
    0x4c8bb5f0, 0x4625090f, 0x43bdb089, 0x091dd07b, 0x43ae4626, 0x460ed0fa, 0x4027405e, 0x433e0ff6,
    0xd0714025, 0x42210124, 0x24ffd0fb, 0x04a419ad, 0x95031b2d, 0x052d2501, 0x0adb02db, 0x432b02c9,
    0x43290ac9, 0x0109011e, 0x0f130f04, 0x43210105, 0x0b880117, 0x43910382, 0x468c092d, 0x431e0ba9,
    0x4395038a, 0x039a0bb3, 0x093f4396, 0x03940bba, 0x960243a7, 0x437e462e, 0x46b69704, 0x19d7194e,
    0x9601460c, 0x97054354, 0x1b37437e, 0x1bbf4676, 0x46b60136, 0x19360bbe, 0x46769600, 0x193704bc,
    0xd3019c00, 0x94001c64, 0x2f000064, 0x2601d001, 0x9e024334, 0x43759f04, 0x437e4666, 0x00769604,
    0x006e1934, 0x94001934, 0x9e019c02, 0x4366191c, 0x46ae1b75, 0x46659406, 0x9c051945, 0x436c9505,
    0x46754359, 0x1a699108, 0x9e049101, 0x43514601, 0x91041ba4, 0x9c001a61, 0x06e19107, 0x0961d005,
    0x43112201, 0xe072e002, 0x0961e06e, 0x27009d08, 0x062d0a2c, 0x9d041869, 0x0a2a417c, 0x1869062d,
    0x41629d01, 0x02ad0dac, 0x4154186d, 0x0d919a07, 0x19550292, 0x9a024161, 0x43584664, 0x9b064354,
    0x435a9a05, 0x1b131a12, 0x019b0e9a, 0x417a185b, 0x05000b01, 0x415118c3, 0x06200a22, 0x415a1944,
    0xd200429a, 0x20011c49, 0x42810240, 0x2315d303, 0x43f62602, 0x2314e002, 0x43f62603, 0x1ac52020,
    0x461740a9, 0x40da40af, 0x4620430a, 0x40d89903, 0x18731409, 0x18510519, 0x43389a03, 0x405107d2,
    0xd00940ac, 0x40942201, 0x2200d306, 0x41511c40, 0xd1012c00, 0x00400840, 0x42934a1e, 0x2201d312,
    0x06929803, 0xdb074290, 0x07402005, 0x481a1809, 0x12c04301, 0xe0044041, 0x07402003, 0x0fc11808,
    0x200007c9, 0xbdf0b009, 0x07f12000, 0x1805e7fa, 0x414c460c, 0x426d4811, 0xd31541a0, 0x461a1894,
    0x480e415a, 0x41904264, 0x2001d30e, 0x0540004a, 0xd3094282, 0x4282005a, 0x4059d306, 0x4a080fc9,
    0x200007c9, 0xe7dd4311, 0x49062000, 0x0000e7da, 0x07ff0000, 0x000007fe, 0x7fffffff, 0xffe00000,
    0x7ff00000, 0x7ff80000, 0x20184901, 0xe7febeab, 0x00020026, 0x44484801, 0xbf004770, 0x00000014,
    0x47704770, 0x47704770, 0x4604b538, 0x2000460a, 0xf0004669, 0x2800f894, 0x4602d005, 0x99004620,
    0xf842f000, 0xbd382001, 0x2001b510, 0xf874f000, 0xb5f8bd10, 0xf876f000, 0x20004605, 0x2010460c,
    0x460646c0, 0x42a019a8, 0x4632d90d, 0x20004669, 0xf875f000, 0xd1010007, 0xf85ef000, 0x42a09800,
    0x4605d000, 0xf7ff19c4, 0x1df6ffc5, 0x60054607, 0x00c008f0, 0x46281946, 0xf808f000, 0xd00442b4,
    0x68384631, 0xf0001ba2, 0xbdf8f80f, 0x60012100, 0x60806041, 0x20104770, 0xb5104770, 0xf824f000,
    0xf81cf000, 0x4770bd10, 0x4603b510, 0xe0016840, 0x68404603, 0xd0012800, 0xd3f94288, 0x18c36818,
    0xd006428b, 0x08d81ccb, 0x1d0000c0, 0x1a521a41, 0x1d084601, 0xf7ff600a, 0xbd10f884, 0x2800b510,
    0xf7ffd001, 0xbd10ff81, 0x2100b510, 0xf000a002, 0x2001f843, 0x0000bd10, 0x41474953, 0x3a545242,
    0x6e624120, 0x616d726f, 0x6574206c, 0x6e696d72, 0x6f697461, 0x0000006e, 0xf000b510, 0xf7fff847,
    0xbd10ffdd, 0x46694b13, 0x03122201, 0x444b1882, 0x601a428a, 0x4611d801, 0x60194770, 0xb5704770,
    0x466d4c0c, 0x444c2601, 0x68230336, 0x189a1bad, 0xd80e42aa, 0x19901aad, 0x19750fee, 0x18aa106d,
    0xd9004290, 0x1dc04610, 0x08c2600b, 0x1ad000d2, 0xbd706022, 0x00000010, 0x4605b570, 0x200a460c,
    0x1c6de000, 0xf83af000, 0xd0062d00, 0x28007828, 0xe002d1f7, 0xf0001c64, 0x2c00f831, 0x7820d002,
    0xd1f72800, 0xf000200a, 0xbd70f829, 0x2801b510, 0x2100d005, 0xf7ffa003, 0x2001ffdf, 0xa109bd10,
    0x0000e7f8, 0x52474953, 0x4d454d54, 0x754f203a, 0x666f2074, 0x61656820, 0x656d2070, 0x79726f6d,
    0x00000000, 0x6548203a, 0x6d207061, 0x726f6d65, 0x6f632079, 0x70757272, 0x00646574, 0x4669b508,
    0x20037008, 0xbd08beab, 0x0c04b5f8, 0x4626b28b, 0x435eb282, 0x46100c0d, 0x43580c37, 0x19800436,
    0x41792100, 0x436e4616, 0x04360c37, 0x41791980, 0x436e4626, 0xbdf81989, 0x4674b430, 0x78251e64,
    0x42ab1c64, 0x461dd200, 0x005b5d63, 0xbc3018e3, 0x00004718, 0x00000003, 0x000000ff, 0x00000001,
    0x00000000, 0x00000016, 0x00000000, 0x00000000, 0x00030502, 0x1c0f1607, 0x00030502, 0x18090f14,
    0x00030502, 0x0d09180f, 0x00030502, 0x18090f14, 0x00030502, 0x110b0918, 0x00030502, 0x0d091810,
    0x00030502, 0x090b1810, 0x00030502, 0x180a0f14, 0x00030502, 0x0a181009, 0x00030502, 0x0d091810,
    0x00030502, 0x0d091810, 0x00030502, 0x0d1a180f, 0x00030502, 0x0d091810, 0x00030502, 0x0d091810,
    0x00030502, 0x14131810, 0x00030502, 0x1c1a1810, 0x00030502, 0x1c1a1810, 0x00030502, 0x14131810,
    0x00030502, 0x14131810, 0x00030502, 0x14131810, 0x00030502, 0x0f0c0911, 0x00030502, 0x0f0c0911,
    0x00030502, 0x0d091810, 0x00030502, 0x0d091810, 0x00030502, 0x180d0f14, 0x00030502, 0x0d091810,
    0x00030502, 0x0d091810, 0x00030502, 0x0d091810, 0x00030502, 0x0d091810, 0x00030502, 0x0d091810,
    0x00030502, 0x0d091810, 0x00030502, 0x0d091810, 0x00030502, 0x0d091810, 0x00030502, 0x0d091810,
    0x00030502, 0x0d091810, 0x00030502, 0x180d0f14, 0x00030502, 0x180d0f14, 0x00030502, 0x0d091810,
    0x00030502, 0x0d091810, 0x00030502, 0x0d091810, 0x00030502, 0x0d091810, 0x00030502, 0x180d0f14,
    0x00030502, 0x180d0f14, 0x00030502, 0x0d091813, 0x00030502, 0x0d091813, 0x00030502, 0x0d091813,
    0x00030502, 0x0d091813, 0x00030502, 0x18130f14, 0x00030502, 0x18130f14, 0x00030502, 0x0d091810,
    0x00030502, 0x0d091810, 0x00030502, 0x0b181a17, 0x00030502, 0x0b181a17, 0x00030502, 0x0b181a17,
    0x00030502, 0x0b181a17, 0x00030502, 0x0b181a17, 0x00030502, 0x0b181a17, 0x00030502, 0x0b181a17,
    0x00030502, 0x07180a0c, 0x00030502, 0x07180a0c, 0x00030502, 0x0d1a1018, 0x00030502, 0x0d1a1018,
    0x00030502, 0x181a0f14, 0x00030502, 0x0d130f1a, 0x00030502, 0x0d18131a, 0x00030502, 0x0d18131a,
    0x00030502, 0x0d18131a, 0x00030502, 0x181a0f14, 0x00030502, 0x1c0f0b09, 0x00030502, 0x1c110c09,
    0x00030502, 0x1c110c09, 0x00030502, 0x1c110a09, 0x00030502, 0x1c110a09, 0x00030502, 0x1c0a0f0c,
    0x00030502, 0x1c0a0f0c, 0x00030502, 0x1c0a0f0c, 0x00030502, 0x1c120f0c, 0x00030502, 0x1812090c,
    0x00030502, 0x1a12010d, 0x7e7f8080, 0x7a7b7c7d, 0x76777879, 0x73747576, 0x70717172, 0x6d6e6e6f,
    0x6a6b6c6c, 0x6868696a, 0x65666667, 0x63636464, 0x60616162, 0x5e5f5f60, 0x5c5d5d5e, 0x5a5b5b5c,
    0x5859595a, 0x56575758, 0x55555556, 0x53535454, 0x51525252, 0x50505051, 0x4e4f4f4f, 0x4d4d4d4e,
    0x4b4c4c4c, 0x4a4a4b4b, 0x4949494a, 0x47484848, 0x46474747, 0x45454646, 0x44444445, 0x43434344,
    0x42424243, 0x41414142, 0x00000000, 0x20230000, 0x00000000, 0x00000000, 0x00000000, 0x00000000
    ],

    # Relative function addresses
    'pc_init': 0x20200093,
    'pc_unInit': 0x2020012b,
    'pc_program_page': 0x20200169,
    'pc_erase_sector': 0x20200145,
    'pc_eraseAll': 0x2020012f,

    'static_base' : 0x20200000 + 0x00000020 + 0x00002a88,
    'begin_stack' : 0x20204000,
    'begin_data' : 0x20240000,
    'page_size' : 0x100,
    'analyzer_supported' : False,
    'analyzer_address' : 0x00000000,
    'page_buffers' : [0x20244000, 0x20244100],   # Enable double buffering
    'min_program_length' : 0x100,
}

FLASH_ALGO = {
    'load_address' : 0x20000000,

    # Flash algorithm as a hex string
    'instructions': [
    0xE00ABE00, 0x062D780D, 0x24084068, 0xD3000040, 0x1E644058, 0x1C49D1FA, 0x2A001E52, 0x4770D1F2,
    0x28001e40, 0x4770d1fc, 0x4605b570, 0x4616460c, 0xcc0fe002, 0x3e10c50f, 0xd2fa2e10, 0xd3022e08,
    0xc503cc03, 0x2e043e08, 0xcc01d307, 0x1f36c501, 0x7821e003, 0x1c647029, 0x1e761c6d, 0xbd70d2f9,
    0x4770ba40, 0x4770ba40, 0x4770ba40, 0x4770ba40, 0x4770ba40, 0x4770ba40, 0x4770bac0, 0x4770bac0,
    0x4770bac0, 0x4770bac0, 0x4770bac0, 0x4770bac0, 0xb5f04770, 0xb0894d7e, 0x6868444d, 0x1c40497e,
    0x487c6068, 0x68002400, 0xd1484288, 0x6941487b, 0xd51f03c9, 0x32804a79, 0xf3bf6054, 0x69418f4f,
    0x041b2301, 0x61414399, 0x8f4ff3bf, 0x4e746810, 0x0c490101, 0x0d8704c0, 0x463805ca, 0x07820c93,
    0x6372431a, 0xd2fa1e40, 0xd2f51e49, 0x8f4ff3bf, 0x8f6ff3bf, 0x2000496b, 0x185201c2, 0x1c406014,
    0xd3f9284f, 0x02402001, 0x49666008, 0x31801c40, 0xf0006008, 0x4964fe14, 0x44792214, 0xf7ff4668,
    0x4668ff83, 0xfa4af000, 0x4478485f, 0x68003818, 0xa8079007, 0xfadef000, 0x485c495d, 0x485d6048,
    0x20216384, 0x0240495c, 0xf0016288, 0x485bfa6b, 0xaa059005, 0x68292001, 0xfa77f001, 0xd1052800,
    0x68292001, 0xfa77f001, 0xd0002800, 0xb0092001, 0x2000bdf0, 0x48464770, 0x4448b510, 0x20016801,
    0xfa97f001, 0xd0002800, 0xbd102001, 0x0709210d, 0x1842b510, 0xd1090500, 0x2301483d, 0x68014448,
    0x2001031b, 0xfa70f001, 0xd0002800, 0xbd102001, 0xb0c3b5f0, 0x4604460e, 0x25004617, 0xa80321fc,
    0xfbe7f001, 0x44784838, 0x684038b4, 0x200d9002, 0x18240700, 0x0e000620, 0x2003d024, 0x18210700,
    0xa802b2e2, 0xfb76f001, 0x21ffb2e0, 0x1a0a3101, 0x9000a902, 0x92011840, 0xf0014639, 0x9800fb6b,
    0x1a22ab02, 0x44484822, 0x20016801, 0xfa29f001, 0xd1382800, 0x21ff9801, 0x31011904, 0x1a08b2e0,
    0x1a3619c7, 0x90000a30, 0x0228e00c, 0x190219c3, 0x44484817, 0x20016801, 0xfa13f001, 0xd1222800,
    0x1c6d9800, 0xd8f042a8, 0x42b5022d, 0xb2f2d01e, 0xa80219e9, 0xfb3ef001, 0xb2f0192c, 0x18222103,
    0x18510709, 0x320122ff, 0xab021a12, 0xf00118c0, 0x4807fb31, 0x44484622, 0xab026801, 0xf0012001,
    0x2800f9f0, 0x2001d002, 0xbdf0b043, 0xe7fb2000, 0x00000004, 0x40c84800, 0x001170a0, 0xe000ed00,
    0xe000ef40, 0x40cc0000, 0x000026ce, 0x00000503, 0x40cc0a00, 0x40c04000, 0x40c0c040, 0xc0000007,
    0x4604b5fe, 0xd0032804, 0xd0012c05, 0xfaf8f001, 0xd03c2c04, 0x21002003, 0xf93ff001, 0x06400101,
    0x0f4e0e40, 0x2c049002, 0x2003d033, 0xf0012130, 0xf002f934, 0x9101f81a, 0x2c049000, 0x2003d02b,
    0xf0012120, 0xf002f92a, 0x4604f810, 0x460d2001, 0xf00240b0, 0x4606f802, 0x4620460f, 0x9b014629,
    0xf0019a00, 0x4604fd91, 0x9802460d, 0xfffdf001, 0x462b4622, 0xfd58f001, 0x4bfe2200, 0xfff8f001,
    0x463b4632, 0xfd80f001, 0xffa2f001, 0x2002bdfe, 0x2002e7c1, 0x2002e7ca, 0xb510e7d2, 0xf0020003,
    0x0406fa33, 0x28261513, 0x48f32f2a, 0x68006801, 0x03c0b2cc, 0x21010f80, 0x40811c40, 0x48ef0049,
    0xfb09f001, 0xe0194360, 0xbd1048ed, 0x318049ea, 0x06446908, 0x0e646808, 0x6a090080, 0x00890880,
    0xf0010889, 0x49e5faf8, 0x43481900, 0x48e5e006, 0x2004bd10, 0x2005e000, 0xff82f7ff, 0xd1f72800,
    0xfa7ef001, 0x2802b510, 0x2803d001, 0x223fd116, 0x409a00cb, 0xd0022802, 0xd10f2803, 0x48d6e005,
    0x6b013040, 0x401148d8, 0x48d3e003, 0x48d56b01, 0x40d94011, 0x3a0c460a, 0xd9012a17, 0xfa60f001,
    0xd0042900, 0xfac7f001, 0x43482112, 0x2000bd10, 0x4601bd10, 0x48c9b510, 0xf002000b, 0x111ff9d5,
    0x161b1619, 0x1f1d1d16, 0x2a25231f, 0x502e2e2c, 0x36343230, 0x403b3939, 0x49494747, 0x00504b4b,
    0x680048c2, 0xd5010740, 0xbd1048c1, 0xbd1048c1, 0xbd1048c1, 0xbd1048c1, 0xe0002000, 0xf7ff2002,
    0xe02aff7c, 0xe0002100, 0x20022101, 0xffaaf7ff, 0x2102e023, 0x2103e7f9, 0x2003e7f7, 0x2100e7ef,
    0x2101e004, 0x2102e002, 0x2103e000, 0xe7ed2003, 0xe7e42001, 0xf7ff2001, 0x0840ff60, 0x2001e00d,
    0xff5bf7ff, 0xf0012105, 0xe006fa76, 0xe7d62004, 0xe7d42005, 0xf0002000, 0x2800f804, 0xf001d1cb,
    0xb530f9ff, 0x49a6b0ff, 0x4604b0a0, 0x00d2224f, 0x46684479, 0xfda8f7ff, 0x01e148a2, 0x68081809,
    0x05406809, 0xb2cd0f40, 0x466a00e1, 0x5c081889, 0xf7ff1c6d, 0x4629ff8e, 0xfa4df001, 0xd1012800,
    0xf9def001, 0xb020b07f, 0xb5f8bd30, 0x68404604, 0x39684601, 0xd9012968, 0xf9d2f001, 0x68294d86,
    0x4e924f91, 0xd50a0489, 0x43c96829, 0x4208b2c9, 0x7820d105, 0x03c06829, 0x03c04388, 0xf000d029,
    0x6829fdbb, 0x22014628, 0x43910752, 0x0f92044a, 0x2203d004, 0x43910352, 0x60014331, 0x79254a84,
    0x78224011, 0x07924604, 0x43150bd2, 0x4981430d, 0x6005430d, 0x8f4ff3bf, 0x8f6ff3bf, 0x201e4970,
    0xf833f001, 0x00806820, 0x433dd5fc, 0x602543b5, 0x6828bdf8, 0xd4020440, 0x43386828, 0x68286028,
    0xd5f50040, 0x43b06828, 0xbdf86028, 0x68084962, 0x43904a71, 0x4310054a, 0x47706008, 0x4d5eb5f8,
    0x35404604, 0x4f6d6828, 0x02004e68, 0x6838d515, 0x42886861, 0x6938d111, 0x428868a1, 0x6828d10d,
    0xd4030480, 0x14716828, 0x60284308, 0x00406828, 0x6828d502, 0x602843b0, 0xf000bdf8, 0x6b28fd5d,
    0x4308495f, 0x68286328, 0x4208495e, 0x43c9d003, 0x43304008, 0x485c6028, 0x68606238, 0x68a06038,
    0x48526138, 0x60283008, 0x201e4c45, 0xf0004621, 0x4856ffdc, 0x46216028, 0xf00020fa, 0x4954ffd6,
    0x68286029, 0xd5fc0080, 0x03402001, 0x60294301, 0x602943b1, 0x494a6b28, 0x400843c9, 0xbdf86328,
    0x30404835, 0x4a466b01, 0x63014311, 0x06026801, 0x60014311, 0x4a476801, 0x60014011, 0xb5f84770,
    0x4611468c, 0x2917390c, 0x2802d803, 0x2803d003, 0xf001d006, 0x4839f915, 0x38104c38, 0xe0033c30,
    0x4c254825, 0x34203030, 0x46636801, 0x254000db, 0x4029409d, 0x68079100, 0x409e2680, 0x60074337,
    0x213f6807, 0x438f4099, 0x06916007, 0x68020e89, 0x43114099, 0x68216001, 0x23024662, 0x40594093,
    0x68016021, 0x600143b1, 0x9a006801, 0x42914029, 0xbdf8d0fa, 0x4c10b5f8, 0x69204605, 0x02804e1b,
    0x7828d504, 0x43886921, 0xd05a0740, 0xfcd4f000, 0x491b6b20, 0x63204308, 0x49147828, 0x0f400740,
    0x18453110, 0x4f066125, 0x4639201e, 0xff5df000, 0xe0334819, 0x4176e360, 0x40c84200, 0x016e3600,
    0x3b9aca00, 0x1c9c3800, 0x1f78a400, 0x40c840c0, 0x00f42400, 0x003d0900, 0x02dc6c00, 0x17d78400,
    0x0000232c, 0x40cc0000, 0x00004000, 0x40000000, 0xfffe7f00, 0x40002000, 0x20006000, 0x40c84280,
    0x80808080, 0x00802000, 0x0fffffff, 0x40800808, 0x40800008, 0xff7fdff7, 0x00200800, 0x61254305,
    0x201e4639, 0xff21f000, 0x02c02001, 0x61254385, 0x00806920, 0x48fed5fc, 0x61254305, 0x612543b5,
    0x6920bdf8, 0xd4040480, 0x21016920, 0x43080349, 0x69206120, 0xd5f30040, 0x43b06920, 0xbdf86120,
    0xb5104770, 0xd0022900, 0x03522201, 0x2200e000, 0x039b2301, 0x290018d2, 0x2104d001, 0x2108e000,
    0xfea7f000, 0xb510bd10, 0xd0012900, 0xe0002104, 0x22012108, 0xf0000412, 0xbd10fe9c, 0x2900b510,
    0x2104d001, 0x2108e000, 0x05922201, 0xfe91f000, 0xb510bd10, 0xd0012900, 0xe0002104, 0x22012108,
    0xf0000352, 0xbd10fe86, 0x460db570, 0x21014604, 0xffeff7ff, 0x462849d7, 0xfec7f000, 0x46202100,
    0xffe7f7ff, 0xb510bd70, 0xd0012900, 0xe0002104, 0x22012108, 0xf00003d2, 0xbd10fe6c, 0xb081b5ff,
    0x461d9f0a, 0x4604460e, 0xd0022801, 0xf7ff2100, 0x463affa8, 0x46202130, 0xfe5bf000, 0x46202120,
    0xf0009a03, 0x2000fe56, 0x90004601, 0x4632237f, 0xf0004620, 0x2c01fe56, 0x2019d008, 0x90002307,
    0x462a065b, 0x46202100, 0xfe4bf000, 0x46202101, 0xffa4f7ff, 0x206449b7, 0xfe87f000, 0x46202101,
    0xff7ff7ff, 0xbdf0b005, 0x49b32201, 0x28000392, 0xd0026808, 0x60084310, 0x43904770, 0x2201e7fb,
    0x035249ad, 0x68082800, 0x4310d002, 0x47706008, 0xe7fb4390, 0x4604b5f8, 0xf0002600, 0x4da6fbcd,
    0x49a66828, 0xd1004381, 0x21012601, 0xf7ff2002, 0x2e00ff6a, 0x2000d011, 0xffe1f7ff, 0xf7ff2001,
    0x2100ffd3, 0xf7ff2002, 0x2100ff8e, 0xf7ff2002, 0x2100ff48, 0xf7ff2002, 0x2001ff61, 0xffcff7ff,
    0x900068a0, 0x78217863, 0x68622002, 0xff86f7ff, 0x200221e1, 0xff68f7ff, 0x00806828, 0x2101d5fc,
    0xf7ff2002, 0x2000ff70, 0xffaef7ff, 0x20022100, 0xff39f7ff, 0xb510bdf8, 0x0004460a, 0x2051d005,
    0x42820100, 0x2a14d801, 0xf000d801, 0x4984ff89, 0x210060a1, 0xd3034282, 0x20367061, 0xe0427020,
    0x00c02051, 0xd3074282, 0x21187061, 0xf0004610, 0x7020ffe2, 0xe035487b, 0x304520ff, 0xd3084282,
    0x70602001, 0x4610210c, 0xffd5f000, 0x48767020, 0x2aa2e028, 0x2002d308, 0x21067060, 0xf0004610,
    0x7020ffca, 0xe01d4871, 0xd3082a51, 0x70602003, 0x46104601, 0xffbff000, 0x486d7020, 0x2a28e012,
    0x2004d308, 0x00507060, 0xf0002103, 0x7020ffb4, 0xe0074868, 0x70602005, 0x21030090, 0xffabf000,
    0x48657020, 0x20004341, 0xbd106061, 0x2200b50e, 0x92004601, 0x46689201, 0xf7ff9202, 0x2800ff9c,
    0x2001d001, 0x4668bd0e, 0xff54f7ff, 0xbd0e2000, 0x2000b510, 0xff43f7ff, 0xf7ff2001, 0x2100ff35,
    0xf7ff2002, 0x2100fef0, 0xf7ff2002, 0x2100feaa, 0xf7ff2002, 0xbd10fec3, 0x22014947, 0x03923140,
    0x69082800, 0x4310d002, 0x47706108, 0xe7fb4390, 0x22014941, 0x03523140, 0x69082800, 0x4310d002,
    0x47706108, 0xe7fb4390, 0x460ab510, 0xd0050004, 0x01802027, 0xd8014282, 0xd2012a9c, 0xfef0f000,
    0x42822103, 0x7021d302, 0xe02120d0, 0x01402027, 0xd3024282, 0x210c7021, 0x2027e006, 0x42820100,
    0x2000d306, 0x21067020, 0xf0004610, 0xe00fff44, 0x303920ff, 0xd3034282, 0x70202001, 0xe7f32103,
    0x2a9c2002, 0xd9027020, 0x21030050, 0x2068e7ed, 0x20006060, 0xb51cbd10, 0x46684601, 0xffc4f7ff,
    0xd0012800, 0xbd1c2001, 0xf7ff4668, 0x2000fcde, 0xb5f8bd1c, 0x26004604, 0xfaaef000, 0x35404d16,
    0x49166928, 0xd1004381, 0x21012601, 0xf7ff2003, 0x2e00fe4a, 0x2000d011, 0xff9af7ff, 0xf7ff2001,
    0x2100ff8b, 0xf7ff2003, 0x2100fe6e, 0xf7ff2003, 0x2100fe28, 0xf7ff2003, 0x2001fe41, 0xff88f7ff,
    0x900068a0, 0x78217863, 0x68622003, 0x0000e016, 0x00002008, 0x3b9aca00, 0x40c84300, 0x20002000,
    0x0fffffff, 0x00aaaaaa, 0x01555555, 0x02aaaaaa, 0x05555555, 0x0aaaaaaa, 0x15555554, 0xfe4ef7ff,
    0x200321e1, 0xfe30f7ff, 0x00806928, 0x2101d5fc, 0xf7ff2003, 0x2000fe38, 0xff4ef7ff, 0x20032100,
    0xfe01f7ff, 0xb50ebdf8, 0x46012200, 0x92019200, 0x92024668, 0xfebff7ff, 0xd0012800, 0xbd0e2001,
    0xf7ff4668, 0x2000ff96, 0xb510bd0e, 0xf7ff2000, 0x2001ff3f, 0xff30f7ff, 0x20032100, 0xfe13f7ff,
    0x20032100, 0xfdcdf7ff, 0x20032100, 0xfde6f7ff, 0x2201bd10, 0x039249df, 0x68082800, 0x4310d002,
    0x47706008, 0xe7fb4390, 0x49da2201, 0x28000652, 0xd0026808, 0x60084310, 0x43904770, 0x2201e7fb,
    0x069249d4, 0x68082800, 0x4310d002, 0x47706008, 0xe7fb4390, 0x49cf2201, 0x28000352, 0xd0026808,
    0x60084310, 0x43904770, 0x49cae7fb, 0x00806808, 0x4770d5fc, 0x4604b538, 0xf9fef000, 0x46082101,
    0xfda1f7ff, 0xf7ff2001, 0x48c3ffe5, 0x4ac39000, 0x23002129, 0xf7ff2001, 0x21e1fdd1, 0xf7ff2001,
    0xf7fffdb3, 0x2101ffe2, 0xf7ff4608, 0x2000fdbc, 0xffaff7ff, 0xf7ff7820, 0x7860ffb7, 0xffbff7ff,
    0x20012100, 0xfd7ff7ff, 0xb510bd38, 0xf7ff2000, 0x2000ffc1, 0xffa8f7ff, 0xf7ff2000, 0x2001ffb0,
    0xff97f7ff, 0x20012100, 0xfd9df7ff, 0x20012100, 0xfd57f7ff, 0x20012100, 0xfd70f7ff, 0x48a8bd10,
    0x06c96a01, 0x2114d404, 0x6a016201, 0xd5fc0049, 0xb5104770, 0xd3012802, 0xfdb2f000, 0x790b780a,
    0xd0012a00, 0x06122201, 0x884a4313, 0x0412794c, 0x43220124, 0x798a4313, 0x43130212, 0x031279ca,
    0x01c24313, 0x18104897, 0x7a4b6003, 0x011b7a0a, 0x7a8b431a, 0x021b7acc, 0x43230324, 0x7b0b431a,
    0x431a041b, 0x051b7b4b, 0x7b8b431a, 0x061b7bc9, 0x0709431a, 0x6102430a, 0xb570bd10, 0x6900488a,
    0x0fca0581, 0x0fc905c1, 0x2a00b2c0, 0x4c87d011, 0x62222220, 0x06000289, 0x63214301, 0x00d56a20,
    0x62204328, 0x20014982, 0xfbfff000, 0x43a86a20, 0xbd706220, 0x3040487a, 0x08496901, 0x61010049,
    0x22016a01, 0x62014311, 0x01cb4770, 0x49792201, 0x18590612, 0x1252600a, 0x10d2604a, 0x680a608a,
    0x02520a52, 0x680a600a, 0x0dc005c0, 0x041b2303, 0x43024318, 0x4870600a, 0x6c086088, 0xd0fc2800,
    0x00806c08, 0x20014770, 0xb5f84770, 0x2400460f, 0xf0002073, 0x2001f8b1, 0x07c04e68, 0x4d6763b0,
    0x35804867, 0x46396268, 0xf0004866, 0x2900fd94, 0x2107d114, 0x0589b280, 0xd01a2816, 0x280ddc08,
    0x280fd021, 0x2810d00c, 0x2814d00d, 0xe00dd106, 0xd0112819, 0xd012281e, 0xd01328f0, 0xbdf82000,
    0x05a42401, 0x2401e00f, 0xe00c05e4, 0x05a42403, 0x2401e009, 0xe0060624, 0x05a42405, 0x2403e003,
    0xe00005e4, 0x6a28460c, 0x43204388, 0x20016228, 0x62a80400, 0x62682040, 0x63b00600, 0x60702000,
    0x28006a28, 0x2001dafc, 0x4944bdf8, 0x31802040, 0x48426288, 0x04026b01, 0x63014311, 0x20014770,
    0xb5f84770, 0x2400460f, 0xf0002073, 0x2001f855, 0x07c04e3d, 0x4d3c63b0, 0x35804839, 0x46396268,
    0xf0004838, 0x2900fd38, 0x2107d114, 0x0589b280, 0xd01a2816, 0x280ddc08, 0x280fd021, 0x2810d00c,
    0x2814d00d, 0xe00dd106, 0xd0112819, 0xd012281e, 0xd01328f0, 0xbdf82000, 0x05a42401, 0x2401e00f,
    0xe00c05e4, 0x05a42403, 0x2401e009, 0xe0060624, 0x05a42405, 0x2403e003, 0xe00005e4, 0x6a28460c,
    0x43204388, 0x20016228, 0x62a80400, 0x62682040, 0x63b00600, 0x60702000, 0x28006a28, 0x2001dafc,
    0x4919bdf8, 0x31802040, 0x48176288, 0x22016b01, 0x43110792, 0x47706301, 0x48140142, 0x18102101,
    0x400a6802, 0xd0044291, 0xf3bf6001, 0xf3bf8f4f, 0x47708f6f, 0x40c842c0, 0x0fffffff, 0x0aaaaaaa,
    0x40c84000, 0x40cc4000, 0x40c84100, 0x40c84900, 0x3b9aca00, 0x40150000, 0x01008000, 0x40434000,
    0x00201000, 0x1c9c3800, 0x40438000, 0x40cc6000, 0x6a114abc, 0x061b2301, 0x06004399, 0x62104308,
    0x4ab84770, 0x23076a11, 0x0740029b, 0x0cc04399, 0x62104308, 0x4ab34770, 0x04c06911, 0x03490b49,
    0x43080cc0, 0x47706110, 0x69114aae, 0x09090109, 0x43080700, 0x47706110, 0x69114aaa, 0x061b230f,
    0x43990700, 0x43080900, 0x47706110, 0x2800b500, 0x2001d004, 0xfff0f7ff, 0xe00a2001, 0xd0042900,
    0xf7ff2005, 0x2005ffe9, 0x2000e003, 0xffe4f7ff, 0xf7ff2000, 0xbd00ffd9, 0x2100b500, 0xf7ff2001,
    0x20ffffe5, 0xf7ff30ef, 0x2001ffc6, 0xffb9f7ff, 0xf7ff2001, 0xbd00ffad, 0x2100b510, 0xf0004608,
    0x22fffa4c, 0x42903206, 0x2100d01b, 0xf0004608, 0x2100fa40, 0xf0004608, 0x20fffa40, 0x30014989,
    0x60083140, 0x6a014888, 0x04122201, 0x62014391, 0x20014986, 0xfa79f000, 0x6b014882, 0x43112210,
    0xbd106301, 0xb5102200, 0x46104611, 0xfa21f000, 0x229dbd10, 0x0092b510, 0x1a802114, 0xfc3bf000,
    0x3a404a78, 0x231f6b11, 0x06c0051b, 0x09c04399, 0x63104308, 0xb570bd10, 0x3c404c72, 0x08406920,
    0x61200040, 0x267d4d71, 0x462900f6, 0xf0004630, 0x6922fa4c, 0x04c02001, 0x61224302, 0x46304629,
    0xfa43f000, 0x21206920, 0x61204308, 0x46304629, 0xfa3bf000, 0x21046920, 0x61204308, 0xb570bd70,
    0x3c404c60, 0x21056920, 0x61204388, 0x267d4d5f, 0x462900f6, 0xf0004630, 0x6922fa28, 0x43822020,
    0x46296122, 0xf0004630, 0x6920fa20, 0x04c92101, 0x61204388, 0xb570bd70, 0x3c404c52, 0x21016b20,
    0x63204308, 0x267d4d51, 0x462900f6, 0xf0004630, 0x6b22fa0c, 0x04402001, 0x63224302, 0x46304629,
    0xfa03f000, 0x21016b20, 0x43080489, 0x46296320, 0xf0004630, 0x6b20f9fa, 0x43882104, 0xbd706320,
    0x4c40b570, 0x6b203c40, 0x43082105, 0x4d3f6320, 0x00f6267d, 0x46304629, 0xf9e7f000, 0x20016b22,
    0x43820480, 0x46296322, 0xf0004630, 0x6b20f9de, 0x04492101, 0x63204388, 0xb570bd70, 0x3c404c31,
    0x08406920, 0x61200040, 0x21016b20, 0x63204308, 0x267d4d2e, 0x462900f6, 0xf0004630, 0x6922f9c6,
    0x04c02001, 0x61224302, 0x21016b20, 0x43080449, 0x46296320, 0xf0004630, 0x6920f9b8, 0x43082120,
    0x6b206120, 0x43080349, 0x46296320, 0xf0004630, 0x6921f9ac, 0x43012004, 0x6b216121, 0x63214381,
    0xb570bd70, 0x3c404c17, 0x20046921, 0x61214381, 0x43016b21, 0x69206321, 0x00400840, 0x6b206120,
    0x43082101, 0x4d116320, 0x00f6267d, 0x46304629, 0xf98bf000, 0x20016b22, 0x43820480, 0x69206322,
    0x43882120, 0x46296120, 0xf0004630, 0x6b20f97e, 0x04492101, 0x63204388, 0x00896920, 0x61204388,
    0x0000bd70, 0x40c84540, 0x40c84800, 0x3b9aca00, 0x25ffb5f8, 0x461f3501, 0x11e6022c, 0xf0010003,
    0x0507fa53, 0xa7784c23, 0x00f2f4d1, 0x29004897, 0x6a014b97, 0x43a1d00a, 0x6a016201, 0x02090a09,
    0x62014311, 0x68186307, 0xe0cc4060, 0x432143a1, 0x6a016201, 0x02090a09, 0x62014311, 0x40606818,
    0x488a6018, 0xe0f03040, 0x2900488a, 0xd0116901, 0x40316903, 0x610343a3, 0x0a1b6903, 0x4313021b,
    0x62076103, 0x406a6902, 0x69026102, 0x428a4032, 0xe0ccd0fb, 0x40316903, 0x432343a3, 0x69036103,
    0x021b0a1b, 0x61034313, 0x406a6902, 0x69026102, 0x428a4032, 0x6b00d0fb, 0x4877bdf8, 0x68012900,
    0x6803d014, 0x43a34031, 0x68036003, 0x021b0a1b, 0x60034313, 0x68026107, 0x6002406a, 0x40326802,
    0xd0fb428a, 0x43a16801, 0xe0a06001, 0x40316803, 0x432343a3, 0x68036003, 0x021b0a1b, 0x60034313,
    0x406a6802, 0x68026002, 0x428a4032, 0x6a00d0fb, 0x4861bdf8, 0x6b012900, 0x6b03d016, 0x43a34031,
    0x6b036303, 0x021b0a1b, 0x63034313, 0x32404a5a, 0x6b026017, 0x6302406a, 0x40326b02, 0xd0fb428a,
    0x43a16b01, 0xe0726301, 0x40316b03, 0x432343a3, 0x6b036303, 0x021b0a1b, 0x63034313, 0x406a6b02,
    0x6b026302, 0x428a4032, 0x484bd0fb, 0xe0483040, 0x2900484a, 0xd0116a01, 0x40316a03, 0x620343a3,
    0x0a1b6a03, 0x4313021b, 0x63076203, 0x406a6a02, 0x6a026202, 0x428a4032, 0xe048d0fb, 0x40316a03,
    0x432343a3, 0x6a036203, 0x021b0a1b, 0x62034313, 0x406a6a02, 0x6a026202, 0x428a4032, 0x4837d0fb,
    0xe0423040, 0x03ed2501, 0x29004836, 0x6a014b34, 0x43a1d00b, 0x6a016201, 0x02090a09, 0x62014311,
    0x68186307, 0x60184068, 0x43a1e021, 0x62014321, 0x0a096a01, 0x43110209, 0x68186201, 0x60184068,
    0x69004826, 0xe022bdf8, 0x4b21e7ff, 0x33402501, 0x29004824, 0xd00c6a01, 0x620143a1, 0x0a096a01,
    0x43110209, 0x63076201, 0x40686b18, 0x20006318, 0x43a1e7e8, 0x62014321, 0x0a096a01, 0x43110209,
    0x6b186201, 0x63184068, 0x68004814, 0xf000e7da, 0x4613f9b7, 0x2101460a, 0x2300e6da, 0x4619460a,
    0xb5ffe6d6, 0x461cb081, 0x460f4615, 0x98019e0a, 0xfff3f7ff, 0x43a040b5, 0x43284025, 0x46394602,
    0xf7ff9801, 0xb005ffe6, 0x0000bdf0, 0x40c84800, 0x40c84500, 0x40c84840, 0x40c84880, 0x40c84900,
    0x40c84400, 0x40c848c0, 0x460cb570, 0x1e4d1900, 0x43a81e40, 0x1d001900, 0xf8aaf000, 0xd0071e02,
    0x19101ce4, 0x1f0143a8, 0x800b4b14, 0x804a1a82, 0xb510bd70, 0x1f004601, 0x4b108802, 0xd103429a,
    0x1a088840, 0xf8c5f000, 0xb510bd10, 0xd00b2800, 0xf8d2f001, 0x23004a0a, 0xf8e1f000, 0x43db2300,
    0x1a1b2200, 0xd201418a, 0xf95af000, 0x23002203, 0xf8d5f000, 0xf7fe1800, 0xbd10fd03, 0x00003039,
    0x000f4240, 0x68004836, 0x483606c1, 0xd5024448, 0x04092121, 0x2101e001, 0x69c90549, 0x47706001,
    0x44494930, 0x68096809, 0x4b2e4708, 0x681b444b, 0x6a5b68db, 0x4a2b4718, 0x6812444a, 0x685268d2,
    0xb4304710, 0x444c4c27, 0x68e46824, 0x46a468a4, 0x4760bc30, 0x9c04b538, 0x4c229400, 0x6824444c,
    0x696468e4, 0xbd3847a0, 0x4c1eb430, 0x6824444c, 0x692468e4, 0xbc3046a4, 0x4b1a4760, 0x681b444b,
    0x6a9b68db, 0x4b174718, 0x681b444b, 0x6adb68db, 0x4a144718, 0x6812444a, 0x68d268d2, 0x4a114710,
    0x6812444a, 0x69d268d2, 0xb4304710, 0x444c4c0d, 0x68e46824, 0x46a46a24, 0x4760bc30, 0x68094908,
    0xd50106c9, 0xe0004908, 0x47084908, 0x4c05b430, 0x6824444c, 0x6b6468e4, 0xbc3046a4, 0x00004760,
    0x40c84800, 0x0000000c, 0x0021a3af, 0x0020426b, 0x4606b570, 0xff36f000, 0x46306805, 0x08c4300b,
    0x42b400e4, 0x2000d801, 0x6868bd70, 0x2800462a, 0x6801d018, 0xd31142a1, 0x4621460b, 0x428b3108,
    0x6843d308, 0x604b1901, 0x1b1b6803, 0x6051600b, 0xe0016004, 0x60516841, 0xbd701d00, 0x68404602,
    0xd1e62800, 0x46284621, 0xff16f000, 0xd1dc2800, 0xb570bd70, 0x1f054604, 0xff04f000, 0x2c006800,
    0x6844d01a, 0x4620e001, 0x2c006864, 0x42acd001, 0x6801d3f9, 0x42aa180a, 0x682ad104, 0x18894605,
    0xe0006001, 0x68286045, 0x42a11941, 0x6861d105, 0x68216069, 0x60281840, 0x606cbd70, 0xb5febd70,
    0x46104605, 0x4318460c, 0x46aed073, 0x2000468c, 0x46011aad, 0xd367419c, 0x27004666, 0x463d2401,
    0x419d1ab6, 0x4613d302, 0x2421463a, 0x46764665, 0x0c36042f, 0x0c2d433e, 0x419d1ab6, 0x0c15d304,
    0x432b041b, 0x34100412, 0x46764665, 0x0a36062f, 0x0a2d433e, 0x419d1ab6, 0x0e15d304, 0x432b021b,
    0x34080212, 0x46764665, 0x0936072f, 0x092d433e, 0x419d1ab6, 0x0f15d304, 0x432b011b, 0x1d240112,
    0x46764665, 0x08b607af, 0x08ad433e, 0x419d1ab6, 0x0f95d304, 0x432b009b, 0x1ca40092, 0x46764665,
    0x087607ef, 0x086d433e, 0x419d1ab6, 0x1892d31a, 0x1c64415b, 0x1800e016, 0x46654676, 0x1ab74149,
    0x9101419d, 0xd3099000, 0x1ab14660, 0x468e4198, 0x98004684, 0x25009901, 0x41691c40, 0x085207dd,
    0x085b432a, 0xd5e61e64, 0x46634672, 0xbdf0b003, 0x2000e7ff, 0x46c04601, 0x462a46c0, 0xe7f54623,
    0x2000b510, 0xfeb1f000, 0x46c046c0, 0xf0002001, 0xbd10fe53, 0x2a04b5f8, 0x0783d32c, 0x780bd012,
    0x70031c49, 0x1e521c40, 0xd00b0783, 0x1c49780b, 0x1c407003, 0x07831e52, 0x780bd004, 0x70031c49,
    0x1e521c40, 0x0f9b078b, 0x1ac9d005, 0x232000df, 0xc9081bde, 0xf7fee00a, 0xbdf8fb87, 0xc908461d,
    0x461c40fd, 0x432c40b4, 0x1f12c010, 0xd2f52a04, 0x1ac908f3, 0xd4f01e52, 0x1c49780b, 0x1c407003,
    0xd4ea1e52, 0x1c49780b, 0x1c407003, 0xd4e42a01, 0x70017809, 0xe001bdf8, 0x1f09c004, 0xd2fb2904,
    0xd501078b, 0x1c808002, 0xd00007c9, 0x47707002, 0xd00b2900, 0xd00207c3, 0x1c407002, 0x29021e49,
    0x0783d304, 0x8002d502, 0x1e891c80, 0x2200e7e3, 0x2200e7ee, 0x2200e7df, 0x428b0903, 0x0a03d32c,
    0xd311428b, 0x469c2300, 0x4603e04e, 0xd43c430b, 0x08432200, 0xd331428b, 0x428b0903, 0x0a03d31c,
    0xd301428b, 0xe03f4694, 0x428b09c3, 0x01cbd301, 0x41521ac0, 0x428b0983, 0x018bd301, 0x41521ac0,
    0x428b0943, 0x014bd301, 0x41521ac0, 0x428b0903, 0x010bd301, 0x41521ac0, 0x428b08c3, 0x00cbd301,
    0x41521ac0, 0x428b0883, 0x008bd301, 0x41521ac0, 0x428b0843, 0x004bd301, 0x41521ac0, 0xd2001a41,
    0x41524601, 0x47704610, 0x0fcae05d, 0x4249d000, 0xd3001003, 0x40534240, 0x469c2200, 0x428b0903,
    0x0a03d32d, 0xd312428b, 0x018922fc, 0x0a03ba12, 0xd30c428b, 0x11920189, 0xd308428b, 0x11920189,
    0xd304428b, 0xd03a0189, 0xe0001192, 0x09c30989, 0xd301428b, 0x1ac001cb, 0x09834152, 0xd301428b,
    0x1ac0018b, 0x09434152, 0xd301428b, 0x1ac0014b, 0x09034152, 0xd301428b, 0x1ac0010b, 0x08c34152,
    0xd301428b, 0x1ac000cb, 0x08834152, 0xd301428b, 0x1ac0008b, 0xd2d94152, 0x428b0843, 0x004bd301,
    0x41521ac0, 0xd2001a41, 0x46634601, 0x105b4152, 0xd3014610, 0x2b004240, 0x4249d500, 0x46634770,
    0xd300105b, 0xb5014240, 0x46c02000, 0xbd0246c0, 0x460cb5f0, 0x419c1a87, 0x1a86460d, 0xd203419d,
    0x41a11bc0, 0x416319d2, 0x0d1f0d0d, 0x1bec4ec9, 0xd05c43ae, 0x0d7f057f, 0x2f002600, 0x052dd051,
    0x43a946ac, 0x230102dd, 0x051b0aed, 0x2320431d, 0xd4241b1b, 0x40e74617, 0x41711838, 0x40e7462f,
    0x409d19cc, 0x46311828, 0x0d0c4161, 0x4461d127, 0xd513409a, 0x1c402300, 0x00524159, 0x0840d10e,
    0x22010040, 0x0552004b, 0xd30742d3, 0x07402005, 0xd50142c1, 0xe00049b0, 0x463049b0, 0x006bbdf0,
    0xd0002a00, 0x189a2201, 0x3c20231f, 0xd5011b1b, 0xe0022300, 0x182840e5, 0x0d0c4171, 0x2401d0d7,
    0x19090524, 0x07cd07c4, 0x08490840, 0x43280fe4, 0x2c004461, 0x2400d0d5, 0x41611c40, 0xd1d0409a,
    0x056ae7cd, 0xd1d90d52, 0x07c90fc9, 0x4d9ce7d5, 0x2800004e, 0x2401d001, 0x2400e000, 0x005f1934,
    0xd0012a00, 0xe0002601, 0x19be2600, 0x42ac4f95, 0x42aed80c, 0x42b4d80a, 0x2501d006, 0x42ec056d,
    0x4610d0bc, 0xbdf04619, 0xd0fc4299, 0x46392000, 0xb5f0bdf0, 0x460cb41f, 0x1a81460f, 0x4625419f,
    0x419d1a86, 0x2501d206, 0x406f07ed, 0x41bc1a40, 0x417b1852, 0x0d1e0d21, 0x46ac1b8d, 0x438d4d7d,
    0x0575d07e, 0xd07c0d6d, 0x46ae050d, 0x031b43ac, 0x4d790b1b, 0x419d4252, 0x46662320, 0xd3141b9b,
    0x40f74617, 0x26001838, 0x41749000, 0x462f4660, 0x19e44107, 0x409d9800, 0x41661828, 0x4671d416,
    0x409a1871, 0x2301d525, 0x1893e020, 0x416a462a, 0x2b000052, 0x1c52d000, 0x3e20231e, 0x2b001b9b,
    0x4135dd61, 0x182817ee, 0xd5e84166, 0x41761800, 0x409a1c5b, 0x1c40d300, 0x18f3054b, 0x40e32415,
    0x2b00d310, 0x4671d052, 0x0fd31871, 0xd30218c0, 0xb0051c49, 0x2301bdf0, 0x429a07db, 0x0840d1f9,
    0xe7f60040, 0x05522201, 0xd10118b2, 0xd0532800, 0x0acc2301, 0x439902db, 0x2a001e89, 0x0c02d108,
    0x0c02d003, 0x39100400, 0x4602e002, 0x39202000, 0x0b552300, 0x0212d105, 0x0b552308, 0x0152d101,
    0x0c55230d, 0x0112d101, 0x0cd51d1b, 0x0092d101, 0x0d151c9b, 0x0052d101, 0x25201c5b, 0xe0011aee,
    0xe022e02c, 0x40f54605, 0x40984315, 0x07e11aca, 0x05131869, 0x2a0018c9, 0x2003dabb, 0x18080740,
    0x07c90fc1, 0x4671e00f, 0xe7b21861, 0x084007f1, 0x10714308, 0x004a4471, 0x2301d00c, 0x429a055b,
    0x0fc9d2a7, 0x200007c9, 0x0549e7a3, 0xd1260d49, 0xe0012000, 0xd1f42800, 0xe79a2100, 0x00664d28,
    0xd0012800, 0xe0002101, 0x18712100, 0x2a00005f, 0x2601d001, 0x2600e000, 0x4f2219be, 0xd81042a9,
    0xd80e42ae, 0xd00842b1, 0x056d2501, 0xd00642e9, 0x02a84619, 0x46104041, 0x429ce77b, 0x4621d001,
    0x2000e777, 0xe7744639, 0x460cb510, 0xd505405c, 0x07e42401, 0xf7ff4063, 0xbd10ff0c, 0xfe78f7ff,
    0xb430bd10, 0x405c460c, 0x2401d504, 0x406307e4, 0xe66dbc30, 0xe6fcbc30, 0x2401b570, 0x406107e4,
    0x405d460d, 0xf7ffd402, 0xbd70fe63, 0xf7ff4063, 0xbd70fef0, 0x000007ff, 0xfff00000, 0x7ff00000,
    0xffe00000, 0x7ff80000, 0x4df9b5f0, 0x091e090c, 0x402e402c, 0xb08b462f, 0xd07d43a7, 0xd0fc43b5,
    0xd07a2e00, 0xd0fc2c00, 0x405d460d, 0x43250fed, 0x25ff1bac, 0x196404ad, 0x02c59408, 0x02c90d44,
    0x0d504321, 0x430402dc, 0x07c02001, 0x43044301, 0x910102d2, 0x0c110c20, 0x468c9500, 0x46230401,
    0x910a438b, 0x04094661, 0x438a4de2, 0x0a019109, 0x1949447d, 0x78093980, 0x434d4605, 0x186d2601,
    0x1b7505f6, 0x0cc94369, 0x1c890b64, 0x1864434c, 0x1b2c01b5, 0x042e0c25, 0x434c43b4, 0x0c21434d,
    0x09891869, 0x9d009105, 0xd00107e9, 0x07c92101, 0x462c9104, 0x08679901, 0x432f07cd, 0x9d05084c,
    0x434d0be1, 0x46290c2d, 0x43754666, 0x460d9502, 0x462e4345, 0x1b7d9d02, 0x950041b4, 0x435d460d,
    0x0c2c9401, 0x460d042f, 0x46ae4355, 0x433d0c2d, 0x94039502, 0x9c044675, 0x1b64042d, 0xd2069404,
    0x9d039c02, 0x26001c64, 0x95034175, 0xac019402, 0x9f00cc70, 0xe0011b7d, 0xe162e13e, 0x040941b4,
    0x91079500, 0x99059401, 0x436108a4, 0x460c0c09, 0x0b654344, 0x460c04e7, 0x43744666, 0x0b6446a6,
    0x9503433c, 0x46749402, 0x04e49d04, 0x94041b2c, 0x9c02d206, 0x1c649d03, 0x41752600, 0x94029503,
    0xcc70ac01, 0x1b7d9f00, 0x940141b4, 0x435c460c, 0x0f659500, 0x460c00e7, 0x46a64354, 0x433c0f64,
    0x94029503, 0x9d044674, 0x1b2c00e4, 0xd2069404, 0x9d039c02, 0x26001c64, 0x95034175, 0xac019402,
    0x9f00cc70, 0x41b41b7d, 0x06af09ae, 0x00c99d04, 0x432f09ad, 0x06a49d04, 0x950406ad, 0x43349d07,
    0x91071869, 0x0be19d05, 0x0c2d434d, 0x46664629, 0x95024375, 0x4345460d, 0x9d02462e, 0x41b41b7d,
    0x460d9500, 0x9401435d, 0x042f0c2c, 0x4355460d, 0x0c2d46ae, 0x9502433d, 0x46759403, 0x042d9c04,
    0x94041b64, 0x9c02d206, 0x1c649d03, 0x41752600, 0x94029503, 0xcc70ac01, 0x1b7d9f00, 0x940141b4,
    0x9406058c, 0x0a899c07, 0x95001861, 0x9c019107, 0x08a49905, 0x0c094361, 0x4345460d, 0x04ef0b68,
    0x4664460d, 0x0b6c4365, 0x9f04433c, 0x1b7d04ed, 0xd20246ac, 0x1c642500, 0x9e004168, 0x1b379d01,
    0x46084185, 0x95014358, 0x0f46460c, 0x435400c5, 0x43280f60, 0x00e44665, 0xd2021b2c, 0x1c402500,
    0x9d01416e, 0x41b51a3f, 0x09bd4628, 0x43280680, 0x09a04684, 0x024e06bf, 0x06a49906, 0x43071876,
    0x20009d07, 0x990a4168, 0x43199007, 0x9b092000, 0x46609000, 0x431319ff, 0x0fe24140, 0x433a0064,
    0x460446a4, 0x418c1ad7, 0xd90342a0, 0x90002001, 0x463a4620, 0x18940fc7, 0x46624140, 0x43220fd2,
    0x00644664, 0x9c009404, 0x006446be, 0x46a41ad7, 0x97024604, 0x4677418c, 0xd1012f00, 0xd90442a0,
    0x1c404660, 0x9a024684, 0x0fc44620, 0x189446a6, 0x9a044140, 0x0fd24605, 0x46644322, 0x1ad30064,
    0x4671418d, 0xd1012900, 0xd90242a8, 0x461a4628, 0x43101c64, 0x28002201, 0x4316d000, 0x20000261,
    0x19899b07, 0x415c4604, 0x2c002302, 0x1849db02, 0x41644613, 0x0ace9d08, 0x195d142d, 0x0567054b,
    0x052c0ae1, 0x9908190d, 0x07c9433e, 0x2b004069, 0x4093d008, 0x2200d306, 0x41511c76, 0xd1012b00,
    0x00760876, 0x2d004a1c, 0x4215db43, 0x43aad041, 0x4630d03f, 0xbdf0b00b, 0x460d1806, 0x4c17414d,
    0x41ac4276, 0x1895d31a, 0x415c461c, 0x426d4a13, 0xd31341a2, 0x05522201, 0x42d4004c, 0x005cd10a,
    0xd00b42d4, 0xd502404b, 0x43110292, 0x0049e7e2, 0xe7df0849, 0x0fc94059, 0xe01807c9, 0x49082000,
    0x4805e7d8, 0xd10d4201, 0xd0f74203, 0x0000e7f2, 0x07ff0000, 0x00000bc4, 0x7ff00000, 0xffe00000,
    0x7ff80000, 0x40594811, 0x12c04301, 0x20004041, 0x2301e7c0, 0x069b9a08, 0xdc05429a, 0x07522203,
    0x0fc91889, 0xe7b507c9, 0x07522205, 0x4a071889, 0x12d24311, 0xe7ad4051, 0x4615b570, 0x4602461c,
    0x4628460b, 0xf7ff4621, 0xbd70fddf, 0x7fffffff, 0x02ca4603, 0xb5100d40, 0x150a4310, 0x2401d004,
    0x432007e4, 0xdb052a00, 0x1aa24c0a, 0x40d0d404, 0xdd002aff, 0xbd102000, 0x2b00004a, 0x2001d001,
    0x2000e000, 0x22011810, 0x42d00552, 0x17c8d8f2, 0xbd1043c0, 0x0000041e, 0x0c03221f, 0x220fd102,
    0xd0020400, 0xd0030e03, 0x2000e004, 0x47704601, 0x3a080200, 0xd1010f03, 0x1f120100, 0xd1010f83,
    0x1e920080, 0xdb012800, 0x1e520040, 0x18510512, 0x12c00542, 0x46101841, 0x0fc14770, 0xd00007c9,
    0x22014240, 0x43110792, 0x2101e7d6, 0xe7d30789, 0x4c8bb5f0, 0x4625090f, 0x43bdb089, 0x091dd07b,
    0x43ae4626, 0x460ed0fa, 0x4027405e, 0x433e0ff6, 0xd0714025, 0x42210124, 0x24ffd0fb, 0x04a419ad,
    0x95031b2d, 0x052d2501, 0x0adb02db, 0x432b02c9, 0x43290ac9, 0x0109011e, 0x0f130f04, 0x43210105,
    0x0b880117, 0x43910382, 0x468c092d, 0x431e0ba9, 0x4395038a, 0x039a0bb3, 0x093f4396, 0x03940bba,
    0x960243a7, 0x437e462e, 0x46b69704, 0x19d7194e, 0x9601460c, 0x97054354, 0x1b37437e, 0x1bbf4676,
    0x46b60136, 0x19360bbe, 0x46769600, 0x193704bc, 0xd3019c00, 0x94001c64, 0x2f000064, 0x2601d001,
    0x9e024334, 0x43759f04, 0x437e4666, 0x00769604, 0x006e1934, 0x94001934, 0x9e019c02, 0x4366191c,
    0x46ae1b75, 0x46659406, 0x9c051945, 0x436c9505, 0x46754359, 0x1a699108, 0x9e049101, 0x43514601,
    0x91041ba4, 0x9c001a61, 0x06e19107, 0x0961d005, 0x43112201, 0xe072e002, 0x0961e06e, 0x27009d08,
    0x062d0a2c, 0x9d041869, 0x0a2a417c, 0x1869062d, 0x41629d01, 0x02ad0dac, 0x4154186d, 0x0d919a07,
    0x19550292, 0x9a024161, 0x43584664, 0x9b064354, 0x435a9a05, 0x1b131a12, 0x019b0e9a, 0x417a185b,
    0x05000b01, 0x415118c3, 0x06200a22, 0x415a1944, 0xd200429a, 0x20011c49, 0x42810240, 0x2315d303,
    0x43f62602, 0x2314e002, 0x43f62603, 0x1ac52020, 0x461740a9, 0x40da40af, 0x4620430a, 0x40d89903,
    0x18731409, 0x18510519, 0x43389a03, 0x405107d2, 0xd00940ac, 0x40942201, 0x2200d306, 0x41511c40,
    0xd1012c00, 0x00400840, 0x42934a1e, 0x2201d312, 0x06929803, 0xdb074290, 0x07402005, 0x481a1809,
    0x12c04301, 0xe0044041, 0x07402003, 0x0fc11808, 0x200007c9, 0xbdf0b009, 0x07f12000, 0x1805e7fa,
    0x414c460c, 0x426d4811, 0xd31541a0, 0x461a1894, 0x480e415a, 0x41904264, 0x2001d30e, 0x0540004a,
    0xd3094282, 0x4282005a, 0x4059d306, 0x4a080fc9, 0x200007c9, 0xe7dd4311, 0x49062000, 0x0000e7da,
    0x07ff0000, 0x000007fe, 0x7fffffff, 0xffe00000, 0x7ff00000, 0x7ff80000, 0x20184901, 0xe7febeab,
    0x00020026, 0x44484801, 0xbf004770, 0x00000014, 0x47704770, 0x47704770, 0x4604b538, 0x2000460a,
    0xf0004669, 0x2800f894, 0x4602d005, 0x99004620, 0xf842f000, 0xbd382001, 0x2001b510, 0xf874f000,
    0xb5f8bd10, 0xf876f000, 0x20004605, 0x2010460c, 0x460646c0, 0x42a019a8, 0x4632d90d, 0x20004669,
    0xf875f000, 0xd1010007, 0xf85ef000, 0x42a09800, 0x4605d000, 0xf7ff19c4, 0x1df6ffc5, 0x60054607,
    0x00c008f0, 0x46281946, 0xf808f000, 0xd00442b4, 0x68384631, 0xf0001ba2, 0xbdf8f80f, 0x60012100,
    0x60806041, 0x20104770, 0xb5104770, 0xf824f000, 0xf81cf000, 0x4770bd10, 0x4603b510, 0xe0016840,
    0x68404603, 0xd0012800, 0xd3f94288, 0x18c36818, 0xd006428b, 0x08d81ccb, 0x1d0000c0, 0x1a521a41,
    0x1d084601, 0xf7ff600a, 0xbd10f884, 0x2800b510, 0xf7ffd001, 0xbd10ff81, 0x2100b510, 0xf000a002,
    0x2001f843, 0x0000bd10, 0x41474953, 0x3a545242, 0x6e624120, 0x616d726f, 0x6574206c, 0x6e696d72,
    0x6f697461, 0x0000006e, 0xf000b510, 0xf7fff847, 0xbd10ffdd, 0x46694b13, 0x03122201, 0x444b1882,
    0x601a428a, 0x4611d801, 0x60194770, 0xb5704770, 0x466d4c0c, 0x444c2601, 0x68230336, 0x189a1bad,
    0xd80e42aa, 0x19901aad, 0x19750fee, 0x18aa106d, 0xd9004290, 0x1dc04610, 0x08c2600b, 0x1ad000d2,
    0xbd706022, 0x00000010, 0x4605b570, 0x200a460c, 0x1c6de000, 0xf83af000, 0xd0062d00, 0x28007828,
    0xe002d1f7, 0xf0001c64, 0x2c00f831, 0x7820d002, 0xd1f72800, 0xf000200a, 0xbd70f829, 0x2801b510,
    0x2100d005, 0xf7ffa003, 0x2001ffdf, 0xa109bd10, 0x0000e7f8, 0x52474953, 0x4d454d54, 0x754f203a,
    0x666f2074, 0x61656820, 0x656d2070, 0x79726f6d, 0x00000000, 0x6548203a, 0x6d207061, 0x726f6d65,
    0x6f632079, 0x70757272, 0x00646574, 0x4669b508, 0x20037008, 0xbd08beab, 0x0c04b5f8, 0x4626b28b,
    0x435eb282, 0x46100c0d, 0x43580c37, 0x19800436, 0x41792100, 0x436e4616, 0x04360c37, 0x41791980,
    0x436e4626, 0xbdf81989, 0x4674b430, 0x78251e64, 0x42ab1c64, 0x461dd200, 0x005b5d63, 0xbc3018e3,
    0x00004718, 0x00000003, 0x000000ff, 0x00000001, 0x00000000, 0x00000016, 0x00000000, 0x00000000,
    0x00030502, 0x1c0f1607, 0x00030502, 0x18090f14, 0x00030502, 0x0d09180f, 0x00030502, 0x18090f14,
    0x00030502, 0x110b0918, 0x00030502, 0x0d091810, 0x00030502, 0x090b1810, 0x00030502, 0x180a0f14,
    0x00030502, 0x0a181009, 0x00030502, 0x0d091810, 0x00030502, 0x0d091810, 0x00030502, 0x0d1a180f,
    0x00030502, 0x0d091810, 0x00030502, 0x0d091810, 0x00030502, 0x14131810, 0x00030502, 0x1c1a1810,
    0x00030502, 0x1c1a1810, 0x00030502, 0x14131810, 0x00030502, 0x14131810, 0x00030502, 0x14131810,
    0x00030502, 0x0f0c0911, 0x00030502, 0x0f0c0911, 0x00030502, 0x0d091810, 0x00030502, 0x0d091810,
    0x00030502, 0x180d0f14, 0x00030502, 0x0d091810, 0x00030502, 0x0d091810, 0x00030502, 0x0d091810,
    0x00030502, 0x0d091810, 0x00030502, 0x0d091810, 0x00030502, 0x0d091810, 0x00030502, 0x0d091810,
    0x00030502, 0x0d091810, 0x00030502, 0x0d091810, 0x00030502, 0x0d091810, 0x00030502, 0x180d0f14,
    0x00030502, 0x180d0f14, 0x00030502, 0x0d091810, 0x00030502, 0x0d091810, 0x00030502, 0x0d091810,
    0x00030502, 0x0d091810, 0x00030502, 0x180d0f14, 0x00030502, 0x180d0f14, 0x00030502, 0x0d091813,
    0x00030502, 0x0d091813, 0x00030502, 0x0d091813, 0x00030502, 0x0d091813, 0x00030502, 0x18130f14,
    0x00030502, 0x18130f14, 0x00030502, 0x0d091810, 0x00030502, 0x0d091810, 0x00030502, 0x0b181a17,
    0x00030502, 0x0b181a17, 0x00030502, 0x0b181a17, 0x00030502, 0x0b181a17, 0x00030502, 0x0b181a17,
    0x00030502, 0x0b181a17, 0x00030502, 0x0b181a17, 0x00030502, 0x07180a0c, 0x00030502, 0x07180a0c,
    0x00030502, 0x0d1a1018, 0x00030502, 0x0d1a1018, 0x00030502, 0x181a0f14, 0x00030502, 0x0d130f1a,
    0x00030502, 0x0d18131a, 0x00030502, 0x0d18131a, 0x00030502, 0x0d18131a, 0x00030502, 0x181a0f14,
    0x00030502, 0x1c0f0b09, 0x00030502, 0x1c110c09, 0x00030502, 0x1c110c09, 0x00030502, 0x1c110a09,
    0x00030502, 0x1c110a09, 0x00030502, 0x1c0a0f0c, 0x00030502, 0x1c0a0f0c, 0x00030502, 0x1c0a0f0c,
    0x00030502, 0x1c120f0c, 0x00030502, 0x1812090c, 0x00030502, 0x1a12010d, 0x7e7f8080, 0x7a7b7c7d,
    0x76777879, 0x73747576, 0x70717172, 0x6d6e6e6f, 0x6a6b6c6c, 0x6868696a, 0x65666667, 0x63636464,
    0x60616162, 0x5e5f5f60, 0x5c5d5d5e, 0x5a5b5b5c, 0x5859595a, 0x56575758, 0x55555556, 0x53535454,
    0x51525252, 0x50505051, 0x4e4f4f4f, 0x4d4d4d4e, 0x4b4c4c4c, 0x4a4a4b4b, 0x4949494a, 0x47484848,
    0x46474747, 0x45454646, 0x44444445, 0x43434344, 0x42424243, 0x41414142, 0x00000000, 0x20230000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000
    ],

    # Relative function addresses
    'pc_init': 0x20000093,
    'pc_unInit': 0x20000173,
    'pc_program_page': 0x200001b1,
    'pc_erase_sector': 0x2000018d,
    'pc_eraseAll': 0x20000177,

    'static_base' : 0x20000000 + 0x00000020 + 0x00002ad8,
    'begin_stack' : 0x20004000,
    'begin_data' : 0x20240000,
    'page_size' : 0x100,
    'analyzer_supported' : False,
    'analyzer_address' : 0x00000000,
    'page_buffers' : [0x20244000, 0x20244100],   # Enable double buffering
    'min_program_length' : 0x100,
}

class MIMXRT1176xxxxx_CM7(CoreSightTarget):

    VENDOR = "NXP"

    # Note: itcm, dtcm, and ocram share a single 512 KB block of RAM that can be configurably
    # divided between those regions (this is called FlexRAM). Thus, the memory map regions for
    # each of these RAMs allocate the maximum possible of 512 KB, but that is the maximum and
    # will not actually be available in all regions simultaneously.
    memoryMap = MemoryMap(
        RamRegion(name="itcm",              start=0x00000000, length=0x80000), # 512 KB
        RomRegion(name="romcp",             start=0x00200000, length=0x18000), # 96 KB
        RamRegion(name="semc_alias",        start=0x10000000, length=0x10000000, alias='semc'), # 256 MB
        RamRegion(name="dtcm",              start=0x20000000, length=0x80000), # 512 KB
        RamRegion(name="ocram",             start=0x20200000, length=0x40000), # 512 KB
        RamRegion(name="ocram1",            start=0x20240000, length=0x80000), # 512 KB
        RamRegion(name="ocram2",            start=0x202C0000, length=0x80000), # 512 KB
        RamRegion(name="aips",              start=0x40000000, length=0x10000000),
        FlashRegion(name="flexspi",         start=0x30000000, length=0x1000000, blocksize=0x1000,
            is_boot_memory=True, algo=FLASH_ALGO, page_size=0x100),
        RamRegion(name="semc",              start=0x80000000, end=0xdfffffff, is_external=True)
        )

    def __init__(self, link):
        self.AP_NUM = 0
        super(MIMXRT1176xxxxx_CM7, self).__init__(link, self.memoryMap)
        self._svd_location = SVDFile.from_builtin("MIMXRT1176_cm7.xml")

    def create_init_sequence(self):
        seq = super(MIMXRT1176xxxxx_CM7, self).create_init_sequence()
        seq.wrap_task('discovery',
            lambda seq: seq.replace_task('find_aps', self.find_aps)\
                    .replace_task('create_cores', self.create_cores)
            )
        return seq

    def prepare_cm4_trap_code(self, ap):
        start = 0x20250000
        ap.write32(start, start + 0x20)
        ap.write32(start + 4, 0x23F041)
        ap.write32(IOMUX_LPSR_GPR0, start & 0xFFFF)
        ap.write32(IOMUX_LPSR_GPR1, (start & 0xFFFF0000) >> 16)

    def release_cm4(self, ap):
        # release CM4
        ap.write32(SRC_SCR, 1)
        LOG.debug("CM4 is released")

    def prepare_cm7_trap_code(self, ap):
        start = 0x2001FF00
        ap.write32(start, start + 0x20)
        ap.write32(start + 4, 0x223105)
        ap.write32(IOMUX_LPSR_GPR26, start >> 7)

    def update_sbmr(self, ap):
        sbmr = ap.read32(SRC_SBMR)
        sbmr |= (0xF << 10);
        ap.write32(SRC_SBMR, sbmr)

    def find_aps(self):
        if self.dp.valid_aps is not None:
            return
        self.dp.read_ap(0xFC)
        self.dp.valid_aps = [0,1,2]
        ap = AccessPort.create(self.dp, APv1Address(0))

        self.prepare_cm7_trap_code(ap)
        self.prepare_cm4_trap_code(ap)
        self.release_cm4(ap)
        self.update_sbmr(ap)

    def create_cores(self):
        core0 = CortexM(self.session, self.aps[0], self.memory_map, 0)
        core0.default_reset_type = self.ResetType.SW_CORE

        self.aps[0].core = core0

        core0.init()

        self.add_core(core0)

class MIMXRT1176xxxxx_CM4(CoreSightTarget):

    VENDOR = "NXP"

    # Note: itcm, dtcm, and ocram share a single 512 KB block of RAM that can be configurably
    # divided between those regions (this is called FlexRAM). Thus, the memory map regions for
    # each of these RAMs allocate the maximum possible of 512 KB, but that is the maximum and
    # will not actually be available in all regions simultaneously.
    memoryMap = MemoryMap(
        RamRegion(name="codetcm",           start=0x1ffe0000, length=0x20000), # 512 KB
        RomRegion(name="romcp",             start=0x00200000, length=0x18000), # 96 KB
        RamRegion(name="systemtcm",         start=0x20000000, length=0x20000), # 512 KB
        RamRegion(name="ocram",             start=0x20200000, length=0x40000), # 512 KB
        RamRegion(name="ocram1",            start=0x20240000, length=0x80000), # 512 KB
        RamRegion(name="ocram2",            start=0x202C0000, length=0x80000), # 512 KB
        RamRegion(name="aips",              start=0x40000000, length=0x10000000),
        FlashRegion(name="flexspi",         start=0x30000000, length=0x1000000, blocksize=0x1000,
            is_boot_memory=True, algo=FLASH_ALGO, page_size=0x100),
        FlashRegion(name="flexspi_alias",   start=0x08000000, length=0x1000000, blocksize=0x1000,
            is_boot_memory=False, algo=FLASH_ALGO_CM4, page_size=0x100),
        RamRegion(name="semc",              start=0x80000000, end=0xdfffffff, is_external=True)
        )


    def __init__(self, link):
        self.AP_NUM = 1
        super(MIMXRT1176xxxxx_CM4, self).__init__(link, self.memoryMap)
        self._svd_location = SVDFile.from_builtin("MIMXRT1176_cm4.xml")

    def create_init_sequence(self):
        seq = super(MIMXRT1176xxxxx_CM4, self).create_init_sequence()
        seq.wrap_task('discovery',
            lambda seq: seq.replace_task('find_aps', self.find_aps)\
                    .replace_task('create_cores', self.create_cores)
            )
        return seq

    def prepare_cm4_trap_code(self, ap):
        start = 0x20250000
        ap.write32(start, start + 0x20)
        ap.write32(start + 4, 0x23F041)
        ap.write32(IOMUX_LPSR_GPR0, start & 0xFFFF)
        ap.write32(IOMUX_LPSR_GPR1, (start & 0xFFFF0000) >> 16)

    def release_cm4(self, ap):
        # release CM4
        ap.write32(SRC_SCR, 1)
        LOG.debug("CM4 is released")

    def prepare_cm7_trap_code(self, ap):
        start = 0x2001FF00
        ap.write32(start, start + 0x20)
        ap.write32(start + 4, 0x223105)
        ap.write32(IOMUX_LPSR_GPR26, start >> 7)

    def update_sbmr(self, ap):
        sbmr = ap.read32(SRC_SBMR)
        sbmr |= (0xF << 10);
        ap.write32(SRC_SBMR, sbmr)

    def find_aps(self):
        if self.dp.valid_aps is not None:
            return
        self.dp.read_ap(0xFC)
        self.dp.valid_aps = [0,1,2]
        ap = AccessPort.create(self.dp, APv1Address(0))

        self.prepare_cm7_trap_code(ap)
        self.prepare_cm4_trap_code(ap)
        self.release_cm4(ap)
        self.update_sbmr(ap)

    def create_cores(self):
        core = CortexM(self.session, self.aps[1], self.memory_map, 1)
        core.default_reset_type = self.ResetType.SW_CORE

        self.aps[0].core = core

        core.init()

        self.add_core(core)

