/*
 * Copyright (C) 2021 Apple Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1.  Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 * 2.  Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE INC. AND ITS CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL APPLE INC. OR ITS CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#include "WebProcess.h"

#if PLATFORM(COCOA)
#include "AccessibilityPreferences.h"
#endif
#include "ArgumentCoders.h"
#include "CacheModel.h"
#include "DataReference.h"
#include "Decoder.h"
#if ENABLE(GAMEPAD)
#include "GamepadData.h"
#endif
#include "HandleMessage.h"
#include "IdentifierTypes.h"
#if (ENABLE(MEDIA_STREAM) && ENABLE(SANDBOX_EXTENSIONS))
#include "MediaDeviceSandboxExtensions.h"
#endif
#include "RemoteWorkerInitializationData.h"
#include "SandboxExtension.h"
#if PLATFORM(COCOA)
#include "SharedBufferCopy.h"
#endif
#include "TextCheckerState.h"
#include "UserData.h"
#if PLATFORM(GTK) || PLATFORM(WPE)
#include "UserMessage.h"
#endif
#include "WebCoreArgumentCoders.h"
#include "WebPageCreationParameters.h"
#include "WebPageProxyIdentifier.h"
#include "WebPreferencesStore.h"
#include "WebProcessCreationParameters.h"
#include "WebProcessDataStoreParameters.h"
#include "WebProcessMessages.h"
#include "WebsiteData.h"
#include "WebsiteDataType.h"
#include <WebCore/BackForwardItemIdentifier.h>
#if HAVE(MEDIA_ACCESSIBILITY_FRAMEWORK)
#include <WebCore/CaptionUserPreferences.h>
#endif
#if HAVE(CVDISPLAYLINK)
#include <WebCore/DisplayUpdate.h>
#endif
#if ENABLE(GAMEPAD)
#include <WebCore/GamepadProviderClient.h>
#endif
#if ENABLE(MEDIA_STREAM)
#include <WebCore/MockMediaDevice.h>
#endif
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
#include <WebCore/NetworkStorageSession.h>
#endif
#include <WebCore/PageIdentifier.h>
#include <WebCore/PrewarmInformation.h>
#if ENABLE(SERVICE_WORKER)
#include <WebCore/ProcessQualified.h>
#endif
#include <WebCore/RegistrableDomain.h>
#if PLATFORM(COCOA)
#include <WebCore/ScreenProperties.h>
#endif
#if ENABLE(SERVICE_WORKER)
#include <WebCore/ScriptExecutionContextIdentifier.h>
#endif
#include <WebCore/SecurityOriginData.h>
#if (PLATFORM(COCOA) && ENABLE(CFPREFS_DIRECT_MODE)) || ENABLE(SERVICE_WORKER)
#include <optional>
#endif
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
#include <wtf/HashMap.h>
#endif
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
#include <wtf/HashSet.h>
#endif
#if ENABLE(SERVICE_WORKER)
#include <wtf/ObjectIdentifier.h>
#endif
#include <wtf/OptionSet.h>
#include <wtf/Vector.h>
#include <wtf/WallTime.h>
#include <wtf/text/WTFString.h>

namespace Messages {

namespace WebProcess {

void FetchWebsiteData::callReply(IPC::Decoder& decoder, CompletionHandler<void(WebKit::WebsiteData&&)>&& completionHandler)
{
    std::optional<WebKit::WebsiteData> websiteData;
    decoder >> websiteData;
    if (!websiteData) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*websiteData));
}

void FetchWebsiteData::cancelReply(CompletionHandler<void(WebKit::WebsiteData&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<WebKit::WebsiteData>::create());
}

void FetchWebsiteData::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const WebKit::WebsiteData& websiteData)
{
    encoder.get() << websiteData;
    connection.sendSyncReply(WTFMove(encoder));
}

void DeleteWebsiteData::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void DeleteWebsiteData::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void DeleteWebsiteData::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

void DeleteWebsiteDataForOrigins::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void DeleteWebsiteDataForOrigins::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void DeleteWebsiteDataForOrigins::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

void PrepareToSuspend::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void PrepareToSuspend::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void PrepareToSuspend::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

void EstablishSharedWorkerContextConnectionToNetworkProcess::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void EstablishSharedWorkerContextConnectionToNetworkProcess::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void EstablishSharedWorkerContextConnectionToNetworkProcess::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

#if ENABLE(SERVICE_WORKER)

void EstablishServiceWorkerContextConnectionToNetworkProcess::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void EstablishServiceWorkerContextConnectionToNetworkProcess::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void EstablishServiceWorkerContextConnectionToNetworkProcess::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

void GetActivePagesOriginsForTesting::callReply(IPC::Decoder& decoder, CompletionHandler<void(Vector<String>&&)>&& completionHandler)
{
    std::optional<Vector<String>> activeOrigins;
    decoder >> activeOrigins;
    if (!activeOrigins) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*activeOrigins));
}

void GetActivePagesOriginsForTesting::cancelReply(CompletionHandler<void(Vector<String>&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<Vector<String>>::create());
}

void GetActivePagesOriginsForTesting::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const Vector<String>& activeOrigins)
{
    encoder.get() << activeOrigins;
    connection.sendSyncReply(WTFMove(encoder));
}

void IsJITEnabled::callReply(IPC::Decoder& decoder, CompletionHandler<void(bool&&)>&& completionHandler)
{
    std::optional<bool> enabled;
    decoder >> enabled;
    if (!enabled) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*enabled));
}

void IsJITEnabled::cancelReply(CompletionHandler<void(bool&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<bool>::create());
}

void IsJITEnabled::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, bool enabled)
{
    encoder.get() << enabled;
    connection.sendSyncReply(WTFMove(encoder));
}

void ClearCachedPage::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void ClearCachedPage::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void ClearCachedPage::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)

void SeedResourceLoadStatisticsForTesting::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SeedResourceLoadStatisticsForTesting::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SeedResourceLoadStatisticsForTesting::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)

void SetThirdPartyCookieBlockingMode::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetThirdPartyCookieBlockingMode::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetThirdPartyCookieBlockingMode::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)

void SetDomainsWithCrossPageStorageAccess::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetDomainsWithCrossPageStorageAccess::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetDomainsWithCrossPageStorageAccess::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)

void SendResourceLoadStatisticsDataImmediately::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SendResourceLoadStatisticsDataImmediately::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SendResourceLoadStatisticsDataImmediately::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

} // namespace WebProcess

} // namespace Messages

namespace WebKit {

void WebProcess::didReceiveWebProcessMessage(IPC::Connection& connection, IPC::Decoder& decoder)
{
    if (decoder.messageName() == Messages::WebProcess::InitializeWebProcess::name())
        return IPC::handleMessage<Messages::WebProcess::InitializeWebProcess>(connection, decoder, this, &WebProcess::initializeWebProcess);
    if (decoder.messageName() == Messages::WebProcess::SetWebsiteDataStoreParameters::name())
        return IPC::handleMessage<Messages::WebProcess::SetWebsiteDataStoreParameters>(connection, decoder, this, &WebProcess::setWebsiteDataStoreParameters);
    if (decoder.messageName() == Messages::WebProcess::CreateWebPage::name())
        return IPC::handleMessage<Messages::WebProcess::CreateWebPage>(connection, decoder, this, &WebProcess::createWebPage);
    if (decoder.messageName() == Messages::WebProcess::PrewarmWithDomainInformation::name())
        return IPC::handleMessage<Messages::WebProcess::PrewarmWithDomainInformation>(connection, decoder, this, &WebProcess::prewarmWithDomainInformation);
    if (decoder.messageName() == Messages::WebProcess::SetCacheModel::name())
        return IPC::handleMessage<Messages::WebProcess::SetCacheModel>(connection, decoder, this, &WebProcess::setCacheModel);
    if (decoder.messageName() == Messages::WebProcess::RegisterURLSchemeAsEmptyDocument::name())
        return IPC::handleMessage<Messages::WebProcess::RegisterURLSchemeAsEmptyDocument>(connection, decoder, this, &WebProcess::registerURLSchemeAsEmptyDocument);
    if (decoder.messageName() == Messages::WebProcess::RegisterURLSchemeAsSecure::name())
        return IPC::handleMessage<Messages::WebProcess::RegisterURLSchemeAsSecure>(connection, decoder, this, &WebProcess::registerURLSchemeAsSecure);
    if (decoder.messageName() == Messages::WebProcess::RegisterURLSchemeAsBypassingContentSecurityPolicy::name())
        return IPC::handleMessage<Messages::WebProcess::RegisterURLSchemeAsBypassingContentSecurityPolicy>(connection, decoder, this, &WebProcess::registerURLSchemeAsBypassingContentSecurityPolicy);
    if (decoder.messageName() == Messages::WebProcess::SetDomainRelaxationForbiddenForURLScheme::name())
        return IPC::handleMessage<Messages::WebProcess::SetDomainRelaxationForbiddenForURLScheme>(connection, decoder, this, &WebProcess::setDomainRelaxationForbiddenForURLScheme);
    if (decoder.messageName() == Messages::WebProcess::RegisterURLSchemeAsLocal::name())
        return IPC::handleMessage<Messages::WebProcess::RegisterURLSchemeAsLocal>(connection, decoder, this, &WebProcess::registerURLSchemeAsLocal);
    if (decoder.messageName() == Messages::WebProcess::RegisterURLSchemeAsNoAccess::name())
        return IPC::handleMessage<Messages::WebProcess::RegisterURLSchemeAsNoAccess>(connection, decoder, this, &WebProcess::registerURLSchemeAsNoAccess);
    if (decoder.messageName() == Messages::WebProcess::RegisterURLSchemeAsDisplayIsolated::name())
        return IPC::handleMessage<Messages::WebProcess::RegisterURLSchemeAsDisplayIsolated>(connection, decoder, this, &WebProcess::registerURLSchemeAsDisplayIsolated);
    if (decoder.messageName() == Messages::WebProcess::RegisterURLSchemeAsCORSEnabled::name())
        return IPC::handleMessage<Messages::WebProcess::RegisterURLSchemeAsCORSEnabled>(connection, decoder, this, &WebProcess::registerURLSchemeAsCORSEnabled);
    if (decoder.messageName() == Messages::WebProcess::RegisterURLSchemeAsCachePartitioned::name())
        return IPC::handleMessage<Messages::WebProcess::RegisterURLSchemeAsCachePartitioned>(connection, decoder, this, &WebProcess::registerURLSchemeAsCachePartitioned);
    if (decoder.messageName() == Messages::WebProcess::RegisterURLSchemeAsCanDisplayOnlyIfCanRequest::name())
        return IPC::handleMessage<Messages::WebProcess::RegisterURLSchemeAsCanDisplayOnlyIfCanRequest>(connection, decoder, this, &WebProcess::registerURLSchemeAsCanDisplayOnlyIfCanRequest);
    if (decoder.messageName() == Messages::WebProcess::SetDefaultRequestTimeoutInterval::name())
        return IPC::handleMessage<Messages::WebProcess::SetDefaultRequestTimeoutInterval>(connection, decoder, this, &WebProcess::setDefaultRequestTimeoutInterval);
    if (decoder.messageName() == Messages::WebProcess::SetAlwaysUsesComplexTextCodePath::name())
        return IPC::handleMessage<Messages::WebProcess::SetAlwaysUsesComplexTextCodePath>(connection, decoder, this, &WebProcess::setAlwaysUsesComplexTextCodePath);
    if (decoder.messageName() == Messages::WebProcess::SetShouldUseFontSmoothing::name())
        return IPC::handleMessage<Messages::WebProcess::SetShouldUseFontSmoothing>(connection, decoder, this, &WebProcess::setShouldUseFontSmoothing);
    if (decoder.messageName() == Messages::WebProcess::SetResourceLoadStatisticsEnabled::name())
        return IPC::handleMessage<Messages::WebProcess::SetResourceLoadStatisticsEnabled>(connection, decoder, this, &WebProcess::setResourceLoadStatisticsEnabled);
    if (decoder.messageName() == Messages::WebProcess::ClearResourceLoadStatistics::name())
        return IPC::handleMessage<Messages::WebProcess::ClearResourceLoadStatistics>(connection, decoder, this, &WebProcess::clearResourceLoadStatistics);
    if (decoder.messageName() == Messages::WebProcess::UserPreferredLanguagesChanged::name())
        return IPC::handleMessage<Messages::WebProcess::UserPreferredLanguagesChanged>(connection, decoder, this, &WebProcess::userPreferredLanguagesChanged);
    if (decoder.messageName() == Messages::WebProcess::FullKeyboardAccessModeChanged::name())
        return IPC::handleMessage<Messages::WebProcess::FullKeyboardAccessModeChanged>(connection, decoder, this, &WebProcess::fullKeyboardAccessModeChanged);
#if HAVE(MOUSE_DEVICE_OBSERVATION)
    if (decoder.messageName() == Messages::WebProcess::SetHasMouseDevice::name())
        return IPC::handleMessage<Messages::WebProcess::SetHasMouseDevice>(connection, decoder, this, &WebProcess::setHasMouseDevice);
#endif
#if HAVE(STYLUS_DEVICE_OBSERVATION)
    if (decoder.messageName() == Messages::WebProcess::SetHasStylusDevice::name())
        return IPC::handleMessage<Messages::WebProcess::SetHasStylusDevice>(connection, decoder, this, &WebProcess::setHasStylusDevice);
#endif
    if (decoder.messageName() == Messages::WebProcess::RefreshPlugins::name())
        return IPC::handleMessage<Messages::WebProcess::RefreshPlugins>(connection, decoder, this, &WebProcess::refreshPlugins);
    if (decoder.messageName() == Messages::WebProcess::StartMemorySampler::name())
        return IPC::handleMessage<Messages::WebProcess::StartMemorySampler>(connection, decoder, this, &WebProcess::startMemorySampler);
    if (decoder.messageName() == Messages::WebProcess::StopMemorySampler::name())
        return IPC::handleMessage<Messages::WebProcess::StopMemorySampler>(connection, decoder, this, &WebProcess::stopMemorySampler);
    if (decoder.messageName() == Messages::WebProcess::SetTextCheckerState::name())
        return IPC::handleMessage<Messages::WebProcess::SetTextCheckerState>(connection, decoder, this, &WebProcess::setTextCheckerState);
    if (decoder.messageName() == Messages::WebProcess::SetEnhancedAccessibility::name())
        return IPC::handleMessage<Messages::WebProcess::SetEnhancedAccessibility>(connection, decoder, this, &WebProcess::setEnhancedAccessibility);
    if (decoder.messageName() == Messages::WebProcess::GarbageCollectJavaScriptObjects::name())
        return IPC::handleMessage<Messages::WebProcess::GarbageCollectJavaScriptObjects>(connection, decoder, this, &WebProcess::garbageCollectJavaScriptObjects);
    if (decoder.messageName() == Messages::WebProcess::SetJavaScriptGarbageCollectorTimerEnabled::name())
        return IPC::handleMessage<Messages::WebProcess::SetJavaScriptGarbageCollectorTimerEnabled>(connection, decoder, this, &WebProcess::setJavaScriptGarbageCollectorTimerEnabled);
    if (decoder.messageName() == Messages::WebProcess::SetInjectedBundleParameter::name())
        return IPC::handleMessage<Messages::WebProcess::SetInjectedBundleParameter>(connection, decoder, this, &WebProcess::setInjectedBundleParameter);
    if (decoder.messageName() == Messages::WebProcess::SetInjectedBundleParameters::name())
        return IPC::handleMessage<Messages::WebProcess::SetInjectedBundleParameters>(connection, decoder, this, &WebProcess::setInjectedBundleParameters);
    if (decoder.messageName() == Messages::WebProcess::HandleInjectedBundleMessage::name())
        return IPC::handleMessage<Messages::WebProcess::HandleInjectedBundleMessage>(connection, decoder, this, &WebProcess::handleInjectedBundleMessage);
    if (decoder.messageName() == Messages::WebProcess::FetchWebsiteData::name())
        return IPC::handleMessageAsync<Messages::WebProcess::FetchWebsiteData>(connection, decoder, this, &WebProcess::fetchWebsiteData);
    if (decoder.messageName() == Messages::WebProcess::DeleteWebsiteData::name())
        return IPC::handleMessageAsync<Messages::WebProcess::DeleteWebsiteData>(connection, decoder, this, &WebProcess::deleteWebsiteData);
    if (decoder.messageName() == Messages::WebProcess::DeleteWebsiteDataForOrigins::name())
        return IPC::handleMessageAsync<Messages::WebProcess::DeleteWebsiteDataForOrigins>(connection, decoder, this, &WebProcess::deleteWebsiteDataForOrigins);
    if (decoder.messageName() == Messages::WebProcess::SetHiddenPageDOMTimerThrottlingIncreaseLimit::name())
        return IPC::handleMessage<Messages::WebProcess::SetHiddenPageDOMTimerThrottlingIncreaseLimit>(connection, decoder, this, &WebProcess::setHiddenPageDOMTimerThrottlingIncreaseLimit);
    if (decoder.messageName() == Messages::WebProcess::SetMemoryCacheDisabled::name())
        return IPC::handleMessage<Messages::WebProcess::SetMemoryCacheDisabled>(connection, decoder, this, &WebProcess::setMemoryCacheDisabled);
#if ENABLE(SERVICE_CONTROLS)
    if (decoder.messageName() == Messages::WebProcess::SetEnabledServices::name())
        return IPC::handleMessage<Messages::WebProcess::SetEnabledServices>(connection, decoder, this, &WebProcess::setEnabledServices);
#endif
    if (decoder.messageName() == Messages::WebProcess::EnsureAutomationSessionProxy::name())
        return IPC::handleMessage<Messages::WebProcess::EnsureAutomationSessionProxy>(connection, decoder, this, &WebProcess::ensureAutomationSessionProxy);
    if (decoder.messageName() == Messages::WebProcess::DestroyAutomationSessionProxy::name())
        return IPC::handleMessage<Messages::WebProcess::DestroyAutomationSessionProxy>(connection, decoder, this, &WebProcess::destroyAutomationSessionProxy);
    if (decoder.messageName() == Messages::WebProcess::PrepareToSuspend::name())
        return IPC::handleMessageAsync<Messages::WebProcess::PrepareToSuspend>(connection, decoder, this, &WebProcess::prepareToSuspend);
    if (decoder.messageName() == Messages::WebProcess::ProcessDidResume::name())
        return IPC::handleMessage<Messages::WebProcess::ProcessDidResume>(connection, decoder, this, &WebProcess::processDidResume);
    if (decoder.messageName() == Messages::WebProcess::BackgroundResponsivenessPing::name())
        return IPC::handleMessage<Messages::WebProcess::BackgroundResponsivenessPing>(connection, decoder, this, &WebProcess::backgroundResponsivenessPing);
#if ENABLE(GAMEPAD)
    if (decoder.messageName() == Messages::WebProcess::SetInitialGamepads::name())
        return IPC::handleMessage<Messages::WebProcess::SetInitialGamepads>(connection, decoder, this, &WebProcess::setInitialGamepads);
#endif
#if ENABLE(GAMEPAD)
    if (decoder.messageName() == Messages::WebProcess::GamepadConnected::name())
        return IPC::handleMessage<Messages::WebProcess::GamepadConnected>(connection, decoder, this, &WebProcess::gamepadConnected);
#endif
#if ENABLE(GAMEPAD)
    if (decoder.messageName() == Messages::WebProcess::GamepadDisconnected::name())
        return IPC::handleMessage<Messages::WebProcess::GamepadDisconnected>(connection, decoder, this, &WebProcess::gamepadDisconnected);
#endif
    if (decoder.messageName() == Messages::WebProcess::EstablishSharedWorkerContextConnectionToNetworkProcess::name())
        return IPC::handleMessageAsync<Messages::WebProcess::EstablishSharedWorkerContextConnectionToNetworkProcess>(connection, decoder, this, &WebProcess::establishSharedWorkerContextConnectionToNetworkProcess);
#if ENABLE(SERVICE_WORKER)
    if (decoder.messageName() == Messages::WebProcess::EstablishServiceWorkerContextConnectionToNetworkProcess::name())
        return IPC::handleMessageAsync<Messages::WebProcess::EstablishServiceWorkerContextConnectionToNetworkProcess>(connection, decoder, this, &WebProcess::establishServiceWorkerContextConnectionToNetworkProcess);
#endif
    if (decoder.messageName() == Messages::WebProcess::SetHasSuspendedPageProxy::name())
        return IPC::handleMessage<Messages::WebProcess::SetHasSuspendedPageProxy>(connection, decoder, this, &WebProcess::setHasSuspendedPageProxy);
    if (decoder.messageName() == Messages::WebProcess::SetIsInProcessCache::name())
        return IPC::handleMessage<Messages::WebProcess::SetIsInProcessCache>(connection, decoder, this, &WebProcess::setIsInProcessCache);
    if (decoder.messageName() == Messages::WebProcess::MarkIsNoLongerPrewarmed::name())
        return IPC::handleMessage<Messages::WebProcess::MarkIsNoLongerPrewarmed>(connection, decoder, this, &WebProcess::markIsNoLongerPrewarmed);
    if (decoder.messageName() == Messages::WebProcess::GetActivePagesOriginsForTesting::name())
        return IPC::handleMessageAsync<Messages::WebProcess::GetActivePagesOriginsForTesting>(connection, decoder, this, &WebProcess::getActivePagesOriginsForTesting);
#if PLATFORM(COCOA)
    if (decoder.messageName() == Messages::WebProcess::SetScreenProperties::name())
        return IPC::handleMessage<Messages::WebProcess::SetScreenProperties>(connection, decoder, this, &WebProcess::setScreenProperties);
#endif
#if PLATFORM(MAC)
    if (decoder.messageName() == Messages::WebProcess::ScrollerStylePreferenceChanged::name())
        return IPC::handleMessage<Messages::WebProcess::ScrollerStylePreferenceChanged>(connection, decoder, this, &WebProcess::scrollerStylePreferenceChanged);
#endif
#if PLATFORM(MAC)
    if (decoder.messageName() == Messages::WebProcess::DisplayConfigurationChanged::name())
        return IPC::handleMessage<Messages::WebProcess::DisplayConfigurationChanged>(connection, decoder, this, &WebProcess::displayConfigurationChanged);
#endif
#if PLATFORM(IOS_FAMILY)
    if (decoder.messageName() == Messages::WebProcess::UserInterfaceIdiomDidChange::name())
        return IPC::handleMessage<Messages::WebProcess::UserInterfaceIdiomDidChange>(connection, decoder, this, &WebProcess::userInterfaceIdiomDidChange);
#endif
#if PLATFORM(IOS_FAMILY) && !PLATFORM(MACCATALYST)
    if (decoder.messageName() == Messages::WebProcess::BacklightLevelDidChange::name())
        return IPC::handleMessage<Messages::WebProcess::BacklightLevelDidChange>(connection, decoder, this, &WebProcess::backlightLevelDidChange);
#endif
#if PLATFORM(MAC) || PLATFORM(MACCATALYST)
    if (decoder.messageName() == Messages::WebProcess::ColorPreferencesDidChange::name())
        return IPC::handleMessage<Messages::WebProcess::ColorPreferencesDidChange>(connection, decoder, this, &WebProcess::colorPreferencesDidChange);
#endif
    if (decoder.messageName() == Messages::WebProcess::IsJITEnabled::name())
        return IPC::handleMessageAsync<Messages::WebProcess::IsJITEnabled>(connection, decoder, this, &WebProcess::isJITEnabled);
#if PLATFORM(COCOA)
    if (decoder.messageName() == Messages::WebProcess::SetMediaMIMETypes::name())
        return IPC::handleMessage<Messages::WebProcess::SetMediaMIMETypes>(connection, decoder, this, &WebProcess::setMediaMIMETypes);
#endif
#if (PLATFORM(COCOA) && ENABLE(REMOTE_INSPECTOR))
    if (decoder.messageName() == Messages::WebProcess::EnableRemoteWebInspector::name())
        return IPC::handleMessage<Messages::WebProcess::EnableRemoteWebInspector>(connection, decoder, this, &WebProcess::enableRemoteWebInspector);
#endif
#if ENABLE(MEDIA_STREAM)
    if (decoder.messageName() == Messages::WebProcess::AddMockMediaDevice::name())
        return IPC::handleMessage<Messages::WebProcess::AddMockMediaDevice>(connection, decoder, this, &WebProcess::addMockMediaDevice);
#endif
#if ENABLE(MEDIA_STREAM)
    if (decoder.messageName() == Messages::WebProcess::ClearMockMediaDevices::name())
        return IPC::handleMessage<Messages::WebProcess::ClearMockMediaDevices>(connection, decoder, this, &WebProcess::clearMockMediaDevices);
#endif
#if ENABLE(MEDIA_STREAM)
    if (decoder.messageName() == Messages::WebProcess::RemoveMockMediaDevice::name())
        return IPC::handleMessage<Messages::WebProcess::RemoveMockMediaDevice>(connection, decoder, this, &WebProcess::removeMockMediaDevice);
#endif
#if ENABLE(MEDIA_STREAM)
    if (decoder.messageName() == Messages::WebProcess::ResetMockMediaDevices::name())
        return IPC::handleMessage<Messages::WebProcess::ResetMockMediaDevices>(connection, decoder, this, &WebProcess::resetMockMediaDevices);
#endif
#if (ENABLE(MEDIA_STREAM) && ENABLE(SANDBOX_EXTENSIONS))
    if (decoder.messageName() == Messages::WebProcess::GrantUserMediaDeviceSandboxExtensions::name())
        return IPC::handleMessage<Messages::WebProcess::GrantUserMediaDeviceSandboxExtensions>(connection, decoder, this, &WebProcess::grantUserMediaDeviceSandboxExtensions);
#endif
#if (ENABLE(MEDIA_STREAM) && ENABLE(SANDBOX_EXTENSIONS))
    if (decoder.messageName() == Messages::WebProcess::RevokeUserMediaDeviceSandboxExtensions::name())
        return IPC::handleMessage<Messages::WebProcess::RevokeUserMediaDeviceSandboxExtensions>(connection, decoder, this, &WebProcess::revokeUserMediaDeviceSandboxExtensions);
#endif
    if (decoder.messageName() == Messages::WebProcess::ClearCurrentModifierStateForTesting::name())
        return IPC::handleMessage<Messages::WebProcess::ClearCurrentModifierStateForTesting>(connection, decoder, this, &WebProcess::clearCurrentModifierStateForTesting);
    if (decoder.messageName() == Messages::WebProcess::SetBackForwardCacheCapacity::name())
        return IPC::handleMessage<Messages::WebProcess::SetBackForwardCacheCapacity>(connection, decoder, this, &WebProcess::setBackForwardCacheCapacity);
    if (decoder.messageName() == Messages::WebProcess::ClearCachedPage::name())
        return IPC::handleMessageAsync<Messages::WebProcess::ClearCachedPage>(connection, decoder, this, &WebProcess::clearCachedPage);
#if PLATFORM(GTK) || PLATFORM(WPE)
    if (decoder.messageName() == Messages::WebProcess::SendMessageToWebExtension::name())
        return IPC::handleMessage<Messages::WebProcess::SendMessageToWebExtension>(connection, decoder, this, &WebProcess::sendMessageToWebExtension);
#endif
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
    if (decoder.messageName() == Messages::WebProcess::SeedResourceLoadStatisticsForTesting::name())
        return IPC::handleMessageAsync<Messages::WebProcess::SeedResourceLoadStatisticsForTesting>(connection, decoder, this, &WebProcess::seedResourceLoadStatisticsForTesting);
#endif
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
    if (decoder.messageName() == Messages::WebProcess::SetThirdPartyCookieBlockingMode::name())
        return IPC::handleMessageAsync<Messages::WebProcess::SetThirdPartyCookieBlockingMode>(connection, decoder, this, &WebProcess::setThirdPartyCookieBlockingMode);
#endif
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
    if (decoder.messageName() == Messages::WebProcess::SetDomainsWithUserInteraction::name())
        return IPC::handleMessage<Messages::WebProcess::SetDomainsWithUserInteraction>(connection, decoder, this, &WebProcess::setDomainsWithUserInteraction);
#endif
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
    if (decoder.messageName() == Messages::WebProcess::SetDomainsWithCrossPageStorageAccess::name())
        return IPC::handleMessageAsync<Messages::WebProcess::SetDomainsWithCrossPageStorageAccess>(connection, decoder, this, &WebProcess::setDomainsWithCrossPageStorageAccess);
#endif
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
    if (decoder.messageName() == Messages::WebProcess::SendResourceLoadStatisticsDataImmediately::name())
        return IPC::handleMessageAsync<Messages::WebProcess::SendResourceLoadStatisticsDataImmediately>(connection, decoder, this, &WebProcess::sendResourceLoadStatisticsDataImmediately);
#endif
    if (decoder.messageName() == Messages::WebProcess::GrantAccessToAssetServices::name())
        return IPC::handleMessage<Messages::WebProcess::GrantAccessToAssetServices>(connection, decoder, this, &WebProcess::grantAccessToAssetServices);
    if (decoder.messageName() == Messages::WebProcess::RevokeAccessToAssetServices::name())
        return IPC::handleMessage<Messages::WebProcess::RevokeAccessToAssetServices>(connection, decoder, this, &WebProcess::revokeAccessToAssetServices);
    if (decoder.messageName() == Messages::WebProcess::SwitchFromStaticFontRegistryToUserFontRegistry::name())
        return IPC::handleMessage<Messages::WebProcess::SwitchFromStaticFontRegistryToUserFontRegistry>(connection, decoder, this, &WebProcess::switchFromStaticFontRegistryToUserFontRegistry);
#if PLATFORM(COCOA)
    if (decoder.messageName() == Messages::WebProcess::UnblockServicesRequiredByAccessibility::name())
        return IPC::handleMessage<Messages::WebProcess::UnblockServicesRequiredByAccessibility>(connection, decoder, this, &WebProcess::unblockServicesRequiredByAccessibility);
#endif
#if (PLATFORM(COCOA) && ENABLE(CFPREFS_DIRECT_MODE))
    if (decoder.messageName() == Messages::WebProcess::NotifyPreferencesChanged::name())
        return IPC::handleMessage<Messages::WebProcess::NotifyPreferencesChanged>(connection, decoder, this, &WebProcess::notifyPreferencesChanged);
#endif
#if (PLATFORM(COCOA) && ENABLE(CFPREFS_DIRECT_MODE))
    if (decoder.messageName() == Messages::WebProcess::UnblockPreferenceService::name())
        return IPC::handleMessage<Messages::WebProcess::UnblockPreferenceService>(connection, decoder, this, &WebProcess::unblockPreferenceService);
#endif
#if PLATFORM(COCOA)
    if (decoder.messageName() == Messages::WebProcess::PowerSourceDidChange::name())
        return IPC::handleMessage<Messages::WebProcess::PowerSourceDidChange>(connection, decoder, this, &WebProcess::powerSourceDidChange);
#endif
#if PLATFORM(GTK) && !USE(GTK4)
    if (decoder.messageName() == Messages::WebProcess::SetUseSystemAppearanceForScrollbars::name())
        return IPC::handleMessage<Messages::WebProcess::SetUseSystemAppearanceForScrollbars>(connection, decoder, this, &WebProcess::setUseSystemAppearanceForScrollbars);
#endif
#if PLATFORM(COCOA)
    if (decoder.messageName() == Messages::WebProcess::DidWriteToPasteboardAsynchronously::name())
        return IPC::handleMessage<Messages::WebProcess::DidWriteToPasteboardAsynchronously>(connection, decoder, this, &WebProcess::didWriteToPasteboardAsynchronously);
#endif
#if HAVE(CVDISPLAYLINK)
    if (decoder.messageName() == Messages::WebProcess::DisplayWasRefreshed::name())
        return IPC::handleMessage<Messages::WebProcess::DisplayWasRefreshed>(connection, decoder, this, &WebProcess::displayWasRefreshed);
#endif
#if PLATFORM(MAC)
    if (decoder.messageName() == Messages::WebProcess::SystemWillPowerOn::name())
        return IPC::handleMessage<Messages::WebProcess::SystemWillPowerOn>(connection, decoder, this, &WebProcess::systemWillPowerOn);
#endif
#if PLATFORM(MAC)
    if (decoder.messageName() == Messages::WebProcess::SystemWillSleep::name())
        return IPC::handleMessage<Messages::WebProcess::SystemWillSleep>(connection, decoder, this, &WebProcess::systemWillSleep);
#endif
#if PLATFORM(MAC)
    if (decoder.messageName() == Messages::WebProcess::SystemDidWake::name())
        return IPC::handleMessage<Messages::WebProcess::SystemDidWake>(connection, decoder, this, &WebProcess::systemDidWake);
#endif
#if PLATFORM(COCOA)
    if (decoder.messageName() == Messages::WebProcess::ConsumeAudioComponentRegistrations::name())
        return IPC::handleMessage<Messages::WebProcess::ConsumeAudioComponentRegistrations>(connection, decoder, this, &WebProcess::consumeAudioComponentRegistrations);
#endif
#if PLATFORM(COCOA)
    if (decoder.messageName() == Messages::WebProcess::AccessibilityPreferencesDidChange::name())
        return IPC::handleMessage<Messages::WebProcess::AccessibilityPreferencesDidChange>(connection, decoder, this, &WebProcess::accessibilityPreferencesDidChange);
#endif
#if HAVE(MEDIA_ACCESSIBILITY_FRAMEWORK)
    if (decoder.messageName() == Messages::WebProcess::SetMediaAccessibilityPreferences::name())
        return IPC::handleMessage<Messages::WebProcess::SetMediaAccessibilityPreferences>(connection, decoder, this, &WebProcess::setMediaAccessibilityPreferences);
#endif
    UNUSED_PARAM(connection);
    UNUSED_PARAM(decoder);
#if ENABLE(IPC_TESTING_API)
    if (connection.ignoreInvalidMessageForTesting())
        return;
#endif // ENABLE(IPC_TESTING_API)
    ASSERT_NOT_REACHED_WITH_MESSAGE("Unhandled message %s to %" PRIu64, IPC::description(decoder.messageName()), decoder.destinationID());
}

} // namespace WebKit
