"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.BasicAuthRoutes = void 0;

var _configSchema = require("@osd/config-schema");

var _security_cookie = require("../../../session/security_cookie");

var _common = require("../../../../common");

var _tenant_resolver = require("../../../multitenancy/tenant_resolver");

var _encode_uri_query = require("../../../../../../src/plugins/opensearch_dashboards_utils/common/url/encode_uri_query");

/*
 *   Copyright OpenSearch Contributors
 *
 *   Licensed under the Apache License, Version 2.0 (the "License").
 *   You may not use this file except in compliance with the License.
 *   A copy of the License is located at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   or in the "license" file accompanying this file. This file is distributed
 *   on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 *   express or implied. See the License for the specific language governing
 *   permissions and limitations under the License.
 */
class BasicAuthRoutes {
  constructor(router, config, sessionStorageFactory, securityClient, coreSetup) {
    this.router = router;
    this.config = config;
    this.sessionStorageFactory = sessionStorageFactory;
    this.securityClient = securityClient;
    this.coreSetup = coreSetup;
  }

  setupRoutes() {
    // bootstrap an empty page so that browser app can render the login page
    // using client side routing.
    this.coreSetup.http.resources.register({
      path: _common.LOGIN_PAGE_URI,
      validate: false,
      options: {
        authRequired: false
      }
    }, async (context, request, response) => {
      this.sessionStorageFactory.asScoped(request).clear();
      const clearOldVersionCookie = (0, _security_cookie.clearOldVersionCookieValue)(this.config);
      return response.renderAnonymousCoreApp({
        headers: {
          'set-cookie': clearOldVersionCookie
        }
      });
    }); // login using username and password

    this.router.post({
      path: _common.API_AUTH_LOGIN,
      validate: {
        body: _configSchema.schema.object({
          username: _configSchema.schema.string(),
          password: _configSchema.schema.string()
        })
      },
      options: {
        authRequired: false
      }
    }, async (context, request, response) => {
      var _this$config$multiten, _this$config$multiten2;

      const forbiddenUsernames = this.config.auth.forbidden_usernames;

      if (forbiddenUsernames.indexOf(request.body.username) > -1) {
        context.security_plugin.logger.error(`Denied login for forbidden username ${request.body.username}`);
        return response.badRequest({
          // Cannot login using forbidden user name.
          body: 'Invalid username or password'
        });
      }

      let user;

      try {
        user = await this.securityClient.authenticate(request, {
          username: request.body.username,
          password: request.body.password
        });
      } catch (error) {
        context.security_plugin.logger.error(`Failed authentication: ${error}`);
        return response.unauthorized({
          headers: {
            'www-authenticate': error.message
          }
        });
      }

      this.sessionStorageFactory.asScoped(request).clear();
      const encodedCredentials = Buffer.from(`${request.body.username}:${request.body.password}`).toString('base64');
      const sessionStorage = {
        username: user.username,
        credentials: {
          authHeaderValue: `Basic ${encodedCredentials}`
        },
        authType: 'basicauth',
        isAnonymousAuth: false,
        expiryTime: Date.now() + this.config.session.ttl
      };

      if ((_this$config$multiten = this.config.multitenancy) !== null && _this$config$multiten !== void 0 && _this$config$multiten.enabled) {
        const selectTenant = (0, _tenant_resolver.resolveTenant)(request, user.username, user.tenants, this.config, sessionStorage);
        sessionStorage.tenant = selectTenant;
      }

      this.sessionStorageFactory.asScoped(request).set(sessionStorage);
      return response.ok({
        body: {
          username: user.username,
          tenants: user.tenants,
          roles: user.roles,
          backendroles: user.backendRoles,
          selectedTenants: (_this$config$multiten2 = this.config.multitenancy) !== null && _this$config$multiten2 !== void 0 && _this$config$multiten2.enabled ? sessionStorage.tenant : undefined
        }
      });
    }); // logout

    this.router.post({
      path: _common.API_AUTH_LOGOUT,
      validate: false,
      options: {
        authRequired: false
      }
    }, async (context, request, response) => {
      this.sessionStorageFactory.asScoped(request).clear();
      return response.ok({
        body: {}
      });
    }); // anonymous auth

    this.router.get({
      path: `/auth/anonymous`,
      validate: false,
      options: {
        authRequired: false
      }
    }, async (context, request, response) => {
      if (this.config.auth.anonymous_auth_enabled) {
        var _this$config$multiten3;

        let user; // If the request contains no redirect path, simply redirect to basepath.

        let redirectUrl = this.coreSetup.http.basePath.serverBasePath ? this.coreSetup.http.basePath.serverBasePath : '/';
        const requestQuery = request.url.searchParams;
        const nextUrl = requestQuery === null || requestQuery === void 0 ? void 0 : requestQuery.get('nextUrl');

        if (nextUrl) {
          redirectUrl = nextUrl;
        }

        context.security_plugin.logger.info('The Redirect Path is ' + redirectUrl);

        try {
          user = await this.securityClient.authenticateWithHeaders(request, {});
        } catch (error) {
          context.security_plugin.logger.error(`Failed authentication: ${error}. Redirecting to Login Page`);
          return response.redirected({
            headers: {
              location: `${this.coreSetup.http.basePath.serverBasePath}${_common.LOGIN_PAGE_URI}${nextUrl ? '?nextUrl=' + (0, _encode_uri_query.encodeUriQuery)(redirectUrl) : ''}`
            }
          });
        }

        this.sessionStorageFactory.asScoped(request).clear();
        const sessionStorage = {
          username: user.username,
          authType: 'basicauth',
          isAnonymousAuth: true,
          expiryTime: Date.now() + this.config.session.ttl
        };

        if ((_this$config$multiten3 = this.config.multitenancy) !== null && _this$config$multiten3 !== void 0 && _this$config$multiten3.enabled) {
          const selectTenant = (0, _tenant_resolver.resolveTenant)(request, user.username, user.tenants, this.config, sessionStorage);
          sessionStorage.tenant = selectTenant;
        }

        this.sessionStorageFactory.asScoped(request).set(sessionStorage);
        return response.redirected({
          headers: {
            location: `${redirectUrl}`
          }
        });
      } else {
        context.security_plugin.logger.error('Anonymous auth is disabled. Redirecting to Login Page');
        return response.redirected({
          headers: {
            location: `${this.coreSetup.http.basePath.serverBasePath}${_common.LOGIN_PAGE_URI}`
          }
        });
      }
    });
  }

}

exports.BasicAuthRoutes = BasicAuthRoutes;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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