/* ssl.c
 *
 * Copyright (C) 2006-2021 wolfSSL Inc.
 *
 * This file is part of wolfSSL.
 *
 * wolfSSL is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * wolfSSL is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1335, USA
 */


#ifdef HAVE_CONFIG_H
    #include <config.h>
#endif

#include <wolfssl/wolfcrypt/settings.h>
#if defined(OPENSSL_EXTRA) && !defined(_WIN32)
    /* turn on GNU extensions for XISASCII */
    #undef  _GNU_SOURCE
    #define _GNU_SOURCE
#endif

#if !defined(WOLFCRYPT_ONLY) || defined(OPENSSL_EXTRA) || \
    defined(OPENSSL_EXTRA_X509_SMALL)

#include <wolfssl/internal.h>
#include <wolfssl/error-ssl.h>
#include <wolfssl/wolfcrypt/coding.h>
#include <wolfssl/wolfcrypt/kdf.h>
#ifdef NO_INLINE
    #include <wolfssl/wolfcrypt/misc.h>
#else
    #define WOLFSSL_MISC_INCLUDED
    #include <wolfcrypt/src/misc.c>
#endif

#ifdef HAVE_ERRNO_H
    #include <errno.h>
#endif


#if !defined(WOLFSSL_ALLOW_NO_SUITES) && !defined(WOLFCRYPT_ONLY)
    #if defined(NO_DH) && !defined(HAVE_ECC) && !defined(WOLFSSL_STATIC_RSA) \
                && !defined(WOLFSSL_STATIC_DH) && !defined(WOLFSSL_STATIC_PSK) \
                && !defined(HAVE_CURVE25519) && !defined(HAVE_CURVE448)
        #error "No cipher suites defined because DH disabled, ECC disabled, and no static suites defined. Please see top of README"
    #endif
    #ifdef WOLFSSL_CERT_GEN
        /* need access to Cert struct for creating certificate */
        #include <wolfssl/wolfcrypt/asn_public.h>
    #endif
#endif

#if !defined(WOLFCRYPT_ONLY) && (defined(OPENSSL_EXTRA)     \
    || defined(OPENSSL_EXTRA_X509_SMALL)                    \
    || defined(HAVE_WEBSERVER) || defined(WOLFSSL_KEY_GEN))
    #include <wolfssl/openssl/evp.h>
    /* openssl headers end, wolfssl internal headers next */
#endif

#include <wolfssl/wolfcrypt/wc_encrypt.h>

#ifndef NO_RSA
    #include <wolfssl/wolfcrypt/rsa.h>
#endif

#ifdef OPENSSL_EXTRA
    /* openssl headers begin */
    #include <wolfssl/openssl/ssl.h>
    #include <wolfssl/openssl/aes.h>
#ifndef WOLFCRYPT_ONLY
    #include <wolfssl/openssl/hmac.h>
    #include <wolfssl/openssl/cmac.h>
#endif
    #include <wolfssl/openssl/crypto.h>
    #include <wolfssl/openssl/des.h>
    #include <wolfssl/openssl/bn.h>
    #include <wolfssl/openssl/buffer.h>
    #include <wolfssl/openssl/dh.h>
    #include <wolfssl/openssl/rsa.h>
    #include <wolfssl/openssl/fips_rand.h>
#ifndef WOLFCRYPT_ONLY
    #include <wolfssl/openssl/pem.h>
#endif
    #include <wolfssl/openssl/ec.h>
    #include <wolfssl/openssl/ec25519.h>
    #include <wolfssl/openssl/ed25519.h>
    #include <wolfssl/openssl/ec448.h>
    #include <wolfssl/openssl/ed448.h>
    #include <wolfssl/openssl/ecdsa.h>
    #include <wolfssl/openssl/ecdh.h>
    #include <wolfssl/openssl/err.h>
    #include <wolfssl/openssl/modes.h>
    #include <wolfssl/openssl/opensslv.h>
    #include <wolfssl/openssl/rc4.h>
    #include <wolfssl/openssl/stack.h>
    #include <wolfssl/openssl/x509_vfy.h>
    /* openssl headers end, wolfssl internal headers next */
    #include <wolfssl/wolfcrypt/hmac.h>
    #include <wolfssl/wolfcrypt/random.h>
    #include <wolfssl/wolfcrypt/des3.h>
    #include <wolfssl/wolfcrypt/ecc.h>
    #include <wolfssl/wolfcrypt/md4.h>
    #include <wolfssl/wolfcrypt/md5.h>
    #include <wolfssl/wolfcrypt/arc4.h>
    #include <wolfssl/wolfcrypt/curve25519.h>
    #include <wolfssl/wolfcrypt/ed25519.h>
    #include <wolfssl/wolfcrypt/curve448.h>
    #if defined(HAVE_PQC)
        #include <wolfssl/wolfcrypt/falcon.h>
    #endif
    #if defined(OPENSSL_ALL) || defined(HAVE_STUNNEL)
        #ifdef HAVE_OCSP
            #include <wolfssl/openssl/ocsp.h>
        #endif
        #include <wolfssl/openssl/lhash.h>
        #include <wolfssl/openssl/txt_db.h>
    #endif /* WITH_STUNNEL */
    #if defined(WOLFSSL_SHA512) || defined(WOLFSSL_SHA384)
        #include <wolfssl/wolfcrypt/sha512.h>
    #endif
    #if defined(WOLFCRYPT_HAVE_SRP) && !defined(NO_SHA256) \
        && !defined(WC_NO_RNG)
        #include <wolfssl/wolfcrypt/srp.h>
    #endif
    #if defined(HAVE_FIPS) || defined(HAVE_SELFTEST)
        #include <wolfssl/wolfcrypt/pkcs7.h>
    #endif
    #if defined(OPENSSL_ALL) && defined(HAVE_PKCS7)
        #include <wolfssl/openssl/pkcs7.h>
    #endif /* OPENSSL_ALL && HAVE_PKCS7 */
#endif

#if defined(OPENSSL_EXTRA) || defined(OPENSSL_EXTRA_X509_SMALL)
    #include <wolfssl/openssl/x509v3.h>
    int SetIndividualInternal(WOLFSSL_BIGNUM* bn, mp_int* mpi);
    int SetIndividualExternal(WOLFSSL_BIGNUM** bn, mp_int* mpi);
#endif

#if defined(WOLFSSL_QT)
    #include <wolfssl/wolfcrypt/sha.h>
#endif

#ifdef NO_ASN
    #include <wolfssl/wolfcrypt/dh.h>
#endif
#endif /* !WOLFCRYPT_ONLY || OPENSSL_EXTRA */

/*
 * OPENSSL_COMPATIBLE_DEFAULTS:
 *     Enable default behaviour that is compatible with OpenSSL. For example
 *     SSL_CTX by default doesn't verify the loaded certs. Enabling this
 *     should make porting to new projects easier.
 * WOLFSSL_CHECK_ALERT_ON_ERR:
 *     Check for alerts during the handshake in the event of an error.
 * NO_SESSION_CACHE_REF:
 *     wolfSSL_get_session on a client will return a reference to the internal
 *     ClientCache by default for backwards compatibility. This define will
 *     make wolfSSL_get_session return a reference to ssl->session. The returned
 *     pointer will be freed with the related WOLFSSL object.
 */

#define WOLFSSL_EVP_INCLUDED
#include "wolfcrypt/src/evp.c"

#ifndef WOLFCRYPT_ONLY

#ifdef OPENSSL_EXTRA
    /* Global pointer to constant BN on */
    static WOLFSSL_BIGNUM* bn_one = NULL;

    /* WOLFSSL_NO_OPENSSL_RAND_CB: Allows way to reduce code size for
     *                OPENSSL_EXTRA where RAND callbacks are not used */
    #ifndef WOLFSSL_NO_OPENSSL_RAND_CB
        static const WOLFSSL_RAND_METHOD* gRandMethods = NULL;
        static int gRandMethodsInit = 0;
        static wolfSSL_Mutex gRandMethodMutex;
    #endif /* !WOLFSSL_NO_OPENSSL_RAND_CB */
#endif /* OPENSSL_EXTRA */

#if defined(OPENSSL_EXTRA) && defined(HAVE_ECC)
const WOLF_EC_NIST_NAME kNistCurves[] = {
    {XSTR_SIZEOF("P-192"),   "P-192",   NID_X9_62_prime192v1},
    {XSTR_SIZEOF("P-256"),   "P-256",   NID_X9_62_prime256v1},
    {XSTR_SIZEOF("P-112"),   "P-112",   NID_secp112r1},
    {XSTR_SIZEOF("P-112-2"), "P-112-2", NID_secp112r2},
    {XSTR_SIZEOF("P-128"),   "P-128",   NID_secp128r1},
    {XSTR_SIZEOF("P-128-2"), "P-128-2", NID_secp128r2},
    {XSTR_SIZEOF("P-160"),   "P-160",   NID_secp160r1},
    {XSTR_SIZEOF("P-160-2"), "P-160-2", NID_secp160r2},
    {XSTR_SIZEOF("P-224"),   "P-224",   NID_secp224r1},
    {XSTR_SIZEOF("P-384"),   "P-384",   NID_secp384r1},
    {XSTR_SIZEOF("P-521"),   "P-521",   NID_secp521r1},
    {XSTR_SIZEOF("K-160"),   "K-160",   NID_secp160k1},
    {XSTR_SIZEOF("K-192"),   "K-192",   NID_secp192k1},
    {XSTR_SIZEOF("K-224"),   "K-224",   NID_secp224k1},
    {XSTR_SIZEOF("K-256"),   "K-256",   NID_secp256k1},
    {XSTR_SIZEOF("B-160"),   "B-160",   NID_brainpoolP160r1},
    {XSTR_SIZEOF("B-192"),   "B-192",   NID_brainpoolP192r1},
    {XSTR_SIZEOF("B-224"),   "B-224",   NID_brainpoolP224r1},
    {XSTR_SIZEOF("B-256"),   "B-256",   NID_brainpoolP256r1},
    {XSTR_SIZEOF("B-320"),   "B-320",   NID_brainpoolP320r1},
    {XSTR_SIZEOF("B-384"),   "B-384",   NID_brainpoolP384r1},
    {XSTR_SIZEOF("B-512"),   "B-512",   NID_brainpoolP512r1},
#ifdef HAVE_PQC
    {XSTR_SIZEOF("KYBER_LEVEL1"), "KYBER_LEVEL1", WOLFSSL_KYBER_LEVEL1},
    {XSTR_SIZEOF("KYBER_LEVEL3"), "KYBER_LEVEL3", WOLFSSL_KYBER_LEVEL3},
    {XSTR_SIZEOF("KYBER_LEVEL5"), "KYBER_LEVEL5", WOLFSSL_KYBER_LEVEL5},
    {XSTR_SIZEOF("NTRU_HPS_LEVEL1"), "NTRU_HPS_LEVEL1", WOLFSSL_NTRU_HPS_LEVEL1},
    {XSTR_SIZEOF("NTRU_HPS_LEVEL3"), "NTRU_HPS_LEVEL3", WOLFSSL_NTRU_HPS_LEVEL3},
    {XSTR_SIZEOF("NTRU_HPS_LEVEL5"), "NTRU_HPS_LEVEL5", WOLFSSL_NTRU_HPS_LEVEL5},
    {XSTR_SIZEOF("NTRU_HRSS_LEVEL3"), "NTRU_HRSS_LEVEL3", WOLFSSL_NTRU_HRSS_LEVEL3},
    {XSTR_SIZEOF("SABER_LEVEL1"), "SABER_LEVEL1", WOLFSSL_SABER_LEVEL1},
    {XSTR_SIZEOF("SABER_LEVEL3"), "SABER_LEVEL3", WOLFSSL_SABER_LEVEL3},
    {XSTR_SIZEOF("SABER_LEVEL5"), "SABER_LEVEL5", WOLFSSL_SABER_LEVEL5},
    {XSTR_SIZEOF("KYBER_90S_LEVEL1"), "KYBER_90S_LEVEL1", WOLFSSL_KYBER_90S_LEVEL1},
    {XSTR_SIZEOF("KYBER_90S_LEVEL3"), "KYBER_90S_LEVEL3", WOLFSSL_KYBER_90S_LEVEL3},
    {XSTR_SIZEOF("KYBER_90S_LEVEL5"), "KYBER_90S_LEVEL5", WOLFSSL_KYBER_90S_LEVEL5},
    {XSTR_SIZEOF("P256_NTRU_HPS_LEVEL1"), "P256_NTRU_HPS_LEVEL1", WOLFSSL_P256_NTRU_HPS_LEVEL1},
    {XSTR_SIZEOF("P384_NTRU_HPS_LEVEL3"), "P384_NTRU_HPS_LEVEL3", WOLFSSL_P384_NTRU_HPS_LEVEL3},
    {XSTR_SIZEOF("P521_NTRU_HPS_LEVEL5"), "P521_NTRU_HPS_LEVEL5", WOLFSSL_P521_NTRU_HPS_LEVEL5},
    {XSTR_SIZEOF("P384_NTRU_HRSS_LEVEL3"), "P384_NTRU_HRSS_LEVEL3", WOLFSSL_P384_NTRU_HRSS_LEVEL3},
    {XSTR_SIZEOF("P256_SABER_LEVEL1"), "P256_SABER_LEVEL1", WOLFSSL_P256_SABER_LEVEL1},
    {XSTR_SIZEOF("P384_SABER_LEVEL3"), "P384_SABER_LEVEL3", WOLFSSL_P384_SABER_LEVEL3},
    {XSTR_SIZEOF("P521_SABER_LEVEL5"), "P521_SABER_LEVEL5", WOLFSSL_P521_SABER_LEVEL5},
    {XSTR_SIZEOF("P256_KYBER_LEVEL1"), "P256_KYBER_LEVEL1", WOLFSSL_P256_KYBER_LEVEL1},
    {XSTR_SIZEOF("P384_KYBER_LEVEL3"), "P384_KYBER_LEVEL3", WOLFSSL_P384_KYBER_LEVEL3},
    {XSTR_SIZEOF("P521_KYBER_LEVEL5"), "P521_KYBER_LEVEL5", WOLFSSL_P521_KYBER_LEVEL5},
    {XSTR_SIZEOF("P256_KYBER_90S_LEVEL1"), "P256_KYBER_90S_LEVEL1", WOLFSSL_P256_KYBER_90S_LEVEL1},
    {XSTR_SIZEOF("P384_KYBER_90S_LEVEL3"), "P384_KYBER_90S_LEVEL3", WOLFSSL_P384_KYBER_90S_LEVEL3},
    {XSTR_SIZEOF("P521_KYBER_90S_LEVEL5"), "P521_KYBER_90S_LEVEL5", WOLFSSL_P521_KYBER_90S_LEVEL5},
#endif
    {0, NULL, 0},
};
#endif

#if defined(WOLFSSL_RENESAS_TSIP_TLS) || defined(WOLFSSL_RENESAS_SCEPROTECT)
#include <wolfssl/wolfcrypt/port/Renesas/renesas_cmn.h>
#endif

#ifdef WOLFSSL_SESSION_EXPORT
/* Used to import a serialized TLS session.
 * WARNING: buf contains sensitive information about the state and is best to be
 *          encrypted before storing if stored.
 *
 * @param ssl WOLFSSL structure to import the session into
 * @param buf serialized session
 * @param sz  size of buffer 'buf'
 * @return the number of bytes read from buffer 'buf'
 */
int wolfSSL_tls_import(WOLFSSL* ssl, const unsigned char* buf, unsigned int sz)
{
    if (ssl == NULL || buf == NULL) {
        return BAD_FUNC_ARG;
    }
    return wolfSSL_session_import_internal(ssl, buf, sz, WOLFSSL_EXPORT_TLS);
}


/* Used to export a serialized TLS session.
 * WARNING: buf contains sensitive information about the state and is best to be
 *          encrypted before storing if stored.
 *
 * @param ssl WOLFSSL structure to export the session from
 * @param buf output of serialized session
 * @param sz  size in bytes set in 'buf'
 * @return the number of bytes written into buffer 'buf'
 */
int wolfSSL_tls_export(WOLFSSL* ssl, unsigned char* buf, unsigned int* sz)
{
    if (ssl == NULL || sz == NULL) {
        return BAD_FUNC_ARG;
    }
    return wolfSSL_session_export_internal(ssl, buf, sz, WOLFSSL_EXPORT_TLS);
}

#ifdef WOLFSSL_DTLS
int wolfSSL_dtls_import(WOLFSSL* ssl, const unsigned char* buf, unsigned int sz)
{
    WOLFSSL_ENTER("wolfSSL_session_import");

    if (ssl == NULL || buf == NULL) {
        return BAD_FUNC_ARG;
    }

    /* sanity checks on buffer and protocol are done in internal function */
    return wolfSSL_session_import_internal(ssl, buf, sz, WOLFSSL_EXPORT_DTLS);
}


/* Sets the function to call for serializing the session. This function is
 * called right after the handshake is completed. */
int wolfSSL_CTX_dtls_set_export(WOLFSSL_CTX* ctx, wc_dtls_export func)
{

    WOLFSSL_ENTER("wolfSSL_CTX_dtls_set_export");

    /* purposefully allow func to be NULL */
    if (ctx == NULL) {
        return BAD_FUNC_ARG;
    }

    ctx->dtls_export = func;

    return WOLFSSL_SUCCESS;
}


/* Sets the function in WOLFSSL struct to call for serializing the session. This
 * function is called right after the handshake is completed. */
int wolfSSL_dtls_set_export(WOLFSSL* ssl, wc_dtls_export func)
{

    WOLFSSL_ENTER("wolfSSL_dtls_set_export");

    /* purposefully allow func to be NULL */
    if (ssl == NULL) {
        return BAD_FUNC_ARG;
    }

    ssl->dtls_export = func;

    return WOLFSSL_SUCCESS;
}


/* This function allows for directly serializing a session rather than using
 * callbacks. It has less overhead by removing a temporary buffer and gives
 * control over when the session gets serialized. When using callbacks the
 * session is always serialized immediately after the handshake is finished.
 *
 * buf is the argument to contain the serialized session
 * sz  is the size of the buffer passed in
 * ssl is the WOLFSSL struct to serialize
 * returns the size of serialized session on success, 0 on no action, and
 *         negative value on error */
int wolfSSL_dtls_export(WOLFSSL* ssl, unsigned char* buf, unsigned int* sz)
{
    WOLFSSL_ENTER("wolfSSL_dtls_export");

    if (ssl == NULL || sz == NULL) {
        return BAD_FUNC_ARG;
    }

    if (buf == NULL) {
        *sz = MAX_EXPORT_BUFFER;
        return 0;
    }

    /* if not DTLS do nothing */
    if (!ssl->options.dtls) {
        WOLFSSL_MSG("Currently only DTLS export is supported");
        return 0;
    }

    /* copy over keys, options, and dtls state struct */
    return wolfSSL_session_export_internal(ssl, buf, sz, WOLFSSL_EXPORT_DTLS);
}


/* This function is similar to wolfSSL_dtls_export but only exports the portion
 * of the WOLFSSL structure related to the state of the connection, i.e. peer
 * sequence number, epoch, AEAD state etc.
 *
 * buf is the argument to contain the serialized state, if null then set "sz" to
 *     buffer size required
 * sz  is the size of the buffer passed in
 * ssl is the WOLFSSL struct to serialize
 * returns the size of serialized session on success, 0 on no action, and
 *         negative value on error */
int wolfSSL_dtls_export_state_only(WOLFSSL* ssl, unsigned char* buf,
        unsigned int* sz)
{
    WOLFSSL_ENTER("wolfSSL_dtls_export_state_only");

    if (ssl == NULL || sz == NULL) {
        return BAD_FUNC_ARG;
    }

    if (buf == NULL) {
        *sz = MAX_EXPORT_STATE_BUFFER;
        return 0;
    }

    /* if not DTLS do nothing */
    if (!ssl->options.dtls) {
        WOLFSSL_MSG("Currently only DTLS export state is supported");
        return 0;
    }

    /* copy over keys, options, and dtls state struct */
    return wolfSSL_dtls_export_state_internal(ssl, buf, *sz);
}


/* returns 0 on success */
int wolfSSL_send_session(WOLFSSL* ssl)
{
    int ret;
    byte* buf;
    word32 bufSz = MAX_EXPORT_BUFFER;

    WOLFSSL_ENTER("wolfSSL_send_session");

    if (ssl == NULL) {
        return BAD_FUNC_ARG;
    }

    buf = (byte*)XMALLOC(bufSz, ssl->heap, DYNAMIC_TYPE_TMP_BUFFER);
    if (buf == NULL) {
        return MEMORY_E;
    }

    /* if not DTLS do nothing */
    if (!ssl->options.dtls) {
        XFREE(buf, ssl->heap, DYNAMIC_TYPE_TMP_BUFFER);
        WOLFSSL_MSG("Currently only DTLS export is supported");
        return 0;
    }

    /* copy over keys, options, and dtls state struct */
    ret = wolfSSL_session_export_internal(ssl, buf, &bufSz, WOLFSSL_EXPORT_DTLS);
    if (ret < 0) {
        XFREE(buf, ssl->heap, DYNAMIC_TYPE_TMP_BUFFER);
        return ret;
    }

    /* if no error ret has size of buffer */
    ret = ssl->dtls_export(ssl, buf, ret, NULL);
    if (ret != WOLFSSL_SUCCESS) {
        XFREE(buf, ssl->heap, DYNAMIC_TYPE_TMP_BUFFER);
        return ret;
    }

    XFREE(buf, ssl->heap, DYNAMIC_TYPE_TMP_BUFFER);
    return 0;
}
#endif /* WOLFSSL_DTLS */
#endif /* WOLFSSL_SESSION_EXPORT */

/* prevent multiple mutex initializations */
static volatile WOLFSSL_GLOBAL int initRefCount = 0;
static WOLFSSL_GLOBAL wolfSSL_Mutex count_mutex;   /* init ref count mutex */
static WOLFSSL_GLOBAL int count_mutex_valid = 0;

/* Create a new WOLFSSL_CTX struct and return the pointer to created struct.
   WOLFSSL_METHOD pointer passed in is given to ctx to manage.
   This function frees the passed in WOLFSSL_METHOD struct on failure and on
   success is freed when ctx is freed.
 */
WOLFSSL_CTX* wolfSSL_CTX_new_ex(WOLFSSL_METHOD* method, void* heap)
{
    WOLFSSL_CTX* ctx = NULL;

    WOLFSSL_ENTER("wolfSSL_CTX_new_ex");

    if (initRefCount == 0) {
        /* user no longer forced to call Init themselves */
        int ret = wolfSSL_Init();
        if (ret != WOLFSSL_SUCCESS) {
            WOLFSSL_MSG("wolfSSL_Init failed");
            WOLFSSL_LEAVE("WOLFSSL_CTX_new", 0);
            if (method != NULL) {
                XFREE(method, heap, DYNAMIC_TYPE_METHOD);
            }
            return NULL;
        }
    }

    if (method == NULL)
        return ctx;

    ctx = (WOLFSSL_CTX*)XMALLOC(sizeof(WOLFSSL_CTX), heap, DYNAMIC_TYPE_CTX);
    if (ctx) {
        int ret;

        ret = InitSSL_Ctx(ctx, method, heap);
    #ifdef WOLFSSL_STATIC_MEMORY
        if (heap != NULL) {
            ctx->onHeapHint = 1; /* free the memory back to heap when done */
        }
    #endif
        if (ret < 0) {
            WOLFSSL_MSG("Init CTX failed");
            wolfSSL_CTX_free(ctx);
            ctx = NULL;
        }
#if defined(OPENSSL_EXTRA) && defined(WOLFCRYPT_HAVE_SRP) \
                           && !defined(NO_SHA256) && !defined(WC_NO_RNG)
        else {
            ctx->srp = (Srp*)XMALLOC(sizeof(Srp), heap, DYNAMIC_TYPE_SRP);
            if (ctx->srp == NULL){
                WOLFSSL_MSG("Init CTX failed");
                wolfSSL_CTX_free(ctx);
                return NULL;
            }
            XMEMSET(ctx->srp, 0, sizeof(Srp));
        }
#endif
    }
    else {
        WOLFSSL_MSG("Alloc CTX failed, method freed");
        XFREE(method, heap, DYNAMIC_TYPE_METHOD);
    }

#ifdef OPENSSL_COMPATIBLE_DEFAULTS
    if (ctx) {
        wolfSSL_CTX_set_verify(ctx, SSL_VERIFY_NONE, NULL);
        wolfSSL_CTX_set_mode(ctx, SSL_MODE_AUTO_RETRY);
        if (wolfSSL_CTX_set_min_proto_version(ctx,
                SSL3_VERSION) != WOLFSSL_SUCCESS ||
#ifdef HAVE_ANON
                wolfSSL_CTX_allow_anon_cipher(ctx) != WOLFSSL_SUCCESS ||
#endif
                wolfSSL_CTX_set_group_messages(ctx) != WOLFSSL_SUCCESS) {
            WOLFSSL_MSG("Setting OpenSSL CTX defaults failed");
            wolfSSL_CTX_free(ctx);
            ctx = NULL;
        }
    }
#endif

    WOLFSSL_LEAVE("WOLFSSL_CTX_new", 0);
    return ctx;
}


WOLFSSL_ABI
WOLFSSL_CTX* wolfSSL_CTX_new(WOLFSSL_METHOD* method)
{
#ifdef WOLFSSL_HEAP_TEST
    /* if testing the heap hint then set top level CTX to have test value */
    return wolfSSL_CTX_new_ex(method, (void*)WOLFSSL_HEAP_TEST);
#else
    return wolfSSL_CTX_new_ex(method, NULL);
#endif
}

/* increases CTX reference count to track proper time to "free" */
int wolfSSL_CTX_up_ref(WOLFSSL_CTX* ctx)
{
    int refCount = SSL_CTX_RefCount(ctx, 1);
    return ((refCount > 1) ? WOLFSSL_SUCCESS : WOLFSSL_FAILURE);
}

WOLFSSL_ABI
void wolfSSL_CTX_free(WOLFSSL_CTX* ctx)
{
    WOLFSSL_ENTER("SSL_CTX_free");
    if (ctx) {
#if defined(OPENSSL_EXTRA) && defined(WOLFCRYPT_HAVE_SRP) \
&& !defined(NO_SHA256) && !defined(WC_NO_RNG)
        if (ctx->srp != NULL) {
            if (ctx->srp_password != NULL){
                XFREE(ctx->srp_password, ctx->heap, DYNAMIC_TYPE_SRP);
                ctx->srp_password = NULL;
            }
            wc_SrpTerm(ctx->srp);
            XFREE(ctx->srp, ctx->heap, DYNAMIC_TYPE_SRP);
            ctx->srp = NULL;
        }
#endif
        FreeSSL_Ctx(ctx);
    }

    WOLFSSL_LEAVE("SSL_CTX_free", 0);
}


#ifdef HAVE_ENCRYPT_THEN_MAC
/**
 * Sets whether Encrypt-Then-MAC extension can be negotiated against context.
 * The default value: enabled.
 *
 * ctx  SSL/TLS context.
 * set  Whether to allow or not: 1 is allow and 0 is disallow.
 * returns WOLFSSL_SUCCESS
 */
int wolfSSL_CTX_AllowEncryptThenMac(WOLFSSL_CTX *ctx, int set)
{
    ctx->disallowEncThenMac = !set;
    return WOLFSSL_SUCCESS;
}

/**
 * Sets whether Encrypt-Then-MAC extension can be negotiated against context.
 * The default value comes from context.
 *
 * ctx  SSL/TLS context.
 * set  Whether to allow or not: 1 is allow and 0 is disallow.
 * returns WOLFSSL_SUCCESS
 */
int wolfSSL_AllowEncryptThenMac(WOLFSSL *ssl, int set)
{
    ssl->options.disallowEncThenMac = !set;
    return WOLFSSL_SUCCESS;
}
#endif

#ifdef SINGLE_THREADED
/* no locking in single threaded mode, allow a CTX level rng to be shared with
 * WOLFSSL objects, WOLFSSL_SUCCESS on ok */
int wolfSSL_CTX_new_rng(WOLFSSL_CTX* ctx)
{
    WC_RNG* rng;
    int     ret;

    if (ctx == NULL) {
        return BAD_FUNC_ARG;
    }

    rng = (WC_RNG*)XMALLOC(sizeof(WC_RNG), ctx->heap, DYNAMIC_TYPE_RNG);
    if (rng == NULL) {
        return MEMORY_E;
    }

#ifndef HAVE_FIPS
    ret = wc_InitRng_ex(rng, ctx->heap, ctx->devId);
#else
    ret = wc_InitRng(rng);
#endif
    if (ret != 0) {
        XFREE(rng, ctx->heap, DYNAMIC_TYPE_RNG);
        return ret;
    }

    ctx->rng = rng;
    return WOLFSSL_SUCCESS;
}
#endif


WOLFSSL_ABI
WOLFSSL* wolfSSL_new(WOLFSSL_CTX* ctx)
{
    WOLFSSL* ssl = NULL;
    int ret = 0;

    WOLFSSL_ENTER("SSL_new");

    if (ctx == NULL)
        return ssl;

    ssl = (WOLFSSL*) XMALLOC(sizeof(WOLFSSL), ctx->heap, DYNAMIC_TYPE_SSL);
    if (ssl)
        if ( (ret = InitSSL(ssl, ctx, 0)) < 0) {
            FreeSSL(ssl, ctx->heap);
            ssl = 0;
        }

    WOLFSSL_LEAVE("SSL_new", ret);
    (void)ret;

    return ssl;
}


WOLFSSL_ABI
void wolfSSL_free(WOLFSSL* ssl)
{
    WOLFSSL_ENTER("SSL_free");
    if (ssl)
        FreeSSL(ssl, ssl->ctx->heap);
    WOLFSSL_LEAVE("SSL_free", 0);
}


int wolfSSL_is_server(WOLFSSL* ssl)
{
    if (ssl == NULL)
        return BAD_FUNC_ARG;
    return ssl->options.side == WOLFSSL_SERVER_END;
}

#ifdef HAVE_WRITE_DUP

/*
 * Release resources around WriteDup object
 *
 * ssl WOLFSSL object
 *
 * no return, destruction so make best attempt
*/
void FreeWriteDup(WOLFSSL* ssl)
{
    int doFree = 0;

    WOLFSSL_ENTER("FreeWriteDup");

    if (ssl->dupWrite) {
        if (wc_LockMutex(&ssl->dupWrite->dupMutex) == 0) {
            ssl->dupWrite->dupCount--;
            if (ssl->dupWrite->dupCount == 0) {
                doFree = 1;
            } else {
                WOLFSSL_MSG("WriteDup count not zero, no full free");
            }
            wc_UnLockMutex(&ssl->dupWrite->dupMutex);
        }
    }

    if (doFree) {
        WOLFSSL_MSG("Doing WriteDup full free, count to zero");
        wc_FreeMutex(&ssl->dupWrite->dupMutex);
        XFREE(ssl->dupWrite, ssl->heap, DYNAMIC_TYPE_WRITEDUP);
    }
}


/*
 * duplicate existing ssl members into dup needed for writing
 *
 * dup write only WOLFSSL
 * ssl existing WOLFSSL
 *
 * 0 on success
*/
static int DupSSL(WOLFSSL* dup, WOLFSSL* ssl)
{
    /* shared dupWrite setup */
    ssl->dupWrite = (WriteDup*)XMALLOC(sizeof(WriteDup), ssl->heap,
                                       DYNAMIC_TYPE_WRITEDUP);
    if (ssl->dupWrite == NULL) {
        return MEMORY_E;
    }
    XMEMSET(ssl->dupWrite, 0, sizeof(WriteDup));

    if (wc_InitMutex(&ssl->dupWrite->dupMutex) != 0) {
        XFREE(ssl->dupWrite, ssl->heap, DYNAMIC_TYPE_WRITEDUP);
        ssl->dupWrite = NULL;
        return BAD_MUTEX_E;
    }
    ssl->dupWrite->dupCount = 2;    /* both sides have a count to start */
    dup->dupWrite = ssl->dupWrite; /* each side uses */

    /* copy write parts over to dup writer */
    XMEMCPY(&dup->specs,   &ssl->specs,   sizeof(CipherSpecs));
    XMEMCPY(&dup->options, &ssl->options, sizeof(Options));
    XMEMCPY(&dup->keys,    &ssl->keys,    sizeof(Keys));
    XMEMCPY(&dup->encrypt, &ssl->encrypt, sizeof(Ciphers));
    /* dup side now owns encrypt/write ciphers */
    XMEMSET(&ssl->encrypt, 0, sizeof(Ciphers));

    dup->IOCB_WriteCtx = ssl->IOCB_WriteCtx;
    dup->CBIOSend = ssl->CBIOSend;
#ifdef OPENSSL_EXTRA
    dup->cbioFlag = ssl->cbioFlag;
#endif
    dup->wfd    = ssl->wfd;
    dup->wflags = ssl->wflags;
#ifndef WOLFSSL_AEAD_ONLY
    dup->hmac   = ssl->hmac;
#endif
#ifdef HAVE_TRUNCATED_HMAC
    dup->truncated_hmac = ssl->truncated_hmac;
#endif

    /* unique side dup setup */
    dup->dupSide = WRITE_DUP_SIDE;
    ssl->dupSide = READ_DUP_SIDE;

    return 0;
}


/*
 * duplicate a WOLFSSL object post handshake for writing only
 * turn existing object into read only.  Allows concurrent access from two
 * different threads.
 *
 * ssl existing WOLFSSL object
 *
 * return dup'd WOLFSSL object on success
*/
WOLFSSL* wolfSSL_write_dup(WOLFSSL* ssl)
{
    WOLFSSL* dup = NULL;
    int ret = 0;

    (void)ret;
    WOLFSSL_ENTER("wolfSSL_write_dup");

    if (ssl == NULL) {
        return ssl;
    }

    if (ssl->options.handShakeDone == 0) {
        WOLFSSL_MSG("wolfSSL_write_dup called before handshake complete");
        return NULL;
    }

    if (ssl->dupWrite) {
        WOLFSSL_MSG("wolfSSL_write_dup already called once");
        return NULL;
    }

    dup = (WOLFSSL*) XMALLOC(sizeof(WOLFSSL), ssl->ctx->heap, DYNAMIC_TYPE_SSL);
    if (dup) {
        if ( (ret = InitSSL(dup, ssl->ctx, 1)) < 0) {
            FreeSSL(dup, ssl->ctx->heap);
            dup = NULL;
        } else if ( (ret = DupSSL(dup, ssl)) < 0) {
            FreeSSL(dup, ssl->ctx->heap);
            dup = NULL;
        }
    }

    WOLFSSL_LEAVE("wolfSSL_write_dup", ret);

    return dup;
}


/*
 * Notify write dup side of fatal error or close notify
 *
 * ssl WOLFSSL object
 * err Notify err
 *
 * 0 on success
*/
int NotifyWriteSide(WOLFSSL* ssl, int err)
{
    int ret;

    WOLFSSL_ENTER("NotifyWriteSide");

    ret = wc_LockMutex(&ssl->dupWrite->dupMutex);
    if (ret == 0) {
        ssl->dupWrite->dupErr = err;
        ret = wc_UnLockMutex(&ssl->dupWrite->dupMutex);
    }

    return ret;
}


#endif /* HAVE_WRITE_DUP */


#ifdef HAVE_POLY1305
/* set if to use old poly 1 for yes 0 to use new poly */
int wolfSSL_use_old_poly(WOLFSSL* ssl, int value)
{
    (void)ssl;
    (void)value;

#ifndef WOLFSSL_NO_TLS12
    WOLFSSL_ENTER("SSL_use_old_poly");
    WOLFSSL_MSG("Warning SSL connection auto detects old/new and this function"
            "is depreciated");
    ssl->options.oldPoly = (word16)value;
    WOLFSSL_LEAVE("SSL_use_old_poly", 0);
#endif
    return 0;
}
#endif


WOLFSSL_ABI
int wolfSSL_set_fd(WOLFSSL* ssl, int fd)
{
    int ret;

    WOLFSSL_ENTER("SSL_set_fd");

    if (ssl == NULL) {
        return BAD_FUNC_ARG;
    }

    ret = wolfSSL_set_read_fd(ssl, fd);
    if (ret == WOLFSSL_SUCCESS) {
        ret = wolfSSL_set_write_fd(ssl, fd);
    }

    return ret;
}


int wolfSSL_set_read_fd(WOLFSSL* ssl, int fd)
{
    WOLFSSL_ENTER("SSL_set_read_fd");

    if (ssl == NULL) {
        return BAD_FUNC_ARG;
    }

    ssl->rfd = fd;      /* not used directly to allow IO callbacks */
    ssl->IOCB_ReadCtx  = &ssl->rfd;

    #ifdef WOLFSSL_DTLS
        if (ssl->options.dtls) {
            ssl->IOCB_ReadCtx = &ssl->buffers.dtlsCtx;
            ssl->buffers.dtlsCtx.rfd = fd;
        }
    #endif

    WOLFSSL_LEAVE("SSL_set_read_fd", WOLFSSL_SUCCESS);
    return WOLFSSL_SUCCESS;
}


int wolfSSL_set_write_fd(WOLFSSL* ssl, int fd)
{
    WOLFSSL_ENTER("SSL_set_write_fd");

    if (ssl == NULL) {
        return BAD_FUNC_ARG;
    }

    ssl->wfd = fd;      /* not used directly to allow IO callbacks */
    ssl->IOCB_WriteCtx  = &ssl->wfd;

    #ifdef WOLFSSL_DTLS
        if (ssl->options.dtls) {
            ssl->IOCB_WriteCtx = &ssl->buffers.dtlsCtx;
            ssl->buffers.dtlsCtx.wfd = fd;
        }
    #endif

    WOLFSSL_LEAVE("SSL_set_write_fd", WOLFSSL_SUCCESS);
    return WOLFSSL_SUCCESS;
}


/**
  * Get the name of cipher at priority level passed in.
  */
char* wolfSSL_get_cipher_list(int priority)
{
    const CipherSuiteInfo* ciphers = GetCipherNames();

    if (priority >= GetCipherNamesSize() || priority < 0) {
        return 0;
    }

    return (char*)ciphers[priority].name;
}


/**
  * Get the name of cipher at priority level passed in.
  */
char* wolfSSL_get_cipher_list_ex(WOLFSSL* ssl, int priority)
{

    if (ssl == NULL) {
        return NULL;
    }
    else {
        const char* cipher;

        if ((cipher = wolfSSL_get_cipher_name_internal(ssl)) != NULL) {
            if (priority == 0) {
                return (char*)cipher;
            }
            else {
                return NULL;
            }
        }
        else {
            return wolfSSL_get_cipher_list(priority);
        }
    }
}


int wolfSSL_get_ciphers(char* buf, int len)
{
    const CipherSuiteInfo* ciphers = GetCipherNames();
    int ciphersSz = GetCipherNamesSize();
    int i;
    int cipherNameSz;

    if (buf == NULL || len <= 0)
        return BAD_FUNC_ARG;

    /* Add each member to the buffer delimited by a : */
    for (i = 0; i < ciphersSz; i++) {
        cipherNameSz = (int)XSTRLEN(ciphers[i].name);
        if (cipherNameSz + 1 < len) {
            XSTRNCPY(buf, ciphers[i].name, len);
            buf += cipherNameSz;

            if (i < ciphersSz - 1)
                *buf++ = ':';
            *buf = 0;

            len -= cipherNameSz + 1;
        }
        else
            return BUFFER_E;
    }
    return WOLFSSL_SUCCESS;
}


#ifndef NO_ERROR_STRINGS
/* places a list of all supported cipher suites in TLS_* format into "buf"
 * return WOLFSSL_SUCCESS on success */
int wolfSSL_get_ciphers_iana(char* buf, int len)
{
    const CipherSuiteInfo* ciphers = GetCipherNames();
    int ciphersSz = GetCipherNamesSize();
    int i;
    int cipherNameSz;

    if (buf == NULL || len <= 0)
        return BAD_FUNC_ARG;

    /* Add each member to the buffer delimited by a : */
    for (i = 0; i < ciphersSz; i++) {
#ifndef NO_CIPHER_SUITE_ALIASES
        if (ciphers[i].flags & WOLFSSL_CIPHER_SUITE_FLAG_NAMEALIAS)
            continue;
#endif
        cipherNameSz = (int)XSTRLEN(ciphers[i].name_iana);
        if (cipherNameSz + 1 < len) {
            XSTRNCPY(buf, ciphers[i].name_iana, len);
            buf += cipherNameSz;

            if (i < ciphersSz - 1)
                *buf++ = ':';
            *buf = 0;

            len -= cipherNameSz + 1;
        }
        else
            return BUFFER_E;
    }
    return WOLFSSL_SUCCESS;
}
#endif /* NO_ERROR_STRINGS */


const char* wolfSSL_get_shared_ciphers(WOLFSSL* ssl, char* buf, int len)
{
    const char* cipher;

    if (ssl == NULL)
        return NULL;

    cipher = wolfSSL_get_cipher_name_iana(ssl);
    len = min(len, (int)(XSTRLEN(cipher) + 1));
    XMEMCPY(buf, cipher, len);
    return buf;
}

int wolfSSL_get_fd(const WOLFSSL* ssl)
{
    int fd = -1;
    WOLFSSL_ENTER("SSL_get_fd");
    if (ssl) {
        fd = ssl->rfd;
    }
    WOLFSSL_LEAVE("SSL_get_fd", fd);
    return fd;
}


int wolfSSL_dtls(WOLFSSL* ssl)
{
    int dtlsOpt = 0;
    if (ssl)
        dtlsOpt = ssl->options.dtls;
    return dtlsOpt;
}

#if !defined(NO_CERTS)
/* Set whether mutual authentication is required for connections.
 * Server side only.
 *
 * ctx  The SSL/TLS CTX object.
 * req  1 to indicate required and 0 when not.
 * returns BAD_FUNC_ARG when ctx is NULL, SIDE_ERROR when not a server and
 * 0 on success.
 */
int wolfSSL_CTX_mutual_auth(WOLFSSL_CTX* ctx, int req)
{
    if (ctx == NULL)
        return BAD_FUNC_ARG;
    if (ctx->method->side == WOLFSSL_CLIENT_END)
        return SIDE_ERROR;

    ctx->mutualAuth = (byte)req;

    return 0;
}

/* Set whether mutual authentication is required for the connection.
 * Server side only.
 *
 * ssl  The SSL/TLS object.
 * req  1 to indicate required and 0 when not.
 * returns BAD_FUNC_ARG when ssl is NULL, or not using TLS v1.3,
 * SIDE_ERROR when not a client and 0 on success.
 */
int wolfSSL_mutual_auth(WOLFSSL* ssl, int req)
{
    if (ssl == NULL)
        return BAD_FUNC_ARG;
    if (ssl->options.side == WOLFSSL_SERVER_END)
        return SIDE_ERROR;

    ssl->options.mutualAuth = (word16)req;

    return 0;
}
#endif /* NO_CERTS */

#ifdef WOLFSSL_WOLFSENTRY_HOOKS

int wolfSSL_CTX_set_AcceptFilter(
    WOLFSSL_CTX *ctx,
    NetworkFilterCallback_t AcceptFilter,
    void *AcceptFilter_arg)
{
    if (ctx == NULL)
        return BAD_FUNC_ARG;
    ctx->AcceptFilter = AcceptFilter;
    ctx->AcceptFilter_arg = AcceptFilter_arg;
    return 0;
}

int wolfSSL_set_AcceptFilter(
    WOLFSSL *ssl,
    NetworkFilterCallback_t AcceptFilter,
    void *AcceptFilter_arg)
{
    if (ssl == NULL)
        return BAD_FUNC_ARG;
    ssl->AcceptFilter = AcceptFilter;
    ssl->AcceptFilter_arg = AcceptFilter_arg;
    return 0;
}

int wolfSSL_CTX_set_ConnectFilter(
    WOLFSSL_CTX *ctx,
    NetworkFilterCallback_t ConnectFilter,
    void *ConnectFilter_arg)
{
    if (ctx == NULL)
        return BAD_FUNC_ARG;
    ctx->ConnectFilter = ConnectFilter;
    ctx->ConnectFilter_arg = ConnectFilter_arg;
    return 0;
}

int wolfSSL_set_ConnectFilter(
    WOLFSSL *ssl,
    NetworkFilterCallback_t ConnectFilter,
    void *ConnectFilter_arg)
{
    if (ssl == NULL)
        return BAD_FUNC_ARG;
    ssl->ConnectFilter = ConnectFilter;
    ssl->ConnectFilter_arg = ConnectFilter_arg;
    return 0;
}

#endif /* WOLFSSL_WOLFSENTRY_HOOKS */

#ifndef WOLFSSL_LEANPSK
int wolfSSL_dtls_set_peer(WOLFSSL* ssl, void* peer, unsigned int peerSz)
{
#ifdef WOLFSSL_DTLS
    void* sa;

    if (ssl == NULL)
        return WOLFSSL_FAILURE;

    sa = (void*)XMALLOC(peerSz, ssl->heap, DYNAMIC_TYPE_SOCKADDR);
    if (sa != NULL) {
        if (ssl->buffers.dtlsCtx.peer.sa != NULL) {
            XFREE(ssl->buffers.dtlsCtx.peer.sa,ssl->heap,DYNAMIC_TYPE_SOCKADDR);
            ssl->buffers.dtlsCtx.peer.sa = NULL;
        }
        XMEMCPY(sa, peer, peerSz);
        ssl->buffers.dtlsCtx.peer.sa = sa;
        ssl->buffers.dtlsCtx.peer.sz = peerSz;
        return WOLFSSL_SUCCESS;
    }
    return WOLFSSL_FAILURE;
#else
    (void)ssl;
    (void)peer;
    (void)peerSz;
    return WOLFSSL_NOT_IMPLEMENTED;
#endif
}

int wolfSSL_dtls_get_peer(WOLFSSL* ssl, void* peer, unsigned int* peerSz)
{
#ifdef WOLFSSL_DTLS
    if (ssl == NULL) {
        return WOLFSSL_FAILURE;
    }

    if (peer != NULL && peerSz != NULL
            && *peerSz >= ssl->buffers.dtlsCtx.peer.sz
            && ssl->buffers.dtlsCtx.peer.sa != NULL) {
        *peerSz = ssl->buffers.dtlsCtx.peer.sz;
        XMEMCPY(peer, ssl->buffers.dtlsCtx.peer.sa, *peerSz);
        return WOLFSSL_SUCCESS;
    }
    return WOLFSSL_FAILURE;
#else
    (void)ssl;
    (void)peer;
    (void)peerSz;
    return WOLFSSL_NOT_IMPLEMENTED;
#endif
}


#if defined(WOLFSSL_SCTP) && defined(WOLFSSL_DTLS)

int wolfSSL_CTX_dtls_set_sctp(WOLFSSL_CTX* ctx)
{
    WOLFSSL_ENTER("wolfSSL_CTX_dtls_set_sctp()");

    if (ctx == NULL)
        return BAD_FUNC_ARG;

    ctx->dtlsSctp = 1;
    return WOLFSSL_SUCCESS;
}


int wolfSSL_dtls_set_sctp(WOLFSSL* ssl)
{
    WOLFSSL_ENTER("wolfSSL_dtls_set_sctp()");

    if (ssl == NULL)
        return BAD_FUNC_ARG;

    ssl->options.dtlsSctp = 1;
    return WOLFSSL_SUCCESS;
}

#endif /* WOLFSSL_DTLS && WOLFSSL_SCTP */

#if (defined(WOLFSSL_SCTP) || defined(WOLFSSL_DTLS_MTU)) && \
                                                           defined(WOLFSSL_DTLS)

int wolfSSL_CTX_dtls_set_mtu(WOLFSSL_CTX* ctx, word16 newMtu)
{
    if (ctx == NULL || newMtu > MAX_RECORD_SIZE)
        return BAD_FUNC_ARG;

    ctx->dtlsMtuSz = newMtu;
    return WOLFSSL_SUCCESS;
}


int wolfSSL_dtls_set_mtu(WOLFSSL* ssl, word16 newMtu)
{
    if (ssl == NULL)
        return BAD_FUNC_ARG;

    if (newMtu > MAX_RECORD_SIZE) {
        ssl->error = BAD_FUNC_ARG;
        return WOLFSSL_FAILURE;
    }

    ssl->dtlsMtuSz = newMtu;
    return WOLFSSL_SUCCESS;
}

#endif /* WOLFSSL_DTLS && (WOLFSSL_SCTP || WOLFSSL_DTLS_MTU) */

#ifdef WOLFSSL_SRTP

static const WOLFSSL_SRTP_PROTECTION_PROFILE gSrtpProfiles[] = {
    /* AES CCM 128, Salt:112-bits, Auth HMAC-SHA1 Tag: 80-bits
     * (master_key:128bits + master_salt:112bits) * 2 = 480 bits (60) */
    {"SRTP_AES128_CM_SHA1_80", SRTP_AES128_CM_SHA1_80, (((128 + 112) * 2) / 8) },
    /* AES CCM 128, Salt:112-bits, Auth HMAC-SHA1 Tag: 32-bits
     * (master_key:128bits + master_salt:112bits) * 2 = 480 bits (60) */
    {"SRTP_AES128_CM_SHA1_32", SRTP_AES128_CM_SHA1_32, (((128 + 112) * 2) / 8) },
    /* NULL Cipher, Salt:112-bits, Auth HMAC-SHA1 Tag 80-bits */
    {"SRTP_NULL_SHA1_80", SRTP_NULL_SHA1_80, ((112 * 2) / 8)},
    /* NULL Cipher, Salt:112-bits, Auth HMAC-SHA1 Tag 32-bits */
    {"SRTP_NULL_SHA1_32", SRTP_NULL_SHA1_32, ((112 * 2) / 8)},
    /* AES GCM 128, Salt: 96-bits, Auth GCM Tag 128-bits
     * (master_key:128bits + master_salt:96bits) * 2 = 448 bits (56) */
    {"SRTP_AEAD_AES_128_GCM", SRTP_AEAD_AES_128_GCM, (((128 + 96) * 2) / 8) },
    /* AES GCM 256, Salt: 96-bits, Auth GCM Tag 128-bits
     * (master_key:256bits + master_salt:96bits) * 2 = 704 bits (88) */
    {"SRTP_AEAD_AES_256_GCM", SRTP_AEAD_AES_256_GCM, (((256 + 96) * 2) / 8) },
};

static const WOLFSSL_SRTP_PROTECTION_PROFILE* DtlsSrtpFindProfile(
    const char* profile_str, word32 profile_str_len, unsigned long id)
{
    int i;
    const WOLFSSL_SRTP_PROTECTION_PROFILE* profile = NULL;
    for (i=0;
         i<(int)(sizeof(gSrtpProfiles)/sizeof(WOLFSSL_SRTP_PROTECTION_PROFILE));
         i++) {
        if (profile_str != NULL) {
            word32 srtp_profile_len = (word32)XSTRLEN(gSrtpProfiles[i].name);
            if (srtp_profile_len == profile_str_len &&
                XMEMCMP(gSrtpProfiles[i].name, profile_str, profile_str_len)
                                                                         == 0) {
                profile = &gSrtpProfiles[i];
                break;
            }
        }
        else if (id != 0 && gSrtpProfiles[i].id == id) {
            profile = &gSrtpProfiles[i];
            break;
        }
    }
    return profile;
}

/* profile_str: accepts ":" colon separated list of SRTP profiles */
static int DtlsSrtpSelProfiles(word16* id, const char* profile_str)
{
    const WOLFSSL_SRTP_PROTECTION_PROFILE* profile;
    const char *current, *next = NULL;
    word32 length = 0, current_length;

    *id = 0; /* reset destination ID's */

    if (profile_str == NULL) {
        return WOLFSSL_FAILURE;
    }

    /* loop on end of line or colon ":" */
    next = profile_str;
    length = (word32)XSTRLEN(profile_str);
    do {
        current = next;
        next = XSTRSTR(current, ":");
        current_length = (!next) ? (word32)XSTRLEN(current)
                                 : (word32)(next - current);
        if (current_length < length)
            length = current_length;
        profile = DtlsSrtpFindProfile(current, current_length, 0);
        if (profile != NULL) {
            *id |= (1 << profile->id); /* selected bit based on ID */
        }
    } while (next != NULL && next++); /* ++ needed to skip ':' */
    return WOLFSSL_SUCCESS;
}

int wolfSSL_CTX_set_tlsext_use_srtp(WOLFSSL_CTX* ctx, const char* profile_str)
{
    int ret = WOLFSSL_FAILURE;
    if (ctx != NULL) {
        ret = DtlsSrtpSelProfiles(&ctx->dtlsSrtpProfiles, profile_str);
    }
    return ret;
}
int wolfSSL_set_tlsext_use_srtp(WOLFSSL* ssl, const char* profile_str)
{
    int ret = WOLFSSL_FAILURE;
    if (ssl != NULL) {
        ret = DtlsSrtpSelProfiles(&ssl->dtlsSrtpProfiles, profile_str);
    }
    return ret;
}

const WOLFSSL_SRTP_PROTECTION_PROFILE* wolfSSL_get_selected_srtp_profile(
    WOLFSSL* ssl)
{
    const WOLFSSL_SRTP_PROTECTION_PROFILE* profile = NULL;
    if (ssl) {
        profile = DtlsSrtpFindProfile(NULL, 0, ssl->dtlsSrtpId);
    }
    return profile;
}
#ifndef NO_WOLFSSL_STUB
WOLF_STACK_OF(WOLFSSL_SRTP_PROTECTION_PROFILE)* wolfSSL_get_srtp_profiles(
    WOLFSSL* ssl)
{
    /* Not yet implemented - should return list of available SRTP profiles
     * ssl->dtlsSrtpProfiles */
    (void)ssl;
    return NULL;
}
#endif

int wolfSSL_export_dtls_srtp_keying_material(WOLFSSL* ssl,
    unsigned char* out, size_t* olen)
{
    int ret = WOLFSSL_FAILURE;
    const char* label = "EXTRACTOR-dtls_srtp";
    const WOLFSSL_SRTP_PROTECTION_PROFILE* profile = NULL;
    byte seed[SEED_LEN];

    if (ssl == NULL || olen == NULL) {
        return BAD_FUNC_ARG;
    }

    profile = DtlsSrtpFindProfile(NULL, 0, ssl->dtlsSrtpId);
    if (profile == NULL) {
        WOLFSSL_MSG("Not using DTLS SRTP");
        return EXT_MISSING;
    }
    if (out == NULL) {
        *olen = profile->kdfBits;
        return LENGTH_ONLY_E;
    }

    if (*olen < (size_t)profile->kdfBits) {
        return BUFFER_E;
    }

#ifdef WOLFSSL_HAVE_PRF
    XMEMCPY(seed, ssl->arrays->clientRandom, RAN_LEN);
    XMEMCPY(seed + RAN_LEN, ssl->arrays->serverRandom, RAN_LEN);

    PRIVATE_KEY_UNLOCK();
    ret = wc_PRF_TLS(out, profile->kdfBits,   /* out: generated keys / salt */
        ssl->arrays->masterSecret, SECRET_LEN,  /* existing master secret */
        (const byte*)label, (int)XSTRLEN(label),/* label */
        seed, SEED_LEN,                         /* seed: client/server random */
        IsAtLeastTLSv1_2(ssl), ssl->specs.mac_algorithm,
        ssl->heap, INVALID_DEVID);
    if (ret == 0) {
        *olen = profile->kdfBits;
        ret = WOLFSSL_SUCCESS;
    }
    PRIVATE_KEY_LOCK();
#else
    /* Pseudo random function must be enabled in the configuration */
    ret = PRF_MISSING;
#endif

    return ret;
}

#endif /* WOLFSSL_SRTP */


#ifdef WOLFSSL_DTLS_DROP_STATS

int wolfSSL_dtls_get_drop_stats(WOLFSSL* ssl,
                                word32* macDropCount, word32* replayDropCount)
{
    int ret;

    WOLFSSL_ENTER("wolfSSL_dtls_get_drop_stats()");

    if (ssl == NULL)
        ret = BAD_FUNC_ARG;
    else {
        ret = WOLFSSL_SUCCESS;
        if (macDropCount != NULL)
            *macDropCount = ssl->macDropCount;
        if (replayDropCount != NULL)
            *replayDropCount = ssl->replayDropCount;
    }

    WOLFSSL_LEAVE("wolfSSL_dtls_get_drop_stats()", ret);
    return ret;
}

#endif /* WOLFSSL_DTLS_DROP_STATS */


#if defined(WOLFSSL_MULTICAST)

int wolfSSL_CTX_mcast_set_member_id(WOLFSSL_CTX* ctx, word16 id)
{
    int ret = 0;

    WOLFSSL_ENTER("wolfSSL_CTX_mcast_set_member_id()");

    if (ctx == NULL || id > 255)
        ret = BAD_FUNC_ARG;

    if (ret == 0) {
        ctx->haveEMS = 0;
        ctx->haveMcast = 1;
        ctx->mcastID = (byte)id;
#ifndef WOLFSSL_USER_IO
        ctx->CBIORecv = EmbedReceiveFromMcast;
#endif /* WOLFSSL_USER_IO */

        ret = WOLFSSL_SUCCESS;
    }
    WOLFSSL_LEAVE("wolfSSL_CTX_mcast_set_member_id()", ret);
    return ret;
}

int wolfSSL_mcast_get_max_peers(void)
{
    return WOLFSSL_MULTICAST_PEERS;
}

#ifdef WOLFSSL_DTLS
static WC_INLINE word32 UpdateHighwaterMark(word32 cur, word32 first,
                                         word32 second, word32 high)
{
    word32 newCur = 0;

    if (cur < first)
        newCur = first;
    else if (cur < second)
        newCur = second;
    else if (cur < high)
        newCur = high;

    return newCur;
}
#endif /* WOLFSSL_DTLS */


int wolfSSL_set_secret(WOLFSSL* ssl, word16 epoch,
                       const byte* preMasterSecret, word32 preMasterSz,
                       const byte* clientRandom, const byte* serverRandom,
                       const byte* suite)
{
    int ret = 0;

    WOLFSSL_ENTER("wolfSSL_set_secret()");

    if (ssl == NULL || preMasterSecret == NULL ||
        preMasterSz == 0 || preMasterSz > ENCRYPT_LEN ||
        clientRandom == NULL || serverRandom == NULL || suite == NULL) {

        ret = BAD_FUNC_ARG;
    }

    if (ret == 0 && ssl->arrays->preMasterSecret == NULL) {
        ssl->arrays->preMasterSz = ENCRYPT_LEN;
        ssl->arrays->preMasterSecret = (byte*)XMALLOC(ENCRYPT_LEN, ssl->heap,
            DYNAMIC_TYPE_SECRET);
        if (ssl->arrays->preMasterSecret == NULL) {
            ret = MEMORY_E;
        }
    }

    if (ret == 0) {
        XMEMCPY(ssl->arrays->preMasterSecret, preMasterSecret, preMasterSz);
        XMEMSET(ssl->arrays->preMasterSecret + preMasterSz, 0, ENCRYPT_LEN - preMasterSz);
        ssl->arrays->preMasterSz = preMasterSz;
        XMEMCPY(ssl->arrays->clientRandom, clientRandom, RAN_LEN);
        XMEMCPY(ssl->arrays->serverRandom, serverRandom, RAN_LEN);
        ssl->options.cipherSuite0 = suite[0];
        ssl->options.cipherSuite = suite[1];

        ret = SetCipherSpecs(ssl);
    }

    if (ret == 0)
        ret = MakeTlsMasterSecret(ssl);

    if (ret == 0) {
        ssl->keys.encryptionOn = 1;
        ret = SetKeysSide(ssl, ENCRYPT_AND_DECRYPT_SIDE);
    }

    if (ret == 0) {
        if (ssl->options.dtls) {
        #ifdef WOLFSSL_DTLS
            WOLFSSL_DTLS_PEERSEQ* peerSeq;
            int i;

            ssl->keys.dtls_epoch = epoch;
            for (i = 0, peerSeq = ssl->keys.peerSeq;
                 i < WOLFSSL_DTLS_PEERSEQ_SZ;
                 i++, peerSeq++) {

                peerSeq->nextEpoch = epoch;
                peerSeq->prevSeq_lo = peerSeq->nextSeq_lo;
                peerSeq->prevSeq_hi = peerSeq->nextSeq_hi;
                peerSeq->nextSeq_lo = 0;
                peerSeq->nextSeq_hi = 0;
                XMEMCPY(peerSeq->prevWindow, peerSeq->window, DTLS_SEQ_SZ);
                XMEMSET(peerSeq->window, 0, DTLS_SEQ_SZ);
                peerSeq->highwaterMark = UpdateHighwaterMark(0,
                        ssl->ctx->mcastFirstSeq,
                        ssl->ctx->mcastSecondSeq,
                        ssl->ctx->mcastMaxSeq);
            }
        #else
            (void)epoch;
        #endif
        }
        FreeHandshakeResources(ssl);
        ret = WOLFSSL_SUCCESS;
    }
    else {
        if (ssl)
            ssl->error = ret;
        ret = WOLFSSL_FATAL_ERROR;
    }
    WOLFSSL_LEAVE("wolfSSL_set_secret()", ret);
    return ret;
}


#ifdef WOLFSSL_DTLS

int wolfSSL_mcast_peer_add(WOLFSSL* ssl, word16 peerId, int sub)
{
    WOLFSSL_DTLS_PEERSEQ* p = NULL;
    int ret = WOLFSSL_SUCCESS;
    int i;

    WOLFSSL_ENTER("wolfSSL_mcast_peer_add()");
    if (ssl == NULL || peerId > 255)
        return BAD_FUNC_ARG;

    if (!sub) {
        /* Make sure it isn't already present, while keeping the first
         * open spot. */
        for (i = 0; i < WOLFSSL_DTLS_PEERSEQ_SZ; i++) {
            if (ssl->keys.peerSeq[i].peerId == INVALID_PEER_ID)
                p = &ssl->keys.peerSeq[i];
            if (ssl->keys.peerSeq[i].peerId == peerId) {
                WOLFSSL_MSG("Peer ID already in multicast peer list.");
                p = NULL;
            }
        }

        if (p != NULL) {
            XMEMSET(p, 0, sizeof(WOLFSSL_DTLS_PEERSEQ));
            p->peerId = peerId;
            p->highwaterMark = UpdateHighwaterMark(0,
                ssl->ctx->mcastFirstSeq,
                ssl->ctx->mcastSecondSeq,
                ssl->ctx->mcastMaxSeq);
        }
        else {
            WOLFSSL_MSG("No room in peer list.");
            ret = -1;
        }
    }
    else {
        for (i = 0; i < WOLFSSL_DTLS_PEERSEQ_SZ; i++) {
            if (ssl->keys.peerSeq[i].peerId == peerId)
                p = &ssl->keys.peerSeq[i];
        }

        if (p != NULL) {
            p->peerId = INVALID_PEER_ID;
        }
        else {
            WOLFSSL_MSG("Peer not found in list.");
        }
    }

    WOLFSSL_LEAVE("wolfSSL_mcast_peer_add()", ret);
    return ret;
}


/* If peerId is in the list of peers and its last sequence number is non-zero,
 * return 1, otherwise return 0. */
int wolfSSL_mcast_peer_known(WOLFSSL* ssl, unsigned short peerId)
{
    int known = 0;
    int i;

    WOLFSSL_ENTER("wolfSSL_mcast_peer_known()");

    if (ssl == NULL || peerId > 255) {
        return BAD_FUNC_ARG;
    }

    for (i = 0; i < WOLFSSL_DTLS_PEERSEQ_SZ; i++) {
        if (ssl->keys.peerSeq[i].peerId == peerId) {
            if (ssl->keys.peerSeq[i].nextSeq_hi ||
                ssl->keys.peerSeq[i].nextSeq_lo) {

                known = 1;
            }
            break;
        }
    }

    WOLFSSL_LEAVE("wolfSSL_mcast_peer_known()", known);
    return known;
}


int wolfSSL_CTX_mcast_set_highwater_cb(WOLFSSL_CTX* ctx, word32 maxSeq,
                                       word32 first, word32 second,
                                       CallbackMcastHighwater cb)
{
    if (ctx == NULL || (second && first > second) ||
        first > maxSeq || second > maxSeq || cb == NULL) {

        return BAD_FUNC_ARG;
    }

    ctx->mcastHwCb = cb;
    ctx->mcastFirstSeq = first;
    ctx->mcastSecondSeq = second;
    ctx->mcastMaxSeq = maxSeq;

    return WOLFSSL_SUCCESS;
}


int wolfSSL_mcast_set_highwater_ctx(WOLFSSL* ssl, void* ctx)
{
    if (ssl == NULL || ctx == NULL)
        return BAD_FUNC_ARG;

    ssl->mcastHwCbCtx = ctx;

    return WOLFSSL_SUCCESS;
}

#endif /* WOLFSSL_DTLS */

#endif /* WOLFSSL_MULTICAST */


#endif /* WOLFSSL_LEANPSK */


/* return underlying connect or accept, WOLFSSL_SUCCESS on ok */
int wolfSSL_negotiate(WOLFSSL* ssl)
{
    int err = WOLFSSL_FATAL_ERROR;

    WOLFSSL_ENTER("wolfSSL_negotiate");
#ifndef NO_WOLFSSL_SERVER
    if (ssl->options.side == WOLFSSL_SERVER_END) {
#ifdef WOLFSSL_TLS13
        if (IsAtLeastTLSv1_3(ssl->version))
            err = wolfSSL_accept_TLSv13(ssl);
        else
#endif
            err = wolfSSL_accept(ssl);
    }
#endif

#ifndef NO_WOLFSSL_CLIENT
    if (ssl->options.side == WOLFSSL_CLIENT_END) {
#ifdef WOLFSSL_TLS13
        if (IsAtLeastTLSv1_3(ssl->version))
            err = wolfSSL_connect_TLSv13(ssl);
        else
#endif
            err = wolfSSL_connect(ssl);
    }
#endif

    (void)ssl;

    WOLFSSL_LEAVE("wolfSSL_negotiate", err);

    return err;
}


WOLFSSL_ABI
WC_RNG* wolfSSL_GetRNG(WOLFSSL* ssl)
{
    if (ssl) {
        return ssl->rng;
    }

    return NULL;
}


#ifndef WOLFSSL_LEANPSK
/* object size based on build */
int wolfSSL_GetObjectSize(void)
{
#ifdef SHOW_SIZES
    printf("sizeof suites           = %lu\n", (unsigned long)sizeof(Suites));
    printf("sizeof ciphers(2)       = %lu\n", (unsigned long)sizeof(Ciphers));
#ifndef NO_RC4
    printf("\tsizeof arc4         = %lu\n", (unsigned long)sizeof(Arc4));
#endif
    printf("\tsizeof aes          = %lu\n", (unsigned long)sizeof(Aes));
#ifndef NO_DES3
    printf("\tsizeof des3         = %lu\n", (unsigned long)sizeof(Des3));
#endif
#ifdef HAVE_CHACHA
    printf("\tsizeof chacha       = %lu\n", (unsigned long)sizeof(ChaCha));
#endif
    printf("sizeof cipher specs     = %lu\n", (unsigned long)sizeof(CipherSpecs));
    printf("sizeof keys             = %lu\n", (unsigned long)sizeof(Keys));
    printf("sizeof Hashes(2)        = %lu\n", (unsigned long)sizeof(Hashes));
#ifndef NO_MD5
    printf("\tsizeof MD5          = %lu\n", (unsigned long)sizeof(wc_Md5));
#endif
#ifndef NO_SHA
    printf("\tsizeof SHA          = %lu\n", (unsigned long)sizeof(wc_Sha));
#endif
#ifdef WOLFSSL_SHA224
    printf("\tsizeof SHA224       = %lu\n", (unsigned long)sizeof(wc_Sha224));
#endif
#ifndef NO_SHA256
    printf("\tsizeof SHA256       = %lu\n", (unsigned long)sizeof(wc_Sha256));
#endif
#ifdef WOLFSSL_SHA384
    printf("\tsizeof SHA384       = %lu\n", (unsigned long)sizeof(wc_Sha384));
#endif
#ifdef WOLFSSL_SHA384
    printf("\tsizeof SHA512       = %lu\n", (unsigned long)sizeof(wc_Sha512));
#endif
    printf("sizeof Buffers          = %lu\n", (unsigned long)sizeof(Buffers));
    printf("sizeof Options          = %lu\n", (unsigned long)sizeof(Options));
    printf("sizeof Arrays           = %lu\n", (unsigned long)sizeof(Arrays));
#ifndef NO_RSA
    printf("sizeof RsaKey           = %lu\n", (unsigned long)sizeof(RsaKey));
#endif
#ifdef HAVE_ECC
    printf("sizeof ecc_key          = %lu\n", (unsigned long)sizeof(ecc_key));
#endif
    printf("sizeof WOLFSSL_CIPHER    = %lu\n", (unsigned long)sizeof(WOLFSSL_CIPHER));
    printf("sizeof WOLFSSL_SESSION   = %lu\n", (unsigned long)sizeof(WOLFSSL_SESSION));
    printf("sizeof WOLFSSL           = %lu\n", (unsigned long)sizeof(WOLFSSL));
    printf("sizeof WOLFSSL_CTX       = %lu\n", (unsigned long)sizeof(WOLFSSL_CTX));
#endif

    return sizeof(WOLFSSL);
}

int wolfSSL_CTX_GetObjectSize(void)
{
    return sizeof(WOLFSSL_CTX);
}

int wolfSSL_METHOD_GetObjectSize(void)
{
    return sizeof(WOLFSSL_METHOD);
}
#endif


#ifdef WOLFSSL_STATIC_MEMORY

int wolfSSL_CTX_load_static_memory(WOLFSSL_CTX** ctx, wolfSSL_method_func method,
                                   unsigned char* buf, unsigned int sz,
                                   int flag, int maxSz)
{
    WOLFSSL_HEAP*      heap;
    WOLFSSL_HEAP_HINT* hint;
    word32 idx = 0;

    if (ctx == NULL || buf == NULL) {
        return BAD_FUNC_ARG;
    }

    if (*ctx == NULL && method == NULL) {
        return BAD_FUNC_ARG;
    }

    if (*ctx == NULL || (*ctx)->heap == NULL) {
        if (sizeof(WOLFSSL_HEAP) + sizeof(WOLFSSL_HEAP_HINT) > sz - idx) {
            return BUFFER_E; /* not enough memory for structures */
        }
        heap = (WOLFSSL_HEAP*)buf;
        idx += sizeof(WOLFSSL_HEAP);
        if (wolfSSL_init_memory_heap(heap) != 0) {
            return WOLFSSL_FAILURE;
        }
        hint = (WOLFSSL_HEAP_HINT*)(buf + idx);
        idx += sizeof(WOLFSSL_HEAP_HINT);
        XMEMSET(hint, 0, sizeof(WOLFSSL_HEAP_HINT));
        hint->memory = heap;

        if (*ctx && (*ctx)->heap == NULL) {
            (*ctx)->heap = (void*)hint;
        }
    }
    else {
#ifdef WOLFSSL_HEAP_TEST
        /* do not load in memory if test has been set */
        if ((*ctx)->heap == (void*)WOLFSSL_HEAP_TEST) {
            return WOLFSSL_SUCCESS;
        }
#endif
        hint = (WOLFSSL_HEAP_HINT*)((*ctx)->heap);
        heap = hint->memory;
    }

    if (wolfSSL_load_static_memory(buf + idx, sz - idx, flag, heap) != 1) {
        WOLFSSL_MSG("Error partitioning memory");
        return WOLFSSL_FAILURE;
    }

    /* create ctx if needed */
    if (*ctx == NULL) {
        *ctx = wolfSSL_CTX_new_ex(method(hint), hint);
        if (*ctx == NULL) {
            WOLFSSL_MSG("Error creating ctx");
            return WOLFSSL_FAILURE;
        }
    }

    /* determine what max applies too */
    if (flag & WOLFMEM_IO_POOL || flag & WOLFMEM_IO_POOL_FIXED) {
        heap->maxIO = maxSz;
    }
    else { /* general memory used in handshakes */
        heap->maxHa = maxSz;
    }

    heap->flag |= flag;

    (void)maxSz;
    (void)method;

    return WOLFSSL_SUCCESS;
}


int wolfSSL_is_static_memory(WOLFSSL* ssl, WOLFSSL_MEM_CONN_STATS* mem_stats)
{
    if (ssl == NULL) {
        return BAD_FUNC_ARG;
    }
    WOLFSSL_ENTER("wolfSSL_is_static_memory");

    /* fill out statistics if wanted and WOLFMEM_TRACK_STATS flag */
    if (mem_stats != NULL && ssl->heap != NULL) {
        WOLFSSL_HEAP_HINT* hint = ((WOLFSSL_HEAP_HINT*)(ssl->heap));
        WOLFSSL_HEAP* heap      = hint->memory;
        if (heap->flag & WOLFMEM_TRACK_STATS && hint->stats != NULL) {
            XMEMCPY(mem_stats, hint->stats, sizeof(WOLFSSL_MEM_CONN_STATS));
        }
    }

    return (ssl->heap) ? 1 : 0;
}


int wolfSSL_CTX_is_static_memory(WOLFSSL_CTX* ctx, WOLFSSL_MEM_STATS* mem_stats)
{
    if (ctx == NULL) {
        return BAD_FUNC_ARG;
    }
    WOLFSSL_ENTER("wolfSSL_CTX_is_static_memory");

    /* fill out statistics if wanted */
    if (mem_stats != NULL && ctx->heap != NULL) {
        WOLFSSL_HEAP* heap = ((WOLFSSL_HEAP_HINT*)(ctx->heap))->memory;
        if (wolfSSL_GetMemStats(heap, mem_stats) != 1) {
            return MEMORY_E;
        }
    }

    return (ctx->heap) ? 1 : 0;
}

#endif /* WOLFSSL_STATIC_MEMORY */


/* return max record layer size plaintext input size */
int wolfSSL_GetMaxOutputSize(WOLFSSL* ssl)
{
    WOLFSSL_ENTER("wolfSSL_GetMaxOutputSize");

    if (ssl == NULL)
        return BAD_FUNC_ARG;

    if (ssl->options.handShakeState != HANDSHAKE_DONE) {
        WOLFSSL_MSG("Handshake not complete yet");
        return BAD_FUNC_ARG;
    }

    return wolfSSL_GetMaxFragSize(ssl, OUTPUT_RECORD_SIZE);
}


/* return record layer size of plaintext input size */
int wolfSSL_GetOutputSize(WOLFSSL* ssl, int inSz)
{
    int maxSize;

    WOLFSSL_ENTER("wolfSSL_GetOutputSize");

    if (inSz < 0)
        return BAD_FUNC_ARG;

    maxSize = wolfSSL_GetMaxOutputSize(ssl);
    if (maxSize < 0)
        return maxSize;   /* error */
    if (inSz > maxSize)
        return INPUT_SIZE_E;

    return BuildMessage(ssl, NULL, 0, NULL, inSz, application_data, 0, 1, 0, CUR_ORDER);
}


#ifdef HAVE_ECC
int wolfSSL_CTX_SetMinEccKey_Sz(WOLFSSL_CTX* ctx, short keySz)
{
    if (ctx == NULL || keySz < 0 || keySz % 8 != 0) {
        WOLFSSL_MSG("Key size must be divisible by 8 or ctx was null");
        return BAD_FUNC_ARG;
    }

    ctx->minEccKeySz     = keySz / 8;
#ifndef NO_CERTS
    ctx->cm->minEccKeySz = keySz / 8;
#endif
    return WOLFSSL_SUCCESS;
}


int wolfSSL_SetMinEccKey_Sz(WOLFSSL* ssl, short keySz)
{
    if (ssl == NULL || keySz < 0 || keySz % 8 != 0) {
        WOLFSSL_MSG("Key size must be divisible by 8 or ssl was null");
        return BAD_FUNC_ARG;
    }

    ssl->options.minEccKeySz = keySz / 8;
    return WOLFSSL_SUCCESS;
}

#endif /* HAVE_ECC */

#ifndef NO_RSA
int wolfSSL_CTX_SetMinRsaKey_Sz(WOLFSSL_CTX* ctx, short keySz)
{
    if (ctx == NULL || keySz < 0 || keySz % 8 != 0) {
        WOLFSSL_MSG("Key size must be divisible by 8 or ctx was null");
        return BAD_FUNC_ARG;
    }

    ctx->minRsaKeySz     = keySz / 8;
    ctx->cm->minRsaKeySz = keySz / 8;
    return WOLFSSL_SUCCESS;
}


int wolfSSL_SetMinRsaKey_Sz(WOLFSSL* ssl, short keySz)
{
    if (ssl == NULL || keySz < 0 || keySz % 8 != 0) {
        WOLFSSL_MSG("Key size must be divisible by 8 or ssl was null");
        return BAD_FUNC_ARG;
    }

    ssl->options.minRsaKeySz = keySz / 8;
    return WOLFSSL_SUCCESS;
}
#endif /* !NO_RSA */

#ifndef NO_DH
/* server Diffie-Hellman parameters, WOLFSSL_SUCCESS on ok */
int wolfSSL_SetTmpDH(WOLFSSL* ssl, const unsigned char* p, int pSz,
                    const unsigned char* g, int gSz)
{
    WOLFSSL_ENTER("wolfSSL_SetTmpDH");

    if (ssl == NULL || p == NULL || g == NULL)
        return BAD_FUNC_ARG;

    if ((word16)pSz < ssl->options.minDhKeySz)
        return DH_KEY_SIZE_E;
    if ((word16)pSz > ssl->options.maxDhKeySz)
        return DH_KEY_SIZE_E;

    /* this function is for server only */
    if (ssl->options.side == WOLFSSL_CLIENT_END)
        return SIDE_ERROR;

    #if !defined(WOLFSSL_OLD_PRIME_CHECK) && !defined(HAVE_FIPS) && \
        !defined(HAVE_SELFTEST)
        ssl->options.dhKeyTested = 0;
        ssl->options.dhDoKeyTest = 1;
    #endif

    if (ssl->buffers.serverDH_P.buffer && ssl->buffers.weOwnDH) {
        XFREE(ssl->buffers.serverDH_P.buffer, ssl->heap, DYNAMIC_TYPE_PUBLIC_KEY);
        ssl->buffers.serverDH_P.buffer = NULL;
    }
    if (ssl->buffers.serverDH_G.buffer && ssl->buffers.weOwnDH) {
        XFREE(ssl->buffers.serverDH_G.buffer, ssl->heap, DYNAMIC_TYPE_PUBLIC_KEY);
        ssl->buffers.serverDH_G.buffer = NULL;
    }

    ssl->buffers.weOwnDH = 1;  /* SSL owns now */
    ssl->buffers.serverDH_P.buffer = (byte*)XMALLOC(pSz, ssl->heap,
                                                    DYNAMIC_TYPE_PUBLIC_KEY);
    if (ssl->buffers.serverDH_P.buffer == NULL)
            return MEMORY_E;

    ssl->buffers.serverDH_G.buffer = (byte*)XMALLOC(gSz, ssl->heap,
                                                    DYNAMIC_TYPE_PUBLIC_KEY);
    if (ssl->buffers.serverDH_G.buffer == NULL) {
        XFREE(ssl->buffers.serverDH_P.buffer, ssl->heap, DYNAMIC_TYPE_PUBLIC_KEY);
        ssl->buffers.serverDH_P.buffer = NULL;
        return MEMORY_E;
    }

    ssl->buffers.serverDH_P.length = pSz;
    ssl->buffers.serverDH_G.length = gSz;

    XMEMCPY(ssl->buffers.serverDH_P.buffer, p, pSz);
    XMEMCPY(ssl->buffers.serverDH_G.buffer, g, gSz);

    ssl->options.haveDH = 1;

    if (ssl->options.side != WOLFSSL_NEITHER_END) {
        word16 havePSK;
        word16 haveRSA;
        int    keySz   = 0;

    #ifndef NO_PSK
        havePSK = ssl->options.havePSK;
    #else
        havePSK = 0;
    #endif
    #ifdef NO_RSA
        haveRSA = 0;
    #else
        haveRSA = 1;
    #endif
    #ifndef NO_CERTS
        keySz = ssl->buffers.keySz;
    #endif
        InitSuites(ssl->suites, ssl->version, keySz, haveRSA, havePSK,
                   ssl->options.haveDH, ssl->options.haveECDSAsig,
                   ssl->options.haveECC, ssl->options.haveStaticECC,
                   ssl->options.haveFalconSig, ssl->options.haveAnon,
                   ssl->options.side);
    }

    WOLFSSL_LEAVE("wolfSSL_SetTmpDH", 0);

    return WOLFSSL_SUCCESS;
}


#if !defined(WOLFSSL_OLD_PRIME_CHECK) && !defined(HAVE_FIPS) && \
    !defined(HAVE_SELFTEST)
/* Enables or disables the session's DH key prime test. */
int wolfSSL_SetEnableDhKeyTest(WOLFSSL* ssl, int enable)
{
    WOLFSSL_ENTER("wolfSSL_SetEnableDhKeyTest");

    if (ssl == NULL)
        return BAD_FUNC_ARG;

    if (!enable)
        ssl->options.dhDoKeyTest = 0;
    else
        ssl->options.dhDoKeyTest = 1;

    WOLFSSL_LEAVE("wolfSSL_SetEnableDhKeyTest", WOLFSSL_SUCCESS);
    return WOLFSSL_SUCCESS;
}
#endif


/* server ctx Diffie-Hellman parameters, WOLFSSL_SUCCESS on ok */
int wolfSSL_CTX_SetTmpDH(WOLFSSL_CTX* ctx, const unsigned char* p, int pSz,
                         const unsigned char* g, int gSz)
{
    WOLFSSL_ENTER("wolfSSL_CTX_SetTmpDH");
    if (ctx == NULL || p == NULL || g == NULL) return BAD_FUNC_ARG;

    if ((word16)pSz < ctx->minDhKeySz)
        return DH_KEY_SIZE_E;
    if ((word16)pSz > ctx->maxDhKeySz)
        return DH_KEY_SIZE_E;

    #if !defined(WOLFSSL_OLD_PRIME_CHECK) && !defined(HAVE_FIPS) && \
        !defined(HAVE_SELFTEST)
    {
        WC_RNG rng;
        int error, freeKey = 0;
    #ifdef WOLFSSL_SMALL_STACK
        DhKey *checkKey = (DhKey*)XMALLOC(sizeof(DhKey), NULL, DYNAMIC_TYPE_DH);
        if (checkKey == NULL)
            return MEMORY_E;
    #else
        DhKey checkKey[1];
    #endif

        error = wc_InitRng(&rng);
        if (!error)
            error = wc_InitDhKey(checkKey);
        if (!error) {
            freeKey = 1;
            error = wc_DhSetCheckKey(checkKey,
                                 p, pSz, g, gSz, NULL, 0, 0, &rng);
        }
        if (freeKey)
            wc_FreeDhKey(checkKey);
    #ifdef WOLFSSL_SMALL_STACK
        XFREE(checkKey, NULL, DYNAMIC_TYPE_DH);
    #endif
        wc_FreeRng(&rng);
        if (error)
            return error;

        ctx->dhKeyTested = 1;
    }
    #endif

    XFREE(ctx->serverDH_P.buffer, ctx->heap, DYNAMIC_TYPE_PUBLIC_KEY);
    ctx->serverDH_P.buffer = NULL;
    XFREE(ctx->serverDH_G.buffer, ctx->heap, DYNAMIC_TYPE_PUBLIC_KEY);
    ctx->serverDH_G.buffer = NULL;

    ctx->serverDH_P.buffer = (byte*)XMALLOC(pSz, ctx->heap, DYNAMIC_TYPE_PUBLIC_KEY);
    if (ctx->serverDH_P.buffer == NULL)
       return MEMORY_E;

    ctx->serverDH_G.buffer = (byte*)XMALLOC(gSz, ctx->heap, DYNAMIC_TYPE_PUBLIC_KEY);
    if (ctx->serverDH_G.buffer == NULL) {
        XFREE(ctx->serverDH_P.buffer, ctx->heap, DYNAMIC_TYPE_PUBLIC_KEY);
        ctx->serverDH_P.buffer = NULL;
        return MEMORY_E;
    }

    ctx->serverDH_P.length = pSz;
    ctx->serverDH_G.length = gSz;

    XMEMCPY(ctx->serverDH_P.buffer, p, pSz);
    XMEMCPY(ctx->serverDH_G.buffer, g, gSz);

    ctx->haveDH = 1;

    WOLFSSL_LEAVE("wolfSSL_CTX_SetTmpDH", 0);
    return WOLFSSL_SUCCESS;
}


int wolfSSL_CTX_SetMinDhKey_Sz(WOLFSSL_CTX* ctx, word16 keySz_bits)
{
    if (ctx == NULL || keySz_bits > 16000 || keySz_bits % 8 != 0)
        return BAD_FUNC_ARG;

    ctx->minDhKeySz = keySz_bits / 8;
    return WOLFSSL_SUCCESS;
}


int wolfSSL_SetMinDhKey_Sz(WOLFSSL* ssl, word16 keySz_bits)
{
    if (ssl == NULL || keySz_bits > 16000 || keySz_bits % 8 != 0)
        return BAD_FUNC_ARG;

    ssl->options.minDhKeySz = keySz_bits / 8;
    return WOLFSSL_SUCCESS;
}


int wolfSSL_CTX_SetMaxDhKey_Sz(WOLFSSL_CTX* ctx, word16 keySz_bits)
{
    if (ctx == NULL || keySz_bits > 16000 || keySz_bits % 8 != 0)
        return BAD_FUNC_ARG;

    ctx->maxDhKeySz = keySz_bits / 8;
    return WOLFSSL_SUCCESS;
}


int wolfSSL_SetMaxDhKey_Sz(WOLFSSL* ssl, word16 keySz_bits)
{
    if (ssl == NULL || keySz_bits > 16000 || keySz_bits % 8 != 0)
        return BAD_FUNC_ARG;

    ssl->options.maxDhKeySz = keySz_bits / 8;
    return WOLFSSL_SUCCESS;
}


int wolfSSL_GetDhKey_Sz(WOLFSSL* ssl)
{
    if (ssl == NULL)
        return BAD_FUNC_ARG;

    return (ssl->options.dhKeySz * 8);
}

#endif /* !NO_DH */


WOLFSSL_ABI
int wolfSSL_write(WOLFSSL* ssl, const void* data, int sz)
{
    int ret;

    WOLFSSL_ENTER("SSL_write()");

    if (ssl == NULL || data == NULL || sz < 0)
        return BAD_FUNC_ARG;

#ifdef WOLFSSL_EARLY_DATA
    if (ssl->earlyData != no_early_data && (ret = wolfSSL_negotiate(ssl)) < 0) {
        ssl->error = ret;
        return WOLFSSL_FATAL_ERROR;
    }
    ssl->earlyData = no_early_data;
#endif

#ifdef HAVE_WRITE_DUP
    { /* local variable scope */
        int dupErr = 0;   /* local copy */

        ret = 0;

        if (ssl->dupWrite && ssl->dupSide == READ_DUP_SIDE) {
            WOLFSSL_MSG("Read dup side cannot write");
            return WRITE_DUP_WRITE_E;
        }
        if (ssl->dupWrite) {
            if (wc_LockMutex(&ssl->dupWrite->dupMutex) != 0) {
                return BAD_MUTEX_E;
            }
            dupErr = ssl->dupWrite->dupErr;
            ret = wc_UnLockMutex(&ssl->dupWrite->dupMutex);
        }

        if (ret != 0) {
            ssl->error = ret;  /* high priority fatal error */
            return WOLFSSL_FATAL_ERROR;
        }
        if (dupErr != 0) {
            WOLFSSL_MSG("Write dup error from other side");
            ssl->error = dupErr;
            return WOLFSSL_FATAL_ERROR;
        }
    }
#endif

#ifdef HAVE_ERRNO_H
    errno = 0;
#endif

    #ifdef OPENSSL_EXTRA
    if (ssl->CBIS != NULL) {
        ssl->CBIS(ssl, SSL_CB_WRITE, WOLFSSL_SUCCESS);
        ssl->cbmode = SSL_CB_WRITE;
    }
    #endif
    ret = SendData(ssl, data, sz);

    WOLFSSL_LEAVE("SSL_write()", ret);

    if (ret < 0)
        return WOLFSSL_FATAL_ERROR;
    else
        return ret;
}

static int wolfSSL_read_internal(WOLFSSL* ssl, void* data, int sz, int peek)
{
    int ret;

    WOLFSSL_ENTER("wolfSSL_read_internal()");

    if (ssl == NULL || data == NULL || sz < 0)
        return BAD_FUNC_ARG;

#if defined(WOLFSSL_ERROR_CODE_OPENSSL) && defined(OPENSSL_EXTRA)
    /* This additional logic is meant to simulate following openSSL behavior:
     * After bidirectional SSL_shutdown complete, SSL_read returns 0 and
     * SSL_get_error_code returns SSL_ERROR_ZERO_RETURN.
     * This behavior is used to know the disconnect of the underlying
     * transport layer.
     *
     * In this logic, CBIORecv is called with a read size of 0 to check the
     * transport layer status. It also returns WOLFSSL_FAILURE so that
     * SSL_read does not return a positive number on failure.
     */

    /* make sure bidirectional TLS shutdown completes */
    if (ssl->error == WOLFSSL_ERROR_SYSCALL) {
        /* ask the underlying transport the connection is closed */
        if (ssl->CBIORecv(ssl, (char*)data, 0, ssl->IOCB_ReadCtx) ==
                                            WOLFSSL_CBIO_ERR_CONN_CLOSE) {
            ssl->options.isClosed = 1;
            ssl->error = WOLFSSL_ERROR_ZERO_RETURN;
        }
        return WOLFSSL_FAILURE;
    }
#endif

#ifdef HAVE_WRITE_DUP
    if (ssl->dupWrite && ssl->dupSide == WRITE_DUP_SIDE) {
        WOLFSSL_MSG("Write dup side cannot read");
        return WRITE_DUP_READ_E;
    }
#endif

#ifdef HAVE_ERRNO_H
        errno = 0;
#endif

#ifdef WOLFSSL_DTLS
    if (ssl->options.dtls) {
        ssl->dtls_expected_rx = max(sz + DTLS_MTU_ADDITIONAL_READ_BUFFER,
                MAX_MTU);
#ifdef WOLFSSL_SCTP
        if (ssl->options.dtlsSctp)
#endif
#if defined(WOLFSSL_SCTP) || defined(WOLFSSL_DTLS_MTU)
            /* Add some bytes so that we can operate with slight difference
             * in set MTU size on each peer */
            ssl->dtls_expected_rx = max(ssl->dtls_expected_rx,
                    ssl->dtlsMtuSz + (word32)DTLS_MTU_ADDITIONAL_READ_BUFFER);
#endif
    }
#endif

    ret = ReceiveData(ssl, (byte*)data, sz, peek);

#ifdef HAVE_WRITE_DUP
    if (ssl->dupWrite) {
        if (ssl->error != 0 && ssl->error != WANT_READ
        #ifdef WOLFSSL_ASYNC_CRYPT
            && ssl->error != WC_PENDING_E
        #endif
        ) {
            int notifyErr;

            WOLFSSL_MSG("Notifying write side of fatal read error");
            notifyErr  = NotifyWriteSide(ssl, ssl->error);
            if (notifyErr < 0) {
                ret = ssl->error = notifyErr;
            }
        }
    }
#endif

    WOLFSSL_LEAVE("wolfSSL_read_internal()", ret);

    if (ret < 0)
        return WOLFSSL_FATAL_ERROR;
    else
        return ret;
}


int wolfSSL_peek(WOLFSSL* ssl, void* data, int sz)
{
    WOLFSSL_ENTER("wolfSSL_peek()");

    return wolfSSL_read_internal(ssl, data, sz, TRUE);
}


WOLFSSL_ABI
int wolfSSL_read(WOLFSSL* ssl, void* data, int sz)
{
    WOLFSSL_ENTER("wolfSSL_read()");

    #ifdef OPENSSL_EXTRA
    if (ssl == NULL) {
        return BAD_FUNC_ARG;
    }
    if (ssl->CBIS != NULL) {
        ssl->CBIS(ssl, SSL_CB_READ, WOLFSSL_SUCCESS);
        ssl->cbmode = SSL_CB_READ;
    }
    #endif
    return wolfSSL_read_internal(ssl, data, sz, FALSE);
}


#ifdef WOLFSSL_MULTICAST

int wolfSSL_mcast_read(WOLFSSL* ssl, word16* id, void* data, int sz)
{
    int ret = 0;

    WOLFSSL_ENTER("wolfSSL_mcast_read()");

    if (ssl == NULL)
        return BAD_FUNC_ARG;

    ret = wolfSSL_read_internal(ssl, data, sz, FALSE);
    if (ssl->options.dtls && ssl->options.haveMcast && id != NULL)
        *id = ssl->keys.curPeerId;
    return ret;
}

#endif /* WOLFSSL_MULTICAST */


/* helpers to set the device id, WOLFSSL_SUCCESS on ok */
WOLFSSL_ABI
int wolfSSL_SetDevId(WOLFSSL* ssl, int devId)
{
    if (ssl == NULL)
        return BAD_FUNC_ARG;

    ssl->devId = devId;

    return WOLFSSL_SUCCESS;
}

WOLFSSL_ABI
int wolfSSL_CTX_SetDevId(WOLFSSL_CTX* ctx, int devId)
{
    if (ctx == NULL)
        return BAD_FUNC_ARG;

    ctx->devId = devId;

    return WOLFSSL_SUCCESS;
}

/* helpers to get device id and heap */
WOLFSSL_ABI
int wolfSSL_CTX_GetDevId(WOLFSSL_CTX* ctx, WOLFSSL* ssl)
{
    int devId = INVALID_DEVID;
    if (ssl != NULL)
        devId = ssl->devId;
    if (ctx != NULL && devId == INVALID_DEVID)
        devId = ctx->devId;
    return devId;
}
void* wolfSSL_CTX_GetHeap(WOLFSSL_CTX* ctx, WOLFSSL* ssl)
{
    void* heap = NULL;
    if (ctx != NULL)
        heap = ctx->heap;
    else if (ssl != NULL)
        heap = ssl->heap;
    return heap;
}


#ifdef HAVE_SNI

WOLFSSL_ABI
int wolfSSL_UseSNI(WOLFSSL* ssl, byte type, const void* data, word16 size)
{
    if (ssl == NULL)
        return BAD_FUNC_ARG;

    return TLSX_UseSNI(&ssl->extensions, type, data, size, ssl->heap);
}


WOLFSSL_ABI
int wolfSSL_CTX_UseSNI(WOLFSSL_CTX* ctx, byte type, const void* data,
                                                                    word16 size)
{
    if (ctx == NULL)
        return BAD_FUNC_ARG;

    return TLSX_UseSNI(&ctx->extensions, type, data, size, ctx->heap);
}

#ifndef NO_WOLFSSL_SERVER

void wolfSSL_SNI_SetOptions(WOLFSSL* ssl, byte type, byte options)
{
    if (ssl && ssl->extensions)
        TLSX_SNI_SetOptions(ssl->extensions, type, options);
}


void wolfSSL_CTX_SNI_SetOptions(WOLFSSL_CTX* ctx, byte type, byte options)
{
    if (ctx && ctx->extensions)
        TLSX_SNI_SetOptions(ctx->extensions, type, options);
}


byte wolfSSL_SNI_Status(WOLFSSL* ssl, byte type)
{
    return TLSX_SNI_Status(ssl ? ssl->extensions : NULL, type);
}


word16 wolfSSL_SNI_GetRequest(WOLFSSL* ssl, byte type, void** data)
{
    if (data)
        *data = NULL;

    if (ssl && ssl->extensions)
        return TLSX_SNI_GetRequest(ssl->extensions, type, data);

    return 0;
}


int wolfSSL_SNI_GetFromBuffer(const byte* clientHello, word32 helloSz,
                              byte type, byte* sni, word32* inOutSz)
{
    if (clientHello && helloSz > 0 && sni && inOutSz && *inOutSz > 0)
        return TLSX_SNI_GetFromBuffer(clientHello, helloSz, type, sni, inOutSz);

    return BAD_FUNC_ARG;
}

#endif /* NO_WOLFSSL_SERVER */

#endif /* HAVE_SNI */


#ifdef HAVE_TRUSTED_CA

WOLFSSL_API int wolfSSL_UseTrustedCA(WOLFSSL* ssl, byte type,
            const byte* certId, word32 certIdSz)
{
    if (ssl == NULL)
        return BAD_FUNC_ARG;

    if (type == WOLFSSL_TRUSTED_CA_PRE_AGREED) {
        if (certId != NULL || certIdSz != 0)
            return BAD_FUNC_ARG;
    }
    else if (type == WOLFSSL_TRUSTED_CA_X509_NAME) {
        if (certId == NULL || certIdSz == 0)
            return BAD_FUNC_ARG;
    }
    #ifndef NO_SHA
    else if (type == WOLFSSL_TRUSTED_CA_KEY_SHA1 ||
            type == WOLFSSL_TRUSTED_CA_CERT_SHA1) {
        if (certId == NULL || certIdSz != WC_SHA_DIGEST_SIZE)
            return BAD_FUNC_ARG;
    }
    #endif
    else
        return BAD_FUNC_ARG;

    return TLSX_UseTrustedCA(&ssl->extensions,
            type, certId, certIdSz, ssl->heap);
}

#endif /* HAVE_TRUSTED_CA */


#ifdef HAVE_MAX_FRAGMENT
#ifndef NO_WOLFSSL_CLIENT

int wolfSSL_UseMaxFragment(WOLFSSL* ssl, byte mfl)
{
    if (ssl == NULL)
        return BAD_FUNC_ARG;

#ifdef WOLFSSL_ALLOW_MAX_FRAGMENT_ADJUST
    /* The following is a non-standard way to reconfigure the max packet size
        post-handshake for wolfSSL_write/wolfSSL_read */
    if (ssl->options.handShakeState == HANDSHAKE_DONE) {
        switch (mfl) {
            case WOLFSSL_MFL_2_8 : ssl->max_fragment =  256; break;
            case WOLFSSL_MFL_2_9 : ssl->max_fragment =  512; break;
            case WOLFSSL_MFL_2_10: ssl->max_fragment = 1024; break;
            case WOLFSSL_MFL_2_11: ssl->max_fragment = 2048; break;
            case WOLFSSL_MFL_2_12: ssl->max_fragment = 4096; break;
            case WOLFSSL_MFL_2_13: ssl->max_fragment = 8192; break;
            default: ssl->max_fragment = MAX_RECORD_SIZE; break;
        }
        return WOLFSSL_SUCCESS;
    }
#endif /* WOLFSSL_MAX_FRAGMENT_ADJUST */

    /* This call sets the max fragment TLS extension, which gets sent to server.
        The server_hello response is what sets the `ssl->max_fragment` in
        TLSX_MFL_Parse */
    return TLSX_UseMaxFragment(&ssl->extensions, mfl, ssl->heap);
}


int wolfSSL_CTX_UseMaxFragment(WOLFSSL_CTX* ctx, byte mfl)
{
    if (ctx == NULL)
        return BAD_FUNC_ARG;

    return TLSX_UseMaxFragment(&ctx->extensions, mfl, ctx->heap);
}

#endif /* NO_WOLFSSL_CLIENT */
#endif /* HAVE_MAX_FRAGMENT */

#ifdef HAVE_TRUNCATED_HMAC
#ifndef NO_WOLFSSL_CLIENT

int wolfSSL_UseTruncatedHMAC(WOLFSSL* ssl)
{
    if (ssl == NULL)
        return BAD_FUNC_ARG;

    return TLSX_UseTruncatedHMAC(&ssl->extensions, ssl->heap);
}


int wolfSSL_CTX_UseTruncatedHMAC(WOLFSSL_CTX* ctx)
{
    if (ctx == NULL)
        return BAD_FUNC_ARG;

    return TLSX_UseTruncatedHMAC(&ctx->extensions, ctx->heap);
}

#endif /* NO_WOLFSSL_CLIENT */
#endif /* HAVE_TRUNCATED_HMAC */

#ifdef HAVE_CERTIFICATE_STATUS_REQUEST

int wolfSSL_UseOCSPStapling(WOLFSSL* ssl, byte status_type, byte options)
{
    if (ssl == NULL || ssl->options.side != WOLFSSL_CLIENT_END)
        return BAD_FUNC_ARG;

    return TLSX_UseCertificateStatusRequest(&ssl->extensions, status_type,
                                          options, NULL, ssl->heap, ssl->devId);
}


int wolfSSL_CTX_UseOCSPStapling(WOLFSSL_CTX* ctx, byte status_type,
                                                                   byte options)
{
    if (ctx == NULL || ctx->method->side != WOLFSSL_CLIENT_END)
        return BAD_FUNC_ARG;

    return TLSX_UseCertificateStatusRequest(&ctx->extensions, status_type,
                                          options, NULL, ctx->heap, ctx->devId);
}

#endif /* HAVE_CERTIFICATE_STATUS_REQUEST */

#ifdef HAVE_CERTIFICATE_STATUS_REQUEST_V2

int wolfSSL_UseOCSPStaplingV2(WOLFSSL* ssl, byte status_type, byte options)
{
    if (ssl == NULL || ssl->options.side != WOLFSSL_CLIENT_END)
        return BAD_FUNC_ARG;

    return TLSX_UseCertificateStatusRequestV2(&ssl->extensions, status_type,
                                                options, ssl->heap, ssl->devId);
}


int wolfSSL_CTX_UseOCSPStaplingV2(WOLFSSL_CTX* ctx, byte status_type,
                                                                   byte options)
{
    if (ctx == NULL || ctx->method->side != WOLFSSL_CLIENT_END)
        return BAD_FUNC_ARG;

    return TLSX_UseCertificateStatusRequestV2(&ctx->extensions, status_type,
                                                options, ctx->heap, ctx->devId);
}

#endif /* HAVE_CERTIFICATE_STATUS_REQUEST_V2 */

/* Elliptic Curves */
#if defined(HAVE_SUPPORTED_CURVES)

static int isValidCurveGroup(word16 name)
{
    switch (name) {
        case WOLFSSL_ECC_SECP160K1:
        case WOLFSSL_ECC_SECP160R1:
        case WOLFSSL_ECC_SECP160R2:
        case WOLFSSL_ECC_SECP192K1:
        case WOLFSSL_ECC_SECP192R1:
        case WOLFSSL_ECC_SECP224K1:
        case WOLFSSL_ECC_SECP224R1:
        case WOLFSSL_ECC_SECP256K1:
        case WOLFSSL_ECC_SECP256R1:
        case WOLFSSL_ECC_SECP384R1:
        case WOLFSSL_ECC_SECP521R1:
        case WOLFSSL_ECC_BRAINPOOLP256R1:
        case WOLFSSL_ECC_BRAINPOOLP384R1:
        case WOLFSSL_ECC_BRAINPOOLP512R1:
        case WOLFSSL_ECC_X25519:
        case WOLFSSL_ECC_X448:

        case WOLFSSL_FFDHE_2048:
        case WOLFSSL_FFDHE_3072:
        case WOLFSSL_FFDHE_4096:
        case WOLFSSL_FFDHE_6144:
        case WOLFSSL_FFDHE_8192:

#ifdef HAVE_PQC
        case WOLFSSL_KYBER_LEVEL1:
        case WOLFSSL_KYBER_LEVEL3:
        case WOLFSSL_KYBER_LEVEL5:
        case WOLFSSL_NTRU_HPS_LEVEL1:
        case WOLFSSL_NTRU_HPS_LEVEL3:
        case WOLFSSL_NTRU_HPS_LEVEL5:
        case WOLFSSL_NTRU_HRSS_LEVEL3:
        case WOLFSSL_SABER_LEVEL1:
        case WOLFSSL_SABER_LEVEL3:
        case WOLFSSL_SABER_LEVEL5:
        case WOLFSSL_KYBER_90S_LEVEL1:
        case WOLFSSL_KYBER_90S_LEVEL3:
        case WOLFSSL_KYBER_90S_LEVEL5:
        case WOLFSSL_P256_NTRU_HPS_LEVEL1:
        case WOLFSSL_P384_NTRU_HPS_LEVEL3:
        case WOLFSSL_P521_NTRU_HPS_LEVEL5:
        case WOLFSSL_P384_NTRU_HRSS_LEVEL3:
        case WOLFSSL_P256_SABER_LEVEL1:
        case WOLFSSL_P384_SABER_LEVEL3:
        case WOLFSSL_P521_SABER_LEVEL5:
        case WOLFSSL_P256_KYBER_LEVEL1:
        case WOLFSSL_P384_KYBER_LEVEL3:
        case WOLFSSL_P521_KYBER_LEVEL5:
        case WOLFSSL_P256_KYBER_90S_LEVEL1:
        case WOLFSSL_P384_KYBER_90S_LEVEL3:
        case WOLFSSL_P521_KYBER_90S_LEVEL5:
#endif
            return 1;

        default:
            return 0;
    }
}

int wolfSSL_UseSupportedCurve(WOLFSSL* ssl, word16 name)
{
    if (ssl == NULL || !isValidCurveGroup(name))
        return BAD_FUNC_ARG;

    ssl->options.userCurves = 1;

    return TLSX_UseSupportedCurve(&ssl->extensions, name, ssl->heap);
}


int wolfSSL_CTX_UseSupportedCurve(WOLFSSL_CTX* ctx, word16 name)
{
    if (ctx == NULL || !isValidCurveGroup(name))
        return BAD_FUNC_ARG;

    ctx->userCurves = 1;

    return TLSX_UseSupportedCurve(&ctx->extensions, name, ctx->heap);
}

#if defined(OPENSSL_EXTRA) && defined(WOLFSSL_TLS13)
int  wolfSSL_CTX_set1_groups(WOLFSSL_CTX* ctx, int* groups,
                                        int count)
{
    int i;
    int _groups[WOLFSSL_MAX_GROUP_COUNT];
    WOLFSSL_ENTER("wolfSSL_CTX_set1_groups");
    if (count == 0) {
        WOLFSSL_MSG("Group count is zero");
        return WOLFSSL_FAILURE;
    }
    for (i = 0; i < count; i++) {
        if (isValidCurveGroup((word16)groups[i])) {
            _groups[i] = groups[i];
        }
#ifdef HAVE_ECC
        else {
            /* groups may be populated with curve NIDs */
            int oid = nid2oid(groups[i], oidCurveType);
            int name = (int)GetCurveByOID(oid);
            if (name == 0) {
                WOLFSSL_MSG("Invalid group name");
                return WOLFSSL_FAILURE;
            }
            _groups[i] = name;
        }
#else
        else {
            WOLFSSL_MSG("Invalid group name");
            return WOLFSSL_FAILURE;
        }
#endif
    }
    return wolfSSL_CTX_set_groups(ctx, _groups, count) == WOLFSSL_SUCCESS ?
            WOLFSSL_SUCCESS : WOLFSSL_FAILURE;
}

int  wolfSSL_set1_groups(WOLFSSL* ssl, int* groups, int count)
{
    int i;
    int _groups[WOLFSSL_MAX_GROUP_COUNT];
    WOLFSSL_ENTER("wolfSSL_CTX_set1_groups");
    if (count == 0) {
        WOLFSSL_MSG("Group count is zero");
        return WOLFSSL_FAILURE;
    }
    for (i = 0; i < count; i++) {
        if (isValidCurveGroup((word16)groups[i])) {
            _groups[i] = groups[i];
        }
#ifdef HAVE_ECC
        else {
            /* groups may be populated with curve NIDs */
            int oid = nid2oid(groups[i], oidCurveType);
            int name = (int)GetCurveByOID(oid);
            if (name == 0) {
                WOLFSSL_MSG("Invalid group name");
                return WOLFSSL_FAILURE;
            }
            _groups[i] = name;
        }
#else
        else {
            WOLFSSL_MSG("Invalid group name");
            return WOLFSSL_FAILURE;
        }
#endif
    }
    return wolfSSL_set_groups(ssl, _groups, count) == WOLFSSL_SUCCESS ?
            WOLFSSL_SUCCESS : WOLFSSL_FAILURE;
}
#endif /* OPENSSL_EXTRA && WOLFSSL_TLS13 */
#endif /* HAVE_SUPPORTED_CURVES */

/* Application-Layer Protocol Negotiation */
#ifdef HAVE_ALPN

WOLFSSL_ABI
int wolfSSL_UseALPN(WOLFSSL* ssl, char *protocol_name_list,
                    word32 protocol_name_listSz, byte options)
{
    char    *list, *ptr, **token;
    word16  len;
    int     idx = 0;
    int     ret = WOLFSSL_FAILURE;

    WOLFSSL_ENTER("wolfSSL_UseALPN");

    if (ssl == NULL || protocol_name_list == NULL)
        return BAD_FUNC_ARG;

    if (protocol_name_listSz > (WOLFSSL_MAX_ALPN_NUMBER *
                                WOLFSSL_MAX_ALPN_PROTO_NAME_LEN +
                                WOLFSSL_MAX_ALPN_NUMBER)) {
        WOLFSSL_MSG("Invalid arguments, protocol name list too long");
        return BAD_FUNC_ARG;
    }

    if (!(options & WOLFSSL_ALPN_CONTINUE_ON_MISMATCH) &&
        !(options & WOLFSSL_ALPN_FAILED_ON_MISMATCH)) {
            WOLFSSL_MSG("Invalid arguments, options not supported");
            return BAD_FUNC_ARG;
        }


    list = (char *)XMALLOC(protocol_name_listSz+1, ssl->heap,
                           DYNAMIC_TYPE_ALPN);
    if (list == NULL) {
        WOLFSSL_MSG("Memory failure");
        return MEMORY_ERROR;
    }

    token = (char **)XMALLOC(sizeof(char *) * (WOLFSSL_MAX_ALPN_NUMBER+1), ssl->heap, DYNAMIC_TYPE_ALPN);
    if (token == NULL) {
        XFREE(list, ssl->heap, DYNAMIC_TYPE_ALPN);
        WOLFSSL_MSG("Memory failure");
        return MEMORY_ERROR;
    }
    XMEMSET(token, 0, sizeof(char *) * (WOLFSSL_MAX_ALPN_NUMBER+1));

    XSTRNCPY(list, protocol_name_list, protocol_name_listSz);
    list[protocol_name_listSz] = '\0';

    /* read all protocol name from the list */
    token[idx] = XSTRTOK(list, ",", &ptr);
    while (idx < WOLFSSL_MAX_ALPN_NUMBER && token[idx] != NULL)
        token[++idx] = XSTRTOK(NULL, ",", &ptr);

    /* add protocol name list in the TLS extension in reverse order */
    while ((idx--) > 0) {
        len = (word16)XSTRLEN(token[idx]);

        ret = TLSX_UseALPN(&ssl->extensions, token[idx], len, options,
                                                                     ssl->heap);
        if (ret != WOLFSSL_SUCCESS) {
            WOLFSSL_MSG("TLSX_UseALPN failure");
            break;
        }
    }

    XFREE(token, ssl->heap, DYNAMIC_TYPE_ALPN);
    XFREE(list, ssl->heap, DYNAMIC_TYPE_ALPN);

    return ret;
}

int wolfSSL_ALPN_GetProtocol(WOLFSSL* ssl, char **protocol_name, word16 *size)
{
    return TLSX_ALPN_GetRequest(ssl ? ssl->extensions : NULL,
                               (void **)protocol_name, size);
}

int wolfSSL_ALPN_GetPeerProtocol(WOLFSSL* ssl, char **list, word16 *listSz)
{
    if (list == NULL || listSz == NULL)
        return BAD_FUNC_ARG;

    if (ssl->alpn_client_list == NULL)
        return BUFFER_ERROR;

    *listSz = (word16)XSTRLEN(ssl->alpn_client_list);
    if (*listSz == 0)
        return BUFFER_ERROR;

    *list = (char *)XMALLOC((*listSz)+1, ssl->heap, DYNAMIC_TYPE_TLSX);
    if (*list == NULL)
        return MEMORY_ERROR;

    XSTRNCPY(*list, ssl->alpn_client_list, (*listSz)+1);
    (*list)[*listSz] = 0;

    return WOLFSSL_SUCCESS;
}


/* used to free memory allocated by wolfSSL_ALPN_GetPeerProtocol */
int wolfSSL_ALPN_FreePeerProtocol(WOLFSSL* ssl, char **list)
{
    if (ssl == NULL) {
        return BAD_FUNC_ARG;
    }

    XFREE(*list, ssl->heap, DYNAMIC_TYPE_TLSX);
    *list = NULL;

    return WOLFSSL_SUCCESS;
}

#endif /* HAVE_ALPN */

/* Secure Renegotiation */
#ifdef HAVE_SERVER_RENEGOTIATION_INFO

/* user is forcing ability to use secure renegotiation, we discourage it */
int wolfSSL_UseSecureRenegotiation(WOLFSSL* ssl)
{
    int ret = BAD_FUNC_ARG;

    if (ssl)
        ret = TLSX_UseSecureRenegotiation(&ssl->extensions, ssl->heap);

    if (ret == WOLFSSL_SUCCESS) {
        TLSX* extension = TLSX_Find(ssl->extensions, TLSX_RENEGOTIATION_INFO);

        if (extension)
            ssl->secure_renegotiation = (SecureRenegotiation*)extension->data;
    }

    return ret;
}

int wolfSSL_CTX_UseSecureRenegotiation(WOLFSSL_CTX* ctx)
{
    if (ctx == NULL)
        return BAD_FUNC_ARG;

    ctx->useSecureReneg = 1;
    return WOLFSSL_SUCCESS;
}


/* do a secure renegotiation handshake, user forced, we discourage */
static int _Rehandshake(WOLFSSL* ssl)
{
    int ret;

    if (ssl == NULL)
        return BAD_FUNC_ARG;

    if (ssl->secure_renegotiation == NULL) {
        WOLFSSL_MSG("Secure Renegotiation not forced on by user");
        return SECURE_RENEGOTIATION_E;
    }

    if (ssl->secure_renegotiation->enabled == 0) {
        WOLFSSL_MSG("Secure Renegotiation not enabled at extension level");
        return SECURE_RENEGOTIATION_E;
    }

    /* If the client started the renegotiation, the server will already
     * have processed the client's hello. */
    if (ssl->options.side != WOLFSSL_SERVER_END ||
        ssl->options.acceptState != ACCEPT_FIRST_REPLY_DONE) {

        if (ssl->options.handShakeState != HANDSHAKE_DONE) {
            if (!ssl->options.handShakeDone) {
                WOLFSSL_MSG("Can't renegotiate until initial "
                            "handshake complete");
                return SECURE_RENEGOTIATION_E;
            }
            else {
                WOLFSSL_MSG("Renegotiation already started. "
                            "Moving it forward.");
                ret = wolfSSL_negotiate(ssl);
                if (ret == WOLFSSL_SUCCESS)
                    ssl->secure_rene_count++;
                return ret;
            }
        }

#ifndef NO_FORCE_SCR_SAME_SUITE
        /* force same suite */
        if (ssl->suites) {
            ssl->suites->suiteSz = SUITE_LEN;
            ssl->suites->suites[0] = ssl->options.cipherSuite0;
            ssl->suites->suites[1] = ssl->options.cipherSuite;
        }
#endif

        /* reset handshake states */
        ssl->options.sendVerify = 0;
        ssl->options.serverState = NULL_STATE;
        ssl->options.clientState = NULL_STATE;
        ssl->options.connectState  = CONNECT_BEGIN;
        ssl->options.acceptState   = ACCEPT_BEGIN_RENEG;
        ssl->options.handShakeState = NULL_STATE;
        ssl->options.processReply  = 0;  /* TODO, move states in internal.h */

        XMEMSET(&ssl->msgsReceived, 0, sizeof(ssl->msgsReceived));

        ssl->secure_renegotiation->cache_status = SCR_CACHE_NEEDED;

#if !defined(NO_WOLFSSL_SERVER) && defined(HAVE_SECURE_RENEGOTIATION)
        if (ssl->options.side == WOLFSSL_SERVER_END) {
            ret = SendHelloRequest(ssl);
            if (ret != 0) {
                ssl->error = ret;
                return WOLFSSL_FATAL_ERROR;
            }
        }
#endif /* !NO_WOLFSSL_SERVER && HAVE_SECURE_RENEGOTIATION */

        ret = InitHandshakeHashes(ssl);
        if (ret != 0) {
            ssl->error = ret;
            return WOLFSSL_FATAL_ERROR;
        }
    }
    ret = wolfSSL_negotiate(ssl);
    if (ret == WOLFSSL_SUCCESS)
        ssl->secure_rene_count++;
    return ret;
}


/* do a secure renegotiation handshake, user forced, we discourage */
int wolfSSL_Rehandshake(WOLFSSL* ssl)
{
    int ret;
    WOLFSSL_ENTER("wolfSSL_Rehandshake");

    if (ssl == NULL)
        return WOLFSSL_FAILURE;

#ifdef HAVE_SESSION_TICKET
    ret = WOLFSSL_SUCCESS;
#endif

    if (ssl->options.side == WOLFSSL_SERVER_END) {
        /* Reset option to send certificate verify. */
        ssl->options.sendVerify = 0;
    }
    else {
        /* Reset resuming flag to do full secure handshake. */
        ssl->options.resuming = 0;
        #ifdef HAVE_SESSION_TICKET
            /* Clearing the ticket. */
            ret = wolfSSL_UseSessionTicket(ssl);
        #endif
    }
    /* CLIENT/SERVER: Reset peer authentication for full secure handshake. */
    ssl->options.peerAuthGood = 0;

#ifdef HAVE_SESSION_TICKET
    if (ret == WOLFSSL_SUCCESS)
#endif
        ret = _Rehandshake(ssl);

    return ret;
}


#ifndef NO_WOLFSSL_CLIENT

/* do a secure resumption handshake, user forced, we discourage */
int wolfSSL_SecureResume(WOLFSSL* ssl)
{
    WOLFSSL_ENTER("wolfSSL_SecureResume");

    if (ssl == NULL)
        return BAD_FUNC_ARG;

    if (ssl->options.side == WOLFSSL_SERVER_END) {
        ssl->error = SIDE_ERROR;
        return WOLFSSL_FATAL_ERROR;
    }

    return _Rehandshake(ssl);
}

#endif /* NO_WOLFSSL_CLIENT */

long wolfSSL_SSL_get_secure_renegotiation_support(WOLFSSL* ssl)
{
    WOLFSSL_ENTER("wolfSSL_SSL_get_secure_renegotiation_support");

    if (!ssl || !ssl->secure_renegotiation)
        return WOLFSSL_FAILURE;
    return ssl->secure_renegotiation->enabled;
}

#endif /* HAVE_SECURE_RENEGOTIATION_INFO */

#if defined(HAVE_SESSION_TICKET)
/* Session Ticket */

#if !defined(NO_WOLFSSL_SERVER)
int wolfSSL_CTX_NoTicketTLSv12(WOLFSSL_CTX* ctx)
{
    if (ctx == NULL)
        return BAD_FUNC_ARG;

    ctx->noTicketTls12 = 1;

    return WOLFSSL_SUCCESS;
}

int wolfSSL_NoTicketTLSv12(WOLFSSL* ssl)
{
    if (ssl == NULL)
        return BAD_FUNC_ARG;

    ssl->options.noTicketTls12 = 1;

    return WOLFSSL_SUCCESS;
}

/* WOLFSSL_SUCCESS on ok */
int wolfSSL_CTX_set_TicketEncCb(WOLFSSL_CTX* ctx, SessionTicketEncCb cb)
{
    if (ctx == NULL)
        return BAD_FUNC_ARG;

    ctx->ticketEncCb = cb;

    return WOLFSSL_SUCCESS;
}

/* set hint interval, WOLFSSL_SUCCESS on ok */
int wolfSSL_CTX_set_TicketHint(WOLFSSL_CTX* ctx, int hint)
{
    if (ctx == NULL)
        return BAD_FUNC_ARG;

    ctx->ticketHint = hint;

    return WOLFSSL_SUCCESS;
}

/* set user context, WOLFSSL_SUCCESS on ok */
int wolfSSL_CTX_set_TicketEncCtx(WOLFSSL_CTX* ctx, void* userCtx)
{
    if (ctx == NULL)
        return BAD_FUNC_ARG;

    ctx->ticketEncCtx = userCtx;

    return WOLFSSL_SUCCESS;
}

/* get user context - returns userCtx on success, NULL on failure */
void* wolfSSL_CTX_get_TicketEncCtx(WOLFSSL_CTX* ctx)
{
    if (ctx == NULL)
        return NULL;

    return ctx->ticketEncCtx;
}

#ifdef WOLFSSL_TLS13
/* set the maximum number of tickets to send
 * return WOLFSSL_SUCCESS on success and WOLFSSL_FAILURE on fail
 */
int wolfSSL_CTX_set_num_tickets(WOLFSSL_CTX* ctx, size_t mxTickets)
{
    if (ctx == NULL)
        return WOLFSSL_FAILURE;

    ctx->maxTicketTls13 = (unsigned int)mxTickets;
    return WOLFSSL_SUCCESS;
}

/* get the maximum number of tickets to send
 * return number of tickets set to be sent
 */
size_t wolfSSL_CTX_get_num_tickets(WOLFSSL_CTX* ctx)
{
    if (ctx == NULL)
        return 0;

    return (size_t)ctx->maxTicketTls13;
}
#endif /* WOLFSSL_TLS13 */
#endif /* !NO_WOLFSSL_SERVER */

#if !defined(NO_WOLFSSL_CLIENT)
int wolfSSL_UseSessionTicket(WOLFSSL* ssl)
{
    if (ssl == NULL)
        return BAD_FUNC_ARG;

    return TLSX_UseSessionTicket(&ssl->extensions, NULL, ssl->heap);
}

int wolfSSL_CTX_UseSessionTicket(WOLFSSL_CTX* ctx)
{
    if (ctx == NULL)
        return BAD_FUNC_ARG;

    return TLSX_UseSessionTicket(&ctx->extensions, NULL, ctx->heap);
}

WOLFSSL_API int wolfSSL_get_SessionTicket(WOLFSSL* ssl,
                                          byte* buf, word32* bufSz)
{
    if (ssl == NULL || buf == NULL || bufSz == NULL || *bufSz == 0)
        return BAD_FUNC_ARG;

    if (ssl->session->ticketLen <= *bufSz) {
        XMEMCPY(buf, ssl->session->ticket, ssl->session->ticketLen);
        *bufSz = ssl->session->ticketLen;
    }
    else
        *bufSz = 0;

    return WOLFSSL_SUCCESS;
}

WOLFSSL_API int wolfSSL_set_SessionTicket(WOLFSSL* ssl, const byte* buf,
                                          word32 bufSz)
{
    if (ssl == NULL || (buf == NULL && bufSz > 0))
        return BAD_FUNC_ARG;

    if (bufSz > 0) {
        /* Ticket will fit into static ticket */
        if (bufSz <= SESSION_TICKET_LEN) {
            if (ssl->session->ticketLenAlloc > 0) {
                XFREE(ssl->session->ticket, ssl->session->heap,
                      DYNAMIC_TYPE_SESSION_TICK);
                ssl->session->ticketLenAlloc = 0;
                ssl->session->ticket = ssl->session->_staticTicket;
            }
        }
        else { /* Ticket requires dynamic ticket storage */
            if (ssl->session->ticketLen < bufSz) { /* is dyn buffer big enough */
                if (ssl->session->ticketLenAlloc > 0) {
                    XFREE(ssl->session->ticket, ssl->session->heap,
                          DYNAMIC_TYPE_SESSION_TICK);
                }
                ssl->session->ticket = (byte*)XMALLOC(bufSz, ssl->session->heap,
                        DYNAMIC_TYPE_SESSION_TICK);
                if(ssl->session->ticket == NULL) {
                    ssl->session->ticket = ssl->session->_staticTicket;
                    ssl->session->ticketLenAlloc = 0;
                    return MEMORY_ERROR;
                }
                ssl->session->ticketLenAlloc = (word16)bufSz;
            }
        }
        XMEMCPY(ssl->session->ticket, buf, bufSz);
    }
    ssl->session->ticketLen = (word16)bufSz;

    return WOLFSSL_SUCCESS;
}


WOLFSSL_API int wolfSSL_set_SessionTicket_cb(WOLFSSL* ssl,
                                            CallbackSessionTicket cb, void* ctx)
{
    if (ssl == NULL)
        return BAD_FUNC_ARG;

    ssl->session_ticket_cb = cb;
    ssl->session_ticket_ctx = ctx;

    return WOLFSSL_SUCCESS;
}
#endif /* !NO_WOLFSSL_CLIENT */

#endif /* HAVE_SESSION_TICKET */


#ifdef HAVE_EXTENDED_MASTER
#ifndef NO_WOLFSSL_CLIENT

int wolfSSL_CTX_DisableExtendedMasterSecret(WOLFSSL_CTX* ctx)
{
    if (ctx == NULL)
        return BAD_FUNC_ARG;

    ctx->haveEMS = 0;

    return WOLFSSL_SUCCESS;
}


int wolfSSL_DisableExtendedMasterSecret(WOLFSSL* ssl)
{
    if (ssl == NULL)
        return BAD_FUNC_ARG;

    ssl->options.haveEMS = 0;

    return WOLFSSL_SUCCESS;
}

#endif
#endif


#ifndef WOLFSSL_LEANPSK

int wolfSSL_send(WOLFSSL* ssl, const void* data, int sz, int flags)
{
    int ret;
    int oldFlags;

    WOLFSSL_ENTER("wolfSSL_send()");

    if (ssl == NULL || data == NULL || sz < 0)
        return BAD_FUNC_ARG;

    oldFlags = ssl->wflags;

    ssl->wflags = flags;
    ret = wolfSSL_write(ssl, data, sz);
    ssl->wflags = oldFlags;

    WOLFSSL_LEAVE("wolfSSL_send()", ret);

    return ret;
}


int wolfSSL_recv(WOLFSSL* ssl, void* data, int sz, int flags)
{
    int ret;
    int oldFlags;

    WOLFSSL_ENTER("wolfSSL_recv()");

    if (ssl == NULL || data == NULL || sz < 0)
        return BAD_FUNC_ARG;

    oldFlags = ssl->rflags;

    ssl->rflags = flags;
    ret = wolfSSL_read(ssl, data, sz);
    ssl->rflags = oldFlags;

    WOLFSSL_LEAVE("wolfSSL_recv()", ret);

    return ret;
}
#endif


/* WOLFSSL_SUCCESS on ok */
WOLFSSL_ABI
int wolfSSL_shutdown(WOLFSSL* ssl)
{
    int  ret = WOLFSSL_FATAL_ERROR;
    WOLFSSL_ENTER("SSL_shutdown()");

    if (ssl == NULL)
        return WOLFSSL_FATAL_ERROR;

    if (ssl->options.quietShutdown) {
        WOLFSSL_MSG("quiet shutdown, no close notify sent");
        ret = WOLFSSL_SUCCESS;
    }
    else {
        /* try to send close notify, not an error if can't */
        if (!ssl->options.isClosed && !ssl->options.connReset &&
                                      !ssl->options.sentNotify) {
            ssl->error = SendAlert(ssl, alert_warning, close_notify);
            if (ssl->error < 0) {
                WOLFSSL_ERROR(ssl->error);
                return WOLFSSL_FATAL_ERROR;
            }
            ssl->options.sentNotify = 1;  /* don't send close_notify twice */
            if (ssl->options.closeNotify)
                ret = WOLFSSL_SUCCESS;
            else {
                ret = WOLFSSL_SHUTDOWN_NOT_DONE;
                WOLFSSL_LEAVE("SSL_shutdown()", ret);
                return ret;
            }
        }

#ifdef WOLFSSL_SHUTDOWNONCE
        if (ssl->options.isClosed || ssl->options.connReset) {
            /* Shutdown has already occurred.
             * Caller is free to ignore this error. */
            return SSL_SHUTDOWN_ALREADY_DONE_E;
        }
#endif

        /* call wolfSSL_shutdown again for bidirectional shutdown */
        if (ssl->options.sentNotify && !ssl->options.closeNotify) {
            ret = ProcessReply(ssl);
            if (ret == ZERO_RETURN) {
                /* simulate OpenSSL behavior */
                ssl->error = WOLFSSL_ERROR_SYSCALL;
                ret = WOLFSSL_SUCCESS;
            } else if (ssl->error == WOLFSSL_ERROR_NONE) {
                ret = WOLFSSL_SHUTDOWN_NOT_DONE;
            } else {
                WOLFSSL_ERROR(ssl->error);
                ret = WOLFSSL_FATAL_ERROR;
            }
        }
    }

#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_WPAS_SMALL)
    /* reset WOLFSSL structure state for possible re-use */
    if (ret == WOLFSSL_SUCCESS) {
        if (wolfSSL_clear(ssl) != WOLFSSL_SUCCESS) {
            WOLFSSL_MSG("could not clear WOLFSSL");
            ret = WOLFSSL_FATAL_ERROR;
        }
    }
#endif

    WOLFSSL_LEAVE("SSL_shutdown()", ret);

    return ret;
}


/* get current error state value */
int wolfSSL_state(WOLFSSL* ssl)
{
    if (ssl == NULL) {
        return BAD_FUNC_ARG;
    }

    return ssl->error;
}


WOLFSSL_ABI
int wolfSSL_get_error(WOLFSSL* ssl, int ret)
{
    WOLFSSL_ENTER("SSL_get_error");

    if (ret > 0)
        return WOLFSSL_ERROR_NONE;
    if (ssl == NULL)
        return BAD_FUNC_ARG;

    WOLFSSL_LEAVE("SSL_get_error", ssl->error);

    /* make sure converted types are handled in SetErrorString() too */
    if (ssl->error == WANT_READ)
        return WOLFSSL_ERROR_WANT_READ;         /* convert to OpenSSL type */
    else if (ssl->error == WANT_WRITE)
        return WOLFSSL_ERROR_WANT_WRITE;        /* convert to OpenSSL type */
    else if (ssl->error == ZERO_RETURN)
        return WOLFSSL_ERROR_ZERO_RETURN;       /* convert to OpenSSL type */
    return ssl->error;
}


/* retrieve alert history, WOLFSSL_SUCCESS on ok */
int wolfSSL_get_alert_history(WOLFSSL* ssl, WOLFSSL_ALERT_HISTORY *h)
{
    if (ssl && h) {
        *h = ssl->alert_history;
    }
    return WOLFSSL_SUCCESS;
}

#ifdef OPENSSL_EXTRA
/* returns SSL_WRITING, SSL_READING or SSL_NOTHING */
int wolfSSL_want(WOLFSSL* ssl)
{
    int rw_state = SSL_NOTHING;
    if (ssl) {
        if (ssl->error == WANT_READ)
            rw_state = SSL_READING;
        else if (ssl->error == WANT_WRITE)
            rw_state = SSL_WRITING;
    }
    return rw_state;
}
#endif

/* return TRUE if current error is want read */
int wolfSSL_want_read(WOLFSSL* ssl)
{
    WOLFSSL_ENTER("SSL_want_read");
    if (ssl->error == WANT_READ)
        return 1;

    return 0;
}


/* return TRUE if current error is want write */
int wolfSSL_want_write(WOLFSSL* ssl)
{
    WOLFSSL_ENTER("SSL_want_write");
    if (ssl->error == WANT_WRITE)
        return 1;

    return 0;
}


char* wolfSSL_ERR_error_string(unsigned long errNumber, char* data)
{
    static char tmp[WOLFSSL_MAX_ERROR_SZ] = {0};

    WOLFSSL_ENTER("ERR_error_string");
    if (data) {
        SetErrorString((int)errNumber, data);
        return data;
    }
    else {
        SetErrorString((int)errNumber, tmp);
        return tmp;
    }
}


void wolfSSL_ERR_error_string_n(unsigned long e, char* buf, unsigned long len)
{
    WOLFSSL_ENTER("wolfSSL_ERR_error_string_n");
    if (len >= WOLFSSL_MAX_ERROR_SZ)
        wolfSSL_ERR_error_string(e, buf);
    else {
        char tmp[WOLFSSL_MAX_ERROR_SZ];

        WOLFSSL_MSG("Error buffer too short, truncating");
        if (len) {
            wolfSSL_ERR_error_string(e, tmp);
            XMEMCPY(buf, tmp, len-1);
            buf[len-1] = '\0';
        }
    }
}


/* don't free temporary arrays at end of handshake */
void wolfSSL_KeepArrays(WOLFSSL* ssl)
{
    if (ssl)
        ssl->options.saveArrays = 1;
}


/* user doesn't need temporary arrays anymore, Free */
void wolfSSL_FreeArrays(WOLFSSL* ssl)
{
    if (ssl && ssl->options.handShakeState == HANDSHAKE_DONE) {
        ssl->options.saveArrays = 0;
        FreeArrays(ssl, 1);
    }
}

/* Set option to indicate that the resources are not to be freed after
 * handshake.
 *
 * ssl  The SSL/TLS object.
 * returns BAD_FUNC_ARG when ssl is NULL and 0 on success.
 */
int wolfSSL_KeepHandshakeResources(WOLFSSL* ssl)
{
    if (ssl == NULL)
        return BAD_FUNC_ARG;

    ssl->options.keepResources = 1;

    return 0;
}

/* Free the handshake resources after handshake.
 *
 * ssl  The SSL/TLS object.
 * returns BAD_FUNC_ARG when ssl is NULL and 0 on success.
 */
int wolfSSL_FreeHandshakeResources(WOLFSSL* ssl)
{
    if (ssl == NULL)
        return BAD_FUNC_ARG;

    FreeHandshakeResources(ssl);

    return 0;
}

/* Use the client's order of preference when matching cipher suites.
 *
 * ssl  The SSL/TLS context object.
 * returns BAD_FUNC_ARG when ssl is NULL and 0 on success.
 */
int wolfSSL_CTX_UseClientSuites(WOLFSSL_CTX* ctx)
{
    if (ctx == NULL)
        return BAD_FUNC_ARG;

    ctx->useClientOrder = 1;

    return 0;
}

/* Use the client's order of preference when matching cipher suites.
 *
 * ssl  The SSL/TLS object.
 * returns BAD_FUNC_ARG when ssl is NULL and 0 on success.
 */
int wolfSSL_UseClientSuites(WOLFSSL* ssl)
{
    if (ssl == NULL)
        return BAD_FUNC_ARG;

    ssl->options.useClientOrder = 1;

    return 0;
}

#ifdef WOLFSSL_DTLS
const byte* wolfSSL_GetDtlsMacSecret(WOLFSSL* ssl, int verify, int epochOrder)
{
#ifndef WOLFSSL_AEAD_ONLY
    Keys* keys = NULL;

    (void)epochOrder;

    if (ssl == NULL)
        return NULL;

#ifdef HAVE_SECURE_RENEGOTIATION
    switch (epochOrder) {
    case PEER_ORDER:
        if (IsDtlsMsgSCRKeys(ssl))
            keys = &ssl->secure_renegotiation->tmp_keys;
        else
            keys = &ssl->keys;
        break;
    case PREV_ORDER:
        keys = &ssl->keys;
        break;
    case CUR_ORDER:
        if (DtlsUseSCRKeys(ssl))
            keys = &ssl->secure_renegotiation->tmp_keys;
        else
            keys = &ssl->keys;
        break;
    default:
        WOLFSSL_MSG("Unknown epoch order");
        return NULL;
    }
#else
    keys = &ssl->keys;
#endif

    if ( (ssl->options.side == WOLFSSL_CLIENT_END && !verify) ||
         (ssl->options.side == WOLFSSL_SERVER_END &&  verify) )
        return keys->client_write_MAC_secret;
    else
        return keys->server_write_MAC_secret;
#else
    (void)ssl;
    (void)verify;
    (void)epochOrder;

    return NULL;
#endif
}
#endif /* WOLFSSL_DTLS */

const byte* wolfSSL_GetMacSecret(WOLFSSL* ssl, int verify)
{
#ifndef WOLFSSL_AEAD_ONLY
    if (ssl == NULL)
        return NULL;

    if ( (ssl->options.side == WOLFSSL_CLIENT_END && !verify) ||
         (ssl->options.side == WOLFSSL_SERVER_END &&  verify) )
        return ssl->keys.client_write_MAC_secret;
    else
        return ssl->keys.server_write_MAC_secret;
#else
    (void)ssl;
    (void)verify;

    return NULL;
#endif
}


#ifdef ATOMIC_USER

void  wolfSSL_CTX_SetMacEncryptCb(WOLFSSL_CTX* ctx, CallbackMacEncrypt cb)
{
    if (ctx)
        ctx->MacEncryptCb = cb;
}


void  wolfSSL_SetMacEncryptCtx(WOLFSSL* ssl, void *ctx)
{
    if (ssl)
        ssl->MacEncryptCtx = ctx;
}


void* wolfSSL_GetMacEncryptCtx(WOLFSSL* ssl)
{
    if (ssl)
        return ssl->MacEncryptCtx;

    return NULL;
}


void  wolfSSL_CTX_SetDecryptVerifyCb(WOLFSSL_CTX* ctx, CallbackDecryptVerify cb)
{
    if (ctx)
        ctx->DecryptVerifyCb = cb;
}


void  wolfSSL_SetDecryptVerifyCtx(WOLFSSL* ssl, void *ctx)
{
    if (ssl)
        ssl->DecryptVerifyCtx = ctx;
}


void* wolfSSL_GetDecryptVerifyCtx(WOLFSSL* ssl)
{
    if (ssl)
        return ssl->DecryptVerifyCtx;

    return NULL;
}

#if defined(HAVE_ENCRYPT_THEN_MAC) && !defined(WOLFSSL_AEAD_ONLY)
/**
 * Set the callback, against the context, that encrypts then MACs.
 *
 * ctx  SSL/TLS context.
 * cb   Callback function to use with Encrypt-Then-MAC.
 */
void  wolfSSL_CTX_SetEncryptMacCb(WOLFSSL_CTX* ctx, CallbackEncryptMac cb)
{
    if (ctx)
        ctx->EncryptMacCb = cb;
}

/**
 * Set the context to use with callback that encrypts then MACs.
 *
 * ssl  SSL/TLS object.
 * ctx  Callback function's context.
 */
void  wolfSSL_SetEncryptMacCtx(WOLFSSL* ssl, void *ctx)
{
    if (ssl)
        ssl->EncryptMacCtx = ctx;
}

/**
 * Get the context being used with callback that encrypts then MACs.
 *
 * ssl  SSL/TLS object.
 * returns callback function's context or NULL if SSL/TLS object is NULL.
 */
void* wolfSSL_GetEncryptMacCtx(WOLFSSL* ssl)
{
    if (ssl)
        return ssl->EncryptMacCtx;

    return NULL;
}


/**
 * Set the callback, against the context, that MAC verifies then decrypts.
 *
 * ctx  SSL/TLS context.
 * cb   Callback function to use with Encrypt-Then-MAC.
 */
void  wolfSSL_CTX_SetVerifyDecryptCb(WOLFSSL_CTX* ctx, CallbackVerifyDecrypt cb)
{
    if (ctx)
        ctx->VerifyDecryptCb = cb;
}

/**
 * Set the context to use with callback that MAC verifies then decrypts.
 *
 * ssl  SSL/TLS object.
 * ctx  Callback function's context.
 */
void  wolfSSL_SetVerifyDecryptCtx(WOLFSSL* ssl, void *ctx)
{
    if (ssl)
        ssl->VerifyDecryptCtx = ctx;
}

/**
 * Get the context being used with callback that MAC verifies then decrypts.
 *
 * ssl  SSL/TLS object.
 * returns callback function's context or NULL if SSL/TLS object is NULL.
 */
void* wolfSSL_GetVerifyDecryptCtx(WOLFSSL* ssl)
{
    if (ssl)
        return ssl->VerifyDecryptCtx;

    return NULL;
}
#endif /* HAVE_ENCRYPT_THEN_MAC !WOLFSSL_AEAD_ONLY */



const byte* wolfSSL_GetClientWriteKey(WOLFSSL* ssl)
{
    if (ssl)
        return ssl->keys.client_write_key;

    return NULL;
}


const byte* wolfSSL_GetClientWriteIV(WOLFSSL* ssl)
{
    if (ssl)
        return ssl->keys.client_write_IV;

    return NULL;
}


const byte* wolfSSL_GetServerWriteKey(WOLFSSL* ssl)
{
    if (ssl)
        return ssl->keys.server_write_key;

    return NULL;
}


const byte* wolfSSL_GetServerWriteIV(WOLFSSL* ssl)
{
    if (ssl)
        return ssl->keys.server_write_IV;

    return NULL;
}

int wolfSSL_GetKeySize(WOLFSSL* ssl)
{
    if (ssl)
        return ssl->specs.key_size;

    return BAD_FUNC_ARG;
}


int wolfSSL_GetIVSize(WOLFSSL* ssl)
{
    if (ssl)
        return ssl->specs.iv_size;

    return BAD_FUNC_ARG;
}


int wolfSSL_GetBulkCipher(WOLFSSL* ssl)
{
    if (ssl)
        return ssl->specs.bulk_cipher_algorithm;

    return BAD_FUNC_ARG;
}


int wolfSSL_GetCipherType(WOLFSSL* ssl)
{
    if (ssl == NULL)
        return BAD_FUNC_ARG;

#ifndef WOLFSSL_AEAD_ONLY
    if (ssl->specs.cipher_type == block)
        return WOLFSSL_BLOCK_TYPE;
    if (ssl->specs.cipher_type == stream)
        return WOLFSSL_STREAM_TYPE;
#endif
    if (ssl->specs.cipher_type == aead)
        return WOLFSSL_AEAD_TYPE;

    return -1;
}


int wolfSSL_GetCipherBlockSize(WOLFSSL* ssl)
{
    if (ssl == NULL)
        return BAD_FUNC_ARG;

    return ssl->specs.block_size;
}


int wolfSSL_GetAeadMacSize(WOLFSSL* ssl)
{
    if (ssl == NULL)
        return BAD_FUNC_ARG;

    return ssl->specs.aead_mac_size;
}


int wolfSSL_IsTLSv1_1(WOLFSSL* ssl)
{
    if (ssl == NULL)
        return BAD_FUNC_ARG;

    if (ssl->options.tls1_1)
        return 1;

    return 0;
}


int wolfSSL_GetSide(WOLFSSL* ssl)
{
    if (ssl)
        return ssl->options.side;

    return BAD_FUNC_ARG;
}


int wolfSSL_GetHmacSize(WOLFSSL* ssl)
{
    /* AEAD ciphers don't have HMAC keys */
    if (ssl)
        return (ssl->specs.cipher_type != aead) ? ssl->specs.hash_size : 0;

    return BAD_FUNC_ARG;
}

#ifdef WORD64_AVAILABLE
int wolfSSL_GetPeerSequenceNumber(WOLFSSL* ssl, word64 *seq)
{
    if ((ssl == NULL) || (seq == NULL))
        return BAD_FUNC_ARG;

    *seq = ((word64)ssl->keys.peer_sequence_number_hi << 32) |
                    ssl->keys.peer_sequence_number_lo;
    return !(*seq);
}

int wolfSSL_GetSequenceNumber(WOLFSSL* ssl, word64 *seq)
{
    if ((ssl == NULL) || (seq == NULL))
        return BAD_FUNC_ARG;

    *seq = ((word64)ssl->keys.sequence_number_hi << 32) |
                    ssl->keys.sequence_number_lo;
    return !(*seq);
}
#endif

#endif /* ATOMIC_USER */

#ifndef NO_CERTS

WOLFSSL_CERT_MANAGER* wolfSSL_CTX_GetCertManager(WOLFSSL_CTX* ctx)
{
    WOLFSSL_CERT_MANAGER* cm = NULL;
    if (ctx)
        cm = ctx->cm;
    return cm;
}

WOLFSSL_CERT_MANAGER* wolfSSL_CertManagerNew_ex(void* heap)
{
    WOLFSSL_CERT_MANAGER* cm;

    WOLFSSL_ENTER("wolfSSL_CertManagerNew");

    cm = (WOLFSSL_CERT_MANAGER*) XMALLOC(sizeof(WOLFSSL_CERT_MANAGER), heap,
                                         DYNAMIC_TYPE_CERT_MANAGER);
    if (cm) {
        XMEMSET(cm, 0, sizeof(WOLFSSL_CERT_MANAGER));
        cm->refCount = 1;

        if (wc_InitMutex(&cm->caLock) != 0) {
            WOLFSSL_MSG("Bad mutex init");
            wolfSSL_CertManagerFree(cm);
            return NULL;
        }
        #ifndef SINGLE_THREADED
        if (wc_InitMutex(&cm->refMutex) != 0) {
            WOLFSSL_MSG("Bad mutex init");
            wolfSSL_CertManagerFree(cm);
            return NULL;
        }
        #endif

        #ifdef WOLFSSL_TRUST_PEER_CERT
        if (wc_InitMutex(&cm->tpLock) != 0) {
            WOLFSSL_MSG("Bad mutex init");
            wolfSSL_CertManagerFree(cm);
            return NULL;
        }
        #endif

        /* set default minimum key size allowed */
        #ifndef NO_RSA
            cm->minRsaKeySz = MIN_RSAKEY_SZ;
        #endif
        #ifdef HAVE_ECC
            cm->minEccKeySz = MIN_ECCKEY_SZ;
        #endif
        #ifdef HAVE_PQC
            cm->minFalconKeySz = MIN_FALCONKEY_SZ;
        #endif

            cm->heap = heap;
    }

    return cm;
}


WOLFSSL_CERT_MANAGER* wolfSSL_CertManagerNew(void)
{
    return wolfSSL_CertManagerNew_ex(NULL);
}


void wolfSSL_CertManagerFree(WOLFSSL_CERT_MANAGER* cm)
{
    int doFree = 0;
    WOLFSSL_ENTER("wolfSSL_CertManagerFree");

    if (cm) {
        #ifndef SINGLE_THREADED
        if (wc_LockMutex(&cm->refMutex) != 0) {
            WOLFSSL_MSG("Couldn't lock cm mutex");
        }
        #endif
        cm->refCount--;
        if (cm->refCount == 0)
            doFree = 1;
        #ifndef SINGLE_THREADED
        wc_UnLockMutex(&cm->refMutex);
        #endif
        if (doFree) {
            #ifdef HAVE_CRL
                if (cm->crl)
                    FreeCRL(cm->crl, 1);
            #endif
            #ifdef HAVE_OCSP
                if (cm->ocsp)
                    FreeOCSP(cm->ocsp, 1);
                XFREE(cm->ocspOverrideURL, cm->heap, DYNAMIC_TYPE_URL);
            #if !defined(NO_WOLFSSL_SERVER) && \
                (defined(HAVE_CERTIFICATE_STATUS_REQUEST) || \
                 defined(HAVE_CERTIFICATE_STATUS_REQUEST_V2))
                if (cm->ocsp_stapling)
                    FreeOCSP(cm->ocsp_stapling, 1);
            #endif
            #endif
            FreeSignerTable(cm->caTable, CA_TABLE_SIZE, cm->heap);
            wc_FreeMutex(&cm->caLock);

            #ifdef WOLFSSL_TRUST_PEER_CERT
            FreeTrustedPeerTable(cm->tpTable, TP_TABLE_SIZE, cm->heap);
            wc_FreeMutex(&cm->tpLock);
            #endif
            #ifndef SINGLE_THREADED
            if (wc_FreeMutex(&cm->refMutex) != 0) {
                WOLFSSL_MSG("Couldn't free refMutex mutex");
            }
            #endif
            XFREE(cm, cm->heap, DYNAMIC_TYPE_CERT_MANAGER);
        }
    }

}

int wolfSSL_CertManager_up_ref(WOLFSSL_CERT_MANAGER* cm)
{
    if (cm) {
#ifndef SINGLE_THREADED
        if (wc_LockMutex(&cm->refMutex) != 0) {
            WOLFSSL_MSG("Failed to lock cm mutex");
            return WOLFSSL_FAILURE;
        }
#endif
        cm->refCount++;
#ifndef SINGLE_THREADED
        wc_UnLockMutex(&cm->refMutex);
#endif

        return WOLFSSL_SUCCESS;
    }

    return WOLFSSL_FAILURE;
}

#if defined(OPENSSL_EXTRA) && !defined(NO_FILESYSTEM)
#if defined(WOLFSSL_SIGNER_DER_CERT)
/******************************************************************************
* wolfSSL_CertManagerGetCerts - retrieve stack of X509 certificates in a
* certificate manager (CM).
*
* RETURNS:
* returns stack of X509 certs on success, otherwise returns a NULL.
*/
WOLFSSL_STACK* wolfSSL_CertManagerGetCerts(WOLFSSL_CERT_MANAGER* cm)
{
    WOLFSSL_STACK* sk = NULL;
    int numCerts = 0;
    DerBuffer** certBuffers = NULL;
    const byte* derBuffer = NULL;
    Signer* signers = NULL;
    word32  row = 0;
    WOLFSSL_X509* x509 = NULL;
    int i = 0;
    int ret = 0;

    if (cm == NULL)
        return NULL;

    sk = wolfSSL_sk_X509_new();
    if (sk == NULL)
        goto error;

    if (wc_LockMutex(&cm->caLock) != 0)
        goto error;

    /* Iterate once to get the number of certs, for memory allocation
       purposes. */
    for (row = 0; row < CA_TABLE_SIZE; row++) {
        signers = cm->caTable[row];
        while (signers && signers->derCert && signers->derCert->buffer) {
            ++numCerts;
            signers = signers->next;
        }
    }

    if (numCerts == 0) {
        wc_UnLockMutex(&cm->caLock);
        goto error;
    }

    certBuffers = (DerBuffer**)XMALLOC(sizeof(DerBuffer*) * numCerts, cm->heap,
                                       DYNAMIC_TYPE_TMP_BUFFER);
    if (certBuffers == NULL) {
        wc_UnLockMutex(&cm->caLock);
        goto error;
    }
    XMEMSET(certBuffers, 0, sizeof(DerBuffer*) * numCerts);

    /* Copy the certs locally so that we can release the caLock. If the lock is
       held when wolfSSL_d2i_X509 is called, GetCA will also try to get the
       lock, leading to deadlock. */
    for (row = 0; row < CA_TABLE_SIZE; row++) {
        signers = cm->caTable[row];
        while (signers && signers->derCert && signers->derCert->buffer) {
            ret = AllocDer(&certBuffers[i], signers->derCert->length, CA_TYPE,
                           cm->heap);
            if (ret < 0) {
                wc_UnLockMutex(&cm->caLock);
                goto error;
            }

            XMEMCPY(certBuffers[i]->buffer, signers->derCert->buffer,
                    signers->derCert->length);
            certBuffers[i]->length = signers->derCert->length;

            ++i;
            signers = signers->next;
        }
    }

    wc_UnLockMutex(&cm->caLock);

    for (i = 0; i < numCerts; ++i) {
        derBuffer = certBuffers[i]->buffer;
        wolfSSL_d2i_X509(&x509, &derBuffer, certBuffers[i]->length);
        if (x509 == NULL)
            goto error;

        if (wolfSSL_sk_X509_push(sk, x509) != WOLFSSL_SUCCESS)
            goto error;
    }

    for (i = 0; i < numCerts && certBuffers[i] != NULL; ++i) {
        FreeDer(&certBuffers[i]);
    }

    XFREE(certBuffers, cm->heap, DYNAMIC_TYPE_TMP_BUFFER);

    return sk;

error:
    if (sk)
        wolfSSL_sk_X509_pop_free(sk, NULL);

    if (certBuffers != NULL) {
        for (i = 0; i < numCerts && certBuffers[i] != NULL; ++i) {
            FreeDer(&certBuffers[i]);
        }
    }

    if (certBuffers)
        XFREE(certBuffers, cm->heap, DYNAMIC_TYPE_TMP_BUFFER);

    return NULL;
}

#endif /* WOLFSSL_SIGNER_DER_CERT */
#endif /* OPENSSL_EXTRA && !NO_FILESYSTEM */

/* Unload the CA signer list */
int wolfSSL_CertManagerUnloadCAs(WOLFSSL_CERT_MANAGER* cm)
{
    WOLFSSL_ENTER("wolfSSL_CertManagerUnloadCAs");

    if (cm == NULL)
        return BAD_FUNC_ARG;

    if (wc_LockMutex(&cm->caLock) != 0)
        return BAD_MUTEX_E;

    FreeSignerTable(cm->caTable, CA_TABLE_SIZE, cm->heap);

    wc_UnLockMutex(&cm->caLock);


    return WOLFSSL_SUCCESS;
}


#ifdef WOLFSSL_TRUST_PEER_CERT
int wolfSSL_CertManagerUnload_trust_peers(WOLFSSL_CERT_MANAGER* cm)
{
    WOLFSSL_ENTER("wolfSSL_CertManagerUnload_trust_peers");

    if (cm == NULL)
        return BAD_FUNC_ARG;

    if (wc_LockMutex(&cm->tpLock) != 0)
        return BAD_MUTEX_E;

    FreeTrustedPeerTable(cm->tpTable, TP_TABLE_SIZE, cm->heap);

    wc_UnLockMutex(&cm->tpLock);


    return WOLFSSL_SUCCESS;
}
#endif /* WOLFSSL_TRUST_PEER_CERT */

#endif /* NO_CERTS */

#if !defined(NO_FILESYSTEM) && !defined(NO_STDIO_FILESYSTEM)

void wolfSSL_ERR_print_errors_fp(XFILE fp, int err)
{
    char data[WOLFSSL_MAX_ERROR_SZ + 1];

    WOLFSSL_ENTER("wolfSSL_ERR_print_errors_fp");
    SetErrorString(err, data);
    XFPRINTF(fp, "%s", data);
}

#if defined(OPENSSL_EXTRA) || defined(DEBUG_WOLFSSL_VERBOSE)
void wolfSSL_ERR_dump_errors_fp(XFILE fp)
{
    wc_ERR_print_errors_fp(fp);
}

void wolfSSL_ERR_print_errors_cb (int (*cb)(const char *str, size_t len,
                                            void *u), void *u)
{
    wc_ERR_print_errors_cb(cb, u);
}
#endif
#endif

/*
 * TODO This ssl parameter needs to be changed to const once our ABI checker
 *      stops flagging qualifier additions as ABI breaking.
 */
WOLFSSL_ABI
int wolfSSL_pending(WOLFSSL* ssl)
{
    WOLFSSL_ENTER("SSL_pending");
    if (ssl == NULL)
        return WOLFSSL_FAILURE;

    return ssl->buffers.clearOutputBuffer.length;
}

int wolfSSL_has_pending(const WOLFSSL* ssl)
{
    WOLFSSL_ENTER("wolfSSL_has_pending");
    if (ssl == NULL)
        return WOLFSSL_FAILURE;

    return ssl->buffers.clearOutputBuffer.length > 0;
}

#ifndef WOLFSSL_LEANPSK
/* turn on handshake group messages for context */
int wolfSSL_CTX_set_group_messages(WOLFSSL_CTX* ctx)
{
    if (ctx == NULL)
       return BAD_FUNC_ARG;

    ctx->groupMessages = 1;

    return WOLFSSL_SUCCESS;
}
#endif


#ifndef NO_WOLFSSL_CLIENT
/* connect enough to get peer cert chain */
int wolfSSL_connect_cert(WOLFSSL* ssl)
{
    int  ret;

    if (ssl == NULL)
        return WOLFSSL_FAILURE;

    ssl->options.certOnly = 1;
    ret = wolfSSL_connect(ssl);
    ssl->options.certOnly   = 0;

    return ret;
}
#endif


#ifndef WOLFSSL_LEANPSK
/* turn on handshake group messages for ssl object */
int wolfSSL_set_group_messages(WOLFSSL* ssl)
{
    if (ssl == NULL)
       return BAD_FUNC_ARG;

    ssl->options.groupMessages = 1;

    return WOLFSSL_SUCCESS;
}


/* make minVersion the internal equivalent SSL version */
static int SetMinVersionHelper(byte* minVersion, int version)
{
#ifdef NO_TLS
    (void)minVersion;
#endif

    switch (version) {
#if defined(WOLFSSL_ALLOW_SSLV3) && !defined(NO_OLD_TLS)
        case WOLFSSL_SSLV3:
            *minVersion = SSLv3_MINOR;
            break;
#endif

#ifndef NO_TLS
    #ifndef NO_OLD_TLS
        #ifdef WOLFSSL_ALLOW_TLSV10
        case WOLFSSL_TLSV1:
            *minVersion = TLSv1_MINOR;
            break;
        #endif

        case WOLFSSL_TLSV1_1:
            *minVersion = TLSv1_1_MINOR;
            break;
    #endif
    #ifndef WOLFSSL_NO_TLS12
        case WOLFSSL_TLSV1_2:
            *minVersion = TLSv1_2_MINOR;
            break;
    #endif
#endif
    #ifdef WOLFSSL_TLS13
        case WOLFSSL_TLSV1_3:
            *minVersion = TLSv1_3_MINOR;
            break;
    #endif

        default:
            WOLFSSL_MSG("Bad function argument");
            return BAD_FUNC_ARG;
    }

    return WOLFSSL_SUCCESS;
}


/* Set minimum downgrade version allowed, WOLFSSL_SUCCESS on ok */
WOLFSSL_ABI
int wolfSSL_CTX_SetMinVersion(WOLFSSL_CTX* ctx, int version)
{
    WOLFSSL_ENTER("wolfSSL_CTX_SetMinVersion");

    if (ctx == NULL) {
        WOLFSSL_MSG("Bad function argument");
        return BAD_FUNC_ARG;
    }

    return SetMinVersionHelper(&ctx->minDowngrade, version);
}


/* Set minimum downgrade version allowed, WOLFSSL_SUCCESS on ok */
int wolfSSL_SetMinVersion(WOLFSSL* ssl, int version)
{
    WOLFSSL_ENTER("wolfSSL_SetMinVersion");

    if (ssl == NULL) {
        WOLFSSL_MSG("Bad function argument");
        return BAD_FUNC_ARG;
    }

    return SetMinVersionHelper(&ssl->options.minDowngrade, version);
}


/* Function to get version as WOLFSSL_ enum value for wolfSSL_SetVersion */
int wolfSSL_GetVersion(const WOLFSSL* ssl)
{
    if (ssl == NULL)
        return BAD_FUNC_ARG;

    if (ssl->version.major == SSLv3_MAJOR) {
        switch (ssl->version.minor) {
            case SSLv3_MINOR :
                return WOLFSSL_SSLV3;
            case TLSv1_MINOR :
                return WOLFSSL_TLSV1;
            case TLSv1_1_MINOR :
                return WOLFSSL_TLSV1_1;
            case TLSv1_2_MINOR :
                return WOLFSSL_TLSV1_2;
            case TLSv1_3_MINOR :
                return WOLFSSL_TLSV1_3;
            default:
                break;
        }
    }

    return VERSION_ERROR;
}

int wolfSSL_SetVersion(WOLFSSL* ssl, int version)
{
    word16 haveRSA = 1;
    word16 havePSK = 0;
    int    keySz   = 0;

    WOLFSSL_ENTER("wolfSSL_SetVersion");

    if (ssl == NULL) {
        WOLFSSL_MSG("Bad function argument");
        return BAD_FUNC_ARG;
    }

    switch (version) {
#if defined(WOLFSSL_ALLOW_SSLV3) && !defined(NO_OLD_TLS)
        case WOLFSSL_SSLV3:
            ssl->version = MakeSSLv3();
            break;
#endif

#ifndef NO_TLS
    #ifndef NO_OLD_TLS
        #ifdef WOLFSSL_ALLOW_TLSV10
        case WOLFSSL_TLSV1:
            ssl->version = MakeTLSv1();
            break;
        #endif

        case WOLFSSL_TLSV1_1:
            ssl->version = MakeTLSv1_1();
            break;
    #endif
    #ifndef WOLFSSL_NO_TLS12
        case WOLFSSL_TLSV1_2:
            ssl->version = MakeTLSv1_2();
            break;
    #endif
#endif
#ifdef WOLFSSL_TLS13
        case WOLFSSL_TLSV1_3:
            ssl->version = MakeTLSv1_3();
            break;

#endif

        default:
            WOLFSSL_MSG("Bad function argument");
            return BAD_FUNC_ARG;
    }

    #ifdef NO_RSA
        haveRSA = 0;
    #endif
    #ifndef NO_PSK
        havePSK = ssl->options.havePSK;
    #endif
    #ifndef NO_CERTS
        keySz = ssl->buffers.keySz;
    #endif

    InitSuites(ssl->suites, ssl->version, keySz, haveRSA, havePSK,
               ssl->options.haveDH, ssl->options.haveECDSAsig,
               ssl->options.haveECC, ssl->options.haveStaticECC,
               ssl->options.haveFalconSig, ssl->options.haveAnon,
               ssl->options.side);

    return WOLFSSL_SUCCESS;
}
#endif /* !leanpsk */


#if !defined(NO_CERTS) || !defined(NO_SESSION_CACHE)

/* Make a work from the front of random hash */
static WC_INLINE word32 MakeWordFromHash(const byte* hashID)
{
    return ((word32)hashID[0] << 24) | ((word32)hashID[1] << 16) |
           ((word32)hashID[2] <<  8) |  (word32)hashID[3];
}

#endif /* !NO_CERTS || !NO_SESSION_CACHE */


#ifndef NO_CERTS

/* hash is the SHA digest of name, just use first 32 bits as hash */
static WC_INLINE word32 HashSigner(const byte* hash)
{
    return MakeWordFromHash(hash) % CA_TABLE_SIZE;
}


/* does CA already exist on signer list */
int AlreadySigner(WOLFSSL_CERT_MANAGER* cm, byte* hash)
{
    Signer* signers;
    int     ret = 0;
    word32  row;

    if (cm == NULL || hash == NULL) {
        return ret;
    }

    row = HashSigner(hash);

    if (wc_LockMutex(&cm->caLock) != 0) {
        return ret;
    }
    signers = cm->caTable[row];
    while (signers) {
        byte* subjectHash;

    #ifndef NO_SKID
        subjectHash = signers->subjectKeyIdHash;
    #else
        subjectHash = signers->subjectNameHash;
    #endif

        if (XMEMCMP(hash, subjectHash, SIGNER_DIGEST_SIZE) == 0) {
            ret = 1; /* success */
            break;
        }
        signers = signers->next;
    }
    wc_UnLockMutex(&cm->caLock);

    return ret;
}


#ifdef WOLFSSL_TRUST_PEER_CERT
/* hash is the SHA digest of name, just use first 32 bits as hash */
static WC_INLINE word32 TrustedPeerHashSigner(const byte* hash)
{
    return MakeWordFromHash(hash) % TP_TABLE_SIZE;
}

/* does trusted peer already exist on signer list */
int AlreadyTrustedPeer(WOLFSSL_CERT_MANAGER* cm, DecodedCert* cert)
{
    TrustedPeerCert* tp;
    int     ret = 0;
    word32  row = TrustedPeerHashSigner(cert->subjectHash);

    if (wc_LockMutex(&cm->tpLock) != 0)
        return  ret;
    tp = cm->tpTable[row];
    while (tp) {
        if (XMEMCMP(cert->subjectHash, tp->subjectNameHash,
                SIGNER_DIGEST_SIZE) == 0)
            ret = 1;
    #ifndef NO_SKID
        if (cert->extSubjKeyIdSet) {
            /* Compare SKID as well if available */
            if (ret == 1 && XMEMCMP(cert->extSubjKeyId, tp->subjectKeyIdHash,
                    SIGNER_DIGEST_SIZE) != 0)
                ret = 0;
        }
    #endif
        if (ret == 1)
            break;
        tp = tp->next;
    }
    wc_UnLockMutex(&cm->tpLock);

    return ret;
}


/* return Trusted Peer if found, otherwise NULL
    type is what to match on
 */
TrustedPeerCert* GetTrustedPeer(void* vp, DecodedCert* cert)
{
    WOLFSSL_CERT_MANAGER* cm = (WOLFSSL_CERT_MANAGER*)vp;
    TrustedPeerCert* ret = NULL;
    TrustedPeerCert* tp  = NULL;
    word32  row;

    if (cm == NULL || cert == NULL)
        return NULL;

    row = TrustedPeerHashSigner(cert->subjectHash);

    if (wc_LockMutex(&cm->tpLock) != 0)
        return ret;

    tp = cm->tpTable[row];
    while (tp) {
        if (XMEMCMP(cert->subjectHash, tp->subjectNameHash,
                SIGNER_DIGEST_SIZE) == 0)
            ret = tp;
    #ifndef NO_SKID
        if (cert->extSubjKeyIdSet) {
            /* Compare SKID as well if available */
            if (ret != NULL && XMEMCMP(cert->extSubjKeyId, tp->subjectKeyIdHash,
                    SIGNER_DIGEST_SIZE) != 0)
                ret = NULL;
        }
    #endif
        if (ret != NULL)
            break;
        tp = tp->next;
    }
    wc_UnLockMutex(&cm->tpLock);

    return ret;
}


int MatchTrustedPeer(TrustedPeerCert* tp, DecodedCert* cert)
{
    if (tp == NULL || cert == NULL)
        return BAD_FUNC_ARG;

    /* subject key id or subject hash has been compared when searching
       tpTable for the cert from function GetTrustedPeer */

    /* compare signatures */
    if (tp->sigLen == cert->sigLength) {
        if (XMEMCMP(tp->sig, cert->signature, cert->sigLength)) {
            return WOLFSSL_FAILURE;
        }
    }
    else {
        return WOLFSSL_FAILURE;
    }

    return WOLFSSL_SUCCESS;
}
#endif /* WOLFSSL_TRUST_PEER_CERT */


/* return CA if found, otherwise NULL */
Signer* GetCA(void* vp, byte* hash)
{
    WOLFSSL_CERT_MANAGER* cm = (WOLFSSL_CERT_MANAGER*)vp;
    Signer* ret = NULL;
    Signer* signers;
    word32  row = 0;

    if (cm == NULL || hash == NULL)
        return NULL;

    row = HashSigner(hash);

    if (wc_LockMutex(&cm->caLock) != 0)
        return ret;

    signers = cm->caTable[row];
    while (signers) {
        byte* subjectHash;
        #ifndef NO_SKID
            subjectHash = signers->subjectKeyIdHash;
        #else
            subjectHash = signers->subjectNameHash;
        #endif
        if (XMEMCMP(hash, subjectHash, SIGNER_DIGEST_SIZE) == 0) {
            ret = signers;
            break;
        }
        signers = signers->next;
    }
    wc_UnLockMutex(&cm->caLock);

    return ret;
}


#ifndef NO_SKID
/* return CA if found, otherwise NULL. Walk through hash table. */
Signer* GetCAByName(void* vp, byte* hash)
{
    WOLFSSL_CERT_MANAGER* cm = (WOLFSSL_CERT_MANAGER*)vp;
    Signer* ret = NULL;
    Signer* signers;
    word32  row;

    if (cm == NULL)
        return NULL;

    if (wc_LockMutex(&cm->caLock) != 0)
        return ret;

    for (row = 0; row < CA_TABLE_SIZE && ret == NULL; row++) {
        signers = cm->caTable[row];
        while (signers && ret == NULL) {
            if (XMEMCMP(hash, signers->subjectNameHash,
                        SIGNER_DIGEST_SIZE) == 0) {
                ret = signers;
            }
            signers = signers->next;
        }
    }
    wc_UnLockMutex(&cm->caLock);

    return ret;
}
#endif


#ifdef WOLFSSL_TRUST_PEER_CERT
/* add a trusted peer cert to linked list */
int AddTrustedPeer(WOLFSSL_CERT_MANAGER* cm, DerBuffer** pDer, int verify)
{
    int ret, row;
    TrustedPeerCert* peerCert;
    DecodedCert* cert;
    DerBuffer*   der = *pDer;

    WOLFSSL_MSG("Adding a Trusted Peer Cert");

    cert = (DecodedCert*)XMALLOC(sizeof(DecodedCert), cm->heap,
                                 DYNAMIC_TYPE_DCERT);
    if (cert == NULL) {
        FreeDer(&der);
        return MEMORY_E;
    }

    InitDecodedCert(cert, der->buffer, der->length, cm->heap);
    if ((ret = ParseCert(cert, TRUSTED_PEER_TYPE, verify, cm)) != 0) {
        FreeDecodedCert(cert);
        XFREE(cert, NULL, DYNAMIC_TYPE_DCERT);
        FreeDer(&der);
        return ret;
    }
    WOLFSSL_MSG("\tParsed new trusted peer cert");

    peerCert = (TrustedPeerCert*)XMALLOC(sizeof(TrustedPeerCert), cm->heap,
                                                             DYNAMIC_TYPE_CERT);
    if (peerCert == NULL) {
        FreeDecodedCert(cert);
        XFREE(cert, cm->heap, DYNAMIC_TYPE_DCERT);
        FreeDer(&der);
        return MEMORY_E;
    }
    XMEMSET(peerCert, 0, sizeof(TrustedPeerCert));

    #ifndef IGNORE_NAME_CONSTRAINTS
        if (peerCert->permittedNames)
            FreeNameSubtrees(peerCert->permittedNames, cm->heap);
        if (peerCert->excludedNames)
            FreeNameSubtrees(peerCert->excludedNames, cm->heap);
    #endif

    if (AlreadyTrustedPeer(cm, cert)) {
        WOLFSSL_MSG("\tAlready have this CA, not adding again");
        FreeTrustedPeer(peerCert, cm->heap);
        (void)ret;
    }
    else {
        /* add trusted peer signature */
        peerCert->sigLen = cert->sigLength;
        peerCert->sig = (byte *)XMALLOC(cert->sigLength, cm->heap,
                                                        DYNAMIC_TYPE_SIGNATURE);
        if (peerCert->sig == NULL) {
            FreeDecodedCert(cert);
            XFREE(cert, cm->heap, DYNAMIC_TYPE_DCERT);
            FreeTrustedPeer(peerCert, cm->heap);
            FreeDer(&der);
            return MEMORY_E;
        }
        XMEMCPY(peerCert->sig, cert->signature, cert->sigLength);

        /* add trusted peer name */
        peerCert->nameLen = cert->subjectCNLen;
        peerCert->name    = cert->subjectCN;
        #ifndef IGNORE_NAME_CONSTRAINTS
            peerCert->permittedNames = cert->permittedNames;
            peerCert->excludedNames  = cert->excludedNames;
        #endif

        /* add SKID when available and hash of name */
        #ifndef NO_SKID
            XMEMCPY(peerCert->subjectKeyIdHash, cert->extSubjKeyId,
                    SIGNER_DIGEST_SIZE);
        #endif
            XMEMCPY(peerCert->subjectNameHash, cert->subjectHash,
                    SIGNER_DIGEST_SIZE);
            peerCert->next    = NULL; /* If Key Usage not set, all uses valid. */
            cert->subjectCN = 0;
        #ifndef IGNORE_NAME_CONSTRAINTS
            cert->permittedNames = NULL;
            cert->excludedNames = NULL;
        #endif

            row = TrustedPeerHashSigner(peerCert->subjectNameHash);

            if (wc_LockMutex(&cm->tpLock) == 0) {
                peerCert->next = cm->tpTable[row];
                cm->tpTable[row] = peerCert;   /* takes ownership */
                wc_UnLockMutex(&cm->tpLock);
            }
            else {
                WOLFSSL_MSG("\tTrusted Peer Cert Mutex Lock failed");
                FreeDecodedCert(cert);
                XFREE(cert, cm->heap, DYNAMIC_TYPE_DCERT);
                FreeTrustedPeer(peerCert, cm->heap);
                FreeDer(&der);
                return BAD_MUTEX_E;
            }
        }

    WOLFSSL_MSG("\tFreeing parsed trusted peer cert");
    FreeDecodedCert(cert);
    XFREE(cert, cm->heap, DYNAMIC_TYPE_DCERT);
    WOLFSSL_MSG("\tFreeing der trusted peer cert");
    FreeDer(&der);
    WOLFSSL_MSG("\t\tOK Freeing der trusted peer cert");
    WOLFSSL_LEAVE("AddTrustedPeer", ret);

    return WOLFSSL_SUCCESS;
}
#endif /* WOLFSSL_TRUST_PEER_CERT */


/* owns der, internal now uses too */
/* type flag ids from user or from chain received during verify
   don't allow chain ones to be added w/o isCA extension */
int AddCA(WOLFSSL_CERT_MANAGER* cm, DerBuffer** pDer, int type, int verify)
{
    int         ret;
    Signer*     signer = NULL;
    word32      row;
    byte*       subjectHash;
#ifdef WOLFSSL_SMALL_STACK
    DecodedCert* cert = NULL;
#else
    DecodedCert  cert[1];
#endif
    DerBuffer*   der = *pDer;

    WOLFSSL_MSG("Adding a CA");

    if (cm == NULL) {
        FreeDer(pDer);
        return BAD_FUNC_ARG;
    }

#ifdef WOLFSSL_SMALL_STACK
    cert = (DecodedCert*)XMALLOC(sizeof(DecodedCert), NULL,
                                 DYNAMIC_TYPE_DCERT);
    if (cert == NULL) {
        FreeDer(pDer);
        return MEMORY_E;
    }
#endif

    InitDecodedCert(cert, der->buffer, der->length, cm->heap);
    ret = ParseCert(cert, CA_TYPE, verify, cm);
    WOLFSSL_MSG("\tParsed new CA");

#ifndef NO_SKID
    subjectHash = cert->extSubjKeyId;
#else
    subjectHash = cert->subjectHash;
#endif

    /* check CA key size */
    if (verify) {
        switch (cert->keyOID) {
            #ifndef NO_RSA
            case RSAk:
                if (cm->minRsaKeySz < 0 ||
                                   cert->pubKeySize < (word16)cm->minRsaKeySz) {
                    ret = RSA_KEY_SIZE_E;
                    WOLFSSL_MSG("\tCA RSA key size error");
                }
                break;
            #endif /* !NO_RSA */
            #ifdef HAVE_ECC
            case ECDSAk:
                if (cm->minEccKeySz < 0 ||
                                   cert->pubKeySize < (word16)cm->minEccKeySz) {
                    ret = ECC_KEY_SIZE_E;
                    WOLFSSL_MSG("\tCA ECC key size error");
                }
                break;
            #endif /* HAVE_ECC */
            #ifdef HAVE_ED25519
            case ED25519k:
                if (cm->minEccKeySz < 0 ||
                                   ED25519_KEY_SIZE < (word16)cm->minEccKeySz) {
                    ret = ECC_KEY_SIZE_E;
                    WOLFSSL_MSG("\tCA ECC key size error");
                }
                break;
            #endif /* HAVE_ED25519 */
            #ifdef HAVE_ED448
            case ED448k:
                if (cm->minEccKeySz < 0 ||
                                     ED448_KEY_SIZE < (word16)cm->minEccKeySz) {
                    ret = ECC_KEY_SIZE_E;
                    WOLFSSL_MSG("\tCA ECC key size error");
                }
                break;
            #endif /* HAVE_ED448 */
            #if defined(HAVE_PQC) && defined(HAVE_FALCON)
            case FALCON_LEVEL1k:
                if (cm->minFalconKeySz < 0 ||
                          FALCON_LEVEL1_KEY_SIZE < (word16)cm->minFalconKeySz) {
                    ret = FALCON_KEY_SIZE_E;
                    WOLFSSL_MSG("\tCA Falcon level 1 key size error");
                }
                break;
            case FALCON_LEVEL5k:
                if (cm->minFalconKeySz < 0 ||
                          FALCON_LEVEL5_KEY_SIZE < (word16)cm->minFalconKeySz) {
                    ret = FALCON_KEY_SIZE_E;
                    WOLFSSL_MSG("\tCA Falcon level 5 key size error");
                }
                break;
            #endif /* HAVE_PQC && HAVE_FALCON */

            default:
                WOLFSSL_MSG("\tNo key size check done on CA");
                break; /* no size check if key type is not in switch */
        }
    }

    if (ret == 0 && cert->isCA == 0 && type != WOLFSSL_USER_CA) {
        WOLFSSL_MSG("\tCan't add as CA if not actually one");
        ret = NOT_CA_ERROR;
    }
#ifndef ALLOW_INVALID_CERTSIGN
    else if (ret == 0 && cert->isCA == 1 && type != WOLFSSL_USER_CA &&
        !cert->selfSigned && (cert->extKeyUsage & KEYUSE_KEY_CERT_SIGN) == 0) {
        /* Intermediate CA certs are required to have the keyCertSign
        * extension set. User loaded root certs are not. */
        WOLFSSL_MSG("\tDoesn't have key usage certificate signing");
        ret = NOT_CA_ERROR;
    }
#endif
    else if (ret == 0 && AlreadySigner(cm, subjectHash)) {
        WOLFSSL_MSG("\tAlready have this CA, not adding again");
        (void)ret;
    }
    else if (ret == 0) {
        /* take over signer parts */
        signer = MakeSigner(cm->heap);
        if (!signer)
            ret = MEMORY_ERROR;
    }
    if (ret == 0 && signer != NULL) {
    #ifdef WOLFSSL_SIGNER_DER_CERT
        ret = AllocDer(&signer->derCert, der->length, der->type, NULL);
    }
    if (ret == 0 && signer != NULL) {
        XMEMCPY(signer->derCert->buffer, der->buffer, der->length);
    #endif
        signer->keyOID         = cert->keyOID;
        if (cert->pubKeyStored) {
            signer->publicKey      = cert->publicKey;
            signer->pubKeySize     = cert->pubKeySize;
        }
        if (cert->subjectCNStored) {
            signer->nameLen        = cert->subjectCNLen;
            signer->name           = cert->subjectCN;
        }
        signer->pathLength     = cert->pathLength;
        signer->maxPathLen     = cert->maxPathLen;
        signer->pathLengthSet  = cert->pathLengthSet;
        signer->selfSigned     = cert->selfSigned;
    #ifndef IGNORE_NAME_CONSTRAINTS
        signer->permittedNames = cert->permittedNames;
        signer->excludedNames  = cert->excludedNames;
    #endif
    #ifndef NO_SKID
        XMEMCPY(signer->subjectKeyIdHash, cert->extSubjKeyId,
                SIGNER_DIGEST_SIZE);
    #endif
        XMEMCPY(signer->subjectNameHash, cert->subjectHash,
                SIGNER_DIGEST_SIZE);
    #ifdef HAVE_OCSP
        XMEMCPY(signer->subjectKeyHash, cert->subjectKeyHash,
                KEYID_SIZE);
    #endif
        signer->keyUsage = cert->extKeyUsageSet ? cert->extKeyUsage
                                                : 0xFFFF;
        signer->next    = NULL; /* If Key Usage not set, all uses valid. */
        cert->publicKey = 0;    /* in case lock fails don't free here.   */
        cert->subjectCN = 0;
    #ifndef IGNORE_NAME_CONSTRAINTS
        cert->permittedNames = NULL;
        cert->excludedNames = NULL;
    #endif

    #ifndef NO_SKID
        row = HashSigner(signer->subjectKeyIdHash);
    #else
        row = HashSigner(signer->subjectNameHash);
    #endif

        if (wc_LockMutex(&cm->caLock) == 0) {
            signer->next = cm->caTable[row];
            cm->caTable[row] = signer;   /* takes ownership */
            wc_UnLockMutex(&cm->caLock);
            if (cm->caCacheCallback)
                cm->caCacheCallback(der->buffer, (int)der->length, type);
        }
        else {
            WOLFSSL_MSG("\tCA Mutex Lock failed");
            ret = BAD_MUTEX_E;
            FreeSigner(signer, cm->heap);
        }
    }
#if defined(WOLFSSL_RENESAS_TSIP_TLS) || defined(WOLFSSL_RENESAS_SCEPROTECT)
    /* Verify CA by TSIP so that generated tsip key is going to be able to */
    /* be used for peer's cert verification                                */
    /* TSIP is only able to handle USER CA, and only one CA.               */
    /* Therefore, it doesn't need to call TSIP again if there is already   */
    /* verified CA.                                                        */
    if ( ret == 0 && signer != NULL ) {
        signer->cm_idx = row;
        if (type == WOLFSSL_USER_CA) {
            if ((ret = wc_Renesas_cmn_RootCertVerify(cert->source, cert->maxIdx,
                 cert->sigCtx.CertAtt.pubkey_n_start,
                 cert->sigCtx.CertAtt.pubkey_n_len - 1,
                 cert->sigCtx.CertAtt.pubkey_e_start,
                cert->sigCtx.CertAtt.pubkey_e_len - 1,
                 row/* cm index */))
                < 0)
                WOLFSSL_MSG("Renesas_RootCertVerify() failed");
            else
                WOLFSSL_MSG("Renesas_RootCertVerify() succeed or skipped");
        }
    }
#endif /* TSIP or SCE */

    WOLFSSL_MSG("\tFreeing Parsed CA");
    FreeDecodedCert(cert);
#ifdef WOLFSSL_SMALL_STACK
    XFREE(cert, NULL, DYNAMIC_TYPE_DCERT);
#endif
    WOLFSSL_MSG("\tFreeing der CA");
    FreeDer(pDer);
    WOLFSSL_MSG("\t\tOK Freeing der CA");

    WOLFSSL_LEAVE("AddCA", ret);

    return ret == 0 ? WOLFSSL_SUCCESS : ret;
}

#endif /* !NO_CERTS */


#ifndef NO_SESSION_CACHE

    /* basic config gives a cache with 33 sessions, adequate for clients and
       embedded servers

       TITAN_SESSION_CACHE allows just over 2 million sessions, for servers
       with titanic amounts of memory with long session ID timeouts and high
       levels of traffic.

       ENABLE_SESSION_CACHE_ROW_LOCK: Allows row level locking for increased
       performance with large session caches

       HUGE_SESSION_CACHE yields 65,791 sessions, for servers under heavy load,
       allows over 13,000 new sessions per minute or over 200 new sessions per
       second

       BIG_SESSION_CACHE yields 20,027 sessions

       MEDIUM_SESSION_CACHE allows 1055 sessions, adequate for servers that
       aren't under heavy load, basically allows 200 new sessions per minute

       SMALL_SESSION_CACHE only stores 6 sessions, good for embedded clients
       or systems where the default of nearly 3kB is too much RAM, this define
       uses less than 500 bytes RAM

       default SESSION_CACHE stores 33 sessions (no XXX_SESSION_CACHE defined)
    */
    #if defined(TITAN_SESSION_CACHE)
        #define SESSIONS_PER_ROW 31
        #define SESSION_ROWS 64937
        #ifdef ENABLE_SESSION_CACHE_ROW_LOCK
        #define ENABLE_SESSION_CACHE_ROW_LOCK
        #endif
    #elif defined(HUGE_SESSION_CACHE)
        #define SESSIONS_PER_ROW 11
        #define SESSION_ROWS 5981
    #elif defined(BIG_SESSION_CACHE)
        #define SESSIONS_PER_ROW 7
        #define SESSION_ROWS 2861
    #elif defined(MEDIUM_SESSION_CACHE)
        #define SESSIONS_PER_ROW 5
        #define SESSION_ROWS 211
    #elif defined(SMALL_SESSION_CACHE)
        #define SESSIONS_PER_ROW 2
        #define SESSION_ROWS 3
    #else
        #define SESSIONS_PER_ROW 3
        #define SESSION_ROWS 11
    #endif
    #define INVALID_SESSION_ROW (-1)

    #ifdef NO_SESSION_CACHE_ROW_LOCK
        #undef ENABLE_SESSION_CACHE_ROW_LOCK
    #endif

    typedef struct SessionRow {
        int nextIdx;                           /* where to place next one   */
        int totalCount;                        /* sessions ever on this row */
        WOLFSSL_SESSION Sessions[SESSIONS_PER_ROW];

    #ifdef ENABLE_SESSION_CACHE_ROW_LOCK
        /* not included in import/export */
        wolfSSL_Mutex row_mutex;
        int mutex_valid;
    #endif
    } SessionRow;
    #define SIZEOF_SESSION_ROW (sizeof(WOLFSSL_SESSION) + (sizeof(int) * 2))

    static WOLFSSL_GLOBAL SessionRow SessionCache[SESSION_ROWS];

    #if defined(WOLFSSL_SESSION_STATS) && defined(WOLFSSL_PEAK_SESSIONS)
        static WOLFSSL_GLOBAL word32 PeakSessions;
    #endif

    #ifdef ENABLE_SESSION_CACHE_ROW_LOCK
    #define SESSION_ROW_LOCK(row)   wc_LockMutex(&(row)->row_mutex)
    #define SESSION_ROW_UNLOCK(row) wc_UnLockMutex(&(row)->row_mutex);
    #else
    static WOLFSSL_GLOBAL wolfSSL_Mutex session_mutex; /* SessionCache mutex */
    static WOLFSSL_GLOBAL int session_mutex_valid = 0;
    #define SESSION_ROW_LOCK(row)   wc_LockMutex(&session_mutex)
    #define SESSION_ROW_UNLOCK(row) wc_UnLockMutex(&session_mutex);
    #endif

    #if !defined(NO_SESSION_CACHE_REF) && defined(NO_CLIENT_CACHE)
    #error ClientCache is required when not using NO_SESSION_CACHE_REF
    #endif

    #ifndef NO_CLIENT_CACHE

        #ifndef CLIENT_SESSIONS_MULTIPLIER
            #ifdef NO_SESSION_CACHE_REF
                #define CLIENT_SESSIONS_MULTIPLIER 1
            #else
                /* ClientSession objects are lightweight (compared to
                 * WOLFSSL_SESSION) so to decrease chance that user will reuse
                 * thse wrong session, increase the ClientCache size. This will
                 * make the entire ClientCache about the size of one
                 * WOLFSSL_SESSION object. */
                #define CLIENT_SESSIONS_MULTIPLIER 8
            #endif
        #endif
        #define CLIENT_SESSIONS_PER_ROW \
                                (SESSIONS_PER_ROW * CLIENT_SESSIONS_MULTIPLIER)
        #define CLIENT_SESSION_ROWS (SESSION_ROWS * CLIENT_SESSIONS_MULTIPLIER)

        #if CLIENT_SESSIONS_PER_ROW > 65535
        #error CLIENT_SESSIONS_PER_ROW too big
        #endif
        #if CLIENT_SESSION_ROWS > 65535
        #error CLIENT_SESSION_ROWS too big
        #endif

        struct ClientSession {
            word16 serverRow;            /* SessionCache Row id */
            word16 serverIdx;            /* SessionCache Idx (column) */
            word32 sessionIDHash;
        };
    #ifndef WOLFSSL_CLIENT_SESSION_DEFINED
        typedef struct ClientSession ClientSession;
        #define WOLFSSL_CLIENT_SESSION_DEFINED
    #endif

        typedef struct ClientRow {
            int nextIdx;                /* where to place next one   */
            int totalCount;             /* sessions ever on this row */
            ClientSession Clients[CLIENT_SESSIONS_PER_ROW];
        } ClientRow;

        static WOLFSSL_GLOBAL ClientRow ClientCache[CLIENT_SESSION_ROWS];
                                                     /* Client Cache */
                                                     /* uses session mutex */

        static WOLFSSL_GLOBAL wolfSSL_Mutex clisession_mutex; /* ClientCache mutex */
        static WOLFSSL_GLOBAL int clisession_mutex_valid = 0;
    #endif /* !NO_CLIENT_CACHE */

#endif /* !NO_SESSION_CACHE */

#if !defined(WC_NO_RNG) && (defined(OPENSSL_EXTRA) || \
    (defined(OPENSSL_EXTRA_X509_SMALL) && !defined(NO_RSA)))

    #define HAVE_GLOBAL_RNG /* consolidate flags for using globalRNG */
    static WC_RNG globalRNG;
    static int initGlobalRNG = 0;
    static wolfSSL_Mutex globalRNGMutex;
    static int globalRNGMutex_valid = 0;

    #if defined(OPENSSL_EXTRA) && defined(HAVE_HASHDRBG)
    static WOLFSSL_DRBG_CTX* gDrbgDefCtx = NULL;
    #endif
#endif

#if defined(OPENSSL_EXTRA) && !defined(WOLFSSL_NO_OPENSSL_RAND_CB)
static int wolfSSL_RAND_InitMutex(void);
#endif

#if defined(OPENSSL_EXTRA) && defined(HAVE_ATEXIT)
static void AtExitCleanup(void)
{
    if (initRefCount > 0) {
        initRefCount = 1;
        (void)wolfSSL_Cleanup();
    }
}
#endif

WOLFSSL_ABI
int wolfSSL_Init(void)
{
    int ret = WOLFSSL_SUCCESS;
#if !defined(NO_SESSION_CACHE) && defined(ENABLE_SESSION_CACHE_ROW_LOCK)
    int i;
#endif

    WOLFSSL_ENTER("wolfSSL_Init");

    #if FIPS_VERSION_GE(5,1)
        ret = wolfCrypt_SetPrivateKeyReadEnable_fips(1, WC_KEYTYPE_ALL);
        if (ret != 0)
            return ret;
        else
            ret = WOLFSSL_SUCCESS;
    #endif

    if (initRefCount == 0) {
        /* Initialize crypto for use with TLS connection */
        if (wolfCrypt_Init() != 0) {
            WOLFSSL_MSG("Bad wolfCrypt Init");
            ret = WC_INIT_E;
        }

#ifdef HAVE_GLOBAL_RNG
        if ((ret == WOLFSSL_SUCCESS) && (wc_InitMutex(&globalRNGMutex) != 0)) {
            WOLFSSL_MSG("Bad Init Mutex rng");
            ret = BAD_MUTEX_E;
        }
        else {
            globalRNGMutex_valid = 1;
        }
#endif

    #ifdef WC_RNG_SEED_CB
        wc_SetSeed_Cb(wc_GenerateSeed);
    #endif

#ifdef OPENSSL_EXTRA
    #ifndef WOLFSSL_NO_OPENSSL_RAND_CB
        if ((ret == WOLFSSL_SUCCESS) && (wolfSSL_RAND_InitMutex() != 0)) {
            ret = BAD_MUTEX_E;
        }
    #endif
        if ((ret == WOLFSSL_SUCCESS) &&
            (wolfSSL_RAND_seed(NULL, 0) != WOLFSSL_SUCCESS)) {
            WOLFSSL_MSG("wolfSSL_RAND_Seed failed");
            ret = WC_INIT_E;
        }
#endif

#ifndef NO_SESSION_CACHE
    #ifdef ENABLE_SESSION_CACHE_ROW_LOCK
        for (i = 0; i < SESSION_ROWS; ++i) {
            SessionCache[i].mutex_valid = 0;
        }
        for (i = 0; (ret == WOLFSSL_SUCCESS) && (i < SESSION_ROWS); ++i) {
            if (wc_InitMutex(&SessionCache[i].row_mutex) != 0) {
                WOLFSSL_MSG("Bad Init Mutex session");
                ret = BAD_MUTEX_E;
            }
            else {
                SessionCache[i].mutex_valid = 1;
            }
        }
    #else
        if ((ret == WOLFSSL_SUCCESS) && (wc_InitMutex(&session_mutex) != 0)) {
            WOLFSSL_MSG("Bad Init Mutex session");
            ret = BAD_MUTEX_E;
        }
        else {
            session_mutex_valid = 1;
        }
    #endif
    #ifndef NO_CLIENT_CACHE
        if ((ret == WOLFSSL_SUCCESS) &&
            (wc_InitMutex(&clisession_mutex) != 0)) {
            WOLFSSL_MSG("Bad Init Mutex session");
            ret = BAD_MUTEX_E;
        }
        else {
            clisession_mutex_valid = 1;
        }
    #endif
#endif
        if ((ret == WOLFSSL_SUCCESS) && (wc_InitMutex(&count_mutex) != 0)) {
            WOLFSSL_MSG("Bad Init Mutex count");
            ret = BAD_MUTEX_E;
        }
        else {
            count_mutex_valid = 1;
        }

#if defined(OPENSSL_EXTRA) && defined(HAVE_ATEXIT)
        /* OpenSSL registers cleanup using atexit */
        if ((ret == WOLFSSL_SUCCESS) && (atexit(AtExitCleanup) != 0)) {
            WOLFSSL_MSG("Bad atexit registration");
            ret = WC_INIT_E;
        }
#endif
    }

    if ((ret == WOLFSSL_SUCCESS) && (wc_LockMutex(&count_mutex) != 0)) {
        WOLFSSL_MSG("Bad Lock Mutex count");
        ret = BAD_MUTEX_E;
    }
    else {
        initRefCount++;
        wc_UnLockMutex(&count_mutex);
    }

    if (ret != WOLFSSL_SUCCESS) {
        initRefCount = 1; /* Force cleanup */
        (void)wolfSSL_Cleanup(); /* Ignore any error from cleanup */
    }

    return ret;
}



#ifndef NO_CERTS

/* process user cert chain to pass during the handshake */
static int ProcessUserChain(WOLFSSL_CTX* ctx, const unsigned char* buff,
                         long sz, int format, int type, WOLFSSL* ssl,
                         long* used, EncryptedInfo* info, int verify)
{
    int ret = 0;
    void* heap = wolfSSL_CTX_GetHeap(ctx, ssl);
#ifdef WOLFSSL_TLS13
    int cnt = 0;
#endif

    if ((type == CA_TYPE) && (ctx == NULL)) {
        WOLFSSL_MSG("Need context for CA load");
        return BAD_FUNC_ARG;
    }

    /* we may have a user cert chain, try to consume */
    if ((type == CERT_TYPE || type == CA_TYPE) && (info->consumed < sz)) {
    #ifdef WOLFSSL_SMALL_STACK
        byte   staticBuffer[1];                 /* force heap usage */
    #else
        byte   staticBuffer[FILE_BUFFER_SIZE];  /* tmp chain buffer */
    #endif
        byte*  chainBuffer = staticBuffer;
        int    dynamicBuffer = 0;
        word32 bufferSz;
        long   consumed = info->consumed;
        word32 idx = 0;
        int    gotOne = 0;

        /* Calculate max possible size, including max headers */
        bufferSz = (word32)(sz - consumed) + (CERT_HEADER_SZ * MAX_CHAIN_DEPTH);
        if (bufferSz > sizeof(staticBuffer)) {
            WOLFSSL_MSG("Growing Tmp Chain Buffer");
            /* will shrink to actual size */
            chainBuffer = (byte*)XMALLOC(bufferSz, heap, DYNAMIC_TYPE_FILE);
            if (chainBuffer == NULL) {
                return MEMORY_E;
            }
            dynamicBuffer = 1;
        }

        WOLFSSL_MSG("Processing Cert Chain");
        while (consumed < sz) {
            DerBuffer* part = NULL;
            word32 remain = (word32)(sz - consumed);
            info->consumed = 0;

            if (format == WOLFSSL_FILETYPE_PEM) {
            #ifdef WOLFSSL_PEM_TO_DER
                ret = PemToDer(buff + consumed, remain, type, &part,
                               heap, info, NULL);
            #else
                ret = NOT_COMPILED_IN;
            #endif
            }
            else {
                int length = remain;
                if (format == WOLFSSL_FILETYPE_ASN1) {
                    /* get length of der (read sequence) */
                    word32 inOutIdx = 0;
                    if (GetSequence(buff + consumed, &inOutIdx, &length,
                            remain) < 0) {
                        ret = ASN_NO_PEM_HEADER;
                    }
                    length += inOutIdx; /* include leading sequence */
                }
                info->consumed = length;
                if (ret == 0) {
                    ret = AllocDer(&part, length, type, heap);
                    if (ret == 0) {
                        XMEMCPY(part->buffer, buff + consumed, length);
                    }
                }
            }
            if (ret == 0) {
                gotOne = 1;
#ifdef WOLFSSL_TLS13
                cnt++;
#endif
                if ((idx + part->length + CERT_HEADER_SZ) > bufferSz) {
                    WOLFSSL_MSG("   Cert Chain bigger than buffer");
                    ret = BUFFER_E;
                }
                else {
                    c32to24(part->length, &chainBuffer[idx]);
                    idx += CERT_HEADER_SZ;
                    XMEMCPY(&chainBuffer[idx], part->buffer, part->length);
                    idx += part->length;
                    consumed  += info->consumed;
                    if (used)
                        *used += info->consumed;
                }

                /* add CA's to certificate manager */
                if (type == CA_TYPE) {
                    /* verify CA unless user set to no verify */
                    ret = AddCA(ctx->cm, &part, WOLFSSL_USER_CA, verify);
                    gotOne = 0; /* don't exit loop for CA type */
                }
            }

            FreeDer(&part);

            if (ret == ASN_NO_PEM_HEADER && gotOne) {
                WOLFSSL_MSG("We got one good cert, so stuff at end ok");
                break;
            }

            if (ret < 0) {
                WOLFSSL_MSG("   Error in Cert in Chain");
                if (dynamicBuffer)
                    XFREE(chainBuffer, heap, DYNAMIC_TYPE_FILE);
                return ret;
            }
            WOLFSSL_MSG("   Consumed another Cert in Chain");
        }
        WOLFSSL_MSG("Finished Processing Cert Chain");

        /* only retain actual size used */
        ret = 0;
        if (idx > 0) {
            if (ssl) {
                if (ssl->buffers.weOwnCertChain) {
                    FreeDer(&ssl->buffers.certChain);
                }
                ret = AllocDer(&ssl->buffers.certChain, idx, type, heap);
                if (ret == 0) {
                    XMEMCPY(ssl->buffers.certChain->buffer, chainBuffer,
                            idx);
                    ssl->buffers.weOwnCertChain = 1;
                }
            #ifdef WOLFSSL_TLS13
                ssl->buffers.certChainCnt = cnt;
            #endif
            } else if (ctx) {
                FreeDer(&ctx->certChain);
                ret = AllocDer(&ctx->certChain, idx, type, heap);
                if (ret == 0) {
                    XMEMCPY(ctx->certChain->buffer, chainBuffer, idx);
                }
            #ifdef WOLFSSL_TLS13
                ctx->certChainCnt = cnt;
            #endif
            }
        }

        if (dynamicBuffer)
            XFREE(chainBuffer, heap, DYNAMIC_TYPE_FILE);
    }

    return ret;
}

static int ProcessBufferTryDecode(WOLFSSL_CTX* ctx, WOLFSSL* ssl, DerBuffer* der,
    int* keySz, word32* idx, int* resetSuites, int* keyFormat, void* heap, int devId)
{
    int ret = 0;

    (void)heap;
    (void)devId;

    if (ctx == NULL && ssl == NULL)
        return BAD_FUNC_ARG;
    if (!der || !keySz || !idx || !resetSuites || !keyFormat)
        return BAD_FUNC_ARG;

#ifndef NO_RSA
    if ((*keyFormat == 0 || *keyFormat == RSAk)) {
        /* make sure RSA key can be used */
    #ifdef WOLFSSL_SMALL_STACK
        RsaKey* key;
    #else
        RsaKey  key[1];
    #endif

    #ifdef WOLFSSL_SMALL_STACK
        key = (RsaKey*)XMALLOC(sizeof(RsaKey), heap, DYNAMIC_TYPE_RSA);
        if (key == NULL)
            return MEMORY_E;
    #endif

        ret = wc_InitRsaKey_ex(key, heap, devId);
        if (ret == 0) {
            *idx = 0;
            ret = wc_RsaPrivateKeyDecode(der->buffer, idx, key, der->length);
        #ifdef WOLF_PRIVATE_KEY_ID
            if (ret != 0 && (devId != INVALID_DEVID
            #ifdef HAVE_PK_CALLBACKS
                || wolfSSL_CTX_IsPrivatePkSet(ctx)
            #endif
            )) {
                /* if using crypto or PK callbacks, try public key decode */
                *idx = 0;
                ret = wc_RsaPublicKeyDecode(der->buffer, idx, key, der->length);
            }
        #endif
            if (ret != 0) {
            #if !defined(HAVE_ECC) && !defined(HAVE_ED25519) && \
                !defined(HAVE_ED448) && !defined(HAVE_PQC)
                WOLFSSL_MSG("RSA decode failed and other algorithms "
                            "not enabled to try");
                ret = WOLFSSL_BAD_FILE;
            #else
                ret = 0; /* continue trying other algorithms */
            #endif
            }
            else {
                /* check that the size of the RSA key is enough */
                int minRsaSz = ssl ? ssl->options.minRsaKeySz :
                    ctx->minRsaKeySz;
                *keySz = wc_RsaEncryptSize((RsaKey*)key);
                if (*keySz < minRsaSz) {
                    ret = RSA_KEY_SIZE_E;
                    WOLFSSL_MSG("Private Key size too small");
                }

                if (ssl) {
                    ssl->buffers.keyType = rsa_sa_algo;
                    ssl->buffers.keySz = *keySz;
                }
                else {
                    ctx->privateKeyType = rsa_sa_algo;
                    ctx->privateKeySz = *keySz;
                }

                *keyFormat = RSAk;

                if (ssl && ssl->options.side == WOLFSSL_SERVER_END) {
                    ssl->options.haveStaticECC = 0;
                    *resetSuites = 1;
                }
            }

            wc_FreeRsaKey(key);
        }

    #ifdef WOLFSSL_SMALL_STACK
        XFREE(key, heap, DYNAMIC_TYPE_RSA);
    #endif
        if (ret != 0)
            return ret;
    }
#endif
#ifdef HAVE_ECC
    if ((*keyFormat == 0 || *keyFormat == ECDSAk)) {
        /* make sure ECC key can be used */
    #ifdef WOLFSSL_SMALL_STACK
        ecc_key* key;
    #else
        ecc_key  key[1];
    #endif

    #ifdef WOLFSSL_SMALL_STACK
        key = (ecc_key*)XMALLOC(sizeof(ecc_key), heap, DYNAMIC_TYPE_ECC);
        if (key == NULL)
            return MEMORY_E;
    #endif

        if (wc_ecc_init_ex(key, heap, devId) == 0) {
            *idx = 0;
            ret = wc_EccPrivateKeyDecode(der->buffer, idx, key, der->length);
        #ifdef WOLF_PRIVATE_KEY_ID
            if (ret != 0 && (devId != INVALID_DEVID
            #ifdef HAVE_PK_CALLBACKS
                || wolfSSL_CTX_IsPrivatePkSet(ctx)
            #endif
            )) {
                /* if using crypto or PK callbacks, try public key decode */
                *idx = 0;
                ret = wc_EccPublicKeyDecode(der->buffer, idx, key, der->length);
            }
        #endif
            if (ret == 0) {
                /* check for minimum ECC key size and then free */
                int minKeySz = ssl ? ssl->options.minEccKeySz :
                                                        ctx->minEccKeySz;
                *keySz = wc_ecc_size(key);
                if (*keySz < minKeySz) {
                    WOLFSSL_MSG("ECC private key too small");
                    ret = ECC_KEY_SIZE_E;
                }

                *keyFormat = ECDSAk;
                if (ssl) {
                    ssl->options.haveStaticECC = 1;
                    ssl->buffers.keyType = ecc_dsa_sa_algo;
                    ssl->buffers.keySz = *keySz;
                }
                else {
                    ctx->haveStaticECC = 1;
                    ctx->privateKeyType = ecc_dsa_sa_algo;
                    ctx->privateKeySz = *keySz;
                }

                if (ssl && ssl->options.side == WOLFSSL_SERVER_END) {
                    *resetSuites = 1;
                }
            }
            else {
                ret = 0; /* continue trying other algorithms */
            }

            wc_ecc_free(key);
        }

    #ifdef WOLFSSL_SMALL_STACK
        XFREE(key, heap, DYNAMIC_TYPE_ECC);
    #endif
        if (ret != 0)
            return ret;
    }
#endif /* HAVE_ECC */
#if defined(HAVE_ED25519) && defined(HAVE_ED25519_KEY_IMPORT)
    if ((*keyFormat == 0 || *keyFormat == ED25519k)) {
        /* make sure Ed25519 key can be used */
    #ifdef WOLFSSL_SMALL_STACK
        ed25519_key* key;
    #else
        ed25519_key  key[1];
    #endif

    #ifdef WOLFSSL_SMALL_STACK
        key = (ed25519_key*)XMALLOC(sizeof(ed25519_key), heap,
                                                      DYNAMIC_TYPE_ED25519);
        if (key == NULL)
            return MEMORY_E;
    #endif

        ret = wc_ed25519_init_ex(key, heap, devId);
        if (ret == 0) {
            *idx = 0;
            ret = wc_Ed25519PrivateKeyDecode(der->buffer, idx, key, der->length);
        #ifdef WOLF_PRIVATE_KEY_ID
            if (ret != 0 && (devId != INVALID_DEVID
            #ifdef HAVE_PK_CALLBACKS
                || wolfSSL_CTX_IsPrivatePkSet(ctx)
            #endif
            )) {
                /* if using crypto or PK callbacks, try public key decode */
                *idx = 0;
                ret = wc_Ed25519PublicKeyDecode(der->buffer, idx, key,
                                                der->length);
            }
        #endif
            if (ret == 0) {
                /* check for minimum key size and then free */
                int minKeySz = ssl ? ssl->options.minEccKeySz :
                                                           ctx->minEccKeySz;
                *keySz = ED25519_KEY_SIZE;
                if (*keySz < minKeySz) {
                    WOLFSSL_MSG("ED25519 private key too small");
                    ret = ECC_KEY_SIZE_E;
                }
                if (ret == 0) {
                    if (ssl) {
                        ssl->buffers.keyType = ed25519_sa_algo;
                        ssl->buffers.keySz = *keySz;
                    }
                    else if (ctx) {
                        ctx->privateKeyType = ed25519_sa_algo;
                        ctx->privateKeySz = *keySz;
                    }

                    *keyFormat = ED25519k;
                    if (ssl != NULL) {
                        /* ED25519 requires caching enabled for tracking message
                         * hash used in EdDSA_Update for signing */
                        ssl->options.cacheMessages = 1;
                        if (ssl->options.side == WOLFSSL_SERVER_END) {
                            *resetSuites = 1;
                        }
                    }
                }
            }
            else {
                ret = 0; /* continue trying other algorithms */
            }

            wc_ed25519_free(key);
        }

    #ifdef WOLFSSL_SMALL_STACK
        XFREE(key, heap, DYNAMIC_TYPE_ED25519);
    #endif
        if (ret != 0)
            return ret;
    }
#endif /* HAVE_ED25519 && HAVE_ED25519_KEY_IMPORT */
#if defined(HAVE_ED448) && defined(HAVE_ED448_KEY_IMPORT)
    if ((*keyFormat == 0 || *keyFormat == ED448k)) {
        /* make sure Ed448 key can be used */
    #ifdef WOLFSSL_SMALL_STACK
        ed448_key* key = NULL;
    #else
        ed448_key  key[1];
    #endif

    #ifdef WOLFSSL_SMALL_STACK
        key = (ed448_key*)XMALLOC(sizeof(ed448_key), heap, DYNAMIC_TYPE_ED448);
        if (key == NULL)
            return MEMORY_E;
    #endif

        ret = wc_ed448_init(key);
        if (ret == 0) {
            *idx = 0;
            ret = wc_Ed448PrivateKeyDecode(der->buffer, idx, key, der->length);
        #ifdef WOLF_PRIVATE_KEY_ID
            if (ret != 0 && (devId != INVALID_DEVID
            #ifdef HAVE_PK_CALLBACKS
                || wolfSSL_CTX_IsPrivatePkSet(ctx)
            #endif
            )) {
                /* if using crypto or PK callbacks, try public key decode */
                *idx = 0;
                ret = wc_Ed448PublicKeyDecode(der->buffer, idx, key,
                                              der->length);
            }
        #endif
            if (ret == 0) {
                /* check for minimum key size and then free */
                int minKeySz = ssl ? ssl->options.minEccKeySz :
                                                               ctx->minEccKeySz;
                *keySz = ED448_KEY_SIZE;
                if (*keySz < minKeySz) {
                    WOLFSSL_MSG("ED448 private key too small");
                    ret = ECC_KEY_SIZE_E;
                }
            }
            if (ret == 0) {
                if (ssl) {
                    ssl->buffers.keyType = ed448_sa_algo;
                    ssl->buffers.keySz = *keySz;
                }
                else if (ctx) {
                    ctx->privateKeyType = ed448_sa_algo;
                    ctx->privateKeySz = *keySz;
                }

                *keyFormat = ED448k;
                if (ssl != NULL) {
                    /* ED448 requires caching enabled for tracking message
                     * hash used in EdDSA_Update for signing */
                    ssl->options.cacheMessages = 1;
                    if (ssl->options.side == WOLFSSL_SERVER_END) {
                        *resetSuites = 1;
                    }
                }
            }

            wc_ed448_free(key);
        }

    #ifdef WOLFSSL_SMALL_STACK
        XFREE(key, heap, DYNAMIC_TYPE_ED448);
    #endif
        if (ret != 0)
            return ret;
    }
#endif /* HAVE_ED448 && HAVE_ED448_KEY_IMPORT */
#if defined(HAVE_PQC) && defined(HAVE_FALCON)
    if (((*keyFormat == 0) || (*keyFormat == FALCON_LEVEL1k) ||
         (*keyFormat == FALCON_LEVEL5k))) {
        /* make sure Falcon key can be used */
        falcon_key* key = (falcon_key*)XMALLOC(sizeof(falcon_key), heap,
                                               DYNAMIC_TYPE_FALCON);
        if (key == NULL) {
            return MEMORY_E;
        }
        ret = wc_falcon_init(key);
        if (ret == 0) {
            if (*keyFormat == FALCON_LEVEL1k) {
                ret = wc_falcon_set_level(key, 1);
            }
            else if (*keyFormat == FALCON_LEVEL5k) {
                ret = wc_falcon_set_level(key, 5);
            }
            else {
                /* What if *keyformat is 0? We might want to do something more
                 * graceful here. */
                wc_falcon_free(key);
                ret = ALGO_ID_E;
            }
        }

        if (ret == 0) {
            *idx = 0;
            ret = wc_falcon_import_private_only(der->buffer, der->length, key);
            if (ret == 0) {
                /* check for minimum key size and then free */
                int minKeySz = ssl ? ssl->options.minFalconKeySz :
                                     ctx->minFalconKeySz;
                *keySz = FALCON_MAX_KEY_SIZE;
                if (*keySz < minKeySz) {
                    WOLFSSL_MSG("Falcon private key too small");
                    ret = FALCON_KEY_SIZE_E;
                }
                if (ssl) {
                    if (*keyFormat == FALCON_LEVEL1k) {
                        ssl->buffers.keyType = falcon_level1_sa_algo;
                    }
                    else {
                        ssl->buffers.keyType = falcon_level5_sa_algo;
                    }
                    ssl->buffers.keySz = *keySz;
                }
                else {
                    if (*keyFormat == FALCON_LEVEL1k) {
                        ctx->privateKeyType = falcon_level1_sa_algo;
                    }
                    else {
                        ctx->privateKeyType = falcon_level5_sa_algo;
                    }
                    ctx->privateKeySz = *keySz;
                }

                if (ssl && ssl->options.side == WOLFSSL_SERVER_END) {
                    *resetSuites = 1;
                }
            }
            wc_falcon_free(key);
        }
        XFREE(key, heap, DYNAMIC_TYPE_FALCON);
        if (ret != 0)
            return ret;
    }
#endif /* HAVE_PQC && HAVE_FALCON */
    return ret;
}

/* process the buffer buff, length sz, into ctx of format and type
   used tracks bytes consumed, userChain specifies a user cert chain
   to pass during the handshake */
int ProcessBuffer(WOLFSSL_CTX* ctx, const unsigned char* buff,
                         long sz, int format, int type, WOLFSSL* ssl,
                         long* used, int userChain, int verify)
{
    DerBuffer*    der = NULL;
    int           ret = 0;
    int           done = 0;
    int           keyFormat = 0;
    int           resetSuites = 0;
    void*         heap = wolfSSL_CTX_GetHeap(ctx, ssl);
    int           devId = wolfSSL_CTX_GetDevId(ctx, ssl);
    word32        idx = 0;
    int           keySz = 0;
#if (defined(WOLFSSL_ENCRYPTED_KEYS) && !defined(NO_PWDBASED)) || \
     defined(HAVE_PKCS8)
    word32        algId = 0;
#endif
#ifdef WOLFSSL_SMALL_STACK
    EncryptedInfo* info = NULL;
#else
    EncryptedInfo  info[1];
#endif

    (void)devId;
    (void)idx;
    (void)keySz;

    if (used)
        *used = sz;     /* used bytes default to sz, PEM chain may shorten*/

    /* check args */
    if (format != WOLFSSL_FILETYPE_ASN1 && format != WOLFSSL_FILETYPE_PEM)
        return WOLFSSL_BAD_FILETYPE;

    if (ctx == NULL && ssl == NULL)
        return BAD_FUNC_ARG;

#ifdef WOLFSSL_SMALL_STACK
    info = (EncryptedInfo*)XMALLOC(sizeof(EncryptedInfo), heap,
                                   DYNAMIC_TYPE_ENCRYPTEDINFO);
    if (info == NULL)
        return MEMORY_E;
#endif

    XMEMSET(info, 0, sizeof(EncryptedInfo));
#if defined(WOLFSSL_ENCRYPTED_KEYS) && !defined(NO_PWDBASED)
    if (ctx) {
        info->passwd_cb       = ctx->passwd_cb;
        info->passwd_userdata = ctx->passwd_userdata;
    }
#endif

    if (format == WOLFSSL_FILETYPE_PEM) {
    #ifdef WOLFSSL_PEM_TO_DER
        ret = PemToDer(buff, sz, type, &der, heap, info, &keyFormat);
    #else
        ret = NOT_COMPILED_IN;
    #endif
    }
    else {
        /* ASN1 (DER) */
        int length = (int)sz;
        if (format == WOLFSSL_FILETYPE_ASN1) {
            /* get length of der (read sequence or octet string) */
            word32 inOutIdx = 0;
            if (GetSequence(buff, &inOutIdx, &length, (word32)sz) >= 0) {
                length += inOutIdx; /* include leading sequence */
            }
            /* get length using octect string (allowed for private key types) */
            else if (type == PRIVATEKEY_TYPE &&
                    GetOctetString(buff, &inOutIdx, &length, (word32)sz) >= 0) {
                length += inOutIdx; /* include leading oct string */
            }
            else {
                ret = ASN_PARSE_E;
            }
        }

        info->consumed = length;

        if (ret == 0) {
            ret = AllocDer(&der, (word32)length, type, heap);
            if (ret == 0) {
                XMEMCPY(der->buffer, buff, length);
            }

        #ifdef HAVE_PKCS8
            /* if private key try and remove PKCS8 header */
            if (type == PRIVATEKEY_TYPE) {
                if ((ret = ToTraditional_ex(der->buffer, der->length,
                                                                 &algId)) > 0) {
                    /* Found PKCS8 header */
                    /* ToTraditional_ex moves buff and returns adjusted length */
                    der->length = ret;
                    keyFormat = algId;
                }
                ret = 0; /* failures should be ignored */
            }
        #endif
        }
    }

    if (used) {
        *used = info->consumed;
    }

    /* process user chain */
    if (ret >= 0) {
        /* Chain should have server cert first, then intermediates, then root.
         * First certificate in chain is processed below after ProcessUserChain
         *   and is loaded into ssl->buffers.certificate.
         * Remainder are processed using ProcessUserChain and are loaded into
         *   ssl->buffers.certChain. */
        if (userChain) {
            ret = ProcessUserChain(ctx, buff, sz, format, type, ssl, used, info,
                                   verify);
            if (ret == ASN_NO_PEM_HEADER) { /* Additional chain is optional */
                unsigned long pemErr;
                CLEAR_ASN_NO_PEM_HEADER_ERROR(pemErr);
                ret = 0;
            }
        }
    }

    /* info is only used for private key with DER or PEM, so free now */
    if (ret < 0 || type != PRIVATEKEY_TYPE) {
    #ifdef WOLFSSL_SMALL_STACK
        XFREE(info, heap, DYNAMIC_TYPE_ENCRYPTEDINFO);
    #endif
    }

    /* check for error */
    if (ret < 0) {
        FreeDer(&der);
        done = 1;
    }

    if (done == 1) {
        /* No operation, just skip the next section */
    }
    /* Handle DER owner */
    else if (type == CA_TYPE) {
        if (ctx == NULL) {
            WOLFSSL_MSG("Need context for CA load");
            FreeDer(&der);
            return BAD_FUNC_ARG;
        }
        /* verify CA unless user set to no verify */
        ret = AddCA(ctx->cm, &der, WOLFSSL_USER_CA, verify);
        done = 1;
    }
#ifdef WOLFSSL_TRUST_PEER_CERT
    else if (type == TRUSTED_PEER_TYPE) {
        /* add trusted peer cert. der is freed within */
        if (ctx != NULL)
            ret = AddTrustedPeer(ctx->cm, &der, !ctx->verifyNone);
        else
            ret = AddTrustedPeer(SSL_CM(ssl), &der, !ssl->options.verifyNone);
        if (ret != WOLFSSL_SUCCESS) {
            WOLFSSL_MSG("Error adding trusted peer");
        }
        done = 1;
    }
#endif /* WOLFSSL_TRUST_PEER_CERT */
    else if (type == CERT_TYPE) {
        if (ssl) {
             /* Make sure previous is free'd */
            if (ssl->buffers.weOwnCert) {
                FreeDer(&ssl->buffers.certificate);
            #ifdef KEEP_OUR_CERT
                wolfSSL_X509_free(ssl->ourCert);
                ssl->ourCert = NULL;
            #endif
            }
            ssl->buffers.certificate = der;
        #ifdef KEEP_OUR_CERT
            ssl->keepCert = 1; /* hold cert for ssl lifetime */
        #endif
            ssl->buffers.weOwnCert = 1;
        }
        else if (ctx) {
            FreeDer(&ctx->certificate); /* Make sure previous is free'd */
        #ifdef KEEP_OUR_CERT
            if (ctx->ourCert) {
                if (ctx->ownOurCert)
                    wolfSSL_X509_free(ctx->ourCert);
                ctx->ourCert = NULL;
            }
        #endif
            ctx->certificate = der;
        }
    }
    else if (type == PRIVATEKEY_TYPE) {
        if (ssl) {
             /* Make sure previous is free'd */
            if (ssl->buffers.weOwnKey) {
                FreeDer(&ssl->buffers.key);
            }
            ssl->buffers.key = der;
            ssl->buffers.weOwnKey = 1;
        }
        else if (ctx) {
            FreeDer(&ctx->privateKey);
            ctx->privateKey = der;
        }
    }
    else {
        FreeDer(&der);
        return WOLFSSL_BAD_CERTTYPE;
    }

    if (done == 1) {
        /* No operation, just skip the next section */
    }
    else if (type == PRIVATEKEY_TYPE) {
        ret = ProcessBufferTryDecode(ctx, ssl, der, &keySz, &idx, &resetSuites,
                &keyFormat, heap, devId);

    #if defined(WOLFSSL_ENCRYPTED_KEYS) && !defined(NO_PWDBASED)
        /* for WOLFSSL_FILETYPE_PEM, PemToDer manages the decryption */
        /* If private key type PKCS8 header wasn't already removed (algoId == 0) */
        if ((ret != 0 || keyFormat == 0)
            && format != WOLFSSL_FILETYPE_PEM && info->passwd_cb && algId == 0)
        {
            int   passwordSz = NAME_SZ;
        #ifndef WOLFSSL_SMALL_STACK
            char  password[NAME_SZ];
        #else
            char* password = (char*)XMALLOC(passwordSz, heap, DYNAMIC_TYPE_STRING);
            if (password == NULL) {
                XFREE(info, heap, DYNAMIC_TYPE_ENCRYPTEDINFO);
                FreeDer(&der);
                return MEMORY_E;
            }
        #endif
            /* get password */
            ret = info->passwd_cb(password, passwordSz, PEM_PASS_READ,
                info->passwd_userdata);
            if (ret >= 0) {
                passwordSz = ret;

                /* PKCS8 decrypt */
                ret = ToTraditionalEnc(der->buffer, der->length,
                                       password, passwordSz, &algId);
                if (ret >= 0) {
                    der->length = ret;
                }
                /* ignore failures and try parsing as unencrypted */

                ForceZero(password, passwordSz);
            }

        #ifdef WOLFSSL_SMALL_STACK
            XFREE(password, heap, DYNAMIC_TYPE_STRING);
        #endif
            ret = ProcessBufferTryDecode(ctx, ssl, der, &keySz, &idx,
                &resetSuites, &keyFormat, heap, devId);
        }
    #endif /* WOLFSSL_ENCRYPTED_KEYS && !NO_PWDBASED */

    #ifdef WOLFSSL_SMALL_STACK
        XFREE(info, heap, DYNAMIC_TYPE_ENCRYPTEDINFO);
    #endif

        if (ret != 0)
            return ret;
        if (keyFormat == 0) {
#ifdef OPENSSL_EXTRA
            /* Reaching this point probably means that the
             * decryption password is wrong */
            if (info->passwd_cb)
                EVPerr(0, EVP_R_BAD_DECRYPT);
#endif
            WOLFSSL_ERROR(WOLFSSL_BAD_FILE);
            return WOLFSSL_BAD_FILE;
        }

        (void)devId;
    }
    else if (type == CERT_TYPE) {
    #ifdef WOLFSSL_SMALL_STACK
        DecodedCert* cert;
    #else
        DecodedCert  cert[1];
    #endif
    #ifdef WOLF_PRIVATE_KEY_ID
        int keyType = 0;
    #endif

    #ifdef WOLFSSL_SMALL_STACK
        cert = (DecodedCert*)XMALLOC(sizeof(DecodedCert), heap,
                                     DYNAMIC_TYPE_DCERT);
        if (cert == NULL)
            return MEMORY_E;
    #endif

        WOLFSSL_MSG("Checking cert signature type");
        InitDecodedCert(cert, der->buffer, der->length, heap);

        if (DecodeToKey(cert, 0) < 0) {
            WOLFSSL_MSG("Decode to key failed");
            FreeDecodedCert(cert);
        #ifdef WOLFSSL_SMALL_STACK
            XFREE(cert, heap, DYNAMIC_TYPE_DCERT);
        #endif
            return WOLFSSL_BAD_FILE;
        }

        if (ssl && ssl->options.side == WOLFSSL_SERVER_END) {
            resetSuites = 1;
        }
        if (ssl && ssl->ctx->haveECDSAsig) {
            WOLFSSL_MSG("SSL layer setting cert, CTX had ECDSA, turning off");
            ssl->options.haveECDSAsig = 0;   /* may turn back on next */
        }

        switch (cert->signatureOID) {
            case CTC_SHAwECDSA:
            case CTC_SHA256wECDSA:
            case CTC_SHA384wECDSA:
            case CTC_SHA512wECDSA:
            case CTC_ED25519:
            case CTC_ED448:
                WOLFSSL_MSG("ECDSA/ED25519/ED448 cert signature");
                if (ssl)
                    ssl->options.haveECDSAsig = 1;
                else if (ctx)
                    ctx->haveECDSAsig = 1;
                break;
            case CTC_FALCON_LEVEL1:
            case CTC_FALCON_LEVEL5:
                WOLFSSL_MSG("Falcon cert signature");
                if (ssl)
                    ssl->options.haveFalconSig = 1;
                else if (ctx)
                    ctx->haveFalconSig = 1;
                break;
            default:
                WOLFSSL_MSG("Not ECDSA cert signature");
                break;
        }

    #if defined(HAVE_ECC) || defined(HAVE_ED25519) || defined(HAVE_ED448) || \
        defined(HAVE_PQC) || !defined(NO_RSA)
        if (ssl) {
        #if defined(HAVE_ECC) || defined(HAVE_ED25519) || \
            (defined(HAVE_CURVE448) && defined(HAVE_ED448))
            ssl->pkCurveOID = cert->pkCurveOID;
        #endif
        #ifndef WC_STRICT_SIG
            if (cert->keyOID == ECDSAk) {
                ssl->options.haveECC = 1;
            }
            #ifndef NO_RSA
            else if (cert->keyOID == RSAk) {
                ssl->options.haveRSA = 1;
            }
            #endif
            #ifdef HAVE_ED25519
                else if (cert->keyOID == ED25519k) {
                    ssl->options.haveECC = 1;
                }
            #endif
            #ifdef HAVE_ED448
                else if (cert->keyOID == ED448k) {
                    ssl->options.haveECC = 1;
                }
            #endif
            #ifdef HAVE_PQC
                else if (cert->keyOID == FALCON_LEVEL1k ||
                         cert->keyOID == FALCON_LEVEL5k) {
                    ssl->options.haveFalconSig = 1;
                }
            #endif
        #else
            ssl->options.haveECC = ssl->options.haveECDSAsig;
        #endif
        }
        else if (ctx) {
        #if defined(HAVE_ECC) || defined(HAVE_ED25519) || defined(HAVE_ED448)
            ctx->pkCurveOID = cert->pkCurveOID;
        #endif
        #ifndef WC_STRICT_SIG
            if (cert->keyOID == ECDSAk) {
                ctx->haveECC = 1;
            }
            #ifndef NO_RSA
            else if (cert->keyOID == RSAk) {
                ctx->haveRSA = 1;
            }
            #endif
            #ifdef HAVE_ED25519
                else if (cert->keyOID == ED25519k) {
                    ctx->haveECC = 1;
                }
            #endif
            #ifdef HAVE_ED448
                else if (cert->keyOID == ED448k) {
                    ctx->haveECC = 1;
                }
            #endif
            #ifdef HAVE_PQC
                else if (cert->keyOID == FALCON_LEVEL1k ||
                         cert->keyOID == FALCON_LEVEL5k) {
                    ctx->haveFalconSig = 1;
                }
            #endif
        #else
            ctx->haveECC = ctx->haveECDSAsig;
        #endif
        }
    #endif

        /* check key size of cert unless specified not to */
        switch (cert->keyOID) {
        #ifndef NO_RSA
            case RSAk:
            #ifdef WOLF_PRIVATE_KEY_ID
                keyType = rsa_sa_algo;
            #endif
                /* Determine RSA key size by parsing public key */
                idx = 0;
                ret = wc_RsaPublicKeyDecode_ex(cert->publicKey, &idx,
                    cert->pubKeySize, NULL, (word32*)&keySz, NULL, NULL);
                if (ret < 0)
                    break;

                if (ssl && !ssl->options.verifyNone) {
                    if (ssl->options.minRsaKeySz < 0 ||
                          keySz < (int)ssl->options.minRsaKeySz) {
                        ret = RSA_KEY_SIZE_E;
                        WOLFSSL_MSG("Certificate RSA key size too small");
                    }
                }
                else if (ctx && !ctx->verifyNone) {
                    if (ctx->minRsaKeySz < 0 ||
                                  keySz < (int)ctx->minRsaKeySz) {
                        ret = RSA_KEY_SIZE_E;
                        WOLFSSL_MSG("Certificate RSA key size too small");
                    }
                }
                break;
        #endif /* !NO_RSA */
        #ifdef HAVE_ECC
            case ECDSAk:
            #ifdef WOLF_PRIVATE_KEY_ID
                keyType = ecc_dsa_sa_algo;
            #endif
                /* Determine ECC key size based on curve */
                keySz = wc_ecc_get_curve_size_from_id(
                    wc_ecc_get_oid(cert->pkCurveOID, NULL, NULL));

                if (ssl && !ssl->options.verifyNone) {
                    if (ssl->options.minEccKeySz < 0 ||
                          keySz < (int)ssl->options.minEccKeySz) {
                        ret = ECC_KEY_SIZE_E;
                        WOLFSSL_MSG("Certificate ECC key size error");
                    }
                }
                else if (ctx && !ctx->verifyNone) {
                    if (ctx->minEccKeySz < 0 ||
                                  keySz < (int)ctx->minEccKeySz) {
                        ret = ECC_KEY_SIZE_E;
                        WOLFSSL_MSG("Certificate ECC key size error");
                    }
                }
                break;
        #endif /* HAVE_ECC */
        #ifdef HAVE_ED25519
            case ED25519k:
            #ifdef WOLF_PRIVATE_KEY_ID
                keyType = ed25519_sa_algo;
            #endif
                /* ED25519 is fixed key size */
                keySz = ED25519_KEY_SIZE;
                if (ssl && !ssl->options.verifyNone) {
                    if (ssl->options.minEccKeySz < 0 ||
                          keySz < (int)ssl->options.minEccKeySz) {
                        ret = ECC_KEY_SIZE_E;
                        WOLFSSL_MSG("Certificate Ed key size error");
                    }
                }
                else if (ctx && !ctx->verifyNone) {
                    if (ctx->minEccKeySz < 0 ||
                                  keySz < (int)ctx->minEccKeySz) {
                        ret = ECC_KEY_SIZE_E;
                        WOLFSSL_MSG("Certificate ECC key size error");
                    }
                }
                break;
        #endif /* HAVE_ED25519 */
        #ifdef HAVE_ED448
            case ED448k:
            #ifdef WOLF_PRIVATE_KEY_ID
                keyType = ed448_sa_algo;
            #endif
                /* ED448 is fixed key size */
                keySz = ED448_KEY_SIZE;
                if (ssl && !ssl->options.verifyNone) {
                    if (ssl->options.minEccKeySz < 0 ||
                          keySz < (int)ssl->options.minEccKeySz) {
                        ret = ECC_KEY_SIZE_E;
                        WOLFSSL_MSG("Certificate Ed key size error");
                    }
                }
                else if (ctx && !ctx->verifyNone) {
                    if (ctx->minEccKeySz < 0 ||
                                  keySz < (int)ctx->minEccKeySz) {
                        ret = ECC_KEY_SIZE_E;
                        WOLFSSL_MSG("Certificate ECC key size error");
                    }
                }
                break;
        #endif /* HAVE_ED448 */
        #if defined(HAVE_PQC) && defined(HAVE_FALCON)
            case FALCON_LEVEL1k:
            case FALCON_LEVEL5k:
                /* Falcon is fixed key size */
                keySz = FALCON_MAX_KEY_SIZE;
                if (ssl && !ssl->options.verifyNone) {
                    if (ssl->options.minFalconKeySz < 0 ||
                          keySz < (int)ssl->options.minFalconKeySz) {
                        ret = FALCON_KEY_SIZE_E;
                        WOLFSSL_MSG("Certificate Falcon key size error");
                    }
                }
                else if (ctx && !ctx->verifyNone) {
                    if (ctx->minFalconKeySz < 0 ||
                                  keySz < (int)ctx->minFalconKeySz) {
                        ret = FALCON_KEY_SIZE_E;
                        WOLFSSL_MSG("Certificate Falcon key size error");
                    }
                }
                break;
        #endif /* HAVE_PQC && HAVE_FALCON */

            default:
                WOLFSSL_MSG("No key size check done on certificate");
                break; /* do no check if not a case for the key */
        }

    #ifdef WOLF_PRIVATE_KEY_ID
        if (ssl != NULL && ssl->buffers.keyType == 0) {
            ssl->buffers.keyType = keyType;
            ssl->buffers.keySz = keySz;
        }
        else if (ctx != NULL && ctx->privateKeyType == 0) {
            ctx->privateKeyType = keyType;
            ctx->privateKeySz = keySz;
        }
    #endif

        FreeDecodedCert(cert);
    #ifdef WOLFSSL_SMALL_STACK
        XFREE(cert, heap, DYNAMIC_TYPE_DCERT);
    #endif

        if (ret != 0) {
            done = 1;
        }
    }

    if (done == 1) {
    #if !defined(NO_WOLFSSL_CM_VERIFY) && (!defined(NO_WOLFSSL_CLIENT) || \
                                           !defined(WOLFSSL_NO_CLIENT_AUTH))
        if ((type == CA_TYPE) || (type == CERT_TYPE)) {
            /* Call to over-ride status */
            if ((ctx != NULL) && (ctx->cm != NULL) &&
                (ctx->cm->verifyCallback != NULL)) {
                ret = CM_VerifyBuffer_ex(ctx->cm, buff,
                        sz, format, (ret == WOLFSSL_SUCCESS ? 0 : ret));
            }
        }
    #endif /* NO_WOLFSSL_CM_VERIFY */

        return ret;
    }


    if (ssl && resetSuites) {
        word16 havePSK = 0;
        word16 haveRSA = 0;

        #ifndef NO_PSK
        if (ssl->options.havePSK) {
            havePSK = 1;
        }
        #endif
        #ifndef NO_RSA
            haveRSA = 1;
        #endif
            keySz = ssl->buffers.keySz;

        /* let's reset suites */
        InitSuites(ssl->suites, ssl->version, keySz, haveRSA,
                   havePSK, ssl->options.haveDH, ssl->options.haveECDSAsig,
                   ssl->options.haveECC, ssl->options.haveStaticECC,
                   ssl->options.haveFalconSig, ssl->options.haveAnon,
                   ssl->options.side);
    }

    return WOLFSSL_SUCCESS;
}


/* CA PEM file for verification, may have multiple/chain certs to process */
static int ProcessChainBuffer(WOLFSSL_CTX* ctx, const unsigned char* buff,
                        long sz, int format, int type, WOLFSSL* ssl, int verify)
{
    long used   = 0;
    int  ret    = 0;
    int  gotOne = 0;

    WOLFSSL_MSG("Processing CA PEM file");
    while (used < sz) {
        long consumed = 0;

        ret = ProcessBuffer(ctx, buff + used, sz - used, format, type, ssl,
                            &consumed, 0, verify);

        if (ret < 0) {
#if defined(WOLFSSL_WPAS) && defined(HAVE_CRL)
            DerBuffer*    der = NULL;
            EncryptedInfo info;

            WOLFSSL_MSG("Trying a CRL");
            if (PemToDer(buff + used, sz - used, CRL_TYPE, &der, NULL, &info,
                                                                   NULL) == 0) {
                WOLFSSL_MSG("   Processed a CRL");
                wolfSSL_CertManagerLoadCRLBuffer(ctx->cm, der->buffer,
                                            der->length, WOLFSSL_FILETYPE_ASN1);
                FreeDer(&der);
                used += info.consumed;
                continue;
            }
#endif

            if (consumed > 0) { /* Made progress in file */
                WOLFSSL_ERROR(ret);
                WOLFSSL_MSG("CA Parse failed, with progress in file.");
                WOLFSSL_MSG("Search for other certs in file");
            }
            else {
                WOLFSSL_MSG("CA Parse failed, no progress in file.");
                WOLFSSL_MSG("Do not continue search for other certs in file");
                break;
            }
        }
        else {
            WOLFSSL_MSG("   Processed a CA");
            gotOne = 1;
        }
        used += consumed;
    }

    if (gotOne) {
        WOLFSSL_MSG("Processed at least one valid CA. Other stuff OK");
        return WOLFSSL_SUCCESS;
    }
    return ret;
}


static WC_INLINE WOLFSSL_METHOD* cm_pick_method(void)
{
    #ifndef NO_WOLFSSL_CLIENT
        #if !defined(NO_OLD_TLS) && defined(WOLFSSL_ALLOW_SSLV3)
            return wolfSSLv3_client_method();
        #elif !defined(NO_OLD_TLS) && defined(WOLFSSL_ALLOW_TLSV10)
            return wolfTLSv1_client_method();
        #elif !defined(NO_OLD_TLS)
            return wolfTLSv1_1_client_method();
        #elif !defined(WOLFSSL_NO_TLS12)
            return wolfTLSv1_2_client_method();
        #elif defined(WOLFSSL_TLS13)
            return wolfTLSv1_3_client_method();
        #else
            return NULL;
        #endif
    #elif !defined(NO_WOLFSSL_SERVER)
        #if !defined(NO_OLD_TLS) && defined(WOLFSSL_ALLOW_SSLV3)
            return wolfSSLv3_server_method();
        #elif !defined(NO_OLD_TLS) && defined(WOLFSSL_ALLOW_TLSV10)
            return wolfTLSv1_server_method();
        #elif !defined(NO_OLD_TLS)
            return wolfTLSv1_1_server_method();
        #elif !defined(WOLFSSL_NO_TLS12)
            return wolfTLSv1_2_server_method();
        #elif defined(WOLFSSL_TLS13)
            return wolfTLSv1_3_server_method();
        #else
            return NULL;
        #endif
    #else
        return NULL;
    #endif
}


/* like load verify locations, 1 for success, < 0 for error */
int wolfSSL_CertManagerLoadCABuffer(WOLFSSL_CERT_MANAGER* cm,
                                   const unsigned char* in, long sz, int format)
{
    int ret = WOLFSSL_FATAL_ERROR;
    WOLFSSL_CTX* tmp;

    WOLFSSL_ENTER("wolfSSL_CertManagerLoadCABuffer");

    if (cm == NULL) {
        WOLFSSL_MSG("No CertManager error");
        return ret;
    }
    tmp = wolfSSL_CTX_new(cm_pick_method());

    if (tmp == NULL) {
        WOLFSSL_MSG("CTX new failed");
        return ret;
    }

    /* for tmp use */
    wolfSSL_CertManagerFree(tmp->cm);
    tmp->cm = cm;

    ret = wolfSSL_CTX_load_verify_buffer(tmp, in, sz, format);

    /* don't loose our good one */
    tmp->cm = NULL;
    wolfSSL_CTX_free(tmp);

    return ret;
}

#ifdef HAVE_CRL

int wolfSSL_CertManagerLoadCRLBuffer(WOLFSSL_CERT_MANAGER* cm,
                                   const unsigned char* buff, long sz, int type)
{
    WOLFSSL_ENTER("wolfSSL_CertManagerLoadCRLBuffer");
    if (cm == NULL)
        return BAD_FUNC_ARG;

    if (cm->crl == NULL) {
        if (wolfSSL_CertManagerEnableCRL(cm, 0) != WOLFSSL_SUCCESS) {
            WOLFSSL_MSG("Enable CRL failed");
            return WOLFSSL_FATAL_ERROR;
        }
    }

    return BufferLoadCRL(cm->crl, buff, sz, type, VERIFY);
}

int wolfSSL_CertManagerFreeCRL(WOLFSSL_CERT_MANAGER* cm)
{
    WOLFSSL_ENTER("wolfSSL_CertManagerFreeCRL");
    if (cm == NULL)
        return BAD_FUNC_ARG;
    if (cm->crl != NULL){
        FreeCRL(cm->crl, 1);
        cm->crl = NULL;
    }
    return WOLFSSL_SUCCESS;
}

int wolfSSL_CTX_LoadCRLBuffer(WOLFSSL_CTX* ctx, const unsigned char* buff,
                              long sz, int type)
{
    WOLFSSL_ENTER("wolfSSL_CTX_LoadCRLBuffer");

    if (ctx == NULL)
        return BAD_FUNC_ARG;

    return wolfSSL_CertManagerLoadCRLBuffer(ctx->cm, buff, sz, type);
}


int wolfSSL_LoadCRLBuffer(WOLFSSL* ssl, const unsigned char* buff,
                          long sz, int type)
{
    WOLFSSL_ENTER("wolfSSL_LoadCRLBuffer");

    if (ssl == NULL || ssl->ctx == NULL)
        return BAD_FUNC_ARG;

    return wolfSSL_CertManagerLoadCRLBuffer(SSL_CM(ssl), buff, sz, type);
}


#endif /* HAVE_CRL */

/* turn on CRL if off and compiled in, set options */
int wolfSSL_CertManagerEnableCRL(WOLFSSL_CERT_MANAGER* cm, int options)
{
    int ret = WOLFSSL_SUCCESS;

    (void)options;

    WOLFSSL_ENTER("wolfSSL_CertManagerEnableCRL");
    if (cm == NULL)
        return BAD_FUNC_ARG;

    #ifdef HAVE_CRL
        if (cm->crl == NULL) {
            cm->crl = (WOLFSSL_CRL*)XMALLOC(sizeof(WOLFSSL_CRL), cm->heap,
                                            DYNAMIC_TYPE_CRL);
            if (cm->crl == NULL)
                return MEMORY_E;

            if (InitCRL(cm->crl, cm) != 0) {
                WOLFSSL_MSG("Init CRL failed");
                FreeCRL(cm->crl, 1);
                cm->crl = NULL;
                return WOLFSSL_FAILURE;
            }

        #if defined(HAVE_CRL_IO) && defined(USE_WOLFSSL_IO)
            cm->crl->crlIOCb = EmbedCrlLookup;
        #endif
        }

        cm->crlEnabled = 1;
        if (options & WOLFSSL_CRL_CHECKALL)
            cm->crlCheckAll = 1;
    #else
        ret = NOT_COMPILED_IN;
    #endif

    return ret;
}


int wolfSSL_CertManagerDisableCRL(WOLFSSL_CERT_MANAGER* cm)
{
    WOLFSSL_ENTER("wolfSSL_CertManagerDisableCRL");
    if (cm == NULL)
        return BAD_FUNC_ARG;

    cm->crlEnabled = 0;

    return WOLFSSL_SUCCESS;
}

#ifndef NO_WOLFSSL_CM_VERIFY
void wolfSSL_CertManagerSetVerify(WOLFSSL_CERT_MANAGER* cm, VerifyCallback vc)
{
    WOLFSSL_ENTER("wolfSSL_CertManagerSetVerify");
    if (cm == NULL)
        return;

    cm->verifyCallback = vc;
}
#endif /* NO_WOLFSSL_CM_VERIFY */

#if !defined(NO_WOLFSSL_CLIENT) || !defined(WOLFSSL_NO_CLIENT_AUTH)
/* Verify the certificate, WOLFSSL_SUCCESS for ok, < 0 for error */
int CM_VerifyBuffer_ex(WOLFSSL_CERT_MANAGER* cm, const byte* buff,
                                    long sz, int format, int err_val)
{
    int ret = 0;
    DerBuffer* der = NULL;
#ifdef WOLFSSL_SMALL_STACK
    DecodedCert* cert;
#else
    DecodedCert  cert[1];
#endif

    WOLFSSL_ENTER("wolfSSL_CertManagerVerifyBuffer");

#ifdef WOLFSSL_SMALL_STACK
    cert = (DecodedCert*)XMALLOC(sizeof(DecodedCert), cm->heap,
                                 DYNAMIC_TYPE_DCERT);
    if (cert == NULL)
        return MEMORY_E;
#endif

    if (format == WOLFSSL_FILETYPE_PEM) {
#ifdef WOLFSSL_PEM_TO_DER
        ret = PemToDer(buff, sz, CERT_TYPE, &der, cm->heap, NULL, NULL);
        if (ret != 0) {
            FreeDer(&der);
        #ifdef WOLFSSL_SMALL_STACK
            XFREE(cert, cm->heap, DYNAMIC_TYPE_DCERT);
        #endif
            return ret;
        }
        InitDecodedCert(cert, der->buffer, der->length, cm->heap);
#else
        ret = NOT_COMPILED_IN;
#endif
    }
    else {
        InitDecodedCert(cert, buff, (word32)sz, cm->heap);
    }

    if (ret == 0)
        ret = ParseCertRelative(cert, CERT_TYPE, 1, cm);

#if defined(OPENSSL_ALL) || defined(WOLFSSL_QT)
    /* ret needs to be self-singer error for Qt compat */
    if (ret == ASN_NO_SIGNER_E && cert->selfSigned)
        ret = ASN_SELF_SIGNED_E;
#endif

#ifdef HAVE_CRL
    if (ret == 0 && cm->crlEnabled)
        ret = CheckCertCRL(cm->crl, cert);
#endif

#ifndef NO_WOLFSSL_CM_VERIFY
    /* if verify callback has been set */
    if (cm->verifyCallback) {
        buffer certBuf;
    #ifdef WOLFSSL_SMALL_STACK
        ProcPeerCertArgs* args;
        args = (ProcPeerCertArgs*)XMALLOC(
            sizeof(ProcPeerCertArgs), cm->heap, DYNAMIC_TYPE_TMP_BUFFER);
        if (args == NULL) {
            XFREE(cert, cm->heap, DYNAMIC_TYPE_DCERT);
            return MEMORY_E;
        }
    #else
        ProcPeerCertArgs  args[1];
    #endif

        certBuf.buffer = (byte*)buff;
        certBuf.length = (unsigned int)sz;
        XMEMSET(args, 0, sizeof(ProcPeerCertArgs));

        args->totalCerts = 1;
        args->certs = &certBuf;
        args->dCert = cert;
        args->dCertInit = 1;

        if (err_val != 0) {
            ret = err_val;
        }
        ret = DoVerifyCallback(cm, NULL, ret, args);
    #ifdef WOLFSSL_SMALL_STACK
        XFREE(args, cm->heap, DYNAMIC_TYPE_TMP_BUFFER);
    #endif
    }
#else
    (void)err_val;
#endif

    FreeDecodedCert(cert);
    FreeDer(&der);
#ifdef WOLFSSL_SMALL_STACK
    XFREE(cert, cm->heap, DYNAMIC_TYPE_DCERT);
#endif

    return ret == 0 ? WOLFSSL_SUCCESS : ret;
}

/* Verify the certificate, WOLFSSL_SUCCESS for ok, < 0 for error */
int wolfSSL_CertManagerVerifyBuffer(WOLFSSL_CERT_MANAGER* cm, const byte* buff,
                                    long sz, int format)
{
    return CM_VerifyBuffer_ex(cm, buff, sz, format, 0);
}
#endif /* !NO_WOLFSSL_CLIENT || !WOLFSSL_NO_CLIENT_AUTH */

/* turn on OCSP if off and compiled in, set options */
int wolfSSL_CertManagerEnableOCSP(WOLFSSL_CERT_MANAGER* cm, int options)
{
    int ret = WOLFSSL_SUCCESS;

    (void)options;

    WOLFSSL_ENTER("wolfSSL_CertManagerEnableOCSP");
    if (cm == NULL)
        return BAD_FUNC_ARG;

    #ifdef HAVE_OCSP
        if (cm->ocsp == NULL) {
            cm->ocsp = (WOLFSSL_OCSP*)XMALLOC(sizeof(WOLFSSL_OCSP), cm->heap,
                                              DYNAMIC_TYPE_OCSP);
            if (cm->ocsp == NULL)
                return MEMORY_E;

            if (InitOCSP(cm->ocsp, cm) != 0) {
                WOLFSSL_MSG("Init OCSP failed");
                FreeOCSP(cm->ocsp, 1);
                cm->ocsp = NULL;
                return WOLFSSL_FAILURE;
            }
        }
        cm->ocspEnabled = 1;
        if (options & WOLFSSL_OCSP_URL_OVERRIDE)
            cm->ocspUseOverrideURL = 1;
        if (options & WOLFSSL_OCSP_NO_NONCE)
            cm->ocspSendNonce = 0;
        else
            cm->ocspSendNonce = 1;
        if (options & WOLFSSL_OCSP_CHECKALL)
            cm->ocspCheckAll = 1;
        #ifndef WOLFSSL_USER_IO
            cm->ocspIOCb = EmbedOcspLookup;
            cm->ocspRespFreeCb = EmbedOcspRespFree;
            cm->ocspIOCtx = cm->heap;
        #endif /* WOLFSSL_USER_IO */
    #else
        ret = NOT_COMPILED_IN;
    #endif

    return ret;
}


int wolfSSL_CertManagerDisableOCSP(WOLFSSL_CERT_MANAGER* cm)
{
    WOLFSSL_ENTER("wolfSSL_CertManagerDisableOCSP");
    if (cm == NULL)
        return BAD_FUNC_ARG;

    cm->ocspEnabled = 0;

    return WOLFSSL_SUCCESS;
}

/* turn on OCSP Stapling if off and compiled in, set options */
int wolfSSL_CertManagerEnableOCSPStapling(WOLFSSL_CERT_MANAGER* cm)
{
    int ret = WOLFSSL_SUCCESS;

    WOLFSSL_ENTER("wolfSSL_CertManagerEnableOCSPStapling");

    if (cm == NULL)
        return BAD_FUNC_ARG;

#if defined(HAVE_CERTIFICATE_STATUS_REQUEST) \
 || defined(HAVE_CERTIFICATE_STATUS_REQUEST_V2)
    #ifndef NO_WOLFSSL_SERVER
    if (cm->ocsp_stapling == NULL) {
        cm->ocsp_stapling = (WOLFSSL_OCSP*)XMALLOC(sizeof(WOLFSSL_OCSP),
                                               cm->heap, DYNAMIC_TYPE_OCSP);
        if (cm->ocsp_stapling == NULL)
            return MEMORY_E;

        if (InitOCSP(cm->ocsp_stapling, cm) != 0) {
            WOLFSSL_MSG("Init OCSP failed");
            FreeOCSP(cm->ocsp_stapling, 1);
            cm->ocsp_stapling = NULL;
            return WOLFSSL_FAILURE;
        }
    }

    #ifndef WOLFSSL_USER_IO
        cm->ocspIOCb = EmbedOcspLookup;
        cm->ocspRespFreeCb = EmbedOcspRespFree;
        cm->ocspIOCtx = cm->heap;
    #endif /* WOLFSSL_USER_IO */
    #endif /* NO_WOLFSSL_SERVER */
    cm->ocspStaplingEnabled = 1;
#else
    ret = NOT_COMPILED_IN;
#endif

    return ret;
}

int wolfSSL_CertManagerDisableOCSPStapling(WOLFSSL_CERT_MANAGER* cm)
{
    int ret = WOLFSSL_SUCCESS;

    WOLFSSL_ENTER("wolfSSL_CertManagerDisableOCSPStapling");

    if (cm == NULL)
        return BAD_FUNC_ARG;

#if defined(HAVE_CERTIFICATE_STATUS_REQUEST) \
 || defined(HAVE_CERTIFICATE_STATUS_REQUEST_V2)
    cm->ocspStaplingEnabled = 0;
#else
    ret = NOT_COMPILED_IN;
#endif
    return ret;
}

/* require OCSP stapling response */
int wolfSSL_CertManagerEnableOCSPMustStaple(WOLFSSL_CERT_MANAGER* cm)
{
    int ret;

    WOLFSSL_ENTER("wolfSSL_CertManagerEnableOCSPMustStaple");

    if (cm == NULL)
        return BAD_FUNC_ARG;

#if defined(HAVE_CERTIFICATE_STATUS_REQUEST) \
 || defined(HAVE_CERTIFICATE_STATUS_REQUEST_V2)
    #ifndef NO_WOLFSSL_CLIENT
        cm->ocspMustStaple = 1;
    #endif
    ret = WOLFSSL_SUCCESS;
#else
    ret = NOT_COMPILED_IN;
#endif

    return ret;
}

int wolfSSL_CertManagerDisableOCSPMustStaple(WOLFSSL_CERT_MANAGER* cm)
{
    int ret;

    WOLFSSL_ENTER("wolfSSL_CertManagerDisableOCSPMustStaple");

    if (cm == NULL)
        return BAD_FUNC_ARG;

#if defined(HAVE_CERTIFICATE_STATUS_REQUEST) \
 || defined(HAVE_CERTIFICATE_STATUS_REQUEST_V2)
    #ifndef NO_WOLFSSL_CLIENT
        cm->ocspMustStaple = 0;
    #endif
    ret = WOLFSSL_SUCCESS;
#else
    ret = NOT_COMPILED_IN;
#endif
    return ret;
}

#ifdef HAVE_OCSP
/* check CRL if enabled, WOLFSSL_SUCCESS  */
int wolfSSL_CertManagerCheckOCSP(WOLFSSL_CERT_MANAGER* cm, byte* der, int sz)
{
    int ret;
#ifdef WOLFSSL_SMALL_STACK
    DecodedCert* cert = NULL;
#else
    DecodedCert  cert[1];
#endif

    WOLFSSL_ENTER("wolfSSL_CertManagerCheckOCSP");

    if (cm == NULL)
        return BAD_FUNC_ARG;

    if (cm->ocspEnabled == 0)
        return WOLFSSL_SUCCESS;

#ifdef WOLFSSL_SMALL_STACK
    cert = (DecodedCert*)XMALLOC(sizeof(DecodedCert), cm->heap, DYNAMIC_TYPE_DCERT);
    if (cert == NULL)
        return MEMORY_E;
#endif

    InitDecodedCert(cert, der, sz, NULL);

    if ((ret = ParseCertRelative(cert, CERT_TYPE, VERIFY_OCSP, cm)) != 0) {
        WOLFSSL_MSG("ParseCert failed");
    }
    else if ((ret = CheckCertOCSP(cm->ocsp, cert, NULL)) != 0) {
        WOLFSSL_MSG("CheckCertOCSP failed");
    }

    FreeDecodedCert(cert);
#ifdef WOLFSSL_SMALL_STACK
    XFREE(cert, cm->heap, DYNAMIC_TYPE_DCERT);
#endif

    return ret == 0 ? WOLFSSL_SUCCESS : ret;
}

WOLFSSL_API int wolfSSL_CertManagerCheckOCSPResponse(WOLFSSL_CERT_MANAGER *cm,
                                                    byte *response, int responseSz, buffer *responseBuffer,
                                                    CertStatus *status, OcspEntry *entry, OcspRequest *ocspRequest)
{
    int ret;

    WOLFSSL_ENTER("wolfSSL_CertManagerCheckOCSPResponse");
    if (cm == NULL || response == NULL)
        return BAD_FUNC_ARG;
    if (cm->ocspEnabled == 0)
        return WOLFSSL_SUCCESS;

    ret = CheckOcspResponse(cm->ocsp, response, responseSz, responseBuffer, status,
                        entry, ocspRequest);

    return ret == 0 ? WOLFSSL_SUCCESS : ret;
}

int wolfSSL_CertManagerSetOCSPOverrideURL(WOLFSSL_CERT_MANAGER* cm,
                                          const char* url)
{
    WOLFSSL_ENTER("wolfSSL_CertManagerSetOCSPOverrideURL");
    if (cm == NULL)
        return BAD_FUNC_ARG;

    XFREE(cm->ocspOverrideURL, cm->heap, DYNAMIC_TYPE_URL);
    if (url != NULL) {
        int urlSz = (int)XSTRLEN(url) + 1;
        cm->ocspOverrideURL = (char*)XMALLOC(urlSz, cm->heap, DYNAMIC_TYPE_URL);
        if (cm->ocspOverrideURL != NULL) {
            XMEMCPY(cm->ocspOverrideURL, url, urlSz);
        }
        else
            return MEMORY_E;
    }
    else
        cm->ocspOverrideURL = NULL;

    return WOLFSSL_SUCCESS;
}


int wolfSSL_CertManagerSetOCSP_Cb(WOLFSSL_CERT_MANAGER* cm,
                        CbOCSPIO ioCb, CbOCSPRespFree respFreeCb, void* ioCbCtx)
{
    WOLFSSL_ENTER("wolfSSL_CertManagerSetOCSP_Cb");
    if (cm == NULL)
        return BAD_FUNC_ARG;

    cm->ocspIOCb = ioCb;
    cm->ocspRespFreeCb = respFreeCb;
    cm->ocspIOCtx = ioCbCtx;

    return WOLFSSL_SUCCESS;
}


int wolfSSL_EnableOCSP(WOLFSSL* ssl, int options)
{
    WOLFSSL_ENTER("wolfSSL_EnableOCSP");
    if (ssl)
        return wolfSSL_CertManagerEnableOCSP(SSL_CM(ssl), options);
    else
        return BAD_FUNC_ARG;
}

int wolfSSL_DisableOCSP(WOLFSSL* ssl)
{
    WOLFSSL_ENTER("wolfSSL_DisableOCSP");
    if (ssl)
        return wolfSSL_CertManagerDisableOCSP(SSL_CM(ssl));
    else
        return BAD_FUNC_ARG;
}


int wolfSSL_EnableOCSPStapling(WOLFSSL* ssl)
{
    WOLFSSL_ENTER("wolfSSL_EnableOCSPStapling");
    if (ssl)
        return wolfSSL_CertManagerEnableOCSPStapling(SSL_CM(ssl));
    else
        return BAD_FUNC_ARG;
}

int wolfSSL_DisableOCSPStapling(WOLFSSL* ssl)
{
    WOLFSSL_ENTER("wolfSSL_DisableOCSPStapling");
    if (ssl)
        return wolfSSL_CertManagerDisableOCSPStapling(SSL_CM(ssl));
    else
        return BAD_FUNC_ARG;
}

int wolfSSL_SetOCSP_OverrideURL(WOLFSSL* ssl, const char* url)
{
    WOLFSSL_ENTER("wolfSSL_SetOCSP_OverrideURL");
    if (ssl)
        return wolfSSL_CertManagerSetOCSPOverrideURL(SSL_CM(ssl), url);
    else
        return BAD_FUNC_ARG;
}


int wolfSSL_SetOCSP_Cb(WOLFSSL* ssl,
                        CbOCSPIO ioCb, CbOCSPRespFree respFreeCb, void* ioCbCtx)
{
    WOLFSSL_ENTER("wolfSSL_SetOCSP_Cb");
    if (ssl) {
        ssl->ocspIOCtx = ioCbCtx; /* use SSL specific ioCbCtx */
        return wolfSSL_CertManagerSetOCSP_Cb(SSL_CM(ssl),
                                             ioCb, respFreeCb, NULL);
    }
    else
        return BAD_FUNC_ARG;
}


int wolfSSL_CTX_EnableOCSP(WOLFSSL_CTX* ctx, int options)
{
    WOLFSSL_ENTER("wolfSSL_CTX_EnableOCSP");
    if (ctx)
        return wolfSSL_CertManagerEnableOCSP(ctx->cm, options);
    else
        return BAD_FUNC_ARG;
}


int wolfSSL_CTX_DisableOCSP(WOLFSSL_CTX* ctx)
{
    WOLFSSL_ENTER("wolfSSL_CTX_DisableOCSP");
    if (ctx)
        return wolfSSL_CertManagerDisableOCSP(ctx->cm);
    else
        return BAD_FUNC_ARG;
}


int wolfSSL_CTX_SetOCSP_OverrideURL(WOLFSSL_CTX* ctx, const char* url)
{
    WOLFSSL_ENTER("wolfSSL_SetOCSP_OverrideURL");
    if (ctx)
        return wolfSSL_CertManagerSetOCSPOverrideURL(ctx->cm, url);
    else
        return BAD_FUNC_ARG;
}


int wolfSSL_CTX_SetOCSP_Cb(WOLFSSL_CTX* ctx, CbOCSPIO ioCb,
                           CbOCSPRespFree respFreeCb, void* ioCbCtx)
{
    WOLFSSL_ENTER("wolfSSL_CTX_SetOCSP_Cb");
    if (ctx)
        return wolfSSL_CertManagerSetOCSP_Cb(ctx->cm, ioCb,
                                             respFreeCb, ioCbCtx);
    else
        return BAD_FUNC_ARG;
}

#if defined(HAVE_CERTIFICATE_STATUS_REQUEST) \
 || defined(HAVE_CERTIFICATE_STATUS_REQUEST_V2)
int wolfSSL_CTX_EnableOCSPStapling(WOLFSSL_CTX* ctx)
{
    WOLFSSL_ENTER("wolfSSL_CTX_EnableOCSPStapling");
    if (ctx)
        return wolfSSL_CertManagerEnableOCSPStapling(ctx->cm);
    else
        return BAD_FUNC_ARG;
}

int wolfSSL_CTX_DisableOCSPStapling(WOLFSSL_CTX* ctx)
{
    WOLFSSL_ENTER("wolfSSL_CTX_DisableOCSPStapling");
    if (ctx)
        return wolfSSL_CertManagerDisableOCSPStapling(ctx->cm);
    else
        return BAD_FUNC_ARG;
}

int wolfSSL_CTX_EnableOCSPMustStaple(WOLFSSL_CTX* ctx)
{
    WOLFSSL_ENTER("wolfSSL_CTX_EnableOCSPMustStaple");
    if (ctx)
        return wolfSSL_CertManagerEnableOCSPMustStaple(ctx->cm);
    else
        return BAD_FUNC_ARG;
}

int wolfSSL_CTX_DisableOCSPMustStaple(WOLFSSL_CTX* ctx)
{
    WOLFSSL_ENTER("wolfSSL_CTX_DisableOCSPMustStaple");
    if (ctx)
        return wolfSSL_CertManagerDisableOCSPMustStaple(ctx->cm);
    else
        return BAD_FUNC_ARG;
}
#endif /* HAVE_CERTIFICATE_STATUS_REQUEST || HAVE_CERTIFICATE_STATUS_REQUEST_V2 */

#endif /* HAVE_OCSP */

/* macro to get verify settings for AddCA */
#define GET_VERIFY_SETTING_CTX(ctx) \
    ((ctx) && (ctx)->verifyNone ? NO_VERIFY : VERIFY)
#define GET_VERIFY_SETTING_SSL(ssl) \
    ((ssl)->options.verifyNone ? NO_VERIFY : VERIFY)

#ifndef NO_FILESYSTEM

/* process a file with name fname into ctx of format and type
   userChain specifies a user certificate chain to pass during handshake */
int ProcessFile(WOLFSSL_CTX* ctx, const char* fname, int format, int type,
                WOLFSSL* ssl, int userChain, WOLFSSL_CRL* crl, int verify)
{
#ifdef WOLFSSL_SMALL_STACK
    byte   staticBuffer[1]; /* force heap usage */
#else
    byte   staticBuffer[FILE_BUFFER_SIZE];
#endif
    byte*  myBuffer = staticBuffer;
    int    dynamic = 0;
    int    ret;
    long   sz = 0;
    XFILE  file;
    void*  heapHint = wolfSSL_CTX_GetHeap(ctx, ssl);
#ifndef NO_CODING
    const char* header = NULL;
    const char* footer = NULL;
#endif

    (void)crl;
    (void)heapHint;

    if (fname == NULL) return WOLFSSL_BAD_FILE;

    file = XFOPEN(fname, "rb");
    if (file == XBADFILE) return WOLFSSL_BAD_FILE;
    if (XFSEEK(file, 0, XSEEK_END) != 0) {
        XFCLOSE(file);
        return WOLFSSL_BAD_FILE;
    }
    sz = XFTELL(file);
    XREWIND(file);

    if (sz > MAX_WOLFSSL_FILE_SIZE || sz <= 0) {
        WOLFSSL_MSG("ProcessFile file size error");
        XFCLOSE(file);
        return WOLFSSL_BAD_FILE;
    }

    if (sz > (long)sizeof(staticBuffer)) {
        WOLFSSL_MSG("Getting dynamic buffer");
        myBuffer = (byte*)XMALLOC(sz, heapHint, DYNAMIC_TYPE_FILE);
        if (myBuffer == NULL) {
            XFCLOSE(file);
            return WOLFSSL_BAD_FILE;
        }
        dynamic = 1;
    }

    if ((size_t)XFREAD(myBuffer, 1, sz, file) != (size_t)sz)
        ret = WOLFSSL_BAD_FILE;
    else {
        /* Try to detect type by parsing cert header and footer */
        if (type == DETECT_CERT_TYPE) {
#ifndef NO_CODING
            if (wc_PemGetHeaderFooter(CA_TYPE, &header, &footer) == 0 &&
               (XSTRNSTR((char*)myBuffer, header, (int)sz) != NULL)) {
                type = CA_TYPE;
            }
#ifdef HAVE_CRL
            else if (wc_PemGetHeaderFooter(CRL_TYPE, &header, &footer) == 0 &&
                    (XSTRNSTR((char*)myBuffer, header, (int)sz) != NULL)) {
                type = CRL_TYPE;
            }
#endif
            else if (wc_PemGetHeaderFooter(CERT_TYPE, &header, &footer) == 0 &&
                    (XSTRNSTR((char*)myBuffer, header, (int)sz) != NULL)) {
                type = CERT_TYPE;
            }
            else
#endif
            {
                WOLFSSL_MSG("Failed to detect certificate type");
                if (dynamic)
                    XFREE(myBuffer, heapHint, DYNAMIC_TYPE_FILE);
                XFCLOSE(file);
                return WOLFSSL_BAD_CERTTYPE;
            }
        }
        if ((type == CA_TYPE || type == TRUSTED_PEER_TYPE)
                                          && format == WOLFSSL_FILETYPE_PEM) {
            ret = ProcessChainBuffer(ctx, myBuffer, sz, format, type, ssl,
                                     verify);
        }
#ifdef HAVE_CRL
        else if (type == CRL_TYPE)
            ret = BufferLoadCRL(crl, myBuffer, sz, format, verify);
#endif
        else
            ret = ProcessBuffer(ctx, myBuffer, sz, format, type, ssl, NULL,
                                userChain, verify);
    }

    XFCLOSE(file);
    if (dynamic)
        XFREE(myBuffer, heapHint, DYNAMIC_TYPE_FILE);

    return ret;
}

/* loads file then loads each file in path, no c_rehash */
int wolfSSL_CTX_load_verify_locations_ex(WOLFSSL_CTX* ctx, const char* file,
                                     const char* path, word32 flags)
{
    int ret = WOLFSSL_SUCCESS;
#ifndef NO_WOLFSSL_DIR
    int fileRet;
    int successCount = 0;
    int failCount = 0;
#endif
    int verify;

    WOLFSSL_MSG("wolfSSL_CTX_load_verify_locations_ex");

    if (ctx == NULL || (file == NULL && path == NULL)) {
        return WOLFSSL_FAILURE;
    }

    verify = GET_VERIFY_SETTING_CTX(ctx);
    if (flags & WOLFSSL_LOAD_FLAG_DATE_ERR_OKAY)
        verify = VERIFY_SKIP_DATE;

    if (file) {
        ret = ProcessFile(ctx, file, WOLFSSL_FILETYPE_PEM, CA_TYPE, NULL, 0,
                          NULL, verify);
#ifndef NO_WOLFSSL_DIR
        if (ret == WOLFSSL_SUCCESS)
            successCount++;
#endif
#if defined(WOLFSSL_TRUST_PEER_CERT) && defined(OPENSSL_COMPATIBLE_DEFAULTS)
        ret = wolfSSL_CTX_trust_peer_cert(ctx, file, WOLFSSL_FILETYPE_PEM);
        if (ret != WOLFSSL_SUCCESS) {
            WOLFSSL_MSG("wolfSSL_CTX_trust_peer_cert error");
        }
#endif
    }

    if (ret == WOLFSSL_SUCCESS && path) {
#ifndef NO_WOLFSSL_DIR
        char* name = NULL;
    #ifdef WOLFSSL_SMALL_STACK
        ReadDirCtx* readCtx;
        readCtx = (ReadDirCtx*)XMALLOC(sizeof(ReadDirCtx), ctx->heap,
                                                       DYNAMIC_TYPE_DIRCTX);
        if (readCtx == NULL)
            return MEMORY_E;
    #else
        ReadDirCtx readCtx[1];
    #endif

        /* try to load each regular file in path */
        fileRet = wc_ReadDirFirst(readCtx, path, &name);
        while (fileRet == 0 && name) {
            WOLFSSL_MSG(name); /* log file name */
            ret = ProcessFile(ctx, name, WOLFSSL_FILETYPE_PEM, CA_TYPE,
                              NULL, 0, NULL, verify);
            if (ret != WOLFSSL_SUCCESS) {
                /* handle flags for ignoring errors, skipping expired certs or
                   by PEM certificate header error */
                if ( (flags & WOLFSSL_LOAD_FLAG_IGNORE_ERR) ||
                    ((flags & WOLFSSL_LOAD_FLAG_PEM_CA_ONLY) &&
                       (ret == ASN_NO_PEM_HEADER))) {
                    /* Do not fail here if a certificate fails to load,
                       continue to next file */
                    unsigned long err;
                    CLEAR_ASN_NO_PEM_HEADER_ERROR(err);
    #if defined(WOLFSSL_QT)
                    ret = WOLFSSL_SUCCESS;
    #endif
                }
                else {
                    WOLFSSL_ERROR(ret);
                    WOLFSSL_MSG("Load CA file failed, continuing");
                    failCount++;
                }
            }
            else {
    #if defined(WOLFSSL_TRUST_PEER_CERT) && defined(OPENSSL_COMPATIBLE_DEFAULTS)
                ret = wolfSSL_CTX_trust_peer_cert(ctx, file, WOLFSSL_FILETYPE_PEM);
                if (ret != WOLFSSL_SUCCESS) {
                    WOLFSSL_MSG("wolfSSL_CTX_trust_peer_cert error. Ignoring"
                            "this error.");
                }
    #endif
                successCount++;
            }
            fileRet = wc_ReadDirNext(readCtx, path, &name);
        }
        wc_ReadDirClose(readCtx);

        /* pass directory read failure to response code */
        if (fileRet != WC_READDIR_NOFILE) {
            ret = fileRet;
    #if defined(WOLFSSL_QT)
            if (ret == BAD_PATH_ERROR &&
                flags & WOLFSSL_LOAD_FLAG_IGNORE_BAD_PATH_ERR) {
               /* QSslSocket always loads certs in system folder
                * when it is initialized.
                * Compliant with OpenSSL when flag sets.
                */
                ret = WOLFSSL_SUCCESS;
            }
            else {
                /* qssl socket wants to know errors. */
                WOLFSSL_ERROR(ret);
            }
    #endif
        }
        /* report failure if no files were loaded or there were failures */
        else if (successCount == 0 || failCount > 0) {
            /* use existing error code if exists */
    #if defined(WOLFSSL_QT)
            /* compliant with OpenSSL when flag sets*/
            if (!(flags & WOLFSSL_LOAD_FLAG_IGNORE_ZEROFILE))
    #endif
            {
                ret = WOLFSSL_FAILURE;
            }
        }
        else {
            ret = WOLFSSL_SUCCESS;
        }

    #ifdef WOLFSSL_SMALL_STACK
        XFREE(readCtx, ctx->heap, DYNAMIC_TYPE_DIRCTX);
    #endif
#else
        ret = NOT_COMPILED_IN;
        (void)flags;
#endif
    }

    return ret;
}

WOLFSSL_ABI
int wolfSSL_CTX_load_verify_locations(WOLFSSL_CTX* ctx, const char* file,
                                     const char* path)
{
    int ret = wolfSSL_CTX_load_verify_locations_ex(ctx, file, path,
        WOLFSSL_LOAD_VERIFY_DEFAULT_FLAGS);

    return WS_RETURN_CODE(ret,WOLFSSL_FAILURE);
}


#ifdef WOLFSSL_TRUST_PEER_CERT
/* Used to specify a peer cert to match when connecting
    ctx : the ctx structure to load in peer cert
    file: the string name of cert file
    type: type of format such as PEM/DER
 */
int wolfSSL_CTX_trust_peer_cert(WOLFSSL_CTX* ctx, const char* file, int type)
{
    WOLFSSL_ENTER("wolfSSL_CTX_trust_peer_cert");

    if (ctx == NULL || file == NULL) {
        return WOLFSSL_FAILURE;
    }

    return ProcessFile(ctx, file, type, TRUSTED_PEER_TYPE, NULL, 0, NULL,
                       GET_VERIFY_SETTING_CTX(ctx));
}

int wolfSSL_trust_peer_cert(WOLFSSL* ssl, const char* file, int type)
{
    WOLFSSL_ENTER("wolfSSL_trust_peer_cert");

    if (ssl == NULL || file == NULL) {
        return WOLFSSL_FAILURE;
    }

    return ProcessFile(NULL, file, type, TRUSTED_PEER_TYPE, ssl, 0, NULL,
                       GET_VERIFY_SETTING_SSL(ssl));
}
#endif /* WOLFSSL_TRUST_PEER_CERT */


#if !defined(NO_WOLFSSL_CLIENT) || !defined(WOLFSSL_NO_CLIENT_AUTH)
/* Verify the certificate, WOLFSSL_SUCCESS for ok, < 0 for error */
int wolfSSL_CertManagerVerify(WOLFSSL_CERT_MANAGER* cm, const char* fname,
                             int format)
{
    int    ret = WOLFSSL_FATAL_ERROR;
#ifdef WOLFSSL_SMALL_STACK
    byte   staticBuffer[1]; /* force heap usage */
#else
    byte   staticBuffer[FILE_BUFFER_SIZE];
#endif
    byte*  myBuffer = staticBuffer;
    int    dynamic = 0;
    long   sz = 0;
    XFILE  file = XFOPEN(fname, "rb");

    WOLFSSL_ENTER("wolfSSL_CertManagerVerify");

    if (file == XBADFILE) return WOLFSSL_BAD_FILE;
    if(XFSEEK(file, 0, XSEEK_END) != 0) {
        XFCLOSE(file);
        return WOLFSSL_BAD_FILE;
    }
    sz = XFTELL(file);
    XREWIND(file);

    if (sz > MAX_WOLFSSL_FILE_SIZE || sz <= 0) {
        WOLFSSL_MSG("CertManagerVerify file size error");
        XFCLOSE(file);
        return WOLFSSL_BAD_FILE;
    }

    if (sz > (long)sizeof(staticBuffer)) {
        WOLFSSL_MSG("Getting dynamic buffer");
        myBuffer = (byte*) XMALLOC(sz, cm->heap, DYNAMIC_TYPE_FILE);
        if (myBuffer == NULL) {
            XFCLOSE(file);
            return WOLFSSL_BAD_FILE;
        }
        dynamic = 1;
    }

    if ((size_t)XFREAD(myBuffer, 1, sz, file) != (size_t)sz)
        ret = WOLFSSL_BAD_FILE;
    else
        ret = wolfSSL_CertManagerVerifyBuffer(cm, myBuffer, sz, format);

    XFCLOSE(file);
    if (dynamic)
        XFREE(myBuffer, cm->heap, DYNAMIC_TYPE_FILE);

    return ret;
}
#endif

/* like load verify locations, 1 for success, < 0 for error */
int wolfSSL_CertManagerLoadCA(WOLFSSL_CERT_MANAGER* cm, const char* file,
                             const char* path)
{
    int ret = WOLFSSL_FATAL_ERROR;
    WOLFSSL_CTX* tmp;

    WOLFSSL_ENTER("wolfSSL_CertManagerLoadCA");

    if (cm == NULL) {
        WOLFSSL_MSG("No CertManager error");
        return ret;
    }
    tmp = wolfSSL_CTX_new(cm_pick_method());

    if (tmp == NULL) {
        WOLFSSL_MSG("CTX new failed");
        return ret;
    }

    /* for tmp use */
    wolfSSL_CertManagerFree(tmp->cm);
    tmp->cm = cm;

    ret = wolfSSL_CTX_load_verify_locations(tmp, file, path);

    /* don't lose our good one */
    tmp->cm = NULL;
    wolfSSL_CTX_free(tmp);

    return ret;
}


#endif /* NO_FILESYSTEM */

#ifdef HAVE_CRL

/* check CRL if enabled, WOLFSSL_SUCCESS  */
int wolfSSL_CertManagerCheckCRL(WOLFSSL_CERT_MANAGER* cm, byte* der, int sz)
{
    int ret = 0;
#ifdef WOLFSSL_SMALL_STACK
    DecodedCert* cert = NULL;
#else
    DecodedCert  cert[1];
#endif

    WOLFSSL_ENTER("wolfSSL_CertManagerCheckCRL");

    if (cm == NULL)
        return BAD_FUNC_ARG;

    if (cm->crlEnabled == 0)
        return WOLFSSL_SUCCESS;

#ifdef WOLFSSL_SMALL_STACK
    cert = (DecodedCert*)XMALLOC(sizeof(DecodedCert), NULL, DYNAMIC_TYPE_DCERT);
    if (cert == NULL)
        return MEMORY_E;
#endif

    InitDecodedCert(cert, der, sz, NULL);

    if ((ret = ParseCertRelative(cert, CERT_TYPE, VERIFY_CRL, cm)) != 0) {
        WOLFSSL_MSG("ParseCert failed");
    }
    else if ((ret = CheckCertCRL(cm->crl, cert)) != 0) {
        WOLFSSL_MSG("CheckCertCRL failed");
    }

    FreeDecodedCert(cert);
#ifdef WOLFSSL_SMALL_STACK
    XFREE(cert, NULL, DYNAMIC_TYPE_DCERT);
#endif

    return ret == 0 ? WOLFSSL_SUCCESS : ret;
}


int wolfSSL_CertManagerSetCRL_Cb(WOLFSSL_CERT_MANAGER* cm, CbMissingCRL cb)
{
    WOLFSSL_ENTER("wolfSSL_CertManagerSetCRL_Cb");
    if (cm == NULL)
        return BAD_FUNC_ARG;

    cm->cbMissingCRL = cb;

    return WOLFSSL_SUCCESS;
}

#ifdef HAVE_CRL_IO
int wolfSSL_CertManagerSetCRL_IOCb(WOLFSSL_CERT_MANAGER* cm, CbCrlIO cb)
{
    if (cm == NULL)
        return BAD_FUNC_ARG;

    cm->crl->crlIOCb = cb;

    return WOLFSSL_SUCCESS;
}
#endif

#ifndef NO_FILESYSTEM
int wolfSSL_CertManagerLoadCRL(WOLFSSL_CERT_MANAGER* cm, const char* path,
                              int type, int monitor)
{
    WOLFSSL_ENTER("wolfSSL_CertManagerLoadCRL");
    if (cm == NULL)
        return BAD_FUNC_ARG;

    if (cm->crl == NULL) {
        if (wolfSSL_CertManagerEnableCRL(cm, 0) != WOLFSSL_SUCCESS) {
            WOLFSSL_MSG("Enable CRL failed");
            return WOLFSSL_FATAL_ERROR;
        }
    }

    return LoadCRL(cm->crl, path, type, monitor);
}

int wolfSSL_CertManagerLoadCRLFile(WOLFSSL_CERT_MANAGER* cm, const char* file,
                              int type)
{
    WOLFSSL_ENTER("wolfSSL_CertManagerLoadCRLFile");
    if (cm == NULL || file == NULL)
        return BAD_FUNC_ARG;

    if (cm->crl == NULL) {
        if (wolfSSL_CertManagerEnableCRL(cm, 0) != WOLFSSL_SUCCESS) {
            WOLFSSL_MSG("Enable CRL failed");
            return WOLFSSL_FATAL_ERROR;
        }
    }

    return ProcessFile(NULL, file, type, CRL_TYPE, NULL, 0, cm->crl,
            VERIFY);
}
#endif

int wolfSSL_EnableCRL(WOLFSSL* ssl, int options)
{
    WOLFSSL_ENTER("wolfSSL_EnableCRL");
    if (ssl)
        return wolfSSL_CertManagerEnableCRL(SSL_CM(ssl), options);
    else
        return BAD_FUNC_ARG;
}


int wolfSSL_DisableCRL(WOLFSSL* ssl)
{
    WOLFSSL_ENTER("wolfSSL_DisableCRL");
    if (ssl)
        return wolfSSL_CertManagerDisableCRL(SSL_CM(ssl));
    else
        return BAD_FUNC_ARG;
}

#ifndef NO_FILESYSTEM
int wolfSSL_LoadCRL(WOLFSSL* ssl, const char* path, int type, int monitor)
{
    WOLFSSL_ENTER("wolfSSL_LoadCRL");
    if (ssl)
        return wolfSSL_CertManagerLoadCRL(SSL_CM(ssl), path, type, monitor);
    else
        return BAD_FUNC_ARG;
}

int wolfSSL_LoadCRLFile(WOLFSSL* ssl, const char* file, int type)
{
    WOLFSSL_ENTER("wolfSSL_LoadCRL");
    if (ssl)
        return wolfSSL_CertManagerLoadCRLFile(SSL_CM(ssl), file, type);
    else
        return BAD_FUNC_ARG;
}
#endif


int wolfSSL_SetCRL_Cb(WOLFSSL* ssl, CbMissingCRL cb)
{
    WOLFSSL_ENTER("wolfSSL_SetCRL_Cb");
    if (ssl)
        return wolfSSL_CertManagerSetCRL_Cb(SSL_CM(ssl), cb);
    else
        return BAD_FUNC_ARG;
}

#ifdef HAVE_CRL_IO
int wolfSSL_SetCRL_IOCb(WOLFSSL* ssl, CbCrlIO cb)
{
    WOLFSSL_ENTER("wolfSSL_SetCRL_Cb");
    if (ssl)
        return wolfSSL_CertManagerSetCRL_IOCb(SSL_CM(ssl), cb);
    else
        return BAD_FUNC_ARG;
}
#endif

int wolfSSL_CTX_EnableCRL(WOLFSSL_CTX* ctx, int options)
{
    WOLFSSL_ENTER("wolfSSL_CTX_EnableCRL");
    if (ctx)
        return wolfSSL_CertManagerEnableCRL(ctx->cm, options);
    else
        return BAD_FUNC_ARG;
}


int wolfSSL_CTX_DisableCRL(WOLFSSL_CTX* ctx)
{
    WOLFSSL_ENTER("wolfSSL_CTX_DisableCRL");
    if (ctx)
        return wolfSSL_CertManagerDisableCRL(ctx->cm);
    else
        return BAD_FUNC_ARG;
}


#ifndef NO_FILESYSTEM
int wolfSSL_CTX_LoadCRL(WOLFSSL_CTX* ctx, const char* path,
                        int type, int monitor)
{
    WOLFSSL_ENTER("wolfSSL_CTX_LoadCRL");
    if (ctx)
        return wolfSSL_CertManagerLoadCRL(ctx->cm, path, type, monitor);
    else
        return BAD_FUNC_ARG;
}

int wolfSSL_CTX_LoadCRLFile(WOLFSSL_CTX* ctx, const char* file,
                        int type)
{
    WOLFSSL_ENTER("wolfSSL_CTX_LoadCRL");
    if (ctx)
        return wolfSSL_CertManagerLoadCRLFile(ctx->cm, file, type);
    else
        return BAD_FUNC_ARG;
}
#endif


int wolfSSL_CTX_SetCRL_Cb(WOLFSSL_CTX* ctx, CbMissingCRL cb)
{
    WOLFSSL_ENTER("wolfSSL_CTX_SetCRL_Cb");
    if (ctx)
        return wolfSSL_CertManagerSetCRL_Cb(ctx->cm, cb);
    else
        return BAD_FUNC_ARG;
}

#ifdef HAVE_CRL_IO
int wolfSSL_CTX_SetCRL_IOCb(WOLFSSL_CTX* ctx, CbCrlIO cb)
{
    WOLFSSL_ENTER("wolfSSL_CTX_SetCRL_IOCb");
    if (ctx)
        return wolfSSL_CertManagerSetCRL_IOCb(ctx->cm, cb);
    else
        return BAD_FUNC_ARG;
}
#endif


#endif /* HAVE_CRL */


#ifndef NO_FILESYSTEM


#ifdef WOLFSSL_DER_LOAD

/* Add format parameter to allow DER load of CA files */
int wolfSSL_CTX_der_load_verify_locations(WOLFSSL_CTX* ctx, const char* file,
                                          int format)
{
    WOLFSSL_ENTER("wolfSSL_CTX_der_load_verify_locations");
    if (ctx == NULL || file == NULL)
        return WOLFSSL_FAILURE;

    if (ProcessFile(ctx, file, format, CA_TYPE, NULL, 0, NULL,
                    GET_VERIFY_SETTING_CTX(ctx)) == WOLFSSL_SUCCESS) {
        return WOLFSSL_SUCCESS;
    }

    return WOLFSSL_FAILURE;
}

#endif /* WOLFSSL_DER_LOAD */



WOLFSSL_ABI
int wolfSSL_CTX_use_certificate_file(WOLFSSL_CTX* ctx, const char* file,
                                     int format)
{
    WOLFSSL_ENTER("wolfSSL_CTX_use_certificate_file");

    if (ProcessFile(ctx, file, format, CERT_TYPE, NULL, 0, NULL,
                    GET_VERIFY_SETTING_CTX(ctx)) == WOLFSSL_SUCCESS) {
        return WOLFSSL_SUCCESS;
    }

    return WOLFSSL_FAILURE;
}


WOLFSSL_ABI
int wolfSSL_CTX_use_PrivateKey_file(WOLFSSL_CTX* ctx, const char* file,
                                    int format)
{
    WOLFSSL_ENTER("wolfSSL_CTX_use_PrivateKey_file");

    if (ProcessFile(ctx, file, format, PRIVATEKEY_TYPE, NULL, 0, NULL,
                    GET_VERIFY_SETTING_CTX(ctx)) == WOLFSSL_SUCCESS) {
        return WOLFSSL_SUCCESS;
    }

    return WOLFSSL_FAILURE;
}


#endif /* NO_FILESYSTEM */


/* Sets the max chain depth when verifying a certificate chain. Default depth
 * is set to MAX_CHAIN_DEPTH.
 *
 * ctx   WOLFSSL_CTX structure to set depth in
 * depth max depth
 */
void wolfSSL_CTX_set_verify_depth(WOLFSSL_CTX *ctx, int depth) {
    WOLFSSL_ENTER("wolfSSL_CTX_set_verify_depth");

    if (ctx == NULL || depth < 0 || depth > MAX_CHAIN_DEPTH) {
        WOLFSSL_MSG("Bad depth argument, too large or less than 0");
        return;
    }

    ctx->verifyDepth = (byte)depth;
}


/* get cert chaining depth using ssl struct */
long wolfSSL_get_verify_depth(WOLFSSL* ssl)
{
    if(ssl == NULL) {
        return BAD_FUNC_ARG;
    }
#ifndef OPENSSL_EXTRA
    return MAX_CHAIN_DEPTH;
#else
    return ssl->options.verifyDepth;
#endif
}


/* get cert chaining depth using ctx struct */
long wolfSSL_CTX_get_verify_depth(WOLFSSL_CTX* ctx)
{
    if (ctx == NULL) {
        return BAD_FUNC_ARG;
    }
#ifndef OPENSSL_EXTRA
    return MAX_CHAIN_DEPTH;
#else
    return ctx->verifyDepth;
#endif
}


#ifndef NO_FILESYSTEM


WOLFSSL_ABI
int wolfSSL_CTX_use_certificate_chain_file(WOLFSSL_CTX* ctx, const char* file)
{
    /* process up to MAX_CHAIN_DEPTH plus subject cert */
    WOLFSSL_ENTER("wolfSSL_CTX_use_certificate_chain_file");

    if (ProcessFile(ctx, file, WOLFSSL_FILETYPE_PEM, CERT_TYPE, NULL, 1, NULL,
                    GET_VERIFY_SETTING_CTX(ctx)) == WOLFSSL_SUCCESS) {
        return WOLFSSL_SUCCESS;
    }

   return WOLFSSL_FAILURE;
}


int wolfSSL_CTX_use_certificate_chain_file_format(WOLFSSL_CTX* ctx,
                                                  const char* file, int format)
{
    /* process up to MAX_CHAIN_DEPTH plus subject cert */
    WOLFSSL_ENTER("wolfSSL_CTX_use_certificate_chain_file_format");

    if (ProcessFile(ctx, file, format, CERT_TYPE, NULL, 1, NULL,
                    GET_VERIFY_SETTING_CTX(ctx)) == WOLFSSL_SUCCESS) {
        return WOLFSSL_SUCCESS;
    }

   return WOLFSSL_FAILURE;
}


#ifndef NO_DH

/* server Diffie-Hellman parameters */
static int wolfSSL_SetTmpDH_file_wrapper(WOLFSSL_CTX* ctx, WOLFSSL* ssl,
                                        const char* fname, int format)
{
#ifdef WOLFSSL_SMALL_STACK
    byte   staticBuffer[1]; /* force heap usage */
#else
    byte   staticBuffer[FILE_BUFFER_SIZE];
#endif
    byte*  myBuffer = staticBuffer;
    int    dynamic = 0;
    int    ret;
    long   sz = 0;
    XFILE  file;

    if (ctx == NULL || fname == NULL)
        return BAD_FUNC_ARG;

    file = XFOPEN(fname, "rb");
    if (file == XBADFILE) return WOLFSSL_BAD_FILE;
    if(XFSEEK(file, 0, XSEEK_END) != 0) {
        XFCLOSE(file);
        return WOLFSSL_BAD_FILE;
    }
    sz = XFTELL(file);
    XREWIND(file);

    if (sz > MAX_WOLFSSL_FILE_SIZE || sz <= 0) {
        WOLFSSL_MSG("SetTmpDH file size error");
        XFCLOSE(file);
        return WOLFSSL_BAD_FILE;
    }

    if (sz > (long)sizeof(staticBuffer)) {
        WOLFSSL_MSG("Getting dynamic buffer");
        myBuffer = (byte*) XMALLOC(sz, ctx->heap, DYNAMIC_TYPE_FILE);
        if (myBuffer == NULL) {
            XFCLOSE(file);
            return WOLFSSL_BAD_FILE;
        }
        dynamic = 1;
    }

    if ((size_t)XFREAD(myBuffer, 1, sz, file) != (size_t)sz)
        ret = WOLFSSL_BAD_FILE;
    else {
        if (ssl)
            ret = wolfSSL_SetTmpDH_buffer(ssl, myBuffer, sz, format);
        else
            ret = wolfSSL_CTX_SetTmpDH_buffer(ctx, myBuffer, sz, format);
    }

    XFCLOSE(file);
    if (dynamic)
        XFREE(myBuffer, ctx->heap, DYNAMIC_TYPE_FILE);

    return ret;
}

/* server Diffie-Hellman parameters */
int wolfSSL_SetTmpDH_file(WOLFSSL* ssl, const char* fname, int format)
{
    if (ssl == NULL)
        return BAD_FUNC_ARG;

    return wolfSSL_SetTmpDH_file_wrapper(ssl->ctx, ssl, fname, format);
}


/* server Diffie-Hellman parameters */
int wolfSSL_CTX_SetTmpDH_file(WOLFSSL_CTX* ctx, const char* fname, int format)
{
    return wolfSSL_SetTmpDH_file_wrapper(ctx, NULL, fname, format);
}

#endif /* NO_DH */

#endif /* NO_FILESYSTEM */

#ifndef NO_CHECK_PRIVATE_KEY
/* Check private against public in certificate for match
 *
 * Returns WOLFSSL_SUCCESS on good private key
 *         WOLFSSL_FAILURE if mismatched */
static int check_cert_key(DerBuffer* cert, DerBuffer* key, void* heap,
    int devId, int isKeyLabel, int isKeyId)
{
#ifdef WOLFSSL_SMALL_STACK
    DecodedCert* der = NULL;
#else
    DecodedCert  der[1];
#endif
    word32 size;
    byte*  buff;
    int    ret = WOLFSSL_FAILURE;

    WOLFSSL_ENTER("check_cert_key");

    if (cert == NULL || key == NULL) {
        return WOLFSSL_FAILURE;
    }

#ifdef WOLFSSL_SMALL_STACK
    der = (DecodedCert*)XMALLOC(sizeof(DecodedCert), NULL, DYNAMIC_TYPE_DCERT);
    if (der == NULL)
        return MEMORY_E;
#endif

    size = cert->length;
    buff = cert->buffer;
    InitDecodedCert(der, buff, size, heap);
    if (ParseCertRelative(der, CERT_TYPE, NO_VERIFY, NULL) != 0) {
        FreeDecodedCert(der);
    #ifdef WOLFSSL_SMALL_STACK
        XFREE(der, NULL, DYNAMIC_TYPE_DCERT);
    #endif
        return WOLFSSL_FAILURE;
    }

    size = key->length;
    buff = key->buffer;
#ifdef WOLF_PRIVATE_KEY_ID
    if (devId != INVALID_DEVID) {
        int type = 0;
        void *pkey = NULL;

    #ifndef NO_RSA
        if (der->keyOID == RSAk) {
            type = DYNAMIC_TYPE_RSA;
        }
    #endif
    #ifdef HAVE_ECC
        if (der->keyOID == ECDSAk) {
            type = DYNAMIC_TYPE_ECC;
        }
    #endif

        ret = CreateDevPrivateKey(&pkey, buff, size, type,
                                  isKeyLabel, isKeyId, heap, devId);
        #ifdef WOLF_CRYPTO_CB
        if (ret == 0) {
            #ifndef NO_RSA
            if (der->keyOID == RSAk) {
                ret = wc_CryptoCb_RsaCheckPrivKey((RsaKey*)pkey,
                                               der->publicKey, der->pubKeySize);
            }
            #endif
            #ifdef HAVE_ECC
            if (der->keyOID == ECDSAk) {
                ret = wc_CryptoCb_EccCheckPrivKey((ecc_key*)pkey,
                                               der->publicKey, der->pubKeySize);
            }
            #endif
        }
        #else
            /* devId was set, don't check, for now */
            /* TODO: Add callback for private key check? */
        #endif
        if (pkey != NULL) {
        #ifndef NO_RSA
            if (der->keyOID == RSAk) {
                wc_FreeRsaKey((RsaKey*)pkey);
            }
        #endif
        #ifdef HAVE_ECC
            if (der->keyOID == ECDSAk) {
                wc_ecc_free((ecc_key*)pkey);
            }
        #endif
            XFREE(pkey, heap, type);
        }
        if (ret != CRYPTOCB_UNAVAILABLE) {
            ret = (ret == 0) ? WOLFSSL_SUCCESS: WOLFSSL_FAILURE;
        }
    }
    else {
        /* fall through if unavailable */
        ret = CRYPTOCB_UNAVAILABLE;
    }

    if (ret == CRYPTOCB_UNAVAILABLE)
#endif /* WOLF_PRIVATE_KEY_ID */
    {
        ret = wc_CheckPrivateKeyCert(buff, size, der);
        ret = (ret == 1) ? WOLFSSL_SUCCESS: WOLFSSL_FAILURE;
    }
    FreeDecodedCert(der);
#ifdef WOLFSSL_SMALL_STACK
    XFREE(der, NULL, DYNAMIC_TYPE_DCERT);
#endif

    (void)devId;
    (void)isKeyLabel;
    (void)isKeyId;

    return ret;
}

/* Check private against public in certificate for match
 *
 * ctx  WOLFSSL_CTX structure to check private key in
 *
 * Returns WOLFSSL_SUCCESS on good private key
 *         WOLFSSL_FAILURE if mismatched. */
int wolfSSL_CTX_check_private_key(const WOLFSSL_CTX* ctx)
{
    if (ctx == NULL) {
        return WOLFSSL_FAILURE;
    }
    return check_cert_key(ctx->certificate, ctx->privateKey, ctx->heap,
        ctx->privateKeyDevId, ctx->privateKeyLabel, ctx->privateKeyId);
}
#endif /* !NO_CHECK_PRIVATE_KEY */

#ifdef OPENSSL_ALL
/**
 * Return the private key of the WOLFSSL_CTX struct
 * @return WOLFSSL_EVP_PKEY* The caller doesn *NOT*` free the returned object.
 */
WOLFSSL_EVP_PKEY* wolfSSL_CTX_get0_privatekey(const WOLFSSL_CTX* ctx)
{
    const unsigned char *key;
    int type;

    WOLFSSL_ENTER("wolfSSL_CTX_get0_privatekey");

    if (ctx == NULL || ctx->privateKey == NULL ||
            ctx->privateKey->buffer == NULL) {
        WOLFSSL_MSG("Bad parameter or key not set");
        return NULL;
    }

    switch (ctx->privateKeyType) {
#ifndef NO_RSA
        case rsa_sa_algo:
            type = EVP_PKEY_RSA;
            break;
#endif
#ifdef HAVE_ECC
        case ecc_dsa_sa_algo:
            type = EVP_PKEY_EC;
            break;
#endif
        default:
            /* Other key types not supported either as ssl private keys
             * or in the EVP layer */
            WOLFSSL_MSG("Unsupported key type");
            return NULL;
    }

    key = ctx->privateKey->buffer;

    if (ctx->privateKeyPKey != NULL)
        return ctx->privateKeyPKey;
    else
        return wolfSSL_d2i_PrivateKey(type,
                (WOLFSSL_EVP_PKEY**)&ctx->privateKeyPKey, &key,
                (long)ctx->privateKey->length);
}
#endif

#ifdef OPENSSL_EXTRA

WOLFSSL_PKCS8_PRIV_KEY_INFO* wolfSSL_d2i_PKCS8_PKEY(
    WOLFSSL_PKCS8_PRIV_KEY_INFO** pkey, const unsigned char** keyBuf, long keyLen)
{
    WOLFSSL_PKCS8_PRIV_KEY_INFO* pkcs8 = NULL;
#ifdef WOLFSSL_PEM_TO_DER
    int ret;
    DerBuffer* der = NULL;

    if (keyBuf == NULL || *keyBuf == NULL || keyLen <= 0) {
        WOLFSSL_MSG("Bad key PEM/DER args");
        return NULL;
    }

    ret = PemToDer(*keyBuf, keyLen, PRIVATEKEY_TYPE, &der, NULL, NULL, NULL);
    if (ret < 0) {
        WOLFSSL_MSG("Not PEM format");
        ret = AllocDer(&der, (word32)keyLen, PRIVATEKEY_TYPE, NULL);
        if (ret == 0) {
            XMEMCPY(der->buffer, *keyBuf, keyLen);
        }
    }

    if (ret == 0) {
        /* Verify this is PKCS8 Key */
        word32 inOutIdx = 0;
        word32 algId;
        ret = ToTraditionalInline_ex(der->buffer, &inOutIdx, der->length, &algId);
        if (ret >= 0) {
            ret = 0; /* good DER */
        }
    }

    if (ret == 0) {
        pkcs8 = wolfSSL_EVP_PKEY_new();
        if (pkcs8 == NULL)
            ret = MEMORY_E;
    }
    if (ret == 0) {
        pkcs8->pkey.ptr = (char*)XMALLOC(der->length, NULL,
            DYNAMIC_TYPE_PUBLIC_KEY);
        if (pkcs8->pkey.ptr == NULL)
            ret = MEMORY_E;
    }
    if (ret == 0) {
        XMEMCPY(pkcs8->pkey.ptr, der->buffer, der->length);
        pkcs8->pkey_sz = der->length;
    }

    FreeDer(&der);
    if (ret != 0) {
        wolfSSL_EVP_PKEY_free(pkcs8);
        pkcs8 = NULL;
    }
    if (pkey != NULL) {
        *pkey = pkcs8;
    }

#else
    (void)bio;
    (void)pkey;
#endif /* WOLFSSL_PEM_TO_DER */

    return pkcs8;
}


#ifndef NO_BIO
/* put SSL type in extra for now, not very common */

/* Converts a DER format key read from "bio" to a PKCS8 structure.
 *
 * bio  input bio to read DER from
 * pkey If not NULL then this pointer will be overwritten with a new PKCS8
 *      structure.
 *
 * returns a WOLFSSL_PKCS8_PRIV_KEY_INFO pointer on success and NULL in fail
 *         case.
 */
WOLFSSL_PKCS8_PRIV_KEY_INFO* wolfSSL_d2i_PKCS8_PKEY_bio(WOLFSSL_BIO* bio,
        WOLFSSL_PKCS8_PRIV_KEY_INFO** pkey)
{
    WOLFSSL_PKCS8_PRIV_KEY_INFO* pkcs8 = NULL;
#ifdef WOLFSSL_PEM_TO_DER
    unsigned char* mem = NULL;
    int memSz;

    WOLFSSL_ENTER("wolfSSL_d2i_PKCS8_PKEY_bio");

    if (bio == NULL) {
        return NULL;
    }

    if ((memSz = wolfSSL_BIO_get_mem_data(bio, &mem)) < 0) {
        return NULL;
    }

    pkcs8 = wolfSSL_d2i_PKCS8_PKEY(pkey, (const unsigned char**)&mem, memSz);
#else
    (void)bio;
    (void)pkey;
#endif /* WOLFSSL_PEM_TO_DER */

    return pkcs8;
}


/* expecting DER format public key
 *
 * bio  input bio to read DER from
 * out  If not NULL then this pointer will be overwritten with a new
 * WOLFSSL_EVP_PKEY pointer
 *
 * returns a WOLFSSL_EVP_PKEY pointer on success and NULL in fail case.
 */
WOLFSSL_EVP_PKEY* wolfSSL_d2i_PUBKEY_bio(WOLFSSL_BIO* bio,
                                         WOLFSSL_EVP_PKEY** out)
{
    unsigned char* mem;
    long memSz;
    WOLFSSL_EVP_PKEY* pkey = NULL;

    WOLFSSL_ENTER("wolfSSL_d2i_PUBKEY_bio()");

    if (bio == NULL) {
        return NULL;
    }
    (void)out;

    memSz = wolfSSL_BIO_get_len(bio);
    if (memSz <= 0) {
        return NULL;
    }

    mem = (unsigned char*)XMALLOC(memSz, bio->heap, DYNAMIC_TYPE_TMP_BUFFER);
    if (mem == NULL) {
        return NULL;
    }

    if (wolfSSL_BIO_read(bio, mem, (int)memSz) == memSz) {
        pkey = wolfSSL_d2i_PUBKEY(NULL, (const unsigned char**)&mem, memSz);
        if (out != NULL && pkey != NULL) {
            *out = pkey;
        }
    }

    XFREE(mem, bio->heap, DYNAMIC_TYPE_TMP_BUFFER);
    return pkey;
}

#endif /* !NO_BIO */

static WOLFSSL_EVP_PKEY* d2iGenericKey(WOLFSSL_EVP_PKEY** out,
                                 const unsigned char** in, long inSz, int priv)
{

    WOLFSSL_EVP_PKEY* pkey = NULL;
    const unsigned char* mem;
    long memSz = inSz;

    WOLFSSL_ENTER("d2iGenericKey");

    if (in == NULL || *in == NULL || inSz < 0) {
        WOLFSSL_MSG("Bad argument");
        return NULL;
    }
    mem = *in;

    #if !defined(NO_RSA)
    {
        word32 keyIdx = 0;
        int isRsaKey;
    #ifdef WOLFSSL_SMALL_STACK
        RsaKey *rsa = (RsaKey*)XMALLOC(sizeof(RsaKey), NULL, DYNAMIC_TYPE_RSA);
        if (rsa == NULL)
            return NULL;
    #else
        RsaKey rsa[1];
    #endif
        XMEMSET(rsa, 0, sizeof(RsaKey));

        /* test if RSA key */
        if (priv)
            isRsaKey = wc_InitRsaKey(rsa, NULL) == 0 &&
                wc_RsaPrivateKeyDecode(mem, &keyIdx, rsa, (word32)memSz) == 0;
        else
            isRsaKey = wc_InitRsaKey(rsa, NULL) == 0 &&
                wc_RsaPublicKeyDecode(mem, &keyIdx, rsa, (word32)memSz) == 0;
        wc_FreeRsaKey(rsa);
    #ifdef WOLFSSL_SMALL_STACK
        XFREE(rsa, NULL, DYNAMIC_TYPE_RSA);
    #endif

        if (isRsaKey) {
            pkey = wolfSSL_EVP_PKEY_new();
            if (pkey != NULL) {
                pkey->pkey_sz = keyIdx;
                pkey->pkey.ptr = (char*)XMALLOC(memSz, NULL,
                        priv ? DYNAMIC_TYPE_PRIVATE_KEY :
                               DYNAMIC_TYPE_PUBLIC_KEY);
                if (pkey->pkey.ptr == NULL) {
                    wolfSSL_EVP_PKEY_free(pkey);
                    return NULL;
                }
                XMEMCPY(pkey->pkey.ptr, mem, keyIdx);
                pkey->type = EVP_PKEY_RSA;
                if (out != NULL) {
                    *out = pkey;
                }

                pkey->ownRsa = 1;
                pkey->rsa = wolfSSL_RSA_new();
                if (pkey->rsa == NULL) {
                    wolfSSL_EVP_PKEY_free(pkey);
                    return NULL;
                }

                if (wolfSSL_RSA_LoadDer_ex(pkey->rsa,
                        (const unsigned char*)pkey->pkey.ptr,
                        pkey->pkey_sz, priv ? WOLFSSL_RSA_LOAD_PRIVATE
                                            : WOLFSSL_RSA_LOAD_PUBLIC) != 1) {
                    wolfSSL_EVP_PKEY_free(pkey);
                    return NULL;
                }

                return pkey;
            }
            else {
                WOLFSSL_MSG("RSA wolfSSL_EVP_PKEY_new error");
            }
        }
    }
    #endif /* NO_RSA */

    #ifdef HAVE_ECC
    {
        word32  keyIdx = 0;
        int     isEccKey;
    #ifdef WOLFSSL_SMALL_STACK
        ecc_key *ecc = (ecc_key*)XMALLOC(sizeof(ecc_key), NULL, DYNAMIC_TYPE_ECC);
        if (ecc == NULL)
            return NULL;
    #else
        ecc_key ecc[1];
    #endif
        XMEMSET(ecc, 0, sizeof(ecc_key));

        if (priv)
            isEccKey = wc_ecc_init(ecc) == 0 &&
                wc_EccPrivateKeyDecode(mem, &keyIdx, ecc, (word32)memSz) == 0;
        else
            isEccKey = wc_ecc_init(ecc) == 0 &&
                wc_EccPublicKeyDecode(mem, &keyIdx, ecc, (word32)memSz) == 0;
        wc_ecc_free(ecc);
    #ifdef WOLFSSL_SMALL_STACK
        XFREE(ecc, NULL, DYNAMIC_TYPE_ECC);
    #endif

        if (isEccKey) {
            pkey = wolfSSL_EVP_PKEY_new();
            if (pkey != NULL) {
                pkey->pkey_sz = keyIdx;
                pkey->pkey.ptr = (char*)XMALLOC(keyIdx, NULL,
                        priv ? DYNAMIC_TYPE_PRIVATE_KEY :
                               DYNAMIC_TYPE_PUBLIC_KEY);
                if (pkey->pkey.ptr == NULL) {
                    wolfSSL_EVP_PKEY_free(pkey);
                    return NULL;
                }
                XMEMCPY(pkey->pkey.ptr, mem, keyIdx);
                pkey->type = EVP_PKEY_EC;
                if (out != NULL) {
                    *out = pkey;
                }

                pkey->ownEcc = 1;
                pkey->ecc = wolfSSL_EC_KEY_new();
                if (pkey->ecc == NULL) {
                    wolfSSL_EVP_PKEY_free(pkey);
                    return NULL;
                }

                if (wolfSSL_EC_KEY_LoadDer_ex(pkey->ecc,
                        (const unsigned char*)pkey->pkey.ptr,
                        pkey->pkey_sz, priv ? WOLFSSL_RSA_LOAD_PRIVATE
                                            : WOLFSSL_RSA_LOAD_PUBLIC) != 1) {
                    wolfSSL_EVP_PKEY_free(pkey);
                    return NULL;
                }

                return pkey;
            }
            else {
                WOLFSSL_MSG("ECC wolfSSL_EVP_PKEY_new error");
            }
        }
    }
    #endif /* HAVE_ECC */

    #if !defined(NO_DSA)
    {
        word32 keyIdx = 0;
        int     isDsaKey;
    #ifdef WOLFSSL_SMALL_STACK
        DsaKey *dsa = (DsaKey*)XMALLOC(sizeof(DsaKey), NULL, DYNAMIC_TYPE_DSA);
        if (dsa == NULL)
            return NULL;
    #else
        DsaKey dsa[1];
    #endif
        XMEMSET(dsa, 0, sizeof(DsaKey));

        if (priv)
            isDsaKey = wc_InitDsaKey(dsa) == 0 &&
                wc_DsaPrivateKeyDecode(mem, &keyIdx, dsa, (word32)memSz) == 0;
        else
            isDsaKey = wc_InitDsaKey(dsa) == 0 &&
                wc_DsaPublicKeyDecode(mem, &keyIdx, dsa, (word32)memSz) == 0;
        wc_FreeDsaKey(dsa);
    #ifdef WOLFSSL_SMALL_STACK
        XFREE(dsa, NULL, DYNAMIC_TYPE_DSA);
    #endif

        /* test if DSA key */
        if (isDsaKey) {
            pkey = wolfSSL_EVP_PKEY_new();

            if (pkey != NULL) {
                pkey->pkey_sz = keyIdx;
                pkey->pkey.ptr = (char*)XMALLOC(memSz, NULL,
                        priv ? DYNAMIC_TYPE_PRIVATE_KEY :
                               DYNAMIC_TYPE_PUBLIC_KEY);
                if (pkey->pkey.ptr == NULL) {
                    wolfSSL_EVP_PKEY_free(pkey);
                    return NULL;
                }
                XMEMCPY(pkey->pkey.ptr, mem, keyIdx);
                pkey->type = EVP_PKEY_DSA;
                if (out != NULL) {
                    *out = pkey;
                }

                pkey->ownDsa = 1;
                pkey->dsa = wolfSSL_DSA_new();
                if (pkey->dsa == NULL) {
                    wolfSSL_EVP_PKEY_free(pkey);
                    return NULL;
                }

                if (wolfSSL_DSA_LoadDer_ex(pkey->dsa,
                        (const unsigned char*)pkey->pkey.ptr,
                        pkey->pkey_sz, priv ? WOLFSSL_RSA_LOAD_PRIVATE
                                            : WOLFSSL_RSA_LOAD_PUBLIC) != 1) {
                    wolfSSL_EVP_PKEY_free(pkey);
                    return NULL;
                }

                return pkey;
            }
            else {
                WOLFSSL_MSG("DSA wolfSSL_EVP_PKEY_new error");
            }
        }
    }
    #endif /* NO_DSA */

    #if !defined(NO_DH) && (defined(WOLFSSL_QT) || defined(OPENSSL_ALL))
    #if !defined(HAVE_FIPS) || (defined(HAVE_FIPS_VERSION) && \
        (HAVE_FIPS_VERSION > 2))
    {
        int isDhKey;
        word32 keyIdx = 0;
    #ifdef WOLFSSL_SMALL_STACK
        DhKey *dh = (DhKey*)XMALLOC(sizeof(DhKey), NULL, DYNAMIC_TYPE_DH);
        if (dh == NULL)
            return NULL;
    #else
        DhKey dh[1];
    #endif
        XMEMSET(dh, 0, sizeof(DhKey));

        isDhKey = wc_InitDhKey(dh) == 0 &&
                  wc_DhKeyDecode(mem, &keyIdx, dh, (word32)memSz) == 0;
        wc_FreeDhKey(dh);
    #ifdef WOLFSSL_SMALL_STACK
        XFREE(dh, NULL, DYNAMIC_TYPE_DH);
    #endif

        /* test if DH key */
        if (isDhKey) {
            pkey = wolfSSL_EVP_PKEY_new();

            if (pkey != NULL) {
                pkey->pkey_sz = (int)memSz;
                pkey->pkey.ptr = (char*)XMALLOC(memSz, NULL,
                        priv ? DYNAMIC_TYPE_PRIVATE_KEY :
                               DYNAMIC_TYPE_PUBLIC_KEY);
                if (pkey->pkey.ptr == NULL) {
                    wolfSSL_EVP_PKEY_free(pkey);
                    return NULL;
                }
                XMEMCPY(pkey->pkey.ptr, mem, memSz);
                pkey->type = EVP_PKEY_DH;
                if (out != NULL) {
                    *out = pkey;
                }

                pkey->ownDh = 1;
                pkey->dh = wolfSSL_DH_new();
                if (pkey->dh == NULL) {
                    wolfSSL_EVP_PKEY_free(pkey);
                    return NULL;
                }

                if (wolfSSL_DH_LoadDer(pkey->dh,
                            (const unsigned char*)pkey->pkey.ptr,
                            pkey->pkey_sz) != WOLFSSL_SUCCESS) {
                    wolfSSL_EVP_PKEY_free(pkey);
                    return NULL;
                }

                return pkey;
            }
            else {
                WOLFSSL_MSG("DH wolfSSL_EVP_PKEY_new error");
            }
        }
    }
    #endif /* !HAVE_FIPS || HAVE_FIPS_VERSION > 2 */
    #endif /* !NO_DH && (WOLFSSL_QT || OPENSSL_ALL) */

    #if !defined(NO_DH) && defined(OPENSSL_EXTRA) && defined(WOLFSSL_DH_EXTRA)
    #if !defined(HAVE_FIPS) || (defined(HAVE_FIPS_VERSION) && \
            (HAVE_FIPS_VERSION > 2))
    {
        word32  keyIdx = 0;
        DhKey*  key = NULL;
        int ret;
        int elements;
    #ifdef WOLFSSL_SMALL_STACK
        DhKey* dh = (DhKey*)XMALLOC(sizeof(DhKey), NULL, DYNAMIC_TYPE_DH);
        if (dh == NULL)
            return NULL;
    #else
        DhKey  dh[1];
    #endif
        XMEMSET(dh, 0, sizeof(DhKey));

        /* test if DH-public key */
        if (wc_InitDhKey(dh) != 0)
            return NULL;

        ret = wc_DhKeyDecode(mem, &keyIdx, dh, (word32)memSz);
        wc_FreeDhKey(dh);
    #ifdef WOLFSSL_SMALL_STACK
        XFREE(dh, NULL, DYNAMIC_TYPE_DH);
    #endif

        if (ret == 0) {
            pkey = wolfSSL_EVP_PKEY_new();
            if (pkey != NULL) {
                pkey->type     = EVP_PKEY_DH;
                pkey->pkey_sz  = (int)memSz;
                pkey->pkey.ptr = (char*)XMALLOC(memSz, NULL,
                        priv ? DYNAMIC_TYPE_PRIVATE_KEY :
                               DYNAMIC_TYPE_PUBLIC_KEY);
                if (pkey->pkey.ptr == NULL) {
                    wolfSSL_EVP_PKEY_free(pkey);
                    return NULL;
                }
                XMEMCPY(pkey->pkey.ptr, mem, memSz);
                if (out != NULL) {
                    *out = pkey;
                }
                pkey->ownDh = 1;
                pkey->dh = wolfSSL_DH_new();
                if (pkey->dh == NULL) {
                    wolfSSL_EVP_PKEY_free(pkey);
                    return NULL;
                }

                key = (DhKey*)pkey->dh->internal;

                keyIdx = 0;
                if (wc_DhKeyDecode(mem, &keyIdx, key, (word32)memSz) == 0)
                {
                    elements = ELEMENT_P | ELEMENT_G | ELEMENT_Q | ELEMENT_PUB;
                    if (priv)
                        elements |= ELEMENT_PRV;
                    if(SetDhExternal_ex(pkey->dh, elements)
                            == WOLFSSL_SUCCESS ) {
                        return pkey;
                    }
                }
                else {
                    wolfSSL_EVP_PKEY_free(pkey);
                    return NULL;
                }
            }
        }
    }
    #endif /* !HAVE_FIPS || HAVE_FIPS_VERSION > 2 */
    #endif /* !NO_DH &&  OPENSSL_EXTRA && WOLFSSL_DH_EXTRA */

    #ifdef HAVE_PQC
    {
        int isFalcon = 0;
    #ifdef WOLFSSL_SMALL_STACK
        falcon_key *falcon = (falcon_key *)MALLOC(sizeof(falcon_key), NULL,
                                                  DYNAMIC_TYPE_FALCON);
        if (falcon == NULL) {
            return NULL;
        }
    #else
        falcon_key falcon[1];
    #endif

        if (wc_falcon_init(falcon) == 0) {
            /* test if Falcon key */
            if (priv) {
                /* Try level 1 */
                isFalcon = wc_falcon_set_level(falcon, 1) == 0 &&
                           wc_falcon_import_private_only(mem, (word32)memSz,
                                                         falcon) == 0;
                if (!isFalcon) {
                    /* Try level 5 */
                    isFalcon = wc_falcon_set_level(falcon, 5) == 0 &&
                               wc_falcon_import_private_only(mem, (word32)memSz,
                                                             falcon) == 0;
                }
            } else {
                /* Try level 1 */
                isFalcon = wc_falcon_set_level(falcon, 1) == 0 &&
                           wc_falcon_import_public(mem, (word32)memSz, falcon)
                           == 0;

                if (!isFalcon) {
                    /* Try level 5 */
                    isFalcon = wc_falcon_set_level(falcon, 5) == 0 &&
                               wc_falcon_import_public(mem, (word32)memSz,
                                                       falcon) == 0;
                }
            }
            wc_falcon_free(falcon);
        }

    #ifdef WOLFSSL_SMALL_STACK
        XFREE(falcon, NULL, DYNAMIC_TYPE_FALCON);
    #endif
        if (isFalcon) {
            /* Create a fake Falcon EVP_PKEY. In the future, we might integrate
             * Falcon into the compatibility layer. */
            pkey = wolfSSL_EVP_PKEY_new();
            if (pkey == NULL) {
                WOLFSSL_MSG("Falcon wolfSSL_EVP_PKEY_new error");
                return NULL;
            }
            pkey->type = EVP_PKEY_FALCON;
            pkey->pkey.ptr = NULL;
            pkey->pkey_sz = 0;
            return pkey;
        }

    }
    #endif /* HAVE_PQC */

    if (pkey == NULL) {
        WOLFSSL_MSG("wolfSSL_d2i_PUBKEY couldn't determine key type");
    }

    return pkey;

}


/* Converts a DER encoded public key to a WOLFSSL_EVP_PKEY structure.
 *
 * out  pointer to new WOLFSSL_EVP_PKEY structure. Can be NULL
 * in   DER buffer to convert
 * inSz size of in buffer
 *
 * returns a pointer to a new WOLFSSL_EVP_PKEY structure on success and NULL
 *         on fail
 */
WOLFSSL_EVP_PKEY* wolfSSL_d2i_PUBKEY(WOLFSSL_EVP_PKEY** out,
                                     const unsigned char** in, long inSz)
{
    WOLFSSL_ENTER("wolfSSL_d2i_PUBKEY");
    return d2iGenericKey(out, in, inSz, 0);
}

/* helper function to get raw pointer to DER buffer from WOLFSSL_EVP_PKEY */
static int wolfSSL_EVP_PKEY_get_der(const WOLFSSL_EVP_PKEY* key, unsigned char** der)
{
    unsigned char* pt;
    int sz;
    word16 pkcs8HeaderSz;

    if (!key || !key->pkey_sz)
        return WOLFSSL_FATAL_ERROR;

    /* return the key without PKCS8 for compatibility */
    /* if pkcs8HeaderSz is invalid, use 0 and return all of pkey */
    pkcs8HeaderSz = 0;
    if (key->pkey_sz > key->pkcs8HeaderSz)
        pkcs8HeaderSz = key->pkcs8HeaderSz;
    sz = key->pkey_sz - pkcs8HeaderSz;
    if (der) {
        pt = (unsigned char*)key->pkey.ptr;
        if (*der) {
            /* since this function signature has no size value passed in it is
             * assumed that the user has allocated a large enough buffer */
            XMEMCPY(*der, pt + pkcs8HeaderSz, sz);
            *der += sz;
        }
        else {
            *der = (unsigned char*)XMALLOC(sz, NULL, DYNAMIC_TYPE_OPENSSL);
            if (*der == NULL) {
                return WOLFSSL_FATAL_ERROR;
            }
            XMEMCPY(*der, pt + pkcs8HeaderSz, sz);
        }
    }
    return sz;
}

int wolfSSL_i2d_PUBKEY(const WOLFSSL_EVP_PKEY *key, unsigned char **der)
{
    return wolfSSL_EVP_PKEY_get_der(key, der);
}

static WOLFSSL_EVP_PKEY* _d2i_PublicKey(int type, WOLFSSL_EVP_PKEY** out,
    const unsigned char **in, long inSz, int priv)
{
    int ret = 0;
    word32 idx = 0, algId;
    word16 pkcs8HeaderSz = 0;
    WOLFSSL_EVP_PKEY* local;
    int opt;

    (void)opt;

    if (in == NULL || inSz < 0) {
        WOLFSSL_MSG("Bad argument");
        return NULL;
    }

    if (priv == 1) {
        /* Check if input buffer has PKCS8 header. In the case that it does not
         * have a PKCS8 header then do not error out. */
        if ((ret = ToTraditionalInline_ex((const byte*)(*in), &idx,
                                          (word32)inSz, &algId)) > 0) {
            WOLFSSL_MSG("Found PKCS8 header");
            pkcs8HeaderSz = (word16)idx;

            if ((type == EVP_PKEY_RSA && algId != RSAk) ||
                (type == EVP_PKEY_EC && algId != ECDSAk) ||
                (type == EVP_PKEY_DSA && algId != DSAk) ||
                (type == EVP_PKEY_DH && algId != DHk)) {
                WOLFSSL_MSG("PKCS8 does not match EVP key type");
                return NULL;
            }

            (void)idx; /* not used */
        }
        else {
            if (ret != ASN_PARSE_E) {
                WOLFSSL_MSG("Unexpected error with trying to remove PKCS8 "
                    "header");
                return NULL;
            }
        }
    }

    if (out != NULL && *out != NULL) {
        wolfSSL_EVP_PKEY_free(*out);
        *out = NULL;
    }
    local = wolfSSL_EVP_PKEY_new();
    if (local == NULL) {
        return NULL;
    }

    local->type     = type;
    local->pkey_sz  = (int)inSz;
    local->pkcs8HeaderSz = pkcs8HeaderSz;
    local->pkey.ptr = (char*)XMALLOC(inSz, NULL, DYNAMIC_TYPE_PUBLIC_KEY);
    if (local->pkey.ptr == NULL) {
        wolfSSL_EVP_PKEY_free(local);
        local = NULL;
        return NULL;
    }
    else {
        XMEMCPY(local->pkey.ptr, *in, inSz);
    }

    switch (type) {
#ifndef NO_RSA
        case EVP_PKEY_RSA:
            local->ownRsa = 1;
            local->rsa = wolfSSL_RSA_new();
            if (local->rsa == NULL) {
                wolfSSL_EVP_PKEY_free(local);
                return NULL;
            }
            opt = priv ? WOLFSSL_RSA_LOAD_PRIVATE : WOLFSSL_RSA_LOAD_PUBLIC;
            if (wolfSSL_RSA_LoadDer_ex(local->rsa,
                      (const unsigned char*)local->pkey.ptr, local->pkey_sz,
                      opt) != WOLFSSL_SUCCESS) {
                wolfSSL_EVP_PKEY_free(local);
                return NULL;
            }
            break;
#endif /* NO_RSA */
#ifdef HAVE_ECC
        case EVP_PKEY_EC:
            local->ownEcc = 1;
            local->ecc = wolfSSL_EC_KEY_new();
            if (local->ecc == NULL) {
                wolfSSL_EVP_PKEY_free(local);
                return NULL;
            }
            opt = priv ? WOLFSSL_EC_KEY_LOAD_PRIVATE :
                         WOLFSSL_EC_KEY_LOAD_PUBLIC;
            if (wolfSSL_EC_KEY_LoadDer_ex(local->ecc,
                      (const unsigned char*)local->pkey.ptr, local->pkey_sz,
                      opt)
                      != WOLFSSL_SUCCESS) {
                wolfSSL_EVP_PKEY_free(local);
                return NULL;
            }
            break;
#endif /* HAVE_ECC */
#if defined(WOLFSSL_QT) || defined(OPENSSL_ALL) || defined(WOLFSSL_OPENSSH)
#ifndef NO_DSA
        case EVP_PKEY_DSA:
            local->ownDsa = 1;
            local->dsa = wolfSSL_DSA_new();
            if (local->dsa == NULL) {
                wolfSSL_EVP_PKEY_free(local);
                return NULL;
            }
            opt = priv ? WOLFSSL_DSA_LOAD_PRIVATE : WOLFSSL_DSA_LOAD_PUBLIC;
            if (wolfSSL_DSA_LoadDer_ex(local->dsa,
                    (const unsigned char*)local->pkey.ptr, local->pkey_sz,
                    opt)
                    != WOLFSSL_SUCCESS) {
                wolfSSL_EVP_PKEY_free(local);
                return NULL;
            }
            break;
#endif /* NO_DSA */
#ifndef NO_DH
#if !defined(HAVE_FIPS) || (defined(HAVE_FIPS_VERSION) && (HAVE_FIPS_VERSION>2))
        case EVP_PKEY_DH:
            local->ownDh = 1;
            local->dh = wolfSSL_DH_new();
            if (local->dh == NULL) {
                wolfSSL_EVP_PKEY_free(local);
                return NULL;
            }
            if (wolfSSL_DH_LoadDer(local->dh,
                      (const unsigned char*)local->pkey.ptr, local->pkey_sz)
                      != WOLFSSL_SUCCESS) {
                wolfSSL_EVP_PKEY_free(local);
                return NULL;
            }
            break;
#endif /* !HAVE_FIPS || HAVE_FIPS_VERSION > 2 */
#endif /* HAVE_DH */
#endif /* WOLFSSL_QT || OPENSSL_ALL || WOLFSSL_OPENSSH */
        default:
            WOLFSSL_MSG("Unsupported key type");
            wolfSSL_EVP_PKEY_free(local);
            return NULL;
    }

    /* advance pointer with success */
    if (local != NULL) {
        if (local->pkey_sz <= (int)inSz) {
            *in += local->pkey_sz;
        }

        if (out != NULL) {
            *out = local;
        }
    }

    return local;
}

WOLFSSL_EVP_PKEY* wolfSSL_d2i_PublicKey(int type, WOLFSSL_EVP_PKEY** out,
        const unsigned char **in, long inSz)
{
    WOLFSSL_ENTER("wolfSSL_d2i_PublicKey");

    return _d2i_PublicKey(type, out, in, inSz, 0);
}
/* Reads in a DER format key. If PKCS8 headers are found they are stripped off.
 *
 * type  type of key
 * out   newly created WOLFSSL_EVP_PKEY structure
 * in    pointer to input key DER
 * inSz  size of in buffer
 *
 * On success a non null pointer is returned and the pointer in is advanced the
 * same number of bytes read.
 */
WOLFSSL_EVP_PKEY* wolfSSL_d2i_PrivateKey(int type, WOLFSSL_EVP_PKEY** out,
        const unsigned char **in, long inSz)
{
    WOLFSSL_ENTER("wolfSSL_d2i_PrivateKey");

    return _d2i_PublicKey(type, out, in, inSz, 1);
}

#ifdef WOLF_PRIVATE_KEY_ID
/* Create an EVP structure for use with crypto callbacks */
WOLFSSL_EVP_PKEY* wolfSSL_d2i_PrivateKey_id(int type, WOLFSSL_EVP_PKEY** out,
    void* heap, int devId)
{
    WOLFSSL_EVP_PKEY* local;

    if (out != NULL && *out != NULL) {
        wolfSSL_EVP_PKEY_free(*out);
        *out = NULL;
    }

    local = wolfSSL_EVP_PKEY_new_ex(heap);
    if (local == NULL) {
        return NULL;
    }

    local->type     = type;
    local->pkey_sz  = 0;
    local->pkcs8HeaderSz = 0;

    switch (type) {
#ifndef NO_RSA
        case EVP_PKEY_RSA:
        {
            RsaKey* key;
            local->ownRsa = 1;
            local->rsa = wolfSSL_RSA_new_ex(heap, devId);
            if (local->rsa == NULL) {
                wolfSSL_EVP_PKEY_free(local);
                return NULL;
            }
            key = (RsaKey*)local->rsa->internal;
        #ifdef WOLF_CRYPTO_CB
            key->devId = devId;
        #endif
            (void)key;
            local->rsa->inSet = 1;
            break;
        }
#endif /* !NO_RSA */
#ifdef HAVE_ECC
        case EVP_PKEY_EC:
        {
            ecc_key* key;
            local->ownEcc = 1;
            local->ecc = wolfSSL_EC_KEY_new_ex(heap, devId);
            if (local->ecc == NULL) {
                wolfSSL_EVP_PKEY_free(local);
                return NULL;
            }
            key = (ecc_key*)local->ecc->internal;
        #ifdef WOLF_CRYPTO_CB
            key->devId = devId;
        #endif
            key->type = ECC_PRIVATEKEY;
            /* key is required to have a key size / curve set, although
             * actual one used is determined by devId callback function */
            wc_ecc_set_curve(key, ECDHE_SIZE, ECC_CURVE_DEF);

            local->ecc->inSet = 1;
            break;
        }
#endif /* HAVE_ECC */
        default:
            WOLFSSL_MSG("Unsupported private key id type");
            wolfSSL_EVP_PKEY_free(local);
            return NULL;
    }

    if (local != NULL && out != NULL) {
        *out = local;
    }

    return local;
}
#endif /* WOLF_PRIVATE_KEY_ID */

#ifndef NO_CERTS // NOLINT(readability-redundant-preprocessor)

#ifndef NO_CHECK_PRIVATE_KEY
/* Check private against public in certificate for match
 *
 * ssl  WOLFSSL structure to check private key in
 *
 * Returns WOLFSSL_SUCCESS on good private key
 *         WOLFSSL_FAILURE if mismatched. */
int wolfSSL_check_private_key(const WOLFSSL* ssl)
{
    if (ssl == NULL) {
        return WOLFSSL_FAILURE;
    }
    return check_cert_key(ssl->buffers.certificate, ssl->buffers.key, ssl->heap,
        ssl->buffers.keyDevId, ssl->buffers.keyLabel, ssl->buffers.keyId);
}
#endif /* !NO_CHECK_PRIVATE_KEY */

#if defined(OPENSSL_ALL)

int wolfSSL_ASN1_BIT_STRING_set_bit(WOLFSSL_ASN1_BIT_STRING* str, int pos,
    int val)
{
    int bytes_cnt, bit;
    byte* temp;

    if (!str || (val != 0 && val != 1) || pos < 0) {
        return WOLFSSL_FAILURE;
    }

    bytes_cnt = pos/8;
    bit = 1<<(7-(pos%8));

    if (bytes_cnt+1 > str->length) {
        if (!(temp = (byte*)XREALLOC(str->data, bytes_cnt+1, NULL,
                DYNAMIC_TYPE_OPENSSL))) {
            return WOLFSSL_FAILURE;
        }
        XMEMSET(temp+str->length, 0, bytes_cnt+1 - str->length);
        str->data = temp;
        str->length = bytes_cnt+1;
    }

    str->data[bytes_cnt] &= ~bit;
    str->data[bytes_cnt] |= val ? bit : 0;

    return WOLFSSL_SUCCESS;
}

#endif /* OPENSSL_ALL */

#endif /* !NO_CERTS */
#endif /* OPENSSL_EXTRA */

#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_WPAS_SMALL)
WOLFSSL_ASN1_BIT_STRING* wolfSSL_ASN1_BIT_STRING_new(void)
{
    WOLFSSL_ASN1_BIT_STRING* str;

    str = (WOLFSSL_ASN1_BIT_STRING*)XMALLOC(sizeof(WOLFSSL_ASN1_BIT_STRING),
                                                  NULL, DYNAMIC_TYPE_OPENSSL);
    if (str) {
        XMEMSET(str, 0, sizeof(WOLFSSL_ASN1_BIT_STRING));
    }
    return str;
}

void wolfSSL_ASN1_BIT_STRING_free(WOLFSSL_ASN1_BIT_STRING* str)
{
    if (str) {
        if (str->data) {
            XFREE(str->data, NULL, DYNAMIC_TYPE_OPENSSL);
            str->data = NULL;
        }
        XFREE(str, NULL, DYNAMIC_TYPE_OPENSSL);
    }
}

int wolfSSL_ASN1_BIT_STRING_get_bit(const WOLFSSL_ASN1_BIT_STRING* str, int i)
{
    if (!str || !str->data || str->length <= (i/8) || i < 0) {
        return WOLFSSL_FAILURE;
    }

    return (str->data[i/8] & (1<<(7-(i%8)))) ? 1 : 0;
}
#endif /* OPENSSL_EXTRA || WOLFSSL_WPAS_SMALL */

#ifdef OPENSSL_EXTRA

int wolfSSL_use_PrivateKey(WOLFSSL* ssl, WOLFSSL_EVP_PKEY* pkey)
{
    WOLFSSL_ENTER("wolfSSL_use_PrivateKey");
    if (ssl == NULL || pkey == NULL ) {
        return WOLFSSL_FAILURE;
    }

    return wolfSSL_use_PrivateKey_buffer(ssl, (unsigned char*)pkey->pkey.ptr,
                                         pkey->pkey_sz, WOLFSSL_FILETYPE_ASN1);
}


int wolfSSL_use_PrivateKey_ASN1(int pri, WOLFSSL* ssl, const unsigned char* der,
                                long derSz)
{
    WOLFSSL_ENTER("wolfSSL_use_PrivateKey_ASN1");
    if (ssl == NULL || der == NULL ) {
        return WOLFSSL_FAILURE;
    }

    (void)pri; /* type of private key */
    return wolfSSL_use_PrivateKey_buffer(ssl, der, derSz, WOLFSSL_FILETYPE_ASN1);
}
/******************************************************************************
* wolfSSL_CTX_use_PrivateKey_ASN1 - loads a private key buffer into the SSL ctx
*
* RETURNS:
* returns WOLFSSL_SUCCESS on success, otherwise returns WOLFSSL_FAILURE
*/

int wolfSSL_CTX_use_PrivateKey_ASN1(int pri, WOLFSSL_CTX* ctx,
                                            unsigned char* der, long derSz)
{
    WOLFSSL_ENTER("wolfSSL_CTX_use_PrivateKey_ASN1");
    if (ctx == NULL || der == NULL ) {
        return WOLFSSL_FAILURE;
    }

    (void)pri; /* type of private key */
    return wolfSSL_CTX_use_PrivateKey_buffer(ctx, der, derSz, WOLFSSL_FILETYPE_ASN1);
}


#ifndef NO_RSA
int wolfSSL_use_RSAPrivateKey_ASN1(WOLFSSL* ssl, unsigned char* der, long derSz)
{
    WOLFSSL_ENTER("wolfSSL_use_RSAPrivateKey_ASN1");
    if (ssl == NULL || der == NULL ) {
        return WOLFSSL_FAILURE;
    }

    return wolfSSL_use_PrivateKey_buffer(ssl, der, derSz, WOLFSSL_FILETYPE_ASN1);
}
#endif

int wolfSSL_use_certificate(WOLFSSL* ssl, WOLFSSL_X509* x509)
{
    long idx;

    WOLFSSL_ENTER("wolfSSL_use_certificate");
    if (x509 != NULL && ssl != NULL && x509->derCert != NULL) {
        if (ProcessBuffer(NULL, x509->derCert->buffer, x509->derCert->length,
                          WOLFSSL_FILETYPE_ASN1, CERT_TYPE, ssl, &idx, 0,
                          GET_VERIFY_SETTING_SSL(ssl)) == WOLFSSL_SUCCESS) {
            return WOLFSSL_SUCCESS;
        }
    }

    (void)idx;
    return WOLFSSL_FAILURE;
}

#endif /* OPENSSL_EXTRA */

int wolfSSL_use_certificate_ASN1(WOLFSSL* ssl, const unsigned char* der,
                                 int derSz)
{
    long idx;

    WOLFSSL_ENTER("wolfSSL_use_certificate_ASN1");
    if (der != NULL && ssl != NULL) {
        if (ProcessBuffer(NULL, der, derSz, WOLFSSL_FILETYPE_ASN1, CERT_TYPE,
                ssl, &idx, 0, GET_VERIFY_SETTING_SSL(ssl)) == WOLFSSL_SUCCESS) {
            return WOLFSSL_SUCCESS;
        }
    }

    (void)idx;
    return WOLFSSL_FAILURE;
}

#ifndef NO_FILESYSTEM

WOLFSSL_ABI
int wolfSSL_use_certificate_file(WOLFSSL* ssl, const char* file, int format)
{
    WOLFSSL_ENTER("wolfSSL_use_certificate_file");

    if (ssl == NULL) {
        return BAD_FUNC_ARG;
    }

    if (ProcessFile(ssl->ctx, file, format, CERT_TYPE,
                ssl, 0, NULL, GET_VERIFY_SETTING_SSL(ssl)) == WOLFSSL_SUCCESS) {
        return WOLFSSL_SUCCESS;
    }

    return WOLFSSL_FAILURE;
}


WOLFSSL_ABI
int wolfSSL_use_PrivateKey_file(WOLFSSL* ssl, const char* file, int format)
{
    WOLFSSL_ENTER("wolfSSL_use_PrivateKey_file");

    if (ssl == NULL) {
        return BAD_FUNC_ARG;
    }

    if (ProcessFile(ssl->ctx, file, format, PRIVATEKEY_TYPE,
                ssl, 0, NULL, GET_VERIFY_SETTING_SSL(ssl)) == WOLFSSL_SUCCESS) {
        return WOLFSSL_SUCCESS;
    }

    return WOLFSSL_FAILURE;
}


WOLFSSL_ABI
int wolfSSL_use_certificate_chain_file(WOLFSSL* ssl, const char* file)
{
    /* process up to MAX_CHAIN_DEPTH plus subject cert */
    WOLFSSL_ENTER("wolfSSL_use_certificate_chain_file");

    if (ssl == NULL) {
        return BAD_FUNC_ARG;
    }

    if (ProcessFile(ssl->ctx, file, WOLFSSL_FILETYPE_PEM, CERT_TYPE,
               ssl, 1, NULL, GET_VERIFY_SETTING_SSL(ssl)) == WOLFSSL_SUCCESS) {
        return WOLFSSL_SUCCESS;
    }

   return WOLFSSL_FAILURE;
}

int wolfSSL_use_certificate_chain_file_format(WOLFSSL* ssl, const char* file,
                                              int format)
{
    /* process up to MAX_CHAIN_DEPTH plus subject cert */
    WOLFSSL_ENTER("wolfSSL_use_certificate_chain_file_format");

    if (ssl == NULL) {
        return BAD_FUNC_ARG;
    }

    if (ProcessFile(ssl->ctx, file, format, CERT_TYPE, ssl, 1,
                    NULL, GET_VERIFY_SETTING_SSL(ssl)) == WOLFSSL_SUCCESS) {
        return WOLFSSL_SUCCESS;
    }
    return WOLFSSL_FAILURE;
}

#endif /* !NO_FILESYSTEM */

#ifdef HAVE_ECC

/* Set Temp CTX EC-DHE size in octets, can be 14 - 66 (112 - 521 bit) */
int wolfSSL_CTX_SetTmpEC_DHE_Sz(WOLFSSL_CTX* ctx, word16 sz)
{
    if (ctx == NULL)
        return BAD_FUNC_ARG;

    /* if 0 then get from loaded private key */
    if (sz == 0) {
        /* applies only to ECDSA */
        if (ctx->privateKeyType != ecc_dsa_sa_algo)
            return WOLFSSL_SUCCESS;

        if (ctx->privateKeySz == 0) {
            WOLFSSL_MSG("Must set private key/cert first");
            return BAD_FUNC_ARG;
        }

        sz = (word16)ctx->privateKeySz;
    }

    /* check size */
    if (sz < ECC_MINSIZE || sz > ECC_MAXSIZE)
        return BAD_FUNC_ARG;

    ctx->eccTempKeySz = sz;

    return WOLFSSL_SUCCESS;
}


/* Set Temp SSL EC-DHE size in octets, can be 14 - 66 (112 - 521 bit) */
int wolfSSL_SetTmpEC_DHE_Sz(WOLFSSL* ssl, word16 sz)
{
    if (ssl == NULL)
        return BAD_FUNC_ARG;

    /* check size */
    if (sz < ECC_MINSIZE || sz > ECC_MAXSIZE)
        return BAD_FUNC_ARG;

    ssl->eccTempKeySz = sz;

    return WOLFSSL_SUCCESS;
}

#endif /* HAVE_ECC */


#ifdef OPENSSL_EXTRA

#ifndef NO_FILESYSTEM
int wolfSSL_CTX_use_RSAPrivateKey_file(WOLFSSL_CTX* ctx,const char* file,
                                   int format)
{
    WOLFSSL_ENTER("SSL_CTX_use_RSAPrivateKey_file");

    return wolfSSL_CTX_use_PrivateKey_file(ctx, file, format);
}


int wolfSSL_use_RSAPrivateKey_file(WOLFSSL* ssl, const char* file, int format)
{
    WOLFSSL_ENTER("wolfSSL_use_RSAPrivateKey_file");

    return wolfSSL_use_PrivateKey_file(ssl, file, format);
}
#endif /* NO_FILESYSTEM */


/* Copies the master secret over to out buffer. If outSz is 0 returns the size
 * of master secret.
 *
 * ses : a session from completed TLS/SSL handshake
 * out : buffer to hold copy of master secret
 * outSz : size of out buffer
 * returns : number of bytes copied into out buffer on success
 *           less then or equal to 0 is considered a failure case
 */
int wolfSSL_SESSION_get_master_key(const WOLFSSL_SESSION* ses,
        unsigned char* out, int outSz)
{
    int size;

    ses = ClientSessionToSession(ses);

    if (outSz == 0) {
        return SECRET_LEN;
    }

    if (ses == NULL || out == NULL || outSz < 0) {
        return 0;
    }

    if (outSz > SECRET_LEN) {
        size = SECRET_LEN;
    }
    else {
        size = outSz;
    }

    XMEMCPY(out, ses->masterSecret, size);
    return size;
}


int wolfSSL_SESSION_get_master_key_length(const WOLFSSL_SESSION* ses)
{
    (void)ses;
    return SECRET_LEN;
}

#endif /* OPENSSL_EXTRA */

typedef struct {
    byte verifyPeer:1;
    byte verifyNone:1;
    byte failNoCert:1;
    byte failNoCertxPSK:1;
    byte verifyPostHandshake:1;
} SetVerifyOptions;

static SetVerifyOptions ModeToVerifyOptions(int mode)
{
    SetVerifyOptions opts;
    XMEMSET(&opts, 0, sizeof(SetVerifyOptions));

    if (mode != WOLFSSL_VERIFY_DEFAULT) {
        opts.verifyNone = (mode == WOLFSSL_VERIFY_NONE);
        if (!opts.verifyNone) {
            opts.verifyPeer =
                    (mode & WOLFSSL_VERIFY_PEER) != 0;
            opts.failNoCertxPSK =
                    (mode & WOLFSSL_VERIFY_FAIL_EXCEPT_PSK) != 0;
            opts.failNoCert =
                    (mode & WOLFSSL_VERIFY_FAIL_IF_NO_PEER_CERT) != 0;
#if defined(WOLFSSL_TLS13) && defined(WOLFSSL_POST_HANDSHAKE_AUTH)
            opts.verifyPostHandshake =
                    (mode & WOLFSSL_VERIFY_POST_HANDSHAKE) != 0;
#endif
        }
    }

    return opts;
}

WOLFSSL_ABI
void wolfSSL_CTX_set_verify(WOLFSSL_CTX* ctx, int mode, VerifyCallback vc)
{
    SetVerifyOptions opts;

    WOLFSSL_ENTER("wolfSSL_CTX_set_verify");
    if (ctx == NULL)
        return;

    opts = ModeToVerifyOptions(mode);

    ctx->verifyNone     = opts.verifyNone;
    ctx->verifyPeer     = opts.verifyPeer;
    ctx->failNoCert     = opts.failNoCert;
    ctx->failNoCertxPSK = opts.failNoCertxPSK;
#if defined(WOLFSSL_TLS13) && defined(WOLFSSL_POST_HANDSHAKE_AUTH)
    ctx->verifyPostHandshake = opts.verifyPostHandshake;
#endif

    ctx->verifyCallback = vc;
}

#ifdef OPENSSL_ALL
void wolfSSL_CTX_set_cert_verify_callback(WOLFSSL_CTX* ctx,
    CertVerifyCallback cb, void* arg)
{
    WOLFSSL_ENTER("SSL_CTX_set_cert_verify_callback");
    if (ctx == NULL)
        return;

    ctx->verifyCertCb = cb;
    ctx->verifyCertCbArg = arg;
}
#endif


void wolfSSL_set_verify(WOLFSSL* ssl, int mode, VerifyCallback vc)
{
    SetVerifyOptions opts;

    WOLFSSL_ENTER("wolfSSL_set_verify");
    if (ssl == NULL)
        return;

    opts = ModeToVerifyOptions(mode);

    ssl->options.verifyNone = opts.verifyNone;
    ssl->options.verifyPeer = opts.verifyPeer;
    ssl->options.failNoCert = opts.failNoCert;
    ssl->options.failNoCertxPSK = opts.failNoCertxPSK;
#if defined(WOLFSSL_TLS13) && defined(WOLFSSL_POST_HANDSHAKE_AUTH)
    ssl->options.verifyPostHandshake = opts.verifyPostHandshake;
#endif

    ssl->verifyCallback = vc;
}

void wolfSSL_set_verify_result(WOLFSSL *ssl, long v)
{
    WOLFSSL_ENTER("wolfSSL_set_verify_result");

    if (ssl == NULL)
        return;

#ifdef OPENSSL_ALL
    ssl->verifyCallbackResult = v;
#else
    (void)v;
    WOLFSSL_STUB("wolfSSL_set_verify_result");
#endif
}

#if defined(OPENSSL_EXTRA) && !defined(NO_CERTS) && \
    defined(WOLFSSL_TLS13) && defined(WOLFSSL_POST_HANDSHAKE_AUTH)
/* For TLS v1.3 send handshake messages after handshake completes. */
/* Returns 1=WOLFSSL_SUCCESS or 0=WOLFSSL_FAILURE */
int wolfSSL_verify_client_post_handshake(WOLFSSL* ssl)
{
    int ret = wolfSSL_request_certificate(ssl);
    if (ret != WOLFSSL_SUCCESS) {
        if (!IsAtLeastTLSv1_3(ssl->version)) {
            /* specific error of wrong version expected */
            WOLFSSL_ERROR(UNSUPPORTED_PROTO_VERSION);

        }
        else {
            WOLFSSL_ERROR(ret); /* log the error in the error queue */
        }
    }
    return (ret == WOLFSSL_SUCCESS) ? WOLFSSL_SUCCESS : WOLFSSL_FAILURE;
}

int wolfSSL_CTX_set_post_handshake_auth(WOLFSSL_CTX* ctx, int val)
{
    int ret = wolfSSL_CTX_allow_post_handshake_auth(ctx);
    if (ret == 0) {
        ctx->postHandshakeAuth = (val != 0);
    }
    return (ret == 0) ? WOLFSSL_SUCCESS : WOLFSSL_FAILURE;
}
int wolfSSL_set_post_handshake_auth(WOLFSSL* ssl, int val)
{
    int ret = wolfSSL_allow_post_handshake_auth(ssl);
    if (ret == 0) {
        ssl->options.postHandshakeAuth = (val != 0);
    }
    return (ret == 0) ? WOLFSSL_SUCCESS : WOLFSSL_FAILURE;
}
#endif /* OPENSSL_EXTRA && !NO_CERTS && WOLFSSL_TLS13 && WOLFSSL_POST_HANDSHAKE_AUTH */

/* store user ctx for verify callback */
void wolfSSL_SetCertCbCtx(WOLFSSL* ssl, void* ctx)
{
    WOLFSSL_ENTER("wolfSSL_SetCertCbCtx");
    if (ssl)
        ssl->verifyCbCtx = ctx;
}


/* store user ctx for verify callback */
void wolfSSL_CTX_SetCertCbCtx(WOLFSSL_CTX* ctx, void* userCtx)
{
    WOLFSSL_ENTER("wolfSSL_CTX_SetCertCbCtx");
    if (ctx)
        ctx->verifyCbCtx = userCtx;
}


/* store context CA Cache addition callback */
void wolfSSL_CTX_SetCACb(WOLFSSL_CTX* ctx, CallbackCACache cb)
{
    if (ctx && ctx->cm)
        ctx->cm->caCacheCallback = cb;
}


#if defined(PERSIST_CERT_CACHE)

#if !defined(NO_FILESYSTEM)

/* Persist cert cache to file */
int wolfSSL_CTX_save_cert_cache(WOLFSSL_CTX* ctx, const char* fname)
{
    WOLFSSL_ENTER("wolfSSL_CTX_save_cert_cache");

    if (ctx == NULL || fname == NULL)
        return BAD_FUNC_ARG;

    return CM_SaveCertCache(ctx->cm, fname);
}


/* Persist cert cache from file */
int wolfSSL_CTX_restore_cert_cache(WOLFSSL_CTX* ctx, const char* fname)
{
    WOLFSSL_ENTER("wolfSSL_CTX_restore_cert_cache");

    if (ctx == NULL || fname == NULL)
        return BAD_FUNC_ARG;

    return CM_RestoreCertCache(ctx->cm, fname);
}

#endif /* NO_FILESYSTEM */

/* Persist cert cache to memory */
int wolfSSL_CTX_memsave_cert_cache(WOLFSSL_CTX* ctx, void* mem,
                                   int sz, int* used)
{
    WOLFSSL_ENTER("wolfSSL_CTX_memsave_cert_cache");

    if (ctx == NULL || mem == NULL || used == NULL || sz <= 0)
        return BAD_FUNC_ARG;

    return CM_MemSaveCertCache(ctx->cm, mem, sz, used);
}


/* Restore cert cache from memory */
int wolfSSL_CTX_memrestore_cert_cache(WOLFSSL_CTX* ctx, const void* mem, int sz)
{
    WOLFSSL_ENTER("wolfSSL_CTX_memrestore_cert_cache");

    if (ctx == NULL || mem == NULL || sz <= 0)
        return BAD_FUNC_ARG;

    return CM_MemRestoreCertCache(ctx->cm, mem, sz);
}


/* get how big the the cert cache save buffer needs to be */
int wolfSSL_CTX_get_cert_cache_memsize(WOLFSSL_CTX* ctx)
{
    WOLFSSL_ENTER("wolfSSL_CTX_get_cert_cache_memsize");

    if (ctx == NULL)
        return BAD_FUNC_ARG;

    return CM_GetCertCacheMemSize(ctx->cm);
}

#endif /* PERSIST_CERT_CACHE */
#endif /* !NO_CERTS */


#ifndef NO_SESSION_CACHE

WOLFSSL_ABI
WOLFSSL_SESSION* wolfSSL_get_session(WOLFSSL* ssl)
{
    WOLFSSL_ENTER("SSL_get_session");
    if (ssl) {
#ifdef NO_SESSION_CACHE_REF
        return ssl->session;
#else
        if (ssl->options.side == WOLFSSL_CLIENT_END) {
            /* On the client side we want to return a persistant reference for
             * backwards compatibility. */
#ifndef NO_CLIENT_CACHE
            if (ssl->clientSession)
                return (WOLFSSL_SESSION*)ssl->clientSession;
            else {
                /* Try to add a ClientCache entry to associate with the current
                 * session. Ignore any session cache options. */
                int error;
                const byte* id = NULL;
                byte idSz = 0;
                id = ssl->session->sessionID;
                idSz = ssl->session->sessionIDSz;
                if (ssl->session->haveAltSessionID) {
                    id = ssl->session->altSessionID;
                    idSz = ID_LEN;
                }
                error = AddSessionToCache(ssl->session, id, idSz,
                        NULL, ssl->session->side,
#ifdef HAVE_SESSION_TICKET
                        ssl->session->ticketLen > 0,
#else
                        0,
#endif
                        &ssl->clientSession);
                if (error == 0)
                    return (WOLFSSL_SESSION*)ssl->clientSession;
            }
#endif
        }
        else
            return ssl->session;
#endif
    }

    return NULL;
}

/* The get1 version requires caller to call SSL_SESSION_free */
WOLFSSL_SESSION* wolfSSL_get1_session(WOLFSSL* ssl)
{
    WOLFSSL_SESSION* sess = NULL;
    WOLFSSL_ENTER("SSL_get1_session");
    if (ssl != NULL) {
        sess = ssl->session;
        if (sess != NULL) {
            /* increase reference count if allocated session */
            if (sess->type == WOLFSSL_SESSION_TYPE_HEAP) {
                if (wolfSSL_SESSION_up_ref(sess) != WOLFSSL_SUCCESS)
                    sess = NULL;
            }
        }
    }
    return sess;
}


/*
 * Sets the session object to use when establishing a TLS/SSL session using
 * the ssl object. Therefore, this function must be called before
 * wolfSSL_connect. The session object to use can be obtained in a previous
 * TLS/SSL connection using wolfSSL_get_session.
 *
 * This function rejects the session if it has been expired when this function
 * is called. Note that this expiration check is wolfSSL specific and differs
 * from OpenSSL return code behavior.
 *
 * By default, wolfSSL_set_session returns WOLFSSL_SUCCESS on successfully
 * setting the session, WOLFSSL_FAILURE on failure due to the session cache
 * being disabled, or the session has expired.
 *
 * To match OpenSSL return code behavior when session is expired, define
 * OPENSSL_EXTRA and WOLFSSL_ERROR_CODE_OPENSSL. This behavior will return
 * WOLFSSL_SUCCESS even when the session is expired and rejected.
 */
WOLFSSL_ABI
int wolfSSL_set_session(WOLFSSL* ssl, WOLFSSL_SESSION* session)
{
    WOLFSSL_ENTER("SSL_set_session");
    if (session)
        return wolfSSL_SetSession(ssl, session);

    return WOLFSSL_FAILURE;
}


#ifndef NO_CLIENT_CACHE

/* Associate client session with serverID, find existing or store for saving
   if newSession flag on, don't reuse existing session
   WOLFSSL_SUCCESS on ok */
int wolfSSL_SetServerID(WOLFSSL* ssl, const byte* id, int len, int newSession)
{
    WOLFSSL_SESSION* session = NULL;

    WOLFSSL_ENTER("wolfSSL_SetServerID");

    if (ssl == NULL || id == NULL || len <= 0)
        return BAD_FUNC_ARG;

    if (newSession == 0) {
        session = wolfSSL_GetSessionClient(ssl, id, len);
        if (session) {
            if (wolfSSL_SetSession(ssl, session) != WOLFSSL_SUCCESS) {
            #ifdef HAVE_EXT_CACHE
                wolfSSL_FreeSession(session);
            #endif
                WOLFSSL_MSG("wolfSSL_SetSession failed");
                session = NULL;
            }
        }
    }

    if (session == NULL) {
        WOLFSSL_MSG("Valid ServerID not cached already");

        ssl->session->idLen = (word16)min(SERVER_ID_LEN, (word32)len);
        XMEMCPY(ssl->session->serverID, id, ssl->session->idLen);
    }
#ifdef HAVE_EXT_CACHE
    else {
        wolfSSL_FreeSession(session);
    }
#endif

    return WOLFSSL_SUCCESS;
}

#endif /* !NO_CLIENT_CACHE */

#if defined(PERSIST_SESSION_CACHE)

/* for persistence, if changes to layout need to increment and modify
   save_session_cache() and restore_session_cache and memory versions too */
#define WOLFSSL_CACHE_VERSION 2

/* Session Cache Header information */
typedef struct {
    int version;     /* cache layout version id */
    int rows;        /* session rows */
    int columns;     /* session columns */
    int sessionSz;   /* sizeof WOLFSSL_SESSION */
} cache_header_t;

/* current persistence layout is:

   1) cache_header_t
   2) SessionCache
   3) ClientCache

   update WOLFSSL_CACHE_VERSION if change layout for the following
   PERSISTENT_SESSION_CACHE functions
*/


/* get how big the the session cache save buffer needs to be */
int wolfSSL_get_session_cache_memsize(void)
{
    int sz  = (int)(sizeof(SessionCache) + sizeof(cache_header_t));
#ifndef NO_CLIENT_CACHE
    sz += (int)(sizeof(ClientCache));
#endif
    return sz;
}


/* Persist session cache to memory */
int wolfSSL_memsave_session_cache(void* mem, int sz)
{
    int i;
    cache_header_t cache_header;
    SessionRow*    row  = (SessionRow*)((byte*)mem + sizeof(cache_header));

    WOLFSSL_ENTER("wolfSSL_memsave_session_cache");

    if (sz < wolfSSL_get_session_cache_memsize()) {
        WOLFSSL_MSG("Memory buffer too small");
        return BUFFER_E;
    }

    cache_header.version   = WOLFSSL_CACHE_VERSION;
    cache_header.rows      = SESSION_ROWS;
    cache_header.columns   = SESSIONS_PER_ROW;
    cache_header.sessionSz = (int)sizeof(WOLFSSL_SESSION);
    XMEMCPY(mem, &cache_header, sizeof(cache_header));

#ifndef ENABLE_SESSION_CACHE_ROW_LOCK
    if (wc_LockMutex(&session_mutex) != 0) {
        WOLFSSL_MSG("Session cache mutex lock failed");
        return BAD_MUTEX_E;
    }
#endif
    for (i = 0; i < cache_header.rows; ++i) {
    #ifdef ENABLE_SESSION_CACHE_ROW_LOCK
        if (SESSION_ROW_LOCK(&SessionCache[i]) != 0) {
            WOLFSSL_MSG("Session row cache mutex lock failed");
            return BAD_MUTEX_E;
        }
    #endif

        XMEMCPY(row++, &SessionCache[i], SIZEOF_SESSION_ROW);
    #ifdef ENABLE_SESSION_CACHE_ROW_LOCK
        SESSION_ROW_UNLOCK(&SessionCache[i]);
    #endif
    }
#ifndef ENABLE_SESSION_CACHE_ROW_LOCK
    wc_UnLockMutex(&session_mutex);
#endif

#ifndef NO_CLIENT_CACHE
    if (wc_LockMutex(&clisession_mutex) != 0) {
        WOLFSSL_MSG("Client cache mutex lock failed");
        return BAD_MUTEX_E;
    }
    XMEMCPY(row, ClientCache, sizeof(ClientCache));
    wc_UnLockMutex(&clisession_mutex);
#endif

    WOLFSSL_LEAVE("wolfSSL_memsave_session_cache", WOLFSSL_SUCCESS);

    return WOLFSSL_SUCCESS;
}


/* Restore the persistent session cache from memory */
int wolfSSL_memrestore_session_cache(const void* mem, int sz)
{
    int    i;
    cache_header_t cache_header;
    SessionRow*    row  = (SessionRow*)((byte*)mem + sizeof(cache_header));

    WOLFSSL_ENTER("wolfSSL_memrestore_session_cache");

    if (sz < wolfSSL_get_session_cache_memsize()) {
        WOLFSSL_MSG("Memory buffer too small");
        return BUFFER_E;
    }

    XMEMCPY(&cache_header, mem, sizeof(cache_header));
    if (cache_header.version   != WOLFSSL_CACHE_VERSION ||
        cache_header.rows      != SESSION_ROWS ||
        cache_header.columns   != SESSIONS_PER_ROW ||
        cache_header.sessionSz != (int)sizeof(WOLFSSL_SESSION)) {

        WOLFSSL_MSG("Session cache header match failed");
        return CACHE_MATCH_ERROR;
    }

#ifndef ENABLE_SESSION_CACHE_ROW_LOCK
    if (wc_LockMutex(&session_mutex) != 0) {
        WOLFSSL_MSG("Session cache mutex lock failed");
        return BAD_MUTEX_E;
    }
#endif
    for (i = 0; i < cache_header.rows; ++i) {
    #ifdef ENABLE_SESSION_CACHE_ROW_LOCK
        if (SESSION_ROW_LOCK(&SessionCache[i]) != 0) {
            WOLFSSL_MSG("Session row cache mutex lock failed");
            return BAD_MUTEX_E;
        }
    #endif

        XMEMCPY(&SessionCache[i], row++, SIZEOF_SESSION_ROW);
    #ifdef ENABLE_SESSION_CACHE_ROW_LOCK
        SESSION_ROW_UNLOCK(&SessionCache[i]);
    #endif
    }
#ifndef ENABLE_SESSION_CACHE_ROW_LOCK
    wc_UnLockMutex(&session_mutex);
#endif

#ifndef NO_CLIENT_CACHE
    if (wc_LockMutex(&clisession_mutex) != 0) {
        WOLFSSL_MSG("Client cache mutex lock failed");
        return BAD_MUTEX_E;
    }
    XMEMCPY(ClientCache, row, sizeof(ClientCache));
    wc_UnLockMutex(&clisession_mutex);
#endif

    WOLFSSL_LEAVE("wolfSSL_memrestore_session_cache", WOLFSSL_SUCCESS);

    return WOLFSSL_SUCCESS;
}

#if !defined(NO_FILESYSTEM)

/* Persist session cache to file */
/* doesn't use memsave because of additional memory use */
int wolfSSL_save_session_cache(const char *fname)
{
    XFILE  file;
    int    ret;
    int    rc = WOLFSSL_SUCCESS;
    int    i;
    cache_header_t cache_header;

    WOLFSSL_ENTER("wolfSSL_save_session_cache");

    file = XFOPEN(fname, "w+b");
    if (file == XBADFILE) {
        WOLFSSL_MSG("Couldn't open session cache save file");
        return WOLFSSL_BAD_FILE;
    }
    cache_header.version   = WOLFSSL_CACHE_VERSION;
    cache_header.rows      = SESSION_ROWS;
    cache_header.columns   = SESSIONS_PER_ROW;
    cache_header.sessionSz = (int)sizeof(WOLFSSL_SESSION);

    /* cache header */
    ret = (int)XFWRITE(&cache_header, sizeof cache_header, 1, file);
    if (ret != 1) {
        WOLFSSL_MSG("Session cache header file write failed");
        XFCLOSE(file);
        return FWRITE_ERROR;
    }

#ifndef ENABLE_SESSION_CACHE_ROW_LOCK
    if (wc_LockMutex(&session_mutex) != 0) {
        WOLFSSL_MSG("Session cache mutex lock failed");
        XFCLOSE(file);
        return BAD_MUTEX_E;
    }
#endif
    /* session cache */
    for (i = 0; i < cache_header.rows; ++i) {
    #ifdef ENABLE_SESSION_CACHE_ROW_LOCK
        if (SESSION_ROW_LOCK(&SessionCache[i]) != 0) {
            WOLFSSL_MSG("Session row cache mutex lock failed");
            XFCLOSE(file);
            return BAD_MUTEX_E;
        }
    #endif

        ret = (int)XFWRITE(&SessionCache[i], SIZEOF_SESSION_ROW, 1, file);
    #ifdef ENABLE_SESSION_CACHE_ROW_LOCK
        SESSION_ROW_UNLOCK(&SessionCache[i]);
    #endif
        if (ret != 1) {
            WOLFSSL_MSG("Session cache member file write failed");
            rc = FWRITE_ERROR;
            break;
        }
    }
#ifndef ENABLE_SESSION_CACHE_ROW_LOCK
    wc_UnLockMutex(&session_mutex);
#endif

#ifndef NO_CLIENT_CACHE
    /* client cache */
    if (wc_LockMutex(&clisession_mutex) != 0) {
        WOLFSSL_MSG("Client cache mutex lock failed");
        XFCLOSE(file);
        return BAD_MUTEX_E;
    }
    ret = (int)XFWRITE(ClientCache, sizeof(ClientCache), 1, file);
    if (ret != 1) {
        WOLFSSL_MSG("Client cache member file write failed");
        rc = FWRITE_ERROR;
    }
    wc_UnLockMutex(&clisession_mutex);
#endif /* !NO_CLIENT_CACHE */

    XFCLOSE(file);
    WOLFSSL_LEAVE("wolfSSL_save_session_cache", rc);

    return rc;
}


/* Restore the persistent session cache from file */
/* doesn't use memstore because of additional memory use */
int wolfSSL_restore_session_cache(const char *fname)
{
    XFILE  file;
    int    rc = WOLFSSL_SUCCESS;
    int    ret;
    int    i;
    cache_header_t cache_header;

    WOLFSSL_ENTER("wolfSSL_restore_session_cache");

    file = XFOPEN(fname, "rb");
    if (file == XBADFILE) {
        WOLFSSL_MSG("Couldn't open session cache save file");
        return WOLFSSL_BAD_FILE;
    }
    /* cache header */
    ret = (int)XFREAD(&cache_header, sizeof(cache_header), 1, file);
    if (ret != 1) {
        WOLFSSL_MSG("Session cache header file read failed");
        XFCLOSE(file);
        return FREAD_ERROR;
    }
    if (cache_header.version   != WOLFSSL_CACHE_VERSION ||
        cache_header.rows      != SESSION_ROWS ||
        cache_header.columns   != SESSIONS_PER_ROW ||
        cache_header.sessionSz != (int)sizeof(WOLFSSL_SESSION)) {

        WOLFSSL_MSG("Session cache header match failed");
        XFCLOSE(file);
        return CACHE_MATCH_ERROR;
    }

#ifndef ENABLE_SESSION_CACHE_ROW_LOCK
    if (wc_LockMutex(&session_mutex) != 0) {
        WOLFSSL_MSG("Session cache mutex lock failed");
        XFCLOSE(file);
        return BAD_MUTEX_E;
    }
#endif
    /* session cache */
    for (i = 0; i < cache_header.rows; ++i) {
    #ifdef ENABLE_SESSION_CACHE_ROW_LOCK
        if (SESSION_ROW_LOCK(&SessionCache[i]) != 0) {
            WOLFSSL_MSG("Session row cache mutex lock failed");
            XFCLOSE(file);
            return BAD_MUTEX_E;
        }
    #endif

        ret = (int)XFREAD(&SessionCache[i], SIZEOF_SESSION_ROW, 1, file);
    #ifdef ENABLE_SESSION_CACHE_ROW_LOCK
        SESSION_ROW_UNLOCK(&SessionCache[i]);
    #endif
        if (ret != 1) {
            WOLFSSL_MSG("Session cache member file read failed");
            XMEMSET(SessionCache, 0, sizeof SessionCache);
            rc = FREAD_ERROR;
            break;
        }
    }
#ifndef ENABLE_SESSION_CACHE_ROW_LOCK
    wc_UnLockMutex(&session_mutex);
#endif

#ifndef NO_CLIENT_CACHE
    /* client cache */
    if (wc_LockMutex(&clisession_mutex) != 0) {
        WOLFSSL_MSG("Client cache mutex lock failed");
        XFCLOSE(file);
        return BAD_MUTEX_E;
    }
    ret = (int)XFREAD(ClientCache, sizeof(ClientCache), 1, file);
    if (ret != 1) {
        WOLFSSL_MSG("Client cache member file read failed");
        XMEMSET(ClientCache, 0, sizeof ClientCache);
        rc = FREAD_ERROR;
    }
    wc_UnLockMutex(&clisession_mutex);
#endif /* !NO_CLIENT_CACHE */

    XFCLOSE(file);
    WOLFSSL_LEAVE("wolfSSL_restore_session_cache", rc);

    return rc;
}

#endif /* !NO_FILESYSTEM */
#endif /* PERSIST_SESSION_CACHE */
#endif /* NO_SESSION_CACHE */


void wolfSSL_load_error_strings(void)
{
    /* compatibility only */
}


int wolfSSL_library_init(void)
{
    WOLFSSL_ENTER("SSL_library_init");
    if (wolfSSL_Init() == WOLFSSL_SUCCESS)
        return WOLFSSL_SUCCESS;
    else
        return WOLFSSL_FATAL_ERROR;
}


#ifdef HAVE_SECRET_CALLBACK

int wolfSSL_set_session_secret_cb(WOLFSSL* ssl, SessionSecretCb cb, void* ctx)
{
    WOLFSSL_ENTER("wolfSSL_set_session_secret_cb");
    if (ssl == NULL)
        return WOLFSSL_FATAL_ERROR;

    ssl->sessionSecretCb = cb;
    ssl->sessionSecretCtx = ctx;
    /* If using a pre-set key, assume session resumption. */
    ssl->session->sessionIDSz = 0;
    ssl->options.resuming = 1;

    return WOLFSSL_SUCCESS;
}

#endif


#ifndef NO_SESSION_CACHE

/* on by default if built in but allow user to turn off */
WOLFSSL_ABI
long wolfSSL_CTX_set_session_cache_mode(WOLFSSL_CTX* ctx, long mode)
{
    WOLFSSL_ENTER("SSL_CTX_set_session_cache_mode");

    if (ctx == NULL)
        return WOLFSSL_FAILURE;

    if (mode == WOLFSSL_SESS_CACHE_OFF)
        ctx->sessionCacheOff = 1;

    if ((mode & WOLFSSL_SESS_CACHE_NO_AUTO_CLEAR) != 0)
        ctx->sessionCacheFlushOff = 1;

#ifdef HAVE_EXT_CACHE
    if ((mode & WOLFSSL_SESS_CACHE_NO_INTERNAL_STORE) != 0)
        ctx->internalCacheOff = 1;
    if ((mode & WOLFSSL_SESS_CACHE_NO_INTERNAL_LOOKUP) != 0)
        ctx->internalCacheLookupOff = 1;
#endif

    return WOLFSSL_SUCCESS;
}

#endif /* NO_SESSION_CACHE */


#if !defined(NO_CERTS)
#if defined(PERSIST_CERT_CACHE)


#define WOLFSSL_CACHE_CERT_VERSION 1

typedef struct {
    int version;                 /* cache cert layout version id */
    int rows;                    /* hash table rows, CA_TABLE_SIZE */
    int columns[CA_TABLE_SIZE];  /* columns per row on list */
    int signerSz;                /* sizeof Signer object */
} CertCacheHeader;

/* current cert persistence layout is:

   1) CertCacheHeader
   2) caTable

   update WOLFSSL_CERT_CACHE_VERSION if change layout for the following
   PERSIST_CERT_CACHE functions
*/


/* Return memory needed to persist this signer, have lock */
static WC_INLINE int GetSignerMemory(Signer* signer)
{
    int sz = sizeof(signer->pubKeySize) + sizeof(signer->keyOID)
           + sizeof(signer->nameLen)    + sizeof(signer->subjectNameHash);

#if !defined(NO_SKID)
        sz += (int)sizeof(signer->subjectKeyIdHash);
#endif

    /* add dynamic bytes needed */
    sz += signer->pubKeySize;
    sz += signer->nameLen;

    return sz;
}


/* Return memory needed to persist this row, have lock */
static WC_INLINE int GetCertCacheRowMemory(Signer* row)
{
    int sz = 0;

    while (row) {
        sz += GetSignerMemory(row);
        row = row->next;
    }

    return sz;
}


/* get the size of persist cert cache, have lock */
static WC_INLINE int GetCertCacheMemSize(WOLFSSL_CERT_MANAGER* cm)
{
    int sz;
    int i;

    sz = sizeof(CertCacheHeader);

    for (i = 0; i < CA_TABLE_SIZE; i++)
        sz += GetCertCacheRowMemory(cm->caTable[i]);

    return sz;
}


/* Store cert cache header columns with number of items per list, have lock */
static WC_INLINE void SetCertHeaderColumns(WOLFSSL_CERT_MANAGER* cm, int* columns)
{
    int     i;
    Signer* row;

    for (i = 0; i < CA_TABLE_SIZE; i++) {
        int count = 0;
        row = cm->caTable[i];

        while (row) {
            ++count;
            row = row->next;
        }
        columns[i] = count;
    }
}


/* Restore whole cert row from memory, have lock, return bytes consumed,
   < 0 on error, have lock */
static WC_INLINE int RestoreCertRow(WOLFSSL_CERT_MANAGER* cm, byte* current,
                                 int row, int listSz, const byte* end)
{
    int idx = 0;

    if (listSz < 0) {
        WOLFSSL_MSG("Row header corrupted, negative value");
        return PARSE_ERROR;
    }

    while (listSz) {
        Signer* signer;
        byte*   publicKey;
        byte*   start = current + idx;  /* for end checks on this signer */
        int     minSz = sizeof(signer->pubKeySize) + sizeof(signer->keyOID) +
                      sizeof(signer->nameLen) + sizeof(signer->subjectNameHash);
        #ifndef NO_SKID
                minSz += (int)sizeof(signer->subjectKeyIdHash);
        #endif

        if (start + minSz > end) {
            WOLFSSL_MSG("Would overread restore buffer");
            return BUFFER_E;
        }
        signer = MakeSigner(cm->heap);
        if (signer == NULL)
            return MEMORY_E;

        /* pubKeySize */
        XMEMCPY(&signer->pubKeySize, current + idx, sizeof(signer->pubKeySize));
        idx += (int)sizeof(signer->pubKeySize);

        /* keyOID */
        XMEMCPY(&signer->keyOID, current + idx, sizeof(signer->keyOID));
        idx += (int)sizeof(signer->keyOID);

        /* publicKey */
        if (start + minSz + signer->pubKeySize > end) {
            WOLFSSL_MSG("Would overread restore buffer");
            FreeSigner(signer, cm->heap);
            return BUFFER_E;
        }
        publicKey = (byte*)XMALLOC(signer->pubKeySize, cm->heap,
                                   DYNAMIC_TYPE_KEY);
        if (publicKey == NULL) {
            FreeSigner(signer, cm->heap);
            return MEMORY_E;
        }

        XMEMCPY(publicKey, current + idx, signer->pubKeySize);
        signer->publicKey = publicKey;
        idx += signer->pubKeySize;

        /* nameLen */
        XMEMCPY(&signer->nameLen, current + idx, sizeof(signer->nameLen));
        idx += (int)sizeof(signer->nameLen);

        /* name */
        if (start + minSz + signer->pubKeySize + signer->nameLen > end) {
            WOLFSSL_MSG("Would overread restore buffer");
            FreeSigner(signer, cm->heap);
            return BUFFER_E;
        }
        signer->name = (char*)XMALLOC(signer->nameLen, cm->heap,
                                      DYNAMIC_TYPE_SUBJECT_CN);
        if (signer->name == NULL) {
            FreeSigner(signer, cm->heap);
            return MEMORY_E;
        }

        XMEMCPY(signer->name, current + idx, signer->nameLen);
        idx += signer->nameLen;

        /* subjectNameHash */
        XMEMCPY(signer->subjectNameHash, current + idx, SIGNER_DIGEST_SIZE);
        idx += SIGNER_DIGEST_SIZE;

        #ifndef NO_SKID
            /* subjectKeyIdHash */
            XMEMCPY(signer->subjectKeyIdHash, current + idx,SIGNER_DIGEST_SIZE);
            idx += SIGNER_DIGEST_SIZE;
        #endif

        signer->next = cm->caTable[row];
        cm->caTable[row] = signer;

        --listSz;
    }

    return idx;
}


/* Store whole cert row into memory, have lock, return bytes added */
static WC_INLINE int StoreCertRow(WOLFSSL_CERT_MANAGER* cm, byte* current, int row)
{
    int     added  = 0;
    Signer* list   = cm->caTable[row];

    while (list) {
        XMEMCPY(current + added, &list->pubKeySize, sizeof(list->pubKeySize));
        added += (int)sizeof(list->pubKeySize);

        XMEMCPY(current + added, &list->keyOID,     sizeof(list->keyOID));
        added += (int)sizeof(list->keyOID);

        XMEMCPY(current + added, list->publicKey, list->pubKeySize);
        added += list->pubKeySize;

        XMEMCPY(current + added, &list->nameLen, sizeof(list->nameLen));
        added += (int)sizeof(list->nameLen);

        XMEMCPY(current + added, list->name, list->nameLen);
        added += list->nameLen;

        XMEMCPY(current + added, list->subjectNameHash, SIGNER_DIGEST_SIZE);
        added += SIGNER_DIGEST_SIZE;

        #ifndef NO_SKID
            XMEMCPY(current + added, list->subjectKeyIdHash,SIGNER_DIGEST_SIZE);
            added += SIGNER_DIGEST_SIZE;
        #endif

        list = list->next;
    }

    return added;
}


/* Persist cert cache to memory, have lock */
static WC_INLINE int DoMemSaveCertCache(WOLFSSL_CERT_MANAGER* cm,
                                     void* mem, int sz)
{
    int realSz;
    int ret = WOLFSSL_SUCCESS;
    int i;

    WOLFSSL_ENTER("DoMemSaveCertCache");

    realSz = GetCertCacheMemSize(cm);
    if (realSz > sz) {
        WOLFSSL_MSG("Mem output buffer too small");
        ret = BUFFER_E;
    }
    else {
        byte*           current;
        CertCacheHeader hdr;

        hdr.version  = WOLFSSL_CACHE_CERT_VERSION;
        hdr.rows     = CA_TABLE_SIZE;
        SetCertHeaderColumns(cm, hdr.columns);
        hdr.signerSz = (int)sizeof(Signer);

        XMEMCPY(mem, &hdr, sizeof(CertCacheHeader));
        current = (byte*)mem + sizeof(CertCacheHeader);

        for (i = 0; i < CA_TABLE_SIZE; ++i)
            current += StoreCertRow(cm, current, i);
    }

    return ret;
}


#if !defined(NO_FILESYSTEM)

/* Persist cert cache to file */
int CM_SaveCertCache(WOLFSSL_CERT_MANAGER* cm, const char* fname)
{
    XFILE file;
    int   rc = WOLFSSL_SUCCESS;
    int   memSz;
    byte* mem;

    WOLFSSL_ENTER("CM_SaveCertCache");

    file = XFOPEN(fname, "w+b");
    if (file == XBADFILE) {
       WOLFSSL_MSG("Couldn't open cert cache save file");
       return WOLFSSL_BAD_FILE;
    }

    if (wc_LockMutex(&cm->caLock) != 0) {
        WOLFSSL_MSG("wc_LockMutex on caLock failed");
        XFCLOSE(file);
        return BAD_MUTEX_E;
    }

    memSz = GetCertCacheMemSize(cm);
    mem   = (byte*)XMALLOC(memSz, cm->heap, DYNAMIC_TYPE_TMP_BUFFER);
    if (mem == NULL) {
        WOLFSSL_MSG("Alloc for tmp buffer failed");
        rc = MEMORY_E;
    } else {
        rc = DoMemSaveCertCache(cm, mem, memSz);
        if (rc == WOLFSSL_SUCCESS) {
            int ret = (int)XFWRITE(mem, memSz, 1, file);
            if (ret != 1) {
                WOLFSSL_MSG("Cert cache file write failed");
                rc = FWRITE_ERROR;
            }
        }
        XFREE(mem, cm->heap, DYNAMIC_TYPE_TMP_BUFFER);
    }

    wc_UnLockMutex(&cm->caLock);
    XFCLOSE(file);

    return rc;
}


/* Restore cert cache from file */
int CM_RestoreCertCache(WOLFSSL_CERT_MANAGER* cm, const char* fname)
{
    XFILE file;
    int   rc = WOLFSSL_SUCCESS;
    int   ret;
    int   memSz;
    byte* mem;

    WOLFSSL_ENTER("CM_RestoreCertCache");

    file = XFOPEN(fname, "rb");
    if (file == XBADFILE) {
       WOLFSSL_MSG("Couldn't open cert cache save file");
       return WOLFSSL_BAD_FILE;
    }

    if(XFSEEK(file, 0, XSEEK_END) != 0) {
        XFCLOSE(file);
        return WOLFSSL_BAD_FILE;
    }
    memSz = (int)XFTELL(file);
    XREWIND(file);

    if (memSz > MAX_WOLFSSL_FILE_SIZE || memSz <= 0) {
        WOLFSSL_MSG("CM_RestoreCertCache file size error");
        XFCLOSE(file);
        return WOLFSSL_BAD_FILE;
    }

    mem = (byte*)XMALLOC(memSz, cm->heap, DYNAMIC_TYPE_TMP_BUFFER);
    if (mem == NULL) {
        WOLFSSL_MSG("Alloc for tmp buffer failed");
        XFCLOSE(file);
        return MEMORY_E;
    }

    ret = (int)XFREAD(mem, memSz, 1, file);
    if (ret != 1) {
        WOLFSSL_MSG("Cert file read error");
        rc = FREAD_ERROR;
    } else {
        rc = CM_MemRestoreCertCache(cm, mem, memSz);
        if (rc != WOLFSSL_SUCCESS) {
            WOLFSSL_MSG("Mem restore cert cache failed");
        }
    }

    XFREE(mem, cm->heap, DYNAMIC_TYPE_TMP_BUFFER);
    XFCLOSE(file);

    return rc;
}

#endif /* NO_FILESYSTEM */


/* Persist cert cache to memory */
int CM_MemSaveCertCache(WOLFSSL_CERT_MANAGER* cm, void* mem, int sz, int* used)
{
    int ret = WOLFSSL_SUCCESS;

    WOLFSSL_ENTER("CM_MemSaveCertCache");

    if (wc_LockMutex(&cm->caLock) != 0) {
        WOLFSSL_MSG("wc_LockMutex on caLock failed");
        return BAD_MUTEX_E;
    }

    ret = DoMemSaveCertCache(cm, mem, sz);
    if (ret == WOLFSSL_SUCCESS)
        *used  = GetCertCacheMemSize(cm);

    wc_UnLockMutex(&cm->caLock);

    return ret;
}


/* Restore cert cache from memory */
int CM_MemRestoreCertCache(WOLFSSL_CERT_MANAGER* cm, const void* mem, int sz)
{
    int ret = WOLFSSL_SUCCESS;
    int i;
    CertCacheHeader* hdr = (CertCacheHeader*)mem;
    byte*            current = (byte*)mem + sizeof(CertCacheHeader);
    byte*            end     = (byte*)mem + sz;  /* don't go over */

    WOLFSSL_ENTER("CM_MemRestoreCertCache");

    if (current > end) {
        WOLFSSL_MSG("Cert Cache Memory buffer too small");
        return BUFFER_E;
    }

    if (hdr->version  != WOLFSSL_CACHE_CERT_VERSION ||
        hdr->rows     != CA_TABLE_SIZE ||
        hdr->signerSz != (int)sizeof(Signer)) {

        WOLFSSL_MSG("Cert Cache Memory header mismatch");
        return CACHE_MATCH_ERROR;
    }

    if (wc_LockMutex(&cm->caLock) != 0) {
        WOLFSSL_MSG("wc_LockMutex on caLock failed");
        return BAD_MUTEX_E;
    }

    FreeSignerTable(cm->caTable, CA_TABLE_SIZE, cm->heap);

    for (i = 0; i < CA_TABLE_SIZE; ++i) {
        int added = RestoreCertRow(cm, current, i, hdr->columns[i], end);
        if (added < 0) {
            WOLFSSL_MSG("RestoreCertRow error");
            ret = added;
            break;
        }
        current += added;
    }

    wc_UnLockMutex(&cm->caLock);

    return ret;
}


/* get how big the the cert cache save buffer needs to be */
int CM_GetCertCacheMemSize(WOLFSSL_CERT_MANAGER* cm)
{
    int sz;

    WOLFSSL_ENTER("CM_GetCertCacheMemSize");

    if (wc_LockMutex(&cm->caLock) != 0) {
        WOLFSSL_MSG("wc_LockMutex on caLock failed");
        return BAD_MUTEX_E;
    }

    sz = GetCertCacheMemSize(cm);

    wc_UnLockMutex(&cm->caLock);

    return sz;
}

#endif /* PERSIST_CERT_CACHE */
#endif /* NO_CERTS */

#ifdef OPENSSL_EXTRA


/* removes all cipher suites from the list that contain "toRemove"
 * returns the new list size on success
 */
static int wolfSSL_remove_ciphers(char* list, int sz, const char* toRemove)
{
    int idx = 0;
    char* next = (char*)list;
    int totalSz = sz;

    if (list == NULL) {
        return 0;
    }

    do {
        char*  current = next;
        char   name[MAX_SUITE_NAME + 1];
        word32 length;

        next   = XSTRSTR(next, ":");
        length = min(sizeof(name), !next ? (word32)XSTRLEN(current) /* last */
                                         : (word32)(next - current));

        XSTRNCPY(name, current, length);
        name[(length == sizeof(name)) ? length - 1 : length] = 0;

        if (XSTRSTR(name, toRemove)) {
            XMEMMOVE(list + idx, list + idx + length, totalSz - (idx + length));
            totalSz -= length;
            list[totalSz] = '\0';
            next = current;
        }
        else {
            idx += length;
        }
    } while (next++); /* ++ needed to skip ':' */

    return totalSz;
}
/*
 * build enabled cipher list w/ TLS13 or w/o TLS13 suites
 * @param ctx    a pointer to WOLFSSL_CTX structure
 * @param suites currently enabled suites
 * @param onlytlsv13suites flag whether correcting w/ TLS13 suites
 *                         or w/o TLS13 suties
 * @param list   suites list that user wants to update
 * @return suites list on success, otherwise NULL
 */
static char* buildEnabledCipherList(WOLFSSL_CTX* ctx, Suites* suites,
           int tls13Only, const char* list)
{
    word32 idx = 0;
    word32 listsz = 0;
    word32 len = 0;
    word32 ianasz = 0;
    const char* enabledcs = NULL;
    char* locallist = NULL;
    char* head = NULL;
    byte cipherSuite0;
    byte cipherSuite;

    /* sanity check */
    if (ctx == NULL || suites == NULL || list == NULL)
        return NULL;

    if (!suites->setSuites)
        return NULL;

    listsz = (word32)XSTRLEN(list);

    /* calculate necessary buffer length */
    for(idx = 0; idx < suites->suiteSz; idx++) {

        cipherSuite0 = suites->suites[idx];
        cipherSuite  = suites->suites[++idx];

        if (tls13Only && cipherSuite0 == TLS13_BYTE) {
            enabledcs = GetCipherNameInternal(cipherSuite0, cipherSuite);
        }
        else if (!tls13Only && cipherSuite0 != TLS13_BYTE) {
            enabledcs = GetCipherNameInternal(cipherSuite0, cipherSuite);
        }
        else
            continue;

        if (XSTRNCMP(enabledcs, "None", XSTRLEN(enabledcs)) != 0) {
            len += (word32)XSTRLEN(enabledcs) + 2;
        }
    }

    len += listsz + 2;

    /* build string */
    if (len > (listsz + 2)) {
        locallist = (char*)XMALLOC(len, ctx->heap,
                                           DYNAMIC_TYPE_TMP_BUFFER);
        /* sanity check */
        if (!locallist)
            return NULL;

        XMEMSET(locallist, 0, len);

        head = locallist;

        if (!tls13Only)
        {
            /* always tls13 suites in the head position */
            XSTRNCPY(locallist, list, len);
            locallist += listsz;
            *locallist++ = ':';
            *locallist = 0;
            len -= listsz + 1;
        }

        for(idx = 0; idx < suites->suiteSz; idx++) {
            cipherSuite0 = suites->suites[idx];
            cipherSuite  = suites->suites[++idx];

            if (tls13Only && cipherSuite0 == TLS13_BYTE) {
                enabledcs = GetCipherNameInternal(cipherSuite0, cipherSuite);
            }
            else if (!tls13Only && cipherSuite0 != TLS13_BYTE) {
                enabledcs = GetCipherNameInternal(cipherSuite0, cipherSuite);
            }
            else
                continue;

            ianasz = (int)XSTRLEN(enabledcs);
            if (ianasz + 1 < len) {
                XSTRNCPY(locallist, enabledcs, len);
                locallist += ianasz;

                *locallist++ = ':';
                *locallist = 0;
                len -= ianasz + 1;
            }
            else{
                XFREE(locallist, ctx->heap, DYNAMIC_TYPE_TMP_BUFFER);
                return NULL;
            }
        }

        if (tls13Only) {
            XSTRNCPY(locallist, list, len);
            locallist += listsz;
            *locallist = 0;
        }

        return head;
    }
    else
        return NULL;
}

/*
 * check if the list has TLS13 and pre-TLS13 suites
 * @param list cipher suite list that user want to set
 * @return mixed: 0, only pre-TLS13: 1, only TLS13: 2
 */
static int CheckcipherList(const char* list)
{
    int ret;
    int findTLSv13Suites = 0;
    int findbeforeSuites = 0;
    byte cipherSuite0;
    byte cipherSuite1;
    int flags;
    char* next = (char*)list;

    do {
        char*  current = next;
        char   name[MAX_SUITE_NAME + 1];
        word32 length = MAX_SUITE_NAME;
        word32 current_length;

        next   = XSTRSTR(next, ":");

        current_length = (!next) ? (word32)XSTRLEN(current)
                                 : (word32)(next - current);

        if (current_length < length) {
            length = current_length;
        }
        XMEMCPY(name, current, length);
        name[length] = 0;

        ret = wolfSSL_get_cipher_suite_from_name(name, &cipherSuite0,
                                                        &cipherSuite1, &flags);
        if (ret == 0) {
            if (cipherSuite0 == TLS13_BYTE) {
                /* TLSv13 suite */
                findTLSv13Suites = 1;
                break;
            }
            else {
                findbeforeSuites = 1;
                break;
            }
        }
        if (findTLSv13Suites == 1 && findbeforeSuites == 1) {
            /* list has mixed suites */
            return 0;
        }
    }  while (next++); /* ++ needed to skip ':' */

    if (findTLSv13Suites == 0 && findbeforeSuites == 1) {
        return 1;/* only before TLSv13 suites */
    }
    else if (findTLSv13Suites == 1 && findbeforeSuites == 0) {
        return 2;/* only TLSv13 suties */
    }
    else {
        return 0;/* handle as mixed */
    }
}

/* parse some bulk lists like !eNULL / !aNULL
 *
 * returns WOLFSSL_SUCCESS on success and sets the cipher suite list
 */
static int wolfSSL_parse_cipher_list(WOLFSSL_CTX* ctx, Suites* suites,
        const char* list)
{
    int       ret          = 0;
    const int suiteSz      = GetCipherNamesSize();
    char*     next         = (char*)list;
    const CipherSuiteInfo* names = GetCipherNames();
    char*     localList    = NULL;
    int sz = 0;
    int listattribute = 0;
    char*     buildcipherList = NULL;
    int tls13Only = 0;

    if (suites == NULL || list == NULL) {
        WOLFSSL_MSG("NULL argument");
        return WOLFSSL_FAILURE;
    }

    /* does list contain eNULL or aNULL? */
    if (XSTRSTR(list, "aNULL") || XSTRSTR(list, "eNULL")) {
        do {
            char*  current = next;
            char   name[MAX_SUITE_NAME + 1];
            int    i;
            word32 length = MAX_SUITE_NAME;
            word32 current_length;

            next   = XSTRSTR(next, ":");

            current_length = (!next) ? (word32)XSTRLEN(current)
                                     : (word32)(next - current);

            if (current_length < length) {
                length = current_length;
            }
            XMEMCPY(name, current, length);
            name[length] = 0;

            /* check for "not" case */
            if (name[0] == '!' && suiteSz > 0) {
                /* populate list with all suites if not already created */
                if (localList == NULL) {
                    for (i = 0; i < suiteSz; i++) {
                        sz += (int)XSTRLEN(names[i].name) + 2;
                    }
                    localList = (char*)XMALLOC(sz, ctx->heap,
                                                       DYNAMIC_TYPE_TMP_BUFFER);
                    if (localList == NULL) {
                        return WOLFSSL_FAILURE;
                    }
                    wolfSSL_get_ciphers(localList, sz);
                    sz = (int)XSTRLEN(localList);
                }

                if (XSTRSTR(name, "eNULL")) {
                    wolfSSL_remove_ciphers(localList, sz, "-NULL");
                }
            }
        }
        while (next++); /* ++ needed to skip ':' */

        ret = SetCipherList(ctx, suites, localList);
        XFREE(localList, ctx->heap, DYNAMIC_TYPE_TMP_BUFFER);
        return (ret)? WOLFSSL_SUCCESS : WOLFSSL_FAILURE;
    }
    else {

        listattribute = CheckcipherList(list);

        if (listattribute == 0) {
           /* list has mixed(pre-TLSv13 and TLSv13) suites
            * update cipher suites the same as before
            */
            return (SetCipherList(ctx, suites, list)) ? WOLFSSL_SUCCESS :
            WOLFSSL_FAILURE;
        }
        else if (listattribute == 1) {
           /* list has only pre-TLSv13 suites.
            * Only update before TLSv13 suites.
            */
            tls13Only = 1;
        }
        else if (listattribute == 2) {
           /* list has only TLSv13 suites. Only update TLv13 suites
            * simulate set_ciphersuites() compatibility layer API
            */
            tls13Only = 0;
        }

        buildcipherList = buildEnabledCipherList(ctx, ctx->suites,
                                                tls13Only, list);

        if (buildcipherList) {
            ret = SetCipherList(ctx, suites, buildcipherList);
            XFREE(buildcipherList, ctx->heap, DYNAMIC_TYPE_TMP_BUFFER);
        }
        else {
            ret = SetCipherList(ctx, suites, list);
        }

        return ret;
    }
}

#endif


int wolfSSL_CTX_set_cipher_list(WOLFSSL_CTX* ctx, const char* list)
{
    WOLFSSL_ENTER("wolfSSL_CTX_set_cipher_list");

    if (ctx == NULL)
        return WOLFSSL_FAILURE;

    /* alloc/init on demand only */
    if (ctx->suites == NULL) {
        ctx->suites = (Suites*)XMALLOC(sizeof(Suites), ctx->heap,
                                       DYNAMIC_TYPE_SUITES);
        if (ctx->suites == NULL) {
            WOLFSSL_MSG("Memory alloc for Suites failed");
            return WOLFSSL_FAILURE;
        }
        XMEMSET(ctx->suites, 0, sizeof(Suites));
    }

#ifdef OPENSSL_EXTRA
    return wolfSSL_parse_cipher_list(ctx, ctx->suites, list);
#else
    return (SetCipherList(ctx, ctx->suites, list)) ? WOLFSSL_SUCCESS : WOLFSSL_FAILURE;
#endif
}


int wolfSSL_set_cipher_list(WOLFSSL* ssl, const char* list)
{
    WOLFSSL_ENTER("wolfSSL_set_cipher_list");
#ifdef SINGLE_THREADED
    if (ssl->ctx->suites == ssl->suites) {
        ssl->suites = (Suites*)XMALLOC(sizeof(Suites), ssl->heap,
                                       DYNAMIC_TYPE_SUITES);
        if (ssl->suites == NULL) {
            WOLFSSL_MSG("Suites Memory error");
            return MEMORY_E;
        }
        *ssl->suites = *ssl->ctx->suites;
        ssl->options.ownSuites = 1;
    }
#endif

#ifdef OPENSSL_EXTRA
    return wolfSSL_parse_cipher_list(ssl->ctx, ssl->suites, list);
#else
    return (SetCipherList(ssl->ctx, ssl->suites, list)) ? WOLFSSL_SUCCESS : WOLFSSL_FAILURE;
#endif
}

#ifdef HAVE_KEYING_MATERIAL

#define TLS_PRF_LABEL_CLIENT_FINISHED     "client finished"
#define TLS_PRF_LABEL_SERVER_FINISHED     "server finished"
#define TLS_PRF_LABEL_MASTER_SECRET       "master secret"
#define TLS_PRF_LABEL_EXT_MASTER_SECRET   "extended master secret"
#define TLS_PRF_LABEL_KEY_EXPANSION       "key expansion"

static const struct ForbiddenLabels {
    const char* label;
    size_t labelLen;
} forbiddenLabels[] = {
    {TLS_PRF_LABEL_CLIENT_FINISHED, XSTR_SIZEOF(TLS_PRF_LABEL_CLIENT_FINISHED)},
    {TLS_PRF_LABEL_SERVER_FINISHED, XSTR_SIZEOF(TLS_PRF_LABEL_SERVER_FINISHED)},
    {TLS_PRF_LABEL_MASTER_SECRET, XSTR_SIZEOF(TLS_PRF_LABEL_MASTER_SECRET)},
    {TLS_PRF_LABEL_EXT_MASTER_SECRET, XSTR_SIZEOF(TLS_PRF_LABEL_EXT_MASTER_SECRET)},
    {TLS_PRF_LABEL_KEY_EXPANSION, XSTR_SIZEOF(TLS_PRF_LABEL_KEY_EXPANSION)},
    {NULL, 0},
};

/**
 * Implement RFC 5705
 * TLS 1.3 uses a different exporter definition (section 7.5 of RFC 8446)
 * @return WOLFSSL_SUCCESS on success and WOLFSSL_FAILURE on error
 */
int wolfSSL_export_keying_material(WOLFSSL *ssl,
        unsigned char *out, size_t outLen,
        const char *label, size_t labelLen,
        const unsigned char *context, size_t contextLen,
        int use_context)
{
    byte*  seed = NULL;
    word32 seedLen;
    const struct ForbiddenLabels* fl;

    WOLFSSL_ENTER("wolfSSL_export_keying_material");

    if (ssl == NULL || out == NULL || label == NULL ||
            (use_context && contextLen && context == NULL)) {
        WOLFSSL_MSG("Bad argument");
        return WOLFSSL_FAILURE;
    }

    /* clientRandom + serverRandom
     * OR
     * clientRandom + serverRandom + ctx len encoding + ctx */
    seedLen = !use_context ? (word32)SEED_LEN :
                             (word32)SEED_LEN + 2 + (word32)contextLen;

    if (ssl->options.saveArrays == 0 || ssl->arrays == NULL) {
        WOLFSSL_MSG("To export keying material wolfSSL needs to keep handshake "
                    "data. Call wolfSSL_KeepArrays before attempting to "
                    "export keyid material.");
        return WOLFSSL_FAILURE;
    }

    /* check forbidden labels */
    for (fl = &forbiddenLabels[0]; fl->label != NULL; fl++) {
        if (labelLen >= fl->labelLen &&
                XMEMCMP(label, fl->label, fl->labelLen) == 0) {
            WOLFSSL_MSG("Forbidden label");
            return WOLFSSL_FAILURE;
        }
    }

#ifdef WOLFSSL_TLS13
    if (IsAtLeastTLSv1_3(ssl->version)) {
        /* Path for TLS 1.3 */
        if (!use_context) {
            contextLen = 0;
            context = (byte*)""; /* Give valid pointer for 0 length memcpy */
        }

        if (Tls13_Exporter(ssl, out, (word32)outLen, label, labelLen,
                context, contextLen) != 0) {
            WOLFSSL_MSG("Tls13_Exporter error");
            return WOLFSSL_FAILURE;
        }
        return WOLFSSL_SUCCESS;
    }
#endif

    /* Path for <=TLS 1.2 */
    seed = (byte*)XMALLOC(seedLen, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    if (seed == NULL) {
        WOLFSSL_MSG("malloc error");
        return WOLFSSL_FAILURE;
    }

    XMEMCPY(seed,           ssl->arrays->clientRandom, RAN_LEN);
    XMEMCPY(seed + RAN_LEN, ssl->arrays->serverRandom, RAN_LEN);

    if (use_context) {
        /* Encode len in big endian */
        seed[SEED_LEN    ] = (contextLen >> 8) & 0xFF;
        seed[SEED_LEN + 1] = (contextLen) & 0xFF;
        if (contextLen) {
            /* 0 length context is allowed */
            XMEMCPY(seed + SEED_LEN + 2, context, contextLen);
        }
    }

    PRIVATE_KEY_UNLOCK();
    if (wc_PRF_TLS(out, (word32)outLen, ssl->arrays->masterSecret, SECRET_LEN,
            (byte*)label, (word32)labelLen, seed, seedLen, IsAtLeastTLSv1_2(ssl),
            ssl->specs.mac_algorithm, ssl->heap, ssl->devId) != 0) {
        WOLFSSL_MSG("wc_PRF_TLS error");
        PRIVATE_KEY_LOCK();
        XFREE(seed, NULL, DYNAMIC_TYPE_TMP_BUFFER);
        return WOLFSSL_FAILURE;
    }
    PRIVATE_KEY_LOCK();

    XFREE(seed, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    return WOLFSSL_SUCCESS;
}
#endif /* HAVE_KEYING_MATERIAL */

int wolfSSL_dtls_get_using_nonblock(WOLFSSL* ssl)
{
    int useNb = 0;

    if (ssl == NULL)
        return WOLFSSL_FAILURE;

    WOLFSSL_ENTER("wolfSSL_dtls_get_using_nonblock");
    if (ssl->options.dtls) {
#ifdef WOLFSSL_DTLS
        useNb = ssl->options.dtlsUseNonblock;
#endif
    }
    else {
        WOLFSSL_MSG("wolfSSL_dtls_get_using_nonblock() is "
                    "DEPRECATED for non-DTLS use.");
    }
    return useNb;
}


#ifndef WOLFSSL_LEANPSK

void wolfSSL_dtls_set_using_nonblock(WOLFSSL* ssl, int nonblock)
{
    (void)nonblock;

    WOLFSSL_ENTER("wolfSSL_dtls_set_using_nonblock");

    if (ssl == NULL)
        return;

    if (ssl->options.dtls) {
#ifdef WOLFSSL_DTLS
        ssl->options.dtlsUseNonblock = (nonblock != 0);
#endif
    }
    else {
        WOLFSSL_MSG("wolfSSL_dtls_set_using_nonblock() is "
                    "DEPRECATED for non-DTLS use.");
    }
}


#ifdef WOLFSSL_DTLS

int wolfSSL_dtls_get_current_timeout(WOLFSSL* ssl)
{
    int timeout = 0;
    if (ssl)
        timeout = ssl->dtls_timeout;

    WOLFSSL_LEAVE("wolfSSL_dtls_get_current_timeout()", timeout);
    return timeout;
}

int wolfSSL_DTLSv1_get_timeout(WOLFSSL* ssl, WOLFSSL_TIMEVAL* timeleft)
{
    if (ssl && timeleft) {
        XMEMSET(timeleft, 0, sizeof(WOLFSSL_TIMEVAL));
        timeleft->tv_sec = ssl->dtls_timeout;
    }
    return 0;
}

#ifndef NO_WOLFSSL_STUB
int wolfSSL_DTLSv1_handle_timeout(WOLFSSL* ssl)
{
    WOLFSSL_STUB("SSL_DTLSv1_handle_timeout");
    (void)ssl;
    return 0;
}
#endif

#ifndef NO_WOLFSSL_STUB
void wolfSSL_DTLSv1_set_initial_timeout_duration(WOLFSSL* ssl, word32 duration_ms)
{
    WOLFSSL_STUB("SSL_DTLSv1_set_initial_timeout_duration");
    (void)ssl;
    (void)duration_ms;
}
#endif

/* user may need to alter init dtls recv timeout, WOLFSSL_SUCCESS on ok */
int wolfSSL_dtls_set_timeout_init(WOLFSSL* ssl, int timeout)
{
    if (ssl == NULL || timeout < 0)
        return BAD_FUNC_ARG;

    if (timeout > ssl->dtls_timeout_max) {
        WOLFSSL_MSG("Can't set dtls timeout init greater than dtls timeout max");
        return BAD_FUNC_ARG;
    }

    ssl->dtls_timeout_init = timeout;
    ssl->dtls_timeout = timeout;

    return WOLFSSL_SUCCESS;
}


/* user may need to alter max dtls recv timeout, WOLFSSL_SUCCESS on ok */
int wolfSSL_dtls_set_timeout_max(WOLFSSL* ssl, int timeout)
{
    if (ssl == NULL || timeout < 0)
        return BAD_FUNC_ARG;

    if (timeout < ssl->dtls_timeout_init) {
        WOLFSSL_MSG("Can't set dtls timeout max less than dtls timeout init");
        return BAD_FUNC_ARG;
    }

    ssl->dtls_timeout_max = timeout;

    return WOLFSSL_SUCCESS;
}


int wolfSSL_dtls_got_timeout(WOLFSSL* ssl)
{
    int result = WOLFSSL_SUCCESS;
    WOLFSSL_ENTER("wolfSSL_dtls_got_timeout()");

    if (ssl == NULL)
        return WOLFSSL_FATAL_ERROR;

    if ((IsSCR(ssl) || !ssl->options.handShakeDone)) {
        if (DtlsMsgPoolTimeout(ssl) < 0){
            ssl->error = SOCKET_ERROR_E;
            WOLFSSL_ERROR(ssl->error);
            result = WOLFSSL_FATAL_ERROR;
        }
        else if ((result = DtlsMsgPoolSend(ssl, 0)) < 0)  {
            ssl->error = result;
            WOLFSSL_ERROR(result);
            result = WOLFSSL_FATAL_ERROR;
        }
        else {
            /* Reset return value to success */
            result = WOLFSSL_SUCCESS;
        }
    }

    WOLFSSL_LEAVE("wolfSSL_dtls_got_timeout()", result);
    return result;
}


/* retransmit all the saves messages, WOLFSSL_SUCCESS on ok */
int wolfSSL_dtls_retransmit(WOLFSSL* ssl)
{
    WOLFSSL_ENTER("wolfSSL_dtls_retransmit()");

    if (ssl == NULL)
        return WOLFSSL_FATAL_ERROR;

    if (!ssl->options.handShakeDone) {
        int result = DtlsMsgPoolSend(ssl, 0);
        if (result < 0) {
            ssl->error = result;
            WOLFSSL_ERROR(result);
            return WOLFSSL_FATAL_ERROR;
        }
    }

    return 0;
}

#endif /* DTLS */
#endif /* LEANPSK */


#if defined(WOLFSSL_DTLS) && !defined(NO_WOLFSSL_SERVER)

/* Not an SSL function, return 0 for success, error code otherwise */
/* Prereq: ssl's RNG needs to be initialized. */
int wolfSSL_DTLS_SetCookieSecret(WOLFSSL* ssl,
                                 const byte* secret, word32 secretSz)
{
    int ret = 0;

    WOLFSSL_ENTER("wolfSSL_DTLS_SetCookieSecret");

    if (ssl == NULL) {
        WOLFSSL_MSG("need a SSL object");
        return BAD_FUNC_ARG;
    }

    if (secret != NULL && secretSz == 0) {
        WOLFSSL_MSG("can't have a new secret without a size");
        return BAD_FUNC_ARG;
    }

    /* If secretSz is 0, use the default size. */
    if (secretSz == 0)
        secretSz = COOKIE_SECRET_SZ;

    if (secretSz != ssl->buffers.dtlsCookieSecret.length) {
        byte* newSecret;

        if (ssl->buffers.dtlsCookieSecret.buffer != NULL) {
            ForceZero(ssl->buffers.dtlsCookieSecret.buffer,
                      ssl->buffers.dtlsCookieSecret.length);
            XFREE(ssl->buffers.dtlsCookieSecret.buffer,
                  ssl->heap, DYNAMIC_TYPE_NONE);
        }

        newSecret = (byte*)XMALLOC(secretSz, ssl->heap,DYNAMIC_TYPE_COOKIE_PWD);
        if (newSecret == NULL) {
            ssl->buffers.dtlsCookieSecret.buffer = NULL;
            ssl->buffers.dtlsCookieSecret.length = 0;
            WOLFSSL_MSG("couldn't allocate new cookie secret");
            return MEMORY_ERROR;
        }
        ssl->buffers.dtlsCookieSecret.buffer = newSecret;
        ssl->buffers.dtlsCookieSecret.length = secretSz;
    }

    /* If the supplied secret is NULL, randomly generate a new secret. */
    if (secret == NULL) {
        ret = wc_RNG_GenerateBlock(ssl->rng,
                             ssl->buffers.dtlsCookieSecret.buffer, secretSz);
    }
    else
        XMEMCPY(ssl->buffers.dtlsCookieSecret.buffer, secret, secretSz);

    WOLFSSL_LEAVE("wolfSSL_DTLS_SetCookieSecret", 0);
    return ret;
}

#endif /* WOLFSSL_DTLS && !NO_WOLFSSL_SERVER */


/* EITHER SIDE METHODS */
#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_EITHER_SIDE)
    WOLFSSL_METHOD* wolfSSLv23_method(void)
    {
        return wolfSSLv23_method_ex(NULL);
    }
    WOLFSSL_METHOD* wolfSSLv23_method_ex(void* heap)
    {
        WOLFSSL_METHOD* m = NULL;
        WOLFSSL_ENTER("SSLv23_method");
    #if !defined(NO_WOLFSSL_CLIENT)
        m = wolfSSLv23_client_method_ex(heap);
    #elif !defined(NO_WOLFSSL_SERVER)
        m = wolfSSLv23_server_method_ex(heap);
    #else
        (void)heap;
    #endif
        if (m != NULL) {
            m->side = WOLFSSL_NEITHER_END;
        }

        return m;
    }

    #ifdef WOLFSSL_ALLOW_SSLV3
    WOLFSSL_METHOD* wolfSSLv3_method(void)
    {
        return wolfSSLv3_method_ex(NULL);
    }
    WOLFSSL_METHOD* wolfSSLv3_method_ex(void* heap)
    {
        WOLFSSL_METHOD* m = NULL;
        WOLFSSL_ENTER("SSLv3_method");
    #if !defined(NO_WOLFSSL_CLIENT)
        m = wolfSSLv3_client_method_ex(heap);
    #elif !defined(NO_WOLFSSL_SERVER)
        m = wolfSSLv3_server_method_ex(heap);
    #endif
        if (m != NULL) {
            m->side = WOLFSSL_NEITHER_END;
        }

        return m;
    }
    #endif
#endif /* OPENSSL_EXTRA || WOLFSSL_EITHER_SIDE */

/* client only parts */
#ifndef NO_WOLFSSL_CLIENT

    #ifdef OPENSSL_EXTRA
    WOLFSSL_METHOD* wolfSSLv2_client_method(void)
    {
        WOLFSSL_STUB("wolfSSLv2_client_method");
        return NULL;
    }
    #endif

    #if defined(WOLFSSL_ALLOW_SSLV3) && !defined(NO_OLD_TLS)
    WOLFSSL_METHOD* wolfSSLv3_client_method(void)
    {
        return wolfSSLv3_client_method_ex(NULL);
    }
    WOLFSSL_METHOD* wolfSSLv3_client_method_ex(void* heap)
    {
        WOLFSSL_METHOD* method =
                              (WOLFSSL_METHOD*) XMALLOC(sizeof(WOLFSSL_METHOD),
                                                     heap, DYNAMIC_TYPE_METHOD);
        (void)heap;
        WOLFSSL_ENTER("SSLv3_client_method_ex");
        if (method)
            InitSSL_Method(method, MakeSSLv3());
        return method;
    }
    #endif /* WOLFSSL_ALLOW_SSLV3 && !NO_OLD_TLS */


    WOLFSSL_METHOD* wolfSSLv23_client_method(void)
    {
        return wolfSSLv23_client_method_ex(NULL);
    }
    WOLFSSL_METHOD* wolfSSLv23_client_method_ex(void* heap)
    {
        WOLFSSL_METHOD* method =
                              (WOLFSSL_METHOD*) XMALLOC(sizeof(WOLFSSL_METHOD),
                                                     heap, DYNAMIC_TYPE_METHOD);
        (void)heap;
        WOLFSSL_ENTER("SSLv23_client_method_ex");
        if (method) {
    #if !defined(NO_SHA256) || defined(WOLFSSL_SHA384) || defined(WOLFSSL_SHA512)
        #if defined(WOLFSSL_TLS13)
            InitSSL_Method(method, MakeTLSv1_3());
        #elif !defined(WOLFSSL_NO_TLS12)
            InitSSL_Method(method, MakeTLSv1_2());
        #elif !defined(NO_OLD_TLS)
            InitSSL_Method(method, MakeTLSv1_1());
        #endif
    #else
        #ifndef NO_OLD_TLS
            InitSSL_Method(method, MakeTLSv1_1());
        #endif
    #endif
    #if !defined(NO_OLD_TLS) || defined(WOLFSSL_TLS13)
            method->downgrade = 1;
    #endif
        }
        return method;
    }

    /* please see note at top of README if you get an error from connect */
    WOLFSSL_ABI
    int wolfSSL_connect(WOLFSSL* ssl)
    {
    #if !(defined(WOLFSSL_NO_TLS12) && defined(NO_OLD_TLS) && defined(WOLFSSL_TLS13))
        int neededState;
    #endif

        WOLFSSL_ENTER("SSL_connect()");

        #ifdef HAVE_ERRNO_H
            errno = 0;
        #endif

        if (ssl == NULL)
            return BAD_FUNC_ARG;

    #if defined(OPENSSL_EXTRA) || defined(WOLFSSL_EITHER_SIDE)
        if (ssl->options.side == WOLFSSL_NEITHER_END) {
            ssl->error = InitSSL_Side(ssl, WOLFSSL_CLIENT_END);
            if (ssl->error != WOLFSSL_SUCCESS) {
                WOLFSSL_ERROR(ssl->error);
                return WOLFSSL_FATAL_ERROR;
            }
            ssl->error = 0; /* expected to be zero here */
        }

    #ifdef OPENSSL_EXTRA
        if (ssl->CBIS != NULL) {
            ssl->CBIS(ssl, SSL_ST_CONNECT, WOLFSSL_SUCCESS);
            ssl->cbmode = SSL_CB_WRITE;
        }
    #endif
    #endif /* OPENSSL_EXTRA || WOLFSSL_EITHER_SIDE */

    #if defined(WOLFSSL_NO_TLS12) && defined(NO_OLD_TLS) && defined(WOLFSSL_TLS13)
        return wolfSSL_connect_TLSv13(ssl);
    #else
        #ifdef WOLFSSL_TLS13
        if (ssl->options.tls1_3)
            return wolfSSL_connect_TLSv13(ssl);
        #endif

#ifdef WOLFSSL_WOLFSENTRY_HOOKS
        if (ssl->ConnectFilter) {
            wolfSSL_netfilter_decision_t res;
            if ((ssl->ConnectFilter(ssl, ssl->ConnectFilter_arg, &res) ==
                 WOLFSSL_SUCCESS) &&
                (res == WOLFSSL_NETFILTER_REJECT)) {
                WOLFSSL_ERROR(ssl->error = SOCKET_FILTERED_E);
                return WOLFSSL_FATAL_ERROR;
            }
        }
#endif /* WOLFSSL_WOLFSENTRY_HOOKS */

        if (ssl->options.side != WOLFSSL_CLIENT_END) {
            WOLFSSL_ERROR(ssl->error = SIDE_ERROR);
            return WOLFSSL_FATAL_ERROR;
        }

        #ifdef WOLFSSL_DTLS
        if (ssl->version.major == DTLS_MAJOR) {
            ssl->options.dtls   = 1;
            ssl->options.tls    = 1;
            ssl->options.tls1_1 = 1;
        }
        #endif

        if (ssl->buffers.outputBuffer.length > 0
        #ifdef WOLFSSL_ASYNC_CRYPT
            /* do not send buffered or advance state if last error was an
                async pending operation */
            && ssl->error != WC_PENDING_E
        #endif
        ) {
            if ( (ssl->error = SendBuffered(ssl)) == 0) {
                /* fragOffset is non-zero when sending fragments. On the last
                 * fragment, fragOffset is zero again, and the state can be
                 * advanced. */
                if (ssl->fragOffset == 0) {
                    ssl->options.connectState++;
                    WOLFSSL_MSG("connect state: "
                                "Advanced from last buffered fragment send");
                }
                else {
                    WOLFSSL_MSG("connect state: "
                                "Not advanced, more fragments to send");
                }
            }
            else {
                WOLFSSL_ERROR(ssl->error);
                return WOLFSSL_FATAL_ERROR;
            }
        }

        switch (ssl->options.connectState) {

        case CONNECT_BEGIN :
            /* always send client hello first */
            if ( (ssl->error = SendClientHello(ssl)) != 0) {
                WOLFSSL_ERROR(ssl->error);
                return WOLFSSL_FATAL_ERROR;
            }
            ssl->options.connectState = CLIENT_HELLO_SENT;
            WOLFSSL_MSG("connect state: CLIENT_HELLO_SENT");
            FALL_THROUGH;

        case CLIENT_HELLO_SENT :
            neededState = ssl->options.resuming ? SERVER_FINISHED_COMPLETE :
                                          SERVER_HELLODONE_COMPLETE;
            #ifdef WOLFSSL_DTLS
                /* In DTLS, when resuming, we can go straight to FINISHED,
                 * or do a cookie exchange and then skip to FINISHED, assume
                 * we need the cookie exchange first. */
                if (IsDtlsNotSctpMode(ssl))
                    neededState = SERVER_HELLOVERIFYREQUEST_COMPLETE;
            #endif
            /* get response */
            while (ssl->options.serverState < neededState) {
                #ifdef WOLFSSL_TLS13
                    if (ssl->options.tls1_3)
                        return wolfSSL_connect_TLSv13(ssl);
                #endif
                if ( (ssl->error = ProcessReply(ssl)) < 0) {
                    WOLFSSL_ERROR(ssl->error);
                    return WOLFSSL_FATAL_ERROR;
                }
                /* if resumption failed, reset needed state */
                else if (neededState == SERVER_FINISHED_COMPLETE)
                    if (!ssl->options.resuming) {
                    #ifdef WOLFSSL_DTLS
                        if (IsDtlsNotSctpMode(ssl))
                            neededState = SERVER_HELLOVERIFYREQUEST_COMPLETE;
                        else
                    #endif
                            neededState = SERVER_HELLODONE_COMPLETE;
                    }
            }

            ssl->options.connectState = HELLO_AGAIN;
            WOLFSSL_MSG("connect state: HELLO_AGAIN");
            FALL_THROUGH;

        case HELLO_AGAIN :
            if (ssl->options.certOnly)
                return WOLFSSL_SUCCESS;

        #ifdef WOLFSSL_TLS13
            if (ssl->options.tls1_3)
                return wolfSSL_connect_TLSv13(ssl);
        #endif

            #ifdef WOLFSSL_DTLS
            if (ssl->options.serverState ==
                    SERVER_HELLOVERIFYREQUEST_COMPLETE) {
                if (IsDtlsNotSctpMode(ssl)) {
                    /* re-init hashes, exclude first hello and verify request */
                    if ((ssl->error = InitHandshakeHashes(ssl)) != 0) {
                        WOLFSSL_ERROR(ssl->error);
                        return WOLFSSL_FATAL_ERROR;
                    }
                    if ( (ssl->error = SendClientHello(ssl)) != 0) {
                        WOLFSSL_ERROR(ssl->error);
                        return WOLFSSL_FATAL_ERROR;
                    }
                }
            }
            #endif

            ssl->options.connectState = HELLO_AGAIN_REPLY;
            WOLFSSL_MSG("connect state: HELLO_AGAIN_REPLY");
            FALL_THROUGH;

        case HELLO_AGAIN_REPLY :
            #ifdef WOLFSSL_DTLS
                if (IsDtlsNotSctpMode(ssl)) {
                    neededState = ssl->options.resuming ?
                           SERVER_FINISHED_COMPLETE : SERVER_HELLODONE_COMPLETE;

                    /* get response */
                    while (ssl->options.serverState < neededState) {
                        if ( (ssl->error = ProcessReply(ssl)) < 0) {
                            WOLFSSL_ERROR(ssl->error);
                            return WOLFSSL_FATAL_ERROR;
                        }
                        /* if resumption failed, reset needed state */
                        if (neededState == SERVER_FINISHED_COMPLETE) {
                            if (!ssl->options.resuming)
                                neededState = SERVER_HELLODONE_COMPLETE;
                        }
                    }
                }
            #endif

            ssl->options.connectState = FIRST_REPLY_DONE;
            WOLFSSL_MSG("connect state: FIRST_REPLY_DONE");
            FALL_THROUGH;

        case FIRST_REPLY_DONE :
            #if !defined(NO_CERTS) && !defined(WOLFSSL_NO_CLIENT_AUTH)
                #ifdef WOLFSSL_TLS13
                    if (ssl->options.tls1_3)
                        return wolfSSL_connect_TLSv13(ssl);
                #endif
                if (ssl->options.sendVerify) {
                    if ( (ssl->error = SendCertificate(ssl)) != 0) {
                    #ifdef WOLFSSL_CHECK_ALERT_ON_ERR
                        ProcessReplyEx(ssl, 1); /* See if an alert was sent. */
                    #endif
                        WOLFSSL_ERROR(ssl->error);
                        return WOLFSSL_FATAL_ERROR;
                    }
                    WOLFSSL_MSG("sent: certificate");
                }

            #endif
            ssl->options.connectState = FIRST_REPLY_FIRST;
            WOLFSSL_MSG("connect state: FIRST_REPLY_FIRST");
            FALL_THROUGH;

        case FIRST_REPLY_FIRST :
        #ifdef WOLFSSL_TLS13
            if (ssl->options.tls1_3)
                return wolfSSL_connect_TLSv13(ssl);
        #endif
            if (!ssl->options.resuming) {
                if ( (ssl->error = SendClientKeyExchange(ssl)) != 0) {
                #ifdef WOLFSSL_CHECK_ALERT_ON_ERR
                    ProcessReplyEx(ssl, 1); /* See if an alert was sent. */
                #endif
                    WOLFSSL_ERROR(ssl->error);
                    return WOLFSSL_FATAL_ERROR;
                }
                WOLFSSL_MSG("sent: client key exchange");
            }

            ssl->options.connectState = FIRST_REPLY_SECOND;
            WOLFSSL_MSG("connect state: FIRST_REPLY_SECOND");
            FALL_THROUGH;

    #if !defined(WOLFSSL_NO_TLS12) || !defined(NO_OLD_TLS)
        case FIRST_REPLY_SECOND :
            /* CLIENT: Fail-safe for Server Authentication. */
            if (!ssl->options.peerAuthGood) {
                WOLFSSL_MSG("Server authentication did not happen");
                return WOLFSSL_FATAL_ERROR;
            }

            #if !defined(NO_CERTS) && !defined(WOLFSSL_NO_CLIENT_AUTH)
                if (ssl->options.sendVerify) {
                    if ( (ssl->error = SendCertificateVerify(ssl)) != 0) {
                    #ifdef WOLFSSL_CHECK_ALERT_ON_ERR
                        ProcessReplyEx(ssl, 1); /* See if an alert was sent. */
                    #endif
                        WOLFSSL_ERROR(ssl->error);
                        return WOLFSSL_FATAL_ERROR;
                    }
                    WOLFSSL_MSG("sent: certificate verify");
                }
            #endif /* !NO_CERTS && !WOLFSSL_NO_CLIENT_AUTH */
            ssl->options.connectState = FIRST_REPLY_THIRD;
            WOLFSSL_MSG("connect state: FIRST_REPLY_THIRD");
            FALL_THROUGH;

        case FIRST_REPLY_THIRD :
            if ( (ssl->error = SendChangeCipher(ssl)) != 0) {
            #ifdef WOLFSSL_CHECK_ALERT_ON_ERR
                ProcessReplyEx(ssl, 1); /* See if an alert was sent. */
            #endif
                WOLFSSL_ERROR(ssl->error);
                return WOLFSSL_FATAL_ERROR;
            }
            WOLFSSL_MSG("sent: change cipher spec");
            ssl->options.connectState = FIRST_REPLY_FOURTH;
            WOLFSSL_MSG("connect state: FIRST_REPLY_FOURTH");
            FALL_THROUGH;

        case FIRST_REPLY_FOURTH :
            if ( (ssl->error = SendFinished(ssl)) != 0) {
            #ifdef WOLFSSL_CHECK_ALERT_ON_ERR
                ProcessReplyEx(ssl, 1); /* See if an alert was sent. */
            #endif
                WOLFSSL_ERROR(ssl->error);
                return WOLFSSL_FATAL_ERROR;
            }
            WOLFSSL_MSG("sent: finished");
            ssl->options.connectState = FINISHED_DONE;
            WOLFSSL_MSG("connect state: FINISHED_DONE");
            FALL_THROUGH;

        case FINISHED_DONE :
            /* get response */
            while (ssl->options.serverState < SERVER_FINISHED_COMPLETE)
                if ( (ssl->error = ProcessReply(ssl)) < 0) {
                    WOLFSSL_ERROR(ssl->error);
                    return WOLFSSL_FATAL_ERROR;
                }

            ssl->options.connectState = SECOND_REPLY_DONE;
            WOLFSSL_MSG("connect state: SECOND_REPLY_DONE");
            FALL_THROUGH;

        case SECOND_REPLY_DONE:
        #ifndef NO_HANDSHAKE_DONE_CB
            if (ssl->hsDoneCb) {
                int cbret = ssl->hsDoneCb(ssl, ssl->hsDoneCtx);
                if (cbret < 0) {
                    ssl->error = cbret;
                    WOLFSSL_MSG("HandShake Done Cb don't continue error");
                    return WOLFSSL_FATAL_ERROR;
                }
            }
        #endif /* NO_HANDSHAKE_DONE_CB */

            if (!ssl->options.dtls) {
                if (!ssl->options.keepResources) {
                    FreeHandshakeResources(ssl);
                }
            }
        #ifdef WOLFSSL_DTLS
            else {
                ssl->options.dtlsHsRetain = 1;
            }
        #endif /* WOLFSSL_DTLS */

        #if defined(WOLFSSL_ASYNC_CRYPT) && defined(HAVE_SECURE_RENEGOTIATION)
            /* This may be necessary in async so that we don't try to
             * renegotiate again */
            if (ssl->secure_renegotiation && ssl->secure_renegotiation->startScr) {
                ssl->secure_renegotiation->startScr = 0;
            }
        #endif /* WOLFSSL_ASYNC_CRYPT && HAVE_SECURE_RENEGOTIATION */

            WOLFSSL_LEAVE("SSL_connect()", WOLFSSL_SUCCESS);
            return WOLFSSL_SUCCESS;
    #endif /* !WOLFSSL_NO_TLS12 || !NO_OLD_TLS */

        default:
            WOLFSSL_MSG("Unknown connect state ERROR");
            return WOLFSSL_FATAL_ERROR; /* unknown connect state */
        }
    #endif /* !WOLFSSL_NO_TLS12 || !NO_OLD_TLS || !WOLFSSL_TLS13 */
    }

#endif /* NO_WOLFSSL_CLIENT */


/* server only parts */
#ifndef NO_WOLFSSL_SERVER

    #ifdef OPENSSL_EXTRA
    WOLFSSL_METHOD* wolfSSLv2_server_method(void)
    {
        WOLFSSL_STUB("wolfSSLv2_server_method");
        return 0;
    }
    #endif

    #if defined(WOLFSSL_ALLOW_SSLV3) && !defined(NO_OLD_TLS)
    WOLFSSL_METHOD* wolfSSLv3_server_method(void)
    {
        return wolfSSLv3_server_method_ex(NULL);
    }
    WOLFSSL_METHOD* wolfSSLv3_server_method_ex(void* heap)
    {
        WOLFSSL_METHOD* method =
                              (WOLFSSL_METHOD*) XMALLOC(sizeof(WOLFSSL_METHOD),
                                                     heap, DYNAMIC_TYPE_METHOD);
        (void)heap;
        WOLFSSL_ENTER("SSLv3_server_method_ex");
        if (method) {
            InitSSL_Method(method, MakeSSLv3());
            method->side = WOLFSSL_SERVER_END;
        }
        return method;
    }
    #endif /* WOLFSSL_ALLOW_SSLV3 && !NO_OLD_TLS */

    WOLFSSL_METHOD* wolfSSLv23_server_method(void)
    {
        return wolfSSLv23_server_method_ex(NULL);
    }

    WOLFSSL_METHOD* wolfSSLv23_server_method_ex(void* heap)
    {
        WOLFSSL_METHOD* method =
                              (WOLFSSL_METHOD*) XMALLOC(sizeof(WOLFSSL_METHOD),
                                                     heap, DYNAMIC_TYPE_METHOD);
        (void)heap;
        WOLFSSL_ENTER("SSLv23_server_method_ex");
        if (method) {
    #if !defined(NO_SHA256) || defined(WOLFSSL_SHA384) || defined(WOLFSSL_SHA512)
        #ifdef WOLFSSL_TLS13
            InitSSL_Method(method, MakeTLSv1_3());
        #elif !defined(WOLFSSL_NO_TLS12)
            InitSSL_Method(method, MakeTLSv1_2());
        #elif !defined(NO_OLD_TLS)
            InitSSL_Method(method, MakeTLSv1_1());
        #endif
    #else
        #ifndef NO_OLD_TLS
            InitSSL_Method(method, MakeTLSv1_1());
        #else
            #error Must have SHA256, SHA384 or SHA512 enabled for TLS 1.2
        #endif
    #endif
    #if !defined(NO_OLD_TLS) || defined(WOLFSSL_TLS13)
            method->downgrade = 1;
    #endif
            method->side      = WOLFSSL_SERVER_END;
        }
        return method;
    }


    WOLFSSL_ABI
    int wolfSSL_accept(WOLFSSL* ssl)
    {
#if !(defined(WOLFSSL_NO_TLS12) && defined(NO_OLD_TLS) && defined(WOLFSSL_TLS13))
        word16 havePSK = 0;
        word16 haveAnon = 0;
        word16 haveMcast = 0;
#endif

        if (ssl == NULL)
            return WOLFSSL_FATAL_ERROR;

    #if defined(OPENSSL_EXTRA) || defined(WOLFSSL_EITHER_SIDE)
        if (ssl->options.side == WOLFSSL_NEITHER_END) {
            WOLFSSL_MSG("Setting WOLFSSL_SSL to be server side");
            ssl->error = InitSSL_Side(ssl, WOLFSSL_SERVER_END);
            if (ssl->error != WOLFSSL_SUCCESS) {
                WOLFSSL_ERROR(ssl->error);
                return WOLFSSL_FATAL_ERROR;
            }
            ssl->error = 0; /* expected to be zero here */
        }
    #endif /* OPENSSL_EXTRA || WOLFSSL_EITHER_SIDE */

#if defined(WOLFSSL_NO_TLS12) && defined(NO_OLD_TLS) && defined(WOLFSSL_TLS13)
        return wolfSSL_accept_TLSv13(ssl);
#else
    #ifdef WOLFSSL_TLS13
        if (ssl->options.tls1_3)
            return wolfSSL_accept_TLSv13(ssl);
    #endif
        WOLFSSL_ENTER("SSL_accept()");

#ifdef WOLFSSL_WOLFSENTRY_HOOKS
        if (ssl->AcceptFilter) {
            wolfSSL_netfilter_decision_t res;
            if ((ssl->AcceptFilter(ssl, ssl->AcceptFilter_arg, &res) ==
                 WOLFSSL_SUCCESS) &&
                (res == WOLFSSL_NETFILTER_REJECT)) {
                WOLFSSL_ERROR(ssl->error = SOCKET_FILTERED_E);
                return WOLFSSL_FATAL_ERROR;
            }
        }
#endif /* WOLFSSL_WOLFSENTRY_HOOKS */

        #ifdef HAVE_ERRNO_H
            errno = 0;
        #endif

        #ifndef NO_PSK
            havePSK = ssl->options.havePSK;
        #endif
        (void)havePSK;

        #ifdef HAVE_ANON
            haveAnon = ssl->options.haveAnon;
        #endif
        (void)haveAnon;

        #ifdef WOLFSSL_MULTICAST
            haveMcast = ssl->options.haveMcast;
        #endif
        (void)haveMcast;

        if (ssl->options.side != WOLFSSL_SERVER_END) {
            WOLFSSL_ERROR(ssl->error = SIDE_ERROR);
            return WOLFSSL_FATAL_ERROR;
        }

    #ifndef NO_CERTS
        /* in case used set_accept_state after init */
        if (!havePSK && !haveAnon && !haveMcast) {
        #ifdef OPENSSL_EXTRA
            if (ssl->ctx->certSetupCb != NULL) {
                WOLFSSL_MSG("CertSetupCb set. server cert and "
                            "key not checked");
            }
            else
        #endif
            {
                if (!ssl->buffers.certificate ||
                    !ssl->buffers.certificate->buffer) {

                    WOLFSSL_MSG("accept error: server cert required");
                    ssl->error = NO_PRIVATE_KEY;
                    WOLFSSL_ERROR(ssl->error);
                    return WOLFSSL_FATAL_ERROR;
                }

                if (!ssl->buffers.key || !ssl->buffers.key->buffer) {
                    /* allow no private key if using existing key */
                #ifdef WOLF_PRIVATE_KEY_ID
                    if (ssl->devId != INVALID_DEVID
                    #ifdef HAVE_PK_CALLBACKS
                        || wolfSSL_CTX_IsPrivatePkSet(ssl->ctx)
                    #endif
                    ) {
                        WOLFSSL_MSG("Allowing no server private key "
                                    "(external)");
                    }
                    else
                #endif
                    {
                        WOLFSSL_MSG("accept error: server key required");
                        WOLFSSL_ERROR(ssl->error = NO_PRIVATE_KEY);
                        return WOLFSSL_FATAL_ERROR;
                    }
                }
            }
        }
    #endif

    #ifdef WOLFSSL_DTLS
        if (ssl->version.major == DTLS_MAJOR) {
            ssl->options.dtls   = 1;
            ssl->options.tls    = 1;
            ssl->options.tls1_1 = 1;
        }
    #endif

        if (ssl->buffers.outputBuffer.length > 0
        #ifdef WOLFSSL_ASYNC_CRYPT
            /* do not send buffered or advance state if last error was an
                async pending operation */
            && ssl->error != WC_PENDING_E
        #endif
        ) {
            if ( (ssl->error = SendBuffered(ssl)) == 0) {
                /* fragOffset is non-zero when sending fragments. On the last
                 * fragment, fragOffset is zero again, and the state can be
                 * advanced. */
                if (ssl->fragOffset == 0) {
                    ssl->options.acceptState++;
                    WOLFSSL_MSG("accept state: "
                                "Advanced from last buffered fragment send");
                }
                else {
                    WOLFSSL_MSG("accept state: "
                                "Not advanced, more fragments to send");
                }
            }
            else {
                WOLFSSL_ERROR(ssl->error);
                return WOLFSSL_FATAL_ERROR;
            }
        }

        switch (ssl->options.acceptState) {

        case ACCEPT_BEGIN :
#ifdef HAVE_SECURE_RENEGOTIATION
        case ACCEPT_BEGIN_RENEG:
#endif
            /* get response */
            while (ssl->options.clientState < CLIENT_HELLO_COMPLETE)
                if ( (ssl->error = ProcessReply(ssl)) < 0) {
                    WOLFSSL_ERROR(ssl->error);
                    return WOLFSSL_FATAL_ERROR;
                }
#ifdef WOLFSSL_TLS13
            ssl->options.acceptState = ACCEPT_CLIENT_HELLO_DONE;
            WOLFSSL_MSG("accept state ACCEPT_CLIENT_HELLO_DONE");
            FALL_THROUGH;

        case ACCEPT_CLIENT_HELLO_DONE :
            if (ssl->options.tls1_3) {
                return wolfSSL_accept_TLSv13(ssl);
            }
#endif
            ssl->options.acceptState = ACCEPT_FIRST_REPLY_DONE;
            WOLFSSL_MSG("accept state ACCEPT_FIRST_REPLY_DONE");
            FALL_THROUGH;

        case ACCEPT_FIRST_REPLY_DONE :
            if ( (ssl->error = SendServerHello(ssl)) != 0) {
                WOLFSSL_ERROR(ssl->error);
                return WOLFSSL_FATAL_ERROR;
            }
            ssl->options.acceptState = SERVER_HELLO_SENT;
            WOLFSSL_MSG("accept state SERVER_HELLO_SENT");
            FALL_THROUGH;

        case SERVER_HELLO_SENT :
        #ifdef WOLFSSL_TLS13
            if (ssl->options.tls1_3) {
                return wolfSSL_accept_TLSv13(ssl);
            }
        #endif
            #ifndef NO_CERTS
                if (!ssl->options.resuming)
                    if ( (ssl->error = SendCertificate(ssl)) != 0) {
                        WOLFSSL_ERROR(ssl->error);
                        return WOLFSSL_FATAL_ERROR;
                    }
            #endif
            ssl->options.acceptState = CERT_SENT;
            WOLFSSL_MSG("accept state CERT_SENT");
            FALL_THROUGH;

        case CERT_SENT :
            #ifndef NO_CERTS
            if (!ssl->options.resuming)
                if ( (ssl->error = SendCertificateStatus(ssl)) != 0) {
                    WOLFSSL_ERROR(ssl->error);
                    return WOLFSSL_FATAL_ERROR;
                }
            #endif
            ssl->options.acceptState = CERT_STATUS_SENT;
            WOLFSSL_MSG("accept state CERT_STATUS_SENT");
            FALL_THROUGH;

        case CERT_STATUS_SENT :
        #ifdef WOLFSSL_TLS13
            if (ssl->options.tls1_3) {
                return wolfSSL_accept_TLSv13(ssl);
            }
        #endif
            if (!ssl->options.resuming)
                if ( (ssl->error = SendServerKeyExchange(ssl)) != 0) {
                    WOLFSSL_ERROR(ssl->error);
                    return WOLFSSL_FATAL_ERROR;
                }
            ssl->options.acceptState = KEY_EXCHANGE_SENT;
            WOLFSSL_MSG("accept state KEY_EXCHANGE_SENT");
            FALL_THROUGH;

        case KEY_EXCHANGE_SENT :
            #ifndef NO_CERTS
                if (!ssl->options.resuming) {
                    if (ssl->options.verifyPeer) {
                        if ( (ssl->error = SendCertificateRequest(ssl)) != 0) {
                            WOLFSSL_ERROR(ssl->error);
                            return WOLFSSL_FATAL_ERROR;
                        }
                    }
                    else {
                        /* SERVER: Peer auth good if not verifying client. */
                        ssl->options.peerAuthGood = 1;
                    }
                }
            #endif
            ssl->options.acceptState = CERT_REQ_SENT;
            WOLFSSL_MSG("accept state CERT_REQ_SENT");
            FALL_THROUGH;

        case CERT_REQ_SENT :
            if (!ssl->options.resuming)
                if ( (ssl->error = SendServerHelloDone(ssl)) != 0) {
                    WOLFSSL_ERROR(ssl->error);
                    return WOLFSSL_FATAL_ERROR;
                }
            ssl->options.acceptState = SERVER_HELLO_DONE;
            WOLFSSL_MSG("accept state SERVER_HELLO_DONE");
            FALL_THROUGH;

        case SERVER_HELLO_DONE :
            if (!ssl->options.resuming) {
                while (ssl->options.clientState < CLIENT_FINISHED_COMPLETE)
                    if ( (ssl->error = ProcessReply(ssl)) < 0) {
                        WOLFSSL_ERROR(ssl->error);
                        return WOLFSSL_FATAL_ERROR;
                    }
            }
            ssl->options.acceptState = ACCEPT_SECOND_REPLY_DONE;
            WOLFSSL_MSG("accept state  ACCEPT_SECOND_REPLY_DONE");
            FALL_THROUGH;

        case ACCEPT_SECOND_REPLY_DONE :
        #ifndef NO_CERTS
            /* SERVER: When not resuming and verifying peer but no certificate
             * received and not failing when not received then peer auth good.
             */
            if (!ssl->options.resuming && ssl->options.verifyPeer &&
                !ssl->options.havePeerCert && !ssl->options.failNoCert) {
                ssl->options.peerAuthGood = 1;
            }
        #endif /* !NO_CERTS  */
        #ifdef WOLFSSL_NO_CLIENT_AUTH
            if (!ssl->options.resuming) {
                ssl->options.peerAuthGood = 1;
            }
        #endif

#ifdef HAVE_SESSION_TICKET
            if (ssl->options.createTicket && !ssl->options.noTicketTls12) {
                if ( (ssl->error = SendTicket(ssl)) != 0) {
                    WOLFSSL_ERROR(ssl->error);
                    return WOLFSSL_FATAL_ERROR;
                }
            }
#endif /* HAVE_SESSION_TICKET */
            ssl->options.acceptState = TICKET_SENT;
            WOLFSSL_MSG("accept state  TICKET_SENT");
            FALL_THROUGH;

        case TICKET_SENT:
            /* SERVER: Fail-safe for CLient Authentication. */
            if (!ssl->options.peerAuthGood) {
                WOLFSSL_MSG("Client authentication did not happen");
                return WOLFSSL_FATAL_ERROR;
            }

            if ( (ssl->error = SendChangeCipher(ssl)) != 0) {
                WOLFSSL_ERROR(ssl->error);
                return WOLFSSL_FATAL_ERROR;
            }
            ssl->options.acceptState = CHANGE_CIPHER_SENT;
            WOLFSSL_MSG("accept state  CHANGE_CIPHER_SENT");
            FALL_THROUGH;

        case CHANGE_CIPHER_SENT :
            if ( (ssl->error = SendFinished(ssl)) != 0) {
                WOLFSSL_ERROR(ssl->error);
                return WOLFSSL_FATAL_ERROR;
            }

            ssl->options.acceptState = ACCEPT_FINISHED_DONE;
            WOLFSSL_MSG("accept state ACCEPT_FINISHED_DONE");
            FALL_THROUGH;

        case ACCEPT_FINISHED_DONE :
            if (ssl->options.resuming)
                while (ssl->options.clientState < CLIENT_FINISHED_COMPLETE)
                    if ( (ssl->error = ProcessReply(ssl)) < 0) {
                        WOLFSSL_ERROR(ssl->error);
                        return WOLFSSL_FATAL_ERROR;
                    }

            ssl->options.acceptState = ACCEPT_THIRD_REPLY_DONE;
            WOLFSSL_MSG("accept state ACCEPT_THIRD_REPLY_DONE");
            FALL_THROUGH;

        case ACCEPT_THIRD_REPLY_DONE :
#ifndef NO_HANDSHAKE_DONE_CB
            if (ssl->hsDoneCb) {
                int cbret = ssl->hsDoneCb(ssl, ssl->hsDoneCtx);
                if (cbret < 0) {
                    ssl->error = cbret;
                    WOLFSSL_MSG("HandShake Done Cb don't continue error");
                    return WOLFSSL_FATAL_ERROR;
                }
            }
#endif /* NO_HANDSHAKE_DONE_CB */

            if (!ssl->options.dtls) {
                if (!ssl->options.keepResources) {
                    FreeHandshakeResources(ssl);
                }
            }
#ifdef WOLFSSL_DTLS
            else {
                ssl->options.dtlsHsRetain = 1;
            }
#endif /* WOLFSSL_DTLS */

#if defined(WOLFSSL_ASYNC_CRYPT) && defined(HAVE_SECURE_RENEGOTIATION)
            /* This may be necessary in async so that we don't try to
             * renegotiate again */
            if (ssl->secure_renegotiation && ssl->secure_renegotiation->startScr) {
                ssl->secure_renegotiation->startScr = 0;
            }
#endif /* WOLFSSL_ASYNC_CRYPT && HAVE_SECURE_RENEGOTIATION */

#if defined(WOLFSSL_SESSION_EXPORT) && defined(WOLFSSL_DTLS)
            if (ssl->dtls_export) {
                if ((ssl->error = wolfSSL_send_session(ssl)) != 0) {
                    WOLFSSL_MSG("Export DTLS session error");
                    WOLFSSL_ERROR(ssl->error);
                    return WOLFSSL_FATAL_ERROR;
                }
            }
#endif

            WOLFSSL_LEAVE("SSL_accept()", WOLFSSL_SUCCESS);
            return WOLFSSL_SUCCESS;

        default :
            WOLFSSL_MSG("Unknown accept state ERROR");
            return WOLFSSL_FATAL_ERROR;
        }
#endif /* !WOLFSSL_NO_TLS12 */
    }

#endif /* NO_WOLFSSL_SERVER */


#ifndef NO_HANDSHAKE_DONE_CB

int wolfSSL_SetHsDoneCb(WOLFSSL* ssl, HandShakeDoneCb cb, void* user_ctx)
{
    WOLFSSL_ENTER("wolfSSL_SetHsDoneCb");

    if (ssl == NULL)
        return BAD_FUNC_ARG;

    ssl->hsDoneCb  = cb;
    ssl->hsDoneCtx = user_ctx;


    return WOLFSSL_SUCCESS;
}

#endif /* NO_HANDSHAKE_DONE_CB */

WOLFSSL_ABI
int wolfSSL_Cleanup(void)
{
    int ret = WOLFSSL_SUCCESS; /* Only the first error will be returned */
    int release = 0;
#if !defined(NO_SESSION_CACHE) && defined(ENABLE_SESSION_CACHE_ROW_LOCK)
    int i;
#endif

    WOLFSSL_ENTER("wolfSSL_Cleanup");

    if (initRefCount == 0)
        return ret;  /* possibly no init yet, but not failure either way */

    if ((count_mutex_valid == 1) && (wc_LockMutex(&count_mutex) != 0)) {
        WOLFSSL_MSG("Bad Lock Mutex count");
        ret = BAD_MUTEX_E;
    }

    release = initRefCount-- == 1;
    if (initRefCount < 0)
        initRefCount = 0;

    if (count_mutex_valid == 1) {
        wc_UnLockMutex(&count_mutex);
    }

    if (!release)
        return ret;

#ifdef OPENSSL_EXTRA
    if (bn_one) {
        wolfSSL_BN_free(bn_one);
        bn_one = NULL;
    }
#endif

#ifndef NO_SESSION_CACHE
    #ifdef ENABLE_SESSION_CACHE_ROW_LOCK
    for (i = 0; i < SESSION_ROWS; ++i) {
        if ((SessionCache[i].mutex_valid == 1) &&
            (wc_FreeMutex(&SessionCache[i].row_mutex) != 0)) {
            if (ret == WOLFSSL_SUCCESS)
                ret = BAD_MUTEX_E;
        }
        SessionCache[i].mutex_valid = 0;
    }
    #else
    if ((session_mutex_valid == 1) && (wc_FreeMutex(&session_mutex) != 0)) {
        if (ret == WOLFSSL_SUCCESS)
            ret = BAD_MUTEX_E;
    }
    session_mutex_valid = 0;
    #endif
    #ifndef NO_CLIENT_CACHE
    if ((clisession_mutex_valid == 1) &&
        (wc_FreeMutex(&clisession_mutex) != 0)) {
        if (ret == WOLFSSL_SUCCESS)
            ret = BAD_MUTEX_E;
    }
    clisession_mutex_valid = 0;
    #endif
#endif /* !NO_SESSION_CACHE */

    if ((count_mutex_valid == 1) && (wc_FreeMutex(&count_mutex) != 0)) {
        if (ret == WOLFSSL_SUCCESS)
            ret = BAD_MUTEX_E;
    }
    count_mutex_valid = 0;

#ifdef OPENSSL_EXTRA
    wolfSSL_RAND_Cleanup();
#endif

    if (wolfCrypt_Cleanup() != 0) {
        WOLFSSL_MSG("Error with wolfCrypt_Cleanup call");
        if (ret == WOLFSSL_SUCCESS)
            ret = WC_CLEANUP_E;
    }

#if FIPS_VERSION_GE(5,1)
    if (wolfCrypt_SetPrivateKeyReadEnable_fips(0, WC_KEYTYPE_ALL) < 0) {
        if (ret == WOLFSSL_SUCCESS)
            ret = WC_CLEANUP_E;
    }
#endif

#ifdef HAVE_GLOBAL_RNG
    if ((globalRNGMutex_valid == 1) && (wc_FreeMutex(&globalRNGMutex) != 0)) {
        if (ret == WOLFSSL_SUCCESS)
            ret = BAD_MUTEX_E;
    }
    globalRNGMutex_valid = 0;

    #if defined(OPENSSL_EXTRA) && defined(HAVE_HASHDRBG)
    wolfSSL_FIPS_drbg_free(gDrbgDefCtx);
    gDrbgDefCtx = NULL;
    #endif
#endif

    return ret;
}


#ifndef NO_SESSION_CACHE

/* some session IDs aren't random after all, let's make them random */
static WC_INLINE word32 HashSession(const byte* sessionID, word32 len, int* error)
{
    byte digest[WC_MAX_DIGEST_SIZE];

#ifndef NO_MD5
    *error =  wc_Md5Hash(sessionID, len, digest);
#elif !defined(NO_SHA)
    *error =  wc_ShaHash(sessionID, len, digest);
#elif !defined(NO_SHA256)
    *error =  wc_Sha256Hash(sessionID, len, digest);
#else
    #error "We need a digest to hash the session IDs"
#endif

    return *error == 0 ? MakeWordFromHash(digest) : 0; /* 0 on failure */
}


WOLFSSL_ABI
void wolfSSL_flush_sessions(WOLFSSL_CTX* ctx, long tm)
{
    /* static table now, no flushing needed */
    (void)ctx;
    (void)tm;
}


/* set ssl session timeout in seconds */
WOLFSSL_ABI
int wolfSSL_set_timeout(WOLFSSL* ssl, unsigned int to)
{
    if (ssl == NULL)
        return BAD_FUNC_ARG;

    if (to == 0)
        to = WOLFSSL_SESSION_TIMEOUT;
    ssl->timeout = to;

    return WOLFSSL_SUCCESS;
}


/**
 * Sets ctx session timeout in seconds.
 * The timeout value set here should be reflected in the
 * "session ticket lifetime hint" if this API works in the openssl compat-layer.
 * Therefore wolfSSL_CTX_set_TicketHint is called internally.
 * Arguments:
 *  - ctx  WOLFSSL_CTX object which the timeout is set to
 *  - to   timeout value in second
 * Returns:
 *  WOLFSSL_SUCCESS on success, BAD_FUNC_ARG on failure.
 *  When WOLFSSL_ERROR_CODE_OPENSSL is defined, returns previous timeout value
 *  on success, BAD_FUNC_ARG on failure.
 */
WOLFSSL_ABI
int wolfSSL_CTX_set_timeout(WOLFSSL_CTX* ctx, unsigned int to)
{
    #if defined(WOLFSSL_ERROR_CODE_OPENSSL)
    word32 prev_timeout = 0;
    #endif

    int ret = WOLFSSL_SUCCESS;
    (void)ret;

    if (ctx == NULL)
        ret = BAD_FUNC_ARG;

    if (ret == WOLFSSL_SUCCESS) {
    #if defined(WOLFSSL_ERROR_CODE_OPENSSL)
        prev_timeout = ctx->timeout;
    #endif
        if (to == 0) {
            ctx->timeout = WOLFSSL_SESSION_TIMEOUT;
        }
        else {
            ctx->timeout = to;
        }
    }
#if defined(OPENSSL_EXTRA) && defined(HAVE_SESSION_TICKET) && \
   !defined(NO_WOLFSSL_SERVER)
    if (ret == WOLFSSL_SUCCESS) {
        if (to == 0) {
            ret = wolfSSL_CTX_set_TicketHint(ctx, SESSION_TICKET_HINT_DEFAULT);
        }
        else {
            ret = wolfSSL_CTX_set_TicketHint(ctx, to);
        }
    }
#endif /* OPENSSL_EXTRA && HAVE_SESSION_TICKET && !NO_WOLFSSL_SERVER */

#if defined(WOLFSSL_ERROR_CODE_OPENSSL)
    if (ret == WOLFSSL_SUCCESS) {
        return prev_timeout;
    }
    else {
        return ret;
    }
#else
    return ret;
#endif /* WOLFSSL_ERROR_CODE_OPENSSL */
}


#ifndef NO_CLIENT_CACHE

/* Get Session from Client cache based on id/len, return NULL on failure */
WOLFSSL_SESSION* wolfSSL_GetSessionClient(WOLFSSL* ssl, const byte* id, int len)
{
    WOLFSSL_SESSION* ret = NULL;
    word32          row;
    int             idx;
    int             count;
    int             error = 0;
    ClientSession*  clSess;

    WOLFSSL_ENTER("GetSessionClient");

    if (ssl->ctx->sessionCacheOff) {
        WOLFSSL_MSG("Session Cache off");
        return NULL;
    }

    if (ssl->options.side == WOLFSSL_SERVER_END)
        return NULL;

    len = min(SERVER_ID_LEN, (word32)len);

#ifdef HAVE_EXT_CACHE
    if (ssl->ctx->get_sess_cb != NULL) {
        int copy = 0;
        WOLFSSL_MSG("Calling external session cache");
        ret = ssl->ctx->get_sess_cb(ssl, (byte*)id, len, &copy);
        if (ret != NULL) {
            WOLFSSL_MSG("Session found in external cache");
            return ret;
        }
        WOLFSSL_MSG("Session not found in external cache");
    }

    if (ssl->ctx->internalCacheLookupOff) {
        WOLFSSL_MSG("Internal cache turned off");
        return NULL;
    }
#endif

    row = HashSession(id, len, &error) % CLIENT_SESSION_ROWS;
    if (error != 0) {
        WOLFSSL_MSG("Hash session failed");
        return NULL;
    }

    if (wc_LockMutex(&clisession_mutex) != 0) {
        WOLFSSL_MSG("Client cache mutex lock failed");
        return NULL;
    }

    /* start from most recently used */
    count = min((word32)ClientCache[row].totalCount, CLIENT_SESSIONS_PER_ROW);
    idx = ClientCache[row].nextIdx - 1;
    if (idx < 0 || idx >= CLIENT_SESSIONS_PER_ROW) {
        idx = CLIENT_SESSIONS_PER_ROW - 1; /* if back to front, the previous was end */
    }
    clSess = ClientCache[row].Clients;

    for (; count > 0; --count) {
        WOLFSSL_SESSION* current;
        SessionRow* sessRow;

        if (clSess[idx].serverRow >= SESSION_ROWS) {
            WOLFSSL_MSG("Client cache serverRow invalid");
            break;
        }

        /* lock row */
        sessRow = &SessionCache[clSess[idx].serverRow];
        if (SESSION_ROW_LOCK(sessRow) != 0) {
            WOLFSSL_MSG("Session cache row lock failure");
            break;
        }

        current = &sessRow->Sessions[clSess[idx].serverIdx];
        if (XMEMCMP(current->serverID, id, len) == 0) {
            WOLFSSL_MSG("Found a serverid match for client");
            if (LowResTimer() < (current->bornOn + current->timeout)) {
                WOLFSSL_MSG("Session valid");
                ret = current;
                SESSION_ROW_UNLOCK(sessRow);
                break;
            } else {
                WOLFSSL_MSG("Session timed out");  /* could have more for id */
            }
        } else {
            WOLFSSL_MSG("ServerID not a match from client table");
        }
        SESSION_ROW_UNLOCK(sessRow);

        idx = idx > 0 ? idx - 1 : CLIENT_SESSIONS_PER_ROW - 1;
    }

    wc_UnLockMutex(&clisession_mutex);

    return ret;
}

#endif /* !NO_CLIENT_CACHE */

static int SslSessionCacheOff(const WOLFSSL* ssl, const WOLFSSL_SESSION* session)
{
    (void)session;
    return ssl->options.sessionCacheOff
    #if defined(HAVE_SESSION_TICKET) && defined(WOLFSSL_FORCE_CACHE_ON_TICKET)
                && session->ticketLen == 0
    #endif
    #ifdef OPENSSL_EXTRA
                && ssl->options.side != WOLFSSL_CLIENT_END
    #endif
                ;
}


int wolfSSL_GetSessionFromCache(WOLFSSL* ssl, WOLFSSL_SESSION* output)
{
    WOLFSSL_SESSION* sess = NULL;
    const byte*  id = NULL;
    word32       row;
    int          idx;
    int          count;
    int          error = 0;
    SessionRow*  sessRow;
#ifdef HAVE_SESSION_TICKET
#ifndef WOLFSSL_SMALL_STACK
    byte         tmpTicket[PREALLOC_SESSION_TICKET_LEN];
#else
    byte*        tmpTicket = NULL;
#endif
    byte         tmpBufSet = 0;
#endif
#if defined(SESSION_CERTS) && defined(OPENSSL_EXTRA)
    WOLFSSL_X509* peer = NULL;
#endif
    byte         bogusID[ID_LEN];
    byte         bogusIDSz = 0;

    WOLFSSL_ENTER("wolfSSL_GetSessionFromCache");

    if (output == NULL) {
        WOLFSSL_MSG("NULL output");
        return WOLFSSL_FAILURE;
    }

    if (SslSessionCacheOff(ssl, ssl->session))
        return WOLFSSL_FAILURE;

    if (ssl->options.haveSessionId == 0)
        return WOLFSSL_FAILURE;

#ifdef HAVE_SESSION_TICKET
    if (ssl->options.side == WOLFSSL_SERVER_END && ssl->options.useTicket == 1)
        return WOLFSSL_FAILURE;
#endif

    XMEMSET(bogusID, 0, sizeof(bogusID));
    if (!IsAtLeastTLSv1_3(ssl->version) && ssl->arrays != NULL)
        id = ssl->arrays->sessionID;
    else if (ssl->session->haveAltSessionID) {
        id = ssl->session->altSessionID;
        /* We want to restore the bogus ID for TLS compatibility */
        if (output == ssl->session) {
            XMEMCPY(bogusID, ssl->session->sessionID, ID_LEN);
            bogusIDSz = ssl->session->sessionIDSz;
        }
    }
    else
        id = ssl->session->sessionID;


#ifdef HAVE_EXT_CACHE
    if (ssl->ctx->get_sess_cb != NULL) {
        int copy = 0;
        /* Attempt to retrieve the session from the external cache. */
        WOLFSSL_MSG("Calling external session cache");
        sess = ssl->ctx->get_sess_cb(ssl, (byte*)id, ID_LEN, &copy);
        if (sess != NULL) {
            WOLFSSL_MSG("Session found in external cache");
            error = wolfSSL_DupSession(sess, output, 0);
            /* If copy not set then free immediately */
            if (!copy)
                wolfSSL_SESSION_free(sess);
            /* We want to restore the bogus ID for TLS compatibility */
            if (ssl->session->haveAltSessionID &&
                    output == ssl->session) {
                XMEMCPY(ssl->session->sessionID, bogusID, ID_LEN);
                ssl->session->sessionIDSz = bogusIDSz;
            }
            return error;
        }
        WOLFSSL_MSG("Session not found in external cache");
    }

    if (ssl->ctx->internalCacheLookupOff) {
        WOLFSSL_MSG("Internal cache lookup turned off");
        return WOLFSSL_FAILURE;
    }
#endif

    row = HashSession(id, ID_LEN, &error) % SESSION_ROWS;
    if (error != 0) {
        WOLFSSL_MSG("Hash session failed");
        return WOLFSSL_FAILURE;
    }


#ifdef HAVE_SESSION_TICKET
    if (output->ticket == NULL ||
            output->ticketLenAlloc < PREALLOC_SESSION_TICKET_LEN) {
#ifdef WOLFSSL_SMALL_STACK
        tmpTicket = (byte*)XMALLOC(PREALLOC_SESSION_TICKET_LEN, output->heap,
                DYNAMIC_TYPE_TMP_BUFFER);
        if (tmpTicket == NULL) {
            WOLFSSL_MSG("tmpTicket malloc failed");
            return WOLFSSL_FAILURE;
        }
#endif
        if (output->ticketLenAlloc)
            XFREE(output->ticket, output->heap, DYNAMIC_TYPE_SESSION_TICK);
        output->ticket = tmpTicket;
        output->ticketLenAlloc = PREALLOC_SESSION_TICKET_LEN;
        output->ticketLen = 0;
        tmpBufSet = 1;
    }
#endif

#if defined(SESSION_CERTS) && defined(OPENSSL_EXTRA)
    if (output->peer != NULL) {
        wolfSSL_X509_free(output->peer);
        output->peer = NULL;
    }
#endif

    /* lock row */
    sessRow = &SessionCache[row];
    if (SESSION_ROW_LOCK(sessRow) != 0) {
        WOLFSSL_MSG("Session cache row lock failure");
#ifdef HAVE_SESSION_TICKET
        if (tmpBufSet) {
            output->ticket = output->_staticTicket;
            output->ticketLenAlloc = 0;
        }
#ifdef WOLFSSL_SMALL_STACK
        if (tmpTicket != NULL)
            XFREE(tmpTicket, output->heap, DYNAMIC_TYPE_TMP_BUFFER);
#endif
#endif
        return WOLFSSL_FAILURE;
    }

    /* start from most recently used */
    count = min((word32)sessRow->totalCount, SESSIONS_PER_ROW);
    idx = sessRow->nextIdx - 1;
    if (idx < 0 || idx >= SESSIONS_PER_ROW) {
        idx = SESSIONS_PER_ROW - 1; /* if back to front, the previous was end */
    }

    for (; count > 0; --count) {
        WOLFSSL_SESSION* current;

        current = &sessRow->Sessions[idx];
        if (XMEMCMP(current->sessionID, id, ID_LEN) == 0 &&
                current->side == ssl->options.side) {
            WOLFSSL_MSG("Found a session match");
            if (LowResTimer() < (current->bornOn + current->timeout)) {
                WOLFSSL_MSG("Session valid");
                sess = current;
            } else {
                WOLFSSL_MSG("Session timed out");
            }
            break;  /* no more sessionIDs whether valid or not that match */
        } else {
            WOLFSSL_MSG("SessionID not a match at this idx");
        }

        idx = idx > 0 ? idx - 1 : SESSIONS_PER_ROW - 1;
    }

    if (sess != NULL) {
#if defined(SESSION_CERTS) && defined(OPENSSL_EXTRA)
        /* We don't want the peer member. We will free it at the end. */
        if (sess->peer != NULL) {
            peer = sess->peer;
            sess->peer = NULL;
        }
#endif
        error = wolfSSL_DupSession(sess, output, 1);
    }
    else {
        error = WOLFSSL_FAILURE;
    }

    SESSION_ROW_UNLOCK(sessRow);

    /* We want to restore the bogus ID for TLS compatibility */
    if (ssl->session->haveAltSessionID &&
            output == ssl->session) {
        XMEMCPY(ssl->session->sessionID, bogusID, ID_LEN);
        ssl->session->sessionIDSz = bogusIDSz;
    }

#ifdef HAVE_SESSION_TICKET
    if (tmpBufSet) {
        if (error == WOLFSSL_SUCCESS) {
            if (output->ticketLen > SESSION_TICKET_LEN) {
                output->ticket = (byte*)XMALLOC(output->ticketLen, output->heap,
                        DYNAMIC_TYPE_SESSION_TICK);
                if (output->ticket == NULL) {
                    error = WOLFSSL_FAILURE;
                    output->ticket = output->_staticTicket;
                    output->ticketLenAlloc = 0;
                    output->ticketLen = 0;
                }
            }
            else {
                output->ticket = output->_staticTicket;
                output->ticketLenAlloc = 0;
            }
        }
        else {
            output->ticket = output->_staticTicket;
            output->ticketLenAlloc = 0;
            output->ticketLen = 0;
        }
        if (error == WOLFSSL_SUCCESS) {
            XMEMCPY(output->ticket, tmpTicket, output->ticketLen);
        }
    }
#ifdef WOLFSSL_SMALL_STACK
    if (tmpTicket != NULL)
        XFREE(tmpTicket, output->heap, DYNAMIC_TYPE_TMP_BUFFER);
#endif
#endif

#if defined(SESSION_CERTS) && defined(OPENSSL_EXTRA)
    if (peer != NULL) {
        wolfSSL_X509_free(peer);
    }
#endif

    return error;
}

WOLFSSL_SESSION* wolfSSL_GetSession(WOLFSSL* ssl, byte* masterSecret,
        byte restoreSessionCerts)
{
    WOLFSSL_SESSION* ret = NULL;

    (void)restoreSessionCerts; /* Kept for compatibility */

    if (wolfSSL_GetSessionFromCache(ssl, ssl->session) == WOLFSSL_SUCCESS) {
        ret = ssl->session;
    }
    else {
        WOLFSSL_MSG("wolfSSL_GetSessionFromCache did not return a session");
    }

    if (ret != NULL && masterSecret != NULL)
        XMEMCPY(masterSecret, ret->masterSecret, SECRET_LEN);

    return ret;
}

int wolfSSL_SetSession(WOLFSSL* ssl, WOLFSSL_SESSION* session)
{
    SessionRow* sessRow = NULL;
    int ret = WOLFSSL_SUCCESS;

    session = ClientSessionToSession(session);

    if (ssl == NULL || session == NULL) {
        return WOLFSSL_FAILURE;
    }

    if (session->type == WOLFSSL_SESSION_TYPE_CACHE) {
        if (session->cacheRow < SESSION_ROWS) {
            sessRow = &SessionCache[session->cacheRow];
            if (SESSION_ROW_LOCK(sessRow) != 0) {
                WOLFSSL_MSG("Session row lock failed");
                return WOLFSSL_FAILURE;
            }
        }
    }

    if (ret == WOLFSSL_SUCCESS && SslSessionCacheOff(ssl, session)) {
        WOLFSSL_MSG("Session cache off");
        ret = WOLFSSL_FAILURE;
    }

    if (ret == WOLFSSL_SUCCESS && ssl->options.side != WOLFSSL_NEITHER_END &&
            (byte)ssl->options.side != session->side) {
        WOLFSSL_MSG("Setting session for wrong role");
        ret = WOLFSSL_FAILURE;
    }


    if (ret == WOLFSSL_SUCCESS &&
            wolfSSL_DupSession(session, ssl->session, 0) != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("Session duplicate failed");
        ret = WOLFSSL_FAILURE;
    }

    /* Let's copy over the altSessionID for local cache purposes */
    if (ret == WOLFSSL_SUCCESS && session->haveAltSessionID) {
        ssl->session->haveAltSessionID = 1;
        XMEMCPY(ssl->session->altSessionID, session->altSessionID, ID_LEN);
    }

    if (sessRow != NULL) {
        SESSION_ROW_UNLOCK(sessRow);
        sessRow = NULL;
    }

    /* Note: the `session` variable cannot be used below, since the row is
     * un-locked */

    if (ret != WOLFSSL_SUCCESS)
        return ret;

#ifdef OPENSSL_EXTRA
    /* check for application context id */
    if (ssl->sessionCtxSz > 0) {
        if (XMEMCMP(ssl->sessionCtx, ssl->session->sessionCtx, ssl->sessionCtxSz)) {
            /* context id did not match! */
            WOLFSSL_MSG("Session context did not match");
            return WOLFSSL_FAILURE;
        }
    }
#endif /* OPENSSL_EXTRA */

    if (LowResTimer() < (ssl->session->bornOn + ssl->session->timeout)) {
        ssl->options.resuming = 1;
        ssl->options.haveEMS = ssl->session->haveEMS;

#if defined(SESSION_CERTS) || (defined(WOLFSSL_TLS13) && \
                               defined(HAVE_SESSION_TICKET))
        ssl->version              = ssl->session->version;
        if (IsAtLeastTLSv1_3(ssl->version))
            ssl->options.tls1_3 = 1;
#endif
#if defined(SESSION_CERTS) || !defined(NO_RESUME_SUITE_CHECK) || \
                        (defined(WOLFSSL_TLS13) && defined(HAVE_SESSION_TICKET))
        ssl->options.cipherSuite0 = ssl->session->cipherSuite0;
        ssl->options.cipherSuite  = ssl->session->cipherSuite;
#endif
#if defined(OPENSSL_EXTRA) || defined(OPENSSL_EXTRA_X509_SMALL)
        ssl->peerVerifyRet = (unsigned long)ssl->session->peerVerifyRet;
#endif
        ret = WOLFSSL_SUCCESS;
    }
    else {
#if defined(OPENSSL_EXTRA) && defined(WOLFSSL_ERROR_CODE_OPENSSL)
        WOLFSSL_MSG("Session is expired but return success for \
                              OpenSSL compatibility");
        ret = WOLFSSL_SUCCESS;
#else
        ret = WOLFSSL_FAILURE;  /* session timed out */
#endif /* OPENSSL_EXTRA && WOLFSSL_ERROR_CODE_OPENSSL */
    }
    return ret;
}


#ifdef WOLFSSL_SESSION_STATS
static int get_locked_session_stats(word32* active, word32* total,
                                    word32* peak);
#endif

#ifndef NO_CLIENT_CACHE
ClientSession* AddSessionToClientCache(int side, int row, int idx, byte* serverID,
                            word16 idLen, const byte* sessionID,
                            word16 useTicket)
{
    int error = -1;
    word32 clientRow = 0, clientIdx = 0, sessionIDHash = 0;
    (void)useTicket;
    if (side == WOLFSSL_CLIENT_END
            && row != INVALID_SESSION_ROW
            && (idLen
#ifdef HAVE_SESSION_TICKET
                || useTicket == 1
#endif
                || serverID != NULL
                )) {

        WOLFSSL_MSG("Trying to add client cache entry");

        if (idLen) {
            clientRow = HashSession(serverID,
                    idLen, &error) % CLIENT_SESSION_ROWS;
        }
        else if (serverID != NULL) {
            clientRow = HashSession(sessionID,
                    ID_LEN, &error) % CLIENT_SESSION_ROWS;
        }
        else
            error = -1;
        if (error == 0 && wc_LockMutex(&clisession_mutex) == 0) {
            clientIdx = ClientCache[clientRow].nextIdx++;
            ClientCache[clientRow].Clients[clientIdx].serverRow =
                                                            (word16)row;
            ClientCache[clientRow].Clients[clientIdx].serverIdx =
                                                            (word16)idx;
            if (sessionID != NULL) {
                sessionIDHash = HashSession(sessionID, ID_LEN, &error);
                if (error == 0) {
                    ClientCache[clientRow].Clients[clientIdx].sessionIDHash
                        = sessionIDHash;
                }
            }
            if (error == 0) {
                WOLFSSL_MSG("Adding client cache entry");
                if (ClientCache[clientRow].totalCount < CLIENT_SESSIONS_PER_ROW)
                    ClientCache[clientRow].totalCount++;
                ClientCache[clientRow].nextIdx %= CLIENT_SESSIONS_PER_ROW;
            }

            wc_UnLockMutex(&clisession_mutex);
        }
        else {
            WOLFSSL_MSG("Hash session failed");
        }
    }
    else {
        WOLFSSL_MSG("Skipping client cache");
    }
    if (error == 0)
        return &ClientCache[clientRow].Clients[clientIdx];
    else
        return NULL;
}
#endif

/**
 * For backwards compatibility, this API needs to be used in *ALL* functions
 * that access the WOLFSSL_SESSION members directly.
 *
 * This API checks if the passed in session is actually a ClientSession object
 * and returns the matching session cache object. Otherwise just return the
 * input. ClientSession objects only occur in the ClientCache. They are not
 * allocated anywhere else.
 */
WOLFSSL_SESSION* ClientSessionToSession(const WOLFSSL_SESSION* session)
{
    WOLFSSL_ENTER("ClientSessionToSession");
#ifdef NO_SESSION_CACHE_REF
    return (WOLFSSL_SESSION*)session;
#else
#ifndef NO_CLIENT_CACHE
    if (session == NULL)
        return NULL;
    /* Check if session points into ClientCache */
    if ((byte*)session >= (byte*)ClientCache &&
            /* Cast to byte* to make pointer arithmetic work per byte */
            (byte*)session < ((byte*)ClientCache) + sizeof(ClientCache)) {
        ClientSession* clientSession = (ClientSession*)session;
        SessionRow* sessRow = NULL;
        WOLFSSL_SESSION* cacheSession = NULL;
        word32 sessionIDHash = 0;
        int error = 0;
        session = NULL; /* Default to NULL for failure case */
        if (wc_LockMutex(&clisession_mutex) != 0) {
            WOLFSSL_MSG("Client cache mutex lock failed");
            return NULL;
        }
        if (clientSession->serverRow >= SESSION_ROWS ||
                clientSession->serverIdx >= SESSIONS_PER_ROW) {
            WOLFSSL_MSG("Client cache serverRow or serverIdx invalid");
            error = -1;
        }
        if (error == 0) {
            /* Lock row */
            sessRow = &SessionCache[clientSession->serverRow];
            error = SESSION_ROW_LOCK(sessRow);
            if (error != 0) {
                WOLFSSL_MSG("Session cache row lock failure");
                sessRow = NULL;
            }
        }
        if (error == 0) {
            cacheSession = &sessRow->Sessions[clientSession->serverIdx];
            if (cacheSession->sessionIDSz == 0) {
                cacheSession = NULL;
                WOLFSSL_MSG("Session cache entry not set");
                error = -1;
            }
        }
        if (error == 0) {
            /* Calculate the hash of the session ID */
            sessionIDHash = HashSession(cacheSession->sessionID, ID_LEN,
                    &error);
        }
        if (error == 0) {
            /* Check the session ID hash matches */
            error = clientSession->sessionIDHash != sessionIDHash;
        }
        if (error == 0) {
            /* Hashes match */
            session = cacheSession;
            WOLFSSL_MSG("Found session cache matching client session object");
        }
        if (sessRow != NULL) {
            SESSION_ROW_UNLOCK(sessRow);
        }
        wc_UnLockMutex(&clisession_mutex);
        return (WOLFSSL_SESSION*)session;
    }
    else {
        /* Plain WOLFSSL_SESSION object */
        return (WOLFSSL_SESSION*)session;
    }
#else
    return (WOLFSSL_SESSION*)session;
#endif
#endif
}

int AddSessionToCache(WOLFSSL_SESSION* addSession, const byte* id, byte idSz,
        int* sessionIndex, int side, word16 useTicket,
        ClientSession** clientCacheEntry)
{
    WOLFSSL_SESSION* cacheSession = NULL;
    SessionRow* sessRow = NULL;
    word32 idx = 0;
#if defined(SESSION_CERTS) && defined(OPENSSL_EXTRA)
    WOLFSSL_X509* peer = NULL;
#endif
#ifdef HAVE_SESSION_TICKET
    byte*  cacheTicBuff = NULL;
    byte   ticBuffUsed = 0;
    byte*  ticBuff = NULL;
    int    ticLen  = 0;
#endif
    int ret = 0;
    int row;
    int i;
    int overwrite = 0;

    (void)sessionIndex;
    (void)useTicket;
    (void)clientCacheEntry;

    addSession = ClientSessionToSession(addSession);

    if (addSession == NULL || idSz == 0) {
        WOLFSSL_MSG("addSession NULL or idSz == 0");
        return BAD_FUNC_ARG;
    }

    /* Find a position for the new session in cache and use that */
#ifdef HAVE_SESSION_TICKET
    ticLen = addSession->ticketLen;
    /* Alloc Memory here to avoid syscalls during lock */
    if (ticLen > SESSION_TICKET_LEN) {
        ticBuff = (byte*)XMALLOC(ticLen, NULL,
                DYNAMIC_TYPE_SESSION_TICK);
        if (ticBuff == NULL) {
            return MEMORY_E;
        }
    }
#endif
    /* Use the session object in the cache for external cache if required */
    row = (int)(HashSession(id, ID_LEN, &ret) % SESSION_ROWS);
    if (ret != 0) {
        WOLFSSL_MSG("Hash session failed");
    #ifdef HAVE_SESSION_TICKET
        XFREE(ticBuff, NULL, DYNAMIC_TYPE_SESSION_TICK);
    #endif
        return ret;
    }

    sessRow = &SessionCache[row];
    if (SESSION_ROW_LOCK(sessRow) != 0) {
    #ifdef HAVE_SESSION_TICKET
        XFREE(ticBuff, NULL, DYNAMIC_TYPE_SESSION_TICK);
    #endif
        WOLFSSL_MSG("Session row lock failed");
        return BAD_MUTEX_E;
    }

    for (i = 0; i < SESSIONS_PER_ROW && i < sessRow->totalCount; i++) {
        if (XMEMCMP(id,
                sessRow->Sessions[i].sessionID, ID_LEN) == 0 &&
                sessRow->Sessions[i].side == side) {
            WOLFSSL_MSG("Session already exists. Overwriting.");
            overwrite = 1;
            idx = i;
            break;
        }
    }

    if (!overwrite)
        idx = sessRow->nextIdx;
#ifdef SESSION_INDEX
    if (sessionIndex != NULL)
        *sessionIndex = (row << SESSIDX_ROW_SHIFT) | idx;
#endif
    cacheSession = &sessRow->Sessions[idx];
    cacheSession->type = WOLFSSL_SESSION_TYPE_CACHE;
    cacheSession->cacheRow = row;

#if defined(SESSION_CERTS) && defined(OPENSSL_EXTRA)
    /* Save the peer field to free after unlocking the row */
    if (cacheSession->peer != NULL)
        peer = cacheSession->peer;
    cacheSession->peer = NULL;
#endif
#ifdef HAVE_SESSION_TICKET
    /* If we can re-use the existing buffer in cacheSession then we won't touch
     * ticBuff at all making it a very cheap malloc/free. The page on a modern
     * OS will most likely not even be allocated to the process. */
    if (ticBuff != NULL && cacheSession->ticketLenAlloc < ticLen) {
        cacheTicBuff = cacheSession->ticket;
        ticBuffUsed = 1;
        cacheSession->ticket = ticBuff;
        cacheSession->ticketLenAlloc = ticLen;
    }
#endif
#ifdef SESSION_CERTS
    if (overwrite &&
            addSession->chain.count == 0 &&
            cacheSession->chain.count > 0) {
        /* Copy in the certs from the session */
        addSession->chain.count = cacheSession->chain.count;
        XMEMCPY(addSession->chain.certs, cacheSession->chain.certs,
                sizeof(x509_buffer) * cacheSession->chain.count);
    }
#endif /* SESSION_CERTS */
    cacheSession->heap = NULL;
    /* Copy data into the cache object */
    ret = wolfSSL_DupSession(addSession, cacheSession, 1) == WOLFSSL_FAILURE;

    if (ret == 0) {
        /* Increment the totalCount and the nextIdx */
        if (sessRow->totalCount < SESSIONS_PER_ROW)
            sessRow->totalCount++;
        sessRow->nextIdx = (sessRow->nextIdx + 1) % SESSIONS_PER_ROW;
        if (id != addSession->sessionID) {
            /* ssl->session->sessionID may contain the bogus ID or we want the
             * ID from the arrays object */
            XMEMCPY(cacheSession->sessionID, id, ID_LEN);
            cacheSession->sessionIDSz = ID_LEN;
        }
    }
#ifdef HAVE_SESSION_TICKET
    else if (ticBuffUsed) {
        /* Error occured. Need to clean up the ticket buffer. */
        cacheSession->ticket = cacheSession->_staticTicket;
        cacheSession->ticketLenAlloc = 0;
        cacheSession->ticketLen = 0;
    }
#endif

    SESSION_ROW_UNLOCK(sessRow);
    cacheSession = NULL; /* Can't access after unlocked */

#ifndef NO_CLIENT_CACHE
    if (ret == 0) {
        ClientSession* clientCache = AddSessionToClientCache(side, row, idx,
                addSession->serverID, addSession->idLen, id, useTicket);
        if (clientCache != NULL && clientCacheEntry != NULL)
            *clientCacheEntry = clientCache;
    }
#endif

#ifdef HAVE_SESSION_TICKET
    if (ticBuff != NULL && !ticBuffUsed)
        XFREE(ticBuff, NULL, DYNAMIC_TYPE_SESSION_TICK);
    if (cacheTicBuff != NULL)
        XFREE(cacheTicBuff, NULL, DYNAMIC_TYPE_SESSION_TICK);
#endif

#if defined(SESSION_CERTS) && defined(OPENSSL_EXTRA)
    if (peer != NULL) {
        wolfSSL_X509_free(peer);
        peer = NULL; /* Make sure not use after this point */
    }
#endif

    return ret;
}

#ifndef NO_CLIENT_CACHE
#endif

void AddSession(WOLFSSL* ssl)
{
    int    error = 0;
    const byte* id = NULL;
    byte idSz = 0;
    WOLFSSL_SESSION* session = ssl->session;
#ifdef HAVE_EXT_CACHE
    int cbRet = 0;
#endif

    (void)error;

    WOLFSSL_ENTER("AddSession");

    if (SslSessionCacheOff(ssl, session)) {
        WOLFSSL_MSG("Cache off");
        return;
    }

    if (ssl->options.haveSessionId == 0) {
        WOLFSSL_MSG("Don't have session id");
        return;
    }

#if defined(HAVE_SESSION_TICKET) && !defined(OPENSSL_EXTRA)
    /* For the compat layer generate a session object to use */
    if (ssl->options.side == WOLFSSL_SERVER_END && ssl->options.useTicket == 1) {
        WOLFSSL_MSG("Using tickets instead of cache");
        return;
    }
#endif

    if (session->haveAltSessionID) {
        id = session->altSessionID;
        idSz = ID_LEN;
    }
    else {
        if (!IsAtLeastTLSv1_3(ssl->version) && ssl->arrays != NULL) {
            /* Make sure the session ID is available when the user calls any
             * get_session API */
            XMEMCPY(session->sessionID, ssl->arrays->sessionID, ID_LEN);
            session->sessionIDSz = ssl->arrays->sessionIDSz;
        }
        id = session->sessionID;
        idSz = session->sessionIDSz;
    }

    session->timeout = ssl->timeout;
    session->side = (byte)ssl->options.side;
    if (!IsAtLeastTLSv1_3(ssl->version) && ssl->arrays != NULL)
        XMEMCPY(session->masterSecret, ssl->arrays->masterSecret, SECRET_LEN);
    session->haveEMS = ssl->options.haveEMS;
#ifdef OPENSSL_EXTRA
    /* If using compatibility layer then check for and copy over session context
     * id. */
    if (ssl->sessionCtxSz > 0 && ssl->sessionCtxSz < ID_LEN) {
        XMEMCPY(ssl->session->sessionCtx, ssl->sessionCtx, ssl->sessionCtxSz);
        session->sessionCtxSz = ssl->sessionCtxSz;
    }
#endif
    session->timeout = ssl->timeout;
    session->bornOn  = LowResTimer();
#if defined(SESSION_CERTS) || (defined(WOLFSSL_TLS13) && \
                               defined(HAVE_SESSION_TICKET))
    session->version = ssl->version;
#endif
#if defined(SESSION_CERTS) || !defined(NO_RESUME_SUITE_CHECK) || \
                        (defined(WOLFSSL_TLS13) && defined(HAVE_SESSION_TICKET))
    session->cipherSuite0 = ssl->options.cipherSuite0;
    session->cipherSuite = ssl->options.cipherSuite;
#endif
#if defined(OPENSSL_EXTRA) || defined(OPENSSL_EXTRA_X509_SMALL)
    session->peerVerifyRet = (byte)ssl->peerVerifyRet;
#endif
    /* Do this last so that if it fails, the rest of the session is setup. Do
     * this only for the client because if the server doesn't have an ID at
     * this point, it won't on resumption. */
    if (idSz == 0 && ssl->options.side == WOLFSSL_CLIENT_END) {
        WC_RNG* rng = NULL;
        if (ssl->rng != NULL)
            rng = ssl->rng;
#if defined(HAVE_GLOBAL_RNG) && defined(OPENSSL_EXTRA)
        else if (initGlobalRNG == 1 || wolfSSL_RAND_Init() == WOLFSSL_SUCCESS) {
            rng = &globalRNG;
        }
#endif
        if (wc_RNG_GenerateBlock(rng, ssl->session->altSessionID,
                ID_LEN) != 0)
            return;
        ssl->session->haveAltSessionID = 1;
        id = ssl->session->altSessionID;
        idSz = ID_LEN;
    }
    /* Setup done */

#ifdef HAVE_EXT_CACHE
    if (!ssl->options.internalCacheOff)
#endif
    {
        /* Try to add the session to cache. Its ok if we don't succeed. */
        (void)AddSessionToCache(session, id, idSz,
#ifdef SESSION_INDEX
                &ssl->sessionIndex,
#else
                NULL,
#endif
                ssl->options.side,
#ifdef HAVE_SESSION_TICKET
                ssl->options.useTicket,
#else
                0,
#endif
                NULL
                );
    }

#ifdef HAVE_EXT_CACHE
    if (error == 0 && ssl->ctx->new_sess_cb != NULL) {
        wolfSSL_SESSION_up_ref(session);
        cbRet = ssl->ctx->new_sess_cb(ssl, session);
        if (cbRet == 0)
            wolfSSL_FreeSession(session);
    }
#endif

#if defined(WOLFSSL_SESSION_STATS) && defined(WOLFSSL_PEAK_SESSIONS)
    if (error == 0) {
        word32 active = 0;

        error = get_locked_session_stats(&active, NULL, NULL);
        if (error == WOLFSSL_SUCCESS) {
            error = 0;  /* back to this function ok */

            if (PeakSessions < active) {
                PeakSessions = active;
            }
        }
    }
#endif /* WOLFSSL_SESSION_STATS && WOLFSSL_PEAK_SESSIONS */
    (void)error;
}


#ifdef SESSION_INDEX

int wolfSSL_GetSessionIndex(WOLFSSL* ssl)
{
    WOLFSSL_ENTER("wolfSSL_GetSessionIndex");
    WOLFSSL_LEAVE("wolfSSL_GetSessionIndex", ssl->sessionIndex);
    return ssl->sessionIndex;
}


int wolfSSL_GetSessionAtIndex(int idx, WOLFSSL_SESSION* session)
{
    int row, col, result = WOLFSSL_FAILURE;
    SessionRow* sessRow;

    WOLFSSL_ENTER("wolfSSL_GetSessionAtIndex");

    session = ClientSessionToSession(session);

    row = idx >> SESSIDX_ROW_SHIFT;
    col = idx & SESSIDX_IDX_MASK;

    if (session == NULL ||
            row < 0 || row >= SESSION_ROWS || col >= SESSIONS_PER_ROW) {
        return WOLFSSL_FAILURE;
    }

    sessRow = &SessionCache[row];
    if (SESSION_ROW_LOCK(sessRow) != 0) {
        return BAD_MUTEX_E;
    }

    XMEMCPY(session, &sessRow->Sessions[col], sizeof(WOLFSSL_SESSION));
    result = WOLFSSL_SUCCESS;

    SESSION_ROW_UNLOCK(sessRow);

    WOLFSSL_LEAVE("wolfSSL_GetSessionAtIndex", result);
    return result;
}

#endif /* SESSION_INDEX */

#if defined(SESSION_CERTS)

WOLFSSL_X509_CHAIN* wolfSSL_SESSION_get_peer_chain(WOLFSSL_SESSION* session)
{
    WOLFSSL_X509_CHAIN* chain = NULL;

    WOLFSSL_ENTER("wolfSSL_SESSION_get_peer_chain");

    session = ClientSessionToSession(session);

    if (session)
        chain = &session->chain;

    WOLFSSL_LEAVE("wolfSSL_SESSION_get_peer_chain", chain ? 1 : 0);
    return chain;
}


#ifdef OPENSSL_EXTRA
/* gets the peer certificate associated with the session passed in
 * returns null on failure, the caller should not free the returned pointer */
WOLFSSL_X509* wolfSSL_SESSION_get0_peer(WOLFSSL_SESSION* session)
{
    WOLFSSL_ENTER("wolfSSL_SESSION_get_peer_chain");

    session = ClientSessionToSession(session);
    if (session) {
        int count;

        count = wolfSSL_get_chain_count(&session->chain);
        if (count < 1 || count >= MAX_CHAIN_DEPTH) {
            WOLFSSL_MSG("bad count found");
            return NULL;
        }

        if (session->peer == NULL) {
            session->peer = wolfSSL_get_chain_X509(&session->chain, 0);
        }
        return session->peer;
    }
    WOLFSSL_MSG("No session passed in");

    return NULL;
}
#endif /* OPENSSL_EXTRA */
#endif /* SESSION_INDEX && SESSION_CERTS */


#ifdef WOLFSSL_SESSION_STATS

static int get_locked_session_stats(word32* active, word32* total, word32* peak)
{
    int result = WOLFSSL_SUCCESS;
    int i;
    int count;
    int idx;
    word32 now   = 0;
    word32 seen  = 0;
    word32 ticks = LowResTimer();

    WOLFSSL_ENTER("get_locked_session_stats");

#ifndef ENABLE_SESSION_CACHE_ROW_LOCK
    wc_LockMutex(&session_mutex);
#endif
    for (i = 0; i < SESSION_ROWS; i++) {
        SessionRow* row = &SessionCache[i];
    #ifdef ENABLE_SESSION_CACHE_ROW_LOCK
        if (SESSION_ROW_LOCK(row) != 0) {
            WOLFSSL_MSG("Session row cache mutex lock failed");
            return BAD_MUTEX_E;
        }
    #endif

        seen += row->totalCount;

        if (active == NULL) {
            SESSION_ROW_UNLOCK(row);
            continue;
        }

        count = min((word32)row->totalCount, SESSIONS_PER_ROW);
        idx   = row->nextIdx - 1;
        if (idx < 0 || idx >= SESSIONS_PER_ROW) {
            idx = SESSIONS_PER_ROW - 1; /* if back to front previous was end */
        }

        for (; count > 0; --count) {
            /* if not expired then good */
            if (ticks < (row->Sessions[idx].bornOn +
                            row->Sessions[idx].timeout) ) {
                now++;
            }

            idx = idx > 0 ? idx - 1 : SESSIONS_PER_ROW - 1;
        }

    #ifdef ENABLE_SESSION_CACHE_ROW_LOCK
        SESSION_ROW_UNLOCK(row);
    #endif
    }
#ifndef ENABLE_SESSION_CACHE_ROW_LOCK
    wc_UnLockMutex(&session_mutex);
#endif

    if (active) {
        *active = now;
    }
    if (total) {
        *total = seen;
    }

#ifdef WOLFSSL_PEAK_SESSIONS
    if (peak) {
        *peak = PeakSessions;
    }
#else
    (void)peak;
#endif

    WOLFSSL_LEAVE("get_locked_session_stats", result);

    return result;
}


/* return WOLFSSL_SUCCESS on ok */
int wolfSSL_get_session_stats(word32* active, word32* total, word32* peak,
                              word32* maxSessions)
{
    int result = WOLFSSL_SUCCESS;

    WOLFSSL_ENTER("wolfSSL_get_session_stats");

    if (maxSessions) {
        *maxSessions = SESSIONS_PER_ROW * SESSION_ROWS;

        if (active == NULL && total == NULL && peak == NULL)
            return result;  /* we're done */
    }

    /* user must provide at least one query value */
    if (active == NULL && total == NULL && peak == NULL) {
        return BAD_FUNC_ARG;
    }

    result = get_locked_session_stats(active, total, peak);

    WOLFSSL_LEAVE("wolfSSL_get_session_stats", result);

    return result;
}

#endif /* WOLFSSL_SESSION_STATS */


    #ifdef PRINT_SESSION_STATS

    /* WOLFSSL_SUCCESS on ok */
    int wolfSSL_PrintSessionStats(void)
    {
        word32 totalSessionsSeen = 0;
        word32 totalSessionsNow = 0;
        word32 peak = 0;
        word32 maxSessions = 0;
        int    i;
        int    ret;
        double E;               /* expected freq */
        double chiSquare = 0;

        ret = wolfSSL_get_session_stats(&totalSessionsNow, &totalSessionsSeen,
                                        &peak, &maxSessions);
        if (ret != WOLFSSL_SUCCESS)
            return ret;
        printf("Total Sessions Seen = %u\n", totalSessionsSeen);
        printf("Total Sessions Now  = %u\n", totalSessionsNow);
#ifdef WOLFSSL_PEAK_SESSIONS
        printf("Peak  Sessions      = %u\n", peak);
#endif
        printf("Max   Sessions      = %u\n", maxSessions);

        E = (double)totalSessionsSeen / SESSION_ROWS;

        for (i = 0; i < SESSION_ROWS; i++) {
            double diff = SessionCache[i].totalCount - E;
            diff *= diff;                /* square    */
            diff /= E;                   /* normalize */

            chiSquare += diff;
        }
        printf("  chi-square = %5.1f, d.f. = %d\n", chiSquare,
                                                     SESSION_ROWS - 1);
        #if (SESSION_ROWS == 11)
            printf(" .05 p value =  18.3, chi-square should be less\n");
        #elif (SESSION_ROWS == 211)
            printf(".05 p value  = 244.8, chi-square should be less\n");
        #elif (SESSION_ROWS == 5981)
            printf(".05 p value  = 6161.0, chi-square should be less\n");
        #elif (SESSION_ROWS == 3)
            printf(".05 p value  =   6.0, chi-square should be less\n");
        #elif (SESSION_ROWS == 2861)
            printf(".05 p value  = 2985.5, chi-square should be less\n");
        #endif
        printf("\n");

        return ret;
    }

    #endif /* SESSION_STATS */

#else  /* NO_SESSION_CACHE */

WOLFSSL_SESSION* ClientSessionToSession(const WOLFSSL_SESSION* session)
{
    return (WOLFSSL_SESSION*)session;
}

/* No session cache version */
WOLFSSL_SESSION* wolfSSL_GetSession(WOLFSSL* ssl, byte* masterSecret,
        byte restoreSessionCerts)
{
    (void)ssl;
    (void)masterSecret;
    (void)restoreSessionCerts;

    return NULL;
}

#endif /* NO_SESSION_CACHE */


/* call before SSL_connect, if verifying will add name check to
   date check and signature check */
WOLFSSL_ABI
int wolfSSL_check_domain_name(WOLFSSL* ssl, const char* dn)
{
    WOLFSSL_ENTER("wolfSSL_check_domain_name");

    if (ssl == NULL || dn == NULL) {
        WOLFSSL_MSG("Bad function argument: NULL");
        return WOLFSSL_FAILURE;
    }

    if (ssl->buffers.domainName.buffer)
        XFREE(ssl->buffers.domainName.buffer, ssl->heap, DYNAMIC_TYPE_DOMAIN);

    ssl->buffers.domainName.length = (word32)XSTRLEN(dn);
    ssl->buffers.domainName.buffer = (byte*)XMALLOC(
            ssl->buffers.domainName.length + 1, ssl->heap, DYNAMIC_TYPE_DOMAIN);

    if (ssl->buffers.domainName.buffer) {
        unsigned char* domainName = ssl->buffers.domainName.buffer;
        XMEMCPY(domainName, dn, ssl->buffers.domainName.length);
        domainName[ssl->buffers.domainName.length] = '\0';
        return WOLFSSL_SUCCESS;
    }
    else {
        ssl->error = MEMORY_ERROR;
        return WOLFSSL_FAILURE;
    }
}


/* turn on wolfSSL zlib compression
   returns WOLFSSL_SUCCESS for success, else error (not built in)
*/
int wolfSSL_set_compression(WOLFSSL* ssl)
{
    WOLFSSL_ENTER("wolfSSL_set_compression");
    (void)ssl;
#ifdef HAVE_LIBZ
    ssl->options.usingCompression = 1;
    return WOLFSSL_SUCCESS;
#else
    return NOT_COMPILED_IN;
#endif
}


#ifndef USE_WINDOWS_API
    #ifndef NO_WRITEV

        /* simulate writev semantics, doesn't actually do block at a time though
           because of SSL_write behavior and because front adds may be small */
        int wolfSSL_writev(WOLFSSL* ssl, const struct iovec* iov, int iovcnt)
        {
        #ifdef WOLFSSL_SMALL_STACK
            byte   staticBuffer[1]; /* force heap usage */
        #else
            byte   staticBuffer[FILE_BUFFER_SIZE];
        #endif
            byte* myBuffer  = staticBuffer;
            int   dynamic   = 0;
            int   sending   = 0;
            int   idx       = 0;
            int   i;
            int   ret;

            WOLFSSL_ENTER("wolfSSL_writev");

            for (i = 0; i < iovcnt; i++)
                sending += (int)iov[i].iov_len;

            if (sending > (int)sizeof(staticBuffer)) {
                myBuffer = (byte*)XMALLOC(sending, ssl->heap,
                                                           DYNAMIC_TYPE_WRITEV);
                if (!myBuffer)
                    return MEMORY_ERROR;

                dynamic = 1;
            }

            for (i = 0; i < iovcnt; i++) {
                XMEMCPY(&myBuffer[idx], iov[i].iov_base, iov[i].iov_len);
                idx += (int)iov[i].iov_len;
            }

           /* myBuffer may not be initialized fully, but the span up to the
            * sending length will be.
            */
            PRAGMA_GCC_DIAG_PUSH;
            PRAGMA_GCC("GCC diagnostic ignored \"-Wmaybe-uninitialized\"");
            ret = wolfSSL_write(ssl, myBuffer, sending);
            PRAGMA_GCC_DIAG_POP;

            if (dynamic)
                XFREE(myBuffer, ssl->heap, DYNAMIC_TYPE_WRITEV);

            return ret;
        }
    #endif
#endif


#ifdef WOLFSSL_CALLBACKS

    typedef struct itimerval Itimerval;

    /* don't keep calling simple functions while setting up timer and signals
       if no inlining these are the next best */

    #define AddTimes(a, b, c)                       \
        do {                                        \
            c.tv_sec  = a.tv_sec  + b.tv_sec;       \
            c.tv_usec = a.tv_usec + b.tv_usec;      \
            if (c.tv_usec >=  1000000) {            \
                c.tv_sec++;                         \
                c.tv_usec -= 1000000;               \
            }                                       \
        } while (0)


    #define SubtractTimes(a, b, c)                  \
        do {                                        \
            c.tv_sec  = a.tv_sec  - b.tv_sec;       \
            c.tv_usec = a.tv_usec - b.tv_usec;      \
            if (c.tv_usec < 0) {                    \
                c.tv_sec--;                         \
                c.tv_usec += 1000000;               \
            }                                       \
        } while (0)

    #define CmpTimes(a, b, cmp)                     \
        ((a.tv_sec  ==  b.tv_sec) ?                 \
            (a.tv_usec cmp b.tv_usec) :             \
            (a.tv_sec  cmp b.tv_sec))               \


    /* do nothing handler */
    static void myHandler(int signo)
    {
        (void)signo;
        return;
    }


    static int wolfSSL_ex_wrapper(WOLFSSL* ssl, HandShakeCallBack hsCb,
                                 TimeoutCallBack toCb, WOLFSSL_TIMEVAL timeout)
    {
        int       ret        = WOLFSSL_FATAL_ERROR;
        int       oldTimerOn = 0;   /* was timer already on */
        WOLFSSL_TIMEVAL startTime;
        WOLFSSL_TIMEVAL endTime;
        WOLFSSL_TIMEVAL totalTime;
        Itimerval myTimeout;
        Itimerval oldTimeout; /* if old timer adjust from total time to reset */
        struct sigaction act, oact;

        #define ERR_OUT(x) { ssl->hsInfoOn = 0; ssl->toInfoOn = 0; return x; }

        if (hsCb) {
            ssl->hsInfoOn = 1;
            InitHandShakeInfo(&ssl->handShakeInfo, ssl);
        }
        if (toCb) {
            ssl->toInfoOn = 1;
            InitTimeoutInfo(&ssl->timeoutInfo);

            if (gettimeofday(&startTime, 0) < 0)
                ERR_OUT(GETTIME_ERROR);

            /* use setitimer to simulate getitimer, init 0 myTimeout */
            myTimeout.it_interval.tv_sec  = 0;
            myTimeout.it_interval.tv_usec = 0;
            myTimeout.it_value.tv_sec     = 0;
            myTimeout.it_value.tv_usec    = 0;
            if (setitimer(ITIMER_REAL, &myTimeout, &oldTimeout) < 0)
                ERR_OUT(SETITIMER_ERROR);

            if (oldTimeout.it_value.tv_sec || oldTimeout.it_value.tv_usec) {
                oldTimerOn = 1;

                /* is old timer going to expire before ours */
                if (CmpTimes(oldTimeout.it_value, timeout, <)) {
                    timeout.tv_sec  = oldTimeout.it_value.tv_sec;
                    timeout.tv_usec = oldTimeout.it_value.tv_usec;
                }
            }
            myTimeout.it_value.tv_sec  = timeout.tv_sec;
            myTimeout.it_value.tv_usec = timeout.tv_usec;

            /* set up signal handler, don't restart socket send/recv */
            act.sa_handler = myHandler;
            sigemptyset(&act.sa_mask);
            act.sa_flags = 0;
#ifdef SA_INTERRUPT
            act.sa_flags |= SA_INTERRUPT;
#endif
            if (sigaction(SIGALRM, &act, &oact) < 0)
                ERR_OUT(SIGACT_ERROR);

            if (setitimer(ITIMER_REAL, &myTimeout, 0) < 0)
                ERR_OUT(SETITIMER_ERROR);
        }

        /* do main work */
#ifndef NO_WOLFSSL_CLIENT
        if (ssl->options.side == WOLFSSL_CLIENT_END)
            ret = wolfSSL_connect(ssl);
#endif
#ifndef NO_WOLFSSL_SERVER
        if (ssl->options.side == WOLFSSL_SERVER_END)
            ret = wolfSSL_accept(ssl);
#endif

        /* do callbacks */
        if (toCb) {
            if (oldTimerOn) {
                gettimeofday(&endTime, 0);
                SubtractTimes(endTime, startTime, totalTime);
                /* adjust old timer for elapsed time */
                if (CmpTimes(totalTime, oldTimeout.it_value, <))
                    SubtractTimes(oldTimeout.it_value, totalTime,
                                  oldTimeout.it_value);
                else {
                    /* reset value to interval, may be off */
                    oldTimeout.it_value.tv_sec = oldTimeout.it_interval.tv_sec;
                    oldTimeout.it_value.tv_usec =oldTimeout.it_interval.tv_usec;
                }
                /* keep iter the same whether there or not */
            }
            /* restore old handler */
            if (sigaction(SIGALRM, &oact, 0) < 0)
                ret = SIGACT_ERROR;    /* more pressing error, stomp */
            else
                /* use old settings which may turn off (expired or not there) */
                if (setitimer(ITIMER_REAL, &oldTimeout, 0) < 0)
                    ret = SETITIMER_ERROR;

            /* if we had a timeout call callback */
            if (ssl->timeoutInfo.timeoutName[0]) {
                ssl->timeoutInfo.timeoutValue.tv_sec  = timeout.tv_sec;
                ssl->timeoutInfo.timeoutValue.tv_usec = timeout.tv_usec;
                (toCb)(&ssl->timeoutInfo);
            }
            /* clean up */
            FreeTimeoutInfo(&ssl->timeoutInfo, ssl->heap);
            ssl->toInfoOn = 0;
        }
        if (hsCb) {
            FinishHandShakeInfo(&ssl->handShakeInfo);
            (hsCb)(&ssl->handShakeInfo);
            ssl->hsInfoOn = 0;
        }
        return ret;
    }


#ifndef NO_WOLFSSL_CLIENT

    int wolfSSL_connect_ex(WOLFSSL* ssl, HandShakeCallBack hsCb,
                          TimeoutCallBack toCb, WOLFSSL_TIMEVAL timeout)
    {
        WOLFSSL_ENTER("wolfSSL_connect_ex");
        return wolfSSL_ex_wrapper(ssl, hsCb, toCb, timeout);
    }

#endif


#ifndef NO_WOLFSSL_SERVER

    int wolfSSL_accept_ex(WOLFSSL* ssl, HandShakeCallBack hsCb,
                         TimeoutCallBack toCb, WOLFSSL_TIMEVAL timeout)
    {
        WOLFSSL_ENTER("wolfSSL_accept_ex");
        return wolfSSL_ex_wrapper(ssl, hsCb, toCb, timeout);
    }

#endif

#endif /* WOLFSSL_CALLBACKS */


#ifndef NO_PSK

    void wolfSSL_CTX_set_psk_client_callback(WOLFSSL_CTX* ctx,
                                         wc_psk_client_callback cb)
    {
        WOLFSSL_ENTER("SSL_CTX_set_psk_client_callback");

        if (ctx == NULL)
            return;

        ctx->havePSK = 1;
        ctx->client_psk_cb = cb;
    }

    void wolfSSL_set_psk_client_callback(WOLFSSL* ssl,wc_psk_client_callback cb)
    {
        byte haveRSA = 1;
        int  keySz   = 0;

        WOLFSSL_ENTER("SSL_set_psk_client_callback");

        if (ssl == NULL)
            return;

        ssl->options.havePSK = 1;
        ssl->options.client_psk_cb = cb;

        #ifdef NO_RSA
            haveRSA = 0;
        #endif
        #ifndef NO_CERTS
            keySz = ssl->buffers.keySz;
        #endif
        InitSuites(ssl->suites, ssl->version, keySz, haveRSA, TRUE,
                   ssl->options.haveDH, ssl->options.haveECDSAsig,
                   ssl->options.haveECC, ssl->options.haveStaticECC,
                   ssl->options.haveFalconSig, ssl->options.haveAnon,
                   ssl->options.side);
    }
    #ifdef OPENSSL_EXTRA
    /**
     * set call back function for psk session use
     * @param ssl  a pointer to WOLFSSL structure
     * @param cb   a function pointer to wc_psk_use_session_cb
     * @return none
     */
    void wolfSSL_set_psk_use_session_callback(WOLFSSL* ssl,
                                                wc_psk_use_session_cb_func cb)
    {
        WOLFSSL_ENTER("wolfSSL_set_psk_use_session_callback");

        ssl->options.havePSK = 1;
        ssl->options.session_psk_cb = cb;

        WOLFSSL_LEAVE("wolfSSL_set_psk_use_session_callback", WOLFSSL_SUCCESS);
    }
    #endif

    void wolfSSL_CTX_set_psk_server_callback(WOLFSSL_CTX* ctx,
                                         wc_psk_server_callback cb)
    {
        WOLFSSL_ENTER("SSL_CTX_set_psk_server_callback");
        if (ctx == NULL)
            return;
        ctx->havePSK = 1;
        ctx->server_psk_cb = cb;
    }

    void wolfSSL_set_psk_server_callback(WOLFSSL* ssl,wc_psk_server_callback cb)
    {
        byte haveRSA = 1;
        int  keySz   = 0;

        WOLFSSL_ENTER("SSL_set_psk_server_callback");
        if (ssl == NULL)
            return;

        ssl->options.havePSK = 1;
        ssl->options.server_psk_cb = cb;

        #ifdef NO_RSA
            haveRSA = 0;
        #endif
        #ifndef NO_CERTS
            keySz = ssl->buffers.keySz;
        #endif
        InitSuites(ssl->suites, ssl->version, keySz, haveRSA, TRUE,
                   ssl->options.haveDH, ssl->options.haveECDSAsig,
                   ssl->options.haveECC, ssl->options.haveStaticECC,
                   ssl->options.haveFalconSig, ssl->options.haveAnon,
                   ssl->options.side);
    }

    const char* wolfSSL_get_psk_identity_hint(const WOLFSSL* ssl)
    {
        WOLFSSL_ENTER("SSL_get_psk_identity_hint");

        if (ssl == NULL || ssl->arrays == NULL)
            return NULL;

        return ssl->arrays->server_hint;
    }


    const char* wolfSSL_get_psk_identity(const WOLFSSL* ssl)
    {
        WOLFSSL_ENTER("SSL_get_psk_identity");

        if (ssl == NULL || ssl->arrays == NULL)
            return NULL;

        return ssl->arrays->client_identity;
    }

    int wolfSSL_CTX_use_psk_identity_hint(WOLFSSL_CTX* ctx, const char* hint)
    {
        WOLFSSL_ENTER("SSL_CTX_use_psk_identity_hint");
        if (hint == 0)
            ctx->server_hint[0] = '\0';
        else {
            /* Qt does not call CTX_set_*_psk_callbacks where havePSK is set */
            #ifdef WOLFSSL_QT
            ctx->havePSK=1;
            #endif
            XSTRNCPY(ctx->server_hint, hint, MAX_PSK_ID_LEN);
            ctx->server_hint[MAX_PSK_ID_LEN] = '\0'; /* null term */
        }
        return WOLFSSL_SUCCESS;
    }

    int wolfSSL_use_psk_identity_hint(WOLFSSL* ssl, const char* hint)
    {
        WOLFSSL_ENTER("SSL_use_psk_identity_hint");

        if (ssl == NULL || ssl->arrays == NULL)
            return WOLFSSL_FAILURE;

        if (hint == 0)
            ssl->arrays->server_hint[0] = 0;
        else {
            XSTRNCPY(ssl->arrays->server_hint, hint,
                                            sizeof(ssl->arrays->server_hint)-1);
            ssl->arrays->server_hint[sizeof(ssl->arrays->server_hint)-1] = '\0';
        }
        return WOLFSSL_SUCCESS;
    }

    void* wolfSSL_get_psk_callback_ctx(WOLFSSL* ssl)
    {
        return ssl ? ssl->options.psk_ctx : NULL;
    }
    void* wolfSSL_CTX_get_psk_callback_ctx(WOLFSSL_CTX* ctx)
    {
        return ctx ? ctx->psk_ctx : NULL;
    }
    int wolfSSL_set_psk_callback_ctx(WOLFSSL* ssl, void* psk_ctx)
    {
        if (ssl == NULL)
            return WOLFSSL_FAILURE;
        ssl->options.psk_ctx = psk_ctx;
        return WOLFSSL_SUCCESS;
    }
    int wolfSSL_CTX_set_psk_callback_ctx(WOLFSSL_CTX* ctx, void* psk_ctx)
    {
        if (ctx == NULL)
            return WOLFSSL_FAILURE;
        ctx->psk_ctx = psk_ctx;
        return WOLFSSL_SUCCESS;
    }
#endif /* NO_PSK */


#ifdef HAVE_ANON

    int wolfSSL_CTX_allow_anon_cipher(WOLFSSL_CTX* ctx)
    {
        WOLFSSL_ENTER("wolfSSL_CTX_allow_anon_cipher");

        if (ctx == NULL)
            return WOLFSSL_FAILURE;

        ctx->haveAnon = 1;

        return WOLFSSL_SUCCESS;
    }

#endif /* HAVE_ANON */


#ifndef NO_CERTS
/* used to be defined on NO_FILESYSTEM only, but are generally useful */

    int wolfSSL_CTX_load_verify_buffer_ex(WOLFSSL_CTX* ctx,
                                         const unsigned char* in,
                                         long sz, int format, int userChain,
                                         word32 flags)
    {
        int verify;
        int ret = WOLFSSL_FAILURE;

        WOLFSSL_ENTER("wolfSSL_CTX_load_verify_buffer_ex");

        verify = GET_VERIFY_SETTING_CTX(ctx);
        if (flags & WOLFSSL_LOAD_FLAG_DATE_ERR_OKAY)
            verify = VERIFY_SKIP_DATE;

        if (format == WOLFSSL_FILETYPE_PEM)
            ret = ProcessChainBuffer(ctx, in, sz, format, CA_TYPE, NULL,
                                      verify);
        else
            ret = ProcessBuffer(ctx, in, sz, format, CA_TYPE, NULL, NULL,
                                 userChain, verify);
#if defined(WOLFSSL_TRUST_PEER_CERT) && defined(OPENSSL_COMPATIBLE_DEFAULTS)
        if (ret == WOLFSSL_SUCCESS)
            ret = wolfSSL_CTX_trust_peer_buffer(ctx, in, sz, format);
#endif

        WOLFSSL_LEAVE("wolfSSL_CTX_load_verify_buffer_ex", ret);
        return ret;
    }

    /* wolfSSL extension allows DER files to be loaded from buffers as well */
    int wolfSSL_CTX_load_verify_buffer(WOLFSSL_CTX* ctx,
                                       const unsigned char* in,
                                       long sz, int format)
    {
        return wolfSSL_CTX_load_verify_buffer_ex(ctx, in, sz, format, 0,
            WOLFSSL_LOAD_VERIFY_DEFAULT_FLAGS);
    }

    int wolfSSL_CTX_load_verify_chain_buffer_format(WOLFSSL_CTX* ctx,
                                       const unsigned char* in,
                                       long sz, int format)
    {
        return wolfSSL_CTX_load_verify_buffer_ex(ctx, in, sz, format, 1,
            WOLFSSL_LOAD_VERIFY_DEFAULT_FLAGS);
    }


#ifdef WOLFSSL_TRUST_PEER_CERT
    int wolfSSL_CTX_trust_peer_buffer(WOLFSSL_CTX* ctx,
                                       const unsigned char* in,
                                       long sz, int format)
    {
        WOLFSSL_ENTER("wolfSSL_CTX_trust_peer_buffer");

        /* sanity check on arguments */
        if (sz < 0 || in == NULL || ctx == NULL) {
            return BAD_FUNC_ARG;
        }

        if (format == WOLFSSL_FILETYPE_PEM)
            return ProcessChainBuffer(ctx, in, sz, format, TRUSTED_PEER_TYPE,
                                      NULL, GET_VERIFY_SETTING_CTX(ctx));
        else
            return ProcessBuffer(ctx, in, sz, format, TRUSTED_PEER_TYPE, NULL,
                                 NULL, 0, GET_VERIFY_SETTING_CTX(ctx));
    }
#endif /* WOLFSSL_TRUST_PEER_CERT */


    int wolfSSL_CTX_use_certificate_buffer(WOLFSSL_CTX* ctx,
                                 const unsigned char* in, long sz, int format)
    {
        int ret = WOLFSSL_FAILURE;

        WOLFSSL_ENTER("wolfSSL_CTX_use_certificate_buffer");
        ret = ProcessBuffer(ctx, in, sz, format, CERT_TYPE, NULL, NULL, 0,
                             GET_VERIFY_SETTING_CTX(ctx));
        WOLFSSL_LEAVE("wolfSSL_CTX_use_certificate_buffer", ret);
        return ret;
    }


    int wolfSSL_CTX_use_PrivateKey_buffer(WOLFSSL_CTX* ctx,
                                 const unsigned char* in, long sz, int format)
    {
        int ret = WOLFSSL_FAILURE;

        WOLFSSL_ENTER("wolfSSL_CTX_use_PrivateKey_buffer");
        ret = ProcessBuffer(ctx, in, sz, format, PRIVATEKEY_TYPE, NULL, NULL,
                             0, GET_VERIFY_SETTING_CTX(ctx));
        WOLFSSL_LEAVE("wolfSSL_CTX_use_PrivateKey_buffer", ret);
        return ret;
    }

#ifdef WOLF_PRIVATE_KEY_ID
    int wolfSSL_CTX_use_PrivateKey_id(WOLFSSL_CTX* ctx, const unsigned char* id,
                                      long sz, int devId, long keySz)
    {
        int ret = wolfSSL_CTX_use_PrivateKey_Id(ctx, id, sz, devId);

        if (ret == WOLFSSL_SUCCESS)
            ctx->privateKeySz = (word32)keySz;

        return ret;
    }

    int wolfSSL_CTX_use_PrivateKey_Id(WOLFSSL_CTX* ctx, const unsigned char* id,
                                      long sz, int devId)
    {
        int ret = WOLFSSL_FAILURE;

        FreeDer(&ctx->privateKey);
        if (AllocDer(&ctx->privateKey, (word32)sz, PRIVATEKEY_TYPE,
                                                              ctx->heap) == 0) {
            XMEMCPY(ctx->privateKey->buffer, id, sz);
            ctx->privateKeyId = 1;
            if (devId != INVALID_DEVID)
                ctx->privateKeyDevId = devId;
            else
                ctx->privateKeyDevId = ctx->devId;

            ret = WOLFSSL_SUCCESS;
        }

        return ret;
    }

    int wolfSSL_CTX_use_PrivateKey_Label(WOLFSSL_CTX* ctx, const char* label,
                                         int devId)
    {
        int ret = WOLFSSL_FAILURE;
        word32 sz = (word32)XSTRLEN(label) + 1;

        FreeDer(&ctx->privateKey);
        if (AllocDer(&ctx->privateKey, (word32)sz, PRIVATEKEY_TYPE,
                                                              ctx->heap) == 0) {
            XMEMCPY(ctx->privateKey->buffer, label, sz);
            ctx->privateKeyLabel = 1;
            if (devId != INVALID_DEVID)
                ctx->privateKeyDevId = devId;
            else
                ctx->privateKeyDevId = ctx->devId;

            ret = WOLFSSL_SUCCESS;
        }

        return ret;
    }
#endif /* WOLF_PRIVATE_KEY_ID */

    int wolfSSL_CTX_use_certificate_chain_buffer_format(WOLFSSL_CTX* ctx,
                                 const unsigned char* in, long sz, int format)
    {
        WOLFSSL_ENTER("wolfSSL_CTX_use_certificate_chain_buffer_format");
        return ProcessBuffer(ctx, in, sz, format, CERT_TYPE, NULL, NULL, 1,
                             GET_VERIFY_SETTING_CTX(ctx));
    }

    int wolfSSL_CTX_use_certificate_chain_buffer(WOLFSSL_CTX* ctx,
                                 const unsigned char* in, long sz)
    {
        return wolfSSL_CTX_use_certificate_chain_buffer_format(ctx, in, sz,
                                                            WOLFSSL_FILETYPE_PEM);
    }


#ifndef NO_DH

    /* server wrapper for ctx or ssl Diffie-Hellman parameters */
    static int wolfSSL_SetTmpDH_buffer_wrapper(WOLFSSL_CTX* ctx, WOLFSSL* ssl,
                                               const unsigned char* buf,
                                               long sz, int format)
    {
        DerBuffer* der = NULL;
        int    ret      = 0;
        word32 pSz = MAX_DH_SIZE;
        word32 gSz = MAX_DH_SIZE;
    #ifdef WOLFSSL_SMALL_STACK
        byte*  p = NULL;
        byte*  g = NULL;
    #else
        byte   p[MAX_DH_SIZE];
        byte   g[MAX_DH_SIZE];
    #endif

        if (ctx == NULL || buf == NULL)
            return BAD_FUNC_ARG;

        ret = AllocDer(&der, 0, DH_PARAM_TYPE, ctx->heap);
        if (ret != 0) {
            return ret;
        }
        der->buffer = (byte*)buf;
        der->length = (word32)sz;

    #ifdef WOLFSSL_SMALL_STACK
        p = (byte*)XMALLOC(pSz, NULL, DYNAMIC_TYPE_PUBLIC_KEY);
        g = (byte*)XMALLOC(gSz, NULL, DYNAMIC_TYPE_PUBLIC_KEY);

        if (p == NULL || g == NULL) {
            XFREE(p, NULL, DYNAMIC_TYPE_PUBLIC_KEY);
            XFREE(g, NULL, DYNAMIC_TYPE_PUBLIC_KEY);
            return MEMORY_E;
        }
    #endif

        if (format != WOLFSSL_FILETYPE_ASN1 && format != WOLFSSL_FILETYPE_PEM)
            ret = WOLFSSL_BAD_FILETYPE;
        else {
            if (format == WOLFSSL_FILETYPE_PEM) {
#ifdef WOLFSSL_PEM_TO_DER
                FreeDer(&der);
                ret = PemToDer(buf, sz, DH_PARAM_TYPE, &der, ctx->heap,
                               NULL, NULL);
                if (ret < 0) {
                    /* Also try X9.42 format */
                    ret = PemToDer(buf, sz, X942_PARAM_TYPE, &der, ctx->heap,
                               NULL, NULL);
                }
    #ifdef WOLFSSL_WPAS
        #ifndef NO_DSA
                if (ret < 0) {
                    ret = PemToDer(buf, sz, DSA_PARAM_TYPE, &der, ctx->heap,
                               NULL, NULL);
                }
        #endif
    #endif /* WOLFSSL_WPAS */
#else
                ret = NOT_COMPILED_IN;
#endif /* WOLFSSL_PEM_TO_DER */
            }

            if (ret == 0) {
                if (wc_DhParamsLoad(der->buffer, der->length, p, &pSz, g, &gSz) < 0)
                    ret = WOLFSSL_BAD_FILETYPE;
                else if (ssl)
                    ret = wolfSSL_SetTmpDH(ssl, p, pSz, g, gSz);
                else
                    ret = wolfSSL_CTX_SetTmpDH(ctx, p, pSz, g, gSz);
            }
        }

        FreeDer(&der);

    #ifdef WOLFSSL_SMALL_STACK
        XFREE(p, NULL, DYNAMIC_TYPE_PUBLIC_KEY);
        XFREE(g, NULL, DYNAMIC_TYPE_PUBLIC_KEY);
    #endif

        return ret;
    }


    /* server Diffie-Hellman parameters, WOLFSSL_SUCCESS on ok */
    int wolfSSL_SetTmpDH_buffer(WOLFSSL* ssl, const unsigned char* buf, long sz,
                               int format)
    {
        if (ssl == NULL)
            return BAD_FUNC_ARG;

        return wolfSSL_SetTmpDH_buffer_wrapper(ssl->ctx, ssl, buf, sz, format);
    }


    /* server ctx Diffie-Hellman parameters, WOLFSSL_SUCCESS on ok */
    int wolfSSL_CTX_SetTmpDH_buffer(WOLFSSL_CTX* ctx, const unsigned char* buf,
                                   long sz, int format)
    {
        return wolfSSL_SetTmpDH_buffer_wrapper(ctx, NULL, buf, sz, format);
    }

#endif /* NO_DH */


    int wolfSSL_use_certificate_buffer(WOLFSSL* ssl,
                                 const unsigned char* in, long sz, int format)
    {
        WOLFSSL_ENTER("wolfSSL_use_certificate_buffer");
        if (ssl == NULL)
            return BAD_FUNC_ARG;

        return ProcessBuffer(ssl->ctx, in, sz, format, CERT_TYPE, ssl, NULL, 0,
                             GET_VERIFY_SETTING_SSL(ssl));
    }


    int wolfSSL_use_PrivateKey_buffer(WOLFSSL* ssl,
                                 const unsigned char* in, long sz, int format)
    {
        WOLFSSL_ENTER("wolfSSL_use_PrivateKey_buffer");
        if (ssl == NULL)
            return BAD_FUNC_ARG;

        return ProcessBuffer(ssl->ctx, in, sz, format, PRIVATEKEY_TYPE,
                             ssl, NULL, 0, GET_VERIFY_SETTING_SSL(ssl));
    }

#ifdef WOLF_PRIVATE_KEY_ID
    int wolfSSL_use_PrivateKey_id(WOLFSSL* ssl, const unsigned char* id,
                                  long sz, int devId, long keySz)
    {
        int ret = wolfSSL_use_PrivateKey_Id(ssl, id, sz, devId);

        if (ret == WOLFSSL_SUCCESS)
            ssl->buffers.keySz = (word32)keySz;

        return ret;
    }

    int wolfSSL_use_PrivateKey_Id(WOLFSSL* ssl, const unsigned char* id,
                                  long sz, int devId)
    {
        int ret = WOLFSSL_FAILURE;

        if (ssl->buffers.weOwnKey)
            FreeDer(&ssl->buffers.key);
        if (AllocDer(&ssl->buffers.key, (word32)sz, PRIVATEKEY_TYPE,
                                                            ssl->heap) == 0) {
            XMEMCPY(ssl->buffers.key->buffer, id, sz);
            ssl->buffers.weOwnKey = 1;
            ssl->buffers.keyId = 1;
            if (devId != INVALID_DEVID)
                ssl->buffers.keyDevId = devId;
            else
                ssl->buffers.keyDevId = ssl->devId;

            ret = WOLFSSL_SUCCESS;
        }

        return ret;
    }

    int wolfSSL_use_PrivateKey_Label(WOLFSSL* ssl, const char* label, int devId)
    {
        int ret = WOLFSSL_FAILURE;
        word32 sz = (word32)XSTRLEN(label) + 1;

        if (ssl->buffers.weOwnKey)
            FreeDer(&ssl->buffers.key);
        if (AllocDer(&ssl->buffers.key, (word32)sz, PRIVATEKEY_TYPE,
                                                            ssl->heap) == 0) {
            XMEMCPY(ssl->buffers.key->buffer, label, sz);
            ssl->buffers.weOwnKey = 1;
            ssl->buffers.keyLabel = 1;
            if (devId != INVALID_DEVID)
                ssl->buffers.keyDevId = devId;
            else
                ssl->buffers.keyDevId = ssl->devId;

            ret = WOLFSSL_SUCCESS;
        }

        return ret;
    }
#endif /* WOLF_PRIVATE_KEY_ID */

    int wolfSSL_use_certificate_chain_buffer_format(WOLFSSL* ssl,
                                 const unsigned char* in, long sz, int format)
    {
        WOLFSSL_ENTER("wolfSSL_use_certificate_chain_buffer_format");
        if (ssl == NULL)
            return BAD_FUNC_ARG;

        return ProcessBuffer(ssl->ctx, in, sz, format, CERT_TYPE,
                             ssl, NULL, 1, GET_VERIFY_SETTING_SSL(ssl));
    }

    int wolfSSL_use_certificate_chain_buffer(WOLFSSL* ssl,
                                 const unsigned char* in, long sz)
    {
        return wolfSSL_use_certificate_chain_buffer_format(ssl, in, sz,
                                                            WOLFSSL_FILETYPE_PEM);
    }


    /* unload any certs or keys that SSL owns, leave CTX as is
       WOLFSSL_SUCCESS on ok */
    int wolfSSL_UnloadCertsKeys(WOLFSSL* ssl)
    {
        if (ssl == NULL) {
            WOLFSSL_MSG("Null function arg");
            return BAD_FUNC_ARG;
        }

        if (ssl->buffers.weOwnCert && !ssl->keepCert) {
            WOLFSSL_MSG("Unloading cert");
            FreeDer(&ssl->buffers.certificate);
            #ifdef KEEP_OUR_CERT
            wolfSSL_X509_free(ssl->ourCert);
            ssl->ourCert = NULL;
            #endif
            ssl->buffers.weOwnCert = 0;
        }

        if (ssl->buffers.weOwnCertChain) {
            WOLFSSL_MSG("Unloading cert chain");
            FreeDer(&ssl->buffers.certChain);
            ssl->buffers.weOwnCertChain = 0;
        }

        if (ssl->buffers.weOwnKey) {
            WOLFSSL_MSG("Unloading key");
            FreeDer(&ssl->buffers.key);
            ssl->buffers.weOwnKey = 0;
        }

        return WOLFSSL_SUCCESS;
    }


    int wolfSSL_CTX_UnloadCAs(WOLFSSL_CTX* ctx)
    {
        WOLFSSL_ENTER("wolfSSL_CTX_UnloadCAs");

        if (ctx == NULL)
            return BAD_FUNC_ARG;

        return wolfSSL_CertManagerUnloadCAs(ctx->cm);
    }


#ifdef WOLFSSL_TRUST_PEER_CERT
    int wolfSSL_CTX_Unload_trust_peers(WOLFSSL_CTX* ctx)
    {
        WOLFSSL_ENTER("wolfSSL_CTX_Unload_trust_peers");

        if (ctx == NULL)
            return BAD_FUNC_ARG;

        return wolfSSL_CertManagerUnload_trust_peers(ctx->cm);
    }

#ifdef WOLFSSL_LOCAL_X509_STORE
    int wolfSSL_Unload_trust_peers(WOLFSSL* ssl)
    {
        WOLFSSL_ENTER("wolfSSL_CTX_Unload_trust_peers");

        if (ssl == NULL)
            return BAD_FUNC_ARG;

        return wolfSSL_CertManagerUnload_trust_peers(SSL_CM(ssl));
    }
#endif /* WOLFSSL_LOCAL_X509_STORE */
#endif /* WOLFSSL_TRUST_PEER_CERT */
/* old NO_FILESYSTEM end */
#endif /* !NO_CERTS */


#ifdef OPENSSL_EXTRA

    int wolfSSL_add_all_algorithms(void)
    {
        WOLFSSL_ENTER("wolfSSL_add_all_algorithms");
        if (initRefCount != 0 || wolfSSL_Init() == WOLFSSL_SUCCESS)
            return WOLFSSL_SUCCESS;
        else
            return WOLFSSL_FATAL_ERROR;
    }

    int wolfSSL_OpenSSL_add_all_algorithms_noconf(void)
    {
        WOLFSSL_ENTER("wolfSSL_OpenSSL_add_all_algorithms_noconf");

        if  (wolfSSL_add_all_algorithms() == WOLFSSL_FATAL_ERROR)
            return WOLFSSL_FATAL_ERROR;

        return  WOLFSSL_SUCCESS;
    }

    int wolfSSL_OpenSSL_add_all_algorithms_conf(void)
    {
        WOLFSSL_ENTER("wolfSSL_OpenSSL_add_all_algorithms_conf");
        /* This function is currently the same as
        wolfSSL_OpenSSL_add_all_algorithms_noconf since we do not employ
        the use of a wolfssl.cnf type configuration file and is only used for
        OpenSSL compatability. */

        if (wolfSSL_add_all_algorithms() == WOLFSSL_FATAL_ERROR) {
            return WOLFSSL_FATAL_ERROR;
        }
        return WOLFSSL_SUCCESS;
    }

   /* returns previous set cache size which stays constant */
    long wolfSSL_CTX_sess_set_cache_size(WOLFSSL_CTX* ctx, long sz)
    {
        /* cache size fixed at compile time in wolfSSL */
        (void)ctx;
        (void)sz;
        WOLFSSL_MSG("session cache is set at compile time");
        #ifndef NO_SESSION_CACHE
            return (long)(SESSIONS_PER_ROW * SESSION_ROWS);
        #else
            return 0;
        #endif
    }

#endif

#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_EXTRA) || defined(WOLFSSL_WPAS_SMALL)
    void wolfSSL_CTX_set_quiet_shutdown(WOLFSSL_CTX* ctx, int mode)
    {
        WOLFSSL_ENTER("wolfSSL_CTX_set_quiet_shutdown");
        if (mode)
            ctx->quietShutdown = 1;
    }


    void wolfSSL_set_quiet_shutdown(WOLFSSL* ssl, int mode)
    {
        WOLFSSL_ENTER("wolfSSL_CTX_set_quiet_shutdown");
        if (mode)
            ssl->options.quietShutdown = 1;
    }
#endif /* OPENSSL_EXTRA || WOLFSSL_EXTRA || WOLFSSL_WPAS_SMALL */

#ifdef OPENSSL_EXTRA
#ifndef NO_BIO
    void wolfSSL_set_bio(WOLFSSL* ssl, WOLFSSL_BIO* rd, WOLFSSL_BIO* wr)
    {
        WOLFSSL_ENTER("wolfSSL_set_bio");

        if (ssl == NULL) {
            WOLFSSL_MSG("Bad argument, ssl was NULL");
            return;
        }

        /* free any existing WOLFSSL_BIOs in use but don't free those in
         * a chain */
        if (ssl->biord != NULL) {
            if (ssl->biord != ssl->biowr) {
                if (ssl->biowr != NULL && ssl->biowr->prev != NULL)
                    wolfSSL_BIO_free(ssl->biowr);
                ssl->biowr = NULL;
            }
            if (ssl->biord->prev != NULL)
                wolfSSL_BIO_free(ssl->biord);
            ssl->biord = NULL;
        }
        /* set flag obviously */
        if (rd && !(rd->flags & WOLFSSL_BIO_FLAG_READ))
            rd->flags |= WOLFSSL_BIO_FLAG_READ;
        if (wr && !(wr->flags & WOLFSSL_BIO_FLAG_WRITE))
            wr->flags |= WOLFSSL_BIO_FLAG_WRITE;

        ssl->biord = rd;
        ssl->biowr = wr;

        /* set SSL to use BIO callbacks instead */
        if (((ssl->cbioFlag & WOLFSSL_CBIO_RECV) == 0)) {
            ssl->CBIORecv = BioReceive;
        }
        if (((ssl->cbioFlag & WOLFSSL_CBIO_SEND) == 0)) {
            ssl->CBIOSend = BioSend;
        }

        /* User programs should always retry reading from these BIOs */
        if (rd) {
            /* User writes to rd */
            BIO_set_retry_write(rd);
        }
        if (wr) {
            /* User reads from wr */
            BIO_set_retry_read(wr);
        }
    }
#endif /* !NO_BIO */
#endif /* OPENSSL_EXTRA */

#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_EXTRA)
    void wolfSSL_CTX_set_client_CA_list(WOLFSSL_CTX* ctx,
                                       WOLF_STACK_OF(WOLFSSL_X509_NAME)* names)
    {
        WOLFSSL_ENTER("wolfSSL_CTX_set_client_CA_list");
        if (ctx != NULL) {
            wolfSSL_sk_X509_NAME_pop_free(ctx->ca_names, NULL);
            ctx->ca_names = names;
        }
    }

    void wolfSSL_set_client_CA_list(WOLFSSL* ssl,
                                       WOLF_STACK_OF(WOLFSSL_X509_NAME)* names)
    {
        WOLFSSL_ENTER("wolfSSL_set_client_CA_list");
        if (ssl != NULL) {
            if (ssl->ca_names != ssl->ctx->ca_names)
                wolfSSL_sk_X509_NAME_pop_free(ssl->ca_names, NULL);
            ssl->ca_names = names;
        }
    }

    #ifdef OPENSSL_EXTRA
    /* registers client cert callback, called during handshake if server
       requests client auth but user has not loaded client cert/key */
    void wolfSSL_CTX_set_client_cert_cb(WOLFSSL_CTX *ctx, client_cert_cb cb)
    {
        WOLFSSL_ENTER("wolfSSL_CTX_set_client_cert_cb");

        if (ctx != NULL) {
            ctx->CBClientCert = cb;
        }
    }

    void wolfSSL_CTX_set_cert_cb(WOLFSSL_CTX* ctx,
        CertSetupCallback cb, void *arg)
    {
        WOLFSSL_ENTER("wolfSSL_CTX_set_cert_cb");
        if (ctx == NULL)
            return;

        ctx->certSetupCb = cb;
        ctx->certSetupCbArg = arg;
    }

    /**
     * Internal wrapper for calling certSetupCb
     * @param ssl The SSL/TLS Object
     * @return 0 on success
     */
    int CertSetupCbWrapper(WOLFSSL* ssl)
    {
        int ret = 0;
        if (ssl->ctx->certSetupCb != NULL) {
            WOLFSSL_MSG("Calling user cert setup callback");
            ret = ssl->ctx->certSetupCb(ssl, ssl->ctx->certSetupCbArg);
            if (ret == 1) {
                WOLFSSL_MSG("User cert callback returned success");
                ret = 0;
            }
            else if (ret == 0) {
                SendAlert(ssl, alert_fatal, internal_error);
                ret = CLIENT_CERT_CB_ERROR;
            }
            else if (ret < 0) {
                ret = WOLFSSL_ERROR_WANT_X509_LOOKUP;
            }
            else {
                WOLFSSL_MSG("Unexpected user callback return");
                ret = CLIENT_CERT_CB_ERROR;
            }
        }
        return ret;
    }
    #endif /* OPENSSL_EXTRA */

#endif /* OPENSSL_EXTRA || WOLFSSL_EXTRA || HAVE_WEBSERVER */

#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_EXTRA)
    WOLF_STACK_OF(WOLFSSL_X509_NAME)* wolfSSL_CTX_get_client_CA_list(
            const WOLFSSL_CTX *ctx)
    {
        WOLFSSL_ENTER("wolfSSL_CTX_get_client_CA_list");

        if (ctx == NULL) {
            WOLFSSL_MSG("Bad argument passed to wolfSSL_CTX_get_client_CA_list");
            return NULL;
        }

        return ctx->ca_names;
    }

    /* returns the CA's set on server side or the CA's sent from server when
     * on client side */
    WOLF_STACK_OF(WOLFSSL_X509_NAME)* wolfSSL_get_client_CA_list(
            const WOLFSSL* ssl)
    {
        WOLFSSL_ENTER("wolfSSL_get_client_CA_list");

        if (ssl == NULL) {
            WOLFSSL_MSG("Bad argument passed to wolfSSL_get_client_CA_list");
            return NULL;
        }

        return SSL_CA_NAMES(ssl);
    }

    #if !defined(NO_CERTS)
    int wolfSSL_CTX_add_client_CA(WOLFSSL_CTX* ctx, WOLFSSL_X509* x509)
    {
        WOLFSSL_X509_NAME *nameCopy = NULL;

        WOLFSSL_ENTER("wolfSSL_CTX_add_client_CA");

        if (ctx == NULL || x509 == NULL){
            WOLFSSL_MSG("Bad argument");
            return WOLFSSL_FAILURE;
        }

        if (ctx->ca_names == NULL) {
            ctx->ca_names = wolfSSL_sk_X509_NAME_new(NULL);
            if (ctx->ca_names == NULL) {
                WOLFSSL_MSG("wolfSSL_sk_X509_NAME_new error");
                return WOLFSSL_FAILURE;
            }
        }

        nameCopy = wolfSSL_X509_NAME_dup(wolfSSL_X509_get_subject_name(x509));
        if (nameCopy == NULL) {
            WOLFSSL_MSG("wolfSSL_X509_NAME_dup error");
            return WOLFSSL_FAILURE;
        }

        if (wolfSSL_sk_X509_NAME_push(ctx->ca_names, nameCopy) != WOLFSSL_SUCCESS) {
            WOLFSSL_MSG("wolfSSL_sk_X509_NAME_push error");
            wolfSSL_X509_NAME_free(nameCopy);
            return WOLFSSL_FAILURE;
        }

        return WOLFSSL_SUCCESS;
    }
    #endif

    #ifndef NO_BIO
        #if !defined(NO_RSA) && !defined(NO_CERTS)
        WOLF_STACK_OF(WOLFSSL_X509_NAME)* wolfSSL_load_client_CA_file(const char* fname)
        {
            /* The webserver build is using this to load a CA into the server
             * for client authentication as an option. Have this return NULL in
             * that case. If OPENSSL_EXTRA is enabled, go ahead and include
             * the function. */
        #ifdef OPENSSL_EXTRA
            WOLFSSL_STACK *list = NULL;
            WOLFSSL_BIO* bio = NULL;
            WOLFSSL_X509 *cert = NULL;
            WOLFSSL_X509_NAME *nameCopy = NULL;
            unsigned long err = WOLFSSL_FAILURE;

            WOLFSSL_ENTER("wolfSSL_load_client_CA_file");

            bio = wolfSSL_BIO_new_file(fname, "rb");
            if (bio == NULL) {
                WOLFSSL_MSG("wolfSSL_BIO_new_file error");
                goto cleanup;
            }

            list = wolfSSL_sk_X509_NAME_new(NULL);
            if (list == NULL) {
                WOLFSSL_MSG("wolfSSL_sk_X509_NAME_new error");
                goto cleanup;
            }

            /* Read each certificate in the chain out of the file. */
            while (wolfSSL_PEM_read_bio_X509(bio, &cert, NULL, NULL) != NULL) {
                /* Need a persistent copy of the subject name. */
                nameCopy = wolfSSL_X509_NAME_dup(
                        wolfSSL_X509_get_subject_name(cert));
                if (nameCopy == NULL) {
                    WOLFSSL_MSG("wolfSSL_X509_NAME_dup error");
                    goto cleanup;
                }
                /*
                * Original cert will be freed so make sure not to try to access
                * it in the future.
                */
                nameCopy->x509 = NULL;

                if (wolfSSL_sk_X509_NAME_push(list, nameCopy) !=
                        WOLFSSL_SUCCESS) {
                    WOLFSSL_MSG("wolfSSL_sk_X509_NAME_push error");
                    /* Do free in loop because nameCopy is now responsibility
                     * of list to free and adding jumps to cleanup after this
                     * might result in a double free. */
                    wolfSSL_X509_NAME_free(nameCopy);
                    goto cleanup;
                }

                wolfSSL_X509_free(cert);
                cert = NULL;
            }

            CLEAR_ASN_NO_PEM_HEADER_ERROR(err);

            err = WOLFSSL_SUCCESS;
cleanup:
            wolfSSL_X509_free(cert);
            wolfSSL_BIO_free(bio);
            if (err != WOLFSSL_SUCCESS) {
                /* We failed so return NULL */
                wolfSSL_sk_X509_NAME_pop_free(list, NULL);
                list = NULL;
            }
            return list;
        #else
            (void)fname;
            return NULL;
        #endif
        }
        #endif
    #endif /* !NO_BIO */
#endif /* OPENSSL_EXTRA || WOLFSSL_EXTRA */

#ifdef OPENSSL_EXTRA

    #ifndef NO_WOLFSSL_STUB
    int wolfSSL_CTX_set_default_verify_paths(WOLFSSL_CTX* ctx)
    {
        /* TODO:, not needed in goahead */
        (void)ctx;
        WOLFSSL_STUB("SSL_CTX_set_default_verify_paths");
        return SSL_NOT_IMPLEMENTED;
    }
    #endif

    #if defined(WOLFCRYPT_HAVE_SRP) && !defined(NO_SHA256) \
        && !defined(WC_NO_RNG)
    static const byte srp_N[] = {
        0xEE, 0xAF, 0x0A, 0xB9, 0xAD, 0xB3, 0x8D, 0xD6, 0x9C, 0x33, 0xF8,
        0x0A, 0xFA, 0x8F, 0xC5, 0xE8, 0x60, 0x72, 0x61, 0x87, 0x75, 0xFF,
        0x3C, 0x0B, 0x9E, 0xA2, 0x31, 0x4C, 0x9C, 0x25, 0x65, 0x76, 0xD6,
        0x74, 0xDF, 0x74, 0x96, 0xEA, 0x81, 0xD3, 0x38, 0x3B, 0x48, 0x13,
        0xD6, 0x92, 0xC6, 0xE0, 0xE0, 0xD5, 0xD8, 0xE2, 0x50, 0xB9, 0x8B,
        0xE4, 0x8E, 0x49, 0x5C, 0x1D, 0x60, 0x89, 0xDA, 0xD1, 0x5D, 0xC7,
        0xD7, 0xB4, 0x61, 0x54, 0xD6, 0xB6, 0xCE, 0x8E, 0xF4, 0xAD, 0x69,
        0xB1, 0x5D, 0x49, 0x82, 0x55, 0x9B, 0x29, 0x7B, 0xCF, 0x18, 0x85,
        0xC5, 0x29, 0xF5, 0x66, 0x66, 0x0E, 0x57, 0xEC, 0x68, 0xED, 0xBC,
        0x3C, 0x05, 0x72, 0x6C, 0xC0, 0x2F, 0xD4, 0xCB, 0xF4, 0x97, 0x6E,
        0xAA, 0x9A, 0xFD, 0x51, 0x38, 0xFE, 0x83, 0x76, 0x43, 0x5B, 0x9F,
        0xC6, 0x1D, 0x2F, 0xC0, 0xEB, 0x06, 0xE3
    };
    static const byte srp_g[] = {
        0x02
    };

    int wolfSSL_CTX_set_srp_username(WOLFSSL_CTX* ctx, char* username)
    {
        int r = 0;
        SrpSide srp_side = SRP_CLIENT_SIDE;
        byte salt[SRP_SALT_SIZE];

        WOLFSSL_ENTER("wolfSSL_CTX_set_srp_username");
        if (ctx == NULL || ctx->srp == NULL || username==NULL)
            return SSL_FAILURE;

        if (ctx->method->side == WOLFSSL_SERVER_END){
            srp_side = SRP_SERVER_SIDE;
        } else if (ctx->method->side == WOLFSSL_CLIENT_END){
            srp_side = SRP_CLIENT_SIDE;
        } else {
            WOLFSSL_MSG("Init CTX failed");
            return SSL_FAILURE;
        }

        if (wc_SrpInit(ctx->srp, SRP_TYPE_SHA256, srp_side) < 0) {
            WOLFSSL_MSG("Init SRP CTX failed");
            XFREE(ctx->srp, ctx->heap, DYNAMIC_TYPE_SRP);
            ctx->srp = NULL;
            return SSL_FAILURE;
        }
        r = wc_SrpSetUsername(ctx->srp, (const byte*)username,
                              (word32)XSTRLEN(username));
        if (r < 0) {
            WOLFSSL_MSG("fail to set srp username.");
            return SSL_FAILURE;
        }

        /* if wolfSSL_CTX_set_srp_password has already been called, */
        /* execute wc_SrpSetPassword here */
        if (ctx->srp_password != NULL) {
            WC_RNG rng;
            if (wc_InitRng(&rng) < 0){
                WOLFSSL_MSG("wc_InitRng failed");
                return SSL_FAILURE;
            }
            XMEMSET(salt, 0, sizeof(salt)/sizeof(salt[0]));
            r = wc_RNG_GenerateBlock(&rng, salt, sizeof(salt)/sizeof(salt[0]));
            wc_FreeRng(&rng);
            if (r <  0) {
                WOLFSSL_MSG("wc_RNG_GenerateBlock failed");
                return SSL_FAILURE;
            }

            if (wc_SrpSetParams(ctx->srp, srp_N, sizeof(srp_N)/sizeof(srp_N[0]),
                                srp_g, sizeof(srp_g)/sizeof(srp_g[0]),
                                salt, sizeof(salt)/sizeof(salt[0])) < 0) {
                WOLFSSL_MSG("wc_SrpSetParam failed");
                return SSL_FAILURE;
            }
            r = wc_SrpSetPassword(ctx->srp,
                     (const byte*)ctx->srp_password,
                     (word32)XSTRLEN((char *)ctx->srp_password));
            if (r < 0) {
                WOLFSSL_MSG("fail to set srp password.");
                return SSL_FAILURE;
            }

            XFREE(ctx->srp_password, ctx->heap, DYNAMIC_TYPE_SRP);
            ctx->srp_password = NULL;
        }

        return WOLFSSL_SUCCESS;
    }

    int wolfSSL_CTX_set_srp_password(WOLFSSL_CTX* ctx, char* password)
    {
        int r;
        byte salt[SRP_SALT_SIZE];

        WOLFSSL_ENTER("wolfSSL_CTX_set_srp_password");
        if (ctx == NULL || ctx->srp == NULL || password == NULL)
            return SSL_FAILURE;

        if (ctx->srp->user != NULL) {
            WC_RNG rng;
            if (wc_InitRng(&rng) < 0) {
                WOLFSSL_MSG("wc_InitRng failed");
                return SSL_FAILURE;
            }
            XMEMSET(salt, 0, sizeof(salt)/sizeof(salt[0]));
            r = wc_RNG_GenerateBlock(&rng, salt, sizeof(salt)/sizeof(salt[0]));
            wc_FreeRng(&rng);
            if (r <  0) {
                WOLFSSL_MSG("wc_RNG_GenerateBlock failed");
                return SSL_FAILURE;
            }
            if (wc_SrpSetParams(ctx->srp, srp_N, sizeof(srp_N)/sizeof(srp_N[0]),
                                srp_g, sizeof(srp_g)/sizeof(srp_g[0]),
                                salt, sizeof(salt)/sizeof(salt[0])) < 0){
                WOLFSSL_MSG("wc_SrpSetParam failed");
                wc_FreeRng(&rng);
                return SSL_FAILURE;
            }
            r = wc_SrpSetPassword(ctx->srp, (const byte*)password,
                                  (word32)XSTRLEN(password));
            if (r < 0) {
                WOLFSSL_MSG("wc_SrpSetPassword failed.");
                wc_FreeRng(&rng);
                return SSL_FAILURE;
            }
            if (ctx->srp_password != NULL){
                XFREE(ctx->srp_password,NULL,
                      DYNAMIC_TYPE_SRP);
                ctx->srp_password = NULL;
            }
            wc_FreeRng(&rng);
        } else {
            /* save password for wolfSSL_set_srp_username */
            if (ctx->srp_password != NULL)
                XFREE(ctx->srp_password,ctx->heap, DYNAMIC_TYPE_SRP);

            ctx->srp_password = (byte*)XMALLOC(XSTRLEN(password) + 1, ctx->heap,
                                               DYNAMIC_TYPE_SRP);
            if (ctx->srp_password == NULL){
                WOLFSSL_MSG("memory allocation error");
                return SSL_FAILURE;
            }
            XMEMCPY(ctx->srp_password, password, XSTRLEN(password) + 1);
        }
        return WOLFSSL_SUCCESS;
    }

    /**
     * The modulus passed to wc_SrpSetParams in ssl.c is constant so check
     * that the requested strength is less than or equal to the size of the
     * static modulus size.
     * @param ctx Not used
     * @param strength Minimum number of bits for the modulus
     * @return 1 if strength is less than or equal to static modulus
     *         0 if strength is greater than static modulus
     */
    int  wolfSSL_CTX_set_srp_strength(WOLFSSL_CTX *ctx, int strength)
    {
        (void)ctx;
        WOLFSSL_ENTER("wolfSSL_CTX_set_srp_strength");
        if (strength > (int)(sizeof(srp_N)*8)) {
            WOLFSSL_MSG("Bad Parameter");
            return WOLFSSL_FAILURE;
        }
        return WOLFSSL_SUCCESS;
    }

    char* wolfSSL_get_srp_username(WOLFSSL *ssl)
    {
        if (ssl && ssl->ctx && ssl->ctx->srp) {
            return (char*) ssl->ctx->srp->user;
        }
        return NULL;
    }
    #endif /* WOLFCRYPT_HAVE_SRP && !NO_SHA256 && !WC_NO_RNG */

    /* keyblock size in bytes or -1 */
    int wolfSSL_get_keyblock_size(WOLFSSL* ssl)
    {
        if (ssl == NULL)
            return WOLFSSL_FATAL_ERROR;

        return 2 * (ssl->specs.key_size + ssl->specs.iv_size +
                    ssl->specs.hash_size);
    }

#endif /* OPENSSL_EXTRA */

#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_EXTRA) || defined(WOLFSSL_WPAS_SMALL)

    /* store keys returns WOLFSSL_SUCCESS or -1 on error */
    int wolfSSL_get_keys(WOLFSSL* ssl, unsigned char** ms, unsigned int* msLen,
                                     unsigned char** sr, unsigned int* srLen,
                                     unsigned char** cr, unsigned int* crLen)
    {
        if (ssl == NULL || ssl->arrays == NULL)
            return WOLFSSL_FATAL_ERROR;

        *ms = ssl->arrays->masterSecret;
        *sr = ssl->arrays->serverRandom;
        *cr = ssl->arrays->clientRandom;

        *msLen = SECRET_LEN;
        *srLen = RAN_LEN;
        *crLen = RAN_LEN;

        return WOLFSSL_SUCCESS;
    }

    void wolfSSL_set_accept_state(WOLFSSL* ssl)
    {
        WOLFSSL_ENTER("wolfSSL_set_accept_state");

        if (ssl == NULL)
            return;

        if (ssl->options.side == WOLFSSL_CLIENT_END) {
    #ifdef HAVE_ECC
        #ifdef WOLFSSL_SMALL_STACK
            ecc_key* key = NULL;
        #else
            ecc_key key[1];
        #endif
            word32 idx = 0;

        #ifdef WOLFSSL_SMALL_STACK
            key = (ecc_key*)XMALLOC(sizeof(ecc_key), ssl->heap,
                                    DYNAMIC_TYPE_ECC);
            if (key == NULL) {
                WOLFSSL_MSG("Error allocating memory for ecc_key");
            }
        #endif
            if (ssl->options.haveStaticECC && ssl->buffers.key != NULL) {
                if (wc_ecc_init(key) >= 0) {
                    if (wc_EccPrivateKeyDecode(ssl->buffers.key->buffer, &idx,
                            key, ssl->buffers.key->length) != 0) {
                        ssl->options.haveECDSAsig = 0;
                        ssl->options.haveECC = 0;
                        ssl->options.haveStaticECC = 0;
                    }
                    wc_ecc_free(key);
                }
            }
        #ifdef WOLFSSL_SMALL_STACK
            XFREE(key, ssl->heap, DYNAMIC_TYPE_ECC);
        #endif
    #endif

    #ifndef NO_DH
            if (!ssl->options.haveDH && ssl->ctx->haveDH) {
                ssl->buffers.serverDH_P = ssl->ctx->serverDH_P;
                ssl->buffers.serverDH_G = ssl->ctx->serverDH_G;
                ssl->options.haveDH = 1;
            }
    #endif
        }

        if (InitSSL_Side(ssl, WOLFSSL_SERVER_END) != WOLFSSL_SUCCESS) {
            WOLFSSL_MSG("Error initializing server side");
        }
    }

#endif /* OPENSSL_EXTRA || WOLFSSL_EXTRA || WOLFSSL_WPAS_SMALL */

    /* return true if connection established */
    int wolfSSL_is_init_finished(WOLFSSL* ssl)
    {
        if (ssl == NULL)
            return 0;

        if (ssl->options.handShakeState == HANDSHAKE_DONE)
            return 1;

        return 0;
    }

#ifdef OPENSSL_EXTRA

    void wolfSSL_CTX_set_tmp_rsa_callback(WOLFSSL_CTX* ctx,
                                      WOLFSSL_RSA*(*f)(WOLFSSL*, int, int))
    {
        /* wolfSSL verifies all these internally */
        (void)ctx;
        (void)f;
    }


    void wolfSSL_set_shutdown(WOLFSSL* ssl, int opt)
    {
        WOLFSSL_ENTER("wolfSSL_set_shutdown");
        if(ssl==NULL) {
            WOLFSSL_MSG("Shutdown not set. ssl is null");
            return;
        }

        ssl->options.sentNotify =  (opt&WOLFSSL_SENT_SHUTDOWN) > 0;
        ssl->options.closeNotify = (opt&WOLFSSL_RECEIVED_SHUTDOWN) > 0;
    }


    long wolfSSL_CTX_get_options(WOLFSSL_CTX* ctx)
    {
        WOLFSSL_ENTER("wolfSSL_CTX_get_options");
        WOLFSSL_MSG("wolfSSL options are set through API calls and macros");
        if(ctx == NULL)
            return BAD_FUNC_ARG;
        return ctx->mask;
    }

#endif

    static long wolf_set_options(long old_op, long op);
    long wolfSSL_CTX_set_options(WOLFSSL_CTX* ctx, long opt)
    {
        WOLFSSL_ENTER("SSL_CTX_set_options");

        if (ctx == NULL)
            return BAD_FUNC_ARG;

        ctx->mask = wolf_set_options(ctx->mask, opt);

        return ctx->mask;
    }

#ifdef OPENSSL_EXTRA

    long wolfSSL_CTX_clear_options(WOLFSSL_CTX* ctx, long opt)
    {
        WOLFSSL_ENTER("SSL_CTX_clear_options");
        if(ctx == NULL)
            return BAD_FUNC_ARG;
        ctx->mask &= ~opt;
        return ctx->mask;
    }

    int wolfSSL_set_rfd(WOLFSSL* ssl, int rfd)
    {
        WOLFSSL_ENTER("SSL_set_rfd");
        ssl->rfd = rfd;      /* not used directly to allow IO callbacks */

        ssl->IOCB_ReadCtx  = &ssl->rfd;

    #ifdef WOLFSSL_DTLS
        if (ssl->options.dtls) {
            ssl->IOCB_ReadCtx = &ssl->buffers.dtlsCtx;
            ssl->buffers.dtlsCtx.rfd = rfd;
        }
    #endif

        return WOLFSSL_SUCCESS;
    }


    int wolfSSL_set_wfd(WOLFSSL* ssl, int wfd)
    {
        WOLFSSL_ENTER("SSL_set_wfd");
        ssl->wfd = wfd;      /* not used directly to allow IO callbacks */

        ssl->IOCB_WriteCtx  = &ssl->wfd;

        return WOLFSSL_SUCCESS;
    }
#endif /* OPENSSL_EXTRA */

#if !defined(NO_CERTS) && (defined(OPENSSL_EXTRA) || defined(WOLFSSL_WPAS_SMALL))

#if defined(SESSION_CERTS) && defined(OPENSSL_EXTRA)
    /**
     * Implemented in a similar way that ngx_ssl_ocsp_validate does it when
     * SSL_get0_verified_chain is not available.
     * @param ssl WOLFSSL object to extract certs from
     * @return Stack of verified certs
     */
    WOLF_STACK_OF(WOLFSSL_X509) *wolfSSL_get0_verified_chain(const WOLFSSL *ssl)
    {
        WOLF_STACK_OF(WOLFSSL_X509)* chain = NULL;
        WOLFSSL_X509_STORE_CTX* storeCtx = NULL;
        WOLFSSL_X509* peerCert = NULL;

        WOLFSSL_ENTER("wolfSSL_get0_verified_chain");

        if (ssl == NULL || ssl->ctx == NULL) {
            WOLFSSL_MSG("Bad parameter");
            return NULL;
        }

        peerCert = wolfSSL_get_peer_certificate((WOLFSSL*)ssl);
        if (peerCert == NULL) {
            WOLFSSL_MSG("wolfSSL_get_peer_certificate error");
            return NULL;
        }
        /* wolfSSL_get_peer_certificate returns a copy. We want the internal
         * member so that we don't have to worry about free'ing it. We call
         * wolfSSL_get_peer_certificate so that we don't have to worry about
         * setting up the internal pointer. */
        wolfSSL_X509_free(peerCert);
        peerCert = (WOLFSSL_X509*)&ssl->peerCert;
        chain = wolfSSL_get_peer_cert_chain(ssl);
        if (chain == NULL) {
            WOLFSSL_MSG("wolfSSL_get_peer_cert_chain error");
            return NULL;
        }
        storeCtx = wolfSSL_X509_STORE_CTX_new();
        if (storeCtx == NULL) {
            WOLFSSL_MSG("wolfSSL_X509_STORE_CTX_new error");
            return NULL;
        }
        if (wolfSSL_X509_STORE_CTX_init(storeCtx, SSL_STORE(ssl),
                peerCert, chain) != WOLFSSL_SUCCESS) {
            WOLFSSL_MSG("wolfSSL_X509_STORE_CTX_init error");
            wolfSSL_X509_STORE_CTX_free(storeCtx);
            return NULL;
        }
        if (wolfSSL_X509_verify_cert(storeCtx) <= 0) {
            WOLFSSL_MSG("wolfSSL_X509_verify_cert error");
            wolfSSL_X509_STORE_CTX_free(storeCtx);
            return NULL;
        }
        wolfSSL_X509_STORE_CTX_free(storeCtx);
        return chain;
    }
#endif /* SESSION_CERTS && OPENSSL_EXTRA */

    WOLFSSL_X509_STORE* wolfSSL_CTX_get_cert_store(WOLFSSL_CTX* ctx)
    {
        if (ctx == NULL) {
            return NULL;
        }

        if (ctx->x509_store_pt != NULL)
            return ctx->x509_store_pt;
        return &ctx->x509_store;
    }

    void wolfSSL_CTX_set_cert_store(WOLFSSL_CTX* ctx, WOLFSSL_X509_STORE* str)
    {
        WOLFSSL_ENTER("wolfSSL_CTX_set_cert_store");
        if (ctx == NULL || str == NULL || ctx->cm == str->cm) {
            return;
        }

        if (wolfSSL_CertManager_up_ref(str->cm) != WOLFSSL_SUCCESS) {
            WOLFSSL_MSG("wolfSSL_CertManager_up_ref error");
            return;
        }
        /* free cert manager if have one */
        if (ctx->cm != NULL) {
            wolfSSL_CertManagerFree(ctx->cm);
        }
        ctx->cm               = str->cm;
        ctx->x509_store.cm    = str->cm;

        /* free existing store if it exists */
        wolfSSL_X509_STORE_free(ctx->x509_store_pt);
        ctx->x509_store.cache = str->cache;
        ctx->x509_store_pt    = str; /* take ownership of store and free it
                                        with CTX free */
        ctx->cm->x509_store_p = ctx->x509_store_pt;/* CTX has onwership
                                                    and free it with CTX free*/
    }


    int wolfSSL_set0_verify_cert_store(WOLFSSL *ssl, WOLFSSL_X509_STORE* str)
    {
        WOLFSSL_ENTER("wolfSSL_set0_verify_cert_store");

        if (ssl == NULL || str == NULL) {
            WOLFSSL_MSG("Bad parameter");
            return WOLFSSL_FAILURE;
        }

        /* NO-OP when setting existing store */
        if (str == SSL_STORE(ssl))
            return WOLFSSL_SUCCESS;

        /* free existing store if it exists */
        wolfSSL_X509_STORE_free(ssl->x509_store_pt);
        if (str == ssl->ctx->x509_store_pt)
            ssl->x509_store_pt = NULL; /* if setting ctx store then just revert
                                          to using that instead */
        else
            ssl->x509_store_pt = str; /* take ownership of store and free it
                                         with SSL free */
        return WOLFSSL_SUCCESS;
    }


    int wolfSSL_set1_verify_cert_store(WOLFSSL *ssl, WOLFSSL_X509_STORE* str)
    {
        WOLFSSL_ENTER("wolfSSL_set0_verify_cert_store");

        if (ssl == NULL || str == NULL) {
            WOLFSSL_MSG("Bad parameter");
            return WOLFSSL_FAILURE;
        }

        /* NO-OP when setting existing store */
        if (str == SSL_STORE(ssl))
            return WOLFSSL_SUCCESS;

        if (wolfSSL_X509_STORE_up_ref(str) != WOLFSSL_SUCCESS) {
            WOLFSSL_MSG("wolfSSL_X509_STORE_up_ref error");
            return WOLFSSL_FAILURE;
        }

        /* free existing store if it exists */
        wolfSSL_X509_STORE_free(ssl->x509_store_pt);
        if (str == ssl->ctx->x509_store_pt)
            ssl->x509_store_pt = NULL; /* if setting ctx store then just revert
                                          to using that instead */
        else
            ssl->x509_store_pt = str; /* take ownership of store and free it
                                         with SSL free */
        return WOLFSSL_SUCCESS;
    }
#endif /* !NO_CERTS && (OPENSSL_EXTRA || WOLFSSL_WPAS_SMALL) */

#ifdef WOLFSSL_ENCRYPTED_KEYS

    void wolfSSL_CTX_set_default_passwd_cb_userdata(WOLFSSL_CTX* ctx,
                                                   void* userdata)
    {
        WOLFSSL_ENTER("SSL_CTX_set_default_passwd_cb_userdata");
        if (ctx)
            ctx->passwd_userdata = userdata;
    }


    void wolfSSL_CTX_set_default_passwd_cb(WOLFSSL_CTX* ctx, wc_pem_password_cb*
                                           cb)
    {
        WOLFSSL_ENTER("SSL_CTX_set_default_passwd_cb");
        if (ctx)
            ctx->passwd_cb = cb;
    }

    wc_pem_password_cb* wolfSSL_CTX_get_default_passwd_cb(WOLFSSL_CTX *ctx)
    {
        if (ctx == NULL || ctx->passwd_cb == NULL) {
            return NULL;
        }

        return ctx->passwd_cb;
    }


    void* wolfSSL_CTX_get_default_passwd_cb_userdata(WOLFSSL_CTX *ctx)
    {
        if (ctx == NULL) {
            return NULL;
        }

        return ctx->passwd_userdata;
    }

#endif /* WOLFSSL_ENCRYPTED_KEYS */


#if defined(OPENSSL_EXTRA) || defined(HAVE_WEBSERVER)
    int wolfSSL_num_locks(void)
    {
        return 0;
    }

    void wolfSSL_set_locking_callback(void (*f)(int, int, const char*, int))
    {
        WOLFSSL_ENTER("wolfSSL_set_locking_callback");

        if (wc_SetMutexCb(f) != 0) {
            WOLFSSL_MSG("Error when setting mutex call back");
        }
    }


    typedef unsigned long (idCb)(void);
    static idCb* inner_idCb = NULL;

    unsigned long wolfSSL_thread_id(void)
    {
        if (inner_idCb != NULL) {
            return inner_idCb();
        }
        else {
            return 0;
        }
    }


    void wolfSSL_set_id_callback(unsigned long (*f)(void))
    {
        inner_idCb = f;
    }

    unsigned long wolfSSL_ERR_get_error(void)
    {
        WOLFSSL_ENTER("wolfSSL_ERR_get_error");

#ifdef WOLFSSL_HAVE_ERROR_QUEUE
#if defined(OPENSSL_ALL) || defined(WOLFSSL_NGINX) || defined(WOLFSSL_HAPROXY)
        {
            unsigned long ret = wolfSSL_ERR_peek_error_line_data(NULL, NULL,
                                                                 NULL, NULL);
            wc_RemoveErrorNode(-1);
            return ret;
        }
#else
        {
            int ret = wc_PullErrorNode(NULL, NULL, NULL);

            if (ret < 0) {
                if (ret == BAD_STATE_E) return 0; /* no errors in queue */
                WOLFSSL_MSG("Error with pulling error node!");
                WOLFSSL_LEAVE("wolfSSL_ERR_get_error", ret);
                ret = 0 - ret; /* return absolute value of error */

                /* panic and try to clear out nodes */
                wc_ClearErrorNodes();
            }

            return (unsigned long)ret;
        }
#endif
#else
        return (unsigned long)(0 - NOT_COMPILED_IN);
#endif
    }

#ifdef WOLFSSL_HAVE_ERROR_QUEUE
#ifndef NO_BIO
    /* print out and clear all errors */
    void wolfSSL_ERR_print_errors(WOLFSSL_BIO* bio)
    {
        const char* file = NULL;
        const char* reason = NULL;
        int ret;
        int line = 0;
        char buf[WOLFSSL_MAX_ERROR_SZ * 2];

        WOLFSSL_ENTER("wolfSSL_ERR_print_errors");

        if (bio == NULL) {
            WOLFSSL_MSG("BIO passed in was null");
            return;
        }

        do {
        ret = wc_PeekErrorNode(0, &file, &reason, &line);
        if (ret >= 0) {
            const char* r = wolfSSL_ERR_reason_error_string(0 - ret);
            XSNPRINTF(buf, sizeof(buf), "error:%d:wolfSSL library:%s:%s:%d\n",
                    ret, r, file, line);
            wolfSSL_BIO_write(bio, buf, (int)XSTRLEN(buf));
            wc_RemoveErrorNode(0);
        }
        } while (ret >= 0);
        if (wolfSSL_BIO_write(bio, "", 1) != 1) {
            WOLFSSL_MSG("Issue writing final string terminator");
        }
    }
#endif /* !NO_BIO */
#endif /* WOLFSSL_HAVE_ERROR_QUEUE */

#endif /* OPENSSL_EXTRA || HAVE_WEBSERVER */

#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_WPAS_SMALL) || \
    defined(HAVE_SECRET_CALLBACK)
#if !defined(NO_WOLFSSL_SERVER)
/* Return the amount of random bytes copied over or error case.
 * ssl : ssl struct after handshake
 * out : buffer to hold random bytes
 * outSz : either 0 (return max buffer sz) or size of out buffer
 */
size_t wolfSSL_get_server_random(const WOLFSSL *ssl, unsigned char *out,
                                                                   size_t outSz)
{
    size_t size;

    /* return max size of buffer */
    if (outSz == 0) {
        return RAN_LEN;
    }

    if (ssl == NULL || out == NULL) {
        return 0;
    }

    if (ssl->arrays == NULL) {
        WOLFSSL_MSG("Arrays struct not saved after handshake");
        return 0;
    }

    if (outSz > RAN_LEN) {
        size = RAN_LEN;
    }
    else {
        size = outSz;
    }

    XMEMCPY(out, ssl->arrays->serverRandom, size);
    return size;
}
#endif /* !NO_WOLFSSL_SERVER */
#endif /* OPENSSL_EXTRA || WOLFSSL_WPAS_SMALL || HAVE_SECRET_CALLBACK */

#ifdef OPENSSL_EXTRA
#if !defined(NO_WOLFSSL_SERVER)
/* Used to get the peer ephemeral public key sent during the connection
 * NOTE: currently wolfSSL_KeepHandshakeResources(WOLFSSL* ssl) must be called
 *       before the ephemeral key is stored.
 * return WOLFSSL_SUCCESS on success */
int wolfSSL_get_server_tmp_key(const WOLFSSL* ssl, WOLFSSL_EVP_PKEY** pkey)
{
    WOLFSSL_EVP_PKEY* ret = NULL;

    WOLFSSL_ENTER("wolfSSL_get_server_tmp_key");

    if (ssl == NULL || pkey == NULL) {
        WOLFSSL_MSG("Bad argument passed in");
        return WOLFSSL_FAILURE;
    }

#ifdef HAVE_ECC
    if (ssl->peerEccKey != NULL) {
        unsigned char* der;
        const unsigned char* pt;
        unsigned int   derSz = 0;
        int sz;

        PRIVATE_KEY_UNLOCK();
        if (wc_ecc_export_x963(ssl->peerEccKey, NULL, &derSz) !=
                LENGTH_ONLY_E) {
            WOLFSSL_MSG("get ecc der size failed");
            PRIVATE_KEY_LOCK();
            return WOLFSSL_FAILURE;
        }
        PRIVATE_KEY_LOCK();

        derSz += MAX_SEQ_SZ + (2 * MAX_ALGO_SZ) + MAX_SEQ_SZ + TRAILING_ZERO;
        der = (unsigned char*)XMALLOC(derSz, ssl->heap, DYNAMIC_TYPE_KEY);
        if (der == NULL) {
            WOLFSSL_MSG("Memory error");
            return WOLFSSL_FAILURE;
        }

        if ((sz = wc_EccPublicKeyToDer(ssl->peerEccKey, der, derSz, 1)) <= 0) {
            WOLFSSL_MSG("get ecc der failed");
            XFREE(der, ssl->heap, DYNAMIC_TYPE_KEY);
            return WOLFSSL_FAILURE;
        }
        pt = der; /* in case pointer gets advanced */
        ret = wolfSSL_d2i_PUBKEY(NULL, &pt, sz);
        XFREE(der, ssl->heap, DYNAMIC_TYPE_KEY);
    }
#endif

    *pkey = ret;
#ifdef HAVE_ECC
    if (ret != NULL)
        return WOLFSSL_SUCCESS;
    else
#endif
        return WOLFSSL_FAILURE;
}

#endif /* !NO_WOLFSSL_SERVER */

/**
 * This function checks if any compiled in protocol versions are
 * left enabled after calls to set_min or set_max API.
 * @param major The SSL/TLS major version
 * @return WOLFSSL_SUCCESS on valid settings and WOLFSSL_FAILURE when no
 *         protocol versions are left enabled.
 */
static int CheckSslMethodVersion(byte major, unsigned long options)
{
    int sanityConfirmed = 0;

    (void)options;

    switch (major) {
    #ifndef NO_TLS
        case SSLv3_MAJOR:
            #ifdef WOLFSSL_ALLOW_SSLV3
                if (!(options & WOLFSSL_OP_NO_SSLv3)) {
                    sanityConfirmed = 1;
                }
            #endif
            #ifndef NO_OLD_TLS
                if (!(options & WOLFSSL_OP_NO_TLSv1))
                    sanityConfirmed = 1;
                if (!(options & WOLFSSL_OP_NO_TLSv1_1))
                    sanityConfirmed = 1;
            #endif
            #ifndef WOLFSSL_NO_TLS12
                if (!(options & WOLFSSL_OP_NO_TLSv1_2))
                    sanityConfirmed = 1;
            #endif
            #ifdef WOLFSSL_TLS13
                if (!(options & WOLFSSL_OP_NO_TLSv1_3))
                    sanityConfirmed = 1;
            #endif
            break;
    #endif
    #ifdef WOLFSSL_DTLS
        case DTLS_MAJOR:
            sanityConfirmed = 1;
            break;
    #endif
        default:
            WOLFSSL_MSG("Invalid major version");
            return WOLFSSL_FAILURE;
    }
    if (!sanityConfirmed) {
        WOLFSSL_MSG("All compiled in TLS versions disabled");
        return WOLFSSL_FAILURE;
    }
    return WOLFSSL_SUCCESS;
}

/**
 * protoVerTbl holds (D)TLS version numbers in ascending order.
 * Except DTLS versions, the newer version is located in the latter part of
 * the table. This table is referred by wolfSSL_CTX_set_min_proto_version and
 * wolfSSL_CTX_set_max_proto_version.
 */
static const int protoVerTbl[] = {
    SSL3_VERSION,
    TLS1_VERSION,
    TLS1_1_VERSION,
    TLS1_2_VERSION,
    TLS1_3_VERSION,
    DTLS1_VERSION,
    DTLS1_2_VERSION
};
/* number of protocol versions listed in protoVerTbl */
#define NUMBER_OF_PROTOCOLS (sizeof(protoVerTbl)/sizeof(int))

/**
 * wolfSSL_CTX_set_min_proto_version attempts to set the minimum protocol
 * version to use by SSL objects created from this WOLFSSL_CTX.
 * This API guarantees that a version of SSL/TLS lower than specified
 * here will not be allowed. If the version specified is not compiled in
 * then this API sets the lowest compiled in protocol version.
 * This API also accept 0 as version, to set the minimum version automatically.
 * CheckSslMethodVersion() is called to check if any remaining protocol versions
 * are enabled.
 * @param ctx The wolfSSL CONTEXT factory for spawning SSL/TLS objects
 * @param version Any of the following
 *          * 0
 *          * SSL3_VERSION
 *          * TLS1_VERSION
 *          * TLS1_1_VERSION
 *          * TLS1_2_VERSION
 *          * TLS1_3_VERSION
 *          * DTLS1_VERSION
 *          * DTLS1_2_VERSION
 * @return WOLFSSL_SUCCESS on valid settings and WOLFSSL_FAILURE when no
 *         protocol versions are left enabled.
 */
static int Set_CTX_min_proto_version(WOLFSSL_CTX* ctx, int version)
{
    WOLFSSL_ENTER("wolfSSL_CTX_set_min_proto_version_ex");

    if (ctx == NULL) {
        return WOLFSSL_FAILURE;
    }

    switch (version) {
#ifndef NO_TLS
        case SSL3_VERSION:
#if defined(WOLFSSL_ALLOW_SSLV3) && !defined(NO_OLD_TLS)
            ctx->minDowngrade = SSLv3_MINOR;
            break;
#endif
        case TLS1_VERSION:
        #ifdef WOLFSSL_ALLOW_TLSV10
            ctx->minDowngrade = TLSv1_MINOR;
            break;
        #endif
        case TLS1_1_VERSION:
        #ifndef NO_OLD_TLS
            ctx->minDowngrade = TLSv1_1_MINOR;
            break;
        #endif
        case TLS1_2_VERSION:
        #ifndef WOLFSSL_NO_TLS12
            ctx->minDowngrade = TLSv1_2_MINOR;
            break;
        #endif
        case TLS1_3_VERSION:
        #ifdef WOLFSSL_TLS13
            ctx->minDowngrade = TLSv1_3_MINOR;
            break;
        #endif
#endif
#ifdef WOLFSSL_DTLS
        case DTLS1_VERSION:
    #ifndef NO_OLD_TLS
            ctx->minDowngrade = DTLS_MINOR;
            break;
    #endif
        case DTLS1_2_VERSION:
            ctx->minDowngrade = DTLSv1_2_MINOR;
            break;
#endif
        default:
            WOLFSSL_MSG("Unrecognized protocol version or not compiled in");
            return WOLFSSL_FAILURE;
    }

    switch (version) {
#ifndef NO_TLS
    case TLS1_3_VERSION:
        wolfSSL_CTX_set_options(ctx, WOLFSSL_OP_NO_TLSv1_2);
        FALL_THROUGH;
    case TLS1_2_VERSION:
        wolfSSL_CTX_set_options(ctx, WOLFSSL_OP_NO_TLSv1_1);
        FALL_THROUGH;
    case TLS1_1_VERSION:
        wolfSSL_CTX_set_options(ctx, WOLFSSL_OP_NO_TLSv1);
        FALL_THROUGH;
    case TLS1_VERSION:
        wolfSSL_CTX_set_options(ctx, WOLFSSL_OP_NO_SSLv3);
        break;
    case SSL3_VERSION:
    case SSL2_VERSION:
        /* Nothing to do here */
        break;
#endif
#ifdef WOLFSSL_DTLS
    case DTLS1_VERSION:
    case DTLS1_2_VERSION:
        break;
#endif
    default:
        WOLFSSL_MSG("Unrecognized protocol version or not compiled in");
        return WOLFSSL_FAILURE;
    }

    return CheckSslMethodVersion(ctx->method->version.major, ctx->mask);
}

/* Sets the min protocol version allowed with WOLFSSL_CTX
 * returns WOLFSSL_SUCCESS on success */
int wolfSSL_CTX_set_min_proto_version(WOLFSSL_CTX* ctx, int version)
{
    int ret;
    int proto    = 0;
    int maxProto = 0;
    int i;
    int idx = 0;

    WOLFSSL_ENTER("wolfSSL_CTX_set_min_proto_version");

    if (ctx == NULL) {
        return WOLFSSL_FAILURE;
    }
    if (version != 0) {
        proto = version;
        ctx->minProto = 0; /* turn min proto flag off */
        for (i = 0; (unsigned)i < NUMBER_OF_PROTOCOLS; i++) {
            if (protoVerTbl[i] == version) {
                break;
            }
        }
    }
    else {
        /* when 0 is specified as version, try to find out the min version */
        for (i = 0; (unsigned)i < NUMBER_OF_PROTOCOLS; i++) {
            ret = Set_CTX_min_proto_version(ctx, protoVerTbl[i]);
            if (ret == WOLFSSL_SUCCESS) {
                proto = protoVerTbl[i];
                ctx->minProto = 1; /* turn min proto flag on */
                break;
            }
        }
    }

    /* check case where max > min , if so then clear the NO_* options
     * i is the index into the table for proto version used, see if the max
     * proto version index found is smaller */
    maxProto = wolfSSL_CTX_get_max_proto_version(ctx);
    for (idx = 0; (unsigned)idx < NUMBER_OF_PROTOCOLS; idx++) {
        if (protoVerTbl[idx] == maxProto) {
            break;
        }
    }
    if (idx < i) {
        wolfSSL_CTX_clear_options(ctx, WOLFSSL_OP_NO_TLSv1 |
                WOLFSSL_OP_NO_TLSv1_1 | WOLFSSL_OP_NO_TLSv1_2 |
                WOLFSSL_OP_NO_TLSv1_3);
    }

    ret = Set_CTX_min_proto_version(ctx, proto);
    return ret;
}

/**
 * wolfSSL_CTX_set_max_proto_version attempts to set the maximum protocol
 * version to use by SSL objects created from this WOLFSSL_CTX.
 * This API guarantees that a version of SSL/TLS higher than specified
 * here will not be allowed. If the version specified is not compiled in
 * then this API sets the highest compiled in protocol version.
 * This API also accept 0 as version, to set the maximum version automatically.
 * CheckSslMethodVersion() is called to check if any remaining protocol versions
 * are enabled.
 * @param ctx The wolfSSL CONTEXT factory for spawning SSL/TLS objects
 * @param ver Any of the following
 *          * 0
 *          * SSL3_VERSION
 *          * TLS1_VERSION
 *          * TLS1_1_VERSION
 *          * TLS1_2_VERSION
 *          * TLS1_3_VERSION
 *          * DTLS1_VERSION
 *          * DTLS1_2_VERSION
 * @return WOLFSSL_SUCCESS on valid settings and WOLFSSL_FAILURE when no
 *         protocol versions are left enabled.
 */
static int Set_CTX_max_proto_version(WOLFSSL_CTX* ctx, int ver)
{
    WOLFSSL_ENTER("Set_CTX_max_proto_version");

    if (!ctx || !ctx->method) {
        WOLFSSL_MSG("Bad parameter");
        return WOLFSSL_FAILURE;
    }

    switch (ver) {
    case SSL2_VERSION:
        WOLFSSL_MSG("wolfSSL does not support SSLv2");
        return WOLFSSL_FAILURE;
#ifndef NO_TLS
    case SSL3_VERSION:
        wolfSSL_CTX_set_options(ctx, WOLFSSL_OP_NO_TLSv1);
        FALL_THROUGH;
    case TLS1_VERSION:
        wolfSSL_CTX_set_options(ctx, WOLFSSL_OP_NO_TLSv1_1);
        FALL_THROUGH;
    case TLS1_1_VERSION:
        wolfSSL_CTX_set_options(ctx, WOLFSSL_OP_NO_TLSv1_2);
        FALL_THROUGH;
    case TLS1_2_VERSION:
        wolfSSL_CTX_set_options(ctx, WOLFSSL_OP_NO_TLSv1_3);
        FALL_THROUGH;
    case TLS1_3_VERSION:
        /* Nothing to do here */
        break;
#endif
#ifdef WOLFSSL_DTLS
    case DTLS1_VERSION:
    case DTLS1_2_VERSION:
        break;
#endif
    default:
        WOLFSSL_MSG("Unrecognized protocol version or not compiled in");
        return WOLFSSL_FAILURE;
    }

    return CheckSslMethodVersion(ctx->method->version.major, ctx->mask);
}


/* Sets the max protocol version allowed with WOLFSSL_CTX
 * returns WOLFSSL_SUCCESS on success */
int wolfSSL_CTX_set_max_proto_version(WOLFSSL_CTX* ctx, int version)
{
    int i;
    int ret = WOLFSSL_FAILURE;
    int minProto;

    WOLFSSL_ENTER("wolfSSL_CTX_set_max_proto_version");

    if (ctx == NULL) {
        return ret;
    }

    /* clear out flags and reset min protocol version */
    minProto = wolfSSL_CTX_get_min_proto_version(ctx);
    wolfSSL_CTX_clear_options(ctx,
            WOLFSSL_OP_NO_TLSv1 | WOLFSSL_OP_NO_TLSv1_1 |
            WOLFSSL_OP_NO_TLSv1_2 | WOLFSSL_OP_NO_TLSv1_3);
    wolfSSL_CTX_set_min_proto_version(ctx, minProto);
    if (version != 0) {
        ctx->maxProto = 0; /* turn max proto flag off */
        return Set_CTX_max_proto_version(ctx, version);
    }

    /* when 0 is specified as version, try to find out the min version from
     * the bottom to top of the protoverTbl.
     */
    for (i = NUMBER_OF_PROTOCOLS -1; i >= 0; i--) {
        ret = Set_CTX_max_proto_version(ctx, protoVerTbl[i]);
        if (ret == WOLFSSL_SUCCESS) {
            ctx->maxProto = 1; /* turn max proto flag on */
            break;
        }
    }

    return ret;
}


static int Set_SSL_min_proto_version(WOLFSSL* ssl, int ver)
{
    WOLFSSL_ENTER("Set_SSL_min_proto_version");

    if (ssl == NULL) {
        return WOLFSSL_FAILURE;
    }

    switch (ver) {
#ifndef NO_TLS
        case SSL3_VERSION:
#if defined(WOLFSSL_ALLOW_SSLV3) && !defined(NO_OLD_TLS)
            ssl->options.minDowngrade = SSLv3_MINOR;
            break;
#endif
        case TLS1_VERSION:
        #ifdef WOLFSSL_ALLOW_TLSV10
            ssl->options.minDowngrade = TLSv1_MINOR;
            break;
        #endif
        case TLS1_1_VERSION:
        #ifndef NO_OLD_TLS
            ssl->options.minDowngrade = TLSv1_1_MINOR;
            break;
        #endif
        case TLS1_2_VERSION:
        #ifndef WOLFSSL_NO_TLS12
            ssl->options.minDowngrade = TLSv1_2_MINOR;
            break;
        #endif
        case TLS1_3_VERSION:
        #ifdef WOLFSSL_TLS13
            ssl->options.minDowngrade = TLSv1_3_MINOR;
            break;
        #endif
#endif
#ifdef WOLFSSL_DTLS
        case DTLS1_VERSION:
    #ifndef NO_OLD_TLS
            ssl->options.minDowngrade = DTLS_MINOR;
            break;
    #endif
        case DTLS1_2_VERSION:
            ssl->options.minDowngrade = DTLSv1_2_MINOR;
            break;
#endif
        default:
            WOLFSSL_MSG("Unrecognized protocol version or not compiled in");
            return WOLFSSL_FAILURE;
    }

    switch (ver) {
#ifndef NO_TLS
    case TLS1_3_VERSION:
        ssl->options.mask |= WOLFSSL_OP_NO_TLSv1_2;
        FALL_THROUGH;
    case TLS1_2_VERSION:
        ssl->options.mask |= WOLFSSL_OP_NO_TLSv1_1;
        FALL_THROUGH;
    case TLS1_1_VERSION:
        ssl->options.mask |= WOLFSSL_OP_NO_TLSv1;
        FALL_THROUGH;
    case TLS1_VERSION:
        ssl->options.mask |= WOLFSSL_OP_NO_SSLv3;
        break;
    case SSL3_VERSION:
    case SSL2_VERSION:
        /* Nothing to do here */
        break;
#endif
#ifdef WOLFSSL_DTLS
    case DTLS1_VERSION:
    case DTLS1_2_VERSION:
        break;
#endif
    default:
        WOLFSSL_MSG("Unrecognized protocol version or not compiled in");
        return WOLFSSL_FAILURE;
    }

    return CheckSslMethodVersion(ssl->version.major, ssl->options.mask);
}

int wolfSSL_set_min_proto_version(WOLFSSL* ssl, int version)
{
    int i;
    int ret = WOLFSSL_FAILURE;;

    WOLFSSL_ENTER("wolfSSL_set_min_proto_version");

    if (ssl == NULL) {
        return WOLFSSL_FAILURE;
    }
    if (version != 0) {
        return Set_SSL_min_proto_version(ssl, version);
    }

    /* when 0 is specified as version, try to find out the min version */
    for (i= 0; (unsigned)i < NUMBER_OF_PROTOCOLS; i++) {
        ret = Set_SSL_min_proto_version(ssl, protoVerTbl[i]);
        if (ret == WOLFSSL_SUCCESS)
            break;
    }

    return ret;
}

static int Set_SSL_max_proto_version(WOLFSSL* ssl, int ver)
{

    WOLFSSL_ENTER("Set_SSL_max_proto_version");

    if (!ssl) {
        WOLFSSL_MSG("Bad parameter");
        return WOLFSSL_FAILURE;
    }

    switch (ver) {
    case SSL2_VERSION:
        WOLFSSL_MSG("wolfSSL does not support SSLv2");
        return WOLFSSL_FAILURE;
#ifndef NO_TLS
    case SSL3_VERSION:
        ssl->options.mask |= WOLFSSL_OP_NO_TLSv1;
        FALL_THROUGH;
    case TLS1_VERSION:
        ssl->options.mask |= WOLFSSL_OP_NO_TLSv1_1;
        FALL_THROUGH;
    case TLS1_1_VERSION:
        ssl->options.mask |= WOLFSSL_OP_NO_TLSv1_2;
        FALL_THROUGH;
    case TLS1_2_VERSION:
        ssl->options.mask |= WOLFSSL_OP_NO_TLSv1_3;
        FALL_THROUGH;
    case TLS1_3_VERSION:
        /* Nothing to do here */
        break;
#endif
#ifdef WOLFSSL_DTLS
    case DTLS1_VERSION:
    case DTLS1_2_VERSION:
        break;
#endif
    default:
        WOLFSSL_MSG("Unrecognized protocol version or not compiled in");
        return WOLFSSL_FAILURE;
    }

    return CheckSslMethodVersion(ssl->version.major, ssl->options.mask);
}

int wolfSSL_set_max_proto_version(WOLFSSL* ssl, int version)
{
    int i;
    int ret = WOLFSSL_FAILURE;;

    WOLFSSL_ENTER("wolfSSL_set_max_proto_version");

    if (ssl == NULL) {
        return WOLFSSL_FAILURE;
    }
    if (version != 0) {
        return Set_SSL_max_proto_version(ssl, version);
    }

    /* when 0 is specified as version, try to find out the min version from
     * the bottom to top of the protoverTbl.
     */
    for (i = NUMBER_OF_PROTOCOLS -1; i >= 0; i--) {
        ret = Set_SSL_max_proto_version(ssl, protoVerTbl[i]);
        if (ret == WOLFSSL_SUCCESS)
            break;
    }

    return ret;
}

static int GetMinProtoVersion(int minDowngrade)
{
    int ret;

    switch (minDowngrade) {
#ifndef NO_OLD_TLS
    #ifdef WOLFSSL_ALLOW_SSLV3
        case SSLv3_MINOR:
            ret = SSL3_VERSION;
            break;
    #endif
    #ifdef WOLFSSL_ALLOW_TLSV10
        case TLSv1_MINOR:
            ret = TLS1_VERSION;
            break;
    #endif
        case TLSv1_1_MINOR:
            ret = TLS1_1_VERSION;
            break;
#endif
#ifndef WOLFSSL_NO_TLS12
        case TLSv1_2_MINOR:
            ret = TLS1_2_VERSION;
            break;
#endif
#ifdef WOLFSSL_TLS13
        case TLSv1_3_MINOR:
            ret = TLS1_3_VERSION;
            break;
#endif
        default:
            ret = 0;
            break;
    }

    return ret;
}

WOLFSSL_API int wolfSSL_CTX_get_min_proto_version(WOLFSSL_CTX* ctx)
{
    int ret = 0;

    WOLFSSL_ENTER("wolfSSL_CTX_get_min_proto_version");

    if (ctx != NULL) {
        if (ctx->minProto) {
            ret = 0;
        }
        else {
            ret = GetMinProtoVersion(ctx->minDowngrade);
        }
    }
    else {
        ret = GetMinProtoVersion(WOLFSSL_MIN_DOWNGRADE);
    }

    WOLFSSL_LEAVE("wolfSSL_CTX_get_min_proto_version", ret);

    return ret;
}


/* returns the maximum allowed protocol version given the 'options' used
 * returns WOLFSSL_FATAL_ERROR on no match */
static int GetMaxProtoVersion(long options)
{
#ifdef WOLFSSL_TLS13
    if (!(options & WOLFSSL_OP_NO_TLSv1_3))
        return TLS1_3_VERSION;
#endif
#ifndef WOLFSSL_NO_TLS12
    if (!(options & WOLFSSL_OP_NO_TLSv1_2))
        return TLS1_2_VERSION;
#endif
#ifndef NO_OLD_TLS
    if (!(options & WOLFSSL_OP_NO_TLSv1_1))
        return TLS1_1_VERSION;
    #ifdef WOLFSSL_ALLOW_TLSV10
    if (!(options & WOLFSSL_OP_NO_TLSv1))
        return TLS1_VERSION;
    #endif
    #ifdef WOLFSSL_ALLOW_SSLV3
    if (!(options & WOLFSSL_OP_NO_SSLv3))
        return SSL3_VERSION;
    #endif
#endif

    return WOLFSSL_FATAL_ERROR;
}


/* returns the maximum protocol version for 'ctx' */
int wolfSSL_CTX_get_max_proto_version(WOLFSSL_CTX* ctx)
{
    int ret = 0;
    long options = 0; /* default to nothing set */

    WOLFSSL_ENTER("wolfSSL_CTX_get_max_proto_version");

    if (ctx != NULL) {
        options = wolfSSL_CTX_get_options(ctx);
    }

    if ((ctx != NULL) && ctx->maxProto) {
        ret = 0;
    }
    else {
        ret = GetMaxProtoVersion(options);
    }

    WOLFSSL_LEAVE("wolfSSL_CTX_get_max_proto_version", ret);

    if (ret == WOLFSSL_FATAL_ERROR) {
        WOLFSSL_MSG("Error getting max proto version");
        ret = 0; /* setting ret to 0 to match compat return */
    }
    return ret;
}
#endif /* OPENSSL_EXTRA */

#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_WPAS_SMALL) || \
    defined(HAVE_SECRET_CALLBACK)
#if !defined(NO_WOLFSSL_CLIENT)
/* Return the amount of random bytes copied over or error case.
 * ssl : ssl struct after handshake
 * out : buffer to hold random bytes
 * outSz : either 0 (return max buffer sz) or size of out buffer
 */
size_t wolfSSL_get_client_random(const WOLFSSL* ssl, unsigned char* out,
                                                                   size_t outSz)
{
    size_t size;

    /* return max size of buffer */
    if (outSz == 0) {
        return RAN_LEN;
    }

    if (ssl == NULL || out == NULL) {
        return 0;
    }

    if (ssl->arrays == NULL) {
        WOLFSSL_MSG("Arrays struct not saved after handshake");
        return 0;
    }

    if (outSz > RAN_LEN) {
        size = RAN_LEN;
    }
    else {
        size = outSz;
    }

    XMEMCPY(out, ssl->arrays->clientRandom, size);
    return size;
}
#endif /* !NO_WOLFSSL_CLIENT */
#endif /* OPENSSL_EXTRA || WOLFSSL_WPAS_SMALL || HAVE_SECRET_CALLBACK */

#ifdef OPENSSL_EXTRA
    unsigned long wolfSSLeay(void)
    {
        return SSLEAY_VERSION_NUMBER;
    }

    unsigned long wolfSSL_OpenSSL_version_num(void)
    {
        return OPENSSL_VERSION_NUMBER;
    }

    const char* wolfSSLeay_version(int type)
    {
        (void)type;
#if defined(OPENSSL_VERSION_NUMBER) && OPENSSL_VERSION_NUMBER >= 0x10100000L
        return wolfSSL_OpenSSL_version(type);
#else
        return wolfSSL_OpenSSL_version();
#endif
    }


#ifndef NO_MD5
    int wolfSSL_MD5_Init(WOLFSSL_MD5_CTX* md5)
    {
        int ret;
        typedef char md5_test[sizeof(MD5_CTX) >= sizeof(wc_Md5) ? 1 : -1];
        (void)sizeof(md5_test);

        WOLFSSL_ENTER("MD5_Init");
        ret = wc_InitMd5((wc_Md5*)md5);

        /* return 1 on success, 0 otherwise */
        if (ret == 0)
            return 1;

        return 0;
    }


    int wolfSSL_MD5_Update(WOLFSSL_MD5_CTX* md5, const void* input,
                           unsigned long sz)
    {
        int ret;

        WOLFSSL_ENTER("wolfSSL_MD5_Update");
        ret = wc_Md5Update((wc_Md5*)md5, (const byte*)input, (word32)sz);

        /* return 1 on success, 0 otherwise */
        if (ret == 0)
            return 1;

        return 0;
    }


    int wolfSSL_MD5_Final(byte* output, WOLFSSL_MD5_CTX* md5)
    {
        int ret;

        WOLFSSL_ENTER("MD5_Final");
        ret = wc_Md5Final((wc_Md5*)md5, output);

        /* have to actually free the resources (if any) here, because the
         * OpenSSL API doesn't include SHA*_Free().
         */
        wc_Md5Free((wc_Md5*)md5);

        /* return 1 on success, 0 otherwise */
        if (ret == 0)
            return 1;

        return 0;
    }
    /* Apply MD5 transformation to the data */
    int wolfSSL_MD5_Transform(WOLFSSL_MD5_CTX* md5, const unsigned char* data)
    {
        int ret;

       WOLFSSL_ENTER("MD5_Transform");

       /* sanity check */
       if (md5 == NULL || data == NULL) {
            return 0;
       }
       #if defined(BIG_ENDIAN_ORDER)
       {
            ByteReverseWords((word32*)data, (word32*)data, WC_MD5_BLOCK_SIZE);
       }
       #endif

       ret = wc_Md5Transform((wc_Md5*)md5, data);

       /* return 1 on success, 0 otherwise */
        if (ret == 0)
            return 1;
        else
            return 0;
    }

    unsigned char *wolfSSL_MD5(const unsigned char* data, size_t len,
            unsigned char* hash)
    {
        static unsigned char out[WC_MD5_DIGEST_SIZE];

        WOLFSSL_ENTER("wolfSSL_MD5");

        if (hash == NULL)
            hash = out;
        if (wc_Md5Hash(data, (word32)len, hash) != 0) {
            WOLFSSL_MSG("wc_Md5Hash error");
            return NULL;
        }
        return hash;
    }
#endif /* !NO_MD5 */


#ifndef NO_SHA
    int wolfSSL_SHA_Init(WOLFSSL_SHA_CTX* sha)
    {
        int ret;

        typedef char sha_test[sizeof(SHA_CTX) >= sizeof(wc_Sha) ? 1 : -1];
        (void)sizeof(sha_test);

        WOLFSSL_ENTER("SHA_Init");
        ret = wc_InitSha((wc_Sha*)sha);

        /* return 1 on success, 0 otherwise */
        if (ret == 0)
            return 1;

        return 0;
    }


    int wolfSSL_SHA_Update(WOLFSSL_SHA_CTX* sha, const void* input,
                           unsigned long sz)
    {
        int ret;

        WOLFSSL_ENTER("SHA_Update");
        ret = wc_ShaUpdate((wc_Sha*)sha, (const byte*)input, (word32)sz);

        /* return 1 on success, 0 otherwise */
        if (ret == 0)
            return 1;

        return 0;
    }


    int wolfSSL_SHA_Final(byte* output, WOLFSSL_SHA_CTX* sha)
    {
        int ret;

        WOLFSSL_ENTER("SHA_Final");
        ret = wc_ShaFinal((wc_Sha*)sha, output);

        /* have to actually free the resources (if any) here, because the
         * OpenSSL API doesn't include SHA*_Free().
         */
        wc_ShaFree((wc_Sha*)sha);

        /* return 1 on success, 0 otherwise */
        if (ret == 0)
            return 1;

        return 0;
    }

    #if defined(OPENSSL_EXTRA)
    #if !defined(HAVE_SELFTEST) && (!defined(HAVE_FIPS) || \
        (defined(HAVE_FIPS_VERSION) && (HAVE_FIPS_VERSION > 2)))
    /* Apply SHA1 transformation to the data */
    int wolfSSL_SHA_Transform(WOLFSSL_SHA_CTX* sha,
                                         const unsigned char* data)
    {
       int ret;

       WOLFSSL_ENTER("SHA_Transform");
       /* sanity check */
       if (sha == NULL || data == NULL) {
            return 0;
       }
       #if defined(LITTLE_ENDIAN_ORDER)
       {
            ByteReverseWords((word32*)data, (word32*)data, WC_SHA_BLOCK_SIZE);
       }
       #endif
       ret = wc_ShaTransform((wc_Sha*)sha, data);

       /* return 1 on success, 0 otherwise */
        if (ret == 0)
            return 1;
        else
            return 0;
    }
    #endif
    #endif

    int wolfSSL_SHA1_Init(WOLFSSL_SHA_CTX* sha)
    {
        WOLFSSL_ENTER("SHA1_Init");
        return SHA_Init(sha);
    }


    int wolfSSL_SHA1_Update(WOLFSSL_SHA_CTX* sha, const void* input,
                            unsigned long sz)
    {
        WOLFSSL_ENTER("SHA1_Update");
        return SHA_Update(sha, input, sz);
    }


    int wolfSSL_SHA1_Final(byte* output, WOLFSSL_SHA_CTX* sha)
    {
        WOLFSSL_ENTER("SHA1_Final");
        return SHA_Final(output, sha);
    }
    #if defined(OPENSSL_EXTRA)
    #if !defined(HAVE_SELFTEST) && (!defined(HAVE_FIPS) || \
        (defined(HAVE_FIPS_VERSION) && (HAVE_FIPS_VERSION > 2)))
    /* Apply SHA1 transformation to the data */
    int wolfSSL_SHA1_Transform(WOLFSSL_SHA_CTX* sha,
                                         const unsigned char* data)
    {
       WOLFSSL_ENTER("SHA1_Transform");
       return (wolfSSL_SHA_Transform(sha, data));
    }
    #endif
    #endif
#endif /* !NO_SHA */

#ifdef WOLFSSL_SHA224

    int wolfSSL_SHA224_Init(WOLFSSL_SHA224_CTX* sha)
    {
        int ret;

        typedef char sha_test[sizeof(SHA224_CTX) >= sizeof(wc_Sha224) ? 1 : -1];
        (void)sizeof(sha_test);

        WOLFSSL_ENTER("SHA224_Init");
        ret = wc_InitSha224((wc_Sha224*)sha);

        /* return 1 on success, 0 otherwise */
        if (ret == 0)
            return 1;

        return 0;
    }


    int wolfSSL_SHA224_Update(WOLFSSL_SHA224_CTX* sha, const void* input,
                           unsigned long sz)
    {
        int ret;

        WOLFSSL_ENTER("SHA224_Update");
        ret = wc_Sha224Update((wc_Sha224*)sha, (const byte*)input, (word32)sz);

        /* return 1 on success, 0 otherwise */
        if (ret == 0)
            return 1;

        return 0;
    }


    int wolfSSL_SHA224_Final(byte* output, WOLFSSL_SHA224_CTX* sha)
    {
        int ret;

        WOLFSSL_ENTER("SHA224_Final");
        ret = wc_Sha224Final((wc_Sha224*)sha, output);

        /* have to actually free the resources (if any) here, because the
         * OpenSSL API doesn't include SHA*_Free().
         */
        wc_Sha224Free((wc_Sha224*)sha);

        /* return 1 on success, 0 otherwise */
        if (ret == 0)
            return 1;

        return 0;
    }

#endif /* WOLFSSL_SHA224 */


    int wolfSSL_SHA256_Init(WOLFSSL_SHA256_CTX* sha256)
    {
        int ret;

        typedef char sha_test[sizeof(SHA256_CTX) >= sizeof(wc_Sha256) ? 1 : -1];
        (void)sizeof(sha_test);

        WOLFSSL_ENTER("SHA256_Init");
        ret = wc_InitSha256((wc_Sha256*)sha256);

        /* return 1 on success, 0 otherwise */
        if (ret == 0)
            return 1;

        return 0;
    }


    int wolfSSL_SHA256_Update(WOLFSSL_SHA256_CTX* sha, const void* input,
                              unsigned long sz)
    {
        int ret;

        WOLFSSL_ENTER("SHA256_Update");
        ret = wc_Sha256Update((wc_Sha256*)sha, (const byte*)input, (word32)sz);

        /* return 1 on success, 0 otherwise */
        if (ret == 0)
            return 1;

        return 0;
    }


    int wolfSSL_SHA256_Final(byte* output, WOLFSSL_SHA256_CTX* sha)
    {
        int ret;

        WOLFSSL_ENTER("SHA256_Final");
        ret = wc_Sha256Final((wc_Sha256*)sha, output);

        /* have to actually free the resources (if any) here, because the
         * OpenSSL API doesn't include SHA*_Free().
         */
        wc_Sha256Free((wc_Sha256*)sha);

        /* return 1 on success, 0 otherwise */
        if (ret == 0)
            return 1;

        return 0;
    }

    #if defined(OPENSSL_EXTRA)
    #if !defined(HAVE_SELFTEST) && (!defined(HAVE_FIPS) || \
        (defined(HAVE_FIPS_VERSION) && (HAVE_FIPS_VERSION > 2))) && \
        !defined(WOLFSSL_DEVCRYPTO_HASH) && !defined(WOLFSSL_AFALG_HASH)
    /* Apply SHA256 transformation to the data */
    int wolfSSL_SHA256_Transform(WOLFSSL_SHA256_CTX* sha256,
                                                const unsigned char* data)
    {
       int ret;

       WOLFSSL_ENTER("SHA256_Transform");
       /* sanity check */
       if (sha256 == NULL || data == NULL) {
            return 0;
       }
       #if defined(LITTLE_ENDIAN_ORDER)
       {
            ByteReverseWords((word32*)data, (word32*)data, WC_SHA256_BLOCK_SIZE);
       }
       #endif
       ret = wc_Sha256Transform((wc_Sha256*)sha256, data);

       /* return 1 on success, 0 otherwise */
        if (ret == 0)
            return 1;
        else
            return 0;
    }
    #endif
    #endif

#ifdef WOLFSSL_SHA384

    int wolfSSL_SHA384_Init(WOLFSSL_SHA384_CTX* sha)
    {
        int ret;

        typedef char sha_test[sizeof(SHA384_CTX) >= sizeof(wc_Sha384) ? 1 : -1];
        (void)sizeof(sha_test);

        WOLFSSL_ENTER("SHA384_Init");
        ret = wc_InitSha384((wc_Sha384*)sha);

        /* return 1 on success, 0 otherwise */
        if (ret == 0)
            return 1;

        return 0;
    }


    int wolfSSL_SHA384_Update(WOLFSSL_SHA384_CTX* sha, const void* input,
                           unsigned long sz)
    {
        int ret;

        WOLFSSL_ENTER("SHA384_Update");
        ret = wc_Sha384Update((wc_Sha384*)sha, (const byte*)input, (word32)sz);

        /* return 1 on success, 0 otherwise */
        if (ret == 0)
            return 1;

        return 0;
    }


    int wolfSSL_SHA384_Final(byte* output, WOLFSSL_SHA384_CTX* sha)
    {
        int ret;

        WOLFSSL_ENTER("SHA384_Final");
        ret = wc_Sha384Final((wc_Sha384*)sha, output);

        /* have to actually free the resources (if any) here, because the
         * OpenSSL API doesn't include SHA*_Free().
         */
        wc_Sha384Free((wc_Sha384*)sha);

        /* return 1 on success, 0 otherwise */
        if (ret == 0)
            return 1;

        return 0;
    }

#endif /* WOLFSSL_SHA384 */


#ifdef WOLFSSL_SHA512

    int wolfSSL_SHA512_Init(WOLFSSL_SHA512_CTX* sha)
    {
        int ret;

        typedef char sha_test[sizeof(SHA512_CTX) >= sizeof(wc_Sha512) ? 1 : -1];
        (void)sizeof(sha_test);

        WOLFSSL_ENTER("SHA512_Init");
        ret = wc_InitSha512((wc_Sha512*)sha);

        /* return 1 on success, 0 otherwise */
        if (ret == 0)
            return 1;

        return 0;
    }


    int wolfSSL_SHA512_Update(WOLFSSL_SHA512_CTX* sha, const void* input,
                           unsigned long sz)
    {
        int ret;

        WOLFSSL_ENTER("SHA512_Update");
        ret = wc_Sha512Update((wc_Sha512*)sha, (const byte*)input, (word32)sz);

        /* return 1 on success, 0 otherwise */
        if (ret == 0)
            return 1;

        return 0;
    }


    int wolfSSL_SHA512_Final(byte* output, WOLFSSL_SHA512_CTX* sha)
    {
        int ret;

        WOLFSSL_ENTER("SHA512_Final");
        ret = wc_Sha512Final((wc_Sha512*)sha, output);

        /* have to actually free the resources (if any) here, because the
         * OpenSSL API doesn't include SHA*_Free().
         */
        wc_Sha512Free((wc_Sha512*)sha);

        /* return 1 on success, 0 otherwise */
        if (ret == 0)
            return 1;

        return 0;
    }

    #if !defined(HAVE_SELFTEST) && (!defined(HAVE_FIPS) || \
        (defined(HAVE_FIPS_VERSION) && (HAVE_FIPS_VERSION > 2)))
    /* Apply SHA512 transformation to the data */
    int wolfSSL_SHA512_Transform(WOLFSSL_SHA512_CTX* sha512,
                                          const unsigned char* data)
    {
       int ret;

       WOLFSSL_ENTER("SHA512_Transform");
       /* sanity check */
       if (sha512 == NULL || data == NULL) {
            return WOLFSSL_FAILURE;
       }

       ret = wc_Sha512Transform((wc_Sha512*)sha512, data);

       /* return 1 on success, 0 otherwise */
        if (ret == 0)
            return WOLFSSL_SUCCESS;
        else
            return WOLFSSL_FAILURE;
    }
    #endif /* !defined(HAVE_FIPS) || (defined(HAVE_FIPS_VERSION) && \
              (HAVE_FIPS_VERSION > 2)) */

#if !defined(HAVE_FIPS) && !defined(HAVE_SELFTEST)
#if !defined(WOLFSSL_NOSHA512_224)
    int wolfSSL_SHA512_224_Init(WOLFSSL_SHA512_224_CTX* sha)
    {
        int ret;

        WOLFSSL_ENTER("wolfSSL_SHA512_224_Init");
        ret = wc_InitSha512_224((wc_Sha512*)sha);

        /* return WOLFSSL_SUCCESS on success, WOLFSSL_FAILURE otherwise */
        if (ret == 0)
            return WOLFSSL_SUCCESS;

        return WOLFSSL_FAILURE;
    }

    int wolfSSL_SHA512_224_Update(WOLFSSL_SHA512_224_CTX* sha,
                                        const void* input, unsigned long sz)
    {
        int ret;

        WOLFSSL_ENTER("wolfSSL_SHA512_224_Update");
        ret = wc_Sha512_224Update((wc_Sha512*)sha, (const byte*)input, (word32)sz);

        /* return WOLFSSL_SUCCESS on success, WOLFSSL_FAILURE otherwise */
        if (ret == 0)
            return WOLFSSL_SUCCESS;

        return WOLFSSL_FAILURE;
    }

    int wolfSSL_SHA512_224_Final(byte* output, WOLFSSL_SHA512_224_CTX* sha)
    {
        int ret;

        WOLFSSL_ENTER("wolfSSL_SHA512_224_Final");
        ret = wc_Sha512_224Final((wc_Sha512*)sha, output);

        /* return WOLFSSL_SUCCESS on success, WOLFSSL_FAILURE otherwise */
        if (ret == 0)
            return WOLFSSL_SUCCESS;

        return WOLFSSL_FAILURE;
    }

    #if !defined(HAVE_SELFTEST) && (!defined(HAVE_FIPS) || \
        (defined(HAVE_FIPS_VERSION) && (HAVE_FIPS_VERSION > 2)))
    /* Apply SHA512 transformation to the data */
    int wolfSSL_SHA512_224_Transform(WOLFSSL_SHA512_CTX* sha512,
                                          const unsigned char* data)
    {
       int ret;

       WOLFSSL_ENTER("SHA512_224_Transform");
       /* sanity check */
       if (sha512 == NULL || data == NULL) {
            return WOLFSSL_FAILURE;
       }

       ret = wc_Sha512_224Transform((wc_Sha512*)sha512, data);

       /* return 1 on success, 0 otherwise */
        if (ret == 0)
            return WOLFSSL_SUCCESS;
        else
            return WOLFSSL_FAILURE;
    }
    #endif /* !defined(HAVE_FIPS) || (defined(HAVE_FIPS_VERSION) && \
              (HAVE_FIPS_VERSION > 2)) */

#endif /* !WOLFSSL_NOSHA512_224 */
#if !defined(WOLFSSL_NOSHA512_256)
    int wolfSSL_SHA512_256_Init(WOLFSSL_SHA512_256_CTX* sha)
    {
        int ret;

        WOLFSSL_ENTER("wolfSSL_SHA512_256_Init");
        ret = wc_InitSha512_256((wc_Sha512*)sha);

        /* return WOLFSSL_SUCCESS on success, WOLFSSL_FAILURE otherwise */
        if (ret == 0)
            return WOLFSSL_SUCCESS;

        return WOLFSSL_FAILURE;
    }

    int wolfSSL_SHA512_256_Update(WOLFSSL_SHA512_256_CTX* sha,
                                        const void* input, unsigned long sz)
    {
        int ret;

        WOLFSSL_ENTER("wolfSSL_SHA512_256_Update");
        ret = wc_Sha512_256Update((wc_Sha512*)sha, (const byte*)input, (word32)sz);

        /* return WOLFSSL_SUCCESS on success, WOLFSSL_FAILURE otherwise */
        if (ret == 0)
            return WOLFSSL_SUCCESS;

        return WOLFSSL_FAILURE;
    }

    int wolfSSL_SHA512_256_Final(byte* output, WOLFSSL_SHA512_256_CTX* sha)
    {
        int ret;

        WOLFSSL_ENTER("wolfSSL_SHA512_256_Final");
        ret = wc_Sha512_256Final((wc_Sha512*)sha, output);

        /* return WOLFSSL_SUCCESS on success, 0 otherwise */
        if (ret == 0)
            return WOLFSSL_SUCCESS;

        return WOLFSSL_FAILURE;
    }

    #if !defined(HAVE_SELFTEST) && (!defined(HAVE_FIPS) || \
        (defined(HAVE_FIPS_VERSION) && (HAVE_FIPS_VERSION > 2)))
    /* Apply SHA512 transformation to the data */
    int wolfSSL_SHA512_256_Transform(WOLFSSL_SHA512_CTX* sha512,
                                          const unsigned char* data)
    {
       int ret;

       WOLFSSL_ENTER("SHA512_256_Transform");
       /* sanity check */
       if (sha512 == NULL || data == NULL) {
            return WOLFSSL_FAILURE;
       }

       ret = wc_Sha512_256Transform((wc_Sha512*)sha512, data);

       /* return 1 on success, 0 otherwise */
        if (ret == 0)
            return WOLFSSL_SUCCESS;
        else
            return WOLFSSL_FAILURE;
    }
    #endif /* !defined(HAVE_FIPS) || (defined(HAVE_FIPS_VERSION) && \
              (HAVE_FIPS_VERSION > 2)) */

#endif /* !WOLFSSL_NOSHA512_256 */
#endif /* !HAVE_FIPS && !HAVE_SELFTEST */

#endif /* WOLFSSL_SHA512 */

#ifdef WOLFSSL_SHA3
#ifndef WOLFSSL_NOSHA3_224

    int wolfSSL_SHA3_224_Init(WOLFSSL_SHA3_224_CTX* sha)
    {
        int ret;

        typedef char sha_test[sizeof(SHA3_224_CTX) >= sizeof(wc_Sha3) ? 1 : -1];
        (void)sizeof(sha_test);

        WOLFSSL_ENTER("SHA3_224_Init");
        ret = wc_InitSha3_224((wc_Sha3*)sha, NULL, INVALID_DEVID);

        /* return 1 on success, 0 otherwise */
        if (ret == 0)
            return 1;

        return 0;
    }


    int wolfSSL_SHA3_224_Update(WOLFSSL_SHA3_224_CTX* sha, const void* input,
                           unsigned long sz)
    {
        int ret;

        WOLFSSL_ENTER("SHA3_224_Update");
        ret = wc_Sha3_224_Update((wc_Sha3*)sha, (const byte*)input, (word32)sz);

        /* return 1 on success, 0 otherwise */
        if (ret == 0)
            return 1;

        return 0;
    }


    int wolfSSL_SHA3_224_Final(byte* output, WOLFSSL_SHA3_224_CTX* sha)
    {
        int ret;

        WOLFSSL_ENTER("SHA3_224_Final");
        ret = wc_Sha3_224_Final((wc_Sha3*)sha, output);

        /* have to actually free the resources (if any) here, because the
         * OpenSSL API doesn't include SHA*_Free().
         */
        wc_Sha3_224_Free((wc_Sha3*)sha);

        /* return 1 on success, 0 otherwise */
        if (ret == 0)
            return 1;

        return 0;
    }

#endif /* WOLFSSL_NOSHA3_224 */


#ifndef WOLFSSL_NOSHA3_256
    int wolfSSL_SHA3_256_Init(WOLFSSL_SHA3_256_CTX* sha3_256)
    {
        int ret;

        typedef char sha_test[sizeof(SHA3_256_CTX) >= sizeof(wc_Sha3) ? 1 : -1];
        (void)sizeof(sha_test);

        WOLFSSL_ENTER("SHA3_256_Init");
        ret = wc_InitSha3_256((wc_Sha3*)sha3_256, NULL, INVALID_DEVID);

        /* return 1 on success, 0 otherwise */
        if (ret == 0)
            return 1;

        return 0;
    }


    int wolfSSL_SHA3_256_Update(WOLFSSL_SHA3_256_CTX* sha, const void* input,
                              unsigned long sz)
    {
        int ret;

        WOLFSSL_ENTER("SHA3_256_Update");
        ret = wc_Sha3_256_Update((wc_Sha3*)sha, (const byte*)input, (word32)sz);

        /* return 1 on success, 0 otherwise */
        if (ret == 0)
            return 1;

        return 0;
    }


    int wolfSSL_SHA3_256_Final(byte* output, WOLFSSL_SHA3_256_CTX* sha)
    {
        int ret;

        WOLFSSL_ENTER("SHA3_256_Final");
        ret = wc_Sha3_256_Final((wc_Sha3*)sha, output);

        /* have to actually free the resources (if any) here, because the
         * OpenSSL API doesn't include SHA*_Free().
         */
        wc_Sha3_256_Free((wc_Sha3*)sha);

        /* return 1 on success, 0 otherwise */
        if (ret == 0)
            return 1;

        return 0;
    }
#endif /* WOLFSSL_NOSHA3_256 */


    int wolfSSL_SHA3_384_Init(WOLFSSL_SHA3_384_CTX* sha)
    {
        int ret;

        typedef char sha_test[sizeof(SHA3_384_CTX) >= sizeof(wc_Sha3) ? 1 : -1];
        (void)sizeof(sha_test);

        WOLFSSL_ENTER("SHA3_384_Init");
        ret = wc_InitSha3_384((wc_Sha3*)sha, NULL, INVALID_DEVID);

        /* return 1 on success, 0 otherwise */
        if (ret == 0)
            return 1;

        return 0;
    }


    int wolfSSL_SHA3_384_Update(WOLFSSL_SHA3_384_CTX* sha, const void* input,
                           unsigned long sz)
    {
        int ret;

        WOLFSSL_ENTER("SHA3_384_Update");
        ret = wc_Sha3_384_Update((wc_Sha3*)sha, (const byte*)input, (word32)sz);

        /* return 1 on success, 0 otherwise */
        if (ret == 0)
            return 1;

        return 0;
    }


    int wolfSSL_SHA3_384_Final(byte* output, WOLFSSL_SHA3_384_CTX* sha)
    {
        int ret;

        WOLFSSL_ENTER("SHA3_384_Final");
        ret = wc_Sha3_384_Final((wc_Sha3*)sha, output);

        /* have to actually free the resources (if any) here, because the
         * OpenSSL API doesn't include SHA*_Free().
         */
        wc_Sha3_384_Free((wc_Sha3*)sha);

        /* return 1 on success, 0 otherwise */
        if (ret == 0)
            return 1;

        return 0;
    }



#ifndef WOLFSSL_NOSHA3_512

    int wolfSSL_SHA3_512_Init(WOLFSSL_SHA3_512_CTX* sha)
    {
        int ret;

        typedef char sha_test[sizeof(SHA3_512_CTX) >= sizeof(wc_Sha3) ? 1 : -1];
        (void)sizeof(sha_test);

        WOLFSSL_ENTER("SHA3_512_Init");
        ret = wc_InitSha3_512((wc_Sha3*)sha, NULL, INVALID_DEVID);

        /* return 1 on success, 0 otherwise */
        if (ret == 0)
            return 1;

        return 0;
    }


    int wolfSSL_SHA3_512_Update(WOLFSSL_SHA3_512_CTX* sha, const void* input,
                           unsigned long sz)
    {
        int ret;

        WOLFSSL_ENTER("SHA3_512_Update");
        ret = wc_Sha3_512_Update((wc_Sha3*)sha, (const byte*)input, (word32)sz);

        /* return 1 on success, 0 otherwise */
        if (ret == 0)
            return 1;

        return 0;
    }


    int wolfSSL_SHA3_512_Final(byte* output, WOLFSSL_SHA3_512_CTX* sha)
    {
        int ret;

        WOLFSSL_ENTER("SHA3_512_Final");
        ret = wc_Sha3_512_Final((wc_Sha3*)sha, output);

        /* have to actually free the resources (if any) here, because the
         * OpenSSL API doesn't include SHA*_Free().
         */
        wc_Sha3_512_Free((wc_Sha3*)sha);

        /* return 1 on success, 0 otherwise */
        if (ret == 0)
            return 1;

        return 0;
    }

#endif /* WOLFSSL_NOSHA3_512 */
#endif /* WOLFSSL_SHA3 */

    unsigned char* wolfSSL_HMAC(const WOLFSSL_EVP_MD* evp_md, const void* key,
                                int key_len, const unsigned char* d, int n,
                                unsigned char* md, unsigned int* md_len)
    {
        int type;
        int mdlen;
        unsigned char* ret = NULL;
#ifdef WOLFSSL_SMALL_STACK
        Hmac* hmac = NULL;
#else
        Hmac  hmac[1];
#endif
        void* heap = NULL;

        WOLFSSL_ENTER("wolfSSL_HMAC");
        if (!md) {
            WOLFSSL_MSG("Static buffer not supported, pass in md buffer");
            return NULL;  /* no static buffer support */
        }

#ifndef NO_MD5
        if (XSTRNCMP(evp_md, "MD5", 3) == 0) {
            type = WC_MD5;
            mdlen = WC_MD5_DIGEST_SIZE;
        } else
#endif
#ifdef WOLFSSL_SHA224
        if (XSTRNCMP(evp_md, "SHA224", 6) == 0) {
            type = WC_SHA224;
            mdlen = WC_SHA224_DIGEST_SIZE;
        } else
#endif
#ifndef NO_SHA256
        if (XSTRNCMP(evp_md, "SHA256", 6) == 0) {
            type = WC_SHA256;
            mdlen = WC_SHA256_DIGEST_SIZE;
        } else
#endif
#ifdef WOLFSSL_SHA384
        if (XSTRNCMP(evp_md, "SHA384", 6) == 0) {
            type = WC_SHA384;
            mdlen = WC_SHA384_DIGEST_SIZE;
        } else
#endif
#ifdef WOLFSSL_SHA512
        if (XSTRNCMP(evp_md, "SHA512", 6) == 0) {
            type = WC_SHA512;
            mdlen = WC_SHA512_DIGEST_SIZE;
        } else
#endif
#ifdef WOLFSSL_SHA3
    #ifndef WOLFSSL_NOSHA3_224
        if (XSTRNCMP(evp_md, "SHA3_224", 8) == 0) {
            type = WC_SHA3_224;
            mdlen = WC_SHA3_224_DIGEST_SIZE;
        } else
    #endif
    #ifndef WOLFSSL_NOSHA3_256
        if (XSTRNCMP(evp_md, "SHA3_256", 8) == 0) {
            type = WC_SHA3_256;
            mdlen = WC_SHA3_256_DIGEST_SIZE;
        } else
    #endif
        if (XSTRNCMP(evp_md, "SHA3_384", 8) == 0) {
            type = WC_SHA3_384;
            mdlen = WC_SHA3_384_DIGEST_SIZE;
        } else
    #ifndef WOLFSSL_NOSHA3_512
        if (XSTRNCMP(evp_md, "SHA3_512", 8) == 0) {
            type = WC_SHA3_512;
            mdlen = WC_SHA3_512_DIGEST_SIZE;
        } else
    #endif
#endif
#ifndef NO_SHA
        if (XSTRNCMP(evp_md, "SHA", 3) == 0) {
            type = WC_SHA;
            mdlen = WC_SHA_DIGEST_SIZE;
        } else
#endif
        {
            return NULL;
        }

    #ifdef WOLFSSL_SMALL_STACK
        hmac = (Hmac*)XMALLOC(sizeof(Hmac), heap, DYNAMIC_TYPE_HMAC);
        if (hmac == NULL)
            return NULL;
    #endif

        if (wc_HmacInit(hmac, heap, INVALID_DEVID) == 0) {
            if (wc_HmacSetKey(hmac, type, (const byte*)key, key_len) == 0) {
                if (wc_HmacUpdate(hmac, d, n) == 0) {
                    if (wc_HmacFinal(hmac, md) == 0) {
                        if (md_len)
                            *md_len = mdlen;
                        ret = md;
                    }
                }
            }
            wc_HmacFree(hmac);
        }

    #ifdef WOLFSSL_SMALL_STACK
        XFREE(hmac, heap, DYNAMIC_TYPE_HMAC);
    #endif

        (void)evp_md;
        return ret;
    }

    void wolfSSL_ERR_clear_error(void)
    {
        WOLFSSL_ENTER("wolfSSL_ERR_clear_error");
#if defined(OPENSSL_EXTRA) || defined(DEBUG_WOLFSSL_VERBOSE)
        wc_ClearErrorNodes();
#endif
    }

#ifndef NO_DES3
    /* 0 on ok */
    int wolfSSL_DES_key_sched(WOLFSSL_const_DES_cblock* key,
                              WOLFSSL_DES_key_schedule* schedule)
    {
        WOLFSSL_ENTER("wolfSSL_DES_key_sched");

        if (key == NULL || schedule == NULL) {
            WOLFSSL_MSG("Null argument passed in");
        }
        else {
            XMEMCPY(schedule, key, sizeof(WOLFSSL_const_DES_cblock));
        }

        return 0;
    }


    /* intended to behave similar to Kerberos mit_des_cbc_cksum
     * return the last 4 bytes of cipher text */
    WOLFSSL_DES_LONG wolfSSL_DES_cbc_cksum(const unsigned char* in,
            WOLFSSL_DES_cblock* out, long length, WOLFSSL_DES_key_schedule* sc,
            WOLFSSL_const_DES_cblock* iv)
    {
        WOLFSSL_DES_LONG ret;
        unsigned char* tmp;
        unsigned char* data   = (unsigned char*)in;
        long           dataSz = length;
        byte dynamicFlag = 0; /* when padding the buffer created needs free'd */

        WOLFSSL_ENTER("wolfSSL_DES_cbc_cksum");

        if (in == NULL || out == NULL || sc == NULL || iv == NULL) {
            WOLFSSL_MSG("Bad argument passed in");
            return 0;
        }

        /* if input length is not a multiple of DES_BLOCK_SIZE pad with 0s */
        if (dataSz % DES_BLOCK_SIZE) {
            dataSz += DES_BLOCK_SIZE - (dataSz % DES_BLOCK_SIZE);
            data = (unsigned char*)XMALLOC(dataSz, NULL,
                                           DYNAMIC_TYPE_TMP_BUFFER);
            if (data == NULL) {
                WOLFSSL_MSG("Issue creating temporary buffer");
                return 0;
            }
            dynamicFlag = 1; /* set to free buffer at end */
            XMEMCPY(data, in, length);
            XMEMSET(data + length, 0, dataSz - length); /* padding */
        }

        tmp = (unsigned char*)XMALLOC(dataSz, NULL, DYNAMIC_TYPE_TMP_BUFFER);
        if (tmp == NULL) {
            WOLFSSL_MSG("Issue creating temporary buffer");
            if (dynamicFlag == 1) {
                XFREE(data, NULL, DYNAMIC_TYPE_TMP_BUFFER);
            }
            return 0;
        }

        wolfSSL_DES_cbc_encrypt(data, tmp, dataSz, sc,
                (WOLFSSL_DES_cblock*)iv, 1);
        XMEMCPY((unsigned char*)out, tmp + (dataSz - DES_BLOCK_SIZE),
                DES_BLOCK_SIZE);

        ret = (((*((unsigned char*)out + 4) & 0xFF) << 24)|
               ((*((unsigned char*)out + 5) & 0xFF) << 16)|
               ((*((unsigned char*)out + 6) & 0xFF) << 8) |
               (*((unsigned char*)out + 7) & 0xFF));

        XFREE(tmp, NULL, DYNAMIC_TYPE_TMP_BUFFER);
        if (dynamicFlag == 1) {
            XFREE(data, NULL, DYNAMIC_TYPE_TMP_BUFFER);
        }

        return ret;
    }


    void wolfSSL_DES_cbc_encrypt(const unsigned char* input,
                                 unsigned char* output, long length,
                                 WOLFSSL_DES_key_schedule* schedule,
                                 WOLFSSL_DES_cblock* ivec, int enc)
    {
        Des myDes;
        byte lastblock[DES_BLOCK_SIZE];
        int  lb_sz;
        long  blk;

        WOLFSSL_ENTER("DES_cbc_encrypt");

        /* OpenSSL compat, no ret */
        if (wc_Des_SetKey(&myDes, (const byte*)schedule, (const byte*)ivec,
                !enc) != 0) {
            WOLFSSL_MSG("wc_Des_SetKey return error.");
            return;
        }
        lb_sz = length%DES_BLOCK_SIZE;
        blk   = length/DES_BLOCK_SIZE;

        if (enc == DES_ENCRYPT){
            wc_Des_CbcEncrypt(&myDes, output, input, (word32)blk*DES_BLOCK_SIZE);
            if(lb_sz){
                XMEMSET(lastblock, 0, DES_BLOCK_SIZE);
                XMEMCPY(lastblock, input+length-lb_sz, lb_sz);
                wc_Des_CbcEncrypt(&myDes, output+blk*DES_BLOCK_SIZE,
                    lastblock, (word32)DES_BLOCK_SIZE);
            }
        }
        else {
            wc_Des_CbcDecrypt(&myDes, output, input, (word32)blk*DES_BLOCK_SIZE);
            if(lb_sz){
                wc_Des_CbcDecrypt(&myDes, lastblock, input+length-lb_sz, (word32)DES_BLOCK_SIZE);
                XMEMCPY(output+length-lb_sz, lastblock, lb_sz);
            }
        }
    }


    /* WOLFSSL_DES_key_schedule is a unsigned char array of size 8 */
    void wolfSSL_DES_ede3_cbc_encrypt(const unsigned char* input,
                                      unsigned char* output, long sz,
                                      WOLFSSL_DES_key_schedule* ks1,
                                      WOLFSSL_DES_key_schedule* ks2,
                                      WOLFSSL_DES_key_schedule* ks3,
                                      WOLFSSL_DES_cblock* ivec, int enc)
    {
        int ret;
        Des3 des;
        byte key[24];/* EDE uses 24 size key */
        byte lastblock[DES_BLOCK_SIZE];
        int  lb_sz;
        long  blk;

        WOLFSSL_ENTER("wolfSSL_DES_ede3_cbc_encrypt");

        XMEMSET(key, 0, sizeof(key));
        XMEMCPY(key, *ks1, DES_BLOCK_SIZE);
        XMEMCPY(&key[DES_BLOCK_SIZE], *ks2, DES_BLOCK_SIZE);
        XMEMCPY(&key[DES_BLOCK_SIZE * 2], *ks3, DES_BLOCK_SIZE);
        lb_sz = sz%DES_BLOCK_SIZE;
        blk   = sz/DES_BLOCK_SIZE;

        /* OpenSSL compat, no ret */
        (void)wc_Des3Init(&des, NULL, INVALID_DEVID);

        if (enc == DES_ENCRYPT) {
            if (wc_Des3_SetKey(&des, key, (const byte*)ivec,
                    DES_ENCRYPTION) == 0) {
                ret = wc_Des3_CbcEncrypt(&des, output, input, (word32)blk*DES_BLOCK_SIZE);
            #if defined(WOLFSSL_ASYNC_CRYPT)
                ret = wc_AsyncWait(ret, &des.asyncDev, WC_ASYNC_FLAG_NONE);
            #endif
                (void)ret; /* ignore return codes for processing */
                if(lb_sz){
                    XMEMSET(lastblock, 0, DES_BLOCK_SIZE);
                    XMEMCPY(lastblock, input+sz-lb_sz, lb_sz);
                    ret = wc_Des3_CbcEncrypt(&des, output+blk*DES_BLOCK_SIZE,
                        lastblock, (word32)DES_BLOCK_SIZE);
                #if defined(WOLFSSL_ASYNC_CRYPT)
                    ret = wc_AsyncWait(ret, &des.asyncDev, WC_ASYNC_FLAG_NONE);
                #endif
                    (void)ret; /* ignore return codes for processing */
                }
            }
        }
        else {
            if (wc_Des3_SetKey(&des, key, (const byte*)ivec,
                    DES_DECRYPTION) == 0) {
                ret = wc_Des3_CbcDecrypt(&des, output, input, (word32)blk*DES_BLOCK_SIZE);
            #if defined(WOLFSSL_ASYNC_CRYPT)
                ret = wc_AsyncWait(ret, &des.asyncDev, WC_ASYNC_FLAG_NONE);
            #endif
                (void)ret; /* ignore return codes for processing */
                if(lb_sz){
                    ret = wc_Des3_CbcDecrypt(&des, lastblock, input+sz-lb_sz, (word32)DES_BLOCK_SIZE);
                #if defined(WOLFSSL_ASYNC_CRYPT)
                    ret = wc_AsyncWait(ret, &des.asyncDev, WC_ASYNC_FLAG_NONE);
                #endif
                    (void)ret; /* ignore return codes for processing */
                    XMEMCPY(output+sz-lb_sz, lastblock, lb_sz);
                }
            }
        }
        wc_Des3Free(&des);
    }


    /* correctly sets ivec for next call */
    void wolfSSL_DES_ncbc_encrypt(const unsigned char* input,
                     unsigned char* output, long length,
                     WOLFSSL_DES_key_schedule* schedule, WOLFSSL_DES_cblock* ivec,
                     int enc)
    {
        Des myDes;
        byte lastblock[DES_BLOCK_SIZE];
        int  lb_sz;
        long idx = length;
        long blk;

        WOLFSSL_ENTER("DES_ncbc_encrypt");

        /* OpenSSL compat, no ret */
        if (wc_Des_SetKey(&myDes, (const byte*)schedule,
                         (const byte*)ivec, !enc) != 0) {
            WOLFSSL_MSG("wc_Des_SetKey return error.");
            return;
        }

        lb_sz = length%DES_BLOCK_SIZE;
        blk   = length/DES_BLOCK_SIZE;
        idx  -= sizeof(DES_cblock);
        if (lb_sz) {
            idx += DES_BLOCK_SIZE - lb_sz;
        }
        if (enc == DES_ENCRYPT){
            wc_Des_CbcEncrypt(&myDes, output, input,
                    (word32)blk * DES_BLOCK_SIZE);
            if (lb_sz){
                XMEMSET(lastblock, 0, DES_BLOCK_SIZE);
                XMEMCPY(lastblock, input+length-lb_sz, lb_sz);
                wc_Des_CbcEncrypt(&myDes, output + blk * DES_BLOCK_SIZE,
                    lastblock, (word32)DES_BLOCK_SIZE);
            }
            XMEMCPY(ivec, output + idx, sizeof(DES_cblock));
        } else {
            WOLFSSL_DES_cblock tmp;
            XMEMCPY(tmp, input + idx, sizeof(DES_cblock));
            wc_Des_CbcDecrypt(&myDes, output, input,
                    (word32)blk * DES_BLOCK_SIZE);
            if (lb_sz){
                wc_Des_CbcDecrypt(&myDes, lastblock, input + length - lb_sz,
                        (word32)DES_BLOCK_SIZE);
                XMEMCPY(output+length-lb_sz, lastblock, lb_sz);
            }
            XMEMCPY(ivec, tmp, sizeof(WOLFSSL_DES_cblock));
        }

    }

#endif /* NO_DES3 */

    void wolfSSL_ERR_free_strings(void)
    {
        /* handled internally */
    }

    void wolfSSL_cleanup_all_ex_data(void)
    {
        /* nothing to do here */
    }

#endif /* OPENSSL_EXTRA */

#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_WPAS_SMALL)
    int wolfSSL_clear(WOLFSSL* ssl)
    {
        if (ssl == NULL) {
            return WOLFSSL_FAILURE;
        }

        ssl->options.isClosed = 0;
        ssl->options.connReset = 0;
        ssl->options.sentNotify = 0;
        ssl->options.closeNotify = 0;
        ssl->options.sendVerify = 0;
        ssl->options.serverState = NULL_STATE;
        ssl->options.clientState = NULL_STATE;
        ssl->options.connectState = CONNECT_BEGIN;
        ssl->options.acceptState  = ACCEPT_BEGIN;
        ssl->options.handShakeState  = NULL_STATE;
        ssl->options.handShakeDone = 0;
        ssl->options.processReply = 0; /* doProcessInit */

        ssl->keys.encryptionOn = 0;
        XMEMSET(&ssl->msgsReceived, 0, sizeof(ssl->msgsReceived));

        if (ssl->hsHashes)
            (void)InitHandshakeHashes(ssl);

#ifdef SESSION_CERTS
        ssl->session->chain.count = 0;
#endif
#ifdef KEEP_PEER_CERT
        FreeX509(&ssl->peerCert);
        InitX509(&ssl->peerCert, 0, ssl->heap);
#endif

        return WOLFSSL_SUCCESS;
    }

#endif /* OPENSSL_EXTRA || WOLFSSL_WPAS_SMALL */

#if defined(OPENSSL_EXTRA) || defined(HAVE_WEBSERVER)
    long wolfSSL_CTX_set_mode(WOLFSSL_CTX* ctx, long mode)
    {
        /* WOLFSSL_MODE_ACCEPT_MOVING_WRITE_BUFFER is wolfSSL default mode */

        WOLFSSL_ENTER("SSL_CTX_set_mode");
        switch(mode) {
            case SSL_MODE_ENABLE_PARTIAL_WRITE:
                ctx->partialWrite = 1;
                break;
            #if defined(OPENSSL_ALL) || defined(WOLFSSL_QT)
            case SSL_MODE_RELEASE_BUFFERS:
                WOLFSSL_MSG("SSL_MODE_RELEASE_BUFFERS not implemented.");
                break;
            #endif
            case SSL_MODE_AUTO_RETRY:
                ctx->autoRetry = 1;
                break;
            default:
                WOLFSSL_MSG("Mode Not Implemented");
        }

        /* SSL_MODE_AUTO_RETRY
         * Should not return -1 with renegotiation on read/write */

        return mode;
    }

    long wolfSSL_CTX_clear_mode(WOLFSSL_CTX* ctx, long mode)
    {
        /* WOLFSSL_MODE_ACCEPT_MOVING_WRITE_BUFFER is wolfSSL default mode */

        WOLFSSL_ENTER("SSL_CTX_set_mode");
        switch(mode) {
            case SSL_MODE_ENABLE_PARTIAL_WRITE:
                ctx->partialWrite = 0;
                break;
            #if defined(OPENSSL_ALL) || defined(WOLFSSL_QT)
            case SSL_MODE_RELEASE_BUFFERS:
                WOLFSSL_MSG("SSL_MODE_RELEASE_BUFFERS not implemented.");
                break;
            #endif
            case SSL_MODE_AUTO_RETRY:
                ctx->autoRetry = 0;
                break;
            default:
                WOLFSSL_MSG("Mode Not Implemented");
        }

        /* SSL_MODE_AUTO_RETRY
         * Should not return -1 with renegotiation on read/write */

        return 0;
    }
#endif

#ifdef OPENSSL_EXTRA

    #ifndef NO_WOLFSSL_STUB
    long wolfSSL_SSL_get_mode(WOLFSSL* ssl)
    {
        /* TODO: */
        (void)ssl;
        WOLFSSL_STUB("SSL_get_mode");
        return 0;
    }
    #endif

    #ifndef NO_WOLFSSL_STUB
    long wolfSSL_CTX_get_mode(WOLFSSL_CTX* ctx)
    {
        /* TODO: */
        (void)ctx;
        WOLFSSL_STUB("SSL_CTX_get_mode");
        return 0;
    }
    #endif

    #ifndef NO_WOLFSSL_STUB
    void wolfSSL_CTX_set_default_read_ahead(WOLFSSL_CTX* ctx, int m)
    {
        /* TODO: maybe? */
        (void)ctx;
        (void)m;
        WOLFSSL_STUB("SSL_CTX_set_default_read_ahead");
    }
    #endif


    /* Storing app session context id, this value is inherited by WOLFSSL
     * objects created from WOLFSSL_CTX. Any session that is imported with a
     * different session context id will be rejected.
     *
     * ctx         structure to set context in
     * sid_ctx     value of context to set
     * sid_ctx_len length of sid_ctx buffer
     *
     * Returns WOLFSSL_SUCCESS in success case and SSL_FAILURE when failing
     */
    int wolfSSL_CTX_set_session_id_context(WOLFSSL_CTX* ctx,
                                           const unsigned char* sid_ctx,
                                           unsigned int sid_ctx_len)
    {
        WOLFSSL_ENTER("SSL_CTX_set_session_id_context");

        /* No application specific context needed for wolfSSL */
        if (sid_ctx_len > ID_LEN || ctx == NULL || sid_ctx == NULL) {
            return SSL_FAILURE;
        }
        XMEMCPY(ctx->sessionCtx, sid_ctx, sid_ctx_len);
        ctx->sessionCtxSz = (byte)sid_ctx_len;

        return WOLFSSL_SUCCESS;
    }



    /* Storing app session context id. Any session that is imported with a
     * different session context id will be rejected.
     *
     * ssl  structure to set context in
     * id   value of context to set
     * len  length of sid_ctx buffer
     *
     * Returns WOLFSSL_SUCCESS in success case and SSL_FAILURE when failing
     */
    int wolfSSL_set_session_id_context(WOLFSSL* ssl, const unsigned char* id,
                                   unsigned int len)
    {
        WOLFSSL_ENTER("wolfSSL_set_session_id_context");

        if (len > ID_LEN || ssl == NULL || id == NULL) {
            return SSL_FAILURE;
        }
        XMEMCPY(ssl->sessionCtx, id, len);
        ssl->sessionCtxSz = (byte)len;

        return WOLFSSL_SUCCESS;
    }


    long wolfSSL_CTX_sess_get_cache_size(WOLFSSL_CTX* ctx)
    {
        (void)ctx;
        #ifndef NO_SESSION_CACHE
            return (long)(SESSIONS_PER_ROW * SESSION_ROWS);
        #else
            return 0;
        #endif
    }


    /* returns the unsigned error value and increments the pointer into the
     * error queue.
     *
     * file  pointer to file name
     * line  gets set to line number of error when not NULL
     */
    unsigned long wolfSSL_ERR_get_error_line(const char** file, int* line)
    {
    #ifdef WOLFSSL_HAVE_ERROR_QUEUE
        int ret = wc_PullErrorNode(file, NULL, line);
        if (ret < 0) {
            if (ret == BAD_STATE_E) return 0; /* no errors in queue */
            WOLFSSL_MSG("Issue getting error node");
            WOLFSSL_LEAVE("wolfSSL_ERR_get_error_line", ret);
            ret = 0 - ret; /* return absolute value of error */

            /* panic and try to clear out nodes */
            wc_ClearErrorNodes();
        }
        return (unsigned long)ret;
    #else
        (void)file;
        (void)line;

        return 0;
    #endif
    }


#if (defined(DEBUG_WOLFSSL) || defined(OPENSSL_EXTRA)) && \
    (!defined(_WIN32) && !defined(NO_ERROR_QUEUE))
    static const char WOLFSSL_SYS_ACCEPT_T[]  = "accept";
    static const char WOLFSSL_SYS_BIND_T[]    = "bind";
    static const char WOLFSSL_SYS_CONNECT_T[] = "connect";
    static const char WOLFSSL_SYS_FOPEN_T[]   = "fopen";
    static const char WOLFSSL_SYS_FREAD_T[]   = "fread";
    static const char WOLFSSL_SYS_GETADDRINFO_T[] = "getaddrinfo";
    static const char WOLFSSL_SYS_GETSOCKOPT_T[]  = "getsockopt";
    static const char WOLFSSL_SYS_GETSOCKNAME_T[] = "getsockname";
    static const char WOLFSSL_SYS_GETHOSTBYNAME_T[] = "gethostbyname";
    static const char WOLFSSL_SYS_GETNAMEINFO_T[]   = "getnameinfo";
    static const char WOLFSSL_SYS_GETSERVBYNAME_T[] = "getservbyname";
    static const char WOLFSSL_SYS_IOCTLSOCKET_T[]   = "ioctlsocket";
    static const char WOLFSSL_SYS_LISTEN_T[]        = "listen";
    static const char WOLFSSL_SYS_OPENDIR_T[]       = "opendir";
    static const char WOLFSSL_SYS_SETSOCKOPT_T[]    = "setsockopt";
    static const char WOLFSSL_SYS_SOCKET_T[]        = "socket";

    /* switch with int mapped to function name for compatibility */
    static const char* wolfSSL_ERR_sys_func(int fun)
    {
        switch (fun) {
            case WOLFSSL_SYS_ACCEPT:      return WOLFSSL_SYS_ACCEPT_T;
            case WOLFSSL_SYS_BIND:        return WOLFSSL_SYS_BIND_T;
            case WOLFSSL_SYS_CONNECT:     return WOLFSSL_SYS_CONNECT_T;
            case WOLFSSL_SYS_FOPEN:       return WOLFSSL_SYS_FOPEN_T;
            case WOLFSSL_SYS_FREAD:       return WOLFSSL_SYS_FREAD_T;
            case WOLFSSL_SYS_GETADDRINFO: return WOLFSSL_SYS_GETADDRINFO_T;
            case WOLFSSL_SYS_GETSOCKOPT:  return WOLFSSL_SYS_GETSOCKOPT_T;
            case WOLFSSL_SYS_GETSOCKNAME: return WOLFSSL_SYS_GETSOCKNAME_T;
            case WOLFSSL_SYS_GETHOSTBYNAME: return WOLFSSL_SYS_GETHOSTBYNAME_T;
            case WOLFSSL_SYS_GETNAMEINFO: return WOLFSSL_SYS_GETNAMEINFO_T;
            case WOLFSSL_SYS_GETSERVBYNAME: return WOLFSSL_SYS_GETSERVBYNAME_T;
            case WOLFSSL_SYS_IOCTLSOCKET: return WOLFSSL_SYS_IOCTLSOCKET_T;
            case WOLFSSL_SYS_LISTEN:      return WOLFSSL_SYS_LISTEN_T;
            case WOLFSSL_SYS_OPENDIR:     return WOLFSSL_SYS_OPENDIR_T;
            case WOLFSSL_SYS_SETSOCKOPT:  return WOLFSSL_SYS_SETSOCKOPT_T;
            case WOLFSSL_SYS_SOCKET:      return WOLFSSL_SYS_SOCKET_T;
            default:
                return "NULL";
        }
    }
#endif /* DEBUG_WOLFSSL */


    /* @TODO when having an error queue this needs to push to the queue */
    void wolfSSL_ERR_put_error(int lib, int fun, int err, const char* file,
            int line)
    {
        WOLFSSL_ENTER("wolfSSL_ERR_put_error");

        #if !defined(DEBUG_WOLFSSL) && !defined(OPENSSL_EXTRA)
        (void)fun;
        (void)err;
        (void)file;
        (void)line;
        WOLFSSL_MSG("Not compiled in debug mode");
        #elif defined(OPENSSL_EXTRA) && \
                (defined(_WIN32) || defined(NO_ERROR_QUEUE))
        (void)fun;
        (void)file;
        (void)line;
        WOLFSSL_ERROR(err);
        #else
        WOLFSSL_ERROR_LINE(err, wolfSSL_ERR_sys_func(fun), (unsigned int)line,
            file, NULL);
        #endif
        (void)lib;
    }


    /* Similar to wolfSSL_ERR_get_error_line but takes in a flags argument for
     * more flexibility.
     *
     * file  output pointer to file where error happened
     * line  output to line number of error
     * data  output data. Is a string if ERR_TXT_STRING flag is used
     * flags output format of output
     *
     * Returns the error value or 0 if no errors are in the queue
     */
    unsigned long wolfSSL_ERR_get_error_line_data(const char** file, int* line,
                                                  const char** data, int *flags)
    {
#ifdef WOLFSSL_HAVE_ERROR_QUEUE
        int ret;

        WOLFSSL_ENTER("wolfSSL_ERR_get_error_line_data");

        if (flags != NULL)
            *flags = ERR_TXT_STRING; /* Clear the flags */

        ret = wc_PullErrorNode(file, data, line);
        if (ret < 0) {
            if (ret == BAD_STATE_E) return 0; /* no errors in queue */
            WOLFSSL_MSG("Error with pulling error node!");
            WOLFSSL_LEAVE("wolfSSL_ERR_get_error_line_data", ret);
            ret = 0 - ret; /* return absolute value of error */

            /* panic and try to clear out nodes */
            wc_ClearErrorNodes();
        }

        return (unsigned long)ret;
#else
        WOLFSSL_ENTER("wolfSSL_ERR_get_error_line_data");
        WOLFSSL_MSG("Error queue turned off, can not get error line");
        (void)file;
        (void)line;
        (void)data;
        (void)flags;
        return 0;
#endif
    }

#endif /* OPENSSL_EXTRA */


#if (defined(KEEP_PEER_CERT) && defined(SESSION_CERTS)) || \
    (defined(OPENSSL_EXTRA) && defined(SESSION_CERTS))
    /* Decode the X509 DER encoded certificate into a WOLFSSL_X509 object.
     *
     * x509  WOLFSSL_X509 object to decode into.
     * in    X509 DER data.
     * len   Length of the X509 DER data.
     * returns the new certificate on success, otherwise NULL.
     */
    static int DecodeToX509(WOLFSSL_X509* x509, const byte* in, int len)
    {
        int          ret;
    #ifdef WOLFSSL_SMALL_STACK
        DecodedCert* cert;
    #else
        DecodedCert  cert[1];
    #endif
        if (x509 == NULL || in == NULL || len <= 0)
            return BAD_FUNC_ARG;

    #ifdef WOLFSSL_SMALL_STACK
        cert = (DecodedCert*)XMALLOC(sizeof(DecodedCert), NULL,
                                     DYNAMIC_TYPE_DCERT);
        if (cert == NULL)
            return MEMORY_E;
    #endif

        /* Create a DecodedCert object and copy fields into WOLFSSL_X509 object.
         */
        InitDecodedCert(cert, (byte*)in, len, NULL);
        if ((ret = ParseCertRelative(cert, CERT_TYPE, 0, NULL)) == 0) {
        /* Check if x509 was not previously initialized by wolfSSL_X509_new() */
            if (x509->dynamicMemory != TRUE)
                InitX509(x509, 0, NULL);
            ret = CopyDecodedToX509(x509, cert);
            FreeDecodedCert(cert);
        }
    #ifdef WOLFSSL_SMALL_STACK
        XFREE(cert, NULL, DYNAMIC_TYPE_DCERT);
    #endif

        return ret;
    }
#endif /* (KEEP_PEER_CERT & SESSION_CERTS) || (OPENSSL_EXTRA & SESSION_CERTS) */


#ifdef KEEP_PEER_CERT
    WOLFSSL_ABI
    WOLFSSL_X509* wolfSSL_get_peer_certificate(WOLFSSL* ssl)
    {
        WOLFSSL_X509* ret = NULL;
        WOLFSSL_ENTER("SSL_get_peer_certificate");
        if (ssl != NULL) {
            if (ssl->peerCert.issuer.sz)
                ret = wolfSSL_X509_dup(&ssl->peerCert);
#ifdef SESSION_CERTS
            else if (ssl->session->chain.count > 0) {
                if (DecodeToX509(&ssl->peerCert, ssl->session->chain.certs[0].buffer,
                        ssl->session->chain.certs[0].length) == 0) {
                    ret = wolfSSL_X509_dup(&ssl->peerCert);
                }
            }
#endif
        }
        WOLFSSL_LEAVE("SSL_get_peer_certificate", ret != NULL);
        return ret;
    }

#endif /* KEEP_PEER_CERT */

#if defined(SESSION_CERTS) && defined(OPENSSL_EXTRA)
/* Return stack of peer certs.
 * Caller does not need to free return. The stack is Free'd when WOLFSSL* ssl is.
 */
WOLF_STACK_OF(WOLFSSL_X509)* wolfSSL_get_peer_cert_chain(const WOLFSSL* ssl)
{
    WOLFSSL_ENTER("wolfSSL_get_peer_cert_chain");

    if (ssl == NULL)
        return NULL;

    /* Try to populate if NULL or empty */
    if (ssl->peerCertChain == NULL ||
            wolfSSL_sk_X509_num(ssl->peerCertChain) == 0)
        wolfSSL_set_peer_cert_chain((WOLFSSL*) ssl);
    return ssl->peerCertChain;
}

#ifndef WOLFSSL_QT
static int x509GetIssuerFromCM(WOLFSSL_X509 **issuer, WOLFSSL_CERT_MANAGER* cm,
        WOLFSSL_X509 *x);
/**
 * Recursively push the issuer CA chain onto the stack
 * @param cm The cert manager that is queried for the issuer
 * @param x  This cert's issuer will be queried in cm
 * @param sk The issuer is pushed onto this stack
 * @return WOLFSSL_SUCCESS on success
 *         WOLFSSL_FAILURE on no issuer found
 *         WOLFSSL_FATAL_ERROR on a fatal error
 */
static int PushCAx509Chain(WOLFSSL_CERT_MANAGER* cm,
        WOLFSSL_X509 *x, WOLFSSL_STACK* sk)
{
    WOLFSSL_X509* issuer[MAX_CHAIN_DEPTH];
    int i;
    int push = 1;
    int ret = WOLFSSL_SUCCESS;

    for (i = 0; i < MAX_CHAIN_DEPTH; i++) {
        if (x509GetIssuerFromCM(&issuer[i], cm, x)
                != WOLFSSL_SUCCESS)
            break;
        x = issuer[i];
    }
    if (i == 0) /* No further chain found */
        return WOLFSSL_FAILURE;
    i--;
    for (; i >= 0; i--) {
        if (push) {
            if (wolfSSL_sk_X509_push(sk, issuer[i]) != WOLFSSL_SUCCESS) {
                wolfSSL_X509_free(issuer[i]);
                ret = WOLFSSL_FATAL_ERROR;
                push = 0; /* Free the rest of the unpushed certs */
            }
        }
        else {
            wolfSSL_X509_free(issuer[i]);
        }
    }
    return ret;
}
#endif /* !WOLFSSL_QT */

/* Builds up and creates a stack of peer certificates for ssl->peerCertChain
    based off of the ssl session chain. Attempts to place CA certificates
    at the bottom of the stack. Returns stack of WOLFSSL_X509 certs or
    NULL on failure */
WOLF_STACK_OF(WOLFSSL_X509)* wolfSSL_set_peer_cert_chain(WOLFSSL* ssl)
{
    WOLFSSL_STACK* sk;
    WOLFSSL_X509* x509;
    int i = 0;
    int ret;

    WOLFSSL_ENTER("wolfSSL_set_peer_cert_chain");
    if ((ssl == NULL) || (ssl->session->chain.count == 0))
        return NULL;

    sk = wolfSSL_sk_X509_new();
    i = ssl->session->chain.count-1;
    for (; i >= 0; i--) {
        x509 = wolfSSL_X509_new();
        if (x509 == NULL) {
            WOLFSSL_MSG("Error Creating X509");
            wolfSSL_sk_X509_pop_free(sk, NULL);
            return NULL;
        }
        ret = DecodeToX509(x509, ssl->session->chain.certs[i].buffer,
                             ssl->session->chain.certs[i].length);
#if !defined(WOLFSSL_QT)
        if (ret == 0 && i == ssl->session->chain.count-1) {
            /* On the last element in the chain try to add the CA chain
             * first if we have one for this cert */
            if (PushCAx509Chain(SSL_CM(ssl), x509, sk)
                    == WOLFSSL_FATAL_ERROR) {
                ret = WOLFSSL_FATAL_ERROR;
            }
        }
#endif

        if (ret != 0 || wolfSSL_sk_X509_push(sk, x509) != WOLFSSL_SUCCESS) {
            WOLFSSL_MSG("Error decoding cert");
            wolfSSL_X509_free(x509);
            wolfSSL_sk_X509_pop_free(sk, NULL);
            return NULL;
        }
    }

    if (sk == NULL) {
        WOLFSSL_MSG("Null session chain");
    }
#if defined(OPENSSL_ALL)
    else if (ssl->options.side == WOLFSSL_SERVER_END) {
        /* to be compliant with openssl
           first element is kept as peer cert on server side.*/
        wolfSSL_sk_X509_shift(sk);
    }
#endif
    if (ssl->peerCertChain != NULL)
        wolfSSL_sk_X509_pop_free(ssl->peerCertChain, NULL);
    /* This is Free'd when ssl is Free'd */
    ssl->peerCertChain = sk;
    return sk;
}
#endif /* SESSION_CERTS && OPENSSL_EXTRA */

#ifndef NO_CERTS
#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_WPAS_SMALL)

/* create a generic wolfSSL stack node
 * returns a new WOLFSSL_STACK structure on success */
WOLFSSL_STACK* wolfSSL_sk_new_node(void* heap)
{
    WOLFSSL_STACK* sk;
    WOLFSSL_ENTER("wolfSSL_sk_new_node");

    sk = (WOLFSSL_STACK*)XMALLOC(sizeof(WOLFSSL_STACK), heap,
                                                          DYNAMIC_TYPE_OPENSSL);
    if (sk != NULL) {
        XMEMSET(sk, 0, sizeof(*sk));
        sk->heap = heap;
    }

    return sk;
}

/* free's node but does not free internal data such as in->data.x509 */
void wolfSSL_sk_free_node(WOLFSSL_STACK* in)
{
    if (in != NULL) {
        XFREE(in, in->heap, DYNAMIC_TYPE_OPENSSL);
    }
}

/* pushes node "in" onto "stack" and returns pointer to the new stack on success
 * also handles internal "num" for number of nodes on stack
 * return WOLFSSL_SUCCESS on success
 */
int wolfSSL_sk_push_node(WOLFSSL_STACK** stack, WOLFSSL_STACK* in)
{
    if (stack == NULL || in == NULL) {
        return WOLFSSL_FAILURE;
    }

    if (*stack == NULL) {
        in->num = 1;
        *stack = in;
        return WOLFSSL_SUCCESS;
    }

    in->num  = (*stack)->num + 1;
    in->next = *stack;
    *stack   = in;
    return WOLFSSL_SUCCESS;
}

#if defined(OPENSSL_ALL) || defined(WOLFSSL_QT)
static WC_INLINE int compare_WOLFSSL_CIPHER(
    WOLFSSL_CIPHER *a,
    WOLFSSL_CIPHER *b)
{
    if ((a->cipherSuite0 == b->cipherSuite0) &&
        (a->cipherSuite == b->cipherSuite) &&
        (a->ssl == b->ssl) &&
        (XMEMCMP(a->description, b->description, sizeof a->description) == 0) &&
        (a->offset == b->offset) &&
        (a->in_stack == b->in_stack) &&
        (a->bits == b->bits))
        return 0;
    else
        return -1;
}
#endif /* OPENSSL_ALL || WOLFSSL_QT */


/* return 1 on success 0 on fail */
int wolfSSL_sk_push(WOLFSSL_STACK* sk, const void *data)
{
    WOLFSSL_STACK* node;
#if defined(OPENSSL_ALL) || defined(WOLFSSL_QT)
    WOLFSSL_CIPHER ciph;
#endif
    WOLFSSL_ENTER("wolfSSL_sk_push");

    if (!sk) {
        return WOLFSSL_FAILURE;
    }

    /* Check if empty data */
    switch (sk->type) {
        case STACK_TYPE_CIPHER:
#if defined(OPENSSL_ALL) || defined(WOLFSSL_QT)
            /* check if entire struct is zero */
            XMEMSET(&ciph, 0, sizeof(WOLFSSL_CIPHER));
            if (compare_WOLFSSL_CIPHER(&sk->data.cipher, &ciph) == 0) {
                sk->data.cipher = *(WOLFSSL_CIPHER*)data;
                sk->num = 1;
                if (sk->hash_fn) {
                    sk->hash = sk->hash_fn(&sk->data.cipher);
                }
                return WOLFSSL_SUCCESS;
            }
            break;
#endif
        case STACK_TYPE_X509:
        case STACK_TYPE_GEN_NAME:
        case STACK_TYPE_BIO:
        case STACK_TYPE_OBJ:
        case STACK_TYPE_STRING:
        case STACK_TYPE_ACCESS_DESCRIPTION:
        case STACK_TYPE_X509_EXT:
        case STACK_TYPE_NULL:
        case STACK_TYPE_X509_NAME:
        case STACK_TYPE_X509_NAME_ENTRY:
        case STACK_TYPE_CONF_VALUE:
        case STACK_TYPE_X509_INFO:
        case STACK_TYPE_BY_DIR_entry:
        case STACK_TYPE_BY_DIR_hash:
        case STACK_TYPE_X509_OBJ:
        case STACK_TYPE_DIST_POINT:
        case STACK_TYPE_X509_CRL:
        default:
            /* All other types are pointers */
            if (!sk->data.generic) {
                sk->data.generic = (void*)data;
                sk->num = 1;
#ifdef OPENSSL_ALL
                if (sk->hash_fn) {
                    sk->hash = sk->hash_fn(sk->data.generic);
                }
#endif
                return WOLFSSL_SUCCESS;
            }
            break;
    }

    /* stack already has value(s) create a new node and add more */
    node = wolfSSL_sk_new_node(sk->heap);
    if (!node) {
        WOLFSSL_MSG("Memory error");
        return WOLFSSL_FAILURE;
    }

    /* push new x509 onto head of stack */
    node->next      = sk->next;
    node->type      = sk->type;
    sk->next        = node;
    sk->num        += 1;

#ifdef OPENSSL_ALL
    node->comp = sk->comp;
    node->hash_fn = sk->hash_fn;
    node->hash = sk->hash;
    sk->hash = 0;
#endif
    switch (sk->type) {
        case STACK_TYPE_CIPHER:
#if defined(OPENSSL_ALL) || defined(WOLFSSL_QT)
            node->data.cipher = sk->data.cipher;
            sk->data.cipher = *(WOLFSSL_CIPHER*)data;
            if (sk->hash_fn) {
                sk->hash = sk->hash_fn(&sk->data.cipher);
            }
            break;
#endif
        case STACK_TYPE_X509:
        case STACK_TYPE_GEN_NAME:
        case STACK_TYPE_BIO:
        case STACK_TYPE_OBJ:
        case STACK_TYPE_STRING:
        case STACK_TYPE_ACCESS_DESCRIPTION:
        case STACK_TYPE_X509_EXT:
        case STACK_TYPE_NULL:
        case STACK_TYPE_X509_NAME:
        case STACK_TYPE_X509_NAME_ENTRY:
        case STACK_TYPE_CONF_VALUE:
        case STACK_TYPE_X509_INFO:
        case STACK_TYPE_BY_DIR_entry:
        case STACK_TYPE_BY_DIR_hash:
        case STACK_TYPE_X509_OBJ:
        case STACK_TYPE_DIST_POINT:
        case STACK_TYPE_X509_CRL:
        default:
            /* All other types are pointers */
            node->data.generic = sk->data.generic;
            sk->data.generic = (void*)data;
#ifdef OPENSSL_ALL
            if (sk->hash_fn) {
                sk->hash = sk->hash_fn(sk->data.generic);
            }
#endif
            break;
    }

    return WOLFSSL_SUCCESS;
}

#endif /* OPENSSL_EXTRA || WOLFSSL_WPAS_SMALL */

#ifdef OPENSSL_EXTRA

/* returns the node at index "idx", NULL if not found */
WOLFSSL_STACK* wolfSSL_sk_get_node(WOLFSSL_STACK* sk, int idx)
{
    int i;
    WOLFSSL_STACK* ret = NULL;
    WOLFSSL_STACK* current;

    current = sk;
    for (i = 0; i <= idx && current != NULL; i++) {
        if (i == idx) {
            ret = current;
            break;
        }
        current = current->next;
    }
    return ret;
}


#endif /* OPENSSL_EXTRA */

#ifdef OPENSSL_EXTRA

#if defined(XFPRINTF) && !defined(NO_FILESYSTEM) && \
    !defined(NO_STDIO_FILESYSTEM) && (!defined(NO_RSA) || !defined(NO_DSA) || \
    defined(HAVE_ECC))
/* Print the number bn in hex with name field and indentation indent to file fp.
 * Used by wolfSSL_DSA_print_fp and wolfSSL_RSA_print_fp to print DSA and RSA
 * keys and parameters.
 */
static int PrintBNFieldFp(XFILE fp, int indent, const char* field,
                          const WOLFSSL_BIGNUM* bn) {
    static const int HEX_INDENT = 4;
    static const int MAX_DIGITS_PER_LINE = 30;

    int ret = WOLFSSL_SUCCESS;
    int i = 0;
    char* buf = NULL;

    if (fp == XBADFILE || indent < 0 || field == NULL || bn == NULL) {
        ret = BAD_FUNC_ARG;
    }

    if (ret == WOLFSSL_SUCCESS) {
        buf = wolfSSL_BN_bn2hex(bn);
        if (buf == NULL) {
            ret = WOLFSSL_FAILURE;
        }
    }
    if (ret == WOLFSSL_SUCCESS) {
        XFPRINTF(fp, "%*s", indent, "");
        XFPRINTF(fp, "%s:\n", field);
        XFPRINTF(fp, "%*s", indent + HEX_INDENT, "");
        while (buf[i]) {
            if (i != 0) {
                if (i % 2 == 0) {
                    XFPRINTF(fp, ":");
                }
                if (i % MAX_DIGITS_PER_LINE == 0) {
                    XFPRINTF(fp, "\n");
                    XFPRINTF(fp, "%*s", indent + HEX_INDENT, "");
                }
            }
            XFPRINTF(fp, "%c", buf[i++]);
        }
        XFPRINTF(fp, "\n");
    }

    if (buf != NULL) {
        XFREE(buf, NULL, DYNAMIC_TYPE_OPENSSL);
    }

    return ret;
}
#endif /* XFPRINTF && !NO_FILESYSTEM && !NO_STDIO_FILESYSTEM && (!NO_DSA ||
          !NO_RSA || HAVE_ECC)*/

#if defined(HAVE_ECC) && defined(XFPRINTF) && !defined(NO_FILESYSTEM) && \
    !defined(NO_STDIO_FILESYSTEM)
int wolfSSL_EC_KEY_print_fp(XFILE fp, WOLFSSL_EC_KEY* key, int indent)
{
    int ret = WOLFSSL_SUCCESS;
    int bits = 0;
    int priv = 0;
    int nid = 0;
    const char* curve;
    const char* nistName;
    WOLFSSL_BIGNUM* pubBn = NULL;

    WOLFSSL_ENTER("wolfSSL_EC_KEY_print_fp");

    if (fp == XBADFILE || key == NULL || key->group == NULL || indent < 0) {
        ret = WOLFSSL_FAILURE;
    }

    if (ret == WOLFSSL_SUCCESS) {
        bits = wolfSSL_EC_GROUP_order_bits(key->group);
        if (bits <= 0) {
            WOLFSSL_MSG("Failed to get group order bits.");
            ret = WOLFSSL_FAILURE;
        }
    }
    if (ret == WOLFSSL_SUCCESS) {
        XFPRINTF(fp, "%*s", indent, "");
        if (key->priv_key != NULL && !wolfSSL_BN_is_zero(key->priv_key)) {
            XFPRINTF(fp, "Private-Key: (%d bit)\n", bits);
            priv = 1;
        }
        else {
            XFPRINTF(fp, "Public-Key: (%d bit)\n", bits);
        }

        if (priv) {
            ret = PrintBNFieldFp(fp, indent, "priv", key->priv_key);
        }
    }
    if (ret == WOLFSSL_SUCCESS && key->pub_key != NULL && key->pub_key->exSet) {
        pubBn = wolfSSL_EC_POINT_point2bn(key->group, key->pub_key,
                                          POINT_CONVERSION_UNCOMPRESSED, NULL,
                                          NULL);
        if (pubBn == NULL) {
            WOLFSSL_MSG("wolfSSL_EC_POINT_point2bn failed.");
            ret = WOLFSSL_FAILURE;
        }
        else {
            ret = PrintBNFieldFp(fp, indent, "pub", pubBn);
        }
    }
    if (ret == WOLFSSL_SUCCESS) {
        nid = wolfSSL_EC_GROUP_get_curve_name(key->group);
        if (nid > 0) {
            curve = wolfSSL_OBJ_nid2ln(nid);
            if (curve != NULL) {
                XFPRINTF(fp, "%*s", indent, "");
                XFPRINTF(fp, "ASN1 OID: %s\n", curve);
            }
            nistName = wolfSSL_EC_curve_nid2nist(nid);
            if (nistName != NULL) {
                XFPRINTF(fp, "%*s", indent, "");
                XFPRINTF(fp, "NIST CURVE: %s\n", nistName);
            }
        }
    }

    if (pubBn != NULL) {
        wolfSSL_BN_free(pubBn);
    }

    WOLFSSL_LEAVE("wolfSSL_EC_KEY_print_fp", ret);

    return ret;
}
#endif /* HAVE_ECC && XFPRINTF && !NO_FILESYSTEM && !NO_STDIO_FILESYSTEM */

#if defined(OPENSSL_ALL)

void *wolfSSL_lh_retrieve(WOLFSSL_STACK *sk, void *data)
{
    unsigned long hash;

    WOLFSSL_ENTER("wolfSSL_lh_retrieve");

    if (!sk || !data) {
        WOLFSSL_MSG("Bad parameters");
        return NULL;
    }

    if (!sk->hash_fn) {
        WOLFSSL_MSG("No hash function defined");
        return NULL;
    }

    hash = sk->hash_fn(data);

    while (sk) {
        /* Calc hash if not done so yet */
        if (!sk->hash) {
            switch (sk->type) {
                case STACK_TYPE_CIPHER:
                    sk->hash = sk->hash_fn(&sk->data.cipher);
                    break;
                case STACK_TYPE_X509:
                case STACK_TYPE_GEN_NAME:
                case STACK_TYPE_BIO:
                case STACK_TYPE_OBJ:
                case STACK_TYPE_STRING:
                case STACK_TYPE_ACCESS_DESCRIPTION:
                case STACK_TYPE_X509_EXT:
                case STACK_TYPE_NULL:
                case STACK_TYPE_X509_NAME:
                case STACK_TYPE_X509_NAME_ENTRY:
                case STACK_TYPE_CONF_VALUE:
                case STACK_TYPE_X509_INFO:
                case STACK_TYPE_BY_DIR_entry:
                case STACK_TYPE_BY_DIR_hash:
                case STACK_TYPE_X509_OBJ:
                case STACK_TYPE_DIST_POINT:
                case STACK_TYPE_X509_CRL:
                default:
                    sk->hash = sk->hash_fn(sk->data.generic);
                    break;
            }
        }
        if (sk->hash == hash) {
            switch (sk->type) {
                case STACK_TYPE_CIPHER:
                    return &sk->data.cipher;
                case STACK_TYPE_X509:
                case STACK_TYPE_GEN_NAME:
                case STACK_TYPE_BIO:
                case STACK_TYPE_OBJ:
                case STACK_TYPE_STRING:
                case STACK_TYPE_ACCESS_DESCRIPTION:
                case STACK_TYPE_X509_EXT:
                case STACK_TYPE_NULL:
                case STACK_TYPE_X509_NAME:
                case STACK_TYPE_X509_NAME_ENTRY:
                case STACK_TYPE_CONF_VALUE:
                case STACK_TYPE_X509_INFO:
                case STACK_TYPE_BY_DIR_entry:
                case STACK_TYPE_BY_DIR_hash:
                case STACK_TYPE_X509_OBJ:
                case STACK_TYPE_DIST_POINT:
                case STACK_TYPE_X509_CRL:
                default:
                    return sk->data.generic;
            }
        }
        sk = sk->next;
    }

    return NULL;
}

#if defined(HAVE_ECC)
/* Copies ecc_key into new WOLFSSL_EC_KEY object
 *
 * src  : EC_KEY to duplicate. If EC_KEY is not null, create new EC_KEY and copy
 * internal ecc_key from src to dup.
 *
 * Returns pointer to duplicate EC_KEY.
 */
WOLFSSL_EC_KEY *wolfSSL_EC_KEY_dup(const WOLFSSL_EC_KEY *src)
{
    WOLFSSL_EC_KEY *newKey;
    ecc_key *key, *srcKey;
    int ret;

    WOLFSSL_ENTER("wolfSSL_EC_KEY_dup");

    if (src == NULL || src->internal == NULL || src->group == NULL || \
       src->pub_key == NULL || src->priv_key == NULL) {

        WOLFSSL_MSG("src NULL error");
        return NULL;
    }

    newKey = wolfSSL_EC_KEY_new();
    if (newKey == NULL) {
        WOLFSSL_MSG("wolfSSL_EC_KEY_new error");
        return NULL;
    }

    key = (ecc_key*)newKey->internal;
    if (key == NULL) {
        WOLFSSL_MSG("ecc_key NULL error");
        wolfSSL_EC_KEY_free(newKey);
        return NULL;
    }
    srcKey = (ecc_key*)src->internal;

    /* ecc_key */
    /* copy pubkey */
    ret = wc_ecc_copy_point(&srcKey->pubkey, &key->pubkey);
    if (ret != MP_OKAY) {
        WOLFSSL_MSG("wc_ecc_copy_point error");
        wolfSSL_EC_KEY_free(newKey);
        return NULL;
    }

    /* copy private key k */
    ret = mp_copy(&srcKey->k, &key->k);
    if (ret != MP_OKAY) {
        WOLFSSL_MSG("mp_copy error");
        wolfSSL_EC_KEY_free(newKey);
        return NULL;
    }

    /* copy domain parameters */
    if (srcKey->dp) {
        ret = wc_ecc_set_curve(key, 0, srcKey->dp->id);
        if (ret != 0) {
            WOLFSSL_MSG("wc_ecc_set_curve error");
            return NULL;
        }
    }

    key->type  = srcKey->type;
    key->idx   = srcKey->idx;
    key->state = srcKey->state;
    key->flags = srcKey->flags;

    /* Copy group */
    if (newKey->group == NULL) {
        WOLFSSL_MSG("EC_GROUP_new_by_curve_name error");
        wolfSSL_EC_KEY_free(newKey);
        return NULL;
    }

    newKey->group->curve_idx = src->group->curve_idx;
    newKey->group->curve_nid = src->group->curve_nid;
    newKey->group->curve_oid = src->group->curve_oid;

    /* Copy public key */
    if (src->pub_key->internal == NULL || newKey->pub_key->internal == NULL) {
        WOLFSSL_MSG("NULL pub_key error");
        wolfSSL_EC_KEY_free(newKey);
        return NULL;
    }

    /* Copy public key internal */
    ret = wc_ecc_copy_point((ecc_point*)src->pub_key->internal, \
                            (ecc_point*)newKey->pub_key->internal);
    if (ret != MP_OKAY) {
        WOLFSSL_MSG("ecc_copy_point error");
        wolfSSL_EC_KEY_free(newKey);
        return NULL;
    }

    /* Copy X, Y, Z */
    newKey->pub_key->X = wolfSSL_BN_dup(src->pub_key->X);
    if (!newKey->pub_key->X && src->pub_key->X) {
        WOLFSSL_MSG("Error copying EC_POINT");
        wolfSSL_EC_KEY_free(newKey);
        return NULL;
    }
    newKey->pub_key->Y = wolfSSL_BN_dup(src->pub_key->Y);
    if (!newKey->pub_key->Y && src->pub_key->Y) {
        WOLFSSL_MSG("Error copying EC_POINT");
        wolfSSL_EC_KEY_free(newKey);
        return NULL;
    }
    newKey->pub_key->Z = wolfSSL_BN_dup(src->pub_key->Z);
    if (!newKey->pub_key->Z && src->pub_key->Z) {
        WOLFSSL_MSG("Error copying EC_POINT");
        wolfSSL_EC_KEY_free(newKey);
        return NULL;
    }

    newKey->pub_key->inSet = src->pub_key->inSet;
    newKey->pub_key->exSet = src->pub_key->exSet;
    newKey->pkcs8HeaderSz = src->pkcs8HeaderSz;

    /* Copy private key */
    if (src->priv_key->internal == NULL || newKey->priv_key->internal == NULL) {
        WOLFSSL_MSG("NULL priv_key error");
        wolfSSL_EC_KEY_free(newKey);
        return NULL;
    }

    /* Free priv_key before call to newKey function */
    wolfSSL_BN_free(newKey->priv_key);
    newKey->priv_key = wolfSSL_BN_dup(src->priv_key);
    if (newKey->priv_key == NULL) {
        WOLFSSL_MSG("BN_newKey error");
        wolfSSL_EC_KEY_free(newKey);
        return NULL;
    }

    return newKey;

}
#endif /* HAVE_ECC */

#if !defined(NO_DH)
int wolfSSL_DH_check(const WOLFSSL_DH *dh, int *codes)
{
    int isPrime = MP_NO, codeTmp = 0;
    WC_RNG rng;

    WOLFSSL_ENTER("wolfSSL_DH_check");
    if (dh == NULL){
        return WOLFSSL_FAILURE;
    }

    if (dh->g == NULL || dh->g->internal == NULL){
        codeTmp = DH_NOT_SUITABLE_GENERATOR;
    }

    if (dh->p == NULL || dh->p->internal == NULL){
        codeTmp = DH_CHECK_P_NOT_PRIME;
    }
    else
    {
        /* test if dh->p has prime */
        if (wc_InitRng(&rng) == 0){
            mp_prime_is_prime_ex((mp_int*)dh->p->internal,8,&isPrime,&rng);
        }
        else {
            WOLFSSL_MSG("Error initializing rng");
            return WOLFSSL_FAILURE;
        }
        wc_FreeRng(&rng);
        if (isPrime != MP_YES){
            codeTmp = DH_CHECK_P_NOT_PRIME;
        }
    }
    /* User may choose to enter NULL for codes if they don't want to check it*/
    if (codes != NULL){
        *codes = codeTmp;
    }

    /* if codeTmp was set,some check was flagged invalid */
    if (codeTmp){
        return WOLFSSL_FAILURE;
    }

    return WOLFSSL_SUCCESS;
}

#if !defined(HAVE_FIPS) || (defined(HAVE_FIPS_VERSION) && (HAVE_FIPS_VERSION>2))
/* Converts DER encoded DH parameters to a WOLFSSL_DH structure.
 *
 * dh   : structure to copy DH parameters into.
 * pp   : DER encoded DH parameters
 * length   : length to copy
 *
 * Returns pointer to WOLFSSL_DH structure on success, or NULL on failure
 */
WOLFSSL_DH *wolfSSL_d2i_DHparams(WOLFSSL_DH **dh, const unsigned char **pp,
                                                                    long length)
{
    WOLFSSL_DH *newDH = NULL;
    int ret;
    word32 idx = 0;

    WOLFSSL_ENTER("wolfSSL_d2i_DHparams");

    if (pp == NULL || length <= 0) {
        WOLFSSL_MSG("bad argument");
        return NULL;
    }

    if ((newDH = wolfSSL_DH_new()) == NULL) {
        WOLFSSL_MSG("wolfSSL_DH_new() failed");
        return NULL;
    }

    ret = wc_DhKeyDecode(*pp, &idx, (DhKey*)newDH->internal, (word32)length);
    if (ret != 0) {
        WOLFSSL_MSG("DhKeyDecode() failed");
        wolfSSL_DH_free(newDH);
        return NULL;
    }
    newDH->inSet = 1;

    if (SetDhExternal(newDH) != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("SetDhExternal failed");
        wolfSSL_DH_free(newDH);
        return NULL;
    }

    *pp += length;
    if (dh != NULL){
        *dh = newDH;
    }

    return newDH;
}
#endif /* !HAVE_FIPS || HAVE_FIPS_VERSION > 2 */

#define ASN_LEN_SIZE(l)             \
    (((l) < 128) ? 1 : (((l) < 256) ? 2 : 3))

/* Converts internal WOLFSSL_DH structure to DER encoded DH.
 *
 * dh   : structure to copy DH parameters from.
 * out  : DER buffer for DH parameters
 *
 * Returns size of DER on success and WOLFSSL_FAILURE if error
 */
int wolfSSL_i2d_DHparams(const WOLFSSL_DH *dh, unsigned char **out)
{
    word32 len;
    int ret = 0;
    int pSz;
    int gSz;

    WOLFSSL_ENTER("wolfSSL_i2d_DHparams");

    if (dh == NULL) {
        WOLFSSL_MSG("Bad parameters");
        return WOLFSSL_FAILURE;
    }

    /* Get total length */
    pSz = mp_unsigned_bin_size((mp_int*)dh->p->internal);
    gSz = mp_unsigned_bin_size((mp_int*)dh->g->internal);
    len = 1 + ASN_LEN_SIZE(pSz) + mp_leading_bit((mp_int*)dh->p->internal) +
          pSz +
          1 + ASN_LEN_SIZE(gSz) + mp_leading_bit((mp_int*)dh->g->internal) +
          gSz;

    /* Two bytes required for length if ASN.1 SEQ data greater than 127 bytes
     * and less than 256 bytes.
     */
    len += 1 + ASN_LEN_SIZE(len);

    if (out != NULL && *out != NULL) {
        ret = StoreDHparams(*out, &len, (mp_int*)dh->p->internal,
                                        (mp_int*)dh->g->internal);
        if (ret != MP_OKAY) {
            WOLFSSL_MSG("StoreDHparams error");
            len = 0;
        }
        else{
            *out += len;
        }
    }
    return (int)len;
}
#endif /* !NO_DH */

#endif /* OPENSSL_ALL */

#endif /* OPENSSL_EXTRA */

/* OPENSSL_EXTRA is needed for wolfSSL_X509_d21 function
   KEEP_OUR_CERT is to insure ability for returning ssl certificate */
#if (defined(OPENSSL_EXTRA) || defined(OPENSSL_EXTRA_X509_SMALL)) && \
    defined(KEEP_OUR_CERT)
WOLFSSL_X509* wolfSSL_get_certificate(WOLFSSL* ssl)
{
    if (ssl == NULL) {
        return NULL;
    }

    if (ssl->buffers.weOwnCert) {
        if (ssl->ourCert == NULL) {
            if (ssl->buffers.certificate == NULL) {
                WOLFSSL_MSG("Certificate buffer not set!");
                return NULL;
            }
            #ifndef WOLFSSL_X509_STORE_CERTS
            ssl->ourCert = wolfSSL_X509_d2i(NULL,
                                              ssl->buffers.certificate->buffer,
                                              ssl->buffers.certificate->length);
            #endif
        }
        return ssl->ourCert;
    }
    else { /* if cert not owned get parent ctx cert or return null */
        if (ssl->ctx) {
            if (ssl->ctx->ourCert == NULL) {
                if (ssl->ctx->certificate == NULL) {
                    WOLFSSL_MSG("Ctx Certificate buffer not set!");
                    return NULL;
                }
                #ifndef WOLFSSL_X509_STORE_CERTS
                ssl->ctx->ourCert = wolfSSL_X509_d2i(NULL,
                                               ssl->ctx->certificate->buffer,
                                               ssl->ctx->certificate->length);
                #endif
                ssl->ctx->ownOurCert = 1;
            }
            return ssl->ctx->ourCert;
        }
    }

    return NULL;
}

WOLFSSL_X509* wolfSSL_CTX_get0_certificate(WOLFSSL_CTX* ctx)
{
    if (ctx) {
        if (ctx->ourCert == NULL) {
            if (ctx->certificate == NULL) {
                WOLFSSL_MSG("Ctx Certificate buffer not set!");
                return NULL;
            }
            #ifndef WOLFSSL_X509_STORE_CERTS
            ctx->ourCert = wolfSSL_X509_d2i(NULL,
                                           ctx->certificate->buffer,
                                           ctx->certificate->length);
            #endif
            ctx->ownOurCert = 1;
        }
        return ctx->ourCert;
    }
    return NULL;
}
#endif /* OPENSSL_EXTRA && KEEP_OUR_CERT */
#endif /* NO_CERTS */


#if !defined(NO_ASN) && (defined(OPENSSL_EXTRA) || \
        defined(OPENSSL_EXTRA_X509_SMALL))
void wolfSSL_ASN1_OBJECT_free(WOLFSSL_ASN1_OBJECT* obj)
{
    if (obj == NULL) {
        return;
    }
    if ((obj->obj != NULL) && ((obj->dynamic & WOLFSSL_ASN1_DYNAMIC_DATA) != 0)) {
#ifdef WOLFSSL_DEBUG_OPENSSL
        WOLFSSL_MSG("Freeing ASN1 data");
#endif
        XFREE((void*)obj->obj, obj->heap, DYNAMIC_TYPE_ASN1);
        obj->obj = NULL;
    }
    #if defined(WOLFSSL_QT) || defined(OPENSSL_ALL)
    if (obj->pathlen != NULL) {
        wolfSSL_ASN1_INTEGER_free(obj->pathlen);
        obj->pathlen = NULL;
    }
    #endif
    if ((obj->dynamic & WOLFSSL_ASN1_DYNAMIC) != 0) {
#ifdef WOLFSSL_DEBUG_OPENSSL
        WOLFSSL_MSG("Freeing ASN1 OBJECT");
#endif
        XFREE(obj, NULL, DYNAMIC_TYPE_ASN1);
    }
}

WOLFSSL_ASN1_OBJECT* wolfSSL_ASN1_OBJECT_new(void)
{
    WOLFSSL_ASN1_OBJECT* obj;

    obj = (WOLFSSL_ASN1_OBJECT*)XMALLOC(sizeof(WOLFSSL_ASN1_OBJECT), NULL,
                                        DYNAMIC_TYPE_ASN1);
    if (obj == NULL) {
        return NULL;
    }

    XMEMSET(obj, 0, sizeof(WOLFSSL_ASN1_OBJECT));
    obj->d.ia5 = &(obj->d.ia5_internal);
#if defined(WOLFSSL_QT) || defined(OPENSSL_ALL)
    obj->d.iPAddress = &(obj->d.iPAddress_internal);
#endif
    obj->dynamic |= WOLFSSL_ASN1_DYNAMIC;
    return obj;
}

WOLFSSL_ASN1_OBJECT* wolfSSL_ASN1_OBJECT_dup(WOLFSSL_ASN1_OBJECT* obj)
{
    WOLFSSL_ASN1_OBJECT* dupl = NULL;

    WOLFSSL_ENTER("wolfSSL_ASN1_OBJECT_dup");

    if (!obj) {
        WOLFSSL_MSG("Bad parameter");
        return NULL;
    }
    dupl = wolfSSL_ASN1_OBJECT_new();
    if (!dupl) {
        WOLFSSL_MSG("wolfSSL_ASN1_OBJECT_new error");
        return NULL;
    }
    /* Copy data */
    XMEMCPY(dupl->sName, obj->sName, WOLFSSL_MAX_SNAME);
    dupl->type = obj->type;
    dupl->grp = obj->grp;
    dupl->nid = obj->nid;
    dupl->objSz = obj->objSz;
    if (obj->obj) {
        dupl->obj = (const unsigned char*)XMALLOC(
                obj->objSz, NULL, DYNAMIC_TYPE_ASN1);
        if (!dupl->obj) {
            WOLFSSL_MSG("ASN1 obj malloc error");
            wolfSSL_ASN1_OBJECT_free(dupl);
            return NULL;
        }
        XMEMCPY((byte*)dupl->obj, obj->obj, obj->objSz);
        dupl->dynamic |= WOLFSSL_ASN1_DYNAMIC_DATA;
    }
    return dupl;
}
#endif /* !NO_ASN && (OPENSSL_EXTRA || OPENSSL_EXTRA_X509_SMALL) */

#ifndef NO_ASN
#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_WPAS_SMALL)
/* Creates and returns a new WOLFSSL_CIPHER stack. */
WOLFSSL_STACK* wolfSSL_sk_new_asn1_obj(void)
{
    WOLFSSL_STACK* sk;
    WOLFSSL_ENTER("wolfSSL_sk_new_asn1_obj");

    sk = wolfSSL_sk_new_null();
    if (sk == NULL)
        return NULL;
    sk->type = STACK_TYPE_OBJ;

    return sk;
}

/* return 1 on success 0 on fail */
int wolfSSL_sk_ASN1_OBJECT_push(WOLF_STACK_OF(WOLFSSL_ASN1_OBJECT)* sk,
                                              WOLFSSL_ASN1_OBJECT* obj)
{
    WOLFSSL_ENTER("wolfSSL_sk_ASN1_OBJECT_push");

    if (sk == NULL || obj == NULL) {
        return WOLFSSL_FAILURE;
    }

    return wolfSSL_sk_push(sk, obj);
}


WOLFSSL_ASN1_OBJECT* wolfSSL_sk_ASN1_OBJECT_pop(
                                        WOLF_STACK_OF(WOLFSSL_ASN1_OBJECT)* sk)
{
    WOLFSSL_STACK* node;
    WOLFSSL_ASN1_OBJECT* obj;

    if (sk == NULL) {
        return NULL;
    }

    node = sk->next;
    obj = sk->data.obj;

    if (node != NULL) { /* update sk and remove node from stack */
        sk->data.obj = node->data.obj;
        sk->next = node->next;
        XFREE(node, NULL, DYNAMIC_TYPE_ASN1);
    }
    else { /* last obj in stack */
        sk->data.obj = NULL;
    }

    if (sk->num > 0) {
        sk->num -= 1;
    }

    return obj;
}


/* Free the structure for ASN1_OBJECT stack
 *
 * sk  stack to free nodes in
 */
void wolfSSL_sk_ASN1_OBJECT_free(WOLF_STACK_OF(WOLFSSL_ASN1_OBJECT)* sk)
{
    wolfSSL_sk_free(sk);
}

/* Free's all nodes in ASN1_OBJECT stack.
 * This is different then wolfSSL_ASN1_OBJECT_free in that it allows for
 * choosing the function to use when freeing an ASN1_OBJECT stack.
 *
 * sk  stack to free nodes in
 * f   X509 free function
 */
void wolfSSL_sk_ASN1_OBJECT_pop_free(WOLF_STACK_OF(WOLFSSL_ASN1_OBJECT)* sk,
                                     void (*f) (WOLFSSL_ASN1_OBJECT*))
{
    WOLFSSL_ENTER("wolfSSL_sk_ASN1_OBJECT_pop_free");
    wolfSSL_sk_pop_free(sk, (wolfSSL_sk_freefunc)f);
}

#endif /* OPENSSL_EXTRA || WOLFSSL_WPAS_SMALL */
#endif /* !NO_ASN */

#if defined(OPENSSL_EXTRA) || defined(OPENSSL_EXTRA_X509_SMALL)
#ifndef NO_ASN

int wolfSSL_ASN1_STRING_to_UTF8(unsigned char **out, WOLFSSL_ASN1_STRING *in)
{
    /*
       ASN1_STRING_to_UTF8() converts the string in to UTF8 format,
       the converted data is allocated in a buffer in *out.
       The length of out is returned or a negative error code.
       The buffer *out should be free using OPENSSL_free().
       */
    unsigned char* buf;
    unsigned char* inPtr;
    int inLen;

    if (!out || !in) {
        return -1;
    }

    inPtr = wolfSSL_ASN1_STRING_data(in);
    inLen = wolfSSL_ASN1_STRING_length(in);
    if (!inPtr || inLen < 0) {
        return -1;
    }
    buf = (unsigned char*)XMALLOC(inLen + 1, NULL, DYNAMIC_TYPE_OPENSSL);
    if (!buf) {
        return -1;
    }
    XMEMCPY(buf, inPtr, inLen + 1);
    *out = buf;
    return inLen;
}
#endif /* !NO_ASN */
#endif /* OPENSSL_EXTRA || OPENSSL_EXTRA_X509_SMALL */

#if defined(OPENSSL_EXTRA)
#ifndef NO_ASN

int wolfSSL_ASN1_UNIVERSALSTRING_to_string(WOLFSSL_ASN1_STRING *s)
{
    char *idx;
    char *copy;
    WOLFSSL_ENTER("wolfSSL_ASN1_UNIVERSALSTRING_to_string");

    if (!s) {
        WOLFSSL_MSG("Bad parameter");
        return WOLFSSL_FAILURE;
    }

    if (s->type != V_ASN1_UNIVERSALSTRING) {
        WOLFSSL_MSG("Input is not a universal string");
        return WOLFSSL_FAILURE;
    }

    if ((s->length % 4) != 0) {
        WOLFSSL_MSG("Input string must be divisible by 4");
        return WOLFSSL_FAILURE;
    }

    for (idx = s->data; idx < s->data + s->length; idx += 4)
        if ((idx[0] != '\0') || (idx[1] != '\0') || (idx[2] != '\0'))
            break;

    if (idx != s->data + s->length) {
        WOLFSSL_MSG("Wrong string format");
        return WOLFSSL_FAILURE;
    }

    for (copy = idx = s->data; idx < s->data + s->length; idx += 4)
        *copy++ = idx[3];
    *copy = '\0';
    s->length /= 4;
    s->type = V_ASN1_PRINTABLESTRING;
    return WOLFSSL_SUCCESS;
}

/* Returns string representation of ASN1_STRING */
char* wolfSSL_i2s_ASN1_STRING(WOLFSSL_v3_ext_method *method,
    const WOLFSSL_ASN1_STRING *s)
{
    int i;
    int tmpSz = 100;
    int valSz = 5;
    char* tmp;
    char val[5];
    unsigned char* str;

    WOLFSSL_ENTER("wolfSSL_i2s_ASN1_STRING");
    (void)method;

    if(s == NULL || s->data == NULL) {
        WOLFSSL_MSG("Bad Function Argument");
        return NULL;
    }
    str = (unsigned char*)XMALLOC(s->length, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    if (str == NULL) {
        WOLFSSL_MSG("Memory Error");
        return NULL;
    }
    XMEMCPY(str, (unsigned char*)s->data, s->length);

    tmp = (char*)XMALLOC(tmpSz, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    if (tmp == NULL) {
        WOLFSSL_MSG("Memory Error");
        XFREE(str, NULL, DYNAMIC_TYPE_TMP_BUFFER);
        return NULL;
    }
    XMEMSET(tmp, 0, tmpSz);

    for (i = 0; i < tmpSz && i < (s->length - 1); i++) {
        XSNPRINTF(val, valSz - 1, "%02X:", str[i]);
        XSTRNCAT(tmp, val, valSz);
    }
    XSNPRINTF(val, valSz - 1, "%02X", str[i]);
    XSTRNCAT(tmp, val, valSz);
    XFREE(str, NULL, DYNAMIC_TYPE_TMP_BUFFER);

    return tmp;
}
#endif /* NO_ASN */
#endif /* OPENSSL_EXTRA */

#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_WPAS_SMALL)
void wolfSSL_set_connect_state(WOLFSSL* ssl)
{
    WOLFSSL_ENTER("wolfSSL_set_connect_state");
    if (ssl == NULL) {
        WOLFSSL_MSG("WOLFSSL struct pointer passed in was null");
        return;
    }

    #ifndef NO_DH
    /* client creates its own DH parameters on handshake */
    if (ssl->buffers.serverDH_P.buffer && ssl->buffers.weOwnDH) {
        XFREE(ssl->buffers.serverDH_P.buffer, ssl->heap,
            DYNAMIC_TYPE_PUBLIC_KEY);
    }
    ssl->buffers.serverDH_P.buffer = NULL;
    if (ssl->buffers.serverDH_G.buffer && ssl->buffers.weOwnDH) {
        XFREE(ssl->buffers.serverDH_G.buffer, ssl->heap,
            DYNAMIC_TYPE_PUBLIC_KEY);
    }
    ssl->buffers.serverDH_G.buffer = NULL;
    #endif

    if (InitSSL_Side(ssl, WOLFSSL_CLIENT_END) != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("Error initializing client side");
    }
}
#endif /* OPENSSL_EXTRA || WOLFSSL_WPAS_SMALL */


int wolfSSL_get_shutdown(const WOLFSSL* ssl)
{
    int isShutdown = 0;

    WOLFSSL_ENTER("wolfSSL_get_shutdown");

    if (ssl) {
#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_WPAS_SMALL)
        if (ssl->options.handShakeState == NULL_STATE) {
            /* The SSL object was possibly cleared with wolfSSL_clear after
             * a successful shutdown. Simulate a response for a full
             * bidirectional shutdown. */
            isShutdown = WOLFSSL_SENT_SHUTDOWN | WOLFSSL_RECEIVED_SHUTDOWN;
        }
        else
#endif
        {
            /* in OpenSSL, WOLFSSL_SENT_SHUTDOWN = 1, when closeNotifySent   *
             * WOLFSSL_RECEIVED_SHUTDOWN = 2, from close notify or fatal err */
            if (ssl->options.sentNotify)
                isShutdown |= WOLFSSL_SENT_SHUTDOWN;
            if (ssl->options.closeNotify||ssl->options.connReset)
                isShutdown |= WOLFSSL_RECEIVED_SHUTDOWN;
        }

    }
    return isShutdown;
}


int wolfSSL_session_reused(WOLFSSL* ssl)
{
    int resuming = 0;
    WOLFSSL_ENTER("wolfSSL_session_reused");
    if (ssl)
        resuming = ssl->options.resuming;
    WOLFSSL_LEAVE("wolfSSL_session_reused", resuming);
    return resuming;
}

/* return a new malloc'd session with default settings on success */
WOLFSSL_SESSION* wolfSSL_NewSession(void* heap)
{
    WOLFSSL_SESSION* ret = NULL;

    ret = (WOLFSSL_SESSION*)XMALLOC(sizeof(WOLFSSL_SESSION), heap,
            DYNAMIC_TYPE_SESSION);
    if (ret != NULL) {
        XMEMSET(ret, 0, sizeof(WOLFSSL_SESSION));
    #ifndef SINGLE_THREADED
        if (wc_InitMutex(&ret->refMutex) != 0) {
            WOLFSSL_MSG("Error setting up session reference mutex");
            XFREE(ret, ret->heap, DYNAMIC_TYPE_SESSION);
            return NULL;
        }
    #endif
        ret->refCount = 1;
#ifndef NO_SESSION_CACHE
        ret->cacheRow = INVALID_SESSION_ROW; /* not in cache */
#endif
        ret->type = WOLFSSL_SESSION_TYPE_HEAP;
        ret->heap = heap;
        ret->masterSecret = ret->_masterSecret;
    #ifndef NO_CLIENT_CACHE
        ret->serverID = ret->_serverID;
    #endif
    #ifdef OPENSSL_EXTRA
        ret->sessionCtx = ret->_sessionCtx;
    #endif
    #ifdef HAVE_SESSION_TICKET
        ret->ticket = ret->_staticTicket;
    #endif
#ifdef HAVE_STUNNEL
        /* stunnel has this funny mechanism of storing the "is_authenticated"
         * session info in the session ex data. This is basically their
         * default so let's just hard code it. */
        if (wolfSSL_SESSION_set_ex_data(ret, 0, (void *)(-1))
                != WOLFSSL_SUCCESS) {
            WOLFSSL_MSG("Error setting up ex data for stunnel");
            XFREE(ret, NULL, DYNAMIC_TYPE_OPENSSL);
            return NULL;
        }
#endif
    }
    return ret;
}


WOLFSSL_SESSION* wolfSSL_SESSION_new_ex(void* heap)
{
    return wolfSSL_NewSession(heap);
}

WOLFSSL_SESSION* wolfSSL_SESSION_new(void)
{
    return wolfSSL_SESSION_new_ex(NULL);
}

/* add one to session reference count
 * return WOLFSSL_SUCCESS on success and WOLFSSL_FAILURE on error */
int wolfSSL_SESSION_up_ref(WOLFSSL_SESSION* session)
{
    session = ClientSessionToSession(session);

    if (session == NULL || session->type != WOLFSSL_SESSION_TYPE_HEAP)
        return WOLFSSL_FAILURE;

#ifndef SINGLE_THREADED
    if (wc_LockMutex(&session->refMutex) != 0) {
        WOLFSSL_MSG("Failed to lock session mutex");
        return WOLFSSL_FAILURE;
    }
#endif
    session->refCount++;
#ifndef SINGLE_THREADED
    wc_UnLockMutex(&session->refMutex);
#endif
    return WOLFSSL_SUCCESS;
}

/**
 * Deep copy the contents from input to output.
 * @param input         The source of the copy.
 * @param output        The destination of the copy.
 * @param avoidSysCalls If true, then system calls will be avoided or an error
 *                      will be returned if it is not possible to proceed
 *                      without a system call. This is useful for fetching
 *                      sessions from cache. When a cache row is locked, we
 *                      don't want to block other threads with long running
 *                      system calls.
 * @return              WOLFSSL_SUCCESS on success
 *                      WOLFSSL_FAILURE on failure
 */
int wolfSSL_DupSession(const WOLFSSL_SESSION* input, WOLFSSL_SESSION* output,
        int avoidSysCalls)
{
#ifdef HAVE_SESSION_TICKET
    int   ticLenAlloc = 0;
    byte *ticBuff = NULL;
#endif
    const size_t copyOffset = OFFSETOF(WOLFSSL_SESSION, heap) + sizeof(input->heap);
    int ret = WOLFSSL_SUCCESS;

    (void)avoidSysCalls;

    input = ClientSessionToSession(input);
    output = ClientSessionToSession(output);

    if (input == NULL || output == NULL || input == output) {
        WOLFSSL_MSG("input or output are null or same");
        return WOLFSSL_FAILURE;
    }

#ifdef HAVE_SESSION_TICKET
    if (output->ticket != output->_staticTicket) {
        ticBuff = output->ticket;
        ticLenAlloc = output->ticketLenAlloc;
    }
#endif

#if defined(SESSION_CERTS) && defined(OPENSSL_EXTRA)
    if (output->peer != NULL) {
        if (avoidSysCalls) {
            WOLFSSL_MSG("Can't free cert when avoiding syscalls");
            return WOLFSSL_FAILURE;
        }
        wolfSSL_X509_free(output->peer);
        output->peer = NULL;
    }
#endif

    XMEMCPY((byte*)output + copyOffset, (byte*)input + copyOffset,
            sizeof(WOLFSSL_SESSION) - copyOffset);

    /* Set sane values for copy */
    if (output->type != WOLFSSL_SESSION_TYPE_CACHE)
#ifndef NO_SESSION_CACHE
        output->cacheRow = INVALID_SESSION_ROW;
#endif
#if defined(SESSION_CERTS) && defined(OPENSSL_EXTRA)
    if (input->peer != NULL && input->peer->dynamicMemory) {
        if (wolfSSL_X509_up_ref(input->peer) != WOLFSSL_SUCCESS) {
            WOLFSSL_MSG("Can't increase peer cert ref count");
            output->peer = NULL;
        }
    }
    else if (!avoidSysCalls)
        output->peer = wolfSSL_X509_dup(input->peer);
    else
        /* output->peer is not that important to copy */
        output->peer = NULL;
#endif
    output->masterSecret = output->_masterSecret;
#ifndef NO_CLIENT_CACHE
    output->serverID = output->_serverID;
#endif
#ifdef OPENSSL_EXTRA
    output->sessionCtx = output->_sessionCtx;
#endif
#ifdef HAVE_SESSION_TICKET
    if (input->ticketLen > SESSION_TICKET_LEN) {
        /* Need dynamic buffer */
        if (ticBuff == NULL || ticLenAlloc < input->ticketLen) {
            /* allocate new one */
            byte* tmp;
            if (!avoidSysCalls) {
                WOLFSSL_MSG("Failed to allocate memory for ticket when avoiding"
                        " syscalls");
                output->ticket = ticBuff;
                output->ticketLenAlloc = ticLenAlloc;
                output->ticketLen = 0;
                ret = WOLFSSL_FAILURE;
            }
            else {
                tmp = (byte*)XREALLOC(ticBuff, input->ticketLen,
                        output->heap, DYNAMIC_TYPE_SESSION_TICK);
                if (tmp == NULL) {
                    WOLFSSL_MSG("Failed to allocate memory for ticket");
                    XFREE(ticBuff, output->heap, DYNAMIC_TYPE_SESSION_TICK);
                    output->ticket = NULL;
                    output->ticketLen = 0;
                    output->ticketLenAlloc = 0;
                    ret = WOLFSSL_FAILURE;
                }
                else {
                    ticBuff = tmp;
                    ticLenAlloc = input->ticketLen;
                }
            }
        }
        if (ticBuff != NULL && ret == WOLFSSL_SUCCESS) {
            XMEMCPY(ticBuff, input->ticket, input->ticketLen);
            output->ticket = ticBuff;
            output->ticketLenAlloc = ticLenAlloc;
        }
    }
    else {
        /* Default ticket to non dynamic */
        if (avoidSysCalls) {
            /* Try to use ticBuf if available. Caller can later move it to
             * the static buffer. */
            if (ticBuff != NULL) {
                if (ticLenAlloc >= input->ticketLen) {
                    output->ticket = output->_staticTicket;
                    output->ticketLenAlloc = 0;
                }
                else {
                    WOLFSSL_MSG("ticket dynamic buffer too small but we are "
                                "avoiding system calls");
                    ret = WOLFSSL_FAILURE;
                    output->ticket = ticBuff;
                    output->ticketLenAlloc = ticLenAlloc;
                    output->ticketLen = 0;
                }
            }
            else {
                output->ticket = output->_staticTicket;
                output->ticketLenAlloc = 0;
            }
        }
        else {
            if (ticBuff != NULL)
                XFREE(ticBuff, output->heap, DYNAMIC_TYPE_SESSION_TICK);
            output->ticket = output->_staticTicket;
            output->ticketLenAlloc = 0;
        }
        if (input->ticketLenAlloc > 0 && ret == WOLFSSL_SUCCESS) {
            /* Shouldn't happen as session should have placed this in
             * the static buffer */
            XMEMCPY(output->ticket, input->ticket,
                    input->ticketLen);
        }
    }
    ticBuff = NULL;
#endif /* HAVE_SESSION_TICKET */
    return ret;
}

WOLFSSL_SESSION* wolfSSL_SESSION_dup(WOLFSSL_SESSION* session)
{
#ifdef HAVE_EXT_CACHE
    WOLFSSL_SESSION* copy;

    WOLFSSL_ENTER("wolfSSL_SESSION_dup");

    session = ClientSessionToSession(session);
    if (session == NULL)
        return NULL;

#ifdef HAVE_SESSION_TICKET
    if (session->ticketLenAlloc > 0 && !session->ticket) {
        WOLFSSL_MSG("Session dynamic flag is set but ticket pointer is null");
        return NULL;
    }
#endif

    copy = wolfSSL_NewSession(session->heap);
    if (copy != NULL &&
            wolfSSL_DupSession(session, copy, 0) != WOLFSSL_SUCCESS) {
        wolfSSL_FreeSession(copy);
        copy = NULL;
    }
    return copy;
#else
    WOLFSSL_MSG("wolfSSL_SESSION_dup feature not compiled in");
    (void)session;
    return NULL;
#endif /* HAVE_EXT_CACHE */
}

void wolfSSL_FreeSession(WOLFSSL_SESSION* session)
{
    session = ClientSessionToSession(session);
    if (session == NULL)
        return;

    /* refCount will always be 1 or more if created externally.
     * Internal cache sessions don't initialize a refMutex. */
    if (session->refCount > 0) {
#ifndef SINGLE_THREADED
        if (wc_LockMutex(&session->refMutex) != 0) {
            WOLFSSL_MSG("Failed to lock session mutex");
            return;
        }
#endif
        if (session->refCount > 1) {
            session->refCount--;
#ifndef SINGLE_THREADED
            wc_UnLockMutex(&session->refMutex);
#endif
            return;
        }
#ifndef SINGLE_THREADED
        wc_UnLockMutex(&session->refMutex);
        wc_FreeMutex(&session->refMutex);
#endif
    }

#ifdef HAVE_EX_DATA_CLEANUP_HOOKS
    wolfSSL_CRYPTO_cleanup_ex_data(&session->ex_data);
#endif

#if defined(SESSION_CERTS) && defined(OPENSSL_EXTRA)
    if (session->peer) {
        wolfSSL_X509_free(session->peer);
        session->peer = NULL;
    }
#endif

#ifdef HAVE_SESSION_TICKET
    if (session->ticketLenAlloc > 0) {
        XFREE(session->ticket, session->heap, DYNAMIC_TYPE_SESSION_TICK);
    }
#endif

#ifdef HAVE_EX_DATA_CLEANUP_HOOKS
    wolfSSL_CRYPTO_cleanup_ex_data(&session->ex_data);
#endif

    if (session->type == WOLFSSL_SESSION_TYPE_HEAP) {
        XFREE(session, session->heap, DYNAMIC_TYPE_SESSION);
    }
}

void wolfSSL_SESSION_free(WOLFSSL_SESSION* session)
{
    session = ClientSessionToSession(session);
    wolfSSL_FreeSession(session);
}

#ifndef NO_SESSION_CACHE
int wolfSSL_CTX_add_session(WOLFSSL_CTX* ctx, WOLFSSL_SESSION* session)
{
    int    error = 0;
    const byte* id = NULL;
    byte idSz = 0;

    WOLFSSL_ENTER("wolfSSL_CTX_add_session");

    session = ClientSessionToSession(session);
    if (session == NULL)
        return WOLFSSL_FAILURE;

    /* Session cache is global */
    (void)ctx;

    id = session->sessionID;
    idSz = session->sessionIDSz;
    if (session->haveAltSessionID) {
        id = session->altSessionID;
        idSz = ID_LEN;
    }

    error = AddSessionToCache(session, id, idSz,
            NULL, session->side,
#ifdef HAVE_SESSION_TICKET
            session->ticketLen > 0,
#else
            0,
#endif
            NULL);

    return error == 0 ? WOLFSSL_SUCCESS : WOLFSSL_FAILURE;
}
#endif

#if defined(OPENSSL_EXTRA) || defined(HAVE_EXT_CACHE)

/**
* set cipher to WOLFSSL_SESSION from WOLFSSL_CIPHER
* @param session  a pointer to WOLFSSL_SESSION structure
* @param cipher   a function pointer to WOLFSSL_CIPHER
* @return WOLFSSL_SUCCESS on success, otherwise WOLFSSL_FAILURE
*/
int wolfSSL_SESSION_set_cipher(WOLFSSL_SESSION* session,
                                            const WOLFSSL_CIPHER* cipher)
{
    WOLFSSL_ENTER("wolfSSL_SESSION_set_cipher");

    session = ClientSessionToSession(session);
    /* sanity check */
    if (session == NULL || cipher == NULL) {
        WOLFSSL_MSG("bad argument");
        return WOLFSSL_FAILURE;
    }
    session->cipherSuite0 = cipher->cipherSuite0;
    session->cipherSuite  = cipher->cipherSuite;

    WOLFSSL_LEAVE("wolfSSL_SESSION_set_cipher", WOLFSSL_SUCCESS);
    return WOLFSSL_SUCCESS;
}
#endif /* OPENSSL_EXTRA || HAVE_EXT_CACHE */


/* helper function that takes in a protocol version struct and returns string */
static const char* wolfSSL_internal_get_version(const ProtocolVersion* version)
{
    WOLFSSL_ENTER("wolfSSL_get_version");

    if (version == NULL) {
        return "Bad arg";
    }

    if (version->major == SSLv3_MAJOR) {
        switch (version->minor) {
            case SSLv3_MINOR :
                return "SSLv3";
            case TLSv1_MINOR :
                return "TLSv1";
            case TLSv1_1_MINOR :
                return "TLSv1.1";
            case TLSv1_2_MINOR :
                return "TLSv1.2";
            case TLSv1_3_MINOR :
                return "TLSv1.3";
            default:
                return "unknown";
        }
    }
#ifdef WOLFSSL_DTLS
    else if (version->major == DTLS_MAJOR) {
        switch (version->minor) {
            case DTLS_MINOR :
                return "DTLS";
            case DTLSv1_2_MINOR :
                return "DTLSv1.2";
            default:
                return "unknown";
        }
    }
#endif /* WOLFSSL_DTLS */
    return "unknown";
}


const char* wolfSSL_get_version(const WOLFSSL* ssl)
{
    if (ssl == NULL) {
        WOLFSSL_MSG("Bad argument");
        return "unknown";
    }

    return wolfSSL_internal_get_version(&ssl->version);
}


/* current library version */
const char* wolfSSL_lib_version(void)
{
    return LIBWOLFSSL_VERSION_STRING;
}

#ifdef OPENSSL_EXTRA
#if defined(OPENSSL_VERSION_NUMBER) && OPENSSL_VERSION_NUMBER >= 0x10100000L
const char* wolfSSL_OpenSSL_version(int a)
{
    (void)a;
    return "wolfSSL " LIBWOLFSSL_VERSION_STRING;
}
#else
const char* wolfSSL_OpenSSL_version(void)
{
    return "wolfSSL " LIBWOLFSSL_VERSION_STRING;
}
#endif /* WOLFSSL_QT */
#endif


/* current library version in hex */
word32 wolfSSL_lib_version_hex(void)
{
    return LIBWOLFSSL_VERSION_HEX;
}


int wolfSSL_get_current_cipher_suite(WOLFSSL* ssl)
{
    WOLFSSL_ENTER("SSL_get_current_cipher_suite");
    if (ssl)
        return (ssl->options.cipherSuite0 << 8) | ssl->options.cipherSuite;
    return 0;
}

WOLFSSL_CIPHER* wolfSSL_get_current_cipher(WOLFSSL* ssl)
{
    WOLFSSL_ENTER("SSL_get_current_cipher");
    if (ssl) {
        ssl->cipher.cipherSuite0 = ssl->options.cipherSuite0;
        ssl->cipher.cipherSuite  = ssl->options.cipherSuite;
#if defined(OPENSSL_ALL) || defined(WOLFSSL_QT)
        ssl->cipher.bits = ssl->specs.key_size * 8;
#endif
        return &ssl->cipher;
    }
    else
        return NULL;
}


const char* wolfSSL_CIPHER_get_name(const WOLFSSL_CIPHER* cipher)
{
    WOLFSSL_ENTER("wolfSSL_CIPHER_get_name");

    if (cipher == NULL) {
        return NULL;
    }

    #if !defined(WOLFSSL_CIPHER_INTERNALNAME) && !defined(NO_ERROR_STRINGS) && \
        !defined(WOLFSSL_QT)
        return GetCipherNameIana(cipher->cipherSuite0, cipher->cipherSuite);
    #else
        return wolfSSL_get_cipher_name_from_suite(cipher->cipherSuite0,
                cipher->cipherSuite);
    #endif
}

const char*  wolfSSL_CIPHER_get_version(const WOLFSSL_CIPHER* cipher)
{
    WOLFSSL_ENTER("SSL_CIPHER_get_version");

    if (cipher == NULL || cipher->ssl == NULL) {
        return NULL;
    }

    return wolfSSL_get_version(cipher->ssl);
}

const char* wolfSSL_SESSION_CIPHER_get_name(const WOLFSSL_SESSION* session)
{
    session = ClientSessionToSession(session);
    if (session == NULL) {
        return NULL;
    }

#if defined(SESSION_CERTS) || !defined(NO_RESUME_SUITE_CHECK) || \
                        (defined(WOLFSSL_TLS13) && defined(HAVE_SESSION_TICKET))
    #if !defined(WOLFSSL_CIPHER_INTERNALNAME) && !defined(NO_ERROR_STRINGS)
        return GetCipherNameIana(session->cipherSuite0, session->cipherSuite);
    #else
        return GetCipherNameInternal(session->cipherSuite0, session->cipherSuite);
    #endif
#else
    return NULL;
#endif
}

const char* wolfSSL_get_cipher(WOLFSSL* ssl)
{
    WOLFSSL_ENTER("wolfSSL_get_cipher");
    return wolfSSL_CIPHER_get_name(wolfSSL_get_current_cipher(ssl));
}

/* gets cipher name in the format DHE-RSA-... rather then TLS_DHE... */
const char* wolfSSL_get_cipher_name(WOLFSSL* ssl)
{
    /* get access to cipher_name_idx in internal.c */
    return wolfSSL_get_cipher_name_internal(ssl);
}

const char* wolfSSL_get_cipher_name_from_suite(const byte cipherSuite0,
    const byte cipherSuite)
{
    return GetCipherNameInternal(cipherSuite0, cipherSuite);
}

const char* wolfSSL_get_cipher_name_iana_from_suite(const byte cipherSuite0,
        const byte cipherSuite)
{
    return GetCipherNameIana(cipherSuite0, cipherSuite);
}

int wolfSSL_get_cipher_suite_from_name(const char* name, byte* cipherSuite0,
                                       byte* cipherSuite, int *flags) {
    if ((name == NULL) ||
        (cipherSuite0 == NULL) ||
        (cipherSuite == NULL) ||
        (flags == NULL))
        return BAD_FUNC_ARG;
    return GetCipherSuiteFromName(name, cipherSuite0, cipherSuite, flags);
}


#if defined(WOLFSSL_QT) || defined(OPENSSL_ALL)
/* Creates and returns a new WOLFSSL_CIPHER stack. */
WOLFSSL_STACK* wolfSSL_sk_new_cipher(void)
{
    WOLFSSL_STACK* sk;
    WOLFSSL_ENTER("wolfSSL_sk_new_cipher");

    sk = wolfSSL_sk_new_null();
    if (sk == NULL)
        return NULL;
    sk->type = STACK_TYPE_CIPHER;

    return sk;
}

/* return 1 on success 0 on fail */
int wolfSSL_sk_CIPHER_push(WOLF_STACK_OF(WOLFSSL_CIPHER)* sk,
                                                      WOLFSSL_CIPHER* cipher)
{
    return wolfSSL_sk_push(sk, cipher);
}

#ifndef NO_WOLFSSL_STUB
WOLFSSL_CIPHER* wolfSSL_sk_CIPHER_pop(WOLF_STACK_OF(WOLFSSL_CIPHER)* sk)
{
    WOLFSSL_STUB("wolfSSL_sk_CIPHER_pop");
    (void)sk;
    return NULL;
}
#endif /* NO_WOLFSSL_STUB */
#endif /* WOLFSSL_QT || OPENSSL_ALL */

word32 wolfSSL_CIPHER_get_id(const WOLFSSL_CIPHER* cipher)
{
    word16 cipher_id = 0;

    WOLFSSL_ENTER("SSL_CIPHER_get_id");

    if (cipher && cipher->ssl) {
        cipher_id = (cipher->ssl->options.cipherSuite0 << 8) |
                     cipher->ssl->options.cipherSuite;
    }

    return cipher_id;
}

const WOLFSSL_CIPHER* wolfSSL_get_cipher_by_value(word16 value)
{
    const WOLFSSL_CIPHER* cipher = NULL;
    byte cipherSuite0, cipherSuite;
    WOLFSSL_ENTER("SSL_get_cipher_by_value");

    /* extract cipher id information */
    cipherSuite =   (value       & 0xFF);
    cipherSuite0 = ((value >> 8) & 0xFF);

    /* TODO: lookup by cipherSuite0 / cipherSuite */
    (void)cipherSuite0;
    (void)cipherSuite;

    return cipher;
}


#if defined(OPENSSL_ALL)
/* Free the structure for WOLFSSL_CIPHER stack
 *
 * sk  stack to free nodes in
 */
void wolfSSL_sk_CIPHER_free(WOLF_STACK_OF(WOLFSSL_CIPHER)* sk)
{
    WOLFSSL_STACK* node;
    WOLFSSL_STACK* tmp;
    WOLFSSL_ENTER("wolfSSL_sk_CIPHER_free");

    if (sk == NULL)
        return;

    /* parse through stack freeing each node */
    node = sk->next;
    while (node) {
        tmp  = node;
        node = node->next;
        XFREE(tmp, NULL, DYNAMIC_TYPE_OPENSSL);
    }

    /* free head of stack */
    XFREE(sk, NULL, DYNAMIC_TYPE_ASN1);
}
#endif /* OPENSSL_ALL */

#if defined(HAVE_ECC) || defined(HAVE_CURVE25519) || defined(HAVE_CURVE448) || \
                                                                 !defined(NO_DH)
#ifdef HAVE_FFDHE
static const char* wolfssl_ffdhe_name(word16 group)
{
    const char* str = NULL;
    switch (group) {
        case WOLFSSL_FFDHE_2048:
            str = "FFDHE_2048";
            break;
        case WOLFSSL_FFDHE_3072:
            str = "FFDHE_3072";
            break;
        case WOLFSSL_FFDHE_4096:
            str = "FFDHE_4096";
            break;
        case WOLFSSL_FFDHE_6144:
            str = "FFDHE_6144";
            break;
        case WOLFSSL_FFDHE_8192:
            str = "FFDHE_8192";
            break;
        default:
            break;
    }
    return str;
}
#endif
/* Return the name of the curve used for key exchange as a printable string.
 *
 * ssl  The SSL/TLS object.
 * returns NULL if ECDH was not used, otherwise the name as a string.
 */
const char* wolfSSL_get_curve_name(WOLFSSL* ssl)
{
    const char* cName = NULL;

    if (ssl == NULL)
        return NULL;

#ifdef HAVE_FFDHE
    if (ssl->namedGroup != 0) {
        cName = wolfssl_ffdhe_name(ssl->namedGroup);
    }
#endif

#ifdef HAVE_CURVE25519
    if (ssl->ecdhCurveOID == ECC_X25519_OID && cName == NULL) {
        cName = "X25519";
    }
#endif

#ifdef HAVE_CURVE448
    if (ssl->ecdhCurveOID == ECC_X448_OID && cName == NULL) {
        cName = "X448";
    }
#endif

#ifdef HAVE_ECC
    if (ssl->ecdhCurveOID != 0 && cName == NULL) {
        cName = wc_ecc_get_name(wc_ecc_get_oid(ssl->ecdhCurveOID, NULL,
                                NULL));
    }
#endif

    return cName;
}
#endif

#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_WPAS_SMALL) || \
    defined(OPENSSL_EXTRA_X509_SMALL)

    /* Creates a new WOLFSSL_ASN1_STRING structure.
     *
     * returns a pointer to the new structure created on success or NULL if fail
     */
    WOLFSSL_ASN1_STRING* wolfSSL_ASN1_STRING_new(void)
    {
        WOLFSSL_ASN1_STRING* asn1;

#ifdef WOLFSSL_DEBUG_OPENSSL
        WOLFSSL_ENTER("wolfSSL_ASN1_STRING_new");
#endif

        asn1 = (WOLFSSL_ASN1_STRING*)XMALLOC(sizeof(WOLFSSL_ASN1_STRING), NULL,
                DYNAMIC_TYPE_OPENSSL);
        if (asn1 != NULL) {
            XMEMSET(asn1, 0, sizeof(WOLFSSL_ASN1_STRING));
        }

        return asn1; /* no check for null because error case is returning null*/
    }

    /**
     * Used to duplicate a passed in WOLFSSL_ASN1_STRING*
     * @param asn1 WOLFSSL_ASN1_STRING* to be duplicated
     * @return WOLFSSL_ASN1_STRING* the duplicate struct or NULL on error
     */
    WOLFSSL_ASN1_STRING* wolfSSL_ASN1_STRING_dup(WOLFSSL_ASN1_STRING* asn1)
    {
        WOLFSSL_ASN1_STRING* dupl = NULL;

        WOLFSSL_ENTER("wolfSSL_ASN1_STRING_dup");
        if (!asn1) {
            WOLFSSL_MSG("Bad parameter");
            return NULL;
        }

        dupl = wolfSSL_ASN1_STRING_new();
        if (!dupl) {
            WOLFSSL_MSG("wolfSSL_ASN1_STRING_new error");
            return NULL;
        }

        dupl->type = asn1->type;
        dupl->flags = asn1->flags;

        if (wolfSSL_ASN1_STRING_set(dupl, asn1->data, asn1->length)
                != WOLFSSL_SUCCESS) {
            WOLFSSL_MSG("wolfSSL_ASN1_STRING_set error");
            wolfSSL_ASN1_STRING_free(dupl);
            return NULL;
        }

        return dupl;
    }


    /* used to free a WOLFSSL_ASN1_STRING structure */
    void wolfSSL_ASN1_STRING_free(WOLFSSL_ASN1_STRING* asn1)
    {
#ifdef WOLFSSL_DEBUG_OPENSSL
        WOLFSSL_ENTER("wolfSSL_ASN1_STRING_free");
#endif

        if (asn1 != NULL) {
            if (asn1->length > 0 && asn1->data != NULL && asn1->isDynamic) {
                XFREE(asn1->data, NULL, DYNAMIC_TYPE_OPENSSL);
            }
            XFREE(asn1, NULL, DYNAMIC_TYPE_OPENSSL);
        }
    }

    int wolfSSL_ASN1_STRING_cmp(const WOLFSSL_ASN1_STRING *a, const WOLFSSL_ASN1_STRING *b)
    {
        int i;
        WOLFSSL_ENTER("wolfSSL_ASN1_STRING_cmp");

        if (!a || !b) {
            return WOLFSSL_FATAL_ERROR;
        }

        if (a->length != b->length) {
            return a->length - b->length;
        }

        if ((i = XMEMCMP(a->data, b->data, a->length)) != 0) {
            return i;
        }

        return a->type - b->type;
    }
#endif /* OPENSSL_EXTRA || WOLFSSL_WPAS_SMALL */

#if !defined(NO_CERTS) && (defined(OPENSSL_EXTRA) || \
    defined(OPENSSL_EXTRA_X509_SMALL))

    int wolfSSL_ASN1_STRING_copy(WOLFSSL_ASN1_STRING* dest,
                            const WOLFSSL_ASN1_STRING* src)
    {
        if (src == NULL || dest == NULL) {
            return WOLFSSL_FAILURE;
        }
        dest->type = src->type;
        if(wolfSSL_ASN1_STRING_set(dest, src->data, src->length)
                    != WOLFSSL_SUCCESS) {
                return WOLFSSL_FAILURE;
        }
        dest->flags = src->flags;

        return WOLFSSL_SUCCESS;
    }
    /* Creates a new WOLFSSL_ASN1_STRING structure given the input type.
     *
     * type is the type of set when WOLFSSL_ASN1_STRING is created
     *
     * returns a pointer to the new structure created on success or NULL if fail
     */
    WOLFSSL_ASN1_STRING* wolfSSL_ASN1_STRING_type_new(int type)
    {
        WOLFSSL_ASN1_STRING* asn1;

#ifdef WOLFSSL_DEBUG_OPENSSL
        WOLFSSL_ENTER("wolfSSL_ASN1_STRING_type_new");
#endif

        asn1 = wolfSSL_ASN1_STRING_new();
        if (asn1 == NULL) {
            return NULL;
        }
        asn1->type = type;

        return asn1;
    }


/******************************************************************************
* wolfSSL_ASN1_STRING_type - returns the type of <asn1>
*
* RETURNS:
* returns the type set for <asn1>. Otherwise, returns WOLFSSL_FAILURE.
*/
    int wolfSSL_ASN1_STRING_type(const WOLFSSL_ASN1_STRING* asn1)
    {

#ifdef WOLFSSL_DEBUG_OPENSSL
        WOLFSSL_ENTER("wolfSSL_ASN1_STRING_type");
#endif

        if (asn1 == NULL) {
            return WOLFSSL_FAILURE;
        }

        return asn1->type;
    }

#endif /* !NO_CERTS && OPENSSL_EXTRA */

#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_WPAS_SMALL) || \
    defined(OPENSSL_EXTRA_X509_SMALL)
    /* if dataSz is negative then use XSTRLEN to find length of data
     * return WOLFSSL_SUCCESS on success and WOLFSSL_FAILURE on failure */
    /* `data` can be NULL and only buffer will be allocated */
    int wolfSSL_ASN1_STRING_set(WOLFSSL_ASN1_STRING* asn1, const void* data,
            int dataSz)
    {
        int sz;

#ifdef WOLFSSL_DEBUG_OPENSSL
        WOLFSSL_ENTER("wolfSSL_ASN1_STRING_set");
#endif

        if (asn1 == NULL || (data == NULL && dataSz < 0)) {
            return WOLFSSL_FAILURE;
        }

        if (dataSz < 0) {
            sz = (int)XSTRLEN((const char*)data);
        }
        else {
            sz = dataSz;
        }

        if (sz < 0) {
            return WOLFSSL_FAILURE;
        }

        /* free any existing data before copying */
        if (asn1->data != NULL && asn1->isDynamic) {
            XFREE(asn1->data, NULL, DYNAMIC_TYPE_OPENSSL);
            asn1->data = NULL;
        }

        if (sz + 1 > CTC_NAME_SIZE) { /* account for null char */
            /* create new data buffer and copy over */
            asn1->data = (char*)XMALLOC(sz + 1, NULL, DYNAMIC_TYPE_OPENSSL);
            if (asn1->data == NULL) {
                return WOLFSSL_FAILURE;
            }
            asn1->isDynamic = 1;
        }
        else {
            XMEMSET(asn1->strData, 0, CTC_NAME_SIZE);
            asn1->data = asn1->strData;
            asn1->isDynamic = 0;
        }
        if (data != NULL) {
            XMEMCPY(asn1->data, data, sz);
            asn1->data[sz] = '\0';
        }
        asn1->length = sz;

        return WOLFSSL_SUCCESS;
    }
#endif /* OPENSSL_EXTRA || WOLFSSL_WPAS_SMALL */

#ifndef NO_CERTS
#if defined(OPENSSL_EXTRA) || defined(OPENSSL_EXTRA_X509_SMALL)
    const unsigned char* wolfSSL_ASN1_STRING_get0_data(
                                            const WOLFSSL_ASN1_STRING* asn)
    {
        WOLFSSL_ENTER("wolfSSL_ASN1_STRING_get0_data");

        if (asn) {
            return (const unsigned char*)asn->data;
        } else {
            return NULL;
        }
    }
    unsigned char* wolfSSL_ASN1_STRING_data(WOLFSSL_ASN1_STRING* asn)
    {
#ifdef WOLFSSL_DEBUG_OPENSSL
        WOLFSSL_ENTER("wolfSSL_ASN1_STRING_data");
#endif

        if (asn) {
            return (unsigned char*)asn->data;
        }
        else {
            return NULL;
        }
    }


    int wolfSSL_ASN1_STRING_length(WOLFSSL_ASN1_STRING* asn)
    {
#ifdef WOLFSSL_DEBUG_OPENSSL
        WOLFSSL_ENTER("wolfSSL_ASN1_STRING_length");
#endif

        if (asn) {
            return asn->length;
        }
        else {
            return 0;
        }
    }
#endif /* OPENSSL_EXTRA || OPENSSL_EXTRA_X509_SMALL */

#ifdef OPENSSL_EXTRA
#ifndef NO_WOLFSSL_STUB
    WOLFSSL_ASN1_STRING* wolfSSL_d2i_DISPLAYTEXT(WOLFSSL_ASN1_STRING **asn,
                                             const unsigned char **in, long len)
    {
        WOLFSSL_STUB("d2i_DISPLAYTEXT");
        (void)asn;
        (void)in;
        (void)len;
        return NULL;
    }
#endif

#endif /* OPENSSL_EXTRA */

#endif /* !NO_CERTS */

#ifdef OPENSSL_EXTRA
#if defined(OPENSSL_ALL) || defined(WOLFSSL_QT)
/* return authentication NID corresponding to cipher suite
 * @param cipher a pointer to WOLFSSL_CIPHER
 * return NID if found, NID_undef if not found
 */
int wolfSSL_CIPHER_get_auth_nid(const WOLFSSL_CIPHER* cipher)
{
    static const struct authnid {
        const char* alg_name;
        const int  nid;
    } authnid_tbl[] = {
        {"RSA",     NID_auth_rsa},
        {"PSK",     NID_auth_psk},
        {"SRP",     NID_auth_srp},
        {"ECDSA",   NID_auth_ecdsa},
        {"None",    NID_auth_null},
        {NULL,      NID_undef}
    };

    const struct authnid* sa;
    const char* authStr;
    char n[MAX_SEGMENTS][MAX_SEGMENT_SZ] = {{0}};

    if (GetCipherSegment(cipher, n) == NULL) {
        WOLFSSL_MSG("no suitable cipher name found");
        return NID_undef;
    }

    authStr = GetCipherAuthStr(n);

    if (authStr != NULL) {
        for(sa = authnid_tbl; sa->alg_name != NULL; sa++) {
            if (XSTRNCMP(sa->alg_name, authStr, XSTRLEN(sa->alg_name)) == 0) {
                return sa->nid;
            }
        }
    }

    return NID_undef;
}
/* return cipher NID corresponding to cipher suite
 * @param cipher a pointer to WOLFSSL_CIPHER
 * return NID if found, NID_undef if not found
 */
int wolfSSL_CIPHER_get_cipher_nid(const WOLFSSL_CIPHER* cipher)
{
    static const struct ciphernid {
        const char* alg_name;
        const int  nid;
    } ciphernid_tbl[] = {
        {"AESGCM(256)",             NID_aes_256_gcm},
        {"AESGCM(128)",             NID_aes_128_gcm},
        {"AESCCM(128)",             NID_aes_128_ccm},
        {"AES(128)",                NID_aes_128_cbc},
        {"AES(256)",                NID_aes_256_cbc},
        {"CAMELLIA(256)",           NID_camellia_256_cbc},
        {"CAMELLIA(128)",           NID_camellia_128_cbc},
        {"RC4",                     NID_rc4},
        {"3DES",                    NID_des_ede3_cbc},
        {"CHACHA20/POLY1305(256)",  NID_chacha20_poly1305},
        {"None",                    NID_undef},
        {NULL,                      NID_undef}
    };

    const struct ciphernid* c;
    const char* encStr;
    char n[MAX_SEGMENTS][MAX_SEGMENT_SZ] = {{0}};

    WOLFSSL_ENTER("wolfSSL_CIPHER_get_cipher_nid");

    if (GetCipherSegment(cipher, n) == NULL) {
        WOLFSSL_MSG("no suitable cipher name found");
        return NID_undef;
    }

    encStr = GetCipherEncStr(n);

    if (encStr != NULL) {
        for(c = ciphernid_tbl; c->alg_name != NULL; c++) {
            if (XSTRNCMP(c->alg_name, encStr, XSTRLEN(c->alg_name)) == 0) {
                return c->nid;
            }
        }
    }

    return NID_undef;
}
/* return digest NID corresponding to cipher suite
 * @param cipher a pointer to WOLFSSL_CIPHER
 * return NID if found, NID_undef if not found
 */
int wolfSSL_CIPHER_get_digest_nid(const WOLFSSL_CIPHER* cipher)
{
    static const struct macnid {
        const char* alg_name;
        const int  nid;
    } macnid_tbl[] = {
        {"SHA1",    NID_sha1},
        {"SHA256",  NID_sha256},
        {"SHA384",  NID_sha384},
        {NULL,      NID_undef}
    };

    const struct macnid* mc;
    const char* name;
    const char* macStr;
    char n[MAX_SEGMENTS][MAX_SEGMENT_SZ] = {{0}};
    (void)name;

    WOLFSSL_ENTER("wolfSSL_CIPHER_get_digest_nid");

    if ((name = GetCipherSegment(cipher, n)) == NULL) {
        WOLFSSL_MSG("no suitable cipher name found");
        return NID_undef;
    }

    /* in MD5 case, NID will be NID_md5 */
    if (XSTRSTR(name, "MD5") != NULL) {
        return NID_md5;
    }

    macStr = GetCipherMacStr(n);

    if (macStr != NULL) {
        for(mc = macnid_tbl; mc->alg_name != NULL; mc++) {
            if (XSTRNCMP(mc->alg_name, macStr, XSTRLEN(mc->alg_name)) == 0) {
                return mc->nid;
            }
        }
    }

    return NID_undef;
}
/* return key exchange NID corresponding to cipher suite
 * @param cipher a pointer to WOLFSSL_CIPHER
 * return NID if found, NID_undef if not found
 */
int wolfSSL_CIPHER_get_kx_nid(const WOLFSSL_CIPHER* cipher)
{
static const struct kxnid {
        const char* name;
        const int  nid;
    } kxnid_table[] = {
        {"ECDHEPSK",  NID_kx_ecdhe_psk},
        {"ECDH",      NID_kx_ecdhe},
        {"DHEPSK",    NID_kx_dhe_psk},
        {"DH",        NID_kx_dhe},
        {"RSAPSK",    NID_kx_rsa_psk},
        {"SRP",       NID_kx_srp},
        {"EDH",       NID_kx_dhe},
        {"RSA",       NID_kx_rsa},
        {NULL,        NID_undef}
    };

    const struct kxnid* k;
    const char* name;
    const char* keaStr;
    char n[MAX_SEGMENTS][MAX_SEGMENT_SZ] = {{0}};
    (void)name;

    WOLFSSL_ENTER("wolfSSL_CIPHER_get_kx_nid");

    if ((name = GetCipherSegment(cipher, n)) == NULL) {
        WOLFSSL_MSG("no suitable cipher name found");
        return NID_undef;
    }

    /* in TLS 1.3 case, NID will be NID_kx_any */
    if (XSTRNCMP(name, "TLS13", 5) == 0) {
        return NID_kx_any;
    }

    keaStr = GetCipherKeaStr(n);

    if (keaStr != NULL) {
        for(k = kxnid_table; k->name != NULL; k++) {
            if (XSTRNCMP(k->name, keaStr, XSTRLEN(k->name)) == 0) {
                printf("k->name %s k->nid %d\n", k->name, k->nid);
                return k->nid;
            }
        }
    }

    return NID_undef;
}
/* check if cipher suite is AEAD
 * @param cipher a pointer to WOLFSSL_CIPHER
 * return 1 if cipher is AEAD, 0 otherwise
 */
int wolfSSL_CIPHER_is_aead(const WOLFSSL_CIPHER* cipher)
{
    char n[MAX_SEGMENTS][MAX_SEGMENT_SZ] = {{0}};

    WOLFSSL_ENTER("wolfSSL_CIPHER_is_aead");

    if (GetCipherSegment(cipher, n) == NULL) {
        WOLFSSL_MSG("no suitable cipher name found");
        return NID_undef;
    }

    return IsCipherAEAD(n);
}
/* Creates cipher->description based on cipher->offset
 * cipher->offset is set in wolfSSL_get_ciphers_compat when it is added
 * to a stack of ciphers.
 * @param [in] cipher: A cipher from a stack of ciphers.
 * return WOLFSSL_SUCCESS if cipher->description is set, else WOLFSSL_FAILURE
 */
int wolfSSL_sk_CIPHER_description(WOLFSSL_CIPHER* cipher)
{
    int strLen;
    unsigned long offset;
    char* dp;
    const char* name;
    const char *keaStr, *authStr, *encStr, *macStr, *protocol;
    char n[MAX_SEGMENTS][MAX_SEGMENT_SZ] = {{0}};
    int len = MAX_DESCRIPTION_SZ-1;
    const CipherSuiteInfo* cipher_names;
    ProtocolVersion pv;
    WOLFSSL_ENTER("wolfSSL_sk_CIPHER_description");

    if (cipher == NULL)
        return WOLFSSL_FAILURE;

    dp = cipher->description;
    if (dp == NULL)
        return WOLFSSL_FAILURE;

    cipher_names = GetCipherNames();

    offset = cipher->offset;
    if (offset >= (unsigned long)GetCipherNamesSize())
        return WOLFSSL_FAILURE;
    pv.major = cipher_names[offset].major;
    pv.minor = cipher_names[offset].minor;
    protocol = wolfSSL_internal_get_version(&pv);

    if ((name = GetCipherSegment(cipher, n)) == NULL) {
        WOLFSSL_MSG("no suitable cipher name found");
        return WOLFSSL_FAILURE;
    }

    /* keaStr */
    keaStr = GetCipherKeaStr(n);
    /* authStr */
    authStr = GetCipherAuthStr(n);
    /* encStr */
    encStr = GetCipherEncStr(n);
    if ((cipher->bits = SetCipherBits(encStr)) == WOLFSSL_FAILURE) {
       WOLFSSL_MSG("Cipher Bits Not Set.");
    }
    /* macStr */
    macStr = GetCipherMacStr(n);


    /* Build up the string by copying onto the end. */
    XSTRNCPY(dp, name, len);
    dp[len-1] = '\0'; strLen = (int)XSTRLEN(dp);
    len -= strLen; dp += strLen;

    XSTRNCPY(dp, " ", len);
    dp[len-1] = '\0'; strLen = (int)XSTRLEN(dp);
    len -= strLen; dp += strLen;
    XSTRNCPY(dp, protocol, len);
    dp[len-1] = '\0'; strLen = (int)XSTRLEN(dp);
    len -= strLen; dp += strLen;

    XSTRNCPY(dp, " Kx=", len);
    dp[len-1] = '\0'; strLen = (int)XSTRLEN(dp);
    len -= strLen; dp += strLen;
    XSTRNCPY(dp, keaStr, len);
    dp[len-1] = '\0'; strLen = (int)XSTRLEN(dp);
    len -= strLen; dp += strLen;

    XSTRNCPY(dp, " Au=", len);
    dp[len-1] = '\0'; strLen = (int)XSTRLEN(dp);
    len -= strLen; dp += strLen;
    XSTRNCPY(dp, authStr, len);
    dp[len-1] = '\0'; strLen = (int)XSTRLEN(dp);
    len -= strLen; dp += strLen;

    XSTRNCPY(dp, " Enc=", len);
    dp[len-1] = '\0'; strLen = (int)XSTRLEN(dp);
    len -= strLen; dp += strLen;
    XSTRNCPY(dp, encStr, len);
    dp[len-1] = '\0'; strLen = (int)XSTRLEN(dp);
    len -= strLen; dp += strLen;

    XSTRNCPY(dp, " Mac=", len);
    dp[len-1] = '\0'; strLen = (int)XSTRLEN(dp);
    len -= strLen; dp += strLen;
    XSTRNCPY(dp, macStr, len);
    dp[len-1] = '\0';

    return WOLFSSL_SUCCESS;
}
#endif /* OPENSSL_ALL || WOLFSSL_QT */

static WC_INLINE const char* wolfssl_kea_to_string(int kea)
{
    const char* keaStr;

    switch (kea) {
        case no_kea:
            keaStr = "None";
            break;
#ifndef NO_RSA
        case rsa_kea:
            keaStr = "RSA";
            break;
#endif
#ifndef NO_DH
        case diffie_hellman_kea:
            keaStr = "DHE";
            break;
#endif
        case fortezza_kea:
            keaStr = "FZ";
            break;
#ifndef NO_PSK
        case psk_kea:
            keaStr = "PSK";
            break;
    #ifndef NO_DH
        case dhe_psk_kea:
            keaStr = "DHEPSK";
            break;
    #endif
    #ifdef HAVE_ECC
        case ecdhe_psk_kea:
            keaStr = "ECDHEPSK";
            break;
    #endif
#endif
#ifdef HAVE_ECC
        case ecc_diffie_hellman_kea:
            keaStr = "ECDHE";
            break;
        case ecc_static_diffie_hellman_kea:
            keaStr = "ECDH";
            break;
#endif
        default:
            keaStr = "unknown";
            break;
    }

    return keaStr;
}

static WC_INLINE const char* wolfssl_sigalg_to_string(int sig_algo)
{
    const char* authStr;

    switch (sig_algo) {
        case anonymous_sa_algo:
            authStr = "None";
            break;
#ifndef NO_RSA
        case rsa_sa_algo:
            authStr = "RSA";
            break;
    #ifdef WC_RSA_PSS
        case rsa_pss_sa_algo:
            authStr = "RSA-PSS";
            break;
    #endif
#endif
#ifndef NO_DSA
        case dsa_sa_algo:
            authStr = "DSA";
            break;
#endif
#ifdef HAVE_ECC
        case ecc_dsa_sa_algo:
            authStr = "ECDSA";
            break;
#endif
#ifdef HAVE_ED25519
        case ed25519_sa_algo:
            authStr = "Ed25519";
            break;
#endif
#ifdef HAVE_ED448
        case ed448_sa_algo:
            authStr = "Ed448";
            break;
#endif
        default:
            authStr = "unknown";
            break;
    }

    return authStr;
}

static WC_INLINE const char* wolfssl_cipher_to_string(int cipher, int key_size)
{
    const char* encStr;

    (void)key_size;

    switch (cipher) {
        case wolfssl_cipher_null:
            encStr = "None";
            break;
#ifndef NO_RC4
        case wolfssl_rc4:
            encStr = "RC4(128)";
            break;
#endif
#ifndef NO_DES3
        case wolfssl_triple_des:
            encStr = "3DES(168)";
            break;
#endif
#ifndef NO_AES
        case wolfssl_aes:
            if (key_size == 128)
                encStr = "AES(128)";
            else if (key_size == 256)
                encStr = "AES(256)";
            else
                encStr = "AES(?)";
            break;
    #ifdef HAVE_AESGCM
        case wolfssl_aes_gcm:
            if (key_size == 128)
                encStr = "AESGCM(128)";
            else if (key_size == 256)
                encStr = "AESGCM(256)";
            else
                encStr = "AESGCM(?)";
            break;
    #endif
    #ifdef HAVE_AESCCM
        case wolfssl_aes_ccm:
            if (key_size == 128)
                encStr = "AESCCM(128)";
            else if (key_size == 256)
                encStr = "AESCCM(256)";
            else
                encStr = "AESCCM(?)";
            break;
    #endif
#endif
#ifdef HAVE_CHACHA
        case wolfssl_chacha:
            encStr = "CHACHA20/POLY1305(256)";
            break;
#endif
#ifdef HAVE_CAMELLIA
        case wolfssl_camellia:
            if (key_size == 128)
                encStr = "Camellia(128)";
            else if (key_size == 256)
                encStr = "Camellia(256)";
            else
                encStr = "Camellia(?)";
            break;
#endif
        default:
            encStr = "unknown";
            break;
    }

    return encStr;
}

static WC_INLINE const char* wolfssl_mac_to_string(int mac)
{
    const char* macStr;

    switch (mac) {
        case no_mac:
            macStr = "None";
            break;
#ifndef NO_MD5
        case md5_mac:
            macStr = "MD5";
            break;
#endif
#ifndef NO_SHA
        case sha_mac:
            macStr = "SHA1";
            break;
#endif
#ifdef HAVE_SHA224
        case sha224_mac:
            macStr = "SHA224";
            break;
#endif
#ifndef NO_SHA256
        case sha256_mac:
            macStr = "SHA256";
            break;
#endif
#ifdef HAVE_SHA384
        case sha384_mac:
            macStr = "SHA384";
            break;
#endif
#ifdef HAVE_SHA512
        case sha512_mac:
            macStr = "SHA512";
            break;
#endif
        default:
            macStr = "unknown";
            break;
    }

    return macStr;
}

char* wolfSSL_CIPHER_description(const WOLFSSL_CIPHER* cipher, char* in,
                                 int len)
{
    char *ret = in;
    const char *keaStr, *authStr, *encStr, *macStr;
    size_t strLen;
    WOLFSSL_ENTER("wolfSSL_CIPHER_description");

    if (cipher == NULL || in == NULL)
        return NULL;

#if defined(WOLFSSL_QT) || defined(OPENSSL_ALL)
    /* if cipher is in the stack from wolfSSL_get_ciphers_compat then
     * Return the description based on cipher_names[cipher->offset]
     */
    if (cipher->in_stack == TRUE) {
        wolfSSL_sk_CIPHER_description((WOLFSSL_CIPHER*)cipher);
        XSTRNCPY(in,cipher->description,len);
        return ret;
    }
#endif

    /* Get the cipher description based on the SSL session cipher */
    keaStr = wolfssl_kea_to_string(cipher->ssl->specs.kea);
    authStr = wolfssl_sigalg_to_string(cipher->ssl->specs.sig_algo);
    encStr = wolfssl_cipher_to_string(cipher->ssl->specs.bulk_cipher_algorithm,
                                      cipher->ssl->specs.key_size);
    macStr = wolfssl_mac_to_string(cipher->ssl->specs.mac_algorithm);

    /* Build up the string by copying onto the end. */
    XSTRNCPY(in, wolfSSL_CIPHER_get_name(cipher), len);
    in[len-1] = '\0'; strLen = XSTRLEN(in); len -= (int)strLen; in += strLen;

    XSTRNCPY(in, " ", len);
    in[len-1] = '\0'; strLen = XSTRLEN(in); len -= (int)strLen; in += strLen;
    XSTRNCPY(in, wolfSSL_get_version(cipher->ssl), len);
    in[len-1] = '\0'; strLen = XSTRLEN(in); len -= (int)strLen; in += strLen;

    XSTRNCPY(in, " Kx=", len);
    in[len-1] = '\0'; strLen = XSTRLEN(in); len -= (int)strLen; in += strLen;
    XSTRNCPY(in, keaStr, len);
    in[len-1] = '\0'; strLen = XSTRLEN(in); len -= (int)strLen; in += strLen;

    XSTRNCPY(in, " Au=", len);
    in[len-1] = '\0'; strLen = XSTRLEN(in); len -= (int)strLen; in += strLen;
    XSTRNCPY(in, authStr, len);
    in[len-1] = '\0'; strLen = XSTRLEN(in); len -= (int)strLen; in += strLen;

    XSTRNCPY(in, " Enc=", len);
    in[len-1] = '\0'; strLen = XSTRLEN(in); len -= (int)strLen; in += strLen;
    XSTRNCPY(in, encStr, len);
    in[len-1] = '\0'; strLen = XSTRLEN(in); len -= (int)strLen; in += strLen;

    XSTRNCPY(in, " Mac=", len);
    in[len-1] = '\0'; strLen = XSTRLEN(in); len -= (int)strLen; in += strLen;
    XSTRNCPY(in, macStr, len);
    in[len-1] = '\0';

    return ret;
}


#ifndef NO_WOLFSSL_STUB
int wolfSSL_OCSP_parse_url(char* url, char** host, char** port, char** path,
                   int* ssl)
{
    (void)url;
    (void)host;
    (void)port;
    (void)path;
    (void)ssl;
    WOLFSSL_STUB("OCSP_parse_url");
    return 0;
}
#endif

#ifndef NO_MD4

void wolfSSL_MD4_Init(WOLFSSL_MD4_CTX* md4)
{
    /* make sure we have a big enough buffer */
    typedef char ok[sizeof(md4->buffer) >= sizeof(Md4) ? 1 : -1];
    (void) sizeof(ok);

    WOLFSSL_ENTER("MD4_Init");
    wc_InitMd4((Md4*)md4);
}


void wolfSSL_MD4_Update(WOLFSSL_MD4_CTX* md4, const void* data,
                       unsigned long len)
{
    WOLFSSL_ENTER("MD4_Update");
    wc_Md4Update((Md4*)md4, (const byte*)data, (word32)len);
}


void wolfSSL_MD4_Final(unsigned char* digest, WOLFSSL_MD4_CTX* md4)
{
    WOLFSSL_ENTER("MD4_Final");
    wc_Md4Final((Md4*)md4, digest);
}

#endif /* NO_MD4 */

#ifndef NO_WOLFSSL_STUB
void wolfSSL_RAND_screen(void)
{
    WOLFSSL_STUB("RAND_screen");
}
#endif



int wolfSSL_RAND_load_file(const char* fname, long len)
{
    (void)fname;
    /* wolfCrypt provides enough entropy internally or will report error */
    if (len == -1)
        return 1024;
    else
        return (int)len;
}


#ifndef NO_WOLFSSL_STUB
WOLFSSL_COMP_METHOD* wolfSSL_COMP_zlib(void)
{
    WOLFSSL_STUB("COMP_zlib");
    return 0;
}
#endif

#ifndef NO_WOLFSSL_STUB
WOLFSSL_COMP_METHOD* wolfSSL_COMP_rle(void)
{
    WOLFSSL_STUB("COMP_rle");
    return 0;
}
#endif

#ifndef NO_WOLFSSL_STUB
int wolfSSL_COMP_add_compression_method(int method, void* data)
{
    (void)method;
    (void)data;
    WOLFSSL_STUB("COMP_add_compression_method");
    return 0;
}
#endif

/*  wolfSSL_set_dynlock_create_callback
 *  CRYPTO_set_dynlock_create_callback has been deprecated since openSSL 1.0.1.
 *  This function exists for compatibility purposes because wolfSSL satisfies
 *  thread safety without relying on the callback.
 */
void wolfSSL_set_dynlock_create_callback(WOLFSSL_dynlock_value* (*f)(
                                                          const char*, int))
{
    WOLFSSL_STUB("CRYPTO_set_dynlock_create_callback");
    (void)f;
}
/*  wolfSSL_set_dynlock_lock_callback
 *  CRYPTO_set_dynlock_lock_callback has been deprecated since openSSL 1.0.1.
 *  This function exists for compatibility purposes because wolfSSL satisfies
 *  thread safety without relying on the callback.
 */
void wolfSSL_set_dynlock_lock_callback(
             void (*f)(int, WOLFSSL_dynlock_value*, const char*, int))
{
    WOLFSSL_STUB("CRYPTO_set_set_dynlock_lock_callback");
    (void)f;
}
/*  wolfSSL_set_dynlock_destroy_callback
 *  CRYPTO_set_dynlock_destroy_callback has been deprecated since openSSL 1.0.1.
 *  This function exists for compatibility purposes because wolfSSL satisfies
 *  thread safety without relying on the callback.
 */
void wolfSSL_set_dynlock_destroy_callback(
                  void (*f)(WOLFSSL_dynlock_value*, const char*, int))
{
    WOLFSSL_STUB("CRYPTO_set_set_dynlock_destroy_callback");
    (void)f;
}


#endif /* OPENSSL_EXTRA */

#ifdef OPENSSL_EXTRA
#ifndef NO_CERTS

#if !defined(NO_ASN) && !defined(NO_PWDBASED)
/* Copies unencrypted DER key buffer into "der". If "der" is null then the size
 * of buffer needed is returned. If *der == NULL then it allocates a buffer.
 * NOTE: This also advances the "der" pointer to be at the end of buffer.
 *
 * Returns size of key buffer on success
 */
int wolfSSL_i2d_PrivateKey(const WOLFSSL_EVP_PKEY* key, unsigned char** der)
{
    return wolfSSL_EVP_PKEY_get_der(key, der);
}

int wolfSSL_i2d_PublicKey(const WOLFSSL_EVP_PKEY *key, unsigned char **der)
{
    return wolfSSL_EVP_PKEY_get_der(key, der);
}
#endif /* !NO_ASN && !NO_PWDBASED */

#endif /* !NO_CERTS */
#endif /* OPENSSL_EXTRA */

#ifdef OPENSSL_EXTRA

/******************************************************************************
* wolfSSL_CTX_set1_param - set a pointer to the SSL verification parameters
*
* RETURNS:
*   WOLFSSL_SUCCESS on success, otherwise returns WOLFSSL_FAILURE
*   Note: Returns WOLFSSL_SUCCESS, in case either parameter is NULL,
*   same as openssl.
*/
int wolfSSL_CTX_set1_param(WOLFSSL_CTX* ctx, WOLFSSL_X509_VERIFY_PARAM *vpm)
{
    if (ctx == NULL || vpm == NULL)
        return WOLFSSL_SUCCESS;

    return wolfSSL_X509_VERIFY_PARAM_set1(ctx->param, vpm);
}

/******************************************************************************
* wolfSSL_CTX/_get0_param - return a pointer to the SSL verification parameters
*
* RETURNS:
* returns pointer to the SSL verification parameters on success,
* otherwise returns NULL
*/
WOLFSSL_X509_VERIFY_PARAM* wolfSSL_CTX_get0_param(WOLFSSL_CTX* ctx)
{
    if (ctx == NULL) {
        return NULL;
    }

    return ctx->param;
}

WOLFSSL_X509_VERIFY_PARAM* wolfSSL_get0_param(WOLFSSL* ssl)
{
    if (ssl == NULL) {
        return NULL;
    }
    return ssl->param;
}

#endif /* OPENSSL_EXTRA */

#if defined(OPENSSL_EXTRA)
int wolfSSL_i2d_ASN1_INTEGER(WOLFSSL_ASN1_INTEGER* a, unsigned char** out)
{
    int ret = 0;
    word32 idx = 0;
    int len;
    int preAlloc = 1;

    WOLFSSL_ENTER("wolfSSL_i2d_ASN1_INTEGER");

    if (a == NULL || a->data == NULL || a->length <= 0 || out == NULL) {
        WOLFSSL_MSG("Bad parameter.");
        ret = WOLFSSL_FATAL_ERROR;
    }

    if (ret == 0 && *out == NULL) {
        preAlloc = 0;
        *out = (unsigned char*)XMALLOC(a->length, NULL, DYNAMIC_TYPE_ASN1);
        if (*out == NULL) {
            WOLFSSL_MSG("Failed to allocate output buffer.");
            ret = WOLFSSL_FATAL_ERROR;
        }
    }
    if (ret == 0) {
        /*
         * A WOLFSSL_ASN1_INTEGER stores the DER buffer of the integer in its
         * "data" field, but it's only the magnitude of the number (i.e. the
         * sign isn't encoded). The "negative" field is 1 if the value should
         * be interpreted as negative and 0 otherwise. If the value is negative,
         * we need to output the 2's complement of the value in the DER output.
         */
        XMEMCPY(*out, a->data, a->length);
        if (a->negative) {
            if (GetLength(a->data, &idx, &len, a->length) < 0) {
                ret = WOLFSSL_FATAL_ERROR;
            }
            else {
                ++idx;
                for (; (int)idx < a->length; ++idx) {
                    (*out)[idx] = ~(*out)[idx];
                }
                do {
                    --idx;
                    ++(*out)[idx];
                } while ((*out)[idx] == 0);
            }
        }
    }
    if (ret == 0) {
        ret = a->length;
        if (preAlloc) {
            *out += a->length;
        }
    }

    WOLFSSL_LEAVE("wolfSSL_i2d_ASN1_INTEGER", ret);

    return ret;
}

WOLFSSL_ASN1_INTEGER* wolfSSL_d2i_ASN1_INTEGER(WOLFSSL_ASN1_INTEGER** a,
                                               const unsigned char** in,
                                               long inSz)
{
    WOLFSSL_ASN1_INTEGER* ret = NULL;
    int err = 0;
    word32 idx = 0;
    int len;

    WOLFSSL_ENTER("wolfSSL_d2i_ASN1_INTEGER");

    if (in == NULL || *in == NULL || inSz <= 0) {
        WOLFSSL_MSG("Bad parameter");
        err = 1;
    }

    if (err == 0 && (*in)[0] != ASN_INTEGER) {
        WOLFSSL_MSG("Tag doesn't indicate integer type.");
        err = 1;
    }
    if (err == 0) {
        ret = wolfSSL_ASN1_INTEGER_new();
        if (ret == NULL) {
            err = 1;
        }
        else {
            ret->type = V_ASN1_INTEGER;
        }
    }
    if (err == 0 && inSz > (long)sizeof(ret->intData)) {
        ret->data = (unsigned char*)XMALLOC(inSz, NULL, DYNAMIC_TYPE_ASN1);
        if (ret->data == NULL) {
            err = 1;
        }
        else {
            ret->isDynamic = 1;
            ret->dataMax = (word32)inSz;
        }
    }
    if (err == 0) {
        XMEMCPY(ret->data, *in, inSz);
        ret->length = (word32)inSz;
        /* Advance to the end of the length field.*/
        if (GetLength(*in, &idx, &len, (word32)inSz) < 0) {
            err = 1;
        }
        else {
            /* See 2's complement comment in wolfSSL_d2i_ASN1_INTEGER. */
            ret->negative = (*in)[idx+1] & 0x80;
            if (ret->negative) {
                ++idx;
                for (; (int)idx < inSz; ++idx) {
                    ret->data[idx] = ~ret->data[idx];
                }
                do {
                    --idx;
                    ++ret->data[idx];
                } while (ret->data[idx] == 0);
                ret->type |= V_ASN1_NEG_INTEGER;
            }
            if (a != NULL) {
                *a = ret;
            }
        }
    }

    if (err != 0) {
        wolfSSL_ASN1_INTEGER_free(ret);
        ret = NULL;
    }

    return ret;
}
#endif /* OPENSSL_EXTRA */

#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_WPAS_SMALL)
/* Used to create a new WOLFSSL_ASN1_INTEGER structure.
 * returns a pointer to new structure on success and NULL on failure
 */
WOLFSSL_ASN1_INTEGER* wolfSSL_ASN1_INTEGER_new(void)
{
    WOLFSSL_ASN1_INTEGER* a;

    a = (WOLFSSL_ASN1_INTEGER*)XMALLOC(sizeof(WOLFSSL_ASN1_INTEGER), NULL,
                                       DYNAMIC_TYPE_OPENSSL);
    if (a == NULL) {
        return NULL;
    }

    XMEMSET(a, 0, sizeof(WOLFSSL_ASN1_INTEGER));
    a->data    = a->intData;
    a->isDynamic = 0;
    a->dataMax = WOLFSSL_ASN1_INTEGER_MAX;
    a->length  = 0;
    return a;
}


/* free's internal elements of WOLFSSL_ASN1_INTEGER and free's "in" itself */
void wolfSSL_ASN1_INTEGER_free(WOLFSSL_ASN1_INTEGER* in)
{
    if (in != NULL) {
        if (in->isDynamic) {
            XFREE(in->data, NULL, DYNAMIC_TYPE_OPENSSL);
        }
        XFREE(in, NULL, DYNAMIC_TYPE_OPENSSL);
    }
}


/* Duplicate all WOLFSSL_ASN1_INTEGER members from src to dup
 *  src : WOLFSSL_ASN1_INTEGER to duplicate
 *  Returns pointer to duplicate WOLFSSL_ASN1_INTEGER
 */
WOLFSSL_ASN1_INTEGER* wolfSSL_ASN1_INTEGER_dup(const WOLFSSL_ASN1_INTEGER* src)
{
    WOLFSSL_ASN1_INTEGER* copy;
    WOLFSSL_ENTER("wolfSSL_ASN1_INTEGER_dup");
    if (!src)
        return NULL;

    copy = wolfSSL_ASN1_INTEGER_new();

    if (copy == NULL)
        return NULL;

    copy->negative  = src->negative;
    copy->dataMax   = src->dataMax;
    copy->isDynamic = src->isDynamic;
#if defined(OPENSSL_ALL) || defined(WOLFSSL_QT)
    copy->length    = src->length;
#endif
    XSTRNCPY((char*)copy->intData,(const char*)src->intData,WOLFSSL_ASN1_INTEGER_MAX);

    if (copy->isDynamic && src->data && copy->dataMax) {
        copy->data = (unsigned char*)
            XMALLOC(src->dataMax,NULL,DYNAMIC_TYPE_OPENSSL);
        if (copy->data == NULL) {
            wolfSSL_ASN1_INTEGER_free(copy);
            return NULL;
        }
        XMEMCPY(copy->data, src->data, copy->dataMax);
    }
    return copy;
}


/* sets the value of WOLFSSL_ASN1_INTEGER a to the long value v. */
int wolfSSL_ASN1_INTEGER_set(WOLFSSL_ASN1_INTEGER *a, long v)
{
    int ret = WOLFSSL_SUCCESS; /* return 1 for success and 0 for failure */
    int j;
    unsigned int i = 0;
    unsigned char tmp[sizeof(long)+1] = {0};
    int pad = 0;

    if (a != NULL) {
        /* dynamically create data buffer, +2 for type and length */
        a->data = (unsigned char*)XMALLOC((sizeof(long)+1) + 2, NULL,
                DYNAMIC_TYPE_OPENSSL);
        if (a->data == NULL) {
            wolfSSL_ASN1_INTEGER_free(a);
            ret = WOLFSSL_FAILURE;
        }
        else {
            a->dataMax   = (int)(sizeof(long)+1) + 2;
            a->isDynamic = 1;
        }
    }
    else {
        /* Invalid parameter */
        ret = WOLFSSL_FAILURE;
    }


    if (ret != WOLFSSL_FAILURE) {
        /* Set type */
        a->data[i++] = ASN_INTEGER;

        /* Check for negative */
        if (v < 0) {
            a->negative = 1;
            v *= -1;
        }

        /* Create char buffer */
        for (j = 0; j < (int)sizeof(long); j++) {
            if (v == 0) {
                break;
            }
            tmp[j] = (unsigned char)(v & 0xff);
            v >>= 8;
        }

        /* 0 pad to indicate positive number when top bit set. */
        if ((!a->negative) && (j > 0) && (tmp[j-1] & 0x80)) {
            pad = 1;
        }
        /* Set length */
        a->data[i++] = (unsigned char)(((j == 0) ? ++j : j) + pad);
        /* +2 for type and length */
        a->length = j + pad + 2;

        /* Add padding if required. */
        if (pad) {
            a->data[i++] = 0;
        }
        /* Copy to data */
        for (; j > 0; j--) {
            a->data[i++] = tmp[j-1];
        }
    }

    return ret;
}

#endif /* OPENSSL_EXTRA || WOLFSSL_WPAS_SMALL */

#if defined(WOLFSSL_MYSQL_COMPATIBLE) || defined(WOLFSSL_NGINX) || \
    defined(WOLFSSL_HAPROXY) || defined(OPENSSL_EXTRA) || defined(OPENSSL_ALL)
#ifndef NO_ASN_TIME
#ifndef NO_BIO
int wolfSSL_ASN1_TIME_print(WOLFSSL_BIO* bio, const WOLFSSL_ASN1_TIME* asnTime)
{
    char buf[MAX_TIME_STRING_SZ];
    int  ret = WOLFSSL_SUCCESS;

    WOLFSSL_ENTER("wolfSSL_ASN1_TIME_print");

    if (bio == NULL || asnTime == NULL) {
        WOLFSSL_MSG("NULL function argument");
        return WOLFSSL_FAILURE;
    }

    if (wolfSSL_ASN1_TIME_to_string((WOLFSSL_ASN1_TIME*)asnTime, buf,
                sizeof(buf)) == NULL) {
        XMEMSET(buf, 0, MAX_TIME_STRING_SZ);
        XSTRNCPY(buf, "Bad time value", sizeof(buf)-1);
        ret = WOLFSSL_FAILURE;
    }

    if (wolfSSL_BIO_write(bio, buf, (int)XSTRLEN(buf)) <= 0) {
        WOLFSSL_MSG("Unable to write to bio");
        return WOLFSSL_FAILURE;
    }

    return ret;
}
#endif /* !NO_BIO */

char* wolfSSL_ASN1_TIME_to_string(WOLFSSL_ASN1_TIME* t, char* buf, int len)
{
    WOLFSSL_ENTER("wolfSSL_ASN1_TIME_to_string");

    if (t == NULL || buf == NULL || len < 5) {
        WOLFSSL_MSG("Bad argument");
        return NULL;
    }

    if (t->length > len) {
        WOLFSSL_MSG("Length of date is longer then buffer");
        return NULL;
    }

    if (!GetTimeString(t->data, t->type, buf, len)) {
        return NULL;
    }

    return buf;
}

/* Converts a WOLFSSL_ASN1_TIME to a struct tm. Returns WOLFSSL_SUCCESS on
 * success and WOLFSSL_FAILURE on failure. */
static int Asn1TimeToTm(WOLFSSL_ASN1_TIME* asnTime, struct tm* tm)
{
    unsigned char* asn1TimeBuf;
    int asn1TimeBufLen;
    int i = 0;
    int bytesNeeded = 11;

    if (asnTime == NULL) {
        WOLFSSL_MSG("asnTime is NULL");
        return WOLFSSL_FAILURE;
    }
    if (tm == NULL) {
        WOLFSSL_MSG("tm is NULL");
        return WOLFSSL_FAILURE;
    }

    asn1TimeBuf = wolfSSL_ASN1_TIME_get_data(asnTime);
    if (asn1TimeBuf == NULL) {
        WOLFSSL_MSG("Failed to get WOLFSSL_ASN1_TIME buffer.");
        return WOLFSSL_FAILURE;
    }
    asn1TimeBufLen = wolfSSL_ASN1_TIME_get_length(asnTime);
    if (asn1TimeBufLen <= 0) {
        WOLFSSL_MSG("Failed to get WOLFSSL_ASN1_TIME buffer length.");
        return WOLFSSL_FAILURE;
    }
    XMEMSET(tm, 0, sizeof(struct tm));

    /* Convert ASN1_time to struct tm */
    /* Check type */
    if (asnTime->type == ASN_UTC_TIME) {
        /* 2-digit year */
        bytesNeeded += 2;
        if (bytesNeeded > asn1TimeBufLen) {
            WOLFSSL_MSG("WOLFSSL_ASN1_TIME buffer length is invalid.");
            return WOLFSSL_FAILURE;
        }
        if (asn1TimeBuf[bytesNeeded-1] != 'Z') {
            WOLFSSL_MSG("Expecting UTC time.");
            return WOLFSSL_FAILURE;
        }

        tm->tm_year = (asn1TimeBuf[i] - '0') * 10; i++;
        tm->tm_year += asn1TimeBuf[i] - '0'; i++;
        if (tm->tm_year < 70) {
            tm->tm_year += 100;
        }
    }
    else if (asnTime->type == ASN_GENERALIZED_TIME) {
        /* 4-digit year */
        bytesNeeded += 4;
        if (bytesNeeded > asn1TimeBufLen) {
            WOLFSSL_MSG("WOLFSSL_ASN1_TIME buffer length is invalid.");
            return WOLFSSL_FAILURE;
        }
        if (asn1TimeBuf[bytesNeeded-1] != 'Z') {
            WOLFSSL_MSG("Expecting UTC time.");
            return WOLFSSL_FAILURE;
        }

        tm->tm_year = (asn1TimeBuf[i] - '0') * 1000; i++;
        tm->tm_year += (asn1TimeBuf[i] - '0') * 100; i++;
        tm->tm_year += (asn1TimeBuf[i] - '0') * 10; i++;
        tm->tm_year += asn1TimeBuf[i] - '0'; i++;
        tm->tm_year -= 1900;
    }
    else {
        WOLFSSL_MSG("asnTime->type is invalid.");
        return WOLFSSL_FAILURE;
    }

    tm->tm_mon = (asn1TimeBuf[i] - '0') * 10; i++;
    tm->tm_mon += (asn1TimeBuf[i] - '0') - 1; i++; /* January is 0 not 1 */
    tm->tm_mday = (asn1TimeBuf[i] - '0') * 10; i++;
    tm->tm_mday += (asn1TimeBuf[i] - '0'); i++;
    tm->tm_hour = (asn1TimeBuf[i] - '0') * 10; i++;
    tm->tm_hour += (asn1TimeBuf[i] - '0'); i++;
    tm->tm_min = (asn1TimeBuf[i] - '0') * 10; i++;
    tm->tm_min += (asn1TimeBuf[i] - '0'); i++;
    tm->tm_sec = (asn1TimeBuf[i] - '0') * 10; i++;
    tm->tm_sec += (asn1TimeBuf[i] - '0');

#ifdef XMKTIME
    /* Call XMKTIME on tm to get the tm_wday and tm_yday fields populated. */
    XMKTIME(tm);
#endif

    return WOLFSSL_SUCCESS;
}

int wolfSSL_ASN1_TIME_to_tm(const WOLFSSL_ASN1_TIME* asnTime, struct tm* tm)
{
    time_t currentTime;
    struct tm *tmpTs;
#if defined(NEED_TMP_TIME)
    /* for use with gmtime_r */
    struct tm tmpTimeStorage;
    tmpTs = &tmpTimeStorage;
#else
    tmpTs = NULL;
#endif
    (void)tmpTs;

    WOLFSSL_ENTER("wolfSSL_ASN1_TIME_to_tm");

    /* If asnTime is NULL, then the current time is converted. */
    if (asnTime == NULL) {
        if (tm == NULL) {
            WOLFSSL_MSG("asnTime and tm are both NULL");
            return WOLFSSL_FAILURE;
        }

        currentTime = wc_Time(0);
        if (currentTime <= 0) {
            WOLFSSL_MSG("Failed to get current time.");
            return WOLFSSL_FAILURE;
        }

        tm = XGMTIME(&currentTime, tmpTs);
        if (tm == NULL) {
            WOLFSSL_MSG("Failed to convert current time to UTC.");
            return WOLFSSL_FAILURE;
        }

        return WOLFSSL_SUCCESS;
    }

    /* If tm is NULL this function performs a format check on asnTime only. */
    if (tm == NULL) {
        return wolfSSL_ASN1_TIME_check(asnTime);
    }

    return Asn1TimeToTm((WOLFSSL_ASN1_TIME*)asnTime, tm);
}
#endif /* !NO_ASN_TIME */
#endif /* WOLFSSL_MYSQL_COMPATIBLE || WOLFSSL_NGINX || WOLFSSL_HAPROXY ||
    OPENSSL_EXTRA*/


#ifdef OPENSSL_EXTRA

int wolfSSL_ASN1_INTEGER_cmp(const WOLFSSL_ASN1_INTEGER* a,
                             const WOLFSSL_ASN1_INTEGER* b)
{
    int ret = 0;

    WOLFSSL_ENTER("wolfSSL_ASN1_INTEGER_cmp");

    if (a == NULL || b == NULL) {
        WOLFSSL_MSG("Bad parameter.");
        ret = WOLFSSL_FATAL_ERROR;
    }

    if (ret == 0 && ((a->length != b->length) ||
                     ((a->negative == 0) != (b->negative == 0)))) {
        ret = WOLFSSL_FATAL_ERROR;
    }
    if (ret == 0) {
        ret = XMEMCMP(a->data, b->data, a->length);
    }

    WOLFSSL_LEAVE("wolfSSL_ASN1_INTEGER_cmp", ret);

    return ret;
}

long wolfSSL_ASN1_INTEGER_get(const WOLFSSL_ASN1_INTEGER* a)
{
    long ret = 1;
    WOLFSSL_BIGNUM* bn = NULL;

    WOLFSSL_ENTER("ASN1_INTEGER_get");

    if (a == NULL) {
        /* OpenSSL returns 0 when a is NULL and -1 if there is an error. Quoting
         * the documentation:
         *
         * "ASN1_INTEGER_get() also returns the value of a but it returns 0 if a
         * is NULL and -1 on error (which is ambiguous because -1 is a
         * legitimate value for an ASN1_INTEGER). New applications should use
         * ASN1_INTEGER_get_int64() instead."
         * */
        ret = 0;
    }

    if (ret > 0) {
        bn = wolfSSL_ASN1_INTEGER_to_BN(a, NULL);
        if (bn == NULL) {
            ret = -1;
        }
    }
    if (ret > 0) {
        ret = wolfSSL_BN_get_word(bn);
        if (a->negative == 1) {
            ret = -ret;
        }
    }

    if (bn != NULL) {
        wolfSSL_BN_free(bn);
    }

    WOLFSSL_LEAVE("ASN1_INTEGER_get", (int)ret);

    return ret;
}

#endif /* OPENSSL_EXTRA */

#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_WPAS_SMALL)
/* Gets an index to store SSL structure at.
 *
 * Returns positive index on success and negative values on failure
 */
int wolfSSL_get_ex_data_X509_STORE_CTX_idx(void)
{
    WOLFSSL_ENTER("wolfSSL_get_ex_data_X509_STORE_CTX_idx");

    /* store SSL at index 0 */
    return 0;
}
#endif /* OPENSSL_EXTRA || WOLFSSL_WPAS_SMALL */

#ifdef OPENSSL_EXTRA
/* Sets a function callback that will send information about the state of all
 * WOLFSSL objects that have been created by the WOLFSSL_CTX structure passed
 * in.
 *
 * ctx WOLFSSL_CTX structure to set callback function in
 * f   callback function to use
 */
void wolfSSL_CTX_set_info_callback(WOLFSSL_CTX* ctx,
       void (*f)(const WOLFSSL* ssl, int type, int val))
{
    WOLFSSL_ENTER("wolfSSL_CTX_set_info_callback");
    if (ctx == NULL) {
        WOLFSSL_MSG("Bad function argument");
    }
    else {
        ctx->CBIS = f;
    }
}


unsigned long wolfSSL_ERR_peek_error(void)
{
    WOLFSSL_ENTER("wolfSSL_ERR_peek_error");

    return wolfSSL_ERR_peek_error_line_data(NULL, NULL, NULL, NULL);
}

int wolfSSL_ERR_GET_LIB(unsigned long err)
{
    unsigned long value;

    value = (err & 0xFFFFFFL);
    switch (value) {
    case -SSL_R_HTTP_REQUEST:
        return ERR_LIB_SSL;
    case PEM_R_NO_START_LINE:
    case PEM_R_PROBLEMS_GETTING_PASSWORD:
    case PEM_R_BAD_PASSWORD_READ:
    case PEM_R_BAD_DECRYPT:
        return ERR_LIB_PEM;
    case EVP_R_BAD_DECRYPT:
    case EVP_R_BN_DECODE_ERROR:
    case EVP_R_DECODE_ERROR:
    case EVP_R_PRIVATE_KEY_DECODE_ERROR:
        return ERR_LIB_EVP;
    case ASN1_R_HEADER_TOO_LONG:
        return ERR_LIB_ASN1;
    default:
        return 0;
    }
}

/* This function is to find global error values that are the same through out
 * all library version. With wolfSSL having only one set of error codes the
 * return value is pretty straight forward. The only thing needed is all wolfSSL
 * error values are typically negative.
 *
 * Returns the error reason
 */
int wolfSSL_ERR_GET_REASON(unsigned long err)
{
    int ret = (int)err;

    WOLFSSL_ENTER("wolfSSL_ERR_GET_REASON");

#if defined(OPENSSL_ALL) || defined(WOLFSSL_NGINX) || defined(WOLFSSL_HAPROXY)
    /* Nginx looks for this error to know to stop parsing certificates. */
    if (err == ((ERR_LIB_PEM << 24) | PEM_R_NO_START_LINE))
        return PEM_R_NO_START_LINE;
    if (err == ((ERR_LIB_SSL << 24) | -SSL_R_HTTP_REQUEST))
        return SSL_R_HTTP_REQUEST;
#endif
#if defined(OPENSSL_ALL) && defined(WOLFSSL_PYTHON)
    if (err == ((ERR_LIB_ASN1 << 24) | ASN1_R_HEADER_TOO_LONG))
        return ASN1_R_HEADER_TOO_LONG;
#endif

    /* check if error value is in range of wolfSSL errors */
    ret = 0 - ret; /* setting as negative value */
    /* wolfCrypt range is less than MAX (-100)
       wolfSSL range is MIN (-300) and lower */
    if (ret < MAX_CODE_E && ret > MIN_CODE_E) {
        return ret;
    }
    else {
        WOLFSSL_MSG("Not in range of typical error values");
        ret = (int)err;
    }

    return ret;
}

/* returns a string that describes the alert
 *
 * alertID the alert value to look up
 */
const char* wolfSSL_alert_type_string_long(int alertID)
{
    WOLFSSL_ENTER("wolfSSL_alert_type_string_long");

    return AlertTypeToString(alertID);
}


const char* wolfSSL_alert_desc_string_long(int alertID)
{
    WOLFSSL_ENTER("wolfSSL_alert_desc_string_long");

    return AlertTypeToString(alertID);
}

/* Gets the current state of the WOLFSSL structure
 *
 * ssl WOLFSSL structure to get state of
 *
 * Returns a human readable string of the WOLFSSL structure state
 */
const char* wolfSSL_state_string_long(const WOLFSSL* ssl)
{

    static const char* OUTPUT_STR[14][6][3] = {
        {
            {"SSLv3 Initialization","SSLv3 Initialization","SSLv3 Initialization"},
            {"TLSv1 Initialization","TLSv2 Initialization","TLSv2 Initialization"},
            {"TLSv1_1 Initialization","TLSv1_1 Initialization","TLSv1_1 Initialization"},
            {"TLSv1_2 Initialization","TLSv1_2 Initialization","TLSv1_2 Initialization"},
            {"DTLSv1 Initialization","DTLSv1 Initialization","DTLSv1 Initialization"},
            {"DTLSv1_2 Initialization","DTLSv1_2 Initialization","DTLSv1_2 Initialization"},
        },
        {
            {"SSLv3 read Server Hello Verify Request",
             "SSLv3 write Server Hello Verify Request",
             "SSLv3 Server Hello Verify Request"},
            {"TLSv1 read Server Hello Verify Request",
             "TLSv1 write Server Hello Verify Request",
             "TLSv1 Server Hello Verify Request"},
            {"TLSv1_1 read Server Hello Verify Request",
            "TLSv1_1 write Server Hello Verify Request",
             "TLSv1_1 Server Hello Verify Request"},
            {"TLSv1_2 read Server Hello Verify Request",
            "TLSv1_2 write Server Hello Verify Request",
             "TLSv1_2 Server Hello Verify Request"},
            {"DTLSv1 read Server Hello Verify Request",
             "DTLSv1 write Server Hello Verify Request",
             "DTLSv1 Server Hello Verify Request"},
            {"DTLSv1_2 read Server Hello Verify Request",
             "DTLSv1_2 write Server Hello Verify Request",
             "DTLSv1_2 Server Hello Verify Request"},
        },
        {
            {"SSLv3 read Server Hello",
             "SSLv3 write Server Hello",
             "SSLv3 Server Hello"},
            {"TLSv1 read Server Hello",
             "TLSv1 write Server Hello",
             "TLSv1 Server Hello"},
            {"TLSv1_1 read Server Hello",
            "TLSv1_1 write Server Hello",
             "TLSv1_1 Server Hello"},
            {"TLSv1_2 read Server Hello",
            "TLSv1_2 write Server Hello",
             "TLSv1_2 Server Hello"},
            {"DTLSv1 read Server Hello",
            "DTLSv1 write Server Hello",
             "DTLSv1 Server Hello"},
            {"DTLSv1_2 read Server Hello"
             "DTLSv1_2 write Server Hello",
             "DTLSv1_2 Server Hello",
            },
        },
        {
            {"SSLv3 read Server Session Ticket",
             "SSLv3 write Server Session Ticket",
             "SSLv3 Server Session Ticket"},
            {"TLSv1 read Server Session Ticket",
             "TLSv1 write Server Session Ticket",
             "TLSv1 Server Session Ticket"},
            {"TLSv1_1 read Server Session Ticket",
             "TLSv1_1 write Server Session Ticket",
             "TLSv1_1 Server Session Ticket"},
            {"TLSv1_2 read Server Session Ticket",
             "TLSv1_2 write Server Session Ticket",
             "TLSv1_2 Server Session Ticket"},
            {"DTLSv1 read Server Session Ticket",
             "DTLSv1 write Server Session Ticket",
             "DTLSv1 Server Session Ticket"},
            {"DTLSv1_2 read Server Session Ticket",
             "DTLSv1_2 write Server Session Ticket",
             "DTLSv1_2 Server Session Ticket"},
        },
        {
            {"SSLv3 read Server Cert",
             "SSLv3 write Server Cert",
             "SSLv3 Server Cert"},
            {"TLSv1 read Server Cert",
             "TLSv1 write Server Cert",
             "TLSv1 Server Cert"},
            {"TLSv1_1 read Server Cert",
             "TLSv1_1 write Server Cert",
             "TLSv1_1 Server Cert"},
            {"TLSv1_2 read Server Cert",
             "TLSv1_2 write Server Cert",
             "TLSv1_2 Server Cert"},
            {"DTLSv1 read Server Cert",
             "DTLSv1 write Server Cert",
             "DTLSv1 Server Cert"},
            {"DTLSv1_2 read Server Cert",
             "DTLSv1_2 write Server Cert",
             "DTLSv1_2 Server Cert"},
        },
        {
            {"SSLv3 read Server Key Exchange",
             "SSLv3 write Server Key Exchange",
             "SSLv3 Server Key Exchange"},
            {"TLSv1 read Server Key Exchange",
             "TLSv1 write Server Key Exchange",
             "TLSv1 Server Key Exchange"},
            {"TLSv1_1 read Server Key Exchange",
             "TLSv1_1 write Server Key Exchange",
             "TLSv1_1 Server Key Exchange"},
            {"TLSv1_2 read Server Key Exchange",
             "TLSv1_2 write Server Key Exchange",
             "TLSv1_2 Server Key Exchange"},
            {"DTLSv1 read Server Key Exchange",
             "DTLSv1 write Server Key Exchange",
             "DTLSv1 Server Key Exchange"},
            {"DTLSv1_2 read Server Key Exchange",
             "DTLSv1_2 write Server Key Exchange",
             "DTLSv1_2 Server Key Exchange"},
        },
        {
            {"SSLv3 read Server Hello Done",
             "SSLv3 write Server Hello Done",
             "SSLv3 Server Hello Done"},
            {"TLSv1 read Server Hello Done",
             "TLSv1 write Server Hello Done",
             "TLSv1 Server Hello Done"},
            {"TLSv1_1 read Server Hello Done",
             "TLSv1_1 write Server Hello Done",
             "TLSv1_1 Server Hello Done"},
            {"TLSv1_2 read Server Hello Done",
             "TLSv1_2 write Server Hello Done",
             "TLSv1_2 Server Hello Done"},
            {"DTLSv1 read Server Hello Done",
             "DTLSv1 write Server Hello Done",
             "DTLSv1 Server Hello Done"},
            {"DTLSv1_2 read Server Hello Done",
             "DTLSv1_2 write Server Hello Done",
             "DTLSv1_2 Server Hello Done"},
        },
        {
            {"SSLv3 read Server Change CipherSpec",
             "SSLv3 write Server Change CipherSpec",
             "SSLv3 Server Change CipherSpec"},
            {"TLSv1 read Server Change CipherSpec",
             "TLSv1 write Server Change CipherSpec",
             "TLSv1 Server Change CipherSpec"},
            {"TLSv1_1 read Server Change CipherSpec",
             "TLSv1_1 write Server Change CipherSpec",
             "TLSv1_1 Server Change CipherSpec"},
            {"TLSv1_2 read Server Change CipherSpec",
             "TLSv1_2 write Server Change CipherSpec",
             "TLSv1_2 Server Change CipherSpec"},
            {"DTLSv1 read Server Change CipherSpec",
             "DTLSv1 write Server Change CipherSpec",
             "DTLSv1 Server Change CipherSpec"},
            {"DTLSv1_2 read Server Change CipherSpec",
             "DTLSv1_2 write Server Change CipherSpec",
             "DTLSv1_2 Server Change CipherSpec"},
        },
        {
            {"SSLv3 read Server Finished",
             "SSLv3 write Server Finished",
             "SSLv3 Server Finished"},
            {"TLSv1 read Server Finished",
             "TLSv1 write Server Finished",
             "TLSv1 Server Finished"},
            {"TLSv1_1 read Server Finished",
             "TLSv1_1 write Server Finished",
             "TLSv1_1 Server Finished"},
            {"TLSv1_2 read Server Finished",
             "TLSv1_2 write Server Finished",
             "TLSv1_2 Server Finished"},
            {"DTLSv1 read Server Finished",
             "DTLSv1 write Server Finished",
             "DTLSv1 Server Finished"},
            {"DTLSv1_2 read Server Finished",
             "DTLSv1_2 write Server Finished",
             "DTLSv1_2 Server Finished"},
        },
        {
            {"SSLv3 read Client Hello",
             "SSLv3 write Client Hello",
             "SSLv3 Client Hello"},
            {"TLSv1 read Client Hello",
             "TLSv1 write Client Hello",
             "TLSv1 Client Hello"},
            {"TLSv1_1 read Client Hello",
             "TLSv1_1 write Client Hello",
             "TLSv1_1 Client Hello"},
            {"TLSv1_2 read Client Hello",
             "TLSv1_2 write Client Hello",
             "TLSv1_2 Client Hello"},
            {"DTLSv1 read Client Hello",
             "DTLSv1 write Client Hello",
             "DTLSv1 Client Hello"},
            {"DTLSv1_2 read Client Hello",
             "DTLSv1_2 write Client Hello",
             "DTLSv1_2 Client Hello"},
        },
        {
            {"SSLv3 read Client Key Exchange",
             "SSLv3 write Client Key Exchange",
             "SSLv3 Client Key Exchange"},
            {"TLSv1 read Client Key Exchange",
             "TLSv1 write Client Key Exchange",
             "TLSv1 Client Key Exchange"},
            {"TLSv1_1 read Client Key Exchange",
             "TLSv1_1 write Client Key Exchange",
             "TLSv1_1 Client Key Exchange"},
            {"TLSv1_2 read Client Key Exchange",
             "TLSv1_2 write Client Key Exchange",
             "TLSv1_2 Client Key Exchange"},
            {"DTLSv1 read Client Key Exchange",
             "DTLSv1 write Client Key Exchange",
             "DTLSv1 Client Key Exchange"},
            {"DTLSv1_2 read Client Key Exchange",
             "DTLSv1_2 write Client Key Exchange",
             "DTLSv1_2 Client Key Exchange"},
        },
        {
            {"SSLv3 read Client Change CipherSpec",
             "SSLv3 write Client Change CipherSpec",
             "SSLv3 Client Change CipherSpec"},
            {"TLSv1 read Client Change CipherSpec",
             "TLSv1 write Client Change CipherSpec",
             "TLSv1 Client Change CipherSpec"},
            {"TLSv1_1 read Client Change CipherSpec",
             "TLSv1_1 write Client Change CipherSpec",
             "TLSv1_1 Client Change CipherSpec"},
            {"TLSv1_2 read Client Change CipherSpec",
             "TLSv1_2 write Client Change CipherSpec",
             "TLSv1_2 Client Change CipherSpec"},
            {"DTLSv1 read Client Change CipherSpec",
             "DTLSv1 write Client Change CipherSpec",
             "DTLSv1 Client Change CipherSpec"},
            {"DTLSv1_2 read Client Change CipherSpec",
             "DTLSv1_2 write Client Change CipherSpec",
             "DTLSv1_2 Client Change CipherSpec"},
        },
        {
            {"SSLv3 read Client Finished",
             "SSLv3 write Client Finished",
             "SSLv3 Client Finished"},
            {"TLSv1 read Client Finished",
             "TLSv1 write Client Finished",
             "TLSv1 Client Finished"},
            {"TLSv1_1 read Client Finished",
             "TLSv1_1 write Client Finished",
             "TLSv1_1 Client Finished"},
            {"TLSv1_2 read Client Finished",
             "TLSv1_2 write Client Finished",
             "TLSv1_2 Client Finished"},
            {"DTLSv1 read Client Finished",
             "DTLSv1 write Client Finished",
             "DTLSv1 Client Finished"},
            {"DTLSv1_2 read Client Finished",
             "DTLSv1_2 write Client Finished",
             "DTLSv1_2 Client Finished"},
        },
        {
            {"SSLv3 Handshake Done",
             "SSLv3 Handshake Done",
             "SSLv3 Handshake Done"},
            {"TLSv1 Handshake Done",
             "TLSv1 Handshake Done",
             "TLSv1 Handshake Done"},
            {"TLSv1_1 Handshake Done",
             "TLSv1_1 Handshake Done",
             "TLSv1_1 Handshake Done"},
            {"TLSv1_2 Handshake Done",
             "TLSv1_2 Handshake Done",
             "TLSv1_2 Handshake Done"},
            {"DTLSv1 Handshake Done",
             "DTLSv1 Handshake Done",
             "DTLSv1 Handshake Done"},
            {"DTLSv1_2 Handshake Done"
             "DTLSv1_2 Handshake Done"
             "DTLSv1_2 Handshake Done"}
        }
    };
    enum ProtocolVer {
        SSL_V3 = 0,
        TLS_V1,
        TLS_V1_1,
        TLS_V1_2,
        DTLS_V1,
        DTLS_V1_2,
        UNKNOWN = 100
    };

    enum IOMode {
        SS_READ = 0,
        SS_WRITE,
        SS_NEITHER
    };

    enum SslState {
        ss_null_state = 0,
        ss_server_helloverify,
        ss_server_hello,
        ss_sessionticket,
        ss_server_cert,
        ss_server_keyexchange,
        ss_server_hellodone,
        ss_server_changecipherspec,
        ss_server_finished,
        ss_client_hello,
        ss_client_keyexchange,
        ss_client_changecipherspec,
        ss_client_finished,
        ss_handshake_done
    };

    int protocol = 0;
    int cbmode = 0;
    int state = 0;

    WOLFSSL_ENTER("wolfSSL_state_string_long");
    if (ssl == NULL) {
        WOLFSSL_MSG("Null argument passed in");
        return NULL;
    }

    /* Get state of callback */
    if (ssl->cbmode == SSL_CB_MODE_WRITE){
        cbmode =  SS_WRITE;
    } else if (ssl->cbmode == SSL_CB_MODE_READ){
        cbmode =  SS_READ;
    } else {
        cbmode =  SS_NEITHER;
    }

    /* Get protocol version */
    switch (ssl->version.major){
        case SSLv3_MAJOR:
            switch (ssl->version.minor){
                case TLSv1_MINOR:
                    protocol = TLS_V1;
                    break;
                case TLSv1_1_MINOR:
                    protocol = TLS_V1_1;
                    break;
                case TLSv1_2_MINOR:
                    protocol = TLS_V1_2;
                    break;
                case SSLv3_MINOR:
                    protocol = SSL_V3;
                    break;
                default:
                    protocol = UNKNOWN;
            }
            break;
        case DTLS_MAJOR:
            switch (ssl->version.minor){
        case DTLS_MINOR:
            protocol = DTLS_V1;
            break;
        case DTLSv1_2_MINOR:
            protocol = DTLS_V1_2;
            break;
        default:
            protocol = UNKNOWN;
    }
    break;
    default:
        protocol = UNKNOWN;
    }

    /* accept process */
    if (ssl->cbmode == SSL_CB_MODE_READ){
        state = ssl->cbtype;
        switch (state) {
            case hello_verify_request:
                state = ss_server_helloverify;
                break;
            case session_ticket:
                state = ss_sessionticket;
                break;
            case server_hello:
                state = ss_server_hello;
                break;
            case server_hello_done:
                state = ss_server_hellodone;
                break;
            case certificate:
                state = ss_server_cert;
                break;
            case server_key_exchange:
                state = ss_server_keyexchange;
                break;
            case client_hello:
                state = ss_client_hello;
                break;
            case client_key_exchange:
                state = ss_client_keyexchange;
                break;
            case finished:
                if (ssl->options.side == WOLFSSL_SERVER_END)
                    state = ss_client_finished;
                else if (ssl->options.side == WOLFSSL_CLIENT_END)
                    state = ss_server_finished;
                else {
                    WOLFSSL_MSG("Unknown State");
                    state = ss_null_state;
                }
                break;
            default:
                WOLFSSL_MSG("Unknown State");
                state = ss_null_state;
        }
    } else {
        /* Send process */
        if (ssl->options.side == WOLFSSL_SERVER_END)
            state = ssl->options.serverState;
        else
            state = ssl->options.clientState;

        switch(state){
            case SERVER_HELLOVERIFYREQUEST_COMPLETE:
                state = ss_server_helloverify;
                break;
            case SERVER_HELLO_COMPLETE:
                state = ss_server_hello;
                break;
            case SERVER_CERT_COMPLETE:
                state = ss_server_cert;
                break;
            case SERVER_KEYEXCHANGE_COMPLETE:
                state = ss_server_keyexchange;
                break;
            case SERVER_HELLODONE_COMPLETE:
                state = ss_server_hellodone;
                break;
            case SERVER_CHANGECIPHERSPEC_COMPLETE:
                state = ss_server_changecipherspec;
                break;
            case SERVER_FINISHED_COMPLETE:
                state = ss_server_finished;
                break;
            case CLIENT_HELLO_COMPLETE:
                state = ss_client_hello;
                break;
            case CLIENT_KEYEXCHANGE_COMPLETE:
                state = ss_client_keyexchange;
                break;
            case CLIENT_CHANGECIPHERSPEC_COMPLETE:
                state = ss_client_changecipherspec;
                break;
            case CLIENT_FINISHED_COMPLETE:
                state = ss_client_finished;
                break;
            case HANDSHAKE_DONE:
                state = ss_handshake_done;
                break;
            default:
                WOLFSSL_MSG("Unknown State");
                state = ss_null_state;
        }
    }

    if (protocol == UNKNOWN) {
        WOLFSSL_MSG("Unknown protocol");
        return "";
    }
    else {
        return OUTPUT_STR[state][protocol][cbmode];
    }
}

/*
 * Sets default PEM callback password if null is passed into
 * the callback parameter of a PEM_read_bio_* function.
 *
 * Returns callback phrase size on success or WOLFSSL_FAILURE otherwise.
 */
int wolfSSL_PEM_def_callback(char* name, int num, int w, void* key)
{
    int sz;
    (void)w;
    WOLFSSL_ENTER("wolfSSL_PEM_def_callback");

    /* We assume that the user passes a default password as userdata */
    if (key) {
        sz = (int)XSTRLEN((const char*)key);
        sz = (sz > num) ? num : sz;
        XMEMCPY(name, key, sz);
        return sz;
    } else {
        WOLFSSL_MSG("Error, default password cannot be created.");
        return WOLFSSL_FAILURE;
    }
}

#endif /* OPENSSL_EXTRA */

static long wolf_set_options(long old_op, long op)
{
    /* if SSL_OP_ALL then turn all bug workarounds on */
    if ((op & WOLFSSL_OP_ALL) == WOLFSSL_OP_ALL) {
        WOLFSSL_MSG("\tSSL_OP_ALL");
    }

    /* by default cookie exchange is on with DTLS */
    if ((op & WOLFSSL_OP_COOKIE_EXCHANGE) == WOLFSSL_OP_COOKIE_EXCHANGE) {
        WOLFSSL_MSG("\tSSL_OP_COOKIE_EXCHANGE : on by default");
    }

    if ((op & WOLFSSL_OP_NO_SSLv2) == WOLFSSL_OP_NO_SSLv2) {
        WOLFSSL_MSG("\tWOLFSSL_OP_NO_SSLv2 : wolfSSL does not support SSLv2");
    }

#ifdef SSL_OP_NO_TLSv1_3
    if ((op & WOLFSSL_OP_NO_TLSv1_3) == WOLFSSL_OP_NO_TLSv1_3) {
        WOLFSSL_MSG("\tSSL_OP_NO_TLSv1_3");
    }
#endif

    if ((op & WOLFSSL_OP_NO_TLSv1_2) == WOLFSSL_OP_NO_TLSv1_2) {
        WOLFSSL_MSG("\tSSL_OP_NO_TLSv1_2");
    }

    if ((op & WOLFSSL_OP_NO_TLSv1_1) == WOLFSSL_OP_NO_TLSv1_1) {
        WOLFSSL_MSG("\tSSL_OP_NO_TLSv1_1");
    }

    if ((op & WOLFSSL_OP_NO_TLSv1) == WOLFSSL_OP_NO_TLSv1) {
        WOLFSSL_MSG("\tSSL_OP_NO_TLSv1");
    }

    if ((op & WOLFSSL_OP_NO_SSLv3) == WOLFSSL_OP_NO_SSLv3) {
        WOLFSSL_MSG("\tSSL_OP_NO_SSLv3");
    }

    if ((op & WOLFSSL_OP_CIPHER_SERVER_PREFERENCE) ==
            WOLFSSL_OP_CIPHER_SERVER_PREFERENCE) {
        WOLFSSL_MSG("\tWOLFSSL_OP_CIPHER_SERVER_PREFERENCE");
    }

    if ((op & WOLFSSL_OP_NO_COMPRESSION) == WOLFSSL_OP_NO_COMPRESSION) {
    #ifdef HAVE_LIBZ
        WOLFSSL_MSG("SSL_OP_NO_COMPRESSION");
    #else
        WOLFSSL_MSG("SSL_OP_NO_COMPRESSION: compression not compiled in");
    #endif
    }

    return old_op | op;
}

#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_WPAS_SMALL)
long wolfSSL_set_options(WOLFSSL* ssl, long op)
{
    word16 haveRSA = 1;
    word16 havePSK = 0;
    int    keySz   = 0;

    WOLFSSL_ENTER("wolfSSL_set_options");

    if (ssl == NULL) {
        return 0;
    }

    ssl->options.mask = wolf_set_options(ssl->options.mask, op);

#ifdef SSL_OP_NO_TLSv1_3
    if ((ssl->options.mask & SSL_OP_NO_TLSv1_3) == SSL_OP_NO_TLSv1_3) {
        if (ssl->version.minor == TLSv1_3_MINOR)
            ssl->version.minor = TLSv1_2_MINOR;
    }
#endif

    if ((ssl->options.mask & SSL_OP_NO_TLSv1_2) == SSL_OP_NO_TLSv1_2) {
        if (ssl->version.minor == TLSv1_2_MINOR)
            ssl->version.minor = TLSv1_1_MINOR;
    }

    if ((ssl->options.mask & SSL_OP_NO_TLSv1_1) == SSL_OP_NO_TLSv1_1) {
        if (ssl->version.minor == TLSv1_1_MINOR)
            ssl->version.minor = TLSv1_MINOR;
    }

    if ((ssl->options.mask & SSL_OP_NO_TLSv1) == SSL_OP_NO_TLSv1) {
        if (ssl->version.minor == TLSv1_MINOR)
            ssl->version.minor = SSLv3_MINOR;
    }

    if ((ssl->options.mask & WOLFSSL_OP_NO_COMPRESSION) == WOLFSSL_OP_NO_COMPRESSION) {
    #ifdef HAVE_LIBZ
        ssl->options.usingCompression = 0;
    #endif
    }

    /* in the case of a version change the cipher suites should be reset */
#ifndef NO_PSK
    havePSK = ssl->options.havePSK;
#endif
#ifdef NO_RSA
    haveRSA = 0;
#endif
#ifndef NO_CERTS
    keySz = ssl->buffers.keySz;
#endif

    if (ssl->suites != NULL && ssl->options.side != WOLFSSL_NEITHER_END)
        InitSuites(ssl->suites, ssl->version, keySz, haveRSA, havePSK,
                   ssl->options.haveDH, ssl->options.haveECDSAsig,
                   ssl->options.haveECC, ssl->options.haveStaticECC,
                   ssl->options.haveFalconSig, ssl->options.haveAnon,
                   ssl->options.side);

    return ssl->options.mask;
}


long wolfSSL_get_options(const WOLFSSL* ssl)
{
    WOLFSSL_ENTER("wolfSSL_get_options");
    if(ssl == NULL)
        return WOLFSSL_FAILURE;
    return ssl->options.mask;
}

#endif /* OPENSSL_EXTRA || WOLFSSL_WPAS_SMALL */

#if defined(HAVE_SECURE_RENEGOTIATION) \
        || defined(HAVE_SERVER_RENEGOTIATION_INFO)
/* clears the counter for number of renegotiations done
 * returns the current count before it is cleared */
long wolfSSL_clear_num_renegotiations(WOLFSSL *s)
{
    long total;

    WOLFSSL_ENTER("wolfSSL_clear_num_renegotiations");
    if (s == NULL)
        return 0;

    total = s->secure_rene_count;
    s->secure_rene_count = 0;
    return total;
}


/* return the number of renegotiations since wolfSSL_new */
long wolfSSL_total_renegotiations(WOLFSSL *s)
{
    WOLFSSL_ENTER("wolfSSL_total_renegotiations");
    return wolfSSL_num_renegotiations(s);
}


/* return the number of renegotiations since wolfSSL_new */
long wolfSSL_num_renegotiations(WOLFSSL* s)
{
    if (s == NULL) {
        return 0;
    }

    return s->secure_rene_count;
}


/* Is there a renegotiation currently in progress? */
int  wolfSSL_SSL_renegotiate_pending(WOLFSSL *s)
{
    return s && s->options.handShakeDone &&
            s->options.handShakeState != HANDSHAKE_DONE ? 1 : 0;
}
#endif /* HAVE_SECURE_RENEGOTIATION || HAVE_SERVER_RENEGOTIATION_INFO */

#ifdef OPENSSL_EXTRA

long wolfSSL_clear_options(WOLFSSL* ssl, long opt)
{
    WOLFSSL_ENTER("SSL_clear_options");
    if(ssl == NULL)
        return WOLFSSL_FAILURE;
    ssl->options.mask &= ~opt;
    return ssl->options.mask;
}


#ifndef NO_DH
long wolfSSL_set_tmp_dh(WOLFSSL *ssl, WOLFSSL_DH *dh)
{
    int pSz, gSz;
    byte *p, *g;
    int ret = 0;

    WOLFSSL_ENTER("wolfSSL_set_tmp_dh");

    if (!ssl || !dh)
        return BAD_FUNC_ARG;

    /* Get needed size for p and g */
    pSz = wolfSSL_BN_bn2bin(dh->p, NULL);
    gSz = wolfSSL_BN_bn2bin(dh->g, NULL);

    if (pSz <= 0 || gSz <= 0)
        return WOLFSSL_FATAL_ERROR;

    p = (byte*)XMALLOC(pSz, ssl->heap, DYNAMIC_TYPE_PUBLIC_KEY);
    if (!p)
        return MEMORY_E;

    g = (byte*)XMALLOC(gSz, ssl->heap, DYNAMIC_TYPE_PUBLIC_KEY);
    if (!g) {
        XFREE(p, ssl->heap, DYNAMIC_TYPE_PUBLIC_KEY);
        return MEMORY_E;
    }

    pSz = wolfSSL_BN_bn2bin(dh->p, p);
    gSz = wolfSSL_BN_bn2bin(dh->g, g);

    if (pSz >= 0 && gSz >= 0) /* Conversion successful */
        ret = wolfSSL_SetTmpDH(ssl, p, pSz, g, gSz);

    XFREE(p, ssl->heap, DYNAMIC_TYPE_PUBLIC_KEY);
    XFREE(g, ssl->heap, DYNAMIC_TYPE_PUBLIC_KEY);

    return pSz > 0 && gSz > 0 ? ret : WOLFSSL_FATAL_ERROR;
}
#endif /* !NO_DH */


#ifdef HAVE_PK_CALLBACKS
long wolfSSL_set_tlsext_debug_arg(WOLFSSL* ssl, void *arg)
{
    if (ssl == NULL) {
        return WOLFSSL_FAILURE;
    }

    ssl->loggingCtx = arg;
    return WOLFSSL_SUCCESS;
}
#endif /* HAVE_PK_CALLBACKS */

#if defined(OPENSSL_ALL) || defined(WOLFSSL_HAPROXY)
const unsigned char *SSL_SESSION_get0_id_context(const WOLFSSL_SESSION *sess, unsigned int *sid_ctx_length)
{
    sess = ClientSessionToSession(sess);
    return wolfSSL_SESSION_get_id((WOLFSSL_SESSION *)sess, sid_ctx_length);
}
#endif

/*** TBD ***/
#ifndef NO_WOLFSSL_STUB
WOLFSSL_API int wolfSSL_sk_SSL_COMP_zero(WOLFSSL_STACK* st)
{
    (void)st;
    WOLFSSL_STUB("wolfSSL_sk_SSL_COMP_zero");
    /* wolfSSL_set_options(ssl, SSL_OP_NO_COMPRESSION); */
    return WOLFSSL_FAILURE;
}
#endif

#ifdef HAVE_CERTIFICATE_STATUS_REQUEST
long wolfSSL_set_tlsext_status_type(WOLFSSL *s, int type)
{
    WOLFSSL_ENTER("wolfSSL_set_tlsext_status_type");

    if (s == NULL){
        return BAD_FUNC_ARG;
    }

    if (type == TLSEXT_STATUSTYPE_ocsp){
        int r = TLSX_UseCertificateStatusRequest(&s->extensions, (byte)type, 0, s,
                                                             s->heap, s->devId);
        return (long)r;
    } else {
        WOLFSSL_MSG(
       "SSL_set_tlsext_status_type only supports TLSEXT_STATUSTYPE_ocsp type.");
        return SSL_FAILURE;
    }

}

long wolfSSL_get_tlsext_status_type(WOLFSSL *s)
{
    TLSX* extension;

    if (s == NULL)
        return WOLFSSL_FATAL_ERROR;
    extension = TLSX_Find(s->extensions, TLSX_STATUS_REQUEST);
    return extension != NULL ? TLSEXT_STATUSTYPE_ocsp : WOLFSSL_FATAL_ERROR;
}
#endif /* HAVE_CERTIFICATE_STATUS_REQUEST */

#ifndef NO_WOLFSSL_STUB
WOLFSSL_API long wolfSSL_get_tlsext_status_exts(WOLFSSL *s, void *arg)
{
    (void)s;
    (void)arg;
    WOLFSSL_STUB("wolfSSL_get_tlsext_status_exts");
    return WOLFSSL_FAILURE;
}
#endif

/*** TBD ***/
#ifndef NO_WOLFSSL_STUB
WOLFSSL_API long wolfSSL_set_tlsext_status_exts(WOLFSSL *s, void *arg)
{
    (void)s;
    (void)arg;
    WOLFSSL_STUB("wolfSSL_set_tlsext_status_exts");
    return WOLFSSL_FAILURE;
}
#endif

/*** TBD ***/
#ifndef NO_WOLFSSL_STUB
WOLFSSL_API long wolfSSL_get_tlsext_status_ids(WOLFSSL *s, void *arg)
{
    (void)s;
    (void)arg;
    WOLFSSL_STUB("wolfSSL_get_tlsext_status_ids");
    return WOLFSSL_FAILURE;
}
#endif

/*** TBD ***/
#ifndef NO_WOLFSSL_STUB
WOLFSSL_API long wolfSSL_set_tlsext_status_ids(WOLFSSL *s, void *arg)
{
    (void)s;
    (void)arg;
    WOLFSSL_STUB("wolfSSL_set_tlsext_status_ids");
    return WOLFSSL_FAILURE;
}
#endif

/*** TBD ***/
#ifndef NO_WOLFSSL_STUB
WOLFSSL_API int SSL_SESSION_set1_id(WOLFSSL_SESSION *s, const unsigned char *sid, unsigned int sid_len)
{
    (void)s;
    (void)sid;
    (void)sid_len;
    WOLFSSL_STUB("SSL_SESSION_set1_id");
    return WOLFSSL_FAILURE;
}
#endif

#ifndef NO_WOLFSSL_STUB
/*** TBD ***/
WOLFSSL_API int SSL_SESSION_set1_id_context(WOLFSSL_SESSION *s, const unsigned char *sid_ctx, unsigned int sid_ctx_len)
{
    (void)s;
    (void)sid_ctx;
    (void)sid_ctx_len;
    WOLFSSL_STUB("SSL_SESSION_set1_id_context");
    return WOLFSSL_FAILURE;
}
#endif

#if defined(OPENSSL_ALL) || defined(WOLFSSL_APACHE_HTTPD) \
    || defined(WOLFSSL_HAPROXY) || defined(WOLFSSL_WPAS)
/**
 * Set `a` in a smart way.
 *
 * @param a Object to set
 * @param type The type of object in value
 * @param value Object to set
 */
void wolfSSL_ASN1_TYPE_set(WOLFSSL_ASN1_TYPE *a, int type, void *value)
{
    if (!a) {
        return;
    }
    switch (type) {
        case V_ASN1_NULL:
            a->value.ptr = (char *)value;
            break;
        case V_ASN1_SEQUENCE:
            a->value.asn1_string = (WOLFSSL_ASN1_STRING*)value;
            break;
        case V_ASN1_OBJECT:
            a->value.object = (WOLFSSL_ASN1_OBJECT*)value;
            break;
        case V_ASN1_UTCTIME:
            a->value.utctime = (WOLFSSL_ASN1_TIME*)value;
            break;
        case V_ASN1_GENERALIZEDTIME:
            a->value.generalizedtime = (WOLFSSL_ASN1_TIME*)value;
            break;
        default:
            WOLFSSL_MSG("Unknown or unsupported ASN1_TYPE");
            return;
    }
    a->type = type;
}

#endif /* OPENSSL_ALL || WOLFSSL_APACHE_HTTPD || WOLFSSL_HAPROXY || WOLFSSL_WPAS */

#if defined(OPENSSL_ALL) || defined(WOLFSSL_APACHE_HTTPD) \
    || defined(WOLFSSL_HAPROXY) || defined(WOLFSSL_WPAS) \
    || defined(OPENSSL_EXTRA)
/**
 * Allocate a new WOLFSSL_ASN1_TYPE object.
 *
 * @return New zero'ed WOLFSSL_ASN1_TYPE object
 */
WOLFSSL_ASN1_TYPE* wolfSSL_ASN1_TYPE_new(void)
{
    WOLFSSL_ASN1_TYPE* ret = (WOLFSSL_ASN1_TYPE*)XMALLOC(sizeof(WOLFSSL_ASN1_TYPE),
                                                        NULL, DYNAMIC_TYPE_OPENSSL);
    if (!ret)
        return NULL;
    XMEMSET(ret, 0, sizeof(WOLFSSL_ASN1_TYPE));
    return ret;
}

/**
 * Free WOLFSSL_ASN1_TYPE and all its members.
 *
 * @param at Object to free
 */
void wolfSSL_ASN1_TYPE_free(WOLFSSL_ASN1_TYPE* at)
{
    if (at) {
        switch (at->type) {
            case V_ASN1_OBJECT:
                wolfSSL_ASN1_OBJECT_free(at->value.object);
                break;
            case V_ASN1_UTCTIME:
            #ifndef NO_ASN_TIME
                wolfSSL_ASN1_TIME_free(at->value.utctime);
            #endif
                break;
            case V_ASN1_GENERALIZEDTIME:
            #ifndef NO_ASN_TIME
                wolfSSL_ASN1_TIME_free(at->value.generalizedtime);
            #endif
                break;
            case V_ASN1_UTF8STRING:
            case V_ASN1_PRINTABLESTRING:
            case V_ASN1_T61STRING:
            case V_ASN1_IA5STRING:
            case V_ASN1_UNIVERSALSTRING:
            case V_ASN1_SEQUENCE:
                wolfSSL_ASN1_STRING_free(at->value.asn1_string);
                break;
            default:
                WOLFSSL_MSG("Unknown or unsupported ASN1_TYPE");
                break;
        }
        XFREE(at, NULL, DYNAMIC_TYPE_OPENSSL);
    }
}
#endif /* OPENSSL_ALL || WOLFSSL_APACHE_HTTPD || WOLFSSL_HAPROXY || WOLFSSL_WPAS
        || OPENSSL_EXTRA */



#ifndef NO_WOLFSSL_STUB
/*** TBD ***/
WOLFSSL_API WOLFSSL_EVP_PKEY *wolfSSL_get_privatekey(const WOLFSSL *ssl)
{
    (void)ssl;
    WOLFSSL_STUB("SSL_get_privatekey");
    return NULL;
}
#endif

/**
 * Get a textual representation of given WOLFSSL_ASN1_OBJECT then write it to
 * buf at most buf_len bytes.
 *
 * params
 * - buf: buffer where the textual representation is to be written to
 * - buf_len: buffer size in bytes
 * - a: WOLFSSL_ASN1_OBJECT
 *
 * return the string length written on success, WOLFSSL_FAILURE on failure.
 */
WOLFSSL_API int wolfSSL_i2t_ASN1_OBJECT(char *buf, int buf_len,
                                                WOLFSSL_ASN1_OBJECT *a)
{
    WOLFSSL_ENTER("wolfSSL_i2t_ASN1_OBJECT");
    return wolfSSL_OBJ_obj2txt(buf, buf_len, a, 0);
}

WOLFSSL_ASN1_OBJECT *wolfSSL_d2i_ASN1_OBJECT(WOLFSSL_ASN1_OBJECT **a,
                                             const unsigned char **der,
                                             long length)
{
    const unsigned char *d;
    long len;
    int tag, cls;
    WOLFSSL_ASN1_OBJECT* ret = NULL;

    WOLFSSL_ENTER("wolfSSL_d2i_ASN1_OBJECT");

    if (!der || !*der || length <= 0) {
        WOLFSSL_MSG("Bad parameter");
        return NULL;
    }

    d = *der;

    if (wolfSSL_ASN1_get_object(&d, &len, &tag, &cls, length) & 0x80) {
        WOLFSSL_MSG("wolfSSL_ASN1_get_object error");
        return NULL;
    }
    /* d now points to value */

    if (tag != ASN_OBJECT_ID) {
        WOLFSSL_MSG("Not an ASN object");
        return NULL;
    }

    ret = wolfSSL_c2i_ASN1_OBJECT(a, &d, len);
    if (ret)
        *der = d;
    return ret;
}

/**
 * Parse an ASN1 encoded input and output information about the parsed object
 * @param in    ASN1 encoded data. *in is moved to the value of the ASN1 object
 * @param len   Length of parsed ASN1 object
 * @param tag   Tag value of parsed ASN1 object
 * @param cls   Class of parsed ASN1 object
 * @param inLen Length of *in buffer
 * @return int  Depends on which bits are set in the returned int:
 *              0x80 an error occurred during parsing
 *              0x20 parsed object is constructed
 *              0x01 the parsed object length is infinite
 */
int wolfSSL_ASN1_get_object(const unsigned char **in, long *len, int *tag,
                            int *cls, long inLen)
{
    word32 inOutIdx = 0;
    int l;
    byte t;
    int ret = 0x80;

    WOLFSSL_ENTER("wolfSSL_ASN1_get_object");

    if (!in || !*in || !len || !tag || !cls || inLen == 0) {
        WOLFSSL_MSG("Bad parameter");
        return ret;
    }

    if (GetASNTag(*in, &inOutIdx, &t, (word32)inLen) != 0) {
        WOLFSSL_MSG("GetASNTag error");
        return ret;
    }

    if (GetLength(*in, &inOutIdx, &l, (word32)inLen) < 0) {
        WOLFSSL_MSG("GetLength error");
        return ret;
    }

    *tag = t & 0x1F; /* Tag number is 5 lsb */
    *cls = t & 0xC0; /* Class is 2 msb */
    *len = l;
    ret = t & ASN_CONSTRUCTED;

    if (l > (int)(inLen - inOutIdx)) {
        /* Still return other values but indicate error in msb */
        ret |= 0x80;
    }

    *in += inOutIdx;
    return ret;
}

WOLFSSL_ASN1_OBJECT *wolfSSL_c2i_ASN1_OBJECT(WOLFSSL_ASN1_OBJECT **a,
        const unsigned char **pp, long len)
{
    WOLFSSL_ASN1_OBJECT* ret = NULL;

    WOLFSSL_ENTER("wolfSSL_c2i_ASN1_OBJECT");

    if (!pp || !*pp || len <= 0) {
        WOLFSSL_MSG("Bad parameter");
        return NULL;
    }

    if (!(ret = wolfSSL_ASN1_OBJECT_new())) {
        WOLFSSL_MSG("wolfSSL_ASN1_OBJECT_new error");
        return NULL;
    }

    ret->obj = (const unsigned char*)XMALLOC(len, NULL, DYNAMIC_TYPE_ASN1);
    if (!ret->obj) {
        WOLFSSL_MSG("error allocating asn data memory");
        wolfSSL_ASN1_OBJECT_free(ret);
        return NULL;
    }

    XMEMCPY((byte*)ret->obj, *pp, len);
    ret->objSz = (unsigned int)len;
    ret->dynamic |= WOLFSSL_ASN1_DYNAMIC_DATA;

    *pp += len;

    if (a)
        *a = ret;
    return ret;
}

#ifndef NO_BIO
/* Return number of bytes written to BIO on success. 0 on failure. */
WOLFSSL_API int wolfSSL_i2a_ASN1_OBJECT(WOLFSSL_BIO *bp,
                                        WOLFSSL_ASN1_OBJECT *a)
{
    int length = 0;
    word32 idx = 0;
    const char null_str[] = "NULL";

    WOLFSSL_ENTER("wolfSSL_i2a_ASN1_OBJECT");

    if (bp == NULL)
        return WOLFSSL_FAILURE;

    if (a == NULL) {
        /* Write "NULL" */
        if (wolfSSL_BIO_write(bp, null_str, (int)XSTRLEN(null_str)) ==
                (int)XSTRLEN(null_str)) {
            return (int)XSTRLEN(null_str);
        }
        else {
            return WOLFSSL_FAILURE;
        }
    }


    if ((a->obj == NULL) || (a->obj[idx++] != ASN_OBJECT_ID)) {
        WOLFSSL_MSG("Bad ASN1 Object");
        return WOLFSSL_FAILURE;
    }

    if (GetLength((const byte*)a->obj, &idx, &length,
                   a->objSz) < 0 || length < 0) {
        return WOLFSSL_FAILURE;
    }

    if (wolfSSL_BIO_write(bp, a->obj + idx, length) == (int)length) {
        return length;
    }

    return WOLFSSL_FAILURE;
}
#endif /* !NO_BIO */

/* Returns object data for an ASN1_OBJECT */
/* If pp is NULL then only the size is returned */
/* If pp has pointer to pointer then its used directly */
/* If pp has pointer to pointer that is NULL then new variable is allocated */
/* Failure returns WOLFSSL_FAILURE (0) */
int wolfSSL_i2d_ASN1_OBJECT(WOLFSSL_ASN1_OBJECT *a, unsigned char **pp)
{
    byte *p;

    WOLFSSL_ENTER("wolfSSL_i2d_ASN1_OBJECT");

    if (!a || !a->obj) {
        WOLFSSL_MSG("Bad parameters");
        return WOLFSSL_FAILURE;
    }

    if (!pp)
        return a->objSz;

    if (*pp)
        p = *pp;
    else {
        p = (byte*)XMALLOC(a->objSz, NULL, DYNAMIC_TYPE_OPENSSL);
        if (!p) {
            WOLFSSL_MSG("Bad malloc");
            return WOLFSSL_FAILURE;
        }
    }

    XMEMCPY(p, a->obj, a->objSz);
    *pp = p + a->objSz;
    return a->objSz;
}

#if defined(OPENSSL_ALL) || defined(WOLFSSL_HAPROXY) || defined(WOLFSSL_WPAS)
WOLFSSL_API size_t wolfSSL_get_finished(const WOLFSSL *ssl, void *buf, size_t count)
{
    byte len = 0;

    WOLFSSL_ENTER("SSL_get_finished");

    if (!ssl || !buf || count < TLS_FINISHED_SZ) {
        WOLFSSL_MSG("Bad parameter");
        return WOLFSSL_FAILURE;
    }

    if (ssl->options.side == WOLFSSL_SERVER_END) {
        len = ssl->serverFinished_len;
        XMEMCPY(buf, ssl->serverFinished, len);
    }
    else {
        len = ssl->clientFinished_len;
        XMEMCPY(buf, ssl->clientFinished, len);
    }
    return len;
}

WOLFSSL_API size_t wolfSSL_get_peer_finished(const WOLFSSL *ssl, void *buf, size_t count)
{
    byte len = 0;
    WOLFSSL_ENTER("SSL_get_peer_finished");

    if (!ssl || !buf || count < TLS_FINISHED_SZ) {
        WOLFSSL_MSG("Bad parameter");
        return WOLFSSL_FAILURE;
    }

    if (ssl->options.side == WOLFSSL_CLIENT_END) {
        len = ssl->serverFinished_len;
        XMEMCPY(buf, ssl->serverFinished, len);
    }
    else {
        len = ssl->clientFinished_len;
        XMEMCPY(buf, ssl->clientFinished, len);
    }

    return len;
}
#endif /* WOLFSSL_HAPROXY */

#ifndef NO_WOLFSSL_STUB
/*** TBD ***/
WOLFSSL_API void SSL_CTX_set_tmp_dh_callback(WOLFSSL_CTX *ctx, WOLFSSL_DH *(*dh) (WOLFSSL *ssl, int is_export, int keylength))
{
    (void)ctx;
    (void)dh;
    WOLFSSL_STUB("SSL_CTX_set_tmp_dh_callback");
}
#endif

#ifndef NO_WOLFSSL_STUB
/*** TBD ***/
WOLFSSL_API WOLF_STACK_OF(SSL_COMP) *SSL_COMP_get_compression_methods(void)
{
    WOLFSSL_STUB("SSL_COMP_get_compression_methods");
    return NULL;
}
#endif


int wolfSSL_sk_SSL_CIPHER_num(const WOLF_STACK_OF(WOLFSSL_CIPHER)* p)
{
    WOLFSSL_ENTER("wolfSSL_sk_SSL_CIPHER_num");
    if (p == NULL) {
        return WOLFSSL_FATAL_ERROR;
    }
    return (int)p->num;
}

WOLFSSL_API WOLFSSL_CIPHER* wolfSSL_sk_SSL_CIPHER_value(WOLFSSL_STACK* sk, int i)
{
    WOLFSSL_ENTER("wolfSSL_sk_SSL_CIPHER_value");
    return (WOLFSSL_CIPHER*)wolfSSL_sk_value(sk, i);
}

#if !defined(NETOS)
WOLFSSL_API void ERR_load_SSL_strings(void)
{

}
#endif

#ifdef HAVE_OCSP
WOLFSSL_API long wolfSSL_get_tlsext_status_ocsp_resp(WOLFSSL *s, unsigned char **resp)
{
    if (s == NULL || resp == NULL)
        return 0;

    *resp = s->ocspResp;
    return s->ocspRespSz;
}

WOLFSSL_API long wolfSSL_set_tlsext_status_ocsp_resp(WOLFSSL *s, unsigned char *resp, int len)
{
    if (s == NULL)
        return WOLFSSL_FAILURE;

    s->ocspResp   = resp;
    s->ocspRespSz = len;

    return WOLFSSL_SUCCESS;
}
#endif /* HAVE_OCSP */

#ifdef HAVE_MAX_FRAGMENT
#ifndef NO_WOLFSSL_CLIENT
/**
 * Set max fragment tls extension
 * @param c a pointer to WOLFSSL_CTX object
 * @param mode maximum fragment length mode
 * @return 1 on success, otherwise 0 or negative error code
 */
WOLFSSL_API int wolfSSL_CTX_set_tlsext_max_fragment_length(WOLFSSL_CTX *c,
                                                            unsigned char mode)
{
    if (c == NULL || (mode < WOLFSSL_MFL_2_9 || mode > WOLFSSL_MFL_2_12 ))
        return BAD_FUNC_ARG;

    return wolfSSL_CTX_UseMaxFragment(c, mode);
}
/**
 * Set max fragment tls extension
 * @param c a pointer to WOLFSSL object
 * @param mode maximum fragment length mode
 * @return 1 on success, otherwise 0 or negative error code
 */
WOLFSSL_API int wolfSSL_set_tlsext_max_fragment_length(WOLFSSL *s,
                                                            unsigned char mode)
{
    if (s == NULL || (mode < WOLFSSL_MFL_2_9 || mode > WOLFSSL_MFL_2_12 ))
        return BAD_FUNC_ARG;

    return wolfSSL_UseMaxFragment(s, mode);
}
#endif /* NO_WOLFSSL_CLIENT */
#endif /* HAVE_MAX_FRAGMENT */

#endif /* OPENSSL_EXTRA */

#if defined(OPENSSL_EXTRA) || defined(OPENSSL_EXTRA_X509_SMALL)
long wolfSSL_get_verify_result(const WOLFSSL *ssl)
{
    if (ssl == NULL) {
        return WOLFSSL_FAILURE;
    }

    return ssl->peerVerifyRet;
}
#endif

#ifdef OPENSSL_EXTRA

#ifndef NO_WOLFSSL_STUB
/* shows the number of accepts attempted by CTX in it's lifetime */
long wolfSSL_CTX_sess_accept(WOLFSSL_CTX* ctx)
{
    WOLFSSL_STUB("wolfSSL_CTX_sess_accept");
    (void)ctx;
    return 0;
}
#endif

#ifndef NO_WOLFSSL_STUB
/* shows the number of connects attempted CTX in it's lifetime */
long wolfSSL_CTX_sess_connect(WOLFSSL_CTX* ctx)
{
    WOLFSSL_STUB("wolfSSL_CTX_sess_connect");
    (void)ctx;
    return 0;
}
#endif


#ifndef NO_WOLFSSL_STUB
/* shows the number of accepts completed by CTX in it's lifetime */
long wolfSSL_CTX_sess_accept_good(WOLFSSL_CTX* ctx)
{
    WOLFSSL_STUB("wolfSSL_CTX_sess_accept_good");
    (void)ctx;
    return 0;
}
#endif


#ifndef NO_WOLFSSL_STUB
/* shows the number of connects completed by CTX in it's lifetime */
long wolfSSL_CTX_sess_connect_good(WOLFSSL_CTX* ctx)
{
    WOLFSSL_STUB("wolfSSL_CTX_sess_connect_good");
    (void)ctx;
    return 0;
}
#endif


#ifndef NO_WOLFSSL_STUB
/* shows the number of renegotiation accepts attempted by CTX */
long wolfSSL_CTX_sess_accept_renegotiate(WOLFSSL_CTX* ctx)
{
    WOLFSSL_STUB("wolfSSL_CTX_sess_accept_renegotiate");
    (void)ctx;
    return 0;
}
#endif


#ifndef NO_WOLFSSL_STUB
/* shows the number of renegotiation accepts attempted by CTX */
long wolfSSL_CTX_sess_connect_renegotiate(WOLFSSL_CTX* ctx)
{
    WOLFSSL_STUB("wolfSSL_CTX_sess_connect_renegotiate");
    (void)ctx;
    return 0;
}
#endif


#ifndef NO_WOLFSSL_STUB
long wolfSSL_CTX_sess_hits(WOLFSSL_CTX* ctx)
{
    WOLFSSL_STUB("wolfSSL_CTX_sess_hits");
    (void)ctx;
    return 0;
}
#endif


#ifndef NO_WOLFSSL_STUB
long wolfSSL_CTX_sess_cb_hits(WOLFSSL_CTX* ctx)
{
    WOLFSSL_STUB("wolfSSL_CTX_sess_cb_hits");
    (void)ctx;
    return 0;
}
#endif


#ifndef NO_WOLFSSL_STUB
long wolfSSL_CTX_sess_cache_full(WOLFSSL_CTX* ctx)
{
    WOLFSSL_STUB("wolfSSL_CTX_sess_cache_full");
    (void)ctx;
    return 0;
}
#endif


#ifndef NO_WOLFSSL_STUB
long wolfSSL_CTX_sess_misses(WOLFSSL_CTX* ctx)
{
    WOLFSSL_STUB("wolfSSL_CTX_sess_misses");
    (void)ctx;
    return 0;
}
#endif


#ifndef NO_WOLFSSL_STUB
long wolfSSL_CTX_sess_timeouts(WOLFSSL_CTX* ctx)
{
    WOLFSSL_STUB("wolfSSL_CTX_sess_timeouts");
    (void)ctx;
    return 0;
}
#endif


/* Return the total number of sessions */
long wolfSSL_CTX_sess_number(WOLFSSL_CTX* ctx)
{
    word32 total = 0;

    WOLFSSL_ENTER("wolfSSL_CTX_sess_number");
    (void)ctx;

#if defined(WOLFSSL_SESSION_STATS) && !defined(NO_SESSION_CACHE)
    if (wolfSSL_get_session_stats(NULL, &total, NULL, NULL) != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("Error getting session stats");
    }
#else
    WOLFSSL_MSG("Please use macro WOLFSSL_SESSION_STATS for session stats");
#endif

    return (long)total;
}


#ifndef NO_CERTS
long wolfSSL_CTX_add_extra_chain_cert(WOLFSSL_CTX* ctx, WOLFSSL_X509* x509)
{
    byte* chain = NULL;
    long  chainSz = 0;
    int   derSz;
    const byte* der;
    int   ret;
    int   idx = 0;
    DerBuffer *derBuffer = NULL;

    WOLFSSL_ENTER("wolfSSL_CTX_add_extra_chain_cert");

    if (ctx == NULL || x509 == NULL) {
        WOLFSSL_MSG("Bad Argument");
        return WOLFSSL_FAILURE;
    }

    der = wolfSSL_X509_get_der(x509, &derSz);
    if (der == NULL || derSz <= 0) {
        WOLFSSL_MSG("Error getting X509 DER");
        return WOLFSSL_FAILURE;
    }

    if (ctx->certificate == NULL) {
        WOLFSSL_ENTER("wolfSSL_use_certificate_chain_buffer_format");

        /* Process buffer makes first certificate the leaf. */
        ret = ProcessBuffer(ctx, der, derSz, WOLFSSL_FILETYPE_ASN1, CERT_TYPE,
                            NULL, NULL, 1, GET_VERIFY_SETTING_CTX(ctx));
        if (ret != WOLFSSL_SUCCESS) {
            WOLFSSL_LEAVE("wolfSSL_CTX_add_extra_chain_cert", ret);
            return WOLFSSL_FAILURE;
        }
    }
    else {
        /* TODO: Do this elsewhere. */
        ret = AllocDer(&derBuffer, derSz, CERT_TYPE, ctx->heap);
        if (ret != 0) {
            WOLFSSL_MSG("Memory Error");
            return WOLFSSL_FAILURE;
        }
        XMEMCPY(derBuffer->buffer, der, derSz);
        ret = AddCA(ctx->cm, &derBuffer, WOLFSSL_USER_CA,
            GET_VERIFY_SETTING_CTX(ctx));
        if (ret != WOLFSSL_SUCCESS) {
            WOLFSSL_LEAVE("wolfSSL_CTX_add_extra_chain_cert", ret);
            return WOLFSSL_FAILURE;
        }

        /* adding cert to existing chain */
        if (ctx->certChain != NULL && ctx->certChain->length > 0) {
            chainSz += ctx->certChain->length;
        }
        chainSz += OPAQUE24_LEN + derSz;

        chain = (byte*)XMALLOC(chainSz, ctx->heap, DYNAMIC_TYPE_DER);
        if (chain == NULL) {
            WOLFSSL_MSG("Memory Error");
            return WOLFSSL_FAILURE;
        }

        if (ctx->certChain != NULL && ctx->certChain->length > 0) {
            XMEMCPY(chain, ctx->certChain->buffer, ctx->certChain->length);
            idx = ctx->certChain->length;
        }
        c32to24(derSz, chain + idx);
        idx += OPAQUE24_LEN;
        XMEMCPY(chain + idx, der, derSz);
        idx += derSz;
#ifdef WOLFSSL_TLS13
        ctx->certChainCnt++;
#endif

        FreeDer(&ctx->certChain);
        ret = AllocDer(&ctx->certChain, idx, CERT_TYPE, ctx->heap);
        if (ret == 0) {
            XMEMCPY(ctx->certChain->buffer, chain, idx);
        }
    }

    /* on success WOLFSSL_X509 memory is responsibility of ctx */
    wolfSSL_X509_free(x509);
    if (chain != NULL)
        XFREE(chain, ctx->heap, DYNAMIC_TYPE_DER);

    return WOLFSSL_SUCCESS;
}


long wolfSSL_CTX_set_tlsext_status_arg(WOLFSSL_CTX* ctx, void* arg)
{
    if (ctx == NULL || ctx->cm == NULL) {
        return WOLFSSL_FAILURE;
    }

    ctx->cm->ocspIOCtx = arg;
    return WOLFSSL_SUCCESS;
}

#endif /* NO_CERTS */


/* Get the session cache mode for CTX
 *
 * ctx  WOLFSSL_CTX struct to get cache mode from
 *
 * Returns a bit mask that has the session cache mode */
WOLFSSL_API long wolfSSL_CTX_get_session_cache_mode(WOLFSSL_CTX* ctx)
{
    long m = 0;

    WOLFSSL_ENTER("SSL_CTX_set_session_cache_mode");

    if (ctx == NULL) {
        return m;
    }

    if (ctx->sessionCacheOff != 1) {
        m |= SSL_SESS_CACHE_SERVER;
    }

    if (ctx->sessionCacheFlushOff == 1) {
        m |= SSL_SESS_CACHE_NO_AUTO_CLEAR;
    }

#ifdef HAVE_EXT_CACHE
    if (ctx->internalCacheOff == 1) {
        m |= SSL_SESS_CACHE_NO_INTERNAL_STORE;
    }
    if (ctx->internalCacheLookupOff == 1) {
        m |= SSL_SESS_CACHE_NO_INTERNAL_LOOKUP;
    }
#endif

    return m;
}


int wolfSSL_get_read_ahead(const WOLFSSL* ssl)
{
    if (ssl == NULL) {
        return WOLFSSL_FAILURE;
    }

    return ssl->readAhead;
}


int wolfSSL_set_read_ahead(WOLFSSL* ssl, int v)
{
    if (ssl == NULL) {
        return WOLFSSL_FAILURE;
    }

    ssl->readAhead = (byte)v;

    return WOLFSSL_SUCCESS;
}


int wolfSSL_CTX_get_read_ahead(WOLFSSL_CTX* ctx)
{
    if (ctx == NULL) {
        return WOLFSSL_FAILURE;
    }

    return ctx->readAhead;
}


int wolfSSL_CTX_set_read_ahead(WOLFSSL_CTX* ctx, int v)
{
    if (ctx == NULL) {
        return WOLFSSL_FAILURE;
    }

    ctx->readAhead = (byte)v;

    return WOLFSSL_SUCCESS;
}


long wolfSSL_CTX_set_tlsext_opaque_prf_input_callback_arg(WOLFSSL_CTX* ctx,
        void* arg)
{
    if (ctx == NULL) {
        return WOLFSSL_FAILURE;
    }

    ctx->userPRFArg = arg;
    return WOLFSSL_SUCCESS;
}


#ifndef NO_DES3
/* 0 on success */
int wolfSSL_DES_set_key(WOLFSSL_const_DES_cblock* myDes,
                                               WOLFSSL_DES_key_schedule* key)
{
#ifdef WOLFSSL_CHECK_DESKEY
    return wolfSSL_DES_set_key_checked(myDes, key);
#else
    wolfSSL_DES_set_key_unchecked(myDes, key);
    return 0;
#endif
}



/* return true in fail case (1) */
static int DES_check(word32 mask, word32 mask2, unsigned char* key)
{
    word32 value[2];

    /* sanity check on length made in wolfSSL_DES_set_key_checked */
    value[0] = mask;
    value[1] = mask2;
    return (XMEMCMP(value, key, sizeof(value)) == 0)? 1: 0;
}


/* check that the key is odd parity and is not a weak key
 * returns -1 if parity is wrong, -2 if weak/null key and 0 on success */
int wolfSSL_DES_set_key_checked(WOLFSSL_const_DES_cblock* myDes,
                                               WOLFSSL_DES_key_schedule* key)
{
    if (myDes == NULL || key == NULL) {
        WOLFSSL_MSG("Bad argument passed to wolfSSL_DES_set_key_checked");
        return -2;
    }
    else {
        word32 sz = sizeof(WOLFSSL_DES_key_schedule);

        /* sanity check before call to DES_check */
        if (sz != (sizeof(word32) * 2)) {
            WOLFSSL_MSG("Unexpected WOLFSSL_DES_key_schedule size");
            return -2;
        }

        /* check odd parity */
        if (wolfSSL_DES_check_key_parity(myDes) != 1) {
            WOLFSSL_MSG("Odd parity test fail");
            return -1;
        }

        if (wolfSSL_DES_is_weak_key(myDes) == 1) {
            WOLFSSL_MSG("Weak key found");
            return -2;
        }

        /* passed tests, now copy over key */
        XMEMCPY(key, myDes, sizeof(WOLFSSL_const_DES_cblock));

        return 0;
    }
}


/* check is not weak. Weak key list from Nist "Recommendation for the Triple
 * Data Encryption Algorithm (TDEA) Block Cipher"
 *
 * returns 1 if is weak 0 if not
 */
int wolfSSL_DES_is_weak_key(WOLFSSL_const_DES_cblock* key)
{
    word32 mask, mask2;

    WOLFSSL_ENTER("wolfSSL_DES_is_weak_key");

    if (key == NULL) {
        WOLFSSL_MSG("NULL key passed in");
        return 1;
    }

    mask = 0x01010101; mask2 = 0x01010101;
    if (DES_check(mask, mask2, *key)) {
        WOLFSSL_MSG("Weak key found");
        return 1;
    }

    mask = 0xFEFEFEFE; mask2 = 0xFEFEFEFE;
    if (DES_check(mask, mask2, *key)) {
        WOLFSSL_MSG("Weak key found");
        return 1;
    }

    mask = 0xE0E0E0E0; mask2 = 0xF1F1F1F1;
    if (DES_check(mask, mask2, *key)) {
        WOLFSSL_MSG("Weak key found");
        return 1;
    }

    mask = 0x1F1F1F1F; mask2 = 0x0E0E0E0E;
    if (DES_check(mask, mask2, *key)) {
        WOLFSSL_MSG("Weak key found");
        return 1;
    }

    /* semi-weak *key check (list from same Nist paper) */
    mask  = 0x011F011F; mask2 = 0x010E010E;
    if (DES_check(mask, mask2, *key) ||
       DES_check(ByteReverseWord32(mask), ByteReverseWord32(mask2), *key)) {
        WOLFSSL_MSG("Weak key found");
        return 1;
    }

    mask  = 0x01E001E0; mask2 = 0x01F101F1;
    if (DES_check(mask, mask2, *key) ||
       DES_check(ByteReverseWord32(mask), ByteReverseWord32(mask2), *key)) {
        WOLFSSL_MSG("Weak key found");
        return 1;
    }

    mask  = 0x01FE01FE; mask2 = 0x01FE01FE;
    if (DES_check(mask, mask2, *key) ||
       DES_check(ByteReverseWord32(mask), ByteReverseWord32(mask2), *key)) {
        WOLFSSL_MSG("Weak key found");
        return 1;
    }

    mask  = 0x1FE01FE0; mask2 = 0x0EF10EF1;
    if (DES_check(mask, mask2, *key) ||
       DES_check(ByteReverseWord32(mask), ByteReverseWord32(mask2), *key)) {
        WOLFSSL_MSG("Weak key found");
        return 1;
    }

    mask  = 0x1FFE1FFE; mask2 = 0x0EFE0EFE;
    if (DES_check(mask, mask2, *key) ||
       DES_check(ByteReverseWord32(mask), ByteReverseWord32(mask2), *key)) {
        WOLFSSL_MSG("Weak key found");
        return 1;
    }

    return 0;
}


void wolfSSL_DES_set_key_unchecked(WOLFSSL_const_DES_cblock* myDes,
                                               WOLFSSL_DES_key_schedule* key)
{
    if (myDes != NULL && key != NULL) {
        XMEMCPY(key, myDes, sizeof(WOLFSSL_const_DES_cblock));
    }
}


/* Sets the parity of the DES key for use */
void wolfSSL_DES_set_odd_parity(WOLFSSL_DES_cblock* myDes)
{
    word32 i;
    word32 sz = sizeof(WOLFSSL_DES_cblock);

    WOLFSSL_ENTER("wolfSSL_DES_set_odd_parity");

    for (i = 0; i < sz; i++) {
        unsigned char c = (*myDes)[i];
        if ((
            ((c >> 1) & 0x01) ^
            ((c >> 2) & 0x01) ^
            ((c >> 3) & 0x01) ^
            ((c >> 4) & 0x01) ^
            ((c >> 5) & 0x01) ^
            ((c >> 6) & 0x01) ^
            ((c >> 7) & 0x01)) == (c & 0x01)) {
            WOLFSSL_MSG("Flipping parity bit");
            (*myDes)[i] = c ^ 0x01;
        }
    }
}

int wolfSSL_DES_check_key_parity(WOLFSSL_DES_cblock *myDes)
{
    word32 i;
    word32 sz = sizeof(WOLFSSL_DES_cblock);

    WOLFSSL_ENTER("wolfSSL_DES_check_key_parity");

    for (i = 0; i < sz; i++) {
        unsigned char c = (*myDes)[i];
        if ((
            ((c >> 1) & 0x01) ^
            ((c >> 2) & 0x01) ^
            ((c >> 3) & 0x01) ^
            ((c >> 4) & 0x01) ^
            ((c >> 5) & 0x01) ^
            ((c >> 6) & 0x01) ^
            ((c >> 7) & 0x01)) == (c & 0x01)) {
            return 0;
        }
    }
    return 1;
}

#ifdef WOLFSSL_DES_ECB
/* Encrypt or decrypt input message desa with key and get output in desb.
 * if enc is DES_ENCRYPT,input message is encrypted or
 * if enc is DES_DECRYPT,input message is decrypted.
 * */
void wolfSSL_DES_ecb_encrypt(WOLFSSL_DES_cblock* desa,
             WOLFSSL_DES_cblock* desb, WOLFSSL_DES_key_schedule* key, int enc)
{
    Des myDes;

    WOLFSSL_ENTER("wolfSSL_DES_ecb_encrypt");

    if (desa == NULL || key == NULL || desb == NULL ||
        (enc != DES_ENCRYPT && enc != DES_DECRYPT)) {
        WOLFSSL_MSG("Bad argument passed to wolfSSL_DES_ecb_encrypt");
    } else {
        if (wc_Des_SetKey(&myDes, (const byte*) key,
                           (const byte*) NULL, !enc) != 0) {
            WOLFSSL_MSG("wc_Des_SetKey return error.");
            return;
        }
        if (enc == DES_ENCRYPT){
            if (wc_Des_EcbEncrypt(&myDes, (byte*) desb, (const byte*) desa,
                        sizeof(WOLFSSL_DES_cblock)) != 0){
                WOLFSSL_MSG("wc_Des_EcbEncrypt return error.");
            }
        } else {
            if (wc_Des_EcbDecrypt(&myDes, (byte*) desb, (const byte*) desa,
                        sizeof(WOLFSSL_DES_cblock)) != 0){
                WOLFSSL_MSG("wc_Des_EcbDecrpyt return error.");
            }
        }
    }
}
#endif
#endif /* NO_DES3 */

#ifndef NO_RC4
/* Set the key state for Arc4 structure.
 *
 * key  Arc4 structure to use
 * len  length of data buffer
 * data initial state to set Arc4 structure
 */
void wolfSSL_RC4_set_key(WOLFSSL_RC4_KEY* key, int len,
        const unsigned char* data)
{
    typedef char rc4_test[sizeof(WOLFSSL_RC4_KEY) >= sizeof(Arc4) ? 1 : -1];
    (void)sizeof(rc4_test);

    WOLFSSL_ENTER("wolfSSL_RC4_set_key");

    if (key == NULL || len < 0) {
        WOLFSSL_MSG("bad argument passed in");
        return;
    }

    XMEMSET(key, 0, sizeof(WOLFSSL_RC4_KEY));
    wc_Arc4SetKey((Arc4*)key, data, (word32)len);
}


/* Encrypt/decrypt with Arc4 structure.
 *
 * len length of buffer to encrypt/decrypt (in/out)
 * in  buffer to encrypt/decrypt
 * out results of encryption/decryption
 */
void wolfSSL_RC4(WOLFSSL_RC4_KEY* key, size_t len,
        const unsigned char* in, unsigned char* out)
{
    WOLFSSL_ENTER("wolfSSL_RC4");

    if (key == NULL || in == NULL || out == NULL) {
        WOLFSSL_MSG("Bad argument passed in");
        return;
    }

    wc_Arc4Process((Arc4*)key, out, in, (word32)len);
}
#endif /* NO_RC4 */

#ifndef NO_AES

#ifdef WOLFSSL_AES_DIRECT
/* AES encrypt direct, it is expected to be blocks of AES_BLOCK_SIZE for input.
 *
 * input  Data to encrypt
 * output Encrypted data after done
 * key    AES key to use for encryption
 */
void wolfSSL_AES_encrypt(const unsigned char* input, unsigned char* output,
        AES_KEY *key)
{
    WOLFSSL_ENTER("wolfSSL_AES_encrypt");

    if (input == NULL || output == NULL || key == NULL) {
        WOLFSSL_MSG("Null argument passed in");
        return;
    }

#if !defined(HAVE_SELFTEST) && \
    (!defined(HAVE_FIPS) || (defined(FIPS_VERSION_GE) && FIPS_VERSION_GE(5,3)))
    if (wc_AesEncryptDirect((Aes*)key, output, input) != 0) {
        WOLFSSL_MSG("wc_AesEncryptDirect failed");
        return;
    }
#else
    wc_AesEncryptDirect((Aes*)key, output, input);
#endif
}


/* AES decrypt direct, it is expected to be blocks of AES_BLOCK_SIZE for input.
 *
 * input  Data to decrypt
 * output Decrypted data after done
 * key    AES key to use for encryption
 */
void wolfSSL_AES_decrypt(const unsigned char* input, unsigned char* output,
        AES_KEY *key)
{
    WOLFSSL_ENTER("wolfSSL_AES_decrypt");

    if (input == NULL || output == NULL || key == NULL) {
        WOLFSSL_MSG("Null argument passed in");
        return;
    }

#if !defined(HAVE_SELFTEST) && \
    (!defined(HAVE_FIPS) || (defined(FIPS_VERSION_GE) && FIPS_VERSION_GE(5,3)))
    if (wc_AesDecryptDirect((Aes*)key, output, input) != 0) {
        WOLFSSL_MSG("wc_AesDecryptDirect failed");
        return;
    }
#else
    wc_AesDecryptDirect((Aes*)key, output, input);
#endif
}
#endif /* WOLFSSL_AES_DIRECT */

/* Setup of an AES key to use for encryption.
 *
 * key  key in bytes to use for encryption
 * bits size of key in bits
 * aes  AES structure to initialize
 */
int wolfSSL_AES_set_encrypt_key(const unsigned char *key, const int bits,
        AES_KEY *aes)
{
    typedef char aes_test[sizeof(AES_KEY) >= sizeof(Aes) ? 1 : -1];
    (void)sizeof(aes_test);

    WOLFSSL_ENTER("wolfSSL_AES_set_encrypt_key");

    if (key == NULL || aes == NULL) {
        WOLFSSL_MSG("Null argument passed in");
        return -1;
    }

    XMEMSET(aes, 0, sizeof(AES_KEY));
    if (wc_AesSetKey((Aes*)aes, key, ((bits)/8), NULL, AES_ENCRYPT) != 0) {
        WOLFSSL_MSG("Error in setting AES key");
        return -1;
    }
    return 0;
}


/* Setup of an AES key to use for decryption.
 *
 * key  key in bytes to use for decryption
 * bits size of key in bits
 * aes  AES structure to initialize
 */
int wolfSSL_AES_set_decrypt_key(const unsigned char *key, const int bits,
        AES_KEY *aes)
{
    typedef char aes_test[sizeof(AES_KEY) >= sizeof(Aes) ? 1 : -1];
    (void)sizeof(aes_test);

    WOLFSSL_ENTER("wolfSSL_AES_set_decrypt_key");

    if (key == NULL || aes == NULL) {
        WOLFSSL_MSG("Null argument passed in");
        return -1;
    }

    XMEMSET(aes, 0, sizeof(AES_KEY));
    if (wc_AesSetKey((Aes*)aes, key, ((bits)/8), NULL, AES_DECRYPT) != 0) {
        WOLFSSL_MSG("Error in setting AES key");
        return -1;
    }
    return 0;
}


#ifdef HAVE_AES_ECB
/* Encrypt/decrypt a 16 byte block of data using the key passed in.
 *
 * in  buffer to encrypt/decrypt
 * out buffer to hold result of encryption/decryption
 * key AES structure to use with encryption/decryption
 * enc AES_ENCRPT for encryption and AES_DECRYPT for decryption
 */
void wolfSSL_AES_ecb_encrypt(const unsigned char *in, unsigned char* out,
                             AES_KEY *key, const int enc)
{
    Aes* aes;

    WOLFSSL_ENTER("wolfSSL_AES_ecb_encrypt");

    if (key == NULL || in == NULL || out == NULL) {
        WOLFSSL_MSG("Error, Null argument passed in");
        return;
    }

    aes = (Aes*)key;
    if (enc == AES_ENCRYPT) {
        if (wc_AesEcbEncrypt(aes, out, in, AES_BLOCK_SIZE) != 0) {
            WOLFSSL_MSG("Error with AES CBC encrypt");
        }
    }
    else {
    #ifdef HAVE_AES_DECRYPT
        if (wc_AesEcbDecrypt(aes, out, in, AES_BLOCK_SIZE) != 0) {
            WOLFSSL_MSG("Error with AES CBC decrypt");
        }
    #else
        WOLFSSL_MSG("AES decryption not compiled in");
    #endif
    }
}
#endif /* HAVE_AES_ECB */

#ifdef HAVE_AES_CBC
/* Encrypt data using key and iv passed in. iv gets updated to most recent iv
 * state after encryption/decryption.
 *
 * in  buffer to encrypt/decrypt
 * out buffer to hold result of encryption/decryption
 * len length of input buffer
 * key AES structure to use with encryption/decryption
 * iv  iv to use with operation
 * enc 1 for encryption and 0 for decryption
 */
void wolfSSL_AES_cbc_encrypt(const unsigned char *in, unsigned char* out,
        size_t len, AES_KEY *key, unsigned char* iv, const int enc)
{
    Aes* aes;

    WOLFSSL_ENTER("wolfSSL_AES_cbc_encrypt");

    if (key == NULL || in == NULL || out == NULL || iv == NULL || len == 0) {
        WOLFSSL_MSG("Error, Null argument passed in");
        return;
    }

    aes = (Aes*)key;
    if (wc_AesSetIV(aes, (const byte*)iv) != 0) {
        WOLFSSL_MSG("Error with setting iv");
        return;
    }

    if (enc == AES_ENCRYPT) {
        if (wc_AesCbcEncrypt(aes, out, in, (word32)len) != 0) {
            WOLFSSL_MSG("Error with AES CBC encrypt");
            return;
        }
    }
    else {
        if (wc_AesCbcDecrypt(aes, out, in, (word32)len) != 0) {
            WOLFSSL_MSG("Error with AES CBC decrypt");
            return;
        }
    }

    /* to be compatible copy iv to iv buffer after completing operation */
    XMEMCPY(iv, (byte*)(aes->reg), AES_BLOCK_SIZE);
}
#endif /* HAVE_AES_CBC */


/* Encrypt data using CFB mode with key and iv passed in. iv gets updated to
 * most recent iv state after encryption/decryption.
 *
 * in  buffer to encrypt/decrypt
 * out buffer to hold result of encryption/decryption
 * len length of input buffer
 * key AES structure to use with encryption/decryption
 * iv  iv to use with operation
 * num contains the amount of block used
 * enc AES_ENCRYPT for encryption and AES_DECRYPT for decryption
 */
void wolfSSL_AES_cfb128_encrypt(const unsigned char *in, unsigned char* out,
        size_t len, AES_KEY *key, unsigned char* iv, int* num,
        const int enc)
{
#ifndef WOLFSSL_AES_CFB
    WOLFSSL_MSG("CFB mode not enabled please use macro WOLFSSL_AES_CFB");
    (void)in;
    (void)out;
    (void)len;
    (void)key;
    (void)iv;
    (void)num;
    (void)enc;

    return;
#else
    Aes* aes;

    WOLFSSL_ENTER("wolfSSL_AES_cbc_encrypt");
    if (key == NULL || in == NULL || out == NULL || iv == NULL) {
        WOLFSSL_MSG("Error, Null argument passed in");
        return;
    }

    aes = (Aes*)key;
    if (wc_AesSetIV(aes, (const byte*)iv) != 0) {
        WOLFSSL_MSG("Error with setting iv");
        return;
    }

    if (enc == AES_ENCRYPT) {
        if (wc_AesCfbEncrypt(aes, out, in, (word32)len) != 0) {
            WOLFSSL_MSG("Error with AES CBC encrypt");
            return;
        }
    }
    else {
        if (wc_AesCfbDecrypt(aes, out, in, (word32)len) != 0) {
            WOLFSSL_MSG("Error with AES CBC decrypt");
            return;
        }
    }

    /* to be compatible copy iv to iv buffer after completing operation */
    XMEMCPY(iv, (byte*)(aes->reg), AES_BLOCK_SIZE);

    /* store number of left over bytes to num */
    *num = (aes->left)? AES_BLOCK_SIZE - aes->left : 0;
#endif /* WOLFSSL_AES_CFB */
}

/* wc_AesKey*Wrap_ex API not available in FIPS and SELFTEST */
#if defined(HAVE_AES_KEYWRAP) && !defined(HAVE_FIPS) && !defined(HAVE_SELFTEST)
int wolfSSL_AES_wrap_key(AES_KEY *key, const unsigned char *iv,
                 unsigned char *out,
                 const unsigned char *in, unsigned int inlen)
{
    int ret;

    WOLFSSL_ENTER("wolfSSL_AES_wrap_key");

    if (out == NULL || in == NULL) {
        WOLFSSL_MSG("Error, Null argument passed in");
        return WOLFSSL_FAILURE;
    }

    ret = wc_AesKeyWrap_ex((Aes*)key, in, inlen, out, inlen + KEYWRAP_BLOCK_SIZE, iv);

    return ret < 0 ? WOLFSSL_FAILURE : ret;
}

int wolfSSL_AES_unwrap_key(AES_KEY *key, const unsigned char *iv,
                   unsigned char *out,
                   const unsigned char *in, unsigned int inlen)
{
    int ret;

    WOLFSSL_ENTER("wolfSSL_AES_wrap_key");

    if (out == NULL || in == NULL) {
        WOLFSSL_MSG("Error, Null argument passed in");
        return WOLFSSL_FAILURE;
    }

    ret = wc_AesKeyUnWrap_ex((Aes*)key, in, inlen, out, inlen + KEYWRAP_BLOCK_SIZE, iv);

    return ret < 0 ? WOLFSSL_FAILURE : ret;
}
#endif /* HAVE_AES_KEYWRAP && !HAVE_FIPS && !HAVE_SELFTEST */

#ifdef HAVE_CTS
/*
 * Ciphertext stealing interface compatible with RFC2040 and RFC3962.
 */
size_t wolfSSL_CRYPTO_cts128_encrypt(const unsigned char *in,
        unsigned char *out, size_t len, const void *key,
        unsigned char *iv, WOLFSSL_CBC128_CB cbc)
{
    byte lastBlk[WOLFSSL_CTS128_BLOCK_SZ];
    int lastBlkLen = len % WOLFSSL_CTS128_BLOCK_SZ;
    WOLFSSL_ENTER("wolfSSL_CRYPTO_cts128_encrypt");

    if (in == NULL || out == NULL || len < WOLFSSL_CTS128_BLOCK_SZ ||
            cbc == NULL) {
        WOLFSSL_MSG("Bad parameter");
        return WOLFSSL_FAILURE;
    }

    if (lastBlkLen == 0)
        lastBlkLen = WOLFSSL_CTS128_BLOCK_SZ;

    /* Encrypt data up to last block */
    (*cbc)(in, out, len - lastBlkLen, key, iv, AES_ENCRYPT);

    /* Move to last block */
    in += len - lastBlkLen;
    out += len - lastBlkLen;

    /* RFC2040: Pad Pn with zeros at the end to create P of length BB. */
    XMEMCPY(lastBlk, in, lastBlkLen);
    XMEMSET(lastBlk + lastBlkLen, 0, WOLFSSL_CTS128_BLOCK_SZ - lastBlkLen);
    /* RFC2040: Select the first Ln bytes of En-1 to create Cn */
    XMEMCPY(out, out - WOLFSSL_CTS128_BLOCK_SZ, lastBlkLen);
    (*cbc)(lastBlk, out - WOLFSSL_CTS128_BLOCK_SZ, WOLFSSL_CTS128_BLOCK_SZ,
            key, iv, AES_ENCRYPT);

    return len;
}

size_t wolfSSL_CRYPTO_cts128_decrypt(const unsigned char *in,
        unsigned char *out, size_t len, const void *key,
        unsigned char *iv, WOLFSSL_CBC128_CB cbc)
{
    byte lastBlk[WOLFSSL_CTS128_BLOCK_SZ];
    byte prevBlk[WOLFSSL_CTS128_BLOCK_SZ];
    int lastBlkLen = len % WOLFSSL_CTS128_BLOCK_SZ;
    WOLFSSL_ENTER("wolfSSL_CRYPTO_cts128_decrypt");

    if (in == NULL || out == NULL || len <= WOLFSSL_CTS128_BLOCK_SZ ||
            cbc == NULL) {
        WOLFSSL_MSG("Bad parameter");
        return WOLFSSL_FAILURE;
    }

    if (lastBlkLen == 0)
        lastBlkLen = WOLFSSL_CTS128_BLOCK_SZ;

    /* Decrypt up to last two blocks */
    (*cbc)(in, out, len - lastBlkLen - WOLFSSL_CTS128_BLOCK_SZ, key, iv,
            AES_DECRYPTION);

    /* Move to last two blocks */
    in += len - lastBlkLen - WOLFSSL_CTS128_BLOCK_SZ;
    out += len - lastBlkLen - WOLFSSL_CTS128_BLOCK_SZ;

    /* RFC2040: Decrypt Cn-1 to create Dn.
     * Use 0 buffer as IV to do straight decryption.
     * This places the Cn-1 block at lastBlk */
    XMEMSET(lastBlk, 0, WOLFSSL_CTS128_BLOCK_SZ);
    (*cbc)(in, prevBlk, WOLFSSL_CTS128_BLOCK_SZ, key, lastBlk, AES_DECRYPT);
    /* RFC2040: Append the tail (BB minus Ln) bytes of Xn to Cn
     *          to create En. */
    XMEMCPY(prevBlk, in + WOLFSSL_CTS128_BLOCK_SZ, lastBlkLen);
    /* Cn and Cn-1 can now be decrypted */
    (*cbc)(prevBlk, out, WOLFSSL_CTS128_BLOCK_SZ, key, iv, AES_DECRYPT);
    (*cbc)(lastBlk, lastBlk, WOLFSSL_CTS128_BLOCK_SZ, key, iv, AES_DECRYPT);
    XMEMCPY(out + WOLFSSL_CTS128_BLOCK_SZ, lastBlk, lastBlkLen);
    return len;
}
#endif /* HAVE_CTS */
#endif /* NO_AES */

#ifndef NO_ASN_TIME
#ifndef NO_BIO
int wolfSSL_ASN1_UTCTIME_print(WOLFSSL_BIO* bio, const WOLFSSL_ASN1_UTCTIME* a)
{
    WOLFSSL_ENTER("ASN1_UTCTIME_print");
    if (bio == NULL || a == NULL) {
        return WOLFSSL_FAILURE;
    }
    if (a->type != ASN_UTC_TIME) {
        WOLFSSL_MSG("Error, not UTC_TIME");
        return WOLFSSL_FAILURE;
    }

    return wolfSSL_ASN1_TIME_print(bio, a);
}

#endif /* !NO_BIO */

/* Checks the ASN1 syntax of "a"
 * returns WOLFSSL_SUCCESS (1)  if correct otherwise WOLFSSL_FAILURE (0) */
int wolfSSL_ASN1_TIME_check(const WOLFSSL_ASN1_TIME* a)
{
    char buf[MAX_TIME_STRING_SZ];

    WOLFSSL_ENTER("wolfSSL_ASN1_TIME_check");

    /* if can parse the WOLFSSL_ASN1_TIME passed in then consider syntax good */
    if (wolfSSL_ASN1_TIME_to_string((WOLFSSL_ASN1_TIME*)a, buf,
                MAX_TIME_STRING_SZ) == NULL) {
        return WOLFSSL_FAILURE;
    }
    return WOLFSSL_SUCCESS;
}

/*
 * Convert time to Unix time (GMT).
 */
static long long TimeToUnixTime(int sec, int min, int hour, int mday, int mon,
                                int year)
{
    /* Number of cumulative days from the previous months, starting from
     * beginning of January. */
    static const int monthDaysCumulative [12] = {
        0, 31, 59, 90, 120, 151, 181, 212, 243, 273, 304, 334
    };
    int leapDays = year;

    if (mon <= 1) {
        --leapDays;
    }
    leapDays = leapDays / 4 - leapDays / 100 + leapDays / 400 - 1969 / 4 +
               1969 / 100 - 1969 / 400;

    return ((((long long) (year - 1970) * 365 + leapDays +
           monthDaysCumulative[mon] + mday - 1) * 24 + hour) * 60 + min) * 60 +
           sec;
}

int wolfSSL_ASN1_TIME_diff(int *days, int *secs, const WOLFSSL_ASN1_TIME *from,
    const WOLFSSL_ASN1_TIME *to)
{
    const int SECS_PER_DAY = 24 * 60 * 60;
    struct tm fromTm_s, *fromTmGmt = &fromTm_s;
    struct tm toTm_s, *toTmGmt = &toTm_s;
    time_t currTime;
    long long fromSecs;
    long long toSecs;
    double diffSecs;
    struct tm *tmpTs;
#if defined(NEED_TMP_TIME)
    /* for use with gmtime_r */
    struct tm tmpTimeStorage;
    tmpTs = &tmpTimeStorage;
#else
    tmpTs = NULL;
#endif
    (void)tmpTs;

    WOLFSSL_ENTER("wolfSSL_ASN1_TIME_diff");

    if (days == NULL) {
        WOLFSSL_MSG("days is NULL");
        return WOLFSSL_FAILURE;
    }
    if (secs == NULL) {
        WOLFSSL_MSG("secs is NULL");
        return WOLFSSL_FAILURE;
    }

    if (from == NULL && to == NULL) {
        *days = 0;
        *secs = 0;
        return WOLFSSL_SUCCESS;
    }

    if (from == NULL) {
        currTime = wc_Time(0);
        fromTmGmt = XGMTIME(&currTime, tmpTs);
        if (fromTmGmt == NULL) {
            WOLFSSL_MSG("XGMTIME for from time failed.");
            return WOLFSSL_FAILURE;
        }
    }
    else if (wolfSSL_ASN1_TIME_to_tm(from, fromTmGmt) != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("Failed to convert from time to struct tm.");
        return WOLFSSL_FAILURE;
    }

    /* We use TimeToUnixTime here instead of XMKTIME to avoid the Year 2038
     * Problem on platforms where time_t is 32 bits. struct tm stores the year
     * as years since 1900, so we add 1900 to the year. */
    fromSecs = TimeToUnixTime(fromTmGmt->tm_sec, fromTmGmt->tm_min,
                          fromTmGmt->tm_hour, fromTmGmt->tm_mday,
                          fromTmGmt->tm_mon, fromTmGmt->tm_year + 1900);

    if (to == NULL) {
        currTime = wc_Time(0);
        toTmGmt = XGMTIME(&currTime, tmpTs);
        if (toTmGmt == NULL) {
            WOLFSSL_MSG("XGMTIME for to time failed.");
            return WOLFSSL_FAILURE;
        }
    }
    else if (wolfSSL_ASN1_TIME_to_tm(to, toTmGmt) != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("Failed to convert to time to struct tm.");
        return WOLFSSL_FAILURE;
    }

    toSecs = TimeToUnixTime(toTmGmt->tm_sec, toTmGmt->tm_min, toTmGmt->tm_hour,
                        toTmGmt->tm_mday, toTmGmt->tm_mon,
                        toTmGmt->tm_year + 1900);

    diffSecs = (double)(toSecs - fromSecs);
    *days = (int) (diffSecs / SECS_PER_DAY);
    *secs = (int) (diffSecs - (((double)*days) * SECS_PER_DAY));

    return WOLFSSL_SUCCESS;
}

int wolfSSL_ASN1_TIME_compare(const WOLFSSL_ASN1_TIME *a,
                              const WOLFSSL_ASN1_TIME *b)
{
    int ret;
    int days;
    int secs;

    WOLFSSL_ENTER("wolfSSL_ASN1_TIME_compare");

    if (wolfSSL_ASN1_TIME_diff(&days, &secs, a, b) != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("Failed to get time difference.");
        ret = -2;
    }
    else {
        if (days == 0 && secs == 0) {
            /* a and b are the same time. */
            ret = 0;
        }
        else if (days >= 0 && secs >= 0) {
            /* a is before b. */
            ret = -1;
        }
        else if (days <= 0 && secs <= 0) {
            /* a is after b. */
            ret = 1;
        }
        else {
            WOLFSSL_MSG("Incoherent time difference.");
            ret = -2;
        }
    }

    WOLFSSL_LEAVE("wolfSSL_ASN1_TIME_compare", ret);

    return ret;
}
#endif /* !NO_ASN_TIME */

#ifndef NO_WOLFSSL_STUB
WOLFSSL_ASN1_TIME *wolfSSL_ASN1_TIME_set(WOLFSSL_ASN1_TIME *s, time_t t)
{
    WOLFSSL_STUB("wolfSSL_ASN1_TIME_set");
    (void)s;
    (void)t;
    return s;
}
#endif /* !NO_WOLFSSL_STUB */

int wolfSSL_ASN1_TIME_set_string(WOLFSSL_ASN1_TIME *s, const char *str)
{
    int slen;
    WOLFSSL_ENTER("wolfSSL_ASN1_TIME_set_string");

    if (!str) {
        WOLFSSL_MSG("Bad parameter");
        return WOLFSSL_FAILURE;
    }
    slen = (int)XSTRLEN(str)+1;
    if (slen > CTC_DATE_SIZE) {
        WOLFSSL_MSG("Date string too long");
        return WOLFSSL_FAILURE;
    }
    if (s) {
        XMEMCPY(s->data, str, slen);
        s->length = slen - 1; /* do not include null terminator in length */
        s->type = slen == ASN_UTC_TIME_SIZE ? V_ASN1_UTCTIME :
            V_ASN1_GENERALIZEDTIME;
    }
    return WOLFSSL_SUCCESS;
}

#ifndef NO_BIO

/* Return the month as a string.
 *
 * n  The number of the month as a two characters (1 based).
 * returns the month as a string.
 */
static WC_INLINE const char* MonthStr(const char* n)
{
    static const char monthStr[12][4] = {
            "Jan", "Feb", "Mar", "Apr", "May", "Jun",
            "Jul", "Aug", "Sep", "Oct", "Nov", "Dec" };
    return monthStr[(n[0] - '0') * 10 + (n[1] - '0') - 1];
}

int wolfSSL_ASN1_GENERALIZEDTIME_print(WOLFSSL_BIO* bio,
    const WOLFSSL_ASN1_GENERALIZEDTIME* asnTime)
{
    const char* p;
    WOLFSSL_ENTER("wolfSSL_ASN1_GENERALIZEDTIME_print");

    if (bio == NULL || asnTime == NULL)
        return BAD_FUNC_ARG;

    if (asnTime->type != ASN_GENERALIZED_TIME) {
        WOLFSSL_MSG("Error, not GENERALIZED_TIME");
        return WOLFSSL_FAILURE;
    }
    p = (const char *)(asnTime->data);
    /* GetTimeString not always available. */
    if (wolfSSL_BIO_write(bio, MonthStr(p + 4), 3) <= 0)
        return WOLFSSL_FAILURE;
    if (wolfSSL_BIO_write(bio, " ", 1) <= 0)
        return WOLFSSL_FAILURE;

    /* Day */
    if (wolfSSL_BIO_write(bio, p + 6, 2) <= 0)
        return WOLFSSL_FAILURE;
    if (wolfSSL_BIO_write(bio, " ", 1) <= 0)
        return WOLFSSL_FAILURE;

    /* Hour */
    if (wolfSSL_BIO_write(bio, p + 8, 2) <= 0)
        return WOLFSSL_FAILURE;
    if (wolfSSL_BIO_write(bio, ":", 1) <= 0)
        return WOLFSSL_FAILURE;

    /* Min */
    if (wolfSSL_BIO_write(bio, p + 10, 2) <= 0)
        return WOLFSSL_FAILURE;
    if (wolfSSL_BIO_write(bio, ":", 1) <= 0)
        return WOLFSSL_FAILURE;

    /* Secs */
    if (wolfSSL_BIO_write(bio, p + 12, 2) <= 0)
        return WOLFSSL_FAILURE;
    if (wolfSSL_BIO_write(bio, " ", 1) <= 0)
        return WOLFSSL_FAILURE;
    if (wolfSSL_BIO_write(bio, p, 4) <= 0)
        return WOLFSSL_FAILURE;

    return 0;
}
#endif /* !NO_BIO */

void wolfSSL_ASN1_GENERALIZEDTIME_free(WOLFSSL_ASN1_TIME* asn1Time)
{
    WOLFSSL_ENTER("wolfSSL_ASN1_GENERALIZEDTIME_free");
    if (asn1Time == NULL)
        return;
    XMEMSET(asn1Time->data, 0, sizeof(asn1Time->data));
}

#endif /* OPENSSL_EXTRA */

#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_WPAS_SMALL)
int wolfSSL_sk_num(const WOLFSSL_STACK* sk)
{
    WOLFSSL_ENTER("wolfSSL_sk_num");
    if (sk == NULL)
        return 0;
    return (int)sk->num;
}

void* wolfSSL_sk_value(const WOLFSSL_STACK* sk, int i)
{
    WOLFSSL_ENTER("wolfSSL_sk_value");

    for (; sk != NULL && i > 0; i--)
        sk = sk->next;
    if (sk == NULL)
        return NULL;

    switch (sk->type) {
        case STACK_TYPE_X509:
            return (void*)sk->data.x509;
        case STACK_TYPE_GEN_NAME:
            return (void*)sk->data.gn;
        case STACK_TYPE_BIO:
            return (void*)sk->data.bio;
        case STACK_TYPE_OBJ:
            return (void*)sk->data.obj;
        case STACK_TYPE_STRING:
            return (void*)sk->data.string;
        case STACK_TYPE_CIPHER:
            return (void*)&sk->data.cipher;
        case STACK_TYPE_ACCESS_DESCRIPTION:
            return (void*)sk->data.access;
        case STACK_TYPE_X509_EXT:
            return (void*)sk->data.ext;
        case STACK_TYPE_NULL:
            return (void*)sk->data.generic;
        case STACK_TYPE_X509_NAME:
            return (void*)sk->data.name;
        case STACK_TYPE_X509_NAME_ENTRY:
            return (void*)sk->data.name_entry;
        case STACK_TYPE_CONF_VALUE:
            return (void*)sk->data.conf;
        case STACK_TYPE_X509_INFO:
            return (void*)sk->data.info;
        case STACK_TYPE_BY_DIR_entry:
            return (void*)sk->data.dir_entry;
        case STACK_TYPE_BY_DIR_hash:
            return (void*)sk->data.dir_hash;
        case STACK_TYPE_X509_OBJ:
            return (void*)sk->data.x509_obj;
        case STACK_TYPE_DIST_POINT:
            return (void*)sk->data.dp;
        case STACK_TYPE_X509_CRL:
            return (void*)sk->data.crl;
        default:
            return (void*)sk->data.generic;
    }
}

/* copies over data of "in" to "out" */
static void wolfSSL_CIPHER_copy(WOLFSSL_CIPHER* in, WOLFSSL_CIPHER* out)
{
    if (in == NULL || out == NULL)
        return;

    *out = *in;
}

WOLFSSL_STACK* wolfSSL_sk_dup(WOLFSSL_STACK* sk)
{

    WOLFSSL_STACK* ret = NULL;
    WOLFSSL_STACK* last = NULL;

    WOLFSSL_ENTER("wolfSSL_sk_dup");

    while (sk) {
        WOLFSSL_STACK* cur = wolfSSL_sk_new_node(sk->heap);

        if (!cur) {
            WOLFSSL_MSG("wolfSSL_sk_new_node error");
            goto error;
        }

        if (!ret) {
            /* Set first node */
            ret = cur;
        }

        if (last) {
            last->next = cur;
        }

        XMEMCPY(cur, sk, sizeof(WOLFSSL_STACK));

        /* We will allocate new memory for this */
        XMEMSET(&cur->data, 0, sizeof(cur->data));
        cur->next = NULL;

        switch (sk->type) {
            case STACK_TYPE_X509:
                if (!sk->data.x509)
                    break;
                cur->data.x509 = wolfSSL_X509_dup(sk->data.x509);
                if (!cur->data.x509) {
                    WOLFSSL_MSG("wolfSSL_X509_dup error");
                    goto error;
                }
                break;
            case STACK_TYPE_CIPHER:
                wolfSSL_CIPHER_copy(&sk->data.cipher, &cur->data.cipher);
                break;
            case STACK_TYPE_GEN_NAME:
                if (!sk->data.gn)
                    break;
                cur->data.gn = wolfSSL_GENERAL_NAME_dup(sk->data.gn);
                if (!cur->data.gn) {
                    WOLFSSL_MSG("wolfSSL_GENERAL_NAME_new error");
                    goto error;
                }
                break;
            case STACK_TYPE_OBJ:
                if (!sk->data.obj)
                    break;
                cur->data.obj = wolfSSL_ASN1_OBJECT_dup(sk->data.obj);
                if (!cur->data.obj) {
                    WOLFSSL_MSG("wolfSSL_ASN1_OBJECT_dup error");
                    goto error;
                }
                break;
            case STACK_TYPE_BIO:
            case STACK_TYPE_STRING:
            case STACK_TYPE_ACCESS_DESCRIPTION:
            case STACK_TYPE_X509_EXT:
            case STACK_TYPE_NULL:
            case STACK_TYPE_X509_NAME:
            case STACK_TYPE_X509_NAME_ENTRY:
            case STACK_TYPE_CONF_VALUE:
            case STACK_TYPE_X509_INFO:
            case STACK_TYPE_BY_DIR_entry:
            case STACK_TYPE_BY_DIR_hash:
            case STACK_TYPE_X509_OBJ:
            case STACK_TYPE_DIST_POINT:
            case STACK_TYPE_X509_CRL:
            default:
                WOLFSSL_MSG("Unsupported stack type");
                goto error;
        }

        sk = sk->next;
        last = cur;
    }
    return ret;

error:
    if (ret) {
        wolfSSL_sk_GENERAL_NAME_free(ret);
    }
    return NULL;
}

/* Free the just the stack structure */
void wolfSSL_sk_free(WOLFSSL_STACK* sk)
{
    WOLFSSL_ENTER("wolfSSL_sk_free");

    while (sk != NULL) {
        WOLFSSL_STACK* next = sk->next;
        XFREE(sk, NULL, DYNAMIC_TYPE_OPENSSL);
        sk = next;
    }
}

/* Frees each node in the stack and frees the stack.
 */
void wolfSSL_sk_GENERIC_pop_free(WOLFSSL_STACK* sk,
    void (*f) (void*))
{
    WOLFSSL_ENTER("wolfSSL_sk_GENERIC_pop_free");
    wolfSSL_sk_pop_free(sk, (wolfSSL_sk_freefunc)f);
}

/* return 1 on success 0 on fail */
int wolfSSL_sk_GENERIC_push(WOLFSSL_STACK* sk, void* generic)
{
    WOLFSSL_ENTER("wolfSSL_sk_GENERIC_push");

    return wolfSSL_sk_push(sk, generic);
}
void wolfSSL_sk_GENERIC_free(WOLFSSL_STACK* sk)
{
    wolfSSL_sk_free(sk);
}
#endif /* OPENSSL_EXTRA || WOLFSSL_WPAS_SMALL */

#ifdef OPENSSL_EXTRA

/* Free all nodes in a stack including the pushed objects */
void wolfSSL_sk_pop_free(WOLF_STACK_OF(WOLFSSL_ASN1_OBJECT)* sk,
                                                       wolfSSL_sk_freefunc func)
{
    WOLFSSL_ENTER("wolfSSL_sk_pop_free");

    if (sk == NULL) {
        /* pop_free can be called with NULL, do not print bad argument */
        return;
    }
    #if defined(WOLFSSL_QT)
    /* In Qt v15.5, it calls OPENSSL_sk_free(xxx, OPENSSL_sk_free).
    *  By using OPENSSL_sk_free for free causes access violation.
    *  Therefore, switching free func to wolfSSL_ACCESS_DESCRIPTION_free
    *  is needed even the func isn't NULL.
    */
    if (sk->type == STACK_TYPE_ACCESS_DESCRIPTION) {
        func = (wolfSSL_sk_freefunc)wolfSSL_ACCESS_DESCRIPTION_free;
    }
    #endif
    if (func == NULL) {
        switch(sk->type) {
            case STACK_TYPE_ACCESS_DESCRIPTION:
            #if defined(OPENSSL_ALL)
                func = (wolfSSL_sk_freefunc)wolfSSL_ACCESS_DESCRIPTION_free;
            #endif
                break;
            case STACK_TYPE_X509:
                func = (wolfSSL_sk_freefunc)wolfSSL_X509_free;
                break;
            case STACK_TYPE_X509_OBJ:
            #ifdef OPENSSL_ALL
                func = (wolfSSL_sk_freefunc)wolfSSL_X509_OBJECT_free;
            #endif
                break;
            case STACK_TYPE_OBJ:
                func = (wolfSSL_sk_freefunc)wolfSSL_ASN1_OBJECT_free;
                break;
            case STACK_TYPE_DIST_POINT:
                func = (wolfSSL_sk_freefunc)wolfSSL_DIST_POINT_free;
                break;
            case STACK_TYPE_GEN_NAME:
                func = (wolfSSL_sk_freefunc)wolfSSL_GENERAL_NAME_free;
                break;
            #if defined(WOLFSSL_NGINX) || defined(WOLFSSL_HAPROXY) || \
                defined(OPENSSL_EXTRA) || defined(OPENSSL_ALL)
            case STACK_TYPE_STRING:
                func = (wolfSSL_sk_freefunc)wolfSSL_WOLFSSL_STRING_free;
                break;
            #endif
            case STACK_TYPE_X509_NAME:
            #if (defined(OPENSSL_EXTRA) || defined(OPENSSL_EXTRA_X509_SMALL)) \
                && !defined(WOLFCRYPT_ONLY)
                func = (wolfSSL_sk_freefunc)wolfSSL_X509_NAME_free;
            #endif
                break;
            case STACK_TYPE_X509_NAME_ENTRY:
            #if (defined(OPENSSL_EXTRA) || defined(OPENSSL_EXTRA_X509_SMALL)) \
                && !defined(WOLFCRYPT_ONLY)
                func = (wolfSSL_sk_freefunc)wolfSSL_X509_NAME_ENTRY_free;
            #endif
                break;
            case STACK_TYPE_X509_EXT:
            #ifdef OPENSSL_ALL
                func = (wolfSSL_sk_freefunc)wolfSSL_X509_EXTENSION_free;
            #endif
                break;
            case STACK_TYPE_CONF_VALUE:
            #ifdef OPENSSL_ALL
                func = (wolfSSL_sk_freefunc)wolfSSL_X509V3_conf_free;
            #endif
                break;
            case STACK_TYPE_X509_INFO:
            #if defined(OPENSSL_ALL)
                func = (wolfSSL_sk_freefunc)wolfSSL_X509_INFO_free;
            #endif
                break;
            case STACK_TYPE_BIO:
#if !defined(NO_BIO)
                func = (wolfSSL_sk_freefunc)wolfSSL_BIO_vfree;
#endif
                break;
            case STACK_TYPE_BY_DIR_entry:
#if defined(OPENSSL_ALL) && !defined(NO_FILESYSTEM) && !defined(NO_WOLFSSL_DIR)
                func = (wolfSSL_sk_freefunc)wolfSSL_BY_DIR_entry_free;
#endif
                break;
            case STACK_TYPE_BY_DIR_hash:
#if defined(OPENSSL_ALL) && !defined(NO_FILESYSTEM) && !defined(NO_WOLFSSL_DIR)
                func = (wolfSSL_sk_freefunc)wolfSSL_BY_DIR_HASH_free;
#endif
                break;
            case STACK_TYPE_X509_CRL:
#if defined(HAVE_CRL) && (defined(OPENSSL_EXTRA) || defined(WOLFSSL_WPAS_SMALL))
                func = (wolfSSL_sk_freefunc)wolfSSL_X509_CRL_free;
#endif
                break;
            case STACK_TYPE_CIPHER:
            case STACK_TYPE_NULL:
            default:
                break;
        }
    }

    while (sk != NULL) {
        WOLFSSL_STACK* next = sk->next;

        if (func != NULL) {
            if (sk->type != STACK_TYPE_CIPHER)
                func(sk->data.generic);
        }
        XFREE(sk, NULL, DYNAMIC_TYPE_OPENSSL);
        sk = next;
    }
}
#endif /* OPENSSL_EXTRA */

#if defined(OPENSSL_EXTRA) || defined(WOLFSSL_WPAS_SMALL)
/* Creates and returns a new null stack. */
WOLFSSL_STACK* wolfSSL_sk_new_null(void)
{
    WOLFSSL_STACK* sk;
    WOLFSSL_ENTER("wolfSSL_sk_new_null");

    sk = (WOLFSSL_STACK*)XMALLOC(sizeof(WOLFSSL_STACK), NULL,
                                 DYNAMIC_TYPE_OPENSSL);
    if (sk == NULL) {
        WOLFSSL_MSG("WOLFSSL_STACK memory error");
        return NULL;
    }

    XMEMSET(sk, 0, sizeof(WOLFSSL_STACK));
    sk->type = STACK_TYPE_NULL;

    return sk;
}

int wolfSSL_sk_SSL_COMP_num(WOLF_STACK_OF(WOLFSSL_COMP)* sk)
{
    if (sk == NULL)
        return 0;
    return (int)sk->num;
}

#endif /* OPENSSL_EXTRA || WOLFSSL_WPAS_SMALL */

#if !defined(NO_SESSION_CACHE) && (defined(OPENSSL_EXTRA) || \
        defined(HAVE_EXT_CACHE))
/* stunnel 4.28 needs
 *
 * Callback that is called if a session tries to resume but could not find
 * the session to resume it.
 */
void wolfSSL_CTX_sess_set_get_cb(WOLFSSL_CTX* ctx,
                    WOLFSSL_SESSION*(*f)(WOLFSSL*, const unsigned char*, int, int*))
{
    if (ctx == NULL)
        return;

#ifdef HAVE_EXT_CACHE
    ctx->get_sess_cb = f;
#else
    (void)f;
#endif
}

void wolfSSL_CTX_sess_set_new_cb(WOLFSSL_CTX* ctx,
                             int (*f)(WOLFSSL*, WOLFSSL_SESSION*))
{
    if (ctx == NULL)
        return;

#ifdef HAVE_EXT_CACHE
    ctx->new_sess_cb = f;
#else
    (void)f;
#endif
}

void wolfSSL_CTX_sess_set_remove_cb(WOLFSSL_CTX* ctx, void (*f)(WOLFSSL_CTX*,
                                                        WOLFSSL_SESSION*))
{
    if (ctx == NULL)
        return;

#ifdef HAVE_EXT_CACHE
    ctx->rem_sess_cb = f;
#else
    (void)f;
#endif
}


/*
 *
 * Note: It is expected that the importing and exporting function have been
 *       built with the same settings. For example if session tickets was
 *       enabled with the wolfSSL library exporting a session then it is
 *       expected to be turned on with the wolfSSL library importing the session.
 */
int wolfSSL_i2d_SSL_SESSION(WOLFSSL_SESSION* sess, unsigned char** p)
{
    int size = 0;
#ifdef HAVE_EXT_CACHE
    int idx = 0;
#ifdef SESSION_CERTS
    int i;
#endif
    unsigned char *data;

    WOLFSSL_ENTER("wolfSSL_i2d_SSL_SESSION");

    sess = ClientSessionToSession(sess);
    if (sess == NULL) {
        return BAD_FUNC_ARG;
    }

    /* side | bornOn | timeout | sessionID len | sessionID | masterSecret |
     * haveEMS  */
    size += OPAQUE8_LEN + OPAQUE32_LEN + OPAQUE32_LEN + OPAQUE8_LEN +
            sess->sessionIDSz + SECRET_LEN + OPAQUE8_LEN;
    /* altSessionID */
    size += OPAQUE8_LEN + (sess->haveAltSessionID ? ID_LEN : 0);
#ifdef SESSION_CERTS
    /* Peer chain */
    size += OPAQUE8_LEN;
    for (i = 0; i < sess->chain.count; i++)
        size += OPAQUE16_LEN + sess->chain.certs[i].length;
#endif
#if defined(SESSION_CERTS) || (defined(WOLFSSL_TLS13) && \
                               defined(HAVE_SESSION_TICKET))
    /* Protocol version */
    size += OPAQUE16_LEN;
#endif
#if defined(SESSION_CERTS) || !defined(NO_RESUME_SUITE_CHECK) || \
                        (defined(WOLFSSL_TLS13) && defined(HAVE_SESSION_TICKET))
    /* cipher suite */
    size += OPAQUE16_LEN;
#endif
#ifndef NO_CLIENT_CACHE
    /* ServerID len | ServerID */
    size += OPAQUE16_LEN + sess->idLen;
#endif
#ifdef OPENSSL_EXTRA
    /* session context ID len | session context ID */
    size += OPAQUE8_LEN + sess->sessionCtxSz;
#endif
#if defined(OPENSSL_EXTRA) || defined(OPENSSL_EXTRA_X509_SMALL)
    /* peerVerifyRet */
    size += OPAQUE8_LEN;
#endif
#ifdef WOLFSSL_TLS13
    /* namedGroup */
    size += OPAQUE16_LEN;
#endif
#if defined(HAVE_SESSION_TICKET) || !defined(NO_PSK)
#ifdef WOLFSSL_TLS13
    /* ticketSeen | ticketAdd */
    size += OPAQUE32_LEN + OPAQUE32_LEN;
    /* ticketNonce */
    size += OPAQUE8_LEN + sess->ticketNonce.len;
#endif
#ifdef WOLFSSL_EARLY_DATA
    size += OPAQUE32_LEN;
#endif
#endif
#ifdef HAVE_SESSION_TICKET
    /* ticket len | ticket */
    size += OPAQUE16_LEN + sess->ticketLen;
#endif

    if (p != NULL) {
        if (*p == NULL)
            *p = (unsigned char*)XMALLOC(size, NULL, DYNAMIC_TYPE_OPENSSL);
        if (*p == NULL)
            return 0;
        data = *p;

        data[idx++] = sess->side;
        c32toa(sess->bornOn, data + idx); idx += OPAQUE32_LEN;
        c32toa(sess->timeout, data + idx); idx += OPAQUE32_LEN;
        data[idx++] = sess->sessionIDSz;
        XMEMCPY(data + idx, sess->sessionID, sess->sessionIDSz);
        idx += sess->sessionIDSz;
        XMEMCPY(data + idx, sess->masterSecret, SECRET_LEN); idx += SECRET_LEN;
        data[idx++] = (byte)sess->haveEMS;
        data[idx++] = sess->haveAltSessionID ? ID_LEN : 0;
        if (sess->haveAltSessionID) {
            XMEMCPY(data + idx, sess->altSessionID, ID_LEN);
            idx += ID_LEN;
        }
#ifdef SESSION_CERTS
        data[idx++] = (byte)sess->chain.count;
        for (i = 0; i < sess->chain.count; i++) {
            c16toa((word16)sess->chain.certs[i].length, data + idx);
            idx += OPAQUE16_LEN;
            XMEMCPY(data + idx, sess->chain.certs[i].buffer,
                    sess->chain.certs[i].length);
            idx += sess->chain.certs[i].length;
        }
#endif
#if defined(SESSION_CERTS) || (defined(WOLFSSL_TLS13) && \
                               defined(HAVE_SESSION_TICKET))
        data[idx++] = sess->version.major;
        data[idx++] = sess->version.minor;
#endif
#if defined(SESSION_CERTS) || !defined(NO_RESUME_SUITE_CHECK) || \
                        (defined(WOLFSSL_TLS13) && defined(HAVE_SESSION_TICKET))
        data[idx++] = sess->cipherSuite0;
        data[idx++] = sess->cipherSuite;
#endif
#ifndef NO_CLIENT_CACHE
        c16toa(sess->idLen, data + idx); idx += OPAQUE16_LEN;
        XMEMCPY(data + idx, sess->serverID, sess->idLen);
        idx += sess->idLen;
#endif
#ifdef OPENSSL_EXTRA
        data[idx++] = sess->sessionCtxSz;
        XMEMCPY(data + idx, sess->sessionCtx, sess->sessionCtxSz);
        idx += sess->sessionCtxSz;
#endif
#if defined(OPENSSL_EXTRA) || defined(OPENSSL_EXTRA_X509_SMALL)
        data[idx++] = sess->peerVerifyRet;
#endif
#ifdef WOLFSSL_TLS13
        c16toa(sess->namedGroup, data + idx);
        idx += OPAQUE16_LEN;
#endif
#if defined(HAVE_SESSION_TICKET) || !defined(NO_PSK)
#ifdef WOLFSSL_TLS13
    c32toa(sess->ticketSeen, data + idx);
    idx += OPAQUE32_LEN;
    c32toa(sess->ticketAdd, data + idx);
    idx += OPAQUE32_LEN;
    data[idx++] = sess->ticketNonce.len;
    XMEMCPY(data + idx, sess->ticketNonce.data, sess->ticketNonce.len);
    idx += sess->ticketNonce.len;
#endif
#ifdef WOLFSSL_EARLY_DATA
        c32toa(sess->maxEarlyDataSz, data + idx);
        idx += OPAQUE32_LEN;
#endif
#endif
#ifdef HAVE_SESSION_TICKET
        c16toa(sess->ticketLen, data + idx); idx += OPAQUE16_LEN;
        XMEMCPY(data + idx, sess->ticket, sess->ticketLen);
        idx += sess->ticketLen;
#endif
    }
#endif

    (void)sess;
    (void)p;
#ifdef HAVE_EXT_CACHE
    (void)idx;
#endif

    return size;
}


/* TODO: no function to free new session.
 *
 * Note: It is expected that the importing and exporting function have been
 *       built with the same settings. For example if session tickets was
 *       enabled with the wolfSSL library exporting a session then it is
 *       expected to be turned on with the wolfSSL library importing the session.
 */
WOLFSSL_SESSION* wolfSSL_d2i_SSL_SESSION(WOLFSSL_SESSION** sess,
                                const unsigned char** p, long i)
{
    WOLFSSL_SESSION* s = NULL;
    int ret = 0;
#if defined(HAVE_EXT_CACHE)
    int idx;
    byte* data;
#ifdef SESSION_CERTS
    int j;
    word16 length;
#endif
#endif /* HAVE_EXT_CACHE */

    (void)p;
    (void)i;
    (void)ret;
    (void)sess;

#ifdef HAVE_EXT_CACHE
    if (p == NULL || *p == NULL)
        return NULL;

    s = wolfSSL_SESSION_new();
    if (s == NULL)
        return NULL;

    idx = 0;
    data = (byte*)*p;

    /* side | bornOn | timeout | sessionID len */
    if (i < OPAQUE8_LEN + OPAQUE32_LEN + OPAQUE32_LEN + OPAQUE8_LEN) {
        ret = BUFFER_ERROR;
        goto end;
    }
    s->side = data[idx++];
    ato32(data + idx, &s->bornOn); idx += OPAQUE32_LEN;
    ato32(data + idx, &s->timeout); idx += OPAQUE32_LEN;
    s->sessionIDSz = data[idx++];

    /* sessionID | secret | haveEMS | haveAltSessionID */
    if (i - idx < s->sessionIDSz + SECRET_LEN + OPAQUE8_LEN + OPAQUE8_LEN) {
        ret = BUFFER_ERROR;
        goto end;
    }
    XMEMCPY(s->sessionID, data + idx, s->sessionIDSz);
    idx  += s->sessionIDSz;
    XMEMCPY(s->masterSecret, data + idx, SECRET_LEN); idx += SECRET_LEN;
    s->haveEMS = data[idx++];
    if (data[idx] != ID_LEN && data[idx] != 0) {
        ret = BUFFER_ERROR;
        goto end;
    }
    s->haveAltSessionID = data[idx++] == ID_LEN;

    /* altSessionID */
    if (s->haveAltSessionID) {
        if (i - idx < ID_LEN) {
            ret = BUFFER_ERROR;
            goto end;
        }
        XMEMCPY(s->altSessionID, data + idx, ID_LEN); idx += ID_LEN;
    }

#ifdef SESSION_CERTS
    /* Certificate chain */
    if (i - idx == 0) {
        ret = BUFFER_ERROR;
        goto end;
    }
    s->chain.count = data[idx++];
    for (j = 0; j < s->chain.count; j++) {
        if (i - idx < OPAQUE16_LEN) {
            ret = BUFFER_ERROR;
            goto end;
        }
        ato16(data + idx, &length); idx += OPAQUE16_LEN;
        s->chain.certs[j].length = length;
        if (i - idx < length) {
            ret = BUFFER_ERROR;
            goto end;
        }
        XMEMCPY(s->chain.certs[j].buffer, data + idx, length);
        idx += length;
    }
#endif
#if defined(SESSION_CERTS) || (defined(WOLFSSL_TLS13) && \
                               defined(HAVE_SESSION_TICKET))
    /* Protocol Version */
    if (i - idx < OPAQUE16_LEN) {
        ret = BUFFER_ERROR;
        goto end;
    }
    s->version.major = data[idx++];
    s->version.minor = data[idx++];
#endif
#if defined(SESSION_CERTS) || !defined(NO_RESUME_SUITE_CHECK) || \
                        (defined(WOLFSSL_TLS13) && defined(HAVE_SESSION_TICKET))
    /* Cipher suite */
    if (i - idx < OPAQUE16_LEN) {
        ret = BUFFER_ERROR;
        goto end;
    }
    s->cipherSuite0 = data[idx++];
    s->cipherSuite = data[idx++];
#endif
#ifndef NO_CLIENT_CACHE
    /* ServerID len */
    if (i - idx < OPAQUE16_LEN) {
        ret = BUFFER_ERROR;
        goto end;
    }
    ato16(data + idx, &s->idLen); idx += OPAQUE16_LEN;

    /* ServerID */
    if (i - idx < s->idLen) {
        ret = BUFFER_ERROR;
        goto end;
    }
    XMEMCPY(s->serverID, data + idx, s->idLen); idx += s->idLen;
#endif
#ifdef OPENSSL_EXTRA
    /* byte for length of session context ID */
    if (i - idx < OPAQUE8_LEN) {
        ret = BUFFER_ERROR;
        goto end;
    }
    s->sessionCtxSz = data[idx++];

    /* app session context ID */
    if (i - idx < s->sessionCtxSz) {
        ret = BUFFER_ERROR;
        goto end;
    }
    XMEMCPY(s->sessionCtx, data + idx, s->sessionCtxSz); idx += s->sessionCtxSz;
#endif
#if defined(OPENSSL_EXTRA) || defined(OPENSSL_EXTRA_X509_SMALL)
    /* byte for peerVerifyRet */
    if (i - idx < OPAQUE8_LEN) {
        ret = BUFFER_ERROR;
        goto end;
    }
    s->peerVerifyRet = data[idx++];
#endif
#ifdef WOLFSSL_TLS13
    if (i - idx < OPAQUE16_LEN) {
        ret = BUFFER_ERROR;
        goto end;
    }
    ato16(data + idx, &s->namedGroup);
    idx += OPAQUE16_LEN;
#endif
#if defined(HAVE_SESSION_TICKET) || !defined(NO_PSK)
#ifdef WOLFSSL_TLS13
    if (i - idx < (OPAQUE32_LEN * 2)) {
        ret = BUFFER_ERROR;
        goto end;
    }
    ato32(data + idx, &s->ticketSeen);
    idx += OPAQUE32_LEN;
    ato32(data + idx, &s->ticketAdd);
    idx += OPAQUE32_LEN;
    if (i - idx < OPAQUE8_LEN) {
        ret = BUFFER_ERROR;
        goto end;
    }
    s->ticketNonce.len = data[idx++];

    if (i - idx < s->ticketNonce.len) {
        ret = BUFFER_ERROR;
        goto end;
    }
    XMEMCPY(s->ticketNonce.data, data + idx, s->ticketNonce.len);
    idx += s->ticketNonce.len;
#endif
#ifdef WOLFSSL_EARLY_DATA
    if (i - idx < OPAQUE32_LEN) {
        ret = BUFFER_ERROR;
        goto end;
    }
    ato32(data + idx, &s->maxEarlyDataSz);
    idx += OPAQUE32_LEN;
#endif
#endif
#ifdef HAVE_SESSION_TICKET
    /* ticket len */
    if (i - idx < OPAQUE16_LEN) {
        ret = BUFFER_ERROR;
        goto end;
    }
    ato16(data + idx, &s->ticketLen); idx += OPAQUE16_LEN;

    /* Dispose of ol dynamic ticket and ensure space for new ticket. */
    if (s->ticketLenAlloc > 0) {
        XFREE(s->ticket, NULL, DYNAMIC_TYPE_SESSION_TICK);
    }
    if (s->ticketLen <= SESSION_TICKET_LEN)
        s->ticket = s->_staticTicket;
    else {
        s->ticket = (byte*)XMALLOC(s->ticketLen, NULL,
                                   DYNAMIC_TYPE_SESSION_TICK);
        if (s->ticket == NULL) {
            ret = MEMORY_ERROR;
            goto end;
        }
        s->ticketLenAlloc = (word16)s->ticketLen;
    }

    /* ticket */
    if (i - idx < s->ticketLen) {
        ret = BUFFER_ERROR;
        goto end;
    }
    XMEMCPY(s->ticket, data + idx, s->ticketLen); idx += s->ticketLen;
#endif
    (void)idx;

    if (sess != NULL) {
        *sess = s;
    }

    *p += idx;

end:
    if (ret != 0 && (sess == NULL || *sess != s)) {
        wolfSSL_SESSION_free(s);
        s = NULL;
    }
#endif /* HAVE_EXT_CACHE */
    return s;
}

/* Check if there is a session ticket associated with this WOLFSSL_SESSION.
 *
 * sess - pointer to WOLFSSL_SESSION struct
 *
 * Returns 1 if has session ticket, otherwise 0 */
int wolfSSL_SESSION_has_ticket(const WOLFSSL_SESSION* sess)
{
    WOLFSSL_ENTER("wolfSSL_SESSION_has_ticket");
#ifdef HAVE_SESSION_TICKET
    sess = ClientSessionToSession(sess);
    if (sess) {
        if ((sess->ticketLen > 0) && (sess->ticket != NULL)) {
            return WOLFSSL_SUCCESS;
        }
    }
#else
    (void)sess;
#endif
    return WOLFSSL_FAILURE;
}

unsigned long wolfSSL_SESSION_get_ticket_lifetime_hint(
                  const WOLFSSL_SESSION* sess)
{
    WOLFSSL_ENTER("wolfSSL_SESSION_get_ticket_lifetime_hint");
    sess = ClientSessionToSession(sess);
    if (sess) {
        return sess->timeout;
    }
    return 0;
}

long wolfSSL_SESSION_get_timeout(const WOLFSSL_SESSION* sess)
{
    long timeout = 0;
    WOLFSSL_ENTER("wolfSSL_SESSION_get_timeout");
    sess = ClientSessionToSession(sess);
    if (sess)
        timeout = sess->timeout;
    return timeout;
}


long wolfSSL_SESSION_get_time(const WOLFSSL_SESSION* sess)
{
    long bornOn = 0;
    WOLFSSL_ENTER("wolfSSL_SESSION_get_time");
    sess = ClientSessionToSession(sess);
    if (sess)
        bornOn = sess->bornOn;
    return bornOn;
}

long wolfSSL_SSL_SESSION_set_timeout(WOLFSSL_SESSION* ses, long t)
{
    word32 tmptime;

    ses = ClientSessionToSession(ses);
    if (ses == NULL || t < 0) {
        return BAD_FUNC_ARG;
    }

    tmptime = t & 0xFFFFFFFF;
    ses->timeout = tmptime;

    return WOLFSSL_SUCCESS;
}

#endif /* !NO_SESSION_CACHE && OPENSSL_EXTRA || HAVE_EXT_CACHE */

#ifdef OPENSSL_EXTRA

#if defined(HAVE_EX_DATA) && !defined(NO_FILESYSTEM)
int wolfSSL_cmp_peer_cert_to_file(WOLFSSL* ssl, const char *fname)
{
    int ret = WOLFSSL_FATAL_ERROR;

    WOLFSSL_ENTER("wolfSSL_cmp_peer_cert_to_file");
    if (ssl != NULL && fname != NULL)
    {
    #ifdef WOLFSSL_SMALL_STACK
        byte           staticBuffer[1]; /* force heap usage */
    #else
        byte           staticBuffer[FILE_BUFFER_SIZE];
    #endif
        byte*          myBuffer  = staticBuffer;
        int            dynamic   = 0;
        XFILE          file;
        long           sz        = 0;
        WOLFSSL_CTX*   ctx       = ssl->ctx;
        WOLFSSL_X509*  peer_cert = &ssl->peerCert;
        DerBuffer*     fileDer = NULL;

        file = XFOPEN(fname, "rb");
        if (file == XBADFILE)
            return WOLFSSL_BAD_FILE;

        if (XFSEEK(file, 0, XSEEK_END) != 0) {
            XFCLOSE(file);
            return WOLFSSL_BAD_FILE;
        }
        sz = XFTELL(file);
        XREWIND(file);

        if (sz > MAX_WOLFSSL_FILE_SIZE || sz < 0) {
            WOLFSSL_MSG("cmp_peer_cert_to_file size error");
            XFCLOSE(file);
            return WOLFSSL_BAD_FILE;
        }

        if (sz > (long)sizeof(staticBuffer)) {
            WOLFSSL_MSG("Getting dynamic buffer");
            myBuffer = (byte*)XMALLOC(sz, ctx->heap, DYNAMIC_TYPE_FILE);
            dynamic = 1;
        }

        if ((myBuffer != NULL) &&
            (sz > 0) &&
            (XFREAD(myBuffer, 1, sz, file) == (size_t)sz) &&
            (PemToDer(myBuffer, (long)sz, CERT_TYPE,
                      &fileDer, ctx->heap, NULL, NULL) == 0) &&
            (fileDer->length != 0) &&
            (fileDer->length == peer_cert->derCert->length) &&
            (XMEMCMP(peer_cert->derCert->buffer, fileDer->buffer,
                                                fileDer->length) == 0))
        {
            ret = 0;
        }

        FreeDer(&fileDer);

        if (dynamic)
            XFREE(myBuffer, ctx->heap, DYNAMIC_TYPE_FILE);

        XFCLOSE(file);
    }

    return ret;
}
#endif
#endif /* OPENSSL_EXTRA */
#if defined(OPENSSL_EXTRA) || defined(OPENSSL_EXTRA_X509_SMALL)
const WOLFSSL_ObjectInfo wolfssl_object_info[] = {
#ifndef NO_CERTS
    /* oidCertExtType */
    { NID_basic_constraints, BASIC_CA_OID, oidCertExtType, "basicConstraints",
                                                "X509v3 Basic Constraints"},
    { NID_subject_alt_name, ALT_NAMES_OID, oidCertExtType, "subjectAltName",
                                         "X509v3 Subject Alternative Name"},
    { NID_crl_distribution_points, CRL_DIST_OID, oidCertExtType, "crlDistributionPoints",
                                          "X509v3 CRL Distribution Points"},
    { NID_info_access, AUTH_INFO_OID, oidCertExtType, "authorityInfoAccess",
                                            "Authority Information Access"},
    { NID_authority_key_identifier, AUTH_KEY_OID, oidCertExtType,
               "authorityKeyIdentifier", "X509v3 Authority Key Identifier"},
    { NID_subject_key_identifier, SUBJ_KEY_OID, oidCertExtType,
                   "subjectKeyIdentifier", "X509v3 Subject Key Identifier"},
    { NID_key_usage, KEY_USAGE_OID, oidCertExtType, "keyUsage",
                                                        "X509v3 Key Usage"},
    { NID_inhibit_any_policy, INHIBIT_ANY_OID, oidCertExtType,
                           "inhibitAnyPolicy", "X509v3 Inhibit Any Policy"},
    { NID_ext_key_usage, EXT_KEY_USAGE_OID, oidCertExtType,
                           "extendedKeyUsage", "X509v3 Extended Key Usage"},
    { NID_name_constraints, NAME_CONS_OID, oidCertExtType,
                              "nameConstraints", "X509v3 Name Constraints"},
    { NID_certificate_policies, CERT_POLICY_OID, oidCertExtType,
                      "certificatePolicies", "X509v3 Certificate Policies"},

    /* oidCertAuthInfoType */
    { NID_ad_OCSP, AIA_OCSP_OID, oidCertAuthInfoType, "OCSP",
                                            "OCSP"},
    { NID_ad_ca_issuers, AIA_CA_ISSUER_OID, oidCertAuthInfoType,
                                                 "caIssuers", "CA Issuers"},

    /* oidCertPolicyType */
    { NID_any_policy, CP_ANY_OID, oidCertPolicyType, "anyPolicy",
                                                       "X509v3 Any Policy"},

    /* oidCertAltNameType */
    { NID_hw_name_oid, HW_NAME_OID, oidCertAltNameType, "Hardware name",""},

    /* oidCertKeyUseType */
    { NID_anyExtendedKeyUsage, EKU_ANY_OID, oidCertKeyUseType,
                           "anyExtendedKeyUsage", "Any Extended Key Usage"},
    { EKU_SERVER_AUTH_OID, EKU_SERVER_AUTH_OID, oidCertKeyUseType,
                             "serverAuth", "TLS Web Server Authentication"},
    { EKU_CLIENT_AUTH_OID, EKU_CLIENT_AUTH_OID, oidCertKeyUseType,
                             "clientAuth", "TLS Web Client Authentication"},
    { EKU_OCSP_SIGN_OID, EKU_OCSP_SIGN_OID, oidCertKeyUseType,
                                             "OCSPSigning", "OCSP Signing"},

    /* oidCertNameType */
    { NID_commonName, NID_commonName, oidCertNameType, "CN", "commonName"},
    { NID_surname, NID_surname, oidCertNameType, "SN", "surname"},
    { NID_serialNumber, NID_serialNumber, oidCertNameType, "serialNumber",
                                                            "serialNumber"},
    { NID_userId, NID_userId, oidCertNameType, "UID", "userid"},
    { NID_countryName, NID_countryName, oidCertNameType, "C", "countryName"},
    { NID_localityName, NID_localityName, oidCertNameType, "L", "localityName"},
    { NID_stateOrProvinceName, NID_stateOrProvinceName, oidCertNameType, "ST",
                                                        "stateOrProvinceName"},
    { NID_streetAddress, NID_streetAddress, oidCertNameType, "street",
                                                        "streetAddress"},
    { NID_organizationName, NID_organizationName, oidCertNameType, "O",
                                                        "organizationName"},
    { NID_organizationalUnitName, NID_organizationalUnitName, oidCertNameType,
                                                "OU", "organizationalUnitName"},
    { NID_emailAddress, NID_emailAddress, oidCertNameType, "emailAddress",
                                                            "emailAddress"},
    { NID_domainComponent, NID_domainComponent, oidCertNameType, "DC",
                                                            "domainComponent"},
    { NID_favouriteDrink, NID_favouriteDrink, oidCertNameType, "favouriteDrink",
                                                            "favouriteDrink"},
    { NID_businessCategory, NID_businessCategory, oidCertNameType, "businessCategory",
                                                            "businessCategory"},
    { NID_jurisdictionCountryName, NID_jurisdictionCountryName, oidCertNameType, "jurisdictionC",
                                                            "jurisdictionCountryName"},
    { NID_jurisdictionStateOrProvinceName, NID_jurisdictionStateOrProvinceName,
            oidCertNameType, "jurisdictionST", "jurisdictionStateOrProvinceName"},
    { NID_postalCode, NID_postalCode, oidCertNameType, "postalCode", "postalCode"},
    { NID_userId, NID_userId, oidCertNameType, "UID", "userId"},

#ifdef WOLFSSL_CERT_REQ
    { NID_pkcs9_challengePassword, CHALLENGE_PASSWORD_OID,
            oidCsrAttrType, "challengePassword", "challengePassword"},
    { NID_pkcs9_contentType, PKCS9_CONTENT_TYPE_OID,
        oidCsrAttrType, "contentType", "contentType" },
#endif
#endif
#ifdef OPENSSL_EXTRA /* OPENSSL_EXTRA_X509_SMALL only needs the above */
        /* oidHashType */
    #ifdef WOLFSSL_MD2
        { NID_md2, MD2h, oidHashType, "MD2", "md2"},
    #endif
    #ifdef WOLFSSL_MD5
        { NID_md5, MD5h, oidHashType, "MD5", "md5"},
    #endif
    #ifndef NO_SHA
        { NID_sha1, SHAh, oidHashType, "SHA1", "sha1"},
    #endif
    #ifdef WOLFSSL_SHA224
        { NID_sha224, SHA224h, oidHashType, "SHA224", "sha224"},
    #endif
    #ifndef NO_SHA256
        { NID_sha256, SHA256h, oidHashType, "SHA256", "sha256"},
    #endif
    #ifdef WOLFSSL_SHA384
        { NID_sha384, SHA384h, oidHashType, "SHA384", "sha384"},
    #endif
    #ifdef WOLFSSL_SHA512
        { NID_sha512, SHA512h, oidHashType, "SHA512", "sha512"},
    #endif
    #ifdef WOLFSSL_SHA3
        #ifndef WOLFSSL_NOSHA3_224
        { NID_sha3_224, SHA3_224h, oidHashType, "SHA3-224", "sha3-224"},
        #endif
        #ifndef WOLFSSL_NOSHA3_256
        { NID_sha3_256, SHA3_256h, oidHashType, "SHA3-256", "sha3-256"},
        #endif
        #ifndef WOLFSSL_NOSHA3_384
        { NID_sha3_384, SHA3_384h, oidHashType, "SHA3-384", "sha3-384"},
        #endif
        #ifndef WOLFSSL_NOSHA3_512
        { NID_sha3_512, SHA3_512h, oidHashType, "SHA3-512", "sha3-512"},
        #endif
    #endif /* WOLFSSL_SHA3 */
        /* oidSigType */
    #ifndef NO_DSA
        #ifndef NO_SHA
        { NID_dsaWithSHA1, CTC_SHAwDSA, oidSigType, "DSA-SHA1", "dsaWithSHA1"},
        { NID_dsa_with_SHA256, CTC_SHA256wDSA, oidSigType, "dsa_with_SHA256",
                                                        "dsa_with_SHA256"},
        #endif
    #endif /* NO_DSA */
    #ifndef NO_RSA
        #ifdef WOLFSSL_MD2
        { NID_md2WithRSAEncryption, CTC_MD2wRSA, oidSigType, "RSA-MD2",
                                                        "md2WithRSAEncryption"},
        #endif
        #ifndef NO_MD5
        { NID_md5WithRSAEncryption, CTC_MD5wRSA, oidSigType, "RSA-MD5",
                                                        "md5WithRSAEncryption"},
        #endif
        #ifndef NO_SHA
        { NID_sha1WithRSAEncryption, CTC_SHAwRSA, oidSigType, "RSA-SHA1",
                                                       "sha1WithRSAEncryption"},
        #endif
        #ifdef WOLFSSL_SHA224
        { NID_sha224WithRSAEncryption, CTC_SHA224wRSA, oidSigType, "RSA-SHA224",
                                                     "sha224WithRSAEncryption"},
        #endif
        #ifndef NO_SHA256
        { NID_sha256WithRSAEncryption, CTC_SHA256wRSA, oidSigType, "RSA-SHA256",
                                                     "sha256WithRSAEncryption"},
        #endif
        #ifdef WOLFSSL_SHA384
        { NID_sha384WithRSAEncryption, CTC_SHA384wRSA, oidSigType, "RSA-SHA384",
                                                     "sha384WithRSAEncryption"},
        #endif
        #ifdef WOLFSSL_SHA512
        { NID_sha512WithRSAEncryption, CTC_SHA512wRSA, oidSigType, "RSA-SHA512",
                                                     "sha512WithRSAEncryption"},
        #endif
        #ifdef WOLFSSL_SHA3
        #ifndef WOLFSSL_NOSHA3_224
        { NID_RSA_SHA3_224, CTC_SHA3_224wRSA, oidSigType, "RSA-SHA3-224",
                                                     "sha3-224WithRSAEncryption"},
        #endif
        #ifndef WOLFSSL_NOSHA3_256
        { NID_RSA_SHA3_256, CTC_SHA3_256wRSA, oidSigType, "RSA-SHA3-256",
                                                     "sha3-256WithRSAEncryption"},
        #endif
        #ifndef WOLFSSL_NOSHA3_384
        { NID_RSA_SHA3_384, CTC_SHA3_384wRSA, oidSigType, "RSA-SHA3-384",
                                                     "sha3-384WithRSAEncryption"},
        #endif
        #ifndef WOLFSSL_NOSHA3_512
        { NID_RSA_SHA3_512, CTC_SHA3_512wRSA, oidSigType, "RSA-SHA3-512",
                                                     "sha3-512WithRSAEncryption"},
        #endif
        #endif
    #endif /* NO_RSA */
    #ifdef HAVE_ECC
        #ifndef NO_SHA
        { NID_ecdsa_with_SHA1, CTC_SHAwECDSA, oidSigType, "ecdsa-with-SHA1", "shaWithECDSA"},
        #endif
        #ifdef WOLFSSL_SHA224
        { NID_ecdsa_with_SHA224, CTC_SHA224wECDSA, oidSigType, "ecdsa-with-SHA224","sha224WithECDSA"},
        #endif
        #ifndef NO_SHA256
        { NID_ecdsa_with_SHA256, CTC_SHA256wECDSA, oidSigType, "ecdsa-with-SHA256","sha256WithECDSA"},
        #endif
        #ifdef WOLFSSL_SHA384
        { NID_ecdsa_with_SHA384, CTC_SHA384wECDSA, oidSigType, "ecdsa-with-SHA384","sha384WithECDSA"},
        #endif
        #ifdef WOLFSSL_SHA512
        { NID_ecdsa_with_SHA512, CTC_SHA512wECDSA, oidSigType, "ecdsa-with-SHA512","sha512WithECDSA"},
        #endif
        #ifdef WOLFSSL_SHA3
        #ifndef WOLFSSL_NOSHA3_224
        { NID_ecdsa_with_SHA3_224, CTC_SHA3_224wECDSA, oidSigType, "id-ecdsa-with-SHA3-224",
                "ecdsa_with_SHA3-224"},
        #endif
        #ifndef WOLFSSL_NOSHA3_256
        { NID_ecdsa_with_SHA3_256, CTC_SHA3_256wECDSA, oidSigType, "id-ecdsa-with-SHA3-256",
                "ecdsa_with_SHA3-256"},
        #endif
        #ifndef WOLFSSL_NOSHA3_384
        { NID_ecdsa_with_SHA3_384, CTC_SHA3_384wECDSA, oidSigType, "id-ecdsa-with-SHA3-384",
                "ecdsa_with_SHA3-384"},
        #endif
        #ifndef WOLFSSL_NOSHA3_512
        { NID_ecdsa_with_SHA3_512, CTC_SHA3_512wECDSA, oidSigType, "id-ecdsa-with-SHA3-512",
                "ecdsa_with_SHA3-512"},
        #endif
        #endif
    #endif /* HAVE_ECC */

        /* oidKeyType */
    #ifndef NO_DSA
        { NID_dsa, DSAk, oidKeyType, "DSA", "dsaEncryption"},
    #endif /* NO_DSA */
    #ifndef NO_RSA
        { NID_rsaEncryption, RSAk, oidKeyType, "rsaEncryption", "rsaEncryption"},
    #endif /* NO_RSA */
    #ifdef HAVE_ECC
        { NID_X9_62_id_ecPublicKey, ECDSAk, oidKeyType, "id-ecPublicKey",
                                                        "id-ecPublicKey"},
    #endif /* HAVE_ECC */
    #ifndef NO_DH
        { NID_dhKeyAgreement, DHk, oidKeyType, "dhKeyAgreement", "dhKeyAgreement"},
    #endif
    #ifdef HAVE_ED448
        { NID_ED448, ED448k,  oidKeyType, "ED448", "ED448"},
    #endif
    #ifdef HAVE_ED25519
        { NID_ED25519, ED25519k,  oidKeyType, "ED25519", "ED25519"},
    #endif
    #ifdef HAVE_PQC
        { CTC_FALCON_LEVEL1, FALCON_LEVEL1k,  oidKeyType, "Falcon Level 1",
                                                          "Falcon Level 1"},
        { CTC_FALCON_LEVEL5, FALCON_LEVEL5k,  oidKeyType, "Falcon Level 5",
                                                          "Falcon Level 5"},
    #endif

        /* oidCurveType */
    #ifdef HAVE_ECC
        { NID_X9_62_prime192v1, ECC_SECP192R1_OID, oidCurveType, "prime192v1", "prime192v1"},
        { NID_X9_62_prime192v2, ECC_PRIME192V2_OID, oidCurveType, "prime192v2", "prime192v2"},
        { NID_X9_62_prime192v3, ECC_PRIME192V3_OID, oidCurveType, "prime192v3", "prime192v3"},

        { NID_X9_62_prime239v1, ECC_PRIME239V1_OID, oidCurveType, "prime239v1", "prime239v1"},
        { NID_X9_62_prime239v2, ECC_PRIME239V2_OID, oidCurveType, "prime239v2", "prime239v2"},
        { NID_X9_62_prime239v3, ECC_PRIME239V3_OID, oidCurveType, "prime239v3", "prime239v3"},

        { NID_X9_62_prime256v1, ECC_SECP256R1_OID, oidCurveType, "prime256v1", "prime256v1"},

        { NID_secp112r1, ECC_SECP112R1_OID,  oidCurveType, "secp112r1", "secp112r1"},
        { NID_secp112r2, ECC_SECP112R2_OID,  oidCurveType, "secp112r2", "secp112r2"},

        { NID_secp128r1, ECC_SECP128R1_OID,  oidCurveType, "secp128r1", "secp128r1"},
        { NID_secp128r2, ECC_SECP128R2_OID,  oidCurveType, "secp128r2", "secp128r2"},

        { NID_secp160r1, ECC_SECP160R1_OID,  oidCurveType, "secp160r1", "secp160r1"},
        { NID_secp160r2, ECC_SECP160R2_OID,  oidCurveType, "secp160r2", "secp160r2"},

        { NID_secp224r1, ECC_SECP224R1_OID,  oidCurveType, "secp224r1", "secp224r1"},
        { NID_secp384r1, ECC_SECP384R1_OID,  oidCurveType, "secp384r1", "secp384r1"},
        { NID_secp521r1, ECC_SECP521R1_OID,  oidCurveType, "secp521r1", "secp521r1"},

        { NID_secp160k1, ECC_SECP160K1_OID,  oidCurveType, "secp160k1", "secp160k1"},
        { NID_secp192k1, ECC_SECP192K1_OID,  oidCurveType, "secp192k1", "secp192k1"},
        { NID_secp224k1, ECC_SECP224K1_OID,  oidCurveType, "secp224k1", "secp224k1"},
        { NID_secp256k1, ECC_SECP256K1_OID,  oidCurveType, "secp256k1", "secp256k1"},

        { NID_brainpoolP160r1, ECC_BRAINPOOLP160R1_OID,  oidCurveType, "brainpoolP160r1", "brainpoolP160r1"},
        { NID_brainpoolP192r1, ECC_BRAINPOOLP192R1_OID,  oidCurveType, "brainpoolP192r1", "brainpoolP192r1"},
        { NID_brainpoolP224r1, ECC_BRAINPOOLP224R1_OID,  oidCurveType, "brainpoolP224r1", "brainpoolP224r1"},
        { NID_brainpoolP256r1, ECC_BRAINPOOLP256R1_OID,  oidCurveType, "brainpoolP256r1", "brainpoolP256r1"},
        { NID_brainpoolP320r1, ECC_BRAINPOOLP320R1_OID,  oidCurveType, "brainpoolP320r1", "brainpoolP320r1"},
        { NID_brainpoolP384r1, ECC_BRAINPOOLP384R1_OID,  oidCurveType, "brainpoolP384r1", "brainpoolP384r1"},
        { NID_brainpoolP512r1, ECC_BRAINPOOLP512R1_OID,  oidCurveType, "brainpoolP512r1", "brainpoolP512r1"},
    #endif /* HAVE_ECC */

        /* oidBlkType */
    #ifdef WOLFSSL_AES_128
        { AES128CBCb, AES128CBCb, oidBlkType, "AES-128-CBC", "aes-128-cbc"},
    #endif
    #ifdef WOLFSSL_AES_192
        { AES192CBCb, AES192CBCb, oidBlkType, "AES-192-CBC", "aes-192-cbc"},
    #endif
    #ifdef WOLFSSL_AES_256
        { AES256CBCb, AES256CBCb, oidBlkType, "AES-256-CBC", "aes-256-cbc"},
    #endif
    #ifndef NO_DES3
        { NID_des, DESb, oidBlkType, "DES-CBC", "des-cbc"},
        { NID_des3, DES3b, oidBlkType, "DES-EDE3-CBC", "des-ede3-cbc"},
    #endif /* !NO_DES3 */

        /* oidOcspType */
    #ifdef HAVE_OCSP
        { NID_id_pkix_OCSP_basic, OCSP_BASIC_OID, oidOcspType, "basicOCSPResponse",
                                                         "Basic OCSP Response"},
        { OCSP_NONCE_OID, OCSP_NONCE_OID, oidOcspType, "Nonce",
                                                                  "OCSP Nonce"},
    #endif /* HAVE_OCSP */

    #ifndef NO_PWDBASED
        /* oidKdfType */
        { PBKDF2_OID, PBKDF2_OID, oidKdfType, "PBKDFv2", "PBKDF2"},

        /* oidPBEType */
        { PBE_SHA1_RC4_128, PBE_SHA1_RC4_128, oidPBEType,
                                 "PBE-SHA1-RC4-128", "pbeWithSHA1And128BitRC4"},
        { PBE_SHA1_DES, PBE_SHA1_DES, oidPBEType, "PBE-SHA1-DES",
                                                       "pbeWithSHA1AndDES-CBC"},
        { PBE_SHA1_DES3, PBE_SHA1_DES3, oidPBEType, "PBE-SHA1-3DES",
                                            "pbeWithSHA1And3-KeyTripleDES-CBC"},
    #endif

        /* oidKeyWrapType */
    #ifdef WOLFSSL_AES_128
        { AES128_WRAP, AES128_WRAP, oidKeyWrapType, "AES-128 wrap", "aes128-wrap"},
    #endif
    #ifdef WOLFSSL_AES_192
        { AES192_WRAP, AES192_WRAP, oidKeyWrapType, "AES-192 wrap", "aes192-wrap"},
    #endif
    #ifdef WOLFSSL_AES_256
        { AES256_WRAP, AES256_WRAP, oidKeyWrapType, "AES-256 wrap", "aes256-wrap"},
    #endif

    #ifndef NO_PKCS7
        #ifndef NO_DH
        /* oidCmsKeyAgreeType */
            #ifndef NO_SHA
        { dhSinglePass_stdDH_sha1kdf_scheme, dhSinglePass_stdDH_sha1kdf_scheme,
                oidCmsKeyAgreeType, "dhSinglePass-stdDH-sha1kdf-scheme", "dhSinglePass-stdDH-sha1kdf-scheme"},
            #endif
            #ifdef WOLFSSL_SHA224
        { dhSinglePass_stdDH_sha224kdf_scheme,
                dhSinglePass_stdDH_sha224kdf_scheme, oidCmsKeyAgreeType,
                "dhSinglePass-stdDH-sha224kdf-scheme", "dhSinglePass-stdDH-sha224kdf-scheme"},
            #endif
            #ifndef NO_SHA256
        { dhSinglePass_stdDH_sha256kdf_scheme,
                        dhSinglePass_stdDH_sha256kdf_scheme, oidCmsKeyAgreeType,
                        "dhSinglePass-stdDH-sha256kdf-scheme", "dhSinglePass-stdDH-sha256kdf-scheme"},
            #endif
            #ifdef WOLFSSL_SHA384
        { dhSinglePass_stdDH_sha384kdf_scheme,
                        dhSinglePass_stdDH_sha384kdf_scheme, oidCmsKeyAgreeType,
                        "dhSinglePass-stdDH-sha384kdf-scheme", "dhSinglePass-stdDH-sha384kdf-scheme"},
            #endif
            #ifdef WOLFSSL_SHA512
        { dhSinglePass_stdDH_sha512kdf_scheme,
                        dhSinglePass_stdDH_sha512kdf_scheme, oidCmsKeyAgreeType,
                        "dhSinglePass-stdDH-sha512kdf-scheme", "dhSinglePass-stdDH-sha512kdf-scheme"},
            #endif
        #endif
    #endif
    #if defined(WOLFSSL_APACHE_HTTPD)
        /* "1.3.6.1.5.5.7.8.7" */
        { NID_id_on_dnsSRV, NID_id_on_dnsSRV, oidCertNameType,
            WOLFSSL_SN_DNS_SRV, WOLFSSL_LN_DNS_SRV },

        /* "1.3.6.1.4.1.311.20.2.3" */
        { NID_ms_upn, WOLFSSL_MS_UPN_SUM, oidCertExtType, WOLFSSL_SN_MS_UPN,
            WOLFSSL_LN_MS_UPN },

        /* "1.3.6.1.5.5.7.1.24" */
        { NID_tlsfeature, WOLFSSL_TLS_FEATURE_SUM, oidTlsExtType,
            WOLFSSL_SN_TLS_FEATURE, WOLFSSL_LN_TLS_FEATURE },
    #endif
#endif /* OPENSSL_EXTRA */
};

#define WOLFSSL_OBJECT_INFO_SZ \
                (sizeof(wolfssl_object_info) / sizeof(*wolfssl_object_info))
const size_t wolfssl_object_info_sz = WOLFSSL_OBJECT_INFO_SZ;
#endif
#if defined(OPENSSL_EXTRA) && \
    !defined(NO_RSA) && !defined(HAVE_USER_RSA) && !defined(HAVE_FAST_RSA)
WC_RNG* WOLFSSL_RSA_GetRNG(WOLFSSL_RSA *rsa, WC_RNG **tmpRNG, int *initTmpRng)
{
    WC_RNG* rng = NULL;

    if (!rsa || !initTmpRng) {
        return NULL;
    }
    *initTmpRng = 0;

#if !defined(HAVE_FIPS) && !defined(HAVE_USER_RSA) && \
    !defined(HAVE_FAST_RSA) && defined(WC_RSA_BLINDING)
    rng = ((RsaKey*)rsa->internal)->rng;
#endif
    if (tmpRNG != NULL
    #if !defined(HAVE_FIPS) && !defined(HAVE_USER_RSA) && \
        !defined(HAVE_FAST_RSA) && defined(WC_RSA_BLINDING)
        && rng == NULL
    #endif
        ) {
        if (*tmpRNG == NULL) {
#ifdef WOLFSSL_SMALL_STACK
            *tmpRNG = (WC_RNG*)XMALLOC(sizeof(WC_RNG), NULL, DYNAMIC_TYPE_TMP_BUFFER);
            if (*tmpRNG == NULL)
                return NULL;
#else
            WOLFSSL_MSG("*tmpRNG is null");
            return NULL;
#endif
        }

        if (wc_InitRng(*tmpRNG) == 0) {
            rng = *tmpRNG;
            *initTmpRng = 1;
        }
        else {
            WOLFSSL_MSG("Bad RNG Init, trying global");
            if (initGlobalRNG == 0)
                WOLFSSL_MSG("Global RNG no Init");
            else
                rng = &globalRNG;
#ifdef WOLFSSL_SMALL_STACK
            if (*tmpRNG)
                XFREE(*tmpRNG, NULL, DYNAMIC_TYPE_TMP_BUFFER);
            *tmpRNG = NULL;
#endif
        }
    }
    return rng;
}
#endif



#ifdef OPENSSL_EXTRA

WOLFSSL_ASN1_INTEGER* wolfSSL_BN_to_ASN1_INTEGER(const WOLFSSL_BIGNUM *bn, WOLFSSL_ASN1_INTEGER *ai)
{
    WOLFSSL_ASN1_INTEGER* a;
    int len;
    const int extraTagSz = MAX_LENGTH_SZ + 1;
    byte intTag[MAX_LENGTH_SZ + 1];
    int idx = 0;
    WOLFSSL_ENTER("wolfSSL_BN_to_ASN1_INTEGER");

    if (ai == NULL) {
        a = wolfSSL_ASN1_INTEGER_new();

        if (a == NULL)
            return NULL;

        a->type = V_ASN1_INTEGER;
    }
    else {
        a = ai;
    }
    if (a) {
        if (wolfSSL_BN_is_negative(bn) && !wolfSSL_BN_is_zero(bn)) {
            a->type |= V_ASN1_NEG_INTEGER;
            a->negative = 1;
        }

        len = wolfSSL_BN_num_bytes(bn);
        if (len == 0)
            len = 1;

        /* allocate buffer */
        if (len + extraTagSz > (int)sizeof(a->intData)) {
            /* create new data buffer and copy over */
            a->data = (byte*)XMALLOC(len + extraTagSz, NULL,
                    DYNAMIC_TYPE_OPENSSL);
            if (a->data == NULL) {
                if (a != ai)
                    wolfSSL_ASN1_INTEGER_free(a);
                return NULL;
            }
            a->isDynamic = 1;
        }
        else {
            XMEMSET(a->intData, 0, sizeof(a->intData));
            a->data = a->intData;
            a->isDynamic = 0;
        }

        /* populate data */
        if (wolfSSL_BN_is_zero(bn)) {
            a->data[0] = 0;
        }
        else {
            len = wolfSSL_BN_bn2bin(bn, a->data);
            if (len < 0) {
                wolfSSL_ASN1_INTEGER_free(a);
                return NULL;
            }
        }
        a->length = len;

        /* Write ASN tag */
        idx = SetASNInt(a->length, a->data[0], intTag);
        XMEMMOVE(a->data + idx, a->data, a->length);
        XMEMCPY(a->data, intTag, idx);
        a->dataMax = a->length += idx;
    }

    return a;
}

#ifdef OPENSSL_ALL
void *wolfSSL_ASN1_item_new(const WOLFSSL_ASN1_ITEM *tpl)
{
    void *ret = NULL;
    const WOLFSSL_ASN1_TEMPLATE *member = NULL;
    size_t i;
    WOLFSSL_ENTER("wolfSSL_ASN1_item_new");
    if (!tpl) {
        return NULL;
    }
    if (!(ret = (void *)XMALLOC(tpl->size, NULL, DYNAMIC_TYPE_OPENSSL))) {
        return NULL;
    }
    XMEMSET(ret, 0, tpl->size);
    for (member = tpl->members, i = 0; i < tpl->mcount;
            member++, i++) {
        switch (member->type) {
            case WOLFSSL_X509_ALGOR_ASN1:
            {
                WOLFSSL_X509_ALGOR* algor = wolfSSL_X509_ALGOR_new();
                if (!algor) {
                    goto error;
                }
                *(WOLFSSL_X509_ALGOR**)(((byte*)ret) + member->offset) = algor;
                break;
            }
            case WOLFSSL_ASN1_BIT_STRING_ASN1:
            {
                WOLFSSL_ASN1_BIT_STRING* bit_str = wolfSSL_ASN1_BIT_STRING_new();
                if (!bit_str) {
                    goto error;
                }
                *(WOLFSSL_ASN1_BIT_STRING**)(((byte*)ret) + member->offset) = bit_str;
                break;
            }
            default:
                WOLFSSL_MSG("Type not supported in wolfSSL_ASN1_item_new");
                goto error;
        }
    }
    return ret;
error:
    wolfSSL_ASN1_item_free(ret, tpl);
    return NULL;
}

void wolfSSL_ASN1_item_free(void *val, const WOLFSSL_ASN1_ITEM *tpl)
{
    const WOLFSSL_ASN1_TEMPLATE *member = NULL;
    size_t i;
    WOLFSSL_ENTER("wolfSSL_ASN1_item_free");
    if (val) {
        for (member = tpl->members, i = 0; i < tpl->mcount;
                member++, i++) {
            switch (member->type) {
                case WOLFSSL_X509_ALGOR_ASN1:
                {
                    WOLFSSL_X509_ALGOR* algor = *(WOLFSSL_X509_ALGOR**)
                                                 (((byte*)val) + member->offset);
                    if (algor) {
                        wolfSSL_X509_ALGOR_free(algor);
                    }
                    break;
                }
                case WOLFSSL_ASN1_BIT_STRING_ASN1:
                {
                    WOLFSSL_ASN1_BIT_STRING* bit_str = *(WOLFSSL_ASN1_BIT_STRING**)
                                                        (((byte*)val) + member->offset);
                    if (bit_str) {
                        wolfSSL_ASN1_BIT_STRING_free(bit_str);
                    }
                    break;
                }
                default:
                    WOLFSSL_MSG("Type not supported in wolfSSL_ASN1_item_free");
            }
        }
        XFREE(val, NULL, DYNAMIC_TYPE_OPENSSL);
    }
}

#define bufLenOrNull(buf, len) ((buf) ? (buf) + (len) : NULL)

static int i2dProcessMembers(const void *src, byte *buf,
                          const WOLFSSL_ASN1_TEMPLATE *members, size_t mcount)
{
    const WOLFSSL_ASN1_TEMPLATE *member = NULL;
    int len = 0, ret;
    size_t i;
    WOLFSSL_ENTER("processMembers");
    for (member = members, i = 0; i < mcount; member++, i++) {
        switch (member->type) {
            case WOLFSSL_X509_ALGOR_ASN1:
            {
                word32 oid = 0;
                word32 idx = 0;
                const WOLFSSL_X509_ALGOR* algor = *(const WOLFSSL_X509_ALGOR**)
                                                   (((byte*)src) + member->offset);
                if (!algor->algorithm) {
                    WOLFSSL_LEAVE("processMembers", WOLFSSL_FAILURE);
                    return WOLFSSL_FAILURE;
                }

                if (GetObjectId(algor->algorithm->obj, &idx, &oid,
                        algor->algorithm->grp, algor->algorithm->objSz) < 0) {
                    WOLFSSL_MSG("Issue getting OID of object");
                    return -1;
                }

                ret = SetAlgoID(oid, bufLenOrNull(buf, len),
                                algor->algorithm->grp, 0);
                if (!ret) {
                    return WOLFSSL_FAILURE;
                }
                len += ret;
                break;
            }
            case WOLFSSL_ASN1_BIT_STRING_ASN1:
            {
                const WOLFSSL_ASN1_BIT_STRING* bit_str;
                bit_str = *(const WOLFSSL_ASN1_BIT_STRING**)
                           (((byte*)src) + member->offset);
                len += SetBitString(bit_str->length, 0, bufLenOrNull(buf, len));
                if (buf && bit_str->data) {
                    XMEMCPY(buf + len, bit_str->data, bit_str->length);
                }
                len += bit_str->length;
                break;
            }
            default:
                WOLFSSL_MSG("Type not support in processMembers");
                WOLFSSL_LEAVE("processMembers", WOLFSSL_FAILURE);
                return WOLFSSL_FAILURE;
        }
    }
    WOLFSSL_LEAVE("processMembers", len);
    return len;
}

static int wolfSSL_ASN1_item_i2d_1(const void *src, byte *buf,
                                       const WOLFSSL_ASN1_ITEM *tpl, int *len)
{
    *len = 0;

    switch (tpl->type) {
        case ASN_SEQUENCE:
        {
            int seq_len = i2dProcessMembers(src, NULL, tpl->members,
                                         tpl->mcount);
            if (seq_len == WOLFSSL_FAILURE)
                return WOLFSSL_FAILURE;
            *len += SetSequence(seq_len, bufLenOrNull(buf, *len));
            if (buf) {
                if (i2dProcessMembers(src, bufLenOrNull(buf, *len), tpl->members,
                                      tpl->mcount) != seq_len) {
                    WOLFSSL_MSG("Inconsistent sequence length");
                    return WOLFSSL_FAILURE;
                }
            }
            *len += seq_len;
            break;
        }
        default:
            WOLFSSL_MSG("Type not supported in wolfSSL_ASN1_item_i2d");
            return WOLFSSL_FAILURE;
    }

    return WOLFSSL_SUCCESS;
}

int wolfSSL_ASN1_item_i2d(const void *src, byte **dest,
                          const WOLFSSL_ASN1_ITEM *tpl)
{
    int len;
    byte *buf = NULL;

    WOLFSSL_ENTER("wolfSSL_ASN1_item_i2d");

    if ((src == NULL) || (tpl == NULL))
        goto error;

    if (wolfSSL_ASN1_item_i2d_1(src, NULL, tpl, &len) != WOLFSSL_SUCCESS)
        goto error;

    if (dest == NULL) {
        WOLFSSL_LEAVE("wolfSSL_ASN1_item_i2d", WOLFSSL_SUCCESS);
        return len;
    }

    if (*dest == NULL) {
        buf = (byte*)XMALLOC(len, NULL, DYNAMIC_TYPE_ASN1);
        if (buf == NULL)
            goto error;
    } else
        buf = *dest;

    if (wolfSSL_ASN1_item_i2d_1(src, buf, tpl, &len) != WOLFSSL_SUCCESS)
        goto error;

    if (*dest == NULL)
        *dest = buf;
    else {
        /* XXX *dest length is not checked because the user is responsible
         * for providing a long enough buffer
         */
        XMEMCPY(*dest, buf, len);
    }

    WOLFSSL_LEAVE("wolfSSL_ASN1_item_i2d", len);
    return len;
error:
    if (buf) {
        XFREE(buf, NULL, DYNAMIC_TYPE_ASN1);
    }
    WOLFSSL_LEAVE("wolfSSL_ASN1_item_i2d", WOLFSSL_FAILURE);
    return WOLFSSL_FAILURE;
}

#endif /* OPENSSL_ALL */

#ifndef NO_DH

static void InitwolfSSL_DH(WOLFSSL_DH* dh)
{
    if (dh) {
        XMEMSET(dh, 0, sizeof(WOLFSSL_DH));
    }
}


WOLFSSL_DH* wolfSSL_DH_new(void)
{
    WOLFSSL_DH* external;
    DhKey*     key;

    WOLFSSL_ENTER("wolfSSL_DH_new");

    key = (DhKey*) XMALLOC(sizeof(DhKey), NULL, DYNAMIC_TYPE_DH);
    if (key == NULL) {
        WOLFSSL_MSG("wolfSSL_DH_new malloc DhKey failure");
        return NULL;
    }

    external = (WOLFSSL_DH*) XMALLOC(sizeof(WOLFSSL_DH), NULL,
                                    DYNAMIC_TYPE_DH);
    if (external == NULL) {
        WOLFSSL_MSG("wolfSSL_DH_new malloc WOLFSSL_DH failure");
        XFREE(key, NULL, DYNAMIC_TYPE_DH);
        return NULL;
    }

    InitwolfSSL_DH(external);

    external->refCount = 1;
#ifndef SINGLE_THREADED
    if (wc_InitMutex(&external->refMutex) != 0) {
        WOLFSSL_MSG("wc_InitMutex WOLFSSL_DH failure");
        XFREE(key, NULL, DYNAMIC_TYPE_DH);
        XFREE(external, NULL, DYNAMIC_TYPE_DH);
        return NULL;
    }
#endif

    if (wc_InitDhKey(key) != 0) {
        WOLFSSL_MSG("wolfSSL_DH_new InitDhKey failure");
        XFREE(key, NULL, DYNAMIC_TYPE_DH);
        XFREE(external, NULL, DYNAMIC_TYPE_DH);
        return NULL;
    }
    external->internal = key;
    external->priv_key = wolfSSL_BN_new();
    external->pub_key = wolfSSL_BN_new();

    return external;
}

WOLFSSL_DH* wolSSL_DH_new_by_nid(int nid)
{
    WOLFSSL_DH* dh = NULL;
    int err = 0;
#if defined(HAVE_PUBLIC_FFDHE) || (defined(HAVE_FIPS) && FIPS_VERSION_EQ(2,0))
    const DhParams* params = NULL;
    WOLFSSL_BIGNUM* pBn = NULL;
    WOLFSSL_BIGNUM* gBn = NULL;
    WOLFSSL_BIGNUM* qBn = NULL;
#elif !defined(HAVE_PUBLIC_FFDHE) && (!defined(HAVE_FIPS) || \
      FIPS_VERSION_GT(2,0))
    int name = 0;
#ifdef HAVE_FFDHE_Q
    int elements = ELEMENT_P | ELEMENT_G | ELEMENT_Q;
#else
    int elements = ELEMENT_P | ELEMENT_G;
#endif /* HAVE_FFDHE_Q */
#endif /* HAVE_PUBLIC_FFDHE || (HAVE_FIPS && HAVE_FIPS_VERSION == 2) */

    WOLFSSL_ENTER("wolfSSL_DH_new_by_nid");

/* HAVE_PUBLIC_FFDHE not required to expose wc_Dh_ffdhe* functions in FIPS v2
 * module */
#if defined(HAVE_PUBLIC_FFDHE) || (defined(HAVE_FIPS) && FIPS_VERSION_EQ(2,0))
    switch (nid) {
#ifdef HAVE_FFDHE_2048
    case NID_ffdhe2048:
        params = wc_Dh_ffdhe2048_Get();
        break;
#endif /* HAVE_FFDHE_2048 */
#ifdef HAVE_FFDHE_3072
    case NID_ffdhe3072:
        params = wc_Dh_ffdhe3072_Get();
        break;
#endif /* HAVE_FFDHE_3072 */
#ifdef HAVE_FFDHE_4096
    case NID_ffdhe4096:
        params = wc_Dh_ffdhe4096_Get();
        break;
#endif /* HAVE_FFDHE_4096 */
    default:
        break;
    }
    if (params == NULL) {
        WOLFSSL_MSG("Unable to find DH params for nid.");
        err = 1;
    }
    if (err == 0) {
        dh = wolfSSL_DH_new();
        if (dh == NULL) {
            WOLFSSL_MSG("Failed to create WOLFSSL_DH.");
            err = 1;
        }
    }
    if (err == 0) {
        pBn = wolfSSL_BN_bin2bn(params->p, params->p_len, NULL);
        if (pBn == NULL) {
            WOLFSSL_MSG("Error converting p hex to WOLFSSL_BIGNUM.");
            err = 1;
        }
    }
    if (err == 0) {
        gBn = wolfSSL_BN_bin2bn(params->g, params->g_len, NULL);
        if (gBn == NULL) {
            WOLFSSL_MSG("Error converting g hex to WOLFSSL_BIGNUM.");
            err = 1;
        }
    }
#ifdef HAVE_FFDHE_Q
    if (err == 0) {
        qBn = wolfSSL_BN_bin2bn(params->q, params->q_len, NULL);
        if (qBn == NULL) {
            WOLFSSL_MSG("Error converting q hex to WOLFSSL_BIGNUM.");
            err = 1;
        }
    }
#endif
#if defined(OPENSSL_ALL) || defined(OPENSSL_VERSION_NUMBER) && \
    OPENSSL_VERSION_NUMBER >= 0x10100000L
    if (err == 0 && wolfSSL_DH_set0_pqg(dh, pBn, qBn, gBn) != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("Failed to set DH params.");
        err = 1;
    }
#else
    if (err == 0) {
        dh->p = pBn;
        dh->q = qBn;
        dh->g = gBn;
        if (SetDhInternal(dh) != WOLFSSL_SUCCESS) {
            WOLFSSL_MSG("Failed to set internal DH params.");
            err = 1;
        }
    }
#endif /* OPENSSL_ALL || OPENSSL_VERSION_NUMBER >= 0x10100000L */

    if (err == 1) {
        wolfSSL_BN_free(pBn);
        wolfSSL_BN_free(gBn);
        wolfSSL_BN_free(qBn);
    }
/* FIPS v2 and lower doesn't support wc_DhSetNamedKey. */
#elif !defined(HAVE_PUBLIC_FFDHE) && (!defined(HAVE_FIPS) || \
      FIPS_VERSION_GT(2,0))
    switch (nid) {
#ifdef HAVE_FFDHE_2048
    case NID_ffdhe2048:
        name = WC_FFDHE_2048;
        break;
#endif /* HAVE_FFDHE_2048 */
#ifdef HAVE_FFDHE_3072
    case NID_ffdhe3072:
        name = WC_FFDHE_3072;
        break;
#endif /* HAVE_FFDHE_3072 */
#ifdef HAVE_FFDHE_4096
    case NID_ffdhe4096:
        name = WC_FFDHE_4096;
        break;
#endif /* HAVE_FFDHE_4096 */
    default:
        err = 1;
        WOLFSSL_MSG("Unable to find DH params for nid.");
        break;
    }
    if (err == 0) {
        dh = wolfSSL_DH_new();
        if (dh == NULL) {
            WOLFSSL_MSG("Failed to create WOLFSSL_DH.");
            err = 1;
        }
    }
    if (err == 0 && wc_DhSetNamedKey((DhKey*)dh->internal, name) != 0) {
        WOLFSSL_MSG("wc_DhSetNamedKey failed.");
        err = 1;
    }
    if (err == 0 && SetDhExternal_ex(dh, elements) != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("Failed to set external DH params.");
        err = 1;
    }
#else
    /* Unsupported configuration. */
    err = 1;
#endif /* HAVE_PUBLIC_FFDHE || (HAVE_FIPS && HAVE_FIPS_VERSION == 2) */

    if (err == 1 && dh != NULL) {
        wolfSSL_DH_free(dh);
        dh = NULL;
    }

    WOLFSSL_LEAVE("wolfSSL_DH_new_by_nid", err);

    return dh;
}

void wolfSSL_DH_free(WOLFSSL_DH* dh)
{
    int doFree = 0;

    WOLFSSL_ENTER("wolfSSL_DH_free");

    if (dh) {

    #ifndef SINGLE_THREADED
        if (wc_LockMutex(&dh->refMutex) != 0) {
            WOLFSSL_MSG("Could not lock DH mutex");
        }
    #endif
        /* only free if all references to it are done */
        dh->refCount--;
        if (dh->refCount == 0) {
            doFree = 1;
        }
    #ifndef SINGLE_THREADED
        wc_UnLockMutex(&dh->refMutex);
    #endif

        if (doFree == 0) {
            return;
        }

        if (dh->internal) {
            wc_FreeDhKey((DhKey*)dh->internal);
            XFREE(dh->internal, NULL, DYNAMIC_TYPE_DH);
            dh->internal = NULL;
        }
        wolfSSL_BN_free(dh->priv_key);
        wolfSSL_BN_free(dh->pub_key);
        wolfSSL_BN_free(dh->g);
        wolfSSL_BN_free(dh->p);
        wolfSSL_BN_free(dh->q);
        InitwolfSSL_DH(dh);  /* set back to NULLs for safety */

        XFREE(dh, NULL, DYNAMIC_TYPE_DH);
    }
}

int wolfSSL_DH_up_ref(WOLFSSL_DH* dh)
{
    WOLFSSL_ENTER("wolfSSL_DH_up_ref");

    if (dh) {
    #ifndef SINGLE_THREADED
        if (wc_LockMutex(&dh->refMutex) != 0) {
            WOLFSSL_MSG("Failed to lock DH mutex");
        }
    #endif
        dh->refCount++;
    #ifndef SINGLE_THREADED
        wc_UnLockMutex(&dh->refMutex);
    #endif
        return WOLFSSL_SUCCESS;
    }

    return WOLFSSL_FAILURE;
}

int SetDhInternal(WOLFSSL_DH* dh)
{
    int            ret = WOLFSSL_FATAL_ERROR;
    int            pSz = 1024;
    int            gSz = 1024;
#ifdef WOLFSSL_DH_EXTRA
    int            privSz = 256; /* Up to 2048-bit */
    int            pubSz  = 256;
#endif
#ifdef WOLFSSL_SMALL_STACK
    unsigned char* p   = NULL;
    unsigned char* g   = NULL;
    #ifdef WOLFSSL_DH_EXTRA
        unsigned char* priv_key = NULL;
        unsigned char* pub_key = NULL;
    #endif
#else
    unsigned char  p[1024];
    unsigned char  g[1024];
    #ifdef WOLFSSL_DH_EXTRA
        unsigned char priv_key[256];
        unsigned char pub_key[256];
    #endif
#endif

    WOLFSSL_ENTER("SetDhInternal");

    if (dh == NULL || dh->p == NULL || dh->g == NULL)
        WOLFSSL_MSG("Bad function arguments");
    else if (wolfSSL_BN_bn2bin(dh->p, NULL) > pSz)
        WOLFSSL_MSG("Bad p internal size");
    else if (wolfSSL_BN_bn2bin(dh->g, NULL) > gSz)
        WOLFSSL_MSG("Bad g internal size");
#ifdef WOLFSSL_DH_EXTRA
    else if (wolfSSL_BN_bn2bin(dh->priv_key, NULL) > privSz)
        WOLFSSL_MSG("Bad private key internal size");
    else if (wolfSSL_BN_bn2bin(dh->pub_key, NULL) > privSz)
        WOLFSSL_MSG("Bad public key internal size");
#endif
    else {
    #ifdef WOLFSSL_SMALL_STACK
        p = (unsigned char*)XMALLOC(pSz, NULL, DYNAMIC_TYPE_PUBLIC_KEY);
        g = (unsigned char*)XMALLOC(gSz, NULL, DYNAMIC_TYPE_PUBLIC_KEY);
        #ifdef WOLFSSL_DH_EXTRA
            priv_key = (unsigned char*)XMALLOC(privSz, NULL,
                DYNAMIC_TYPE_PRIVATE_KEY);
            pub_key  = (unsigned char*)XMALLOC(pubSz, NULL,
                DYNAMIC_TYPE_PUBLIC_KEY);
        #endif

        if (p == NULL || g == NULL) {
            XFREE(p, NULL, DYNAMIC_TYPE_PUBLIC_KEY);
            XFREE(g, NULL, DYNAMIC_TYPE_PUBLIC_KEY);
            return ret;
        }
    #endif /* WOLFSSL_SMALL_STACK */

        /* Free so that mp_init's don't leak */
        wc_FreeDhKey((DhKey*)dh->internal);

    #ifdef WOLFSSL_DH_EXTRA
        privSz = wolfSSL_BN_bn2bin(dh->priv_key, priv_key);
        pubSz  = wolfSSL_BN_bn2bin(dh->pub_key,  pub_key);
        if (privSz <= 0) {
            WOLFSSL_MSG("No private key size.");
        }
        if (pubSz <= 0) {
            WOLFSSL_MSG("No public key size.");
        }
        if (privSz > 0 || pubSz > 0) {
            ret = wc_DhImportKeyPair((DhKey*)dh->internal, priv_key, privSz,
                                     pub_key, pubSz);
            if (ret == 0) {
                ret = WOLFSSL_SUCCESS;
            }
            else {
                WOLFSSL_MSG("Failed setting private or public key.");
                ret = WOLFSSL_FAILURE;
            }
        }
    #endif /* WOLFSSL_DH_EXTRA */

        pSz = wolfSSL_BN_bn2bin(dh->p, p);
        gSz = wolfSSL_BN_bn2bin(dh->g, g);

        if (pSz <= 0 || gSz <= 0)
            WOLFSSL_MSG("Bad BN2bin set");
        else if (wc_DhSetKey((DhKey*)dh->internal, p, pSz, g, gSz) < 0)
            WOLFSSL_MSG("Bad DH SetKey");
        else {
            dh->inSet = 1;
            ret = WOLFSSL_SUCCESS;
        }

    #ifdef WOLFSSL_SMALL_STACK
        XFREE(p, NULL, DYNAMIC_TYPE_PUBLIC_KEY);
        XFREE(g, NULL, DYNAMIC_TYPE_PUBLIC_KEY);
        #ifdef WOLFSSL_DH_EXTRA
            XFREE(priv_key, NULL, DYNAMIC_TYPE_PRIVATE_KEY);
            XFREE(pub_key,  NULL, DYNAMIC_TYPE_PUBLIC_KEY);
        #endif
    #endif
    }

    return ret;
}

#if !defined(NO_DH) && (defined(WOLFSSL_QT) || defined(OPENSSL_ALL) \
    || defined(WOLFSSL_OPENSSH)) || defined(OPENSSL_EXTRA)

#ifdef WOLFSSL_DH_EXTRA
WOLFSSL_DH* wolfSSL_DH_dup(WOLFSSL_DH* dh)
{
    WOLFSSL_DH* ret = NULL;

    WOLFSSL_ENTER("wolfSSL_DH_dup");

    if (!dh) {
        WOLFSSL_MSG("Bad parameter");
        return NULL;
    }

    if (dh->inSet == 0 && SetDhInternal(dh) != WOLFSSL_SUCCESS){
        WOLFSSL_MSG("Bad DH set internal");
        return NULL;
    }

    if (!(ret = wolfSSL_DH_new())) {
        WOLFSSL_MSG("wolfSSL_DH_new error");
        return NULL;
    }

    if (wc_DhKeyCopy((DhKey*)dh->internal, (DhKey*)ret->internal) != MP_OKAY) {
        WOLFSSL_MSG("wc_DhKeyCopy error");
        wolfSSL_DH_free(ret);
        return NULL;
    }
    ret->inSet = 1;

    if (SetDhExternal(ret) != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("SetDhExternal error");
        wolfSSL_DH_free(ret);
        return NULL;
    }

    return ret;
}
#endif /* WOLFSSL_DH_EXTRA */

/* Set the members of DhKey into WOLFSSL_DH
 * Specify elements to set via the 2nd parmeter
 */
int SetDhExternal_ex(WOLFSSL_DH *dh, int elm)
{
    DhKey *key;
    WOLFSSL_MSG("Entering SetDhExternal_ex");

    if (dh == NULL || dh->internal == NULL) {
        WOLFSSL_MSG("dh key NULL error");
        return WOLFSSL_FATAL_ERROR;
    }

    key = (DhKey*)dh->internal;

    if (elm & ELEMENT_P) {
        if (SetIndividualExternal(&dh->p, &key->p) != WOLFSSL_SUCCESS) {
            WOLFSSL_MSG("dh param p error");
            return WOLFSSL_FATAL_ERROR;
        }
    }
    if (elm & ELEMENT_Q) {
        if (SetIndividualExternal(&dh->q, &key->q) != WOLFSSL_SUCCESS) {
            WOLFSSL_MSG("dh param q error");
            return WOLFSSL_FATAL_ERROR;
        }
    }
    if (elm & ELEMENT_G) {
        if (SetIndividualExternal(&dh->g, &key->g) != WOLFSSL_SUCCESS) {
            WOLFSSL_MSG("dh param g error");
            return WOLFSSL_FATAL_ERROR;
        }
    }
#ifdef WOLFSSL_DH_EXTRA
    if (elm & ELEMENT_PRV) {
        if (SetIndividualExternal(&dh->priv_key, &key->priv) !=
                                                      WOLFSSL_SUCCESS) {
            WOLFSSL_MSG("No DH Private Key");
            return WOLFSSL_FATAL_ERROR;
        }
    }
    if (elm & ELEMENT_PUB) {
        if (SetIndividualExternal(&dh->pub_key, &key->pub) != WOLFSSL_SUCCESS) {
            WOLFSSL_MSG("No DH Public Key");
            return WOLFSSL_FATAL_ERROR;
        }
    }
#endif /* WOLFSSL_DH_EXTRA */

    dh->exSet = 1;

    return WOLFSSL_SUCCESS;
}
/* Set the members of DhKey into WOLFSSL_DH
 * DhKey was populated from wc_DhKeyDecode
 * p, g, pub_key and pri_key are set.
 */
int SetDhExternal(WOLFSSL_DH *dh)
{
    int elements = ELEMENT_P | ELEMENT_G | ELEMENT_PUB | ELEMENT_PRV;
    WOLFSSL_MSG("Entering SetDhExternal");
    return SetDhExternal_ex(dh, elements);
}
#endif /* !NO_DH && (WOLFSSL_QT || OPENSSL_ALL) */

/* return code compliant with OpenSSL :
 *   DH prime size in bytes if success, 0 if error
 */
int wolfSSL_DH_size(WOLFSSL_DH* dh)
{
    WOLFSSL_MSG("wolfSSL_DH_size");

    if (dh == NULL)
        return WOLFSSL_FATAL_ERROR;

    return wolfSSL_BN_num_bytes(dh->p);
}

/* This sets a big number with the 768-bit prime from RFC 2409.
 *
 * bn  if not NULL then the big number structure is used. If NULL then a new
 *     big number structure is created.
 *
 * Returns a WOLFSSL_BIGNUM structure on success and NULL with failure.
 */
WOLFSSL_BIGNUM* wolfSSL_DH_768_prime(WOLFSSL_BIGNUM* bn)
{
    const char prm[] = {
        "FFFFFFFFFFFFFFFFC90FDAA22168C234"
        "C4C6628B80DC1CD129024E088A67CC74"
        "020BBEA63B139B22514A08798E3404DD"
        "EF9519B3CD3A431B302B0A6DF25F1437"
        "4FE1356D6D51C245E485B576625E7EC6"
        "F44C42E9A63A3620FFFFFFFFFFFFFFFF"
    };

    WOLFSSL_ENTER("wolfSSL_DH_768_prime");

    if (wolfSSL_BN_hex2bn(&bn, prm) != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("Error converting DH 768 prime to big number");
        return NULL;
    }

    return bn;
}

/* This sets a big number with the 1024-bit prime from RFC 2409.
 *
 * bn  if not NULL then the big number structure is used. If NULL then a new
 *     big number structure is created.
 *
 * Returns a WOLFSSL_BIGNUM structure on success and NULL with failure.
 */
WOLFSSL_BIGNUM* wolfSSL_DH_1024_prime(WOLFSSL_BIGNUM* bn)
{
    const char prm[] = {
        "FFFFFFFFFFFFFFFFC90FDAA22168C234"
        "C4C6628B80DC1CD129024E088A67CC74"
        "020BBEA63B139B22514A08798E3404DD"
        "EF9519B3CD3A431B302B0A6DF25F1437"
        "4FE1356D6D51C245E485B576625E7EC6"
        "F44C42E9A637ED6B0BFF5CB6F406B7ED"
        "EE386BFB5A899FA5AE9F24117C4B1FE6"
        "49286651ECE65381FFFFFFFFFFFFFFFF"
    };

    WOLFSSL_ENTER("wolfSSL_DH_1024_prime");

    if (wolfSSL_BN_hex2bn(&bn, prm) != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("Error converting DH 1024 prime to big number");
        return NULL;
    }

    return bn;
}

/* This sets a big number with the 1536-bit prime from RFC 3526.
 *
 * bn  if not NULL then the big number structure is used. If NULL then a new
 *     big number structure is created.
 *
 * Returns a WOLFSSL_BIGNUM structure on success and NULL with failure.
 */
WOLFSSL_BIGNUM* wolfSSL_DH_1536_prime(WOLFSSL_BIGNUM* bn)
{
    const char prm[] = {
        "FFFFFFFFFFFFFFFFC90FDAA22168C234"
        "C4C6628B80DC1CD129024E088A67CC74"
        "020BBEA63B139B22514A08798E3404DD"
        "EF9519B3CD3A431B302B0A6DF25F1437"
        "4FE1356D6D51C245E485B576625E7EC6"
        "F44C42E9A637ED6B0BFF5CB6F406B7ED"
        "EE386BFB5A899FA5AE9F24117C4B1FE6"
        "49286651ECE45B3DC2007CB8A163BF05"
        "98DA48361C55D39A69163FA8FD24CF5F"
        "83655D23DCA3AD961C62F356208552BB"
        "9ED529077096966D670C354E4ABC9804"
        "F1746C08CA237327FFFFFFFFFFFFFFFF"
    };

    WOLFSSL_ENTER("wolfSSL_DH_1536_prime");

    if (wolfSSL_BN_hex2bn(&bn, prm) != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("Error converting DH 1536 prime to big number");
        return NULL;
    }

    return bn;
}

/* This sets a big number with the 2048-bit prime from RFC 3526.
 *
 * bn  if not NULL then the big number structure is used. If NULL then a new
 *     big number structure is created.
 *
 * Returns a WOLFSSL_BIGNUM structure on success and NULL with failure.
 */
WOLFSSL_BIGNUM* wolfSSL_DH_2048_prime(WOLFSSL_BIGNUM* bn)
{
    const char prm[] = {
        "FFFFFFFFFFFFFFFFC90FDAA22168C234"
        "C4C6628B80DC1CD129024E088A67CC74"
        "020BBEA63B139B22514A08798E3404DD"
        "EF9519B3CD3A431B302B0A6DF25F1437"
        "4FE1356D6D51C245E485B576625E7EC6"
        "F44C42E9A637ED6B0BFF5CB6F406B7ED"
        "EE386BFB5A899FA5AE9F24117C4B1FE6"
        "49286651ECE45B3DC2007CB8A163BF05"
        "98DA48361C55D39A69163FA8FD24CF5F"
        "83655D23DCA3AD961C62F356208552BB"
        "9ED529077096966D670C354E4ABC9804"
        "F1746C08CA18217C32905E462E36CE3B"
        "E39E772C180E86039B2783A2EC07A28F"
        "B5C55DF06F4C52C9DE2BCBF695581718"
        "3995497CEA956AE515D2261898FA0510"
        "15728E5A8AACAA68FFFFFFFFFFFFFFFF"
    };

    WOLFSSL_ENTER("wolfSSL_DH_2048_prime");

    if (wolfSSL_BN_hex2bn(&bn, prm) != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("Error converting DH 2048 prime to big number");
        return NULL;
    }

    return bn;
}

/* This sets a big number with the 3072-bit prime from RFC 3526.
 *
 * bn  if not NULL then the big number structure is used. If NULL then a new
 *     big number structure is created.
 *
 * Returns a WOLFSSL_BIGNUM structure on success and NULL with failure.
 */
WOLFSSL_BIGNUM* wolfSSL_DH_3072_prime(WOLFSSL_BIGNUM* bn)
{
    const char prm[] = {
        "FFFFFFFFFFFFFFFFC90FDAA22168C234"
        "C4C6628B80DC1CD129024E088A67CC74"
        "020BBEA63B139B22514A08798E3404DD"
        "EF9519B3CD3A431B302B0A6DF25F1437"
        "4FE1356D6D51C245E485B576625E7EC6"
        "F44C42E9A637ED6B0BFF5CB6F406B7ED"
        "EE386BFB5A899FA5AE9F24117C4B1FE6"
        "49286651ECE45B3DC2007CB8A163BF05"
        "98DA48361C55D39A69163FA8FD24CF5F"
        "83655D23DCA3AD961C62F356208552BB"
        "9ED529077096966D670C354E4ABC9804"
        "F1746C08CA18217C32905E462E36CE3B"
        "E39E772C180E86039B2783A2EC07A28F"
        "B5C55DF06F4C52C9DE2BCBF695581718"
        "3995497CEA956AE515D2261898FA0510"
        "15728E5A8AAAC42DAD33170D04507A33"
        "A85521ABDF1CBA64ECFB850458DBEF0A"
        "8AEA71575D060C7DB3970F85A6E1E4C7"
        "ABF5AE8CDB0933D71E8C94E04A25619D"
        "CEE3D2261AD2EE6BF12FFA06D98A0864"
        "D87602733EC86A64521F2B18177B200C"
        "BBE117577A615D6C770988C0BAD946E2"
        "08E24FA074E5AB3143DB5BFCE0FD108E"
        "4B82D120A93AD2CAFFFFFFFFFFFFFFFF"
    };

    WOLFSSL_ENTER("wolfSSL_DH_3072_prime");

    if (wolfSSL_BN_hex2bn(&bn, prm) != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("Error converting DH 3072 prime to big number");
        return NULL;
    }

    return bn;
}

/* This sets a big number with the 4096-bit prime from RFC 3526.
 *
 * bn  if not NULL then the big number structure is used. If NULL then a new
 *     big number structure is created.
 *
 * Returns a WOLFSSL_BIGNUM structure on success and NULL with failure.
 */
WOLFSSL_BIGNUM* wolfSSL_DH_4096_prime(WOLFSSL_BIGNUM* bn)
{
    const char prm[] = {
        "FFFFFFFFFFFFFFFFC90FDAA22168C234"
        "C4C6628B80DC1CD129024E088A67CC74"
        "020BBEA63B139B22514A08798E3404DD"
        "EF9519B3CD3A431B302B0A6DF25F1437"
        "4FE1356D6D51C245E485B576625E7EC6"
        "F44C42E9A637ED6B0BFF5CB6F406B7ED"
        "EE386BFB5A899FA5AE9F24117C4B1FE6"
        "49286651ECE45B3DC2007CB8A163BF05"
        "98DA48361C55D39A69163FA8FD24CF5F"
        "83655D23DCA3AD961C62F356208552BB"
        "9ED529077096966D670C354E4ABC9804"
        "F1746C08CA18217C32905E462E36CE3B"
        "E39E772C180E86039B2783A2EC07A28F"
        "B5C55DF06F4C52C9DE2BCBF695581718"
        "3995497CEA956AE515D2261898FA0510"
        "15728E5A8AAAC42DAD33170D04507A33"
        "A85521ABDF1CBA64ECFB850458DBEF0A"
        "8AEA71575D060C7DB3970F85A6E1E4C7"
        "ABF5AE8CDB0933D71E8C94E04A25619D"
        "CEE3D2261AD2EE6BF12FFA06D98A0864"
        "D87602733EC86A64521F2B18177B200C"
        "BBE117577A615D6C770988C0BAD946E2"
        "08E24FA074E5AB3143DB5BFCE0FD108E"
        "4B82D120A92108011A723C12A787E6D7"
        "88719A10BDBA5B2699C327186AF4E23C"
        "1A946834B6150BDA2583E9CA2AD44CE8"
        "DBBBC2DB04DE8EF92E8EFC141FBECAA6"
        "287C59474E6BC05D99B2964FA090C3A2"
        "233BA186515BE7ED1F612970CEE2D7AF"
        "B81BDD762170481CD0069127D5B05AA9"
        "93B4EA988D8FDDC186FFB7DC90A6C08F"
        "4DF435C934063199FFFFFFFFFFFFFFFF"
    };

    WOLFSSL_ENTER("wolfSSL_DH_4096_prime");

    if (wolfSSL_BN_hex2bn(&bn, prm) != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("Error converting DH 4096 prime to big number");
        return NULL;
    }

    return bn;
}

/* This sets a big number with the 6144-bit prime from RFC 3526.
 *
 * bn  if not NULL then the big number structure is used. If NULL then a new
 *     big number structure is created.
 *
 * Returns a WOLFSSL_BIGNUM structure on success and NULL with failure.
 */
WOLFSSL_BIGNUM* wolfSSL_DH_6144_prime(WOLFSSL_BIGNUM* bn)
{
    const char prm[] = {
        "FFFFFFFFFFFFFFFFC90FDAA22168C234"
        "C4C6628B80DC1CD129024E088A67CC74"
        "020BBEA63B139B22514A08798E3404DD"
        "EF9519B3CD3A431B302B0A6DF25F1437"
        "4FE1356D6D51C245E485B576625E7EC6"
        "F44C42E9A637ED6B0BFF5CB6F406B7ED"
        "EE386BFB5A899FA5AE9F24117C4B1FE6"
        "49286651ECE45B3DC2007CB8A163BF05"
        "98DA48361C55D39A69163FA8FD24CF5F"
        "83655D23DCA3AD961C62F356208552BB"
        "9ED529077096966D670C354E4ABC9804"
        "F1746C08CA18217C32905E462E36CE3B"
        "E39E772C180E86039B2783A2EC07A28F"
        "B5C55DF06F4C52C9DE2BCBF695581718"
        "3995497CEA956AE515D2261898FA0510"
        "15728E5A8AAAC42DAD33170D04507A33"
        "A85521ABDF1CBA64ECFB850458DBEF0A"
        "8AEA71575D060C7DB3970F85A6E1E4C7"
        "ABF5AE8CDB0933D71E8C94E04A25619D"
        "CEE3D2261AD2EE6BF12FFA06D98A0864"
        "D87602733EC86A64521F2B18177B200C"
        "BBE117577A615D6C770988C0BAD946E2"
        "08E24FA074E5AB3143DB5BFCE0FD108E"
        "4B82D120A92108011A723C12A787E6D7"
        "88719A10BDBA5B2699C327186AF4E23C"
        "1A946834B6150BDA2583E9CA2AD44CE8"
        "DBBBC2DB04DE8EF92E8EFC141FBECAA6"
        "287C59474E6BC05D99B2964FA090C3A2"
        "233BA186515BE7ED1F612970CEE2D7AF"
        "B81BDD762170481CD0069127D5B05AA9"
        "93B4EA988D8FDDC186FFB7DC90A6C08F"
        "4DF435C93402849236C3FAB4D27C7026"
        "C1D4DCB2602646DEC9751E763DBA37BD"
        "F8FF9406AD9E530EE5DB382F413001AE"
        "B06A53ED9027D831179727B0865A8918"
        "DA3EDBEBCF9B14ED44CE6CBACED4BB1B"
        "DB7F1447E6CC254B332051512BD7AF42"
        "6FB8F401378CD2BF5983CA01C64B92EC"
        "F032EA15D1721D03F482D7CE6E74FEF6"
        "D55E702F46980C82B5A84031900B1C9E"
        "59E7C97FBEC7E8F323A97A7E36CC88BE"
        "0F1D45B7FF585AC54BD407B22B4154AA"
        "CC8F6D7EBF48E1D814CC5ED20F8037E0"
        "A79715EEF29BE32806A1D58BB7C5DA76"
        "F550AA3D8A1FBFF0EB19CCB1A313D55C"
        "DA56C9EC2EF29632387FE8D76E3C0468"
        "043E8F663F4860EE12BF2D5B0B7474D6"
        "E694F91E6DCC4024FFFFFFFFFFFFFFFF"
    };

    WOLFSSL_ENTER("wolfSSL_DH_6144_prime");

    if (wolfSSL_BN_hex2bn(&bn, prm) != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("Error converting DH 6144 prime to big number");
        return NULL;
    }

    return bn;
}


/* This sets a big number with the 8192-bit prime from RFC 3526.
 *
 * bn  if not NULL then the big number structure is used. If NULL then a new
 *     big number structure is created.
 *
 * Returns a WOLFSSL_BIGNUM structure on success and NULL with failure.
 */
WOLFSSL_BIGNUM* wolfSSL_DH_8192_prime(WOLFSSL_BIGNUM* bn)
{
    const char prm[] = {
        "FFFFFFFFFFFFFFFFC90FDAA22168C234"
        "C4C6628B80DC1CD129024E088A67CC74"
        "020BBEA63B139B22514A08798E3404DD"
        "EF9519B3CD3A431B302B0A6DF25F1437"
        "4FE1356D6D51C245E485B576625E7EC6"
        "F44C42E9A637ED6B0BFF5CB6F406B7ED"
        "EE386BFB5A899FA5AE9F24117C4B1FE6"
        "49286651ECE45B3DC2007CB8A163BF05"
        "98DA48361C55D39A69163FA8FD24CF5F"
        "83655D23DCA3AD961C62F356208552BB"
        "9ED529077096966D670C354E4ABC9804"
        "F1746C08CA18217C32905E462E36CE3B"
        "E39E772C180E86039B2783A2EC07A28F"
        "B5C55DF06F4C52C9DE2BCBF695581718"
        "3995497CEA956AE515D2261898FA0510"
        "15728E5A8AAAC42DAD33170D04507A33"
        "A85521ABDF1CBA64ECFB850458DBEF0A"
        "8AEA71575D060C7DB3970F85A6E1E4C7"
        "ABF5AE8CDB0933D71E8C94E04A25619D"
        "CEE3D2261AD2EE6BF12FFA06D98A0864"
        "D87602733EC86A64521F2B18177B200C"
        "BBE117577A615D6C770988C0BAD946E2"
        "08E24FA074E5AB3143DB5BFCE0FD108E"
        "4B82D120A92108011A723C12A787E6D7"
        "88719A10BDBA5B2699C327186AF4E23C"
        "1A946834B6150BDA2583E9CA2AD44CE8"
        "DBBBC2DB04DE8EF92E8EFC141FBECAA6"
        "287C59474E6BC05D99B2964FA090C3A2"
        "233BA186515BE7ED1F612970CEE2D7AF"
        "B81BDD762170481CD0069127D5B05AA9"
        "93B4EA988D8FDDC186FFB7DC90A6C08F"
        "4DF435C93402849236C3FAB4D27C7026"
        "C1D4DCB2602646DEC9751E763DBA37BD"
        "F8FF9406AD9E530EE5DB382F413001AE"
        "B06A53ED9027D831179727B0865A8918"
        "DA3EDBEBCF9B14ED44CE6CBACED4BB1B"
        "DB7F1447E6CC254B332051512BD7AF42"
        "6FB8F401378CD2BF5983CA01C64B92EC"
        "F032EA15D1721D03F482D7CE6E74FEF6"
        "D55E702F46980C82B5A84031900B1C9E"
        "59E7C97FBEC7E8F323A97A7E36CC88BE"
        "0F1D45B7FF585AC54BD407B22B4154AA"
        "CC8F6D7EBF48E1D814CC5ED20F8037E0"
        "A79715EEF29BE32806A1D58BB7C5DA76"
        "F550AA3D8A1FBFF0EB19CCB1A313D55C"
        "DA56C9EC2EF29632387FE8D76E3C0468"
        "043E8F663F4860EE12BF2D5B0B7474D6"
        "E694F91E6DBE115974A3926F12FEE5E4"
        "38777CB6A932DF8CD8BEC4D073B931BA"
        "3BC832B68D9DD300741FA7BF8AFC47ED"
        "2576F6936BA424663AAB639C5AE4F568"
        "3423B4742BF1C978238F16CBE39D652D"
        "E3FDB8BEFC848AD922222E04A4037C07"
        "13EB57A81A23F0C73473FC646CEA306B"
        "4BCBC8862F8385DDFA9D4B7FA2C087E8"
        "79683303ED5BDD3A062B3CF5B3A278A6"
        "6D2A13F83F44F82DDF310EE074AB6A36"
        "4597E899A0255DC164F31CC50846851D"
        "F9AB48195DED7EA1B1D510BD7EE74D73"
        "FAF36BC31ECFA268359046F4EB879F92"
        "4009438B481C6CD7889A002ED5EE382B"
        "C9190DA6FC026E479558E4475677E9AA"
        "9E3050E2765694DFC81F56E880B96E71"
        "60C980DD98EDD3DFFFFFFFFFFFFFFFFF"
    };

    WOLFSSL_ENTER("wolfSSL_DH_8192_prime");

    if (wolfSSL_BN_hex2bn(&bn, prm) != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("Error converting DH 8192 prime to big number");
        return NULL;
    }

    return bn;
}

/* The functions inside the macro guard below are fine to use with FIPS provided
 * WOLFSSL_DH_EXTRA isn't defined. That define will cause SetDhInternal to have
 * a call to wc_DhImportKeyPair, which isn't defined in the FIPS v2 module. */
#if !defined(HAVE_FIPS) || (defined(HAVE_FIPS) && !defined(WOLFSSL_DH_EXTRA)) \
 || (defined(HAVE_FIPS_VERSION) && (HAVE_FIPS_VERSION>2))
/* return code compliant with OpenSSL :
 *   1 if success, 0 if error
 */
int wolfSSL_DH_generate_key(WOLFSSL_DH* dh)
{
    int            ret    = WOLFSSL_FAILURE;
    word32         pubSz  = 0;
    word32         privSz = 0;
    int            initTmpRng = 0;
    WC_RNG*        rng    = NULL;
#ifdef WOLFSSL_SMALL_STACK
    WC_RNG*        tmpRNG;
#else
    WC_RNG         tmpRNG[1];
#endif
    unsigned char* pub    = NULL;
    unsigned char* priv   = NULL;

    WOLFSSL_MSG("wolfSSL_DH_generate_key");

#ifdef WOLFSSL_SMALL_STACK
    tmpRNG = (WC_RNG*)XMALLOC(sizeof(WC_RNG), NULL, DYNAMIC_TYPE_RNG);
    if (tmpRNG == NULL) {
        XFREE(tmpRNG, NULL, DYNAMIC_TYPE_RNG);
        return ret;
    }
#endif

    if (dh == NULL || dh->p == NULL || dh->g == NULL)
        WOLFSSL_MSG("Bad function arguments");
    else if (dh->inSet == 0 && SetDhInternal(dh) != WOLFSSL_SUCCESS)
            WOLFSSL_MSG("Bad DH set internal");
    else if (wc_InitRng(tmpRNG) == 0) {
        rng = tmpRNG;
        initTmpRng = 1;
    }
    else {
        WOLFSSL_MSG("Bad RNG Init, trying global");
        if (initGlobalRNG == 0)
            WOLFSSL_MSG("Global RNG no Init");
        else
            rng = &globalRNG;
    }

    if (rng) {
        pubSz = wolfSSL_BN_num_bytes(dh->p);
        if (dh->length) {
            privSz = dh->length/8; /* to bytes */
        } else {
            privSz = pubSz;
        }
        if (pubSz > 0) {
            pub = (unsigned char*)XMALLOC(pubSz,
                    NULL, DYNAMIC_TYPE_PUBLIC_KEY);
        }
        if (privSz > 0) {
            priv = (unsigned char*)XMALLOC(privSz,
                    NULL, DYNAMIC_TYPE_PRIVATE_KEY);
        }
        PRIVATE_KEY_UNLOCK();
        if (pub == NULL || priv == NULL) {
            WOLFSSL_MSG("Unable to malloc memory");
        }
        else if (wc_DhGenerateKeyPair((DhKey*)dh->internal, rng, priv, &privSz,
                                                               pub, &pubSz) < 0)
            WOLFSSL_MSG("Bad wc_DhGenerateKeyPair");
        else {
            if (dh->pub_key)
                wolfSSL_BN_free(dh->pub_key);

            dh->pub_key = wolfSSL_BN_new();
            if (dh->pub_key == NULL) {
                WOLFSSL_MSG("Bad DH new pub");
            }
            if (dh->priv_key)
                wolfSSL_BN_free(dh->priv_key);

            dh->priv_key = wolfSSL_BN_new();

            if (dh->priv_key == NULL) {
                WOLFSSL_MSG("Bad DH new priv");
            }

            if (dh->pub_key && dh->priv_key) {
               if (wolfSSL_BN_bin2bn(pub, pubSz, dh->pub_key) == NULL)
                   WOLFSSL_MSG("Bad DH bn2bin error pub");
               else if (wolfSSL_BN_bin2bn(priv, privSz, dh->priv_key) == NULL)
                   WOLFSSL_MSG("Bad DH bn2bin error priv");
               else
                   ret = WOLFSSL_SUCCESS;
            }
        }
        PRIVATE_KEY_LOCK();
    }

    if (initTmpRng)
        wc_FreeRng(tmpRNG);

#ifdef WOLFSSL_SMALL_STACK
    XFREE(tmpRNG, NULL, DYNAMIC_TYPE_RNG);
#endif
    XFREE(pub,    NULL, DYNAMIC_TYPE_PUBLIC_KEY);
    XFREE(priv,   NULL, DYNAMIC_TYPE_PRIVATE_KEY);

    return ret;
}


/* return code compliant with OpenSSL :
 *   size of shared secret if success, -1 if error
 */
int wolfSSL_DH_compute_key(unsigned char* key, const WOLFSSL_BIGNUM* otherPub,
                          WOLFSSL_DH* dh)
{
    int            ret    = WOLFSSL_FATAL_ERROR;
    word32         keySz  = 0;
    int            pubSz  = 1024;
    int            privSz = 1024;
#ifdef WOLFSSL_SMALL_STACK
    unsigned char* pub;
    unsigned char* priv   = NULL;
#else
    unsigned char  pub [1024];
    unsigned char  priv[1024];
#endif

    WOLFSSL_MSG("wolfSSL_DH_compute_key");

#ifdef WOLFSSL_SMALL_STACK
    pub = (unsigned char*)XMALLOC(pubSz, NULL, DYNAMIC_TYPE_PUBLIC_KEY);
    if (pub == NULL)
        return ret;

    priv = (unsigned char*)XMALLOC(privSz, NULL, DYNAMIC_TYPE_PRIVATE_KEY);
    if (priv == NULL) {
        XFREE(pub, NULL, DYNAMIC_TYPE_PUBLIC_KEY);
        return ret;
    }
#endif

    if (dh == NULL || dh->priv_key == NULL || otherPub == NULL)
        WOLFSSL_MSG("Bad function arguments");
    else if ((keySz = (word32)DH_size(dh)) == 0)
        WOLFSSL_MSG("Bad DH_size");
    else if (wolfSSL_BN_bn2bin(dh->priv_key, NULL) > (int)privSz)
        WOLFSSL_MSG("Bad priv internal size");
    else if (wolfSSL_BN_bn2bin(otherPub, NULL) > (int)pubSz)
        WOLFSSL_MSG("Bad otherPub size");
    else {
        privSz = wolfSSL_BN_bn2bin(dh->priv_key, priv);
        pubSz  = wolfSSL_BN_bn2bin(otherPub, pub);
        if (dh->inSet == 0 && SetDhInternal(dh) != WOLFSSL_SUCCESS){
            WOLFSSL_MSG("Bad DH set internal");
        }
        PRIVATE_KEY_UNLOCK();
        if (privSz <= 0 || pubSz <= 0)
            WOLFSSL_MSG("Bad BN2bin set");
        else if (wc_DhAgree((DhKey*)dh->internal, key, &keySz,
                            priv, privSz, pub, pubSz) < 0)
            WOLFSSL_MSG("wc_DhAgree failed");
        else
            ret = (int)keySz;
        PRIVATE_KEY_LOCK();
    }

#ifdef WOLFSSL_SMALL_STACK
    XFREE(pub,  NULL, DYNAMIC_TYPE_PUBLIC_KEY);
    XFREE(priv, NULL, DYNAMIC_TYPE_PRIVATE_KEY);
#endif

    WOLFSSL_LEAVE("wolfSSL_DH_compute_key", ret);

    return ret;
}


#if defined(OPENSSL_ALL) || \
    defined(OPENSSL_VERSION_NUMBER) && OPENSSL_VERSION_NUMBER >= 0x10100000L
int wolfSSL_DH_set_length(WOLFSSL_DH *dh, long len)
{
    WOLFSSL_ENTER("wolfSSL_DH_set_length");

    /* len is checked at generation */
    if (dh == NULL) {
        WOLFSSL_MSG("Bad function arguments");
        return WOLFSSL_FAILURE;
    }

    dh->length = (int)len;
    return WOLFSSL_SUCCESS;
}

/* ownership of p,q,and g get taken over by "dh" on success and should be free'd
 * with a call to wolfSSL_DH_free -- not individually.
 *
 * returns WOLFSSL_SUCCESS on success
 */
int wolfSSL_DH_set0_pqg(WOLFSSL_DH *dh, WOLFSSL_BIGNUM *p,
    WOLFSSL_BIGNUM *q, WOLFSSL_BIGNUM *g)
{
    int ret;
    WOLFSSL_ENTER("wolfSSL_DH_set0_pqg");

    /* q can be NULL */
    if (dh == NULL || p == NULL || g == NULL) {
        WOLFSSL_MSG("Bad function arguments");
        return WOLFSSL_FAILURE;
    }

    /* free existing internal DH structure and recreate with new p / g */
    if (dh->inSet) {
#ifndef HAVE_SELFTEST
        ret = wc_FreeDhKey((DhKey*)dh->internal);
        if (ret != 0) {
            WOLFSSL_MSG("Unable to free internal DH key");
            return WOLFSSL_FAILURE;
        }
#else
        /* Selftest code has this API with a void return type */
        wc_FreeDhKey((DhKey*)dh->internal);
#endif
    }

    wolfSSL_BN_free(dh->p);
    wolfSSL_BN_free(dh->q);
    wolfSSL_BN_free(dh->g);

    dh->p = p;
    dh->q = q;
    dh->g = g;

    ret = SetDhInternal(dh);
    if (ret != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("Unable to set internal DH key");
        dh->p = NULL;
        dh->q = NULL;
        dh->g = NULL;
        dh->inSet = 0;
        return WOLFSSL_FAILURE;
    }

    return WOLFSSL_SUCCESS;
}
#endif /* OPENSSL_ALL || (v1.1.0 or later) */
#endif /* !HAVE_FIPS || (HAVE_FIPS && !WOLFSSL_DH_EXTRA) ||
        * HAVE_FIPS_VERSION > 2 */

void wolfSSL_DH_get0_key(const WOLFSSL_DH *dh,
        const WOLFSSL_BIGNUM **pub_key, const WOLFSSL_BIGNUM **priv_key)
{
    WOLFSSL_ENTER("wolfSSL_DH_get0_key");

    if (dh != NULL) {
        if (pub_key != NULL && dh->pub_key != NULL &&
                wolfSSL_BN_is_zero(dh->pub_key) != WOLFSSL_SUCCESS)
            *pub_key = dh->pub_key;
        if (priv_key != NULL && dh->priv_key != NULL &&
                wolfSSL_BN_is_zero(dh->priv_key) != WOLFSSL_SUCCESS)
            *priv_key = dh->priv_key;
    }
}

int wolfSSL_DH_set0_key(WOLFSSL_DH *dh, WOLFSSL_BIGNUM *pub_key,
        WOLFSSL_BIGNUM *priv_key)
{
    WOLFSSL_ENTER("wolfSSL_DH_set0_key");

    if (dh == NULL)
        return WOLFSSL_FAILURE;

    if (pub_key != NULL) {
        wolfSSL_BN_free(dh->pub_key);
        dh->pub_key = pub_key;
    }

    if (priv_key != NULL) {
        wolfSSL_BN_free(dh->priv_key);
        dh->priv_key = priv_key;
    }

    if (dh->p == NULL || dh->g == NULL)
        return WOLFSSL_SUCCESS; /* Allow loading parameters afterwards */
    else
        return SetDhInternal(dh);
}

/* See RFC 5114 section 2.3, "2048-bit MODP Group with 256-bit Prime Order
 * Subgroup." */
WOLFSSL_DH* wolfSSL_DH_get_2048_256(void)
{
    WOLFSSL_DH* ret;
    int err = 0;
    const byte pHex[] = {
        0x87, 0xA8, 0xE6, 0x1D, 0xB4, 0xB6, 0x66, 0x3C, 0xFF, 0xBB, 0xD1, 0x9C,
        0x65, 0x19, 0x59, 0x99, 0x8C, 0xEE, 0xF6, 0x08, 0x66, 0x0D, 0xD0, 0xF2,
        0x5D, 0x2C, 0xEE, 0xD4, 0x43, 0x5E, 0x3B, 0x00, 0xE0, 0x0D, 0xF8, 0xF1,
        0xD6, 0x19, 0x57, 0xD4, 0xFA, 0xF7, 0xDF, 0x45, 0x61, 0xB2, 0xAA, 0x30,
        0x16, 0xC3, 0xD9, 0x11, 0x34, 0x09, 0x6F, 0xAA, 0x3B, 0xF4, 0x29, 0x6D,
        0x83, 0x0E, 0x9A, 0x7C, 0x20, 0x9E, 0x0C, 0x64, 0x97, 0x51, 0x7A, 0xBD,
        0x5A, 0x8A, 0x9D, 0x30, 0x6B, 0xCF, 0x67, 0xED, 0x91, 0xF9, 0xE6, 0x72,
        0x5B, 0x47, 0x58, 0xC0, 0x22, 0xE0, 0xB1, 0xEF, 0x42, 0x75, 0xBF, 0x7B,
        0x6C, 0x5B, 0xFC, 0x11, 0xD4, 0x5F, 0x90, 0x88, 0xB9, 0x41, 0xF5, 0x4E,
        0xB1, 0xE5, 0x9B, 0xB8, 0xBC, 0x39, 0xA0, 0xBF, 0x12, 0x30, 0x7F, 0x5C,
        0x4F, 0xDB, 0x70, 0xC5, 0x81, 0xB2, 0x3F, 0x76, 0xB6, 0x3A, 0xCA, 0xE1,
        0xCA, 0xA6, 0xB7, 0x90, 0x2D, 0x52, 0x52, 0x67, 0x35, 0x48, 0x8A, 0x0E,
        0xF1, 0x3C, 0x6D, 0x9A, 0x51, 0xBF, 0xA4, 0xAB, 0x3A, 0xD8, 0x34, 0x77,
        0x96, 0x52, 0x4D, 0x8E, 0xF6, 0xA1, 0x67, 0xB5, 0xA4, 0x18, 0x25, 0xD9,
        0x67, 0xE1, 0x44, 0xE5, 0x14, 0x05, 0x64, 0x25, 0x1C, 0xCA, 0xCB, 0x83,
        0xE6, 0xB4, 0x86, 0xF6, 0xB3, 0xCA, 0x3F, 0x79, 0x71, 0x50, 0x60, 0x26,
        0xC0, 0xB8, 0x57, 0xF6, 0x89, 0x96, 0x28, 0x56, 0xDE, 0xD4, 0x01, 0x0A,
        0xBD, 0x0B, 0xE6, 0x21, 0xC3, 0xA3, 0x96, 0x0A, 0x54, 0xE7, 0x10, 0xC3,
        0x75, 0xF2, 0x63, 0x75, 0xD7, 0x01, 0x41, 0x03, 0xA4, 0xB5, 0x43, 0x30,
        0xC1, 0x98, 0xAF, 0x12, 0x61, 0x16, 0xD2, 0x27, 0x6E, 0x11, 0x71, 0x5F,
        0x69, 0x38, 0x77, 0xFA, 0xD7, 0xEF, 0x09, 0xCA, 0xDB, 0x09, 0x4A, 0xE9,
        0x1E, 0x1A, 0x15, 0x97
    };
    const byte gHex[] = {
        0x3F, 0xB3, 0x2C, 0x9B, 0x73, 0x13, 0x4D, 0x0B, 0x2E, 0x77, 0x50, 0x66,
        0x60, 0xED, 0xBD, 0x48, 0x4C, 0xA7, 0xB1, 0x8F, 0x21, 0xEF, 0x20, 0x54,
        0x07, 0xF4, 0x79, 0x3A, 0x1A, 0x0B, 0xA1, 0x25, 0x10, 0xDB, 0xC1, 0x50,
        0x77, 0xBE, 0x46, 0x3F, 0xFF, 0x4F, 0xED, 0x4A, 0xAC, 0x0B, 0xB5, 0x55,
        0xBE, 0x3A, 0x6C, 0x1B, 0x0C, 0x6B, 0x47, 0xB1, 0xBC, 0x37, 0x73, 0xBF,
        0x7E, 0x8C, 0x6F, 0x62, 0x90, 0x12, 0x28, 0xF8, 0xC2, 0x8C, 0xBB, 0x18,
        0xA5, 0x5A, 0xE3, 0x13, 0x41, 0x00, 0x0A, 0x65, 0x01, 0x96, 0xF9, 0x31,
        0xC7, 0x7A, 0x57, 0xF2, 0xDD, 0xF4, 0x63, 0xE5, 0xE9, 0xEC, 0x14, 0x4B,
        0x77, 0x7D, 0xE6, 0x2A, 0xAA, 0xB8, 0xA8, 0x62, 0x8A, 0xC3, 0x76, 0xD2,
        0x82, 0xD6, 0xED, 0x38, 0x64, 0xE6, 0x79, 0x82, 0x42, 0x8E, 0xBC, 0x83,
        0x1D, 0x14, 0x34, 0x8F, 0x6F, 0x2F, 0x91, 0x93, 0xB5, 0x04, 0x5A, 0xF2,
        0x76, 0x71, 0x64, 0xE1, 0xDF, 0xC9, 0x67, 0xC1, 0xFB, 0x3F, 0x2E, 0x55,
        0xA4, 0xBD, 0x1B, 0xFF, 0xE8, 0x3B, 0x9C, 0x80, 0xD0, 0x52, 0xB9, 0x85,
        0xD1, 0x82, 0xEA, 0x0A, 0xDB, 0x2A, 0x3B, 0x73, 0x13, 0xD3, 0xFE, 0x14,
        0xC8, 0x48, 0x4B, 0x1E, 0x05, 0x25, 0x88, 0xB9, 0xB7, 0xD2, 0xBB, 0xD2,
        0xDF, 0x01, 0x61, 0x99, 0xEC, 0xD0, 0x6E, 0x15, 0x57, 0xCD, 0x09, 0x15,
        0xB3, 0x35, 0x3B, 0xBB, 0x64, 0xE0, 0xEC, 0x37, 0x7F, 0xD0, 0x28, 0x37,
        0x0D, 0xF9, 0x2B, 0x52, 0xC7, 0x89, 0x14, 0x28, 0xCD, 0xC6, 0x7E, 0xB6,
        0x18, 0x4B, 0x52, 0x3D, 0x1D, 0xB2, 0x46, 0xC3, 0x2F, 0x63, 0x07, 0x84,
        0x90, 0xF0, 0x0E, 0xF8, 0xD6, 0x47, 0xD1, 0x48, 0xD4, 0x79, 0x54, 0x51,
        0x5E, 0x23, 0x27, 0xCF, 0xEF, 0x98, 0xC5, 0x82, 0x66, 0x4B, 0x4C, 0x0F,
        0x6C, 0xC4, 0x16, 0x59
    };
    const byte qHex[] = {
        0x8C, 0xF8, 0x36, 0x42, 0xA7, 0x09, 0xA0, 0x97, 0xB4, 0x47, 0x99, 0x76,
        0x40, 0x12, 0x9D, 0xA2, 0x99, 0xB1, 0xA4, 0x7D, 0x1E, 0xB3, 0x75, 0x0B,
        0xA3, 0x08, 0xB0, 0xFE, 0x64, 0xF5, 0xFB, 0xD3
    };
    WOLFSSL_BIGNUM* pBn = NULL;
    WOLFSSL_BIGNUM* gBn = NULL;
    WOLFSSL_BIGNUM* qBn = NULL;

    ret = wolfSSL_DH_new();
    if (ret == NULL) {
        err = 1;
    }
    if (err == 0) {
        pBn = wolfSSL_BN_bin2bn(pHex, (int)sizeof(pHex), NULL);
        if (pBn == NULL) {
            WOLFSSL_MSG("Error converting p hex to WOLFSSL_BIGNUM.");
            err = 1;
        }
    }
    if (err == 0) {
        gBn = wolfSSL_BN_bin2bn(gHex, (int)sizeof(gHex), NULL);
        if (gBn == NULL) {
            WOLFSSL_MSG("Error converting g hex to WOLFSSL_BIGNUM.");
            err = 1;
        }
    }
    if (err == 0) {
        qBn = wolfSSL_BN_bin2bn(qHex, (int)sizeof(qHex), NULL);
        if (qBn == NULL) {
            WOLFSSL_MSG("Error converting q hex to WOLFSSL_BIGNUM.");
            err = 1;
        }
    }
    if (err == 0) {
    #if defined(OPENSSL_ALL) || \
        defined(OPENSSL_VERSION_NUMBER) && OPENSSL_VERSION_NUMBER >= 0x10100000L
        if (wolfSSL_DH_set0_pqg(ret, pBn, qBn, gBn) != WOLFSSL_SUCCESS) {
            WOLFSSL_MSG("Error setting DH parameters.");
            err = 1;
        }
    #else
        ret->p = pBn;
        ret->q = qBn;
        ret->g = gBn;

        if (SetDhInternal(ret) != WOLFSSL_SUCCESS) {
            WOLFSSL_MSG("Error setting DH parameters.");
            err = 1;
        }
    #endif
    }

    if (err == 1) {
        wolfSSL_BN_free(pBn);
        wolfSSL_BN_free(gBn);
        wolfSSL_BN_free(qBn);
        wolfSSL_DH_free(ret);
        ret = NULL;
    }

    return ret;
}

#endif /* NO_DH */
#endif /* OPENSSL_EXTRA */

#ifndef NO_DSA
#if defined(OPENSSL_EXTRA) && defined(XFPRINTF) && !defined(NO_FILESYSTEM) && \
    !defined(NO_STDIO_FILESYSTEM)
/* return code compliant with OpenSSL :
 *   1 if success, 0 if error
 */
int wolfSSL_DSA_print_fp(XFILE fp, WOLFSSL_DSA* dsa, int indent)
{
    int ret = WOLFSSL_SUCCESS;
    int pBits;

    WOLFSSL_ENTER("wolfSSL_DSA_print_fp");

    if (fp == XBADFILE || dsa == NULL) {
        ret = WOLFSSL_FAILURE;
    }

    if (ret == WOLFSSL_SUCCESS && dsa->p != NULL) {
        pBits = wolfSSL_BN_num_bits(dsa->p);
        if (pBits == WOLFSSL_FAILURE) {
            ret = WOLFSSL_FAILURE;
        }
        else {
            XFPRINTF(fp, "%*s", indent, "");
            XFPRINTF(fp, "Private-Key: (%d bit)\n", pBits);
        }
    }
    if (ret == WOLFSSL_SUCCESS && dsa->priv_key != NULL) {
        ret = PrintBNFieldFp(fp, indent, "priv", dsa->priv_key);
    }
    if (ret == WOLFSSL_SUCCESS && dsa->pub_key != NULL) {
        ret = PrintBNFieldFp(fp, indent, "pub", dsa->pub_key);
    }
    if (ret == WOLFSSL_SUCCESS && dsa->p != NULL) {
        ret = PrintBNFieldFp(fp, indent, "P", dsa->p);
    }
    if (ret == WOLFSSL_SUCCESS && dsa->q != NULL) {
        ret = PrintBNFieldFp(fp, indent, "Q", dsa->q);
    }
    if (ret == WOLFSSL_SUCCESS && dsa->g != NULL) {
        ret = PrintBNFieldFp(fp, indent, "G", dsa->g);
    }

    WOLFSSL_LEAVE("wolfSSL_DSA_print_fp", ret);

    return ret;
}
#endif /* OPENSSL_EXTRA && XSNPRINTF && !NO_FILESYSTEM && NO_STDIO_FILESYSTEM */

#if defined(OPENSSL_EXTRA) || defined(OPENSSL_EXTRA_X509_SMALL)
static void InitwolfSSL_DSA(WOLFSSL_DSA* dsa)
{
    if (dsa) {
        dsa->p        = NULL;
        dsa->q        = NULL;
        dsa->g        = NULL;
        dsa->pub_key  = NULL;
        dsa->priv_key = NULL;
        dsa->internal = NULL;
        dsa->inSet    = 0;
        dsa->exSet    = 0;
    }
}


WOLFSSL_DSA* wolfSSL_DSA_new(void)
{
    WOLFSSL_DSA* external;
    DsaKey*     key;

    WOLFSSL_MSG("wolfSSL_DSA_new");

    key = (DsaKey*) XMALLOC(sizeof(DsaKey), NULL, DYNAMIC_TYPE_DSA);
    if (key == NULL) {
        WOLFSSL_MSG("wolfSSL_DSA_new malloc DsaKey failure");
        return NULL;
    }

    external = (WOLFSSL_DSA*) XMALLOC(sizeof(WOLFSSL_DSA), NULL,
                                    DYNAMIC_TYPE_DSA);
    if (external == NULL) {
        WOLFSSL_MSG("wolfSSL_DSA_new malloc WOLFSSL_DSA failure");
        XFREE(key, NULL, DYNAMIC_TYPE_DSA);
        return NULL;
    }

    InitwolfSSL_DSA(external);
    if (wc_InitDsaKey(key) != 0) {
        WOLFSSL_MSG("wolfSSL_DSA_new InitDsaKey failure");
        XFREE(key, NULL, DYNAMIC_TYPE_DSA);
        wolfSSL_DSA_free(external);
        return NULL;
    }
    external->internal = key;

    return external;
}


void wolfSSL_DSA_free(WOLFSSL_DSA* dsa)
{
    WOLFSSL_MSG("wolfSSL_DSA_free");

    if (dsa) {
        if (dsa->internal) {
            FreeDsaKey((DsaKey*)dsa->internal);
            XFREE(dsa->internal, NULL, DYNAMIC_TYPE_DSA);
            dsa->internal = NULL;
        }
        wolfSSL_BN_free(dsa->priv_key);
        wolfSSL_BN_free(dsa->pub_key);
        wolfSSL_BN_free(dsa->g);
        wolfSSL_BN_free(dsa->q);
        wolfSSL_BN_free(dsa->p);
        InitwolfSSL_DSA(dsa);  /* set back to NULLs for safety */

        XFREE(dsa, NULL, DYNAMIC_TYPE_DSA);

        /* dsa = NULL, don't try to access or double free it */
    }
}

/* wolfSSL -> OpenSSL */
int SetDsaExternal(WOLFSSL_DSA* dsa)
{
    DsaKey* key;
    WOLFSSL_MSG("Entering SetDsaExternal");

    if (dsa == NULL || dsa->internal == NULL) {
        WOLFSSL_MSG("dsa key NULL error");
        return WOLFSSL_FATAL_ERROR;
    }

    key = (DsaKey*)dsa->internal;

    if (SetIndividualExternal(&dsa->p, &key->p) != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("dsa p key error");
        return WOLFSSL_FATAL_ERROR;
    }

    if (SetIndividualExternal(&dsa->q, &key->q) != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("dsa q key error");
        return WOLFSSL_FATAL_ERROR;
    }

    if (SetIndividualExternal(&dsa->g, &key->g) != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("dsa g key error");
        return WOLFSSL_FATAL_ERROR;
    }

    if (SetIndividualExternal(&dsa->pub_key, &key->y) != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("dsa y key error");
        return WOLFSSL_FATAL_ERROR;
    }

    if (SetIndividualExternal(&dsa->priv_key, &key->x) != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("dsa x key error");
        return WOLFSSL_FATAL_ERROR;
    }

    dsa->exSet = 1;

    return WOLFSSL_SUCCESS;
}
#endif /* OPENSSL_EXTRA || OPENSSL_EXTRA_X509_SMALL */
#endif /* !NO_DSA */

#if !defined(NO_DSA) && defined(OPENSSL_EXTRA)
/* Openssl -> WolfSSL */
int SetDsaInternal(WOLFSSL_DSA* dsa)
{
    DsaKey* key;
    WOLFSSL_MSG("Entering SetDsaInternal");

    if (dsa == NULL || dsa->internal == NULL) {
        WOLFSSL_MSG("dsa key NULL error");
        return WOLFSSL_FATAL_ERROR;
    }

    key = (DsaKey*)dsa->internal;

    if (dsa->p != NULL &&
        SetIndividualInternal(dsa->p, &key->p) != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("rsa p key error");
        return WOLFSSL_FATAL_ERROR;
    }

    if (dsa->q != NULL &&
        SetIndividualInternal(dsa->q, &key->q) != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("rsa q key error");
        return WOLFSSL_FATAL_ERROR;
    }

    if (dsa->g != NULL &&
        SetIndividualInternal(dsa->g, &key->g) != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("rsa g key error");
        return WOLFSSL_FATAL_ERROR;
    }

    if (dsa->pub_key != NULL) {
        if (SetIndividualInternal(dsa->pub_key, &key->y) != WOLFSSL_SUCCESS) {
            WOLFSSL_MSG("rsa pub_key error");
            return WOLFSSL_FATAL_ERROR;
        }

        /* public key */
        key->type = DSA_PUBLIC;
    }

    if (dsa->priv_key != NULL) {
        if (SetIndividualInternal(dsa->priv_key, &key->x) != WOLFSSL_SUCCESS) {
            WOLFSSL_MSG("rsa priv_key error");
            return WOLFSSL_FATAL_ERROR;
        }

        /* private key */
        key->type = DSA_PRIVATE;
    }

    dsa->inSet = 1;

    return WOLFSSL_SUCCESS;
}
#endif /* !NO_DSA && OPENSSL_EXTRA */


#ifdef OPENSSL_EXTRA
#if !defined(NO_RSA)

/* return wolfSSL native error codes. */
static int wolfSSL_RSA_generate_key_native(WOLFSSL_RSA* rsa, int bits, WOLFSSL_BIGNUM* bn,
                                void* cb)
{
    int ret;

    (void)cb;
    (void)bn;
    (void)bits;

    WOLFSSL_ENTER("wolfSSL_RSA_generate_key_native");

    if (rsa == NULL || rsa->internal == NULL) {
        /* bit size checked during make key call */
        WOLFSSL_MSG("bad arguments");
        return BAD_FUNC_ARG;
    }

#ifdef WOLFSSL_KEY_GEN
    {
    #ifdef WOLFSSL_SMALL_STACK
        WC_RNG* rng;
    #else
        WC_RNG  rng[1];
    #endif

    #ifdef WOLFSSL_SMALL_STACK
        rng = (WC_RNG*)XMALLOC(sizeof(WC_RNG), NULL, DYNAMIC_TYPE_RNG);
        if (rng == NULL)
            return MEMORY_E;
    #endif

        if ((ret = wc_InitRng(rng)) < 0)
            WOLFSSL_MSG("RNG init failed");
        else if ((ret = wc_MakeRsaKey((RsaKey*)rsa->internal, bits,
                    wolfSSL_BN_get_word(bn), rng)) != MP_OKAY)
            WOLFSSL_MSG("wc_MakeRsaKey failed");
        else if ((ret = SetRsaExternal(rsa)) != WOLFSSL_SUCCESS)
            WOLFSSL_MSG("SetRsaExternal failed");
        else {
            rsa->inSet = 1;
            ret = WOLFSSL_ERROR_NONE;
        }

        wc_FreeRng(rng);
    #ifdef WOLFSSL_SMALL_STACK
        XFREE(rng, NULL, DYNAMIC_TYPE_RNG);
    #endif
    }
#else
    WOLFSSL_MSG("No Key Gen built in");
    ret = NOT_COMPILED_IN;
#endif
    return ret;
}

/* Generates a RSA key of length len
 *
 * len  length of RSA key i.e. 2048
 * e    e to use when generating RSA key
 * f    callback function for generation details
 * data user callback argument
 *
 * Note: Because of wc_MakeRsaKey an RSA key size generated can be slightly
 *       rounded down. For example generating a key of size 2999 with e =
 *       65537 will make a key of size 374 instead of 375.
 * Returns a new RSA key on success and NULL on failure
 */
WOLFSSL_RSA* wolfSSL_RSA_generate_key(int len, unsigned long e,
                                      void(*f)(int, int, void*), void* data)
{
    WOLFSSL_RSA*    rsa = NULL;
    WOLFSSL_BIGNUM* bn  = NULL;

    WOLFSSL_ENTER("wolfSSL_RSA_generate_key");

    (void)f;
    (void)data;

    if (len < 0) {
        WOLFSSL_MSG("Bad argument: length was less than 0");
        return NULL;
    }

    bn = wolfSSL_BN_new();
    if (bn == NULL) {
        WOLFSSL_MSG("Error creating big number");
        return NULL;
    }

    if (wolfSSL_BN_set_word(bn, e) != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("Error using e value");
        wolfSSL_BN_free(bn);
        return NULL;
    }

    rsa = wolfSSL_RSA_new();
    if (rsa == NULL) {
        WOLFSSL_MSG("memory error");
    }
    else {
#ifdef HAVE_FIPS
        for (;;)
#endif
        {
            int gen_ret = wolfSSL_RSA_generate_key_native(rsa, len, bn, NULL);
            if (gen_ret != WOLFSSL_ERROR_NONE) {
#ifdef HAVE_FIPS
                if (gen_ret == PRIME_GEN_E)
                    continue;
#endif
                wolfSSL_RSA_free(rsa);
                rsa = NULL;
            }
#ifdef HAVE_FIPS
            break;
#endif
        }
    }
    wolfSSL_BN_free(bn);

    return rsa;
}

/* return compliant with OpenSSL
 *   1 if success, 0 if error
 */
int wolfSSL_RSA_generate_key_ex(WOLFSSL_RSA* rsa, int bits, WOLFSSL_BIGNUM* bn,
                                void* cb)
{
#ifdef HAVE_FIPS
    for (;;)
#endif
    {
        int gen_ret = wolfSSL_RSA_generate_key_native(rsa, bits, bn, cb);
        if (gen_ret == WOLFSSL_ERROR_NONE)
            return WOLFSSL_SUCCESS;
#ifdef HAVE_FIPS
        else if (gen_ret == PRIME_GEN_E)
            continue;
#endif
        else
            return WOLFSSL_FAILURE;
    }
}
#endif /* NO_RSA */

#ifndef NO_DSA
/* return code compliant with OpenSSL :
 *   1 if success, 0 if error
 */
int wolfSSL_DSA_generate_key(WOLFSSL_DSA* dsa)
{
    int ret = WOLFSSL_FAILURE;

    WOLFSSL_ENTER("wolfSSL_DSA_generate_key");

    if (dsa == NULL || dsa->internal == NULL) {
        WOLFSSL_MSG("Bad arguments");
        return WOLFSSL_FAILURE;
    }

    if (dsa->inSet == 0) {
        WOLFSSL_MSG("No DSA internal set, do it");

        if (SetDsaInternal(dsa) != WOLFSSL_SUCCESS) {
            WOLFSSL_MSG("SetDsaInternal failed");
            return ret;
        }
    }

#ifdef WOLFSSL_KEY_GEN
    {
        int initTmpRng = 0;
        WC_RNG *rng = NULL;
#ifdef WOLFSSL_SMALL_STACK
        WC_RNG *tmpRNG;
#else
        WC_RNG tmpRNG[1];
#endif

#ifdef WOLFSSL_SMALL_STACK
        tmpRNG = (WC_RNG*)XMALLOC(sizeof(WC_RNG), NULL, DYNAMIC_TYPE_RNG);
        if (tmpRNG == NULL)
            return WOLFSSL_FATAL_ERROR;
#endif
        if (wc_InitRng(tmpRNG) == 0) {
            rng = tmpRNG;
            initTmpRng = 1;
        }
        else {
            WOLFSSL_MSG("Bad RNG Init, trying global");
            if (initGlobalRNG == 0)
                WOLFSSL_MSG("Global RNG no Init");
            else
                rng = &globalRNG;
        }

        if (rng) {
            /* These were allocated above by SetDsaInternal(). They should
             * be cleared before wc_MakeDsaKey() which reinitializes
             * x and y. */
            mp_clear(&((DsaKey*)dsa->internal)->x);
            mp_clear(&((DsaKey*)dsa->internal)->y);

            if (wc_MakeDsaKey(rng, (DsaKey*)dsa->internal) != MP_OKAY)
                WOLFSSL_MSG("wc_MakeDsaKey failed");
            else if (SetDsaExternal(dsa) != WOLFSSL_SUCCESS)
                WOLFSSL_MSG("SetDsaExternal failed");
            else
                ret = WOLFSSL_SUCCESS;
        }

        if (initTmpRng)
            wc_FreeRng(tmpRNG);

#ifdef WOLFSSL_SMALL_STACK
        XFREE(tmpRNG, NULL, DYNAMIC_TYPE_RNG);
#endif
    }
#else /* WOLFSSL_KEY_GEN */
    WOLFSSL_MSG("No Key Gen built in");
#endif
    return ret;
}


/* Returns a pointer to a new WOLFSSL_DSA structure on success and NULL on fail
 */
WOLFSSL_DSA* wolfSSL_DSA_generate_parameters(int bits, unsigned char* seed,
        int seedLen, int* counterRet, unsigned long* hRet,
        WOLFSSL_BN_CB cb, void* CBArg)
{
    WOLFSSL_DSA* dsa;

    WOLFSSL_ENTER("wolfSSL_DSA_generate_parameters()");

    (void)cb;
    (void)CBArg;
    dsa = wolfSSL_DSA_new();
    if (dsa == NULL) {
        return NULL;
    }

    if (wolfSSL_DSA_generate_parameters_ex(dsa, bits, seed, seedLen,
                                  counterRet, hRet, NULL) != WOLFSSL_SUCCESS) {
        wolfSSL_DSA_free(dsa);
        return NULL;
    }

    return dsa;
}


/* return code compliant with OpenSSL :
 *   1 if success, 0 if error
 */
int wolfSSL_DSA_generate_parameters_ex(WOLFSSL_DSA* dsa, int bits,
                                       unsigned char* seed, int seedLen,
                                       int* counterRet,
                                       unsigned long* hRet, void* cb)
{
    int ret = WOLFSSL_FAILURE;

    (void)bits;
    (void)seed;
    (void)seedLen;
    (void)counterRet;
    (void)hRet;
    (void)cb;

    WOLFSSL_ENTER("wolfSSL_DSA_generate_parameters_ex");

    if (dsa == NULL || dsa->internal == NULL) {
        WOLFSSL_MSG("Bad arguments");
        return WOLFSSL_FAILURE;
    }

#ifdef WOLFSSL_KEY_GEN
    {
        int initTmpRng = 0;
        WC_RNG *rng = NULL;
#ifdef WOLFSSL_SMALL_STACK
        WC_RNG *tmpRNG;
#else
        WC_RNG tmpRNG[1];
#endif

#ifdef WOLFSSL_SMALL_STACK
        tmpRNG = (WC_RNG*)XMALLOC(sizeof(WC_RNG), NULL, DYNAMIC_TYPE_RNG);
        if (tmpRNG == NULL)
            return WOLFSSL_FATAL_ERROR;
#endif
        if (wc_InitRng(tmpRNG) == 0) {
            rng = tmpRNG;
            initTmpRng = 1;
        }
        else {
            WOLFSSL_MSG("Bad RNG Init, trying global");
            if (initGlobalRNG == 0)
                WOLFSSL_MSG("Global RNG no Init");
            else
                rng = &globalRNG;
        }

        if (rng) {
            if (wc_MakeDsaParameters(rng, bits,
                                     (DsaKey*)dsa->internal) != MP_OKAY)
                WOLFSSL_MSG("wc_MakeDsaParameters failed");
            else if (SetDsaExternal(dsa) != WOLFSSL_SUCCESS)
                WOLFSSL_MSG("SetDsaExternal failed");
            else
                ret = WOLFSSL_SUCCESS;
        }

        if (initTmpRng)
            wc_FreeRng(tmpRNG);

#ifdef WOLFSSL_SMALL_STACK
        XFREE(tmpRNG, NULL, DYNAMIC_TYPE_RNG);
#endif
    }
#else /* WOLFSSL_KEY_GEN */
    WOLFSSL_MSG("No Key Gen built in");
#endif

    return ret;
}

void wolfSSL_DSA_get0_pqg(const WOLFSSL_DSA *d, const WOLFSSL_BIGNUM **p,
        const WOLFSSL_BIGNUM **q, const WOLFSSL_BIGNUM **g)
{
    WOLFSSL_ENTER("wolfSSL_DSA_get0_pqg");
    if (d != NULL) {
        if (p != NULL)
            *p = d->p;
        if (q != NULL)
            *q = d->q;
        if (g != NULL)
            *g = d->g;
    }
}

int wolfSSL_DSA_set0_pqg(WOLFSSL_DSA *d, WOLFSSL_BIGNUM *p,
        WOLFSSL_BIGNUM *q, WOLFSSL_BIGNUM *g)
{
    WOLFSSL_ENTER("wolfSSL_DSA_set0_pqg");
    if (d == NULL || p == NULL || q == NULL || g == NULL) {
        WOLFSSL_MSG("Bad parameter");
        return WOLFSSL_FAILURE;
    }
    wolfSSL_BN_free(d->p);
    wolfSSL_BN_free(d->q);
    wolfSSL_BN_free(d->g);
    d->p = p;
    d->q = q;
    d->g = g;
    return WOLFSSL_SUCCESS;
}

void wolfSSL_DSA_get0_key(const WOLFSSL_DSA *d,
        const WOLFSSL_BIGNUM **pub_key, const WOLFSSL_BIGNUM **priv_key)
{
    WOLFSSL_ENTER("wolfSSL_DSA_get0_key");
    if (d != NULL) {
        if (pub_key != NULL)
            *pub_key = d->pub_key;
        if (priv_key != NULL)
            *priv_key = d->priv_key;
    }
}

int wolfSSL_DSA_set0_key(WOLFSSL_DSA *d, WOLFSSL_BIGNUM *pub_key,
        WOLFSSL_BIGNUM *priv_key)
{
    WOLFSSL_ENTER("wolfSSL_DSA_set0_key");

    /* The private key may be NULL */
    if (pub_key == NULL) {
        WOLFSSL_MSG("Bad parameter");
        return WOLFSSL_FAILURE;
    }

    wolfSSL_BN_free(d->pub_key);
    wolfSSL_BN_free(d->priv_key);
    d->pub_key = pub_key;
    d->priv_key = priv_key;

    return WOLFSSL_SUCCESS;
}

WOLFSSL_DSA_SIG* wolfSSL_DSA_SIG_new(void)
{
    WOLFSSL_DSA_SIG* sig;
    WOLFSSL_ENTER("wolfSSL_DSA_SIG_new");
    sig = (WOLFSSL_DSA_SIG*)XMALLOC(sizeof(WOLFSSL_DSA_SIG), NULL, DYNAMIC_TYPE_OPENSSL);
    if (sig)
        XMEMSET(sig, 0, sizeof(WOLFSSL_DSA_SIG));
    return sig;
}

void wolfSSL_DSA_SIG_free(WOLFSSL_DSA_SIG *sig)
{
    WOLFSSL_ENTER("wolfSSL_DSA_SIG_free");
    if (sig) {
        if (sig->r) {
            wolfSSL_BN_free(sig->r);
        }
        if (sig->s) {
            wolfSSL_BN_free(sig->s);
        }
        XFREE(sig, NULL, DYNAMIC_TYPE_OPENSSL);
    }
}

void wolfSSL_DSA_SIG_get0(const WOLFSSL_DSA_SIG *sig,
        const WOLFSSL_BIGNUM **r, const WOLFSSL_BIGNUM **s)
{
    WOLFSSL_ENTER("wolfSSL_DSA_SIG_get0");
    if (sig != NULL) {
        *r = sig->r;
        *s = sig->s;
    }
}

int wolfSSL_DSA_SIG_set0(WOLFSSL_DSA_SIG *sig, WOLFSSL_BIGNUM *r,
        WOLFSSL_BIGNUM *s)
{
    WOLFSSL_ENTER("wolfSSL_DSA_SIG_set0");
    if (r == NULL || s == NULL) {
        WOLFSSL_MSG("Bad parameter");
        return WOLFSSL_FAILURE;
    }

    wolfSSL_BN_clear_free(sig->r);
    wolfSSL_BN_clear_free(sig->s);
    sig->r = r;
    sig->s = s;

    return WOLFSSL_SUCCESS;
}

#ifndef HAVE_SELFTEST
/**
 *
 * @param sig The input signature to encode
 * @param out The output buffer. If *out is NULL then a new buffer is
 *            allocated. Otherwise the output is written to the buffer.
 * @return length on success and -1 on error
 */
int wolfSSL_i2d_DSA_SIG(const WOLFSSL_DSA_SIG *sig, byte **out)
{
    /* Space for sequence + two asn ints */
    byte buf[MAX_SEQ_SZ + 2*(ASN_TAG_SZ + MAX_LENGTH_SZ + DSA_MAX_HALF_SIZE)];
    word32 bufLen = sizeof(buf);

    WOLFSSL_ENTER("wolfSSL_i2d_DSA_SIG");

    if (sig == NULL || sig->r == NULL || sig->s == NULL ||
            out == NULL) {
        WOLFSSL_MSG("Bad function arguments");
        return WOLFSSL_FATAL_ERROR;
    }

    if (StoreECC_DSA_Sig(buf, &bufLen,
            (mp_int*)sig->r->internal, (mp_int*)sig->s->internal) != 0) {
        WOLFSSL_MSG("StoreECC_DSA_Sig error");
        return WOLFSSL_FATAL_ERROR;
    }

    if (*out == NULL) {
        byte* tmp = (byte*)XMALLOC(bufLen, NULL, DYNAMIC_TYPE_ASN1);
        if (tmp == NULL) {
            WOLFSSL_MSG("malloc error");
            return WOLFSSL_FATAL_ERROR;
        }
        *out = tmp;
    }

    XMEMCPY(*out, buf, bufLen);

    return (int)bufLen;
}

/**
 * Same as wolfSSL_DSA_SIG_new but also initializes the internal bignums as well.
 * @return New WOLFSSL_DSA_SIG with r and s created as well
 */
static WOLFSSL_DSA_SIG* wolfSSL_DSA_SIG_new_bn(void)
{
    WOLFSSL_DSA_SIG* ret;

    if ((ret = wolfSSL_DSA_SIG_new()) == NULL) {
        WOLFSSL_MSG("wolfSSL_DSA_SIG_new error");
        return NULL;
    }

    if ((ret->r = wolfSSL_BN_new()) == NULL) {
        WOLFSSL_MSG("wolfSSL_BN_new error");
        wolfSSL_DSA_SIG_free(ret);
        return NULL;
    }

    if ((ret->s = wolfSSL_BN_new()) == NULL) {
        WOLFSSL_MSG("wolfSSL_BN_new error");
        wolfSSL_DSA_SIG_free(ret);
        return NULL;
    }

    return ret;
}

/**
 * This parses a DER encoded ASN.1 structure. The ASN.1 encoding is:
 * ASN1_SEQUENCE
 *   ASN1_INTEGER (DSA r)
 *   ASN1_INTEGER (DSA s)
 * Alternatively, if the input is DSA_160_SIG_SIZE or DSA_256_SIG_SIZE in
 * length then this API interprets this as two unsigned binary numbers.
 * @param sig    If non-null then free'd first and then newly created
 *               WOLFSSL_DSA_SIG is assigned
 * @param pp     Input buffer that is moved forward on success
 * @param length Length of input buffer
 * @return Newly created WOLFSSL_DSA_SIG on success or NULL on failure
 */
WOLFSSL_DSA_SIG* wolfSSL_d2i_DSA_SIG(WOLFSSL_DSA_SIG **sig,
        const unsigned char **pp, long length)
{
    WOLFSSL_DSA_SIG* ret;
    mp_int* r;
    mp_int* s;

    WOLFSSL_ENTER("wolfSSL_d2i_DSA_SIG");

    if (pp == NULL || *pp == NULL || length < 0) {
        WOLFSSL_MSG("Bad function arguments");
        return NULL;
    }

    if ((ret = wolfSSL_DSA_SIG_new_bn()) == NULL) {
        WOLFSSL_MSG("wolfSSL_DSA_SIG_new_bn error");
        return NULL;
    }

    r = (mp_int*)ret->r->internal;
    s = (mp_int*)ret->s->internal;

    if (DecodeECC_DSA_Sig(*pp, (word32)length, r, s) != 0) {
        if (length == DSA_160_SIG_SIZE || length == DSA_256_SIG_SIZE) {
            /* Two raw numbers of length/2 size each */
            if (mp_read_unsigned_bin(r, *pp, (int)length/2) != 0) {
                WOLFSSL_MSG("r mp_read_unsigned_bin error");
                wolfSSL_DSA_SIG_free(ret);
                return NULL;
            }

            if (mp_read_unsigned_bin(s, *pp + (length/2), (int)length/2) != 0) {
                WOLFSSL_MSG("s mp_read_unsigned_bin error");
                wolfSSL_DSA_SIG_free(ret);
                return NULL;
            }

            *pp += length;
        }
        else {
            WOLFSSL_MSG("DecodeECC_DSA_Sig error");
            wolfSSL_DSA_SIG_free(ret);
            return NULL;
        }
    }
    else {
        /* DecodeECC_DSA_Sig success move pointer forward */
#ifndef NO_STRICT_ECDSA_LEN
        *pp += length;
#else
        {
            /* We need to figure out how much to move by ourselves */
            word32 idx = 0;
            int len = 0;
            if (GetSequence(*pp, &idx, &len, (word32)length) < 0) {
                WOLFSSL_MSG("GetSequence error");
                wolfSSL_DSA_SIG_free(ret);
                return NULL;
            }
            *pp += len;
        }
#endif
    }

    if (sig != NULL) {
        if (*sig != NULL)
            wolfSSL_DSA_SIG_free(*sig);
        *sig = ret;
    }

    return ret;
}
#endif

/* return WOLFSSL_SUCCESS on success, < 0 otherwise */
int wolfSSL_DSA_do_sign(const unsigned char* d, unsigned char* sigRet,
                       WOLFSSL_DSA* dsa)
{
    int     ret = WOLFSSL_FATAL_ERROR;
    int     initTmpRng = 0;
    WC_RNG* rng = NULL;
#ifdef WOLFSSL_SMALL_STACK
    WC_RNG* tmpRNG = NULL;
#else
    WC_RNG  tmpRNG[1];
#endif

    WOLFSSL_ENTER("wolfSSL_DSA_do_sign");

    if (d == NULL || sigRet == NULL || dsa == NULL) {
        WOLFSSL_MSG("Bad function arguments");
        return ret;
    }

    if (dsa->inSet == 0) {
        WOLFSSL_MSG("No DSA internal set, do it");
        if (SetDsaInternal(dsa) != WOLFSSL_SUCCESS) {
            WOLFSSL_MSG("SetDsaInternal failed");
            return ret;
        }
    }

#ifdef WOLFSSL_SMALL_STACK
    tmpRNG = (WC_RNG*)XMALLOC(sizeof(WC_RNG), NULL, DYNAMIC_TYPE_RNG);
    if (tmpRNG == NULL)
        return WOLFSSL_FATAL_ERROR;
#endif

    if (wc_InitRng(tmpRNG) == 0) {
        rng = tmpRNG;
        initTmpRng = 1;
    }
    else {
        WOLFSSL_MSG("Bad RNG Init, trying global");
        if (initGlobalRNG == 0)
            WOLFSSL_MSG("Global RNG no Init");
        else
            rng = &globalRNG;
    }

    if (rng) {
        if (wc_DsaSign(d, sigRet, (DsaKey*)dsa->internal, rng) < 0)
            WOLFSSL_MSG("DsaSign failed");
        else
            ret = WOLFSSL_SUCCESS;
    }

    if (initTmpRng)
        wc_FreeRng(tmpRNG);
#ifdef WOLFSSL_SMALL_STACK
    XFREE(tmpRNG, NULL, DYNAMIC_TYPE_RNG);
#endif

    return ret;
}

#ifndef HAVE_SELFTEST
WOLFSSL_DSA_SIG* wolfSSL_DSA_do_sign_ex(const unsigned char* digest,
                                        int inLen, WOLFSSL_DSA* dsa)
{
    byte sigBin[DSA_MAX_SIG_SIZE];
    const byte *tmp = sigBin;
    int sigLen;

    WOLFSSL_ENTER("wolfSSL_DSA_do_sign_ex");

    if (!digest || !dsa || inLen != WC_SHA_DIGEST_SIZE) {
        WOLFSSL_MSG("Bad function arguments");
        return NULL;
    }

    if (wolfSSL_DSA_do_sign(digest, sigBin, dsa) != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("wolfSSL_DSA_do_sign error");
        return NULL;
    }

    if (dsa->internal == NULL) {
        WOLFSSL_MSG("dsa->internal is null");
        return NULL;
    }

    sigLen = mp_unsigned_bin_size(&((DsaKey*)dsa->internal)->q);
    if (sigLen <= 0) {
        WOLFSSL_MSG("mp_unsigned_bin_size error");
        return NULL;
    }

    /* 2 * sigLen for the two points r and s */
    return wolfSSL_d2i_DSA_SIG(NULL, &tmp, 2 * sigLen);
}
#endif /* !HAVE_SELFTEST */

int wolfSSL_DSA_do_verify(const unsigned char* d, unsigned char* sig,
                        WOLFSSL_DSA* dsa, int *dsacheck)
{
    int    ret = WOLFSSL_FATAL_ERROR;

    WOLFSSL_ENTER("wolfSSL_DSA_do_verify");

    if (d == NULL || sig == NULL || dsa == NULL) {
        WOLFSSL_MSG("Bad function arguments");
        return WOLFSSL_FATAL_ERROR;
    }
    if (dsa->inSet == 0)
    {
        WOLFSSL_MSG("No DSA internal set, do it");

        if (SetDsaInternal(dsa) != WOLFSSL_SUCCESS) {
            WOLFSSL_MSG("SetDsaInternal failed");
            return WOLFSSL_FATAL_ERROR;
        }
    }

    ret = DsaVerify(d, sig, (DsaKey*)dsa->internal, dsacheck);
    if (ret != 0 || *dsacheck != 1) {
        WOLFSSL_MSG("DsaVerify failed");
        return ret;
    }

    return WOLFSSL_SUCCESS;
}


int wolfSSL_DSA_bits(const WOLFSSL_DSA *d)
{
    if (!d)
        return WOLFSSL_FAILURE;
    if (!d->exSet && SetDsaExternal((WOLFSSL_DSA*)d) != WOLFSSL_SUCCESS)
        return WOLFSSL_FAILURE;
    return wolfSSL_BN_num_bits(d->p);
}

#ifndef HAVE_SELFTEST
int wolfSSL_DSA_do_verify_ex(const unsigned char* digest, int digest_len,
                             WOLFSSL_DSA_SIG* sig, WOLFSSL_DSA* dsa)
{
    int dsacheck, sz;
    byte sigBin[DSA_MAX_SIG_SIZE];
    byte* sigBinPtr = sigBin;
    DsaKey* key;
    int qSz;

    WOLFSSL_ENTER("wolfSSL_DSA_do_verify_ex");

    if (!digest || !sig || !dsa || digest_len != WC_SHA_DIGEST_SIZE) {
        WOLFSSL_MSG("Bad function arguments");
        return WOLFSSL_FAILURE;
    }

    if (!sig->r || !sig->s) {
        WOLFSSL_MSG("No signature found in DSA_SIG");
        return WOLFSSL_FAILURE;
    }

    if (dsa->inSet == 0) {
        WOLFSSL_MSG("No DSA internal set, do it");
        if (SetDsaInternal(dsa) != WOLFSSL_SUCCESS) {
            WOLFSSL_MSG("SetDsaInternal failed");
            return WOLFSSL_FAILURE;
        }
    }

    key = (DsaKey*)dsa->internal;

    if (key == NULL) {
        WOLFSSL_MSG("dsa->internal is null");
        return WOLFSSL_FAILURE;
    }

    qSz = mp_unsigned_bin_size(&key->q);
    if (qSz < 0 || qSz > DSA_MAX_HALF_SIZE) {
        WOLFSSL_MSG("mp_unsigned_bin_size error");
        return WOLFSSL_FAILURE;
    }

    /* read r */
    /* front pad with zeros */
    if ((sz = wolfSSL_BN_num_bytes(sig->r)) < 0 || sz > DSA_MAX_HALF_SIZE)
        return WOLFSSL_FAILURE;
    while (sz++ < qSz)
        *sigBinPtr++ = 0;
    if (wolfSSL_BN_bn2bin(sig->r, sigBinPtr) == WOLFSSL_FATAL_ERROR)
        return WOLFSSL_FAILURE;

    /* Move to s */
    sigBinPtr = sigBin + qSz;

    /* read s */
    /* front pad with zeros */
    if ((sz = wolfSSL_BN_num_bytes(sig->s)) < 0 || sz > DSA_MAX_HALF_SIZE)
        return WOLFSSL_FAILURE;
    while (sz++ < qSz)
        *sigBinPtr++ = 0;
    if (wolfSSL_BN_bn2bin(sig->s, sigBinPtr) == WOLFSSL_FATAL_ERROR)
        return WOLFSSL_FAILURE;

    if (wolfSSL_DSA_do_verify(digest, sigBin, dsa, &dsacheck) != WOLFSSL_SUCCESS ||
            dsacheck != 1)
        return WOLFSSL_FAILURE;

    return WOLFSSL_SUCCESS;
}
#endif /* !HAVE_SELFTEST */

WOLFSSL_API int wolfSSL_i2d_DSAparams(const WOLFSSL_DSA* dsa,
    unsigned char** out)
{
    int ret = 0;
    word32 derLen = 0;
    int preAllocated = 1;
    DsaKey* key = NULL;

    WOLFSSL_ENTER("wolfSSL_i2d_DSAparams");

    if (dsa == NULL || dsa->internal == NULL || out == NULL) {
        ret = BAD_FUNC_ARG;
    }

    if (ret == 0) {
        key = (DsaKey*)dsa->internal;
        ret = wc_DsaKeyToParamsDer_ex(key, NULL, &derLen);
        if (ret == LENGTH_ONLY_E) {
            ret = 0;
        }
    }
    if (ret == 0 && *out == NULL) {
        /* If we're allocating out for the caller, we don't increment out just
           past the end of the DER buffer. If out is already allocated, we do.
           (OpenSSL convention) */
        preAllocated = 0;
        *out = (unsigned char*)XMALLOC(derLen, key->heap, DYNAMIC_TYPE_OPENSSL);
        if (*out == NULL) {
            ret = MEMORY_E;
        }
    }
    if (ret == 0) {
        ret = wc_DsaKeyToParamsDer_ex(key, *out, &derLen);
    }
    if (ret >= 0 && preAllocated == 1) {
        *out += derLen;
    }

    if (ret < 0 && preAllocated == 0) {
        XFREE(*out, key->heap, DYNAMIC_TYPE_OPENSSL);
    }

    WOLFSSL_LEAVE("wolfSSL_i2d_DSAparams", ret);

    return ret;
}

WOLFSSL_API WOLFSSL_DSA* wolfSSL_d2i_DSAparams(WOLFSSL_DSA** dsa,
    const unsigned char** der, long derLen)
{
    WOLFSSL_DSA* ret = NULL;
    int err = 0;
    word32 idx = 0;
    int asnLen;
    DsaKey* internalKey = NULL;

    WOLFSSL_ENTER("wolfSSL_d2i_DSAparams");

    if (der == NULL || *der == NULL || derLen <= 0) {
        err = 1;
    }
    if (err == 0) {
        ret = wolfSSL_DSA_new();
        err = ret == NULL;
    }
    if (err == 0) {
        err = GetSequence(*der, &idx, &asnLen, (word32)derLen) <= 0;
    }
    if (err == 0) {
        internalKey = (DsaKey*)ret->internal;
        err = GetInt(&internalKey->p, *der, &idx, (word32)derLen) != 0;
    }
    if (err == 0) {
        err = GetInt(&internalKey->q, *der, &idx, (word32)derLen) != 0;
    }
    if (err == 0) {
        err = GetInt(&internalKey->g, *der, &idx, (word32)derLen) != 0;
    }
    if (err == 0) {
        err = SetIndividualExternal(&ret->p, &internalKey->p)
                != WOLFSSL_SUCCESS;
    }
    if (err == 0) {
        err = SetIndividualExternal(&ret->q, &internalKey->q)
                != WOLFSSL_SUCCESS;
    }
    if (err == 0) {
        err = SetIndividualExternal(&ret->g, &internalKey->g)
                != WOLFSSL_SUCCESS;
    }
    if (err == 0 && dsa != NULL) {
        *dsa = ret;
    }

    if (err != 0 && ret != NULL) {
        wolfSSL_DSA_free(ret);
        ret = NULL;
    }

    return ret;
}
#endif /* NO_DSA */

#if !defined(NO_RSA) && !defined(HAVE_USER_RSA)

#ifdef DEBUG_SIGN
static void DEBUG_SIGN_msg(const char *title, const unsigned char *out, unsigned int outlen)
{
    const unsigned char *pt;
    printf("%s[%d] = \n", title, (int)outlen);
    outlen = outlen>100?100:outlen;
    for (pt = out; pt < out + outlen;
            printf("%c", ((*pt)&0x6f)>='A'?((*pt)&0x6f):'.'), pt++);
    printf("\n");
}
#else
#define DEBUG_SIGN_msg(a,b,c)
#endif

static int nid2HashSum(int type) {
    switch (type) {
    #ifdef WOLFSSL_MD2
        case NID_md2:       type = MD2h;    break;
    #endif
    #ifndef NO_MD5
        case NID_md5:       type = MD5h;    break;
    #endif
    #ifndef NO_SHA
        case NID_sha1:      type = SHAh;    break;
    #endif
    #ifndef NO_SHA256
        case NID_sha256:    type = SHA256h; break;
    #endif
    #ifdef WOLFSSL_SHA384
        case NID_sha384:    type = SHA384h; break;
    #endif
    #ifdef WOLFSSL_SHA512
        case NID_sha512:    type = SHA512h; break;
    #endif
    #ifndef WOLFSSL_NOSHA3_224
        case NID_sha3_224:  type = SHA3_224h; break;
    #endif
    #ifndef WOLFSSL_NOSHA3_256
        case NID_sha3_256:  type = SHA3_256h; break;
    #endif
    #ifndef WOLFSSL_NOSHA3_384
        case NID_sha3_384:  type = SHA3_384h; break;
    #endif
    #ifndef WOLFSSL_NOSHA3_512
        case NID_sha3_512:  type = SHA3_512h; break;
    #endif
        default:
            WOLFSSL_MSG("This NID (md type) not configured or not implemented");
            return 0;
    }
    return type;
}

/* return WOLFSSL_SUCCESS on ok, 0 otherwise */
int wolfSSL_RSA_sign(int type, const unsigned char* m,
                           unsigned int mLen, unsigned char* sigRet,
                           unsigned int* sigLen, WOLFSSL_RSA* rsa)
{
    return wolfSSL_RSA_sign_ex(type, m, mLen, sigRet, sigLen, rsa, 1);
}

int wolfSSL_RSA_sign_ex(int type, const unsigned char* m,
                           unsigned int mLen, unsigned char* sigRet,
                           unsigned int* sigLen, WOLFSSL_RSA* rsa, int flag)
{
    if (sigLen != NULL)
        *sigLen = RSA_MAX_SIZE / CHAR_BIT; /* No size checking in this API */
    return wolfSSL_RSA_sign_generic_padding(type, m, mLen, sigRet, sigLen,
            rsa, flag, RSA_PKCS1_PADDING);
}

/**
 * Sign a message with the chosen message digest, padding, and RSA key.
 * @param type      Hash NID
 * @param m         Message to sign. Most likely this will be the digest of
 *                  the message to sign
 * @param mLen      Length of message to sign
 * @param sigRet    Output buffer
 * @param sigLen    On Input: length of sigRet buffer
 *                  On Output: length of data written to sigRet
 * @param rsa       RSA key used to sign the input
 * @param flag      1: Output the signature
 *                  0: Output the value that the unpadded signature should be
 *                     compared to. Note: for RSA_PKCS1_PSS_PADDING the
 *                     wc_RsaPSS_CheckPadding_ex function should be used to check
 *                     the output of a *Verify* function.
 * @param padding   Padding to use. Only RSA_PKCS1_PSS_PADDING and
 *                  RSA_PKCS1_PADDING are currently supported for signing.
 * @return          WOLFSSL_SUCCESS on success and WOLFSSL_FAILURE on error
 */
int wolfSSL_RSA_sign_generic_padding(int type, const unsigned char* m,
                           unsigned int mLen, unsigned char* sigRet,
                           unsigned int* sigLen, WOLFSSL_RSA* rsa, int flag,
                           int padding)
{
    word32  outLen;
    word32  signSz;
    int     initTmpRng = 0;
    WC_RNG* rng        = NULL;
    int     ret        = 0;
#ifdef WOLFSSL_SMALL_STACK
    WC_RNG* tmpRNG     = NULL;
    byte*   encodedSig = NULL;
#else
    WC_RNG  tmpRNG[1];
    byte    encodedSig[MAX_ENCODED_SIG_SZ];
#endif

    WOLFSSL_ENTER("wolfSSL_RSA_sign_generic_padding");

    if (m == NULL || sigRet == NULL || sigLen == NULL || rsa == NULL) {
        WOLFSSL_MSG("Bad function arguments");
        return WOLFSSL_FAILURE;
    }
    DEBUG_SIGN_msg("Message to Sign", m, mLen);

    if (rsa->inSet == 0) {
        WOLFSSL_MSG("No RSA internal set, do it");

        if (SetRsaInternal(rsa) != WOLFSSL_SUCCESS) {
            WOLFSSL_MSG("SetRsaInternal failed");
            return WOLFSSL_FAILURE;
        }
    }

    type = nid2HashSum(type);

    outLen = (word32)wolfSSL_BN_num_bytes(rsa->n);

#ifdef WOLFSSL_SMALL_STACK
    tmpRNG = (WC_RNG*)XMALLOC(sizeof(WC_RNG), NULL, DYNAMIC_TYPE_RNG);
    if (tmpRNG == NULL)
        return WOLFSSL_FAILURE;

    encodedSig = (byte*)XMALLOC(MAX_ENCODED_SIG_SZ, NULL,
                                                   DYNAMIC_TYPE_SIGNATURE);
    if (encodedSig == NULL) {
        XFREE(tmpRNG, NULL, DYNAMIC_TYPE_RNG);
        return WOLFSSL_FAILURE;
    }
#endif

    if (outLen == 0) {
        WOLFSSL_MSG("Bad RSA size");
    }
    else if (outLen > *sigLen) {
        WOLFSSL_MSG("Output buffer too small");
        return WOLFSSL_FAILURE;
    }
    else if (wc_InitRng(tmpRNG) == 0) {
        rng = tmpRNG;
        initTmpRng = 1;
    }
    else {
        WOLFSSL_MSG("Bad RNG Init, trying global");

        if (initGlobalRNG == 0)
            WOLFSSL_MSG("Global RNG no Init");
        else
            rng = &globalRNG;
    }

    if (rng) {
        if (flag != 0) {
            switch (padding) {
#ifdef WC_RSA_NO_PADDING
            case RSA_NO_PADDING:
                WOLFSSL_MSG("RSA_NO_PADDING not supported for signing");
                ret = BAD_FUNC_ARG;
                break;
#endif
#if defined(WC_RSA_PSS) && !defined(HAVE_SELFTEST) && (!defined(HAVE_FIPS) || \
    (defined(FIPS_VERSION_GE) && FIPS_VERSION_GE(5,1)))
            case RSA_PKCS1_PSS_PADDING:
            {
                enum wc_HashType hType = wc_OidGetHash(type);
#ifndef WOLFSSL_PSS_SALT_LEN_DISCOVER
                WOLFSSL_MSG("Using RSA-PSS with hash length salt. "
                            "OpenSSL uses max length by default.");
#endif
                ret = wc_RsaPSS_Sign_ex(m, mLen, sigRet, outLen,
                        hType, wc_hash2mgf(hType),
#ifndef WOLFSSL_PSS_SALT_LEN_DISCOVER
                        RSA_PSS_SALT_LEN_DEFAULT,
#else
                        RSA_PSS_SALT_LEN_DISCOVER,
#endif
                        (RsaKey*)rsa->internal, rng);
                break;
            }
#endif
#ifndef WC_NO_RSA_OAEP
            case RSA_PKCS1_OAEP_PADDING:
            {
                WOLFSSL_MSG("RSA_PKCS1_OAEP_PADDING not supported for signing");
                ret = BAD_FUNC_ARG;
                break;
            }
#endif
            case RSA_PKCS1_PADDING:
                signSz = wc_EncodeSignature(encodedSig, m, mLen, type);
                if (signSz == 0) {
                    WOLFSSL_MSG("Bad Encode Signature");
                }
                DEBUG_SIGN_msg("Encoded Message", encodedSig, signSz);
                ret = wc_RsaSSL_Sign(encodedSig, signSz, sigRet, outLen,
                                (RsaKey*)rsa->internal, rng);
                break;
            default:
                WOLFSSL_MSG("Unsupported padding");
                ret = BAD_FUNC_ARG;
                break;
            }
            if (ret <= 0) {
                WOLFSSL_MSG("Bad Rsa Sign");
                ret = 0;
            }
            else {
                *sigLen = (unsigned int)ret;
                ret = WOLFSSL_SUCCESS;
                DEBUG_SIGN_msg("Signature", sigRet, *sigLen);
            }
        } else {
            switch (padding) {
            case RSA_NO_PADDING:
            case RSA_PKCS1_PSS_PADDING:
            case RSA_PKCS1_OAEP_PADDING:
                ret = WOLFSSL_SUCCESS;
                XMEMCPY(sigRet, m, mLen);
                *sigLen = mLen;
                break;
            case RSA_PKCS1_PADDING:
            default:
                signSz = wc_EncodeSignature(encodedSig, m, mLen, type);
                if (signSz == 0) {
                    WOLFSSL_MSG("Bad Encode Signature");
                }
                ret = WOLFSSL_SUCCESS;
                XMEMCPY(sigRet, encodedSig, signSz);
                *sigLen = signSz;
                break;
            }
        }
    }

    if (initTmpRng)
        wc_FreeRng(tmpRNG);

#ifdef WOLFSSL_SMALL_STACK
    XFREE(tmpRNG,     NULL, DYNAMIC_TYPE_RNG);
    XFREE(encodedSig, NULL, DYNAMIC_TYPE_SIGNATURE);
#endif

    if (ret == WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("wolfSSL_RSA_sign_generic_padding success");
    }
    else {
        WOLFSSL_LEAVE("wolfSSL_RSA_sign_generic_padding", ret);
        WOLFSSL_MSG("wolfSSL_RSA_sign_generic_padding failed. "
                    "Returning WOLFSSL_FAILURE.");
        ret = WOLFSSL_FAILURE;
    }
    return ret;
}

/* returns WOLFSSL_SUCCESS on successful verify and WOLFSSL_FAILURE on fail */
int wolfSSL_RSA_verify(int type, const unsigned char* m,
                               unsigned int mLen, const unsigned char* sig,
                               unsigned int sigLen, WOLFSSL_RSA* rsa)
{
    return wolfSSL_RSA_verify_ex(type, m, mLen, sig, sigLen, rsa, RSA_PKCS1_PADDING);
}

/* returns WOLFSSL_SUCCESS on successful verify and WOLFSSL_FAILURE on fail */
int wolfSSL_RSA_verify_ex(int type, const unsigned char* m,
                               unsigned int mLen, const unsigned char* sig,
                               unsigned int sigLen, WOLFSSL_RSA* rsa,
                               int padding) {
    int     ret = WOLFSSL_FAILURE;
    unsigned char *sigRet = NULL;
    unsigned char *sigDec = NULL;
    unsigned int   len = sigLen;
    int verLen;
#if (!defined(HAVE_FIPS) || (defined(FIPS_VERSION_GE) && \
    FIPS_VERSION_GE(5,1))) && !defined(HAVE_SELFTEST)
    int hSum = nid2HashSum(type);
    enum wc_HashType hType;
#endif

    WOLFSSL_ENTER("wolfSSL_RSA_verify");
    if ((m == NULL) || (sig == NULL)) {
        WOLFSSL_MSG("Bad function arguments");
        return WOLFSSL_FAILURE;
    }
    sigDec = (unsigned char *)XMALLOC(sigLen, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    if (sigDec == NULL) {
        WOLFSSL_MSG("Memory failure");
        goto cleanup;
    }
    if (padding != RSA_PKCS1_PSS_PADDING) {
        sigRet = (unsigned char *)XMALLOC(sigLen, NULL, DYNAMIC_TYPE_TMP_BUFFER);
        if (sigRet == NULL) {
            WOLFSSL_MSG("Memory failure");
            goto cleanup;
        }
        /* get non-encrypted signature to be compared with decrypted signature */
        if (wolfSSL_RSA_sign_generic_padding(type, m, mLen, sigRet, &len, rsa,
                0, padding) <= 0) {
            WOLFSSL_MSG("Message Digest Error");
            goto cleanup;
        }
        DEBUG_SIGN_msg("Encoded Message", sigRet, len);
    }
    else {
        DEBUG_SIGN_msg("Encoded Message", m, mLen);
    }
    /* decrypt signature */
#if (!defined(HAVE_FIPS) || (defined(FIPS_VERSION_GE) && \
    FIPS_VERSION_GE(5,1))) && !defined(HAVE_SELFTEST)
    hType = wc_OidGetHash(hSum);
    if ((verLen = wc_RsaSSL_Verify_ex2(sig, sigLen, (unsigned char *)sigDec,
            sigLen, (RsaKey*)rsa->internal, padding, hType)) <= 0) {
        WOLFSSL_MSG("RSA Decrypt error");
        goto cleanup;
    }
#else
    verLen = wc_RsaSSL_Verify(sig, sigLen, (unsigned char *)sigDec, sigLen,
        (RsaKey*)rsa->internal);
#endif
    DEBUG_SIGN_msg("Decrypted Signature", sigDec, ret);
#if defined(WC_RSA_PSS) && !defined(HAVE_SELFTEST) && (!defined(HAVE_FIPS) || \
    (defined(FIPS_VERSION_GE) && FIPS_VERSION_GE(5,1)))
    if (padding == RSA_PKCS1_PSS_PADDING) {
        if (wc_RsaPSS_CheckPadding_ex(m, mLen, sigDec, verLen,
                hType,
#ifndef WOLFSSL_PSS_SALT_LEN_DISCOVER
                RSA_PSS_SALT_LEN_DEFAULT,
#else
                RSA_PSS_SALT_LEN_DISCOVER,
#endif
                mp_count_bits(&((RsaKey*)rsa->internal)->n)) != 0) {
            WOLFSSL_MSG("wc_RsaPSS_CheckPadding_ex error");
            goto cleanup;
        }
    }
    else
#endif /* !defined(HAVE_FIPS) && !defined(HAVE_SELFTEST) */
    if ((int)len != verLen || XMEMCMP(sigRet, sigDec, verLen) != 0) {
        WOLFSSL_MSG("wolfSSL_RSA_verify_ex failed");
        goto cleanup;
    }

    WOLFSSL_MSG("wolfSSL_RSA_verify_ex success");
    ret = WOLFSSL_SUCCESS;
cleanup:
    if (sigRet)
        XFREE(sigRet, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    if (sigDec)
        XFREE(sigDec, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    return ret;
}

void wolfSSL_RSA_get0_crt_params(const WOLFSSL_RSA *r,
        const WOLFSSL_BIGNUM **dmp1, const WOLFSSL_BIGNUM **dmq1,
        const WOLFSSL_BIGNUM **iqmp)
{
    WOLFSSL_ENTER("wolfSSL_RSA_get0_crt_params");

    if (r != NULL) {
        if (dmp1 != NULL)
            *dmp1 = r->dmp1;
        if (dmq1 != NULL)
            *dmq1 = r->dmq1;
        if (iqmp != NULL)
            *iqmp = r->iqmp;
    } else {
        if (dmp1 != NULL)
            *dmp1 = NULL;
        if (dmq1 != NULL)
            *dmq1 = NULL;
        if (iqmp != NULL)
            *iqmp = NULL;
    }
}

int wolfSSL_RSA_set0_crt_params(WOLFSSL_RSA *r, WOLFSSL_BIGNUM *dmp1,
                                WOLFSSL_BIGNUM *dmq1, WOLFSSL_BIGNUM *iqmp)
{
    WOLFSSL_ENTER("wolfSSL_RSA_set0_crt_params");

    /* If a param is null in r then it must be non-null in the
     * corresponding user input. */
    if (r == NULL || (r->dmp1 == NULL && dmp1 == NULL) ||
            (r->dmq1 == NULL && dmq1 == NULL) ||
            (r->iqmp == NULL && iqmp == NULL)) {
        WOLFSSL_MSG("Bad parameters");
        return WOLFSSL_FAILURE;
    }

    if (dmp1 != NULL) {
        wolfSSL_BN_clear_free(r->dmp1);
        r->dmp1 = dmp1;
    }
    if (dmq1 != NULL) {
        wolfSSL_BN_clear_free(r->dmq1);
        r->dmq1 = dmq1;
    }
    if (iqmp != NULL) {
        wolfSSL_BN_clear_free(r->iqmp);
        r->iqmp = iqmp;
    }

    return SetRsaInternal(r) == WOLFSSL_SUCCESS ?
            WOLFSSL_SUCCESS : WOLFSSL_FAILURE;
}

void wolfSSL_RSA_get0_factors(const WOLFSSL_RSA *r, const WOLFSSL_BIGNUM **p,
                              const WOLFSSL_BIGNUM **q)
{
    WOLFSSL_ENTER("wolfSSL_RSA_get0_factors");

    if (r != NULL) {
        if (p != NULL)
            *p = r->p;
        if (q != NULL)
            *q = r->q;
    } else {
        if (p != NULL)
            *p = NULL;
        if (q != NULL)
            *q = NULL;
    }
}

int wolfSSL_RSA_set0_factors(WOLFSSL_RSA *r, WOLFSSL_BIGNUM *p, WOLFSSL_BIGNUM *q)
{
    WOLFSSL_ENTER("wolfSSL_RSA_set0_factors");

    /* If a param is null in r then it must be non-null in the
     * corresponding user input. */
    if (r == NULL || (r->p == NULL && p == NULL) ||
            (r->q == NULL && q == NULL)) {
        WOLFSSL_MSG("Bad parameters");
        return WOLFSSL_FAILURE;
    }

    if (p != NULL) {
        wolfSSL_BN_clear_free(r->p);
        r->p = p;
    }
    if (q != NULL) {
        wolfSSL_BN_clear_free(r->q);
        r->q = q;
    }

    return SetRsaInternal(r) == WOLFSSL_SUCCESS ?
            WOLFSSL_SUCCESS : WOLFSSL_FAILURE;
}

void wolfSSL_RSA_get0_key(const WOLFSSL_RSA *r, const WOLFSSL_BIGNUM **n,
    const WOLFSSL_BIGNUM **e, const WOLFSSL_BIGNUM **d)
{
    WOLFSSL_ENTER("wolfSSL_RSA_get0_key");

    if (r != NULL) {
        if (n != NULL)
            *n = r->n;
        if (e != NULL)
            *e = r->e;
        if (d != NULL)
            *d = r->d;
    } else {
        if (n != NULL)
            *n = NULL;
        if (e != NULL)
            *e = NULL;
        if (d != NULL)
            *d = NULL;
    }
}

/* generate p-1 and q-1, WOLFSSL_SUCCESS on ok */
int wolfSSL_RSA_GenAdd(WOLFSSL_RSA* rsa)
{
    int    err;
    mp_int tmp;

    WOLFSSL_MSG("wolfSSL_RsaGenAdd");

    if (rsa == NULL || rsa->p == NULL || rsa->q == NULL || rsa->d == NULL ||
                       rsa->dmp1 == NULL || rsa->dmq1 == NULL) {
        WOLFSSL_MSG("rsa no init error");
        return WOLFSSL_FATAL_ERROR;
    }

    if (mp_init(&tmp) != MP_OKAY) {
        WOLFSSL_MSG("mp_init error");
        return WOLFSSL_FATAL_ERROR;
    }

    err = mp_sub_d((mp_int*)rsa->p->internal, 1, &tmp);
    if (err != MP_OKAY) {
        WOLFSSL_MSG("mp_sub_d error");
    }
    else
        err = mp_mod((mp_int*)rsa->d->internal, &tmp,
                     (mp_int*)rsa->dmp1->internal);

    if (err != MP_OKAY) {
        WOLFSSL_MSG("mp_mod error");
    }
    else
        err = mp_sub_d((mp_int*)rsa->q->internal, 1, &tmp);
    if (err != MP_OKAY) {
        WOLFSSL_MSG("mp_sub_d error");
    }
    else
        err = mp_mod((mp_int*)rsa->d->internal, &tmp,
                     (mp_int*)rsa->dmq1->internal);

    mp_clear(&tmp);

    if (err == MP_OKAY)
        return WOLFSSL_SUCCESS;
    else
        return WOLFSSL_FATAL_ERROR;
}
#endif /* !NO_RSA && !HAVE_USER_RSA */

WOLFSSL_HMAC_CTX* wolfSSL_HMAC_CTX_new(void)
{
    WOLFSSL_HMAC_CTX* hmac_ctx = (WOLFSSL_HMAC_CTX*)XMALLOC(
        sizeof(WOLFSSL_HMAC_CTX), NULL, DYNAMIC_TYPE_OPENSSL);
    if (hmac_ctx != NULL) {
        XMEMSET(hmac_ctx, 0, sizeof(WOLFSSL_HMAC_CTX));
    }
    return hmac_ctx;
}

int wolfSSL_HMAC_CTX_Init(WOLFSSL_HMAC_CTX* ctx)
{
    WOLFSSL_MSG("wolfSSL_HMAC_CTX_Init");

    if (ctx != NULL) {
        /* wc_HmacSetKey sets up ctx->hmac */
        XMEMSET(ctx, 0, sizeof(WOLFSSL_HMAC_CTX));
    }

    return WOLFSSL_SUCCESS;
}


int wolfSSL_HMAC_Init_ex(WOLFSSL_HMAC_CTX* ctx, const void* key,
                             int keylen, const EVP_MD* type, WOLFSSL_ENGINE* e)
{
    WOLFSSL_ENTER("wolfSSL_HMAC_Init_ex");

    /* WOLFSSL_ENGINE not used, call wolfSSL_HMAC_Init */
    (void)e;
    return wolfSSL_HMAC_Init(ctx, key, keylen, type);
}


/* helper function for Deep copy of internal wolfSSL hmac structure
 * returns WOLFSSL_SUCCESS on success */
int wolfSSL_HmacCopy(Hmac* des, Hmac* src)
{
    void* heap;
    int ret;

#ifndef HAVE_FIPS
    heap = src->heap;
#else
    heap = NULL;
#endif
    if (wc_HmacInit(des, heap, 0) != 0) {
        return WOLFSSL_FAILURE;
    }

    /* requires that hash structures have no dynamic parts to them */
    switch (src->macType) {
    #ifndef NO_MD5
        case WC_MD5:
            ret = wc_Md5Copy(&src->hash.md5, &des->hash.md5);
            break;
    #endif /* !NO_MD5 */

    #ifndef NO_SHA
        case WC_SHA:
            ret = wc_ShaCopy(&src->hash.sha, &des->hash.sha);
            break;
    #endif /* !NO_SHA */

    #ifdef WOLFSSL_SHA224
        case WC_SHA224:
            ret = wc_Sha224Copy(&src->hash.sha224, &des->hash.sha224);
            break;
    #endif /* WOLFSSL_SHA224 */

    #ifndef NO_SHA256
        case WC_SHA256:
            ret = wc_Sha256Copy(&src->hash.sha256, &des->hash.sha256);
            break;
    #endif /* !NO_SHA256 */

    #ifdef WOLFSSL_SHA384
        case WC_SHA384:
            ret = wc_Sha384Copy(&src->hash.sha384, &des->hash.sha384);
            break;
    #endif /* WOLFSSL_SHA384 */
    #ifdef WOLFSSL_SHA512
        case WC_SHA512:
            ret = wc_Sha512Copy(&src->hash.sha512, &des->hash.sha512);
            break;
    #endif /* WOLFSSL_SHA512 */
#ifdef WOLFSSL_SHA3
    #ifndef WOLFSSL_NOSHA3_224
        case WC_SHA3_224:
            ret = wc_Sha3_224_Copy(&src->hash.sha3, &des->hash.sha3);
            break;
    #endif /* WOLFSSL_NO_SHA3_224 */
    #ifndef WOLFSSL_NOSHA3_256
        case WC_SHA3_256:
            ret = wc_Sha3_256_Copy(&src->hash.sha3, &des->hash.sha3);
            break;
    #endif /* WOLFSSL_NO_SHA3_256 */
    #ifndef WOLFSSL_NOSHA3_384
        case WC_SHA3_384:
            ret = wc_Sha3_384_Copy(&src->hash.sha3, &des->hash.sha3);
            break;
    #endif /* WOLFSSL_NO_SHA3_384 */
    #ifndef WOLFSSL_NOSHA3_512
        case WC_SHA3_512:
            ret = wc_Sha3_512_Copy(&src->hash.sha3, &des->hash.sha3);
            break;
    #endif /* WOLFSSL_NO_SHA3_512 */
#endif /* WOLFSSL_SHA3 */

        default:
            return WOLFSSL_FAILURE;
    }

    if (ret != 0)
        return WOLFSSL_FAILURE;

    XMEMCPY((byte*)des->ipad, (byte*)src->ipad, WC_HMAC_BLOCK_SIZE);
    XMEMCPY((byte*)des->opad, (byte*)src->opad, WC_HMAC_BLOCK_SIZE);
    XMEMCPY((byte*)des->innerHash, (byte*)src->innerHash, WC_MAX_DIGEST_SIZE);
#ifndef HAVE_FIPS
    des->heap    = heap;
#endif
    des->macType = src->macType;
    des->innerHashKeyed = src->innerHashKeyed;

#ifdef WOLFSSL_ASYNC_CRYPT
    XMEMCPY(&des->asyncDev, &src->asyncDev, sizeof(WC_ASYNC_DEV));
    des->keyLen = src->keyLen;
    #ifdef HAVE_CAVIUM
        des->data = (byte*)XMALLOC(src->dataLen, des->heap,
                DYNAMIC_TYPE_HMAC);
        if (des->data == NULL) {
            return BUFFER_E;
        }
        XMEMCPY(des->data, src->data, src->dataLen);
        des->dataLen = src->dataLen;
    #endif /* HAVE_CAVIUM */
#endif /* WOLFSSL_ASYNC_CRYPT */
        return WOLFSSL_SUCCESS;
}


/* Deep copy of information from src to des structure
 *
 * des destination to copy information to
 * src structure to get information from
 *
 * Returns WOLFSSL_SUCCESS on success and WOLFSSL_FAILURE on error
 */
int wolfSSL_HMAC_CTX_copy(WOLFSSL_HMAC_CTX* des, WOLFSSL_HMAC_CTX* src)
{
    WOLFSSL_ENTER("wolfSSL_HMAC_CTX_copy");

    if (des == NULL || src == NULL) {
        return WOLFSSL_FAILURE;
    }

    des->type = src->type;
    XMEMCPY((byte *)&des->save_ipad, (byte *)&src->hmac.ipad,
                                        WC_HMAC_BLOCK_SIZE);
    XMEMCPY((byte *)&des->save_opad, (byte *)&src->hmac.opad,
                                        WC_HMAC_BLOCK_SIZE);

    return wolfSSL_HmacCopy(&des->hmac, &src->hmac);
}


#if defined(HAVE_FIPS) && \
    (!defined(HAVE_FIPS_VERSION) || (HAVE_FIPS_VERSION < 2))

static int _HMAC_Init(Hmac* hmac, int type, void* heap)
{
    int ret = 0;

    switch (type) {
    #ifndef NO_MD5
        case WC_MD5:
            ret = wc_InitMd5(&hmac->hash.md5);
            break;
    #endif /* !NO_MD5 */

    #ifndef NO_SHA
        case WC_SHA:
            ret = wc_InitSha(&hmac->hash.sha);
            break;
    #endif /* !NO_SHA */

    #ifdef WOLFSSL_SHA224
        case WC_SHA224:
            ret = wc_InitSha224(&hmac->hash.sha224);
            break;
    #endif /* WOLFSSL_SHA224 */

    #ifndef NO_SHA256
        case WC_SHA256:
            ret = wc_InitSha256(&hmac->hash.sha256);
            break;
    #endif /* !NO_SHA256 */

    #ifdef WOLFSSL_SHA384
        case WC_SHA384:
            ret = wc_InitSha384(&hmac->hash.sha384);
            break;
    #endif /* WOLFSSL_SHA384 */
    #ifdef WOLFSSL_SHA512
        case WC_SHA512:
            ret = wc_InitSha512(&hmac->hash.sha512);
            break;
    #endif /* WOLFSSL_SHA512 */

    #ifdef WOLFSSL_SHA3
        case WC_SHA3_224:
            ret = wc_InitSha3_224(&hmac->hash.sha3, heap, INVALID_DEVID);
            break;
        case WC_SHA3_256:
            ret = wc_InitSha3_256(&hmac->hash.sha3, heap, INVALID_DEVID);
            break;
        case WC_SHA3_384:
            ret = wc_InitSha3_384(&hmac->hash.sha3, heap, INVALID_DEVID);
            break;
        case WC_SHA3_512:
            ret = wc_InitSha3_512(&hmac->hash.sha3, heap, INVALID_DEVID);
            break;
    #endif

        default:
            ret = BAD_FUNC_ARG;
            break;
    }

    (void)heap;

    return ret;
}

#else
    #define _HMAC_Init _InitHmac
#endif


int wolfSSL_HMAC_Init(WOLFSSL_HMAC_CTX* ctx, const void* key, int keylen,
                  const EVP_MD* type)
{
    int hmac_error = 0;
    void* heap = NULL;
    int inited;

    WOLFSSL_MSG("wolfSSL_HMAC_Init");

    if (ctx == NULL) {
        WOLFSSL_MSG("no ctx on init");
        return WOLFSSL_FAILURE;
    }

#ifndef HAVE_FIPS
    heap = ctx->hmac.heap;
#endif

    if (type) {
        WOLFSSL_MSG("init has type");

#ifndef NO_MD5
        if (XSTRNCMP(type, "MD5", 3) == 0) {
            WOLFSSL_MSG("md5 hmac");
            ctx->type = WC_MD5;
        }
        else
#endif
#ifdef WOLFSSL_SHA224
        if (XSTRNCMP(type, "SHA224", 6) == 0) {
            WOLFSSL_MSG("sha224 hmac");
            ctx->type = WC_SHA224;
        }
        else
#endif
#ifndef NO_SHA256
        if (XSTRNCMP(type, "SHA256", 6) == 0) {
            WOLFSSL_MSG("sha256 hmac");
            ctx->type = WC_SHA256;
        }
        else
#endif
#ifdef WOLFSSL_SHA384
        if (XSTRNCMP(type, "SHA384", 6) == 0) {
            WOLFSSL_MSG("sha384 hmac");
            ctx->type = WC_SHA384;
        }
        else
#endif
#ifdef WOLFSSL_SHA512
        if (XSTRNCMP(type, "SHA512", 6) == 0) {
            WOLFSSL_MSG("sha512 hmac");
            ctx->type = WC_SHA512;
        }
        else
#endif
#ifdef WOLFSSL_SHA3
    #ifndef WOLFSSL_NOSHA3_224
        if (XSTRNCMP(type, "SHA3_224", 8) == 0) {
            WOLFSSL_MSG("sha3_224 hmac");
            ctx->type = WC_SHA3_224;
        }
        else
    #endif
    #ifndef WOLFSSL_NOSHA3_256
        if (XSTRNCMP(type, "SHA3_256", 8) == 0) {
            WOLFSSL_MSG("sha3_256 hmac");
            ctx->type = WC_SHA3_256;
        }
        else
    #endif
        if (XSTRNCMP(type, "SHA3_384", 8) == 0) {
            WOLFSSL_MSG("sha3_384 hmac");
            ctx->type = WC_SHA3_384;
        }
        else
    #ifndef WOLFSSL_NOSHA3_512
        if (XSTRNCMP(type, "SHA3_512", 8) == 0) {
            WOLFSSL_MSG("sha3_512 hmac");
            ctx->type = WC_SHA3_512;
        }
        else
    #endif
#endif

#ifndef NO_SHA
        /* has to be last since would pick or 256, 384, or 512 too */
        if (XSTRNCMP(type, "SHA", 3) == 0) {
            WOLFSSL_MSG("sha hmac");
            ctx->type = WC_SHA;
        }
        else
#endif
        {
            WOLFSSL_MSG("bad init type");
            return WOLFSSL_FAILURE;
        }
    }

    /* Check if init has been called before */
    inited = (ctx->hmac.macType != WC_HASH_TYPE_NONE);
    /* Free if needed */
    if (inited) {
        wc_HmacFree(&ctx->hmac);
    }
    if (key != NULL) {
        WOLFSSL_MSG("keying hmac");

        if (wc_HmacInit(&ctx->hmac, NULL, INVALID_DEVID) == 0) {
            hmac_error = wc_HmacSetKey(&ctx->hmac, ctx->type, (const byte*)key,
                                       (word32)keylen);
            if (hmac_error < 0){
                /* in FIPS mode a key < 14 characters will fail here */
                WOLFSSL_MSG("hmac set key error");
                WOLFSSL_ERROR(hmac_error);
                wc_HmacFree(&ctx->hmac);
                return WOLFSSL_FAILURE;
            }
            XMEMCPY((byte *)&ctx->save_ipad, (byte *)&ctx->hmac.ipad,
                                        WC_HMAC_BLOCK_SIZE);
            XMEMCPY((byte *)&ctx->save_opad, (byte *)&ctx->hmac.opad,
                                        WC_HMAC_BLOCK_SIZE);
        }
        /* OpenSSL compat, no error */
    }
    else if (!inited) {
        return WOLFSSL_FAILURE;
    }
    else if (ctx->type >= 0) { /* MD5 == 0 */
        WOLFSSL_MSG("recover hmac");
        if (wc_HmacInit(&ctx->hmac, NULL, INVALID_DEVID) == 0) {
            ctx->hmac.macType = (byte)ctx->type;
            ctx->hmac.innerHashKeyed = 0;
            XMEMCPY((byte *)&ctx->hmac.ipad, (byte *)&ctx->save_ipad,
                                       WC_HMAC_BLOCK_SIZE);
            XMEMCPY((byte *)&ctx->hmac.opad, (byte *)&ctx->save_opad,
                                       WC_HMAC_BLOCK_SIZE);
            if ((hmac_error = _HMAC_Init(&ctx->hmac, ctx->hmac.macType, heap))
                    !=0) {
                WOLFSSL_MSG("hmac init error");
                WOLFSSL_ERROR(hmac_error);
                return WOLFSSL_FAILURE;
            }
        }
    }

    (void)hmac_error;

    return WOLFSSL_SUCCESS;
}


int wolfSSL_HMAC_Update(WOLFSSL_HMAC_CTX* ctx, const unsigned char* data,
                    int len)
{
    int hmac_error = 0;

    WOLFSSL_MSG("wolfSSL_HMAC_Update");

    if (ctx == NULL) {
        WOLFSSL_MSG("no ctx");
        return WOLFSSL_FAILURE;
    }

    if (data) {
        WOLFSSL_MSG("updating hmac");
        hmac_error = wc_HmacUpdate(&ctx->hmac, data, (word32)len);
        if (hmac_error < 0){
            WOLFSSL_MSG("hmac update error");
            return WOLFSSL_FAILURE;
        }
    }

    return WOLFSSL_SUCCESS;
}


int wolfSSL_HMAC_Final(WOLFSSL_HMAC_CTX* ctx, unsigned char* hash,
                   unsigned int* len)
{
    int hmac_error;

    WOLFSSL_MSG("wolfSSL_HMAC_Final");

    /* "len" parameter is optional. */
    if (ctx == NULL || hash == NULL) {
        WOLFSSL_MSG("invalid parameter");
        return WOLFSSL_FAILURE;
    }

    WOLFSSL_MSG("final hmac");
    hmac_error = wc_HmacFinal(&ctx->hmac, hash);
    if (hmac_error < 0){
        WOLFSSL_MSG("final hmac error");
        return WOLFSSL_FAILURE;
    }

    if (len) {
        WOLFSSL_MSG("setting output len");
        switch (ctx->type) {
            #ifndef NO_MD5
            case WC_MD5:
                *len = WC_MD5_DIGEST_SIZE;
                break;
            #endif

            #ifndef NO_SHA
            case WC_SHA:
                *len = WC_SHA_DIGEST_SIZE;
                break;
            #endif

            #ifdef WOLFSSL_SHA224
            case WC_SHA224:
                *len = WC_SHA224_DIGEST_SIZE;
                break;
            #endif

            #ifndef NO_SHA256
            case WC_SHA256:
                *len = WC_SHA256_DIGEST_SIZE;
                break;
            #endif

            #ifdef WOLFSSL_SHA384
            case WC_SHA384:
                *len = WC_SHA384_DIGEST_SIZE;
                break;
            #endif

            #ifdef WOLFSSL_SHA512
            case WC_SHA512:
                *len = WC_SHA512_DIGEST_SIZE;
                break;
            #endif

        #ifdef WOLFSSL_SHA3
            #ifndef WOLFSSL_NOSHA3_224
            case WC_SHA3_224:
                *len = WC_SHA3_224_DIGEST_SIZE;
                break;
            #endif
            #ifndef WOLFSSL_NOSHA3_256
            case WC_SHA3_256:
                *len = WC_SHA3_256_DIGEST_SIZE;
                break;
            #endif
            #ifndef WOLFSSL_NOSHA3_384
            case WC_SHA3_384:
                *len = WC_SHA3_384_DIGEST_SIZE;
                break;
            #endif
            #ifndef WOLFSSL_NOSHA3_512
            case WC_SHA3_512:
                *len = WC_SHA3_512_DIGEST_SIZE;
                break;
            #endif
        #endif

            default:
                WOLFSSL_MSG("bad hmac type");
                return WOLFSSL_FAILURE;
        }
    }

    return WOLFSSL_SUCCESS;
}


int wolfSSL_HMAC_cleanup(WOLFSSL_HMAC_CTX* ctx)
{
    WOLFSSL_MSG("wolfSSL_HMAC_cleanup");

    if (ctx) {
        wc_HmacFree(&ctx->hmac);
    }

    return WOLFSSL_SUCCESS;
}

void wolfSSL_HMAC_CTX_cleanup(WOLFSSL_HMAC_CTX* ctx)
{
    if (ctx) {
        wolfSSL_HMAC_cleanup(ctx);
    }
}

void wolfSSL_HMAC_CTX_free(WOLFSSL_HMAC_CTX* ctx)
{
    if (ctx) {
        wolfSSL_HMAC_CTX_cleanup(ctx);
        XFREE(ctx, NULL, DYNAMIC_TYPE_OPENSSL);
    }
}

size_t wolfSSL_HMAC_size(const WOLFSSL_HMAC_CTX *ctx)
{
    if (!ctx) {
        return 0;
    }

    return (size_t)wc_HashGetDigestSize((enum wc_HashType)ctx->hmac.macType);
}

const WOLFSSL_EVP_MD *wolfSSL_HMAC_CTX_get_md(const WOLFSSL_HMAC_CTX *ctx)
{
    if (!ctx) {
        return NULL;
    }

    return wolfSSL_macType2EVP_md((enum wc_HashType)ctx->type);
}

#if defined(WOLFSSL_CMAC) && defined(OPENSSL_EXTRA) && \
    defined(WOLFSSL_AES_DIRECT)
WOLFSSL_CMAC_CTX* wolfSSL_CMAC_CTX_new(void)
{
    WOLFSSL_CMAC_CTX* ctx = NULL;

    ctx = (WOLFSSL_CMAC_CTX*)XMALLOC(sizeof(WOLFSSL_CMAC_CTX), NULL,
                                     DYNAMIC_TYPE_OPENSSL);
    if (ctx != NULL) {
        ctx->internal = (Cmac*)XMALLOC(sizeof(Cmac), NULL, DYNAMIC_TYPE_CMAC);
        if (ctx->internal == NULL) {
            XFREE(ctx, NULL, DYNAMIC_TYPE_OPENSSL);
            ctx = NULL;
        }
    }
    if (ctx != NULL) {
        ctx->cctx = wolfSSL_EVP_CIPHER_CTX_new();
        if (ctx->cctx == NULL) {
            XFREE(ctx->internal, NULL, DYNAMIC_TYPE_CMAC);
            XFREE(ctx, NULL, DYNAMIC_TYPE_OPENSSL);
            ctx = NULL;
        }
    }

    return ctx;
}

void wolfSSL_CMAC_CTX_free(WOLFSSL_CMAC_CTX *ctx)
{
    if (ctx != NULL) {
        if (ctx->internal != NULL) {
            XFREE(ctx->internal, NULL, DYNAMIC_TYPE_CMAC);
        }
        if (ctx->cctx != NULL) {
            wolfSSL_EVP_CIPHER_CTX_free(ctx->cctx);
        }
        XFREE(ctx, NULL, DYNAMIC_TYPE_OPENSSL);
    }
}

WOLFSSL_EVP_CIPHER_CTX* wolfSSL_CMAC_CTX_get0_cipher_ctx(WOLFSSL_CMAC_CTX* ctx)
{
    WOLFSSL_EVP_CIPHER_CTX* cctx = NULL;

    if (ctx != NULL) {
        cctx = ctx->cctx;
    }

    return cctx;
}

int wolfSSL_CMAC_Init(WOLFSSL_CMAC_CTX* ctx, const void *key, size_t keyLen,
                      const WOLFSSL_EVP_CIPHER* cipher, WOLFSSL_ENGINE* engine)
{
    int ret = WOLFSSL_SUCCESS;

    (void)engine;

    WOLFSSL_ENTER("wolfSSL_CMAC_Init");

    if (ctx == NULL || cipher == NULL || (
            cipher != EVP_AES_128_CBC &&
            cipher != EVP_AES_192_CBC &&
            cipher != EVP_AES_256_CBC)) {
        ret = WOLFSSL_FAILURE;
    }

    if (ret == WOLFSSL_SUCCESS) {
        ret = wc_InitCmac((Cmac*)ctx->internal, (const byte*)key,
                          (word32)keyLen, WC_CMAC_AES, NULL);
        if (ret != 0) {
            ret = WOLFSSL_FAILURE;
        }
        else {
            ret = WOLFSSL_SUCCESS;
        }
    }
    if (ret == WOLFSSL_SUCCESS) {
        ret = wolfSSL_EVP_CipherInit(ctx->cctx, cipher, (const byte*)key, NULL,
                                     1);
    }

    WOLFSSL_LEAVE("wolfSSL_CMAC_Init", ret);

    return ret;
}

int wolfSSL_CMAC_Update(WOLFSSL_CMAC_CTX* ctx, const void* data, size_t len)
{
    int ret = WOLFSSL_SUCCESS;

    WOLFSSL_ENTER("wolfSSL_CMAC_Update");

    if (ctx == NULL || ctx->internal == NULL) {
        ret = WOLFSSL_FAILURE;
    }

    if (ret == WOLFSSL_SUCCESS) {
        if (data) {
            ret = wc_CmacUpdate((Cmac*)ctx->internal, (const byte*)data,
                                (word32)len);
            if (ret != 0){
                ret = WOLFSSL_FAILURE;
            }
            else {
                ret = WOLFSSL_SUCCESS;
            }
        }
    }

    WOLFSSL_LEAVE("wolfSSL_CMAC_Update", ret);

    return ret;
}

int wolfSSL_CMAC_Final(WOLFSSL_CMAC_CTX* ctx, unsigned char* out,
                       size_t* len)
{
    int ret = WOLFSSL_SUCCESS;
    int blockSize;

    WOLFSSL_ENTER("wolfSSL_CMAC_Final");

    if (ctx == NULL || ctx->cctx == NULL || ctx->internal == NULL ||
                                                                  len == NULL) {
        ret = WOLFSSL_FAILURE;
    }

    if (ret == WOLFSSL_SUCCESS) {
        blockSize = EVP_CIPHER_CTX_block_size(ctx->cctx);
        if (blockSize <= 0) {
            ret = WOLFSSL_FAILURE;
        }
        else {
            *len = blockSize;
        }
    }
    if (ret == WOLFSSL_SUCCESS) {
        word32 len32 = (word32)*len;

        ret = wc_CmacFinal((Cmac*)ctx->internal, out, &len32);
        *len = (size_t)len32;
        if (ret != 0) {
            ret = WOLFSSL_FAILURE;
        }
        else {
            ret = WOLFSSL_SUCCESS;
        }
    }

    WOLFSSL_LEAVE("wolfSSL_CMAC_Final", ret);

    return ret;
}
#endif /* WOLFSSL_CMAC && OPENSSL_EXTRA && WOLFSSL_AES_DIRECT */
#endif /* OPENSSL_EXTRA */

#if defined(OPENSSL_EXTRA) || defined(OPENSSL_EXTRA_X509_SMALL)
/* Free the dynamically allocated data.
 *
 * p  Pointer to dynamically allocated memory.
 */
void wolfSSL_OPENSSL_free(void* p)
{
    WOLFSSL_MSG("wolfSSL_OPENSSL_free");

    XFREE(p, NULL, DYNAMIC_TYPE_OPENSSL);
}
#endif /* OPENSSL_EXTRA || OPENSSL_EXTRA_X509_SMALL */

#ifdef OPENSSL_EXTRA

void *wolfSSL_OPENSSL_malloc(size_t a)
{
    return (void *)XMALLOC(a, NULL, DYNAMIC_TYPE_OPENSSL);
}

int wolfSSL_OPENSSL_hexchar2int(unsigned char c)
{
    /* 'char' is unsigned on some platforms. */
    return (int)(signed char)HexCharToByte((char)c);
}

unsigned char *wolfSSL_OPENSSL_hexstr2buf(const char *str, long *len)
{
    unsigned char* targetBuf;
    int srcDigitHigh = 0;
    int srcDigitLow = 0;
    size_t srcLen;
    size_t srcIdx = 0;
    long targetIdx = 0;

    srcLen = XSTRLEN(str);
    targetBuf = (unsigned char*)XMALLOC(srcLen / 2, NULL, DYNAMIC_TYPE_OPENSSL);
    if (targetBuf == NULL) {
        return NULL;
    }

    while (srcIdx < srcLen) {
        if (str[srcIdx] == ':') {
            srcIdx++;
            continue;
        }

        srcDigitHigh = wolfSSL_OPENSSL_hexchar2int(str[srcIdx++]);
        srcDigitLow = wolfSSL_OPENSSL_hexchar2int(str[srcIdx++]);
        if (srcDigitHigh < 0 || srcDigitLow < 0) {
            WOLFSSL_MSG("Invalid hex character.");
            XFREE(targetBuf, NULL, DYNAMIC_TYPE_OPENSSL);
            return NULL;
        }

        targetBuf[targetIdx++] = (unsigned char)((srcDigitHigh << 4) | srcDigitLow);
    }

    if (len != NULL)
        *len = targetIdx;

    return targetBuf;
}

int wolfSSL_OPENSSL_init_ssl(word64 opts, const OPENSSL_INIT_SETTINGS *settings)
{
    (void)opts;
    (void)settings;
    return wolfSSL_library_init();
}

int wolfSSL_OPENSSL_init_crypto(word64 opts, const OPENSSL_INIT_SETTINGS* settings)
{
    (void)opts;
    (void)settings;
    return wolfSSL_library_init();
}

#if defined(WOLFSSL_KEY_GEN) && defined(WOLFSSL_PEM_TO_DER)

static int EncryptDerKey(byte *der, int *derSz, const EVP_CIPHER* cipher,
                         unsigned char* passwd, int passwdSz, byte **cipherInfo,
                         int maxDerSz)
{
    int ret, paddingSz;
    word32 idx, cipherInfoSz;
#ifdef WOLFSSL_SMALL_STACK
    EncryptedInfo* info = NULL;
#else
    EncryptedInfo  info[1];
#endif

    WOLFSSL_ENTER("EncryptDerKey");

    if (der == NULL || derSz == NULL || cipher == NULL ||
        passwd == NULL || cipherInfo == NULL)
        return BAD_FUNC_ARG;

#ifdef WOLFSSL_SMALL_STACK
    info = (EncryptedInfo*)XMALLOC(sizeof(EncryptedInfo), NULL,
                                   DYNAMIC_TYPE_ENCRYPTEDINFO);
    if (info == NULL) {
        WOLFSSL_MSG("malloc failed");
        return WOLFSSL_FAILURE;
    }
#endif

    XMEMSET(info, 0, sizeof(EncryptedInfo));

    /* set the cipher name on info */
    XSTRNCPY(info->name, cipher, NAME_SZ-1);
    info->name[NAME_SZ-1] = '\0'; /* null term */

    ret = wc_EncryptedInfoGet(info, info->name);
    if (ret != 0) {
        WOLFSSL_MSG("unsupported cipher");
    #ifdef WOLFSSL_SMALL_STACK
        XFREE(info, NULL, DYNAMIC_TYPE_ENCRYPTEDINFO);
    #endif
        return WOLFSSL_FAILURE;
    }

    /* Generate a random salt */
    if (wolfSSL_RAND_bytes(info->iv, info->ivSz) != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("generate iv failed");
#ifdef WOLFSSL_SMALL_STACK
        XFREE(info, NULL, DYNAMIC_TYPE_ENCRYPTEDINFO);
#endif
        return WOLFSSL_FAILURE;
    }

    /* add the padding before encryption */
    paddingSz = ((*derSz)/info->ivSz + 1) * info->ivSz - (*derSz);
    if (paddingSz == 0)
        paddingSz = info->ivSz;
    if (maxDerSz < *derSz + paddingSz) {
        WOLFSSL_MSG("not enough DER buffer allocated");
#ifdef WOLFSSL_SMALL_STACK
        XFREE(info, NULL, DYNAMIC_TYPE_ENCRYPTEDINFO);
#endif
        return WOLFSSL_FAILURE;
    }
    XMEMSET(der+(*derSz), (byte)paddingSz, paddingSz);
    (*derSz) += paddingSz;

    /* encrypt buffer */
    if (wc_BufferKeyEncrypt(info, der, *derSz, passwd, passwdSz, WC_MD5) != 0) {
        WOLFSSL_MSG("encrypt key failed");
#ifdef WOLFSSL_SMALL_STACK
        XFREE(info, NULL, DYNAMIC_TYPE_ENCRYPTEDINFO);
#endif
        return WOLFSSL_FAILURE;
    }

    /* create cipher info : 'cipher_name,Salt(hex)' */
    cipherInfoSz = (word32)(2*info->ivSz + XSTRLEN(info->name) + 2);
    *cipherInfo = (byte*)XMALLOC(cipherInfoSz, NULL,
                                DYNAMIC_TYPE_STRING);
    if (*cipherInfo == NULL) {
        WOLFSSL_MSG("malloc failed");
#ifdef WOLFSSL_SMALL_STACK
        XFREE(info, NULL, DYNAMIC_TYPE_ENCRYPTEDINFO);
#endif
        return WOLFSSL_FAILURE;
    }
    XSTRNCPY((char*)*cipherInfo, info->name, cipherInfoSz);
    XSTRNCAT((char*)*cipherInfo, ",", 2);

    idx = (word32)XSTRLEN((char*)*cipherInfo);
    cipherInfoSz -= idx;
    ret = Base16_Encode(info->iv, info->ivSz, *cipherInfo+idx, &cipherInfoSz);

#ifdef WOLFSSL_SMALL_STACK
    XFREE(info, NULL, DYNAMIC_TYPE_ENCRYPTEDINFO);
#endif
    if (ret != 0) {
        WOLFSSL_MSG("Base16_Encode failed");
        XFREE(*cipherInfo, NULL, DYNAMIC_TYPE_STRING);
        return WOLFSSL_FAILURE;
    }

    return WOLFSSL_SUCCESS;
}
#endif /* WOLFSSL_KEY_GEN || WOLFSSL_PEM_TO_DER */

#if defined(WOLFSSL_KEY_GEN) && !defined(NO_RSA) && !defined(HAVE_USER_RSA)
static int wolfSSL_RSA_To_Der(WOLFSSL_RSA* rsa, byte** outBuf, int publicKey, void* heap)
{
    int derSz  = 0;
    int ret;
    byte* derBuf;

    WOLFSSL_ENTER("wolfSSL_RSA_To_Der");

    if (!rsa || (publicKey != 0 && publicKey != 1)) {
        WOLFSSL_LEAVE("wolfSSL_RSA_To_Der", BAD_FUNC_ARG);
        return BAD_FUNC_ARG;
    }

    if (rsa->inSet == 0) {
        if ((ret = SetRsaInternal(rsa)) != WOLFSSL_SUCCESS) {
            WOLFSSL_MSG("SetRsaInternal() Failed");
            WOLFSSL_LEAVE("wolfSSL_RSA_To_Der", ret);
            return ret;
        }
    }

    if (publicKey) {
        if ((derSz = wc_RsaPublicKeyDerSize((RsaKey *)rsa->internal, 1)) < 0) {
            WOLFSSL_MSG("wc_RsaPublicKeyDerSize failed");
            WOLFSSL_LEAVE("wolfSSL_RSA_To_Der", derSz);
            return derSz;
        }
    }
    else {
        if ((derSz = wc_RsaKeyToDer((RsaKey*)rsa->internal, NULL, 0)) < 0) {
            WOLFSSL_MSG("wc_RsaKeyToDer failed");
            WOLFSSL_LEAVE("wolfSSL_RSA_To_Der", derSz);
            return derSz;
        }
    }

    if (outBuf) {
        if (!(derBuf = (byte*)XMALLOC(derSz, heap, DYNAMIC_TYPE_TMP_BUFFER))) {
            WOLFSSL_MSG("malloc failed");
            WOLFSSL_LEAVE("wolfSSL_RSA_To_Der", MEMORY_ERROR);
            return MEMORY_ERROR;
        }

        /* Key to DER */
        if (publicKey) {
            derSz = wc_RsaKeyToPublicDer((RsaKey*)rsa->internal, derBuf, derSz);
        }
        else {
            derSz = wc_RsaKeyToDer((RsaKey*)rsa->internal, derBuf, derSz);
        }

        if (derSz < 0) {
            WOLFSSL_MSG("wc_RsaKeyToPublicDer failed");
            XFREE(derBuf, heap, DYNAMIC_TYPE_TMP_BUFFER);
        }
        else {
            if (*outBuf) {
                XMEMCPY(*outBuf, derBuf, derSz);
                XFREE(derBuf, heap, DYNAMIC_TYPE_TMP_BUFFER);
            }
            else {
                *outBuf = derBuf;
            }
        }
    }

    (void)heap; /* unused if memory is disabled */
    WOLFSSL_LEAVE("wolfSSL_RSA_To_Der", derSz);
    return derSz;
}
#endif

#ifndef NO_BIO
#if (defined(OPENSSL_EXTRA) || defined(OPENSSL_ALL)) && !defined(NO_RSA)
/* Takes a WOLFSSL_RSA key and writes it out to a WOLFSSL_BIO
 *
 * bio    the WOLFSSL_BIO to write to
 * key    the WOLFSSL_RSA key to write out
 * cipher cipher used
 * passwd password string if used
 * len    length of password string
 * cb     password callback to use
 * arg    null terminated string for passphrase
 */
int wolfSSL_PEM_write_bio_RSAPrivateKey(WOLFSSL_BIO* bio, WOLFSSL_RSA* key,
                                        const WOLFSSL_EVP_CIPHER* cipher,
                                        unsigned char* passwd, int len,
                                        wc_pem_password_cb* cb, void* arg)
{
    int ret;
    WOLFSSL_EVP_PKEY* pkey;

    WOLFSSL_ENTER("wolfSSL_PEM_write_bio_RSAPrivateKey");

    if (bio == NULL || key == NULL) {
        WOLFSSL_MSG("Bad Function Arguments");
        return WOLFSSL_FAILURE;
    }

    pkey = wolfSSL_EVP_PKEY_new_ex(bio->heap);
    if (pkey == NULL) {
        WOLFSSL_MSG("wolfSSL_EVP_PKEY_new_ex failed");
        return WOLFSSL_FAILURE;
    }

    pkey->type   = EVP_PKEY_RSA;
    pkey->rsa    = key;
    pkey->ownRsa = 0;
#if defined(WOLFSSL_KEY_GEN) && !defined(NO_RSA) && !defined(HAVE_USER_RSA)
    /* similar to how wolfSSL_PEM_write_mem_RSAPrivateKey finds DER of key */
    {
        int derSz;
        byte* derBuf = NULL;

        if ((derSz = wolfSSL_RSA_To_Der(key, &derBuf, 0, bio->heap)) < 0) {
            WOLFSSL_MSG("wolfSSL_RSA_To_Der failed");
            return WOLFSSL_FAILURE;
        }

        if (derBuf == NULL) {
            WOLFSSL_MSG("wolfSSL_RSA_To_Der failed to get buffer");
            return WOLFSSL_FAILURE;
        }

        pkey->pkey.ptr = (char*)XMALLOC(derSz, bio->heap,
                DYNAMIC_TYPE_TMP_BUFFER);
        if (pkey->pkey.ptr == NULL) {
            WOLFSSL_MSG("key malloc failed");
            XFREE(derBuf, bio->heap, DYNAMIC_TYPE_TMP_BUFFER);
            wolfSSL_EVP_PKEY_free(pkey);
            return WOLFSSL_FAILURE;
        }
        pkey->pkey_sz = derSz;
        XMEMCPY(pkey->pkey.ptr, derBuf, derSz);
        XFREE(derBuf, bio->heap, DYNAMIC_TYPE_TMP_BUFFER);
    }
#endif /* WOLFSSL_KEY_GEN && !NO_RSA && !HAVE_USER_RSA */

    ret = wolfSSL_PEM_write_bio_PrivateKey(bio, pkey, cipher, passwd, len,
                                        cb, arg);

    wolfSSL_EVP_PKEY_free(pkey);

    return ret;
}

#if defined(WOLFSSL_KEY_GEN) && !defined(NO_RSA) && !defined(HAVE_USER_RSA)
/* forward declaration for wolfSSL_PEM_write_bio_RSA_PUBKEY */
static int WriteBioPUBKEY(WOLFSSL_BIO* bio, WOLFSSL_EVP_PKEY* key);

/* Takes an RSA public key and writes it out to a WOLFSSL_BIO
 * Returns WOLFSSL_SUCCESS or WOLFSSL_FAILURE
 */
int wolfSSL_PEM_write_bio_RSA_PUBKEY(WOLFSSL_BIO* bio, WOLFSSL_RSA* rsa)
{
    int ret = 0;
    WOLFSSL_EVP_PKEY* pkey = NULL;

    WOLFSSL_ENTER("wolfSSL_PEM_write_bio_RSA_PUBKEY");

    if (bio == NULL || rsa == NULL) {
        WOLFSSL_MSG("Bad Function Arguments");
        return WOLFSSL_FAILURE;
    }

    /* Initialize pkey structure */
    pkey = wolfSSL_EVP_PKEY_new_ex(bio->heap);
    if (pkey == NULL) {
        WOLFSSL_MSG("wolfSSL_EVP_PKEY_new_ex failed");
        return WOLFSSL_FAILURE;
    }

    pkey->type   = EVP_PKEY_RSA;
    pkey->rsa    = rsa;
    pkey->ownRsa = 0;

    ret = WriteBioPUBKEY(bio, pkey);
    wolfSSL_EVP_PKEY_free(pkey);

    return ret;
}

#ifndef NO_FILESYSTEM
int wolfSSL_PEM_write_RSAPublicKey(XFILE fp, WOLFSSL_RSA* key)
{
    int ret = WOLFSSL_SUCCESS;
    WOLFSSL_BIO* bio = NULL;

    WOLFSSL_ENTER("wolfSSL_PEM_write_RSAPublicKey");

    if (fp == XBADFILE || key == NULL) {
        WOLFSSL_MSG("Bad argument.");
        ret = WOLFSSL_FAILURE;
    }

    if (ret == WOLFSSL_SUCCESS) {
        bio = wolfSSL_BIO_new(wolfSSL_BIO_s_file());
        if (bio == NULL) {
            WOLFSSL_MSG("wolfSSL_BIO_new failed.");
            ret = WOLFSSL_FAILURE;
        }
        else if (wolfSSL_BIO_set_fp(bio, fp, BIO_NOCLOSE) != WOLFSSL_SUCCESS) {
            WOLFSSL_MSG("wolfSSL_BIO_set_fp failed.");
            ret = WOLFSSL_FAILURE;
        }
    }
    if (ret == WOLFSSL_SUCCESS && wolfSSL_PEM_write_bio_RSA_PUBKEY(bio, key)
        != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("wolfSSL_PEM_write_bio_RSA_PUBKEY failed.");
        ret = WOLFSSL_FAILURE;
    }

    if (bio != NULL) {
        wolfSSL_BIO_free(bio);
    }

    WOLFSSL_LEAVE("wolfSSL_PEM_write_RSAPublicKey", ret);

    return ret;
}
#endif /* !NO_FILESYSTEM */
#endif /* WOLFSSL_KEY_GEN && !NO_RSA && !HAVE_USER_RSA */


/* Reads an RSA public key from a WOLFSSL_BIO into a WOLFSSL_RSA
 * Returns WOLFSSL_SUCCESS or WOLFSSL_FAILURE
 */
WOLFSSL_RSA *wolfSSL_PEM_read_bio_RSA_PUBKEY(WOLFSSL_BIO* bio,WOLFSSL_RSA** rsa,
                                             wc_pem_password_cb* cb,
                                             void *pass)
{
    WOLFSSL_EVP_PKEY* pkey;
    WOLFSSL_RSA* local;

    WOLFSSL_ENTER("wolfSSL_PEM_read_bio_RSA_PUBKEY");

    pkey = wolfSSL_PEM_read_bio_PUBKEY(bio, NULL, cb, pass);
    if (pkey == NULL) {
        return NULL;
    }

    /* Since the WOLFSSL_RSA structure is being taken from WOLFSSL_EVP_PKEY the
     * flag indicating that the WOLFSSL_RSA structure is owned should be FALSE
     * to avoid having it free'd */
    pkey->ownRsa = 0;
    local = pkey->rsa;
    if (rsa != NULL){
        *rsa = local;
    }

    wolfSSL_EVP_PKEY_free(pkey);
    return local;
}

#endif /* defined(OPENSSL_EXTRA) || defined(OPENSSL_ALL) && !defined(NO_RSA) */

static int WriteBioPUBKEY(WOLFSSL_BIO* bio, WOLFSSL_EVP_PKEY* key)
{
    int ret;
    int pemSz;
    byte* pemBuf;
    int derSz = 0;
    byte* derBuf = NULL;

    if (bio == NULL || key == NULL) {
        WOLFSSL_MSG("Bad parameters");
        return WOLFSSL_FAILURE;
    }

    switch (key->type) {
#if defined(WOLFSSL_KEY_GEN) && !defined(NO_RSA) && !defined(HAVE_USER_RSA)
        case EVP_PKEY_RSA:
            if ((derSz = wolfSSL_RSA_To_Der(key->rsa, &derBuf, 1, bio->heap))
                    < 0) {
                WOLFSSL_MSG("wolfSSL_RSA_To_Der failed");
                break;
            }
            break;
#endif /* WOLFSSL_KEY_GEN && !NO_RSA && !HAVE_USER_RSA */
#if !defined(NO_DSA) && !defined(HAVE_SELFTEST) && (defined(WOLFSSL_KEY_GEN) || \
        defined(WOLFSSL_CERT_GEN))
        case EVP_PKEY_DSA:
            if (key->dsa == NULL) {
                WOLFSSL_MSG("key->dsa is null");
                break;
            }
            derSz = MAX_DSA_PUBKEY_SZ;
            derBuf = (byte*)XMALLOC(derSz, bio->heap, DYNAMIC_TYPE_TMP_BUFFER);
            if (derBuf == NULL) {
                WOLFSSL_MSG("malloc failed");
                break;
            }
            /* Key to DER */
            derSz = wc_DsaKeyToPublicDer((DsaKey*)key->dsa->internal, derBuf,
                    derSz);
            if (derSz < 0) {
                WOLFSSL_MSG("wc_DsaKeyToDer failed");
                break;
            }
            break;
#endif /* !NO_DSA && !HAVE_SELFTEST && (WOLFSSL_KEY_GEN || WOLFSSL_CERT_GEN) */
#if defined(HAVE_ECC) && defined(HAVE_ECC_KEY_EXPORT)
        case EVP_PKEY_EC:
        {
            if (key->ecc == NULL) {
                WOLFSSL_MSG("key->ecc is null");
                break;
            }
            derSz = wc_EccPublicKeyDerSize((ecc_key*)key->ecc->internal, 1);
            if (derSz <= 0) {
                WOLFSSL_MSG("wc_EccPublicKeyDerSize failed");
                break;
            }
            derBuf = (byte*)XMALLOC(derSz, bio->heap, DYNAMIC_TYPE_TMP_BUFFER);
            if (derBuf == NULL) {
                WOLFSSL_MSG("malloc failed");
                break;
            }
            derSz = wc_EccPublicKeyToDer((ecc_key*)key->ecc->internal, derBuf,
                    derSz, 1);
            if (derSz < 0) {
                WOLFSSL_MSG("wc_EccPublicKeyToDer failed");
                break;
            }
            break;
        }
#endif /* HAVE_ECC && HAVE_ECC_KEY_EXPORT */
#if !defined(NO_DH) && (defined(WOLFSSL_QT) || defined(OPENSSL_ALL))
        case EVP_PKEY_DH:
            WOLFSSL_MSG("Writing DH PUBKEY not supported!");
            break;
#endif /* !NO_DH && (WOLFSSL_QT || OPENSSL_ALL) */
        default:
            WOLFSSL_MSG("Unknown Key type!");
            break;
    }

    if (derBuf == NULL || derSz <= 0) {
        if (derBuf != NULL)
            XFREE(derBuf, bio->heap, DYNAMIC_TYPE_DER);
        return WOLFSSL_FAILURE;
    }

    pemSz = wc_DerToPem(derBuf, derSz, NULL, 0, PUBLICKEY_TYPE);
    if (pemSz < 0) {
        WOLFSSL_LEAVE("WriteBioPUBKEY", pemSz);
        XFREE(derBuf, bio->heap, DYNAMIC_TYPE_DER);
        return WOLFSSL_FAILURE;
    }

    pemBuf = (byte*)XMALLOC(pemSz, bio->heap, DYNAMIC_TYPE_TMP_BUFFER);
    if (pemBuf == NULL) {
        WOLFSSL_LEAVE("WriteBioPUBKEY", pemSz);
        XFREE(derBuf, bio->heap, DYNAMIC_TYPE_DER);
        return WOLFSSL_FAILURE;
    }

    ret = wc_DerToPem(derBuf, derSz, pemBuf, pemSz, PUBLICKEY_TYPE);
    XFREE(derBuf, bio->heap, DYNAMIC_TYPE_DER);
    if (ret < 0) {
        WOLFSSL_LEAVE("WriteBioPUBKEY", ret);
        XFREE(pemBuf, bio->heap, DYNAMIC_TYPE_TMP_BUFFER);
        return WOLFSSL_FAILURE;
    }

    ret = wolfSSL_BIO_write(bio, pemBuf, pemSz);
    XFREE(pemBuf, bio->heap, DYNAMIC_TYPE_TMP_BUFFER);
    if (ret != pemSz) {
        WOLFSSL_MSG("Unable to write full PEM to BIO");
        return WOLFSSL_FAILURE;
    }

    return WOLFSSL_SUCCESS;
}

/* Takes a public key and writes it out to a WOLFSSL_BIO
 * Returns WOLFSSL_SUCCESS or WOLFSSL_FAILURE
 */
int wolfSSL_PEM_write_bio_PUBKEY(WOLFSSL_BIO* bio, WOLFSSL_EVP_PKEY* key)
{
    WOLFSSL_ENTER("wolfSSL_PEM_write_bio_PUBKEY");

    return WriteBioPUBKEY(bio, key);
}

/* Takes a private key and writes it out to a WOLFSSL_BIO
 * Returns WOLFSSL_SUCCESS or WOLFSSL_FAILURE
 */
int wolfSSL_PEM_write_bio_PrivateKey(WOLFSSL_BIO* bio, WOLFSSL_EVP_PKEY* key,
                                     const WOLFSSL_EVP_CIPHER* cipher,
                                     unsigned char* passwd, int len,
                                     wc_pem_password_cb* cb, void* arg)
{
    byte* keyDer;
    int pemSz;
    int type;
    int ret;
    byte* tmp;

    (void)cipher;
    (void)passwd;
    (void)len;
    (void)cb;
    (void)arg;

    WOLFSSL_ENTER("wolfSSL_PEM_write_bio_PrivateKey");

    if (bio == NULL || key == NULL) {
        WOLFSSL_MSG("Bad Function Arguments");
        return WOLFSSL_FAILURE;
    }

    keyDer = (byte*)key->pkey.ptr;

    switch (key->type) {
#ifndef NO_RSA
        case EVP_PKEY_RSA:
            type = PRIVATEKEY_TYPE;
            break;
#endif

#ifndef NO_DSA
        case EVP_PKEY_DSA:
            type = DSA_PRIVATEKEY_TYPE;
            break;
#endif

#ifdef HAVE_ECC
        case EVP_PKEY_EC:
            type = ECC_PRIVATEKEY_TYPE;
            break;
#endif

#if !defined(NO_DH) && (defined(WOLFSSL_QT) || defined(OPENSSL_ALL))
        case EVP_PKEY_DH:
            type = DH_PRIVATEKEY_TYPE;
            break;
#endif

        default:
            WOLFSSL_MSG("Unknown Key type!");
            type = PRIVATEKEY_TYPE;
    }

    pemSz = wc_DerToPem(keyDer, key->pkey_sz, NULL, 0, type);
    if (pemSz < 0) {
        WOLFSSL_LEAVE("wolfSSL_PEM_write_bio_PrivateKey", pemSz);
        return WOLFSSL_FAILURE;
    }
    tmp = (byte*)XMALLOC(pemSz, bio->heap, DYNAMIC_TYPE_OPENSSL);
    if (tmp == NULL) {
        return MEMORY_E;
    }

    ret = wc_DerToPem(keyDer, key->pkey_sz, tmp, pemSz, type);
    if (ret < 0) {
        WOLFSSL_LEAVE("wolfSSL_PEM_write_bio_PrivateKey", ret);
        XFREE(tmp, bio->heap, DYNAMIC_TYPE_OPENSSL);
        return WOLFSSL_FAILURE;
    }

    ret = wolfSSL_BIO_write(bio, tmp, pemSz);
    XFREE(tmp, bio->heap, DYNAMIC_TYPE_OPENSSL);
    if (ret != pemSz) {
        WOLFSSL_MSG("Unable to write full PEM to BIO");
        return WOLFSSL_FAILURE;
    }

    return WOLFSSL_SUCCESS;
}
#endif /* !NO_BIO */

#if (defined(WOLFSSL_KEY_GEN) && !defined(NO_RSA) && !defined(HAVE_USER_RSA)) && \
    (defined(WOLFSSL_PEM_TO_DER) || defined(WOLFSSL_DER_TO_PEM))

/* return code compliant with OpenSSL :
 *   1 if success, 0 if error
 */
int wolfSSL_PEM_write_mem_RSAPrivateKey(RSA* rsa, const EVP_CIPHER* cipher,
                                        unsigned char* passwd, int passwdSz,
                                        unsigned char **pem, int *plen)
{
    byte *derBuf = NULL, *tmp, *cipherInfo = NULL;
    int  derSz = 0;
    const int type = PRIVATEKEY_TYPE;
    const char* header = NULL;
    const char* footer = NULL;

    WOLFSSL_ENTER("wolfSSL_PEM_write_mem_RSAPrivateKey");

    if (pem == NULL || plen == NULL || rsa == NULL || rsa->internal == NULL) {
        WOLFSSL_MSG("Bad function arguments");
        return WOLFSSL_FAILURE;
    }

    if (wc_PemGetHeaderFooter(type, &header, &footer) != 0)
        return WOLFSSL_FAILURE;

    if (rsa->inSet == 0) {
        WOLFSSL_MSG("No RSA internal set, do it");

        if (SetRsaInternal(rsa) != WOLFSSL_SUCCESS) {
            WOLFSSL_MSG("SetRsaInternal failed");
            return WOLFSSL_FAILURE;
        }
    }

    if ((derSz = wolfSSL_RSA_To_Der(rsa, &derBuf, 0, rsa->heap)) < 0) {
        WOLFSSL_MSG("wolfSSL_RSA_To_Der failed");
        return WOLFSSL_FAILURE;
    }

    /* encrypt DER buffer if required */
    if (passwd != NULL && passwdSz > 0 && cipher != NULL) {
        int ret;
        int blockSz = wolfSSL_EVP_CIPHER_block_size(cipher);
        byte *tmpBuf;

        /* Add space for padding */
        if (!(tmpBuf = (byte*)XREALLOC(derBuf, derSz + blockSz, rsa->heap,
                DYNAMIC_TYPE_TMP_BUFFER))) {
            WOLFSSL_MSG("Extending DER buffer failed");
            XFREE(derBuf, NULL, DYNAMIC_TYPE_DER);
            return WOLFSSL_FAILURE;
        }
        derBuf = tmpBuf;

        ret = EncryptDerKey(derBuf, &derSz, cipher,
                            passwd, passwdSz, &cipherInfo, derSz + blockSz);
        if (ret != WOLFSSL_SUCCESS) {
            WOLFSSL_MSG("EncryptDerKey failed");
            XFREE(derBuf, rsa->heap, DYNAMIC_TYPE_DER);
            return ret;
        }

        /* tmp buffer with a max size */
        *plen = (derSz * 2) + (int)XSTRLEN(header) + 1 +
            (int)XSTRLEN(footer) + 1 + HEADER_ENCRYPTED_KEY_SIZE;
    }
    else {
        /* tmp buffer with a max size */
        *plen = (derSz * 2) + (int)XSTRLEN(header) + 1 +
            (int)XSTRLEN(footer) + 1;
    }

    tmp = (byte*)XMALLOC(*plen, NULL, DYNAMIC_TYPE_PEM);
    if (tmp == NULL) {
        WOLFSSL_MSG("malloc failed");
        XFREE(derBuf, rsa->heap, DYNAMIC_TYPE_DER);
        if (cipherInfo != NULL)
            XFREE(cipherInfo, NULL, DYNAMIC_TYPE_STRING);
        return WOLFSSL_FAILURE;
    }

    /* DER to PEM */
    *plen = wc_DerToPemEx(derBuf, derSz, tmp, *plen, cipherInfo, type);
    if (*plen <= 0) {
        WOLFSSL_MSG("wc_DerToPemEx failed");
        XFREE(derBuf, rsa->heap, DYNAMIC_TYPE_DER);
        XFREE(tmp, NULL, DYNAMIC_TYPE_PEM);
        if (cipherInfo != NULL)
            XFREE(cipherInfo, NULL, DYNAMIC_TYPE_STRING);
        return WOLFSSL_FAILURE;
    }
    XFREE(derBuf, rsa->heap, DYNAMIC_TYPE_DER);
    if (cipherInfo != NULL)
        XFREE(cipherInfo, NULL, DYNAMIC_TYPE_STRING);

    *pem = (byte*)XMALLOC((*plen)+1, NULL, DYNAMIC_TYPE_KEY);
    if (*pem == NULL) {
        WOLFSSL_MSG("malloc failed");
        XFREE(tmp, NULL, DYNAMIC_TYPE_PEM);
        return WOLFSSL_FAILURE;
    }
    XMEMSET(*pem, 0, (*plen)+1);

    if (XMEMCPY(*pem, tmp, *plen) == NULL) {
        WOLFSSL_MSG("XMEMCPY failed");
        XFREE(pem, NULL, DYNAMIC_TYPE_KEY);
        XFREE(tmp, NULL, DYNAMIC_TYPE_PEM);
        return WOLFSSL_FAILURE;
    }
    XFREE(tmp, NULL, DYNAMIC_TYPE_PEM);

    return WOLFSSL_SUCCESS;
}


#ifndef NO_FILESYSTEM
/* return code compliant with OpenSSL :
 *   1 if success, 0 if error
 */
int wolfSSL_PEM_write_RSAPrivateKey(XFILE fp, WOLFSSL_RSA *rsa,
                                    const EVP_CIPHER *enc,
                                    unsigned char *kstr, int klen,
                                    wc_pem_password_cb *cb, void *u)
{
    byte *pem;
    int  plen, ret;

    (void)cb;
    (void)u;

    WOLFSSL_MSG("wolfSSL_PEM_write_RSAPrivateKey");

    if (fp == XBADFILE || rsa == NULL || rsa->internal == NULL)
    {
        WOLFSSL_MSG("Bad function arguments");
        return WOLFSSL_FAILURE;
    }

    ret = wolfSSL_PEM_write_mem_RSAPrivateKey(rsa, enc, kstr, klen, &pem, &plen);
    if (ret != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("wolfSSL_PEM_write_mem_RSAPrivateKey failed");
        return WOLFSSL_FAILURE;
    }

    ret = (int)XFWRITE(pem, plen, 1, fp);
    if (ret != 1) {
        WOLFSSL_MSG("RSA private key file write failed");
        return WOLFSSL_FAILURE;
    }

    XFREE(pem, NULL, DYNAMIC_TYPE_KEY);
    return WOLFSSL_SUCCESS;
}
#endif /* NO_FILESYSTEM */
#endif /* WOLFSSL_KEY_GEN && !NO_RSA && !HAVE_USER_RSA && WOLFSSL_PEM_TO_DER */

/* Colon separated list of <public key>+<digest> algorithms.
 * Replaces list in context.
 */
int wolfSSL_CTX_set1_sigalgs_list(WOLFSSL_CTX* ctx, const char* list)
{
    WOLFSSL_MSG("wolfSSL_CTX_set1_sigalg_list");

    if (ctx == NULL || list == NULL) {
        WOLFSSL_MSG("Bad function arguments");
        return WOLFSSL_FAILURE;
    }

    /* alloc/init on demand only */
    if (ctx->suites == NULL) {
        ctx->suites = (Suites*)XMALLOC(sizeof(Suites), ctx->heap,
                                       DYNAMIC_TYPE_SUITES);
        if (ctx->suites == NULL) {
            WOLFSSL_MSG("Memory alloc for Suites failed");
            return WOLFSSL_FAILURE;
        }
        XMEMSET(ctx->suites, 0, sizeof(Suites));
    }

    return SetSuitesHashSigAlgo(ctx->suites, list);
}

/* Colon separated list of <public key>+<digest> algorithms.
 * Replaces list in SSL.
 */
int wolfSSL_set1_sigalgs_list(WOLFSSL* ssl, const char* list)
{
    WOLFSSL_MSG("wolfSSL_set1_sigalg_list");

    if (ssl == NULL) {
        WOLFSSL_MSG("Bad function arguments");
        return WOLFSSL_FAILURE;
    }

#ifdef SINGLE_THREADED
    if (ssl->ctx->suites == ssl->suites) {
        ssl->suites = (Suites*)XMALLOC(sizeof(Suites), ssl->heap,
                                       DYNAMIC_TYPE_SUITES);
        if (ssl->suites == NULL) {
            WOLFSSL_MSG("Suites Memory error");
            return MEMORY_E;
        }
        *ssl->suites = *ssl->ctx->suites;
        ssl->options.ownSuites = 1;
    }
#endif
    if (ssl == NULL || list == NULL) {
        WOLFSSL_MSG("Bad function arguments");
        return WOLFSSL_FAILURE;
    }

    return SetSuitesHashSigAlgo(ssl->suites, list);
}

struct WOLFSSL_HashSigInfo {
    int hashAlgo;
    int sigAlgo;
    int nid;
}  wolfssl_hash_sig_info[] =
{
#ifndef NO_RSA
    #ifndef NO_SHA256
        { sha256_mac, rsa_sa_algo, CTC_SHA256wRSA },
    #endif
    #ifdef WOLFSSL_SHA384
        { sha384_mac, rsa_sa_algo, CTC_SHA384wRSA },
    #endif
    #ifdef WOLFSSL_SHA512
        { sha512_mac, rsa_sa_algo, CTC_SHA512wRSA },
    #endif
    #ifdef WOLFSSL_SHA224
        { sha224_mac, rsa_sa_algo, CTC_SHA224wRSA },
    #endif
    #ifndef NO_SHA
        { sha_mac,    rsa_sa_algo, CTC_SHAwRSA },
    #endif
    #ifdef WC_RSA_PSS
        #ifndef NO_SHA256
            { sha256_mac, rsa_pss_sa_algo, CTC_SHA256wRSA },
        #endif
        #ifdef WOLFSSL_SHA384
            { sha384_mac, rsa_pss_sa_algo, CTC_SHA384wRSA },
        #endif
        #ifdef WOLFSSL_SHA512
            { sha512_mac, rsa_pss_sa_algo, CTC_SHA512wRSA },
        #endif
        #ifdef WOLFSSL_SHA224
            { sha224_mac, rsa_pss_sa_algo, CTC_SHA224wRSA },
        #endif
    #endif
#endif
#ifdef HAVE_ECC
    #ifndef NO_SHA256
        { sha256_mac, ecc_dsa_sa_algo, CTC_SHA256wECDSA },
    #endif
    #ifdef WOLFSSL_SHA384
        { sha384_mac, ecc_dsa_sa_algo, CTC_SHA384wECDSA },
    #endif
    #ifdef WOLFSSL_SHA512
        { sha512_mac, ecc_dsa_sa_algo, CTC_SHA512wECDSA },
    #endif
    #ifdef WOLFSSL_SHA224
        { sha224_mac, ecc_dsa_sa_algo, CTC_SHA224wECDSA },
    #endif
    #ifndef NO_SHA
        { sha_mac,    ecc_dsa_sa_algo, CTC_SHAwECDSA },
    #endif
#endif
#ifdef HAVE_ED25519
    { no_mac, ed25519_sa_algo, CTC_ED25519 },
#endif
#ifdef HAVE_ED448
    { no_mac, ed448_sa_algo, CTC_ED448 },
#endif
#ifdef HAVE_PQC
    { no_mac, falcon_level1_sa_algo, CTC_FALCON_LEVEL1 },
    { no_mac, falcon_level5_sa_algo, CTC_FALCON_LEVEL5 },
#endif
#ifndef NO_DSA
    #ifndef NO_SHA
        { sha_mac,    dsa_sa_algo, CTC_SHAwDSA },
    #endif
#endif
};
#define WOLFSSL_HASH_SIG_INFO_SZ \
    (int)(sizeof(wolfssl_hash_sig_info)/sizeof(*wolfssl_hash_sig_info))

int wolfSSL_get_signature_nid(WOLFSSL *ssl, int* nid)
{
    int i;
    int ret = WOLFSSL_FAILURE;

    WOLFSSL_MSG("wolfSSL_get_signature_nid");

    if (ssl == NULL) {
        WOLFSSL_MSG("Bad function arguments");
        return WOLFSSL_FAILURE;
    }

    for (i = 0; i < WOLFSSL_HASH_SIG_INFO_SZ; i++) {
        if (ssl->suites->hashAlgo == wolfssl_hash_sig_info[i].hashAlgo &&
                     ssl->suites->sigAlgo == wolfssl_hash_sig_info[i].sigAlgo) {
            *nid = wolfssl_hash_sig_info[i].nid;
            ret = WOLFSSL_SUCCESS;
            break;
        }
    }

    return ret;
}

#ifdef HAVE_ECC

#ifdef ALT_ECC_SIZE
static int SetIndividualInternalEcc(WOLFSSL_BIGNUM* bn, mp_int* mpi)
{
    WOLFSSL_MSG("Entering SetIndividualInternal");

    if (bn == NULL || bn->internal == NULL) {
        WOLFSSL_MSG("bn NULL error");
        return WOLFSSL_FATAL_ERROR;
    }

    if (mpi == NULL) {
        WOLFSSL_MSG("mpi NULL error");
        return WOLFSSL_FATAL_ERROR;
    }

    if (mp_copy((mp_int*)bn->internal, mpi) != MP_OKAY) {
        WOLFSSL_MSG("mp_copy error");
        return WOLFSSL_FATAL_ERROR;
    }

    return WOLFSSL_SUCCESS;
}
#endif /* ALT_ECC_SIZE */

/* EC_POINT Openssl -> WolfSSL */
static int SetECPointInternal(WOLFSSL_EC_POINT *p)
{
    ecc_point* point;
    WOLFSSL_ENTER("SetECPointInternal");

    if (p == NULL || p->internal == NULL) {
        WOLFSSL_MSG("ECPoint NULL error");
        return WOLFSSL_FATAL_ERROR;
    }

    point = (ecc_point*)p->internal;

#ifndef ALT_ECC_SIZE
    if (p->X != NULL && SetIndividualInternal(p->X, point->x) != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("ecc point X error");
        return WOLFSSL_FATAL_ERROR;
    }

    if (p->Y != NULL && SetIndividualInternal(p->Y, point->y) != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("ecc point Y error");
        return WOLFSSL_FATAL_ERROR;
    }

    if (p->Z != NULL && SetIndividualInternal(p->Z, point->z) != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("ecc point Z error");
        return WOLFSSL_FATAL_ERROR;
    }
#else
    if (p->X != NULL && SetIndividualInternalEcc(p->X, point->x) != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("ecc point X error");
        return WOLFSSL_FATAL_ERROR;
    }

    if (p->Y != NULL && SetIndividualInternalEcc(p->Y, point->y) != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("ecc point Y error");
        return WOLFSSL_FATAL_ERROR;
    }

    if (p->Z != NULL && SetIndividualInternalEcc(p->Z, point->z) != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("ecc point Z error");
        return WOLFSSL_FATAL_ERROR;
    }
#endif

    p->inSet = 1;

    return WOLFSSL_SUCCESS;
}

/* EC_POINT WolfSSL -> OpenSSL */
static int SetECPointExternal(WOLFSSL_EC_POINT *p)
{
    ecc_point* point;

    WOLFSSL_ENTER("SetECPointExternal");

    if (p == NULL || p->internal == NULL) {
        WOLFSSL_MSG("ECPoint NULL error");
        return WOLFSSL_FATAL_ERROR;
    }

    point = (ecc_point*)p->internal;

    if (SetIndividualExternal(&p->X, point->x) != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("ecc point X error");
        return WOLFSSL_FATAL_ERROR;
    }

    if (SetIndividualExternal(&p->Y, point->y) != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("ecc point Y error");
        return WOLFSSL_FATAL_ERROR;
    }

    if (SetIndividualExternal(&p->Z, point->z) != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("ecc point Z error");
        return WOLFSSL_FATAL_ERROR;
    }

    p->exSet = 1;

    return WOLFSSL_SUCCESS;
}


/* EC_KEY wolfSSL -> OpenSSL */
int SetECKeyExternal(WOLFSSL_EC_KEY* eckey)
{
    ecc_key* key;

    WOLFSSL_ENTER("SetECKeyExternal");

    if (eckey == NULL || eckey->internal == NULL) {
        WOLFSSL_MSG("ec key NULL error");
        return WOLFSSL_FATAL_ERROR;
    }

    key = (ecc_key*)eckey->internal;

    /* set group (OID, nid and idx) */
    eckey->group->curve_oid = ecc_sets[key->idx].oidSum;
    eckey->group->curve_nid = EccEnumToNID(ecc_sets[key->idx].id);
    eckey->group->curve_idx = key->idx;

    if (eckey->pub_key->internal != NULL) {
        /* set the internal public key */
        if (wc_ecc_copy_point(&key->pubkey,
                             (ecc_point*)eckey->pub_key->internal) != MP_OKAY) {
            WOLFSSL_MSG("SetECKeyExternal ecc_copy_point failed");
            return WOLFSSL_FATAL_ERROR;
        }

        /* set the external pubkey (point) */
        if (SetECPointExternal(eckey->pub_key) != WOLFSSL_SUCCESS) {
            WOLFSSL_MSG("SetECKeyExternal SetECPointExternal failed");
            return WOLFSSL_FATAL_ERROR;
        }
    }

    /* set the external privkey */
    if (key->type == ECC_PRIVATEKEY) {
        if (SetIndividualExternal(&eckey->priv_key, &key->k) != WOLFSSL_SUCCESS) {
            WOLFSSL_MSG("ec priv key error");
            return WOLFSSL_FATAL_ERROR;
        }
    }

    eckey->exSet = 1;

    return WOLFSSL_SUCCESS;
}

/* EC_KEY Openssl -> WolfSSL */
int SetECKeyInternal(WOLFSSL_EC_KEY* eckey)
{
    ecc_key* key;

    WOLFSSL_ENTER("SetECKeyInternal");

    if (eckey == NULL || eckey->internal == NULL || eckey->group == NULL) {
        WOLFSSL_MSG("ec key NULL error");
        return WOLFSSL_FATAL_ERROR;
    }

    key = (ecc_key*)eckey->internal;

    /* validate group */
    if ((eckey->group->curve_idx < 0) ||
        (wc_ecc_is_valid_idx(eckey->group->curve_idx) == 0)) {
        WOLFSSL_MSG("invalid curve idx");
        return WOLFSSL_FATAL_ERROR;
    }

    /* set group (idx of curve and corresponding domain parameters) */
    key->idx = eckey->group->curve_idx;
    key->dp = &ecc_sets[key->idx];

    /* set pubkey (point) */
    if (eckey->pub_key != NULL) {
        if (SetECPointInternal(eckey->pub_key) != WOLFSSL_SUCCESS) {
            WOLFSSL_MSG("ec key pub error");
            return WOLFSSL_FATAL_ERROR;
        }

        /* copy over the public point to key */
        if (wc_ecc_copy_point((ecc_point*)eckey->pub_key->internal, &key->pubkey) != MP_OKAY) {
            WOLFSSL_MSG("wc_ecc_copy_point error");
            return WOLFSSL_FATAL_ERROR;
        }

        /* public key */
        key->type = ECC_PUBLICKEY;
    }

    /* set privkey */
    if (eckey->priv_key != NULL) {
        if (SetIndividualInternal(eckey->priv_key, &key->k) != WOLFSSL_SUCCESS) {
            WOLFSSL_MSG("ec key priv error");
            return WOLFSSL_FATAL_ERROR;
        }

        /* private key */
        if (!mp_iszero(&key->k))
            key->type = ECC_PRIVATEKEY;
    }

    eckey->inSet = 1;

    return WOLFSSL_SUCCESS;
}

WOLFSSL_EC_POINT *wolfSSL_EC_KEY_get0_public_key(const WOLFSSL_EC_KEY *key)
{
    WOLFSSL_ENTER("wolfSSL_EC_KEY_get0_public_key");

    if (key == NULL) {
        WOLFSSL_MSG("wolfSSL_EC_KEY_get0_public_key Bad arguments");
        return NULL;
    }

    return key->pub_key;
}

const WOLFSSL_EC_GROUP *wolfSSL_EC_KEY_get0_group(const WOLFSSL_EC_KEY *key)
{
    WOLFSSL_ENTER("wolfSSL_EC_KEY_get0_group");

    if (key == NULL) {
        WOLFSSL_MSG("wolfSSL_EC_KEY_get0_group Bad arguments");
        return NULL;
    }

    return key->group;
}


/* return code compliant with OpenSSL :
 *   1 if success, 0 if error
 */
int wolfSSL_EC_KEY_set_private_key(WOLFSSL_EC_KEY *key,
                                   const WOLFSSL_BIGNUM *priv_key)
{
    WOLFSSL_ENTER("wolfSSL_EC_KEY_set_private_key");

    if (key == NULL || priv_key == NULL) {
        WOLFSSL_MSG("Bad arguments");
        return WOLFSSL_FAILURE;
    }

    /* free key if previously set */
    if (key->priv_key != NULL)
        wolfSSL_BN_free(key->priv_key);

    key->priv_key = wolfSSL_BN_dup(priv_key);
    if (key->priv_key == NULL) {
        WOLFSSL_MSG("key ecc priv key NULL");
        return WOLFSSL_FAILURE;
    }

    if (SetECKeyInternal(key) != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("SetECKeyInternal failed");
        wolfSSL_BN_free(key->priv_key);
        return WOLFSSL_FAILURE;
    }

    return WOLFSSL_SUCCESS;
}


WOLFSSL_BIGNUM *wolfSSL_EC_KEY_get0_private_key(const WOLFSSL_EC_KEY *key)
{
    WOLFSSL_ENTER("wolfSSL_EC_KEY_get0_private_key");

    if (key == NULL) {
        WOLFSSL_MSG("wolfSSL_EC_KEY_get0_private_key Bad arguments");
        return NULL;
    }

    if (wolfSSL_BN_is_zero(key->priv_key)) {
        /* return NULL if not set */
        return NULL;
    }

    return key->priv_key;
}

WOLFSSL_EC_KEY *wolfSSL_EC_KEY_new_by_curve_name(int nid)
{
    WOLFSSL_EC_KEY *key;
    int x;
    int eccEnum = NIDToEccEnum(nid);

    WOLFSSL_ENTER("wolfSSL_EC_KEY_new_by_curve_name");

    key = wolfSSL_EC_KEY_new();
    if (key == NULL) {
        WOLFSSL_MSG("wolfSSL_EC_KEY_new failure");
        return NULL;
    }

    /* set the nid of the curve */
    key->group->curve_nid = nid;

    if (eccEnum != -1) {
        /* search and set the corresponding internal curve idx */
        for (x = 0; ecc_sets[x].size != 0; x++)
            if (ecc_sets[x].id == eccEnum) {
                key->group->curve_idx = x;
                key->group->curve_oid = ecc_sets[x].oidSum;
                break;
            }
    }

    return key;
}

const char* wolfSSL_EC_curve_nid2nist(int nid)
{
    const WOLF_EC_NIST_NAME* nist_name;
    for (nist_name = kNistCurves; nist_name->name != NULL; nist_name++) {
        if (nist_name->nid == nid) {
            return nist_name->name;
        }
    }
    return NULL;
}

/**
 * return nist curve id
 * @param name nist curve name
 * @return nist curve id when found, 0 when not found
 */
int wolfSSL_EC_curve_nist2nid(const char* name)
{
    const WOLF_EC_NIST_NAME* nist_name;
    for (nist_name = kNistCurves; nist_name->name != NULL; nist_name++) {
        if (XSTRCMP(nist_name->name, name) == 0) {
            return nist_name->nid;
        }
    }
    return 0;
}

#if defined(WOLFSSL_TLS13) && defined(HAVE_SUPPORTED_CURVES)
static int populate_groups(int* groups, int max_count, char *list)
{
    char *end;
    int len;
    int count = 0;
    const WOLF_EC_NIST_NAME* nist_name;

    if (!groups || !list) {
        return -1;
    }

    for (end = list; ; list = ++end) {
        if (count > max_count) {
            WOLFSSL_MSG("Too many curves in list");
            return -1;
        }
        while (*end != ':' && *end != '\0') end++;
        len = (int)(end - list); /* end points to char after end
                                  * of curve name so no need for -1 */
        if ((len < kNistCurves_MIN_NAME_LEN) ||
                (len > kNistCurves_MAX_NAME_LEN)) {
            WOLFSSL_MSG("Unrecognized curve name in list");
            return -1;
        }
        for (nist_name = kNistCurves; nist_name->name != NULL; nist_name++) {
            if (len == nist_name->name_len &&
                    XSTRNCMP(list, nist_name->name, nist_name->name_len) == 0) {
                break;
            }
        }
        if (!nist_name->name) {
            WOLFSSL_MSG("Unrecognized curve name in list");
            return -1;
        }
        groups[count++] = nist_name->nid;
        if (*end == '\0') break;
    }

    return count;
}

int wolfSSL_CTX_set1_groups_list(WOLFSSL_CTX *ctx, char *list)
{
    int groups[WOLFSSL_MAX_GROUP_COUNT];
    int count;

    if (!ctx || !list) {
        return WOLFSSL_FAILURE;
    }

    if ((count = populate_groups(groups,
            WOLFSSL_MAX_GROUP_COUNT, list)) == -1) {
        return WOLFSSL_FAILURE;
    }

    return wolfSSL_CTX_set1_groups(ctx, groups, count);
}

int wolfSSL_set1_groups_list(WOLFSSL *ssl, char *list)
{
    int groups[WOLFSSL_MAX_GROUP_COUNT];
    int count;

    if (!ssl || !list) {
        return WOLFSSL_FAILURE;
    }

    if ((count = populate_groups(groups,
            WOLFSSL_MAX_GROUP_COUNT, list)) == -1) {
        return WOLFSSL_FAILURE;
    }

    return wolfSSL_set1_groups(ssl, groups, count);
}
#endif /* WOLFSSL_TLS13 */

static void InitwolfSSL_ECKey(WOLFSSL_EC_KEY* key)
{
    if (key) {
        key->group    = NULL;
        key->pub_key  = NULL;
        key->priv_key = NULL;
        key->internal = NULL;
        key->inSet    = 0;
        key->exSet    = 0;
        key->form     = POINT_CONVERSION_UNCOMPRESSED;
    }
}

WOLFSSL_EC_KEY *wolfSSL_EC_KEY_new_ex(void* heap, int devId)
{
    WOLFSSL_EC_KEY *external;
    WOLFSSL_ENTER("wolfSSL_EC_KEY_new");

    external = (WOLFSSL_EC_KEY*)XMALLOC(sizeof(WOLFSSL_EC_KEY), heap,
                                        DYNAMIC_TYPE_ECC);
    if (external == NULL) {
        WOLFSSL_MSG("wolfSSL_EC_KEY_new malloc WOLFSSL_EC_KEY failure");
        return NULL;
    }
    XMEMSET(external, 0, sizeof(WOLFSSL_EC_KEY));
    external->heap = heap;

    InitwolfSSL_ECKey(external);

    external->refCount = 1;
#ifndef SINGLE_THREADED
    if (wc_InitMutex(&external->refMutex) != 0) {
        WOLFSSL_MSG("wc_InitMutex WOLFSSL_EC_KEY failure");
        XFREE(external, heap, DYNAMIC_TYPE_ECC);
        return NULL;
    }
#endif

    external->internal = (ecc_key*)XMALLOC(sizeof(ecc_key), heap,
                                           DYNAMIC_TYPE_ECC);
    if (external->internal == NULL) {
        WOLFSSL_MSG("wolfSSL_EC_KEY_new malloc ecc key failure");
        goto error;
    }
    XMEMSET(external->internal, 0, sizeof(ecc_key));

    if (wc_ecc_init_ex((ecc_key*)external->internal, heap, devId) != 0) {
        WOLFSSL_MSG("wolfSSL_EC_KEY_new init ecc key failure");
        goto error;
    }

    /* Group unknown at creation */
    external->group = wolfSSL_EC_GROUP_new_by_curve_name(NID_undef);
    if (external->group == NULL) {
        WOLFSSL_MSG("wolfSSL_EC_KEY_new malloc WOLFSSL_EC_GROUP failure");
        goto error;
    }

    /* public key */
    external->pub_key = wolfSSL_EC_POINT_new(external->group);
    if (external->pub_key == NULL) {
        WOLFSSL_MSG("wolfSSL_EC_POINT_new failure");
        goto error;
    }

    /* private key */
    external->priv_key = wolfSSL_BN_new();
    if (external->priv_key == NULL) {
        WOLFSSL_MSG("wolfSSL_BN_new failure");
        goto error;
    }

    return external;
error:
    wolfSSL_EC_KEY_free(external);
    return NULL;
}

WOLFSSL_EC_KEY *wolfSSL_EC_KEY_new(void)
{
    return wolfSSL_EC_KEY_new_ex(NULL, INVALID_DEVID);
}

void wolfSSL_EC_KEY_free(WOLFSSL_EC_KEY *key)
{
    int doFree = 0;

    WOLFSSL_ENTER("wolfSSL_EC_KEY_free");

    if (key != NULL) {
        void* heap = key->heap;

    #ifndef SINGLE_THREADED
        if (wc_LockMutex(&key->refMutex) != 0) {
            WOLFSSL_MSG("Could not lock EC_KEY mutex");
            return;
        }
    #endif

        /* only free if all references to it are done */
        key->refCount--;
        if (key->refCount == 0) {
            doFree = 1;
        }
    #ifndef SINGLE_THREADED
        wc_UnLockMutex(&key->refMutex);
    #endif

        if (doFree == 0) {
            return;
        }

        if (key->internal != NULL) {
            wc_ecc_free((ecc_key*)key->internal);
            XFREE(key->internal, heap, DYNAMIC_TYPE_ECC);
        }
        wolfSSL_BN_free(key->priv_key);
        wolfSSL_EC_POINT_free(key->pub_key);
        wolfSSL_EC_GROUP_free(key->group);
        InitwolfSSL_ECKey(key); /* set back to NULLs for safety */

        XFREE(key, heap, DYNAMIC_TYPE_ECC);
        (void)heap;
        /* key = NULL, don't try to access or double free it */
    }
}

/* Increments ref count of WOLFSSL_EC_KEY.
 * Return WOLFSSL_SUCCESS on success, WOLFSSL_FAILURE on error */
int wolfSSL_EC_KEY_up_ref(WOLFSSL_EC_KEY* key)
{
    if (key) {
    #ifndef SINGLE_THREADED
        if (wc_LockMutex(&key->refMutex) != 0) {
            WOLFSSL_MSG("Failed to lock EC_KEY mutex");
        }
    #endif
        key->refCount++;
    #ifndef SINGLE_THREADED
        wc_UnLockMutex(&key->refMutex);
    #endif
        return WOLFSSL_SUCCESS;
    }

    return WOLFSSL_FAILURE;
}

/* set the group in WOLFSSL_EC_KEY and return WOLFSSL_SUCCESS on success */
int wolfSSL_EC_KEY_set_group(WOLFSSL_EC_KEY *key, WOLFSSL_EC_GROUP *group)
{
    if (key == NULL || group == NULL)
        return WOLFSSL_FAILURE;

    WOLFSSL_ENTER("wolfSSL_EC_KEY_set_group");

    if (key->group != NULL) {
        /* free the current group */
        wolfSSL_EC_GROUP_free(key->group);
    }

    key->group = wolfSSL_EC_GROUP_dup(group);
    if (key->group == NULL) {
        return WOLFSSL_FAILURE;
    }

    return WOLFSSL_SUCCESS;
}


int wolfSSL_EC_KEY_generate_key(WOLFSSL_EC_KEY *key)
{
    int     initTmpRng = 0;
    int     eccEnum;
    WC_RNG* rng = NULL;
#ifdef WOLFSSL_SMALL_STACK
    WC_RNG* tmpRNG = NULL;
#else
    WC_RNG  tmpRNG[1];
#endif
    int ret;
    ecc_key* ecKey;

    WOLFSSL_ENTER("wolfSSL_EC_KEY_generate_key");

    if (key == NULL || key->internal == NULL ||
        key->group == NULL) {
        WOLFSSL_MSG("wolfSSL_EC_KEY_generate_key Bad arguments");
        return 0;
    }
    if (key->group->curve_idx < 0) {
        /* generate key using the default curve */
        /* group should be set, but to retain compat use index 0 */
        key->group->curve_idx = ECC_CURVE_DEF;
    }

#ifdef WOLFSSL_SMALL_STACK
    tmpRNG = (WC_RNG*)XMALLOC(sizeof(WC_RNG), NULL, DYNAMIC_TYPE_RNG);
    if (tmpRNG == NULL)
        return 0;
#endif

    if (wc_InitRng(tmpRNG) == 0) {
        rng = tmpRNG;
        initTmpRng = 1;
    }
    else {
        WOLFSSL_MSG("Bad RNG Init, trying global");
        if (initGlobalRNG == 0)
            WOLFSSL_MSG("Global RNG no Init");
        else
            rng = &globalRNG;
    }

    if (rng == NULL) {
        WOLFSSL_MSG("wolfSSL_EC_KEY_generate_key failed to set RNG");
#ifdef WOLFSSL_SMALL_STACK
        XFREE(tmpRNG, NULL, DYNAMIC_TYPE_RNG);
#endif
        return 0;
    }

    /* NIDToEccEnum returns -1 for invalid NID so if key->group->curve_nid
     * is 0 then pass ECC_CURVE_DEF as arg */
    ecKey = (ecc_key*)key->internal;
    eccEnum = key->group->curve_nid ?
            NIDToEccEnum(key->group->curve_nid) : ECC_CURVE_DEF;
    ret = wc_ecc_make_key_ex(rng, 0, ecKey, eccEnum);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &ecKey->asyncDev, WC_ASYNC_FLAG_NONE);
#endif

    if (ret != 0) {
        WOLFSSL_MSG("wolfSSL_EC_KEY_generate_key wc_ecc_make_key failed");
#ifdef WOLFSSL_SMALL_STACK
        XFREE(tmpRNG, NULL, DYNAMIC_TYPE_RNG);
#endif
        return 0;
    }

    if (initTmpRng)
        wc_FreeRng(tmpRNG);
#ifdef WOLFSSL_SMALL_STACK
    XFREE(tmpRNG, NULL, DYNAMIC_TYPE_RNG);
#endif

    if (SetECKeyExternal(key) != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("wolfSSL_EC_KEY_generate_key SetECKeyExternal failed");
        return 0;
    }

    return 1;
}

#ifndef NO_WOLFSSL_STUB
void wolfSSL_EC_KEY_set_asn1_flag(WOLFSSL_EC_KEY *key, int asn1_flag)
{
    (void)key;
    (void)asn1_flag;

    WOLFSSL_ENTER("wolfSSL_EC_KEY_set_asn1_flag");
    WOLFSSL_STUB("EC_KEY_set_asn1_flag");
}
#endif

static int setupPoint(const WOLFSSL_EC_POINT *p) {
    if (!p) {
        return WOLFSSL_FAILURE;
    }
    if (p->inSet == 0) {
        WOLFSSL_MSG("No ECPoint internal set, do it");

        if (SetECPointInternal((WOLFSSL_EC_POINT *)p) != WOLFSSL_SUCCESS) {
            WOLFSSL_MSG("SetECPointInternal SetECPointInternal failed");
            return WOLFSSL_FAILURE;
        }
    }
    return WOLFSSL_SUCCESS;
}

/* return code compliant with OpenSSL :
 *   1 if success, 0 if error
 */
int wolfSSL_EC_KEY_set_public_key(WOLFSSL_EC_KEY *key,
                                  const WOLFSSL_EC_POINT *pub)
{
    ecc_point *pub_p, *key_p;

    WOLFSSL_ENTER("wolfSSL_EC_KEY_set_public_key");

    if (key == NULL || key->internal == NULL ||
        pub == NULL || pub->internal == NULL) {
        WOLFSSL_MSG("wolfSSL_EC_KEY_set_public_key Bad arguments");
        return WOLFSSL_FAILURE;
    }

    if (key->inSet == 0) {
        if (SetECKeyInternal(key) != WOLFSSL_SUCCESS) {
            WOLFSSL_MSG("SetECKeyInternal failed");
            return WOLFSSL_FAILURE;
        }
    }

    if (setupPoint(pub) != WOLFSSL_SUCCESS) {
        return WOLFSSL_FAILURE;
    }

    pub_p = (ecc_point*)pub->internal;
    key_p = (ecc_point*)key->pub_key->internal;

    /* create new point if required */
    if (key_p == NULL)
        key_p = wc_ecc_new_point();

    if (key_p == NULL) {
        WOLFSSL_MSG("key ecc point NULL");
        return WOLFSSL_FAILURE;
    }

    if (wc_ecc_copy_point(pub_p, key_p) != MP_OKAY) {
        WOLFSSL_MSG("ecc_copy_point failure");
        return WOLFSSL_FAILURE;
    }

    if (SetECPointExternal(key->pub_key) != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("SetECKeyInternal failed");
        return WOLFSSL_FAILURE;
    }

    if (SetECKeyInternal(key) != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("SetECKeyInternal failed");
        return WOLFSSL_FAILURE;
    }

    wolfSSL_EC_POINT_dump("pub", pub);
    wolfSSL_EC_POINT_dump("key->pub_key", key->pub_key);

    return WOLFSSL_SUCCESS;
}

int wolfSSL_EC_KEY_check_key(const WOLFSSL_EC_KEY *key)
{
    WOLFSSL_ENTER("wolfSSL_EC_KEY_check_key");

    if (key == NULL || key->internal == NULL) {
        WOLFSSL_MSG("Bad parameter");
        return WOLFSSL_FAILURE;
    }

    if (key->inSet == 0) {
        if (SetECKeyInternal((WOLFSSL_EC_KEY*)key) != WOLFSSL_SUCCESS) {
            WOLFSSL_MSG("SetECKeyInternal failed");
            return WOLFSSL_FAILURE;
        }
    }

    return wc_ecc_check_key((ecc_key*)key->internal) == 0 ?
            WOLFSSL_SUCCESS : WOLFSSL_FAILURE;
}
/* End EC_KEY */

/* Calculate and return maximum size of the ECDSA signature for the curve */
int wolfSSL_ECDSA_size(const WOLFSSL_EC_KEY *key)
{
    const EC_GROUP *group;
    int bits, bytes;
    word32 headerSz = SIG_HEADER_SZ; /* 2*ASN_TAG + 2*LEN(ENUM) */

    if (key == NULL) {
        return WOLFSSL_FAILURE;
    }

    if ((group = wolfSSL_EC_KEY_get0_group(key)) == NULL) {
        return WOLFSSL_FAILURE;
    }
    if ((bits = wolfSSL_EC_GROUP_order_bits(group)) == 0) {
        /* group is not set */
        return WOLFSSL_FAILURE;
    }

    bytes = (bits + 7) / 8;  /* bytes needed to hold bits */
    return headerSz +
            ECC_MAX_PAD_SZ + /* possible leading zeroes in r and s */
            bytes + bytes;   /* r and s */
}

int wolfSSL_ECDSA_sign(int type,
    const unsigned char *digest, int digestSz,
    unsigned char *sig, unsigned int *sigSz, WOLFSSL_EC_KEY *key)
{
    int ret = WOLFSSL_SUCCESS;
    WC_RNG* rng = NULL;
#ifdef WOLFSSL_SMALL_STACK
    WC_RNG* tmpRNG = NULL;
#else
    WC_RNG  tmpRNG[1];
#endif
    int initTmpRng = 0;

    WOLFSSL_ENTER("wolfSSL_ECDSA_sign");

    if (!key) {
        return WOLFSSL_FAILURE;
    }

#ifdef WOLFSSL_SMALL_STACK
    tmpRNG = (WC_RNG*)XMALLOC(sizeof(WC_RNG), NULL, DYNAMIC_TYPE_RNG);
    if (tmpRNG == NULL)
        return WOLFSSL_FAILURE;
#endif

    if (wc_InitRng(tmpRNG) == 0) {
        rng = tmpRNG;
        initTmpRng = 1;
    }
    else {
        WOLFSSL_MSG("Bad RNG Init, trying global");
        if (initGlobalRNG == 0) {
            WOLFSSL_MSG("Global RNG no Init");
        }
        else {
            rng = &globalRNG;
        }
    }
    if (rng) {
        if (wc_ecc_sign_hash(digest, digestSz, sig, sigSz, rng,
                (ecc_key*)key->internal) != 0) {
            ret = WOLFSSL_FAILURE;
        }
        if (initTmpRng) {
            wc_FreeRng(tmpRNG);
        }
    } else {
        ret = WOLFSSL_FAILURE;
    }

#ifdef WOLFSSL_SMALL_STACK
    if (tmpRNG)
        XFREE(tmpRNG, NULL, DYNAMIC_TYPE_RNG);
#endif

    (void)type;
    return ret;
}

int wolfSSL_ECDSA_verify(int type,
    const unsigned char *digest, int digestSz,
    const unsigned char *sig, int sigSz, WOLFSSL_EC_KEY *key)
{
    int ret = WOLFSSL_SUCCESS;
    int verify = 0;

    WOLFSSL_ENTER("wolfSSL_ECDSA_verify");

    if (key == NULL) {
        return WOLFSSL_FAILURE;
    }

    if (wc_ecc_verify_hash(sig, sigSz, digest, digestSz,
            &verify, (ecc_key*)key->internal) != 0) {
        ret = WOLFSSL_FAILURE;
    }
    if (ret == WOLFSSL_SUCCESS && verify != 1) {
        WOLFSSL_MSG("wolfSSL_ECDSA_verify failed");
        ret = WOLFSSL_FAILURE;
    }

    (void)type;
    return ret;
}

#ifndef HAVE_SELFTEST
/* ECC point compression types were not included in selftest ecc.h */

char* wolfSSL_EC_POINT_point2hex(const WOLFSSL_EC_GROUP* group,
                                 const WOLFSSL_EC_POINT* point, int form,
                                 WOLFSSL_BN_CTX* ctx)
{
    static const char* hexDigit = "0123456789ABCDEF";
    char* hex = NULL;
    int id;
    int i, sz, len;

    (void)ctx;

    if (group == NULL || point == NULL)
        return NULL;

    id = wc_ecc_get_curve_id(group->curve_idx);

    if ((sz = wc_ecc_get_curve_size_from_id(id)) < 0)
        return NULL;

    len = sz + 1;
    if (form == POINT_CONVERSION_UNCOMPRESSED)
        len += sz;

    hex = (char*)XMALLOC(2 * len + 1, NULL, DYNAMIC_TYPE_ECC);
    if (hex == NULL)
        return NULL;
    XMEMSET(hex, 0, 2 * len + 1);

    /* Put in x-ordinate after format byte. */
    i = sz - mp_unsigned_bin_size((mp_int*)point->X->internal) + 1;
    if (mp_to_unsigned_bin((mp_int*)point->X->internal, (byte*)(hex + i)) < 0) {
        XFREE(hex,  NULL, DYNAMIC_TYPE_ECC);
        return NULL;
    }

    if (form == POINT_CONVERSION_COMPRESSED) {
        hex[0] = mp_isodd((mp_int*)point->Y->internal) ? ECC_POINT_COMP_ODD :
                                                         ECC_POINT_COMP_EVEN;
    }
    else {
        hex[0] = ECC_POINT_UNCOMP;
        /* Put in y-ordinate after x-ordinate */
        i = 1 + 2 * sz - mp_unsigned_bin_size((mp_int*)point->Y->internal);
        if (mp_to_unsigned_bin((mp_int*)point->Y->internal,
                                                        (byte*)(hex + i)) < 0) {
            XFREE(hex,  NULL, DYNAMIC_TYPE_ECC);
            return NULL;
        }
    }

    for (i = len-1; i >= 0; i--) {
        byte b = hex[i];
        hex[i * 2 + 1] = hexDigit[b  & 0xf];
        hex[i * 2    ] = hexDigit[b >>   4];
    }

    return hex;
}

#endif /* HAVE_SELFTEST */

void wolfSSL_EC_POINT_dump(const char *msg, const WOLFSSL_EC_POINT *p)
{
#if defined(DEBUG_WOLFSSL)
    char *num;

    WOLFSSL_ENTER("wolfSSL_EC_POINT_dump");

    if (!WOLFSSL_IS_DEBUG_ON() || wolfSSL_GetLoggingCb()) {
        return;
    }

    if (p == NULL) {
        printf("%s = NULL", msg);
        return;
    }

    printf("%s:\n\tinSet=%d, exSet=%d\n", msg, p->inSet, p->exSet);
    num = wolfSSL_BN_bn2hex(p->X);
    printf("\tX = %s\n", num);
    XFREE(num, NULL, DYNAMIC_TYPE_OPENSSL);
    num = wolfSSL_BN_bn2hex(p->Y);
    printf("\tY = %s\n", num);
    XFREE(num, NULL, DYNAMIC_TYPE_OPENSSL);
    num = wolfSSL_BN_bn2hex(p->Z);
    printf("\tZ = %s\n", num);
    XFREE(num, NULL, DYNAMIC_TYPE_OPENSSL);
#else
    (void)msg;
    (void)p;
#endif
}

/* Start EC_GROUP */

/* return code compliant with OpenSSL :
 *   0 if equal, 1 if not and -1 in case of error
 */
int wolfSSL_EC_GROUP_cmp(const WOLFSSL_EC_GROUP *a, const WOLFSSL_EC_GROUP *b,
                         WOLFSSL_BN_CTX *ctx)
{
    (void)ctx;

    WOLFSSL_ENTER("wolfSSL_EC_GROUP_cmp");

    if (a == NULL || b == NULL) {
        WOLFSSL_MSG("wolfSSL_EC_GROUP_cmp Bad arguments");
        return WOLFSSL_FATAL_ERROR;
    }

    /* ok */
    if ((a->curve_idx == b->curve_idx) && (a->curve_nid == b->curve_nid))
        return 0;

    /* ko */
    return 1;
}

WOLFSSL_EC_GROUP *wolfSSL_EC_GROUP_dup(const WOLFSSL_EC_GROUP *src)
{
    if (!src)
        return NULL;
    return wolfSSL_EC_GROUP_new_by_curve_name(src->curve_nid);
}

#endif /* HAVE_ECC */
#endif /* OPENSSL_EXTRA */

#if defined(HAVE_ECC) && (defined(OPENSSL_EXTRA) || defined(OPENSSL_EXTRA_X509_SMALL))
const WOLFSSL_EC_METHOD* wolfSSL_EC_GROUP_method_of(
                                                const WOLFSSL_EC_GROUP *group)
{
    return group;
}

int wolfSSL_EC_METHOD_get_field_type(const WOLFSSL_EC_METHOD *meth)
{
    if (meth) {
        return NID_X9_62_prime_field;
    }
    return WOLFSSL_FAILURE;
}

void wolfSSL_EC_GROUP_free(WOLFSSL_EC_GROUP *group)
{
    WOLFSSL_ENTER("wolfSSL_EC_GROUP_free");

    XFREE(group, NULL, DYNAMIC_TYPE_ECC);
    /* group = NULL, don't try to access or double free it */
}
#endif

#ifdef OPENSSL_EXTRA
#ifdef HAVE_ECC
#ifndef NO_WOLFSSL_STUB
void wolfSSL_EC_GROUP_set_asn1_flag(WOLFSSL_EC_GROUP *group, int flag)
{
    (void)group;
    (void)flag;

    WOLFSSL_ENTER("wolfSSL_EC_GROUP_set_asn1_flag");
    WOLFSSL_STUB("EC_GROUP_set_asn1_flag");
}
#endif

WOLFSSL_EC_GROUP *wolfSSL_EC_GROUP_new_by_curve_name(int nid)
{
    WOLFSSL_EC_GROUP *g;
    int x, eccEnum;

    WOLFSSL_ENTER("wolfSSL_EC_GROUP_new_by_curve_name");

    /* curve group */
    g = (WOLFSSL_EC_GROUP*)XMALLOC(sizeof(WOLFSSL_EC_GROUP), NULL,
                                    DYNAMIC_TYPE_ECC);
    if (g == NULL) {
        WOLFSSL_MSG("wolfSSL_EC_GROUP_new_by_curve_name malloc failure");
        return NULL;
    }
    XMEMSET(g, 0, sizeof(WOLFSSL_EC_GROUP));

    /* set the nid of the curve */
    g->curve_nid = nid;
    g->curve_idx = -1;

    /* If NID passed in is OpenSSL type, convert it to ecc_curve_id enum */
    eccEnum = NIDToEccEnum(nid);
    if (eccEnum != -1) {
        /* search and set the corresponding internal curve idx */
        for (x = 0; ecc_sets[x].size != 0; x++) {
            if (ecc_sets[x].id == eccEnum) {
                g->curve_idx = x;
                g->curve_oid = ecc_sets[x].oidSum;
                break;
            }
        }
    }

    return g;
}

/* return code compliant with OpenSSL :
 *   the curve nid if success, 0 if error
 */
int wolfSSL_EC_GROUP_get_curve_name(const WOLFSSL_EC_GROUP *group)
{
    int nid;
    WOLFSSL_ENTER("wolfSSL_EC_GROUP_get_curve_name");

    if (group == NULL) {
        WOLFSSL_MSG("wolfSSL_EC_GROUP_get_curve_name Bad arguments");
        return WOLFSSL_FAILURE;
    }

    /* If curve_nid is ECC Enum type, return corresponding OpenSSL nid */
    if ((nid = EccEnumToNID(group->curve_nid)) != -1)
        return nid;

    return group->curve_nid;
}

/* return code compliant with OpenSSL :
 *   the degree of the curve if success, 0 if error
 */
int wolfSSL_EC_GROUP_get_degree(const WOLFSSL_EC_GROUP *group)
{
    int nid;
    int tmp;

    WOLFSSL_ENTER("wolfSSL_EC_GROUP_get_degree");

    if (group == NULL || group->curve_idx < 0) {
        WOLFSSL_MSG("wolfSSL_EC_GROUP_get_degree Bad arguments");
        return WOLFSSL_FAILURE;
    }

    /* If curve_nid passed in is an ecc_curve_id enum, convert it to the
        corresponding OpenSSL NID */
    tmp = EccEnumToNID(group->curve_nid);
    if (tmp != -1){
        nid = tmp;
    }
    else{
        nid = group->curve_nid;
    }

    switch(nid) {
        case NID_secp112r1:
        case NID_secp112r2:
            return 112;
        case NID_secp128r1:
        case NID_secp128r2:
            return 128;
        case NID_secp160k1:
        case NID_secp160r1:
        case NID_secp160r2:
        case NID_brainpoolP160r1:
            return 160;
        case NID_secp192k1:
        case NID_brainpoolP192r1:
        case NID_X9_62_prime192v1:
            return 192;
        case NID_secp224k1:
        case NID_secp224r1:
        case NID_brainpoolP224r1:
            return 224;
        case NID_secp256k1:
        case NID_brainpoolP256r1:
        case NID_X9_62_prime256v1:
            return 256;
        case NID_brainpoolP320r1:
            return 320;
        case NID_secp384r1:
        case NID_brainpoolP384r1:
            return 384;
        case NID_secp521r1:
            return 521;
        case NID_brainpoolP512r1:
            return 512;
        default:
            return WOLFSSL_FAILURE;
    }
}

/* Converts OpenSSL NID value of ECC curves to the associated enum values in
   ecc_curve_id, used by ecc_sets[].*/
int NIDToEccEnum(int n)
{
    WOLFSSL_ENTER("NIDToEccEnum()");

    switch(n) {
        case NID_X9_62_prime192v1:
            return ECC_SECP192R1;
        case NID_X9_62_prime192v2:
            return ECC_PRIME192V2;
        case NID_X9_62_prime192v3:
            return ECC_PRIME192V3;
        case NID_X9_62_prime239v1:
            return ECC_PRIME239V1;
        case NID_X9_62_prime239v2:
            return ECC_PRIME239V2;
        case NID_X9_62_prime239v3:
            return ECC_PRIME239V3;
        case NID_X9_62_prime256v1:
            return ECC_SECP256R1;
        case NID_secp112r1:
            return ECC_SECP112R1;
        case NID_secp112r2:
            return ECC_SECP112R2;
        case NID_secp128r1:
            return ECC_SECP128R1;
        case NID_secp128r2:
            return ECC_SECP128R2;
        case NID_secp160r1:
            return ECC_SECP160R1;
        case NID_secp160r2:
            return ECC_SECP160R2;
        case NID_secp224r1:
            return ECC_SECP224R1;
        case NID_secp384r1:
            return ECC_SECP384R1;
        case NID_secp521r1:
            return ECC_SECP521R1;
        case NID_secp160k1:
            return ECC_SECP160K1;
        case NID_secp192k1:
            return ECC_SECP192K1;
        case NID_secp224k1:
            return ECC_SECP224K1;
        case NID_secp256k1:
            return ECC_SECP256K1;
        case NID_brainpoolP160r1:
            return ECC_BRAINPOOLP160R1;
        case NID_brainpoolP192r1:
            return ECC_BRAINPOOLP192R1;
        case NID_brainpoolP224r1:
            return ECC_BRAINPOOLP224R1;
        case NID_brainpoolP256r1:
            return ECC_BRAINPOOLP256R1;
        case NID_brainpoolP320r1:
            return ECC_BRAINPOOLP320R1;
        case NID_brainpoolP384r1:
            return ECC_BRAINPOOLP384R1;
        case NID_brainpoolP512r1:
            return ECC_BRAINPOOLP512R1;
        default:
            WOLFSSL_MSG("NID not found");
            return -1;
    }
}

/* return code compliant with OpenSSL :
 *   1 if success, 0 if error
 */
int wolfSSL_EC_GROUP_get_order(const WOLFSSL_EC_GROUP *group,
                               WOLFSSL_BIGNUM *order, WOLFSSL_BN_CTX *ctx)
{
    (void)ctx;

    if (group == NULL || order == NULL || order->internal == NULL) {
        WOLFSSL_MSG("wolfSSL_EC_GROUP_get_order NULL error");
        return WOLFSSL_FAILURE;
    }

    if (mp_init((mp_int*)order->internal) != MP_OKAY) {
        WOLFSSL_MSG("wolfSSL_EC_GROUP_get_order mp_init failure");
        return WOLFSSL_FAILURE;
    }

    if (mp_read_radix((mp_int*)order->internal,
                  ecc_sets[group->curve_idx].order, MP_RADIX_HEX) != MP_OKAY) {
        WOLFSSL_MSG("wolfSSL_EC_GROUP_get_order mp_read order failure");
        mp_clear((mp_int*)order->internal);
        return WOLFSSL_FAILURE;
    }

    return WOLFSSL_SUCCESS;
}

int wolfSSL_EC_GROUP_order_bits(const WOLFSSL_EC_GROUP *group)
{
    int ret;
    mp_int order;

    if (group == NULL || group->curve_idx < 0) {
        WOLFSSL_MSG("wolfSSL_EC_GROUP_order_bits NULL error");
        return 0;
    }

    ret = mp_init(&order);
    if (ret == 0) {
        ret = mp_read_radix(&order, ecc_sets[group->curve_idx].order,
            MP_RADIX_HEX);
        if (ret == 0)
            ret = mp_count_bits(&order);
        mp_clear(&order);
    }

    return ret;
}

/* End EC_GROUP */

/* Start EC_POINT */

/* return code compliant with OpenSSL :
 *   1 if success, 0 if error
 */
int wolfSSL_ECPoint_i2d(const WOLFSSL_EC_GROUP *group,
                        const WOLFSSL_EC_POINT *p,
                        unsigned char *out, unsigned int *len)
{
    int err;

    WOLFSSL_ENTER("wolfSSL_ECPoint_i2d");

    if (group == NULL || p == NULL || len == NULL) {
        WOLFSSL_MSG("wolfSSL_ECPoint_i2d NULL error");
        return WOLFSSL_FAILURE;
    }

    if (setupPoint(p) != WOLFSSL_SUCCESS) {
        return WOLFSSL_FAILURE;
    }

    if (out != NULL) {
        wolfSSL_EC_POINT_dump("i2d p", p);
    }

    err = wc_ecc_export_point_der(group->curve_idx, (ecc_point*)p->internal,
                                  out, len);
    if (err != MP_OKAY && !(out == NULL && err == LENGTH_ONLY_E)) {
        WOLFSSL_MSG("wolfSSL_ECPoint_i2d wc_ecc_export_point_der failed");
        return WOLFSSL_FAILURE;
    }

    return WOLFSSL_SUCCESS;
}

/* return code compliant with OpenSSL :
 *   1 if success, 0 if error
 */
int wolfSSL_ECPoint_d2i(unsigned char *in, unsigned int len,
                        const WOLFSSL_EC_GROUP *group, WOLFSSL_EC_POINT *p)
{
    WOLFSSL_ENTER("wolfSSL_ECPoint_d2i");

    if (group == NULL || p == NULL || p->internal == NULL || in == NULL) {
        WOLFSSL_MSG("wolfSSL_ECPoint_d2i NULL error");
        return WOLFSSL_FAILURE;
    }

#if !defined(HAVE_SELFTEST) && (!defined(HAVE_FIPS) || \
    (defined(HAVE_FIPS_VERSION) && (HAVE_FIPS_VERSION>2)))
    if (wc_ecc_import_point_der_ex(in, len, group->curve_idx,
                                   (ecc_point*)p->internal, 0) != MP_OKAY) {
        WOLFSSL_MSG("wc_ecc_import_point_der_ex failed");
        return WOLFSSL_FAILURE;
    }
#else
    /* ECC_POINT_UNCOMP is not defined CAVP self test so use magic number */
    if (in[0] == 0x04) {
        if (wc_ecc_import_point_der(in, len, group->curve_idx,
                                    (ecc_point*)p->internal) != MP_OKAY) {
            WOLFSSL_MSG("wc_ecc_import_point_der failed");
            return WOLFSSL_FAILURE;
        }
    }
    else {
        WOLFSSL_MSG("Only uncompressed points supported with HAVE_SELFTEST");
        return WOLFSSL_FAILURE;
    }
#endif

    /* Set new external point */
    if (SetECPointExternal(p) != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("SetECPointExternal failed");
        return WOLFSSL_FAILURE;
    }

    wolfSSL_EC_POINT_dump("d2i p", p);

    return WOLFSSL_SUCCESS;
}

size_t wolfSSL_EC_POINT_point2oct(const WOLFSSL_EC_GROUP *group,
                                  const WOLFSSL_EC_POINT *p,
                                  char form,
                                  byte *buf, size_t len, WOLFSSL_BN_CTX *ctx)
{
    word32 min_len = (word32)len;
#if !defined(HAVE_SELFTEST) && (!defined(HAVE_FIPS) || \
    (defined(HAVE_FIPS_VERSION) && (HAVE_FIPS_VERSION > 2)))
    int compressed = form == POINT_CONVERSION_COMPRESSED ? 1 : 0;
#endif /* !HAVE_SELFTEST */

    WOLFSSL_ENTER("EC_POINT_point2oct");

    if (!group || !p) {
        return WOLFSSL_FAILURE;
    }

    if (setupPoint(p) != WOLFSSL_SUCCESS) {
        return WOLFSSL_FAILURE;
    }

    if (wolfSSL_EC_POINT_is_at_infinity(group, p)) {
        /* encodes to a single 0 octet */
        if (buf != NULL) {
            if (len < 1) {
                ECerr(EC_F_EC_GFP_SIMPLE_POINT2OCT, EC_R_BUFFER_TOO_SMALL);
                return WOLFSSL_FAILURE;
            }
            buf[0] = 0;
        }
        return 1;
    }

    if (form != POINT_CONVERSION_UNCOMPRESSED
#ifndef HAVE_SELFTEST
            && form != POINT_CONVERSION_COMPRESSED
#endif /* !HAVE_SELFTEST */
            ) {
        WOLFSSL_MSG("Unsupported curve form");
        return WOLFSSL_FAILURE;
    }

#if !defined(HAVE_SELFTEST) && (!defined(HAVE_FIPS) || \
    (defined(HAVE_FIPS_VERSION) && (HAVE_FIPS_VERSION > 2)))
    if (wc_ecc_export_point_der_ex(group->curve_idx, (ecc_point*)p->internal,
               buf, &min_len, compressed) != (buf ? MP_OKAY : LENGTH_ONLY_E)) {
        return WOLFSSL_FAILURE;
    }
#else
    if (wc_ecc_export_point_der(group->curve_idx, (ecc_point*)p->internal,
                                buf, &min_len) != (buf ? MP_OKAY : LENGTH_ONLY_E)) {
        return WOLFSSL_FAILURE;
    }
#endif /* !HAVE_SELFTEST */

    (void)ctx;

    return (size_t)min_len;
}


int wolfSSL_EC_POINT_oct2point(const WOLFSSL_EC_GROUP *group,
                               WOLFSSL_EC_POINT *p, const unsigned char *buf,
                               size_t len, WOLFSSL_BN_CTX *ctx)
{
    WOLFSSL_ENTER("wolfSSL_EC_POINT_oct2point");

    if (!group || !p) {
        return WOLFSSL_FAILURE;
    }

    (void)ctx;

    return wolfSSL_ECPoint_d2i((unsigned char*)buf, (unsigned int)len, group, p);
}


WOLFSSL_EC_KEY *wolfSSL_o2i_ECPublicKey(WOLFSSL_EC_KEY **a, const unsigned char **in,
                                        long len)
{
    WOLFSSL_EC_KEY* ret;

    WOLFSSL_ENTER("wolfSSL_o2i_ECPublicKey");

    if (!a || !*a || !(*a)->group || !in || !*in || len <= 0) {
        WOLFSSL_MSG("wolfSSL_o2i_ECPublicKey Bad arguments");
        return NULL;
    }

    ret = *a;

    if (wolfSSL_EC_POINT_oct2point(ret->group, ret->pub_key, *in, len, NULL)
            != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("wolfSSL_EC_POINT_oct2point error");
        return NULL;
    }

    *in += len;
    return ret;
}

int wolfSSL_i2o_ECPublicKey(const WOLFSSL_EC_KEY *in, unsigned char **out)
{
    size_t len;
    unsigned char *tmp = NULL;
    char form;
    WOLFSSL_ENTER("wolfSSL_i2o_ECPublicKey");

    if (!in) {
        WOLFSSL_MSG("wolfSSL_i2o_ECPublicKey Bad arguments");
        return WOLFSSL_FAILURE;
    }

    if (!in->exSet) {
        if (SetECKeyExternal((WOLFSSL_EC_KEY*)in) != WOLFSSL_SUCCESS) {
            WOLFSSL_MSG("SetECKeyExternal failure");
            return WOLFSSL_FAILURE;
        }
    }

#ifdef HAVE_COMP_KEY
    /* Default to compressed form if not set */
    form = in->form == POINT_CONVERSION_UNCOMPRESSED ?
            POINT_CONVERSION_UNCOMPRESSED:
            POINT_CONVERSION_COMPRESSED;
#else
    form = POINT_CONVERSION_UNCOMPRESSED;
#endif

    len = wolfSSL_EC_POINT_point2oct(in->group, in->pub_key, form,
                                     NULL, 0, NULL);

    if (len != WOLFSSL_FAILURE && out) {
        if (!*out) {
            if (!(tmp = (unsigned char*)XMALLOC(len, NULL,
                                                DYNAMIC_TYPE_OPENSSL))) {
                WOLFSSL_MSG("malloc failed");
                return WOLFSSL_FAILURE;
            }
            *out = tmp;
        }

        if (wolfSSL_EC_POINT_point2oct(in->group, in->pub_key, form, *out,
                                       len, NULL) == WOLFSSL_FAILURE) {
            if (tmp) {
                XFREE(tmp, NULL, DYNAMIC_TYPE_OPENSSL);
                *out = NULL;
            }
            return WOLFSSL_FAILURE;
        }

        if (!tmp) {
            /* Move buffer forward if it was not alloced in this function */
            *out += len;
        }
    }

    return (int)len;
}

#ifdef HAVE_ECC_KEY_IMPORT
WOLFSSL_EC_KEY *wolfSSL_d2i_ECPrivateKey(WOLFSSL_EC_KEY **key, const unsigned char **in,
                                         long len)
{
    word32 idx = 0;
    WOLFSSL_EC_KEY *eckey = NULL;
    WOLFSSL_ENTER("wolfSSL_d2i_ECPrivateKey");

    if (!in || !*in || len <= 0) {
        WOLFSSL_MSG("wolfSSL_d2i_ECPrivateKey Bad arguments");
        return NULL;
    }

    if (!(eckey = wolfSSL_EC_KEY_new())) {
        WOLFSSL_MSG("wolfSSL_EC_KEY_new error");
        return NULL;
    }

    if (wc_EccPrivateKeyDecode(*in, &idx, (ecc_key*)eckey->internal,
            (word32)len) != 0) {
        WOLFSSL_MSG("wc_EccPrivateKeyDecode error");
        goto error;
    }

    eckey->inSet = 1;

    if (SetECKeyExternal(eckey) != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("SetECKeyExternal error");
        goto error;
    }

    if (key) {
        *key = eckey;
    }

    return eckey;

error:
    wolfSSL_EC_KEY_free(eckey);
    return NULL;
}
#endif /* HAVE_ECC_KEY_IMPORT */

int wolfSSL_i2d_ECPrivateKey(const WOLFSSL_EC_KEY *in, unsigned char **out)
{
    word32 len;
    byte* buf = NULL;
    WOLFSSL_ENTER("wolfSSL_i2d_ECPrivateKey");

    if (!in) {
        WOLFSSL_MSG("wolfSSL_i2d_ECPrivateKey Bad arguments");
        return WOLFSSL_FAILURE;
    }

    if (!in->inSet && SetECKeyInternal(
            (WOLFSSL_EC_KEY*)in) != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("SetECKeyInternal error");
        return WOLFSSL_FAILURE;
    }

    if ((len = wc_EccKeyDerSize((ecc_key*)in->internal, 0)) <= 0) {
        WOLFSSL_MSG("wc_EccKeyDerSize error");
        return WOLFSSL_FAILURE;
    }

    if (out) {
        if (!(buf = (byte*)XMALLOC(len, NULL, DYNAMIC_TYPE_TMP_BUFFER))) {
            WOLFSSL_MSG("tmp buffer malloc error");
            return WOLFSSL_FAILURE;
        }

        if (wc_EccPrivateKeyToDer((ecc_key*)in->internal, buf, len) < 0) {
            WOLFSSL_MSG("wc_EccPrivateKeyToDer error");
            XFREE(buf, NULL, DYNAMIC_TYPE_TMP_BUFFER);
            return WOLFSSL_FAILURE;
        }

        if (*out) {
            XMEMCPY(*out, buf, len);
            XFREE(buf, NULL, DYNAMIC_TYPE_TMP_BUFFER);
        }
        else {
            *out = buf;
        }
    }

    return (int)len;
}

void wolfSSL_EC_KEY_set_conv_form(WOLFSSL_EC_KEY *eckey, char form)
{
    if (eckey && (form == POINT_CONVERSION_UNCOMPRESSED
#ifdef HAVE_COMP_KEY
                  || form == POINT_CONVERSION_COMPRESSED
#endif
                  )) {
        eckey->form = form;
    } else {
        WOLFSSL_MSG("Incorrect form or HAVE_COMP_KEY not compiled in");
    }
}

point_conversion_form_t wolfSSL_EC_KEY_get_conv_form(const WOLFSSL_EC_KEY* key)
{
    if (key != NULL) {
        return key->form;
    }

    return -1;
}

/* wolfSSL_EC_POINT_point2bn should return "in" if not null */
WOLFSSL_BIGNUM *wolfSSL_EC_POINT_point2bn(const WOLFSSL_EC_GROUP *group,
                                          const WOLFSSL_EC_POINT *p,
                                          char form,
                                          WOLFSSL_BIGNUM *in, WOLFSSL_BN_CTX *ctx)
{
    size_t len;
    byte *buf;
    WOLFSSL_BIGNUM *ret = NULL;

    WOLFSSL_ENTER("wolfSSL_EC_POINT_oct2point");

    if (!group || !p) {
        return NULL;
    }

    if ((len = wolfSSL_EC_POINT_point2oct(group, p, form,
                                          NULL, 0, ctx)) == WOLFSSL_FAILURE) {
        return NULL;
    }

    if (!(buf = (byte*)XMALLOC(len, NULL, DYNAMIC_TYPE_TMP_BUFFER))) {
        WOLFSSL_MSG("malloc failed");
        return NULL;
    }

    if (wolfSSL_EC_POINT_point2oct(group, p, form,
                                   buf, len, ctx) == len) {
        ret = wolfSSL_BN_bin2bn(buf, (int)len, in);
    }

    XFREE(buf, NULL, DYNAMIC_TYPE_TMP_BUFFER);

    return ret;
}

#if defined(USE_ECC_B_PARAM) && (!defined(HAVE_FIPS) || \
    (defined(HAVE_FIPS_VERSION) && (HAVE_FIPS_VERSION > 2)))
int wolfSSL_EC_POINT_is_on_curve(const WOLFSSL_EC_GROUP *group,
                                 const WOLFSSL_EC_POINT *point,
                                 WOLFSSL_BN_CTX *ctx)
{
    (void)ctx;
    WOLFSSL_ENTER("wolfSSL_EC_POINT_is_on_curve");

    if (!group || !point) {
        WOLFSSL_MSG("Invalid arguments");
        return WOLFSSL_FAILURE;
    }

    if (!point->inSet && SetECPointInternal((WOLFSSL_EC_POINT*)point)) {
        WOLFSSL_MSG("SetECPointInternal error");
        return WOLFSSL_FAILURE;
    }

    return wc_ecc_point_is_on_curve((ecc_point*)point->internal, group->curve_idx)
            == MP_OKAY ? WOLFSSL_SUCCESS : WOLFSSL_FAILURE;
}
#endif /* USE_ECC_B_PARAM && (!HAVE_FIPS || HAVE_FIPS_VERSION > 2) */

WOLFSSL_EC_POINT *wolfSSL_EC_POINT_new(const WOLFSSL_EC_GROUP *group)
{
    WOLFSSL_EC_POINT *p;

    WOLFSSL_ENTER("wolfSSL_EC_POINT_new");

    if (group == NULL) {
        WOLFSSL_MSG("wolfSSL_EC_POINT_new NULL error");
        return NULL;
    }

    p = (WOLFSSL_EC_POINT *)XMALLOC(sizeof(WOLFSSL_EC_POINT), NULL,
                                    DYNAMIC_TYPE_ECC);
    if (p == NULL) {
        WOLFSSL_MSG("wolfSSL_EC_POINT_new malloc ecc point failure");
        return NULL;
    }
    XMEMSET(p, 0, sizeof(WOLFSSL_EC_POINT));

    p->internal = wc_ecc_new_point();
    if (p->internal == NULL) {
        WOLFSSL_MSG("ecc_new_point failure");
        XFREE(p, NULL, DYNAMIC_TYPE_ECC);
        return NULL;
    }

    return p;
}

#ifndef WOLFSSL_SP_MATH
/* return code compliant with OpenSSL :
 *   1 if success, 0 if error
 */
int wolfSSL_EC_POINT_get_affine_coordinates_GFp(const WOLFSSL_EC_GROUP *group,
                                                const WOLFSSL_EC_POINT *point,
                                                WOLFSSL_BIGNUM *x,
                                                WOLFSSL_BIGNUM *y,
                                                WOLFSSL_BN_CTX *ctx)
{
    mp_digit mp;
#ifdef WOLFSSL_SMALL_STACK
    mp_int* modulus = NULL;
#else
    mp_int modulus[1];
#endif
    (void)ctx;

    WOLFSSL_ENTER("wolfSSL_EC_POINT_get_affine_coordinates_GFp");

    if (group == NULL || point == NULL || point->internal == NULL ||
        x == NULL || y == NULL || wolfSSL_EC_POINT_is_at_infinity(group, point)) {
        WOLFSSL_MSG("wolfSSL_EC_POINT_get_affine_coordinates_GFp NULL error");
        return WOLFSSL_FAILURE;
    }

    if (setupPoint(point) != WOLFSSL_SUCCESS) {
        return WOLFSSL_FAILURE;
    }

#ifdef WOLFSSL_SMALL_STACK
    modulus = (mp_int*)XMALLOC(sizeof(mp_int), NULL, DYNAMIC_TYPE_BIGINT);
    if (modulus == NULL) {
        return WOLFSSL_FAILURE;
    }
#endif

    if (!wolfSSL_BN_is_one(point->Z)) {
        if (mp_init(modulus) != MP_OKAY) {
            WOLFSSL_MSG("mp_init failed");
        #ifdef WOLFSSL_SMALL_STACK
            XFREE(modulus, NULL, DYNAMIC_TYPE_BIGINT);
        #endif
            return WOLFSSL_FAILURE;
        }
        /* Map the Jacobian point back to affine space */
        if (mp_read_radix(modulus, ecc_sets[group->curve_idx].prime, MP_RADIX_HEX) != MP_OKAY) {
            WOLFSSL_MSG("mp_read_radix failed");
            mp_clear(modulus);
        #ifdef WOLFSSL_SMALL_STACK
            XFREE(modulus, NULL, DYNAMIC_TYPE_BIGINT);
        #endif
            return WOLFSSL_FAILURE;
        }
        if (mp_montgomery_setup(modulus, &mp) != MP_OKAY) {
            WOLFSSL_MSG("mp_montgomery_setup failed");
            mp_clear(modulus);
        #ifdef WOLFSSL_SMALL_STACK
            XFREE(modulus, NULL, DYNAMIC_TYPE_BIGINT);
        #endif
            return WOLFSSL_FAILURE;
        }
        if (ecc_map((ecc_point*)point->internal, modulus, mp) != MP_OKAY) {
            WOLFSSL_MSG("ecc_map failed");
            mp_clear(modulus);
        #ifdef WOLFSSL_SMALL_STACK
            XFREE(modulus, NULL, DYNAMIC_TYPE_BIGINT);
        #endif
            return WOLFSSL_FAILURE;
        }
        if (SetECPointExternal((WOLFSSL_EC_POINT *)point) != WOLFSSL_SUCCESS) {
            WOLFSSL_MSG("SetECPointExternal failed");
            mp_clear(modulus);
        #ifdef WOLFSSL_SMALL_STACK
            XFREE(modulus, NULL, DYNAMIC_TYPE_BIGINT);
        #endif
            return WOLFSSL_FAILURE;
        }

        mp_clear(modulus);
    }

    BN_copy(x, point->X);
    BN_copy(y, point->Y);

#ifdef WOLFSSL_SMALL_STACK
    XFREE(modulus, NULL, DYNAMIC_TYPE_BIGINT);
#endif

    return WOLFSSL_SUCCESS;
}
#endif

int wolfSSL_EC_POINT_set_affine_coordinates_GFp(const WOLFSSL_EC_GROUP *group,
                                                WOLFSSL_EC_POINT *point,
                                                const WOLFSSL_BIGNUM *x,
                                                const WOLFSSL_BIGNUM *y,
                                                WOLFSSL_BN_CTX *ctx)
{
    (void)ctx;
    WOLFSSL_ENTER("wolfSSL_EC_POINT_set_affine_coordinates_GFp");

    if (group == NULL || point == NULL || point->internal == NULL ||
        x == NULL || y == NULL) {
        WOLFSSL_MSG("wolfSSL_EC_POINT_set_affine_coordinates_GFp NULL error");
        return WOLFSSL_FAILURE;
    }

    if (!point->X) {
        point->X = wolfSSL_BN_new();
    }
    if (!point->Y) {
        point->Y = wolfSSL_BN_new();
    }
    if (!point->Z) {
        point->Z = wolfSSL_BN_new();
    }
    if (!point->X || !point->Y || !point->Z) {
        WOLFSSL_MSG("wolfSSL_BN_new failed");
        return WOLFSSL_FAILURE;
    }

    BN_copy(point->X, x);
    BN_copy(point->Y, y);
    BN_copy(point->Z, wolfSSL_BN_value_one());

    if (SetECPointInternal((WOLFSSL_EC_POINT *)point) != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("SetECPointInternal failed");
        return WOLFSSL_FAILURE;
    }

    return WOLFSSL_SUCCESS;
}

#if !defined(WOLFSSL_ATECC508A) && !defined(WOLFSSL_ATECC608A) && \
    !defined(HAVE_SELFTEST) && !defined(WOLFSSL_SP_MATH)
int wolfSSL_EC_POINT_add(const WOLFSSL_EC_GROUP *group, WOLFSSL_EC_POINT *r,
                         const WOLFSSL_EC_POINT *p1,
                         const WOLFSSL_EC_POINT *p2, WOLFSSL_BN_CTX *ctx)
{
#ifdef WOLFSSL_SMALL_STACK
    mp_int* a = NULL;
    mp_int* prime = NULL;
    mp_int* mu = NULL;
#else
    mp_int a[1];
    mp_int prime[1];
    mp_int mu[1];
#endif
    mp_digit mp = 0;
    ecc_point* montP1 = NULL;
    ecc_point* montP2 = NULL;
    ecc_point* eccP1;
    ecc_point* eccP2;
    int ret = WOLFSSL_FAILURE;

    (void)ctx;

    if (!group || !r || !p1 || !p2) {
        WOLFSSL_MSG("wolfSSL_EC_POINT_add error");
        return WOLFSSL_FAILURE;
    }

    if (setupPoint(r) != WOLFSSL_SUCCESS ||
        setupPoint(p1) != WOLFSSL_SUCCESS ||
        setupPoint(p2) != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("setupPoint error");
        return WOLFSSL_FAILURE;
    }

#ifdef WOLFSSL_SMALL_STACK
    a = (mp_int*)XMALLOC(sizeof(mp_int), NULL, DYNAMIC_TYPE_BIGINT);
    if (a == NULL) {
        WOLFSSL_MSG("Failed to allocate memory for mp_int a");
        return WOLFSSL_FAILURE;
    }
    prime = (mp_int*)XMALLOC(sizeof(mp_int), NULL, DYNAMIC_TYPE_BIGINT);
    if (prime == NULL) {
        WOLFSSL_MSG("Failed to allocate memory for mp_int prime");
        XFREE(a, NULL, DYNAMIC_TYPE_BIGINT);
        return WOLFSSL_FAILURE;
    }
    mu = (mp_int*)XMALLOC(sizeof(mp_int), NULL, DYNAMIC_TYPE_BIGINT);
    if (mu == NULL) {
        WOLFSSL_MSG("Failed to allocate memory for mp_int mu");
        XFREE(a, NULL, DYNAMIC_TYPE_BIGINT);
        XFREE(prime, NULL, DYNAMIC_TYPE_BIGINT);
        return WOLFSSL_FAILURE;
    }
    XMEMSET(a, 0, sizeof(mp_int));
    XMEMSET(prime, 0, sizeof(mp_int));
    XMEMSET(mu, 0, sizeof(mp_int));
#endif

    /* read the curve prime and a */
    if (mp_init_multi(prime, a, mu, NULL, NULL, NULL) != MP_OKAY) {
        WOLFSSL_MSG("mp_init_multi error");
        goto cleanup;
    }

    if (mp_read_radix(a, ecc_sets[group->curve_idx].Af, MP_RADIX_HEX)
            != MP_OKAY) {
        WOLFSSL_MSG("mp_read_radix a error");
        goto cleanup;
    }

    if (mp_read_radix(prime, ecc_sets[group->curve_idx].prime, MP_RADIX_HEX)
            != MP_OKAY) {
        WOLFSSL_MSG("mp_read_radix prime error");
        goto cleanup;
    }

    if (mp_montgomery_setup(prime, &mp) != MP_OKAY) {
        WOLFSSL_MSG("mp_montgomery_setup nqm error");
        goto cleanup;
    }

    eccP1 = (ecc_point*)p1->internal;
    eccP2 = (ecc_point*)p2->internal;

    if (!(montP1 = wc_ecc_new_point_h(NULL)) ||
            !(montP2 = wc_ecc_new_point_h(NULL))) {
        WOLFSSL_MSG("wc_ecc_new_point_h nqm error");
        goto cleanup;
    }

    if ((mp_montgomery_calc_normalization(mu, prime)) != MP_OKAY) {
        WOLFSSL_MSG("mp_montgomery_calc_normalization error");
        goto cleanup;
    }

    /* Convert to Montgomery form */
    if (mp_cmp_d(mu, 1) == MP_EQ) {
        if (wc_ecc_copy_point(eccP1, montP1) != MP_OKAY ||
                wc_ecc_copy_point(eccP2, montP2) != MP_OKAY) {
            WOLFSSL_MSG("wc_ecc_copy_point error");
            goto cleanup;
        }
    } else {
        if (mp_mulmod(eccP1->x, mu, prime, montP1->x) != MP_OKAY ||
                mp_mulmod(eccP1->y, mu, prime, montP1->y) != MP_OKAY ||
                mp_mulmod(eccP1->z, mu, prime, montP1->z) != MP_OKAY) {
            WOLFSSL_MSG("mp_mulmod error");
            goto cleanup;
        }
        if (mp_mulmod(eccP2->x, mu, prime, montP2->x) != MP_OKAY ||
                mp_mulmod(eccP2->y, mu, prime, montP2->y) != MP_OKAY ||
                mp_mulmod(eccP2->z, mu, prime, montP2->z) != MP_OKAY) {
            WOLFSSL_MSG("mp_mulmod error");
            goto cleanup;
        }
    }

    if (ecc_projective_add_point(montP1, montP2, (ecc_point*)r->internal,
            a, prime, mp) != MP_OKAY) {
        WOLFSSL_MSG("ecc_projective_add_point error");
        goto cleanup;
    }

    if (ecc_map((ecc_point*)r->internal, prime, mp) != MP_OKAY) {
        WOLFSSL_MSG("ecc_map error");
        goto cleanup;
    }

    ret = WOLFSSL_SUCCESS;
cleanup:
    mp_clear(a);
    mp_clear(prime);
    mp_clear(mu);
    wc_ecc_del_point_h(montP1, NULL);
    wc_ecc_del_point_h(montP2, NULL);
#ifdef WOLFSSL_SMALL_STACK
    XFREE(a, NULL, DYNAMIC_TYPE_BIGINT);
    XFREE(prime, NULL, DYNAMIC_TYPE_BIGINT);
    XFREE(mu, NULL, DYNAMIC_TYPE_BIGINT);
#endif
    return ret;
}

/* Calculate the value: generator * n + q * m
 * return code compliant with OpenSSL :
 *   1 if success, 0 if error
 */
int wolfSSL_EC_POINT_mul(const WOLFSSL_EC_GROUP *group, WOLFSSL_EC_POINT *r,
                         const WOLFSSL_BIGNUM *n, const WOLFSSL_EC_POINT *q,
                         const WOLFSSL_BIGNUM *m, WOLFSSL_BN_CTX *ctx)
{
#ifdef WOLFSSL_SMALL_STACK
    mp_int* a = NULL;
    mp_int* prime = NULL;
#else
    mp_int a[1], prime[1];
#endif
    int ret = WOLFSSL_FAILURE;
    ecc_point* result = NULL;
    ecc_point* tmp = NULL;

    (void)ctx;

    WOLFSSL_ENTER("wolfSSL_EC_POINT_mul");

    if (!group || !r) {
        WOLFSSL_MSG("wolfSSL_EC_POINT_mul NULL error");
        return WOLFSSL_FAILURE;
    }

#ifdef WOLFSSL_SMALL_STACK
    a = (mp_int*)XMALLOC(sizeof(mp_int), NULL, DYNAMIC_TYPE_BIGINT);
    if (a == NULL)  {
        return WOLFSSL_FAILURE;
    }
    prime = (mp_int*)XMALLOC(sizeof(mp_int), NULL, DYNAMIC_TYPE_BIGINT);
    if (prime == NULL)  {
        XFREE(a, NULL, DYNAMIC_TYPE_BIGINT);
        return WOLFSSL_FAILURE;
    }
#endif

    if (!(result = wc_ecc_new_point())) {
        WOLFSSL_MSG("wolfSSL_EC_POINT_new error");
        return WOLFSSL_FAILURE;
    }

    /* read the curve prime and a */
    if (mp_init_multi(prime, a, NULL, NULL, NULL, NULL) != MP_OKAY) {
        WOLFSSL_MSG("mp_init_multi error");
        goto cleanup;
    }

    if (q && setupPoint(q) != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("setupPoint error");
        goto cleanup;
    }

    if (mp_read_radix(prime, ecc_sets[group->curve_idx].prime, MP_RADIX_HEX)
            != MP_OKAY) {
        WOLFSSL_MSG("mp_read_radix prime error");
        goto cleanup;
    }

    if (mp_read_radix(a, ecc_sets[group->curve_idx].Af, MP_RADIX_HEX)
            != MP_OKAY) {
        WOLFSSL_MSG("mp_read_radix a error");
        goto cleanup;
    }

    if (n) {
        /* load generator */
    #if !defined(HAVE_FIPS) || (defined(HAVE_FIPS_VERSION) && (HAVE_FIPS_VERSION>2))
        if (wc_ecc_get_generator(result, group->curve_idx)
                != MP_OKAY) {
            WOLFSSL_MSG("wc_ecc_get_generator error");
            goto cleanup;
        }
    #else
        /* wc_ecc_get_generator is not defined in the FIPS v2 module. */
        if (mp_read_radix(result->x, ecc_sets[group->curve_idx].Gx, MP_RADIX_HEX)
                != MP_OKAY) {
            WOLFSSL_MSG("mp_read_radix Gx error");
            goto cleanup;
        }
        if (mp_read_radix(result->y, ecc_sets[group->curve_idx].Gy, MP_RADIX_HEX)
                != MP_OKAY) {
            WOLFSSL_MSG("mp_read_radix Gy error");
            goto cleanup;
        }
        if (mp_set(result->z, 1) != MP_OKAY) {
            WOLFSSL_MSG("mp_set Gz error");
            goto cleanup;
        }
    #endif /* !HAVE_FIPS || HAVE_FIPS_VERSION > 2 */
    }

    if (n && q && m) {
        /* r = generator * n + q * m */
#ifdef ECC_SHAMIR
        if (ecc_mul2add(result, (mp_int*)n->internal,
                        (ecc_point*)q->internal, (mp_int*)m->internal,
                        result, a, prime, NULL)
                != MP_OKAY) {
            WOLFSSL_MSG("ecc_mul2add error");
            goto cleanup;
        }
#else
        mp_digit mp = 0;
        if (mp_montgomery_setup(prime, &mp) != MP_OKAY) {
            WOLFSSL_MSG("mp_montgomery_setup nqm error");
            goto cleanup;
        }
        if (!(tmp = wc_ecc_new_point())) {
            WOLFSSL_MSG("wolfSSL_EC_POINT_new nqm error");
            goto cleanup;
        }
        /* r = generator * n */
        if (wc_ecc_mulmod((mp_int*)n->internal, result, result, a, prime, 0)
                != MP_OKAY) {
            WOLFSSL_MSG("wc_ecc_mulmod nqm error");
            goto cleanup;
        }
        /* tmp = q * m */
        if (wc_ecc_mulmod((mp_int*)m->internal, (ecc_point*)q->internal,
                tmp, a, prime, 0) != MP_OKAY) {
            WOLFSSL_MSG("wc_ecc_mulmod nqm error");
            goto cleanup;
        }
        /* result = result + tmp */
        if (ecc_projective_add_point(tmp, result, result, a, prime, mp)
                != MP_OKAY) {
            WOLFSSL_MSG("wc_ecc_mulmod nqm error");
            goto cleanup;
        }
        if (ecc_map(result, prime, mp) != MP_OKAY) {
            WOLFSSL_MSG("ecc_map nqm error");
            goto cleanup;
        }
#endif
    }
    else if (n) {
        /* r = generator * n */
        if (wc_ecc_mulmod((mp_int*)n->internal, result, result, a, prime, 1)
                != MP_OKAY) {
            WOLFSSL_MSG("wc_ecc_mulmod gn error");
            goto cleanup;
        }
    }
    else if (q && m) {
        /* r = q * m */
        if (wc_ecc_mulmod((mp_int*)m->internal, (ecc_point*)q->internal,
                           result, a, prime, 1) != MP_OKAY) {
            WOLFSSL_MSG("wc_ecc_mulmod qm error");
            goto cleanup;
        }
    }

    /* copy to destination */
    if (wc_ecc_copy_point(result, (ecc_point*)r->internal)) {
        WOLFSSL_MSG("wc_ecc_copy_point error");
        goto cleanup;
    }
    r->inSet = 1;
    if (SetECPointExternal(r) != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("SetECPointExternal error");
        goto cleanup;
    }

    ret = WOLFSSL_SUCCESS;
cleanup:
    mp_clear(a);
    mp_clear(prime);
    wc_ecc_del_point(result);
    wc_ecc_del_point(tmp);
#ifdef WOLFSSL_SMALL_STACK
    XFREE(a, NULL, DYNAMIC_TYPE_BIGINT);
    XFREE(prime, NULL, DYNAMIC_TYPE_BIGINT);
#endif
    return ret;
}
#endif /* !WOLFSSL_ATECC508A && !WOLFSSL_ATECC608A && !HAVE_SELFTEST &&
        * !WOLFSSL_SP_MATH */

/* (x, y) -> (x, -y) */
int wolfSSL_EC_POINT_invert(const WOLFSSL_EC_GROUP *group, WOLFSSL_EC_POINT *a,
                            WOLFSSL_BN_CTX *ctx)
{
    ecc_point* p;
#ifdef WOLFSSL_SMALL_STACK
    mp_int* prime = NULL;
#else
    mp_int prime[1];
#endif

    (void)ctx;

    WOLFSSL_ENTER("wolfSSL_EC_POINT_invert");

    if (!group || !a || !a->internal || setupPoint(a) != WOLFSSL_SUCCESS) {
        return WOLFSSL_FAILURE;
    }

    p = (ecc_point*)a->internal;

#ifdef WOLFSSL_SMALL_STACK
    prime = (mp_int*)XMALLOC(sizeof(mp_int), NULL, DYNAMIC_TYPE_BIGINT);
    if (prime == NULL) {
        return WOLFSSL_FAILURE;
    }
#endif

    /* read the curve prime and a */
    if (mp_init_multi(prime, NULL, NULL, NULL, NULL, NULL) != MP_OKAY) {
        WOLFSSL_MSG("mp_init_multi error");
    #ifdef WOLFSSL_SMALL_STACK
        XFREE(prime, NULL, DYNAMIC_TYPE_BIGINT);
    #endif
        return WOLFSSL_FAILURE;
    }

    if (mp_sub(prime, p->y, p->y) != MP_OKAY) {
        WOLFSSL_MSG("mp_sub error");
    #ifdef WOLFSSL_SMALL_STACK
        XFREE(prime, NULL, DYNAMIC_TYPE_BIGINT);
    #endif
        return WOLFSSL_FAILURE;
    }

    if (SetECPointExternal(a) != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("SetECPointExternal error");
    #ifdef WOLFSSL_SMALL_STACK
        XFREE(prime, NULL, DYNAMIC_TYPE_BIGINT);
    #endif
        return WOLFSSL_FAILURE;
    }

#ifdef WOLFSSL_SMALL_STACK
    XFREE(prime, NULL, DYNAMIC_TYPE_BIGINT);
#endif

    return WOLFSSL_SUCCESS;
}

void wolfSSL_EC_POINT_clear_free(WOLFSSL_EC_POINT *p)
{
    WOLFSSL_ENTER("wolfSSL_EC_POINT_clear_free");

    wolfSSL_EC_POINT_free(p);
}

/* return code compliant with OpenSSL :
 *   0 if equal, 1 if not and -1 in case of error
 */
int wolfSSL_EC_POINT_cmp(const WOLFSSL_EC_GROUP *group,
                         const WOLFSSL_EC_POINT *a, const WOLFSSL_EC_POINT *b,
                         WOLFSSL_BN_CTX *ctx)
{
    int ret;

    (void)ctx;

    WOLFSSL_ENTER("wolfSSL_EC_POINT_cmp");

    if (group == NULL || a == NULL || a->internal == NULL || b == NULL ||
        b->internal == NULL) {
        WOLFSSL_MSG("wolfSSL_EC_POINT_cmp Bad arguments");
        return WOLFSSL_FATAL_ERROR;
    }

    ret = wc_ecc_cmp_point((ecc_point*)a->internal, (ecc_point*)b->internal);
    if (ret == MP_EQ)
        return 0;
    else if (ret == MP_LT || ret == MP_GT)
        return 1;

    return WOLFSSL_FATAL_ERROR;
}

int wolfSSL_EC_POINT_copy(WOLFSSL_EC_POINT *dest, const WOLFSSL_EC_POINT *src)
{
    WOLFSSL_ENTER("wolfSSL_EC_POINT_copy");

    if (!dest || !src) {
        return WOLFSSL_FAILURE;
    }

    if (setupPoint(src) != WOLFSSL_SUCCESS) {
        return WOLFSSL_FAILURE;
    }

    if (wc_ecc_copy_point((ecc_point*) dest->internal,
                          (ecc_point*) src->internal) != MP_OKAY) {
        return WOLFSSL_FAILURE;
    }

    dest->inSet = 1;

    if (SetECPointExternal(dest) != WOLFSSL_SUCCESS) {
        return WOLFSSL_FAILURE;
    }

    return WOLFSSL_SUCCESS;
}
#endif /* HAVE_ECC */
#endif /* OPENSSL_EXTRA */

#if defined(HAVE_ECC) && (defined(OPENSSL_EXTRA) || defined(OPENSSL_EXTRA_X509_SMALL))
void wolfSSL_EC_POINT_free(WOLFSSL_EC_POINT *p)
{
    WOLFSSL_ENTER("wolfSSL_EC_POINT_free");

    if (p != NULL) {
        if (p->internal != NULL) {
            wc_ecc_del_point((ecc_point*)p->internal);
            p->internal = NULL;
        }

        wolfSSL_BN_free(p->X);
        wolfSSL_BN_free(p->Y);
        wolfSSL_BN_free(p->Z);
        p->X = NULL;
        p->Y = NULL;
        p->Z = NULL;
        p->inSet = p->exSet = 0;

        XFREE(p, NULL, DYNAMIC_TYPE_ECC);
        /* p = NULL, don't try to access or double free it */
    }
}
#endif

#ifdef OPENSSL_EXTRA
#ifdef HAVE_ECC
/* return code compliant with OpenSSL :
 *   1 if point at infinity, 0 else
 */
int wolfSSL_EC_POINT_is_at_infinity(const WOLFSSL_EC_GROUP *group,
                                    const WOLFSSL_EC_POINT *point)
{
    int ret;

    WOLFSSL_ENTER("wolfSSL_EC_POINT_is_at_infinity");

    if (group == NULL || point == NULL || point->internal == NULL) {
        WOLFSSL_MSG("wolfSSL_EC_POINT_is_at_infinity NULL error");
        return WOLFSSL_FAILURE;
    }

    if (setupPoint(point) != WOLFSSL_SUCCESS) {
        return WOLFSSL_FAILURE;
    }

    ret = wc_ecc_point_is_at_infinity((ecc_point*)point->internal);
    if (ret < 0) {
        WOLFSSL_MSG("ecc_point_is_at_infinity failure");
        return WOLFSSL_FAILURE;
    }

    return ret;
}

/* End EC_POINT */

#if !defined(HAVE_FIPS) || (defined(HAVE_FIPS_VERSION) && (HAVE_FIPS_VERSION>2))
size_t wolfSSL_EC_get_builtin_curves(WOLFSSL_EC_BUILTIN_CURVE *r, size_t nitems)
{
    size_t i, min_nitems;
#ifdef HAVE_SELFTEST
    size_t ecc_sets_count;
    for (i = 0; ecc_sets[i].size != 0 && ecc_sets[i].name != NULL; i++);
    ecc_sets_count = i;
#endif

    if (r == NULL || nitems == 0)
        return ecc_sets_count;

    min_nitems = nitems < ecc_sets_count ? nitems : ecc_sets_count;

    for (i = 0; i < min_nitems; i++) {
        r[i].nid = EccEnumToNID(ecc_sets[i].id);
        r[i].comment = wolfSSL_OBJ_nid2sn(r[i].nid);
    }

    return min_nitems;
}
#endif /* !HAVE_FIPS || HAVE_FIPS_VERSION > 2 */

/* Start ECDSA_SIG */
void wolfSSL_ECDSA_SIG_free(WOLFSSL_ECDSA_SIG *sig)
{
    WOLFSSL_ENTER("wolfSSL_ECDSA_SIG_free");

    if (sig) {
        wolfSSL_BN_free(sig->r);
        wolfSSL_BN_free(sig->s);

        XFREE(sig, NULL, DYNAMIC_TYPE_ECC);
    }
}

WOLFSSL_ECDSA_SIG *wolfSSL_ECDSA_SIG_new(void)
{
    WOLFSSL_ECDSA_SIG *sig;

    WOLFSSL_ENTER("wolfSSL_ECDSA_SIG_new");

    sig = (WOLFSSL_ECDSA_SIG*) XMALLOC(sizeof(WOLFSSL_ECDSA_SIG), NULL,
                                       DYNAMIC_TYPE_ECC);
    if (sig == NULL) {
        WOLFSSL_MSG("wolfSSL_ECDSA_SIG_new malloc ECDSA signature failure");
        return NULL;
    }

    sig->s = NULL;
    sig->r = wolfSSL_BN_new();
    if (sig->r == NULL) {
        WOLFSSL_MSG("wolfSSL_ECDSA_SIG_new malloc ECDSA r failure");
        wolfSSL_ECDSA_SIG_free(sig);
        return NULL;
    }

    sig->s = wolfSSL_BN_new();
    if (sig->s == NULL) {
        WOLFSSL_MSG("wolfSSL_ECDSA_SIG_new malloc ECDSA s failure");
        wolfSSL_ECDSA_SIG_free(sig);
        return NULL;
    }

    return sig;
}

void wolfSSL_ECDSA_SIG_get0(const WOLFSSL_ECDSA_SIG* sig,
    const WOLFSSL_BIGNUM** r, const WOLFSSL_BIGNUM** s)
{
    if (sig == NULL) {
        return;
    }

    if (r != NULL) {
        *r = sig->r;
    }
    if (s != NULL) {
        *s = sig->s;
    }
}

int wolfSSL_ECDSA_SIG_set0(WOLFSSL_ECDSA_SIG* sig, WOLFSSL_BIGNUM* r,
    WOLFSSL_BIGNUM* s)
{
    if (sig == NULL || r == NULL || s == NULL) {
        return WOLFSSL_FAILURE;
    }

    wolfSSL_BN_free(sig->r);
    wolfSSL_BN_free(sig->s);

    sig->r = r;
    sig->s = s;

    return WOLFSSL_SUCCESS;
}

/* return signature structure on success, NULL otherwise */
WOLFSSL_ECDSA_SIG *wolfSSL_ECDSA_do_sign(const unsigned char *d, int dlen,
                                         WOLFSSL_EC_KEY *key)
{
    WOLFSSL_ECDSA_SIG *sig = NULL;
    int     initTmpRng = 0;
    WC_RNG* rng = NULL;
#ifdef WOLFSSL_SMALL_STACK
    WC_RNG* tmpRNG = NULL;
    byte*   out = NULL;
    mp_int* sig_r = NULL;
    mp_int* sig_s = NULL;
#else
    WC_RNG  tmpRNG[1];
    byte    out[ECC_BUFSIZE];
    mp_int sig_r[1], sig_s[1];
#endif
    word32 outlen = ECC_BUFSIZE;

    WOLFSSL_ENTER("wolfSSL_ECDSA_do_sign");

    if (d == NULL || key == NULL || key->internal == NULL) {
        WOLFSSL_MSG("wolfSSL_ECDSA_do_sign Bad arguments");
        return NULL;
    }

    /* set internal key if not done */
    if (key->inSet == 0)
    {
        WOLFSSL_MSG("wolfSSL_ECDSA_do_sign No EC key internal set, do it");

        if (SetECKeyInternal(key) != WOLFSSL_SUCCESS) {
            WOLFSSL_MSG("wolfSSL_ECDSA_do_sign SetECKeyInternal failed");
            return NULL;
        }
    }

#ifdef WOLFSSL_SMALL_STACK
    tmpRNG = (WC_RNG*)XMALLOC(sizeof(WC_RNG), NULL, DYNAMIC_TYPE_RNG);
    if (tmpRNG == NULL)
        return NULL;
    out = (byte*)XMALLOC(outlen, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    if (out == NULL) {
        XFREE(tmpRNG, NULL, DYNAMIC_TYPE_RNG);
        return NULL;
    }
    sig_r = (mp_int*)XMALLOC(sizeof(mp_int), NULL, DYNAMIC_TYPE_BIGINT);
    if (sig_r == NULL) {
        XFREE(out, NULL, DYNAMIC_TYPE_TMP_BUFFER);
        XFREE(tmpRNG, NULL, DYNAMIC_TYPE_RNG);
        return NULL;
    }
    sig_s = (mp_int*)XMALLOC(sizeof(mp_int), NULL, DYNAMIC_TYPE_BIGINT);
    if (sig_s == NULL) {
        XFREE(sig_r, NULL, DYNAMIC_TYPE_BIGINT);
        XFREE(out, NULL, DYNAMIC_TYPE_TMP_BUFFER);
        XFREE(tmpRNG, NULL, DYNAMIC_TYPE_RNG);
        return NULL;
    }
#endif

    if (wc_InitRng(tmpRNG) == 0) {
        rng = tmpRNG;
        initTmpRng = 1;
    }
    else {
        WOLFSSL_MSG("wolfSSL_ECDSA_do_sign Bad RNG Init, trying global");
        if (initGlobalRNG == 0)
            WOLFSSL_MSG("wolfSSL_ECDSA_do_sign Global RNG no Init");
        else
            rng = &globalRNG;
    }

    if (rng) {
        /* use wc_ecc_sign_hash because it supports crypto callbacks */
        if (wc_ecc_sign_hash(d, dlen, out, &outlen, rng,
                                                (ecc_key*)key->internal) == 0) {
            if (mp_init_multi(sig_r, sig_s, NULL, NULL, NULL, NULL) == MP_OKAY) {
                /* put signature blob in ECDSA structure */
                if (DecodeECC_DSA_Sig(out, outlen, sig_r, sig_s) == 0) {
                    sig = wolfSSL_ECDSA_SIG_new();
                    if (sig == NULL)
                        WOLFSSL_MSG("wolfSSL_ECDSA_SIG_new failed");
                    else if (SetIndividualExternal(&sig->r, sig_r) != WOLFSSL_SUCCESS) {
                        WOLFSSL_MSG("ecdsa r key error");
                        wolfSSL_ECDSA_SIG_free(sig);
                        sig = NULL;
                    }
                    else if (SetIndividualExternal(&sig->s, sig_s)!=WOLFSSL_SUCCESS){
                        WOLFSSL_MSG("ecdsa s key error");
                        wolfSSL_ECDSA_SIG_free(sig);
                        sig = NULL;
                    }
                }
                mp_free(sig_r);
                mp_free(sig_s);
            }
        }
        else {
            WOLFSSL_MSG("wc_ecc_sign_hash_ex failed");
        }
    }

    if (initTmpRng)
        wc_FreeRng(tmpRNG);
#ifdef WOLFSSL_SMALL_STACK
    XFREE(sig_s, NULL, DYNAMIC_TYPE_BIGINT);
    XFREE(sig_r, NULL, DYNAMIC_TYPE_BIGINT);
    XFREE(out, NULL, DYNAMIC_TYPE_TMP_BUFFER);
    XFREE(tmpRNG, NULL, DYNAMIC_TYPE_RNG);
#endif

    return sig;
}

/* return code compliant with OpenSSL :
 *   1 for a valid signature, 0 for an invalid signature and -1 on error
 */
int wolfSSL_ECDSA_do_verify(const unsigned char *d, int dlen,
                            const WOLFSSL_ECDSA_SIG *sig, WOLFSSL_EC_KEY *key)
{
    int check_sign = 0;

    WOLFSSL_ENTER("wolfSSL_ECDSA_do_verify");

    if (d == NULL || sig == NULL || key == NULL || key->internal == NULL) {
        WOLFSSL_MSG("wolfSSL_ECDSA_do_verify Bad arguments");
        return WOLFSSL_FATAL_ERROR;
    }

    /* set internal key if not done */
    if (key->inSet == 0)
    {
        WOLFSSL_MSG("No EC key internal set, do it");

        if (SetECKeyInternal(key) != WOLFSSL_SUCCESS) {
            WOLFSSL_MSG("SetECKeyInternal failed");
            return WOLFSSL_FATAL_ERROR;
        }
    }

    if (wc_ecc_verify_hash_ex((mp_int*)sig->r->internal,
                              (mp_int*)sig->s->internal, d, dlen, &check_sign,
                              (ecc_key *)key->internal) != MP_OKAY) {
        WOLFSSL_MSG("wc_ecc_verify_hash failed");
        return WOLFSSL_FATAL_ERROR;
    }
    else if (check_sign == 0) {
        WOLFSSL_MSG("wc_ecc_verify_hash incorrect signature detected");
        return WOLFSSL_FAILURE;
    }

    return WOLFSSL_SUCCESS;
}

WOLFSSL_ECDSA_SIG *wolfSSL_d2i_ECDSA_SIG(WOLFSSL_ECDSA_SIG **sig,
                                         const unsigned char **pp, long len)
{
    WOLFSSL_ECDSA_SIG *s = NULL;

    if (pp == NULL)
        return NULL;

    if (sig != NULL)
        s = *sig;
    if (s == NULL) {
        s = wolfSSL_ECDSA_SIG_new();
        if (s == NULL)
            return NULL;
    }

    /* DecodeECC_DSA_Sig calls mp_init, so free these */
    mp_free((mp_int*)s->r->internal);
    mp_free((mp_int*)s->s->internal);

    if (DecodeECC_DSA_Sig(*pp, (word32)len, (mp_int*)s->r->internal,
                                          (mp_int*)s->s->internal) != MP_OKAY) {
        if (sig == NULL || *sig == NULL)
            wolfSSL_ECDSA_SIG_free(s);
        return NULL;
    }

    *pp += len;
    if (sig != NULL)
        *sig = s;
    return s;
}

int wolfSSL_i2d_ECDSA_SIG(const WOLFSSL_ECDSA_SIG *sig, unsigned char **pp)
{
    word32 len;

    if (sig == NULL)
        return 0;

    /* ASN.1: SEQ + INT + INT
     *   ASN.1 Integer must be a positive value - prepend zero if number has
     *   top bit set.
     */
    len = 2 + mp_leading_bit((mp_int*)sig->r->internal) +
              mp_unsigned_bin_size((mp_int*)sig->r->internal) +
          2 + mp_leading_bit((mp_int*)sig->s->internal) +
              mp_unsigned_bin_size((mp_int*)sig->s->internal);
    /* Two bytes required for length if ASN.1 SEQ data greater than 127 bytes
     * and less than 256 bytes.
     */
    len = 1 + ((len > 127) ? 2 : 1) + len;
    if (pp != NULL && *pp != NULL) {
        if (StoreECC_DSA_Sig(*pp, &len, (mp_int*)sig->r->internal,
                                        (mp_int*)sig->s->internal) != MP_OKAY) {
            len = 0;
        }
        else
            *pp += len;
    }

    return (int)len;
}
/* End ECDSA_SIG */

/* Start ECDH */
/* return code compliant with OpenSSL :
 *   length of computed key if success, -1 if error
 */
int wolfSSL_ECDH_compute_key(void *out, size_t outlen,
                             const WOLFSSL_EC_POINT *pub_key,
                             WOLFSSL_EC_KEY *ecdh,
                             void *(*KDF) (const void *in, size_t inlen,
                                           void *out, size_t *outlen))
{
    word32 len;
    ecc_key* key;
    int ret;
#if defined(ECC_TIMING_RESISTANT) && !defined(HAVE_SELFTEST) && \
    (!defined(HAVE_FIPS) || (defined(FIPS_VERSION_GE) && FIPS_VERSION_GE(5,0)))
    int setGlobalRNG = 0;
#endif
    (void)KDF;

    WOLFSSL_ENTER("wolfSSL_ECDH_compute_key");

    if (out == NULL || pub_key == NULL || pub_key->internal == NULL ||
        ecdh == NULL || ecdh->internal == NULL) {
        WOLFSSL_MSG("Bad function arguments");
        return WOLFSSL_FATAL_ERROR;
    }

    /* set internal key if not done */
    if (ecdh->inSet == 0)
    {
        WOLFSSL_MSG("No EC key internal set, do it");

        if (SetECKeyInternal(ecdh) != WOLFSSL_SUCCESS) {
            WOLFSSL_MSG("SetECKeyInternal failed");
            return WOLFSSL_FATAL_ERROR;
        }
    }

    len = (word32)outlen;
    key = (ecc_key*)ecdh->internal;

#if defined(ECC_TIMING_RESISTANT) && !defined(HAVE_SELFTEST) && \
    (!defined(HAVE_FIPS) || (defined(FIPS_VERSION_GE) && FIPS_VERSION_GE(5,0)))
    if (key->rng == NULL) {
        if (initGlobalRNG == 0 && wolfSSL_RAND_Init() != WOLFSSL_SUCCESS) {
            WOLFSSL_MSG("No RNG to use");
            return WOLFSSL_FATAL_ERROR;
        }
        key->rng = &globalRNG;
        setGlobalRNG = 1;
    }
#endif
    PRIVATE_KEY_UNLOCK();
    ret = wc_ecc_shared_secret_ssh(key, (ecc_point*)pub_key->internal,
            (byte *)out, &len);
    PRIVATE_KEY_LOCK();
#if defined(ECC_TIMING_RESISTANT) && !defined(HAVE_SELFTEST) && \
    (!defined(HAVE_FIPS) || (defined(FIPS_VERSION_GE) && FIPS_VERSION_GE(5,0)))
    if (setGlobalRNG)
        key->rng = NULL;
#endif
    if (ret != MP_OKAY) {
        WOLFSSL_MSG("wc_ecc_shared_secret failed");
        return WOLFSSL_FATAL_ERROR;
    }

    return len;
}
/* End ECDH */

#if !defined(NO_FILESYSTEM)

#ifndef NO_BIO

#ifdef WOLFSSL_KEY_GEN
/* return code compliant with OpenSSL :
 *   1 if success, 0 if error
 */
int wolfSSL_PEM_write_EC_PUBKEY(XFILE fp, WOLFSSL_EC_KEY* key)
{
    int ret = WOLFSSL_SUCCESS;
    WOLFSSL_BIO* bio = NULL;

    WOLFSSL_ENTER("wolfSSL_PEM_write_EC_PUBKEY");

    if (fp == XBADFILE || key == NULL) {
        WOLFSSL_MSG("Bad argument.");
        ret = WOLFSSL_FAILURE;
    }

    if (ret == WOLFSSL_SUCCESS) {
        bio = wolfSSL_BIO_new(wolfSSL_BIO_s_file());
        if (bio == NULL) {
            WOLFSSL_MSG("wolfSSL_BIO_new failed.");
            ret = WOLFSSL_FAILURE;
        }
        else if (wolfSSL_BIO_set_fp(bio, fp, BIO_NOCLOSE) != WOLFSSL_SUCCESS) {
            WOLFSSL_MSG("wolfSSL_BIO_set_fp failed.");
            ret = WOLFSSL_FAILURE;
        }
    }
    if (ret == WOLFSSL_SUCCESS && wolfSSL_PEM_write_bio_EC_PUBKEY(bio, key)
        != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("wolfSSL_PEM_write_bio_EC_PUBKEY failed.");
        ret = WOLFSSL_FAILURE;
    }

    if (bio != NULL) {
        wolfSSL_BIO_free(bio);
    }

    WOLFSSL_LEAVE("wolfSSL_PEM_write_EC_PUBKEY", ret);

    return ret;
}
#endif

/* Uses the same format of input as wolfSSL_PEM_read_bio_PrivateKey but expects
 * the results to be an EC key.
 *
 * bio  structure to read EC private key from
 * ec   if not null is then set to the result
 * cb   password callback for reading PEM
 * pass password string
 *
 * returns a pointer to a new WOLFSSL_EC_KEY struct on success and NULL on fail
 */

WOLFSSL_EC_KEY* wolfSSL_PEM_read_bio_EC_PUBKEY(WOLFSSL_BIO* bio,
                                               WOLFSSL_EC_KEY** ec,
                                               wc_pem_password_cb* cb,
                                               void *pass)
{
    WOLFSSL_EVP_PKEY* pkey;
    WOLFSSL_EC_KEY* local;

    WOLFSSL_ENTER("wolfSSL_PEM_read_bio_EC_PUBKEY");

    pkey = wolfSSL_PEM_read_bio_PUBKEY(bio, NULL, cb, pass);
    if (pkey == NULL) {
        return NULL;
    }

    /* Since the WOLFSSL_EC_KEY structure is being taken from WOLFSSL_EVP_PKEY the
     * flag indicating that the WOLFSSL_EC_KEY structure is owned should be FALSE
     * to avoid having it free'd */
    pkey->ownEcc = 0;
    local = pkey->ecc;
    if (ec != NULL) {
        *ec = local;
    }

    wolfSSL_EVP_PKEY_free(pkey);
    return local;
}

/* Reads a private EC key from a WOLFSSL_BIO into a WOLFSSL_EC_KEY.
 * Returns WOLFSSL_SUCCESS or WOLFSSL_FAILURE
 */
WOLFSSL_EC_KEY* wolfSSL_PEM_read_bio_ECPrivateKey(WOLFSSL_BIO* bio,
                                                  WOLFSSL_EC_KEY** ec,
                                                  wc_pem_password_cb* cb,
                                                  void *pass)
{
    WOLFSSL_EVP_PKEY* pkey;
    WOLFSSL_EC_KEY* local;

    WOLFSSL_ENTER("wolfSSL_PEM_read_bio_ECPrivateKey");

    pkey = wolfSSL_PEM_read_bio_PrivateKey(bio, NULL, cb, pass);
    if (pkey == NULL) {
        return NULL;
    }

    /* Since the WOLFSSL_EC_KEY structure is being taken from WOLFSSL_EVP_PKEY the
     * flag indicating that the WOLFSSL_EC_KEY structure is owned should be FALSE
     * to avoid having it free'd */
    pkey->ownEcc = 0;
    local = pkey->ecc;
    if (ec != NULL) {
        *ec = local;
    }

    wolfSSL_EVP_PKEY_free(pkey);
    return local;
}
#endif /* !NO_BIO */
#endif /* NO_FILESYSTEM */

#if defined(WOLFSSL_KEY_GEN)
#ifndef NO_BIO
/* Takes a public WOLFSSL_EC_KEY and writes it out to WOLFSSL_BIO
 * Returns WOLFSSL_SUCCESS or WOLFSSL_FAILURE
 */
int wolfSSL_PEM_write_bio_EC_PUBKEY(WOLFSSL_BIO* bio, WOLFSSL_EC_KEY* ec)
{
    int ret = 0;
    WOLFSSL_EVP_PKEY* pkey;

    WOLFSSL_ENTER("wolfSSL_PEM_write_bio_EC_PUBKEY");

    if (bio == NULL || ec == NULL) {
        WOLFSSL_MSG("Bad Function Arguments");
        return WOLFSSL_FAILURE;
    }

    /* Initialize pkey structure */
    pkey = wolfSSL_EVP_PKEY_new_ex(bio->heap);
    if (pkey == NULL) {
        WOLFSSL_MSG("wolfSSL_EVP_PKEY_new_ex failed");
        return WOLFSSL_FAILURE;
    }

    /* Set pkey info */
    pkey->ecc    = ec;
    pkey->ownEcc = 0; /* pkey does not own ECC */
    pkey->type = EVP_PKEY_EC;

    if((ret = WriteBioPUBKEY(bio, pkey)) != WOLFSSL_SUCCESS){
        WOLFSSL_MSG("wolfSSL_PEM_write_bio_PUBKEY failed");
    }
    wolfSSL_EVP_PKEY_free(pkey);

    return ret;
}

/* return code compliant with OpenSSL :
 *   1 if success, 0 if error
 */
int wolfSSL_PEM_write_bio_ECPrivateKey(WOLFSSL_BIO* bio, WOLFSSL_EC_KEY* ec,
                                       const EVP_CIPHER* cipher,
                                       unsigned char* passwd, int len,
                                       wc_pem_password_cb* cb, void* arg)
{
    int ret = 0, der_max_len = 0, derSz = 0;
    byte *derBuf;
    WOLFSSL_EVP_PKEY* pkey;
    WOLFSSL_ENTER("WOLFSSL_PEM_write_bio_ECPrivateKey");

    if (bio == NULL || ec == NULL) {
        WOLFSSL_MSG("Bad Function Arguments");
        return WOLFSSL_FAILURE;
    }

    /* Initialize pkey structure */
    pkey = wolfSSL_EVP_PKEY_new_ex(bio->heap);
    if (pkey == NULL) {
        WOLFSSL_MSG("wolfSSL_EVP_PKEY_new_ex failed");
        return WOLFSSL_FAILURE;
    }

    /* Set pkey info */
    pkey->ecc    = ec;
    pkey->ownEcc = 0; /* pkey does not own ECC */
    pkey->type = EVP_PKEY_EC;

    /* 4 > size of pub, priv + ASN.1 additional informations
     */
    der_max_len = 4 * wc_ecc_size((ecc_key*)ec->internal) + AES_BLOCK_SIZE;

    derBuf = (byte*)XMALLOC(der_max_len, bio->heap, DYNAMIC_TYPE_TMP_BUFFER);
    if (derBuf == NULL) {
        WOLFSSL_MSG("Malloc failed");
        wolfSSL_EVP_PKEY_free(pkey);
        return WOLFSSL_FAILURE;
    }

    /* convert key to der format */
    derSz = wc_EccKeyToDer((ecc_key*)ec->internal, derBuf, der_max_len);
    if (derSz < 0) {
        WOLFSSL_MSG("wc_EccKeyToDer failed");
        XFREE(derBuf, NULL, DYNAMIC_TYPE_DER);
        wolfSSL_EVP_PKEY_free(pkey);
        return WOLFSSL_FAILURE;
    }

    pkey->pkey.ptr = (char*)XMALLOC(derSz, bio->heap, DYNAMIC_TYPE_TMP_BUFFER);
    if (pkey->pkey.ptr == NULL) {
        WOLFSSL_MSG("key malloc failed");
        XFREE(derBuf, bio->heap, DYNAMIC_TYPE_TMP_BUFFER);
        wolfSSL_EVP_PKEY_free(pkey);
        return WOLFSSL_FAILURE;
    }

    /* add der info to the evp key */
    pkey->pkey_sz = derSz;
    XMEMCPY(pkey->pkey.ptr, derBuf, derSz);
    XFREE(derBuf, bio->heap, DYNAMIC_TYPE_TMP_BUFFER);

    ret = wolfSSL_PEM_write_bio_PrivateKey(bio, pkey, cipher, passwd, len,
                                        cb, arg);
    wolfSSL_EVP_PKEY_free(pkey);

    return ret;
}

#endif /* !NO_BIO */

/* return code compliant with OpenSSL :
 *   1 if success, 0 if error
 */
int wolfSSL_PEM_write_mem_ECPrivateKey(WOLFSSL_EC_KEY* ecc,
                                       const EVP_CIPHER* cipher,
                                       unsigned char* passwd, int passwdSz,
                                       unsigned char **pem, int *plen)
{
#if defined(WOLFSSL_PEM_TO_DER) || defined(WOLFSSL_DER_TO_PEM)
    byte *derBuf, *tmp, *cipherInfo = NULL;
    int  der_max_len = 0, derSz = 0;
    const int type = ECC_PRIVATEKEY_TYPE;
    const char* header = NULL;
    const char* footer = NULL;

    WOLFSSL_MSG("wolfSSL_PEM_write_mem_ECPrivateKey");

    if (pem == NULL || plen == NULL || ecc == NULL || ecc->internal == NULL) {
        WOLFSSL_MSG("Bad function arguments");
        return WOLFSSL_FAILURE;
    }

    if (wc_PemGetHeaderFooter(type, &header, &footer) != 0)
        return WOLFSSL_FAILURE;

    if (ecc->inSet == 0) {
        WOLFSSL_MSG("No ECC internal set, do it");

        if (SetECKeyInternal(ecc) != WOLFSSL_SUCCESS) {
            WOLFSSL_MSG("SetECKeyInternal failed");
            return WOLFSSL_FAILURE;
        }
    }

    /* 4 > size of pub, priv + ASN.1 additional information */
    der_max_len = 4 * wc_ecc_size((ecc_key*)ecc->internal) + AES_BLOCK_SIZE;

    derBuf = (byte*)XMALLOC(der_max_len, NULL, DYNAMIC_TYPE_DER);
    if (derBuf == NULL) {
        WOLFSSL_MSG("malloc failed");
        return WOLFSSL_FAILURE;
    }

    /* Key to DER */
    derSz = wc_EccKeyToDer((ecc_key*)ecc->internal, derBuf, der_max_len);
    if (derSz < 0) {
        WOLFSSL_MSG("wc_EccKeyToDer failed");
        XFREE(derBuf, NULL, DYNAMIC_TYPE_DER);
        return WOLFSSL_FAILURE;
    }

    /* encrypt DER buffer if required */
    if (passwd != NULL && passwdSz > 0 && cipher != NULL) {
        int ret;

        ret = EncryptDerKey(derBuf, &derSz, cipher,
                            passwd, passwdSz, &cipherInfo, der_max_len);
        if (ret != WOLFSSL_SUCCESS) {
            WOLFSSL_MSG("EncryptDerKey failed");
            XFREE(derBuf, NULL, DYNAMIC_TYPE_DER);
            return ret;
        }

        /* tmp buffer with a max size */
        *plen = (derSz * 2) + (int)XSTRLEN(header) + 1 +
            (int)XSTRLEN(footer) + 1 + HEADER_ENCRYPTED_KEY_SIZE;
    }
    else { /* tmp buffer with a max size */
        *plen = (derSz * 2) + (int)XSTRLEN(header) + 1 +
            (int)XSTRLEN(footer) + 1;
    }

    tmp = (byte*)XMALLOC(*plen, NULL, DYNAMIC_TYPE_PEM);
    if (tmp == NULL) {
        WOLFSSL_MSG("malloc failed");
        XFREE(derBuf, NULL, DYNAMIC_TYPE_DER);
        if (cipherInfo != NULL)
            XFREE(cipherInfo, NULL, DYNAMIC_TYPE_STRING);
        return WOLFSSL_FAILURE;
    }

    /* DER to PEM */
    *plen = wc_DerToPemEx(derBuf, derSz, tmp, *plen, cipherInfo, type);
    if (*plen <= 0) {
        WOLFSSL_MSG("wc_DerToPemEx failed");
        XFREE(derBuf, NULL, DYNAMIC_TYPE_DER);
        XFREE(tmp, NULL, DYNAMIC_TYPE_PEM);
        if (cipherInfo != NULL)
            XFREE(cipherInfo, NULL, DYNAMIC_TYPE_STRING);
        return WOLFSSL_FAILURE;
    }
    XFREE(derBuf, NULL, DYNAMIC_TYPE_DER);
    if (cipherInfo != NULL)
        XFREE(cipherInfo, NULL, DYNAMIC_TYPE_STRING);

    *pem = (byte*)XMALLOC((*plen)+1, NULL, DYNAMIC_TYPE_KEY);
    if (*pem == NULL) {
        WOLFSSL_MSG("malloc failed");
        XFREE(tmp, NULL, DYNAMIC_TYPE_PEM);
        return WOLFSSL_FAILURE;
    }
    XMEMSET(*pem, 0, (*plen)+1);

    if (XMEMCPY(*pem, tmp, *plen) == NULL) {
        WOLFSSL_MSG("XMEMCPY failed");
        XFREE(pem, NULL, DYNAMIC_TYPE_KEY);
        XFREE(tmp, NULL, DYNAMIC_TYPE_PEM);
        return WOLFSSL_FAILURE;
    }
    XFREE(tmp, NULL, DYNAMIC_TYPE_PEM);

    return WOLFSSL_SUCCESS;
#else
    (void)ecc;
    (void)cipher;
    (void)passwd;
    (void)passwdSz;
    (void)pem;
    (void)plen;
    return WOLFSSL_FAILURE;
#endif /* WOLFSSL_PEM_TO_DER || WOLFSSL_DER_TO_PEM */
}

#ifndef NO_FILESYSTEM
/* return code compliant with OpenSSL :
 *   1 if success, 0 if error
 */
int wolfSSL_PEM_write_ECPrivateKey(XFILE fp, WOLFSSL_EC_KEY *ecc,
                                   const EVP_CIPHER *enc,
                                   unsigned char *kstr, int klen,
                                   wc_pem_password_cb *cb, void *u)
{
    byte *pem;
    int  plen, ret;

    (void)cb;
    (void)u;

    WOLFSSL_MSG("wolfSSL_PEM_write_ECPrivateKey");

    if (fp == XBADFILE || ecc == NULL || ecc->internal == NULL) {
        WOLFSSL_MSG("Bad function arguments");
        return WOLFSSL_FAILURE;
    }

    ret = wolfSSL_PEM_write_mem_ECPrivateKey(ecc, enc, kstr, klen, &pem, &plen);
    if (ret != WOLFSSL_SUCCESS) {
        WOLFSSL_MSG("wolfSSL_PEM_write_mem_ECPrivateKey failed");
        return WOLFSSL_FAILURE;
    }

    ret = (int)XFWRITE(pem, plen, 1, fp);
    if (ret != 1) {
        WOLFSSL_MSG("ECC private key file write failed");
        return WOLFSSL_FAILURE;
    }

    XFREE(pem, NULL, DYNAMIC_TYPE_KEY);
    return WOLFSSL_SUCCESS;
}

#endif /* NO_FILESYSTEM */
#endif /* defined(WOLFSSL_KEY_GEN) */

#endif /* HAVE_ECC */


#ifndef NO_DSA

#if defined(WOLFSSL_KEY_GEN)
#ifndef NO_BIO

/* Takes a DSA Privatekey and writes it out to a WOLFSSL_BIO
 * Returns WOLFSSL_SUCCESS or WOLFSSL_FAILURE
 */
int wolfSSL_PEM_write_bio_DSAPrivateKey(WOLFSSL_BIO* bio, WOLFSSL_DSA* dsa,
                                       const EVP_CIPHER* cipher,
                                       unsigned char* passwd, int len,
                                       wc_pem_password_cb* cb, void* arg)
{
    int ret = 0, der_max_len = 0, derSz = 0;
    byte *derBuf;
    WOLFSSL_EVP_PKEY* pkey;

    WOLFSSL_ENTER("wolfSSL_PEM_write_bio_DSAPrivateKey");

    if (bio == NULL || dsa == NULL) {
        WOLFSSL_MSG("Bad Function Arguments");
        return WOLFSSL_FAILURE;
    }

    pkey = wolfSSL_EVP_PKEY_new_ex(bio->heap);
    if (pkey == NULL) {
        WOLFSSL_MSG("wolfSSL_EVP_PKEY_new_ex failed");
        return WOLFSSL_FAILURE;
    }

    pkey->type   = EVP_PKEY_DSA;
    pkey->dsa    = dsa;
    pkey->ownDsa = 0;

    /* 4 > size of pub, priv, p, q, g + ASN.1 additional information */
    der_max_len = MAX_DSA_PRIVKEY_SZ;

    derBuf = (byte*)XMALLOC(der_max_len, bio->heap, DYNAMIC_TYPE_TMP_BUFFER);
    if (derBuf == NULL) {
        WOLFSSL_MSG("Malloc failed");
        wolfSSL_EVP_PKEY_free(pkey);
        return WOLFSSL_FAILURE;
    }

    /* convert key to der format */
    derSz = wc_DsaKeyToDer((DsaKey*)dsa->internal, derBuf, der_max_len);
    if (derSz < 0) {
        WOLFSSL_MSG("wc_DsaKeyToDer failed");
        XFREE(derBuf, NULL, DYNAMIC_TYPE_DER);
        wolfSSL_EVP_PKEY_free(pkey);
        return WOLFSSL_FAILURE;
    }

    pkey->pkey.ptr = (char*)XMALLOC(derSz, bio->heap, DYNAMIC_TYPE_TMP_BUFFER);
    if (pkey->pkey.ptr == NULL) {
        WOLFSSL_MSG("key malloc failed");
        XFREE(derBuf, bio->heap, DYNAMIC_TYPE_TMP_BUFFER);
        wolfSSL_EVP_PKEY_free(pkey);
        return WOLFSSL_FAILURE;
    }

    /* add der info to the evp key */
    pkey->pkey_sz = derSz;
    XMEMCPY(pkey->pkey.ptr, derBuf, derSz);
    XFREE(derBuf, bio->heap, DYNAMIC_TYPE_TMP_BUFFER);

    ret = wolfSSL_PEM_write_bio_PrivateKey(bio, pkey, cipher, passwd, len,
                                        cb, arg);
    wolfSSL_EVP_PKEY_free(pkey);

    return ret;
}

#ifndef HAVE_SELFTEST
/* Takes a DSA public key and writes it out to a WOLFSSL_BIO
 * Returns WOLFSSL_SUCCESS or WOLFSSL_FAILURE
 */
int wolfSSL_PEM_write_bio_DSA_PUBKEY(WOLFSSL_BIO* bio, WOLFSSL_DSA* dsa)
{
    int ret = 0;
    WOLFSSL_EVP_PKEY* pkey;

    WOLFSSL_ENTER("wolfSSL_PEM_write_bio_DSA_PUBKEY");

    if (bio == NULL || dsa ==