/*
 * Copyright (C) 2021 Apple Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1.  Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 * 2.  Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE INC. AND ITS CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL APPLE INC. OR ITS CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#include "DownloadProxy.h"

#include "ArgumentCoders.h"
#include "DataReference.h"
#include "Decoder.h"
#include "DownloadID.h"
#include "DownloadProxyMessages.h"
#include "HandleMessage.h"
#include "IdentifierTypes.h"
#include "SandboxExtension.h"
#include "WebCoreArgumentCoders.h"
#include <WebCore/AuthenticationChallenge.h>
#include <WebCore/ResourceError.h>
#include <WebCore/ResourceRequest.h>
#include <WebCore/ResourceResponse.h>
#include <wtf/text/WTFString.h>

namespace Messages {

namespace DownloadProxy {

void DecideDestinationWithSuggestedFilename::callReply(IPC::Decoder& decoder, CompletionHandler<void(String&&, WebKit::SandboxExtension::Handle&&, WebKit::AllowOverwrite&&)>&& completionHandler)
{
    std::optional<String> filename;
    decoder >> filename;
    if (!filename) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    std::optional<WebKit::SandboxExtension::Handle> handle;
    decoder >> handle;
    if (!handle) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    std::optional<WebKit::AllowOverwrite> allowOverwrite;
    decoder >> allowOverwrite;
    if (!allowOverwrite) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*filename), WTFMove(*handle), WTFMove(*allowOverwrite));
}

void DecideDestinationWithSuggestedFilename::cancelReply(CompletionHandler<void(String&&, WebKit::SandboxExtension::Handle&&, WebKit::AllowOverwrite&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<String>::create(), IPC::AsyncReplyError<WebKit::SandboxExtension::Handle>::create(), IPC::AsyncReplyError<WebKit::AllowOverwrite>::create());
}

void DecideDestinationWithSuggestedFilename::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const String& filename, const WebKit::SandboxExtension::Handle& handle, WebKit::AllowOverwrite allowOverwrite)
{
    encoder.get() << filename;
    encoder.get() << handle;
    encoder.get() << allowOverwrite;
    connection.sendSyncReply(WTFMove(encoder));
}

} // namespace DownloadProxy

} // namespace Messages

namespace WebKit {

void DownloadProxy::didReceiveMessage(IPC::Connection& connection, IPC::Decoder& decoder)
{
    Ref protectedThis { *this };
    if (decoder.messageName() == Messages::DownloadProxy::DidStart::name())
        return IPC::handleMessage<Messages::DownloadProxy::DidStart>(connection, decoder, this, &DownloadProxy::didStart);
    if (decoder.messageName() == Messages::DownloadProxy::DidReceiveAuthenticationChallenge::name())
        return IPC::handleMessage<Messages::DownloadProxy::DidReceiveAuthenticationChallenge>(connection, decoder, this, &DownloadProxy::didReceiveAuthenticationChallenge);
    if (decoder.messageName() == Messages::DownloadProxy::WillSendRequest::name())
        return IPC::handleMessage<Messages::DownloadProxy::WillSendRequest>(connection, decoder, this, &DownloadProxy::willSendRequest);
    if (decoder.messageName() == Messages::DownloadProxy::DecideDestinationWithSuggestedFilename::name())
        return IPC::handleMessageAsync<Messages::DownloadProxy::DecideDestinationWithSuggestedFilename>(connection, decoder, this, &DownloadProxy::decideDestinationWithSuggestedFilename);
    if (decoder.messageName() == Messages::DownloadProxy::DidReceiveData::name())
        return IPC::handleMessage<Messages::DownloadProxy::DidReceiveData>(connection, decoder, this, &DownloadProxy::didReceiveData);
    if (decoder.messageName() == Messages::DownloadProxy::DidCreateDestination::name())
        return IPC::handleMessage<Messages::DownloadProxy::DidCreateDestination>(connection, decoder, this, &DownloadProxy::didCreateDestination);
    if (decoder.messageName() == Messages::DownloadProxy::DidFinish::name())
        return IPC::handleMessage<Messages::DownloadProxy::DidFinish>(connection, decoder, this, &DownloadProxy::didFinish);
    if (decoder.messageName() == Messages::DownloadProxy::DidFail::name())
        return IPC::handleMessage<Messages::DownloadProxy::DidFail>(connection, decoder, this, &DownloadProxy::didFail);
    UNUSED_PARAM(connection);
    UNUSED_PARAM(decoder);
#if ENABLE(IPC_TESTING_API)
    if (connection.ignoreInvalidMessageForTesting())
        return;
#endif // ENABLE(IPC_TESTING_API)
    ASSERT_NOT_REACHED_WITH_MESSAGE("Unhandled message %s to %" PRIu64, IPC::description(decoder.messageName()), decoder.destinationID());
}

} // namespace WebKit
