/*
 * Copyright (C) 2021 Apple Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1.  Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 * 2.  Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE INC. AND ITS CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL APPLE INC. OR ITS CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#include "NetworkConnectionToWebProcess.h"

#include "ArgumentCoders.h"
#include "Decoder.h"
#include "DownloadID.h"
#include "HandleMessage.h"
#include "NavigatingToAppBoundDomain.h"
#include "NetworkConnectionToWebProcessMessages.h"
#include "NetworkResourceLoadIdentifier.h"
#include "NetworkResourceLoadParameters.h"
#include "SandboxExtension.h"
#include "WebCoreArgumentCoders.h"
#include "WebPageProxyIdentifier.h"
#include <WebCore/BlobPart.h>
#include <WebCore/ClientOrigin.h>
#include <WebCore/Cookie.h>
#include <WebCore/CookieJar.h>
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
#include <WebCore/DocumentStorageAccess.h>
#endif
#include <WebCore/FrameIdentifier.h>
#include <WebCore/LoadSchedulingMode.h>
#include <WebCore/MessagePortIdentifier.h>
#include <WebCore/MessageWithMessagePorts.h>
#include <WebCore/NetworkLoadInformation.h>
#include <WebCore/NetworkLoadMetrics.h>
#include <WebCore/NetworkStorageSession.h>
#include <WebCore/PageIdentifier.h>
#include <WebCore/PolicyContainer.h>
#if ENABLE(SERVICE_WORKER)
#include <WebCore/ProcessQualified.h>
#endif
#if ENABLE(WEB_RTC)
#include <WebCore/RTCDataChannelIdentifier.h>
#endif
#include <WebCore/RegistrableDomain.h>
#include <WebCore/ResourceError.h>
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
#include <WebCore/ResourceLoadStatistics.h>
#endif
#include <WebCore/ResourceLoaderIdentifier.h>
#include <WebCore/ResourceRequest.h>
#include <WebCore/ResourceResponse.h>
#include <WebCore/SameSiteInfo.h>
#if ENABLE(SERVICE_WORKER)
#include <WebCore/ScriptExecutionContextIdentifier.h>
#endif
#include <WebCore/ShouldRelaxThirdPartyCookieBlocking.h>
#include <WebCore/WebSocketIdentifier.h>
#include <optional>
#include <wtf/Expected.h>
#if HAVE(COOKIE_CHANGE_LISTENER_API)
#include <wtf/HashSet.h>
#endif
#if ENABLE(SERVICE_WORKER)
#include <wtf/ObjectIdentifier.h>
#endif
#include <wtf/Seconds.h>
#include <wtf/URLHash.h>
#include <wtf/Vector.h>
#include <wtf/text/WTFString.h>

namespace Messages {

namespace NetworkConnectionToWebProcess {

void PerformSynchronousLoad::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const WebCore::ResourceError& error, const WebCore::ResourceResponse& response, const Vector<uint8_t>& data)
{
    encoder.get() << error;
    encoder.get() << response;
    encoder.get() << data;
    connection.sendSyncReply(WTFMove(encoder));
}

void TestProcessIncomingSyncMessagesWhenWaitingForSyncReply::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, bool handled)
{
    encoder.get() << handled;
    connection.sendSyncReply(WTFMove(encoder));
}

void SendH2Ping::callReply(IPC::Decoder& decoder, CompletionHandler<void(Expected<Seconds, WebCore::ResourceError>&&)>&& completionHandler)
{
    std::optional<Expected<Seconds, WebCore::ResourceError>> result;
    decoder >> result;
    if (!result) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*result));
}

void SendH2Ping::cancelReply(CompletionHandler<void(Expected<Seconds, WebCore::ResourceError>&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<Expected<Seconds, WebCore::ResourceError>>::create());
}

void SendH2Ping::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const Expected<Seconds, WebCore::ResourceError>& result)
{
    encoder.get() << result;
    connection.sendSyncReply(WTFMove(encoder));
}

void IsResourceLoadFinished::callReply(IPC::Decoder& decoder, CompletionHandler<void(bool&&)>&& completionHandler)
{
    std::optional<bool> isFinished;
    decoder >> isFinished;
    if (!isFinished) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*isFinished));
}

void IsResourceLoadFinished::cancelReply(CompletionHandler<void(bool&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<bool>::create());
}

void IsResourceLoadFinished::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, bool isFinished)
{
    encoder.get() << isFinished;
    connection.sendSyncReply(WTFMove(encoder));
}

void CookiesForDOM::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const String& cookieString, bool didAccessSecureCookies)
{
    encoder.get() << cookieString;
    encoder.get() << didAccessSecureCookies;
    connection.sendSyncReply(WTFMove(encoder));
}

void CookieRequestHeaderFieldValue::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const String& cookieString, bool didAccessSecureCookies)
{
    encoder.get() << cookieString;
    encoder.get() << didAccessSecureCookies;
    connection.sendSyncReply(WTFMove(encoder));
}

void GetRawCookies::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const Vector<WebCore::Cookie>& cookies)
{
    encoder.get() << cookies;
    connection.sendSyncReply(WTFMove(encoder));
}

void DomCookiesForHost::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const Vector<WebCore::Cookie>& cookies)
{
    encoder.get() << cookies;
    connection.sendSyncReply(WTFMove(encoder));
}

void BlobSize::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, uint64_t resultSize)
{
    encoder.get() << resultSize;
    connection.sendSyncReply(WTFMove(encoder));
}

void WriteBlobsToTemporaryFilesForIndexedDB::callReply(IPC::Decoder& decoder, CompletionHandler<void(Vector<String>&&)>&& completionHandler)
{
    std::optional<Vector<String>> fileNames;
    decoder >> fileNames;
    if (!fileNames) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*fileNames));
}

void WriteBlobsToTemporaryFilesForIndexedDB::cancelReply(CompletionHandler<void(Vector<String>&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<Vector<String>>::create());
}

void WriteBlobsToTemporaryFilesForIndexedDB::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const Vector<String>& fileNames)
{
    encoder.get() << fileNames;
    connection.sendSyncReply(WTFMove(encoder));
}

#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)

void ResourceLoadStatisticsUpdated::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void ResourceLoadStatisticsUpdated::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void ResourceLoadStatisticsUpdated::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)

void HasStorageAccess::callReply(IPC::Decoder& decoder, CompletionHandler<void(bool&&)>&& completionHandler)
{
    std::optional<bool> hasStorageAccess;
    decoder >> hasStorageAccess;
    if (!hasStorageAccess) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*hasStorageAccess));
}

void HasStorageAccess::cancelReply(CompletionHandler<void(bool&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<bool>::create());
}

void HasStorageAccess::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, bool hasStorageAccess)
{
    encoder.get() << hasStorageAccess;
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)

void RequestStorageAccess::callReply(IPC::Decoder& decoder, CompletionHandler<void(WebCore::RequestStorageAccessResult&&)>&& completionHandler)
{
    std::optional<WebCore::RequestStorageAccessResult> result;
    decoder >> result;
    if (!result) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*result));
}

void RequestStorageAccess::cancelReply(CompletionHandler<void(WebCore::RequestStorageAccessResult&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<WebCore::RequestStorageAccessResult>::create());
}

void RequestStorageAccess::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const WebCore::RequestStorageAccessResult& result)
{
    encoder.get() << result;
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

void GetNetworkLoadInformationResponse::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const WebCore::ResourceResponse& response)
{
    encoder.get() << response;
    connection.sendSyncReply(WTFMove(encoder));
}

void GetNetworkLoadIntermediateInformation::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const Vector<WebCore::NetworkTransactionInformation>& transactions)
{
    encoder.get() << transactions;
    connection.sendSyncReply(WTFMove(encoder));
}

void TakeNetworkLoadInformationMetrics::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const WebCore::NetworkLoadMetrics& networkMetrics)
{
    encoder.get() << networkMetrics;
    connection.sendSyncReply(WTFMove(encoder));
}

#if ENABLE(SERVICE_WORKER)

void EstablishSWContextConnection::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void EstablishSWContextConnection::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void EstablishSWContextConnection::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

void EstablishSharedWorkerContextConnection::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void EstablishSharedWorkerContextConnection::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void EstablishSharedWorkerContextConnection::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

void CreateRTCProvider::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void CreateRTCProvider::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void CreateRTCProvider::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

#if ENABLE(WEB_RTC)

void ConnectToRTCDataChannelRemoteSource::callReply(IPC::Decoder& decoder, CompletionHandler<void(std::optional<bool>&&)>&& completionHandler)
{
    std::optional<std::optional<bool>> result;
    decoder >> result;
    if (!result) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*result));
}

void ConnectToRTCDataChannelRemoteSource::cancelReply(CompletionHandler<void(std::optional<bool>&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<std::optional<bool>>::create());
}

void ConnectToRTCDataChannelRemoteSource::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const std::optional<bool>& result)
{
    encoder.get() << result;
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

void TakeAllMessagesForPort::callReply(IPC::Decoder& decoder, CompletionHandler<void(Vector<WebCore::MessageWithMessagePorts>&&, uint64_t&&)>&& completionHandler)
{
    std::optional<Vector<WebCore::MessageWithMessagePorts>> messages;
    decoder >> messages;
    if (!messages) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    std::optional<uint64_t> messageBatchIdentifier;
    decoder >> messageBatchIdentifier;
    if (!messageBatchIdentifier) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*messages), WTFMove(*messageBatchIdentifier));
}

void TakeAllMessagesForPort::cancelReply(CompletionHandler<void(Vector<WebCore::MessageWithMessagePorts>&&, uint64_t&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<Vector<WebCore::MessageWithMessagePorts>>::create(), IPC::AsyncReplyError<uint64_t>::create());
}

void TakeAllMessagesForPort::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const Vector<WebCore::MessageWithMessagePorts>& messages, uint64_t messageBatchIdentifier)
{
    encoder.get() << messages;
    encoder.get() << messageBatchIdentifier;
    connection.sendSyncReply(WTFMove(encoder));
}

void CheckRemotePortForActivity::callReply(IPC::Decoder& decoder, CompletionHandler<void(bool&&)>&& completionHandler)
{
    std::optional<bool> hasActivity;
    decoder >> hasActivity;
    if (!hasActivity) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*hasActivity));
}

void CheckRemotePortForActivity::cancelReply(CompletionHandler<void(bool&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<bool>::create());
}

void CheckRemotePortForActivity::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, bool hasActivity)
{
    encoder.get() << hasActivity;
    connection.sendSyncReply(WTFMove(encoder));
}

#if PLATFORM(MAC)

void GetProcessDisplayName::callReply(IPC::Decoder& decoder, CompletionHandler<void(String&&)>&& completionHandler)
{
    std::optional<String> displayName;
    decoder >> displayName;
    if (!displayName) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*displayName));
}

void GetProcessDisplayName::cancelReply(CompletionHandler<void(String&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<String>::create());
}

void GetProcessDisplayName::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const String& displayName)
{
    encoder.get() << displayName;
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

} // namespace NetworkConnectionToWebProcess

} // namespace Messages

namespace WebKit {

void NetworkConnectionToWebProcess::didReceiveNetworkConnectionToWebProcessMessage(IPC::Connection& connection, IPC::Decoder& decoder)
{
    Ref protectedThis { *this };
    if (decoder.messageName() == Messages::NetworkConnectionToWebProcess::ScheduleResourceLoad::name())
        return IPC::handleMessage<Messages::NetworkConnectionToWebProcess::ScheduleResourceLoad>(connection, decoder, this, &NetworkConnectionToWebProcess::scheduleResourceLoad);
    if (decoder.messageName() == Messages::NetworkConnectionToWebProcess::LoadPing::name())
        return IPC::handleMessage<Messages::NetworkConnectionToWebProcess::LoadPing>(connection, decoder, this, &NetworkConnectionToWebProcess::loadPing);
    if (decoder.messageName() == Messages::NetworkConnectionToWebProcess::RemoveLoadIdentifier::name())
        return IPC::handleMessage<Messages::NetworkConnectionToWebProcess::RemoveLoadIdentifier>(connection, decoder, this, &NetworkConnectionToWebProcess::removeLoadIdentifier);
    if (decoder.messageName() == Messages::NetworkConnectionToWebProcess::PageLoadCompleted::name())
        return IPC::handleMessage<Messages::NetworkConnectionToWebProcess::PageLoadCompleted>(connection, decoder, this, &NetworkConnectionToWebProcess::pageLoadCompleted);
    if (decoder.messageName() == Messages::NetworkConnectionToWebProcess::BrowsingContextRemoved::name())
        return IPC::handleMessage<Messages::NetworkConnectionToWebProcess::BrowsingContextRemoved>(connection, decoder, this, &NetworkConnectionToWebProcess::browsingContextRemoved);
    if (decoder.messageName() == Messages::NetworkConnectionToWebProcess::PrefetchDNS::name())
        return IPC::handleMessage<Messages::NetworkConnectionToWebProcess::PrefetchDNS>(connection, decoder, this, &NetworkConnectionToWebProcess::prefetchDNS);
    if (decoder.messageName() == Messages::NetworkConnectionToWebProcess::SendH2Ping::name())
        return IPC::handleMessageAsync<Messages::NetworkConnectionToWebProcess::SendH2Ping>(connection, decoder, this, &NetworkConnectionToWebProcess::sendH2Ping);
    if (decoder.messageName() == Messages::NetworkConnectionToWebProcess::PreconnectTo::name())
        return IPC::handleMessage<Messages::NetworkConnectionToWebProcess::PreconnectTo>(connection, decoder, this, &NetworkConnectionToWebProcess::preconnectTo);
    if (decoder.messageName() == Messages::NetworkConnectionToWebProcess::IsResourceLoadFinished::name())
        return IPC::handleMessageAsync<Messages::NetworkConnectionToWebProcess::IsResourceLoadFinished>(connection, decoder, this, &NetworkConnectionToWebProcess::isResourceLoadFinished);
    if (decoder.messageName() == Messages::NetworkConnectionToWebProcess::StartDownload::name())
        return IPC::handleMessage<Messages::NetworkConnectionToWebProcess::StartDownload>(connection, decoder, this, &NetworkConnectionToWebProcess::startDownload);
    if (decoder.messageName() == Messages::NetworkConnectionToWebProcess::ConvertMainResourceLoadToDownload::name())
        return IPC::handleMessage<Messages::NetworkConnectionToWebProcess::ConvertMainResourceLoadToDownload>(connection, decoder, this, &NetworkConnectionToWebProcess::convertMainResourceLoadToDownload);
    if (decoder.messageName() == Messages::NetworkConnectionToWebProcess::SetCookiesFromDOM::name())
        return IPC::handleMessage<Messages::NetworkConnectionToWebProcess::SetCookiesFromDOM>(connection, decoder, this, &NetworkConnectionToWebProcess::setCookiesFromDOM);
    if (decoder.messageName() == Messages::NetworkConnectionToWebProcess::SetRawCookie::name())
        return IPC::handleMessage<Messages::NetworkConnectionToWebProcess::SetRawCookie>(connection, decoder, this, &NetworkConnectionToWebProcess::setRawCookie);
    if (decoder.messageName() == Messages::NetworkConnectionToWebProcess::DeleteCookie::name())
        return IPC::handleMessage<Messages::NetworkConnectionToWebProcess::DeleteCookie>(connection, decoder, this, &NetworkConnectionToWebProcess::deleteCookie);
#if HAVE(COOKIE_CHANGE_LISTENER_API)
    if (decoder.messageName() == Messages::NetworkConnectionToWebProcess::UnsubscribeFromCookieChangeNotifications::name())
        return IPC::handleMessage<Messages::NetworkConnectionToWebProcess::UnsubscribeFromCookieChangeNotifications>(connection, decoder, this, &NetworkConnectionToWebProcess::unsubscribeFromCookieChangeNotifications);
#endif
    if (decoder.messageName() == Messages::NetworkConnectionToWebProcess::RegisterFileBlobURL::name())
        return IPC::handleMessage<Messages::NetworkConnectionToWebProcess::RegisterFileBlobURL>(connection, decoder, this, &NetworkConnectionToWebProcess::registerFileBlobURL);
    if (decoder.messageName() == Messages::NetworkConnectionToWebProcess::RegisterBlobURL::name())
        return IPC::handleMessage<Messages::NetworkConnectionToWebProcess::RegisterBlobURL>(connection, decoder, this, &NetworkConnectionToWebProcess::registerBlobURL);
    if (decoder.messageName() == Messages::NetworkConnectionToWebProcess::RegisterBlobURLFromURL::name())
        return IPC::handleMessage<Messages::NetworkConnectionToWebProcess::RegisterBlobURLFromURL>(connection, decoder, this, &NetworkConnectionToWebProcess::registerBlobURLFromURL);
    if (decoder.messageName() == Messages::NetworkConnectionToWebProcess::RegisterBlobURLOptionallyFileBacked::name())
        return IPC::handleMessage<Messages::NetworkConnectionToWebProcess::RegisterBlobURLOptionallyFileBacked>(connection, decoder, this, &NetworkConnectionToWebProcess::registerBlobURLOptionallyFileBacked);
    if (decoder.messageName() == Messages::NetworkConnectionToWebProcess::RegisterBlobURLForSlice::name())
        return IPC::handleMessage<Messages::NetworkConnectionToWebProcess::RegisterBlobURLForSlice>(connection, decoder, this, &NetworkConnectionToWebProcess::registerBlobURLForSlice);
    if (decoder.messageName() == Messages::NetworkConnectionToWebProcess::UnregisterBlobURL::name())
        return IPC::handleMessage<Messages::NetworkConnectionToWebProcess::UnregisterBlobURL>(connection, decoder, this, &NetworkConnectionToWebProcess::unregisterBlobURL);
    if (decoder.messageName() == Messages::NetworkConnectionToWebProcess::WriteBlobsToTemporaryFilesForIndexedDB::name())
        return IPC::handleMessageAsync<Messages::NetworkConnectionToWebProcess::WriteBlobsToTemporaryFilesForIndexedDB>(connection, decoder, this, &NetworkConnectionToWebProcess::writeBlobsToTemporaryFilesForIndexedDB);
    if (decoder.messageName() == Messages::NetworkConnectionToWebProcess::RegisterBlobURLHandle::name())
        return IPC::handleMessage<Messages::NetworkConnectionToWebProcess::RegisterBlobURLHandle>(connection, decoder, this, &NetworkConnectionToWebProcess::registerBlobURLHandle);
    if (decoder.messageName() == Messages::NetworkConnectionToWebProcess::UnregisterBlobURLHandle::name())
        return IPC::handleMessage<Messages::NetworkConnectionToWebProcess::UnregisterBlobURLHandle>(connection, decoder, this, &NetworkConnectionToWebProcess::unregisterBlobURLHandle);
    if (decoder.messageName() == Messages::NetworkConnectionToWebProcess::SetCaptureExtraNetworkLoadMetricsEnabled::name())
        return IPC::handleMessage<Messages::NetworkConnectionToWebProcess::SetCaptureExtraNetworkLoadMetricsEnabled>(connection, decoder, this, &NetworkConnectionToWebProcess::setCaptureExtraNetworkLoadMetricsEnabled);
    if (decoder.messageName() == Messages::NetworkConnectionToWebProcess::CreateSocketStream::name())
        return IPC::handleMessage<Messages::NetworkConnectionToWebProcess::CreateSocketStream>(connection, decoder, this, &NetworkConnectionToWebProcess::createSocketStream);
    if (decoder.messageName() == Messages::NetworkConnectionToWebProcess::CreateSocketChannel::name())
        return IPC::handleMessage<Messages::NetworkConnectionToWebProcess::CreateSocketChannel>(connection, decoder, this, &NetworkConnectionToWebProcess::createSocketChannel);
    if (decoder.messageName() == Messages::NetworkConnectionToWebProcess::ClearPageSpecificData::name())
        return IPC::handleMessage<Messages::NetworkConnectionToWebProcess::ClearPageSpecificData>(connection, decoder, this, &NetworkConnectionToWebProcess::clearPageSpecificData);
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
    if (decoder.messageName() == Messages::NetworkConnectionToWebProcess::RemoveStorageAccessForFrame::name())
        return IPC::handleMessage<Messages::NetworkConnectionToWebProcess::RemoveStorageAccessForFrame>(connection, decoder, this, &NetworkConnectionToWebProcess::removeStorageAccessForFrame);
#endif
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
    if (decoder.messageName() == Messages::NetworkConnectionToWebProcess::LogUserInteraction::name())
        return IPC::handleMessage<Messages::NetworkConnectionToWebProcess::LogUserInteraction>(connection, decoder, this, &NetworkConnectionToWebProcess::logUserInteraction);
#endif
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
    if (decoder.messageName() == Messages::NetworkConnectionToWebProcess::ResourceLoadStatisticsUpdated::name())
        return IPC::handleMessageAsync<Messages::NetworkConnectionToWebProcess::ResourceLoadStatisticsUpdated>(connection, decoder, this, &NetworkConnectionToWebProcess::resourceLoadStatisticsUpdated);
#endif
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
    if (decoder.messageName() == Messages::NetworkConnectionToWebProcess::HasStorageAccess::name())
        return IPC::handleMessageAsync<Messages::NetworkConnectionToWebProcess::HasStorageAccess>(connection, decoder, this, &NetworkConnectionToWebProcess::hasStorageAccess);
#endif
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
    if (decoder.messageName() == Messages::NetworkConnectionToWebProcess::RequestStorageAccess::name())
        return IPC::handleMessageAsync<Messages::NetworkConnectionToWebProcess::RequestStorageAccess>(connection, decoder, this, &NetworkConnectionToWebProcess::requestStorageAccess);
#endif
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
    if (decoder.messageName() == Messages::NetworkConnectionToWebProcess::RequestStorageAccessUnderOpener::name())
        return IPC::handleMessage<Messages::NetworkConnectionToWebProcess::RequestStorageAccessUnderOpener>(connection, decoder, this, &NetworkConnectionToWebProcess::requestStorageAccessUnderOpener);
#endif
    if (decoder.messageName() == Messages::NetworkConnectionToWebProcess::AddOriginAccessAllowListEntry::name())
        return IPC::handleMessage<Messages::NetworkConnectionToWebProcess::AddOriginAccessAllowListEntry>(connection, decoder, this, &NetworkConnectionToWebProcess::addOriginAccessAllowListEntry);
    if (decoder.messageName() == Messages::NetworkConnectionToWebProcess::RemoveOriginAccessAllowListEntry::name())
        return IPC::handleMessage<Messages::NetworkConnectionToWebProcess::RemoveOriginAccessAllowListEntry>(connection, decoder, this, &NetworkConnectionToWebProcess::removeOriginAccessAllowListEntry);
    if (decoder.messageName() == Messages::NetworkConnectionToWebProcess::ResetOriginAccessAllowLists::name())
        return IPC::handleMessage<Messages::NetworkConnectionToWebProcess::ResetOriginAccessAllowLists>(connection, decoder, this, &NetworkConnectionToWebProcess::resetOriginAccessAllowLists);
#if ENABLE(SERVICE_WORKER)
    if (decoder.messageName() == Messages::NetworkConnectionToWebProcess::EstablishSWContextConnection::name())
        return IPC::handleMessageAsync<Messages::NetworkConnectionToWebProcess::EstablishSWContextConnection>(connection, decoder, this, &NetworkConnectionToWebProcess::establishSWContextConnection);
#endif
#if ENABLE(SERVICE_WORKER)
    if (decoder.messageName() == Messages::NetworkConnectionToWebProcess::CloseSWContextConnection::name())
        return IPC::handleMessage<Messages::NetworkConnectionToWebProcess::CloseSWContextConnection>(connection, decoder, this, &NetworkConnectionToWebProcess::closeSWContextConnection);
#endif
    if (decoder.messageName() == Messages::NetworkConnectionToWebProcess::EstablishSharedWorkerContextConnection::name())
        return IPC::handleMessageAsync<Messages::NetworkConnectionToWebProcess::EstablishSharedWorkerContextConnection>(connection, decoder, this, &NetworkConnectionToWebProcess::establishSharedWorkerContextConnection);
    if (decoder.messageName() == Messages::NetworkConnectionToWebProcess::CloseSharedWorkerContextConnection::name())
        return IPC::handleMessage<Messages::NetworkConnectionToWebProcess::CloseSharedWorkerContextConnection>(connection, decoder, this, &NetworkConnectionToWebProcess::closeSharedWorkerContextConnection);
    if (decoder.messageName() == Messages::NetworkConnectionToWebProcess::CreateRTCProvider::name())
        return IPC::handleMessageAsync<Messages::NetworkConnectionToWebProcess::CreateRTCProvider>(connection, decoder, this, &NetworkConnectionToWebProcess::createRTCProvider);
#if ENABLE(WEB_RTC)
    if (decoder.messageName() == Messages::NetworkConnectionToWebProcess::ConnectToRTCDataChannelRemoteSource::name())
        return IPC::handleMessageAsync<Messages::NetworkConnectionToWebProcess::ConnectToRTCDataChannelRemoteSource>(connection, decoder, this, &NetworkConnectionToWebProcess::connectToRTCDataChannelRemoteSource);
#endif
    if (decoder.messageName() == Messages::NetworkConnectionToWebProcess::UpdateQuotaBasedOnSpaceUsageForTesting::name())
        return IPC::handleMessage<Messages::NetworkConnectionToWebProcess::UpdateQuotaBasedOnSpaceUsageForTesting>(connection, decoder, this, &NetworkConnectionToWebProcess::updateQuotaBasedOnSpaceUsageForTesting);
    if (decoder.messageName() == Messages::NetworkConnectionToWebProcess::CreateNewMessagePortChannel::name())
        return IPC::handleMessage<Messages::NetworkConnectionToWebProcess::CreateNewMessagePortChannel>(connection, decoder, this, &NetworkConnectionToWebProcess::createNewMessagePortChannel);
    if (decoder.messageName() == Messages::NetworkConnectionToWebProcess::EntangleLocalPortInThisProcessToRemote::name())
        return IPC::handleMessage<Messages::NetworkConnectionToWebProcess::EntangleLocalPortInThisProcessToRemote>(connection, decoder, this, &NetworkConnectionToWebProcess::entangleLocalPortInThisProcessToRemote);
    if (decoder.messageName() == Messages::NetworkConnectionToWebProcess::MessagePortDisentangled::name())
        return IPC::handleMessage<Messages::NetworkConnectionToWebProcess::MessagePortDisentangled>(connection, decoder, this, &NetworkConnectionToWebProcess::messagePortDisentangled);
    if (decoder.messageName() == Messages::NetworkConnectionToWebProcess::MessagePortClosed::name())
        return IPC::handleMessage<Messages::NetworkConnectionToWebProcess::MessagePortClosed>(connection, decoder, this, &NetworkConnectionToWebProcess::messagePortClosed);
    if (decoder.messageName() == Messages::NetworkConnectionToWebProcess::TakeAllMessagesForPort::name())
        return IPC::handleMessageAsync<Messages::NetworkConnectionToWebProcess::TakeAllMessagesForPort>(connection, decoder, this, &NetworkConnectionToWebProcess::takeAllMessagesForPort);
    if (decoder.messageName() == Messages::NetworkConnectionToWebProcess::PostMessageToRemote::name())
        return IPC::handleMessage<Messages::NetworkConnectionToWebProcess::PostMessageToRemote>(connection, decoder, this, &NetworkConnectionToWebProcess::postMessageToRemote);
    if (decoder.messageName() == Messages::NetworkConnectionToWebProcess::CheckRemotePortForActivity::name())
        return IPC::handleMessageAsync<Messages::NetworkConnectionToWebProcess::CheckRemotePortForActivity>(connection, decoder, this, &NetworkConnectionToWebProcess::checkRemotePortForActivity);
    if (decoder.messageName() == Messages::NetworkConnectionToWebProcess::DidDeliverMessagePortMessages::name())
        return IPC::handleMessage<Messages::NetworkConnectionToWebProcess::DidDeliverMessagePortMessages>(connection, decoder, this, &NetworkConnectionToWebProcess::didDeliverMessagePortMessages);
    if (decoder.messageName() == Messages::NetworkConnectionToWebProcess::RegisterURLSchemesAsCORSEnabled::name())
        return IPC::handleMessage<Messages::NetworkConnectionToWebProcess::RegisterURLSchemesAsCORSEnabled>(connection, decoder, this, &NetworkConnectionToWebProcess::registerURLSchemesAsCORSEnabled);
    if (decoder.messageName() == Messages::NetworkConnectionToWebProcess::SetCORSDisablingPatterns::name())
        return IPC::handleMessage<Messages::NetworkConnectionToWebProcess::SetCORSDisablingPatterns>(connection, decoder, this, &NetworkConnectionToWebProcess::setCORSDisablingPatterns);
#if PLATFORM(MAC)
    if (decoder.messageName() == Messages::NetworkConnectionToWebProcess::GetProcessDisplayName::name())
        return IPC::handleMessageAsync<Messages::NetworkConnectionToWebProcess::GetProcessDisplayName>(connection, decoder, this, &NetworkConnectionToWebProcess::getProcessDisplayName);
#endif
#if PLATFORM(MAC)
    if (decoder.messageName() == Messages::NetworkConnectionToWebProcess::UpdateActivePages::name())
        return IPC::handleMessage<Messages::NetworkConnectionToWebProcess::UpdateActivePages>(connection, decoder, this, &NetworkConnectionToWebProcess::updateActivePages);
#endif
    if (decoder.messageName() == Messages::NetworkConnectionToWebProcess::SetResourceLoadSchedulingMode::name())
        return IPC::handleMessage<Messages::NetworkConnectionToWebProcess::SetResourceLoadSchedulingMode>(connection, decoder, this, &NetworkConnectionToWebProcess::setResourceLoadSchedulingMode);
    if (decoder.messageName() == Messages::NetworkConnectionToWebProcess::PrioritizeResourceLoads::name())
        return IPC::handleMessage<Messages::NetworkConnectionToWebProcess::PrioritizeResourceLoads>(connection, decoder, this, &NetworkConnectionToWebProcess::prioritizeResourceLoads);
    UNUSED_PARAM(connection);
    UNUSED_PARAM(decoder);
#if ENABLE(IPC_TESTING_API)
    if (connection.ignoreInvalidMessageForTesting())
        return;
#endif // ENABLE(IPC_TESTING_API)
    ASSERT_NOT_REACHED_WITH_MESSAGE("Unhandled message %s to %" PRIu64, IPC::description(decoder.messageName()), decoder.destinationID());
}

bool NetworkConnectionToWebProcess::didReceiveSyncNetworkConnectionToWebProcessMessage(IPC::Connection& connection, IPC::Decoder& decoder, UniqueRef<IPC::Encoder>& replyEncoder)
{
    Ref protectedThis { *this };
    if (decoder.messageName() == Messages::NetworkConnectionToWebProcess::PerformSynchronousLoad::name())
        return IPC::handleMessageSynchronous<Messages::NetworkConnectionToWebProcess::PerformSynchronousLoad>(connection, decoder, replyEncoder, this, &NetworkConnectionToWebProcess::performSynchronousLoad);
    if (decoder.messageName() == Messages::NetworkConnectionToWebProcess::TestProcessIncomingSyncMessagesWhenWaitingForSyncReply::name())
        return IPC::handleMessageSynchronous<Messages::NetworkConnectionToWebProcess::TestProcessIncomingSyncMessagesWhenWaitingForSyncReply>(connection, decoder, replyEncoder, this, &NetworkConnectionToWebProcess::testProcessIncomingSyncMessagesWhenWaitingForSyncReply);
    if (decoder.messageName() == Messages::NetworkConnectionToWebProcess::CookiesForDOM::name())
        return IPC::handleMessageSynchronous<Messages::NetworkConnectionToWebProcess::CookiesForDOM>(connection, decoder, replyEncoder, this, &NetworkConnectionToWebProcess::cookiesForDOM);
    if (decoder.messageName() == Messages::NetworkConnectionToWebProcess::CookieRequestHeaderFieldValue::name())
        return IPC::handleMessageSynchronous<Messages::NetworkConnectionToWebProcess::CookieRequestHeaderFieldValue>(connection, decoder, replyEncoder, this, &NetworkConnectionToWebProcess::cookieRequestHeaderFieldValue);
    if (decoder.messageName() == Messages::NetworkConnectionToWebProcess::GetRawCookies::name())
        return IPC::handleMessageSynchronous<Messages::NetworkConnectionToWebProcess::GetRawCookies>(connection, decoder, replyEncoder, this, &NetworkConnectionToWebProcess::getRawCookies);
    if (decoder.messageName() == Messages::NetworkConnectionToWebProcess::DomCookiesForHost::name())
        return IPC::handleMessageSynchronous<Messages::NetworkConnectionToWebProcess::DomCookiesForHost>(connection, decoder, replyEncoder, this, &NetworkConnectionToWebProcess::domCookiesForHost);
    if (decoder.messageName() == Messages::NetworkConnectionToWebProcess::BlobSize::name())
        return IPC::handleMessageSynchronous<Messages::NetworkConnectionToWebProcess::BlobSize>(connection, decoder, replyEncoder, this, &NetworkConnectionToWebProcess::blobSize);
    if (decoder.messageName() == Messages::NetworkConnectionToWebProcess::GetNetworkLoadInformationResponse::name())
        return IPC::handleMessageSynchronous<Messages::NetworkConnectionToWebProcess::GetNetworkLoadInformationResponse>(connection, decoder, replyEncoder, this, &NetworkConnectionToWebProcess::getNetworkLoadInformationResponse);
    if (decoder.messageName() == Messages::NetworkConnectionToWebProcess::GetNetworkLoadIntermediateInformation::name())
        return IPC::handleMessageSynchronous<Messages::NetworkConnectionToWebProcess::GetNetworkLoadIntermediateInformation>(connection, decoder, replyEncoder, this, &NetworkConnectionToWebProcess::getNetworkLoadIntermediateInformation);
    if (decoder.messageName() == Messages::NetworkConnectionToWebProcess::TakeNetworkLoadInformationMetrics::name())
        return IPC::handleMessageSynchronous<Messages::NetworkConnectionToWebProcess::TakeNetworkLoadInformationMetrics>(connection, decoder, replyEncoder, this, &NetworkConnectionToWebProcess::takeNetworkLoadInformationMetrics);
    UNUSED_PARAM(connection);
    UNUSED_PARAM(decoder);
    UNUSED_PARAM(replyEncoder);
#if ENABLE(IPC_TESTING_API)
    if (connection.ignoreInvalidMessageForTesting())
        return false;
#endif // ENABLE(IPC_TESTING_API)
    ASSERT_NOT_REACHED_WITH_MESSAGE("Unhandled synchronous message %s to %" PRIu64, description(decoder.messageName()), decoder.destinationID());
    return false;
}

} // namespace WebKit
