/*
 * Copyright (C) 2021 Apple Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1.  Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 * 2.  Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE INC. AND ITS CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL APPLE INC. OR ITS CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#if ENABLE(SERVICE_WORKER)
#include "WebSWContextManagerConnection.h"

#include "ArgumentCoders.h"
#include "DataReference.h"
#include "Decoder.h"
#include "FormDataReference.h"
#include "HandleMessage.h"
#include "WebCoreArgumentCoders.h"
#include "WebPreferencesStore.h"
#include "WebSWContextManagerConnectionMessages.h"
#include <WebCore/FetchIdentifier.h>
#include <WebCore/FetchOptions.h>
#include <WebCore/MessageWithMessagePorts.h>
#include <WebCore/ResourceRequest.h>
#if ENABLE(SHAREABLE_RESOURCE) && PLATFORM(COCOA)
#include <WebCore/ScriptBuffer.h>
#endif
#include <WebCore/ServiceWorkerClientData.h>
#include <WebCore/ServiceWorkerContextData.h>
#include <WebCore/ServiceWorkerData.h>
#include <WebCore/ServiceWorkerIdentifier.h>
#include <WebCore/ServiceWorkerTypes.h>
#include <WebCore/WorkerThreadMode.h>
#include <optional>
#if ENABLE(SHAREABLE_RESOURCE) && PLATFORM(COCOA)
#include <wtf/HashMap.h>
#endif
#if ENABLE(SHAREABLE_RESOURCE) && PLATFORM(COCOA)
#include <wtf/URLHash.h>
#endif
#include <wtf/Vector.h>
#include <wtf/text/WTFString.h>

namespace Messages {

namespace WebSWContextManagerConnection {

void FirePushEvent::callReply(IPC::Decoder& decoder, CompletionHandler<void(bool&&)>&& completionHandler)
{
    std::optional<bool> result;
    decoder >> result;
    if (!result) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*result));
}

void FirePushEvent::cancelReply(CompletionHandler<void(bool&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<bool>::create());
}

void FirePushEvent::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, bool result)
{
    encoder.get() << result;
    connection.sendSyncReply(WTFMove(encoder));
}

} // namespace WebSWContextManagerConnection

} // namespace Messages

namespace WebKit {

void WebSWContextManagerConnection::didReceiveMessage(IPC::Connection& connection, IPC::Decoder& decoder)
{
    if (decoder.messageName() == Messages::WebSWContextManagerConnection::InstallServiceWorker::name())
        return IPC::handleMessage<Messages::WebSWContextManagerConnection::InstallServiceWorker>(connection, decoder, this, &WebSWContextManagerConnection::installServiceWorker);
    if (decoder.messageName() == Messages::WebSWContextManagerConnection::UpdateAppInitiatedValue::name())
        return IPC::handleMessage<Messages::WebSWContextManagerConnection::UpdateAppInitiatedValue>(connection, decoder, this, &WebSWContextManagerConnection::updateAppInitiatedValue);
    if (decoder.messageName() == Messages::WebSWContextManagerConnection::StartFetch::name())
        return IPC::handleMessage<Messages::WebSWContextManagerConnection::StartFetch>(connection, decoder, this, &WebSWContextManagerConnection::startFetch);
    if (decoder.messageName() == Messages::WebSWContextManagerConnection::CancelFetch::name())
        return IPC::handleMessage<Messages::WebSWContextManagerConnection::CancelFetch>(connection, decoder, this, &WebSWContextManagerConnection::cancelFetch);
    if (decoder.messageName() == Messages::WebSWContextManagerConnection::ContinueDidReceiveFetchResponse::name())
        return IPC::handleMessage<Messages::WebSWContextManagerConnection::ContinueDidReceiveFetchResponse>(connection, decoder, this, &WebSWContextManagerConnection::continueDidReceiveFetchResponse);
    if (decoder.messageName() == Messages::WebSWContextManagerConnection::PostMessageToServiceWorker::name())
        return IPC::handleMessage<Messages::WebSWContextManagerConnection::PostMessageToServiceWorker>(connection, decoder, this, &WebSWContextManagerConnection::postMessageToServiceWorker);
    if (decoder.messageName() == Messages::WebSWContextManagerConnection::FireInstallEvent::name())
        return IPC::handleMessage<Messages::WebSWContextManagerConnection::FireInstallEvent>(connection, decoder, this, &WebSWContextManagerConnection::fireInstallEvent);
    if (decoder.messageName() == Messages::WebSWContextManagerConnection::FireActivateEvent::name())
        return IPC::handleMessage<Messages::WebSWContextManagerConnection::FireActivateEvent>(connection, decoder, this, &WebSWContextManagerConnection::fireActivateEvent);
    if (decoder.messageName() == Messages::WebSWContextManagerConnection::FirePushEvent::name())
        return IPC::handleMessageAsync<Messages::WebSWContextManagerConnection::FirePushEvent>(connection, decoder, this, &WebSWContextManagerConnection::firePushEvent);
    if (decoder.messageName() == Messages::WebSWContextManagerConnection::TerminateWorker::name())
        return IPC::handleMessage<Messages::WebSWContextManagerConnection::TerminateWorker>(connection, decoder, this, &WebSWContextManagerConnection::terminateWorker);
#if ENABLE(SHAREABLE_RESOURCE) && PLATFORM(COCOA)
    if (decoder.messageName() == Messages::WebSWContextManagerConnection::DidSaveScriptsToDisk::name())
        return IPC::handleMessage<Messages::WebSWContextManagerConnection::DidSaveScriptsToDisk>(connection, decoder, this, &WebSWContextManagerConnection::didSaveScriptsToDisk);
#endif
    if (decoder.messageName() == Messages::WebSWContextManagerConnection::MatchAllCompleted::name())
        return IPC::handleMessage<Messages::WebSWContextManagerConnection::MatchAllCompleted>(connection, decoder, this, &WebSWContextManagerConnection::matchAllCompleted);
    if (decoder.messageName() == Messages::WebSWContextManagerConnection::SetUserAgent::name())
        return IPC::handleMessage<Messages::WebSWContextManagerConnection::SetUserAgent>(connection, decoder, this, &WebSWContextManagerConnection::setUserAgent);
    if (decoder.messageName() == Messages::WebSWContextManagerConnection::UpdatePreferencesStore::name())
        return IPC::handleMessage<Messages::WebSWContextManagerConnection::UpdatePreferencesStore>(connection, decoder, this, &WebSWContextManagerConnection::updatePreferencesStore);
    if (decoder.messageName() == Messages::WebSWContextManagerConnection::Close::name())
        return IPC::handleMessage<Messages::WebSWContextManagerConnection::Close>(connection, decoder, this, &WebSWContextManagerConnection::close);
    if (decoder.messageName() == Messages::WebSWContextManagerConnection::SetThrottleState::name())
        return IPC::handleMessage<Messages::WebSWContextManagerConnection::SetThrottleState>(connection, decoder, this, &WebSWContextManagerConnection::setThrottleState);
    if (decoder.messageName() == Messages::WebSWContextManagerConnection::ConvertFetchToDownload::name())
        return IPC::handleMessage<Messages::WebSWContextManagerConnection::ConvertFetchToDownload>(connection, decoder, this, &WebSWContextManagerConnection::convertFetchToDownload);
    UNUSED_PARAM(connection);
    UNUSED_PARAM(decoder);
#if ENABLE(IPC_TESTING_API)
    if (connection.ignoreInvalidMessageForTesting())
        return;
#endif // ENABLE(IPC_TESTING_API)
    ASSERT_NOT_REACHED_WITH_MESSAGE("Unhandled message %s to %" PRIu64, IPC::description(decoder.messageName()), decoder.destinationID());
}

} // namespace WebKit

#endif // ENABLE(SERVICE_WORKER)
