"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SamlAuthentication = void 0;

var _querystring = require("querystring");

var _security_cookie = require("../../../session/security_cookie");

var _routes = require("./routes");

var _authentication_type = require("../authentication_type");

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

class SamlAuthentication extends _authentication_type.AuthenticationType {
  constructor(config, sessionStorageFactory, router, esClient, coreSetup, logger) {
    super(config, sessionStorageFactory, router, esClient, coreSetup, logger);

    _defineProperty(this, "type", 'saml');

    this.setupRoutes();
  }

  generateNextUrl(request) {
    const path = this.coreSetup.http.basePath.serverBasePath + (request.url.path || '/app/opensearch-dashboards');
    return (0, _querystring.escape)(path);
  }

  redirectToLoginUri(request, toolkit) {
    const nextUrl = this.generateNextUrl(request);
    const clearOldVersionCookie = (0, _security_cookie.clearOldVersionCookieValue)(this.config);
    return toolkit.redirected({
      location: `${this.coreSetup.http.basePath.serverBasePath}/auth/saml/login?nextUrl=${nextUrl}`,
      'set-cookie': clearOldVersionCookie
    });
  }

  setupRoutes() {
    const samlAuthRoutes = new _routes.SamlAuthRoutes(this.router, this.config, this.sessionStorageFactory, this.securityClient, this.coreSetup);
    samlAuthRoutes.setupRoutes();
  }

  requestIncludesAuthInfo(request) {
    return request.headers[SamlAuthentication.AUTH_HEADER_NAME] ? true : false;
  }

  getAdditionalAuthHeader(request) {
    return {};
  }

  getCookie(request, authInfo) {
    return {
      username: authInfo.user_name,
      credentials: {
        authHeaderValue: request.headers[SamlAuthentication.AUTH_HEADER_NAME]
      },
      authType: this.type,
      expiryTime: Date.now() + this.config.session.ttl
    };
  }

  async isValidCookie(cookie) {
    var _cookie$credentials;

    return cookie.authType === this.type && cookie.username && cookie.expiryTime && ((_cookie$credentials = cookie.credentials) === null || _cookie$credentials === void 0 ? void 0 : _cookie$credentials.authHeaderValue);
  }

  handleUnauthedRequest(request, response, toolkit) {
    if (this.isPageRequest(request)) {
      return this.redirectToLoginUri(request, toolkit);
    } else {
      return response.unauthorized();
    }
  }

  buildAuthHeaderFromCookie(cookie) {
    var _cookie$credentials2;

    const headers = {};
    headers[SamlAuthentication.AUTH_HEADER_NAME] = (_cookie$credentials2 = cookie.credentials) === null || _cookie$credentials2 === void 0 ? void 0 : _cookie$credentials2.authHeaderValue;
    return headers;
  }

}

exports.SamlAuthentication = SamlAuthentication;

_defineProperty(SamlAuthentication, "AUTH_HEADER_NAME", 'authorization');
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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