"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.migrateTenantIndices = migrateTenantIndices;
exports.setupIndexTemplate = setupIndexTemplate;

var _core = require("../../../../src/core/server/saved_objects/migrations/core");

var _build_index_map = require("../../../../src/core/server/saved_objects/migrations/core/build_index_map");

var _opensearch_dashboards_migrator = require("../../../../src/core/server/saved_objects/migrations/opensearch_dashboards/opensearch_dashboards_migrator");

/*
 *   Copyright OpenSearch Contributors
 *
 *   Licensed under the Apache License, Version 2.0 (the "License").
 *   You may not use this file except in compliance with the License.
 *   A copy of the License is located at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   or in the "license" file accompanying this file. This file is distributed
 *   on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 *   express or implied. See the License for the specific language governing
 *   permissions and limitations under the License.
 */
async function setupIndexTemplate(esClient, opensearchDashboardsIndex, typeRegistry, logger) {
  const mappings = (0, _core.buildActiveMappings)((0, _opensearch_dashboards_migrator.mergeTypes)(typeRegistry.getAllTypes()));

  try {
    await esClient.indices.putTemplate({
      name: 'tenant_template',
      body: {
        index_patterns: [opensearchDashboardsIndex + '_-*_*', opensearchDashboardsIndex + '_0*_*', opensearchDashboardsIndex + '_1*_*', opensearchDashboardsIndex + '_2*_*', opensearchDashboardsIndex + '_3*_*', opensearchDashboardsIndex + '_4*_*', opensearchDashboardsIndex + '_5*_*', opensearchDashboardsIndex + '_6*_*', opensearchDashboardsIndex + '_7*_*', opensearchDashboardsIndex + '_8*_*', opensearchDashboardsIndex + '_9*_*'],
        settings: {
          number_of_shards: 1
        },
        mappings
      }
    });
  } catch (error) {
    logger.error(error);
    throw error;
  }
}

async function migrateTenantIndices(opensearchDashboardsVersion, migrationClient, securityClient, typeRegistry, serializer, logger) {
  let tenentInfo;

  try {
    tenentInfo = await securityClient.getTenantInfoWithInternalUser();
  } catch (error) {
    logger.error(error);
    throw error;
  } // follows the same approach in opensearch_dashboards_migrator.ts to initiate DocumentMigrator here
  // see: https://tiny.amazon.com/foi0x1wt/githelaskibablobe4c1srccore


  const documentMigrator = new _core.DocumentMigrator({
    opensearchDashboardsVersion,
    typeRegistry,
    log: logger
  });

  for (const indexName of Object.keys(tenentInfo)) {
    const indexMap = (0, _build_index_map.createIndexMap)({
      opensearchDashboardsIndexName: indexName,
      indexMap: (0, _opensearch_dashboards_migrator.mergeTypes)(typeRegistry.getAllTypes()),
      registry: typeRegistry
    }); // follows the same aporach in opensearch_dashboards_mirator.ts to construct IndexMigrator
    // see: https://tiny.amazon.com/9cdcchz5/githelaskibablobe4c1srccore
    //
    // FIXME: hard code batchSize, pollInterval, and scrollDuration for now
    //        they are used to fetched from `migration.xxx` config, which is not accessible from new playform

    const indexMigrator = new _core.IndexMigrator({
      batchSize: 100,
      client: migrationClient,
      documentMigrator,
      index: indexName,
      log: logger,
      mappingProperties: indexMap[indexName].typeMappings,
      pollInterval: 1500,
      // millisec
      scrollDuration: '15m',
      serializer,
      obsoleteIndexTemplatePattern: undefined,
      convertToAliasScript: indexMap[indexName].script
    });

    try {
      await indexMigrator.migrate();
    } catch (error) {
      logger.error(error); // fail early, exit the kibana process
      // NOTE: according to https://github.com/elastic/kibana/issues/41983 ,
      //       PR https://github.com/elastic/kibana/pull/75819 , API to allow plugins
      //       to set status will be available in 7.10, for now, we fail OpenSearchDashboards
      //       process to indicate index migration error. Customer can fix their
      //       tenant indices in ES then restart OpenSearchDashboards.

      process.exit(1);
    }
  }
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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