package PAR::StrippedPARL::Dynamic;
use 5.008001;
use strict;
use warnings;
our $VERSION = '0.958';

use base 'PAR::StrippedPARL::Base';

our $Data_Pos = tell DATA;

=head1 NAME

PAR::StrippedPARL::Dynamic - Data package containing a dynamic PARL

=head1 SYNOPSIS

  # For details, see PAR::StrippedPARL::Base.
  PAR::StrippedPARL::Dynamic->write_parl($file) or die "Some error...";

=head1 DESCRIPTION

This class is internal to PAR. Do not use it outside of PAR.

This class is basically just a container for a dynamic binary PAR loader
which doesn't include the PAR code like the F<parl> or F<parl.exe>
you are used to. If you're really curious, I'll tell you it is
just a copy of the F<myldr/par> (or F<myldr/par.exe>) file.

The data is appended during the C<make> phase of the PAR build process,
but only if applicable: If you perl is static, you won't get the dynamic
loader.

If the binary data isn't appended during the build process, the class
methods will return the empty list.

=head1 CLASS METHODS

Inherits the methods from L<PAR::StrippedPARL::Base>.

=cut

sub _data_pos {
    my $class = shift;
    return $Data_Pos;
}

=head1 AUTHORS

Steffen Mueller E<lt>smueller@cpan.orgE<gt>,
Audrey Tang E<lt>cpan@audreyt.orgE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright 2006-2009 by Steffen Mueller E<lt>smueller@cpan.orgE<gt>.

This program is free software; you can redistribute it and/or 
modify it under the same terms as Perl itself.

See F<LICENSE>.

=cut

$^V eq 5.36.0 
    or die sprintf("Perl (%s) version (%vd) doesn't match the version (5.36.0) ".
                   "that PAR::Packer was built with; please rebuild PAR::Packer", 
                   $^X, $^V);

1;

__DATA__
M?T5,1@(!`0````````````,`MP`!````P!X```````!``````````+#!````
M`````````$``.``)`$``&@`9``8````$````0`````````!``````````$``
M````````^`$```````#X`0````````@``````````P````0````X`@``````
M`#@"````````.`(````````;`````````!L``````````0`````````!````
M!0```````````````````````````````````""L````````(*P`````````
M``$```````$````&````6+L```````!8NP$``````%B[`0``````*`4`````
M``!8%0```````````0```````@````8```!HNP```````&B[`0``````:+L!
M```````@`@```````"`"````````"``````````$````!````%0"````````
M5`(```````!4`@```````$0`````````1``````````$`````````%#E=&0$
M````\*0```````#PI````````/"D````````_`````````#\``````````0`
M````````4>5T9`8`````````````````````````````````````````````
M````````````$`````````!2Y71D!````%B[````````6+L!``````!8NP$`
M`````*@$````````J`0````````!`````````"]L:6(O;&0M;&EN=7@M86%R
M8V@V-"YS;RXQ```$````%`````,```!'3E4`UHHDW9'2I?#QJ+Z3OTFY64B,
MG/P$````$`````$```!'3E4```````,````'`````````"X```!*````!```
M``@````A$!``!<`$A"@"$`6`AD"$@@D`!`D@!(B(@!!!N@`""$H`````````
M3````$X`````````3P````````!0````40````````!2````4P```%0```!5
M`````````%<``````````````%@`````````60``````````````````````
M``!:````6P```%P```!=`````````%X``````````````&``````````80``
M`&(```````````````````!C````9`````````!E```````````````NQ9NV
M:W^:?`PZEPL5C"7,0_K7AZ/@^TE#1=7LB0=PT]EQ6!R_VE<;`:\TZ'ZIG)O/
MWD<MP^4_+-5DU'PS8MOMX:<""EGOEWEOQ=\?:PE#UG94/)0A_^0.N^.2?*U+
MX\"S`8MO:QJ>4.-'Q@%FK\QF^Z%L8```````````````````````````````
M`````````````P`+`,`;`````````````````````````P`6``#``0``````
M```````````H`0``$@````````````````````````#>`0``$@``````````
M``````````````"`````$@````````````````````````"P````$@``````
M``````````````````"%`@``$@````````````````````````#S`0``$@``
M``````````````````````!E`0``$@`````````````````````````!````
M(`````````````````````````"C`@``$@````````````````````````"A
M`0``$@````````````````````````"C`P``$@``````````````````````
M``"G````$@````````````````````````#F`@``$0``````````````````
M``````!3````$@````````````````````````!N`@``$@``````````````
M``````````"[````$@````````````````````````!!`@``$@``````````
M``````````````">`@``$@````````````````````````!)`@``$@``````
M``````````````````"&`0``(@`````````````````````````0````(```
M```````````````````````@`@``$@`````````````````````````(`0``
M$@`````````````````````````L````(``````````````````````````7
M`@``$@`````````````````````````!`@``$@``````````````````````
M``!V`@``$@````````````````````````"]`@``$@``````````````````
M``````!G`@``$0````````````````````````!3`@``$@``````````````
M``````````!X`0``$@````````````````````````#U````%@``````````
M``````````````"7````$@````````````````````````!O````$@``````
M```````````````````G`@``$@````````````````````````!4`0``$@``
M``````````````````````#8````$@`````````````````````````)`@``
M$@````````````````````````"-`@``$@````````````````````````#4
M`@``$@````````````````````````"5`0``$@``````````````````````
M``#"`0``$@````````````````````````#)````$@``````````````````
M``````"L`@``$@````````````````````````#G````$@``````````````
M```````````^`0``$@````````````````````````!C````$@``````````
M``````````````!U`@``$@````````````````````````#0`0``$@``````
M``````````````````#E`0``$@````````````````````````!]`@``$@``
M```````````````````````Z`@``$@`````````````````````````3`0``
M$@````````````````````````":`0``$@`````````````````````````R
M`0``$@````````````````````````#$`@``$@``````````````````````
M```0`@``$@````````````````````````!&````$0``````````````````
M```````M`0``$@````````````````````````",````$@``````````````
M``````````!%`0``$@````````````````````````"H`0``$@``````````
M```````````````M`@``$@````````````````````````"P`0``$@``````
M``````````````````#*`@``$0````````````````````````#7`0``$@``
M``````````````````````#+`@``(0`````````````````````````S`@``
M$@````````````````````````!9`@``$@````````````````````````#,
M`@``$0````````````````````````#Z`0``$@``````````````````````
M```C`P``$@`-`-`H````````!`````````"]`0``$@`-`$`<````````;`(`
M``````#)`P``(``6``#``0`````````````````5!```$@`-`,!@````````
M2`````````#B`P``$@`-`!!A````````'`$```````!@!```$@`-`-`L````
M````S`0````````P!```$``6`(#``0````````````````!6!```$0`6`'C`
M`0``````"``````````)!```$``7`(#``0````````````````#]`P``$``7
M`+#0`0````````````````#^`P``$``7`+#0`0````````````````"1`P``
M$@`-`$`R````````,``````````C!```$@`-`-`S````````:`@```````#M
M`P``$@`-`(`G````````-`````````!!`P``$@`-`$`\````````;`(`````
M``#-`P``$@`-`,`>````````-`````````!E`P``$@`-`+0G````````%`$`
M``````#U`P``$``7`+#0`0````````````````"L`P``$@`-`+`^````````
M1`$```````#'`P``$``6``#``0````````````````#4`P``$``7`(#``0``
M``````````````!)!```$@`-`-0H````````D`(````````>!```$``7`+#0
M`0````````````````"X`P``$0`/`$!D````````!`````````!7`P``$@`-
M`*`Q````````F`````````"'`P``$@`-`#!B````````V`$````````T`P``
M$@`-`&0K````````)`````````![`P``$@`-`)`K````````.`$````````W
M!```$@`-`'`R````````8`$`````````7U]G;6]N7W-T87)T7U\`7TE435]D
M97)E9VES=&5R5$U#;&]N951A8FQE`%])5$U?<F5G:7-T97)434-L;VYE5&%B
M;&4`4$Q?9&]?=6YD=6UP`%!E<FQ?9W9?9F5T8VAP=@!097)L7VUG7W-E=`!0
M97)L7V%T9F]R:U]L;V-K`%!E<FQ?9V5T7W-V`%!E<FQ?;F5W6%,`8F]O=%]$
M>6YA3&]A9&5R`'!E<FQ?<G5N`'!E<FQ?<&%R<V4`4&5R;%]S=E]S971P=@!0
M97)L7W-V7W-E='!V;@!P97)L7V-O;G-T<G5C=`!P97)L7V1E<W1R=6-T`%!,
M7V-U<G)E;G1?8V]N=&5X=`!P97)L7V%L;&]C`%!E<FQ?8W)O86M?;F]C;VYT
M97AT`'!E<FQ?9G)E90!097)L7W-A9F5S>7-M86QL;V,`4&5R;%]S>7-?:6YI
M=#,`4&5R;%]N97=84U]F;&%G<P!097)L7V%T9F]R:U]U;FQO8VL`4&5R;%]S
M>7-?=&5R;0!?7V-X85]F:6YA;&EZ90!R96%D`'-T<F-H<@!G971P:60`;65M
M;6]V90!?7VQI8F-?<W1A<G1?;6%I;@!?7V9P<FEN=&9?8VAK`'-T<F-M<`!O
M<&5N-C0`=6YL:6YK`%]?<W1R;F-P>5]C:&L`<W1R=&]K`&UE;7-E=`!S=')N
M8VUP`&=E='5I9`!M96UC:'(`8VQO<V5D:7(`<W1R<W1R`')M9&ER`&-L;W-E
M`'-T<FQE;@!S=')D=7``<F5A;&QO8P!R96%D9&ER-C0`86)O<G0`7U]S<')I
M;G1F7V-H:P!S=&1E<G(`;65M8W!Y`&QS=&%T-C0`;W!E;F1I<@!L<V5E:S8T
M`%]?97)R;F]?;&]C871I;VX`97AI=`!R96%D;&EN:P!?7W-T86-K7V-H:U]F
M86EL`&%C8V5S<P!M:V1I<@!?7V5N=FER;VX`7U]R96=I<W1E<E]A=&9O<FL`
M7U]S=&%C:U]C:&M?9W5A<F0`;&EB<&5R;"YS;P!L:6)C+G-O+C8`;&0M;&EN
M=7@M86%R8V@V-"YS;RXQ`'!A<E]C=7)R96YT7V5X96,`<&%R7V)A<V5N86UE
M`%A37TEN=&5R;F%L<U]005)?0D]/5`!P87)?96YV7V-L96%N`'!A<E]C=7)R
M96YT7V5X96-?<')O8P!P87)?9&ER;F%M90!S:&%?9FEN86P`9V5T7W5S97)N
M86UE7V9R;VU?9V5T<'=U:60`<&%R7V-L96%N=7``7TE/7W-T9&EN7W5S960`
M7U]D871A7W-T87)T`%]?8G-S7W-T87)T7U\`<VAA7W5P9&%T90!X<U]I;FET
M`%]?96YD7U\`7U]B<W-?96YD7U\`7U]B<W-?<W1A<G0`<VAA7VEN:70`7V5N
M9`!P87)?;6MT;7!D:7(`7V5D871A`'!A<E]S971U<%]L:6)P871H`'!A<E]F
M:6YD<')O9P!M>5]P87)?<&P`<&%R7VEN:71?96YV`$=,24)#7S(N,3<`1TQ)
M0D-?,BXS-`!'3$E"0U\R+C,S`"]U<W(O;&EB+W!E<FPU+S4N,S8O8V]R95]P
M97)L+T-/4D4```````````$``@`!``$``@`"``$``0`"``(``@`!``,``0`"
M``$``@`"``(``@`!``(``0`!``(``@`$``(``@`"``$``0`!``$``@`!``$`
M`@`"``(``@`"``$``@`!``(``0`$``(``@`"``(``0`"``$``@`"``$``@`!
M``$``@`"``4``@`"``(``@`"``(``@`!``$``0`!``$``0`!``$``0`!``$`
M`0`!``$``0`!``$``0`!``$``0`!``$``0`!``$``0`!``$``0`!``T#```0
M````(````)>1E@8```,`;00````````!``,``P,``!``````````M)&6!@``
M!0!X!```$````+.1E@8```0`@P0``!````"7D98&```"`&T$````````6+L!
M```````#!````````-`?````````8+L!```````#!````````(`?````````
M.+X!```````#!````````#!B````````V+X!```````#!````````!!A````
M````0+\!```````#!````````!!D````````4+\!```````#!````````,!@
M````````\+\!```````#!````````$`<````````",`!```````#!```````
M``C``0``````&,`!```````#!````````/!G````````(,`!```````#!```
M`````+!D````````,,`!```````#!````````*!E````````.,`!```````#
M!````````-!G````````0,`!```````#!````````-AG````````2,`!````
M```#!````````.!G````````4,`!```````#!````````.AG````````8,`!
M```````#!````````+AG````````:,`!```````#!````````,!G````````
M>,`!```````#!````````/AG````````P+T!```````!!````P``````````
M````R+T!```````!!```!```````````````T+T!```````!!```!0``````
M````````V+T!```````!!```!@``````````````X+T!```````!!```!P``
M````````````Z+T!```````!!```"```````````````\+T!```````!!```
M"0``````````````^+T!```````!!```"@```````````````+X!```````!
M!```"P``````````````"+X!```````!!```#```````````````$+X!````
M```!!```#0``````````````&+X!```````!!```#@``````````````(+X!
M```````!!```#P``````````````*+X!```````!!```$```````````````
M,+X!```````!!```$0``````````````0+X!```````!!```$@``````````
M````2+X!```````!!```$P``````````````4+X!```````!!```%```````
M````````6+X!```````!!```%0``````````````8+X!```````!!```%@``
M````````````:+X!```````!!```%P``````````````<+X!```````!!```
M&```````````````>+X!```````!!```&0``````````````@+X!```````!
M!```&@``````````````B+X!```````!!```&P``````````````D+X!````
M```!!```'```````````````F+X!```````!!```'0``````````````H+X!
M```````!!```'@``````````````J+X!```````!!```'P``````````````
ML+X!```````!!```(0``````````````N+X!```````&!```(@``````````
M````P+X!```````!!```(P``````````````R+X!```````!!```)```````
M````````T+X!```````!!```)0``````````````X+X!```````!!```)@``
M````````````Z+X!```````!!```)P``````````````\+X!```````!!```
M*```````````````^+X!```````!!```*0```````````````+\!```````!
M!```*@``````````````"+\!```````!!```*P``````````````$+\!````
M```!!```+```````````````&+\!```````!!```+0``````````````(+\!
M```````!!```+@``````````````*+\!```````!!```+P``````````````
M,+\!```````!!```,```````````````.+\!```````!!```,0``````````
M````2+\!```````!!```,@``````````````6+\!```````!!```,P``````
M````````8+\!```````!!```-```````````````:+\!```````!!```-0``
M````````````<+\!```````!!```-@``````````````>+\!```````!!```
M-P``````````````@+\!```````!!```.```````````````B+\!```````!
M!```.0``````````````D+\!```````!!```.@``````````````F+\!````
M```!!```.P``````````````H+\!```````!!```/```````````````J+\!
M```````!!```/0``````````````L+\!```````!!```/@``````````````
MN+\!```````!!```/P``````````````P+\!```````!!```0```````````
M````R+\!```````!!```00``````````````T+\!```````!!```1```````
M````````V+\!```````!!```1@``````````````X+\!```````!!```1P``
M````````````Z+\!```````!!```2```````````````^+\!```````!!```
M20``````````````H+T!```````"!```"@``````````````J+T!```````"
M!```(```````````````L+T!```````"!```0@``````````````'R`#U?U[
MOZG]`P"1R@``E/U[P:C``U_6``````````#P>[^IT```T!'.1OD08C:1(`(?
MUA\@`]4?(`/5'R`#U=```-`1TD;Y$((VD2`"'];0``#0$=9&^1"B-I$@`A_6
MT```T!':1OD0PC:1(`(?U@````````````````````#]>[JIPP``T&,01_G]
M`P"1\U,!J?5;`JGBAP.IXN,`D>$#`9'@3P"YX#,!D60`0/GD+P#Y!`"`TL,`
M`-!CW$?Y8``_UL(``-!"^$;YP```T`!D1_GA`P*JPP``T&.@1_E@`#_6P```
MT`#01_D``$`Y@`H`-,```-``7$?Y0=`[U=,``/`B:&#X]$]`N4$H4CF`$@`1
ME`X`$2$$`#)!*!(YP0``T"'$1_D`?'V3(``_UN,#`*KF(T#YP```\&)"`I'E
M3T"Y!#Q`^2$``+#'`$#Y(:`>D2```+``P!Z19P0`J4,$`/ED``&IOP0`<>T*
M`%0A`(#2'R`#U2`,`!'$>&'X(00`D638(/B_``%K;/__5*4,`!%U0@*1?]@E
M^,4``-"E[$;YX@,4*J`.0/D!``"P!`"`TB$`')&@`#_6]`,`*@`&`#1U0@*1
M]E,!D<$``-`AE$?YH`Y`^2``/]8@``"PX0,6J@#`%)&.``"48`4`M'-"`I'!
M``#0(>!&^6`.0/D@`#_6P```T`!81_D``#_6P```T``01_GB+T#Y`0!`^4(`
M`>L!`(#2P04`5/5;0JG@`Q0J\U-!J?U[QJC``U_6P```T``\1_G3``#P```_
MUF)"`I%`#`#YP`,`M,$``-`A=$?Y(``_UL```-``7$?Y0=`[U2)H8/A?W"@Y
MHO__%Z`.0/G!``#0(0Q'^2``/];T`P`JS/__%Z`*0/E@``"T%`@`E-/__Q?A
M`Q:J(```L`#@$I%;``"4H`H`^:#Y_[0,"`"4R___%X4`@%*R__\7P0``T"$H
M1_D@`(!2(``_UL```-``D$?Y```_UA\@`]4?(`/5'R`#U1\@`]4?(`/5'R`#
MU1T`@-(>`(#2Y0,`JN$#0/GB(P"1Y@,`D<```-``^$?Y`P"`T@0`@-)-__^7
M2/__E\```-``_$;Y0```M$#__Q?``U_6'R`#U1\@`]7```#P```"D<$``/`A
M``*1/P``Z\```%3!``#0(31'^6$``+3P`P&J``(?UL`#7];```#P```"D<$`
M`/`A``*1(0``RR+\?]-!#(&+(?Q!D\$``+3"``#00D!'^6(``+3P`P*J``(?
MUL`#7]8?(`/5_7N^J?T#`)'S"P#YTP``\&`"0CE``0`UP0``T"$P1_F!``"T
MP```\``$0/D@`#_6V/__ER``@%)@`@(Y\PM`^?U[PJC``U_6'R`#U1\@`]7<
M__\7'R`#U1\@`]4?(`/5_7N[J?T#`)'S4P&I(`<`M/=C`ZGW`P"J0(`,$`#T
M1_GY(P#Y&0!`^9D&`+3U6P*I]@,!JA4`@%+@`D`Y'_0`<0080'H!`P!4,P-`
M^?,!`+38``"P&$M'^;5^0)/T`QFJX@,5JN$#%ZK@`Q.J``,_UH```#5@:G4X
M'_0`<2`"`%23CD#X\_[_M?5;0JGW8T.I^2-`^>`#$ZKS4T&I_7O%J,`#7];U
M`Q>JH!Y`.!_T`'$$&$!ZH?__5+4"%TOC__\7@`(9R[4&`)%S`A6+`/Q#D\`"
M`+G@`Q.J\U-!J?5;0JGW8T.I^2-`^?U[Q:C``U_6$P"`TNC__Q?W8T.I$P"`
MTODC0/GD__\7'R`#U1\@`]7]>[6IP@``L$(01_G]`P"1\PL`^>&C`)'S`P"J
M0P!`^>-7`/D#`(#2P@``L$*D1_E``#_6(`(`->$[0+D"`)12(0P4$C\00'$D
M$$)ZH`$`5,$``+`A$$?YXU=`^2(`0/EC``+K`@"`TL$!`%3S"T#Y_7O+J,`#
M7]8``(!2]?__%\(``+!"4$?YX`,3JD$`@%)``#_6'P``<>`7GQKM__\7P```
ML`"01_D``#_6'R`#U<$``+`A$$?Y_7NSJ?T#`)$B`$#YXF<`^0(`@-+```"T
M\U,!J?,#`*H!`$`Y80$`-?-30:G```"P`!!'^>)G0/D!`$#Y0@`!ZP$`@-)A
M"P!4_7O-J,`#7];!``"P(;1'^2``/];T`P"J0/[_M/5;`JG6``"PUBY'^>`#
M%*K``C_60`<`M`%,0#D53`"1/[@`<8$``%2A!D`Y`?__-!\@`]4`3$`Y'[@`
M<6`'`%3W8P.IX`,3JM@``+`8[T?Y``,_UO<#`*K@`Q6J``,_UL$``+`AF$?Y
MX`(`"P`(`!$`?$"3(``_UL8``+#&\$?YY0,5J@(`@)+U`P"J(0"`4N0#$ZHC
M``"08R`1D<``/];"``"P0DQ'^>`#%:KA(P&10``_UA\$`#&@``!4X%M`N0`,
M%!(?$$!QP`,`5,$``+`AY$;YX`,5JB``/];!``"P(=1'^>`#%:H@`#_6]V-#
MJ>`#%*K``C_6`/G_M<$``+`A1$?YX`,4JB``/];!``"P(6A'^>`#$ZH@`#_6
M\U-!J?5;0JFI__\7H`9`.1^X`'&!^/]4H`I`.:#V_S3!__\7X`,5JI3__Y?E
M__\7P```L`"01_GS4P&I]5L"J?=C`ZD``#_6#`B"TO]C+,O#``"P8Q!'^>+_
M@=+]>P"I_0,`D?-3`:GS`P"J]`,!JO5;`JGUXP"1X0,5JF``0/G@'PCY``"`
MTL,``+!C`$?Y(```D`!`$9%@`#_6'P``<<T``%0$!`!1X@,`*J#*9#A``P`U
MY`(`-<```+``7$?Y0]`[U>(#%*KA`Q.J8&A@^,,``+!C($?Y8``_UL```+``
M$$?YXA](^0$`0/E"``'K`0"`TH$#`%3]>T"I#`B"TO-30:GU6T*I_V,LB\`#
M7];B`P0J'R`#U<,``+!CS$?Y5GQ`D^(#%JK@`Q6JX06`4F``/]8@_/^TP```
ML`!<1_E$T#O5XP,6JN(#%:KA`Q.J@&A@^,0``+"$C$?Y@``_UM___Q?```"P
M`)!'^0``/]8?(`/5'R`#U?U[N:G]`P"1\U,!J?0#`:KS`P"J]5L"J?=C`ZGY
M:P2I(`!`.<$``+`A$$?Y'_0`<=<``+#W[D?Y(@!`^>(W`/D"`(#2E!:4FN`#
M%*K@`C_6]@,`JN&3`9'@`Q.JN?[_EV`(`+3U`P"JX`(_UM\"`.LI!P!4]V>`
MN=@``+`8]T?YV@``L%J;1_GW\GW38`)`.1_T`'$$&$!ZP`P`5.0#$ZJ#'$`X
M?_0`<6080'JA__]4A``3RX1\0),5`T#YP`H`D0``!(M``S_6H&HW^*`"`+0!
M`T#YX`,3JB-H=_@!%$`X80``.3_T`'$D&$!ZP```5`(40#AB'``X7_0`<408
M0'J!__]4H0>`4@``@-)A%``X@FI@.&)H(#@`!`"1HO__-<```+``$$?YXC=`
M^0$`0/E"``'K`0"`TF$(`%3S4T&I]5M"J?=C0ZGY:T2I_7O'J,`#7]8!`(#2
M@FIA.*)J(3@A!`"1HO__->[__Q?8``"P&/='^?LK`/D;`T#Y8`-`^<`%`+3@
M`QNJ%0"`4A\@`]4!C$#XX@,5*K4&`!&A__^U0`P`$0!\?9/!``#0(I!`N8("
M`#7:``"P6IM'^2(`@%(BD`"Y0`,_UOD#`*I`!`"TPP``L&/@1_FW?GV3X0,;
MJN(#%ZI@`#_6&0,`^:`&`!'U9P"Y/]L@^/LK0/FD__\7P@``L$(D1_GA`P"J
MX`,;JD``/]8``P#Y^0,`JL`!`+3:``"P6IM'^;=^?9/P__\7!`"`TJ'__Q<`
M`H#2%0"`4MK__Q?```"P`)!'^?LK`/D``#_6^RM`^;+__Q<?(`/5_7N^J4,`
M`)`A``"0_0,`D?-3`:E4T#O5TP``L'->1_G"``"P0F!'^6.@$I'$``"PA-A'
M^2&`$9&`:G/X@``_UH!J<_@D``"0\U-!J82`'I']>\*HXP,$JM```+`0<D?Y
M`@``L"$``)!"`#&1(0`2D04`@%(``A_6'R`#U1\@`]7```"P`%Q'^430.]7"
M``"P0F!'^4,``)#0``"P$-I'^6.@$I&`:&#X(0``D"&`$9$``A_6#`J$TO]C
M+,O```"P`!!'^0$(@M+]>P"I_0,`D?-3`:GSXP"1]1,`^?5C(8L!`$#YX2<0
M^0$`@-+```"P``1'^0``/];D`P`JQ@``L,;P1_DB`(+2(0"`4N`#$ZHE``"0
M(P``D*5@$I%C@!*1P``_UL,``+!C`$?YX`,3JN$#%:H"`(+28``_UB`$^#?!
M``"P(9A'^?,#`*H`!``1`'Q`DR``/];T`P"J(`,`M,,``+!C&$?Y<WY`D^$#
M%:KB`Q.J8``_UI]J,SC```"P`!!'^>(G4/D!`$#Y0@`!ZP$`@-(!`0!4_7M`
MJ>`#%*KS4T&I#`J$TO430/G_8RR+P`-?UL```+``D$?Y```_UA0`@-+N__\7
M'R`#U1\@`]6Y__\7#"*"TO]C+,O"``"P0A!'^?U[`*G]`P"1\U,!J?0#`*KU
M6P*I]0,!JN$3`I%``$#YX(<(^0``@-(@``"0`.`2D;/]_Y?S`P"JP@``L$+`
M1_G@`Q2JX06`4D``/]9@`@"TX0,4JB```)``(!.1V/[_E^`#%*K!``"P(1!'
M^>.'2/DB`$#Y8P`"ZP(`@-)A#P!4_7M`J0PB@M+S4T&I]5M"J?]C+(O``U_6
M^6L$J>`#%*HV``"0V0``L#GO1_GW8P.IV@``L%J[1_G[<P6IUP``L/?V1OGZ
M/P#Y(`,_UO@#`*K@`Q6J0`,_UL%B$Y'@`C_6_`,`JL`'`+3```"P`%!'^?4C
M!)';``"P>_-'^>`[`/G:``"P6D]'^<```+``K$?YX#<`^8`#0#F@!@`TTP``
MM.(W0/GA`QRJX`,3JD``/];`!``TX`,<JB`#/]8"!`!1@<,BBX/+8CA_O`!Q
M(0$`5!\@`]7@`P(J0@0`43\``#F!PR*+@\MB.'^\`'%`__]4`0<`$2````L?
M_#]Q#`,`5.0#'*KF`Q2J)0``D",``)"E@!.18Z`3D0(`@M(A`(!2X`,5JF`#
M/];A(P*1X`,5JD`#/]:@```UX)M`N0`,%!(?($!QH`(`5,%B$Y$``(#2X`(_
MUOP#`*K@^?^U]V-#J?EK1*G[<T6IFO__%W,#`+1@`D`Y'[@`<:```%0D``"0
M(`"`4IS`$I':__\78`9`.<#]_S3Z__\7XCM`^>`#%:HA`(!20``_U@#]_S7A
M`Q6J(```D``@$Y%A_O^7X3]`^>`#%:H@`#_6]V-#J?EK1*G[<T6IA/__%R0`
M`)`@`(!2G,`2D</__Q?```"P`)!'^?=C`ZGY:P2I^W,%J0``/]8"`$`YX0,`
MJL(``#0A!`"17[P`<0`0@9HB`$`Y@O__-<`#7]8?(`/5'R`#U2`)`+3]>[ZI
M_0,`D?,+`/GS`P"J`0!`.4$&`#3!``"P(>Q'^2``/]8"!`#18@("BW\"`NNC
M``!4$```%$($`-%_`@+KP`4`5$``0#D?O`!Q8/__5'\"`NNC``!4!P``%$($
M`-%_`@+KH`0`5$``0#D?O`!Q8?__5'\"`NNA``!4'P``%$``0#D?O`!Q@0``
M5$($`-%_`@+K8___5$(`$\L``(#200@`D3\$0/%(`0!4P```T`3``I'@`P2J
M0@0`D<0``+"$L$?YX0,3J@,`@M*``#_6\PM`^?U[PJC``U_6\PM`^2```)#]
M>\*H`,`2D=```+`0ND?Y``(?UF`"0#D?O`!QX/[_5/,+0/D@``"0_7O"J`"`
M$Y'0``"P$+I'^0`"'];0``"P$+I'^2```)``P!*1``(?UA\@`]4?(`/5_7NZ
MJ<```+``$$?Y_0,`D?-3`:DT``"0\U,!D?5;`JF4PA.1]V,#J?DC`/G5``"P
MM?9'^0$`0/GA+P#Y`0"`TN$#$ZK@`Q2JLOS_E^`!`+2B`D#YXU>`N4`,`XL!
M!$#Y07@C^.'^_[0!"$#Y`00`^0`@`)&A__^UX0,3JN`#%*JD_/^78/[_M=4`
M`+"U]D?Y-```D)0"%)'A`Q.JX`,4JIS\_Y?@`0"TH@)`^>-7@+E`#`.+`01`
M^4%X(_CA_O^T`0A`^0$$`/D`(`"1H?__M>$#$ZK@`Q2JCOS_EV#^_[75``"P
MM?9'^30``)"4(A21X0,3JN`#%*J&_/^7X`$`M*("0/GC5X"Y0`P#BP$$0/E!
M>"/XX?[_M`$(0/D!!`#Y`"``D:'__[7A`Q.JX`,4JGC\_Y=@_O^UU0``L+7V
M1_DT``"0E&(4D>$#$ZK@`Q2J</S_E^`!`+2B`D#YXU>`N4`,`XL!!$#Y07@C
M^.'^_[0!"$#Y`00`^0`@`)&A__^UX0,3JN`#%*IB_/^78/[_M=8``+#6]D?Y
M-```D)6"%)'A`Q.JX`,5JEK\_Y?@`0"TP@)`^>-7@+E`#`.+`01`^4%X(_CA
M_O^T`0A`^0$$`/D`(`"1H?__M>$#$ZK@`Q6J3/S_EV#^_[76``"PUO9'^34`
M`)"UPA21X0,3JN`#%:I$_/^7X`$`M,("0/GC5X"Y0`P#BP$$0/E!>"/XX?[_
MM`$(0/D!!`#Y`"``D:'__[7A`Q.JX`,5JC;\_Y=@_O^UUP``L/?V1_DU``"0
MMN(2D>$#$ZK@`Q:J+OS_E^`!`+3B`D#YXU>`N4`,`XL!!$#Y07@C^.'^_[0!
M"$#Y`00`^0`@`)&A__^UX0,3JN`#%JH@_/^78/[_M=@``+`8]T?Y-@``D-<"
M%9'A`Q.JX`,7JAC\_Y?@`0"T`@-`^>-7@+E`#`.+`01`^4%X(_CA_O^T`0A`
M^0$$`/D`(`"1H?__M>$#$ZK@`Q>J"OS_EV#^_[79``"P.?='^3<``)#X0A61
MX0,3JN`#&*H"_/^7X`$`M"(#0/GC5X"Y0`P#BP$$0/E!>"/XX?[_M`$(0/D!
M!`#Y`"``D:'__[7A`Q.JX`,8JO3[_Y=@_O^UV0``D#GW1_D8``#P&(,5D>$#
M$ZK@`QBJ[/O_E^`!`+0B`T#YXU>`N4`,`XL!!$#Y07@C^.'^_[0!"$#Y`00`
M^0`@`)&A__^UX0,3JN`#&*K>^_^78/[_M=D``)`Y]T?Y&```\!@C$Y'A`Q.J
MX`,8JM;[_Y?@`0"T(@-`^>-7@+E`#`.+`01`^4%X(_CA_O^T`0A`^0$$`/D`
M(`"1H?__M>$#$ZK@`QBJR/O_EV#^_[7A`Q.J````\`#`%9'#^_^7X0,`JF``
M`+3@0A61[_S_E^$#$ZH```#P`"`6D;O[_Y?A`P"J@```M````/``@!:1YOS_
ME^$#$ZH```#P`,`6D;+[_Y?A`P"J8`(`M*#B$I'>_/^7P```D``01_GB+T#Y
M`0!`^4(``>L!`(#280(`5/5;0JF`@A21\U-!J0$``/#W8T.I(6`7D?DC0/G]
M>\:HSOS_%^$#$ZH```#P```7D9K[_Y?A`P"J8/W_M,`"%9'&_/^7Z/__%\``
M`)``D$?Y```_UA\@`]7]>[VI_0,`D<$``)`A$$?Y\PL`^=,``+`B`$#YXA<`
M^0(`@-)@$D"Y'P0`,6`!`%3!``"0(1!'^>,70/DB`$#Y8P`"ZP(`@-(A`@!4
M\PM`^?U[PZC``U_6X9,`D0```/```!61=_O_E^$#`*H``(!2H0``M"``0#D?
MP`!Q!!A`>N`'GQI@$@"YZO__%\```)``D$?Y```_UA\@`]4?(`/5P```D`!X
M1_G]>[^I_0,`D0``/];!``"0(0A'^2``/]9```"T``!`^?U[P:C``U_6_7N[
MJ<$``)`A$$?Y_0,`D?-3`:GT`P"J]5L"J14``/"V@A>1(@!`^>(G`/D"`(#2
MX`,6JN$3`9%.^_^7@```M`$`0#GS`P"JP0$`-<```)``$$?YXB=`^0$`0/E"
M``'K`0"`TF$'`%3A`Q2JH((7D?-30:GU6T*I_7O%J&W\_Q?"``"00CA'^>$#
M%*I``#_6@`$`M,```)``$$?YXB=`^0$`0/E"``'K`0"`TB$%`%3S4T&I]5M"
MJ?U[Q:C``U_6]QL`^>`#%*K7``"0]^Y'^>`"/];U`P"JX`,3JN`"/];!``"0
M(9A'^:`"`(L`"`"1(``_UL<``)#G\$?YY@,3J@(`@)(A`(!2Y`,4JO,#`*H%
M``#P`P``\*5@$Y%CH!.1X``_UL```)``$$?YXB=`^0$`0/E"``'K`0"`TL$`
M`%3W&T#YX0,3JN`#%JK)__\7]QL`^<```)``D$?Y```_U@PXD-+_8RS+P0``
MD"$01_G]>P&I_4,`D?MS!JG;``"P8D,`D4.``)'S4P*I1IQ`J?5;`ZGV`P.1
M1!1%J?=C!*GW,P*1X#L`^6`$0*UB",`]`P*0TN-C(XL@`$#Y8-0`^0``@-+A
M`Q>J0PQ`^0```/!",$#Y`.`2D<`&`*WF'PFIXU,`^>27"JGB7P#YXCN`/>#Z
M_Y>```"T`0!`.?,#`*I!)P`UP```D`!X1_GY:P6IX#\`^0``/];!``"0(0A'
M^2``/]:@$@"T&0!`^7D2`+3:``"06N]'^>`#&:I``S_6`/A_TP`$`)'5``"0
MM9I'^:`"/];\`P"J)`-`.?,#`*KD)@`TU```D)3R1_D8``#P&.,7D1\@`]7@
M`QRJXP,8J@(`@)(A`(!2@`(_UIP+`)$D'T`X)/__-0```/#6(@"1`(`6D0,`
M`!3`AD#XX`$`M.$#%ZJO^O^7^`,`JF#__[0!`$`Y(?__-.[Z_Y?@_O\TP0``
MD"&X1_G@`QBJ(``_UOD#`*K@`0"UU@``D-:Z1_GX0P*1&0-`^5D!`+0@`T`Y
M@`L`-.`#&:K>^O^78`H`-1D'0/D8(P"1.?__M1\@`]7@`QFJ0`,_UO8#`*K@
M`Q.J0`,_UM8"``O6$A`1V```D!C+1_G<?D"3X`,<JJ`"/]8!``#PYP,3JB6`
M$Y'D`QFJ!@``\`,``/#&`!B18R`8D0(`@)+V`P"J(0"`4H`"/];@`Q:J`3B`
M4@`#/]8?!``QX0``5,```)``?$?Y```_U@4`0+F_1`!Q81@`5,(``)!"I$?Y
MX<,#D>`#%JI``#_6\P,`*J```#7@`T&Y``P4$A\00'&`"0!4X3M`^0(``/#`
M``"0`%1'^>0#%JHC`$#Y0F`9D0``0/DA`(!2Q0``D*6(1_F@`#_6%0"`TL``
M`)``$$?Y`0*0TN)C(8M#U$#Y`0!`^6,``>L!`(#2`2P`5/U[0:G@`Q6J\U-"
MJ0PXD-+U6T.I]V-$J?EK1:G[<T:I_V,LB\`#7];@`QFJ&",`D<`"/];Y`P"J
M8/3_M*W__Q<9`(#2J___%^!70/GTHP*1TP``D'.Z1_E``0"T'R`#U>$#%ZHW
M^O^78```M`$`0#D!`@`U@`9`^90B`)$@__^UU0``D+6:1_F@`8#2V@``D%KO
M1_F@`C_6_`,`J@4``/#S`P"JN<`7D60*@%)B__\78`(_UOD#`*K@_?^TV@``
MD%KO1_E``S_6`/A_TP`$`)%2__\7X#]`^?D+0;D``#_6/P,`:R'V_U3@`T&Y
M`"``$A\`!W&A]?]4X`,<JJ`"/]8!``#P]0,`JB#`&I'A`Q>J#/K_E^$#`*H@
M'`"TX#M`^0``0/E$_/^7_`,`JH`;`+3@`QRJ0`,_UN(#`*H?#`!Q30(`5-D`
M`)`YKT?Y`'Q`DP`0`-$!``#P@`,`BR'@&I'B/P#Y(`,_U@`!`#7B/T#Y7Q``
M<:`9`%1"%`!1@,MB.!^\`'$@&0!48!-`N1\$`#%`#`!4H`T`--L``)![!T?Y
M`0``\````/`A8!>1```5D13[_Y<```#P%X`3D1H``/`9``#P8`,_UCFC&Y'F
M`P`JY0,7JD>#'I'D`Q:JX`,5J@,``/`"`("28V`;D2$`@%*``C_6W```D)Q_
M1_D2```4@`,_U@``0+D?1`!Q80(`5&`#/]9S!@`108,>D>$#`/GF`P`JYP,3
M*N4#%ZKD`Q:JXP,9JN`#%:H"`("2(0"`4H`"/];@`Q6J`3B`4@`#/]8?!``Q
M8/W_5,$``)`AU$?YX`,6JB``/];A`Q6J````\`#@$I'C^O^7X`,5JE7^_Y=7
M__\7X3M`^>0#%JK```"0`%1'^0(``/`C`$#Y0F`8D0``0/DA`(!2Q@``D,:(
M1_G``#_62?__%T;^_Y?```"0`!!'^0("D-+B8R*+0]1`^0$`0/EC``'K`0"`
MTF$5`%3]>T&IX`,3JO-30JD,.)#2]5M#J?=C1*G[<T:I_V,LB]```)`0ND?Y
M``(?UM0``)"4\D?YU?[_%P```/#A`Q>J```5D8?Y_Y=`#@"T``!`.1_``'$$
M&$!ZX`>?&F`3`+G`\O\U'R`#U<(``)!"Z$?YX`,<J@$`@%)``#_6]P,`*L#Q
M_S38``"0&/-&^4(`@%+9``"0.8='^2$"@)+SXP:1``,_UN$#$ZK@`Q<JP@"`
MTB`#/];@XT8Y@```->#G1CD?#`%QH`H`5`(`@%(!`(#2X`,7*@`#/];```"0
M`*A'^=H``)!:;T?Y```_UO@#`*H%```4X@,`*N$#$ZK@`QBJ0`,_UN$#$ZK@
M`Q<J`@"0TB`#/]8?``!Q[/[_5,$``)`AY$?YX`,7*O/#!9'Z(P:1]P,3JOL#
M&JH@`#_66:,`D<(``)!"'$?YX0,8JA@``/`8XQ>1X`,3JD``/]8?(`/5Y!9`
M..`#&ZKC`QBJ>PL`D0(`@)(A`(!2@`(_UG\#&>L!__]4_\,&.0```/#F`QJJ
M!8`3D>0#%JK@`Q6J!P``\`,``/#G@!Z18R`;D0(`@)(A`(!2@`(_UG#__Q?@
M.T#Y'`!`^2/__Q?@.T#Y`@1`^<+F_[3@`P*JXCL`^4`#/]8?#`#Q*>;_5.([
M0/D`$`#1`0``\"$`&Y%```"+(`,_UA\``''B.T#YG!."FB?__Q=_$P"YE?__
M%^#K1CD?!`%Q0?7_5.#O1CD?#`%QX?3_5.#S1CD?(`%Q@?3_5.#W1CD?%`%Q
M(?3_5.`#%RI"`(!2(0>`D@`#/];A`Q.JX`,7*B(%@-(@`S_6````\.8#$ZH%
M@!.1Y`,6JN`#%:H'``#P`P``\.>`'I%C(!N1`@"`DB$`@%*``C_6./__%\``
M`)``D$?Y```_UL```)``D$?Y^6L%J0``/]8?(`/5'R`#U?U[NJD```#P`(`4
MD?T#`)'S4P&I5-`[U=,``)!S7D?Y]5L"J?93`9'A`Q:J]V,#J?DC`/G"``"0
M0A!'^95J<_A#`$#YXR\`^0,`@-*BPD(YHL8".=/X_Y>```"T`0!`.3_$`''`
M#`!4"OS_EY5J<_C@`Q6JV```D!@71_D!``#P8P"`4B$`')$B`(!2``,_UO4#
M`*J@"0"TH0I`^0(``/"`:G/X0B`<D2$`0/G9``"0.2-'^2`#/]:@"D#Y`0!`
M^2`,0+D`"[`W@&IS^`$``/!C`(!2(4`<D2(`@%+7``"P``,_UO4#`*I@`0"T
MX0,6J@```/``(!.1JOC_E^(#`*I`"`"TH0I`^8!J<_@A`$#Y(`,_UL,``)!C
MZ$;Y(@"`4I5J<_@!``#P(6`<D>`#%:I@`#_6`@``\.$#`*I"P!R1X`,5JB`#
M/]:":G/X]$("D>$#%JH```#P`.`2D5_$`CF0^/^7@`H`^?,#`*I``P"TP@``
MD$+(1_D!.(!2X`,3JD``/]8!!``1/P0`<02(47JA!0!4P```D``01_GB+T#Y
M`0!`^4(``>L!`(#2800`5/-30:GU6T*I]V-#J?DC0/G]>\:HP`-?UMD``)`Y
M(T?YOO__%X`&0/EN_?^7@`H`^?,#`*I@_/^UP0``D"&\1_D```#P`.`<D2``
M/]8`!$`Y@//_-)G__Q?@0@*1H@I`^0$$0/E``$#Y(0!`^47Y_Y>]__\7@&IS
M^,(``)!"G$?Y0``_UJ7__Q?```"0`)!'^0``/];W0@*1XP,`*L0``)"$O$?Y
M````\.$&0/GB`Q.J`(`=D2$`0/F``#_6'R`#U?U[O*G!``"0(1!'^?T#`)'S
M4P&I]`,`JO5;`JDB`$#YXA\`^0(`@-(N^_^7`0!`.?,#`*K!```T``0`D3^\
M`'%S$H":`0!`.8'__S76``"PGP(`\?4'GQK`$D"Y'P0`,<`$`%0?``!QX`>?
M&K4"``J5`0`UP```D``01_GB'T#Y`0!`^4(``>L!`(#2804`5/-30:GU6T*I
M_7O$J,`#7]:``D`Y@/[_-,,``)!C2$?Y`0``\.`#$ZHA`!B1@@"`TF``/]:`
M_?\UP```D``01_GB'T#Y`0!`^4(``>L!`(#2P0(`5/5;0JG@`Q2J\U-!J?U[
MQ*A_^/\7````\.'3`)$``!61"_C_ET`!`+0!`$`Y/P``<>`'GQH_P`!QX0>?
M&@```0K`$@"Y%0`5"M'__Q??$@"YT/__%\```)``D$?Y```_UA\@`]4`!``4
M'R`#U1D`0/D!^/\7````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````'R`#
MU1\@`]7_@PK1`>``D:,``-!C$$?YXD,`D?U[`*D$X`&1_0,`D5'D`&_PY0!O
M90!`^>5/`?D%`(#2A^0`;^,#`JHC`$!M(@3!/&,X@`Y!!&!O0`18;V,(("Y"
M!$AO)E183P`<,4YDI"`O(51(3\,<,$X`'*).(1PG3@`<HTX%'*%.9(R?3)\`
M`>O!_?]4\!]!K4*@"9'F#T*MX>,!D>430ZWA(\`]\>4`;Q\@`]4B@-D\(4``
MD2``WSP`'")N`!PC;@`<,&[P'*=.QQRF3F8<HTX"5$%/``1A;Z,<I4Z%'*1.
M)!RA3D`<H$X!'#%.(8"`/#\``NO!_?]4!RA`J20SC](()$&I1%"K\N4_0:DC
M=)W2XOQ;T^S\0M-"%`>JC'D'JB$!",HCVZWR(0`*BB$`"<JE``2+(D`BBT']
M0M,+$$#Y(7@*JBU\0))"``6+I0$(RD(`"POF0T*II0`'BJ4`",KO`02+3OQ;
MTZ4`"8O.%0*J(0`,RJ4`#XLA``**I0`."T_\0M,A``W*XGD"JL8`!(NN_%O3
M(0`(B\X5!:HA``:+A@$"RB$`#@N,?4"2Q@`%BJ[\0M/&``S*$`($B\8`#8LO
M_%O3S7D%JN\5`:KN&T#Y10`-RL8`$(M"?$"2Q@`/"Z4``8JE``+*,/Q"T\X!
M!(NE``R+S_Q;TPQZ`:KO%0:JI0`.BZ4`#PNA`0S*S_Q"TR$`!HKF>0:JK7U`
MDNZ_0ZDA``W*(0`"B[#\0M/"`02+KOQ;TR$``HN"`0;*0@`%BHQ]0)+.%06J
M0@`,RBX`#@L%>@6JX0$$BT(`#8M"``&+P0`%RM!\0)(A``Z*(0`0RLW]6],A
M``R+K14.JN^S1*E-``T+QOU"T\YX#JJF?$"2X@$$BZ_]6],A``*+H@`.RN\5
M#:I"``V*+P`/"Z7]0M-"``;*@0$$BZ5X#:I"`!"+0@`!B^S]6]/!`07*\/U"
MTXP5#ZHA``^*#WH/JDP`#`OMPT6ISGU`DB$`#LHA``:+AOU;T\84#*JB`02+
M$`($BR$``HNB``_*+0`&"Z5\0))"``R*AOU"TT(`!<K&>`RJH?U;TT(`#HM"
M`!"++!0-JK#]0M/A`0;*(0`-B@UZ#:KNPT:I3``,"^]]0)(A``_*(0`%BX7]
M6].E%`RJP@$$BQ`"!(LA``*+P@`-RBX`!0O&?$"20@`,BH7]0M-"``;*I7@,
MJL']6]-"``^+0@`0BRP4#JK0_4+3H0$%RB$`#HH.>@ZJ[\-'J4P`#`NM?4"2
M(0`-RB$`!HN&_5O3QA0,JN(!!(L0`@2+(0`"BZ(`#LHO``8+I7Q`DD(`#(J&
M_4+30@`%RL9X#*KA_5O30@`-BT(`$(LL%`^J\/U"T\$!!LHA``^*#WH/JNW#
M2*E,``P+SGU`DB$`#LHA``6+A?U;TZ44#*JB`02+$`($BR$``HO"``_*+0`%
M"X7]0M.E>`RJ0@`,BJ']6]/&?$"2+!0-JN$!!<HA``V*[WU`DD(`!LHA``_*
M0@`.BR$`!HONFTFI0@`0BTP`#`NP_4+3#7H-JI#]0M/"`02+COU;TR$``HO.
M%0RJH@`-RBX`#@NE?$"20@`,BD(`!<H,>@RJP0`$BT(`#XO&_5O30@`!B^]7
M0/FA`0S*QA0.JJU]0)+2``(+(0`.BB$`#<KD`02+(0`%B]#]0M-%_EO3#GH.
MJN];0/DA``2+I102JD9`)HNP``$+A`$.RH0`!DKB`0.+A``-BP;^6]--_D+3
MA``"B^)?0/FR>1*JQA00JB5`)8O/``0+P0$2RB4`!4I!``.+HD`LBPS^0M/E
M_5O3C7D0JD(``8NE%`^JX6-`^;$``@N&0":+1`(-RH0`!DKL_4+3)@`#BX%`
M+HLD_EO3CWD/JB$`!HN$%!&JYF=`^9```0M%0"6+H@$/RD4`!4HN_D+3T7D1
MJJQ`,HO"``.+!?Y;T^9K0/F"`0*+I100JB1`)(OA`1'*K@`""R0`!$H,_D+3
MP0`#BX9`+8N0>1"JP0`!B^9O0/G$_5O314`EBX04#JHB`A#*C0`!"T(`!4K&
M``.+0D`OB\S]0M..>0ZJ0@`&B^9S0/FE_5O3I10-JB1`)(L!`@[*KP`""R$`
M!$JL_4+3C7D-JL8``XLA0#&+Y/U;TR$`!HM%0"6+YG=`^<(!#<J$%`^J10`%
M2I$``0NL0#"+\/U"T\(``XOF>T#Y#WH/JB7^6],D0"2+H0$/RH(!`HNE%!&J
M)``$2K```@O!``.++/Y"TX9`+HN,>1&JP0`!B^9_0/E%0"6+!/Y;T^(!#,J$
M%!"J0@`%2HX``0O&``.+0D`MBPW^0M/Q@T#Y0@`&BZ9Y$*HD0"2+Q?U;TX$!
M!LJE%`ZJ(0`$2JT``@LP`@.+(4`OB\_]0M,A`!"+[GD.JO"'0/FD_5O3A!0-
MJD5`)8NQ_4+3P@`.RH\``0LM>@VJ10`%2@("`XOPBT#Y)$`DB\$!#<JL0"R+
M)``$2N7]6].E%`^JAD`FB_']0M,!`@.+P0`!BX(!`HLF>@^JK``""T5`)8NB
M`0;*0@`%2H3]6]-"0"Z+A!0,JO"[4:F1_4+3CP`!"RQZ#*HD0"2+P0`,RN7]
M6],A``1*\?U"TZ44#ZH0`@.++WH/JD(`$(LA0"V+[<=2J;```@O.`0.+14`E
MBP3^6],A``Z+A!00JH(!#\J.``$+0@`%2D)`)HNE`0.+#?Y"TT4`!8O&_5O3
MHGD0JL84#JHD0"2+X0$"RLT`!0LD``1*(0(#BX1`+(O1_4+3\)]`^80``8NL
M_5O3(7H.JHP5#:JF0":+10`!RHX!!`NE``9*$`(#BZ5`+XNO_4+3[WD-JJ4`
M$(OPHT#YQOU;T\84#JJ,0"R+)``/RLT`!0N,``Q*T?U"TP0"`XN"02*+L/U;
MTT(`!(LD>@ZJID`FB^RG0/D0%@VJY0$$RD(`$`NE``9*KOU"TZ%`(8O%>0VJ
MA@$#BT[\0M.,``6J0_Q;TV,4`JJ,`0**PGD"JB$`!HOM.U6I(0`#"Z,`!`J&
MFY?2C`$#JF;CL?*#02^++_Q"TZP!!HLM_%O3K14!JF,`#(NL``*J8P`-"XP!
M`8I-``4*C`$-JN%Y`:K.`0:+A$$DBVW\6]-,``&JK14#JH0`#HN$``T+C`$#
MBBT``@IN_$+3C`$-JHW\6].%026+S'D#JN\[5JFC%02J+0`,JN\!!HO.`0:+
MI0`/BX_\0M.C``,+I0$$BHT!`0JE``VJ;?Q;TZ5`(HOB>02JK14#JH0!`JJE
M``Z+A``#BJ4`#0M-``P*A``-JFW\0M.$0"&+H7D#JN\W5ZFN_%O3SA4%JD,`
M`:IC``6*[P$&BZT!!HN$``^+K_Q"TX0`#@LN``(*8P`.JN5Y!:IC0"R+COQ;
MT\X5!*HL``6J8P`-BXP!!(KMPT#Y8P`."ZX``0J/_$+3C`$.JN1Y!*J"02*+
M;OQ;TZP!!HO.%0.J[<=`^4(`#(NL``2J0@`."XP!`XJ.``4*C`$.JF_\0M/C
M>0.JK0$&BX%!(8M._%O3SA4"JHP``ZHA``V+C`$"BB$`#@MN``0*C`$.JD[\
M0M/">0*JA4$EB^T[6:DO_$+3K`$&BRW\6].M%0&JI0`,BVP``JJE``T+C`$!
MBDT``PJ,`0VJX7D!JLX!!HN$022+K?Q;TTP``:JM%06JA``.BX0`#0N,`06*
M+0`""J[\0M.,`0VJC?Q;TX-!(XO,>06J[SM:J:T5!*HE``RJI0`$BN\!!HO.
M`0:+8P`/BX_\0M-C``T+C0$!"J4`#:KD>02JHD`BBVW\6].M%0.JA0$$JD(`
M#HNE``.*0@`-"XT`#`JE``VJ;?Q"TZ%`(8NE>0.J[S=;J4[\6]/.%0*J@P`%
MJF,``HKO`0:+K0$&BR$`#XNO``0*(0`."V,`#ZI._$+3;$`LB\)Y`JHC_%O3
MC`$-BV,4`:JM``*J+_Q"TX,!`PNL`0&*X7D!JDT`!0KN/URIC`$-JH1!)(MM
M_%O3K14#JLP!!HLN``(*A``,BTP``:J$``T+C`$#BHP!#JIM_$+3K7D#JN\!
M!HN%026+COQ;T^SK0/FC``^+SA4$JB4`#:IC``X+I0`$BJX!`0J/_$+3I0`.
MJF[\6].B0"*+A0$&B^1Y!*K.%0.J0@`%BZ4!!*I"``X+;OQ"TZ4``XK#>0.J
M[+M=J8\`#0JE``^J3_Q"TZ5`(8M!_%O3(10"JHP!!HNE``R+C``#JJ4``0N,
M`0**80`$"N)Y`JJ,`0&J4``#"H%!+8O-`0:+K/Q;TR$`#8OO]T#Y;0`"JHP5
M!:JM`06*C@$!"ZT!$*JD022+L/Q"T^T!!HO&_5O3[_M`^80`#8O&%`ZJ#7H%
MJM$`!`LL0"R+00`-RL4ZF-(A``Q*14RY\B%`(XOL`06+T/U"TR/^6]/O_T#Y
M#GH.JF,4$:HA``R+;``!"X9`)HND`0[*(T`CBX0`!DKF`06+@D`BBR_^0M.$
M_5O3\7D1JD(`!HN$%`RJY@-!^9(``@O!`1'*1$`DBR,``TK!``6+9D`MBXW]
M0M-#_EO3L'D,JL8``8MC%!*JX0=!^6\`!@LB`A#*3/Y"TT(`!$HD``6+0D`N
MB^']6].->1*J1``$B^(+0?DA%`^J+@`$"\-`(XL&`@W*[/U"T\,``TI&``6+
M8T`QB\+]6]./>0^J8P`&B^8/0?E"%`ZJ40`#"X%`(8ND`0_*8D`BBX$``4K$
M``6++$`PB]#]0M,A_EO3#GH.JN830?F$`02+(101JN,!#LHP``0+8@`"2BS^
M0M/#``6+1D`MBP+^6].,>1&JPP`#B^870?E"%!"J30`#"X%`(8O$`0S*$?Y"
MTX$``4K&``6+)$`OBZ']6]/O&T'YA``&BR$4#:HF>A"J8D`BBS``!`N#`0;*
M[P$%BV(``DJQ_4+30T`NBP+^6]/N'T'Y+7H-JF,`#XM"%!"J3P`#"X%`(8O$
M``W*8D`BBX$``4K$`06++$`LBP[^0M/A_5O3T'D0JH0!!(LA%`^J["-!^3$`
M!`NC`1#*@4`ABV(``DJ#`06+1D`FB^S]0M,B_EO3CWD/JL,``XM"%!&JYB=!
M^4X``PL$`@_*8D`BBX$``4K&``6+)$`MBRW^0M/!_5O3A``&B^PK0?FF>1&J
M(10.JN,!!LHM``0+8@`"2HP!!8M#0#"+T/U"TZ+]6],.>@ZJ8P`,B^PO0?E"
M%`VJ40`#"X%`(8O$``[*8D`BBX$``4J$`06++$`OBZ_]0M,A_EO3[7D-JH0!
M!(LA%!&J[#-!^3``!`O#`0W*+_Y"TV(``DJ,`06+0D`FBP/^6]-&``R+8Q00
MJN(W0?GL>1&J@4`ABV\`!@ND`0S*PT`CBX0``4I!``6+A$`NB^+]6],._D+3
MA``!B_$[0?G!>1"J0A0/JH8!`<I.``0+PP`#2C`"!8MF0"V+\?U"T\/]6]/M
M/T'Y8Q0.JB]Z#ZK&`!"+<``&"X)`(HLD``_*QD`CBX0``DJB`06+C$`LB\W]
M0M,$_EO3C`$"B^)#0?F$%!"JKGD.JH,`#`OM`0[*YT`OBZT!!DI&``6+H4$A
MBV+\6]-"%`.J(0`&B^U'0?DA``(+`OY"TX9!)(M">!"J)/Q;T\P!`LJ$%`&J
MC`$&2J8!!8ME_$+3Q@`'BZ-X`ZJ$``R+IP``L.<01_DA``H+0@`)"\4!"PN$
M``8+8P`("P0$`*D#"`&I!1``^>)/0?GA`$#Y0@`!ZP$`@-*!``!4_7M`J?^#
M"I'``U_6H```L`"01_D``#_6H0``L"&81_G]>[^I`!"`TOT#`)$@`#_6'Q@`
M^2(``)`A``"0'W@`N4$TP3TB``"0(C#!/4`XP3T"!`"M``B`/?U[P:C``U_6
M'R`#U1\@`]7]>[NI0GQ`D_T#`)'S4P&I%.``D?,#`:KU6P*I]0,`JO8#`JKW
M8P.I^2,`^06,0JD`>$"YI`P""[\`!.MCE(.:8G1"BZ2*`JG@`P`UN```L!@;
M1_G?_@!Q[04`5-8"`5'9?@93-P<`$?=J>M-W`A>+8`9`K>`#%:J`!@"M8`9!
MK7,"`9&`!@&MFOO_EW\"%^L!__]4UAH92^$#%ZK@`Q2JPGY`DP`#/]:V>@"Y
M\U-!J?5;0JGW8T.I^2-`^?U[Q:C``U_6%PB`4O<"`$O_`A9K@,(@B[@``+`8
M&T?Y]]*6&OE^0)/B`QFJ``,_UJ!Z0+G@`@`+H'H`N1\``7&!_?]4U@(72W,"
M&8O@`Q6J>?O_E]'__Q?W`Q.JX/__%Q\@`]7]>[RIX@^`$OT#`)'S4P&I]`,`
MJO,#`:KU6P*I-N``D>@;`/TU%$#Y*!A`_:`B`U,!!``1PLH@.,#"(8L_X`!Q
MS0L`5*,``+!C_$?Y`@B`4D(``4L!`(!20GQ`DV``/];@`Q.J6_O_EW\V`/D`
M!`!/P`(`K<`*@#T%!7!_``5H?P0%>'^A`F>>X`,3JB,$:'\B!'!_H`0#;B$$
M>'^`!`5N``4';F`$"6Y`!`MN(`0-;J`>#TY@.@#]1OO_EV$"0/G@`Q.JL```
ML!#61_DA_%C3@0(`.6$"0/DA_%#3@08`.6$"0/DA_$C3@0H`.6$"0/F!#@`Y
M809`^2'\6-.!$@`Y809`^2'\4-.!%@`Y809`^2'\2-.!&@`Y809`^8$>`#EA
M"D#Y(?Q8TX$B`#EA"D#Y(?Q0TX$F`#EA"D#Y(?Q(TX$J`#EA"D#Y@2X`.6$.
M0/DA_%C3@3(`.6$.0/DA_%#3@38`.6$.0/DA_$C3@3H`.6$.0/F!/@`Y81)`
M^2'\6-.!0@`Y81)`^2'\4-.!1@`Y81)`^2'\2-.!2@`Y81)`^8%.`#GS4T&I
M]5M"J>@;0/W]>\2H``(?UJ,``+!C_$?Y`@>`4D(``4L!`(!20GQ`DV``/]:I
M__\7'R`#U1\@`]6C``#0L```L!""1_EC!$#Y``(?UA\@`]7]>[^I_0,`D?U[
MP:C``U_6```````````!``(``````"5S+R5S````+W!R;V,O<V5L9B]E>&4`
M`$1Y;F%,;V%D97(Z.F)O;W1?1'EN84QO861E<@``````26YT97)N86QS.CI0
M05(Z.D)/3U0`````97AE```````O<')O8R\E:2\E<P``````+@````````!0
M05)?5$5-4```````````4$%27U!23T=.04U%`````#H`````````+P``````
M```E<R5S)7,``%!%4DPU3$E"``````````!015),3$E"`%!%4DPU3U!4````
M``````!015),24\``%!!4E])3DE424%,25I%1`!005)?4U!!5TY%1```````
M4$%27T-,14%.`````````%!!4E]$14)51P````````!005)?0T%#2$4`````
M````4$%27T=,3T)!3%]$14)51P``````````4$%27T=,3T)!3%]435!$25(`
M````````4$%27U1-4$1)4@```````%!!4E]'3$]"04Q?5$5-4`!005)?1TQ/
M0D%,7T-,14%.```````````Q`````````$Q$7TQ)0E)!4EE?4$%42`!365-4
M14T``"4P,G@`````<&%R+0`````E<R5S)7,E<P``````````)7,Z(&-R96%T
M:6]N(&]F('!R:79A=&4@<W5B9&ER96-T;W)Y("5S(&9A:6QE9"`H97)R;F\]
M)6DI"@```````"5S.B!P<FEV871E('-U8F1I<F5C=&]R>2`E<R!I<R!U;G-A
M9F4@*'!L96%S92!R96UO=F4@:70@86YD(')E=')Y('EO=7(@;W!E<F%T:6]N
M*0H```````!0051(`````'!A<FP`````+G!A<@`````E<R5S8V%C:&4M)7,E
M<P``)7,E<W1E;7`M)74E<P```"5S)7-T96UP+25U+25U)7,``````````!@`
M````````<&5R;``````P`````````%!!4CHZ4&%C:V5R.CI615)324].````
M`#$N,#4U````56YA8FQE('1O(&-R96%T92!C86-H92!D:7)E8W1O<GD`````
M`````"5S.B!C<F5A=&EO;B!O9B!P<FEV871E(&-A8VAE('-U8F1I<F5C=&]R
M>2`E<R!F86EL960@*&5R<FYO/25I*0H``````````"UE````````+2T`````
M``!54T52`````%5315).04U%``````````!435!$25(``%1%35!$25(`5$5-
M4`````!435```````"]T;7``````979A;"`G97AE8R`O=7-R+V)I;B]P97)L
M("`M4R`D,"`D>S$K(B1`(GTG"FEF(#`[(",@;F]T(')U;FYI;F<@=6YD97(@
M<V]M92!S:&5L;`IP86-K86=E(%]?<&%R7W!L.PIM>2`H)%!!4E]-04=)0RP@
M)'!A<E]T96UP+"`D<')O9VYA;64L($!T;7!F:6QE+"`E36]D=6QE0V%C:&4I
M.PI%3D0@>R!I9B`H)$5.5GM005)?0TQ%04Y]*2!["G)E<75I<F4@1FEL93HZ
M5&5M<#L*<F5Q=6ER92!&:6QE.CI"87-E;F%M93L*<F5Q=6ER92!&:6QE.CI3
M<&5C.PIM>2`D=&]P9&ER(#T@1FEL93HZ0F%S96YA;64Z.F1I<FYA;64H)'!A
M<E]T96UP*3L*;W5T<RAQ<5M296UO=FEN9R!F:6QE<R!I;B`B)'!A<E]T96UP
M(ETI.PI&:6QE.CI&:6YD.CIF:6YD9&5P=&@H<W5B('L@*"`M9"`I(#\@<FUD
M:7(@.B!U;FQI;FL@?2P@)'!A<E]T96UP*3L*<FUD:7(@)'!A<E]T96UP.PII
M9B`H+60@)'!A<E]T96UP("8F("1>3R!N92`G35-7:6XS,B<I('L*;7D@)'1M
M<"`](&YE=R!&:6QE.CI496UP*`I414U03$%412`]/B`G=&UP6%A86%@G+`I$
M25(@/3X@1FEL93HZ0F%S96YA;64Z.F1I<FYA;64H)'1O<&1I<BDL"E-51D9)
M6"`]/B`G+F-M9"<L"E5.3$E.2R`]/B`P+`HI.PIM>2`D9FEL96YA;64@/2`D
M=&UP+3YF:6QE;F%M93L*<')I;G0@)'1M<"`\/"(N+BXB.PIX/3$[('=H:6QE
M(%L@7"1X("UL="`Q,"!=.R!D;PIR;2`M<F8@)R1P87)?=&5M<"<*:68@6R!<
M(2`M9"`G)'!A<E]T96UP)R!=.R!T:&5N"F)R96%K"F9I"G-L965P(#$*>#U@
M97AP<B!<)'@@*R`Q8`ID;VYE"G)M("<D9FEL96YA;64G"BXN+@IC;&]S92`D
M=&UP.PIC:&UO9"`P-S`P+"`D9FEL96YA;64["FUY("1C;60@/2`B)&9I;&5N
M86UE(#XO9&5V+VYU;&P@,CXF,2`F(CL*<WES=&5M*"1C;60I.PIO=71S*'%Q
M6U-P87=N960@8F%C:V=R;W5N9"!P<F]C97-S('1O('!E<F9O<FT@8VQE86YU
M<#H@)&9I;&5N86UE72D["GT*?2!]"D)%1TE.('L*)%!!4E]-04=)0R`](")<
M;E!!4BYP;5QN(CL*26YT97)N86QS.CI005(Z.D)/3U0H*2!I9B!D969I;F5D
M("9);G1E<FYA;',Z.E!!4CHZ0D]/5#L*979A;"!["E]P87)?:6YI=%]E;G8H
M*3L*;7D@)'%U:65T(#T@(21%3E9[4$%27T1%0E5'?3L*;7D@)7-Y<R`]("@*
M<&%T:%]S97`@("`@/3X@*"1>3R`]?B`O7DU35VEN+R`_("<[)R`Z("<Z)RDL
M"E]E>&4@("`@("`@(#T^("@D7D\@/7X@+UXH/SI-4U=I;GQ/4S)\8WEG=VEN
M*2\@/R`G+F5X92<@.B`G)RDL"E]D96QI;2`@("`@(#T^("@D7D\@/7X@+UY-
M4U=I;GQ/4S(O(#\@)UQ<)R`Z("<O)RDL"BD["E]S971?<')O9VYA;64H*3L*
M;W5T<RAQ<5M<)'!R;V=N86UE(#T@(B1P<F]G;F%M92)=*3L*7W-E=%]P87)?
M=&5M<"@I.PIO=71S*'%Q6UPD14Y6>U!!4E]414U0?2`]("(D14Y6>U!!4E]4
M14U0?2)=*3L*;7D@*"1S=&%R=%]P;W,L("1D871A7W!O<RD["DU!1TE#.B![
M"FQO8V%L("1324=[7U]705).7U]](#T@<W5B('M].PIU;FQE<W,@*&]P96X@
M7T9(+"`G/#IR87<G+"`D<')O9VYA;64I('L*;W5T<RAQ<5M#86XG="!R96%D
M(&9R;VT@9FEL92`B)'!R;V=N86UE(ETI.R`@(R!D;VXG="!U<V4@)"$@:&5R
M92!A<R!I="!R97%U:7)E<R!%<G)N;RYP;0IL87-T($U!1TE#.PI]"FUY("1C
M:'5N:U]S:7IE(#T@-C0@*B`Q,#(T.PIM>2`D8G5F.PIM>2`D<VEZ92`]("US
M(%]&2#L*;7D@)&UA9VEC7W!O<R`]("1S:7IE("T@)'-I>F4@)2`D8VAU;FM?
M<VEZ93L@(R!.3U1%.B`D;6%G:6-?<&]S(&ES(&$@;75L=&EP;&4@;V8@)&-H
M=6YK7W-I>F4*=VAI;&4@*"1M86=I8U]P;W,@/CT@,"D@>PIS965K(%]&2"P@
M)&UA9VEC7W!O<RP@,#L*<F5A9"!?1D@L("1B=68L("1C:'5N:U]S:7IE("L@
M;&5N9W1H*"1005)?34%'24,I.PII9B`H*&UY("1I(#T@<FEN9&5X*"1B=68L
M("1005)?34%'24,I*2`^/2`P*2!["B1M86=I8U]P;W,@*ST@)&D["FQA<W0[
M"GT*)&UA9VEC7W!O<R`M/2`D8VAU;FM?<VEZ93L*?0II9B`H)&UA9VEC7W!O
M<R`\(#`I('L*;W5T<RAQ<5M#86XG="!F:6YD(&UA9VEC('-T<FEN9R`B)%!!
M4E]-04=)0R(@:6X@9FEL92`B)'!R;V=N86UE(ETI.PIL87-T($U!1TE#.PI]
M"F]U=',H(D9O=6YD(%!!4B!M86=I8R!A="!P;W-I=&EO;B`D;6%G:6-?<&]S
M(BD["G-E96L@7T9(+"`D;6%G:6-?<&]S("T@-"P@,#L*<F5A9"!?1D@L("1B
M=68L(#0["FUY("1O9F9S970@/2!U;G!A8VLH(DXB+"`D8G5F*3L*;W5T<R@B
M3V9F<V5T(&9R;VT@<W1A<G0@;V8@1DE,17,@:7,@)&]F9G-E="(I.PIS965K
M(%]&2"P@)&UA9VEC7W!O<R`M(#0@+2`D;V9F<V5T+"`P.PHD9&%T85]P;W,@
M/2!T96QL(%]&2#L*;7D@)7)E<75I<F5?;&ES=#L*<F5A9"!?1D@L("1B=68L
M(#0[("`@("`@("`@("`@("`@("`@("`@("`@("`@(R!R96%D('1H92!F:7)S
M="`B1DE,12(*=VAI;&4@*"1B=68@97$@(D9)3$4B*2!["G)E860@7T9(+"`D
M8G5F+"`T.PIR96%D(%]&2"P@)&)U9BP@=6YP86-K*").(BP@)&)U9BD["FUY
M("1F=6QL;F%M92`]("1B=68["F]U=',H<7%;56YP86-K:6YG($9)3$4@(B1F
M=6QL;F%M92(N+BY=*3L*;7D@)&-R8R`]("@@)&9U;&QN86UE(#U^('-\7BA;
M82UF7&1=>SA]*2]\?"`I(#\@)#$@.B!U;F1E9CL*;7D@*"1B87-E;F%M92P@
M)&5X="D@/2`H)&)U9B`]?B!M?"@_.BXJ+RD_*"XJ*2A<+BXJ*7PI.PIR96%D
M(%]&2"P@)&)U9BP@-#L*<F5A9"!?1D@L("1B=68L('5N<&%C:R@B3B(L("1B
M=68I.PII9B`H9&5F:6YE9"@D97AT*2!A;F0@)&5X="`A?B`O7"XH/SIP;7QP
M;'QI>'QA;"DD+VDI('L*;7D@)&9I;&5N86UE(#T@7W-A=F5?87,H(B1C<F,D
M97AT(BP@)&)U9BP@,#<U-2D["B1005(Z.DAE879Y.CI&=6QL0V%C:&5[)&9U
M;&QN86UE?2`]("1F:6QE;F%M93L*)%!!4CHZ2&5A=GDZ.D9U;&Q#86-H97LD
M9FEL96YA;65](#T@)&9U;&QN86UE.PI]"F5L<VEF("@@)&9U;&QN86UE(#U^
M(&U\7B\_<VAL:6(O?"!A;F0@9&5F:6YE9"`D14Y6>U!!4E]414U0?2`I('L*
M;7D@)&9I;&5N86UE(#T@7W-A=F5?87,H(B1B87-E;F%M921E>'0B+"`D8G5F
M+"`P-S4U*3L*;W5T<R@B4TA,24(Z("1F:6QE;F%M95QN(BD["GT*96QS92![
M"B1-;V1U;&5#86-H97LD9G5L;&YA;65](#T@>PIB=68@/3X@)&)U9BP*8W)C
M(#T^("1C<F,L"FYA;64@/3X@)&9U;&QN86UE+`I].PHD<F5Q=6ER95]L:7-T
M>R1F=6QL;F%M97TK*SL*?0IR96%D(%]&2"P@)&)U9BP@-#L*?0IL;V-A;"!`
M24Y#(#T@*'-U8B!["FUY("@D<V5L9BP@)&UO9'5L92D@/2!`7SL*<F5T=7)N
M(&EF(')E9B`D;6]D=6QE(&]R("$D;6]D=6QE.PIM>2`D:6YF;R`]("1-;V1U
M;&5#86-H97LD;6]D=6QE?3L*<F5T=7)N('5N;&5S<R`D:6YF;SL*9&5L971E
M("1R97%U:7)E7VQI<W1[)&UO9'5L97T["B1)3D-[)&UO9'5L97T@/2`B+VQO
M861E<B\D:6YF;R\D;6]D=6QE(CL*:68@*"1%3E9[4$%27T-,14%.?2!A;F0@
M9&5F:6YE9"@F24\Z.D9I;&4Z.FYE=RDI('L*;7D@)&9H(#T@24\Z.D9I;&4M
M/FYE=U]T;7!F:6QE(&]R(&1I92`B0V%N)W0@8W)E871E('1E;7`@9FEL93H@
M)"$B.PHD9F@M/F)I;FUO9&4H*3L*)&9H+3YP<FEN="@D:6YF;RT^>V)U9GTI
M.PHD9F@M/G-E96LH,"P@,"D["G)E='5R;B`D9F@["GT*96QS92!["FUY("1F
M:6QE;F%M92`](%]S879E7V%S*"(D:6YF;RT^>V-R8WTN<&TB+"`D:6YF;RT^
M>V)U9GTI.PHD:6YF;RT^>V9I;&5](#T@)&9I;&5N86UE.PIO<&5N(&UY("1F
M:"P@)SPZ<F%W)RP@)&9I;&5N86UE(&]R(&1I92!Q<5M#86XG="!R96%D("(D
M9FEL96YA;64B.B`D(5T["G)E='5R;B`D9F@["GT*9&EE(")";V]T<W1R87!P
M:6YG(&9A:6QE9#H@8V%N)W0@9FEN9"!M;V1U;&4@)&UO9'5L92$B.PI]+"!`
M24Y#*3L*<F5Q=6ER92!84TQO861E<CL*<F5Q=6ER92!005(Z.DAE879Y.PIR
M97%U:7)E($-A<G`Z.DAE879Y.PIR97%U:7)E($5X<&]R=&5R.CI(96%V>3L*
M4$%2.CI(96%V>3HZ7VEN:71?9'EN86QO861E<B@I.PIR97%U:7)E($E/.CI&
M:6QE.PIW:&EL92`H;7D@)&9I;&5N86UE(#T@*'-O<G0@:V5Y<R`E<F5Q=6ER
M95]L:7-T*5LP72D@>PIU;FQE<W,@*"1)3D-[)&9I;&5N86UE?2!O<B`D9FEL
M96YA;64@/7X@+T)31%!!3B\I('L*:68@*"1F:6QE;F%M92`]?B`O7"YP;6,_
M)"]I*2!["G)E<75I<F4@)&9I;&5N86UE.PI]"F5L<V4@>PID;R`D9FEL96YA
M;64@=6YL97-S("1F:6QE;F%M92`]?B`O<VET96-U<W1O;6EZ95PN<&PD+SL*
M?0I]"F1E;&5T92`D<F5Q=6ER95]L:7-T>R1F:6QE;F%M97T["GT*=6YL97-S
M("@D8G5F(&5Q(")02UPP,#-<,#`T(BD@>PIO=71S*'%Q6TYO('II<"!F;W5N
M9"!A9G1E<B!&24Q%('-E8W1I;VX@:6X@9FEL92`B)'!R;V=N86UE(ETI.PIL
M87-T($U!1TE#(#L*?0HD<W1A<G1?<&]S(#T@*'1E;&P@7T9(*2`M(#0[("`@
M("`@("`@("`@("`@(",@<W1A<G0@;V8@>FEP"GT*;7D@0'!A<E]A<F=S.PIM
M>2`H)&]U="P@)&)U;F1L92P@)&QO9V9H+"`D8V%C:&5?;F%M92D["F1E;&5T
M92`D14Y6>U!!4E]!4%!?4D554T5].R`C('-A;FET:7IE("A2155312!M87D@
M8F4@82!S96-U<FET>2!P<F]B;&5M*0HD<75I970@/2`P('5N;&5S<R`D14Y6
M>U!!4E]$14)51WT["FEF("@A)'-T87)T7W!O<R!O<B`H)$%21U9;,%T@97$@
M)RTM<&%R+6]P=&EO;G,G("8F('-H:69T*2D@>PIM>2`E9&ES=%]C;60@/2!Q
M=R@*<"`@(&)L:6)?=&]?<&%R"FD@("!I;G-T86QL7W!A<@IU("`@=6YI;G-T
M86QL7W!A<@IS("`@<VEG;E]P87(*=B`@('9E<FEF>5]P87(**3L*:68@*$!!
M4D=6(&%N9"`D05)'5ELP72!E<2`G+2UR975S92<I('L*<VAI9G0@0$%21U8[
M"B1%3E9[4$%27T%04%]2155317T@/2!S:&EF="!`05)'5CL*?0IE;'-E('L@
M(R!N;W)M86P@<&%R;"!B96AA=FEO=7(*;7D@0&%D9%]T;U]I;F,["G=H:6QE
M("A`05)'5BD@>PHD05)'5ELP72`]?B`O7BTH6T%)34]"3&)Q<&EU<U1V72DH
M+BHI+R!O<B!L87-T.PII9B`H)#$@97$@)TDG*2!["G!U<V@@0&%D9%]T;U]I
M;F,L("0R.PI]"F5L<VEF("@D,2!E<2`G32<I('L*979A;"`B=7-E("0R(CL*
M?0IE;'-I9B`H)#$@97$@)T$G*2!["G5N<VAI9G0@0'!A<E]A<F=S+"`D,CL*
M?0IE;'-I9B`H)#$@97$@)T\G*2!["B1O=70@/2`D,CL*?0IE;'-I9B`H)#$@
M97$@)V(G*2!["B1B=6YD;&4@/2`G<VET92<["GT*96QS:68@*"0Q(&5Q("="
M)RD@>PHD8G5N9&QE(#T@)V%L;"<["GT*96QS:68@*"0Q(&5Q("=Q)RD@>PHD
M<75I970@/2`Q.PI]"F5L<VEF("@D,2!E<2`G3"<I('L*;W!E;B`D;&]G9F@L
M("(^/B(L("0R(&]R(&1I92!Q<5M#86XG="!O<&5N(&QO9R!F:6QE("(D,B(Z
M("0A73L*?0IE;'-I9B`H)#$@97$@)U0G*2!["B1C86-H95]N86UE(#T@)#([
M"GT*<VAI9G0H0$%21U8I.PII9B`H;7D@)&-M9"`]("1D:7-T7V-M9'LD,7TI
M('L*9&5L971E("1%3E9[)U!!4E]414U0)WT["FEN:71?:6YC*"D["G)E<75I
M<F4@4$%2.CI$:7-T.PHF>R)005(Z.D1I<W0Z.B1C;60B?2@I('5N;&5S<R!`
M05)'5CL*)GLB4$%2.CI$:7-T.CHD8VUD(GTH)%\I(&9O<B!`05)'5CL*97AI
M=#L*?0I]"G5N<VAI9G0@0$E.0RP@0&%D9%]T;U]I;F,["GT*?0II9B`H)&]U
M="D@>PI["G)E<75I<F4@24\Z.D9I;&4["G)E<75I<F4@07)C:&EV93HZ6FEP
M.PIR97%U:7)E($1I9V5S=#HZ4TA!.PI]"FUY("1P87(@/2!S:&EF="A`05)'
M5BD["FUY("1Z:7`["FEF("AD969I;F5D("1P87(I('L*;W!E;B!M>2`D9F@L
M("<\.G)A=R<L("1P87(@;W(@9&EE('%Q6T-A;B=T(&9I;F0@<&%R(&9I;&4@
M(B1P87(B.B`D(5T["F)L97-S*"1F:"P@)TE/.CI&:6QE)RD["B1Z:7`@/2!!
M<F-H:79E.CI::7`M/FYE=SL**"`D>FEP+3YR96%D1G)O;49I;&5(86YD;&4H
M)&9H+"`D<&%R*2`]/2!!<F-H:79E.CI::7`Z.D%:7T]+*"D@*0IO<B!D:64@
M<7%;17)R;W(@<F5A9&EN9R!Z:7`@87)C:&EV92`B)'!A<B)=.PI]"FEF("AD
M969I;F5D("1P87(I('L*;W!E;B!M>2`D<&@L("<\.G)A=R<L("1P87(@;W(@
M9&EE('%Q6T-A;B=T(')E860@<&%R(&9I;&4@(B1P87(B.B`D(5T["FUY("1B
M=68["G)E860@)'!H+"`D8G5F+"`T.PID:64@<7%;(B1P87(B(&ES(&YO="!A
M('!A<B!F:6QE72!U;FQE<W,@)&)U9B!E<2`B4$M<,#`S7#`P-"(["F-L;W-E
M("1P:#L*?0I#<F5A=&50871H*"1O=70I(#L*;7D@)&9H(#T@24\Z.D9I;&4M
M/FYE=R@*)&]U="P*24\Z.D9I;&4Z.D]?0U)%050H*2!\($E/.CI&:6QE.CI/
M7U=23TY,62@I('P@24\Z.D9I;&4Z.D]?5%)53D,H*2P*,#<W-RP**2!O<B!D
M:64@<7%;0V%N)W0@8W)E871E(&9I;&4@(B1O=70B.B`D(5T["B1F:"T^8FEN
M;6]D92@I.PIS965K(%]&2"P@,"P@,#L*;7D@)&QO861E<CL*:68@*&1E9FEN
M960@)&1A=&%?<&]S*2!["G)E860@7T9(+"`D;&]A9&5R+"`D9&%T85]P;W,[
M"GT@96QS92!["FQO8V%L("0O(#T@=6YD968["B1L;V%D97(@/2`\7T9(/CL*
M?0II9B`H(21%3E9[4$%27U9%4D)!5$E-?2!A;F0@)&QO861E<B`]?B`O7B@_
M.B,A?%Q`<F5M*2\I('L*<F5Q=6ER92!005(Z.D9I;'1E<CHZ4&]D4W1R:7`[
M"E!!4CHZ1FEL=&5R.CI0;V13=')I<"T^87!P;'DH7"1L;V%D97(L("0P*3L*
M?0HD9F@M/G!R:6YT*"1L;V%D97(I"F]R(&1I92!Q<5M%<G)O<B!W<FET:6YG
M(&QO861E<B!T;R`B)&]U="(Z("0A73L*:68@*"1B=6YD;&4I('L*<F5Q=6ER
M92!005(Z.DAE879Y.PI005(Z.DAE879Y.CI?:6YI=%]D>6YA;&]A9&5R*"D[
M"FEN:71?:6YC*"D["G)E<75I<F5?;6]D=6QE<R@I.PIM>2!`:6YC(#T@9W)E
M<"![("$O0E-$4$%.+R!]"F=R97`@>PHH)&)U;F1L92!N92`G<VET92<I(&]R
M"B@D7R!N92`D0V]N9FEG.CI#;VYF:6=[87)C:&QI8F5X<'T@86YD"B1?(&YE
M("1#;VYF:6<Z.D-O;F9I9WMP<FEV;&EB97AP?2D["GT@0$E.0SL*<WPO*WPO
M?&<L('-\+R1\?"!F;W)E86-H($!I;F,["FUY("5F:6QE<SL*)&9I;&5S>R1?
M?2LK(&9O<B!`1'EN84QO861E<CHZ9&Q?<VAA<F5D7V]B:F5C=',L('9A;'5E
M<R`E24Y#.PIM>2`D;&EB7V5X="`]("1#;VYF:6<Z.D-O;F9I9WML:6)?97AT
M?3L@("`@("`@("`C(%A86"!L:6)?97AT('9S(&1L97AT(#\*;7D@)7=R:71T
M96X["F9O<F5A8V@@;7D@)&ME>2`H<V]R="!K97ES("5F:6QE<RD@>PIM>2`H
M)&9I;&4L("1N86UE*3L*:68@*&1E9FEN960H;7D@)&9C(#T@)%!!4CHZ2&5A
M=GDZ.D9U;&Q#86-H97LD:V5Y?2DI('L**"1F:6QE+"`D;F%M92D@/2`H)&ME
M>2P@)&9C*3L*?0IE;'-E('L*9F]R96%C:"!M>2`D9&ER("A`:6YC*2!["FEF
M("@D:V5Y(#U^(&U\7EQ1)&1I<EQ%+R@N*BDD?&DI('L**"1F:6QE+"`D;F%M
M92D@/2`H)&ME>2P@)#$I.PIL87-T.PI]"FEF("@D:V5Y(#U^(&U\7B]L;V%D
M97(O6UXO72LO*"XJ*21\*2!["FEF("AM>2`D<F5F(#T@)$UO9'5L94-A8VAE
M>R0Q?2D@>PHH)&9I;&4L("1N86UE*2`]("@D<F5F+"`D,2D["FQA<W0["GT*
M:68@*"UF("(D9&ER+R0Q(BD@>PHH)&9I;&4L("1N86UE*2`]("@B)&1I<B\D
M,2(L("0Q*3L*;&%S=#L*?0I]"GT*?0IN97AT('5N;&5S<R!D969I;F5D("1N
M86UE.PIN97AT(&EF("1W<FET=&5N>R1N86UE?2LK.PIN97AT(&EF("%R968H
M)&9I;&4I(&%N9"`D9FEL92`]?B`O7"Y<421L:6)?97AT7$4D+VD["F]U=',H
M<W!R:6YT9BAQ<5M086-K:6YG($9)3$4@(B5S(BXN+ETL(')E9B`D9FEL92`_
M("1F:6QE+3Y[;F%M97T@.B`D9FEL92DI.PIM>2`D8V]N=&5N=#L*:68@*')E
M9B@D9FEL92DI('L*)&-O;G1E;G0@/2`D9FEL92T^>V)U9GT["GT*96QS92![
M"FQO8V%L("0O(#T@=6YD968["F]P96X@;7D@)'1H+"`G/#IR87<G+"`D9FEL
M92!O<B!D:64@<7%;0V%N)W0@<F5A9"`B)&9I;&4B.B`D(5T["B1C;VYT96YT
M(#T@/"1T:#X["F-L;W-E("1T:#L*4$%2.CI&:6QT97(Z.E!O9%-T<FEP+3YA
M<'!L>2A<)&-O;G1E;G0L("(\96UB961D960^+R1N86UE(BD*:68@(21%3E9[
M4$%27U9%4D)!5$E-?2!A;F0@)&YA;64@/7X@+UPN*#\Z<&U\:7A\86PI)"]I
M.PI005(Z.D9I;'1E<CHZ4&%T8VA#;VYT96YT+3YN97<M/F%P<&QY*%PD8V]N
M=&5N="P@)&9I;&4L("1N86UE*3L*?0HD9F@M/G!R:6YT*")&24Q%(BP*<&%C
M:R@G3B<L(&QE;F=T:"@D;F%M92D@*R`Y*2P*<W!R:6YT9B@B)3`X>"\E<R(L
M($%R8VAI=F4Z.EII<#HZ8V]M<'5T94-20S,R*"1C;VYT96YT*2P@)&YA;64I
M+`IP86-K*"=.)RP@;&5N9W1H*"1C;VYT96YT*2DL"B1C;VYT96YT*0IO<B!D
M:64@<7%;17)R;W(@=W)I=&EN9R!E;6)E9&1E9"!&24Q%('1O("(D;W5T(CH@
M)"%=.PIO=71S*'%Q6U=R:71T96X@87,@(B1N86UE(ETI.PI]"GT*:68@*"1Z
M:7`I('L*)'II<"T^=W)I=&54;T9I;&5(86YD;&4H)&9H*2`]/2!!<F-H:79E
M.CI::7`Z.D%:7T]+"F]R(&1I92!Q<5M%<G)O<B!W<FET:6YG('II<"!P87)T
M(&]F("(D;W5T(ET["GT*)&-A8VAE7VYA;64@/2!S=6)S='(@)&-A8VAE7VYA
M;64L(#`L(#0P.PII9B`H(21C86-H95]N86UE(&%N9"!M>2`D;71I;64@/2`H
M<W1A="@D;W5T*2E;.5TI('L*;7D@)&-T>"`]($1I9V5S=#HZ4TA!+3YN97<H
M,2D["F]P96X@;7D@)'1H+"`B/#IR87<B+"`D;W5T.PHD8W1X+3YA9&1F:6QE
M*"1T:"D["F-L;W-E("1T:#L*)&-A8VAE7VYA;64@/2`D8W1X+3YH97AD:6=E
M<W0["GT*)&-A8VAE7VYA;64@+CT@(EPP(B!X("@T,"`M(&QE;F=T:"`D8V%C
M:&5?;F%M92D["B1C86-H95]N86UE("X](")<,$-!0TA%(CL*;7D@)&]F9G-E
M="`]("1F:"T^=&5L;"`K(&QE;F=T:"@D8V%C:&5?;F%M92D@+2!L96YG=&@H
M)&QO861E<BD["B1F:"T^<')I;G0H)&-A8VAE7VYA;64L"G!A8VLH)TXG+"`D
M;V9F<V5T*2P*)%!!4E]-04=)0RD*)B8@)&9H+3YC;&]S90IO<B!D:64@<7%;
M17)R;W(@=W)I=&EN9R!T<F%I;&5R(&]F("(D;W5T(CH@)"%=.PIC:&UO9"`P
M-S4U+"`D;W5T.PIE>&ET.PI]"GL*;&%S="!U;FQE<W,@9&5F:6YE9"`D<W1A
M<G1?<&]S.PI?9FEX7W!R;V=N86UE*"D["G)E<75I<F4@4$%2.PI005(Z.DAE
M879Y.CI?:6YI=%]D>6YA;&]A9&5R*"D["GL*<F5Q=6ER92!&:6QE.CI&:6YD
M.PIR97%U:7)E($%R8VAI=F4Z.EII<#L*?0IM>2`D9F@@/2!)3SHZ1FEL92T^
M;F5W.R`@("`@("`@("`@("`@("`@("`@("`@("`@("`@(R!!<F-H:79E.CI:
M:7`@;W!E<F%T97,@;VX@86X@24\Z.DAA;F1L90HD9F@M/F9D;W!E;BAF:6QE
M;F\H7T9(*2P@)W(G*2!O<B!D:64@<7%;9F1O<&5N*"D@9F%I;&5D.B`D(5T[
M"D%R8VAI=F4Z.EII<#HZ<V5T0VAU;FM3:7IE*"US(%]&2"D["FUY("1Z:7`@
M/2!!<F-H:79E.CI::7`M/FYE=SL**"1Z:7`M/G)E861&<F]M1FEL94AA;F1L
M92@D9F@L("1P<F]G;F%M92D@/3T@07)C:&EV93HZ6FEP.CI!6E]/2R@I*0IO
M<B!D:64@<7%;17)R;W(@<F5A9&EN9R!Z:7`@87)C:&EV92`B)'!R;V=N86UE
M(ET["D%R8VAI=F4Z.EII<#HZ<V5T0VAU;FM3:7IE*#8T("H@,3`R-"D["B1Q
M=6EE="`]("$D14Y6>U!!4E]$14)51WT["F]U=',H(E)E861I;F<@345402YY
M;6PN+BXB*3L*:68@*&UY("1M971A(#T@)'II<"T^8V]N=&5N=',H)TU%5$$N
M>6UL)RDI('L*)&UE=&$@/7X@<R\N*EYP87(Z7',J)"\O;7,["B1M971A(#U^
M(',O7EQ3+BHO+VUS.PII9B`H;7D@*"1C;&5A;BD@/2`D;65T82`]?B`O7EQS
M*V-L96%N.B`H+BHI)"]M*2!["B1C;&5A;B`]?B`O7EQS*GQ<<RHD+V<["B1%
M3E9[4$%27T-,14%.?2`](#$@:68@)&-L96%N.PI]"GT*<'5S:"!`4$%2.CI,
M:6)#86-H92P@)'II<#L*)%!!4CHZ3&EB0V%C:&5[)'!R;V=N86UE?2`]("1Z
M:7`["F]U=',H(D5X=')A8W1I;F<@>FEP+BXN(BD["FEF("AD969I;F5D("1%
M3E9[4$%27U1%35!]*2![(",@<VAO=6QD(&)E('-E="!A="!T:&ES('!O:6YT
M(0IF;W)E86-H(&UY("1M96UB97(@*"`D>FEP+3YM96UB97)S("D@>PIN97AT
M(&EF("1M96UB97(M/FES1&ER96-T;W)Y.PIM>2`D;65M8F5R7VYA;64@/2`D
M;65M8F5R+3YF:6QE3F%M93L*;F5X="!U;FQE<W,@)&UE;6)E<E]N86UE(#U^
M(&U["EX*+S]S:&QI8B\**#\Z)$-O;F9I9SHZ0V]N9FEG>W9E<G-I;VY]+RD_
M"B@_.B1#;VYF:6<Z.D-O;F9I9WMA<F-H;F%M97TO*3\**%M>+UTK*0HD"GUX
M.PIM>2`D97AT<F%C=%]N86UE(#T@)#$["FUY("1D97-T7VYA;64@/2!&:6QE
M.CI3<&5C+3YC871F:6QE*"1%3E9[4$%27U1%35!]+"`D97AT<F%C=%]N86UE
M*3L*:68@*"UF("1D97-T7VYA;64@)B8@+7,@7R`]/2`D;65M8F5R+3YU;F-O
M;7!R97-S9613:7IE*"DI('L*;W5T<RAQ<5M3:VEP<&EN9R`B)&UE;6)E<E]N
M86UE(B!S:6YC92!I="!A;')E861Y(&5X:7-T<R!A="`B)&1E<W1?;F%M92)=
M*3L*?2!E;'-E('L*;W5T<RAQ<5M%>'1R86-T:6YG("(D;65M8F5R7VYA;64B
M('1O("(D9&5S=%]N86UE(BXN+ETI.PHH)&UE;6)E<BT^97AT<F%C=%1O1FEL
M94YA;65D*"1D97-T7VYA;64I(#T]($%R8VAI=F4Z.EII<#HZ05I?3TLH*2D*
M;W(@9&EE('%Q6T5R<F]R(&5X=')A8W1I;F<@>FEP(&UE;6)E<B!T;R`B)&1E
M<W1?;F%M92)=.PIC:&UO9"@P-34U+"`D9&5S=%]N86UE*2!I9B`D7D\@97$@
M(FAP=7@B.PI]"GT*?0I]"G5N;&5S<R`H)%!!4CHZ3&EB0V%C:&5[)'!R;V=N
M86UE?2D@>PID:64@/#P@(BXB('5N;&5S<R!`05)'5CL*57-A9V4Z("0P(%L@
M+4%L:6(N<&%R(%T@6R`M261I<B!=(%L@+4UM;V1U;&4@72!;('-R8RYP87(@
M72!;('!R;V=R86TN<&P@70HD,"!;("U"?"UB(%T@6RU/;W5T9FEL95T@<W)C
M+G!A<@HN"B1%3E9[4$%27U!23T=.04U%?2`]("1P<F]G;F%M92`]("0P(#T@
M<VAI9G0H0$%21U8I.PI]"G-U8B!#<F5A=&50871H('L*;7D@*"1N86UE*2`]
M($!?.PIR97%U:7)E($9I;&4Z.D)A<V5N86UE.PIM>2`H)&)A<V5N86UE+"`D
M<&%T:"P@)&5X="D@/2!&:6QE.CI"87-E;F%M93HZ9FEL97!A<G-E*"1N86UE
M+"`H)UPN+BHG*2D["G)E<75I<F4@1FEL93HZ4&%T:#L*1FEL93HZ4&%T:#HZ
M;6MP871H*"1P871H*2!U;FQE<W,H+64@)'!A=&@I.R`C(&UK<&%T:"!D:65S
M('=I=&@@97)R;W(*?0IS=6(@<F5Q=6ER95]M;V1U;&5S('L*<F5Q=6ER92!L
M:6(["G)E<75I<F4@1'EN84QO861E<CL*<F5Q=6ER92!I;G1E9V5R.PIR97%U
M:7)E('-T<FEC=#L*<F5Q=6ER92!W87)N:6YG<SL*<F5Q=6ER92!V87)S.PIR
M97%U:7)E($-A<G`["G)E<75I<F4@0V%R<#HZ2&5A=GD["G)E<75I<F4@17)R
M;F\["G)E<75I<F4@17AP;W)T97(Z.DAE879Y.PIR97%U:7)E($5X<&]R=&5R
M.PIR97%U:7)E($9C;G1L.PIR97%U:7)E($9I;&4Z.E1E;7`["G)E<75I<F4@
M1FEL93HZ4W!E8SL*<F5Q=6ER92!84TQO861E<CL*<F5Q=6ER92!#;VYF:6<[
M"G)E<75I<F4@24\Z.DAA;F1L93L*<F5Q=6ER92!)3SHZ1FEL93L*<F5Q=6ER
M92!#;VUP<F5S<SHZ6FQI8CL*<F5Q=6ER92!!<F-H:79E.CI::7`["G)E<75I
M<F4@1&EG97-T.CI32$$["G)E<75I<F4@4$%2.PIR97%U:7)E(%!!4CHZ2&5A
M=GD["G)E<75I<F4@4$%2.CI$:7-T.PIR97%U:7)E(%!!4CHZ1FEL=&5R.CI0
M;V13=')I<#L*<F5Q=6ER92!005(Z.D9I;'1E<CHZ4&%T8VA#;VYT96YT.PIR
M97%U:7)E(&%T=')I8G5T97,["F5V86P@>R!R97%U:7)E($-W9"!].PIE=F%L
M('L@<F5Q=6ER92!7:6XS,B!].PIE=F%L('L@<F5Q=6ER92!38V%L87(Z.E5T
M:6P@?3L*979A;"![(')E<75I<F4@07)C:&EV93HZ56YZ:7`Z.D)U<G-T('T[
M"F5V86P@>R!R97%U:7)E(%1I93HZ2&%S:#HZ3F%M961#87!T=7)E('T["F5V
M86P@>R!R97%U:7)E(%!E<FQ)3SL@<F5Q=6ER92!097)L24\Z.G-C86QA<B!]
M.PIE=F%L('L@<F5Q=6ER92!U=&8X('T["GT*<W5B(%]S971?<&%R7W1E;7`@
M>PII9B`H9&5F:6YE9"`D14Y6>U!!4E]414U0?2!A;F0@)$5.5GM005)?5$5-
M4'T@/7X@+R@N*RDO*2!["B1P87)?=&5M<"`]("0Q.PIR971U<FX["GT*9F]R
M96%C:"!M>2`D<&%T:"`H"BAM87`@)$5.5GLD7WTL('%W*"!005)?5$U01$E2
M(%1-4$1)4B!414U01$E2(%1%35`@5$U0("DI+`IQ=R@@0SI<7%1%35`@+W1M
M<"`N("D**2!["FYE>'0@=6YL97-S(&1E9FEN960@)'!A=&@@86YD("UD("1P
M871H(&%N9"`M=R`D<&%T:#L*;7D@)'5S97)N86UE.PIM>2`D<'=U:60["F5V
M86P@>R@D<'=U:60I(#T@9V5T<'=U:60H)#XI(&EF(&1E9FEN960@)#X[?3L*
M:68@*"!D969I;F5D*"97:6XS,CHZ3&]G:6Y.86UE*2`I('L*)'5S97)N86UE
M(#T@)E=I;C,R.CI,;V=I;DYA;64["GT*96QS:68@*&1E9FEN960@)'!W=6ED
M*2!["B1U<V5R;F%M92`]("1P=W5I9#L*?0IE;'-E('L*)'5S97)N86UE(#T@
M)$5.5GM54T523D%-17T@?'P@)$5.5GM54T52?2!\?"`G4UE35$5-)SL*?0HD
M=7-E<FYA;64@/7X@<R]<5R]?+V<["FUY("1S=&UP9&ER(#T@(B1P871H)'-Y
M<WM?9&5L:6U]<&%R+2(N=6YP86-K*")(*B(L("1U<V5R;F%M92D["FUK9&ER
M("1S=&UP9&ER+"`P-S4U.PII9B`H(21%3E9[4$%27T-,14%.?2!A;F0@;7D@
M)&UT:6UE(#T@*'-T870H)'!R;V=N86UE*2E;.5TI('L*;W!E;B!M>2`D9F@L
M("(\.G)A=R(L("1P<F]G;F%M92!O<B!D:64@<7%;0V%N)W0@<F5A9"`B)'!R
M;V=N86UE(CH@)"%=.PIS965K("1F:"P@+3$X+"`R.PIM>2`D8G5F.PIR96%D
M("1F:"P@)&)U9BP@-CL*:68@*"1B=68@97$@(EPP0T%#2$4B*2!["G-E96L@
M)&9H+"`M-3@L(#(["G)E860@)&9H+"`D8G5F+"`T,3L*)&)U9B`]?B!S+UPP
M+R]G.PHD<W1M<&1I<B`N/2`B)'-Y<WM?9&5L:6U]8V%C:&4M)&)U9B(["GT*
M96QS92!["FUY("1D:6=E<W0@/2!E=F%L"GL*<F5Q=6ER92!$:6=E<W0Z.E-(
M03L*;7D@)&-T>"`]($1I9V5S=#HZ4TA!+3YN97<H,2D["F]P96X@;7D@)&9H
M+"`B/#IR87<B+"`D<')O9VYA;64@;W(@9&EE('%Q6T-A;B=T(')E860@(B1P
M<F]G;F%M92(Z("0A73L*)&-T>"T^861D9FEL92@D9F@I.PIC;&]S92@D9F@I
M.PHD8W1X+3YH97AD:6=E<W0["GT@?'P@)&UT:6UE.PHD<W1M<&1I<B`N/2`B
M)'-Y<WM?9&5L:6U]8V%C:&4M)&1I9V5S="(["GT*8VQO<V4H)&9H*3L*?0IE
M;'-E('L*)$5.5GM005)?0TQ%04Y](#T@,3L*)'-T;7!D:7(@+CT@(B1S>7-[
M7V1E;&EM?71E;7`M)"0B.PI]"B1%3E9[4$%27U1%35!](#T@)'-T;7!D:7([
M"FUK9&ER("1S=&UP9&ER+"`P-S4U.PIL87-T.PI]"B1P87)?=&5M<"`]("0Q
M(&EF("1%3E9[4$%27U1%35!](&%N9"`D14Y6>U!!4E]414U0?2`]?B`O*"XK
M*2\["GT*<W5B(%]S879E7V%S('L*;7D@*"1N86UE+"`D8V]N=&5N=',L("1M
M;V1E*2`]($!?.PIM>2`D9G5L;&YA;64@/2`B)'!A<E]T96UP+R1N86UE(CL*
M=6YL97-S("@M92`D9G5L;&YA;64I('L*;7D@)'1E;7!N86UE(#T@(B1F=6QL
M;F%M92XD)"(["FUY("1F:#L**&]P96X@)&9H+"`G/CIR87<G+"`D=&5M<&YA
M;64I"B8F("AP<FEN="`D9F@@)&-O;G1E;G1S*0HF)B`H8VQO<V4@)&9H*0IO
M<B!D:64@<7%;17)R;W(@=W)I=&EN9R`B)'1E;7!N86UE(CH@)"%=.PIC:&UO
M9"`D;6]D92P@)'1E;7!N86UE(&EF(&1E9FEN960@)&UO9&4["G)E;F%M92@D
M=&5M<&YA;64L("1F=6QL;F%M92D@;W(@=6YL:6YK*"1T96UP;F%M92D["GT*
M<F5T=7)N("1F=6QL;F%M93L*?0IS=6(@7W-E=%]P<F]G;F%M92!["FEF("AD
M969I;F5D("1%3E9[4$%27U!23T=.04U%?2!A;F0@)$5.5GM005)?4%)/1TY!
M345](#U^("\H+BLI+RD@>PHD<')O9VYA;64@/2`D,3L*?0HD<')O9VYA;64@
M?'P]("0P.PII9B`H)$5.5GM005)?5$5-4'T@86YD(&EN9&5X*"1P<F]G;F%M
M92P@)$5.5GM005)?5$5-4'TI(#X](#`I('L*)'!R;V=N86UE(#T@<W5B<W1R
M*"1P<F]G;F%M92P@<FEN9&5X*"1P<F]G;F%M92P@)'-Y<WM?9&5L:6U]*2`K
M(#$I.PI]"FEF("@A)$5.5GM005)?4%)/1TY!345](&]R(&EN9&5X*"1P<F]G
M;F%M92P@)'-Y<WM?9&5L:6U]*2`^/2`P*2!["FEF("AO<&5N(&UY("1F:"P@
M)SPG+"`D<')O9VYA;64I('L*<F5T=7)N(&EF("US("1F:#L*?0II9B`H+7,@
M(B1P<F]G;F%M921S>7-[7V5X97TB*2!["B1P<F]G;F%M92`N/2`D<WES>U]E
M>&5].PIR971U<FX["GT*?0IF;W)E86-H(&UY("1D:7(@*'-P;&ET("]<421S
M>7-[<&%T:%]S97!]7$4O+"`D14Y6>U!!5$A]*2!["FYE>'0@:68@97AI<W1S
M("1%3E9[4$%27U1%35!](&%N9"`D9&ER(&5Q("1%3E9[4$%27U1%35!].PHD
M9&ER(#U^(',O7%$D<WES>U]D96QI;7U<120O+SL**"@D<')O9VYA;64@/2`B
M)&1I<B1S>7-[7V1E;&EM?21P<F]G;F%M921S>7-[7V5X97TB*2P@;&%S="D*
M:68@+7,@(B1D:7(D<WES>U]D96QI;7TD<')O9VYA;64D<WES>U]E>&5](CL*
M*"@D<')O9VYA;64@/2`B)&1I<B1S>7-[7V1E;&EM?21P<F]G;F%M92(I+"!L
M87-T*0II9B`M<R`B)&1I<B1S>7-[7V1E;&EM?21P<F]G;F%M92(["GT*?0IS
M=6(@7V9I>%]P<F]G;F%M92!["B0P(#T@)'!R;V=N86UE('Q\/2`D14Y6>U!!
M4E]04D]'3D%-17T["FEF("AI;F1E>"@D<')O9VYA;64L("1S>7-[7V1E;&EM
M?2D@/"`P*2!["B1P<F]G;F%M92`]("(N)'-Y<WM?9&5L:6U])'!R;V=N86UE
M(CL*?0IM>2`D<'=D(#T@*&1E9FEN960@)D-W9#HZ9V5T8W=D*2`_($-W9#HZ
M9V5T8W=D*"D*.B`H*&1E9FEN960@)E=I;C,R.CI'971#=V0I(#\@5VEN,S(Z
M.D=E=$-W9"@I(#H@8'!W9&`I.PIC:&]M<"@D<'=D*3L*)'!R;V=N86UE(#U^
M(',O7B@_/5PN7"X_7%$D<WES>U]D96QI;7U<12DO)'!W9"1S>7-[7V1E;&EM
M?2\["B1%3E9[4$%27U!23T=.04U%?2`]("1P<F]G;F%M93L*?0IS=6(@7W!A
M<E]I;FET7V5N=B!["FEF("@@)$5.5GM005)?24Y)5$E!3$E:141]*RL@/3T@
M,2`I('L*<F5T=7)N.PI](&5L<V4@>PHD14Y6>U!!4E])3DE424%,25I%1'T@
M/2`R.PI]"F9O<B`H<7<H(%-005=.140@5$5-4"!#3$5!3B!$14)51R!#04-(
M12!04D]'3D%-12`I("D@>PID96QE=&4@)$5.5GLB4$%27R1?(GT["GT*9F]R
M("AQ=R\@5$U01$E2(%1%35`@0TQ%04X@1$5"54<@+RD@>PHD14Y6>R)005)?
M)%\B?2`]("1%3E9[(E!!4E]'3$]"04Q?)%\B?2!I9B!E>&ES=',@)$5.5GLB
M4$%27T=,3T)!3%\D7R)].PI]"FEF("@D14Y6>U!!4E]414U0?2D@>PID96QE
M=&4@)$5.5GM005)?0TQ%04Y].PI]"GT*<W5B(&]U=',@>PIR971U<FX@:68@
M)'%U:65T.PII9B`H)&QO9V9H*2!["G!R:6YT("1L;V=F:"`B0%]<;B(["GT*
M96QS92!["G!R:6YT(")`7UQN(CL*?0I]"G-U8B!I;FET7VEN8R!["G)E<75I
M<F4@0V]N9FEG.PIP=7-H($!)3D,L(&=R97`@9&5F:6YE9"P@;6%P("1#;VYF
M:6<Z.D-O;F9I9WLD7WTL('%W*`IA<F-H;&EB97AP('!R:79L:6)E>'`@<VET
M96%R8VAE>'`@<VET96QI8F5X<`IV96YD;W)A<F-H97AP('9E;F1O<FQI8F5X
M<`HI.PI]"G!A8VMA9V4@;6%I;CL*<F5Q=6ER92!005(["G5N<VAI9G0@0$E.
M0RP@7"9005(Z.F9I;F1?<&%R.PI005(M/FEM<&]R="A`<&%R7V%R9W,I.PID
M:64@<7%;<&%R+G!L.B!#86XG="!O<&5N('!E<FP@<V-R:7!T("(D<')O9VYA
M;64B.B!.;R!S=6-H(&9I;&4@;W(@9&ER96-T;W)Y7&Y="G5N;&5S<R`M92`D
M<')O9VYA;64["F1O("1P<F]G;F%M93L*0T]213HZ97AI="@D,2D@:68@*"1`
M(#U^+UY?5$M?15A)5%]<*"A<9"LI7"DO*3L*9&EE("1`(&EF("1`.PI].PHD
M.CI?7T524D]2(#T@)$`@:68@)$`["GT*0T]213HZ97AI="@D,2D@:68@*"0Z
M.E]?15)23U(@/7XO7E]42U]%6$E47UPH*%QD*RE<*2\I.PID:64@)#HZ7U]%
M4E)/4B!I9B`D.CI?7T524D]2.PHQ.PI?7T5.1%]?"@!P97)L>'-I+F,`````
M```````````````!(T5G`````(FKS>\`````_MRZF`````!V5#(0`````/#A
MTL,````````````````!&P,[_````!X```!0=___+`8``-!Y__\4`0``('K_
M_R@!``!0>O__/`$``)!Z__]0`0``X'K__W0!``#P>O__B`$```!\___\`0``
ML'S__R0"``"`?O__B`(``+!____``@``$(+__Q`#``"0@O__-`,``,2"__](
M`P``X(/__WP#``#D@___D`,``'2&__\0!```H(;__R0$``#@A___6`0``+",
M__^4!```4(W__[@$``"`C?__V`0``.".__\<!0``4)?__[0%``#`F?__\`4`
M`!"K__]@!@``T+O__X0&```@O/__I`8``$"]___<!@``(+___Q@'````````
M$``````````!>E(`!'@>`1L,'P`0````&````+1X__\T`````$$''A`````L
M````\'C__S``````````$````$`````,>?__/``````````@````5````#AY
M__](`````$$.()T$G@-"DP).WMW3#@`````0````>````&1Y__\$````````
M`'````",````8'G__P@!````00Y0G0J>"4*3")0'0I@#EP1$F0)#E@65!E?6
MU4'8UT'90][=T]0.`$$.4),(E`>5!I8%EP28`YD"G0J>"4_6U4'8UT'90=[=
MT]0.`$$.4),(E`>="IX)0I<$F`.9`D'8UT+9)``````!``#\>?__K`````!!
M#K`!G1:>%423%%@*WMW3#@!!"P```&`````H`0``A'K__]`!````0P[0`9T:
MGAE&E!>3&$34TTC>W0X`00[0`9,8E!>=&IX91I85E190F!.7%&O8UTP*U--!
MUM5!"T:7%)@30]/4U=;7V$.4%Y,80985E19!F!.7%``T````C`$``/![__\H
M`0```$(.P"!$G8@$GH<$0I.&!)2%!$.5A`26@P1F"MW>U=;3U`X`00L``$P`
M``#$`0``Z'S__UP"````00YPG0Z>#4*3#)0+294*E@F7")@'F0::!0)("M[=
MV=K7V-76T]0.`$$+29L$80K;00M,VT*;!$/;0YL$0ML`(````!0"``#X?O__
M>`````!!#B"=!)X#1),"E`%0WMW3U`X`$````#@"``!4?___-``````````P
M````3`(``'1___\4`0```$(.T$!$G8H(GHD(0I.(")2'"$*5A@AU"MW>U=/4
M#@!!"P``$````(`"``!<@/__!`````````!\````E`(``$R`__^0`@```$(.
MD")#G:($GJ$$0I.@!)2?!$*5G@26G01@"MW>U=;3U`X`00M!FID$F9H$19B;
M!)><!$.<EP2;F`0"20K8UT':V4'<VT$+6`K8UT':V4'<VT$+1-?8V=K;W$.8
MFP27G`1!FID$F9H$09R7!)N8!!`````4`P``7(+__R0`````````,````"@#
M``!T@O__.`$```!"#B"=!)X#0I,"<PK>W=,.`$$+0PK>W=,.`$0+1M[=TPX`
M`#@```!<`P``@(/__\P$````00Y@G0R>"T23"I0)1I4(E@>7!I@%F00#&P$*
MWMW9U]C5UM/4#@!!"P```"````"8`P``%(C__Y@`````00XPG0:>!423!%`*
MWMW3#@!!"QP```"\`P``D(C__S``````0PX0G0*>`4C>W0X`````0````-P#
M``"@B/__8`$```!!#E"="IX)1),(E`="E0:6!5@*WMW5UM/4#@!!"T\*WMW5
MUM/4#@!!"T&7!&#71)<$``"4````(`0``+R)__]H"````$(.P(,"0YVV()ZU
M($*;K""<JR!&D[0@E+,@E;(@EK$@0Y>P()BO(%R:K2"9KB`"CPK:V4+=WMO<
MU]C5UM/4#@!!"P*DV=I2W=[;W-?8U=;3U`X`0P[`@P*3M""4LR"5LB"6L2"7
ML""8KR"9KB":K2";K""<JR"=MB">M2`"G]G:0YJM()FN(#@```"X!```E)'_
M_VP"````00Y@G0R>"T23"I0)1)4(E@=$EP:8!9D$`F0*WMW9U]C5UM/4#@!!
M"P```#@```#T!```R)/__T0!````00Y`G0B>!T23!I0%0I4$E@-@"M[=U=;3
MU`X`00M5"M[=U=;3U`X`00L``#`````P!0``'''__VP"````00Y@G0R>"T63
M"I0)E0B6!P)I"M[=U=;3U`X`00L````````@````9`4``*BD___`$````$$.
MH`5&G52>4P,E!`K=W@X`00L<````B`4``$2U__](`````$,.$)T"G@%.WMT.
M`````#0```"H!0``=+7__QP!````00Y0G0J>"4.3")0'0Y4&E@5$EP28`YD"
M90K>W=G7V-76T]0.`$$+.````.`%``!<MO__V`$```!!#D"=")X'0Y,&E`5#
ME026`T(%2`("9`K>W=76T]0&2`X`00L`````````$````!P&````N/__%```
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``````````````````````````````````````````````#0'P```````(`?
M`````````0````````#X`@````````$``````````P,````````!````````
M``T#````````#P````````".!`````````P`````````P!L````````-````
M`````"1D````````&0````````!8NP$``````!L`````````"``````````:
M`````````&"[`0``````'``````````(`````````/7^_V\`````F`(`````
M```%`````````*`-````````!@````````#X`P````````H`````````L00`
M```````+`````````!@`````````%0````````````````````,`````````
MB+T!```````"`````````$@`````````%``````````'`````````!<`````
M````>!L````````'`````````(`3````````"`````````#X!P````````D`
M````````&``````````8````````````````````^___;P`````!```(````
M`/[__V\`````(!,```````#___]O``````(`````````\/__;P````!2$@``
M`````/G__V\`````$@``````````````````````````````````````````
M````````````````````````````````````````````````````````````
M`````````````````````````````````````````````.`;````````X!L`
M``````#@&P```````&B[`0``````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````,&(`````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M`````````````````````````````````````````!!A````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M$&0``````````````````,!@````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``````````````````````````````````````````````````````!`'```
M```````````````````````````(P`$``````/____\`````\&<```````"P
M9```````````````````H&4```````#09P```````-AG````````X&<`````
M``#H9P``````````````````N&<```````#`9P``````````````````^&<`
M``````!'0T,Z("A'3E4I(#$R+C$N,`!'0T,Z("AC<F]S<W1O;VPM3D<@,2XR
M-2XP+C(X7V(X.&0S,S@I(#$R+C$N,```+G-H<W1R=&%B`"YI;G1E<G``+FYO
M=&4N9VYU+F)U:6QD+6ED`"YN;W1E+D%"22UT86<`+F=N=2YH87-H`"YD>6YS
M>6T`+F1Y;G-T<@`N9VYU+G9E<G-I;VX`+F=N=2YV97)S:6]N7W(`+G)E;&$N
M9'EN`"YR96QA+G!L=``N:6YI=``N=&5X=``N9FEN:0`N<F]D871A`"YE:%]F
M<F%M95]H9'(`+F5H7V9R86UE`"YI;FET7V%R<F%Y`"YF:6YI7V%R<F%Y`"YD
M>6YA;6EC`"YG;W0`+F1A=&$`+F)S<P`N8V]M;65N=```````````````````
M````````````````````````````````````````````````````````````
M``````````````````L````!`````@`````````X`@```````#@"````````
M&P````````````````````$````````````````````3````!P````(`````
M````5`(```````!4`@```````"0````````````````````$````````````
M````````)@````<````"`````````'@"````````>`(````````@````````
M````````````!````````````````````#0```#V__]O`@````````"8`@``
M`````)@"````````7`$````````%``````````@````````````````````^
M````"P````(`````````^`,```````#X`P```````*@)````````!@````,`
M```(`````````!@`````````1@````,````"`````````*`-````````H`T`
M``````"Q!````````````````````0```````````````````$X```#___]O
M`@````````!2$@```````%(2````````S@`````````%``````````(`````
M`````@````````!;````_O__;P(`````````(!,````````@$P```````&``
M````````!@````(````(````````````````````:@````0````"````````
M`(`3````````@!,```````#X!P````````4`````````"``````````8````
M`````'0````$````0@````````!X&P```````'@;````````2``````````%
M````%0````@`````````&`````````!^`````0````8`````````P!L`````
M``#`&P```````!@````````````````````$````````````````````>0``
M``$````&`````````.`;````````X!L```````!0````````````````````
M$````````````````````(0````!````!@````````!`'````````$`<````
M````Y$<``````````````````$````````````````````"*`````0````8`
M````````)&0````````D9````````!0````````````````````$````````
M````````````D`````$````"`````````$!D````````0&0```````"P0```
M````````````````$````````````````````)@````!`````@````````#P
MI````````/"D````````_`````````````````````0`````````````````
M``"F`````0````(`````````\*4```````#PI0```````#`&````````````
M```````(````````````````````L`````X````#`````````%B[`0``````
M6+L````````(````````````````````"``````````(`````````+P````/
M`````P````````!@NP$``````&"[````````"`````````````````````@`
M````````"`````````#(````!@````,`````````:+L!``````!HNP``````
M`"`"````````!@`````````(`````````!``````````T0````$````#````
M`````(B]`0``````B+T```````!X`@``````````````````"``````````(
M`````````-8````!`````P``````````P`$```````#`````````@```````
M``````````````@```````````````````#<````"`````,`````````@,`!
M``````"`P````````#`0```````````````````0````````````````````
MX0````$````P````````````````````@,`````````_````````````````
M`````0`````````!``````````$````#`````````````````````````+_`
F````````Z@````````````````````$`````````````````````
