# -*- coding: utf-8 -*-

"""Data containing all current emoji
   Extracted from https://unicode.org/Public/emoji/latest/emoji-test.txt
   and https://www.unicode.org/Public/UCD/latest/ucd/emoji/emoji-variation-sequences.txt
   See utils/get_codes_from_unicode_emoji_data_files.py

   +----------------+-------------+------------------+-------------------+
   | Emoji Version  |    Date     | Unicode Version  | Data File Comment |
   +----------------+-------------+------------------+-------------------+
   | N/A            | 2010-10-11  | Unicode 6.0      | E0.6              |
   | N/A            | 2014-06-16  | Unicode 7.0      | E0.7              |
   | Emoji 1.0      | 2015-06-09  | Unicode 8.0      | E1.0              |
   | Emoji 2.0      | 2015-11-12  | Unicode 8.0      | E2.0              |
   | Emoji 3.0      | 2016-06-03  | Unicode 9.0      | E3.0              |
   | Emoji 4.0      | 2016-11-22  | Unicode 9.0      | E4.0              |
   | Emoji 5.0      | 2017-06-20  | Unicode 10.0     | E5.0              |
   | Emoji 11.0     | 2018-05-21  | Unicode 11.0     | E11.0             |
   | Emoji 12.0     | 2019-03-05  | Unicode 12.0     | E12.0             |
   | Emoji 12.1     | 2019-10-21  | Unicode 12.1     | E12.1             |
   | Emoji 13.0     | 2020-03-10  | Unicode 13.0     | E13.0             |
   | Emoji 13.1     | 2020-09-15  | Unicode 13.0     | E13.1             |
   | Emoji 14.0     | 2021-09-14  | Unicode 14.0     | E14.0             |

                          http://www.unicode.org/reports/tr51/#Versioning

"""

__all__ = [
    'EMOJI_DATA', 'STATUS'
]

component = 1
fully_qualified = 2
minimally_qualified = 3
unqualified = 4

STATUS = {
    "component" : component,
    "fully_qualified" : fully_qualified,
    "minimally_qualified" : minimally_qualified,
    "unqualified" : unqualified
}

EMOJI_DATA = {
    u'\U0001F947': {
        'en' : u':1st_place_medal:',
        'status' : fully_qualified,
        'E' : 3,
        'de': u':goldmedaille:',
        'es': u':medalla_de_oro:',
        'fr': u':médaille_d’or:',
        'pt': u':medalha_de_ouro:',
        'it': u':medaglia_d’oro:'
    },
    u'\U0001F948': {
        'en' : u':2nd_place_medal:',
        'status' : fully_qualified,
        'E' : 3,
        'de': u':silbermedaille:',
        'es': u':medalla_de_plata:',
        'fr': u':médaille_d’argent:',
        'pt': u':medalha_de_prata:',
        'it': u':medaglia_d’argento:'
    },
    u'\U0001F949': {
        'en' : u':3rd_place_medal:',
        'status' : fully_qualified,
        'E' : 3,
        'de': u':bronzemedaille:',
        'es': u':medalla_de_bronce:',
        'fr': u':médaille_de_bronze:',
        'pt': u':medalha_de_bronze:',
        'it': u':medaglia_di_bronzo:'
    },
    u'\U0001F18E': {
        'en' : u':AB_button_(blood_type):',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':ab:'],
        'de': u':großbuchstaben_ab_in_rotem_quadrat:',
        'es': u':grupo_sanguíneo_ab:',
        'fr': u':groupe_sanguin_ab:',
        'pt': u':botão_ab_(tipo_sanguíneo):',
        'it': u':gruppo_sanguigno_ab:'
    },
    u'\U0001F3E7': {
        'en' : u':ATM_sign:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':atm:'],
        'de': u':symbol_geldautomat:',
        'es': u':señal_de_cajero_automático:',
        'fr': u':distributeur_de_billets:',
        'pt': u':símbolo_de_caixa_automático:',
        'it': u':simbolo_dello_sportello_bancomat:'
    },
    u'\U0001F170\U0000FE0F': {
        'en' : u':A_button_(blood_type):',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':a:'],
        'variant': True,
        'de': u':großbuchstabe_a_in_rotem_quadrat:',
        'es': u':grupo_sanguíneo_a:',
        'fr': u':groupe_sanguin_a:',
        'pt': u':botão_a_(tipo_sanguíneo):',
        'it': u':gruppo_sanguigno_a:'
    },
    u'\U0001F170': {
        'en' : u':A_button_(blood_type):',
        'status' : unqualified,
        'E' : 0.6,
        'alias' : [u':a:'],
        'variant': True,
        'de': u':großbuchstabe_a_in_rotem_quadrat:',
        'es': u':grupo_sanguíneo_a:',
        'fr': u':groupe_sanguin_a:',
        'pt': u':botão_a_(tipo_sanguíneo):',
        'it': u':gruppo_sanguigno_a:'
    },
    u'\U0001F1E6\U0001F1EB': {
        'en' : u':Afghanistan:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Afghanistan:'],
        'de': u':flagge_afghanistan:',
        'es': u':bandera_afganistán:',
        'fr': u':drapeau_afghanistan:',
        'pt': u':bandeira_afeganistão:',
        'it': u':bandiera_afghanistan:'
    },
    u'\U0001F1E6\U0001F1F1': {
        'en' : u':Albania:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Albania:'],
        'de': u':flagge_albanien:',
        'es': u':bandera_albania:',
        'fr': u':drapeau_albanie:',
        'pt': u':bandeira_albânia:',
        'it': u':bandiera_albania:'
    },
    u'\U0001F1E9\U0001F1FF': {
        'en' : u':Algeria:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Algeria:'],
        'de': u':flagge_algerien:',
        'es': u':bandera_argelia:',
        'fr': u':drapeau_algérie:',
        'pt': u':bandeira_argélia:',
        'it': u':bandiera_algeria:'
    },
    u'\U0001F1E6\U0001F1F8': {
        'en' : u':American_Samoa:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_American_Samoa:'],
        'de': u':flagge_amerikanisch-samoa:',
        'es': u':bandera_samoa_americana:',
        'fr': u':drapeau_samoa_américaines:',
        'pt': u':bandeira_samoa_americana:',
        'it': u':bandiera_samoa_americane:'
    },
    u'\U0001F1E6\U0001F1E9': {
        'en' : u':Andorra:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Andorra:'],
        'de': u':flagge_andorra:',
        'es': u':bandera_andorra:',
        'fr': u':drapeau_andorre:',
        'pt': u':bandeira_andorra:',
        'it': u':bandiera_andorra:'
    },
    u'\U0001F1E6\U0001F1F4': {
        'en' : u':Angola:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Angola:'],
        'de': u':flagge_angola:',
        'es': u':bandera_angola:',
        'fr': u':drapeau_angola:',
        'pt': u':bandeira_angola:',
        'it': u':bandiera_angola:'
    },
    u'\U0001F1E6\U0001F1EE': {
        'en' : u':Anguilla:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Anguilla:'],
        'de': u':flagge_anguilla:',
        'es': u':bandera_anguila:',
        'fr': u':drapeau_anguilla:',
        'pt': u':bandeira_anguila:',
        'it': u':bandiera_anguilla:'
    },
    u'\U0001F1E6\U0001F1F6': {
        'en' : u':Antarctica:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Antarctica:'],
        'de': u':flagge_antarktis:',
        'es': u':bandera_antártida:',
        'fr': u':drapeau_antarctique:',
        'pt': u':bandeira_antártida:',
        'it': u':bandiera_antartide:'
    },
    u'\U0001F1E6\U0001F1EC': {
        'en' : u':Antigua_&_Barbuda:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Antigua_&_Barbuda:'],
        'de': u':flagge_antigua_und_barbuda:',
        'es': u':bandera_antigua_y_barbuda:',
        'fr': u':drapeau_antigua-et-barbuda:',
        'pt': u':bandeira_antígua_e_barbuda:',
        'it': u':bandiera_antigua_e_barbuda:'
    },
    u'\U00002652': {
        'en' : u':Aquarius:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':aquarius:'],
        'variant': True,
        'de': u':wassermann_(sternzeichen):',
        'es': u':acuario:',
        'fr': u':verseau:',
        'pt': u':signo_de_aquário:',
        'it': u':segno_zodiacale_dell’acquario:'
    },
    u'\U0001F1E6\U0001F1F7': {
        'en' : u':Argentina:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Argentina:'],
        'de': u':flagge_argentinien:',
        'es': u':bandera_argentina:',
        'fr': u':drapeau_argentine:',
        'pt': u':bandeira_argentina:',
        'it': u':bandiera_argentina:'
    },
    u'\U00002648': {
        'en' : u':Aries:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':aries:'],
        'variant': True,
        'de': u':widder_(sternzeichen):',
        'es': u':aries:',
        'fr': u':bélier_zodiaque:',
        'pt': u':signo_de_áries:',
        'it': u':segno_zodiacale_dell’ariete:'
    },
    u'\U0001F1E6\U0001F1F2': {
        'en' : u':Armenia:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Armenia:'],
        'de': u':flagge_armenien:',
        'es': u':bandera_armenia:',
        'fr': u':drapeau_arménie:',
        'pt': u':bandeira_armênia:',
        'it': u':bandiera_armenia:'
    },
    u'\U0001F1E6\U0001F1FC': {
        'en' : u':Aruba:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Aruba:'],
        'de': u':flagge_aruba:',
        'es': u':bandera_aruba:',
        'fr': u':drapeau_aruba:',
        'pt': u':bandeira_aruba:',
        'it': u':bandiera_aruba:'
    },
    u'\U0001F1E6\U0001F1E8': {
        'en' : u':Ascension_Island:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Ascension_Island:'],
        'de': u':flagge_ascension:',
        'es': u':bandera_isla_de_la_ascensión:',
        'fr': u':drapeau_île_de_l’ascension:',
        'pt': u':bandeira_ilha_de_ascensão:',
        'it': u':bandiera_isola_ascensione:'
    },
    u'\U0001F1E6\U0001F1FA': {
        'en' : u':Australia:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Australia:'],
        'de': u':flagge_australien:',
        'es': u':bandera_australia:',
        'fr': u':drapeau_australie:',
        'pt': u':bandeira_austrália:',
        'it': u':bandiera_australia:'
    },
    u'\U0001F1E6\U0001F1F9': {
        'en' : u':Austria:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Austria:'],
        'de': u':flagge_österreich:',
        'es': u':bandera_austria:',
        'fr': u':drapeau_autriche:',
        'pt': u':bandeira_áustria:',
        'it': u':bandiera_austria:'
    },
    u'\U0001F1E6\U0001F1FF': {
        'en' : u':Azerbaijan:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Azerbaijan:'],
        'de': u':flagge_aserbaidschan:',
        'es': u':bandera_azerbaiyán:',
        'fr': u':drapeau_azerbaïdjan:',
        'pt': u':bandeira_azerbaijão:',
        'it': u':bandiera_azerbaigian:'
    },
    u'\U0001F519': {
        'en' : u':BACK_arrow:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':back:'],
        'de': u':back-pfeil:',
        'es': u':flecha_back:',
        'fr': u':flèche_retour:',
        'pt': u':seta_back:',
        'it': u':freccia_back:'
    },
    u'\U0001F171\U0000FE0F': {
        'en' : u':B_button_(blood_type):',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':b:'],
        'variant': True,
        'de': u':großbuchstabe_b_in_rotem_quadrat:',
        'es': u':grupo_sanguíneo_b:',
        'fr': u':groupe_sanguin_b:',
        'pt': u':botão_b_(tipo_sanguíneo):',
        'it': u':gruppo_sanguigno_b:'
    },
    u'\U0001F171': {
        'en' : u':B_button_(blood_type):',
        'status' : unqualified,
        'E' : 0.6,
        'alias' : [u':b:'],
        'variant': True,
        'de': u':großbuchstabe_b_in_rotem_quadrat:',
        'es': u':grupo_sanguíneo_b:',
        'fr': u':groupe_sanguin_b:',
        'pt': u':botão_b_(tipo_sanguíneo):',
        'it': u':gruppo_sanguigno_b:'
    },
    u'\U0001F1E7\U0001F1F8': {
        'en' : u':Bahamas:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Bahamas:'],
        'de': u':flagge_bahamas:',
        'es': u':bandera_bahamas:',
        'fr': u':drapeau_bahamas:',
        'pt': u':bandeira_bahamas:',
        'it': u':bandiera_bahamas:'
    },
    u'\U0001F1E7\U0001F1ED': {
        'en' : u':Bahrain:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Bahrain:'],
        'de': u':flagge_bahrain:',
        'es': u':bandera_baréin:',
        'fr': u':drapeau_bahreïn:',
        'pt': u':bandeira_bahrein:',
        'it': u':bandiera_bahrein:'
    },
    u'\U0001F1E7\U0001F1E9': {
        'en' : u':Bangladesh:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Bangladesh:'],
        'de': u':flagge_bangladesch:',
        'es': u':bandera_bangladés:',
        'fr': u':drapeau_bangladesh:',
        'pt': u':bandeira_bangladesh:',
        'it': u':bandiera_bangladesh:'
    },
    u'\U0001F1E7\U0001F1E7': {
        'en' : u':Barbados:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Barbados:'],
        'de': u':flagge_barbados:',
        'es': u':bandera_barbados:',
        'fr': u':drapeau_barbade:',
        'pt': u':bandeira_barbados:',
        'it': u':bandiera_barbados:'
    },
    u'\U0001F1E7\U0001F1FE': {
        'en' : u':Belarus:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Belarus:'],
        'de': u':flagge_belarus:',
        'es': u':bandera_bielorrusia:',
        'fr': u':drapeau_biélorussie:',
        'pt': u':bandeira_bielorrússia:',
        'it': u':bandiera_bielorussia:'
    },
    u'\U0001F1E7\U0001F1EA': {
        'en' : u':Belgium:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Belgium:'],
        'de': u':flagge_belgien:',
        'es': u':bandera_bélgica:',
        'fr': u':drapeau_belgique:',
        'pt': u':bandeira_bélgica:',
        'it': u':bandiera_belgio:'
    },
    u'\U0001F1E7\U0001F1FF': {
        'en' : u':Belize:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Belize:'],
        'de': u':flagge_belize:',
        'es': u':bandera_belice:',
        'fr': u':drapeau_belize:',
        'pt': u':bandeira_belize:',
        'it': u':bandiera_belize:'
    },
    u'\U0001F1E7\U0001F1EF': {
        'en' : u':Benin:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Benin:'],
        'de': u':flagge_benin:',
        'es': u':bandera_benín:',
        'fr': u':drapeau_bénin:',
        'pt': u':bandeira_benin:',
        'it': u':bandiera_benin:'
    },
    u'\U0001F1E7\U0001F1F2': {
        'en' : u':Bermuda:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Bermuda:'],
        'de': u':flagge_bermuda:',
        'es': u':bandera_bermudas:',
        'fr': u':drapeau_bermudes:',
        'pt': u':bandeira_bermudas:',
        'it': u':bandiera_bermuda:'
    },
    u'\U0001F1E7\U0001F1F9': {
        'en' : u':Bhutan:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Bhutan:'],
        'de': u':flagge_bhutan:',
        'es': u':bandera_bután:',
        'fr': u':drapeau_bhoutan:',
        'pt': u':bandeira_butão:',
        'it': u':bandiera_bhutan:'
    },
    u'\U0001F1E7\U0001F1F4': {
        'en' : u':Bolivia:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Bolivia:'],
        'de': u':flagge_bolivien:',
        'es': u':bandera_bolivia:',
        'fr': u':drapeau_bolivie:',
        'pt': u':bandeira_bolívia:',
        'it': u':bandiera_bolivia:'
    },
    u'\U0001F1E7\U0001F1E6': {
        'en' : u':Bosnia_&_Herzegovina:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Bosnia_&_Herzegovina:'],
        'de': u':flagge_bosnien_und_herzegowina:',
        'es': u':bandera_bosnia_y_herzegovina:',
        'fr': u':drapeau_bosnie-herzégovine:',
        'pt': u':bandeira_bósnia_e_herzegovina:',
        'it': u':bandiera_bosnia_ed_erzegovina:'
    },
    u'\U0001F1E7\U0001F1FC': {
        'en' : u':Botswana:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Botswana:'],
        'de': u':flagge_botsuana:',
        'es': u':bandera_botsuana:',
        'fr': u':drapeau_botswana:',
        'pt': u':bandeira_botsuana:',
        'it': u':bandiera_botswana:'
    },
    u'\U0001F1E7\U0001F1FB': {
        'en' : u':Bouvet_Island:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Bouvet_Island:'],
        'de': u':flagge_bouvetinsel:',
        'es': u':bandera_isla_bouvet:',
        'fr': u':drapeau_île_bouvet:',
        'pt': u':bandeira_ilha_bouvet:',
        'it': u':bandiera_isola_bouvet:'
    },
    u'\U0001F1E7\U0001F1F7': {
        'en' : u':Brazil:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Brazil:'],
        'de': u':flagge_brasilien:',
        'es': u':bandera_brasil:',
        'fr': u':drapeau_brésil:',
        'pt': u':bandeira_brasil:',
        'it': u':bandiera_brasile:'
    },
    u'\U0001F1EE\U0001F1F4': {
        'en' : u':British_Indian_Ocean_Territory:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_British_Indian_Ocean_Territory:'],
        'de': u':flagge_britisches_territorium_im_indischen_ozean:',
        'es': u':bandera_territorio_británico_del_océano_índico:',
        'fr': u':drapeau_territoire_britannique_de_l’océan_indien:',
        'pt': u':bandeira_território_britânico_do_oceano_índico:',
        'it': u':bandiera_territorio_britannico_dell’oceano_indiano:'
    },
    u'\U0001F1FB\U0001F1EC': {
        'en' : u':British_Virgin_Islands:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_British_Virgin_Islands:'],
        'de': u':flagge_britische_jungferninseln:',
        'es': u':bandera_islas_vírgenes_británicas:',
        'fr': u':drapeau_îles_vierges_britanniques:',
        'pt': u':bandeira_ilhas_virgens_britânicas:',
        'it': u':bandiera_isole_vergini_britanniche:'
    },
    u'\U0001F1E7\U0001F1F3': {
        'en' : u':Brunei:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Brunei:'],
        'de': u':flagge_brunei_darussalam:',
        'es': u':bandera_brunéi:',
        'fr': u':drapeau_brunéi_darussalam:',
        'pt': u':bandeira_brunei:',
        'it': u':bandiera_brunei:'
    },
    u'\U0001F1E7\U0001F1EC': {
        'en' : u':Bulgaria:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Bulgaria:'],
        'de': u':flagge_bulgarien:',
        'es': u':bandera_bulgaria:',
        'fr': u':drapeau_bulgarie:',
        'pt': u':bandeira_bulgária:',
        'it': u':bandiera_bulgaria:'
    },
    u'\U0001F1E7\U0001F1EB': {
        'en' : u':Burkina_Faso:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Burkina_Faso:'],
        'de': u':flagge_burkina_faso:',
        'es': u':bandera_burkina_faso:',
        'fr': u':drapeau_burkina_faso:',
        'pt': u':bandeira_burquina_faso:',
        'it': u':bandiera_burkina_faso:'
    },
    u'\U0001F1E7\U0001F1EE': {
        'en' : u':Burundi:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Burundi:'],
        'de': u':flagge_burundi:',
        'es': u':bandera_burundi:',
        'fr': u':drapeau_burundi:',
        'pt': u':bandeira_burundi:',
        'it': u':bandiera_burundi:'
    },
    u'\U0001F191': {
        'en' : u':CL_button:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':cl:'],
        'de': u':großbuchstaben_cl_in_rotem_quadrat:',
        'es': u':borrar:',
        'fr': u':bouton_effacer:',
        'pt': u':botão_cl:',
        'it': u':pulsante_cl:'
    },
    u'\U0001F192': {
        'en' : u':COOL_button:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':cool:'],
        'de': u':wort_cool_in_blauem_quadrat:',
        'es': u':botón_cool:',
        'fr': u':bouton_cool:',
        'pt': u':botão_cool:',
        'it': u':pulsante_cool:'
    },
    u'\U0001F1F0\U0001F1ED': {
        'en' : u':Cambodia:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Cambodia:'],
        'de': u':flagge_kambodscha:',
        'es': u':bandera_camboya:',
        'fr': u':drapeau_cambodge:',
        'pt': u':bandeira_camboja:',
        'it': u':bandiera_cambogia:'
    },
    u'\U0001F1E8\U0001F1F2': {
        'en' : u':Cameroon:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Cameroon:'],
        'de': u':flagge_kamerun:',
        'es': u':bandera_camerún:',
        'fr': u':drapeau_cameroun:',
        'pt': u':bandeira_camarões:',
        'it': u':bandiera_camerun:'
    },
    u'\U0001F1E8\U0001F1E6': {
        'en' : u':Canada:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Canada:'],
        'de': u':flagge_kanada:',
        'es': u':bandera_canadá:',
        'fr': u':drapeau_canada:',
        'pt': u':bandeira_canadá:',
        'it': u':bandiera_canada:'
    },
    u'\U0001F1EE\U0001F1E8': {
        'en' : u':Canary_Islands:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Canary_Islands:'],
        'de': u':flagge_kanarische_inseln:',
        'es': u':bandera_canarias:',
        'fr': u':drapeau_îles_canaries:',
        'pt': u':bandeira_ilhas_canárias:',
        'it': u':bandiera_isole_canarie:'
    },
    u'\U0000264B': {
        'en' : u':Cancer:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':cancer:'],
        'variant': True,
        'de': u':krebs_(sternzeichen):',
        'es': u':cáncer:',
        'fr': u':cancer:',
        'pt': u':signo_de_câncer:',
        'it': u':segno_zodiacale_del_cancro:'
    },
    u'\U0001F1E8\U0001F1FB': {
        'en' : u':Cape_Verde:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Cape_Verde:'],
        'de': u':flagge_cabo_verde:',
        'es': u':bandera_cabo_verde:',
        'fr': u':drapeau_cap-vert:',
        'pt': u':bandeira_cabo_verde:',
        'it': u':bandiera_capo_verde:'
    },
    u'\U00002651': {
        'en' : u':Capricorn:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':capricorn:'],
        'variant': True,
        'de': u':steinbock_(sternzeichen):',
        'es': u':capricornio:',
        'fr': u':capricorne:',
        'pt': u':signo_de_capricórnio:',
        'it': u':segno_zodiacale_del_capricorno:'
    },
    u'\U0001F1E7\U0001F1F6': {
        'en' : u':Caribbean_Netherlands:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Caribbean_Netherlands:'],
        'de': u':flagge_bonaire,_sint_eustatius_und_saba:',
        'es': u':bandera_caribe_neerlandés:',
        'fr': u':drapeau_pays-bas_caribéens:',
        'pt': u':bandeira_países_baixos_caribenhos:',
        'it': u':bandiera_caraibi_olandesi:'
    },
    u'\U0001F1F0\U0001F1FE': {
        'en' : u':Cayman_Islands:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Cayman_Islands:'],
        'de': u':flagge_kaimaninseln:',
        'es': u':bandera_islas_caimán:',
        'fr': u':drapeau_îles_caïmans:',
        'pt': u':bandeira_ilhas_cayman:',
        'it': u':bandiera_isole_cayman:'
    },
    u'\U0001F1E8\U0001F1EB': {
        'en' : u':Central_African_Republic:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Central_African_Republic:'],
        'de': u':flagge_zentralafrikanische_republik:',
        'es': u':bandera_república_centroafricana:',
        'fr': u':drapeau_république_centrafricaine:',
        'pt': u':bandeira_república_centro-africana:',
        'it': u':bandiera_repubblica_centrafricana:'
    },
    u'\U0001F1EA\U0001F1E6': {
        'en' : u':Ceuta_&_Melilla:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Ceuta_&_Melilla:'],
        'de': u':flagge_ceuta_und_melilla:',
        'es': u':bandera_ceuta_y_melilla:',
        'fr': u':drapeau_ceuta_et_melilla:',
        'pt': u':bandeira_ceuta_e_melilla:',
        'it': u':bandiera_ceuta_e_melilla:'
    },
    u'\U0001F1F9\U0001F1E9': {
        'en' : u':Chad:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Chad:'],
        'de': u':flagge_tschad:',
        'es': u':bandera_chad:',
        'fr': u':drapeau_tchad:',
        'pt': u':bandeira_chade:',
        'it': u':bandiera_ciad:'
    },
    u'\U0001F1E8\U0001F1F1': {
        'en' : u':Chile:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Chile:'],
        'de': u':flagge_chile:',
        'es': u':bandera_chile:',
        'fr': u':drapeau_chili:',
        'pt': u':bandeira_chile:',
        'it': u':bandiera_cile:'
    },
    u'\U0001F1E8\U0001F1F3': {
        'en' : u':China:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':flag_for_China:'],
        'de': u':flagge_china:',
        'es': u':bandera_china:',
        'fr': u':drapeau_chine:',
        'pt': u':bandeira_china:',
        'it': u':bandiera_cina:'
    },
    u'\U0001F1E8\U0001F1FD': {
        'en' : u':Christmas_Island:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Christmas_Island:'],
        'de': u':flagge_weihnachtsinsel:',
        'es': u':bandera_isla_de_navidad:',
        'fr': u':drapeau_île_christmas:',
        'pt': u':bandeira_ilha_christmas:',
        'it': u':bandiera_isola_christmas:'
    },
    u'\U0001F384': {
        'en' : u':Christmas_tree:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':christmas_tree:'],
        'de': u':weihnachtsbaum:',
        'es': u':árbol_de_navidad:',
        'fr': u':sapin_de_noël:',
        'pt': u':árvore_de_natal:',
        'it': u':albero_di_natale:'
    },
    u'\U0001F1E8\U0001F1F5': {
        'en' : u':Clipperton_Island:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Clipperton_Island:'],
        'de': u':flagge_clipperton-insel:',
        'es': u':bandera_isla_clipperton:',
        'fr': u':drapeau_île_clipperton:',
        'pt': u':bandeira_ilha_de_clipperton:',
        'it': u':bandiera_isola_di_clipperton:'
    },
    u'\U0001F1E8\U0001F1E8': {
        'en' : u':Cocos_(Keeling)_Islands:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Cocos__Islands:'],
        'de': u':flagge_kokosinseln:',
        'es': u':bandera_islas_cocos:',
        'fr': u':drapeau_îles_cocos:',
        'pt': u':bandeira_ilhas_cocos_(keeling):',
        'it': u':bandiera_isole_cocos_(keeling):'
    },
    u'\U0001F1E8\U0001F1F4': {
        'en' : u':Colombia:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Colombia:'],
        'de': u':flagge_kolumbien:',
        'es': u':bandera_colombia:',
        'fr': u':drapeau_colombie:',
        'pt': u':bandeira_colômbia:',
        'it': u':bandiera_colombia:'
    },
    u'\U0001F1F0\U0001F1F2': {
        'en' : u':Comoros:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Comoros:'],
        'de': u':flagge_komoren:',
        'es': u':bandera_comoras:',
        'fr': u':drapeau_comores:',
        'pt': u':bandeira_comores:',
        'it': u':bandiera_comore:'
    },
    u'\U0001F1E8\U0001F1EC': {
        'en' : u':Congo_-_Brazzaville:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Congo____Brazzaville:'],
        'de': u':flagge_kongo-brazzaville:',
        'es': u':bandera_congo:',
        'fr': u':drapeau_congo-brazzaville:',
        'pt': u':bandeira_república_do_congo:',
        'it': u':bandiera_congo-brazzaville:'
    },
    u'\U0001F1E8\U0001F1E9': {
        'en' : u':Congo_-_Kinshasa:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Congo____Kinshasa:'],
        'de': u':flagge_kongo-kinshasa:',
        'es': u':bandera_república_democrática_del_congo:',
        'fr': u':drapeau_congo-kinshasa:',
        'pt': u':bandeira_congo_–_kinshasa:',
        'it': u':bandiera_congo_–_kinshasa:'
    },
    u'\U0001F1E8\U0001F1F0': {
        'en' : u':Cook_Islands:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Cook_Islands:'],
        'de': u':flagge_cookinseln:',
        'es': u':bandera_islas_cook:',
        'fr': u':drapeau_îles_cook:',
        'pt': u':bandeira_ilhas_cook:',
        'it': u':bandiera_isole_cook:'
    },
    u'\U0001F1E8\U0001F1F7': {
        'en' : u':Costa_Rica:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Costa_Rica:'],
        'de': u':flagge_costa_rica:',
        'es': u':bandera_costa_rica:',
        'fr': u':drapeau_costa_rica:',
        'pt': u':bandeira_costa_rica:',
        'it': u':bandiera_costa_rica:'
    },
    u'\U0001F1ED\U0001F1F7': {
        'en' : u':Croatia:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Croatia:'],
        'de': u':flagge_kroatien:',
        'es': u':bandera_croacia:',
        'fr': u':drapeau_croatie:',
        'pt': u':bandeira_croácia:',
        'it': u':bandiera_croazia:'
    },
    u'\U0001F1E8\U0001F1FA': {
        'en' : u':Cuba:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Cuba:'],
        'de': u':flagge_kuba:',
        'es': u':bandera_cuba:',
        'fr': u':drapeau_cuba:',
        'pt': u':bandeira_cuba:',
        'it': u':bandiera_cuba:'
    },
    u'\U0001F1E8\U0001F1FC': {
        'en' : u':Curaçao:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Curaçao:'],
        'de': u':flagge_curaçao:',
        'es': u':bandera_curazao:',
        'fr': u':drapeau_curaçao:',
        'pt': u':bandeira_curaçao:',
        'it': u':bandiera_curaçao:'
    },
    u'\U0001F1E8\U0001F1FE': {
        'en' : u':Cyprus:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Cyprus:'],
        'de': u':flagge_zypern:',
        'es': u':bandera_chipre:',
        'fr': u':drapeau_chypre:',
        'pt': u':bandeira_chipre:',
        'it': u':bandiera_cipro:'
    },
    u'\U0001F1E8\U0001F1FF': {
        'en' : u':Czechia:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Czech_Republic:'],
        'de': u':flagge_tschechien:',
        'es': u':bandera_chequia:',
        'fr': u':drapeau_tchéquie:',
        'pt': u':bandeira_tchéquia:',
        'it': u':bandiera_cechia:'
    },
    u'\U0001F1E8\U0001F1EE': {
        'en' : u':Côte_d’Ivoire:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Côte_d’Ivoire:'],
        'de': u':flagge_côte_d’ivoire:',
        'es': u':bandera_côte_d’ivoire:',
        'fr': u':drapeau_côte_d’ivoire:',
        'pt': u':bandeira_costa_do_marfim:',
        'it': u':bandiera_costa_d’avorio:'
    },
    u'\U0001F1E9\U0001F1F0': {
        'en' : u':Denmark:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Denmark:'],
        'de': u':flagge_dänemark:',
        'es': u':bandera_dinamarca:',
        'fr': u':drapeau_danemark:',
        'pt': u':bandeira_dinamarca:',
        'it': u':bandiera_danimarca:'
    },
    u'\U0001F1E9\U0001F1EC': {
        'en' : u':Diego_Garcia:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Diego_Garcia:'],
        'de': u':flagge_diego_garcia:',
        'es': u':bandera_diego_garcía:',
        'fr': u':drapeau_diego_garcia:',
        'pt': u':bandeira_diego_garcia:',
        'it': u':bandiera_diego_garcia:'
    },
    u'\U0001F1E9\U0001F1EF': {
        'en' : u':Djibouti:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Djibouti:'],
        'de': u':flagge_dschibuti:',
        'es': u':bandera_yibuti:',
        'fr': u':drapeau_djibouti:',
        'pt': u':bandeira_djibuti:',
        'it': u':bandiera_gibuti:'
    },
    u'\U0001F1E9\U0001F1F2': {
        'en' : u':Dominica:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Dominica:'],
        'de': u':flagge_dominica:',
        'es': u':bandera_dominica:',
        'fr': u':drapeau_dominique:',
        'pt': u':bandeira_dominica:',
        'it': u':bandiera_dominica:'
    },
    u'\U0001F1E9\U0001F1F4': {
        'en' : u':Dominican_Republic:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Dominican_Republic:'],
        'de': u':flagge_dominikanische_republik:',
        'es': u':bandera_república_dominicana:',
        'fr': u':drapeau_république_dominicaine:',
        'pt': u':bandeira_república_dominicana:',
        'it': u':bandiera_repubblica_dominicana:'
    },
    u'\U0001F51A': {
        'en' : u':END_arrow:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':end:'],
        'de': u':end-pfeil:',
        'es': u':flecha_end:',
        'fr': u':flèche_fin:',
        'pt': u':seta_end:',
        'it': u':freccia_end:'
    },
    u'\U0001F1EA\U0001F1E8': {
        'en' : u':Ecuador:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Ecuador:'],
        'de': u':flagge_ecuador:',
        'es': u':bandera_ecuador:',
        'fr': u':drapeau_équateur:',
        'pt': u':bandeira_equador:',
        'it': u':bandiera_ecuador:'
    },
    u'\U0001F1EA\U0001F1EC': {
        'en' : u':Egypt:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Egypt:'],
        'de': u':flagge_ägypten:',
        'es': u':bandera_egipto:',
        'fr': u':drapeau_égypte:',
        'pt': u':bandeira_egito:',
        'it': u':bandiera_egitto:'
    },
    u'\U0001F1F8\U0001F1FB': {
        'en' : u':El_Salvador:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_El_Salvador:'],
        'de': u':flagge_el_salvador:',
        'es': u':bandera_el_salvador:',
        'fr': u':drapeau_salvador:',
        'pt': u':bandeira_el_salvador:',
        'it': u':bandiera_el_salvador:'
    },
    u'\U0001F3F4\U000E0067\U000E0062\U000E0065\U000E006E\U000E0067\U000E007F': {
        'en' : u':England:',
        'status' : fully_qualified,
        'E' : 5,
        'alias' : [u':england:'],
        'de': u':flagge_england:',
        'es': u':bandera_inglaterra:',
        'fr': u':drapeau_angleterre:',
        'pt': u':bandeira_inglaterra:',
        'it': u':bandiera_inghilterra:'
    },
    u'\U0001F1EC\U0001F1F6': {
        'en' : u':Equatorial_Guinea:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Equatorial_Guinea:'],
        'de': u':flagge_äquatorialguinea:',
        'es': u':bandera_guinea_ecuatorial:',
        'fr': u':drapeau_guinée_équatoriale:',
        'pt': u':bandeira_guiné_equatorial:',
        'it': u':bandiera_guinea_equatoriale:'
    },
    u'\U0001F1EA\U0001F1F7': {
        'en' : u':Eritrea:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Eritrea:'],
        'de': u':flagge_eritrea:',
        'es': u':bandera_eritrea:',
        'fr': u':drapeau_érythrée:',
        'pt': u':bandeira_eritreia:',
        'it': u':bandiera_eritrea:'
    },
    u'\U0001F1EA\U0001F1EA': {
        'en' : u':Estonia:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Estonia:'],
        'de': u':flagge_estland:',
        'es': u':bandera_estonia:',
        'fr': u':drapeau_estonie:',
        'pt': u':bandeira_estônia:',
        'it': u':bandiera_estonia:'
    },
    u'\U0001F1F8\U0001F1FF': {
        'en' : u':Eswatini:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Swaziland:'],
        'de': u':flagge_eswatini:',
        'es': u':bandera_esuatini:',
        'fr': u':drapeau_eswatini:',
        'pt': u':bandeira_essuatíni:',
        'it': u':bandiera_swaziland:'
    },
    u'\U0001F1EA\U0001F1F9': {
        'en' : u':Ethiopia:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Ethiopia:'],
        'de': u':flagge_äthiopien:',
        'es': u':bandera_etiopía:',
        'fr': u':drapeau_éthiopie:',
        'pt': u':bandeira_etiópia:',
        'it': u':bandiera_etiopia:'
    },
    u'\U0001F1EA\U0001F1FA': {
        'en' : u':European_Union:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_European_Union:'],
        'de': u':flagge_europäische_union:',
        'es': u':bandera_unión_europea:',
        'fr': u':drapeau_union_européenne:',
        'pt': u':bandeira_união_europeia:',
        'it': u':bandiera_unione_europea:'
    },
    u'\U0001F193': {
        'en' : u':FREE_button:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':free:'],
        'de': u':wort_free_in_blauem_quadrat:',
        'es': u':botón_free:',
        'fr': u':bouton_gratuit:',
        'pt': u':botão_free:',
        'it': u':pulsante_free:'
    },
    u'\U0001F1EB\U0001F1F0': {
        'en' : u':Falkland_Islands:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Falkland_Islands:'],
        'de': u':flagge_falklandinseln:',
        'es': u':bandera_islas_malvinas:',
        'fr': u':drapeau_îles_malouines:',
        'pt': u':bandeira_ilhas_malvinas:',
        'it': u':bandiera_isole_falkland:'
    },
    u'\U0001F1EB\U0001F1F4': {
        'en' : u':Faroe_Islands:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Faroe_Islands:'],
        'de': u':flagge_färöer:',
        'es': u':bandera_islas_feroe:',
        'fr': u':drapeau_îles_féroé:',
        'pt': u':bandeira_ilhas_faroe:',
        'it': u':bandiera_isole_fær_øer:'
    },
    u'\U0001F1EB\U0001F1EF': {
        'en' : u':Fiji:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Fiji:'],
        'de': u':flagge_fidschi:',
        'es': u':bandera_fiyi:',
        'fr': u':drapeau_fidji:',
        'pt': u':bandeira_fiji:',
        'it': u':bandiera_figi:'
    },
    u'\U0001F1EB\U0001F1EE': {
        'en' : u':Finland:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Finland:'],
        'de': u':flagge_finnland:',
        'es': u':bandera_finlandia:',
        'fr': u':drapeau_finlande:',
        'pt': u':bandeira_finlândia:',
        'it': u':bandiera_finlandia:'
    },
    u'\U0001F1EB\U0001F1F7': {
        'en' : u':France:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':flag_for_France:'],
        'de': u':flagge_frankreich:',
        'es': u':bandera_francia:',
        'fr': u':drapeau_france:',
        'pt': u':bandeira_frança:',
        'it': u':bandiera_francia:'
    },
    u'\U0001F1EC\U0001F1EB': {
        'en' : u':French_Guiana:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_French_Guiana:'],
        'de': u':flagge_französisch-guayana:',
        'es': u':bandera_guayana_francesa:',
        'fr': u':drapeau_guyane_française:',
        'pt': u':bandeira_guiana_francesa:',
        'it': u':bandiera_guyana_francese:'
    },
    u'\U0001F1F5\U0001F1EB': {
        'en' : u':French_Polynesia:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_French_Polynesia:'],
        'de': u':flagge_französisch-polynesien:',
        'es': u':bandera_polinesia_francesa:',
        'fr': u':drapeau_polynésie_française:',
        'pt': u':bandeira_polinésia_francesa:',
        'it': u':bandiera_polinesia_francese:'
    },
    u'\U0001F1F9\U0001F1EB': {
        'en' : u':French_Southern_Territories:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_French_Southern_Territories:'],
        'de': u':flagge_französische_süd-_und_antarktisgebiete:',
        'es': u':bandera_territorios_australes_franceses:',
        'fr': u':drapeau_terres_australes_françaises:',
        'pt': u':bandeira_territórios_franceses_do_sul:',
        'it': u':bandiera_terre_australi_francesi:'
    },
    u'\U0001F1EC\U0001F1E6': {
        'en' : u':Gabon:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Gabon:'],
        'de': u':flagge_gabun:',
        'es': u':bandera_gabón:',
        'fr': u':drapeau_gabon:',
        'pt': u':bandeira_gabão:',
        'it': u':bandiera_gabon:'
    },
    u'\U0001F1EC\U0001F1F2': {
        'en' : u':Gambia:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Gambia:'],
        'de': u':flagge_gambia:',
        'es': u':bandera_gambia:',
        'fr': u':drapeau_gambie:',
        'pt': u':bandeira_gâmbia:',
        'it': u':bandiera_gambia:'
    },
    u'\U0000264A': {
        'en' : u':Gemini:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':gemini:'],
        'variant': True,
        'de': u':zwilling_(sternzeichen):',
        'es': u':géminis:',
        'fr': u':gémeaux:',
        'pt': u':signo_de_gêmeos:',
        'it': u':segno_zodiacale_dei_gemelli:'
    },
    u'\U0001F1EC\U0001F1EA': {
        'en' : u':Georgia:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Georgia:'],
        'de': u':flagge_georgien:',
        'es': u':bandera_georgia:',
        'fr': u':drapeau_géorgie:',
        'pt': u':bandeira_geórgia:',
        'it': u':bandiera_georgia:'
    },
    u'\U0001F1E9\U0001F1EA': {
        'en' : u':Germany:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':flag_for_Germany:'],
        'de': u':flagge_deutschland:',
        'es': u':bandera_alemania:',
        'fr': u':drapeau_allemagne:',
        'pt': u':bandeira_alemanha:',
        'it': u':bandiera_germania:'
    },
    u'\U0001F1EC\U0001F1ED': {
        'en' : u':Ghana:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Ghana:'],
        'de': u':flagge_ghana:',
        'es': u':bandera_ghana:',
        'fr': u':drapeau_ghana:',
        'pt': u':bandeira_gana:',
        'it': u':bandiera_ghana:'
    },
    u'\U0001F1EC\U0001F1EE': {
        'en' : u':Gibraltar:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Gibraltar:'],
        'de': u':flagge_gibraltar:',
        'es': u':bandera_gibraltar:',
        'fr': u':drapeau_gibraltar:',
        'pt': u':bandeira_gibraltar:',
        'it': u':bandiera_gibilterra:'
    },
    u'\U0001F1EC\U0001F1F7': {
        'en' : u':Greece:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Greece:'],
        'de': u':flagge_griechenland:',
        'es': u':bandera_grecia:',
        'fr': u':drapeau_grèce:',
        'pt': u':bandeira_grécia:',
        'it': u':bandiera_grecia:'
    },
    u'\U0001F1EC\U0001F1F1': {
        'en' : u':Greenland:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Greenland:'],
        'de': u':flagge_grönland:',
        'es': u':bandera_groenlandia:',
        'fr': u':drapeau_groenland:',
        'pt': u':bandeira_groenlândia:',
        'it': u':bandiera_groenlandia:'
    },
    u'\U0001F1EC\U0001F1E9': {
        'en' : u':Grenada:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Grenada:'],
        'de': u':flagge_grenada:',
        'es': u':bandera_granada:',
        'fr': u':drapeau_grenade:',
        'pt': u':bandeira_granada:',
        'it': u':bandiera_grenada:'
    },
    u'\U0001F1EC\U0001F1F5': {
        'en' : u':Guadeloupe:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Guadeloupe:'],
        'de': u':flagge_guadeloupe:',
        'es': u':bandera_guadalupe:',
        'fr': u':drapeau_guadeloupe:',
        'pt': u':bandeira_guadalupe:',
        'it': u':bandiera_guadalupa:'
    },
    u'\U0001F1EC\U0001F1FA': {
        'en' : u':Guam:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Guam:'],
        'de': u':flagge_guam:',
        'es': u':bandera_guam:',
        'fr': u':drapeau_guam:',
        'pt': u':bandeira_guam:',
        'it': u':bandiera_guam:'
    },
    u'\U0001F1EC\U0001F1F9': {
        'en' : u':Guatemala:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Guatemala:'],
        'de': u':flagge_guatemala:',
        'es': u':bandera_guatemala:',
        'fr': u':drapeau_guatemala:',
        'pt': u':bandeira_guatemala:',
        'it': u':bandiera_guatemala:'
    },
    u'\U0001F1EC\U0001F1EC': {
        'en' : u':Guernsey:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Guernsey:'],
        'de': u':flagge_guernsey:',
        'es': u':bandera_guernsey:',
        'fr': u':drapeau_guernesey:',
        'pt': u':bandeira_guernsey:',
        'it': u':bandiera_guernsey:'
    },
    u'\U0001F1EC\U0001F1F3': {
        'en' : u':Guinea:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Guinea:'],
        'de': u':flagge_guinea:',
        'es': u':bandera_guinea:',
        'fr': u':drapeau_guinée:',
        'pt': u':bandeira_guiné:',
        'it': u':bandiera_guinea:'
    },
    u'\U0001F1EC\U0001F1FC': {
        'en' : u':Guinea-Bissau:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Guinea__Bissau:'],
        'de': u':flagge_guinea-bissau:',
        'es': u':bandera_guinea-bisáu:',
        'fr': u':drapeau_guinée-bissau:',
        'pt': u':bandeira_guiné-bissau:',
        'it': u':bandiera_guinea-bissau:'
    },
    u'\U0001F1EC\U0001F1FE': {
        'en' : u':Guyana:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Guyana:'],
        'de': u':flagge_guyana:',
        'es': u':bandera_guyana:',
        'fr': u':drapeau_guyana:',
        'pt': u':bandeira_guiana:',
        'it': u':bandiera_guyana:'
    },
    u'\U0001F1ED\U0001F1F9': {
        'en' : u':Haiti:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Haiti:'],
        'de': u':flagge_haiti:',
        'es': u':bandera_haití:',
        'fr': u':drapeau_haïti:',
        'pt': u':bandeira_haiti:',
        'it': u':bandiera_haiti:'
    },
    u'\U0001F1ED\U0001F1F2': {
        'en' : u':Heard_&_McDonald_Islands:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Heard_&_McDonald_Islands:'],
        'de': u':flagge_heard_und_mcdonaldinseln:',
        'es': u':bandera_islas_heard_y_mcdonald:',
        'fr': u':drapeau_îles_heard_et_mcdonald:',
        'pt': u':bandeira_ilhas_heard_e_mcdonald:',
        'it': u':bandiera_isole_heard_e_mcdonald:'
    },
    u'\U0001F1ED\U0001F1F3': {
        'en' : u':Honduras:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Honduras:'],
        'de': u':flagge_honduras:',
        'es': u':bandera_honduras:',
        'fr': u':drapeau_honduras:',
        'pt': u':bandeira_honduras:',
        'it': u':bandiera_honduras:'
    },
    u'\U0001F1ED\U0001F1F0': {
        'en' : u':Hong_Kong_SAR_China:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Hong_Kong:'],
        'de': u':flagge_sonderverwaltungsregion_hongkong:',
        'es': u':bandera_rae_de_hong_kong_(china):',
        'fr': u':drapeau_r.a.s._chinoise_de_hong_kong:',
        'pt': u':bandeira_hong_kong_rae_da_china:',
        'it': u':bandiera_ras_di_hong_kong:'
    },
    u'\U0001F1ED\U0001F1FA': {
        'en' : u':Hungary:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Hungary:'],
        'de': u':flagge_ungarn:',
        'es': u':bandera_hungría:',
        'fr': u':drapeau_hongrie:',
        'pt': u':bandeira_hungria:',
        'it': u':bandiera_ungheria:'
    },
    u'\U0001F194': {
        'en' : u':ID_button:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':id:'],
        'de': u':großbuchstaben_id_in_lila_quadrat:',
        'es': u':símbolo_de_identificación:',
        'fr': u':bouton_identifiant:',
        'pt': u':botão_id:',
        'it': u':pulsante_id:'
    },
    u'\U0001F1EE\U0001F1F8': {
        'en' : u':Iceland:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Iceland:'],
        'de': u':flagge_island:',
        'es': u':bandera_islandia:',
        'fr': u':drapeau_islande:',
        'pt': u':bandeira_islândia:',
        'it': u':bandiera_islanda:'
    },
    u'\U0001F1EE\U0001F1F3': {
        'en' : u':India:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_India:'],
        'de': u':flagge_indien:',
        'es': u':bandera_india:',
        'fr': u':drapeau_inde:',
        'pt': u':bandeira_índia:',
        'it': u':bandiera_india:'
    },
    u'\U0001F1EE\U0001F1E9': {
        'en' : u':Indonesia:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Indonesia:'],
        'de': u':flagge_indonesien:',
        'es': u':bandera_indonesia:',
        'fr': u':drapeau_indonésie:',
        'pt': u':bandeira_indonésia:',
        'it': u':bandiera_indonesia:'
    },
    u'\U0001F1EE\U0001F1F7': {
        'en' : u':Iran:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Iran:'],
        'de': u':flagge_iran:',
        'es': u':bandera_irán:',
        'fr': u':drapeau_iran:',
        'pt': u':bandeira_irã:',
        'it': u':bandiera_iran:'
    },
    u'\U0001F1EE\U0001F1F6': {
        'en' : u':Iraq:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Iraq:'],
        'de': u':flagge_irak:',
        'es': u':bandera_irak:',
        'fr': u':drapeau_irak:',
        'pt': u':bandeira_iraque:',
        'it': u':bandiera_iraq:'
    },
    u'\U0001F1EE\U0001F1EA': {
        'en' : u':Ireland:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Ireland:'],
        'de': u':flagge_irland:',
        'es': u':bandera_irlanda:',
        'fr': u':drapeau_irlande:',
        'pt': u':bandeira_irlanda:',
        'it': u':bandiera_irlanda:'
    },
    u'\U0001F1EE\U0001F1F2': {
        'en' : u':Isle_of_Man:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Isle_of_Man:'],
        'de': u':flagge_isle_of_man:',
        'es': u':bandera_isla_de_man:',
        'fr': u':drapeau_île_de_man:',
        'pt': u':bandeira_ilha_de_man:',
        'it': u':bandiera_isola_di_man:'
    },
    u'\U0001F1EE\U0001F1F1': {
        'en' : u':Israel:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Israel:'],
        'de': u':flagge_israel:',
        'es': u':bandera_israel:',
        'fr': u':drapeau_israël:',
        'pt': u':bandeira_israel:',
        'it': u':bandiera_israele:'
    },
    u'\U0001F1EE\U0001F1F9': {
        'en' : u':Italy:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':flag_for_Italy:'],
        'de': u':flagge_italien:',
        'es': u':bandera_italia:',
        'fr': u':drapeau_italie:',
        'pt': u':bandeira_itália:',
        'it': u':bandiera_italia:'
    },
    u'\U0001F1EF\U0001F1F2': {
        'en' : u':Jamaica:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Jamaica:'],
        'de': u':flagge_jamaika:',
        'es': u':bandera_jamaica:',
        'fr': u':drapeau_jamaïque:',
        'pt': u':bandeira_jamaica:',
        'it': u':bandiera_giamaica:'
    },
    u'\U0001F1EF\U0001F1F5': {
        'en' : u':Japan:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':flag_for_Japan:'],
        'de': u':flagge_japan:',
        'es': u':bandera_japón:',
        'fr': u':drapeau_japon:',
        'pt': u':bandeira_japão:',
        'it': u':bandiera_giappone:'
    },
    u'\U0001F251': {
        'en' : u':Japanese_acceptable_button:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':accept:'],
        'de': u':schriftzeichen_für_akzeptieren:',
        'es': u':ideograma_japonés_para_aceptable:',
        'fr': u':bouton_accepter_en_japonais:',
        'pt': u':botão_japonês_de_“aceitável”:',
        'it': u':ideogramma_giapponese_di_“accettabile”:'
    },
    u'\U0001F238': {
        'en' : u':Japanese_application_button:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':u7533:'],
        'de': u':schriftzeichen_für_anwenden:',
        'es': u':ideograma_japonés_para_aplicación:',
        'fr': u':bouton_application_en_japonais:',
        'pt': u':botão_japonês_de_“aplicação”:',
        'it': u':ideogramma_giapponese_di_“candidatura”:'
    },
    u'\U0001F250': {
        'en' : u':Japanese_bargain_button:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':ideograph_advantage:'],
        'de': u':schriftzeichen_für_schnäppchen:',
        'es': u':ideograma_japonés_para_ganga:',
        'fr': u':bouton_bonne_affaire_en_japonais:',
        'pt': u':botão_japonês_de_“barganha”:',
        'it': u':ideogramma_giapponese_di_“occasione”:'
    },
    u'\U0001F3EF': {
        'en' : u':Japanese_castle:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':japanese_castle:'],
        'de': u':japanisches_schloss:',
        'es': u':castillo_japonés:',
        'fr': u':château_japonais:',
        'pt': u':castelo_japonês:',
        'it': u':castello_giapponese:'
    },
    u'\U00003297\U0000FE0F': {
        'en' : u':Japanese_congratulations_button:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':congratulations:'],
        'variant': True,
        'de': u':schriftzeichen_für_gratulation:',
        'es': u':ideograma_japonés_para_enhorabuena:',
        'fr': u':bouton_félicitations_en_japonais:',
        'pt': u':botão_japonês_de_“parabéns”:',
        'it': u':ideogramma_giapponese_di_“congratulazioni”:'
    },
    u'\U00003297': {
        'en' : u':Japanese_congratulations_button:',
        'status' : unqualified,
        'E' : 0.6,
        'alias' : [u':congratulations:'],
        'variant': True,
        'de': u':schriftzeichen_für_gratulation:',
        'es': u':ideograma_japonés_para_enhorabuena:',
        'fr': u':bouton_félicitations_en_japonais:',
        'pt': u':botão_japonês_de_“parabéns”:',
        'it': u':ideogramma_giapponese_di_“congratulazioni”:'
    },
    u'\U0001F239': {
        'en' : u':Japanese_discount_button:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':u5272:'],
        'de': u':schriftzeichen_für_rabatt:',
        'es': u':ideograma_japonés_para_descuento:',
        'fr': u':bouton_réduction_en_japonais:',
        'pt': u':botão_japonês_de_“desconto”:',
        'it': u':ideogramma_giapponese_di_“sconto”:'
    },
    u'\U0001F38E': {
        'en' : u':Japanese_dolls:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':dolls:'],
        'de': u':japanische_puppen:',
        'es': u':muñecas_japonesas:',
        'fr': u':poupées_japonaises:',
        'pt': u':bonecas_japonesas:',
        'it': u':bambole_giapponesi:'
    },
    u'\U0001F21A': {
        'en' : u':Japanese_free_of_charge_button:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':u7121:'],
        'variant': True,
        'de': u':schriftzeichen_für_gratis:',
        'es': u':ideograma_japonés_para_gratis:',
        'fr': u':bouton_gratuit_en_japonais:',
        'pt': u':botão_japonês_de_“gratuito”:',
        'it': u':ideogramma_giapponese_di_“gratis”:'
    },
    u'\U0001F201': {
        'en' : u':Japanese_here_button:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':koko:'],
        'de': u':schriftzeichen_koko:',
        'es': u':ideograma_japonés_para_aquí:',
        'fr': u':bouton_ici_en_japonais:',
        'pt': u':botão_japonês_de_“aqui”:',
        'it': u':ideogramma_giapponese_per_“qui”:'
    },
    u'\U0001F237\U0000FE0F': {
        'en' : u':Japanese_monthly_amount_button:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':u6708:'],
        'variant': True,
        'de': u':schriftzeichen_für_monatsbetrag:',
        'es': u':ideograma_japonés_para_cantidad_mensual:',
        'fr': u':bouton_montant_mensuel_en_japonais:',
        'pt': u':botão_japonês_de_“quantidade_mensal”:',
        'it': u':ideogramma_giapponese_di_“importo_mensile”:'
    },
    u'\U0001F237': {
        'en' : u':Japanese_monthly_amount_button:',
        'status' : unqualified,
        'E' : 0.6,
        'alias' : [u':u6708:'],
        'variant': True,
        'de': u':schriftzeichen_für_monatsbetrag:',
        'es': u':ideograma_japonés_para_cantidad_mensual:',
        'fr': u':bouton_montant_mensuel_en_japonais:',
        'pt': u':botão_japonês_de_“quantidade_mensal”:',
        'it': u':ideogramma_giapponese_di_“importo_mensile”:'
    },
    u'\U0001F235': {
        'en' : u':Japanese_no_vacancy_button:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':u6e80:'],
        'de': u':schriftzeichen_für_kein_zimmer_frei:',
        'es': u':ideograma_japonés_para_completo:',
        'fr': u':bouton_complet_en_japonais:',
        'pt': u':botão_japonês_de_“sem_vagas”:',
        'it': u':ideogramma_giapponese_di_“nessun_posto_libero”:'
    },
    u'\U0001F236': {
        'en' : u':Japanese_not_free_of_charge_button:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':u6709:'],
        'de': u':schriftzeichen_für_nicht_gratis:',
        'es': u':ideograma_japonés_para_de_pago:',
        'fr': u':bouton_pas_gratuit_en_japonais:',
        'pt': u':botão_japonês_de_“não_gratuito”:',
        'it': u':ideogramma_giapponese_di_“a_pagamento”:'
    },
    u'\U0001F23A': {
        'en' : u':Japanese_open_for_business_button:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':u55b6:'],
        'de': u':schriftzeichen_für_geöffnet:',
        'es': u':ideograma_japonés_para_abierto:',
        'fr': u':bouton_ouvert_pour_affaires_en_japonais:',
        'pt': u':botão_japonês_de_“aberto_para_negócios”:',
        'it': u':ideogramma_giapponese_di_“aperto_al_pubblico”:'
    },
    u'\U0001F234': {
        'en' : u':Japanese_passing_grade_button:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':u5408:'],
        'de': u':schriftzeichen_für_note_zum_bestehen:',
        'es': u':ideograma_japonés_para_aprobado:',
        'fr': u':bouton_note_pour_réussir_en_japonais:',
        'pt': u':botão_japonês_de_“nota_mínima”:',
        'it': u':ideogramma_giapponese_di_“voto_di_sufficienza”:'
    },
    u'\U0001F3E3': {
        'en' : u':Japanese_post_office:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':post_office:'],
        'de': u':japanisches_postgebäude:',
        'es': u':oficina_de_correos_japonesa:',
        'fr': u':bureau_de_poste_japonais:',
        'pt': u':correio_japonês:',
        'it': u':ufficio_postale_giapponese:'
    },
    u'\U0001F232': {
        'en' : u':Japanese_prohibited_button:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':u7981:'],
        'de': u':schriftzeichen_für_verbieten:',
        'es': u':ideograma_japonés_para_prohibido:',
        'fr': u':bouton_interdit_en_japonais:',
        'pt': u':botão_japonês_de_“proibido”:',
        'it': u':ideogramma_giapponese_di_“proibito”:'
    },
    u'\U0001F22F': {
        'en' : u':Japanese_reserved_button:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':u6307:'],
        'variant': True,
        'de': u':schriftzeichen_für_reserviert:',
        'es': u':ideograma_japonés_para_reservado:',
        'fr': u':bouton_réservé_en_japonais:',
        'pt': u':botão_japonês_de_“reservado”:',
        'it': u':ideogramma_giapponese_di_“riservato”:'
    },
    u'\U00003299\U0000FE0F': {
        'en' : u':Japanese_secret_button:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':secret:'],
        'variant': True,
        'de': u':schriftzeichen_für_geheimnis:',
        'es': u':ideograma_japonés_para_secreto:',
        'fr': u':bouton_secret_en_japonais:',
        'pt': u':botão_japonês_de_“segredo”:',
        'it': u':ideogramma_giapponese_di_“segreto”:'
    },
    u'\U00003299': {
        'en' : u':Japanese_secret_button:',
        'status' : unqualified,
        'E' : 0.6,
        'alias' : [u':secret:'],
        'variant': True,
        'de': u':schriftzeichen_für_geheimnis:',
        'es': u':ideograma_japonés_para_secreto:',
        'fr': u':bouton_secret_en_japonais:',
        'pt': u':botão_japonês_de_“segredo”:',
        'it': u':ideogramma_giapponese_di_“segreto”:'
    },
    u'\U0001F202\U0000FE0F': {
        'en' : u':Japanese_service_charge_button:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':sa:'],
        'variant': True,
        'de': u':schriftzeichen_sa:',
        'es': u':ideograma_japonés_para_de_cortesía:',
        'fr': u':bouton_frais_de_service_en_japonais:',
        'pt': u':botão_japonês_de_“taxa_de_serviço”:',
        'it': u':ideogramma_giapponese_di_“costo_del_servizio”:'
    },
    u'\U0001F202': {
        'en' : u':Japanese_service_charge_button:',
        'status' : unqualified,
        'E' : 0.6,
        'alias' : [u':sa:'],
        'variant': True,
        'de': u':schriftzeichen_sa:',
        'es': u':ideograma_japonés_para_de_cortesía:',
        'fr': u':bouton_frais_de_service_en_japonais:',
        'pt': u':botão_japonês_de_“taxa_de_serviço”:',
        'it': u':ideogramma_giapponese_di_“costo_del_servizio”:'
    },
    u'\U0001F530': {
        'en' : u':Japanese_symbol_for_beginner:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':beginner:'],
        'de': u':japanisches_anfänger-zeichen:',
        'es': u':símbolo_japonés_para_principiante:',
        'fr': u':symbole_japonais_de_débutant:',
        'pt': u':símbolo_japonês_de_principiante:',
        'it': u':ideogramma_giapponese_di_“principiante”:'
    },
    u'\U0001F233': {
        'en' : u':Japanese_vacancy_button:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':u7a7a:'],
        'de': u':schriftzeichen_für_zimmer_frei:',
        'es': u':ideograma_japonés_para_vacante:',
        'fr': u':bouton_chambres_disponibles_en_japonais:',
        'pt': u':botão_japonês_de_“vago”:',
        'it': u':ideogramma_giapponese_di_“posto_libero”:'
    },
    u'\U0001F1EF\U0001F1EA': {
        'en' : u':Jersey:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Jersey:'],
        'de': u':flagge_jersey:',
        'es': u':bandera_jersey:',
        'fr': u':drapeau_jersey:',
        'pt': u':bandeira_jersey:',
        'it': u':bandiera_jersey:'
    },
    u'\U0001F1EF\U0001F1F4': {
        'en' : u':Jordan:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Jordan:'],
        'de': u':flagge_jordanien:',
        'es': u':bandera_jordania:',
        'fr': u':drapeau_jordanie:',
        'pt': u':bandeira_jordânia:',
        'it': u':bandiera_giordania:'
    },
    u'\U0001F1F0\U0001F1FF': {
        'en' : u':Kazakhstan:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Kazakhstan:'],
        'de': u':flagge_kasachstan:',
        'es': u':bandera_kazajistán:',
        'fr': u':drapeau_kazakhstan:',
        'pt': u':bandeira_cazaquistão:',
        'it': u':bandiera_kazakistan:'
    },
    u'\U0001F1F0\U0001F1EA': {
        'en' : u':Kenya:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Kenya:'],
        'de': u':flagge_kenia:',
        'es': u':bandera_kenia:',
        'fr': u':drapeau_kenya:',
        'pt': u':bandeira_quênia:',
        'it': u':bandiera_kenya:'
    },
    u'\U0001F1F0\U0001F1EE': {
        'en' : u':Kiribati:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Kiribati:'],
        'de': u':flagge_kiribati:',
        'es': u':bandera_kiribati:',
        'fr': u':drapeau_kiribati:',
        'pt': u':bandeira_quiribati:',
        'it': u':bandiera_kiribati:'
    },
    u'\U0001F1FD\U0001F1F0': {
        'en' : u':Kosovo:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Kosovo:'],
        'de': u':flagge_kosovo:',
        'es': u':bandera_kosovo:',
        'fr': u':drapeau_kosovo:',
        'pt': u':bandeira_kosovo:',
        'it': u':bandiera_kosovo:'
    },
    u'\U0001F1F0\U0001F1FC': {
        'en' : u':Kuwait:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Kuwait:'],
        'de': u':flagge_kuwait:',
        'es': u':bandera_kuwait:',
        'fr': u':drapeau_koweït:',
        'pt': u':bandeira_kuwait:',
        'it': u':bandiera_kuwait:'
    },
    u'\U0001F1F0\U0001F1EC': {
        'en' : u':Kyrgyzstan:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Kyrgyzstan:'],
        'de': u':flagge_kirgisistan:',
        'es': u':bandera_kirguistán:',
        'fr': u':drapeau_kirghizistan:',
        'pt': u':bandeira_quirguistão:',
        'it': u':bandiera_kirghizistan:'
    },
    u'\U0001F1F1\U0001F1E6': {
        'en' : u':Laos:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Laos:'],
        'de': u':flagge_laos:',
        'es': u':bandera_laos:',
        'fr': u':drapeau_laos:',
        'pt': u':bandeira_laos:',
        'it': u':bandiera_laos:'
    },
    u'\U0001F1F1\U0001F1FB': {
        'en' : u':Latvia:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Latvia:'],
        'de': u':flagge_lettland:',
        'es': u':bandera_letonia:',
        'fr': u':drapeau_lettonie:',
        'pt': u':bandeira_letônia:',
        'it': u':bandiera_lettonia:'
    },
    u'\U0001F1F1\U0001F1E7': {
        'en' : u':Lebanon:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Lebanon:'],
        'de': u':flagge_libanon:',
        'es': u':bandera_líbano:',
        'fr': u':drapeau_liban:',
        'pt': u':bandeira_líbano:',
        'it': u':bandiera_libano:'
    },
    u'\U0000264C': {
        'en' : u':Leo:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':leo:'],
        'variant': True,
        'de': u':löwe_(sternzeichen):',
        'es': u':leo:',
        'fr': u':lion:',
        'pt': u':signo_de_leão:',
        'it': u':segno_zodiacale_del_leone:'
    },
    u'\U0001F1F1\U0001F1F8': {
        'en' : u':Lesotho:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Lesotho:'],
        'de': u':flagge_lesotho:',
        'es': u':bandera_lesoto:',
        'fr': u':drapeau_lesotho:',
        'pt': u':bandeira_lesoto:',
        'it': u':bandiera_lesotho:'
    },
    u'\U0001F1F1\U0001F1F7': {
        'en' : u':Liberia:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Liberia:'],
        'de': u':flagge_liberia:',
        'es': u':bandera_liberia:',
        'fr': u':drapeau_libéria:',
        'pt': u':bandeira_libéria:',
        'it': u':bandiera_liberia:'
    },
    u'\U0000264E': {
        'en' : u':Libra:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':libra:'],
        'variant': True,
        'de': u':waage_(sternzeichen):',
        'es': u':libra:',
        'fr': u':balance:',
        'pt': u':signo_de_libra:',
        'it': u':segno_zodiacale_della_bilancia:'
    },
    u'\U0001F1F1\U0001F1FE': {
        'en' : u':Libya:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Libya:'],
        'de': u':flagge_libyen:',
        'es': u':bandera_libia:',
        'fr': u':drapeau_libye:',
        'pt': u':bandeira_líbia:',
        'it': u':bandiera_libia:'
    },
    u'\U0001F1F1\U0001F1EE': {
        'en' : u':Liechtenstein:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Liechtenstein:'],
        'de': u':flagge_liechtenstein:',
        'es': u':bandera_liechtenstein:',
        'fr': u':drapeau_liechtenstein:',
        'pt': u':bandeira_liechtenstein:',
        'it': u':bandiera_liechtenstein:'
    },
    u'\U0001F1F1\U0001F1F9': {
        'en' : u':Lithuania:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Lithuania:'],
        'de': u':flagge_litauen:',
        'es': u':bandera_lituania:',
        'fr': u':drapeau_lituanie:',
        'pt': u':bandeira_lituânia:',
        'it': u':bandiera_lituania:'
    },
    u'\U0001F1F1\U0001F1FA': {
        'en' : u':Luxembourg:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Luxembourg:'],
        'de': u':flagge_luxemburg:',
        'es': u':bandera_luxemburgo:',
        'fr': u':drapeau_luxembourg:',
        'pt': u':bandeira_luxemburgo:',
        'it': u':bandiera_lussemburgo:'
    },
    u'\U0001F1F2\U0001F1F4': {
        'en' : u':Macao_SAR_China:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Macau:'],
        'de': u':flagge_sonderverwaltungsregion_macau:',
        'es': u':bandera_rae_de_macao_(china):',
        'fr': u':drapeau_r.a.s._chinoise_de_macao:',
        'pt': u':bandeira_macau_rae_da_china:',
        'it': u':bandiera_ras_di_macao:'
    },
    u'\U0001F1F2\U0001F1EC': {
        'en' : u':Madagascar:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Madagascar:'],
        'de': u':flagge_madagaskar:',
        'es': u':bandera_madagascar:',
        'fr': u':drapeau_madagascar:',
        'pt': u':bandeira_madagascar:',
        'it': u':bandiera_madagascar:'
    },
    u'\U0001F1F2\U0001F1FC': {
        'en' : u':Malawi:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Malawi:'],
        'de': u':flagge_malawi:',
        'es': u':bandera_malaui:',
        'fr': u':drapeau_malawi:',
        'pt': u':bandeira_malaui:',
        'it': u':bandiera_malawi:'
    },
    u'\U0001F1F2\U0001F1FE': {
        'en' : u':Malaysia:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Malaysia:'],
        'de': u':flagge_malaysia:',
        'es': u':bandera_malasia:',
        'fr': u':drapeau_malaisie:',
        'pt': u':bandeira_malásia:',
        'it': u':bandiera_malaysia:'
    },
    u'\U0001F1F2\U0001F1FB': {
        'en' : u':Maldives:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Maldives:'],
        'de': u':flagge_malediven:',
        'es': u':bandera_maldivas:',
        'fr': u':drapeau_maldives:',
        'pt': u':bandeira_maldivas:',
        'it': u':bandiera_maldive:'
    },
    u'\U0001F1F2\U0001F1F1': {
        'en' : u':Mali:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Mali:'],
        'de': u':flagge_mali:',
        'es': u':bandera_mali:',
        'fr': u':drapeau_mali:',
        'pt': u':bandeira_mali:',
        'it': u':bandiera_mali:'
    },
    u'\U0001F1F2\U0001F1F9': {
        'en' : u':Malta:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Malta:'],
        'de': u':flagge_malta:',
        'es': u':bandera_malta:',
        'fr': u':drapeau_malte:',
        'pt': u':bandeira_malta:',
        'it': u':bandiera_malta:'
    },
    u'\U0001F1F2\U0001F1ED': {
        'en' : u':Marshall_Islands:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Marshall_Islands:'],
        'de': u':flagge_marshallinseln:',
        'es': u':bandera_islas_marshall:',
        'fr': u':drapeau_îles_marshall:',
        'pt': u':bandeira_ilhas_marshall:',
        'it': u':bandiera_isole_marshall:'
    },
    u'\U0001F1F2\U0001F1F6': {
        'en' : u':Martinique:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Martinique:'],
        'de': u':flagge_martinique:',
        'es': u':bandera_martinica:',
        'fr': u':drapeau_martinique:',
        'pt': u':bandeira_martinica:',
        'it': u':bandiera_martinica:'
    },
    u'\U0001F1F2\U0001F1F7': {
        'en' : u':Mauritania:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Mauritania:'],
        'de': u':flagge_mauretanien:',
        'es': u':bandera_mauritania:',
        'fr': u':drapeau_mauritanie:',
        'pt': u':bandeira_mauritânia:',
        'it': u':bandiera_mauritania:'
    },
    u'\U0001F1F2\U0001F1FA': {
        'en' : u':Mauritius:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Mauritius:'],
        'de': u':flagge_mauritius:',
        'es': u':bandera_mauricio:',
        'fr': u':drapeau_maurice:',
        'pt': u':bandeira_maurício:',
        'it': u':bandiera_mauritius:'
    },
    u'\U0001F1FE\U0001F1F9': {
        'en' : u':Mayotte:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Mayotte:'],
        'de': u':flagge_mayotte:',
        'es': u':bandera_mayotte:',
        'fr': u':drapeau_mayotte:',
        'pt': u':bandeira_mayotte:',
        'it': u':bandiera_mayotte:'
    },
    u'\U0001F1F2\U0001F1FD': {
        'en' : u':Mexico:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Mexico:'],
        'de': u':flagge_mexiko:',
        'es': u':bandera_méxico:',
        'fr': u':drapeau_mexique:',
        'pt': u':bandeira_méxico:',
        'it': u':bandiera_messico:'
    },
    u'\U0001F1EB\U0001F1F2': {
        'en' : u':Micronesia:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Micronesia:'],
        'de': u':flagge_mikronesien:',
        'es': u':bandera_micronesia:',
        'fr': u':drapeau_états_fédérés_de_micronésie:',
        'pt': u':bandeira_micronésia:',
        'it': u':bandiera_micronesia:'
    },
    u'\U0001F1F2\U0001F1E9': {
        'en' : u':Moldova:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Moldova:'],
        'de': u':flagge_republik_moldau:',
        'es': u':bandera_moldavia:',
        'fr': u':drapeau_moldavie:',
        'pt': u':bandeira_moldova:',
        'it': u':bandiera_moldavia:'
    },
    u'\U0001F1F2\U0001F1E8': {
        'en' : u':Monaco:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Monaco:'],
        'de': u':flagge_monaco:',
        'es': u':bandera_mónaco:',
        'fr': u':drapeau_monaco:',
        'pt': u':bandeira_mônaco:',
        'it': u':bandiera_monaco:'
    },
    u'\U0001F1F2\U0001F1F3': {
        'en' : u':Mongolia:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Mongolia:'],
        'de': u':flagge_mongolei:',
        'es': u':bandera_mongolia:',
        'fr': u':drapeau_mongolie:',
        'pt': u':bandeira_mongólia:',
        'it': u':bandiera_mongolia:'
    },
    u'\U0001F1F2\U0001F1EA': {
        'en' : u':Montenegro:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Montenegro:'],
        'de': u':flagge_montenegro:',
        'es': u':bandera_montenegro:',
        'fr': u':drapeau_monténégro:',
        'pt': u':bandeira_montenegro:',
        'it': u':bandiera_montenegro:'
    },
    u'\U0001F1F2\U0001F1F8': {
        'en' : u':Montserrat:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Montserrat:'],
        'de': u':flagge_montserrat:',
        'es': u':bandera_montserrat:',
        'fr': u':drapeau_montserrat:',
        'pt': u':bandeira_montserrat:',
        'it': u':bandiera_montserrat:'
    },
    u'\U0001F1F2\U0001F1E6': {
        'en' : u':Morocco:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Morocco:'],
        'de': u':flagge_marokko:',
        'es': u':bandera_marruecos:',
        'fr': u':drapeau_maroc:',
        'pt': u':bandeira_marrocos:',
        'it': u':bandiera_marocco:'
    },
    u'\U0001F1F2\U0001F1FF': {
        'en' : u':Mozambique:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Mozambique:'],
        'de': u':flagge_mosambik:',
        'es': u':bandera_mozambique:',
        'fr': u':drapeau_mozambique:',
        'pt': u':bandeira_moçambique:',
        'it': u':bandiera_mozambico:'
    },
    u'\U0001F936': {
        'en' : u':Mrs._Claus:',
        'status' : fully_qualified,
        'E' : 3,
        'alias' : [u':mrs_claus:'],
        'de': u':weihnachtsfrau:',
        'es': u':mamá_noel:',
        'fr': u':mère_noël:',
        'pt': u':mamãe_noel:',
        'it': u':mamma_natale:'
    },
    u'\U0001F936\U0001F3FF': {
        'en' : u':Mrs._Claus_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 3,
        'de': u':weihnachtsfrau_dunkle_hautfarbe:',
        'es': u':mamá_noel_tono_de_piel_oscuro:',
        'fr': u':mère_noël_peau_foncée:',
        'pt': u':mamãe_noel_pele_escura:',
        'it': u':mamma_natale_carnagione_scura:'
    },
    u'\U0001F936\U0001F3FB': {
        'en' : u':Mrs._Claus_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 3,
        'de': u':weihnachtsfrau_helle_hautfarbe:',
        'es': u':mamá_noel_tono_de_piel_claro:',
        'fr': u':mère_noël_peau_claire:',
        'pt': u':mamãe_noel_pele_clara:',
        'it': u':mamma_natale_carnagione_chiara:'
    },
    u'\U0001F936\U0001F3FE': {
        'en' : u':Mrs._Claus_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 3,
        'de': u':weihnachtsfrau_mitteldunkle_hautfarbe:',
        'es': u':mamá_noel_tono_de_piel_oscuro_medio:',
        'fr': u':mère_noël_peau_mate:',
        'pt': u':mamãe_noel_pele_morena_escura:',
        'it': u':mamma_natale_carnagione_abbastanza_scura:'
    },
    u'\U0001F936\U0001F3FC': {
        'en' : u':Mrs._Claus_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 3,
        'de': u':weihnachtsfrau_mittelhelle_hautfarbe:',
        'es': u':mamá_noel_tono_de_piel_claro_medio:',
        'fr': u':mère_noël_peau_moyennement_claire:',
        'pt': u':mamãe_noel_pele_morena_clara:',
        'it': u':mamma_natale_carnagione_abbastanza_chiara:'
    },
    u'\U0001F936\U0001F3FD': {
        'en' : u':Mrs._Claus_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 3,
        'de': u':weihnachtsfrau_mittlere_hautfarbe:',
        'es': u':mamá_noel_tono_de_piel_medio:',
        'fr': u':mère_noël_peau_légèrement_mate:',
        'pt': u':mamãe_noel_pele_morena:',
        'it': u':mamma_natale_carnagione_olivastra:'
    },
    u'\U0001F1F2\U0001F1F2': {
        'en' : u':Myanmar_(Burma):',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Myanmar:'],
        'de': u':flagge_myanmar:',
        'es': u':bandera_myanmar_(birmania):',
        'fr': u':drapeau_myanmar_(birmanie):',
        'pt': u':bandeira_mianmar_(birmânia):',
        'it': u':bandiera_myanmar_(birmania):'
    },
    u'\U0001F195': {
        'en' : u':NEW_button:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':new:'],
        'de': u':wort_new_in_blauem_quadrat:',
        'es': u':botón_new:',
        'fr': u':bouton_nouveau:',
        'pt': u':botão_new:',
        'it': u':pulsante_new:'
    },
    u'\U0001F196': {
        'en' : u':NG_button:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':ng:'],
        'de': u':großbuchstaben_ng_in_blauem_quadrat:',
        'es': u':botón_ng:',
        'fr': u':bouton_pas_bien:',
        'pt': u':botão_ng:',
        'it': u':pulsante_ng:'
    },
    u'\U0001F1F3\U0001F1E6': {
        'en' : u':Namibia:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Namibia:'],
        'de': u':flagge_namibia:',
        'es': u':bandera_namibia:',
        'fr': u':drapeau_namibie:',
        'pt': u':bandeira_namíbia:',
        'it': u':bandiera_namibia:'
    },
    u'\U0001F1F3\U0001F1F7': {
        'en' : u':Nauru:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Nauru:'],
        'de': u':flagge_nauru:',
        'es': u':bandera_nauru:',
        'fr': u':drapeau_nauru:',
        'pt': u':bandeira_nauru:',
        'it': u':bandiera_nauru:'
    },
    u'\U0001F1F3\U0001F1F5': {
        'en' : u':Nepal:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Nepal:'],
        'de': u':flagge_nepal:',
        'es': u':bandera_nepal:',
        'fr': u':drapeau_népal:',
        'pt': u':bandeira_nepal:',
        'it': u':bandiera_nepal:'
    },
    u'\U0001F1F3\U0001F1F1': {
        'en' : u':Netherlands:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Netherlands:'],
        'de': u':flagge_niederlande:',
        'es': u':bandera_países_bajos:',
        'fr': u':drapeau_pays-bas:',
        'pt': u':bandeira_países_baixos:',
        'it': u':bandiera_paesi_bassi:'
    },
    u'\U0001F1F3\U0001F1E8': {
        'en' : u':New_Caledonia:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_New_Caledonia:'],
        'de': u':flagge_neukaledonien:',
        'es': u':bandera_nueva_caledonia:',
        'fr': u':drapeau_nouvelle-calédonie:',
        'pt': u':bandeira_nova_caledônia:',
        'it': u':bandiera_nuova_caledonia:'
    },
    u'\U0001F1F3\U0001F1FF': {
        'en' : u':New_Zealand:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_New_Zealand:'],
        'de': u':flagge_neuseeland:',
        'es': u':bandera_nueva_zelanda:',
        'fr': u':drapeau_nouvelle-zélande:',
        'pt': u':bandeira_nova_zelândia:',
        'it': u':bandiera_nuova_zelanda:'
    },
    u'\U0001F1F3\U0001F1EE': {
        'en' : u':Nicaragua:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Nicaragua:'],
        'de': u':flagge_nicaragua:',
        'es': u':bandera_nicaragua:',
        'fr': u':drapeau_nicaragua:',
        'pt': u':bandeira_nicarágua:',
        'it': u':bandiera_nicaragua:'
    },
    u'\U0001F1F3\U0001F1EA': {
        'en' : u':Niger:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Niger:'],
        'de': u':flagge_niger:',
        'es': u':bandera_níger:',
        'fr': u':drapeau_niger:',
        'pt': u':bandeira_níger:',
        'it': u':bandiera_niger:'
    },
    u'\U0001F1F3\U0001F1EC': {
        'en' : u':Nigeria:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Nigeria:'],
        'de': u':flagge_nigeria:',
        'es': u':bandera_nigeria:',
        'fr': u':drapeau_nigéria:',
        'pt': u':bandeira_nigéria:',
        'it': u':bandiera_nigeria:'
    },
    u'\U0001F1F3\U0001F1FA': {
        'en' : u':Niue:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Niue:'],
        'de': u':flagge_niue:',
        'es': u':bandera_niue:',
        'fr': u':drapeau_niue:',
        'pt': u':bandeira_niue:',
        'it': u':bandiera_niue:'
    },
    u'\U0001F1F3\U0001F1EB': {
        'en' : u':Norfolk_Island:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Norfolk_Island:'],
        'de': u':flagge_norfolkinsel:',
        'es': u':bandera_isla_norfolk:',
        'fr': u':drapeau_île_norfolk:',
        'pt': u':bandeira_ilha_norfolk:',
        'it': u':bandiera_isola_norfolk:'
    },
    u'\U0001F1F0\U0001F1F5': {
        'en' : u':North_Korea:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_North_Korea:'],
        'de': u':flagge_nordkorea:',
        'es': u':bandera_corea_del_norte:',
        'fr': u':drapeau_corée_du_nord:',
        'pt': u':bandeira_coreia_do_norte:',
        'it': u':bandiera_corea_del_nord:'
    },
    u'\U0001F1F2\U0001F1F0': {
        'en' : u':North_Macedonia:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Macedonia:'],
        'de': u':flagge_nordmazedonien:',
        'es': u':bandera_macedonia_del_norte:',
        'fr': u':drapeau_macédoine_du_nord:',
        'pt': u':bandeira_macedônia_do_norte:',
        'it': u':bandiera_macedonia_del_nord:'
    },
    u'\U0001F1F2\U0001F1F5': {
        'en' : u':Northern_Mariana_Islands:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Northern_Mariana_Islands:'],
        'de': u':flagge_nördliche_marianen:',
        'es': u':bandera_islas_marianas_del_norte:',
        'fr': u':drapeau_îles_mariannes_du_nord:',
        'pt': u':bandeira_ilhas_marianas_do_norte:',
        'it': u':bandiera_isole_marianne_settentrionali:'
    },
    u'\U0001F1F3\U0001F1F4': {
        'en' : u':Norway:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Norway:'],
        'de': u':flagge_norwegen:',
        'es': u':bandera_noruega:',
        'fr': u':drapeau_norvège:',
        'pt': u':bandeira_noruega:',
        'it': u':bandiera_norvegia:'
    },
    u'\U0001F197': {
        'en' : u':OK_button:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':ok:'],
        'de': u':großbuchstaben_ok_in_blauem_quadrat:',
        'es': u':botón_ok:',
        'fr': u':bouton_ok:',
        'pt': u':botão_ok:',
        'it': u':pulsante_ok:'
    },
    u'\U0001F44C': {
        'en' : u':OK_hand:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':ok_hand:'],
        'de': u':ok-zeichen:',
        'es': u':señal_de_aprobación_con_la_mano:',
        'fr': u':ok:',
        'pt': u':sinal_de_ok:',
        'it': u':mano_che_fa_ok:'
    },
    u'\U0001F44C\U0001F3FF': {
        'en' : u':OK_hand_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':ok-zeichen_dunkle_hautfarbe:',
        'es': u':señal_de_aprobación_con_la_mano_tono_de_piel_oscuro:',
        'fr': u':ok_peau_foncée:',
        'pt': u':sinal_de_ok_pele_escura:',
        'it': u':mano_che_fa_ok_carnagione_scura:'
    },
    u'\U0001F44C\U0001F3FB': {
        'en' : u':OK_hand_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':ok-zeichen_helle_hautfarbe:',
        'es': u':señal_de_aprobación_con_la_mano_tono_de_piel_claro:',
        'fr': u':ok_peau_claire:',
        'pt': u':sinal_de_ok_pele_clara:',
        'it': u':mano_che_fa_ok_carnagione_chiara:'
    },
    u'\U0001F44C\U0001F3FE': {
        'en' : u':OK_hand_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':ok-zeichen_mitteldunkle_hautfarbe:',
        'es': u':señal_de_aprobación_con_la_mano_tono_de_piel_oscuro_medio:',
        'fr': u':ok_peau_mate:',
        'pt': u':sinal_de_ok_pele_morena_escura:',
        'it': u':mano_che_fa_ok_carnagione_abbastanza_scura:'
    },
    u'\U0001F44C\U0001F3FC': {
        'en' : u':OK_hand_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':ok-zeichen_mittelhelle_hautfarbe:',
        'es': u':señal_de_aprobación_con_la_mano_tono_de_piel_claro_medio:',
        'fr': u':ok_peau_moyennement_claire:',
        'pt': u':sinal_de_ok_pele_morena_clara:',
        'it': u':mano_che_fa_ok_carnagione_abbastanza_chiara:'
    },
    u'\U0001F44C\U0001F3FD': {
        'en' : u':OK_hand_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':ok-zeichen_mittlere_hautfarbe:',
        'es': u':señal_de_aprobación_con_la_mano_tono_de_piel_medio:',
        'fr': u':ok_peau_légèrement_mate:',
        'pt': u':sinal_de_ok_pele_morena:',
        'it': u':mano_che_fa_ok_carnagione_olivastra:'
    },
    u'\U0001F51B': {
        'en' : u':ON!_arrow:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':on:'],
        'de': u':on!-pfeil:',
        'es': u':flecha_on!:',
        'fr': u':flèche_activé:',
        'pt': u':seta_on!:',
        'it': u':freccia_on:'
    },
    u'\U0001F17E\U0000FE0F': {
        'en' : u':O_button_(blood_type):',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':o2:'],
        'variant': True,
        'de': u':großbuchstabe_o_in_rotem_quadrat:',
        'es': u':grupo_sanguíneo_tipo_o:',
        'fr': u':groupe_sanguin_o:',
        'pt': u':botão_o_(tipo_sanguíneo):',
        'it': u':gruppo_sanguigno_0:'
    },
    u'\U0001F17E': {
        'en' : u':O_button_(blood_type):',
        'status' : unqualified,
        'E' : 0.6,
        'alias' : [u':o2:'],
        'variant': True,
        'de': u':großbuchstabe_o_in_rotem_quadrat:',
        'es': u':grupo_sanguíneo_tipo_o:',
        'fr': u':groupe_sanguin_o:',
        'pt': u':botão_o_(tipo_sanguíneo):',
        'it': u':gruppo_sanguigno_0:'
    },
    u'\U0001F1F4\U0001F1F2': {
        'en' : u':Oman:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Oman:'],
        'de': u':flagge_oman:',
        'es': u':bandera_omán:',
        'fr': u':drapeau_oman:',
        'pt': u':bandeira_omã:',
        'it': u':bandiera_oman:'
    },
    u'\U000026CE': {
        'en' : u':Ophiuchus:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':ophiuchus:'],
        'de': u':schlangenträger:',
        'es': u':ofiuco:',
        'fr': u':serpentaire:',
        'pt': u':signo_de_ofiúco:',
        'it': u':segno_zodiacale_dell’ofiuco:'
    },
    u'\U0001F17F\U0000FE0F': {
        'en' : u':P_button:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':parking:'],
        'variant': True,
        'de': u':großbuchstabe_p_in_blauem_quadrat:',
        'es': u':aparcamiento:',
        'fr': u':bouton_p:',
        'pt': u':botão_p:',
        'it': u':pulsante_p:'
    },
    u'\U0001F17F': {
        'en' : u':P_button:',
        'status' : unqualified,
        'E' : 0.6,
        'alias' : [u':parking:'],
        'variant': True,
        'de': u':großbuchstabe_p_in_blauem_quadrat:',
        'es': u':aparcamiento:',
        'fr': u':bouton_p:',
        'pt': u':botão_p:',
        'it': u':pulsante_p:'
    },
    u'\U0001F1F5\U0001F1F0': {
        'en' : u':Pakistan:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Pakistan:'],
        'de': u':flagge_pakistan:',
        'es': u':bandera_pakistán:',
        'fr': u':drapeau_pakistan:',
        'pt': u':bandeira_paquistão:',
        'it': u':bandiera_pakistan:'
    },
    u'\U0001F1F5\U0001F1FC': {
        'en' : u':Palau:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Palau:'],
        'de': u':flagge_palau:',
        'es': u':bandera_palaos:',
        'fr': u':drapeau_palaos:',
        'pt': u':bandeira_palau:',
        'it': u':bandiera_palau:'
    },
    u'\U0001F1F5\U0001F1F8': {
        'en' : u':Palestinian_Territories:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Palestinian_Territories:'],
        'de': u':flagge_palästinensische_autonomiegebiete:',
        'es': u':bandera_territorios_palestinos:',
        'fr': u':drapeau_territoires_palestiniens:',
        'pt': u':bandeira_territórios_palestinos:',
        'it': u':bandiera_territori_palestinesi:'
    },
    u'\U0001F1F5\U0001F1E6': {
        'en' : u':Panama:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Panama:'],
        'de': u':flagge_panama:',
        'es': u':bandera_panamá:',
        'fr': u':drapeau_panama:',
        'pt': u':bandeira_panamá:',
        'it': u':bandiera_panamá:'
    },
    u'\U0001F1F5\U0001F1EC': {
        'en' : u':Papua_New_Guinea:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Papua_New_Guinea:'],
        'de': u':flagge_papua-neuguinea:',
        'es': u':bandera_papúa_nueva_guinea:',
        'fr': u':drapeau_papouasie-nouvelle-guinée:',
        'pt': u':bandeira_papua-nova_guiné:',
        'it': u':bandiera_papua_nuova_guinea:'
    },
    u'\U0001F1F5\U0001F1FE': {
        'en' : u':Paraguay:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Paraguay:'],
        'de': u':flagge_paraguay:',
        'es': u':bandera_paraguay:',
        'fr': u':drapeau_paraguay:',
        'pt': u':bandeira_paraguai:',
        'it': u':bandiera_paraguay:'
    },
    u'\U0001F1F5\U0001F1EA': {
        'en' : u':Peru:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Peru:'],
        'de': u':flagge_peru:',
        'es': u':bandera_perú:',
        'fr': u':drapeau_pérou:',
        'pt': u':bandeira_peru:',
        'it': u':bandiera_perù:'
    },
    u'\U0001F1F5\U0001F1ED': {
        'en' : u':Philippines:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Philippines:'],
        'de': u':flagge_philippinen:',
        'es': u':bandera_filipinas:',
        'fr': u':drapeau_philippines:',
        'pt': u':bandeira_filipinas:',
        'it': u':bandiera_filippine:'
    },
    u'\U00002653': {
        'en' : u':Pisces:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':pisces:'],
        'variant': True,
        'de': u':fische_(sternzeichen):',
        'es': u':piscis:',
        'fr': u':poissons:',
        'pt': u':signo_de_peixes:',
        'it': u':segno_zodiacale_dei_pesci:'
    },
    u'\U0001F1F5\U0001F1F3': {
        'en' : u':Pitcairn_Islands:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Pitcairn_Islands:'],
        'de': u':flagge_pitcairninseln:',
        'es': u':bandera_islas_pitcairn:',
        'fr': u':drapeau_îles_pitcairn:',
        'pt': u':bandeira_ilhas_pitcairn:',
        'it': u':bandiera_isole_pitcairn:'
    },
    u'\U0001F1F5\U0001F1F1': {
        'en' : u':Poland:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Poland:'],
        'de': u':flagge_polen:',
        'es': u':bandera_polonia:',
        'fr': u':drapeau_pologne:',
        'pt': u':bandeira_polônia:',
        'it': u':bandiera_polonia:'
    },
    u'\U0001F1F5\U0001F1F9': {
        'en' : u':Portugal:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Portugal:'],
        'de': u':flagge_portugal:',
        'es': u':bandera_portugal:',
        'fr': u':drapeau_portugal:',
        'pt': u':bandeira_portugal:',
        'it': u':bandiera_portogallo:'
    },
    u'\U0001F1F5\U0001F1F7': {
        'en' : u':Puerto_Rico:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Puerto_Rico:'],
        'de': u':flagge_puerto_rico:',
        'es': u':bandera_puerto_rico:',
        'fr': u':drapeau_porto_rico:',
        'pt': u':bandeira_porto_rico:',
        'it': u':bandiera_portorico:'
    },
    u'\U0001F1F6\U0001F1E6': {
        'en' : u':Qatar:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Qatar:'],
        'de': u':flagge_katar:',
        'es': u':bandera_catar:',
        'fr': u':drapeau_qatar:',
        'pt': u':bandeira_catar:',
        'it': u':bandiera_qatar:'
    },
    u'\U0001F1F7\U0001F1F4': {
        'en' : u':Romania:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Romania:'],
        'de': u':flagge_rumänien:',
        'es': u':bandera_rumanía:',
        'fr': u':drapeau_roumanie:',
        'pt': u':bandeira_romênia:',
        'it': u':bandiera_romania:'
    },
    u'\U0001F1F7\U0001F1FA': {
        'en' : u':Russia:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':flag_for_Russia:'],
        'de': u':flagge_russland:',
        'es': u':bandera_rusia:',
        'fr': u':drapeau_russie:',
        'pt': u':bandeira_rússia:',
        'it': u':bandiera_russia:'
    },
    u'\U0001F1F7\U0001F1FC': {
        'en' : u':Rwanda:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Rwanda:'],
        'de': u':flagge_ruanda:',
        'es': u':bandera_ruanda:',
        'fr': u':drapeau_rwanda:',
        'pt': u':bandeira_ruanda:',
        'it': u':bandiera_ruanda:'
    },
    u'\U0001F1F7\U0001F1EA': {
        'en' : u':Réunion:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Réunion:'],
        'de': u':flagge_réunion:',
        'es': u':bandera_reunión:',
        'fr': u':drapeau_la_réunion:',
        'pt': u':bandeira_reunião:',
        'it': u':bandiera_riunione:'
    },
    u'\U0001F51C': {
        'en' : u':SOON_arrow:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':soon:'],
        'de': u':soon-pfeil:',
        'es': u':flecha_soon:',
        'fr': u':flèche_bientôt:',
        'pt': u':seta_soon:',
        'it': u':freccia_soon:'
    },
    u'\U0001F198': {
        'en' : u':SOS_button:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':sos:'],
        'de': u':sos-zeichen:',
        'es': u':símbolo_de_socorro:',
        'fr': u':bouton_sos:',
        'pt': u':botão_sos:',
        'it': u':pulsante_sos:'
    },
    u'\U00002650': {
        'en' : u':Sagittarius:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':sagittarius:'],
        'variant': True,
        'de': u':schütze_(sternzeichen):',
        'es': u':sagitario:',
        'fr': u':sagittaire:',
        'pt': u':signo_de_sagitário:',
        'it': u':segno_zodiacale_del_saggitario:'
    },
    u'\U0001F1FC\U0001F1F8': {
        'en' : u':Samoa:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Samoa:'],
        'de': u':flagge_samoa:',
        'es': u':bandera_samoa:',
        'fr': u':drapeau_samoa:',
        'pt': u':bandeira_samoa:',
        'it': u':bandiera_samoa:'
    },
    u'\U0001F1F8\U0001F1F2': {
        'en' : u':San_Marino:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_San_Marino:'],
        'de': u':flagge_san_marino:',
        'es': u':bandera_san_marino:',
        'fr': u':drapeau_saint-marin:',
        'pt': u':bandeira_san_marino:',
        'it': u':bandiera_san_marino:'
    },
    u'\U0001F385': {
        'en' : u':Santa_Claus:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':santa:'],
        'de': u':weihnachtsmann:',
        'es': u':papá_noel:',
        'fr': u':père_noël:',
        'pt': u':papai_noel:',
        'it': u':babbo_natale:'
    },
    u'\U0001F385\U0001F3FF': {
        'en' : u':Santa_Claus_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':weihnachtsmann_dunkle_hautfarbe:',
        'es': u':papá_noel_tono_de_piel_oscuro:',
        'fr': u':père_noël_peau_foncée:',
        'pt': u':papai_noel_pele_escura:',
        'it': u':babbo_natale_carnagione_scura:'
    },
    u'\U0001F385\U0001F3FB': {
        'en' : u':Santa_Claus_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':weihnachtsmann_helle_hautfarbe:',
        'es': u':papá_noel_tono_de_piel_claro:',
        'fr': u':père_noël_peau_claire:',
        'pt': u':papai_noel_pele_clara:',
        'it': u':babbo_natale_carnagione_chiara:'
    },
    u'\U0001F385\U0001F3FE': {
        'en' : u':Santa_Claus_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':weihnachtsmann_mitteldunkle_hautfarbe:',
        'es': u':papá_noel_tono_de_piel_oscuro_medio:',
        'fr': u':père_noël_peau_mate:',
        'pt': u':papai_noel_pele_morena_escura:',
        'it': u':babbo_natale_carnagione_abbastanza_scura:'
    },
    u'\U0001F385\U0001F3FC': {
        'en' : u':Santa_Claus_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':weihnachtsmann_mittelhelle_hautfarbe:',
        'es': u':papá_noel_tono_de_piel_claro_medio:',
        'fr': u':père_noël_peau_moyennement_claire:',
        'pt': u':papai_noel_pele_morena_clara:',
        'it': u':babbo_natale_carnagione_abbastanza_chiara:'
    },
    u'\U0001F385\U0001F3FD': {
        'en' : u':Santa_Claus_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':weihnachtsmann_mittlere_hautfarbe:',
        'es': u':papá_noel_tono_de_piel_medio:',
        'fr': u':père_noël_peau_légèrement_mate:',
        'pt': u':papai_noel_pele_morena:',
        'it': u':babbo_natale_carnagione_olivastra:'
    },
    u'\U0001F1F8\U0001F1E6': {
        'en' : u':Saudi_Arabia:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Saudi_Arabia:'],
        'de': u':flagge_saudi-arabien:',
        'es': u':bandera_arabia_saudí:',
        'fr': u':drapeau_arabie_saoudite:',
        'pt': u':bandeira_arábia_saudita:',
        'it': u':bandiera_arabia_saudita:'
    },
    u'\U0000264F': {
        'en' : u':Scorpio:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':scorpius:'],
        'variant': True,
        'de': u':skorpion_(sternzeichen):',
        'es': u':escorpio:',
        'fr': u':scorpion_zodiaque:',
        'pt': u':signo_de_escorpião:',
        'it': u':segno_zodiacale_dello_scorpione:'
    },
    u'\U0001F3F4\U000E0067\U000E0062\U000E0073\U000E0063\U000E0074\U000E007F': {
        'en' : u':Scotland:',
        'status' : fully_qualified,
        'E' : 5,
        'alias' : [u':scotland:'],
        'de': u':flagge_schottland:',
        'es': u':bandera_escocia:',
        'fr': u':drapeau_écosse:',
        'pt': u':bandeira_escócia:',
        'it': u':bandiera_scozia:'
    },
    u'\U0001F1F8\U0001F1F3': {
        'en' : u':Senegal:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Senegal:'],
        'de': u':flagge_senegal:',
        'es': u':bandera_senegal:',
        'fr': u':drapeau_sénégal:',
        'pt': u':bandeira_senegal:',
        'it': u':bandiera_senegal:'
    },
    u'\U0001F1F7\U0001F1F8': {
        'en' : u':Serbia:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Serbia:'],
        'de': u':flagge_serbien:',
        'es': u':bandera_serbia:',
        'fr': u':drapeau_serbie:',
        'pt': u':bandeira_sérvia:',
        'it': u':bandiera_serbia:'
    },
    u'\U0001F1F8\U0001F1E8': {
        'en' : u':Seychelles:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Seychelles:'],
        'de': u':flagge_seychellen:',
        'es': u':bandera_seychelles:',
        'fr': u':drapeau_seychelles:',
        'pt': u':bandeira_seicheles:',
        'it': u':bandiera_seychelles:'
    },
    u'\U0001F1F8\U0001F1F1': {
        'en' : u':Sierra_Leone:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Sierra_Leone:'],
        'de': u':flagge_sierra_leone:',
        'es': u':bandera_sierra_leona:',
        'fr': u':drapeau_sierra_leone:',
        'pt': u':bandeira_serra_leoa:',
        'it': u':bandiera_sierra_leone:'
    },
    u'\U0001F1F8\U0001F1EC': {
        'en' : u':Singapore:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Singapore:'],
        'de': u':flagge_singapur:',
        'es': u':bandera_singapur:',
        'fr': u':drapeau_singapour:',
        'pt': u':bandeira_singapura:',
        'it': u':bandiera_singapore:'
    },
    u'\U0001F1F8\U0001F1FD': {
        'en' : u':Sint_Maarten:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Sint_Maarten:'],
        'de': u':flagge_sint_maarten:',
        'es': u':bandera_sint_maarten:',
        'fr': u':drapeau_saint-martin_(partie_néerlandaise):',
        'pt': u':bandeira_sint_maarten:',
        'it': u':bandiera_sint_maarten:'
    },
    u'\U0001F1F8\U0001F1F0': {
        'en' : u':Slovakia:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Slovakia:'],
        'de': u':flagge_slowakei:',
        'es': u':bandera_eslovaquia:',
        'fr': u':drapeau_slovaquie:',
        'pt': u':bandeira_eslováquia:',
        'it': u':bandiera_slovacchia:'
    },
    u'\U0001F1F8\U0001F1EE': {
        'en' : u':Slovenia:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Slovenia:'],
        'de': u':flagge_slowenien:',
        'es': u':bandera_eslovenia:',
        'fr': u':drapeau_slovénie:',
        'pt': u':bandeira_eslovênia:',
        'it': u':bandiera_slovenia:'
    },
    u'\U0001F1F8\U0001F1E7': {
        'en' : u':Solomon_Islands:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Solomon_Islands:'],
        'de': u':flagge_salomonen:',
        'es': u':bandera_islas_salomón:',
        'fr': u':drapeau_îles_salomon:',
        'pt': u':bandeira_ilhas_salomão:',
        'it': u':bandiera_isole_salomone:'
    },
    u'\U0001F1F8\U0001F1F4': {
        'en' : u':Somalia:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Somalia:'],
        'de': u':flagge_somalia:',
        'es': u':bandera_somalia:',
        'fr': u':drapeau_somalie:',
        'pt': u':bandeira_somália:',
        'it': u':bandiera_somalia:'
    },
    u'\U0001F1FF\U0001F1E6': {
        'en' : u':South_Africa:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_South_Africa:'],
        'de': u':flagge_südafrika:',
        'es': u':bandera_sudáfrica:',
        'fr': u':drapeau_afrique_du_sud:',
        'pt': u':bandeira_áfrica_do_sul:',
        'it': u':bandiera_sudafrica:'
    },
    u'\U0001F1EC\U0001F1F8': {
        'en' : u':South_Georgia_&_South_Sandwich_Islands:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_South_Georgia_&_South_Sandwich_Islands:'],
        'de': u':flagge_südgeorgien_und_die_südlichen_sandwichinseln:',
        'es': u':bandera_islas_georgia_del_sur_y_sandwich_del_sur:',
        'fr': u':drapeau_géorgie_du_sud_et_îles_sandwich_du_sud:',
        'pt': u':bandeira_ilhas_geórgia_do_sul_e_sandwich_do_sul:',
        'it': u':bandiera_georgia_del_sud_e_sandwich_australi:'
    },
    u'\U0001F1F0\U0001F1F7': {
        'en' : u':South_Korea:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':flag_for_South_Korea:'],
        'de': u':flagge_südkorea:',
        'es': u':bandera_corea_del_sur:',
        'fr': u':drapeau_corée_du_sud:',
        'pt': u':bandeira_coreia_do_sul:',
        'it': u':bandiera_corea_del_sud:'
    },
    u'\U0001F1F8\U0001F1F8': {
        'en' : u':South_Sudan:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_South_Sudan:'],
        'de': u':flagge_südsudan:',
        'es': u':bandera_sudán_del_sur:',
        'fr': u':drapeau_soudan_du_sud:',
        'pt': u':bandeira_sudão_do_sul:',
        'it': u':bandiera_sud_sudan:'
    },
    u'\U0001F1EA\U0001F1F8': {
        'en' : u':Spain:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':flag_for_Spain:'],
        'de': u':flagge_spanien:',
        'es': u':bandera_españa:',
        'fr': u':drapeau_espagne:',
        'pt': u':bandeira_espanha:',
        'it': u':bandiera_spagna:'
    },
    u'\U0001F1F1\U0001F1F0': {
        'en' : u':Sri_Lanka:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Sri_Lanka:'],
        'de': u':flagge_sri_lanka:',
        'es': u':bandera_sri_lanka:',
        'fr': u':drapeau_sri_lanka:',
        'pt': u':bandeira_sri_lanka:',
        'it': u':bandiera_sri_lanka:'
    },
    u'\U0001F1E7\U0001F1F1': {
        'en' : u':St._Barthélemy:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_St._Barthélemy:'],
        'de': u':flagge_st._barthélemy:',
        'es': u':bandera_san_bartolomé:',
        'fr': u':drapeau_saint-barthélemy:',
        'pt': u':bandeira_são_bartolomeu:',
        'it': u':bandiera_saint-barthélemy:'
    },
    u'\U0001F1F8\U0001F1ED': {
        'en' : u':St._Helena:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_St._Helena:'],
        'de': u':flagge_st._helena:',
        'es': u':bandera_santa_elena:',
        'fr': u':drapeau_sainte-hélène:',
        'pt': u':bandeira_santa_helena:',
        'it': u':bandiera_sant’elena:'
    },
    u'\U0001F1F0\U0001F1F3': {
        'en' : u':St._Kitts_&_Nevis:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_St._Kitts_&_Nevis:'],
        'de': u':flagge_st._kitts_und_nevis:',
        'es': u':bandera_san_cristóbal_y_nieves:',
        'fr': u':drapeau_saint-christophe-et-niévès:',
        'pt': u':bandeira_são_cristóvão_e_névis:',
        'it': u':bandiera_saint_kitts_e_nevis:'
    },
    u'\U0001F1F1\U0001F1E8': {
        'en' : u':St._Lucia:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_St._Lucia:'],
        'de': u':flagge_st._lucia:',
        'es': u':bandera_santa_lucía:',
        'fr': u':drapeau_sainte-lucie:',
        'pt': u':bandeira_santa_lúcia:',
        'it': u':bandiera_saint_lucia:'
    },
    u'\U0001F1F2\U0001F1EB': {
        'en' : u':St._Martin:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_St._Martin:'],
        'de': u':flagge_st._martin:',
        'es': u':bandera_san_martín:',
        'fr': u':drapeau_saint-martin:',
        'pt': u':bandeira_são_martinho:',
        'it': u':bandiera_saint_martin:'
    },
    u'\U0001F1F5\U0001F1F2': {
        'en' : u':St._Pierre_&_Miquelon:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_St._Pierre_&_Miquelon:'],
        'de': u':flagge_st._pierre_und_miquelon:',
        'es': u':bandera_san_pedro_y_miquelón:',
        'fr': u':drapeau_saint-pierre-et-miquelon:',
        'pt': u':bandeira_são_pedro_e_miquelão:',
        'it': u':bandiera_saint-pierre_e_miquelon:'
    },
    u'\U0001F1FB\U0001F1E8': {
        'en' : u':St._Vincent_&_Grenadines:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_St._Vincent_&_Grenadines:'],
        'de': u':flagge_st._vincent_und_die_grenadinen:',
        'es': u':bandera_san_vicente_y_las_granadinas:',
        'fr': u':drapeau_saint-vincent-et-les-grenadines:',
        'pt': u':bandeira_são_vicente_e_granadinas:',
        'it': u':bandiera_saint_vincent_e_grenadine:'
    },
    u'\U0001F5FD': {
        'en' : u':Statue_of_Liberty:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':statue_of_liberty:'],
        'de': u':freiheitsstatue:',
        'es': u':estatua_de_la_libertad:',
        'fr': u':statue_de_la_liberté:',
        'pt': u':estátua_da_liberdade:',
        'it': u':statua_della_libertà:'
    },
    u'\U0001F1F8\U0001F1E9': {
        'en' : u':Sudan:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Sudan:'],
        'de': u':flagge_sudan:',
        'es': u':bandera_sudán:',
        'fr': u':drapeau_soudan:',
        'pt': u':bandeira_sudão:',
        'it': u':bandiera_sudan:'
    },
    u'\U0001F1F8\U0001F1F7': {
        'en' : u':Suriname:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Suriname:'],
        'de': u':flagge_suriname:',
        'es': u':bandera_surinam:',
        'fr': u':drapeau_suriname:',
        'pt': u':bandeira_suriname:',
        'it': u':bandiera_suriname:'
    },
    u'\U0001F1F8\U0001F1EF': {
        'en' : u':Svalbard_&_Jan_Mayen:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Svalbard_&_Jan_Mayen:'],
        'de': u':flagge_spitzbergen_und_jan_mayen:',
        'es': u':bandera_svalbard_y_jan_mayen:',
        'fr': u':drapeau_svalbard_et_jan_mayen:',
        'pt': u':bandeira_svalbard_e_jan_mayen:',
        'it': u':bandiera_svalbard_e_jan_mayen:'
    },
    u'\U0001F1F8\U0001F1EA': {
        'en' : u':Sweden:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Sweden:'],
        'de': u':flagge_schweden:',
        'es': u':bandera_suecia:',
        'fr': u':drapeau_suède:',
        'pt': u':bandeira_suécia:',
        'it': u':bandiera_svezia:'
    },
    u'\U0001F1E8\U0001F1ED': {
        'en' : u':Switzerland:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Switzerland:'],
        'de': u':flagge_schweiz:',
        'es': u':bandera_suiza:',
        'fr': u':drapeau_suisse:',
        'pt': u':bandeira_suíça:',
        'it': u':bandiera_svizzera:'
    },
    u'\U0001F1F8\U0001F1FE': {
        'en' : u':Syria:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Syria:'],
        'de': u':flagge_syrien:',
        'es': u':bandera_siria:',
        'fr': u':drapeau_syrie:',
        'pt': u':bandeira_síria:',
        'it': u':bandiera_siria:'
    },
    u'\U0001F1F8\U0001F1F9': {
        'en' : u':São_Tomé_&_Príncipe:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_São_Tomé_&_Príncipe:'],
        'de': u':flagge_são_tomé_und_príncipe:',
        'es': u':bandera_santo_tomé_y_príncipe:',
        'fr': u':drapeau_sao_tomé-et-principe:',
        'pt': u':bandeira_são_tomé_e_príncipe:',
        'it': u':bandiera_são_tomé_e_príncipe:'
    },
    u'\U0001F996': {
        'en' : u':T-Rex:',
        'status' : fully_qualified,
        'E' : 5,
        'alias' : [u':t-rex:'],
        'de': u':t-rex:',
        'es': u':t-rex:',
        'fr': u':t-rex:',
        'pt': u':tiranossauro_rex:',
        'it': u':t-rex:'
    },
    u'\U0001F51D': {
        'en' : u':TOP_arrow:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':top:'],
        'de': u':top-pfeil:',
        'es': u':flecha_top:',
        'fr': u':flèche_en_haut:',
        'pt': u':seta_top:',
        'it': u':freccia_top:'
    },
    u'\U0001F1F9\U0001F1FC': {
        'en' : u':Taiwan:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Taiwan:'],
        'de': u':flagge_taiwan:',
        'es': u':bandera_taiwán:',
        'fr': u':drapeau_taïwan:',
        'pt': u':bandeira_taiwan:',
        'it': u':bandiera_taiwan:'
    },
    u'\U0001F1F9\U0001F1EF': {
        'en' : u':Tajikistan:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Tajikistan:'],
        'de': u':flagge_tadschikistan:',
        'es': u':bandera_tayikistán:',
        'fr': u':drapeau_tadjikistan:',
        'pt': u':bandeira_tadjiquistão:',
        'it': u':bandiera_tagikistan:'
    },
    u'\U0001F1F9\U0001F1FF': {
        'en' : u':Tanzania:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Tanzania:'],
        'de': u':flagge_tansania:',
        'es': u':bandera_tanzania:',
        'fr': u':drapeau_tanzanie:',
        'pt': u':bandeira_tanzânia:',
        'it': u':bandiera_tanzania:'
    },
    u'\U00002649': {
        'en' : u':Taurus:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':taurus:'],
        'variant': True,
        'de': u':stier_(sternzeichen):',
        'es': u':tauro:',
        'fr': u':taureau:',
        'pt': u':signo_de_touro:',
        'it': u':segno_zodiacale_del_toro:'
    },
    u'\U0001F1F9\U0001F1ED': {
        'en' : u':Thailand:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Thailand:'],
        'de': u':flagge_thailand:',
        'es': u':bandera_tailandia:',
        'fr': u':drapeau_thaïlande:',
        'pt': u':bandeira_tailândia:',
        'it': u':bandiera_thailandia:'
    },
    u'\U0001F1F9\U0001F1F1': {
        'en' : u':Timor-Leste:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Timor__Leste:'],
        'de': u':flagge_timor-leste:',
        'es': u':bandera_timor-leste:',
        'fr': u':drapeau_timor_oriental:',
        'pt': u':bandeira_timor-leste:',
        'it': u':bandiera_timor_est:'
    },
    u'\U0001F1F9\U0001F1EC': {
        'en' : u':Togo:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Togo:'],
        'de': u':flagge_togo:',
        'es': u':bandera_togo:',
        'fr': u':drapeau_togo:',
        'pt': u':bandeira_togo:',
        'it': u':bandiera_togo:'
    },
    u'\U0001F1F9\U0001F1F0': {
        'en' : u':Tokelau:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Tokelau:'],
        'de': u':flagge_tokelau:',
        'es': u':bandera_tokelau:',
        'fr': u':drapeau_tokelau:',
        'pt': u':bandeira_tokelau:',
        'it': u':bandiera_tokelau:'
    },
    u'\U0001F5FC': {
        'en' : u':Tokyo_tower:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':tokyo_tower:'],
        'de': u':tokyo_tower:',
        'es': u':torre_de_tokio:',
        'fr': u':tour_de_tokyo:',
        'pt': u':torre_de_tóquio:',
        'it': u':torre_di_tokyo:'
    },
    u'\U0001F1F9\U0001F1F4': {
        'en' : u':Tonga:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Tonga:'],
        'de': u':flagge_tonga:',
        'es': u':bandera_tonga:',
        'fr': u':drapeau_tonga:',
        'pt': u':bandeira_tonga:',
        'it': u':bandiera_tonga:'
    },
    u'\U0001F1F9\U0001F1F9': {
        'en' : u':Trinidad_&_Tobago:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Trinidad_&_Tobago:'],
        'de': u':flagge_trinidad_und_tobago:',
        'es': u':bandera_trinidad_y_tobago:',
        'fr': u':drapeau_trinité-et-tobago:',
        'pt': u':bandeira_trinidad_e_tobago:',
        'it': u':bandiera_trinidad_e_tobago:'
    },
    u'\U0001F1F9\U0001F1E6': {
        'en' : u':Tristan_da_Cunha:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Tristan_da_Cunha:'],
        'de': u':flagge_tristan_da_cunha:',
        'es': u':bandera_tristán_de_acuña:',
        'fr': u':drapeau_tristan_da_cunha:',
        'pt': u':bandeira_tristão_da_cunha:',
        'it': u':bandiera_tristan_da_cunha:'
    },
    u'\U0001F1F9\U0001F1F3': {
        'en' : u':Tunisia:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Tunisia:'],
        'de': u':flagge_tunesien:',
        'es': u':bandera_túnez:',
        'fr': u':drapeau_tunisie:',
        'pt': u':bandeira_tunísia:',
        'it': u':bandiera_tunisia:'
    },
    u'\U0001F1F9\U0001F1F7': {
        'en' : u':Turkey:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Turkey:'],
        'de': u':flagge_türkei:',
        'es': u':bandera_turquía:',
        'fr': u':drapeau_turquie:',
        'pt': u':bandeira_turquia:',
        'it': u':bandiera_turchia:'
    },
    u'\U0001F1F9\U0001F1F2': {
        'en' : u':Turkmenistan:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Turkmenistan:'],
        'de': u':flagge_turkmenistan:',
        'es': u':bandera_turkmenistán:',
        'fr': u':drapeau_turkménistan:',
        'pt': u':bandeira_turcomenistão:',
        'it': u':bandiera_turkmenistan:'
    },
    u'\U0001F1F9\U0001F1E8': {
        'en' : u':Turks_&_Caicos_Islands:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Turks_&_Caicos_Islands:'],
        'de': u':flagge_turks-_und_caicosinseln:',
        'es': u':bandera_islas_turcas_y_caicos:',
        'fr': u':drapeau_îles_turques-et-caïques:',
        'pt': u':bandeira_ilhas_turcas_e_caicos:',
        'it': u':bandiera_isole_turks_e_caicos:'
    },
    u'\U0001F1F9\U0001F1FB': {
        'en' : u':Tuvalu:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Tuvalu:'],
        'de': u':flagge_tuvalu:',
        'es': u':bandera_tuvalu:',
        'fr': u':drapeau_tuvalu:',
        'pt': u':bandeira_tuvalu:',
        'it': u':bandiera_tuvalu:'
    },
    u'\U0001F1FA\U0001F1F2': {
        'en' : u':U.S._Outlying_Islands:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_U.S._Outlying_Islands:'],
        'de': u':flagge_amerikanische_überseeinseln:',
        'es': u':bandera_islas_menores_alejadas_de_ee._uu.:',
        'fr': u':drapeau_îles_mineures_éloignées_des_états-unis:',
        'pt': u':bandeira_ilhas_menores_distantes_dos_eua:',
        'it': u':bandiera_altre_isole_americane_del_pacifico:'
    },
    u'\U0001F1FB\U0001F1EE': {
        'en' : u':U.S._Virgin_Islands:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_U.S._Virgin_Islands:'],
        'de': u':flagge_amerikanische_jungferninseln:',
        'es': u':bandera_islas_vírgenes_de_ee._uu.:',
        'fr': u':drapeau_îles_vierges_des_états-unis:',
        'pt': u':bandeira_ilhas_virgens_americanas:',
        'it': u':bandiera_isole_vergini_americane:'
    },
    u'\U0001F199': {
        'en' : u':UP!_button:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':up:'],
        'de': u':schriftzug_up!_im_blauen_quadrat:',
        'es': u':botón_up!:',
        'fr': u':bouton_vers_le_haut:',
        'pt': u':botão_up!:',
        'it': u':pulsante_up!:'
    },
    u'\U0001F1FA\U0001F1EC': {
        'en' : u':Uganda:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Uganda:'],
        'de': u':flagge_uganda:',
        'es': u':bandera_uganda:',
        'fr': u':drapeau_ouganda:',
        'pt': u':bandeira_uganda:',
        'it': u':bandiera_uganda:'
    },
    u'\U0001F1FA\U0001F1E6': {
        'en' : u':Ukraine:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Ukraine:'],
        'de': u':flagge_ukraine:',
        'es': u':bandera_ucrania:',
        'fr': u':drapeau_ukraine:',
        'pt': u':bandeira_ucrânia:',
        'it': u':bandiera_ucraina:'
    },
    u'\U0001F1E6\U0001F1EA': {
        'en' : u':United_Arab_Emirates:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_United_Arab_Emirates:'],
        'de': u':flagge_vereinigte_arabische_emirate:',
        'es': u':bandera_emiratos_árabes_unidos:',
        'fr': u':drapeau_émirats_arabes_unis:',
        'pt': u':bandeira_emirados_árabes_unidos:',
        'it': u':bandiera_emirati_arabi_uniti:'
    },
    u'\U0001F1EC\U0001F1E7': {
        'en' : u':United_Kingdom:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':flag_for_United_Kingdom:'],
        'de': u':flagge_vereinigtes_königreich:',
        'es': u':bandera_reino_unido:',
        'fr': u':drapeau_royaume-uni:',
        'pt': u':bandeira_reino_unido:',
        'it': u':bandiera_regno_unito:'
    },
    u'\U0001F1FA\U0001F1F3': {
        'en' : u':United_Nations:',
        'status' : fully_qualified,
        'E' : 4,
        'alias' : [u':united_nations:'],
        'de': u':flagge_vereinte_nationen:',
        'es': u':bandera_naciones_unidas:',
        'fr': u':drapeau_nations_unies:',
        'pt': u':bandeira_nações_unidas:',
        'it': u':bandiera_nazioni_unite:'
    },
    u'\U0001F1FA\U0001F1F8': {
        'en' : u':United_States:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':flag_for_United_States:'],
        'de': u':flagge_vereinigte_staaten:',
        'es': u':bandera_estados_unidos:',
        'fr': u':drapeau_états-unis:',
        'pt': u':bandeira_estados_unidos:',
        'it': u':bandiera_stati_uniti:'
    },
    u'\U0001F1FA\U0001F1FE': {
        'en' : u':Uruguay:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Uruguay:'],
        'de': u':flagge_uruguay:',
        'es': u':bandera_uruguay:',
        'fr': u':drapeau_uruguay:',
        'pt': u':bandeira_uruguai:',
        'it': u':bandiera_uruguay:'
    },
    u'\U0001F1FA\U0001F1FF': {
        'en' : u':Uzbekistan:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Uzbekistan:'],
        'de': u':flagge_usbekistan:',
        'es': u':bandera_uzbekistán:',
        'fr': u':drapeau_ouzbékistan:',
        'pt': u':bandeira_uzbequistão:',
        'it': u':bandiera_uzbekistan:'
    },
    u'\U0001F19A': {
        'en' : u':VS_button:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':vs:'],
        'de': u':großbuchstaben_vs_in_orangefarbenem_quadrat:',
        'es': u':botón_vs:',
        'fr': u':bouton_vs:',
        'pt': u':botão_vs:',
        'it': u':pulsante_vs:'
    },
    u'\U0001F1FB\U0001F1FA': {
        'en' : u':Vanuatu:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Vanuatu:'],
        'de': u':flagge_vanuatu:',
        'es': u':bandera_vanuatu:',
        'fr': u':drapeau_vanuatu:',
        'pt': u':bandeira_vanuatu:',
        'it': u':bandiera_vanuatu:'
    },
    u'\U0001F1FB\U0001F1E6': {
        'en' : u':Vatican_City:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Vatican_City:'],
        'de': u':flagge_vatikanstadt:',
        'es': u':bandera_ciudad_del_vaticano:',
        'fr': u':drapeau_état_de_la_cité_du_vatican:',
        'pt': u':bandeira_cidade_do_vaticano:',
        'it': u':bandiera_città_del_vaticano:'
    },
    u'\U0001F1FB\U0001F1EA': {
        'en' : u':Venezuela:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Venezuela:'],
        'de': u':flagge_venezuela:',
        'es': u':bandera_venezuela:',
        'fr': u':drapeau_venezuela:',
        'pt': u':bandeira_venezuela:',
        'it': u':bandiera_venezuela:'
    },
    u'\U0001F1FB\U0001F1F3': {
        'en' : u':Vietnam:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Vietnam:'],
        'de': u':flagge_vietnam:',
        'es': u':bandera_vietnam:',
        'fr': u':drapeau_vietnam:',
        'pt': u':bandeira_vietnã:',
        'it': u':bandiera_vietnam:'
    },
    u'\U0000264D': {
        'en' : u':Virgo:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':virgo:'],
        'variant': True,
        'de': u':jungfrau_(sternzeichen):',
        'es': u':virgo:',
        'fr': u':vierge:',
        'pt': u':signo_de_virgem:',
        'it': u':segno_zodiacale_della_vergine:'
    },
    u'\U0001F3F4\U000E0067\U000E0062\U000E0077\U000E006C\U000E0073\U000E007F': {
        'en' : u':Wales:',
        'status' : fully_qualified,
        'E' : 5,
        'alias' : [u':wales:'],
        'de': u':flagge_wales:',
        'es': u':bandera_gales:',
        'fr': u':drapeau_pays_de_galles:',
        'pt': u':bandeira_país_de_gales:',
        'it': u':bandiera_galles:'
    },
    u'\U0001F1FC\U0001F1EB': {
        'en' : u':Wallis_&_Futuna:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Wallis_&_Futuna:'],
        'de': u':flagge_wallis_und_futuna:',
        'es': u':bandera_wallis_y_futuna:',
        'fr': u':drapeau_wallis-et-futuna:',
        'pt': u':bandeira_wallis_e_futuna:',
        'it': u':bandiera_wallis_e_futuna:'
    },
    u'\U0001F1EA\U0001F1ED': {
        'en' : u':Western_Sahara:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Western_Sahara:'],
        'de': u':flagge_westsahara:',
        'es': u':bandera_sáhara_occidental:',
        'fr': u':drapeau_sahara_occidental:',
        'pt': u':bandeira_saara_ocidental:',
        'it': u':bandiera_sahara_occidentale:'
    },
    u'\U0001F1FE\U0001F1EA': {
        'en' : u':Yemen:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Yemen:'],
        'de': u':flagge_jemen:',
        'es': u':bandera_yemen:',
        'fr': u':drapeau_yémen:',
        'pt': u':bandeira_iêmen:',
        'it': u':bandiera_yemen:'
    },
    u'\U0001F1FF\U0001F1F2': {
        'en' : u':Zambia:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Zambia:'],
        'de': u':flagge_sambia:',
        'es': u':bandera_zambia:',
        'fr': u':drapeau_zambie:',
        'pt': u':bandeira_zâmbia:',
        'it': u':bandiera_zambia:'
    },
    u'\U0001F1FF\U0001F1FC': {
        'en' : u':Zimbabwe:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Zimbabwe:'],
        'de': u':flagge_simbabwe:',
        'es': u':bandera_zimbabue:',
        'fr': u':drapeau_zimbabwe:',
        'pt': u':bandeira_zimbábue:',
        'it': u':bandiera_zimbabwe:'
    },
    u'\U0001F9EE': {
        'en' : u':abacus:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':abakus:',
        'es': u':ábaco:',
        'fr': u':abaque:',
        'pt': u':ábaco:',
        'it': u':abaco:'
    },
    u'\U0001FA97': {
        'en' : u':accordion:',
        'status' : fully_qualified,
        'E' : 13,
        'de': u':akkordeon:',
        'es': u':acordeón:',
        'fr': u':accordéon:',
        'pt': u':acordeão:',
        'it': u':fisarmonica:'
    },
    u'\U0001FA79': {
        'en' : u':adhesive_bandage:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':heftpflaster:',
        'es': u':tirita:',
        'fr': u':sparadrap:',
        'pt': u':atadura_adesiva:',
        'it': u':cerotto:'
    },
    u'\U0001F39F\U0000FE0F': {
        'en' : u':admission_tickets:',
        'status' : fully_qualified,
        'E' : 0.7,
        'alias' : [u':tickets:'],
        'variant': True,
        'de': u':eintrittskarten:',
        'es': u':entradas:',
        'fr': u':billet_d’entrée:',
        'pt': u':ingresso_de_cinema:',
        'it': u':biglietto_d’ingresso:'
    },
    u'\U0001F39F': {
        'en' : u':admission_tickets:',
        'status' : unqualified,
        'E' : 0.7,
        'alias' : [u':tickets:'],
        'variant': True,
        'de': u':eintrittskarten:',
        'es': u':entradas:',
        'fr': u':billet_d’entrée:',
        'pt': u':ingresso_de_cinema:',
        'it': u':biglietto_d’ingresso:'
    },
    u'\U0001F6A1': {
        'en' : u':aerial_tramway:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':bergseilbahn:',
        'es': u':teleférico:',
        'fr': u':tramway_aérien:',
        'pt': u':teleférico:',
        'it': u':funivia_aerea:'
    },
    u'\U00002708\U0000FE0F': {
        'en' : u':airplane:',
        'status' : fully_qualified,
        'E' : 0.6,
        'variant': True,
        'de': u':flugzeug:',
        'es': u':avión:',
        'fr': u':avion:',
        'pt': u':avião:',
        'it': u':aeroplano:'
    },
    u'\U00002708': {
        'en' : u':airplane:',
        'status' : unqualified,
        'E' : 0.6,
        'variant': True,
        'de': u':flugzeug:',
        'es': u':avión:',
        'fr': u':avion:',
        'pt': u':avião:',
        'it': u':aeroplano:'
    },
    u'\U0001F6EC': {
        'en' : u':airplane_arrival:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [u':airplane_arriving:', u':flight_arrival:'],
        'de': u':landung_eines_flugzeugs:',
        'es': u':avión_aterrizando:',
        'fr': u':avion_à_l’atterrissage:',
        'pt': u':avião_aterrissando:',
        'it': u':atterraggio:'
    },
    u'\U0001F6EB': {
        'en' : u':airplane_departure:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [u':flight_departure:'],
        'de': u':abflug:',
        'es': u':avión_despegando:',
        'fr': u':avion_au_décollage:',
        'pt': u':avião_decolando:',
        'it': u':decollo:'
    },
    u'\U000023F0': {
        'en' : u':alarm_clock:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':wecker:',
        'es': u':reloj_despertador:',
        'fr': u':réveil:',
        'pt': u':despertador:',
        'it': u':sveglia:'
    },
    u'\U00002697\U0000FE0F': {
        'en' : u':alembic:',
        'status' : fully_qualified,
        'E' : 1,
        'variant': True,
        'de': u':destillierapparat:',
        'es': u':alambique:',
        'fr': u':alambic:',
        'pt': u':alambique:',
        'it': u':alambicco:'
    },
    u'\U00002697': {
        'en' : u':alembic:',
        'status' : unqualified,
        'E' : 1,
        'variant': True,
        'de': u':destillierapparat:',
        'es': u':alambique:',
        'fr': u':alambic:',
        'pt': u':alambique:',
        'it': u':alambicco:'
    },
    u'\U0001F47D': {
        'en' : u':alien:',
        'status' : fully_qualified,
        'E' : 0.6,
        'variant': True,
        'de': u':außerirdischer:',
        'es': u':alienígena:',
        'fr': u':alien:',
        'pt': u':alienígena:',
        'it': u':alieno:'
    },
    u'\U0001F47E': {
        'en' : u':alien_monster:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':space_invader:'],
        'de': u':computerspiel-monster:',
        'es': u':monstruo_alienígena:',
        'fr': u':monstre_de_l’espace:',
        'pt': u':monstro_alienígena:',
        'it': u':mostro_alieno:'
    },
    u'\U0001F691': {
        'en' : u':ambulance:',
        'status' : fully_qualified,
        'E' : 0.6,
        'variant': True,
        'de': u':krankenwagen:',
        'es': u':ambulancia:',
        'fr': u':ambulance:',
        'pt': u':ambulância:',
        'it': u':ambulanza:'
    },
    u'\U0001F3C8': {
        'en' : u':american_football:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':football:'],
        'de': u':football:',
        'es': u':balón_de_fútbol_americano:',
        'fr': u':football_américain:',
        'pt': u':bola_de_futebol_americano:',
        'it': u':football_americano:'
    },
    u'\U0001F3FA': {
        'en' : u':amphora:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':amphore:',
        'es': u':ánfora:',
        'fr': u':amphore:',
        'pt': u':ânfora:',
        'it': u':anfora:'
    },
    u'\U0001FAC0': {
        'en' : u':anatomical_heart:',
        'status' : fully_qualified,
        'E' : 13,
        'de': u':herz_(organ):',
        'es': u':corazón_humano:',
        'fr': u':cœur:',
        'pt': u':coração_humano:',
        'it': u':organo_del_cuore:'
    },
    u'\U00002693': {
        'en' : u':anchor:',
        'status' : fully_qualified,
        'E' : 0.6,
        'variant': True,
        'de': u':anker:',
        'es': u':ancla:',
        'fr': u':ancre:',
        'pt': u':âncora:',
        'it': u':ancora:'
    },
    u'\U0001F4A2': {
        'en' : u':anger_symbol:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':anger:'],
        'de': u':ärger:',
        'es': u':símbolo_de_enfado:',
        'fr': u':symbole_de_colère:',
        'pt': u':símbolo_de_raiva:',
        'it': u':rabbia:'
    },
    u'\U0001F620': {
        'en' : u':angry_face:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':angry:'],
        'de': u':verärgertes_gesicht:',
        'es': u':cara_enfadada:',
        'fr': u':visage_en_colère:',
        'pt': u':rosto_zangado:',
        'it': u':faccina_arrabbiata:'
    },
    u'\U0001F47F': {
        'en' : u':angry_face_with_horns:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':imp:'],
        'de': u':wütendes_gesicht_mit_hörnern:',
        'es': u':cara_enfadada_con_cuernos:',
        'fr': u':visage_en_colère_avec_des_cornes:',
        'pt': u':rosto_zangado_com_chifres:',
        'it': u':faccina_arrabbiata_con_corna:'
    },
    u'\U0001F627': {
        'en' : u':anguished_face:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [u':anguished:'],
        'de': u':qualvolles_gesicht:',
        'es': u':cara_angustiada:',
        'fr': u':visage_angoissé:',
        'pt': u':rosto_angustiado:',
        'it': u':faccina_angosciata:'
    },
    u'\U0001F41C': {
        'en' : u':ant:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':ameise:',
        'es': u':hormiga:',
        'fr': u':fourmi:',
        'pt': u':formiga:',
        'it': u':formica:'
    },
    u'\U0001F4F6': {
        'en' : u':antenna_bars:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':signal_strength:'],
        'de': u':balkenförmige_signalstärkenanzeige:',
        'es': u':barras_de_cobertura:',
        'fr': u':barres_de_réseau:',
        'pt': u':barras_de_sinal:',
        'it': u':segnale_cellulare:'
    },
    u'\U0001F630': {
        'en' : u':anxious_face_with_sweat:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':cold_sweat:'],
        'de': u':besorgtes_gesicht_mit_schweißtropfen:',
        'es': u':cara_con_ansiedad_y_sudor:',
        'fr': u':visage_anxieux_avec_goutte_de_sueur:',
        'pt': u':rosto_ansioso_com_gota_de_suor:',
        'it': u':faccina_sudata_in_ansia:'
    },
    u'\U0001F69B': {
        'en' : u':articulated_lorry:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':sattelzug:',
        'es': u':camión_articulado:',
        'fr': u':semi-remorque:',
        'pt': u':caminhão:',
        'it': u':autoarticolato:'
    },
    u'\U0001F9D1\U0000200D\U0001F3A8': {
        'en' : u':artist:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':künstler(in):',
        'es': u':artista:',
        'fr': u':artiste:',
        'pt': u':artista:',
        'it': u':artista:'
    },
    u'\U0001F9D1\U0001F3FF\U0000200D\U0001F3A8': {
        'en' : u':artist_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':künstler(in)_dunkle_hautfarbe:',
        'es': u':artista_tono_de_piel_oscuro:',
        'fr': u':artiste_peau_foncée:',
        'pt': u':artista_pele_escura:',
        'it': u':artista_carnagione_scura:'
    },
    u'\U0001F9D1\U0001F3FB\U0000200D\U0001F3A8': {
        'en' : u':artist_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':künstler(in)_helle_hautfarbe:',
        'es': u':artista_tono_de_piel_claro:',
        'fr': u':artiste_peau_claire:',
        'pt': u':artista_pele_clara:',
        'it': u':artista_carnagione_chiara:'
    },
    u'\U0001F9D1\U0001F3FE\U0000200D\U0001F3A8': {
        'en' : u':artist_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':künstler(in)_mitteldunkle_hautfarbe:',
        'es': u':artista_tono_de_piel_oscuro_medio:',
        'fr': u':artiste_peau_mate:',
        'pt': u':artista_pele_morena_escura:',
        'it': u':artista_carnagione_abbastanza_scura:'
    },
    u'\U0001F9D1\U0001F3FC\U0000200D\U0001F3A8': {
        'en' : u':artist_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':künstler(in)_mittelhelle_hautfarbe:',
        'es': u':artista_tono_de_piel_claro_medio:',
        'fr': u':artiste_peau_moyennement_claire:',
        'pt': u':artista_pele_morena_clara:',
        'it': u':artista_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9D1\U0001F3FD\U0000200D\U0001F3A8': {
        'en' : u':artist_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':künstler(in)_mittlere_hautfarbe:',
        'es': u':artista_tono_de_piel_medio:',
        'fr': u':artiste_peau_légèrement_mate:',
        'pt': u':artista_pele_morena:',
        'it': u':artista_carnagione_olivastra:'
    },
    u'\U0001F3A8': {
        'en' : u':artist_palette:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':art:'],
        'de': u':mischpalette:',
        'es': u':paleta_de_pintor:',
        'fr': u':palette_de_peinture:',
        'pt': u':paleta_de_tintas:',
        'it': u':tavolozza:'
    },
    u'\U0001F632': {
        'en' : u':astonished_face:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':astonished:'],
        'de': u':erstauntes_gesicht:',
        'es': u':cara_asombrada:',
        'fr': u':visage_stupéfait:',
        'pt': u':rosto_espantado:',
        'it': u':faccina_stupita:'
    },
    u'\U0001F9D1\U0000200D\U0001F680': {
        'en' : u':astronaut:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':astronaut(in):',
        'es': u':astronauta:',
        'fr': u':astronaute:',
        'pt': u':astronauta:',
        'it': u':astronauta:'
    },
    u'\U0001F9D1\U0001F3FF\U0000200D\U0001F680': {
        'en' : u':astronaut_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':astronaut(in)_dunkle_hautfarbe:',
        'es': u':astronauta_tono_de_piel_oscuro:',
        'fr': u':astronaute_peau_foncée:',
        'pt': u':astronauta_pele_escura:',
        'it': u':astronauta_carnagione_scura:'
    },
    u'\U0001F9D1\U0001F3FB\U0000200D\U0001F680': {
        'en' : u':astronaut_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':astronaut(in)_helle_hautfarbe:',
        'es': u':astronauta_tono_de_piel_claro:',
        'fr': u':astronaute_peau_claire:',
        'pt': u':astronauta_pele_clara:',
        'it': u':astronauta_carnagione_chiara:'
    },
    u'\U0001F9D1\U0001F3FE\U0000200D\U0001F680': {
        'en' : u':astronaut_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':astronaut(in)_mitteldunkle_hautfarbe:',
        'es': u':astronauta_tono_de_piel_oscuro_medio:',
        'fr': u':astronaute_peau_mate:',
        'pt': u':astronauta_pele_morena_escura:',
        'it': u':astronauta_carnagione_abbastanza_scura:'
    },
    u'\U0001F9D1\U0001F3FC\U0000200D\U0001F680': {
        'en' : u':astronaut_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':astronaut(in)_mittelhelle_hautfarbe:',
        'es': u':astronauta_tono_de_piel_claro_medio:',
        'fr': u':astronaute_peau_moyennement_claire:',
        'pt': u':astronauta_pele_morena_clara:',
        'it': u':astronauta_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9D1\U0001F3FD\U0000200D\U0001F680': {
        'en' : u':astronaut_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':astronaut(in)_mittlere_hautfarbe:',
        'es': u':astronauta_tono_de_piel_medio:',
        'fr': u':astronaute_peau_légèrement_mate:',
        'pt': u':astronauta_pele_morena:',
        'it': u':astronauta_carnagione_olivastra:'
    },
    u'\U0000269B\U0000FE0F': {
        'en' : u':atom_symbol:',
        'status' : fully_qualified,
        'E' : 1,
        'variant': True,
        'de': u':atomzeichen:',
        'es': u':símbolo_de_átomo:',
        'fr': u':symbole_de_l’atome:',
        'pt': u':símbolo_de_átomo:',
        'it': u':simbolo_dell’atomo:'
    },
    u'\U0000269B': {
        'en' : u':atom_symbol:',
        'status' : unqualified,
        'E' : 1,
        'variant': True,
        'de': u':atomzeichen:',
        'es': u':símbolo_de_átomo:',
        'fr': u':symbole_de_l’atome:',
        'pt': u':símbolo_de_átomo:',
        'it': u':simbolo_dell’atomo:'
    },
    u'\U0001F6FA': {
        'en' : u':auto_rickshaw:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':autorikscha:',
        'es': u':mototaxi:',
        'fr': u':tuk_tuk:',
        'pt': u':automóvel_riquixá:',
        'it': u':risciò_a_motore:'
    },
    u'\U0001F697': {
        'en' : u':automobile:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':red_car:', u':car:'],
        'de': u':auto:',
        'es': u':coche:',
        'fr': u':voiture:',
        'pt': u':carro:',
        'it': u':auto:'
    },
    u'\U0001F951': {
        'en' : u':avocado:',
        'status' : fully_qualified,
        'E' : 3,
        'de': u':avocado:',
        'es': u':aguacate:',
        'fr': u':avocat:',
        'pt': u':abacate:',
        'it': u':avocado:'
    },
    u'\U0001FA93': {
        'en' : u':axe:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':axt:',
        'es': u':hacha:',
        'fr': u':hache:',
        'pt': u':machado:',
        'it': u':ascia:'
    },
    u'\U0001F476': {
        'en' : u':baby:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':baby:',
        'es': u':bebé:',
        'fr': u':bébé:',
        'pt': u':bebê:',
        'it': u':neonato:'
    },
    u'\U0001F47C': {
        'en' : u':baby_angel:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':angel:'],
        'de': u':putte:',
        'es': u':bebé_ángel:',
        'fr': u':bébé_ange:',
        'pt': u':bebê_anjo:',
        'it': u':angioletto:'
    },
    u'\U0001F47C\U0001F3FF': {
        'en' : u':baby_angel_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':putte_dunkle_hautfarbe:',
        'es': u':bebé_ángel_tono_de_piel_oscuro:',
        'fr': u':bébé_ange_peau_foncée:',
        'pt': u':bebê_anjo_pele_escura:',
        'it': u':angioletto_carnagione_scura:'
    },
    u'\U0001F47C\U0001F3FB': {
        'en' : u':baby_angel_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':putte_helle_hautfarbe:',
        'es': u':bebé_ángel_tono_de_piel_claro:',
        'fr': u':bébé_ange_peau_claire:',
        'pt': u':bebê_anjo_pele_clara:',
        'it': u':angioletto_carnagione_chiara:'
    },
    u'\U0001F47C\U0001F3FE': {
        'en' : u':baby_angel_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':putte_mitteldunkle_hautfarbe:',
        'es': u':bebé_ángel_tono_de_piel_oscuro_medio:',
        'fr': u':bébé_ange_peau_mate:',
        'pt': u':bebê_anjo_pele_morena_escura:',
        'it': u':angioletto_carnagione_abbastanza_scura:'
    },
    u'\U0001F47C\U0001F3FC': {
        'en' : u':baby_angel_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':putte_mittelhelle_hautfarbe:',
        'es': u':bebé_ángel_tono_de_piel_claro_medio:',
        'fr': u':bébé_ange_peau_moyennement_claire:',
        'pt': u':bebê_anjo_pele_morena_clara:',
        'it': u':angioletto_carnagione_abbastanza_chiara:'
    },
    u'\U0001F47C\U0001F3FD': {
        'en' : u':baby_angel_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':putte_mittlere_hautfarbe:',
        'es': u':bebé_ángel_tono_de_piel_medio:',
        'fr': u':bébé_ange_peau_légèrement_mate:',
        'pt': u':bebê_anjo_pele_morena:',
        'it': u':angioletto_carnagione_olivastra:'
    },
    u'\U0001F37C': {
        'en' : u':baby_bottle:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':babyflasche:',
        'es': u':biberón:',
        'fr': u':biberon:',
        'pt': u':mamadeira:',
        'it': u':biberon:'
    },
    u'\U0001F424': {
        'en' : u':baby_chick:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':küken:',
        'es': u':pollito:',
        'fr': u':poussin:',
        'pt': u':pintinho_de_perfil:',
        'it': u':pulcino:'
    },
    u'\U0001F476\U0001F3FF': {
        'en' : u':baby_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':baby_dunkle_hautfarbe:',
        'es': u':bebé_tono_de_piel_oscuro:',
        'fr': u':bébé_peau_foncée:',
        'pt': u':bebê_pele_escura:',
        'it': u':neonato_carnagione_scura:'
    },
    u'\U0001F476\U0001F3FB': {
        'en' : u':baby_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':baby_helle_hautfarbe:',
        'es': u':bebé_tono_de_piel_claro:',
        'fr': u':bébé_peau_claire:',
        'pt': u':bebê_pele_clara:',
        'it': u':neonato_carnagione_chiara:'
    },
    u'\U0001F476\U0001F3FE': {
        'en' : u':baby_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':baby_mitteldunkle_hautfarbe:',
        'es': u':bebé_tono_de_piel_oscuro_medio:',
        'fr': u':bébé_peau_mate:',
        'pt': u':bebê_pele_morena_escura:',
        'it': u':neonato_carnagione_abbastanza_scura:'
    },
    u'\U0001F476\U0001F3FC': {
        'en' : u':baby_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':baby_mittelhelle_hautfarbe:',
        'es': u':bebé_tono_de_piel_claro_medio:',
        'fr': u':bébé_peau_moyennement_claire:',
        'pt': u':bebê_pele_morena_clara:',
        'it': u':neonato_carnagione_abbastanza_chiara:'
    },
    u'\U0001F476\U0001F3FD': {
        'en' : u':baby_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':baby_mittlere_hautfarbe:',
        'es': u':bebé_tono_de_piel_medio:',
        'fr': u':bébé_peau_légèrement_mate:',
        'pt': u':bebê_pele_morena:',
        'it': u':neonato_carnagione_olivastra:'
    },
    u'\U0001F6BC': {
        'en' : u':baby_symbol:',
        'status' : fully_qualified,
        'E' : 0.6,
        'variant': True,
        'de': u':symbol_baby:',
        'es': u':señal_de_bebé:',
        'fr': u':symbole_bébé:',
        'pt': u':símbolo_de_bebê:',
        'it': u':simbolo_con_immagine_di_bambino:'
    },
    u'\U0001F447': {
        'en' : u':backhand_index_pointing_down:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':point_down:'],
        'variant': True,
        'de': u':nach_unten_weisender_zeigefinger:',
        'es': u':dorso_de_mano_con_índice_hacia_abajo:',
        'fr': u':main_avec_index_pointant_vers_le_bas:',
        'pt': u':dorso_da_mão_com_dedo_indicador_apontando_para_baixo:',
        'it': u':indice_abbassato:'
    },
    u'\U0001F447\U0001F3FF': {
        'en' : u':backhand_index_pointing_down_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':nach_unten_weisender_zeigefinger_dunkle_hautfarbe:',
        'es': u':dorso_de_mano_con_índice_hacia_abajo_tono_de_piel_oscuro:',
        'fr': u':main_avec_index_pointant_vers_le_bas_peau_foncée:',
        'pt': u':dorso_da_mão_com_dedo_indicador_apontando_para_baixo_pele_escura:',
        'it': u':indice_abbassato_carnagione_scura:'
    },
    u'\U0001F447\U0001F3FB': {
        'en' : u':backhand_index_pointing_down_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':nach_unten_weisender_zeigefinger_helle_hautfarbe:',
        'es': u':dorso_de_mano_con_índice_hacia_abajo_tono_de_piel_claro:',
        'fr': u':main_avec_index_pointant_vers_le_bas_peau_claire:',
        'pt': u':dorso_da_mão_com_dedo_indicador_apontando_para_baixo_pele_clara:',
        'it': u':indice_abbassato_carnagione_chiara:'
    },
    u'\U0001F447\U0001F3FE': {
        'en' : u':backhand_index_pointing_down_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':nach_unten_weisender_zeigefinger_mitteldunkle_hautfarbe:',
        'es': u':dorso_de_mano_con_índice_hacia_abajo_tono_de_piel_oscuro_medio:',
        'fr': u':main_avec_index_pointant_vers_le_bas_peau_mate:',
        'pt': u':dorso_da_mão_com_dedo_indicador_apontando_para_baixo_pele_morena_escura:',
        'it': u':indice_abbassato_carnagione_abbastanza_scura:'
    },
    u'\U0001F447\U0001F3FC': {
        'en' : u':backhand_index_pointing_down_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':nach_unten_weisender_zeigefinger_mittelhelle_hautfarbe:',
        'es': u':dorso_de_mano_con_índice_hacia_abajo_tono_de_piel_claro_medio:',
        'fr': u':main_avec_index_pointant_vers_le_bas_peau_moyennement_claire:',
        'pt': u':dorso_da_mão_com_dedo_indicador_apontando_para_baixo_pele_morena_clara:',
        'it': u':indice_abbassato_carnagione_abbastanza_chiara:'
    },
    u'\U0001F447\U0001F3FD': {
        'en' : u':backhand_index_pointing_down_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':nach_unten_weisender_zeigefinger_mittlere_hautfarbe:',
        'es': u':dorso_de_mano_con_índice_hacia_abajo_tono_de_piel_medio:',
        'fr': u':main_avec_index_pointant_vers_le_bas_peau_légèrement_mate:',
        'pt': u':dorso_da_mão_com_dedo_indicador_apontando_para_baixo_pele_morena:',
        'it': u':indice_abbassato_carnagione_olivastra:'
    },
    u'\U0001F448': {
        'en' : u':backhand_index_pointing_left:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':point_left:'],
        'variant': True,
        'de': u':nach_links_weisender_zeigefinger:',
        'es': u':dorso_de_mano_con_índice_a_la_izquierda:',
        'fr': u':main_avec_index_pointant_à_gauche:',
        'pt': u':dorso_da_mão_com_dedo_indicador_apontando_para_a_esquerda:',
        'it': u':indice_verso_sinistra:'
    },
    u'\U0001F448\U0001F3FF': {
        'en' : u':backhand_index_pointing_left_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':nach_links_weisender_zeigefinger_dunkle_hautfarbe:',
        'es': u':dorso_de_mano_con_índice_a_la_izquierda_tono_de_piel_oscuro:',
        'fr': u':main_avec_index_pointant_à_gauche_peau_foncée:',
        'pt': u':dorso_da_mão_com_dedo_indicador_apontando_para_a_esquerda_pele_escura:',
        'it': u':indice_verso_sinistra_carnagione_scura:'
    },
    u'\U0001F448\U0001F3FB': {
        'en' : u':backhand_index_pointing_left_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':nach_links_weisender_zeigefinger_helle_hautfarbe:',
        'es': u':dorso_de_mano_con_índice_a_la_izquierda_tono_de_piel_claro:',
        'fr': u':main_avec_index_pointant_à_gauche_peau_claire:',
        'pt': u':dorso_da_mão_com_dedo_indicador_apontando_para_a_esquerda_pele_clara:',
        'it': u':indice_verso_sinistra_carnagione_chiara:'
    },
    u'\U0001F448\U0001F3FE': {
        'en' : u':backhand_index_pointing_left_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':nach_links_weisender_zeigefinger_mitteldunkle_hautfarbe:',
        'es': u':dorso_de_mano_con_índice_a_la_izquierda_tono_de_piel_oscuro_medio:',
        'fr': u':main_avec_index_pointant_à_gauche_peau_mate:',
        'pt': u':dorso_da_mão_com_dedo_indicador_apontando_para_a_esquerda_pele_morena_escura:',
        'it': u':indice_verso_sinistra_carnagione_abbastanza_scura:'
    },
    u'\U0001F448\U0001F3FC': {
        'en' : u':backhand_index_pointing_left_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':nach_links_weisender_zeigefinger_mittelhelle_hautfarbe:',
        'es': u':dorso_de_mano_con_índice_a_la_izquierda_tono_de_piel_claro_medio:',
        'fr': u':main_avec_index_pointant_à_gauche_peau_moyennement_claire:',
        'pt': u':dorso_da_mão_com_dedo_indicador_apontando_para_a_esquerda_pele_morena_clara:',
        'it': u':indice_verso_sinistra_carnagione_abbastanza_chiara:'
    },
    u'\U0001F448\U0001F3FD': {
        'en' : u':backhand_index_pointing_left_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':nach_links_weisender_zeigefinger_mittlere_hautfarbe:',
        'es': u':dorso_de_mano_con_índice_a_la_izquierda_tono_de_piel_medio:',
        'fr': u':main_avec_index_pointant_à_gauche_peau_légèrement_mate:',
        'pt': u':dorso_da_mão_com_dedo_indicador_apontando_para_a_esquerda_pele_morena:',
        'it': u':indice_verso_sinistra_carnagione_olivastra:'
    },
    u'\U0001F449': {
        'en' : u':backhand_index_pointing_right:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':point_right:'],
        'variant': True,
        'de': u':nach_rechts_weisender_zeigefinger:',
        'es': u':dorso_de_mano_con_índice_a_la_derecha:',
        'fr': u':main_avec_index_pointant_à_droite:',
        'pt': u':dorso_da_mão_com_dedo_indicador_apontando_para_a_direita:',
        'it': u':indice_verso_destra:'
    },
    u'\U0001F449\U0001F3FF': {
        'en' : u':backhand_index_pointing_right_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':nach_rechts_weisender_zeigefinger_dunkle_hautfarbe:',
        'es': u':dorso_de_mano_con_índice_a_la_derecha_tono_de_piel_oscuro:',
        'fr': u':main_avec_index_pointant_à_droite_peau_foncée:',
        'pt': u':dorso_da_mão_com_dedo_indicador_apontando_para_a_direita_pele_escura:',
        'it': u':indice_verso_destra_carnagione_scura:'
    },
    u'\U0001F449\U0001F3FB': {
        'en' : u':backhand_index_pointing_right_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':nach_rechts_weisender_zeigefinger_helle_hautfarbe:',
        'es': u':dorso_de_mano_con_índice_a_la_derecha_tono_de_piel_claro:',
        'fr': u':main_avec_index_pointant_à_droite_peau_claire:',
        'pt': u':dorso_da_mão_com_dedo_indicador_apontando_para_a_direita_pele_clara:',
        'it': u':indice_verso_destra_carnagione_chiara:'
    },
    u'\U0001F449\U0001F3FE': {
        'en' : u':backhand_index_pointing_right_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':nach_rechts_weisender_zeigefinger_mitteldunkle_hautfarbe:',
        'es': u':dorso_de_mano_con_índice_a_la_derecha_tono_de_piel_oscuro_medio:',
        'fr': u':main_avec_index_pointant_à_droite_peau_mate:',
        'pt': u':dorso_da_mão_com_dedo_indicador_apontando_para_a_direita_pele_morena_escura:',
        'it': u':indice_verso_destra_carnagione_abbastanza_scura:'
    },
    u'\U0001F449\U0001F3FC': {
        'en' : u':backhand_index_pointing_right_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':nach_rechts_weisender_zeigefinger_mittelhelle_hautfarbe:',
        'es': u':dorso_de_mano_con_índice_a_la_derecha_tono_de_piel_claro_medio:',
        'fr': u':main_avec_index_pointant_à_droite_peau_moyennement_claire:',
        'pt': u':dorso_da_mão_com_dedo_indicador_apontando_para_a_direita_pele_morena_clara:',
        'it': u':indice_verso_destra_carnagione_abbastanza_chiara:'
    },
    u'\U0001F449\U0001F3FD': {
        'en' : u':backhand_index_pointing_right_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':nach_rechts_weisender_zeigefinger_mittlere_hautfarbe:',
        'es': u':dorso_de_mano_con_índice_a_la_derecha_tono_de_piel_medio:',
        'fr': u':main_avec_index_pointant_à_droite_peau_légèrement_mate:',
        'pt': u':dorso_da_mão_com_dedo_indicador_apontando_para_a_direita_pele_morena:',
        'it': u':indice_verso_destra_carnagione_olivastra:'
    },
    u'\U0001F446': {
        'en' : u':backhand_index_pointing_up:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':point_up_2:'],
        'variant': True,
        'de': u':nach_oben_weisender_zeigefinger_von_hinten:',
        'es': u':dorso_de_mano_con_índice_hacia_arriba:',
        'fr': u':main_avec_index_pointant_vers_le_haut:',
        'pt': u':dorso_da_mão_com_dedo_indicador_apontando_para_cima:',
        'it': u':indice_alzato:'
    },
    u'\U0001F446\U0001F3FF': {
        'en' : u':backhand_index_pointing_up_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':nach_oben_weisender_zeigefinger_von_hinten_dunkle_hautfarbe:',
        'es': u':dorso_de_mano_con_índice_hacia_arriba_tono_de_piel_oscuro:',
        'fr': u':main_avec_index_pointant_vers_le_haut_peau_foncée:',
        'pt': u':dorso_da_mão_com_dedo_indicador_apontando_para_cima_pele_escura:',
        'it': u':indice_alzato_carnagione_scura:'
    },
    u'\U0001F446\U0001F3FB': {
        'en' : u':backhand_index_pointing_up_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':nach_oben_weisender_zeigefinger_von_hinten_helle_hautfarbe:',
        'es': u':dorso_de_mano_con_índice_hacia_arriba_tono_de_piel_claro:',
        'fr': u':main_avec_index_pointant_vers_le_haut_peau_claire:',
        'pt': u':dorso_da_mão_com_dedo_indicador_apontando_para_cima_pele_clara:',
        'it': u':indice_alzato_carnagione_chiara:'
    },
    u'\U0001F446\U0001F3FE': {
        'en' : u':backhand_index_pointing_up_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':nach_oben_weisender_zeigefinger_von_hinten_mitteldunkle_hautfarbe:',
        'es': u':dorso_de_mano_con_índice_hacia_arriba_tono_de_piel_oscuro_medio:',
        'fr': u':main_avec_index_pointant_vers_le_haut_peau_mate:',
        'pt': u':dorso_da_mão_com_dedo_indicador_apontando_para_cima_pele_morena_escura:',
        'it': u':indice_alzato_carnagione_abbastanza_scura:'
    },
    u'\U0001F446\U0001F3FC': {
        'en' : u':backhand_index_pointing_up_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':nach_oben_weisender_zeigefinger_von_hinten_mittelhelle_hautfarbe:',
        'es': u':dorso_de_mano_con_índice_hacia_arriba_tono_de_piel_claro_medio:',
        'fr': u':main_avec_index_pointant_vers_le_haut_peau_moyennement_claire:',
        'pt': u':dorso_da_mão_com_dedo_indicador_apontando_para_cima_pele_morena_clara:',
        'it': u':indice_alzato_carnagione_abbastanza_chiara:'
    },
    u'\U0001F446\U0001F3FD': {
        'en' : u':backhand_index_pointing_up_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':nach_oben_weisender_zeigefinger_von_hinten_mittlere_hautfarbe:',
        'es': u':dorso_de_mano_con_índice_hacia_arriba_tono_de_piel_medio:',
        'fr': u':main_avec_index_pointant_vers_le_haut_peau_légèrement_mate:',
        'pt': u':dorso_da_mão_com_dedo_indicador_apontando_para_cima_pele_morena:',
        'it': u':indice_alzato_carnagione_olivastra:'
    },
    u'\U0001F392': {
        'en' : u':backpack:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':school_satchel:'],
        'de': u':schulranzen:',
        'es': u':mochila_escolar:',
        'fr': u':cartable:',
        'pt': u':mochila:',
        'it': u':zaino:'
    },
    u'\U0001F953': {
        'en' : u':bacon:',
        'status' : fully_qualified,
        'E' : 3,
        'de': u':bacon:',
        'es': u':beicon:',
        'fr': u':lard:',
        'pt': u':bacon:',
        'it': u':pancetta:'
    },
    u'\U0001F9A1': {
        'en' : u':badger:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':dachs:',
        'es': u':tejón:',
        'fr': u':blaireau:',
        'pt': u':texugo:',
        'it': u':tasso:'
    },
    u'\U0001F3F8': {
        'en' : u':badminton:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [u':badminton_racquet_and_shuttlecock:'],
        'de': u':badminton:',
        'es': u':bádminton:',
        'fr': u':badminton:',
        'pt': u':badminton:',
        'it': u':badminton:'
    },
    u'\U0001F96F': {
        'en' : u':bagel:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':bagel:',
        'es': u':bagel:',
        'fr': u':bagel:',
        'pt': u':rosca:',
        'it': u':bagel:'
    },
    u'\U0001F6C4': {
        'en' : u':baggage_claim:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':gepäckausgabe:',
        'es': u':recogida_de_equipajes:',
        'fr': u':retrait_des_bagages:',
        'pt': u':restituição_de_bagagem:',
        'it': u':simbolo_del_ritiro_bagagli:'
    },
    u'\U0001F956': {
        'en' : u':baguette_bread:',
        'status' : fully_qualified,
        'E' : 3,
        'de': u':baguette:',
        'es': u':baguete:',
        'fr': u':baguette:',
        'pt': u':baguete:',
        'it': u':baguette:'
    },
    u'\U00002696\U0000FE0F': {
        'en' : u':balance_scale:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [u':scales:'],
        'variant': True,
        'de': u':waage:',
        'es': u':balanza:',
        'fr': u':balance_à_poids:',
        'pt': u':balança:',
        'it': u':bilancia_a_doppio_piatto:'
    },
    u'\U00002696': {
        'en' : u':balance_scale:',
        'status' : unqualified,
        'E' : 1,
        'alias' : [u':scales:'],
        'variant': True,
        'de': u':waage:',
        'es': u':balanza:',
        'fr': u':balance_à_poids:',
        'pt': u':balança:',
        'it': u':bilancia_a_doppio_piatto:'
    },
    u'\U0001F9B2': {
        'en' : u':bald:',
        'status' : component,
        'E' : 11,
        'de': u':glatze:',
        'es': u':sin_pelo:',
        'fr': u':chauve:',
        'pt': u':careca:',
        'it': u':calvo:'
    },
    u'\U0001FA70': {
        'en' : u':ballet_shoes:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':ballettschuhe:',
        'es': u':zapatillas_de_ballet:',
        'fr': u':chaussons_de_danse:',
        'pt': u':sapatilha_de_balé:',
        'it': u':scarpette_da_ballerina:'
    },
    u'\U0001F388': {
        'en' : u':balloon:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':luftballon:',
        'es': u':globo:',
        'fr': u':ballon_gonflable:',
        'pt': u':balão:',
        'it': u':palloncino:'
    },
    u'\U0001F5F3\U0000FE0F': {
        'en' : u':ballot_box_with_ballot:',
        'status' : fully_qualified,
        'E' : 0.7,
        'alias' : [u':ballot_box:'],
        'variant': True,
        'de': u':urne_mit_wahlzettel:',
        'es': u':urna_con_papeleta:',
        'fr': u':urne_électorale:',
        'pt': u':urna_eleitoral_com_cédula:',
        'it': u':urna_elettorale_con_scheda:'
    },
    u'\U0001F5F3': {
        'en' : u':ballot_box_with_ballot:',
        'status' : unqualified,
        'E' : 0.7,
        'alias' : [u':ballot_box:'],
        'variant': True,
        'de': u':urne_mit_wahlzettel:',
        'es': u':urna_con_papeleta:',
        'fr': u':urne_électorale:',
        'pt': u':urna_eleitoral_com_cédula:',
        'it': u':urna_elettorale_con_scheda:'
    },
    u'\U0001F34C': {
        'en' : u':banana:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':banane:',
        'es': u':plátano:',
        'fr': u':banane:',
        'pt': u':banana:',
        'it': u':banana:'
    },
    u'\U0001FA95': {
        'en' : u':banjo:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':banjo:',
        'es': u':banjo:',
        'fr': u':banjo:',
        'pt': u':banjo:',
        'it': u':banjo:'
    },
    u'\U0001F3E6': {
        'en' : u':bank:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':bank:',
        'es': u':banco:',
        'fr': u':banque:',
        'pt': u':banco:',
        'it': u':banca:'
    },
    u'\U0001F4CA': {
        'en' : u':bar_chart:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':balkendiagramm:',
        'es': u':gráfico_de_barras:',
        'fr': u':graphiques_à_barres:',
        'pt': u':gráfico_de_barras:',
        'it': u':grafico_a_barre:'
    },
    u'\U0001F488': {
        'en' : u':barber_pole:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':barber:'],
        'de': u':barbershop-säule:',
        'es': u':poste_de_barbero:',
        'fr': u':enseigne_de_barbier:',
        'pt': u':barbearia:',
        'it': u':barbiere:'
    },
    u'\U000026BE': {
        'en' : u':baseball:',
        'status' : fully_qualified,
        'E' : 0.6,
        'variant': True,
        'de': u':baseball:',
        'es': u':béisbol:',
        'fr': u':baseball:',
        'pt': u':bola_de_beisebol:',
        'it': u':palla_da_baseball:'
    },
    u'\U0001F9FA': {
        'en' : u':basket:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':korb:',
        'es': u':cesta:',
        'fr': u':panier:',
        'pt': u':cesta:',
        'it': u':cesto:'
    },
    u'\U0001F3C0': {
        'en' : u':basketball:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':basketball:',
        'es': u':balón_de_baloncesto:',
        'fr': u':basket:',
        'pt': u':bola_de_basquete:',
        'it': u':palla_da_pallacanestro:'
    },
    u'\U0001F987': {
        'en' : u':bat:',
        'status' : fully_qualified,
        'E' : 3,
        'de': u':fledermaus:',
        'es': u':murciélago:',
        'fr': u':chauve-souris:',
        'pt': u':morcego:',
        'it': u':pipistrello:'
    },
    u'\U0001F6C1': {
        'en' : u':bathtub:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':badewanne:',
        'es': u':bañera:',
        'fr': u':baignoire:',
        'pt': u':banheira:',
        'it': u':vasca:'
    },
    u'\U0001F50B': {
        'en' : u':battery:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':batterie:',
        'es': u':pila:',
        'fr': u':pile:',
        'pt': u':pilha:',
        'it': u':batteria:'
    },
    u'\U0001F3D6\U0000FE0F': {
        'en' : u':beach_with_umbrella:',
        'status' : fully_qualified,
        'E' : 0.7,
        'alias' : [u':beach_umbrella:'],
        'variant': True,
        'de': u':strand_mit_sonnenschirm:',
        'es': u':playa_y_sombrilla:',
        'fr': u':plage_avec_parasol:',
        'pt': u':praia_e_guarda-sol:',
        'it': u':spiaggia_con_ombrellone:'
    },
    u'\U0001F3D6': {
        'en' : u':beach_with_umbrella:',
        'status' : unqualified,
        'E' : 0.7,
        'alias' : [u':beach_umbrella:'],
        'variant': True,
        'de': u':strand_mit_sonnenschirm:',
        'es': u':playa_y_sombrilla:',
        'fr': u':plage_avec_parasol:',
        'pt': u':praia_e_guarda-sol:',
        'it': u':spiaggia_con_ombrellone:'
    },
    u'\U0001F601': {
        'en' : u':beaming_face_with_smiling_eyes:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':grin:'],
        'de': u':strahlendes_gesicht_mit_lachenden_augen:',
        'es': u':cara_radiante_con_ojos_sonrientes:',
        'fr': u':visage_souriant_aux_yeux_rieurs:',
        'pt': u':rosto_contente_com_olhos_sorridentes:',
        'it': u':faccina_raggiante_con_occhi_felici:'
    },
    u'\U0001FAD8': {
        'en' : u':beans:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001F43B': {
        'en' : u':bear:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':bär:',
        'es': u':oso:',
        'fr': u':ours:',
        'pt': u':rosto_de_urso:',
        'it': u':orso:'
    },
    u'\U0001F493': {
        'en' : u':beating_heart:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':heartbeat:'],
        'de': u':schlagendes_herz:',
        'es': u':corazón_latiendo:',
        'fr': u':cœur_battant:',
        'pt': u':coração_pulsante:',
        'it': u':cuore_che_batte:'
    },
    u'\U0001F9AB': {
        'en' : u':beaver:',
        'status' : fully_qualified,
        'E' : 13,
        'de': u':biber:',
        'es': u':castor:',
        'fr': u':castor:',
        'pt': u':castor:',
        'it': u':castoro:'
    },
    u'\U0001F6CF\U0000FE0F': {
        'en' : u':bed:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': u':bett:',
        'es': u':cama:',
        'fr': u':lit:',
        'pt': u':cama:',
        'it': u':letto:'
    },
    u'\U0001F6CF': {
        'en' : u':bed:',
        'status' : unqualified,
        'E' : 0.7,
        'variant': True,
        'de': u':bett:',
        'es': u':cama:',
        'fr': u':lit:',
        'pt': u':cama:',
        'it': u':letto:'
    },
    u'\U0001F37A': {
        'en' : u':beer_mug:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':beer:'],
        'de': u':bierkrug:',
        'es': u':jarra_de_cerveza:',
        'fr': u':chope:',
        'pt': u':cerveja:',
        'it': u':boccale_di_birra:'
    },
    u'\U0001FAB2': {
        'en' : u':beetle:',
        'status' : fully_qualified,
        'E' : 13,
        'de': u':käfer:',
        'es': u':escarabajo:',
        'fr': u':scarabée:',
        'pt': u':besouro:',
        'it': u':coleottero:'
    },
    u'\U0001F514': {
        'en' : u':bell:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':glocke:',
        'es': u':campana:',
        'fr': u':cloche:',
        'pt': u':sino:',
        'it': u':campana:'
    },
    u'\U0001FAD1': {
        'en' : u':bell_pepper:',
        'status' : fully_qualified,
        'E' : 13,
        'de': u':paprika:',
        'es': u':pimiento:',
        'fr': u':poivron:',
        'pt': u':pimentão:',
        'it': u':peperone:'
    },
    u'\U0001F515': {
        'en' : u':bell_with_slash:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [u':no_bell:'],
        'de': u':durchgestrichene_glocke:',
        'es': u':campana_con_signo_de_cancelación:',
        'fr': u':alarme_désactivée:',
        'pt': u':sino_silenciado:',
        'it': u':campana_sbarrata:'
    },
    u'\U0001F6CE\U0000FE0F': {
        'en' : u':bellhop_bell:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': u':rezeptionsklingel:',
        'es': u':timbre_de_hotel:',
        'fr': u':cloche_de_comptoir:',
        'pt': u':sineta:',
        'it': u':campanello_da_hotel:'
    },
    u'\U0001F6CE': {
        'en' : u':bellhop_bell:',
        'status' : unqualified,
        'E' : 0.7,
        'variant': True,
        'de': u':rezeptionsklingel:',
        'es': u':timbre_de_hotel:',
        'fr': u':cloche_de_comptoir:',
        'pt': u':sineta:',
        'it': u':campanello_da_hotel:'
    },
    u'\U0001F371': {
        'en' : u':bento_box:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':bento:'],
        'de': u':bento-box:',
        'es': u':caja_de_bento:',
        'fr': u':boîte_déjeuner:',
        'pt': u':bentô:',
        'it': u':bento_box:'
    },
    u'\U0001F9C3': {
        'en' : u':beverage_box:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':trinkpäckchen:',
        'es': u':tetrabrik:',
        'fr': u':briquette_de_jus:',
        'pt': u':suco_de_caixa:',
        'it': u':bevanda_monodose:'
    },
    u'\U0001F6B2': {
        'en' : u':bicycle:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':bike:'],
        'variant': True,
        'de': u':fahrrad:',
        'es': u':bicicleta:',
        'fr': u':vélo:',
        'pt': u':bicicleta:',
        'it': u':bicicletta:'
    },
    u'\U0001F459': {
        'en' : u':bikini:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':bikini:',
        'es': u':bikini:',
        'fr': u':bikini:',
        'pt': u':biquíni:',
        'it': u':bikini:'
    },
    u'\U0001F9E2': {
        'en' : u':billed_cap:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':baseballmütze:',
        'es': u':gorra_con_visera:',
        'fr': u':casquette_américaine:',
        'pt': u':boné:',
        'it': u':cappello_con_visiera:'
    },
    u'\U00002623\U0000FE0F': {
        'en' : u':biohazard:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [u':biohazard_sign:'],
        'variant': True,
        'de': u':biogefährdung:',
        'es': u':riesgo_biológico:',
        'fr': u':danger_biologique:',
        'pt': u':risco_biológico:',
        'it': u':simbolo_del_rischio_biologico:'
    },
    u'\U00002623': {
        'en' : u':biohazard:',
        'status' : unqualified,
        'E' : 1,
        'alias' : [u':biohazard_sign:'],
        'variant': True,
        'de': u':biogefährdung:',
        'es': u':riesgo_biológico:',
        'fr': u':danger_biologique:',
        'pt': u':risco_biológico:',
        'it': u':simbolo_del_rischio_biologico:'
    },
    u'\U0001F426': {
        'en' : u':bird:',
        'status' : fully_qualified,
        'E' : 0.6,
        'variant': True,
        'de': u':vogel:',
        'es': u':pájaro:',
        'fr': u':oiseau:',
        'pt': u':pássaro:',
        'it': u':uccello:'
    },
    u'\U0001F382': {
        'en' : u':birthday_cake:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':birthday:'],
        'de': u':geburtstagskuchen:',
        'es': u':tarta_de_cumpleaños:',
        'fr': u':gâteau_d’anniversaire:',
        'pt': u':bolo_de_aniversário:',
        'it': u':torta_di_compleanno:'
    },
    u'\U0001F9AC': {
        'en' : u':bison:',
        'status' : fully_qualified,
        'E' : 13,
        'de': u':bison:',
        'es': u':bisonte:',
        'fr': u':bison:',
        'pt': u':bisão:',
        'it': u':bisonte:'
    },
    u'\U0001FAE6': {
        'en' : u':biting_lip:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001F408\U0000200D\U00002B1B': {
        'en' : u':black_cat:',
        'status' : fully_qualified,
        'E' : 13,
        'de': u':schwarze_katze:',
        'es': u':gato_negro:',
        'fr': u':chat_noir:',
        'pt': u':gato_preto:',
        'it': u':gatto_nero:'
    },
    u'\U000026AB': {
        'en' : u':black_circle:',
        'status' : fully_qualified,
        'E' : 0.6,
        'variant': True,
        'de': u':schwarzer_kreis:',
        'es': u':círculo_negro:',
        'fr': u':disque_noir:',
        'pt': u':círculo_preto:',
        'it': u':cerchio_nero:'
    },
    u'\U0001F3F4': {
        'en' : u':black_flag:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [u':waving_black_flag:'],
        'de': u':schwarze_flagge:',
        'es': u':bandera_negra:',
        'fr': u':drapeau_noir:',
        'pt': u':bandeira_preta:',
        'it': u':bandiera_nera:'
    },
    u'\U0001F5A4': {
        'en' : u':black_heart:',
        'status' : fully_qualified,
        'E' : 3,
        'de': u':schwarzes_herz:',
        'es': u':corazón_negro:',
        'fr': u':cœur_noir:',
        'pt': u':coração_preto:',
        'it': u':cuore_nero:'
    },
    u'\U00002B1B': {
        'en' : u':black_large_square:',
        'status' : fully_qualified,
        'E' : 0.6,
        'variant': True,
        'de': u':großes_schwarzes_quadrat:',
        'es': u':cuadrado_negro_grande:',
        'fr': u':grand_carré_noir:',
        'pt': u':quadrado_preto_grande:',
        'it': u':quadrato_nero_grande:'
    },
    u'\U000025FE': {
        'en' : u':black_medium-small_square:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':black_medium_small_square:'],
        'variant': True,
        'de': u':mittelkleines_schwarzes_quadrat:',
        'es': u':cuadrado_negro_mediano-pequeño:',
        'fr': u':carré_petit_moyen_noir:',
        'pt': u':quadrado_preto_médio_menor:',
        'it': u':quadrato_nero_medio-piccolo:'
    },
    u'\U000025FC\U0000FE0F': {
        'en' : u':black_medium_square:',
        'status' : fully_qualified,
        'E' : 0.6,
        'variant': True,
        'de': u':mittelgroßes_schwarzes_quadrat:',
        'es': u':cuadrado_negro_mediano:',
        'fr': u':carré_moyen_noir:',
        'pt': u':quadrado_preto_médio:',
        'it': u':quadrato_nero_medio:'
    },
    u'\U000025FC': {
        'en' : u':black_medium_square:',
        'status' : unqualified,
        'E' : 0.6,
        'variant': True,
        'de': u':mittelgroßes_schwarzes_quadrat:',
        'es': u':cuadrado_negro_mediano:',
        'fr': u':carré_moyen_noir:',
        'pt': u':quadrado_preto_médio:',
        'it': u':quadrato_nero_medio:'
    },
    u'\U00002712\U0000FE0F': {
        'en' : u':black_nib:',
        'status' : fully_qualified,
        'E' : 0.6,
        'variant': True,
        'de': u':schwarzer_federhalter:',
        'es': u':pluma_negra:',
        'fr': u':stylo_plume_noir:',
        'pt': u':ponta_de_caneta_tinteiro:',
        'it': u':pennino:'
    },
    u'\U00002712': {
        'en' : u':black_nib:',
        'status' : unqualified,
        'E' : 0.6,
        'variant': True,
        'de': u':schwarzer_federhalter:',
        'es': u':pluma_negra:',
        'fr': u':stylo_plume_noir:',
        'pt': u':ponta_de_caneta_tinteiro:',
        'it': u':pennino:'
    },
    u'\U000025AA\U0000FE0F': {
        'en' : u':black_small_square:',
        'status' : fully_qualified,
        'E' : 0.6,
        'variant': True,
        'de': u':kleines_schwarzes_quadrat:',
        'es': u':cuadrado_negro_pequeño:',
        'fr': u':petit_carré_noir:',
        'pt': u':quadrado_preto_pequeno:',
        'it': u':quadrato_nero_piccolo:'
    },
    u'\U000025AA': {
        'en' : u':black_small_square:',
        'status' : unqualified,
        'E' : 0.6,
        'variant': True,
        'de': u':kleines_schwarzes_quadrat:',
        'es': u':cuadrado_negro_pequeño:',
        'fr': u':petit_carré_noir:',
        'pt': u':quadrado_preto_pequeno:',
        'it': u':quadrato_nero_piccolo:'
    },
    u'\U0001F532': {
        'en' : u':black_square_button:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':schwarze_quadratische_schaltfläche:',
        'es': u':botón_cuadrado_con_borde_negro:',
        'fr': u':carré_noir:',
        'pt': u':botão_quadrado_preto:',
        'it': u':tasto_quadrato_bianco_con_bordo_nero:'
    },
    u'\U0001F33C': {
        'en' : u':blossom:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':gelbe_blüte:',
        'es': u':flor:',
        'fr': u':bourgeon:',
        'pt': u':flor:',
        'it': u':fiore:'
    },
    u'\U0001F421': {
        'en' : u':blowfish:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':kugelfisch:',
        'es': u':pez_globo:',
        'fr': u':poisson-lune:',
        'pt': u':baiacu:',
        'it': u':pesce_palla:'
    },
    u'\U0001F4D8': {
        'en' : u':blue_book:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':blaues_buch:',
        'es': u':libro_azul:',
        'fr': u':livre_bleu:',
        'pt': u':livro_azul:',
        'it': u':libro_blu:'
    },
    u'\U0001F535': {
        'en' : u':blue_circle:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':large_blue_circle:'],
        'de': u':blauer_kreis:',
        'es': u':círculo_azul_grande:',
        'fr': u':disque_bleu:',
        'pt': u':círculo_azul:',
        'it': u':cerchio_blu:'
    },
    u'\U0001F499': {
        'en' : u':blue_heart:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':blaues_herz:',
        'es': u':corazón_azul:',
        'fr': u':cœur_bleu:',
        'pt': u':coração_azul:',
        'it': u':cuore_azzurro:'
    },
    u'\U0001F7E6': {
        'en' : u':blue_square:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':blaues_viereck:',
        'es': u':cuadrado_azul:',
        'fr': u':carré_bleu:',
        'pt': u':quadrado_azul:',
        'it': u':quadrato_blu:'
    },
    u'\U0001FAD0': {
        'en' : u':blueberries:',
        'status' : fully_qualified,
        'E' : 13,
        'de': u':blaubeeren:',
        'es': u':arándanos:',
        'fr': u':myrtilles:',
        'pt': u':mirtilos:',
        'it': u':mirtilli:'
    },
    u'\U0001F417': {
        'en' : u':boar:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':wildschwein:',
        'es': u':jabalí:',
        'fr': u':sanglier:',
        'pt': u':javali:',
        'it': u':cinghiale:'
    },
    u'\U0001F4A3': {
        'en' : u':bomb:',
        'status' : fully_qualified,
        'E' : 0.6,
        'variant': True,
        'de': u':bombe:',
        'es': u':bomba:',
        'fr': u':bombe:',
        'pt': u':bomba:',
        'it': u':bomba:'
    },
    u'\U0001F9B4': {
        'en' : u':bone:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':knochen:',
        'es': u':hueso:',
        'fr': u':os:',
        'pt': u':osso:',
        'it': u':osso:'
    },
    u'\U0001F516': {
        'en' : u':bookmark:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':lesezeichen:',
        'es': u':marcapáginas:',
        'fr': u':marque-page:',
        'pt': u':marcador_de_página:',
        'it': u':segnalibro:'
    },
    u'\U0001F4D1': {
        'en' : u':bookmark_tabs:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':pagemarker:',
        'es': u':marcadores:',
        'fr': u':signets:',
        'pt': u':marcadores_de_página:',
        'it': u':etichette_segnalibro:'
    },
    u'\U0001F4DA': {
        'en' : u':books:',
        'status' : fully_qualified,
        'E' : 0.6,
        'variant': True,
        'de': u':bücherstapel:',
        'es': u':libros:',
        'fr': u':livres:',
        'pt': u':livros:',
        'it': u':libri:'
    },
    u'\U0001FA83': {
        'en' : u':boomerang:',
        'status' : fully_qualified,
        'E' : 13,
        'de': u':bumerang:',
        'es': u':bumerán:',
        'fr': u':boomerang:',
        'pt': u':bumerangue:',
        'it': u':boomerang:'
    },
    u'\U0001F37E': {
        'en' : u':bottle_with_popping_cork:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [u':champagne:'],
        'de': u':flasche_mit_knallendem_korken:',
        'es': u':botella_descorchada:',
        'fr': u':bouteille_de_champagne:',
        'pt': u':garrafa_de_champanhe:',
        'it': u':bottiglia_stappata:'
    },
    u'\U0001F490': {
        'en' : u':bouquet:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':blumenstrauß:',
        'es': u':ramo_de_flores:',
        'fr': u':bouquet:',
        'pt': u':buquê:',
        'it': u':mazzo_di_fiori:'
    },
    u'\U0001F3F9': {
        'en' : u':bow_and_arrow:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':pfeil_und_bogen:',
        'es': u':arco_y_flecha:',
        'fr': u':arc_et_flèche:',
        'pt': u':arco_e_flecha:',
        'it': u':arco_e_freccia:'
    },
    u'\U0001F963': {
        'en' : u':bowl_with_spoon:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':schüssel_mit_löffel:',
        'es': u':cuenco_con_cuchara:',
        'fr': u':bol_avec_cuillère:',
        'pt': u':tigela_com_colher:',
        'it': u':ciotola_con_cucchiaio:'
    },
    u'\U0001F3B3': {
        'en' : u':bowling:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':bowling:',
        'es': u':bolos:',
        'fr': u':bowling:',
        'pt': u':boliche:',
        'it': u':bowling:'
    },
    u'\U0001F94A': {
        'en' : u':boxing_glove:',
        'status' : fully_qualified,
        'E' : 3,
        'de': u':boxhandschuh:',
        'es': u':guante_de_boxeo:',
        'fr': u':gant_de_boxe:',
        'pt': u':luva_de_boxe:',
        'it': u':guantone_da_pugilato:'
    },
    u'\U0001F466': {
        'en' : u':boy:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':junge:',
        'es': u':niño:',
        'fr': u':garçon:',
        'pt': u':menino:',
        'it': u':bambino:'
    },
    u'\U0001F466\U0001F3FF': {
        'en' : u':boy_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':junge_dunkle_hautfarbe:',
        'es': u':niño_tono_de_piel_oscuro:',
        'fr': u':garçon_peau_foncée:',
        'pt': u':menino_pele_escura:',
        'it': u':bambino_carnagione_scura:'
    },
    u'\U0001F466\U0001F3FB': {
        'en' : u':boy_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':junge_helle_hautfarbe:',
        'es': u':niño_tono_de_piel_claro:',
        'fr': u':garçon_peau_claire:',
        'pt': u':menino_pele_clara:',
        'it': u':bambino_carnagione_chiara:'
    },
    u'\U0001F466\U0001F3FE': {
        'en' : u':boy_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':junge_mitteldunkle_hautfarbe:',
        'es': u':niño_tono_de_piel_oscuro_medio:',
        'fr': u':garçon_peau_mate:',
        'pt': u':menino_pele_morena_escura:',
        'it': u':bambino_carnagione_abbastanza_scura:'
    },
    u'\U0001F466\U0001F3FC': {
        'en' : u':boy_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':junge_mittelhelle_hautfarbe:',
        'es': u':niño_tono_de_piel_claro_medio:',
        'fr': u':garçon_peau_moyennement_claire:',
        'pt': u':menino_pele_morena_clara:',
        'it': u':bambino_carnagione_abbastanza_chiara:'
    },
    u'\U0001F466\U0001F3FD': {
        'en' : u':boy_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':junge_mittlere_hautfarbe:',
        'es': u':niño_tono_de_piel_medio:',
        'fr': u':garçon_peau_légèrement_mate:',
        'pt': u':menino_pele_morena:',
        'it': u':bambino_carnagione_olivastra:'
    },
    u'\U0001F9E0': {
        'en' : u':brain:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':gehirn:',
        'es': u':cerebro:',
        'fr': u':cerveau:',
        'pt': u':cérebro:',
        'it': u':cervello:'
    },
    u'\U0001F35E': {
        'en' : u':bread:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':brot:',
        'es': u':pan_de_molde:',
        'fr': u':pain:',
        'pt': u':pão:',
        'it': u':pane_in_cassetta:'
    },
    u'\U0001F931': {
        'en' : u':breast-feeding:',
        'status' : fully_qualified,
        'E' : 5,
        'alias' : [u':breast_feeding:'],
        'de': u':stillen:',
        'es': u':lactancia_materna:',
        'fr': u':allaitement:',
        'pt': u':amamentando:',
        'it': u':allattare:'
    },
    u'\U0001F931\U0001F3FF': {
        'en' : u':breast-feeding_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':stillen_dunkle_hautfarbe:',
        'es': u':lactancia_materna_tono_de_piel_oscuro:',
        'fr': u':allaitement_peau_foncée:',
        'pt': u':amamentando_pele_escura:',
        'it': u':allattare_carnagione_scura:'
    },
    u'\U0001F931\U0001F3FB': {
        'en' : u':breast-feeding_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':stillen_helle_hautfarbe:',
        'es': u':lactancia_materna_tono_de_piel_claro:',
        'fr': u':allaitement_peau_claire:',
        'pt': u':amamentando_pele_clara:',
        'it': u':allattare_carnagione_chiara:'
    },
    u'\U0001F931\U0001F3FE': {
        'en' : u':breast-feeding_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':stillen_mitteldunkle_hautfarbe:',
        'es': u':lactancia_materna_tono_de_piel_oscuro_medio:',
        'fr': u':allaitement_peau_mate:',
        'pt': u':amamentando_pele_morena_escura:',
        'it': u':allattare_carnagione_abbastanza_scura:'
    },
    u'\U0001F931\U0001F3FC': {
        'en' : u':breast-feeding_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':stillen_mittelhelle_hautfarbe:',
        'es': u':lactancia_materna_tono_de_piel_claro_medio:',
        'fr': u':allaitement_peau_moyennement_claire:',
        'pt': u':amamentando_pele_morena_clara:',
        'it': u':allattare_carnagione_abbastanza_chiara:'
    },
    u'\U0001F931\U0001F3FD': {
        'en' : u':breast-feeding_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':stillen_mittlere_hautfarbe:',
        'es': u':lactancia_materna_tono_de_piel_medio:',
        'fr': u':allaitement_peau_légèrement_mate:',
        'pt': u':amamentando_pele_morena:',
        'it': u':allattare_carnagione_olivastra:'
    },
    u'\U0001F9F1': {
        'en' : u':brick:',
        'status' : fully_qualified,
        'E' : 11,
        'alias' : [u':bricks:'],
        'de': u':ziegelstein:',
        'es': u':ladrillo:',
        'fr': u':brique:',
        'pt': u':tijolo:',
        'it': u':mattoni:'
    },
    u'\U0001F309': {
        'en' : u':bridge_at_night:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':brücke_vor_nachthimmel:',
        'es': u':puente_de_noche:',
        'fr': u':pont_de_nuit:',
        'pt': u':ponte_à_noite:',
        'it': u':ponte_di_notte:'
    },
    u'\U0001F4BC': {
        'en' : u':briefcase:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':aktentasche:',
        'es': u':maletín:',
        'fr': u':porte-documents:',
        'pt': u':maleta:',
        'it': u':valigetta_24_ore:'
    },
    u'\U0001FA72': {
        'en' : u':briefs:',
        'status' : fully_qualified,
        'E' : 12,
        'alias' : [u':swim_brief:'],
        'de': u':slip:',
        'es': u':ropa_interior:',
        'fr': u':slip:',
        'pt': u':cueca:',
        'it': u':slip:'
    },
    u'\U0001F506': {
        'en' : u':bright_button:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [u':high_brightness:'],
        'de': u':taste_heller:',
        'es': u':brillo_alto:',
        'fr': u':luminosité_élevée:',
        'pt': u':botão_de_aumentar_brilho:',
        'it': u':luminosità_elevata:'
    },
    u'\U0001F966': {
        'en' : u':broccoli:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':brokkoli:',
        'es': u':brócoli:',
        'fr': u':broccoli:',
        'pt': u':brócolis:',
        'it': u':broccoli:'
    },
    u'\U0001F494': {
        'en' : u':broken_heart:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':gebrochenes_herz:',
        'es': u':corazón_roto:',
        'fr': u':cœur_brisé:',
        'pt': u':coração_partido:',
        'it': u':cuore_infranto:'
    },
    u'\U0001F9F9': {
        'en' : u':broom:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':besen:',
        'es': u':escoba:',
        'fr': u':balai:',
        'pt': u':vassoura:',
        'it': u':scopa:'
    },
    u'\U0001F7E4': {
        'en' : u':brown_circle:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':brauner_kreis:',
        'es': u':círculo_marrón:',
        'fr': u':disque_marron:',
        'pt': u':círculo_marrom:',
        'it': u':cerchio_marrone:'
    },
    u'\U0001F90E': {
        'en' : u':brown_heart:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':braunes_herz:',
        'es': u':corazón_marrón:',
        'fr': u':cœur_marron:',
        'pt': u':coração_marrom:',
        'it': u':cuore_marrone:'
    },
    u'\U0001F7EB': {
        'en' : u':brown_square:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':braunes_viereck:',
        'es': u':cuadrado_marrón:',
        'fr': u':carré_marron:',
        'pt': u':quadrado_marrom:',
        'it': u':quadrato_marrone:'
    },
    u'\U0001F9CB': {
        'en' : u':bubble_tea:',
        'status' : fully_qualified,
        'E' : 13,
        'de': u':bubble_tea:',
        'es': u':té_de_burbujas:',
        'fr': u':thé_aux_perles:',
        'pt': u':chá_perolado:',
        'it': u':bubble_tea:'
    },
    u'\U0001FAE7': {
        'en' : u':bubbles:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001FAA3': {
        'en' : u':bucket:',
        'status' : fully_qualified,
        'E' : 13,
        'de': u':eimer:',
        'es': u':cubo:',
        'fr': u':seau:',
        'pt': u':balde:',
        'it': u':secchio:'
    },
    u'\U0001F41B': {
        'en' : u':bug:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':raupe:',
        'es': u':insecto:',
        'fr': u':chenille:',
        'pt': u':inseto:',
        'it': u':insetto:'
    },
    u'\U0001F3D7\U0000FE0F': {
        'en' : u':building_construction:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': u':kran:',
        'es': u':construcción:',
        'fr': u':construction_d’un_bâtiment:',
        'pt': u':construção:',
        'it': u':edificio_in_costruzione:'
    },
    u'\U0001F3D7': {
        'en' : u':building_construction:',
        'status' : unqualified,
        'E' : 0.7,
        'variant': True,
        'de': u':kran:',
        'es': u':construcción:',
        'fr': u':construction_d’un_bâtiment:',
        'pt': u':construção:',
        'it': u':edificio_in_costruzione:'
    },
    u'\U0001F685': {
        'en' : u':bullet_train:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':bullettrain_front:'],
        'de': u':hochgeschwindigkeitszug:',
        'es': u':tren_bala:',
        'fr': u':train_à_grande_vitesse:',
        'pt': u':trem_de_alta_velocidade_japonês:',
        'it': u':treno_alta_velocità_punta_arrotondata:'
    },
    u'\U0001F3AF': {
        'en' : u':bullseye:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':dart:'],
        'de': u':darts:',
        'es': u':diana:',
        'fr': u':dans_le_mille:',
        'pt': u':no_alvo:',
        'it': u':bersaglio:'
    },
    u'\U0001F32F': {
        'en' : u':burrito:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':burrito:',
        'es': u':burrito:',
        'fr': u':burrito:',
        'pt': u':burrito:',
        'it': u':burrito:'
    },
    u'\U0001F68C': {
        'en' : u':bus:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':bus:',
        'es': u':autobús:',
        'fr': u':bus:',
        'pt': u':ônibus:',
        'it': u':bus:'
    },
    u'\U0001F68F': {
        'en' : u':bus_stop:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':busstop:'],
        'de': u':bushaltestelle:',
        'es': u':parada_de_autobús:',
        'fr': u':arrêt_de_bus:',
        'pt': u':ponto_de_ônibus:',
        'it': u':fermata_dell’autobus:'
    },
    u'\U0001F464': {
        'en' : u':bust_in_silhouette:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':silhouette_einer_büste:',
        'es': u':silueta_de_busto:',
        'fr': u':silhouette_de_buste:',
        'pt': u':silhueta_de_busto:',
        'it': u':profilo_di_persona:'
    },
    u'\U0001F465': {
        'en' : u':busts_in_silhouette:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':silhouette_mehrerer_büsten:',
        'es': u':dos_siluetas_de_bustos:',
        'fr': u':silhouettes_de_bustes:',
        'pt': u':silhueta_de_bustos:',
        'it': u':profilo_di_due_persone:'
    },
    u'\U0001F9C8': {
        'en' : u':butter:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':butter:',
        'es': u':mantequilla:',
        'fr': u':beurre:',
        'pt': u':manteiga:',
        'it': u':burro:'
    },
    u'\U0001F98B': {
        'en' : u':butterfly:',
        'status' : fully_qualified,
        'E' : 3,
        'de': u':schmetterling:',
        'es': u':mariposa:',
        'fr': u':papillon:',
        'pt': u':borboleta:',
        'it': u':farfalla:'
    },
    u'\U0001F335': {
        'en' : u':cactus:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':kaktus:',
        'es': u':cactus:',
        'fr': u':cactus:',
        'pt': u':cacto:',
        'it': u':cactus:'
    },
    u'\U0001F4C5': {
        'en' : u':calendar:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':date:'],
        'de': u':kalender:',
        'es': u':calendario:',
        'fr': u':calendrier:',
        'pt': u':calendário:',
        'it': u':calendario:'
    },
    u'\U0001F919': {
        'en' : u':call_me_hand:',
        'status' : fully_qualified,
        'E' : 3,
        'de': u':ruf-mich-an-handzeichen:',
        'es': u':mano_haciendo_el_gesto_de_llamar:',
        'fr': u':signe_appel_téléphonique_avec_les_doigts:',
        'pt': u':sinal_me_liga:',
        'it': u':mano_con_gesto_di_chiamata:'
    },
    u'\U0001F919\U0001F3FF': {
        'en' : u':call_me_hand_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 3,
        'de': u':ruf-mich-an-handzeichen_dunkle_hautfarbe:',
        'es': u':mano_haciendo_el_gesto_de_llamar_tono_de_piel_oscuro:',
        'fr': u':signe_appel_téléphonique_avec_les_doigts_peau_foncée:',
        'pt': u':sinal_me_liga_pele_escura:',
        'it': u':mano_con_gesto_di_chiamata_carnagione_scura:'
    },
    u'\U0001F919\U0001F3FB': {
        'en' : u':call_me_hand_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 3,
        'de': u':ruf-mich-an-handzeichen_helle_hautfarbe:',
        'es': u':mano_haciendo_el_gesto_de_llamar_tono_de_piel_claro:',
        'fr': u':signe_appel_téléphonique_avec_les_doigts_peau_claire:',
        'pt': u':sinal_me_liga_pele_clara:',
        'it': u':mano_con_gesto_di_chiamata_carnagione_chiara:'
    },
    u'\U0001F919\U0001F3FE': {
        'en' : u':call_me_hand_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 3,
        'de': u':ruf-mich-an-handzeichen_mitteldunkle_hautfarbe:',
        'es': u':mano_haciendo_el_gesto_de_llamar_tono_de_piel_oscuro_medio:',
        'fr': u':signe_appel_téléphonique_avec_les_doigts_peau_mate:',
        'pt': u':sinal_me_liga_pele_morena_escura:',
        'it': u':mano_con_gesto_di_chiamata_carnagione_abbastanza_scura:'
    },
    u'\U0001F919\U0001F3FC': {
        'en' : u':call_me_hand_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 3,
        'de': u':ruf-mich-an-handzeichen_mittelhelle_hautfarbe:',
        'es': u':mano_haciendo_el_gesto_de_llamar_tono_de_piel_claro_medio:',
        'fr': u':signe_appel_téléphonique_avec_les_doigts_peau_moyennement_claire:',
        'pt': u':sinal_me_liga_pele_morena_clara:',
        'it': u':mano_con_gesto_di_chiamata_carnagione_abbastanza_chiara:'
    },
    u'\U0001F919\U0001F3FD': {
        'en' : u':call_me_hand_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 3,
        'de': u':ruf-mich-an-handzeichen_mittlere_hautfarbe:',
        'es': u':mano_haciendo_el_gesto_de_llamar_tono_de_piel_medio:',
        'fr': u':signe_appel_téléphonique_avec_les_doigts_peau_légèrement_mate:',
        'pt': u':sinal_me_liga_pele_morena:',
        'it': u':mano_con_gesto_di_chiamata_carnagione_olivastra:'
    },
    u'\U0001F42A': {
        'en' : u':camel:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [u':dromedary_camel:'],
        'de': u':dromedar:',
        'es': u':dromedario:',
        'fr': u':dromadaire:',
        'pt': u':camelo:',
        'it': u':dromedario:'
    },
    u'\U0001F4F7': {
        'en' : u':camera:',
        'status' : fully_qualified,
        'E' : 0.6,
        'variant': True,
        'de': u':fotoapparat:',
        'es': u':cámara_de_fotos:',
        'fr': u':appareil_photo:',
        'pt': u':câmera:',
        'it': u':fotocamera:'
    },
    u'\U0001F4F8': {
        'en' : u':camera_with_flash:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [u':camera_flash:'],
        'de': u':fotoapparat_mit_blitz:',
        'es': u':cámara_con_flash:',
        'fr': u':appareil_photo_avec_flash:',
        'pt': u':câmera_com_flash:',
        'it': u':fotocamera_con_flash:'
    },
    u'\U0001F3D5\U0000FE0F': {
        'en' : u':camping:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': u':camping:',
        'es': u':camping:',
        'fr': u':camping:',
        'pt': u':acampamento:',
        'it': u':campeggio:'
    },
    u'\U0001F3D5': {
        'en' : u':camping:',
        'status' : unqualified,
        'E' : 0.7,
        'variant': True,
        'de': u':camping:',
        'es': u':camping:',
        'fr': u':camping:',
        'pt': u':acampamento:',
        'it': u':campeggio:'
    },
    u'\U0001F56F\U0000FE0F': {
        'en' : u':candle:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': u':kerze:',
        'es': u':vela:',
        'fr': u':bougie:',
        'pt': u':vela:',
        'it': u':candela:'
    },
    u'\U0001F56F': {
        'en' : u':candle:',
        'status' : unqualified,
        'E' : 0.7,
        'variant': True,
        'de': u':kerze:',
        'es': u':vela:',
        'fr': u':bougie:',
        'pt': u':vela:',
        'it': u':candela:'
    },
    u'\U0001F36C': {
        'en' : u':candy:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':bonbon:',
        'es': u':caramelo:',
        'fr': u':bonbon:',
        'pt': u':bala:',
        'it': u':caramella:'
    },
    u'\U0001F96B': {
        'en' : u':canned_food:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':konserve:',
        'es': u':comida_enlatada:',
        'fr': u':aliments_en_conserve:',
        'pt': u':comida_enlatada:',
        'it': u':cibo_in_scatola:'
    },
    u'\U0001F6F6': {
        'en' : u':canoe:',
        'status' : fully_qualified,
        'E' : 3,
        'de': u':kanu:',
        'es': u':canoa:',
        'fr': u':canoë:',
        'pt': u':canoa:',
        'it': u':canoa:'
    },
    u'\U0001F5C3\U0000FE0F': {
        'en' : u':card_file_box:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': u':karteikasten:',
        'es': u':archivador_de_tarjetas:',
        'fr': u':boîte_à_dossiers:',
        'pt': u':caixa_de_arquivos:',
        'it': u':schedario_da_tavolo:'
    },
    u'\U0001F5C3': {
        'en' : u':card_file_box:',
        'status' : unqualified,
        'E' : 0.7,
        'variant': True,
        'de': u':karteikasten:',
        'es': u':archivador_de_tarjetas:',
        'fr': u':boîte_à_dossiers:',
        'pt': u':caixa_de_arquivos:',
        'it': u':schedario_da_tavolo:'
    },
    u'\U0001F4C7': {
        'en' : u':card_index:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':rotationskartei:',
        'es': u':organizador_de_fichas:',
        'fr': u':carnet_d’adresses:',
        'pt': u':índice_de_cartões:',
        'it': u':schedario:'
    },
    u'\U0001F5C2\U0000FE0F': {
        'en' : u':card_index_dividers:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': u':karteibahnen:',
        'es': u':separador_de_fichas:',
        'fr': u':intercalaires:',
        'pt': u':divisores_de_pastas:',
        'it': u':divisori_per_schedario:'
    },
    u'\U0001F5C2': {
        'en' : u':card_index_dividers:',
        'status' : unqualified,
        'E' : 0.7,
        'variant': True,
        'de': u':karteibahnen:',
        'es': u':separador_de_fichas:',
        'fr': u':intercalaires:',
        'pt': u':divisores_de_pastas:',
        'it': u':divisori_per_schedario:'
    },
    u'\U0001F3A0': {
        'en' : u':carousel_horse:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':karussellpferd:',
        'es': u':caballo_de_tiovivo:',
        'fr': u':cheval_de_manège:',
        'pt': u':carrossel:',
        'it': u':cavallo_da_giostra:'
    },
    u'\U0001F38F': {
        'en' : u':carp_streamer:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':flags:'],
        'de': u':traditionelle_japanische_windsäcke:',
        'es': u':banderín_de_carpas:',
        'fr': u':koinobori:',
        'pt': u':bandeira_de_carpas:',
        'it': u':aquilone_a_forma_di_carpa:'
    },
    u'\U0001FA9A': {
        'en' : u':carpentry_saw:',
        'status' : fully_qualified,
        'E' : 13,
        'de': u':handsäge:',
        'es': u':sierra_de_carpintería:',
        'fr': u':scie:',
        'pt': u':serrote:',
        'it': u':sega_da_falegname:'
    },
    u'\U0001F955': {
        'en' : u':carrot:',
        'status' : fully_qualified,
        'E' : 3,
        'de': u':karotte:',
        'es': u':zanahoria:',
        'fr': u':carotte:',
        'pt': u':cenoura:',
        'it': u':carota:'
    },
    u'\U0001F3F0': {
        'en' : u':castle:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':european_castle:'],
        'de': u':schloss:',
        'es': u':castillo_europeo:',
        'fr': u':château:',
        'pt': u':castelo:',
        'it': u':castello:'
    },
    u'\U0001F408': {
        'en' : u':cat:',
        'status' : fully_qualified,
        'E' : 0.7,
        'alias' : [u':cat2:'],
        'variant': True,
        'de': u':katze:',
        'es': u':gato:',
        'fr': u':chat:',
        'pt': u':gato:',
        'it': u':gatto:'
    },
    u'\U0001F431': {
        'en' : u':cat_face:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':cat:'],
        'de': u':katzengesicht:',
        'es': u':cara_de_gato:',
        'fr': u':tête_de_chat:',
        'pt': u':rosto_de_gato:',
        'it': u':muso_di_gatto:'
    },
    u'\U0001F639': {
        'en' : u':cat_with_tears_of_joy:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':joy_cat:'],
        'de': u':katze_mit_freudentränen:',
        'es': u':gato_llorando_de_risa:',
        'fr': u':chat_qui_pleure_de_joie:',
        'pt': u':rosto_de_gato_com_lágrimas_de_alegria:',
        'it': u':gatto_con_lacrime_di_gioia:'
    },
    u'\U0001F63C': {
        'en' : u':cat_with_wry_smile:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':smirk_cat:'],
        'de': u':verwegen_lächelnde_katze:',
        'es': u':gato_haciendo_una_mueca:',
        'fr': u':chat_avec_sourire_en_coin:',
        'pt': u':rosto_de_gato_com_sorriso_irônico:',
        'it': u':gatto_con_sorriso_sarcastico:'
    },
    u'\U000026D3\U0000FE0F': {
        'en' : u':chains:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': u':ketten:',
        'es': u':cadenas:',
        'fr': u':chaînes:',
        'pt': u':correntes:',
        'it': u':catene:'
    },
    u'\U000026D3': {
        'en' : u':chains:',
        'status' : unqualified,
        'E' : 0.7,
        'variant': True,
        'de': u':ketten:',
        'es': u':cadenas:',
        'fr': u':chaînes:',
        'pt': u':correntes:',
        'it': u':catene:'
    },
    u'\U0001FA91': {
        'en' : u':chair:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':stuhl:',
        'es': u':silla:',
        'fr': u':chaise:',
        'pt': u':cadeira:',
        'it': u':sedia:'
    },
    u'\U0001F4C9': {
        'en' : u':chart_decreasing:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':chart_with_downwards_trend:'],
        'de': u':abwärtstrend:',
        'es': u':gráfica_de_evolución_descendente:',
        'fr': u':graphique_en_baisse:',
        'pt': u':gráfico_caindo:',
        'it': u':grafico_con_andamento_negativo:'
    },
    u'\U0001F4C8': {
        'en' : u':chart_increasing:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':chart_with_upwards_trend:'],
        'de': u':aufwärtstrend:',
        'es': u':gráfica_de_evolución_ascendente:',
        'fr': u':graphique_en_hausse:',
        'pt': u':gráfico_subindo:',
        'it': u':grafico_con_andamento_positivo:'
    },
    u'\U0001F4B9': {
        'en' : u':chart_increasing_with_yen:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':chart:'],
        'de': u':steigender_trend_mit_yen-zeichen:',
        'es': u':mercado_al_alza:',
        'fr': u':courbe_avec_yen_en_hausse:',
        'pt': u':gráfico_subindo_com_iene:',
        'it': u':grafico_ascendente:'
    },
    u'\U00002611\U0000FE0F': {
        'en' : u':check_box_with_check:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':ballot_box_with_check:'],
        'variant': True,
        'de': u':abstimmungsfeld_mit_häkchen:',
        'es': u':casilla_con_marca_de_verificación:',
        'fr': u':case_cochée:',
        'pt': u':caixa_de_seleção_marcada_com_tique:',
        'it': u':riquadro_con_spunta:'
    },
    u'\U00002611': {
        'en' : u':check_box_with_check:',
        'status' : unqualified,
        'E' : 0.6,
        'alias' : [u':ballot_box_with_check:'],
        'variant': True,
        'de': u':abstimmungsfeld_mit_häkchen:',
        'es': u':casilla_con_marca_de_verificación:',
        'fr': u':case_cochée:',
        'pt': u':caixa_de_seleção_marcada_com_tique:',
        'it': u':riquadro_con_spunta:'
    },
    u'\U00002714\U0000FE0F': {
        'en' : u':check_mark:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':heavy_check_mark:'],
        'variant': True,
        'de': u':kräftiges_häkchen:',
        'es': u':marca_de_verificación:',
        'fr': u':coche:',
        'pt': u':marca_de_seleção:',
        'it': u':segno_di_spunta_nero:'
    },
    u'\U00002714': {
        'en' : u':check_mark:',
        'status' : unqualified,
        'E' : 0.6,
        'alias' : [u':heavy_check_mark:'],
        'variant': True,
        'de': u':kräftiges_häkchen:',
        'es': u':marca_de_verificación:',
        'fr': u':coche:',
        'pt': u':marca_de_seleção:',
        'it': u':segno_di_spunta_nero:'
    },
    u'\U00002705': {
        'en' : u':check_mark_button:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':white_check_mark:'],
        'de': u':weißes_häkchen:',
        'es': u':botón_de_marca_de_verificación:',
        'fr': u':bouton_coché:',
        'pt': u':marca_de_seleção_branca:',
        'it': u':segno_di_spunta:'
    },
    u'\U0001F9C0': {
        'en' : u':cheese_wedge:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [u':cheese:'],
        'de': u':käsestück:',
        'es': u':cuña_de_queso:',
        'fr': u':part_de_fromage:',
        'pt': u':queijo:',
        'it': u':fetta_di_formaggio:'
    },
    u'\U0001F3C1': {
        'en' : u':chequered_flag:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':checkered_flag:'],
        'de': u':zielflagge:',
        'es': u':bandera_de_cuadros:',
        'fr': u':drapeau_à_damier:',
        'pt': u':bandeira_quadriculada:',
        'it': u':bandiera_a_scacchi:'
    },
    u'\U0001F352': {
        'en' : u':cherries:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':kirschen:',
        'es': u':cerezas:',
        'fr': u':cerises:',
        'pt': u':cereja:',
        'it': u':ciliegie:'
    },
    u'\U0001F338': {
        'en' : u':cherry_blossom:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':kirschblüte:',
        'es': u':flor_de_cerezo:',
        'fr': u':fleur_de_cerisier:',
        'pt': u':flor_de_cerejeira:',
        'it': u':fiore_di_ciliegio:'
    },
    u'\U0000265F\U0000FE0F': {
        'en' : u':chess_pawn:',
        'status' : fully_qualified,
        'E' : 11,
        'variant': True,
        'de': u':bauer_schach:',
        'es': u':peón_de_ajedrez:',
        'fr': u':pion_d’échec:',
        'pt': u':peão_de_xadrez:',
        'it': u':pedina_degli_scacchi:'
    },
    u'\U0000265F': {
        'en' : u':chess_pawn:',
        'status' : unqualified,
        'E' : 11,
        'variant': True,
        'de': u':bauer_schach:',
        'es': u':peón_de_ajedrez:',
        'fr': u':pion_d’échec:',
        'pt': u':peão_de_xadrez:',
        'it': u':pedina_degli_scacchi:'
    },
    u'\U0001F330': {
        'en' : u':chestnut:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':kastanie:',
        'es': u':castaña:',
        'fr': u':châtaigne:',
        'pt': u':castanha:',
        'it': u':castagna:'
    },
    u'\U0001F414': {
        'en' : u':chicken:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':huhn:',
        'es': u':gallina:',
        'fr': u':poule:',
        'pt': u':galinha:',
        'it': u':gallina:'
    },
    u'\U0001F9D2': {
        'en' : u':child:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':kind:',
        'es': u':infante:',
        'fr': u':enfant:',
        'pt': u':criança:',
        'it': u':bimbo:'
    },
    u'\U0001F9D2\U0001F3FF': {
        'en' : u':child_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':kind_dunkle_hautfarbe:',
        'es': u':infante_tono_de_piel_oscuro:',
        'fr': u':enfant_peau_foncée:',
        'pt': u':criança_pele_escura:',
        'it': u':bimbo_carnagione_scura:'
    },
    u'\U0001F9D2\U0001F3FB': {
        'en' : u':child_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':kind_helle_hautfarbe:',
        'es': u':infante_tono_de_piel_claro:',
        'fr': u':enfant_peau_claire:',
        'pt': u':criança_pele_clara:',
        'it': u':bimbo_carnagione_chiara:'
    },
    u'\U0001F9D2\U0001F3FE': {
        'en' : u':child_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':kind_mitteldunkle_hautfarbe:',
        'es': u':infante_tono_de_piel_oscuro_medio:',
        'fr': u':enfant_peau_mate:',
        'pt': u':criança_pele_morena_escura:',
        'it': u':bimbo_carnagione_abbastanza_scura:'
    },
    u'\U0001F9D2\U0001F3FC': {
        'en' : u':child_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':kind_mittelhelle_hautfarbe:',
        'es': u':infante_tono_de_piel_claro_medio:',
        'fr': u':enfant_peau_moyennement_claire:',
        'pt': u':criança_pele_morena_clara:',
        'it': u':bimbo_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9D2\U0001F3FD': {
        'en' : u':child_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':kind_mittlere_hautfarbe:',
        'es': u':infante_tono_de_piel_medio:',
        'fr': u':enfant_peau_légèrement_mate:',
        'pt': u':criança_pele_morena:',
        'it': u':bimbo_carnagione_olivastra:'
    },
    u'\U0001F6B8': {
        'en' : u':children_crossing:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':kinder_überqueren_die_straße:',
        'es': u':niños_cruzando:',
        'fr': u':traversée_d’enfants:',
        'pt': u':crianças_atravessando:',
        'it': u':attraversamento_bambini:'
    },
    u'\U0001F43F\U0000FE0F': {
        'en' : u':chipmunk:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': u':streifenhörnchen:',
        'es': u':ardilla:',
        'fr': u':écureuil:',
        'pt': u':esquilo:',
        'it': u':scoiattolo:'
    },
    u'\U0001F43F': {
        'en' : u':chipmunk:',
        'status' : unqualified,
        'E' : 0.7,
        'variant': True,
        'de': u':streifenhörnchen:',
        'es': u':ardilla:',
        'fr': u':écureuil:',
        'pt': u':esquilo:',
        'it': u':scoiattolo:'
    },
    u'\U0001F36B': {
        'en' : u':chocolate_bar:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':schokoladentafel:',
        'es': u':tableta_de_chocolate:',
        'fr': u':barre_chocolatée:',
        'pt': u':chocolate:',
        'it': u':cioccolato:'
    },
    u'\U0001F962': {
        'en' : u':chopsticks:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':essstäbchen:',
        'es': u':palillos:',
        'fr': u':baguettes:',
        'pt': u':hashi:',
        'it': u':bacchette:'
    },
    u'\U000026EA': {
        'en' : u':church:',
        'status' : fully_qualified,
        'E' : 0.6,
        'variant': True,
        'de': u':kirche:',
        'es': u':iglesia:',
        'fr': u':église:',
        'pt': u':igreja:',
        'it': u':chiesa:'
    },
    u'\U0001F6AC': {
        'en' : u':cigarette:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':smoking:'],
        'de': u':zigarette:',
        'es': u':cigarrillo:',
        'fr': u':cigarette:',
        'pt': u':cigarro:',
        'it': u':sigaretta:'
    },
    u'\U0001F3A6': {
        'en' : u':cinema:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':kinosymbol:',
        'es': u':cine:',
        'fr': u':cinéma:',
        'pt': u':cinema:',
        'it': u':simbolo_del_cinema:'
    },
    u'\U000024C2\U0000FE0F': {
        'en' : u':circled_M:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':m:'],
        'variant': True,
        'de': u':buchstabe_m_in_kreis:',
        'es': u':m_en_círculo:',
        'fr': u':m_encerclé:',
        'pt': u':círculo_com_a_letra_m:',
        'it': u':pulsante_m_cerchiata:'
    },
    u'\U000024C2': {
        'en' : u':circled_M:',
        'status' : unqualified,
        'E' : 0.6,
        'alias' : [u':m:'],
        'variant': True,
        'de': u':buchstabe_m_in_kreis:',
        'es': u':m_en_círculo:',
        'fr': u':m_encerclé:',
        'pt': u':círculo_com_a_letra_m:',
        'it': u':pulsante_m_cerchiata:'
    },
    u'\U0001F3AA': {
        'en' : u':circus_tent:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':zirkuszelt:',
        'es': u':carpa_de_circo:',
        'fr': u':chapiteau:',
        'pt': u':circo:',
        'it': u':circo:'
    },
    u'\U0001F3D9\U0000FE0F': {
        'en' : u':cityscape:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': u':skyline:',
        'es': u':paisaje_urbano:',
        'fr': u':ville:',
        'pt': u':cidade:',
        'it': u':paesaggio_urbano:'
    },
    u'\U0001F3D9': {
        'en' : u':cityscape:',
        'status' : unqualified,
        'E' : 0.7,
        'variant': True,
        'de': u':skyline:',
        'es': u':paisaje_urbano:',
        'fr': u':ville:',
        'pt': u':cidade:',
        'it': u':paesaggio_urbano:'
    },
    u'\U0001F306': {
        'en' : u':cityscape_at_dusk:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':city_sunset:'],
        'de': u':abendstimmung_in_der_stadt:',
        'es': u':ciudad_al_atardecer:',
        'fr': u':ville_au_crépuscule:',
        'pt': u':cidade_ao_anoitecer:',
        'it': u':città_al_tramonto:'
    },
    u'\U0001F5DC\U0000FE0F': {
        'en' : u':clamp:',
        'status' : fully_qualified,
        'E' : 0.7,
        'alias' : [u':compression:'],
        'variant': True,
        'de': u':schraubzwinge:',
        'es': u':tornillo_de_banco:',
        'fr': u':serre-joint:',
        'pt': u':braçadeira:',
        'it': u':morsetto:'
    },
    u'\U0001F5DC': {
        'en' : u':clamp:',
        'status' : unqualified,
        'E' : 0.7,
        'alias' : [u':compression:'],
        'variant': True,
        'de': u':schraubzwinge:',
        'es': u':tornillo_de_banco:',
        'fr': u':serre-joint:',
        'pt': u':braçadeira:',
        'it': u':morsetto:'
    },
    u'\U0001F3AC': {
        'en' : u':clapper_board:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':clapper:'],
        'variant': True,
        'de': u':filmklappe:',
        'es': u':claqueta:',
        'fr': u':clap:',
        'pt': u':claquete:',
        'it': u':ciak:'
    },
    u'\U0001F44F': {
        'en' : u':clapping_hands:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':clap:'],
        'de': u':klatschende_hände:',
        'es': u':manos_aplaudiendo:',
        'fr': u':applaudissements:',
        'pt': u':mãos_aplaudindo:',
        'it': u':mani_che_applaudono:'
    },
    u'\U0001F44F\U0001F3FF': {
        'en' : u':clapping_hands_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':klatschende_hände_dunkle_hautfarbe:',
        'es': u':manos_aplaudiendo_tono_de_piel_oscuro:',
        'fr': u':applaudissements_peau_foncée:',
        'pt': u':mãos_aplaudindo_pele_escura:',
        'it': u':mani_che_applaudono_carnagione_scura:'
    },
    u'\U0001F44F\U0001F3FB': {
        'en' : u':clapping_hands_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':klatschende_hände_helle_hautfarbe:',
        'es': u':manos_aplaudiendo_tono_de_piel_claro:',
        'fr': u':applaudissements_peau_claire:',
        'pt': u':mãos_aplaudindo_pele_clara:',
        'it': u':mani_che_applaudono_carnagione_chiara:'
    },
    u'\U0001F44F\U0001F3FE': {
        'en' : u':clapping_hands_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':klatschende_hände_mitteldunkle_hautfarbe:',
        'es': u':manos_aplaudiendo_tono_de_piel_oscuro_medio:',
        'fr': u':applaudissements_peau_mate:',
        'pt': u':mãos_aplaudindo_pele_morena_escura:',
        'it': u':mani_che_applaudono_carnagione_abbastanza_scura:'
    },
    u'\U0001F44F\U0001F3FC': {
        'en' : u':clapping_hands_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':klatschende_hände_mittelhelle_hautfarbe:',
        'es': u':manos_aplaudiendo_tono_de_piel_claro_medio:',
        'fr': u':applaudissements_peau_moyennement_claire:',
        'pt': u':mãos_aplaudindo_pele_morena_clara:',
        'it': u':mani_che_applaudono_carnagione_abbastanza_chiara:'
    },
    u'\U0001F44F\U0001F3FD': {
        'en' : u':clapping_hands_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':klatschende_hände_mittlere_hautfarbe:',
        'es': u':manos_aplaudiendo_tono_de_piel_medio:',
        'fr': u':applaudissements_peau_légèrement_mate:',
        'pt': u':mãos_aplaudindo_pele_morena:',
        'it': u':mani_che_applaudono_carnagione_olivastra:'
    },
    u'\U0001F3DB\U0000FE0F': {
        'en' : u':classical_building:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': u':antikes_gebäude:',
        'es': u':edificio_clásico:',
        'fr': u':monument_classique:',
        'pt': u':prédio_grego:',
        'it': u':edificio_classico:'
    },
    u'\U0001F3DB': {
        'en' : u':classical_building:',
        'status' : unqualified,
        'E' : 0.7,
        'variant': True,
        'de': u':antikes_gebäude:',
        'es': u':edificio_clásico:',
        'fr': u':monument_classique:',
        'pt': u':prédio_grego:',
        'it': u':edificio_classico:'
    },
    u'\U0001F37B': {
        'en' : u':clinking_beer_mugs:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':beers:'],
        'de': u':bierkrüge:',
        'es': u':jarras_de_cerveza_brindando:',
        'fr': u':chopes_de_bière:',
        'pt': u':canecas_de_cerveja:',
        'it': u':boccali_di_birra:'
    },
    u'\U0001F942': {
        'en' : u':clinking_glasses:',
        'status' : fully_qualified,
        'E' : 3,
        'de': u':sektgläser:',
        'es': u':copas_brindando:',
        'fr': u':trinquer:',
        'pt': u':taças_brindando:',
        'it': u':brindisi:'
    },
    u'\U0001F4CB': {
        'en' : u':clipboard:',
        'status' : fully_qualified,
        'E' : 0.6,
        'variant': True,
        'de': u':zwischenablage:',
        'es': u':portapapeles:',
        'fr': u':porte-bloc:',
        'pt': u':prancheta:',
        'it': u':portablocco:'
    },
    u'\U0001F503': {
        'en' : u':clockwise_vertical_arrows:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':arrows_clockwise:'],
        'de': u':kreisförmige_pfeile_im_uhrzeigersinn:',
        'es': u':flechas_verticales_en_sentido_horario:',
        'fr': u':flèches_dans_le_sens_horaire:',
        'pt': u':setas_verticais_no_sentido_horário:',
        'it': u':frecce_verticali_che_ruotano_in_senso_orario:'
    },
    u'\U0001F4D5': {
        'en' : u':closed_book:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':geschlossenes_buch:',
        'es': u':libro_cerrado:',
        'fr': u':livre_fermé:',
        'pt': u':livro_fechado:',
        'it': u':libro_chiuso:'
    },
    u'\U0001F4EA': {
        'en' : u':closed_mailbox_with_lowered_flag:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':mailbox_closed:'],
        'variant': True,
        'de': u':geschlossener_briefkasten_ohne_post:',
        'es': u':buzón_cerrado_con_la_bandera_bajada:',
        'fr': u':boîte_aux_lettres_fermée_drapeau_baissé:',
        'pt': u':caixa_de_correio_fechada_com_bandeira_abaixada:',
        'it': u':cassetta_postale_chiusa_bandierina_abbassata:'
    },
    u'\U0001F4EB': {
        'en' : u':closed_mailbox_with_raised_flag:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':mailbox:'],
        'variant': True,
        'de': u':geschlossener_briefkasten_mit_post:',
        'es': u':buzón_cerrado_con_la_bandera_levantada:',
        'fr': u':boîte_aux_lettres_fermée_drapeau_levé:',
        'pt': u':caixa_de_correio_fechada_com_bandeira_levantada:',
        'it': u':cassetta_postale_chiusa_bandierina_alzata:'
    },
    u'\U0001F302': {
        'en' : u':closed_umbrella:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':geschlossener_regenschirm:',
        'es': u':paraguas_cerrado:',
        'fr': u':parapluie_fermé:',
        'pt': u':guarda-chuva_fechado:',
        'it': u':ombrello_chiuso:'
    },
    u'\U00002601\U0000FE0F': {
        'en' : u':cloud:',
        'status' : fully_qualified,
        'E' : 0.6,
        'variant': True,
        'de': u':wolke:',
        'es': u':nube:',
        'fr': u':nuage:',
        'pt': u':nuvem:',
        'it': u':nuvola:'
    },
    u'\U00002601': {
        'en' : u':cloud:',
        'status' : unqualified,
        'E' : 0.6,
        'variant': True,
        'de': u':wolke:',
        'es': u':nube:',
        'fr': u':nuage:',
        'pt': u':nuvem:',
        'it': u':nuvola:'
    },
    u'\U0001F329\U0000FE0F': {
        'en' : u':cloud_with_lightning:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': u':wolke_mit_blitz:',
        'es': u':nube_con_rayo:',
        'fr': u':nuage_avec_éclair:',
        'pt': u':nuvem_com_trovão:',
        'it': u':fulmini:'
    },
    u'\U0001F329': {
        'en' : u':cloud_with_lightning:',
        'status' : unqualified,
        'E' : 0.7,
        'variant': True,
        'de': u':wolke_mit_blitz:',
        'es': u':nube_con_rayo:',
        'fr': u':nuage_avec_éclair:',
        'pt': u':nuvem_com_trovão:',
        'it': u':fulmini:'
    },
    u'\U000026C8\U0000FE0F': {
        'en' : u':cloud_with_lightning_and_rain:',
        'status' : fully_qualified,
        'E' : 0.7,
        'alias' : [u':thunder_cloud_and_rain:'],
        'variant': True,
        'de': u':wolke_mit_blitz_und_regen:',
        'es': u':nube_con_rayo_y_lluvia:',
        'fr': u':nuage_avec_éclair_et_pluie:',
        'pt': u':chuva_com_trovão:',
        'it': u':temporale:'
    },
    u'\U000026C8': {
        'en' : u':cloud_with_lightning_and_rain:',
        'status' : unqualified,
        'E' : 0.7,
        'alias' : [u':thunder_cloud_and_rain:'],
        'variant': True,
        'de': u':wolke_mit_blitz_und_regen:',
        'es': u':nube_con_rayo_y_lluvia:',
        'fr': u':nuage_avec_éclair_et_pluie:',
        'pt': u':chuva_com_trovão:',
        'it': u':temporale:'
    },
    u'\U0001F327\U0000FE0F': {
        'en' : u':cloud_with_rain:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': u':wolke_mit_regen:',
        'es': u':nube_con_lluvia:',
        'fr': u':nuage_avec_pluie:',
        'pt': u':nuvem_com_chuva:',
        'it': u':pioggia:'
    },
    u'\U0001F327': {
        'en' : u':cloud_with_rain:',
        'status' : unqualified,
        'E' : 0.7,
        'variant': True,
        'de': u':wolke_mit_regen:',
        'es': u':nube_con_lluvia:',
        'fr': u':nuage_avec_pluie:',
        'pt': u':nuvem_com_chuva:',
        'it': u':pioggia:'
    },
    u'\U0001F328\U0000FE0F': {
        'en' : u':cloud_with_snow:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': u':wolke_mit_schnee:',
        'es': u':nube_con_nieve:',
        'fr': u':nuage_avec_neige:',
        'pt': u':nuvem_com_neve:',
        'it': u':neve:'
    },
    u'\U0001F328': {
        'en' : u':cloud_with_snow:',
        'status' : unqualified,
        'E' : 0.7,
        'variant': True,
        'de': u':wolke_mit_schnee:',
        'es': u':nube_con_nieve:',
        'fr': u':nuage_avec_neige:',
        'pt': u':nuvem_com_neve:',
        'it': u':neve:'
    },
    u'\U0001F921': {
        'en' : u':clown_face:',
        'status' : fully_qualified,
        'E' : 3,
        'de': u':clown-gesicht:',
        'es': u':cara_de_payaso:',
        'fr': u':visage_de_clown:',
        'pt': u':rosto_de_palhaço:',
        'it': u':faccina_pagliaccio:'
    },
    u'\U00002663\U0000FE0F': {
        'en' : u':club_suit:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':clubs:'],
        'variant': True,
        'de': u':kreuz:',
        'es': u':palo_de_tréboles:',
        'fr': u':trèfle_cartes:',
        'pt': u':naipe_de_paus:',
        'it': u':fiori:'
    },
    u'\U00002663': {
        'en' : u':club_suit:',
        'status' : unqualified,
        'E' : 0.6,
        'alias' : [u':clubs:'],
        'variant': True,
        'de': u':kreuz:',
        'es': u':palo_de_tréboles:',
        'fr': u':trèfle_cartes:',
        'pt': u':naipe_de_paus:',
        'it': u':fiori:'
    },
    u'\U0001F45D': {
        'en' : u':clutch_bag:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':pouch:'],
        'de': u':clutch:',
        'es': u':bolso_de_mano:',
        'fr': u':pochette:',
        'pt': u':bolsa_pequena:',
        'it': u':pochette:'
    },
    u'\U0001F9E5': {
        'en' : u':coat:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':mantel:',
        'es': u':abrigo:',
        'fr': u':manteau:',
        'pt': u':casaco:',
        'it': u':cappotto:'
    },
    u'\U0001FAB3': {
        'en' : u':cockroach:',
        'status' : fully_qualified,
        'E' : 13,
        'de': u':kakerlake:',
        'es': u':cucaracha:',
        'fr': u':cafard:',
        'pt': u':barata:',
        'it': u':scarafaggio:'
    },
    u'\U0001F378': {
        'en' : u':cocktail_glass:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':cocktail:'],
        'variant': True,
        'de': u':cocktailglas:',
        'es': u':copa_de_cóctel:',
        'fr': u':cocktail:',
        'pt': u':coquetel:',
        'it': u':cocktail:'
    },
    u'\U0001F965': {
        'en' : u':coconut:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':kokosnuss:',
        'es': u':coco:',
        'fr': u':noix_de_coco:',
        'pt': u':coco:',
        'it': u':cocco:'
    },
    u'\U000026B0\U0000FE0F': {
        'en' : u':coffin:',
        'status' : fully_qualified,
        'E' : 1,
        'variant': True,
        'de': u':sarg:',
        'es': u':ataúd:',
        'fr': u':cercueil:',
        'pt': u':caixão:',
        'it': u':bara:'
    },
    u'\U000026B0': {
        'en' : u':coffin:',
        'status' : unqualified,
        'E' : 1,
        'variant': True,
        'de': u':sarg:',
        'es': u':ataúd:',
        'fr': u':cercueil:',
        'pt': u':caixão:',
        'it': u':bara:'
    },
    u'\U0001FA99': {
        'en' : u':coin:',
        'status' : fully_qualified,
        'E' : 13,
        'de': u':münze:',
        'es': u':moneda:',
        'fr': u':pièce:',
        'pt': u':moeda:',
        'it': u':moneta:'
    },
    u'\U0001F976': {
        'en' : u':cold_face:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':frierendes_gesicht:',
        'es': u':cara_con_frío:',
        'fr': u':visage_bleu_et_froid:',
        'pt': u':rosto_gelado:',
        'it': u':faccina_congelata:'
    },
    u'\U0001F4A5': {
        'en' : u':collision:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':boom:'],
        'de': u':zusammenstoß:',
        'es': u':colisión:',
        'fr': u':explosion:',
        'pt': u':colisão:',
        'it': u':collisione:'
    },
    u'\U00002604\U0000FE0F': {
        'en' : u':comet:',
        'status' : fully_qualified,
        'E' : 1,
        'variant': True,
        'de': u':komet:',
        'es': u':meteorito:',
        'fr': u':comète:',
        'pt': u':cometa:',
        'it': u':cometa:'
    },
    u'\U00002604': {
        'en' : u':comet:',
        'status' : unqualified,
        'E' : 1,
        'variant': True,
        'de': u':komet:',
        'es': u':meteorito:',
        'fr': u':comète:',
        'pt': u':cometa:',
        'it': u':cometa:'
    },
    u'\U0001F9ED': {
        'en' : u':compass:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':kompass:',
        'es': u':brújula:',
        'fr': u':boussole:',
        'pt': u':bússola:',
        'it': u':bussola:'
    },
    u'\U0001F4BD': {
        'en' : u':computer_disk:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':minidisc:'],
        'de': u':minidisc:',
        'es': u':minidisc:',
        'fr': u':disque_d’ordinateur:',
        'pt': u':minidisc:',
        'it': u':minidisc:'
    },
    u'\U0001F5B1\U0000FE0F': {
        'en' : u':computer_mouse:',
        'status' : fully_qualified,
        'E' : 0.7,
        'alias' : [u':three_button_mouse:'],
        'variant': True,
        'de': u':computermaus:',
        'es': u':ratón_de_ordenador:',
        'fr': u':souris_d’ordinateur:',
        'pt': u':mouse:',
        'it': u':mouse:'
    },
    u'\U0001F5B1': {
        'en' : u':computer_mouse:',
        'status' : unqualified,
        'E' : 0.7,
        'alias' : [u':three_button_mouse:'],
        'variant': True,
        'de': u':computermaus:',
        'es': u':ratón_de_ordenador:',
        'fr': u':souris_d’ordinateur:',
        'pt': u':mouse:',
        'it': u':mouse:'
    },
    u'\U0001F38A': {
        'en' : u':confetti_ball:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':konfettiball:',
        'es': u':bola_de_confeti:',
        'fr': u':confettis:',
        'pt': u':confete:',
        'it': u':coriandoli:'
    },
    u'\U0001F616': {
        'en' : u':confounded_face:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':confounded:'],
        'de': u':verwirrtes_gesicht:',
        'es': u':cara_de_frustración:',
        'fr': u':visage_déconcerté:',
        'pt': u':rosto_perplexo:',
        'it': u':faccina_frustrata:'
    },
    u'\U0001F615': {
        'en' : u':confused_face:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [u':confused:'],
        'de': u':verwundertes_gesicht:',
        'es': u':cara_de_confusión:',
        'fr': u':visage_confus:',
        'pt': u':rosto_confuso:',
        'it': u':faccina_confusa:'
    },
    u'\U0001F6A7': {
        'en' : u':construction:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':baustellenschild:',
        'es': u':obras:',
        'fr': u':travaux:',
        'pt': u':em_construção:',
        'it': u':area_cantiere:'
    },
    u'\U0001F477': {
        'en' : u':construction_worker:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':bauarbeiter(in):',
        'es': u':profesional_de_la_construcción:',
        'fr': u':personnel_du_bâtiment:',
        'pt': u':trabalhador_de_construção_civil:',
        'it': u':operaio_edile:'
    },
    u'\U0001F477\U0001F3FF': {
        'en' : u':construction_worker_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':bauarbeiter(in)_dunkle_hautfarbe:',
        'es': u':profesional_de_la_construcción_tono_de_piel_oscuro:',
        'fr': u':personnel_du_bâtiment_peau_foncée:',
        'pt': u':trabalhador_de_construção_civil_pele_escura:',
        'it': u':operaio_edile_carnagione_scura:'
    },
    u'\U0001F477\U0001F3FB': {
        'en' : u':construction_worker_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':bauarbeiter(in)_helle_hautfarbe:',
        'es': u':profesional_de_la_construcción_tono_de_piel_claro:',
        'fr': u':personnel_du_bâtiment_peau_claire:',
        'pt': u':trabalhador_de_construção_civil_pele_clara:',
        'it': u':operaio_edile_carnagione_chiara:'
    },
    u'\U0001F477\U0001F3FE': {
        'en' : u':construction_worker_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':bauarbeiter(in)_mitteldunkle_hautfarbe:',
        'es': u':profesional_de_la_construcción_tono_de_piel_oscuro_medio:',
        'fr': u':personnel_du_bâtiment_peau_mate:',
        'pt': u':trabalhador_de_construção_civil_pele_morena_escura:',
        'it': u':operaio_edile_carnagione_abbastanza_scura:'
    },
    u'\U0001F477\U0001F3FC': {
        'en' : u':construction_worker_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':bauarbeiter(in)_mittelhelle_hautfarbe:',
        'es': u':profesional_de_la_construcción_tono_de_piel_claro_medio:',
        'fr': u':personnel_du_bâtiment_peau_moyennement_claire:',
        'pt': u':trabalhador_de_construção_civil_pele_morena_clara:',
        'it': u':operaio_edile_carnagione_abbastanza_chiara:'
    },
    u'\U0001F477\U0001F3FD': {
        'en' : u':construction_worker_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':bauarbeiter(in)_mittlere_hautfarbe:',
        'es': u':profesional_de_la_construcción_tono_de_piel_medio:',
        'fr': u':personnel_du_bâtiment_peau_légèrement_mate:',
        'pt': u':trabalhador_de_construção_civil_pele_morena:',
        'it': u':operaio_edile_carnagione_olivastra:'
    },
    u'\U0001F39B\U0000FE0F': {
        'en' : u':control_knobs:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': u':drehregler:',
        'es': u':ruedas_de_control:',
        'fr': u':boutons_de_réglage:',
        'pt': u':botões_giratórios:',
        'it': u':manopole_di_controllo:'
    },
    u'\U0001F39B': {
        'en' : u':control_knobs:',
        'status' : unqualified,
        'E' : 0.7,
        'variant': True,
        'de': u':drehregler:',
        'es': u':ruedas_de_control:',
        'fr': u':boutons_de_réglage:',
        'pt': u':botões_giratórios:',
        'it': u':manopole_di_controllo:'
    },
    u'\U0001F3EA': {
        'en' : u':convenience_store:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':bedarfsartikelgeschäft:',
        'es': u':tienda_24_horas:',
        'fr': u':supérette:',
        'pt': u':loja_de_conveniência:',
        'it': u':minimarket:'
    },
    u'\U0001F9D1\U0000200D\U0001F373': {
        'en' : u':cook:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':koch/köchin:',
        'es': u':chef:',
        'fr': u':cuisinier_(tous_genres):',
        'pt': u':chef_de_cozinha:',
        'it': u':persona_che_cucina:'
    },
    u'\U0001F9D1\U0001F3FF\U0000200D\U0001F373': {
        'en' : u':cook_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':koch/köchin_dunkle_hautfarbe:',
        'es': u':chef_tono_de_piel_oscuro:',
        'fr': u':cuisinier_(tous_genres)_peau_foncée:',
        'pt': u':chef_de_cozinha_pele_escura:',
        'it': u':persona_che_cucina_carnagione_scura:'
    },
    u'\U0001F9D1\U0001F3FB\U0000200D\U0001F373': {
        'en' : u':cook_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':koch/köchin_helle_hautfarbe:',
        'es': u':chef_tono_de_piel_claro:',
        'fr': u':cuisinier_(tous_genres)_peau_claire:',
        'pt': u':chef_de_cozinha_pele_clara:',
        'it': u':persona_che_cucina_carnagione_chiara:'
    },
    u'\U0001F9D1\U0001F3FE\U0000200D\U0001F373': {
        'en' : u':cook_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':koch/köchin_mitteldunkle_hautfarbe:',
        'es': u':chef_tono_de_piel_oscuro_medio:',
        'fr': u':cuisinier_(tous_genres)_peau_mate:',
        'pt': u':chef_de_cozinha_pele_morena_escura:',
        'it': u':persona_che_cucina_carnagione_abbastanza_scura:'
    },
    u'\U0001F9D1\U0001F3FC\U0000200D\U0001F373': {
        'en' : u':cook_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':koch/köchin_mittelhelle_hautfarbe:',
        'es': u':chef_tono_de_piel_claro_medio:',
        'fr': u':cuisinier_(tous_genres)_peau_moyennement_claire:',
        'pt': u':chef_de_cozinha_pele_morena_clara:',
        'it': u':persona_che_cucina_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9D1\U0001F3FD\U0000200D\U0001F373': {
        'en' : u':cook_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':koch/köchin_mittlere_hautfarbe:',
        'es': u':chef_tono_de_piel_medio:',
        'fr': u':cuisinier_(tous_genres)_peau_légèrement_mate:',
        'pt': u':chef_de_cozinha_pele_morena:',
        'it': u':persona_che_cucina_carnagione_olivastra:'
    },
    u'\U0001F35A': {
        'en' : u':cooked_rice:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':rice:'],
        'de': u':reis_in_schüssel:',
        'es': u':arroz_cocido:',
        'fr': u':bol_de_riz:',
        'pt': u':arroz_cozido:',
        'it': u':riso_bollito:'
    },
    u'\U0001F36A': {
        'en' : u':cookie:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':keks:',
        'es': u':galleta:',
        'fr': u':cookie:',
        'pt': u':biscoito:',
        'it': u':biscotto:'
    },
    u'\U0001F373': {
        'en' : u':cooking:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':egg:', u':fried_egg:'],
        'de': u':spiegelei_in_bratpfanne:',
        'es': u':cocinar:',
        'fr': u':œuf_au_plat:',
        'pt': u':ovo_frito:',
        'it': u':cucinare:'
    },
    u'\U000000A9\U0000FE0F': {
        'en' : u':copyright:',
        'status' : fully_qualified,
        'E' : 0.6,
        'variant': True,
        'de': u':copyright:',
        'es': u':copyright:',
        'fr': u':symbole_copyright:',
        'pt': u':símbolo_de_copyright:',
        'it': u':copyright:'
    },
    u'\U000000A9': {
        'en' : u':copyright:',
        'status' : unqualified,
        'E' : 0.6,
        'variant': True,
        'de': u':copyright:',
        'es': u':copyright:',
        'fr': u':symbole_copyright:',
        'pt': u':símbolo_de_copyright:',
        'it': u':copyright:'
    },
    u'\U0001FAB8': {
        'en' : u':coral:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001F6CB\U0000FE0F': {
        'en' : u':couch_and_lamp:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': u':sofa_und_lampe:',
        'es': u':sofá_y_lámpara:',
        'fr': u':canapé_et_lampe:',
        'pt': u':sofá_e_luminária:',
        'it': u':divano_e_lampada:'
    },
    u'\U0001F6CB': {
        'en' : u':couch_and_lamp:',
        'status' : unqualified,
        'E' : 0.7,
        'variant': True,
        'de': u':sofa_und_lampe:',
        'es': u':sofá_y_lámpara:',
        'fr': u':canapé_et_lampe:',
        'pt': u':sofá_e_luminária:',
        'it': u':divano_e_lampada:'
    },
    u'\U0001F504': {
        'en' : u':counterclockwise_arrows_button:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [u':arrows_counterclockwise:'],
        'de': u':kreisförmige_pfeile_gegen_den_uhrzeigersinn:',
        'es': u':flechas_en_sentido_antihorario:',
        'fr': u':flèches_dans_le_sens_antihoraire:',
        'pt': u':botão_de_setas_em_sentido_anti-horário:',
        'it': u':pulsante_con_frecce_che_ruotano_in_senso_antiorario:'
    },
    u'\U0001F491': {
        'en' : u':couple_with_heart:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':liebespaar:',
        'es': u':pareja_enamorada:',
        'fr': u':couple_avec_cœur:',
        'pt': u':casal_apaixonado:',
        'it': u':coppia_con_cuore:'
    },
    u'\U0001F491\U0001F3FF': {
        'en' : u':couple_with_heart_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F491\U0001F3FB': {
        'en' : u':couple_with_heart_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468': {
        'en' : u':couple_with_heart_man_man:',
        'status' : fully_qualified,
        'E' : 2,
        'de': u':liebespaar_mann,_mann:',
        'es': u':pareja_enamorada_hombre_y_hombre:',
        'fr': u':couple_avec_cœur_homme_et_homme:',
        'pt': u':casal_apaixonado_homem_e_homem:',
        'it': u':coppia_con_cuore_uomo_e_uomo:'
    },
    u'\U0001F468\U0000200D\U00002764\U0000200D\U0001F468': {
        'en' : u':couple_with_heart_man_man:',
        'status' : minimally_qualified,
        'E' : 2
    },
    u'\U0001F468\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FF': {
        'en' : u':couple_with_heart_man_man_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FF': {
        'en' : u':couple_with_heart_man_man_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FB': {
        'en' : u':couple_with_heart_man_man_dark_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FB': {
        'en' : u':couple_with_heart_man_man_dark_skin_tone_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FE': {
        'en' : u':couple_with_heart_man_man_dark_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FE': {
        'en' : u':couple_with_heart_man_man_dark_skin_tone_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FC': {
        'en' : u':couple_with_heart_man_man_dark_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FC': {
        'en' : u':couple_with_heart_man_man_dark_skin_tone_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FD': {
        'en' : u':couple_with_heart_man_man_dark_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FD': {
        'en' : u':couple_with_heart_man_man_dark_skin_tone_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FB': {
        'en' : u':couple_with_heart_man_man_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FB': {
        'en' : u':couple_with_heart_man_man_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FF': {
        'en' : u':couple_with_heart_man_man_light_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FF': {
        'en' : u':couple_with_heart_man_man_light_skin_tone_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FE': {
        'en' : u':couple_with_heart_man_man_light_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FE': {
        'en' : u':couple_with_heart_man_man_light_skin_tone_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FC': {
        'en' : u':couple_with_heart_man_man_light_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FC': {
        'en' : u':couple_with_heart_man_man_light_skin_tone_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FD': {
        'en' : u':couple_with_heart_man_man_light_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FD': {
        'en' : u':couple_with_heart_man_man_light_skin_tone_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FE': {
        'en' : u':couple_with_heart_man_man_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FE': {
        'en' : u':couple_with_heart_man_man_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FF': {
        'en' : u':couple_with_heart_man_man_medium-dark_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FF': {
        'en' : u':couple_with_heart_man_man_medium-dark_skin_tone_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FB': {
        'en' : u':couple_with_heart_man_man_medium-dark_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FB': {
        'en' : u':couple_with_heart_man_man_medium-dark_skin_tone_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FC': {
        'en' : u':couple_with_heart_man_man_medium-dark_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FC': {
        'en' : u':couple_with_heart_man_man_medium-dark_skin_tone_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FD': {
        'en' : u':couple_with_heart_man_man_medium-dark_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FD': {
        'en' : u':couple_with_heart_man_man_medium-dark_skin_tone_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FC': {
        'en' : u':couple_with_heart_man_man_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FC': {
        'en' : u':couple_with_heart_man_man_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FF': {
        'en' : u':couple_with_heart_man_man_medium-light_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FF': {
        'en' : u':couple_with_heart_man_man_medium-light_skin_tone_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FB': {
        'en' : u':couple_with_heart_man_man_medium-light_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FB': {
        'en' : u':couple_with_heart_man_man_medium-light_skin_tone_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FE': {
        'en' : u':couple_with_heart_man_man_medium-light_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FE': {
        'en' : u':couple_with_heart_man_man_medium-light_skin_tone_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FD': {
        'en' : u':couple_with_heart_man_man_medium-light_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FD': {
        'en' : u':couple_with_heart_man_man_medium-light_skin_tone_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FD': {
        'en' : u':couple_with_heart_man_man_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FD': {
        'en' : u':couple_with_heart_man_man_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FF': {
        'en' : u':couple_with_heart_man_man_medium_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FF': {
        'en' : u':couple_with_heart_man_man_medium_skin_tone_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FB': {
        'en' : u':couple_with_heart_man_man_medium_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FB': {
        'en' : u':couple_with_heart_man_man_medium_skin_tone_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FE': {
        'en' : u':couple_with_heart_man_man_medium_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FE': {
        'en' : u':couple_with_heart_man_man_medium_skin_tone_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FC': {
        'en' : u':couple_with_heart_man_man_medium_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FC': {
        'en' : u':couple_with_heart_man_man_medium_skin_tone_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F491\U0001F3FE': {
        'en' : u':couple_with_heart_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F491\U0001F3FC': {
        'en' : u':couple_with_heart_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F491\U0001F3FD': {
        'en' : u':couple_with_heart_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F9D1\U0001F3FB': {
        'en' : u':couple_with_heart_person_person_dark_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F9D1\U0001F3FB': {
        'en' : u':couple_with_heart_person_person_dark_skin_tone_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F9D1\U0001F3FE': {
        'en' : u':couple_with_heart_person_person_dark_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F9D1\U0001F3FE': {
        'en' : u':couple_with_heart_person_person_dark_skin_tone_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F9D1\U0001F3FC': {
        'en' : u':couple_with_heart_person_person_dark_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F9D1\U0001F3FC': {
        'en' : u':couple_with_heart_person_person_dark_skin_tone_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F9D1\U0001F3FD': {
        'en' : u':couple_with_heart_person_person_dark_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F9D1\U0001F3FD': {
        'en' : u':couple_with_heart_person_person_dark_skin_tone_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F9D1\U0001F3FF': {
        'en' : u':couple_with_heart_person_person_light_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F9D1\U0001F3FF': {
        'en' : u':couple_with_heart_person_person_light_skin_tone_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F9D1\U0001F3FE': {
        'en' : u':couple_with_heart_person_person_light_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F9D1\U0001F3FE': {
        'en' : u':couple_with_heart_person_person_light_skin_tone_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F9D1\U0001F3FC': {
        'en' : u':couple_with_heart_person_person_light_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F9D1\U0001F3FC': {
        'en' : u':couple_with_heart_person_person_light_skin_tone_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F9D1\U0001F3FD': {
        'en' : u':couple_with_heart_person_person_light_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F9D1\U0001F3FD': {
        'en' : u':couple_with_heart_person_person_light_skin_tone_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F9D1\U0001F3FF': {
        'en' : u':couple_with_heart_person_person_medium-dark_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F9D1\U0001F3FF': {
        'en' : u':couple_with_heart_person_person_medium-dark_skin_tone_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F9D1\U0001F3FB': {
        'en' : u':couple_with_heart_person_person_medium-dark_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F9D1\U0001F3FB': {
        'en' : u':couple_with_heart_person_person_medium-dark_skin_tone_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F9D1\U0001F3FC': {
        'en' : u':couple_with_heart_person_person_medium-dark_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F9D1\U0001F3FC': {
        'en' : u':couple_with_heart_person_person_medium-dark_skin_tone_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F9D1\U0001F3FD': {
        'en' : u':couple_with_heart_person_person_medium-dark_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F9D1\U0001F3FD': {
        'en' : u':couple_with_heart_person_person_medium-dark_skin_tone_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F9D1\U0001F3FF': {
        'en' : u':couple_with_heart_person_person_medium-light_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F9D1\U0001F3FF': {
        'en' : u':couple_with_heart_person_person_medium-light_skin_tone_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F9D1\U0001F3FB': {
        'en' : u':couple_with_heart_person_person_medium-light_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F9D1\U0001F3FB': {
        'en' : u':couple_with_heart_person_person_medium-light_skin_tone_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F9D1\U0001F3FE': {
        'en' : u':couple_with_heart_person_person_medium-light_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F9D1\U0001F3FE': {
        'en' : u':couple_with_heart_person_person_medium-light_skin_tone_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F9D1\U0001F3FD': {
        'en' : u':couple_with_heart_person_person_medium-light_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F9D1\U0001F3FD': {
        'en' : u':couple_with_heart_person_person_medium-light_skin_tone_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F9D1\U0001F3FF': {
        'en' : u':couple_with_heart_person_person_medium_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F9D1\U0001F3FF': {
        'en' : u':couple_with_heart_person_person_medium_skin_tone_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F9D1\U0001F3FB': {
        'en' : u':couple_with_heart_person_person_medium_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F9D1\U0001F3FB': {
        'en' : u':couple_with_heart_person_person_medium_skin_tone_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F9D1\U0001F3FE': {
        'en' : u':couple_with_heart_person_person_medium_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F9D1\U0001F3FE': {
        'en' : u':couple_with_heart_person_person_medium_skin_tone_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F9D1\U0001F3FC': {
        'en' : u':couple_with_heart_person_person_medium_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F9D1\U0001F3FC': {
        'en' : u':couple_with_heart_person_person_medium_skin_tone_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468': {
        'en' : u':couple_with_heart_woman_man:',
        'status' : fully_qualified,
        'E' : 2,
        'de': u':liebespaar_frau,_mann:',
        'es': u':pareja_enamorada_mujer_y_hombre:',
        'fr': u':couple_avec_cœur_femme_et_homme:',
        'pt': u':casal_apaixonado_mulher_e_homem:',
        'it': u':coppia_con_cuore_donna_e_uomo:'
    },
    u'\U0001F469\U0000200D\U00002764\U0000200D\U0001F468': {
        'en' : u':couple_with_heart_woman_man:',
        'status' : minimally_qualified,
        'E' : 2
    },
    u'\U0001F469\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FF': {
        'en' : u':couple_with_heart_woman_man_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FF': {
        'en' : u':couple_with_heart_woman_man_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FB': {
        'en' : u':couple_with_heart_woman_man_dark_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FB': {
        'en' : u':couple_with_heart_woman_man_dark_skin_tone_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FE': {
        'en' : u':couple_with_heart_woman_man_dark_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FE': {
        'en' : u':couple_with_heart_woman_man_dark_skin_tone_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FC': {
        'en' : u':couple_with_heart_woman_man_dark_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FC': {
        'en' : u':couple_with_heart_woman_man_dark_skin_tone_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FD': {
        'en' : u':couple_with_heart_woman_man_dark_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FD': {
        'en' : u':couple_with_heart_woman_man_dark_skin_tone_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FB': {
        'en' : u':couple_with_heart_woman_man_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FB': {
        'en' : u':couple_with_heart_woman_man_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FF': {
        'en' : u':couple_with_heart_woman_man_light_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FF': {
        'en' : u':couple_with_heart_woman_man_light_skin_tone_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FE': {
        'en' : u':couple_with_heart_woman_man_light_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FE': {
        'en' : u':couple_with_heart_woman_man_light_skin_tone_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FC': {
        'en' : u':couple_with_heart_woman_man_light_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FC': {
        'en' : u':couple_with_heart_woman_man_light_skin_tone_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FD': {
        'en' : u':couple_with_heart_woman_man_light_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FD': {
        'en' : u':couple_with_heart_woman_man_light_skin_tone_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FE': {
        'en' : u':couple_with_heart_woman_man_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FE': {
        'en' : u':couple_with_heart_woman_man_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FF': {
        'en' : u':couple_with_heart_woman_man_medium-dark_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FF': {
        'en' : u':couple_with_heart_woman_man_medium-dark_skin_tone_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FB': {
        'en' : u':couple_with_heart_woman_man_medium-dark_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FB': {
        'en' : u':couple_with_heart_woman_man_medium-dark_skin_tone_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FC': {
        'en' : u':couple_with_heart_woman_man_medium-dark_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FC': {
        'en' : u':couple_with_heart_woman_man_medium-dark_skin_tone_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FD': {
        'en' : u':couple_with_heart_woman_man_medium-dark_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FD': {
        'en' : u':couple_with_heart_woman_man_medium-dark_skin_tone_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FC': {
        'en' : u':couple_with_heart_woman_man_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FC': {
        'en' : u':couple_with_heart_woman_man_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FF': {
        'en' : u':couple_with_heart_woman_man_medium-light_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FF': {
        'en' : u':couple_with_heart_woman_man_medium-light_skin_tone_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FB': {
        'en' : u':couple_with_heart_woman_man_medium-light_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FB': {
        'en' : u':couple_with_heart_woman_man_medium-light_skin_tone_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FE': {
        'en' : u':couple_with_heart_woman_man_medium-light_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FE': {
        'en' : u':couple_with_heart_woman_man_medium-light_skin_tone_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FD': {
        'en' : u':couple_with_heart_woman_man_medium-light_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FD': {
        'en' : u':couple_with_heart_woman_man_medium-light_skin_tone_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FD': {
        'en' : u':couple_with_heart_woman_man_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FD': {
        'en' : u':couple_with_heart_woman_man_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FF': {
        'en' : u':couple_with_heart_woman_man_medium_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FF': {
        'en' : u':couple_with_heart_woman_man_medium_skin_tone_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FB': {
        'en' : u':couple_with_heart_woman_man_medium_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FB': {
        'en' : u':couple_with_heart_woman_man_medium_skin_tone_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FE': {
        'en' : u':couple_with_heart_woman_man_medium_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FE': {
        'en' : u':couple_with_heart_woman_man_medium_skin_tone_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FC': {
        'en' : u':couple_with_heart_woman_man_medium_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FC': {
        'en' : u':couple_with_heart_woman_man_medium_skin_tone_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469': {
        'en' : u':couple_with_heart_woman_woman:',
        'status' : fully_qualified,
        'E' : 2,
        'de': u':liebespaar_frau,_frau:',
        'es': u':pareja_enamorada_mujer_y_mujer:',
        'fr': u':couple_avec_cœur_femme_et_femme:',
        'pt': u':casal_apaixonado_mulher_e_mulher:',
        'it': u':coppia_con_cuore_donna_e_donna:'
    },
    u'\U0001F469\U0000200D\U00002764\U0000200D\U0001F469': {
        'en' : u':couple_with_heart_woman_woman:',
        'status' : minimally_qualified,
        'E' : 2
    },
    u'\U0001F469\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FF': {
        'en' : u':couple_with_heart_woman_woman_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FF': {
        'en' : u':couple_with_heart_woman_woman_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FB': {
        'en' : u':couple_with_heart_woman_woman_dark_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FB': {
        'en' : u':couple_with_heart_woman_woman_dark_skin_tone_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FE': {
        'en' : u':couple_with_heart_woman_woman_dark_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FE': {
        'en' : u':couple_with_heart_woman_woman_dark_skin_tone_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FC': {
        'en' : u':couple_with_heart_woman_woman_dark_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FC': {
        'en' : u':couple_with_heart_woman_woman_dark_skin_tone_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FD': {
        'en' : u':couple_with_heart_woman_woman_dark_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FD': {
        'en' : u':couple_with_heart_woman_woman_dark_skin_tone_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FB': {
        'en' : u':couple_with_heart_woman_woman_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FB': {
        'en' : u':couple_with_heart_woman_woman_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FF': {
        'en' : u':couple_with_heart_woman_woman_light_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FF': {
        'en' : u':couple_with_heart_woman_woman_light_skin_tone_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FE': {
        'en' : u':couple_with_heart_woman_woman_light_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FE': {
        'en' : u':couple_with_heart_woman_woman_light_skin_tone_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FC': {
        'en' : u':couple_with_heart_woman_woman_light_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FC': {
        'en' : u':couple_with_heart_woman_woman_light_skin_tone_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FD': {
        'en' : u':couple_with_heart_woman_woman_light_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FD': {
        'en' : u':couple_with_heart_woman_woman_light_skin_tone_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FE': {
        'en' : u':couple_with_heart_woman_woman_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FE': {
        'en' : u':couple_with_heart_woman_woman_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FF': {
        'en' : u':couple_with_heart_woman_woman_medium-dark_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FF': {
        'en' : u':couple_with_heart_woman_woman_medium-dark_skin_tone_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FB': {
        'en' : u':couple_with_heart_woman_woman_medium-dark_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FB': {
        'en' : u':couple_with_heart_woman_woman_medium-dark_skin_tone_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FC': {
        'en' : u':couple_with_heart_woman_woman_medium-dark_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FC': {
        'en' : u':couple_with_heart_woman_woman_medium-dark_skin_tone_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FD': {
        'en' : u':couple_with_heart_woman_woman_medium-dark_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FD': {
        'en' : u':couple_with_heart_woman_woman_medium-dark_skin_tone_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FC': {
        'en' : u':couple_with_heart_woman_woman_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FC': {
        'en' : u':couple_with_heart_woman_woman_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FF': {
        'en' : u':couple_with_heart_woman_woman_medium-light_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FF': {
        'en' : u':couple_with_heart_woman_woman_medium-light_skin_tone_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FB': {
        'en' : u':couple_with_heart_woman_woman_medium-light_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FB': {
        'en' : u':couple_with_heart_woman_woman_medium-light_skin_tone_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FE': {
        'en' : u':couple_with_heart_woman_woman_medium-light_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FE': {
        'en' : u':couple_with_heart_woman_woman_medium-light_skin_tone_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FD': {
        'en' : u':couple_with_heart_woman_woman_medium-light_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FD': {
        'en' : u':couple_with_heart_woman_woman_medium-light_skin_tone_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FD': {
        'en' : u':couple_with_heart_woman_woman_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FD': {
        'en' : u':couple_with_heart_woman_woman_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FF': {
        'en' : u':couple_with_heart_woman_woman_medium_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FF': {
        'en' : u':couple_with_heart_woman_woman_medium_skin_tone_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FB': {
        'en' : u':couple_with_heart_woman_woman_medium_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FB': {
        'en' : u':couple_with_heart_woman_woman_medium_skin_tone_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FE': {
        'en' : u':couple_with_heart_woman_woman_medium_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FE': {
        'en' : u':couple_with_heart_woman_woman_medium_skin_tone_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FC': {
        'en' : u':couple_with_heart_woman_woman_medium_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FC': {
        'en' : u':couple_with_heart_woman_woman_medium_skin_tone_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F404': {
        'en' : u':cow:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [u':cow2:'],
        'de': u':kuh:',
        'es': u':vaca:',
        'fr': u':vache:',
        'pt': u':vaca:',
        'it': u':mucca:'
    },
    u'\U0001F42E': {
        'en' : u':cow_face:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':cow:'],
        'de': u':kuhgesicht:',
        'es': u':cara_de_vaca:',
        'fr': u':tête_de_vache:',
        'pt': u':rosto_de_vaca:',
        'it': u':muso_di_mucca:'
    },
    u'\U0001F920': {
        'en' : u':cowboy_hat_face:',
        'status' : fully_qualified,
        'E' : 3,
        'de': u':gesicht_mit_cowboyhut:',
        'es': u':cara_con_sombrero_de_vaquero:',
        'fr': u':visage_avec_chapeau_de_cowboy:',
        'pt': u':rosto_com_chapéu_de_caubói:',
        'it': u':faccina_con_cappello_da_cowboy:'
    },
    u'\U0001F980': {
        'en' : u':crab:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':krebs:',
        'es': u':cangrejo:',
        'fr': u':crabe:',
        'pt': u':caranguejo:',
        'it': u':granchio:'
    },
    u'\U0001F58D\U0000FE0F': {
        'en' : u':crayon:',
        'status' : fully_qualified,
        'E' : 0.7,
        'alias' : [u':lower_left_crayon:'],
        'variant': True,
        'de': u':wachsmalstift:',
        'es': u':lápiz_de_cera:',
        'fr': u':crayon_pastel:',
        'pt': u':giz_de_cera:',
        'it': u':pastello_a_cera:'
    },
    u'\U0001F58D': {
        'en' : u':crayon:',
        'status' : unqualified,
        'E' : 0.7,
        'alias' : [u':lower_left_crayon:'],
        'variant': True,
        'de': u':wachsmalstift:',
        'es': u':lápiz_de_cera:',
        'fr': u':crayon_pastel:',
        'pt': u':giz_de_cera:',
        'it': u':pastello_a_cera:'
    },
    u'\U0001F4B3': {
        'en' : u':credit_card:',
        'status' : fully_qualified,
        'E' : 0.6,
        'variant': True,
        'de': u':kreditkarte:',
        'es': u':tarjeta_de_crédito:',
        'fr': u':carte_bancaire:',
        'pt': u':cartão_de_crédito:',
        'it': u':carta_di_credito:'
    },
    u'\U0001F319': {
        'en' : u':crescent_moon:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':mondsichel:',
        'es': u':luna:',
        'fr': u':croissant_de_lune:',
        'pt': u':lua_crescente:',
        'it': u':spicchio_di_luna:'
    },
    u'\U0001F997': {
        'en' : u':cricket:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':grille:',
        'es': u':grillo:',
        'fr': u':criquet:',
        'pt': u':grilo:',
        'it': u':grillo:'
    },
    u'\U0001F3CF': {
        'en' : u':cricket_game:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [u':cricket_bat_and_ball:'],
        'de': u':kricket:',
        'es': u':críquet:',
        'fr': u':cricket:',
        'pt': u':críquete:',
        'it': u':cricket:'
    },
    u'\U0001F40A': {
        'en' : u':crocodile:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':krokodil:',
        'es': u':cocodrilo:',
        'fr': u':crocodile:',
        'pt': u':crocodilo:',
        'it': u':coccodrillo:'
    },
    u'\U0001F950': {
        'en' : u':croissant:',
        'status' : fully_qualified,
        'E' : 3,
        'de': u':croissant:',
        'es': u':cruasán:',
        'fr': u':croissant:',
        'pt': u':croissant:',
        'it': u':croissant:'
    },
    u'\U0000274C': {
        'en' : u':cross_mark:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':x:'],
        'de': u':kreuzzeichen:',
        'es': u':marca_de_cruz:',
        'fr': u':croix:',
        'pt': u':xis:',
        'it': u':croce:'
    },
    u'\U0000274E': {
        'en' : u':cross_mark_button:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':negative_squared_cross_mark:'],
        'de': u':kreuzsymbol_im_quadrat:',
        'es': u':botón_con_marca_de_cruz:',
        'fr': u':bouton_croix:',
        'pt': u':botão_de_xis:',
        'it': u':croce_con_quadrato:'
    },
    u'\U0001F91E': {
        'en' : u':crossed_fingers:',
        'status' : fully_qualified,
        'E' : 3,
        'de': u':hand_mit_gekreuzten_fingern:',
        'es': u':dedos_cruzados:',
        'fr': u':doigts_croisés:',
        'pt': u':dedos_cruzados:',
        'it': u':dita_incrociate:'
    },
    u'\U0001F91E\U0001F3FF': {
        'en' : u':crossed_fingers_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 3,
        'de': u':hand_mit_gekreuzten_fingern_dunkle_hautfarbe:',
        'es': u':dedos_cruzados_tono_de_piel_oscuro:',
        'fr': u':doigts_croisés_peau_foncée:',
        'pt': u':dedos_cruzados_pele_escura:',
        'it': u':dita_incrociate_carnagione_scura:'
    },
    u'\U0001F91E\U0001F3FB': {
        'en' : u':crossed_fingers_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 3,
        'de': u':hand_mit_gekreuzten_fingern_helle_hautfarbe:',
        'es': u':dedos_cruzados_tono_de_piel_claro:',
        'fr': u':doigts_croisés_peau_claire:',
        'pt': u':dedos_cruzados_pele_clara:',
        'it': u':dita_incrociate_carnagione_chiara:'
    },
    u'\U0001F91E\U0001F3FE': {
        'en' : u':crossed_fingers_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 3,
        'de': u':hand_mit_gekreuzten_fingern_mitteldunkle_hautfarbe:',
        'es': u':dedos_cruzados_tono_de_piel_oscuro_medio:',
        'fr': u':doigts_croisés_peau_mate:',
        'pt': u':dedos_cruzados_pele_morena_escura:',
        'it': u':dita_incrociate_carnagione_abbastanza_scura:'
    },
    u'\U0001F91E\U0001F3FC': {
        'en' : u':crossed_fingers_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 3,
        'de': u':hand_mit_gekreuzten_fingern_mittelhelle_hautfarbe:',
        'es': u':dedos_cruzados_tono_de_piel_claro_medio:',
        'fr': u':doigts_croisés_peau_moyennement_claire:',
        'pt': u':dedos_cruzados_pele_morena_clara:',
        'it': u':dita_incrociate_carnagione_abbastanza_chiara:'
    },
    u'\U0001F91E\U0001F3FD': {
        'en' : u':crossed_fingers_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 3,
        'de': u':hand_mit_gekreuzten_fingern_mittlere_hautfarbe:',
        'es': u':dedos_cruzados_tono_de_piel_medio:',
        'fr': u':doigts_croisés_peau_légèrement_mate:',
        'pt': u':dedos_cruzados_pele_morena:',
        'it': u':dita_incrociate_carnagione_olivastra:'
    },
    u'\U0001F38C': {
        'en' : u':crossed_flags:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':überkreuzte_flaggen:',
        'es': u':banderas_cruzadas:',
        'fr': u':drapeaux_croisés:',
        'pt': u':bandeiras_cruzadas:',
        'it': u':bandiere_del_giappone_incrociate:'
    },
    u'\U00002694\U0000FE0F': {
        'en' : u':crossed_swords:',
        'status' : fully_qualified,
        'E' : 1,
        'variant': True,
        'de': u':gekreuzte_schwerter:',
        'es': u':espadas_cruzadas:',
        'fr': u':épées_croisées:',
        'pt': u':espadas_cruzadas:',
        'it': u':spade_incrociate:'
    },
    u'\U00002694': {
        'en' : u':crossed_swords:',
        'status' : unqualified,
        'E' : 1,
        'variant': True,
        'de': u':gekreuzte_schwerter:',
        'es': u':espadas_cruzadas:',
        'fr': u':épées_croisées:',
        'pt': u':espadas_cruzadas:',
        'it': u':spade_incrociate:'
    },
    u'\U0001F451': {
        'en' : u':crown:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':krone:',
        'es': u':corona:',
        'fr': u':couronne:',
        'pt': u':coroa:',
        'it': u':corona:'
    },
    u'\U0001FA7C': {
        'en' : u':crutch:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001F63F': {
        'en' : u':crying_cat:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':crying_cat_face:'],
        'de': u':weinende_katze:',
        'es': u':gato_llorando:',
        'fr': u':chat_qui_pleure:',
        'pt': u':rosto_de_gato_chorando:',
        'it': u':gatto_che_piange:'
    },
    u'\U0001F622': {
        'en' : u':crying_face:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':cry:'],
        'de': u':weinendes_gesicht:',
        'es': u':cara_llorando:',
        'fr': u':visage_qui_pleure:',
        'pt': u':rosto_chorando:',
        'it': u':faccina_che_piange:'
    },
    u'\U0001F52E': {
        'en' : u':crystal_ball:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':kristallkugel:',
        'es': u':bola_de_cristal:',
        'fr': u':boule_de_cristal:',
        'pt': u':bola_de_cristal:',
        'it': u':sfera_di_cristallo:'
    },
    u'\U0001F952': {
        'en' : u':cucumber:',
        'status' : fully_qualified,
        'E' : 3,
        'de': u':gurke:',
        'es': u':pepino:',
        'fr': u':concombre:',
        'pt': u':pepino:',
        'it': u':cetriolo:'
    },
    u'\U0001F964': {
        'en' : u':cup_with_straw:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':becher_mit_strohhalm:',
        'es': u':vaso_con_pajita:',
        'fr': u':gobelet_avec_paille:',
        'pt': u':copo_com_canudo:',
        'it': u':bicchiere_con_cannuccia:'
    },
    u'\U0001F9C1': {
        'en' : u':cupcake:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':cupcake:',
        'es': u':magdalena:',
        'fr': u':cupcake:',
        'pt': u':cupcake:',
        'it': u':cupcake:'
    },
    u'\U0001F94C': {
        'en' : u':curling_stone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':curlingstein:',
        'es': u':piedra_de_curling:',
        'fr': u':pierre_de_curling:',
        'pt': u':pedra_de_curling:',
        'it': u':stone_da_curling:'
    },
    u'\U0001F9B1': {
        'en' : u':curly_hair:',
        'status' : component,
        'E' : 11,
        'de': u':lockiges_haar:',
        'es': u':pelo_rizado:',
        'fr': u':cheveux_bouclés:',
        'pt': u':cabelo_cacheado:',
        'it': u':capelli_ricci:'
    },
    u'\U000027B0': {
        'en' : u':curly_loop:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':schleife:',
        'es': u':bucle:',
        'fr': u':boucle:',
        'pt': u':laço_encaracolado:',
        'it': u':occhiello:'
    },
    u'\U0001F4B1': {
        'en' : u':currency_exchange:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':geldwechsel:',
        'es': u':cambio_de_divisas:',
        'fr': u':conversion_de_devise:',
        'pt': u':câmbio_de_moeda:',
        'it': u':cambio_valuta:'
    },
    u'\U0001F35B': {
        'en' : u':curry_rice:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':curry:'],
        'de': u':reis_mit_curry:',
        'es': u':arroz_con_curry:',
        'fr': u':riz_au_curry:',
        'pt': u':arroz_com_curry:',
        'it': u':riso_al_curry:'
    },
    u'\U0001F36E': {
        'en' : u':custard:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':pudding:',
        'es': u':flan:',
        'fr': u':crème_renversée:',
        'pt': u':pudim:',
        'it': u':budino:'
    },
    u'\U0001F6C3': {
        'en' : u':customs:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':zollkontrolle:',
        'es': u':aduana:',
        'fr': u':douane:',
        'pt': u':alfândega:',
        'it': u':simbolo_della_dogana:'
    },
    u'\U0001F969': {
        'en' : u':cut_of_meat:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':fleischstück:',
        'es': u':corte_de_carne:',
        'fr': u':morceau_de_viande:',
        'pt': u':corte_de_carne:',
        'it': u':taglio_di_carne:'
    },
    u'\U0001F300': {
        'en' : u':cyclone:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':wirbel:',
        'es': u':ciclón:',
        'fr': u':cyclone:',
        'pt': u':ciclone:',
        'it': u':ciclone:'
    },
    u'\U0001F5E1\U0000FE0F': {
        'en' : u':dagger:',
        'status' : fully_qualified,
        'E' : 0.7,
        'alias' : [u':dagger_knife:'],
        'variant': True,
        'de': u':dolch:',
        'es': u':puñal:',
        'fr': u':dague:',
        'pt': u':adaga:',
        'it': u':pugnale:'
    },
    u'\U0001F5E1': {
        'en' : u':dagger:',
        'status' : unqualified,
        'E' : 0.7,
        'alias' : [u':dagger_knife:'],
        'variant': True,
        'de': u':dolch:',
        'es': u':puñal:',
        'fr': u':dague:',
        'pt': u':adaga:',
        'it': u':pugnale:'
    },
    u'\U0001F361': {
        'en' : u':dango:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':dango:',
        'es': u':dango:',
        'fr': u':brochette_de_bonbons:',
        'pt': u':dango:',
        'it': u':dolcetto:'
    },
    u'\U0001F3FF': {
        'en' : u':dark_skin_tone:',
        'status' : component,
        'E' : 1,
        'alias' : [u':emoji_modifier_fitzpatrick_type__6:'],
        'de': u':dunkle_hautfarbe:',
        'es': u':tono_de_piel_oscuro:',
        'fr': u':peau_foncée:',
        'pt': u':pele_escura:',
        'it': u':carnagione_scura:'
    },
    u'\U0001F4A8': {
        'en' : u':dashing_away:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':dash:'],
        'de': u':staubwolke:',
        'es': u':salir_corriendo:',
        'fr': u':décamper:',
        'pt': u':rapidez:',
        'it': u':nuvola_di_polvere:'
    },
    u'\U0001F9CF\U0000200D\U00002642\U0000FE0F': {
        'en' : u':deaf_man:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':gehörloser_mann:',
        'es': u':hombre_sordo:',
        'fr': u':homme_sourd:',
        'pt': u':homem_surdo:',
        'it': u':uomo_con_problemi_di_udito:'
    },
    u'\U0001F9CF\U0000200D\U00002642': {
        'en' : u':deaf_man:',
        'status' : minimally_qualified,
        'E' : 12
    },
    u'\U0001F9CF\U0001F3FF\U0000200D\U00002642\U0000FE0F': {
        'en' : u':deaf_man_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':gehörloser_mann_dunkle_hautfarbe:',
        'es': u':hombre_sordo_tono_de_piel_oscuro:',
        'fr': u':homme_sourd_peau_foncée:',
        'pt': u':homem_surdo_pele_escura:',
        'it': u':uomo_con_problemi_di_udito_carnagione_scura:'
    },
    u'\U0001F9CF\U0001F3FF\U0000200D\U00002642': {
        'en' : u':deaf_man_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 12
    },
    u'\U0001F9CF\U0001F3FB\U0000200D\U00002642\U0000FE0F': {
        'en' : u':deaf_man_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':gehörloser_mann_helle_hautfarbe:',
        'es': u':hombre_sordo_tono_de_piel_claro:',
        'fr': u':homme_sourd_peau_claire:',
        'pt': u':homem_surdo_pele_clara:',
        'it': u':uomo_con_problemi_di_udito_carnagione_chiara:'
    },
    u'\U0001F9CF\U0001F3FB\U0000200D\U00002642': {
        'en' : u':deaf_man_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 12
    },
    u'\U0001F9CF\U0001F3FE\U0000200D\U00002642\U0000FE0F': {
        'en' : u':deaf_man_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':gehörloser_mann_mitteldunkle_hautfarbe:',
        'es': u':hombre_sordo_tono_de_piel_oscuro_medio:',
        'fr': u':homme_sourd_peau_mate:',
        'pt': u':homem_surdo_pele_morena_escura:',
        'it': u':uomo_con_problemi_di_udito_carnagione_abbastanza_scura:'
    },
    u'\U0001F9CF\U0001F3FE\U0000200D\U00002642': {
        'en' : u':deaf_man_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 12
    },
    u'\U0001F9CF\U0001F3FC\U0000200D\U00002642\U0000FE0F': {
        'en' : u':deaf_man_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':gehörloser_mann_mittelhelle_hautfarbe:',
        'es': u':hombre_sordo_tono_de_piel_claro_medio:',
        'fr': u':homme_sourd_peau_moyennement_claire:',
        'pt': u':homem_surdo_pele_morena_clara:',
        'it': u':uomo_con_problemi_di_udito_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9CF\U0001F3FC\U0000200D\U00002642': {
        'en' : u':deaf_man_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 12
    },
    u'\U0001F9CF\U0001F3FD\U0000200D\U00002642\U0000FE0F': {
        'en' : u':deaf_man_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':gehörloser_mann_mittlere_hautfarbe:',
        'es': u':hombre_sordo_tono_de_piel_medio:',
        'fr': u':homme_sourd_peau_légèrement_mate:',
        'pt': u':homem_surdo_pele_morena:',
        'it': u':uomo_con_problemi_di_udito_carnagione_olivastra:'
    },
    u'\U0001F9CF\U0001F3FD\U0000200D\U00002642': {
        'en' : u':deaf_man_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 12
    },
    u'\U0001F9CF': {
        'en' : u':deaf_person:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':gehörlose_person:',
        'es': u':persona_sorda:',
        'fr': u':personne_sourde:',
        'pt': u':pessoa_surda:',
        'it': u':persona_con_problemi_di_udito:'
    },
    u'\U0001F9CF\U0001F3FF': {
        'en' : u':deaf_person_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':gehörlose_person_dunkle_hautfarbe:',
        'es': u':persona_sorda_tono_de_piel_oscuro:',
        'fr': u':personne_sourde_peau_foncée:',
        'pt': u':pessoa_surda_pele_escura:',
        'it': u':persona_con_problemi_di_udito_carnagione_scura:'
    },
    u'\U0001F9CF\U0001F3FB': {
        'en' : u':deaf_person_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':gehörlose_person_helle_hautfarbe:',
        'es': u':persona_sorda_tono_de_piel_claro:',
        'fr': u':personne_sourde_peau_claire:',
        'pt': u':pessoa_surda_pele_clara:',
        'it': u':persona_con_problemi_di_udito_carnagione_chiara:'
    },
    u'\U0001F9CF\U0001F3FE': {
        'en' : u':deaf_person_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':gehörlose_person_mitteldunkle_hautfarbe:',
        'es': u':persona_sorda_tono_de_piel_oscuro_medio:',
        'fr': u':personne_sourde_peau_mate:',
        'pt': u':pessoa_surda_pele_morena_escura:',
        'it': u':persona_con_problemi_di_udito_carnagione_abbastanza_scura:'
    },
    u'\U0001F9CF\U0001F3FC': {
        'en' : u':deaf_person_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':gehörlose_person_mittelhelle_hautfarbe:',
        'es': u':persona_sorda_tono_de_piel_claro_medio:',
        'fr': u':personne_sourde_peau_moyennement_claire:',
        'pt': u':pessoa_surda_pele_morena_clara:',
        'it': u':persona_con_problemi_di_udito_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9CF\U0001F3FD': {
        'en' : u':deaf_person_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':gehörlose_person_mittlere_hautfarbe:',
        'es': u':persona_sorda_tono_de_piel_medio:',
        'fr': u':personne_sourde_peau_légèrement_mate:',
        'pt': u':pessoa_surda_pele_morena:',
        'it': u':persona_con_problemi_di_udito_carnagione_olivastra:'
    },
    u'\U0001F9CF\U0000200D\U00002640\U0000FE0F': {
        'en' : u':deaf_woman:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':gehörlose_frau:',
        'es': u':mujer_sorda:',
        'fr': u':femme_sourde:',
        'pt': u':mulher_surda:',
        'it': u':donna_con_problemi_di_udito:'
    },
    u'\U0001F9CF\U0000200D\U00002640': {
        'en' : u':deaf_woman:',
        'status' : minimally_qualified,
        'E' : 12
    },
    u'\U0001F9CF\U0001F3FF\U0000200D\U00002640\U0000FE0F': {
        'en' : u':deaf_woman_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':gehörlose_frau_dunkle_hautfarbe:',
        'es': u':mujer_sorda_tono_de_piel_oscuro:',
        'fr': u':femme_sourde_peau_foncée:',
        'pt': u':mulher_surda_pele_escura:',
        'it': u':donna_con_problemi_di_udito_carnagione_scura:'
    },
    u'\U0001F9CF\U0001F3FF\U0000200D\U00002640': {
        'en' : u':deaf_woman_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 12
    },
    u'\U0001F9CF\U0001F3FB\U0000200D\U00002640\U0000FE0F': {
        'en' : u':deaf_woman_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':gehörlose_frau_helle_hautfarbe:',
        'es': u':mujer_sorda_tono_de_piel_claro:',
        'fr': u':femme_sourde_peau_claire:',
        'pt': u':mulher_surda_pele_clara:',
        'it': u':donna_con_problemi_di_udito_carnagione_chiara:'
    },
    u'\U0001F9CF\U0001F3FB\U0000200D\U00002640': {
        'en' : u':deaf_woman_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 12
    },
    u'\U0001F9CF\U0001F3FE\U0000200D\U00002640\U0000FE0F': {
        'en' : u':deaf_woman_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':gehörlose_frau_mitteldunkle_hautfarbe:',
        'es': u':mujer_sorda_tono_de_piel_oscuro_medio:',
        'fr': u':femme_sourde_peau_mate:',
        'pt': u':mulher_surda_pele_morena_escura:',
        'it': u':donna_con_problemi_di_udito_carnagione_abbastanza_scura:'
    },
    u'\U0001F9CF\U0001F3FE\U0000200D\U00002640': {
        'en' : u':deaf_woman_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 12
    },
    u'\U0001F9CF\U0001F3FC\U0000200D\U00002640\U0000FE0F': {
        'en' : u':deaf_woman_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':gehörlose_frau_mittelhelle_hautfarbe:',
        'es': u':mujer_sorda_tono_de_piel_claro_medio:',
        'fr': u':femme_sourde_peau_moyennement_claire:',
        'pt': u':mulher_surda_pele_morena_clara:',
        'it': u':donna_con_problemi_di_udito_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9CF\U0001F3FC\U0000200D\U00002640': {
        'en' : u':deaf_woman_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 12
    },
    u'\U0001F9CF\U0001F3FD\U0000200D\U00002640\U0000FE0F': {
        'en' : u':deaf_woman_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':gehörlose_frau_mittlere_hautfarbe:',
        'es': u':mujer_sorda_tono_de_piel_medio:',
        'fr': u':femme_sourde_peau_légèrement_mate:',
        'pt': u':mulher_surda_pele_morena:',
        'it': u':donna_con_problemi_di_udito_carnagione_olivastra:'
    },
    u'\U0001F9CF\U0001F3FD\U0000200D\U00002640': {
        'en' : u':deaf_woman_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 12
    },
    u'\U0001F333': {
        'en' : u':deciduous_tree:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':laubbaum:',
        'es': u':árbol_de_hoja_caduca:',
        'fr': u':arbre_à_feuilles_caduques:',
        'pt': u':árvore_caidiça:',
        'it': u':albero_deciduo:'
    },
    u'\U0001F98C': {
        'en' : u':deer:',
        'status' : fully_qualified,
        'E' : 3,
        'de': u':hirsch:',
        'es': u':ciervo:',
        'fr': u':cerf:',
        'pt': u':cervo:',
        'it': u':cervo:'
    },
    u'\U0001F69A': {
        'en' : u':delivery_truck:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':truck:'],
        'de': u':lieferwagen:',
        'es': u':camión_de_reparto:',
        'fr': u':camion_de_livraison:',
        'pt': u':caminhão_de_entrega:',
        'it': u':camion:'
    },
    u'\U0001F3EC': {
        'en' : u':department_store:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':kaufhaus:',
        'es': u':grandes_almacenes:',
        'fr': u':grand_magasin:',
        'pt': u':loja_de_departamentos:',
        'it': u':grande_magazzino:'
    },
    u'\U0001F3DA\U0000FE0F': {
        'en' : u':derelict_house:',
        'status' : fully_qualified,
        'E' : 0.7,
        'alias' : [u':derelict_house_building:'],
        'variant': True,
        'de': u':verfallenes_haus:',
        'es': u':casa_abandonada:',
        'fr': u':maison_abandonnée:',
        'pt': u':casa_abandonada:',
        'it': u':casa_in_rovina:'
    },
    u'\U0001F3DA': {
        'en' : u':derelict_house:',
        'status' : unqualified,
        'E' : 0.7,
        'alias' : [u':derelict_house_building:'],
        'variant': True,
        'de': u':verfallenes_haus:',
        'es': u':casa_abandonada:',
        'fr': u':maison_abandonnée:',
        'pt': u':casa_abandonada:',
        'it': u':casa_in_rovina:'
    },
    u'\U0001F3DC\U0000FE0F': {
        'en' : u':desert:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': u':wüste:',
        'es': u':desierto:',
        'fr': u':désert:',
        'pt': u':deserto:',
        'it': u':deserto:'
    },
    u'\U0001F3DC': {
        'en' : u':desert:',
        'status' : unqualified,
        'E' : 0.7,
        'variant': True,
        'de': u':wüste:',
        'es': u':desierto:',
        'fr': u':désert:',
        'pt': u':deserto:',
        'it': u':deserto:'
    },
    u'\U0001F3DD\U0000FE0F': {
        'en' : u':desert_island:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': u':einsame_insel:',
        'es': u':isla_desierta:',
        'fr': u':île_déserte:',
        'pt': u':ilha_deserta:',
        'it': u':isola_deserta:'
    },
    u'\U0001F3DD': {
        'en' : u':desert_island:',
        'status' : unqualified,
        'E' : 0.7,
        'variant': True,
        'de': u':einsame_insel:',
        'es': u':isla_desierta:',
        'fr': u':île_déserte:',
        'pt': u':ilha_deserta:',
        'it': u':isola_deserta:'
    },
    u'\U0001F5A5\U0000FE0F': {
        'en' : u':desktop_computer:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': u':desktopcomputer:',
        'es': u':ordenador_de_sobremesa:',
        'fr': u':ordinateur_de_bureau:',
        'pt': u':computador_de_mesa:',
        'it': u':computer_fisso:'
    },
    u'\U0001F5A5': {
        'en' : u':desktop_computer:',
        'status' : unqualified,
        'E' : 0.7,
        'variant': True,
        'de': u':desktopcomputer:',
        'es': u':ordenador_de_sobremesa:',
        'fr': u':ordinateur_de_bureau:',
        'pt': u':computador_de_mesa:',
        'it': u':computer_fisso:'
    },
    u'\U0001F575\U0000FE0F': {
        'en' : u':detective:',
        'status' : fully_qualified,
        'E' : 0.7,
        'alias' : [u':sleuth_or_spy:'],
        'variant': True,
        'de': u':detektiv(in):',
        'es': u':detective:',
        'fr': u':détective:',
        'pt': u':detetive:',
        'it': u':detective:'
    },
    u'\U0001F575': {
        'en' : u':detective:',
        'status' : unqualified,
        'E' : 0.7,
        'alias' : [u':sleuth_or_spy:'],
        'variant': True,
        'de': u':detektiv(in):',
        'es': u':detective:',
        'fr': u':détective:',
        'pt': u':detetive:',
        'it': u':detective:'
    },
    u'\U0001F575\U0001F3FF': {
        'en' : u':detective_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 2,
        'de': u':detektiv(in)_dunkle_hautfarbe:',
        'es': u':detective_tono_de_piel_oscuro:',
        'fr': u':détective_peau_foncée:',
        'pt': u':detetive_pele_escura:',
        'it': u':detective_carnagione_scura:'
    },
    u'\U0001F575\U0001F3FB': {
        'en' : u':detective_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 2,
        'de': u':detektiv(in)_helle_hautfarbe:',
        'es': u':detective_tono_de_piel_claro:',
        'fr': u':détective_peau_claire:',
        'pt': u':detetive_pele_clara:',
        'it': u':detective_carnagione_chiara:'
    },
    u'\U0001F575\U0001F3FE': {
        'en' : u':detective_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 2,
        'de': u':detektiv(in)_mitteldunkle_hautfarbe:',
        'es': u':detective_tono_de_piel_oscuro_medio:',
        'fr': u':détective_peau_mate:',
        'pt': u':detetive_pele_morena_escura:',
        'it': u':detective_carnagione_abbastanza_scura:'
    },
    u'\U0001F575\U0001F3FC': {
        'en' : u':detective_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 2,
        'de': u':detektiv(in)_mittelhelle_hautfarbe:',
        'es': u':detective_tono_de_piel_claro_medio:',
        'fr': u':détective_peau_moyennement_claire:',
        'pt': u':detetive_pele_morena_clara:',
        'it': u':detective_carnagione_abbastanza_chiara:'
    },
    u'\U0001F575\U0001F3FD': {
        'en' : u':detective_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 2,
        'de': u':detektiv(in)_mittlere_hautfarbe:',
        'es': u':detective_tono_de_piel_medio:',
        'fr': u':détective_peau_légèrement_mate:',
        'pt': u':detetive_pele_morena:',
        'it': u':detective_carnagione_olivastra:'
    },
    u'\U00002666\U0000FE0F': {
        'en' : u':diamond_suit:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':diamonds:'],
        'variant': True,
        'de': u':karo:',
        'es': u':palo_de_diamantes:',
        'fr': u':carreau:',
        'pt': u':naipe_de_ouros:',
        'it': u':quadri:'
    },
    u'\U00002666': {
        'en' : u':diamond_suit:',
        'status' : unqualified,
        'E' : 0.6,
        'alias' : [u':diamonds:'],
        'variant': True,
        'de': u':karo:',
        'es': u':palo_de_diamantes:',
        'fr': u':carreau:',
        'pt': u':naipe_de_ouros:',
        'it': u':quadri:'
    },
    u'\U0001F4A0': {
        'en' : u':diamond_with_a_dot:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':diamond_shape_with_a_dot_inside:'],
        'de': u':rautenform_mit_punkt:',
        'es': u':rombo_con_pétalo:',
        'fr': u':diamant_avec_un_point:',
        'pt': u':diamante_com_um_ponto:',
        'it': u':petalo_di_fiore:'
    },
    u'\U0001F505': {
        'en' : u':dim_button:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [u':low_brightness:'],
        'de': u':taste_dimmen:',
        'es': u':brillo_bajo:',
        'fr': u':luminosité_faible:',
        'pt': u':botão_de_diminuir_brilho:',
        'it': u':luminosità_bassa:'
    },
    u'\U0001F61E': {
        'en' : u':disappointed_face:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':disappointed:'],
        'de': u':enttäuschtes_gesicht:',
        'es': u':cara_decepcionada:',
        'fr': u':visage_déçu:',
        'pt': u':rosto_desapontado:',
        'it': u':faccina_delusa:'
    },
    u'\U0001F978': {
        'en' : u':disguised_face:',
        'status' : fully_qualified,
        'E' : 13,
        'de': u':verkleidet:',
        'es': u':cara_disfrazada:',
        'fr': u':visage_déguisé:',
        'pt': u':rosto_disfarçado:',
        'it': u':faccina_travestita:'
    },
    u'\U00002797': {
        'en' : u':divide:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':heavy_division_sign:'],
        'de': u':divisionszeichen:',
        'es': u':división:',
        'fr': u':signe_diviser:',
        'pt': u':símbolo_de_divisão:',
        'it': u':simbolo_della_divisione:'
    },
    u'\U0001F93F': {
        'en' : u':diving_mask:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':tauchmaske:',
        'es': u':máscara_de_buceo:',
        'fr': u':masque_de_plongée:',
        'pt': u':máscara_de_mergulho:',
        'it': u':maschera_da_sub:'
    },
    u'\U0001FA94': {
        'en' : u':diya_lamp:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':öllampe:',
        'es': u':lámpara_de_aceite:',
        'fr': u':diya:',
        'pt': u':lâmpada_de_óleo:',
        'it': u':diya:'
    },
    u'\U0001F4AB': {
        'en' : u':dizzy:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':schwindlig:',
        'es': u':símbolo_de_mareo:',
        'fr': u':étourdissement:',
        'pt': u':zonzo:',
        'it': u':stella_con_scia:'
    },
    u'\U0001F9EC': {
        'en' : u':dna:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':dna:',
        'es': u':adn:',
        'fr': u':adn:',
        'pt': u':dna:',
        'it': u':dna:'
    },
    u'\U0001F9A4': {
        'en' : u':dodo:',
        'status' : fully_qualified,
        'E' : 13,
        'de': u':dodo:',
        'es': u':dodo:',
        'fr': u':dodo:',
        'pt': u':dodô:',
        'it': u':dodo:'
    },
    u'\U0001F415': {
        'en' : u':dog:',
        'status' : fully_qualified,
        'E' : 0.7,
        'alias' : [u':dog2:'],
        'variant': True,
        'de': u':hund:',
        'es': u':perro:',
        'fr': u':chien:',
        'pt': u':cachorro:',
        'it': u':cane:'
    },
    u'\U0001F436': {
        'en' : u':dog_face:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':dog:'],
        'de': u':hundegesicht:',
        'es': u':cara_de_perro:',
        'fr': u':tête_de_chien:',
        'pt': u':rosto_de_cachorro:',
        'it': u':muso_di_cane:'
    },
    u'\U0001F4B5': {
        'en' : u':dollar_banknote:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':dollar:'],
        'de': u':dollar-banknote:',
        'es': u':billete_de_dólar:',
        'fr': u':billet_en_dollars:',
        'pt': u':nota_de_dólar:',
        'it': u':banconota_dollaro:'
    },
    u'\U0001F42C': {
        'en' : u':dolphin:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':flipper:'],
        'de': u':delfin:',
        'es': u':delfín:',
        'fr': u':dauphin:',
        'pt': u':golfinho:',
        'it': u':delfino:'
    },
    u'\U0001F6AA': {
        'en' : u':door:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':tür:',
        'es': u':puerta:',
        'fr': u':porte:',
        'pt': u':porta:',
        'it': u':porta:'
    },
    u'\U0001FAE5': {
        'en' : u':dotted_line_face:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001F52F': {
        'en' : u':dotted_six-pointed_star:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':six_pointed_star:'],
        'de': u':hexagramm_mit_punkt:',
        'es': u':estrella_de_seis_puntas:',
        'fr': u':étoile_à_6_branches:',
        'pt': u':estrela_de_seis_pontas:',
        'it': u':stella_a_sei_punte:'
    },
    u'\U000027BF': {
        'en' : u':double_curly_loop:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [u':loop:'],
        'de': u':doppelschleife:',
        'es': u':bucle_doble:',
        'fr': u':double_boucle:',
        'pt': u':loop_encaracolado_duas_vezes:',
        'it': u':doppio_occhiello:'
    },
    u'\U0000203C\U0000FE0F': {
        'en' : u':double_exclamation_mark:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':bangbang:'],
        'variant': True,
        'de': u':doppeltes_ausrufezeichen:',
        'es': u':exclamación_doble:',
        'fr': u':double_point_d’exclamation:',
        'pt': u':dupla_exclamação:',
        'it': u':doppio_punto_esclamativo:'
    },
    u'\U0000203C': {
        'en' : u':double_exclamation_mark:',
        'status' : unqualified,
        'E' : 0.6,
        'alias' : [u':bangbang:'],
        'variant': True,
        'de': u':doppeltes_ausrufezeichen:',
        'es': u':exclamación_doble:',
        'fr': u':double_point_d’exclamation:',
        'pt': u':dupla_exclamação:',
        'it': u':doppio_punto_esclamativo:'
    },
    u'\U0001F369': {
        'en' : u':doughnut:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':donut:',
        'es': u':dónut:',
        'fr': u':doughnut:',
        'pt': u':donut:',
        'it': u':ciambella:'
    },
    u'\U0001F54A\U0000FE0F': {
        'en' : u':dove:',
        'status' : fully_qualified,
        'E' : 0.7,
        'alias' : [u':dove_of_peace:'],
        'variant': True,
        'de': u':taube:',
        'es': u':paloma:',
        'fr': u':colombe:',
        'pt': u':pomba_branca:',
        'it': u':colomba:'
    },
    u'\U0001F54A': {
        'en' : u':dove:',
        'status' : unqualified,
        'E' : 0.7,
        'alias' : [u':dove_of_peace:'],
        'variant': True,
        'de': u':taube:',
        'es': u':paloma:',
        'fr': u':colombe:',
        'pt': u':pomba_branca:',
        'it': u':colomba:'
    },
    u'\U00002199\U0000FE0F': {
        'en' : u':down-left_arrow:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':arrow_lower_left:'],
        'variant': True,
        'de': u':pfeil_nach_links_unten:',
        'es': u':flecha_hacia_la_esquina_inferior_izquierda:',
        'fr': u':flèche_bas_gauche:',
        'pt': u':seta_para_baixo_e_para_a_esquerda:',
        'it': u':freccia_in_basso_a_sinistra:'
    },
    u'\U00002199': {
        'en' : u':down-left_arrow:',
        'status' : unqualified,
        'E' : 0.6,
        'alias' : [u':arrow_lower_left:'],
        'variant': True,
        'de': u':pfeil_nach_links_unten:',
        'es': u':flecha_hacia_la_esquina_inferior_izquierda:',
        'fr': u':flèche_bas_gauche:',
        'pt': u':seta_para_baixo_e_para_a_esquerda:',
        'it': u':freccia_in_basso_a_sinistra:'
    },
    u'\U00002198\U0000FE0F': {
        'en' : u':down-right_arrow:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':arrow_lower_right:'],
        'variant': True,
        'de': u':pfeil_nach_rechts_unten:',
        'es': u':flecha_hacia_la_esquina_inferior_derecha:',
        'fr': u':flèche_bas_droite:',
        'pt': u':seta_para_baixo_e_para_a_direita:',
        'it': u':freccia_in_basso_a_destra:'
    },
    u'\U00002198': {
        'en' : u':down-right_arrow:',
        'status' : unqualified,
        'E' : 0.6,
        'alias' : [u':arrow_lower_right:'],
        'variant': True,
        'de': u':pfeil_nach_rechts_unten:',
        'es': u':flecha_hacia_la_esquina_inferior_derecha:',
        'fr': u':flèche_bas_droite:',
        'pt': u':seta_para_baixo_e_para_a_direita:',
        'it': u':freccia_in_basso_a_destra:'
    },
    u'\U00002B07\U0000FE0F': {
        'en' : u':down_arrow:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':arrow_down:'],
        'variant': True,
        'de': u':pfeil_nach_unten:',
        'es': u':flecha_hacia_abajo:',
        'fr': u':flèche_bas:',
        'pt': u':seta_para_baixo:',
        'it': u':freccia_rivolta_verso_il_basso:'
    },
    u'\U00002B07': {
        'en' : u':down_arrow:',
        'status' : unqualified,
        'E' : 0.6,
        'alias' : [u':arrow_down:'],
        'variant': True,
        'de': u':pfeil_nach_unten:',
        'es': u':flecha_hacia_abajo:',
        'fr': u':flèche_bas:',
        'pt': u':seta_para_baixo:',
        'it': u':freccia_rivolta_verso_il_basso:'
    },
    u'\U0001F613': {
        'en' : u':downcast_face_with_sweat:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':sweat:'],
        'de': u':bedrücktes_gesicht_mit_schweiß:',
        'es': u':cara_con_sudor_frío:',
        'fr': u':visage_démoralisé_avec_goutte_de_sueur:',
        'pt': u':rosto_cabisbaixo_com_gota_de_suor:',
        'it': u':faccina_sudata:'
    },
    u'\U0001F53D': {
        'en' : u':downwards_button:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':arrow_down_small:'],
        'de': u':abwärts-schaltfläche:',
        'es': u':triángulo_hacia_abajo:',
        'fr': u':petit_triangle_bas:',
        'pt': u':botão_apontando_para_baixo:',
        'it': u':pulsante_a_triangolo_rivolto_verso_il_basso:'
    },
    u'\U0001F409': {
        'en' : u':dragon:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':drache:',
        'es': u':dragón:',
        'fr': u':dragon:',
        'pt': u':dragão:',
        'it': u':drago:'
    },
    u'\U0001F432': {
        'en' : u':dragon_face:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':drachengesicht:',
        'es': u':cara_de_dragón:',
        'fr': u':tête_de_dragon:',
        'pt': u':rosto_de_dragão:',
        'it': u':testa_di_drago:'
    },
    u'\U0001F457': {
        'en' : u':dress:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':kleid:',
        'es': u':vestido:',
        'fr': u':robe:',
        'pt': u':vestido:',
        'it': u':vestito:'
    },
    u'\U0001F924': {
        'en' : u':drooling_face:',
        'status' : fully_qualified,
        'E' : 3,
        'de': u':sabberndes_gesicht:',
        'es': u':cara_babeando:',
        'fr': u':visage_qui_bave:',
        'pt': u':rosto_babando:',
        'it': u':faccina_che_sbava:'
    },
    u'\U0001FA78': {
        'en' : u':drop_of_blood:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':blutstropfen:',
        'es': u':gota_de_sangre:',
        'fr': u':goutte_de_sang:',
        'pt': u':gota_de_sangue:',
        'it': u':goccia_di_sangue:'
    },
    u'\U0001F4A7': {
        'en' : u':droplet:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':tropfen:',
        'es': u':gota:',
        'fr': u':goutte_d’eau:',
        'pt': u':gota:',
        'it': u':goccia:'
    },
    u'\U0001F941': {
        'en' : u':drum:',
        'status' : fully_qualified,
        'E' : 3,
        'de': u':trommel:',
        'es': u':tambor:',
        'fr': u':batterie:',
        'pt': u':tambor:',
        'it': u':tamburo:'
    },
    u'\U0001F986': {
        'en' : u':duck:',
        'status' : fully_qualified,
        'E' : 3,
        'de': u':ente:',
        'es': u':pato:',
        'fr': u':canard:',
        'pt': u':pato:',
        'it': u':anatra:'
    },
    u'\U0001F95F': {
        'en' : u':dumpling:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':teigtasche:',
        'es': u':dumpling:',
        'fr': u':boulette_de_pâte:',
        'pt': u':bolinho_asiático:',
        'it': u':raviolo:'
    },
    u'\U0001F4C0': {
        'en' : u':dvd:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':dvd:',
        'es': u':disco_dvd:',
        'fr': u':dvd:',
        'pt': u':dvd:',
        'it': u':dvd:'
    },
    u'\U0001F4E7': {
        'en' : u':e-mail:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':email:', u':e__mail:'],
        'de': u':e-mail:',
        'es': u':correo_electrónico:',
        'fr': u':e-mail:',
        'pt': u':e-mail:',
        'it': u':e-mail:'
    },
    u'\U0001F985': {
        'en' : u':eagle:',
        'status' : fully_qualified,
        'E' : 3,
        'de': u':adler:',
        'es': u':águila:',
        'fr': u':aigle:',
        'pt': u':águia:',
        'it': u':aquila:'
    },
    u'\U0001F442': {
        'en' : u':ear:',
        'status' : fully_qualified,
        'E' : 0.6,
        'variant': True,
        'de': u':ohr:',
        'es': u':oreja:',
        'fr': u':oreille:',
        'pt': u':orelha:',
        'it': u':orecchio:'
    },
    u'\U0001F442\U0001F3FF': {
        'en' : u':ear_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':ohr_dunkle_hautfarbe:',
        'es': u':oreja_tono_de_piel_oscuro:',
        'fr': u':oreille_peau_foncée:',
        'pt': u':orelha_pele_escura:',
        'it': u':orecchio_carnagione_scura:'
    },
    u'\U0001F442\U0001F3FB': {
        'en' : u':ear_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':ohr_helle_hautfarbe:',
        'es': u':oreja_tono_de_piel_claro:',
        'fr': u':oreille_peau_claire:',
        'pt': u':orelha_pele_clara:',
        'it': u':orecchio_carnagione_chiara:'
    },
    u'\U0001F442\U0001F3FE': {
        'en' : u':ear_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':ohr_mitteldunkle_hautfarbe:',
        'es': u':oreja_tono_de_piel_oscuro_medio:',
        'fr': u':oreille_peau_mate:',
        'pt': u':orelha_pele_morena_escura:',
        'it': u':orecchio_carnagione_abbastanza_scura:'
    },
    u'\U0001F442\U0001F3FC': {
        'en' : u':ear_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':ohr_mittelhelle_hautfarbe:',
        'es': u':oreja_tono_de_piel_claro_medio:',
        'fr': u':oreille_peau_moyennement_claire:',
        'pt': u':orelha_pele_morena_clara:',
        'it': u':orecchio_carnagione_abbastanza_chiara:'
    },
    u'\U0001F442\U0001F3FD': {
        'en' : u':ear_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':ohr_mittlere_hautfarbe:',
        'es': u':oreja_tono_de_piel_medio:',
        'fr': u':oreille_peau_légèrement_mate:',
        'pt': u':orelha_pele_morena:',
        'it': u':orecchio_carnagione_olivastra:'
    },
    u'\U0001F33D': {
        'en' : u':ear_of_corn:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':corn:'],
        'de': u':maiskolben:',
        'es': u':espiga_de_maíz:',
        'fr': u':épi_de_maïs:',
        'pt': u':milho:',
        'it': u':pannocchia:'
    },
    u'\U0001F9BB': {
        'en' : u':ear_with_hearing_aid:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':ohr_mit_hörhilfe:',
        'es': u':oreja_con_audífono:',
        'fr': u':oreille_appareillée:',
        'pt': u':ouvido_com_aparelho_auditivo:',
        'it': u':orecchio_con_apparecchio_acustico:'
    },
    u'\U0001F9BB\U0001F3FF': {
        'en' : u':ear_with_hearing_aid_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':ohr_mit_hörhilfe_dunkle_hautfarbe:',
        'es': u':oreja_con_audífono_tono_de_piel_oscuro:',
        'fr': u':oreille_appareillée_peau_foncée:',
        'pt': u':ouvido_com_aparelho_auditivo_pele_escura:',
        'it': u':orecchio_con_apparecchio_acustico_carnagione_scura:'
    },
    u'\U0001F9BB\U0001F3FB': {
        'en' : u':ear_with_hearing_aid_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':ohr_mit_hörhilfe_helle_hautfarbe:',
        'es': u':oreja_con_audífono_tono_de_piel_claro:',
        'fr': u':oreille_appareillée_peau_claire:',
        'pt': u':ouvido_com_aparelho_auditivo_pele_clara:',
        'it': u':orecchio_con_apparecchio_acustico_carnagione_chiara:'
    },
    u'\U0001F9BB\U0001F3FE': {
        'en' : u':ear_with_hearing_aid_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':ohr_mit_hörhilfe_mitteldunkle_hautfarbe:',
        'es': u':oreja_con_audífono_tono_de_piel_oscuro_medio:',
        'fr': u':oreille_appareillée_peau_mate:',
        'pt': u':ouvido_com_aparelho_auditivo_pele_morena_escura:',
        'it': u':orecchio_con_apparecchio_acustico_carnagione_abbastanza_scura:'
    },
    u'\U0001F9BB\U0001F3FC': {
        'en' : u':ear_with_hearing_aid_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':ohr_mit_hörhilfe_mittelhelle_hautfarbe:',
        'es': u':oreja_con_audífono_tono_de_piel_claro_medio:',
        'fr': u':oreille_appareillée_peau_moyennement_claire:',
        'pt': u':ouvido_com_aparelho_auditivo_pele_morena_clara:',
        'it': u':orecchio_con_apparecchio_acustico_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9BB\U0001F3FD': {
        'en' : u':ear_with_hearing_aid_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':ohr_mit_hörhilfe_mittlere_hautfarbe:',
        'es': u':oreja_con_audífono_tono_de_piel_medio:',
        'fr': u':oreille_appareillée_peau_légèrement_mate:',
        'pt': u':ouvido_com_aparelho_auditivo_pele_morena:',
        'it': u':orecchio_con_apparecchio_acustico_carnagione_olivastra:'
    },
    u'\U0001F95A': {
        'en' : u':egg:',
        'status' : fully_qualified,
        'E' : 3,
        'alias' : [u':egg2:'],
        'de': u':ei:',
        'es': u':huevo:',
        'fr': u':œuf:',
        'pt': u':ovo:',
        'it': u':uovo:'
    },
    u'\U0001F346': {
        'en' : u':eggplant:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':aubergine:',
        'es': u':berenjena:',
        'fr': u':aubergine:',
        'pt': u':berinjela:',
        'it': u':melanzana:'
    },
    u'\U00002734\U0000FE0F': {
        'en' : u':eight-pointed_star:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':eight_pointed_black_star:'],
        'variant': True,
        'de': u':stern_mit_acht_zacken:',
        'es': u':estrella_de_ocho_puntas:',
        'fr': u':étoile_huit_branches:',
        'pt': u':estrela_de_oito_pontas:',
        'it': u':stella_stilizzata:'
    },
    u'\U00002734': {
        'en' : u':eight-pointed_star:',
        'status' : unqualified,
        'E' : 0.6,
        'alias' : [u':eight_pointed_black_star:'],
        'variant': True,
        'de': u':stern_mit_acht_zacken:',
        'es': u':estrella_de_ocho_puntas:',
        'fr': u':étoile_huit_branches:',
        'pt': u':estrela_de_oito_pontas:',
        'it': u':stella_stilizzata:'
    },
    u'\U00002733\U0000FE0F': {
        'en' : u':eight-spoked_asterisk:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':eight_spoked_asterisk:'],
        'variant': True,
        'de': u':achtzackiger_stern:',
        'es': u':asterisco_de_ocho_puntas:',
        'fr': u':astérisque_huit_branches:',
        'pt': u':asterisco_de_oito_pontas:',
        'it': u':asterisco:'
    },
    u'\U00002733': {
        'en' : u':eight-spoked_asterisk:',
        'status' : unqualified,
        'E' : 0.6,
        'alias' : [u':eight_spoked_asterisk:'],
        'variant': True,
        'de': u':achtzackiger_stern:',
        'es': u':asterisco_de_ocho_puntas:',
        'fr': u':astérisque_huit_branches:',
        'pt': u':asterisco_de_oito_pontas:',
        'it': u':asterisco:'
    },
    u'\U0001F563': {
        'en' : u':eight-thirty:',
        'status' : fully_qualified,
        'E' : 0.7,
        'alias' : [u':clock830:'],
        'variant': True,
        'de': u':8.30_uhr:',
        'es': u':ocho_y_media:',
        'fr': u':huit_heures_et_demie:',
        'pt': u':oito_e_meia:',
        'it': u':ore_otto_e_mezza:'
    },
    u'\U0001F557': {
        'en' : u':eight_o’clock:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':clock8:'],
        'variant': True,
        'de': u':8.00_uhr:',
        'es': u':8_en_punto:',
        'fr': u':huit_heures:',
        'pt': u':8_horas:',
        'it': u':ore_otto:'
    },
    u'\U000023CF\U0000FE0F': {
        'en' : u':eject_button:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [u':eject_symbol:'],
        'variant': True,
        'de': u':auswerfen:',
        'es': u':expulsar:',
        'fr': u':bouton_éjecter:',
        'pt': u':botão_ejetar:',
        'it': u':pulsante_di_espulsione:'
    },
    u'\U000023CF': {
        'en' : u':eject_button:',
        'status' : unqualified,
        'E' : 1,
        'alias' : [u':eject_symbol:'],
        'variant': True,
        'de': u':auswerfen:',
        'es': u':expulsar:',
        'fr': u':bouton_éjecter:',
        'pt': u':botão_ejetar:',
        'it': u':pulsante_di_espulsione:'
    },
    u'\U0001F50C': {
        'en' : u':electric_plug:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':stromstecker:',
        'es': u':enchufe_eléctrico:',
        'fr': u':câble_avec_fiche_électrique:',
        'pt': u':tomada_elétrica:',
        'it': u':spina_elettrica:'
    },
    u'\U0001F418': {
        'en' : u':elephant:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':elefant:',
        'es': u':elefante:',
        'fr': u':éléphant:',
        'pt': u':elefante:',
        'it': u':elefante:'
    },
    u'\U0001F6D7': {
        'en' : u':elevator:',
        'status' : fully_qualified,
        'E' : 13,
        'de': u':fahrstuhl:',
        'es': u':ascensor:',
        'fr': u':ascenseur:',
        'pt': u':elevador:',
        'it': u':ascensore:'
    },
    u'\U0001F566': {
        'en' : u':eleven-thirty:',
        'status' : fully_qualified,
        'E' : 0.7,
        'alias' : [u':clock1130:'],
        'variant': True,
        'de': u':11.30_uhr:',
        'es': u':once_y_media:',
        'fr': u':onze_heures_et_demie:',
        'pt': u':onze_e_meia:',
        'it': u':ore_undici_e_mezza:'
    },
    u'\U0001F55A': {
        'en' : u':eleven_o’clock:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':clock11:'],
        'variant': True,
        'de': u':11.00_uhr:',
        'es': u':11_en_punto:',
        'fr': u':onze_heures:',
        'pt': u':11_horas:',
        'it': u':ore_undici:'
    },
    u'\U0001F9DD': {
        'en' : u':elf:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':elf(e):',
        'es': u':elfo:',
        'fr': u':elfe:',
        'pt': u':elfo:',
        'it': u':elfo:'
    },
    u'\U0001F9DD\U0001F3FF': {
        'en' : u':elf_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':elf(e)_dunkle_hautfarbe:',
        'es': u':elfo_tono_de_piel_oscuro:',
        'fr': u':elfe_peau_foncée:',
        'pt': u':elfo_pele_escura:',
        'it': u':elfo_carnagione_scura:'
    },
    u'\U0001F9DD\U0001F3FB': {
        'en' : u':elf_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':elf(e)_helle_hautfarbe:',
        'es': u':elfo_tono_de_piel_claro:',
        'fr': u':elfe_peau_claire:',
        'pt': u':elfo_pele_clara:',
        'it': u':elfo_carnagione_chiara:'
    },
    u'\U0001F9DD\U0001F3FE': {
        'en' : u':elf_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':elf(e)_mitteldunkle_hautfarbe:',
        'es': u':elfo_tono_de_piel_oscuro_medio:',
        'fr': u':elfe_peau_mate:',
        'pt': u':elfo_pele_morena_escura:',
        'it': u':elfo_carnagione_abbastanza_scura:'
    },
    u'\U0001F9DD\U0001F3FC': {
        'en' : u':elf_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':elf(e)_mittelhelle_hautfarbe:',
        'es': u':elfo_tono_de_piel_claro_medio:',
        'fr': u':elfe_peau_moyennement_claire:',
        'pt': u':elfo_pele_morena_clara:',
        'it': u':elfo_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9DD\U0001F3FD': {
        'en' : u':elf_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':elf(e)_mittlere_hautfarbe:',
        'es': u':elfo_tono_de_piel_medio:',
        'fr': u':elfe_peau_légèrement_mate:',
        'pt': u':elfo_pele_morena:',
        'it': u':elfo_carnagione_olivastra:'
    },
    u'\U0001FAB9': {
        'en' : u':empty_nest:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U00002709\U0000FE0F': {
        'en' : u':envelope:',
        'status' : fully_qualified,
        'E' : 0.6,
        'variant': True,
        'de': u':briefumschlag:',
        'es': u':sobre:',
        'fr': u':enveloppe:',
        'pt': u':envelope:',
        'it': u':busta:'
    },
    u'\U00002709': {
        'en' : u':envelope:',
        'status' : unqualified,
        'E' : 0.6,
        'variant': True,
        'de': u':briefumschlag:',
        'es': u':sobre:',
        'fr': u':enveloppe:',
        'pt': u':envelope:',
        'it': u':busta:'
    },
    u'\U0001F4E9': {
        'en' : u':envelope_with_arrow:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':umschlag_mit_pfeil:',
        'es': u':sobre_con_flecha:',
        'fr': u':enveloppe_avec_flèche:',
        'pt': u':envelope_com_seta:',
        'it': u':posta_in_uscita:'
    },
    u'\U0001F4B6': {
        'en' : u':euro_banknote:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [u':euro:'],
        'de': u':euro-banknote:',
        'es': u':billete_de_euro:',
        'fr': u':billet_en_euros:',
        'pt': u':nota_de_euro:',
        'it': u':banconota_euro:'
    },
    u'\U0001F332': {
        'en' : u':evergreen_tree:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':nadelbaum:',
        'es': u':árbol_de_hoja_perenne:',
        'fr': u':conifère:',
        'pt': u':conífera:',
        'it': u':albero_sempreverde:'
    },
    u'\U0001F411': {
        'en' : u':ewe:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':sheep:'],
        'de': u':schaf:',
        'es': u':oveja:',
        'fr': u':mouton:',
        'pt': u':ovelha:',
        'it': u':pecora:'
    },
    u'\U00002049\U0000FE0F': {
        'en' : u':exclamation_question_mark:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':interrobang:'],
        'variant': True,
        'de': u':ausrufe-_und_fragezeichen:',
        'es': u':exclamación_e_interrogación:',
        'fr': u':points_d’exclamation_et_d’interrogation:',
        'pt': u':exclamação_com_interrogação:',
        'it': u':punto_esclamativo_e_interrogativo:'
    },
    u'\U00002049': {
        'en' : u':exclamation_question_mark:',
        'status' : unqualified,
        'E' : 0.6,
        'alias' : [u':interrobang:'],
        'variant': True,
        'de': u':ausrufe-_und_fragezeichen:',
        'es': u':exclamación_e_interrogación:',
        'fr': u':points_d’exclamation_et_d’interrogation:',
        'pt': u':exclamação_com_interrogação:',
        'it': u':punto_esclamativo_e_interrogativo:'
    },
    u'\U0001F92F': {
        'en' : u':exploding_head:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':explodierender_kopf:',
        'es': u':cabeza_explotando:',
        'fr': u':tête_qui_explose:',
        'pt': u':cabeça_explodindo:',
        'it': u':testa_che_esplode:'
    },
    u'\U0001F611': {
        'en' : u':expressionless_face:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [u':expressionless:'],
        'de': u':ausdrucksloses_gesicht:',
        'es': u':cara_sin_expresión:',
        'fr': u':visage_sans_expression:',
        'pt': u':rosto_inexpressivo:',
        'it': u':faccina_inespressiva:'
    },
    u'\U0001F441\U0000FE0F': {
        'en' : u':eye:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': u':auge:',
        'es': u':ojo:',
        'fr': u':œil:',
        'pt': u':olho:',
        'it': u':occhio:'
    },
    u'\U0001F441': {
        'en' : u':eye:',
        'status' : unqualified,
        'E' : 0.7,
        'variant': True,
        'de': u':auge:',
        'es': u':ojo:',
        'fr': u':œil:',
        'pt': u':olho:',
        'it': u':occhio:'
    },
    u'\U0001F441\U0000FE0F\U0000200D\U0001F5E8\U0000FE0F': {
        'en' : u':eye_in_speech_bubble:',
        'status' : fully_qualified,
        'E' : 2,
        'de': u':auge_in_sprechblase:',
        'es': u':ojo_en_bocadillo_de_texto:',
        'fr': u':œil_dans_une_bulle_de_bd:',
        'pt': u':olho_no_balão_de_diálogo:',
        'it': u':occhio_nel_fumetto:'
    },
    u'\U0001F441\U0000200D\U0001F5E8\U0000FE0F': {
        'en' : u':eye_in_speech_bubble:',
        'status' : unqualified,
        'E' : 2
    },
    u'\U0001F441\U0000FE0F\U0000200D\U0001F5E8': {
        'en' : u':eye_in_speech_bubble:',
        'status' : unqualified,
        'E' : 2
    },
    u'\U0001F441\U0000200D\U0001F5E8': {
        'en' : u':eye_in_speech_bubble:',
        'status' : unqualified,
        'E' : 2
    },
    u'\U0001F440': {
        'en' : u':eyes:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':augen:',
        'es': u':ojos:',
        'fr': u':yeux:',
        'pt': u':olhos:',
        'it': u':occhi:'
    },
    u'\U0001F618': {
        'en' : u':face_blowing_a_kiss:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':kissing_heart:'],
        'de': u':kuss_zuwerfendes_gesicht:',
        'es': u':cara_lanzando_un_beso:',
        'fr': u':visage_envoyant_un_bisou:',
        'pt': u':rosto_mandando_um_beijo:',
        'it': u':faccina_che_manda_un_bacio:'
    },
    u'\U0001F62E\U0000200D\U0001F4A8': {
        'en' : u':face_exhaling:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F979': {
        'en' : u':face_holding_back_tears:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001F636\U0000200D\U0001F32B\U0000FE0F': {
        'en' : u':face_in_clouds:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F636\U0000200D\U0001F32B': {
        'en' : u':face_in_clouds:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F60B': {
        'en' : u':face_savoring_food:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':yum:'],
        'de': u':sich_die_lippen_leckendes_gesicht:',
        'es': u':cara_saboreando_comida:',
        'fr': u':miam:',
        'pt': u':rosto_saboreando_comida:',
        'it': u':faccina_che_si_lecca_i_baffi:'
    },
    u'\U0001F631': {
        'en' : u':face_screaming_in_fear:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':scream:'],
        'de': u':vor_angst_schreiendes_gesicht:',
        'es': u':cara_gritando_de_miedo:',
        'fr': u':visage_qui_hurle_de_peur:',
        'pt': u':rosto_gritando_de_medo:',
        'it': u':faccina_terrorizzata:'
    },
    u'\U0001F92E': {
        'en' : u':face_vomiting:',
        'status' : fully_qualified,
        'E' : 5,
        'alias' : [u':vomiting_face:'],
        'de': u':kotzendes_gesicht:',
        'es': u':cara_vomitando:',
        'fr': u':visage_qui_vomit:',
        'pt': u':rosto_vomitando:',
        'it': u':faccina_che_vomita:'
    },
    u'\U0001F635': {
        'en' : u':face_with_crossed-out_eyes:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':dizzy_face:'],
        'de': u':benommenes_gesicht:',
        'es': u':cara_mareada:',
        'fr': u':visage_étourdi:',
        'pt': u':rosto_atordoado:',
        'it': u':faccina_frastornata:'
    },
    u'\U0001FAE4': {
        'en' : u':face_with_diagonal_mouth:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001F92D': {
        'en' : u':face_with_hand_over_mouth:',
        'status' : fully_qualified,
        'E' : 5,
        'alias' : [u':hand_over_mouth:'],
        'de': u':verlegen_kicherndes_gesicht:',
        'es': u':cara_con_mano_sobre_la_boca:',
        'fr': u':visage_avec_une_main_sur_la_bouche:',
        'pt': u':rosto_com_a_mão_sobre_a_boca:',
        'it': u':faccina_con_mano_sulla_bocca:'
    },
    u'\U0001F915': {
        'en' : u':face_with_head-bandage:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [u':face_with_head__bandage:', u':face_with_head_bandage:'],
        'de': u':gesicht_mit_kopfverband:',
        'es': u':cara_con_la_cabeza_vendada:',
        'fr': u':visage_avec_bandage_autour_de_la_tête:',
        'pt': u':rosto_com_atadura_na_cabeça:',
        'it': u':faccina_con_la_testa_bendata:'
    },
    u'\U0001F637': {
        'en' : u':face_with_medical_mask:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':mask:'],
        'de': u':gesicht_mit_atemschutzmaske:',
        'es': u':cara_con_mascarilla_médica:',
        'fr': u':visage_avec_masque:',
        'pt': u':rosto_com_máscara_médica:',
        'it': u':faccina_con_mascherina:'
    },
    u'\U0001F9D0': {
        'en' : u':face_with_monocle:',
        'status' : fully_qualified,
        'E' : 5,
        'alias' : [u':monocle_face:'],
        'de': u':gesicht_mit_monokel:',
        'es': u':cara_con_monóculo:',
        'fr': u':visage_avec_un_monocle:',
        'pt': u':rosto_com_monóculo:',
        'it': u':faccina_con_monocolo:'
    },
    u'\U0001FAE2': {
        'en' : u':face_with_open_eyes_and_hand_over_mouth:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001F62E': {
        'en' : u':face_with_open_mouth:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [u':open_mouth:'],
        'de': u':gesicht_mit_offenem_mund:',
        'es': u':cara_con_la_boca_abierta:',
        'fr': u':visage_avec_bouche_ouverte:',
        'pt': u':rosto_com_boca_aberta:',
        'it': u':faccina_con_bocca_aperta:'
    },
    u'\U0001FAE3': {
        'en' : u':face_with_peeking_eye:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001F928': {
        'en' : u':face_with_raised_eyebrow:',
        'status' : fully_qualified,
        'E' : 5,
        'alias' : [u':raised_eyebrow:'],
        'de': u':gesicht_mit_hochgezogenen_augenbrauen:',
        'es': u':cara_con_ceja_alzada:',
        'fr': u':visage_avec_les_sourcils_relevés:',
        'pt': u':rosto_com_sobrancelha_levantada:',
        'it': u':faccia_con_sopracciglia_alzate:'
    },
    u'\U0001F644': {
        'en' : u':face_with_rolling_eyes:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [u':roll_eyes:'],
        'de': u':augen_verdrehendes_gesicht:',
        'es': u':cara_con_ojos_en_blanco:',
        'fr': u':visage_roulant_des_yeux:',
        'pt': u':rosto_com_olhos_revirados:',
        'it': u':faccina_con_occhi_al_cielo:'
    },
    u'\U0001F635\U0000200D\U0001F4AB': {
        'en' : u':face_with_spiral_eyes:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F624': {
        'en' : u':face_with_steam_from_nose:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':triumph:'],
        'de': u':schnaubendes_gesicht:',
        'es': u':cara_resoplando:',
        'fr': u':visage_avec_fumée_sortant_des_narines:',
        'pt': u':rosto_soltando_vapor_pelo_nariz:',
        'it': u':faccina_che_sbuffa:'
    },
    u'\U0001F92C': {
        'en' : u':face_with_symbols_on_mouth:',
        'status' : fully_qualified,
        'E' : 5,
        'alias' : [u':cursing_face:'],
        'de': u':gesicht_mit_symbolen_über_dem_mund:',
        'es': u':cara_con_símbolos_en_la_boca:',
        'fr': u':visage_avec_des_symboles_dans_la_bouche:',
        'pt': u':rosto_com_símbolos_na_boca:',
        'it': u':faccina_con_simboli_sulla_bocca:'
    },
    u'\U0001F602': {
        'en' : u':face_with_tears_of_joy:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':joy:'],
        'de': u':gesicht_mit_freudentränen:',
        'es': u':cara_llorando_de_risa:',
        'fr': u':visage_riant_aux_larmes:',
        'pt': u':rosto_chorando_de_rir:',
        'it': u':faccina_con_lacrime_di_gioia:'
    },
    u'\U0001F912': {
        'en' : u':face_with_thermometer:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':gesicht_mit_fieberthermometer:',
        'es': u':cara_con_termómetro:',
        'fr': u':visage_avec_thermomètre:',
        'pt': u':rosto_com_termômetro:',
        'it': u':faccina_con_termometro:'
    },
    u'\U0001F61B': {
        'en' : u':face_with_tongue:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [u':stuck_out_tongue:'],
        'de': u':gesicht_mit_herausgestreckter_zunge:',
        'es': u':cara_sacando_la_lengua:',
        'fr': u':visage_qui_tire_la_langue:',
        'pt': u':rosto_mostrando_a_língua:',
        'it': u':faccina_che_mostra_la_lingua:'
    },
    u'\U0001F636': {
        'en' : u':face_without_mouth:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [u':no_mouth:'],
        'de': u':gesicht_ohne_mund:',
        'es': u':cara_sin_boca:',
        'fr': u':visage_sans_bouche:',
        'pt': u':rosto_sem_boca:',
        'it': u':faccina_senza_bocca:'
    },
    u'\U0001F3ED': {
        'en' : u':factory:',
        'status' : fully_qualified,
        'E' : 0.6,
        'variant': True,
        'de': u':fabrik:',
        'es': u':fábrica:',
        'fr': u':usine:',
        'pt': u':fábrica:',
        'it': u':fabbrica:'
    },
    u'\U0001F9D1\U0000200D\U0001F3ED': {
        'en' : u':factory_worker:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':fabrikarbeiter(in):',
        'es': u':profesional_industrial:',
        'fr': u':ouvrier_(tous_genres):',
        'pt': u':funcionário_de_fábrica:',
        'it': u':persona_che_lavora_in_fabbrica:'
    },
    u'\U0001F9D1\U0001F3FF\U0000200D\U0001F3ED': {
        'en' : u':factory_worker_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':fabrikarbeiter(in)_dunkle_hautfarbe:',
        'es': u':profesional_industrial_tono_de_piel_oscuro:',
        'fr': u':ouvrier_(tous_genres)_peau_foncée:',
        'pt': u':funcionário_de_fábrica_pele_escura:',
        'it': u':persona_che_lavora_in_fabbrica_carnagione_scura:'
    },
    u'\U0001F9D1\U0001F3FB\U0000200D\U0001F3ED': {
        'en' : u':factory_worker_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':fabrikarbeiter(in)_helle_hautfarbe:',
        'es': u':profesional_industrial_tono_de_piel_claro:',
        'fr': u':ouvrier_(tous_genres)_peau_claire:',
        'pt': u':funcionário_de_fábrica_pele_clara:',
        'it': u':persona_che_lavora_in_fabbrica_carnagione_chiara:'
    },
    u'\U0001F9D1\U0001F3FE\U0000200D\U0001F3ED': {
        'en' : u':factory_worker_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':fabrikarbeiter(in)_mitteldunkle_hautfarbe:',
        'es': u':profesional_industrial_tono_de_piel_oscuro_medio:',
        'fr': u':ouvrier_(tous_genres)_peau_mate:',
        'pt': u':funcionário_de_fábrica_pele_morena_escura:',
        'it': u':persona_che_lavora_in_fabbrica_carnagione_abbastanza_scura:'
    },
    u'\U0001F9D1\U0001F3FC\U0000200D\U0001F3ED': {
        'en' : u':factory_worker_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':fabrikarbeiter(in)_mittelhelle_hautfarbe:',
        'es': u':profesional_industrial_tono_de_piel_claro_medio:',
        'fr': u':ouvrier_(tous_genres)_peau_moyennement_claire:',
        'pt': u':funcionário_de_fábrica_pele_morena_clara:',
        'it': u':persona_che_lavora_in_fabbrica_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9D1\U0001F3FD\U0000200D\U0001F3ED': {
        'en' : u':factory_worker_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':fabrikarbeiter(in)_mittlere_hautfarbe:',
        'es': u':profesional_industrial_tono_de_piel_medio:',
        'fr': u':ouvrier_(tous_genres)_peau_légèrement_mate:',
        'pt': u':funcionário_de_fábrica_pele_morena:',
        'it': u':persona_che_lavora_in_fabbrica_carnagione_olivastra:'
    },
    u'\U0001F9DA': {
        'en' : u':fairy:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':märchenfee:',
        'es': u':hada:',
        'fr': u':personnage_féérique:',
        'pt': u':fada:',
        'it': u':fata:'
    },
    u'\U0001F9DA\U0001F3FF': {
        'en' : u':fairy_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':märchenfee_dunkle_hautfarbe:',
        'es': u':hada_tono_de_piel_oscuro:',
        'fr': u':personnage_féérique_peau_foncée:',
        'pt': u':fada_pele_escura:',
        'it': u':fata_carnagione_scura:'
    },
    u'\U0001F9DA\U0001F3FB': {
        'en' : u':fairy_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':märchenfee_helle_hautfarbe:',
        'es': u':hada_tono_de_piel_claro:',
        'fr': u':personnage_féérique_peau_claire:',
        'pt': u':fada_pele_clara:',
        'it': u':fata_carnagione_chiara:'
    },
    u'\U0001F9DA\U0001F3FE': {
        'en' : u':fairy_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':märchenfee_mitteldunkle_hautfarbe:',
        'es': u':hada_tono_de_piel_oscuro_medio:',
        'fr': u':personnage_féérique_peau_mate:',
        'pt': u':fada_pele_morena_escura:',
        'it': u':fata_carnagione_abbastanza_scura:'
    },
    u'\U0001F9DA\U0001F3FC': {
        'en' : u':fairy_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':märchenfee_mittelhelle_hautfarbe:',
        'es': u':hada_tono_de_piel_claro_medio:',
        'fr': u':personnage_féérique_peau_moyennement_claire:',
        'pt': u':fada_pele_morena_clara:',
        'it': u':fata_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9DA\U0001F3FD': {
        'en' : u':fairy_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':märchenfee_mittlere_hautfarbe:',
        'es': u':hada_tono_de_piel_medio:',
        'fr': u':personnage_féérique_peau_légèrement_mate:',
        'pt': u':fada_pele_morena:',
        'it': u':fata_carnagione_olivastra:'
    },
    u'\U0001F9C6': {
        'en' : u':falafel:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':falafel:',
        'es': u':falafel:',
        'fr': u':falafels:',
        'pt': u':falafel:',
        'it': u':falafel:'
    },
    u'\U0001F342': {
        'en' : u':fallen_leaf:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':laub:',
        'es': u':hojas_caídas:',
        'fr': u':feuille_morte:',
        'pt': u':folhas_caídas:',
        'it': u':foglia_caduta:'
    },
    u'\U0001F46A': {
        'en' : u':family:',
        'status' : fully_qualified,
        'E' : 0.6,
        'variant': True,
        'de': u':familie:',
        'es': u':familia:',
        'fr': u':famille:',
        'pt': u':família:',
        'it': u':famiglia:'
    },
    u'\U0001F468\U0000200D\U0001F466': {
        'en' : u':family_man_boy:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':familie_mann,_junge:',
        'es': u':familia_hombre_y_niño:',
        'fr': u':famille_homme_et_garçon:',
        'pt': u':família_homem_e_menino:',
        'it': u':famiglia_uomo_e_bambino:'
    },
    u'\U0001F468\U0000200D\U0001F466\U0000200D\U0001F466': {
        'en' : u':family_man_boy_boy:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':familie_mann,_junge_und_junge:',
        'es': u':familia_hombre_niño_niño:',
        'fr': u':famille_homme,_garçon_et_garçon:',
        'pt': u':família_homem_menino_e_menino:',
        'it': u':famiglia_uomo_bambino_e_bambino:'
    },
    u'\U0001F468\U0000200D\U0001F467': {
        'en' : u':family_man_girl:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':familie_mann,_mädchen:',
        'es': u':familia_hombre_y_niña:',
        'fr': u':famille_homme_et_fille:',
        'pt': u':família_homem_e_menina:',
        'it': u':famiglia_uomo_e_bambina:'
    },
    u'\U0001F468\U0000200D\U0001F467\U0000200D\U0001F466': {
        'en' : u':family_man_girl_boy:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':familie_mann,_mädchen_und_junge:',
        'es': u':familia_hombre_niña_niño:',
        'fr': u':famille_homme,_fille_et_garçon:',
        'pt': u':família_homem_menina_e_menino:',
        'it': u':famiglia_uomo_bambina_e_bambino:'
    },
    u'\U0001F468\U0000200D\U0001F467\U0000200D\U0001F467': {
        'en' : u':family_man_girl_girl:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':familie_mann,_mädchen_und_mädchen:',
        'es': u':familia_hombre_niña_niña:',
        'fr': u':famille_homme,_fille_et_fille:',
        'pt': u':família_homem_menina_e_menina:',
        'it': u':famiglia_uomo_bambina_e_bambina:'
    },
    u'\U0001F468\U0000200D\U0001F468\U0000200D\U0001F466': {
        'en' : u':family_man_man_boy:',
        'status' : fully_qualified,
        'E' : 2,
        'de': u':familie_mann,_mann_und_junge:',
        'es': u':familia_hombre_hombre_niño:',
        'fr': u':famille_homme,_homme_et_garçon:',
        'pt': u':família_homem_homem_e_menino:',
        'it': u':famiglia_uomo_uomo_e_bambino:'
    },
    u'\U0001F468\U0000200D\U0001F468\U0000200D\U0001F466\U0000200D\U0001F466': {
        'en' : u':family_man_man_boy_boy:',
        'status' : fully_qualified,
        'E' : 2,
        'de': u':familie_mann,_mann,_junge_und_junge:',
        'es': u':familia_hombre_hombre_niño_niño:',
        'fr': u':famille_homme,_homme,_garçon_et_garçon:',
        'pt': u':família_homem_homem_menino_e_menino:',
        'it': u':famiglia_uomo_uomo_bambino_e_bambino:'
    },
    u'\U0001F468\U0000200D\U0001F468\U0000200D\U0001F467': {
        'en' : u':family_man_man_girl:',
        'status' : fully_qualified,
        'E' : 2,
        'de': u':familie_mann,_mann_und_mädchen:',
        'es': u':familia_hombre_hombre_niña:',
        'fr': u':famille_homme,_homme_et_fille:',
        'pt': u':família_homem_homem_e_menina:',
        'it': u':famiglia_uomo_uomo_e_bambina:'
    },
    u'\U0001F468\U0000200D\U0001F468\U0000200D\U0001F467\U0000200D\U0001F466': {
        'en' : u':family_man_man_girl_boy:',
        'status' : fully_qualified,
        'E' : 2,
        'de': u':familie_mann,_mann,_mädchen_und_junge:',
        'es': u':familia_hombre_hombre_niña_niño:',
        'fr': u':famille_homme,_homme,_fille_et_garçon:',
        'pt': u':família_homem_homem_menina_e_menino:',
        'it': u':famiglia_uomo_uomo_bambina_e_bambino:'
    },
    u'\U0001F468\U0000200D\U0001F468\U0000200D\U0001F467\U0000200D\U0001F467': {
        'en' : u':family_man_man_girl_girl:',
        'status' : fully_qualified,
        'E' : 2,
        'de': u':familie_mann,_mann,_mädchen_und_mädchen:',
        'es': u':familia_hombre_hombre_niña_niña:',
        'fr': u':famille_homme,_homme,_fille_et_fille:',
        'pt': u':família_homem_homem_menina_e_menina:',
        'it': u':famiglia_uomo_uomo_bambina_e_bambina:'
    },
    u'\U0001F468\U0000200D\U0001F469\U0000200D\U0001F466': {
        'en' : u':family_man_woman_boy:',
        'status' : fully_qualified,
        'E' : 2,
        'de': u':familie_mann,_frau_und_junge:',
        'es': u':familia_hombre_mujer_niño:',
        'fr': u':famille_homme,_femme_et_garçon:',
        'pt': u':família_homem_mulher_e_menino:',
        'it': u':famiglia_uomo_donna_e_bambino:'
    },
    u'\U0001F468\U0000200D\U0001F469\U0000200D\U0001F466\U0000200D\U0001F466': {
        'en' : u':family_man_woman_boy_boy:',
        'status' : fully_qualified,
        'E' : 2,
        'de': u':familie_mann,_frau,_junge_und_junge:',
        'es': u':familia_hombre_mujer_niño_niño:',
        'fr': u':famille_homme,_femme,_garçon_et_garçon:',
        'pt': u':família_homem_mulher_menino_e_menino:',
        'it': u':famiglia_uomo_donna_bambino_e_bambino:'
    },
    u'\U0001F468\U0000200D\U0001F469\U0000200D\U0001F467': {
        'en' : u':family_man_woman_girl:',
        'status' : fully_qualified,
        'E' : 2,
        'de': u':familie_mann,_frau_und_mädchen:',
        'es': u':familia_hombre_mujer_niña:',
        'fr': u':famille_homme,_femme_et_fille:',
        'pt': u':família_homem_mulher_e_menina:',
        'it': u':famiglia_uomo_donna_e_bambina:'
    },
    u'\U0001F468\U0000200D\U0001F469\U0000200D\U0001F467\U0000200D\U0001F466': {
        'en' : u':family_man_woman_girl_boy:',
        'status' : fully_qualified,
        'E' : 2,
        'de': u':familie_mann,_frau,_mädchen_und_junge:',
        'es': u':familia_hombre_mujer_niña_niño:',
        'fr': u':famille_homme,_femme,_fille_et_garçon:',
        'pt': u':família_homem_mulher_menina_e_menino:',
        'it': u':famiglia_uomo_donna_bambina_e_bambino:'
    },
    u'\U0001F468\U0000200D\U0001F469\U0000200D\U0001F467\U0000200D\U0001F467': {
        'en' : u':family_man_woman_girl_girl:',
        'status' : fully_qualified,
        'E' : 2,
        'de': u':familie_mann,_frau,_mädchen_und_mädchen:',
        'es': u':familia_hombre_mujer_niña_niña:',
        'fr': u':famille_homme,_femme,_fille_et_fille:',
        'pt': u':família_homem_mulher_menina_e_menina:',
        'it': u':famiglia_uomo_donna_bambina_e_bambina:'
    },
    u'\U0001F469\U0000200D\U0001F466': {
        'en' : u':family_woman_boy:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':familie_frau,_junge:',
        'es': u':familia_mujer_y_niño:',
        'fr': u':famille_femme_et_garçon:',
        'pt': u':família_mulher_e_menino:',
        'it': u':famiglia_donna_e_bambino:'
    },
    u'\U0001F469\U0000200D\U0001F466\U0000200D\U0001F466': {
        'en' : u':family_woman_boy_boy:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':familie_frau,_junge_und_junge:',
        'es': u':familia_mujer_niño_niño:',
        'fr': u':famille_femme,_garçon_et_garçon:',
        'pt': u':família_mulher_menino_e_menino:',
        'it': u':famiglia_donna_bambino_e_bambino:'
    },
    u'\U0001F469\U0000200D\U0001F467': {
        'en' : u':family_woman_girl:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':familie_frau,_mädchen:',
        'es': u':familia_mujer_y_niña:',
        'fr': u':famille_femme_et_fille:',
        'pt': u':família_mulher_e_menina:',
        'it': u':famiglia_donna_e_bambina:'
    },
    u'\U0001F469\U0000200D\U0001F467\U0000200D\U0001F466': {
        'en' : u':family_woman_girl_boy:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':familie_frau,_mädchen_und_junge:',
        'es': u':familia_mujer_niña_niño:',
        'fr': u':famille_femme,_fille_et_garçon:',
        'pt': u':família_mulher_menina_e_menino:',
        'it': u':famiglia_donna_bambina_e_bambino:'
    },
    u'\U0001F469\U0000200D\U0001F467\U0000200D\U0001F467': {
        'en' : u':family_woman_girl_girl:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':familie_frau,_mädchen_und_mädchen:',
        'es': u':familia_mujer_niña_niña:',
        'fr': u':famille_femme,_fille_et_fille:',
        'pt': u':família_mulher_menina_e_menina:',
        'it': u':famiglia_donna_bambina_e_bambina:'
    },
    u'\U0001F469\U0000200D\U0001F469\U0000200D\U0001F466': {
        'en' : u':family_woman_woman_boy:',
        'status' : fully_qualified,
        'E' : 2,
        'de': u':familie_frau,_frau_und_junge:',
        'es': u':familia_mujer_mujer_niño:',
        'fr': u':famille_femme,_femme_et_garçon:',
        'pt': u':família_mulher_mulher_e_menino:',
        'it': u':famiglia_donna_donna_e_bambino:'
    },
    u'\U0001F469\U0000200D\U0001F469\U0000200D\U0001F466\U0000200D\U0001F466': {
        'en' : u':family_woman_woman_boy_boy:',
        'status' : fully_qualified,
        'E' : 2,
        'de': u':familie_frau,_frau,_junge_und_junge:',
        'es': u':familia_mujer_mujer_niño_niño:',
        'fr': u':famille_femme,_femme,_garçon_et_garçon:',
        'pt': u':família_mulher_mulher_menino_e_menino:',
        'it': u':famiglia_donna_donna_bambino_e_bambino:'
    },
    u'\U0001F469\U0000200D\U0001F469\U0000200D\U0001F467': {
        'en' : u':family_woman_woman_girl:',
        'status' : fully_qualified,
        'E' : 2,
        'de': u':familie_frau,_frau_und_mädchen:',
        'es': u':familia_mujer_mujer_niña:',
        'fr': u':famille_femme,_femme_et_fille:',
        'pt': u':família_mulher_mulher_e_menina:',
        'it': u':famiglia_donna_donna_e_bambina:'
    },
    u'\U0001F469\U0000200D\U0001F469\U0000200D\U0001F467\U0000200D\U0001F466': {
        'en' : u':family_woman_woman_girl_boy:',
        'status' : fully_qualified,
        'E' : 2,
        'de': u':familie_frau,_frau,_mädchen_und_junge:',
        'es': u':familia_mujer_mujer_niña_niño:',
        'fr': u':famille_femme,_femme,_fille_et_garçon:',
        'pt': u':família_mulher_mulher_menina_e_menino:',
        'it': u':famiglia_donna_donna_bambina_e_bambino:'
    },
    u'\U0001F469\U0000200D\U0001F469\U0000200D\U0001F467\U0000200D\U0001F467': {
        'en' : u':family_woman_woman_girl_girl:',
        'status' : fully_qualified,
        'E' : 2,
        'de': u':familie_frau,_frau,_mädchen_und_mädchen:',
        'es': u':familia_mujer_mujer_niña_niña:',
        'fr': u':famille_femme,_femme,_fille_et_fille:',
        'pt': u':família_mulher_mulher_menina_e_menina:',
        'it': u':famiglia_donna_donna_bambina_e_bambina:'
    },
    u'\U0001F9D1\U0000200D\U0001F33E': {
        'en' : u':farmer:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':bauer/bäuerin:',
        'es': u':profesional_de_la_agricultura:',
        'fr': u':fermier_(tous_genres):',
        'pt': u':agricultor:',
        'it': u':agricoltore:'
    },
    u'\U0001F9D1\U0001F3FF\U0000200D\U0001F33E': {
        'en' : u':farmer_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':bauer/bäuerin_dunkle_hautfarbe:',
        'es': u':profesional_de_la_agricultura_tono_de_piel_oscuro:',
        'fr': u':fermier_(tous_genres)_peau_foncée:',
        'pt': u':agricultor_pele_escura:',
        'it': u':agricoltore_carnagione_scura:'
    },
    u'\U0001F9D1\U0001F3FB\U0000200D\U0001F33E': {
        'en' : u':farmer_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':bauer/bäuerin_helle_hautfarbe:',
        'es': u':profesional_de_la_agricultura_tono_de_piel_claro:',
        'fr': u':fermier_(tous_genres)_peau_claire:',
        'pt': u':agricultor_pele_clara:',
        'it': u':agricoltore_carnagione_chiara:'
    },
    u'\U0001F9D1\U0001F3FE\U0000200D\U0001F33E': {
        'en' : u':farmer_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':bauer/bäuerin_mitteldunkle_hautfarbe:',
        'es': u':profesional_de_la_agricultura_tono_de_piel_oscuro_medio:',
        'fr': u':fermier_(tous_genres)_peau_mate:',
        'pt': u':agricultor_pele_morena_escura:',
        'it': u':agricoltore_carnagione_abbastanza_scura:'
    },
    u'\U0001F9D1\U0001F3FC\U0000200D\U0001F33E': {
        'en' : u':farmer_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':bauer/bäuerin_mittelhelle_hautfarbe:',
        'es': u':profesional_de_la_agricultura_tono_de_piel_claro_medio:',
        'fr': u':fermier_(tous_genres)_peau_moyennement_claire:',
        'pt': u':agricultor_pele_morena_clara:',
        'it': u':agricoltore_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9D1\U0001F3FD\U0000200D\U0001F33E': {
        'en' : u':farmer_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':bauer/bäuerin_mittlere_hautfarbe:',
        'es': u':profesional_de_la_agricultura_tono_de_piel_medio:',
        'fr': u':fermier_(tous_genres)_peau_légèrement_mate:',
        'pt': u':agricultor_pele_morena:',
        'it': u':agricoltore_carnagione_olivastra:'
    },
    u'\U000023E9': {
        'en' : u':fast-forward_button:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':fast_forward:'],
        'variant': True,
        'de': u':doppelpfeile_nach_rechts:',
        'es': u':avance_rápido:',
        'fr': u':bouton_avance_rapide:',
        'pt': u':botão_avançar:',
        'it': u':pulsante_di_avanzamento_rapido:'
    },
    u'\U000023EC': {
        'en' : u':fast_down_button:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':arrow_double_down:'],
        'de': u':doppelpfeile_nach_unten:',
        'es': u':triángulo_doble_hacia_abajo:',
        'fr': u':double_flèche_vers_le_bas:',
        'pt': u':botão_de_avanço_para_baixo:',
        'it': u':doppia_freccia_in_basso:'
    },
    u'\U000023EA': {
        'en' : u':fast_reverse_button:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':rewind:'],
        'variant': True,
        'de': u':zurückspulen:',
        'es': u':rebobinado:',
        'fr': u':bouton_retour_rapide:',
        'pt': u':botão_de_retroceder:',
        'it': u':pulsante_di_riavvolgimento_rapido:'
    },
    u'\U000023EB': {
        'en' : u':fast_up_button:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':arrow_double_up:'],
        'de': u':doppelpfeile_nach_oben:',
        'es': u':triángulo_doble_hacia_arriba:',
        'fr': u':double_flèche_vers_le_haut:',
        'pt': u':botão_de_avanço_para_cima:',
        'it': u':pulsante_doppia_freccia_in_alto:'
    },
    u'\U0001F4E0': {
        'en' : u':fax_machine:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':fax:'],
        'de': u':faxgerät:',
        'es': u':máquina_de_fax:',
        'fr': u':fax:',
        'pt': u':fax:',
        'it': u':fax:'
    },
    u'\U0001F628': {
        'en' : u':fearful_face:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':fearful:'],
        'de': u':ängstliches_gesicht:',
        'es': u':cara_asustada:',
        'fr': u':visage_effrayé:',
        'pt': u':rosto_amedrontado:',
        'it': u':faccina_impaurita:'
    },
    u'\U0001FAB6': {
        'en' : u':feather:',
        'status' : fully_qualified,
        'E' : 13,
        'de': u':feder:',
        'es': u':pluma:',
        'fr': u':plume:',
        'pt': u':pena:',
        'it': u':piuma:'
    },
    u'\U00002640\U0000FE0F': {
        'en' : u':female_sign:',
        'status' : fully_qualified,
        'E' : 4,
        'variant': True,
        'de': u':frauensymbol:',
        'es': u':signo_femenino:',
        'fr': u':symbole_de_la_femme:',
        'pt': u':símbolo_de_feminino:',
        'it': u':simbolo_genere_femminile:'
    },
    u'\U00002640': {
        'en' : u':female_sign:',
        'status' : unqualified,
        'E' : 4,
        'variant': True,
        'de': u':frauensymbol:',
        'es': u':signo_femenino:',
        'fr': u':symbole_de_la_femme:',
        'pt': u':símbolo_de_feminino:',
        'it': u':simbolo_genere_femminile:'
    },
    u'\U0001F3A1': {
        'en' : u':ferris_wheel:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':riesenrad:',
        'es': u':noria_de_feria:',
        'fr': u':grande_roue:',
        'pt': u':roda_gigante:',
        'it': u':ruota_panoramica:'
    },
    u'\U000026F4\U0000FE0F': {
        'en' : u':ferry:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': u':fähre:',
        'es': u':ferri:',
        'fr': u':ferry:',
        'pt': u':balsa:',
        'it': u':traghetto:'
    },
    u'\U000026F4': {
        'en' : u':ferry:',
        'status' : unqualified,
        'E' : 0.7,
        'variant': True,
        'de': u':fähre:',
        'es': u':ferri:',
        'fr': u':ferry:',
        'pt': u':balsa:',
        'it': u':traghetto:'
    },
    u'\U0001F3D1': {
        'en' : u':field_hockey:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [u':field_hockey_stick_and_ball:'],
        'de': u':feldhockey:',
        'es': u':hockey_sobre_hierba:',
        'fr': u':hockey_sur_gazon:',
        'pt': u':hóquei_de_campo:',
        'it': u':hockey_su_prato:'
    },
    u'\U0001F5C4\U0000FE0F': {
        'en' : u':file_cabinet:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': u':aktenschrank:',
        'es': u':archivador:',
        'fr': u':meuble_à_dossiers:',
        'pt': u':gavetas_de_escritório:',
        'it': u':schedario_da_ufficio:'
    },
    u'\U0001F5C4': {
        'en' : u':file_cabinet:',
        'status' : unqualified,
        'E' : 0.7,
        'variant': True,
        'de': u':aktenschrank:',
        'es': u':archivador:',
        'fr': u':meuble_à_dossiers:',
        'pt': u':gavetas_de_escritório:',
        'it': u':schedario_da_ufficio:'
    },
    u'\U0001F4C1': {
        'en' : u':file_folder:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':ordner:',
        'es': u':carpeta_de_archivos:',
        'fr': u':dossier:',
        'pt': u':pasta_de_arquivos:',
        'it': u':cartella_file:'
    },
    u'\U0001F39E\U0000FE0F': {
        'en' : u':film_frames:',
        'status' : fully_qualified,
        'E' : 0.7,
        'alias' : [u':film_strip:'],
        'variant': True,
        'de': u':filmstreifen:',
        'es': u':fotograma_de_película:',
        'fr': u':pellicule:',
        'pt': u':rolo_de_filmes:',
        'it': u':pellicola_cinematografica:'
    },
    u'\U0001F39E': {
        'en' : u':film_frames:',
        'status' : unqualified,
        'E' : 0.7,
        'alias' : [u':film_strip:'],
        'variant': True,
        'de': u':filmstreifen:',
        'es': u':fotograma_de_película:',
        'fr': u':pellicule:',
        'pt': u':rolo_de_filmes:',
        'it': u':pellicola_cinematografica:'
    },
    u'\U0001F4FD\U0000FE0F': {
        'en' : u':film_projector:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': u':filmprojektor:',
        'es': u':proyector_de_cine:',
        'fr': u':projecteur_cinématographique:',
        'pt': u':projetor_de_filmes:',
        'it': u':proiettore_cinematografico:'
    },
    u'\U0001F4FD': {
        'en' : u':film_projector:',
        'status' : unqualified,
        'E' : 0.7,
        'variant': True,
        'de': u':filmprojektor:',
        'es': u':proyector_de_cine:',
        'fr': u':projecteur_cinématographique:',
        'pt': u':projetor_de_filmes:',
        'it': u':proiettore_cinematografico:'
    },
    u'\U0001F525': {
        'en' : u':fire:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':feuer:',
        'es': u':fuego:',
        'fr': u':feu:',
        'pt': u':fogo:',
        'it': u':fuoco:'
    },
    u'\U0001F692': {
        'en' : u':fire_engine:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':feuerwehrauto:',
        'es': u':coche_de_bomberos:',
        'fr': u':camion_de_pompier:',
        'pt': u':carro_do_corpo_de_bombeiros:',
        'it': u':camion_dei_pompieri:'
    },
    u'\U0001F9EF': {
        'en' : u':fire_extinguisher:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':feuerlöscher:',
        'es': u':extintor:',
        'fr': u':extincteur:',
        'pt': u':extintor_de_incêndio:',
        'it': u':estintore:'
    },
    u'\U0001F9E8': {
        'en' : u':firecracker:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':feuerwerkskörper:',
        'es': u':petardo:',
        'fr': u':pétard:',
        'pt': u':bombinha:',
        'it': u':petardo:'
    },
    u'\U0001F9D1\U0000200D\U0001F692': {
        'en' : u':firefighter:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':feuerwehrmann/-frau:',
        'es': u':bombero:',
        'fr': u':pompier:',
        'pt': u':bombeiro:',
        'it': u':pompiere:'
    },
    u'\U0001F9D1\U0001F3FF\U0000200D\U0001F692': {
        'en' : u':firefighter_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':feuerwehrmann/-frau_dunkle_hautfarbe:',
        'es': u':bombero_tono_de_piel_oscuro:',
        'fr': u':pompier_peau_foncée:',
        'pt': u':bombeiro_pele_escura:',
        'it': u':pompiere_carnagione_scura:'
    },
    u'\U0001F9D1\U0001F3FB\U0000200D\U0001F692': {
        'en' : u':firefighter_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':feuerwehrmann/-frau_helle_hautfarbe:',
        'es': u':bombero_tono_de_piel_claro:',
        'fr': u':pompier_peau_claire:',
        'pt': u':bombeiro_pele_clara:',
        'it': u':pompiere_carnagione_chiara:'
    },
    u'\U0001F9D1\U0001F3FE\U0000200D\U0001F692': {
        'en' : u':firefighter_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':feuerwehrmann/-frau_mitteldunkle_hautfarbe:',
        'es': u':bombero_tono_de_piel_oscuro_medio:',
        'fr': u':pompier_peau_mate:',
        'pt': u':bombeiro_pele_morena_escura:',
        'it': u':pompiere_carnagione_abbastanza_scura:'
    },
    u'\U0001F9D1\U0001F3FC\U0000200D\U0001F692': {
        'en' : u':firefighter_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':feuerwehrmann/-frau_mittelhelle_hautfarbe:',
        'es': u':bombero_tono_de_piel_claro_medio:',
        'fr': u':pompier_peau_moyennement_claire:',
        'pt': u':bombeiro_pele_morena_clara:',
        'it': u':pompiere_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9D1\U0001F3FD\U0000200D\U0001F692': {
        'en' : u':firefighter_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':feuerwehrmann/-frau_mittlere_hautfarbe:',
        'es': u':bombero_tono_de_piel_medio:',
        'fr': u':pompier_peau_légèrement_mate:',
        'pt': u':bombeiro_pele_morena:',
        'it': u':pompiere_carnagione_olivastra:'
    },
    u'\U0001F386': {
        'en' : u':fireworks:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':feuerwerk:',
        'es': u':fuegos_artificiales:',
        'fr': u':feu_d’artifice:',
        'pt': u':fogos_de_artifício:',
        'it': u':fuochi_d’artificio:'
    },
    u'\U0001F313': {
        'en' : u':first_quarter_moon:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':zunehmender_halbmond:',
        'es': u':luna_en_cuarto_creciente:',
        'fr': u':premier_quartier_de_lune:',
        'pt': u':quarto_crescente:',
        'it': u':primo_quarto_di_luna:'
    },
    u'\U0001F31B': {
        'en' : u':first_quarter_moon_face:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':first_quarter_moon_with_face:'],
        'de': u':mondsichel_mit_gesicht_links:',
        'es': u':luna_de_cuarto_creciente_con_cara:',
        'fr': u':premier_quartier_de_lune_avec_visage:',
        'pt': u':rosto_da_lua_de_quarto_crescente:',
        'it': u':faccina_primo_quarto_di_luna:'
    },
    u'\U0001F41F': {
        'en' : u':fish:',
        'status' : fully_qualified,
        'E' : 0.6,
        'variant': True,
        'de': u':fisch:',
        'es': u':pez:',
        'fr': u':poisson:',
        'pt': u':peixe:',
        'it': u':pesce:'
    },
    u'\U0001F365': {
        'en' : u':fish_cake_with_swirl:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':fish_cake:'],
        'de': u':fischfrikadelle:',
        'es': u':pastel_de_pescado_japonés:',
        'fr': u':croquette_de_poisson:',
        'pt': u':bolinho_de_peixe:',
        'it': u':tortino_di_pesce_a_spirale:'
    },
    u'\U0001F3A3': {
        'en' : u':fishing_pole:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':fishing_pole_and_fish:'],
        'de': u':angel_mit_fisch:',
        'es': u':caña_de_pescar:',
        'fr': u':pêche_à_la_ligne:',
        'pt': u':pesca:',
        'it': u':canna_da_pesca:'
    },
    u'\U0001F560': {
        'en' : u':five-thirty:',
        'status' : fully_qualified,
        'E' : 0.7,
        'alias' : [u':clock530:'],
        'variant': True,
        'de': u':5.30_uhr:',
        'es': u':cinco_y_media:',
        'fr': u':cinq_heures_et_demie:',
        'pt': u':cinco_e_meia:',
        'it': u':ore_cinque_e_mezza:'
    },
    u'\U0001F554': {
        'en' : u':five_o’clock:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':clock5:'],
        'variant': True,
        'de': u':5.00_uhr:',
        'es': u':5_en_punto:',
        'fr': u':cinq_heures:',
        'pt': u':5_horas:',
        'it': u':ore_cinque:'
    },
    u'\U000026F3': {
        'en' : u':flag_in_hole:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':golf:'],
        'variant': True,
        'de': u':golffahne:',
        'es': u':banderín_en_hoyo:',
        'fr': u':drapeau_de_golf:',
        'pt': u':bandeira_no_buraco:',
        'it': u':bandiera_in_buca:'
    },
    u'\U0001F9A9': {
        'en' : u':flamingo:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':flamingo:',
        'es': u':flamenco:',
        'fr': u':flamant:',
        'pt': u':flamingo:',
        'it': u':fenicottero:'
    },
    u'\U0001F526': {
        'en' : u':flashlight:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':taschenlampe:',
        'es': u':linterna:',
        'fr': u':torche:',
        'pt': u':lanterna:',
        'it': u':torcia:'
    },
    u'\U0001F97F': {
        'en' : u':flat_shoe:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':flacher_schuh:',
        'es': u':bailarina:',
        'fr': u':chaussure_plate:',
        'pt': u':sapatilha:',
        'it': u':ballerina:'
    },
    u'\U0001FAD3': {
        'en' : u':flatbread:',
        'status' : fully_qualified,
        'E' : 13,
        'de': u':fladenbrot:',
        'es': u':pan_sin_levadura:',
        'fr': u':galette:',
        'pt': u':pão_sírio:',
        'it': u':focaccia:'
    },
    u'\U0000269C\U0000FE0F': {
        'en' : u':fleur-de-lis:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [u':fleur_de_lis:', u':fleur__de__lis:'],
        'variant': True,
        'de': u':lilie:',
        'es': u':flor_de_lis:',
        'fr': u':fleur_de_lys:',
        'pt': u':flor-de-lis:',
        'it': u':giglio_araldico:'
    },
    u'\U0000269C': {
        'en' : u':fleur-de-lis:',
        'status' : unqualified,
        'E' : 1,
        'alias' : [u':fleur_de_lis:', u':fleur__de__lis:'],
        'variant': True,
        'de': u':lilie:',
        'es': u':flor_de_lis:',
        'fr': u':fleur_de_lys:',
        'pt': u':flor-de-lis:',
        'it': u':giglio_araldico:'
    },
    u'\U0001F4AA': {
        'en' : u':flexed_biceps:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':muscle:'],
        'de': u':angespannter_bizeps:',
        'es': u':bíceps_flexionado:',
        'fr': u':biceps_contracté:',
        'pt': u':bíceps:',
        'it': u':bicipite:'
    },
    u'\U0001F4AA\U0001F3FF': {
        'en' : u':flexed_biceps_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':angespannter_bizeps_dunkle_hautfarbe:',
        'es': u':bíceps_flexionado_tono_de_piel_oscuro:',
        'fr': u':biceps_contracté_peau_foncée:',
        'pt': u':bíceps_pele_escura:',
        'it': u':bicipite_carnagione_scura:'
    },
    u'\U0001F4AA\U0001F3FB': {
        'en' : u':flexed_biceps_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':angespannter_bizeps_helle_hautfarbe:',
        'es': u':bíceps_flexionado_tono_de_piel_claro:',
        'fr': u':biceps_contracté_peau_claire:',
        'pt': u':bíceps_pele_clara:',
        'it': u':bicipite_carnagione_chiara:'
    },
    u'\U0001F4AA\U0001F3FE': {
        'en' : u':flexed_biceps_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':angespannter_bizeps_mitteldunkle_hautfarbe:',
        'es': u':bíceps_flexionado_tono_de_piel_oscuro_medio:',
        'fr': u':biceps_contracté_peau_mate:',
        'pt': u':bíceps_pele_morena_escura:',
        'it': u':bicipite_carnagione_abbastanza_scura:'
    },
    u'\U0001F4AA\U0001F3FC': {
        'en' : u':flexed_biceps_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':angespannter_bizeps_mittelhelle_hautfarbe:',
        'es': u':bíceps_flexionado_tono_de_piel_claro_medio:',
        'fr': u':biceps_contracté_peau_moyennement_claire:',
        'pt': u':bíceps_pele_morena_clara:',
        'it': u':bicipite_carnagione_abbastanza_chiara:'
    },
    u'\U0001F4AA\U0001F3FD': {
        'en' : u':flexed_biceps_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':angespannter_bizeps_mittlere_hautfarbe:',
        'es': u':bíceps_flexionado_tono_de_piel_medio:',
        'fr': u':biceps_contracté_peau_légèrement_mate:',
        'pt': u':bíceps_pele_morena:',
        'it': u':bicipite_carnagione_olivastra:'
    },
    u'\U0001F4BE': {
        'en' : u':floppy_disk:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':diskette:',
        'es': u':disquete:',
        'fr': u':disquette:',
        'pt': u':disquete:',
        'it': u':floppy_disc:'
    },
    u'\U0001F3B4': {
        'en' : u':flower_playing_cards:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':japanische_blumenkarte:',
        'es': u':cartas_de_flores:',
        'fr': u':jeu_des_fleurs:',
        'pt': u':carta_de_flores:',
        'it': u':carta_da_gioco:'
    },
    u'\U0001F633': {
        'en' : u':flushed_face:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':flushed:'],
        'de': u':errötetes_gesicht_mit_großen_augen:',
        'es': u':cara_sonrojada:',
        'fr': u':visage_qui_rougit:',
        'pt': u':rosto_ruborizado:',
        'it': u':faccina_imbarazzata:'
    },
    u'\U0001FAB0': {
        'en' : u':fly:',
        'status' : fully_qualified,
        'E' : 13,
        'de': u':fliege:',
        'es': u':mosca:',
        'fr': u':mouche:',
        'pt': u':mosca:',
        'it': u':mosca:'
    },
    u'\U0001F94F': {
        'en' : u':flying_disc:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':frisbee:',
        'es': u':disco_volador:',
        'fr': u':disque-volant:',
        'pt': u':frisbee:',
        'it': u':frisbee:'
    },
    u'\U0001F6F8': {
        'en' : u':flying_saucer:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':fliegende_untertasse:',
        'es': u':platillo_volante:',
        'fr': u':soucoupe_volante:',
        'pt': u':disco_voador:',
        'it': u':disco_volante:'
    },
    u'\U0001F32B\U0000FE0F': {
        'en' : u':fog:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': u':nebel:',
        'es': u':niebla:',
        'fr': u':brouillard:',
        'pt': u':neblina:',
        'it': u':nebbia:'
    },
    u'\U0001F32B': {
        'en' : u':fog:',
        'status' : unqualified,
        'E' : 0.7,
        'variant': True,
        'de': u':nebel:',
        'es': u':niebla:',
        'fr': u':brouillard:',
        'pt': u':neblina:',
        'it': u':nebbia:'
    },
    u'\U0001F301': {
        'en' : u':foggy:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':neblig:',
        'es': u':bruma:',
        'fr': u':brume:',
        'pt': u':enevoado:',
        'it': u':nebbioso:'
    },
    u'\U0001F64F': {
        'en' : u':folded_hands:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':pray:'],
        'de': u':betende_hände:',
        'es': u':manos_en_oración:',
        'fr': u':mains_en_prière:',
        'pt': u':mãos_juntas:',
        'it': u':mani_giunte:'
    },
    u'\U0001F64F\U0001F3FF': {
        'en' : u':folded_hands_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':betende_hände_dunkle_hautfarbe:',
        'es': u':manos_en_oración_tono_de_piel_oscuro:',
        'fr': u':mains_en_prière_peau_foncée:',
        'pt': u':mãos_juntas_pele_escura:',
        'it': u':mani_giunte_carnagione_scura:'
    },
    u'\U0001F64F\U0001F3FB': {
        'en' : u':folded_hands_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':betende_hände_helle_hautfarbe:',
        'es': u':manos_en_oración_tono_de_piel_claro:',
        'fr': u':mains_en_prière_peau_claire:',
        'pt': u':mãos_juntas_pele_clara:',
        'it': u':mani_giunte_carnagione_chiara:'
    },
    u'\U0001F64F\U0001F3FE': {
        'en' : u':folded_hands_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':betende_hände_mitteldunkle_hautfarbe:',
        'es': u':manos_en_oración_tono_de_piel_oscuro_medio:',
        'fr': u':mains_en_prière_peau_mate:',
        'pt': u':mãos_juntas_pele_morena_escura:',
        'it': u':mani_giunte_carnagione_abbastanza_scura:'
    },
    u'\U0001F64F\U0001F3FC': {
        'en' : u':folded_hands_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':betende_hände_mittelhelle_hautfarbe:',
        'es': u':manos_en_oración_tono_de_piel_claro_medio:',
        'fr': u':mains_en_prière_peau_moyennement_claire:',
        'pt': u':mãos_juntas_pele_morena_clara:',
        'it': u':mani_giunte_carnagione_abbastanza_chiara:'
    },
    u'\U0001F64F\U0001F3FD': {
        'en' : u':folded_hands_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':betende_hände_mittlere_hautfarbe:',
        'es': u':manos_en_oración_tono_de_piel_medio:',
        'fr': u':mains_en_prière_peau_légèrement_mate:',
        'pt': u':mãos_juntas_pele_morena:',
        'it': u':mani_giunte_carnagione_olivastra:'
    },
    u'\U0001FAD5': {
        'en' : u':fondue:',
        'status' : fully_qualified,
        'E' : 13,
        'de': u':fondue:',
        'es': u':fondue:',
        'fr': u':fondue:',
        'pt': u':fondue:',
        'it': u':fonduta:'
    },
    u'\U0001F9B6': {
        'en' : u':foot:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':fuß:',
        'es': u':pie:',
        'fr': u':pied:',
        'pt': u':pé:',
        'it': u':piede:'
    },
    u'\U0001F9B6\U0001F3FF': {
        'en' : u':foot_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':fuß_dunkle_hautfarbe:',
        'es': u':pie_tono_de_piel_oscuro:',
        'fr': u':pied_peau_foncée:',
        'pt': u':pé_pele_escura:',
        'it': u':piede_carnagione_scura:'
    },
    u'\U0001F9B6\U0001F3FB': {
        'en' : u':foot_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':fuß_helle_hautfarbe:',
        'es': u':pie_tono_de_piel_claro:',
        'fr': u':pied_peau_claire:',
        'pt': u':pé_pele_clara:',
        'it': u':piede_carnagione_chiara:'
    },
    u'\U0001F9B6\U0001F3FE': {
        'en' : u':foot_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':fuß_mitteldunkle_hautfarbe:',
        'es': u':pie_tono_de_piel_oscuro_medio:',
        'fr': u':pied_peau_mate:',
        'pt': u':pé_pele_morena_escura:',
        'it': u':piede_carnagione_abbastanza_scura:'
    },
    u'\U0001F9B6\U0001F3FC': {
        'en' : u':foot_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':fuß_mittelhelle_hautfarbe:',
        'es': u':pie_tono_de_piel_claro_medio:',
        'fr': u':pied_peau_moyennement_claire:',
        'pt': u':pé_pele_morena_clara:',
        'it': u':piede_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9B6\U0001F3FD': {
        'en' : u':foot_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':fuß_mittlere_hautfarbe:',
        'es': u':pie_tono_de_piel_medio:',
        'fr': u':pied_peau_légèrement_mate:',
        'pt': u':pé_pele_morena:',
        'it': u':piede_carnagione_olivastra:'
    },
    u'\U0001F463': {
        'en' : u':footprints:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':fußabdrücke:',
        'es': u':huellas_de_pies:',
        'fr': u':traces_de_pas:',
        'pt': u':pegadas:',
        'it': u':impronta_di_piedi:'
    },
    u'\U0001F374': {
        'en' : u':fork_and_knife:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':gabel_und_messer:',
        'es': u':tenedor_y_cuchillo:',
        'fr': u':fourchette_et_couteau:',
        'pt': u':garfo_e_faca:',
        'it': u':forchetta_e_coltello:'
    },
    u'\U0001F37D\U0000FE0F': {
        'en' : u':fork_and_knife_with_plate:',
        'status' : fully_qualified,
        'E' : 0.7,
        'alias' : [u':plate_with_cutlery:'],
        'variant': True,
        'de': u':teller_mit_messer_und_gabel:',
        'es': u':cuchillo_y_tenedor_con_un_plato:',
        'fr': u':fourchette_et_couteau_avec_assiette:',
        'pt': u':prato_com_talheres:',
        'it': u':piatto_e_posate:'
    },
    u'\U0001F37D': {
        'en' : u':fork_and_knife_with_plate:',
        'status' : unqualified,
        'E' : 0.7,
        'alias' : [u':plate_with_cutlery:'],
        'variant': True,
        'de': u':teller_mit_messer_und_gabel:',
        'es': u':cuchillo_y_tenedor_con_un_plato:',
        'fr': u':fourchette_et_couteau_avec_assiette:',
        'pt': u':prato_com_talheres:',
        'it': u':piatto_e_posate:'
    },
    u'\U0001F960': {
        'en' : u':fortune_cookie:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':glückskeks:',
        'es': u':galleta_de_la_fortuna:',
        'fr': u':biscuit_chinois:',
        'pt': u':biscoito_da_sorte:',
        'it': u':biscotto_della_fortuna:'
    },
    u'\U000026F2': {
        'en' : u':fountain:',
        'status' : fully_qualified,
        'E' : 0.6,
        'variant': True,
        'de': u':springbrunnen:',
        'es': u':fuente:',
        'fr': u':fontaine:',
        'pt': u':fonte:',
        'it': u':fontana:'
    },
    u'\U0001F58B\U0000FE0F': {
        'en' : u':fountain_pen:',
        'status' : fully_qualified,
        'E' : 0.7,
        'alias' : [u':lower_left_fountain_pen:'],
        'variant': True,
        'de': u':füllhalter:',
        'es': u':estilográfica:',
        'fr': u':stylo_plume:',
        'pt': u':caneta_tinteiro:',
        'it': u':penna_stilografica:'
    },
    u'\U0001F58B': {
        'en' : u':fountain_pen:',
        'status' : unqualified,
        'E' : 0.7,
        'alias' : [u':lower_left_fountain_pen:'],
        'variant': True,
        'de': u':füllhalter:',
        'es': u':estilográfica:',
        'fr': u':stylo_plume:',
        'pt': u':caneta_tinteiro:',
        'it': u':penna_stilografica:'
    },
    u'\U0001F55F': {
        'en' : u':four-thirty:',
        'status' : fully_qualified,
        'E' : 0.7,
        'alias' : [u':clock430:'],
        'variant': True,
        'de': u':4.30_uhr:',
        'es': u':cuatro_y_media:',
        'fr': u':quatre_heures_et_demie:',
        'pt': u':quatro_e_meia:',
        'it': u':ore_quattro_e_mezza:'
    },
    u'\U0001F340': {
        'en' : u':four_leaf_clover:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':glücksklee:',
        'es': u':trébol_de_cuatro_hojas:',
        'fr': u':trèfle_à_quatre_feuilles:',
        'pt': u':trevo_de_quatro_folhas:',
        'it': u':quadrifoglio:'
    },
    u'\U0001F553': {
        'en' : u':four_o’clock:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':clock4:'],
        'variant': True,
        'de': u':4.00_uhr:',
        'es': u':4_en_punto:',
        'fr': u':quatre_heures:',
        'pt': u':4_horas:',
        'it': u':ore_quattro:'
    },
    u'\U0001F98A': {
        'en' : u':fox:',
        'status' : fully_qualified,
        'E' : 3,
        'alias' : [u':fox_face:'],
        'de': u':fuchs:',
        'es': u':zorro:',
        'fr': u':renard:',
        'pt': u':rosto_de_raposa:',
        'it': u':volpe:'
    },
    u'\U0001F5BC\U0000FE0F': {
        'en' : u':framed_picture:',
        'status' : fully_qualified,
        'E' : 0.7,
        'alias' : [u':frame_with_picture:'],
        'variant': True,
        'de': u':gerahmtes_bild:',
        'es': u':cuadro_enmarcado:',
        'fr': u':cadre_avec_image:',
        'pt': u':quadro_emoldurado:',
        'it': u':quadro_incorniciato:'
    },
    u'\U0001F5BC': {
        'en' : u':framed_picture:',
        'status' : unqualified,
        'E' : 0.7,
        'alias' : [u':frame_with_picture:'],
        'variant': True,
        'de': u':gerahmtes_bild:',
        'es': u':cuadro_enmarcado:',
        'fr': u':cadre_avec_image:',
        'pt': u':quadro_emoldurado:',
        'it': u':quadro_incorniciato:'
    },
    u'\U0001F35F': {
        'en' : u':french_fries:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':fries:'],
        'de': u':pommes_frites:',
        'es': u':patatas_fritas:',
        'fr': u':frites:',
        'pt': u':batata_frita:',
        'it': u':patatine:'
    },
    u'\U0001F364': {
        'en' : u':fried_shrimp:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':frittierte_garnele:',
        'es': u':gamba_frita:',
        'fr': u':beignet_de_crevette:',
        'pt': u':camarão_frito:',
        'it': u':gambero_fritto:'
    },
    u'\U0001F438': {
        'en' : u':frog:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':frosch:',
        'es': u':rana:',
        'fr': u':grenouille:',
        'pt': u':sapo:',
        'it': u':rana:'
    },
    u'\U0001F425': {
        'en' : u':front-facing_baby_chick:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':hatched_chick:'],
        'de': u':küken_von_vorne:',
        'es': u':pollito_de_frente:',
        'fr': u':poussin_de_face:',
        'pt': u':pintinho_de_frente:',
        'it': u':pulcino_visto_di_fronte:'
    },
    u'\U00002639\U0000FE0F': {
        'en' : u':frowning_face:',
        'status' : fully_qualified,
        'E' : 0.7,
        'alias' : [u':white_frowning_face:'],
        'variant': True,
        'de': u':düsteres_gesicht:',
        'es': u':cara_con_el_ceño_fruncido:',
        'fr': u':visage_mécontent:',
        'pt': u':rosto_descontente:',
        'it': u':faccina_imbronciata:'
    },
    u'\U00002639': {
        'en' : u':frowning_face:',
        'status' : unqualified,
        'E' : 0.7,
        'alias' : [u':white_frowning_face:'],
        'variant': True,
        'de': u':düsteres_gesicht:',
        'es': u':cara_con_el_ceño_fruncido:',
        'fr': u':visage_mécontent:',
        'pt': u':rosto_descontente:',
        'it': u':faccina_imbronciata:'
    },
    u'\U0001F626': {
        'en' : u':frowning_face_with_open_mouth:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [u':frowning:'],
        'de': u':entsetztes_gesicht:',
        'es': u':cara_con_el_ceño_fruncido_y_la_boca_abierta:',
        'fr': u':visage_mécontent_avec_bouche_ouverte:',
        'pt': u':rosto_franzido_com_boca_aberta:',
        'it': u':faccina_imbronciata_con_bocca_aperta:'
    },
    u'\U000026FD': {
        'en' : u':fuel_pump:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':fuelpump:'],
        'variant': True,
        'de': u':tanksäule:',
        'es': u':surtidor_de_gasolina:',
        'fr': u':pompe_à_essence:',
        'pt': u':posto_de_gasolina:',
        'it': u':stazione_di_servizio:'
    },
    u'\U0001F315': {
        'en' : u':full_moon:',
        'status' : fully_qualified,
        'E' : 0.6,
        'variant': True,
        'de': u':vollmond:',
        'es': u':luna_llena:',
        'fr': u':pleine_lune:',
        'pt': u':lua_cheia:',
        'it': u':luna_piena:'
    },
    u'\U0001F31D': {
        'en' : u':full_moon_face:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [u':full_moon_with_face:'],
        'de': u':vollmond_mit_gesicht:',
        'es': u':luna_llena_con_cara:',
        'fr': u':pleine_lune_avec_visage:',
        'pt': u':rosto_da_lua_cheia:',
        'it': u':faccina_luna_piena:'
    },
    u'\U000026B1\U0000FE0F': {
        'en' : u':funeral_urn:',
        'status' : fully_qualified,
        'E' : 1,
        'variant': True,
        'de': u':urne:',
        'es': u':urna_funeraria:',
        'fr': u':urne_funéraire:',
        'pt': u':urna_funerária:',
        'it': u':urna_funeraria:'
    },
    u'\U000026B1': {
        'en' : u':funeral_urn:',
        'status' : unqualified,
        'E' : 1,
        'variant': True,
        'de': u':urne:',
        'es': u':urna_funeraria:',
        'fr': u':urne_funéraire:',
        'pt': u':urna_funerária:',
        'it': u':urna_funeraria:'
    },
    u'\U0001F3B2': {
        'en' : u':game_die:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':spielwürfel:',
        'es': u':dado:',
        'fr': u':dés:',
        'pt': u':jogo_de_dado:',
        'it': u':dado:'
    },
    u'\U0001F9C4': {
        'en' : u':garlic:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':knoblauch:',
        'es': u':ajo:',
        'fr': u':ail:',
        'pt': u':alho:',
        'it': u':aglio:'
    },
    u'\U00002699\U0000FE0F': {
        'en' : u':gear:',
        'status' : fully_qualified,
        'E' : 1,
        'variant': True,
        'de': u':zahnrad:',
        'es': u':engranaje:',
        'fr': u':roue_dentée:',
        'pt': u':engrenagem:',
        'it': u':ingranaggio:'
    },
    u'\U00002699': {
        'en' : u':gear:',
        'status' : unqualified,
        'E' : 1,
        'variant': True,
        'de': u':zahnrad:',
        'es': u':engranaje:',
        'fr': u':roue_dentée:',
        'pt': u':engrenagem:',
        'it': u':ingranaggio:'
    },
    u'\U0001F48E': {
        'en' : u':gem_stone:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':gem:'],
        'de': u':edelstein:',
        'es': u':piedra_preciosa:',
        'fr': u':pierre_précieuse:',
        'pt': u':pedra_preciosa:',
        'it': u':gemma:'
    },
    u'\U0001F9DE': {
        'en' : u':genie:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':flaschengeist:',
        'es': u':genio:',
        'fr': u':génie:',
        'pt': u':gênio:',
        'it': u':genio:'
    },
    u'\U0001F47B': {
        'en' : u':ghost:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':gespenst:',
        'es': u':fantasma:',
        'fr': u':fantôme:',
        'pt': u':fantasma:',
        'it': u':fantasma:'
    },
    u'\U0001F992': {
        'en' : u':giraffe:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':giraffe:',
        'es': u':jirafa:',
        'fr': u':girafe:',
        'pt': u':girafa:',
        'it': u':giraffa:'
    },
    u'\U0001F467': {
        'en' : u':girl:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':mädchen:',
        'es': u':niña:',
        'fr': u':fille:',
        'pt': u':menina:',
        'it': u':bambina:'
    },
    u'\U0001F467\U0001F3FF': {
        'en' : u':girl_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':mädchen_dunkle_hautfarbe:',
        'es': u':niña_tono_de_piel_oscuro:',
        'fr': u':fille_peau_foncée:',
        'pt': u':menina_pele_escura:',
        'it': u':bambina_carnagione_scura:'
    },
    u'\U0001F467\U0001F3FB': {
        'en' : u':girl_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':mädchen_helle_hautfarbe:',
        'es': u':niña_tono_de_piel_claro:',
        'fr': u':fille_peau_claire:',
        'pt': u':menina_pele_clara:',
        'it': u':bambina_carnagione_chiara:'
    },
    u'\U0001F467\U0001F3FE': {
        'en' : u':girl_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':mädchen_mitteldunkle_hautfarbe:',
        'es': u':niña_tono_de_piel_oscuro_medio:',
        'fr': u':fille_peau_mate:',
        'pt': u':menina_pele_morena_escura:',
        'it': u':bambina_carnagione_abbastanza_scura:'
    },
    u'\U0001F467\U0001F3FC': {
        'en' : u':girl_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':mädchen_mittelhelle_hautfarbe:',
        'es': u':niña_tono_de_piel_claro_medio:',
        'fr': u':fille_peau_moyennement_claire:',
        'pt': u':menina_pele_morena_clara:',
        'it': u':bambina_carnagione_abbastanza_chiara:'
    },
    u'\U0001F467\U0001F3FD': {
        'en' : u':girl_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':mädchen_mittlere_hautfarbe:',
        'es': u':niña_tono_de_piel_medio:',
        'fr': u':fille_peau_légèrement_mate:',
        'pt': u':menina_pele_morena:',
        'it': u':bambina_carnagione_olivastra:'
    },
    u'\U0001F95B': {
        'en' : u':glass_of_milk:',
        'status' : fully_qualified,
        'E' : 3,
        'alias' : [u':milk_glass:'],
        'de': u':glas_milch:',
        'es': u':vaso_de_leche:',
        'fr': u':verre_de_lait:',
        'pt': u':copo_de_leite:',
        'it': u':bicchiere_di_latte:'
    },
    u'\U0001F453': {
        'en' : u':glasses:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':eyeglasses:'],
        'variant': True,
        'de': u':brille:',
        'es': u':gafas:',
        'fr': u':lunettes_de_vue:',
        'pt': u':óculos:',
        'it': u':occhiali_da_vista:'
    },
    u'\U0001F30E': {
        'en' : u':globe_showing_Americas:',
        'status' : fully_qualified,
        'E' : 0.7,
        'alias' : [u':earth_americas:'],
        'variant': True,
        'de': u':globus_mit_amerika:',
        'es': u':globo_terráqueo_mostrando_américa:',
        'fr': u':globe_tourné_sur_les_amériques:',
        'pt': u':globo_mostrando_as_américas:',
        'it': u':america:'
    },
    u'\U0001F30F': {
        'en' : u':globe_showing_Asia-Australia:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':earth_asia:'],
        'variant': True,
        'de': u':globus_mit_asien_und_australien:',
        'es': u':globo_terráqueo_mostrando_asia_y_australia:',
        'fr': u':globe_tourné_sur_l’asie_et_l’australie:',
        'pt': u':globo_mostrando_ásia_e_oceania:',
        'it': u':asia_e_australia:'
    },
    u'\U0001F30D': {
        'en' : u':globe_showing_Europe-Africa:',
        'status' : fully_qualified,
        'E' : 0.7,
        'alias' : [u':earth_africa:'],
        'variant': True,
        'de': u':globus_mit_europa_und_afrika:',
        'es': u':globo_terráqueo_mostrando_europa_y_áfrica:',
        'fr': u':globe_tourné_sur_l’afrique_et_l’europe:',
        'pt': u':globo_mostrando_europa_e_áfrica:',
        'it': u':europa_e_africa:'
    },
    u'\U0001F310': {
        'en' : u':globe_with_meridians:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':globus_mit_meridianen:',
        'es': u':globo_terráqueo_con_meridianos:',
        'fr': u':globe_avec_méridiens:',
        'pt': u':globo_com_meridianos:',
        'it': u':globo_con_meridiani:'
    },
    u'\U0001F9E4': {
        'en' : u':gloves:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':handschuhe:',
        'es': u':guantes:',
        'fr': u':gants:',
        'pt': u':luvas:',
        'it': u':guanti:'
    },
    u'\U0001F31F': {
        'en' : u':glowing_star:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':star2:'],
        'de': u':funkelnder_stern:',
        'es': u':estrella_brillante:',
        'fr': u':étoile_brillante:',
        'pt': u':estrela_brilhante:',
        'it': u':stella_che_brilla:'
    },
    u'\U0001F945': {
        'en' : u':goal_net:',
        'status' : fully_qualified,
        'E' : 3,
        'de': u':tor:',
        'es': u':portería:',
        'fr': u':cage:',
        'pt': u':gol:',
        'it': u':porta_da_calcio:'
    },
    u'\U0001F410': {
        'en' : u':goat:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':ziege:',
        'es': u':cabra:',
        'fr': u':chèvre:',
        'pt': u':cabra:',
        'it': u':capra:'
    },
    u'\U0001F47A': {
        'en' : u':goblin:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':japanese_goblin:'],
        'de': u':kobold:',
        'es': u':demonio_japonés_tengu:',
        'fr': u':monstre_japonais:',
        'pt': u':duende_japonês:',
        'it': u':goblin:'
    },
    u'\U0001F97D': {
        'en' : u':goggles:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':schutzbrille:',
        'es': u':gafas_de_protección:',
        'fr': u':lunettes:',
        'pt': u':óculos_de_proteção:',
        'it': u':maschera:'
    },
    u'\U0001F98D': {
        'en' : u':gorilla:',
        'status' : fully_qualified,
        'E' : 3,
        'de': u':gorilla:',
        'es': u':gorila:',
        'fr': u':gorille:',
        'pt': u':gorila:',
        'it': u':gorilla:'
    },
    u'\U0001F393': {
        'en' : u':graduation_cap:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':mortar_board:'],
        'variant': True,
        'de': u':doktorhut:',
        'es': u':birrete:',
        'fr': u':toque_universitaire:',
        'pt': u':chapéu_de_formatura:',
        'it': u':cappello_da_laureato:'
    },
    u'\U0001F347': {
        'en' : u':grapes:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':trauben:',
        'es': u':uvas:',
        'fr': u':raisin:',
        'pt': u':uvas:',
        'it': u':uva:'
    },
    u'\U0001F34F': {
        'en' : u':green_apple:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':grüner_apfel:',
        'es': u':manzana_verde:',
        'fr': u':pomme_verte:',
        'pt': u':maçã_verde:',
        'it': u':mela_verde:'
    },
    u'\U0001F4D7': {
        'en' : u':green_book:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':grünes_buch:',
        'es': u':libro_verde:',
        'fr': u':livre_vert:',
        'pt': u':livro_verde:',
        'it': u':libro_verde:'
    },
    u'\U0001F7E2': {
        'en' : u':green_circle:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':grüner_kreis:',
        'es': u':círculo_verde:',
        'fr': u':disque_vert:',
        'pt': u':círculo_verde:',
        'it': u':cerchio_verde:'
    },
    u'\U0001F49A': {
        'en' : u':green_heart:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':grünes_herz:',
        'es': u':corazón_verde:',
        'fr': u':cœur_vert:',
        'pt': u':coração_verde:',
        'it': u':cuore_verde:'
    },
    u'\U0001F957': {
        'en' : u':green_salad:',
        'status' : fully_qualified,
        'E' : 3,
        'de': u':salat:',
        'es': u':ensalada:',
        'fr': u':salade_verte:',
        'pt': u':salada_verde:',
        'it': u':insalata_verde:'
    },
    u'\U0001F7E9': {
        'en' : u':green_square:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':grünes_viereck:',
        'es': u':cuadrado_verde:',
        'fr': u':carré_vert:',
        'pt': u':quadrado_verde:',
        'it': u':quadrato_verde:'
    },
    u'\U0001F62C': {
        'en' : u':grimacing_face:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [u':grimacing:'],
        'de': u':grimassen_schneidendes_gesicht:',
        'es': u':cara_haciendo_una_mueca:',
        'fr': u':visage_grimaçant:',
        'pt': u':rosto_expressando_desagrado:',
        'it': u':faccina_con_smorfia:'
    },
    u'\U0001F63A': {
        'en' : u':grinning_cat:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':smiley_cat:'],
        'de': u':grinsende_katze:',
        'es': u':gato_sonriendo:',
        'fr': u':chat_qui_sourit:',
        'pt': u':rosto_de_gato_sorrindo:',
        'it': u':gatto_che_sorride:'
    },
    u'\U0001F638': {
        'en' : u':grinning_cat_with_smiling_eyes:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':smile_cat:'],
        'de': u':grinsende_katze_mit_lachenden_augen:',
        'es': u':gato_sonriendo_con_ojos_sonrientes:',
        'fr': u':chat_qui_sourit_avec_des_yeux_rieurs:',
        'pt': u':rosto_de_gato_sorrindo_com_olhos_sorridentes:',
        'it': u':gatto_che_sogghigna:'
    },
    u'\U0001F600': {
        'en' : u':grinning_face:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [u':grinning:'],
        'de': u':grinsendes_gesicht:',
        'es': u':cara_sonriendo:',
        'fr': u':visage_rieur:',
        'pt': u':rosto_risonho:',
        'it': u':faccina_con_un_gran_sorriso:'
    },
    u'\U0001F603': {
        'en' : u':grinning_face_with_big_eyes:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':smiley:'],
        'de': u':grinsendes_gesicht_mit_großen_augen:',
        'es': u':cara_sonriendo_con_ojos_grandes:',
        'fr': u':visage_souriant_avec_de_grands_yeux:',
        'pt': u':rosto_risonho_com_olhos_bem_abertos:',
        'it': u':faccina_con_un_gran_sorriso_e_occhi_spalancati:'
    },
    u'\U0001F604': {
        'en' : u':grinning_face_with_smiling_eyes:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':smile:'],
        'de': u':grinsendes_gesicht_mit_lachenden_augen:',
        'es': u':cara_sonriendo_con_ojos_sonrientes:',
        'fr': u':visage_très_souriant_aux_yeux_rieurs:',
        'pt': u':rosto_risonho_com_olhos_sorridentes:',
        'it': u':faccina_con_sorriso_e_occhi_sorridenti:'
    },
    u'\U0001F605': {
        'en' : u':grinning_face_with_sweat:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':sweat_smile:'],
        'de': u':grinsendes_gesicht_mit_schweißtropfen:',
        'es': u':cara_sonriendo_con_sudor_frío:',
        'fr': u':visage_souriant_avec_une_goutte_de_sueur:',
        'pt': u':rosto_risonho_com_gota_de_suor:',
        'it': u':faccina_con_un_gran_sorriso_e_goccia_di_sudore:'
    },
    u'\U0001F606': {
        'en' : u':grinning_squinting_face:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':satisfied:', u':laughing:'],
        'de': u':grinsendes_gesicht_mit_zusammengekniffenen_augen:',
        'es': u':cara_sonriendo_con_los_ojos_cerrados:',
        'fr': u':visage_souriant_avec_yeux_plissés:',
        'pt': u':rosto_risonho_com_olhos_semicerrados:',
        'it': u':sorriso_a_bocca_aperta_con_occhi_chiusi:'
    },
    u'\U0001F497': {
        'en' : u':growing_heart:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':heartpulse:'],
        'de': u':wachsendes_herz:',
        'es': u':corazón_creciente:',
        'fr': u':cœur_grandissant:',
        'pt': u':coração_crescendo:',
        'it': u':cuore_che_cresce:'
    },
    u'\U0001F482': {
        'en' : u':guard:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':guardsman:'],
        'de': u':wachmann/wachfrau:',
        'es': u':guardia:',
        'fr': u':garde:',
        'pt': u':guarda:',
        'it': u':guardia:'
    },
    u'\U0001F482\U0001F3FF': {
        'en' : u':guard_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':wachmann/wachfrau_dunkle_hautfarbe:',
        'es': u':guardia_tono_de_piel_oscuro:',
        'fr': u':garde_peau_foncée:',
        'pt': u':guarda_pele_escura:',
        'it': u':guardia_carnagione_scura:'
    },
    u'\U0001F482\U0001F3FB': {
        'en' : u':guard_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':wachmann/wachfrau_helle_hautfarbe:',
        'es': u':guardia_tono_de_piel_claro:',
        'fr': u':garde_peau_claire:',
        'pt': u':guarda_pele_clara:',
        'it': u':guardia_carnagione_chiara:'
    },
    u'\U0001F482\U0001F3FE': {
        'en' : u':guard_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':wachmann/wachfrau_mitteldunkle_hautfarbe:',
        'es': u':guardia_tono_de_piel_oscuro_medio:',
        'fr': u':garde_peau_mate:',
        'pt': u':guarda_pele_morena_escura:',
        'it': u':guardia_carnagione_abbastanza_scura:'
    },
    u'\U0001F482\U0001F3FC': {
        'en' : u':guard_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':wachmann/wachfrau_mittelhelle_hautfarbe:',
        'es': u':guardia_tono_de_piel_claro_medio:',
        'fr': u':garde_peau_moyennement_claire:',
        'pt': u':guarda_pele_morena_clara:',
        'it': u':guardia_carnagione_abbastanza_chiara:'
    },
    u'\U0001F482\U0001F3FD': {
        'en' : u':guard_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':wachmann/wachfrau_mittlere_hautfarbe:',
        'es': u':guardia_tono_de_piel_medio:',
        'fr': u':garde_peau_légèrement_mate:',
        'pt': u':guarda_pele_morena:',
        'it': u':guardia_carnagione_olivastra:'
    },
    u'\U0001F9AE': {
        'en' : u':guide_dog:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':blindenhund:',
        'es': u':perro_guía:',
        'fr': u':chien_guide:',
        'pt': u':cão-guia:',
        'it': u':cane_guida:'
    },
    u'\U0001F3B8': {
        'en' : u':guitar:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':gitarre:',
        'es': u':guitarra:',
        'fr': u':guitare:',
        'pt': u':guitarra:',
        'it': u':chitarra:'
    },
    u'\U0001F354': {
        'en' : u':hamburger:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':hamburger:',
        'es': u':hamburguesa:',
        'fr': u':hamburger:',
        'pt': u':hambúrguer:',
        'it': u':hamburger:'
    },
    u'\U0001F528': {
        'en' : u':hammer:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':hammer:',
        'es': u':martillo:',
        'fr': u':marteau:',
        'pt': u':martelo:',
        'it': u':martello:'
    },
    u'\U00002692\U0000FE0F': {
        'en' : u':hammer_and_pick:',
        'status' : fully_qualified,
        'E' : 1,
        'variant': True,
        'de': u':hammer_und_pickel:',
        'es': u':martillo_y_pico:',
        'fr': u':marteau_et_pioche:',
        'pt': u':martelo_e_picareta:',
        'it': u':piccone_e_martello:'
    },
    u'\U00002692': {
        'en' : u':hammer_and_pick:',
        'status' : unqualified,
        'E' : 1,
        'variant': True,
        'de': u':hammer_und_pickel:',
        'es': u':martillo_y_pico:',
        'fr': u':marteau_et_pioche:',
        'pt': u':martelo_e_picareta:',
        'it': u':piccone_e_martello:'
    },
    u'\U0001F6E0\U0000FE0F': {
        'en' : u':hammer_and_wrench:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': u':hammer_und_schraubenschlüssel:',
        'es': u':martillo_y_llave_inglesa:',
        'fr': u':marteau_et_clé_à_molette:',
        'pt': u':martelo_e_chave-inglesa:',
        'it': u':martello_e_chiave_inglese:'
    },
    u'\U0001F6E0': {
        'en' : u':hammer_and_wrench:',
        'status' : unqualified,
        'E' : 0.7,
        'variant': True,
        'de': u':hammer_und_schraubenschlüssel:',
        'es': u':martillo_y_llave_inglesa:',
        'fr': u':marteau_et_clé_à_molette:',
        'pt': u':martelo_e_chave-inglesa:',
        'it': u':martello_e_chiave_inglese:'
    },
    u'\U0001FAAC': {
        'en' : u':hamsa:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001F439': {
        'en' : u':hamster:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':hamster:',
        'es': u':hámster:',
        'fr': u':hamster:',
        'pt': u':rosto_de_hamster:',
        'it': u':criceto:'
    },
    u'\U0001F590\U0000FE0F': {
        'en' : u':hand_with_fingers_splayed:',
        'status' : fully_qualified,
        'E' : 0.7,
        'alias' : [u':raised_hand_with_fingers_splayed:'],
        'variant': True,
        'de': u':hand_mit_gespreizten_fingern:',
        'es': u':mano_abierta:',
        'fr': u':main_levée_doigts_écartés:',
        'pt': u':mão_aberta_com_os_dedos_separados:',
        'it': u':mano_aperta:'
    },
    u'\U0001F590': {
        'en' : u':hand_with_fingers_splayed:',
        'status' : unqualified,
        'E' : 0.7,
        'alias' : [u':raised_hand_with_fingers_splayed:'],
        'variant': True,
        'de': u':hand_mit_gespreizten_fingern:',
        'es': u':mano_abierta:',
        'fr': u':main_levée_doigts_écartés:',
        'pt': u':mão_aberta_com_os_dedos_separados:',
        'it': u':mano_aperta:'
    },
    u'\U0001F590\U0001F3FF': {
        'en' : u':hand_with_fingers_splayed_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':hand_mit_gespreizten_fingern_dunkle_hautfarbe:',
        'es': u':mano_abierta_tono_de_piel_oscuro:',
        'fr': u':main_levée_doigts_écartés_peau_foncée:',
        'pt': u':mão_aberta_com_os_dedos_separados_pele_escura:',
        'it': u':mano_aperta_carnagione_scura:'
    },
    u'\U0001F590\U0001F3FB': {
        'en' : u':hand_with_fingers_splayed_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':hand_mit_gespreizten_fingern_helle_hautfarbe:',
        'es': u':mano_abierta_tono_de_piel_claro:',
        'fr': u':main_levée_doigts_écartés_peau_claire:',
        'pt': u':mão_aberta_com_os_dedos_separados_pele_clara:',
        'it': u':mano_aperta_carnagione_chiara:'
    },
    u'\U0001F590\U0001F3FE': {
        'en' : u':hand_with_fingers_splayed_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':hand_mit_gespreizten_fingern_mitteldunkle_hautfarbe:',
        'es': u':mano_abierta_tono_de_piel_oscuro_medio:',
        'fr': u':main_levée_doigts_écartés_peau_mate:',
        'pt': u':mão_aberta_com_os_dedos_separados_pele_morena_escura:',
        'it': u':mano_aperta_carnagione_abbastanza_scura:'
    },
    u'\U0001F590\U0001F3FC': {
        'en' : u':hand_with_fingers_splayed_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':hand_mit_gespreizten_fingern_mittelhelle_hautfarbe:',
        'es': u':mano_abierta_tono_de_piel_claro_medio:',
        'fr': u':main_levée_doigts_écartés_peau_moyennement_claire:',
        'pt': u':mão_aberta_com_os_dedos_separados_pele_morena_clara:',
        'it': u':mano_aperta_carnagione_abbastanza_chiara:'
    },
    u'\U0001F590\U0001F3FD': {
        'en' : u':hand_with_fingers_splayed_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':hand_mit_gespreizten_fingern_mittlere_hautfarbe:',
        'es': u':mano_abierta_tono_de_piel_medio:',
        'fr': u':main_levée_doigts_écartés_peau_légèrement_mate:',
        'pt': u':mão_aberta_com_os_dedos_separados_pele_morena:',
        'it': u':mano_aperta_carnagione_olivastra:'
    },
    u'\U0001FAF0': {
        'en' : u':hand_with_index_finger_and_thumb_crossed:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001FAF0\U0001F3FF': {
        'en' : u':hand_with_index_finger_and_thumb_crossed_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001FAF0\U0001F3FB': {
        'en' : u':hand_with_index_finger_and_thumb_crossed_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001FAF0\U0001F3FE': {
        'en' : u':hand_with_index_finger_and_thumb_crossed_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001FAF0\U0001F3FC': {
        'en' : u':hand_with_index_finger_and_thumb_crossed_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001FAF0\U0001F3FD': {
        'en' : u':hand_with_index_finger_and_thumb_crossed_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001F45C': {
        'en' : u':handbag:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':handtasche:',
        'es': u':bolso:',
        'fr': u':sac_à_main:',
        'pt': u':bolsa_de_mão:',
        'it': u':borsa:'
    },
    u'\U0001F91D': {
        'en' : u':handshake:',
        'status' : fully_qualified,
        'E' : 3,
        'de': u':handschlag:',
        'es': u':apretón_de_manos:',
        'fr': u':poignée_de_main:',
        'pt': u':aperto_de_mãos:',
        'it': u':stretta_di_mano:'
    },
    u'\U0001F91D\U0001F3FF': {
        'en' : u':handshake_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 3
    },
    u'\U0001FAF1\U0001F3FF\U0000200D\U0001FAF2\U0001F3FB': {
        'en' : u':handshake_dark_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001FAF1\U0001F3FF\U0000200D\U0001FAF2\U0001F3FE': {
        'en' : u':handshake_dark_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001FAF1\U0001F3FF\U0000200D\U0001FAF2\U0001F3FC': {
        'en' : u':handshake_dark_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001FAF1\U0001F3FF\U0000200D\U0001FAF2\U0001F3FD': {
        'en' : u':handshake_dark_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001F91D\U0001F3FB': {
        'en' : u':handshake_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 3
    },
    u'\U0001FAF1\U0001F3FB\U0000200D\U0001FAF2\U0001F3FF': {
        'en' : u':handshake_light_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001FAF1\U0001F3FB\U0000200D\U0001FAF2\U0001F3FE': {
        'en' : u':handshake_light_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001FAF1\U0001F3FB\U0000200D\U0001FAF2\U0001F3FC': {
        'en' : u':handshake_light_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001FAF1\U0001F3FB\U0000200D\U0001FAF2\U0001F3FD': {
        'en' : u':handshake_light_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001F91D\U0001F3FE': {
        'en' : u':handshake_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 3
    },
    u'\U0001FAF1\U0001F3FE\U0000200D\U0001FAF2\U0001F3FF': {
        'en' : u':handshake_medium-dark_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001FAF1\U0001F3FE\U0000200D\U0001FAF2\U0001F3FB': {
        'en' : u':handshake_medium-dark_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001FAF1\U0001F3FE\U0000200D\U0001FAF2\U0001F3FC': {
        'en' : u':handshake_medium-dark_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001FAF1\U0001F3FE\U0000200D\U0001FAF2\U0001F3FD': {
        'en' : u':handshake_medium-dark_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001F91D\U0001F3FC': {
        'en' : u':handshake_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 3
    },
    u'\U0001FAF1\U0001F3FC\U0000200D\U0001FAF2\U0001F3FF': {
        'en' : u':handshake_medium-light_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001FAF1\U0001F3FC\U0000200D\U0001FAF2\U0001F3FB': {
        'en' : u':handshake_medium-light_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001FAF1\U0001F3FC\U0000200D\U0001FAF2\U0001F3FE': {
        'en' : u':handshake_medium-light_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001FAF1\U0001F3FC\U0000200D\U0001FAF2\U0001F3FD': {
        'en' : u':handshake_medium-light_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001F91D\U0001F3FD': {
        'en' : u':handshake_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 3
    },
    u'\U0001FAF1\U0001F3FD\U0000200D\U0001FAF2\U0001F3FF': {
        'en' : u':handshake_medium_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001FAF1\U0001F3FD\U0000200D\U0001FAF2\U0001F3FB': {
        'en' : u':handshake_medium_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001FAF1\U0001F3FD\U0000200D\U0001FAF2\U0001F3FE': {
        'en' : u':handshake_medium_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001FAF1\U0001F3FD\U0000200D\U0001FAF2\U0001F3FC': {
        'en' : u':handshake_medium_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001F423': {
        'en' : u':hatching_chick:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':schlüpfendes_küken:',
        'es': u':pollito_rompiendo_el_cascarón:',
        'fr': u':poussin_qui_éclôt:',
        'pt': u':pintinho_chocando:',
        'it': u':pulcino_che_nasce:'
    },
    u'\U0001F3A7': {
        'en' : u':headphone:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':headphones:'],
        'variant': True,
        'de': u':kopfhörer:',
        'es': u':auricular:',
        'fr': u':casque:',
        'pt': u':fones_de_ouvido:',
        'it': u':cuffie:'
    },
    u'\U0001FAA6': {
        'en' : u':headstone:',
        'status' : fully_qualified,
        'E' : 13,
        'de': u':grabstein:',
        'es': u':lápida:',
        'fr': u':pierre_tombale:',
        'pt': u':lápide:',
        'it': u':lapide:'
    },
    u'\U0001F9D1\U0000200D\U00002695\U0000FE0F': {
        'en' : u':health_worker:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':arzt/ärztin:',
        'es': u':profesional_sanitario:',
        'fr': u':professionnel_de_la_santé_(tous_genres):',
        'pt': u':profissional_de_saúde:',
        'it': u':persona_che_lavora_nella_sanità:'
    },
    u'\U0001F9D1\U0000200D\U00002695': {
        'en' : u':health_worker:',
        'status' : minimally_qualified,
        'E' : 12.1
    },
    u'\U0001F9D1\U0001F3FF\U0000200D\U00002695\U0000FE0F': {
        'en' : u':health_worker_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':arzt/ärztin_dunkle_hautfarbe:',
        'es': u':profesional_sanitario_tono_de_piel_oscuro:',
        'fr': u':professionnel_de_la_santé_(tous_genres)_peau_foncée:',
        'pt': u':profissional_de_saúde_pele_escura:',
        'it': u':persona_che_lavora_nella_sanità_carnagione_scura:'
    },
    u'\U0001F9D1\U0001F3FF\U0000200D\U00002695': {
        'en' : u':health_worker_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 12.1
    },
    u'\U0001F9D1\U0001F3FB\U0000200D\U00002695\U0000FE0F': {
        'en' : u':health_worker_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':arzt/ärztin_helle_hautfarbe:',
        'es': u':profesional_sanitario_tono_de_piel_claro:',
        'fr': u':professionnel_de_la_santé_(tous_genres)_peau_claire:',
        'pt': u':profissional_de_saúde_pele_clara:',
        'it': u':persona_che_lavora_nella_sanità_carnagione_chiara:'
    },
    u'\U0001F9D1\U0001F3FB\U0000200D\U00002695': {
        'en' : u':health_worker_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 12.1
    },
    u'\U0001F9D1\U0001F3FE\U0000200D\U00002695\U0000FE0F': {
        'en' : u':health_worker_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':arzt/ärztin_mitteldunkle_hautfarbe:',
        'es': u':profesional_sanitario_tono_de_piel_oscuro_medio:',
        'fr': u':professionnel_de_la_santé_(tous_genres)_peau_mate:',
        'pt': u':profissional_de_saúde_pele_morena_escura:',
        'it': u':persona_che_lavora_nella_sanità_carnagione_abbastanza_scura:'
    },
    u'\U0001F9D1\U0001F3FE\U0000200D\U00002695': {
        'en' : u':health_worker_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 12.1
    },
    u'\U0001F9D1\U0001F3FC\U0000200D\U00002695\U0000FE0F': {
        'en' : u':health_worker_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':arzt/ärztin_mittelhelle_hautfarbe:',
        'es': u':profesional_sanitario_tono_de_piel_claro_medio:',
        'fr': u':professionnel_de_la_santé_(tous_genres)_peau_moyennement_claire:',
        'pt': u':profissional_de_saúde_pele_morena_clara:',
        'it': u':persona_che_lavora_nella_sanità_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9D1\U0001F3FC\U0000200D\U00002695': {
        'en' : u':health_worker_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 12.1
    },
    u'\U0001F9D1\U0001F3FD\U0000200D\U00002695\U0000FE0F': {
        'en' : u':health_worker_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':arzt/ärztin_mittlere_hautfarbe:',
        'es': u':profesional_sanitario_tono_de_piel_medio:',
        'fr': u':professionnel_de_la_santé_(tous_genres)_peau_légèrement_mate:',
        'pt': u':profissional_de_saúde_pele_morena:',
        'it': u':persona_che_lavora_nella_sanità_carnagione_olivastra:'
    },
    u'\U0001F9D1\U0001F3FD\U0000200D\U00002695': {
        'en' : u':health_worker_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 12.1
    },
    u'\U0001F649': {
        'en' : u':hear-no-evil_monkey:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':hear_no_evil:'],
        'de': u':sich_die_ohren_zuhaltendes_affengesicht:',
        'es': u':mono_con_los_oídos_tapados:',
        'fr': u':singe_ne_rien_entendre:',
        'pt': u':macaco_que_não_ouve_nada:',
        'it': u':non_sento:'
    },
    u'\U0001F49F': {
        'en' : u':heart_decoration:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':herzdekoration:',
        'es': u':adorno_de_corazón:',
        'fr': u':décoration_avec_cœur:',
        'pt': u':coração_decorativo:',
        'it': u':decorazione_con_cuore:'
    },
    u'\U00002763\U0000FE0F': {
        'en' : u':heart_exclamation:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [u':heavy_heart_exclamation_mark_ornament:', u':heavy_heart_exclamation:'],
        'variant': True,
        'de': u':herz_als_ausrufezeichen:',
        'es': u':exclamación_de_corazón:',
        'fr': u':cœur_point_d’exclamation:',
        'pt': u':exclamação_de_coração:',
        'it': u':punto_esclamativo_a_cuore:'
    },
    u'\U00002763': {
        'en' : u':heart_exclamation:',
        'status' : unqualified,
        'E' : 1,
        'alias' : [u':heavy_heart_exclamation_mark_ornament:', u':heavy_heart_exclamation:'],
        'variant': True,
        'de': u':herz_als_ausrufezeichen:',
        'es': u':exclamación_de_corazón:',
        'fr': u':cœur_point_d’exclamation:',
        'pt': u':exclamação_de_coração:',
        'it': u':punto_esclamativo_a_cuore:'
    },
    u'\U0001FAF6': {
        'en' : u':heart_hands:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001FAF6\U0001F3FF': {
        'en' : u':heart_hands_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001FAF6\U0001F3FB': {
        'en' : u':heart_hands_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001FAF6\U0001F3FE': {
        'en' : u':heart_hands_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001FAF6\U0001F3FC': {
        'en' : u':heart_hands_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001FAF6\U0001F3FD': {
        'en' : u':heart_hands_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U00002764\U0000FE0F\U0000200D\U0001F525': {
        'en' : u':heart_on_fire:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U00002764\U0000200D\U0001F525': {
        'en' : u':heart_on_fire:',
        'status' : unqualified,
        'E' : 13.1
    },
    u'\U00002665\U0000FE0F': {
        'en' : u':heart_suit:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':hearts:'],
        'variant': True,
        'de': u':herz:',
        'es': u':palo_de_corazones:',
        'fr': u':cœur_cartes:',
        'pt': u':naipe_de_copas:',
        'it': u':cuori:'
    },
    u'\U00002665': {
        'en' : u':heart_suit:',
        'status' : unqualified,
        'E' : 0.6,
        'alias' : [u':hearts:'],
        'variant': True,
        'de': u':herz:',
        'es': u':palo_de_corazones:',
        'fr': u':cœur_cartes:',
        'pt': u':naipe_de_copas:',
        'it': u':cuori:'
    },
    u'\U0001F498': {
        'en' : u':heart_with_arrow:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':cupid:'],
        'de': u':herz_mit_pfeil:',
        'es': u':corazón_con_flecha:',
        'fr': u':cœur_et_flèche:',
        'pt': u':coração_com_flecha:',
        'it': u':cuore_con_freccia:'
    },
    u'\U0001F49D': {
        'en' : u':heart_with_ribbon:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':gift_heart:'],
        'de': u':herz_mit_schleife:',
        'es': u':corazón_con_lazo:',
        'fr': u':cœur_avec_ruban:',
        'pt': u':coração_com_fita:',
        'it': u':cuore_con_fiocco:'
    },
    u'\U0001F4B2': {
        'en' : u':heavy_dollar_sign:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':dollarzeichen:',
        'es': u':símbolo_de_dólar:',
        'fr': u':dollar:',
        'pt': u':cifrão:',
        'it': u':dollaro:'
    },
    u'\U0001F7F0': {
        'en' : u':heavy_equals_sign:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001F994': {
        'en' : u':hedgehog:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':igel:',
        'es': u':erizo:',
        'fr': u':hérisson:',
        'pt': u':porco-espinho:',
        'it': u':riccio:'
    },
    u'\U0001F681': {
        'en' : u':helicopter:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':hubschrauber:',
        'es': u':helicóptero:',
        'fr': u':hélicoptère:',
        'pt': u':helicóptero:',
        'it': u':elicottero:'
    },
    u'\U0001F33F': {
        'en' : u':herb:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':kräuter:',
        'es': u':hierba:',
        'fr': u':feuille:',
        'pt': u':erva:',
        'it': u':pianta:'
    },
    u'\U0001F33A': {
        'en' : u':hibiscus:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':hibiskus:',
        'es': u':flor_de_hibisco:',
        'fr': u':hibiscus:',
        'pt': u':hibisco:',
        'it': u':ibisco:'
    },
    u'\U0001F460': {
        'en' : u':high-heeled_shoe:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':high_heel:'],
        'de': u':stöckelschuh:',
        'es': u':zapato_de_tacón:',
        'fr': u':chaussure_à_talon_haut:',
        'pt': u':sapato_de_salto_alto:',
        'it': u':scarpa_con_il_tacco:'
    },
    u'\U0001F684': {
        'en' : u':high-speed_train:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':bullettrain_side:'],
        'de': u':hochgeschwindigkeitszug_mit_spitzer_nase:',
        'es': u':tren_de_alta_velocidad:',
        'fr': u':tgv:',
        'pt': u':trem_de_alta_velocidade:',
        'it': u':treno_alta_velocità:'
    },
    u'\U000026A1': {
        'en' : u':high_voltage:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':zap:'],
        'variant': True,
        'de': u':hochspannung:',
        'es': u':alto_voltaje:',
        'fr': u':symbole_de_haute_tension:',
        'pt': u':alta_tensão:',
        'it': u':alta_tensione:'
    },
    u'\U0001F97E': {
        'en' : u':hiking_boot:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':wanderstiefel:',
        'es': u':bota_de_senderismo:',
        'fr': u':chaussure_de_randonnée:',
        'pt': u':bota_de_trekking:',
        'it': u':stivale_da_trekking:'
    },
    u'\U0001F6D5': {
        'en' : u':hindu_temple:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':hindutempel:',
        'es': u':templo_hindú:',
        'fr': u':temple_hindou:',
        'pt': u':templo_hindu:',
        'it': u':tempio_indù:'
    },
    u'\U0001F99B': {
        'en' : u':hippopotamus:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':nilpferd:',
        'es': u':hipopótamo:',
        'fr': u':hippopotame:',
        'pt': u':hipopótamo:',
        'it': u':ippopotamo:'
    },
    u'\U0001F573\U0000FE0F': {
        'en' : u':hole:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': u':loch:',
        'es': u':agujero:',
        'fr': u':trou:',
        'pt': u':buraco:',
        'it': u':buco:'
    },
    u'\U0001F573': {
        'en' : u':hole:',
        'status' : unqualified,
        'E' : 0.7,
        'variant': True,
        'de': u':loch:',
        'es': u':agujero:',
        'fr': u':trou:',
        'pt': u':buraco:',
        'it': u':buco:'
    },
    u'\U00002B55': {
        'en' : u':hollow_red_circle:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':o:'],
        'variant': True,
        'de': u':hohler_roter_kreis:',
        'es': u':círculo_rojo_hueco:',
        'fr': u':cercle_rouge:',
        'pt': u':círculo_grande_oco:',
        'it': u':cerchio:'
    },
    u'\U0001F36F': {
        'en' : u':honey_pot:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':honigtopf:',
        'es': u':tarro_de_miel:',
        'fr': u':pot_de_miel:',
        'pt': u':pote_de_mel:',
        'it': u':barattolo_di_miele:'
    },
    u'\U0001F41D': {
        'en' : u':honeybee:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':bee:'],
        'de': u':biene:',
        'es': u':abeja:',
        'fr': u':abeille:',
        'pt': u':abelha:',
        'it': u':ape:'
    },
    u'\U0001FA9D': {
        'en' : u':hook:',
        'status' : fully_qualified,
        'E' : 13,
        'de': u':angelhaken:',
        'es': u':gancho:',
        'fr': u':crochet:',
        'pt': u':gancho:',
        'it': u':gancio:'
    },
    u'\U0001F6A5': {
        'en' : u':horizontal_traffic_light:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':traffic_light:'],
        'de': u':horizontale_verkehrsampel:',
        'es': u':semáforo_horizontal:',
        'fr': u':feu_tricolore_horizontal:',
        'pt': u':semáforo_horizontal:',
        'it': u':semaforo_orizzontale:'
    },
    u'\U0001F40E': {
        'en' : u':horse:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':racehorse:'],
        'de': u':pferd:',
        'es': u':caballo:',
        'fr': u':cheval:',
        'pt': u':cavalo:',
        'it': u':cavallo:'
    },
    u'\U0001F434': {
        'en' : u':horse_face:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':horse:'],
        'de': u':pferdegesicht:',
        'es': u':cara_de_caballo:',
        'fr': u':tête_de_cheval:',
        'pt': u':rosto_de_cavalo:',
        'it': u':muso_di_cavallo:'
    },
    u'\U0001F3C7': {
        'en' : u':horse_racing:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':pferderennen:',
        'es': u':carrera_de_caballos:',
        'fr': u':course_hippique:',
        'pt': u':corrida_de_cavalos:',
        'it': u':ippica:'
    },
    u'\U0001F3C7\U0001F3FF': {
        'en' : u':horse_racing_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':pferderennen_dunkle_hautfarbe:',
        'es': u':carrera_de_caballos_tono_de_piel_oscuro:',
        'fr': u':course_hippique_peau_foncée:',
        'pt': u':corrida_de_cavalos_pele_escura:',
        'it': u':ippica_carnagione_scura:'
    },
    u'\U0001F3C7\U0001F3FB': {
        'en' : u':horse_racing_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':pferderennen_helle_hautfarbe:',
        'es': u':carrera_de_caballos_tono_de_piel_claro:',
        'fr': u':course_hippique_peau_claire:',
        'pt': u':corrida_de_cavalos_pele_clara:',
        'it': u':ippica_carnagione_chiara:'
    },
    u'\U0001F3C7\U0001F3FE': {
        'en' : u':horse_racing_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':pferderennen_mitteldunkle_hautfarbe:',
        'es': u':carrera_de_caballos_tono_de_piel_oscuro_medio:',
        'fr': u':course_hippique_peau_mate:',
        'pt': u':corrida_de_cavalos_pele_morena_escura:',
        'it': u':ippica_carnagione_abbastanza_scura:'
    },
    u'\U0001F3C7\U0001F3FC': {
        'en' : u':horse_racing_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':pferderennen_mittelhelle_hautfarbe:',
        'es': u':carrera_de_caballos_tono_de_piel_claro_medio:',
        'fr': u':course_hippique_peau_moyennement_claire:',
        'pt': u':corrida_de_cavalos_pele_morena_clara:',
        'it': u':ippica_carnagione_abbastanza_chiara:'
    },
    u'\U0001F3C7\U0001F3FD': {
        'en' : u':horse_racing_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':pferderennen_mittlere_hautfarbe:',
        'es': u':carrera_de_caballos_tono_de_piel_medio:',
        'fr': u':course_hippique_peau_légèrement_mate:',
        'pt': u':corrida_de_cavalos_pele_morena:',
        'it': u':ippica_carnagione_olivastra:'
    },
    u'\U0001F3E5': {
        'en' : u':hospital:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':krankenhaus:',
        'es': u':hospital:',
        'fr': u':hôpital:',
        'pt': u':hospital:',
        'it': u':ospedale:'
    },
    u'\U00002615': {
        'en' : u':hot_beverage:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':coffee:'],
        'variant': True,
        'de': u':heißgetränk:',
        'es': u':bebida_caliente:',
        'fr': u':boisson_chaude:',
        'pt': u':café:',
        'it': u':bevanda_calda:'
    },
    u'\U0001F32D': {
        'en' : u':hot_dog:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [u':hotdog:'],
        'de': u':hotdog:',
        'es': u':perrito_caliente:',
        'fr': u':hot_dog:',
        'pt': u':cachorro-quente:',
        'it': u':hot_dog:'
    },
    u'\U0001F975': {
        'en' : u':hot_face:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':schwitzendes_gesicht:',
        'es': u':cara_con_calor:',
        'fr': u':visage_rouge_et_chaud:',
        'pt': u':rosto_fervendo_de_calor:',
        'it': u':faccina_accaldata:'
    },
    u'\U0001F336\U0000FE0F': {
        'en' : u':hot_pepper:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': u':peperoni:',
        'es': u':chile_picante:',
        'fr': u':piment_rouge:',
        'pt': u':pimenta:',
        'it': u':peperoncino:'
    },
    u'\U0001F336': {
        'en' : u':hot_pepper:',
        'status' : unqualified,
        'E' : 0.7,
        'variant': True,
        'de': u':peperoni:',
        'es': u':chile_picante:',
        'fr': u':piment_rouge:',
        'pt': u':pimenta:',
        'it': u':peperoncino:'
    },
    u'\U00002668\U0000FE0F': {
        'en' : u':hot_springs:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':hotsprings:'],
        'variant': True,
        'de': u':heiße_quellen:',
        'es': u':aguas_termales:',
        'fr': u':sources_chaudes:',
        'pt': u':chamas:',
        'it': u':sorgenti_termali:'
    },
    u'\U00002668': {
        'en' : u':hot_springs:',
        'status' : unqualified,
        'E' : 0.6,
        'alias' : [u':hotsprings:'],
        'variant': True,
        'de': u':heiße_quellen:',
        'es': u':aguas_termales:',
        'fr': u':sources_chaudes:',
        'pt': u':chamas:',
        'it': u':sorgenti_termali:'
    },
    u'\U0001F3E8': {
        'en' : u':hotel:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':hotel:',
        'es': u':hotel:',
        'fr': u':hôtel:',
        'pt': u':hotel:',
        'it': u':hotel:'
    },
    u'\U0000231B': {
        'en' : u':hourglass_done:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':hourglass:'],
        'variant': True,
        'de': u':sanduhr:',
        'es': u':reloj_de_arena_sin_tiempo:',
        'fr': u':sablier:',
        'pt': u':ampulheta:',
        'it': u':clessidra:'
    },
    u'\U000023F3': {
        'en' : u':hourglass_not_done:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':hourglass_flowing_sand:'],
        'variant': True,
        'de': u':laufende_sanduhr:',
        'es': u':reloj_de_arena_con_tiempo:',
        'fr': u':sablier_avec_sable_qui_coule:',
        'pt': u':ampulheta_cheia_em_cima:',
        'it': u':clessidra_che_scorre:'
    },
    u'\U0001F3E0': {
        'en' : u':house:',
        'status' : fully_qualified,
        'E' : 0.6,
        'variant': True,
        'de': u':haus:',
        'es': u':casa:',
        'fr': u':pavillon:',
        'pt': u':casa:',
        'it': u':casa:'
    },
    u'\U0001F3E1': {
        'en' : u':house_with_garden:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':haus_mit_garten:',
        'es': u':casa_con_jardín:',
        'fr': u':maison_avec_jardin:',
        'pt': u':casa_com_jardim:',
        'it': u':casa_con_giardino:'
    },
    u'\U0001F3D8\U0000FE0F': {
        'en' : u':houses:',
        'status' : fully_qualified,
        'E' : 0.7,
        'alias' : [u':house_buildings:'],
        'variant': True,
        'de': u':wohnhäuser:',
        'es': u':casas:',
        'fr': u':maisons:',
        'pt': u':casas:',
        'it': u':case:'
    },
    u'\U0001F3D8': {
        'en' : u':houses:',
        'status' : unqualified,
        'E' : 0.7,
        'alias' : [u':house_buildings:'],
        'variant': True,
        'de': u':wohnhäuser:',
        'es': u':casas:',
        'fr': u':maisons:',
        'pt': u':casas:',
        'it': u':case:'
    },
    u'\U0001F4AF': {
        'en' : u':hundred_points:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':100:'],
        'de': u':100_punkte:',
        'es': u':cien_puntos:',
        'fr': u':cent_points:',
        'pt': u':cem_pontos:',
        'it': u':100_punti:'
    },
    u'\U0001F62F': {
        'en' : u':hushed_face:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [u':hushed:'],
        'de': u':verdutztes_gesicht:',
        'es': u':cara_estupefacta:',
        'fr': u':visage_ébahi:',
        'pt': u':rosto_surpreso:',
        'it': u':faccina_sorpresa:'
    },
    u'\U0001F6D6': {
        'en' : u':hut:',
        'status' : fully_qualified,
        'E' : 13,
        'de': u':hütte:',
        'es': u':cabaña:',
        'fr': u':hutte:',
        'pt': u':cabana:',
        'it': u':capanna:'
    },
    u'\U0001F9CA': {
        'en' : u':ice:',
        'status' : fully_qualified,
        'E' : 12,
        'alias' : [u':ice_cube:'],
        'de': u':eiswürfel:',
        'es': u':cubito_de_hielo:',
        'fr': u':glaçon:',
        'pt': u':cubo_de_gelo:',
        'it': u':cubetto_di_ghiaccio:'
    },
    u'\U0001F368': {
        'en' : u':ice_cream:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':eiscreme:',
        'es': u':helado:',
        'fr': u':glace:',
        'pt': u':sorvete:',
        'it': u':coppa_di_gelato:'
    },
    u'\U0001F3D2': {
        'en' : u':ice_hockey:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [u':ice_hockey_stick_and_puck:'],
        'de': u':eishockey:',
        'es': u':hockey_sobre_hielo:',
        'fr': u':hockey_sur_glace:',
        'pt': u':hóquei_no_gelo:',
        'it': u':hockey_su_ghiaccio:'
    },
    u'\U000026F8\U0000FE0F': {
        'en' : u':ice_skate:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': u':schlittschuh:',
        'es': u':patín_de_hielo:',
        'fr': u':patin_à_glace:',
        'pt': u':patins_de_gelo:',
        'it': u':pattinaggio_su_ghiaccio:'
    },
    u'\U000026F8': {
        'en' : u':ice_skate:',
        'status' : unqualified,
        'E' : 0.7,
        'variant': True,
        'de': u':schlittschuh:',
        'es': u':patín_de_hielo:',
        'fr': u':patin_à_glace:',
        'pt': u':patins_de_gelo:',
        'it': u':pattinaggio_su_ghiaccio:'
    },
    u'\U0001FAAA': {
        'en' : u':identification_card:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001F4E5': {
        'en' : u':inbox_tray:',
        'status' : fully_qualified,
        'E' : 0.6,
        'variant': True,
        'de': u':posteingang:',
        'es': u':bandeja_de_entrada:',
        'fr': u':boîte_de_réception:',
        'pt': u':bandeja_de_entrada:',
        'it': u':posta_ricevuta:'
    },
    u'\U0001F4E8': {
        'en' : u':incoming_envelope:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':eingehender_briefumschlag:',
        'es': u':sobre_entrante:',
        'fr': u':message_reçu:',
        'pt': u':envelope_chegando:',
        'it': u':posta_in_arrivo:'
    },
    u'\U0001FAF5': {
        'en' : u':index_pointing_at_the_viewer:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001FAF5\U0001F3FF': {
        'en' : u':index_pointing_at_the_viewer_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001FAF5\U0001F3FB': {
        'en' : u':index_pointing_at_the_viewer_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001FAF5\U0001F3FE': {
        'en' : u':index_pointing_at_the_viewer_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001FAF5\U0001F3FC': {
        'en' : u':index_pointing_at_the_viewer_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001FAF5\U0001F3FD': {
        'en' : u':index_pointing_at_the_viewer_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0000261D\U0000FE0F': {
        'en' : u':index_pointing_up:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':point_up:'],
        'variant': True,
        'de': u':nach_oben_weisender_zeigefinger_von_vorne:',
        'es': u':dedo_índice_hacia_arriba:',
        'fr': u':index_pointant_vers_le_haut:',
        'pt': u':indicador_apontando_para_cima:',
        'it': u':indice_verso_l’alto:'
    },
    u'\U0000261D': {
        'en' : u':index_pointing_up:',
        'status' : unqualified,
        'E' : 0.6,
        'alias' : [u':point_up:'],
        'variant': True,
        'de': u':nach_oben_weisender_zeigefinger_von_vorne:',
        'es': u':dedo_índice_hacia_arriba:',
        'fr': u':index_pointant_vers_le_haut:',
        'pt': u':indicador_apontando_para_cima:',
        'it': u':indice_verso_l’alto:'
    },
    u'\U0000261D\U0001F3FF': {
        'en' : u':index_pointing_up_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':nach_oben_weisender_zeigefinger_von_vorne_dunkle_hautfarbe:',
        'es': u':dedo_índice_hacia_arriba_tono_de_piel_oscuro:',
        'fr': u':index_pointant_vers_le_haut_peau_foncée:',
        'pt': u':indicador_apontando_para_cima_pele_escura:',
        'it': u':indice_verso_l’alto_carnagione_scura:'
    },
    u'\U0000261D\U0001F3FB': {
        'en' : u':index_pointing_up_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':nach_oben_weisender_zeigefinger_von_vorne_helle_hautfarbe:',
        'es': u':dedo_índice_hacia_arriba_tono_de_piel_claro:',
        'fr': u':index_pointant_vers_le_haut_peau_claire:',
        'pt': u':indicador_apontando_para_cima_pele_clara:',
        'it': u':indice_verso_l’alto_carnagione_chiara:'
    },
    u'\U0000261D\U0001F3FE': {
        'en' : u':index_pointing_up_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':nach_oben_weisender_zeigefinger_von_vorne_mitteldunkle_hautfarbe:',
        'es': u':dedo_índice_hacia_arriba_tono_de_piel_oscuro_medio:',
        'fr': u':index_pointant_vers_le_haut_peau_mate:',
        'pt': u':indicador_apontando_para_cima_pele_morena_escura:',
        'it': u':indice_verso_l’alto_carnagione_abbastanza_scura:'
    },
    u'\U0000261D\U0001F3FC': {
        'en' : u':index_pointing_up_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':nach_oben_weisender_zeigefinger_von_vorne_mittelhelle_hautfarbe:',
        'es': u':dedo_índice_hacia_arriba_tono_de_piel_claro_medio:',
        'fr': u':index_pointant_vers_le_haut_peau_moyennement_claire:',
        'pt': u':indicador_apontando_para_cima_pele_morena_clara:',
        'it': u':indice_verso_l’alto_carnagione_abbastanza_chiara:'
    },
    u'\U0000261D\U0001F3FD': {
        'en' : u':index_pointing_up_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':nach_oben_weisender_zeigefinger_von_vorne_mittlere_hautfarbe:',
        'es': u':dedo_índice_hacia_arriba_tono_de_piel_medio:',
        'fr': u':index_pointant_vers_le_haut_peau_légèrement_mate:',
        'pt': u':indicador_apontando_para_cima_pele_morena:',
        'it': u':indice_verso_l’alto_carnagione_olivastra:'
    },
    u'\U0000267E\U0000FE0F': {
        'en' : u':infinity:',
        'status' : fully_qualified,
        'E' : 11,
        'variant': True,
        'de': u':unendlichkeit:',
        'es': u':infinito:',
        'fr': u':infini:',
        'pt': u':infinito:',
        'it': u':simbolo_dell’infinito:'
    },
    u'\U0000267E': {
        'en' : u':infinity:',
        'status' : unqualified,
        'E' : 11,
        'variant': True,
        'de': u':unendlichkeit:',
        'es': u':infinito:',
        'fr': u':infini:',
        'pt': u':infinito:',
        'it': u':simbolo_dell’infinito:'
    },
    u'\U00002139\U0000FE0F': {
        'en' : u':information:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':information_source:'],
        'variant': True,
        'de': u':buchstabe_i_in_blauem_quadrat:',
        'es': u':información:',
        'fr': u':source_d’informations:',
        'pt': u':informações:',
        'it': u':punto_informazioni:'
    },
    u'\U00002139': {
        'en' : u':information:',
        'status' : unqualified,
        'E' : 0.6,
        'alias' : [u':information_source:'],
        'variant': True,
        'de': u':buchstabe_i_in_blauem_quadrat:',
        'es': u':información:',
        'fr': u':source_d’informations:',
        'pt': u':informações:',
        'it': u':punto_informazioni:'
    },
    u'\U0001F524': {
        'en' : u':input_latin_letters:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':abc:'],
        'de': u':eingabesymbol_lateinische_buchstaben:',
        'es': u':alfabeto_latino:',
        'fr': u':alphabet_latin:',
        'pt': u':letras_latinas:',
        'it': u':pulsante_con_lettere_latine:'
    },
    u'\U0001F521': {
        'en' : u':input_latin_lowercase:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':abcd:'],
        'de': u':eingabesymbol_lateinische_kleinbuchstaben:',
        'es': u':letras_latinas_minúsculas:',
        'fr': u':minuscules:',
        'pt': u':letras_latinas_minúsculas:',
        'it': u':pulsante_con_lettere_latine_minuscole:'
    },
    u'\U0001F520': {
        'en' : u':input_latin_uppercase:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':capital_abcd:'],
        'de': u':eingabesymbol_lateinische_großbuchstaben:',
        'es': u':letras_latinas_mayúsculas:',
        'fr': u':majuscules:',
        'pt': u':letras_latinas_maiúsculas:',
        'it': u':pulsante_con_lettere_latine_maiuscole:'
    },
    u'\U0001F522': {
        'en' : u':input_numbers:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':1234:'],
        'de': u':eingabesymbol_zahlen:',
        'es': u':números:',
        'fr': u':saisie_de_chiffres:',
        'pt': u':números:',
        'it': u':pulsante_con_numeri:'
    },
    u'\U0001F523': {
        'en' : u':input_symbols:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':symbols:'],
        'de': u':eingabesymbol_sonderzeichen:',
        'es': u':símbolos:',
        'fr': u':saisie_de_symboles:',
        'pt': u':símbolos:',
        'it': u':pulsante_con_simboli:'
    },
    u'\U0001F383': {
        'en' : u':jack-o-lantern:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':jack_o_lantern:'],
        'de': u':halloweenkürbis:',
        'es': u':calabaza_de_halloween:',
        'fr': u':citrouille:',
        'pt': u':abóbora_de_halloween:',
        'it': u':zucca_di_halloween:'
    },
    u'\U0001FAD9': {
        'en' : u':jar:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001F456': {
        'en' : u':jeans:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':jeans:',
        'es': u':vaqueros:',
        'fr': u':jean:',
        'pt': u':jeans:',
        'it': u':jeans:'
    },
    u'\U0001F0CF': {
        'en' : u':joker:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':black_joker:'],
        'de': u':jokerkarte:',
        'es': u':comodín:',
        'fr': u':carte_joker:',
        'pt': u':curinga:',
        'it': u':jolly:'
    },
    u'\U0001F579\U0000FE0F': {
        'en' : u':joystick:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': u':joystick:',
        'es': u':joystick:',
        'fr': u':manette_de_jeu:',
        'pt': u':joystick:',
        'it': u':joystick:'
    },
    u'\U0001F579': {
        'en' : u':joystick:',
        'status' : unqualified,
        'E' : 0.7,
        'variant': True,
        'de': u':joystick:',
        'es': u':joystick:',
        'fr': u':manette_de_jeu:',
        'pt': u':joystick:',
        'it': u':joystick:'
    },
    u'\U0001F9D1\U0000200D\U00002696\U0000FE0F': {
        'en' : u':judge:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':richter(in):',
        'es': u':fiscal:',
        'fr': u':juge:',
        'pt': u':juiz_no_tribunal:',
        'it': u':giudice:'
    },
    u'\U0001F9D1\U0000200D\U00002696': {
        'en' : u':judge:',
        'status' : minimally_qualified,
        'E' : 12.1
    },
    u'\U0001F9D1\U0001F3FF\U0000200D\U00002696\U0000FE0F': {
        'en' : u':judge_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':richter(in)_dunkle_hautfarbe:',
        'es': u':fiscal_tono_de_piel_oscuro:',
        'fr': u':juge_peau_foncée:',
        'pt': u':juiz_no_tribunal_pele_escura:',
        'it': u':giudice_carnagione_scura:'
    },
    u'\U0001F9D1\U0001F3FF\U0000200D\U00002696': {
        'en' : u':judge_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 12.1
    },
    u'\U0001F9D1\U0001F3FB\U0000200D\U00002696\U0000FE0F': {
        'en' : u':judge_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':richter(in)_helle_hautfarbe:',
        'es': u':fiscal_tono_de_piel_claro:',
        'fr': u':juge_peau_claire:',
        'pt': u':juiz_no_tribunal_pele_clara:',
        'it': u':giudice_carnagione_chiara:'
    },
    u'\U0001F9D1\U0001F3FB\U0000200D\U00002696': {
        'en' : u':judge_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 12.1
    },
    u'\U0001F9D1\U0001F3FE\U0000200D\U00002696\U0000FE0F': {
        'en' : u':judge_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':richter(in)_mitteldunkle_hautfarbe:',
        'es': u':fiscal_tono_de_piel_oscuro_medio:',
        'fr': u':juge_peau_mate:',
        'pt': u':juiz_no_tribunal_pele_morena_escura:',
        'it': u':giudice_carnagione_abbastanza_scura:'
    },
    u'\U0001F9D1\U0001F3FE\U0000200D\U00002696': {
        'en' : u':judge_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 12.1
    },
    u'\U0001F9D1\U0001F3FC\U0000200D\U00002696\U0000FE0F': {
        'en' : u':judge_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':richter(in)_mittelhelle_hautfarbe:',
        'es': u':fiscal_tono_de_piel_claro_medio:',
        'fr': u':juge_peau_moyennement_claire:',
        'pt': u':juiz_no_tribunal_pele_morena_clara:',
        'it': u':giudice_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9D1\U0001F3FC\U0000200D\U00002696': {
        'en' : u':judge_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 12.1
    },
    u'\U0001F9D1\U0001F3FD\U0000200D\U00002696\U0000FE0F': {
        'en' : u':judge_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':richter(in)_mittlere_hautfarbe:',
        'es': u':fiscal_tono_de_piel_medio:',
        'fr': u':juge_peau_légèrement_mate:',
        'pt': u':juiz_no_tribunal_pele_morena:',
        'it': u':giudice_carnagione_olivastra:'
    },
    u'\U0001F9D1\U0001F3FD\U0000200D\U00002696': {
        'en' : u':judge_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 12.1
    },
    u'\U0001F54B': {
        'en' : u':kaaba:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':kaaba:',
        'es': u':kaaba:',
        'fr': u':kaaba:',
        'pt': u':caaba_islã:',
        'it': u':kaaba:'
    },
    u'\U0001F998': {
        'en' : u':kangaroo:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':känguru:',
        'es': u':canguro:',
        'fr': u':kangourou:',
        'pt': u':canguru:',
        'it': u':canguro:'
    },
    u'\U0001F511': {
        'en' : u':key:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':schlüssel:',
        'es': u':llave:',
        'fr': u':clé:',
        'pt': u':chave:',
        'it': u':chiave:'
    },
    u'\U00002328\U0000FE0F': {
        'en' : u':keyboard:',
        'status' : fully_qualified,
        'E' : 1,
        'variant': True,
        'de': u':tastatur:',
        'es': u':teclado:',
        'fr': u':clavier:',
        'pt': u':teclado:',
        'it': u':tastiera:'
    },
    u'\U00002328': {
        'en' : u':keyboard:',
        'status' : unqualified,
        'E' : 1,
        'variant': True,
        'de': u':tastatur:',
        'es': u':teclado:',
        'fr': u':clavier:',
        'pt': u':teclado:',
        'it': u':tastiera:'
    },
    u'\U00000023\U0000FE0F\U000020E3': {
        'en' : u':keycap_#:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':taste_#:',
        'es': u':teclas_#:',
        'fr': u':touches_#:',
        'pt': u':tecla_#:',
        'it': u':tasto_#:'
    },
    u'\U00000023\U000020E3': {
        'en' : u':keycap_#:',
        'status' : unqualified,
        'E' : 0.6,
        'alias' : [u':hash:']
    },
    u'\U0000002A\U0000FE0F\U000020E3': {
        'en' : u':keycap_*:',
        'status' : fully_qualified,
        'E' : 2,
        'de': u':taste_*:',
        'es': u':teclas_*:',
        'fr': u':touches_*:',
        'pt': u':tecla_*:',
        'it': u':tasto_*:'
    },
    u'\U0000002A\U000020E3': {
        'en' : u':keycap_*:',
        'status' : unqualified,
        'E' : 2,
        'alias' : [u':asterisk:']
    },
    u'\U00000030\U0000FE0F\U000020E3': {
        'en' : u':keycap_0:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':zero:'],
        'de': u':taste_0:',
        'es': u':teclas_0:',
        'fr': u':touches_0:',
        'pt': u':tecla_0:',
        'it': u':tasto_0:'
    },
    u'\U00000030\U000020E3': {
        'en' : u':keycap_0:',
        'status' : unqualified,
        'E' : 0.6,
        'alias' : [u':zero:']
    },
    u'\U00000031\U0000FE0F\U000020E3': {
        'en' : u':keycap_1:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':one:'],
        'de': u':taste_1:',
        'es': u':teclas_1:',
        'fr': u':touches_1:',
        'pt': u':tecla_1:',
        'it': u':tasto_1:'
    },
    u'\U00000031\U000020E3': {
        'en' : u':keycap_1:',
        'status' : unqualified,
        'E' : 0.6,
        'alias' : [u':one:']
    },
    u'\U0001F51F': {
        'en' : u':keycap_10:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':ten:', u':keycap_ten:'],
        'de': u':taste_10:',
        'es': u':teclas_10:',
        'fr': u':touches_10:',
        'pt': u':tecla_10:',
        'it': u':tasto_10:'
    },
    u'\U00000032\U0000FE0F\U000020E3': {
        'en' : u':keycap_2:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':two:'],
        'de': u':taste_2:',
        'es': u':teclas_2:',
        'fr': u':touches_2:',
        'pt': u':tecla_2:',
        'it': u':tasto_2:'
    },
    u'\U00000032\U000020E3': {
        'en' : u':keycap_2:',
        'status' : unqualified,
        'E' : 0.6,
        'alias' : [u':two:']
    },
    u'\U00000033\U0000FE0F\U000020E3': {
        'en' : u':keycap_3:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':three:'],
        'de': u':taste_3:',
        'es': u':teclas_3:',
        'fr': u':touches_3:',
        'pt': u':tecla_3:',
        'it': u':tasto_3:'
    },
    u'\U00000033\U000020E3': {
        'en' : u':keycap_3:',
        'status' : unqualified,
        'E' : 0.6,
        'alias' : [u':three:']
    },
    u'\U00000034\U0000FE0F\U000020E3': {
        'en' : u':keycap_4:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':four:'],
        'de': u':taste_4:',
        'es': u':teclas_4:',
        'fr': u':touches_4:',
        'pt': u':tecla_4:',
        'it': u':tasto_4:'
    },
    u'\U00000034\U000020E3': {
        'en' : u':keycap_4:',
        'status' : unqualified,
        'E' : 0.6,
        'alias' : [u':four:']
    },
    u'\U00000035\U0000FE0F\U000020E3': {
        'en' : u':keycap_5:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':five:'],
        'de': u':taste_5:',
        'es': u':teclas_5:',
        'fr': u':touches_5:',
        'pt': u':tecla_5:',
        'it': u':tasto_5:'
    },
    u'\U00000035\U000020E3': {
        'en' : u':keycap_5:',
        'status' : unqualified,
        'E' : 0.6,
        'alias' : [u':five:']
    },
    u'\U00000036\U0000FE0F\U000020E3': {
        'en' : u':keycap_6:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':six:'],
        'de': u':taste_6:',
        'es': u':teclas_6:',
        'fr': u':touches_6:',
        'pt': u':tecla_6:',
        'it': u':tasto_6:'
    },
    u'\U00000036\U000020E3': {
        'en' : u':keycap_6:',
        'status' : unqualified,
        'E' : 0.6,
        'alias' : [u':six:']
    },
    u'\U00000037\U0000FE0F\U000020E3': {
        'en' : u':keycap_7:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':seven:'],
        'de': u':taste_7:',
        'es': u':teclas_7:',
        'fr': u':touches_7:',
        'pt': u':tecla_7:',
        'it': u':tasto_7:'
    },
    u'\U00000037\U000020E3': {
        'en' : u':keycap_7:',
        'status' : unqualified,
        'E' : 0.6,
        'alias' : [u':seven:']
    },
    u'\U00000038\U0000FE0F\U000020E3': {
        'en' : u':keycap_8:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':eight:'],
        'de': u':taste_8:',
        'es': u':teclas_8:',
        'fr': u':touches_8:',
        'pt': u':tecla_8:',
        'it': u':tasto_8:'
    },
    u'\U00000038\U000020E3': {
        'en' : u':keycap_8:',
        'status' : unqualified,
        'E' : 0.6,
        'alias' : [u':eight:']
    },
    u'\U00000039\U0000FE0F\U000020E3': {
        'en' : u':keycap_9:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':nine:'],
        'de': u':taste_9:',
        'es': u':teclas_9:',
        'fr': u':touches_9:',
        'pt': u':tecla_9:',
        'it': u':tasto_9:'
    },
    u'\U00000039\U000020E3': {
        'en' : u':keycap_9:',
        'status' : unqualified,
        'E' : 0.6,
        'alias' : [u':nine:']
    },
    u'\U0001F6F4': {
        'en' : u':kick_scooter:',
        'status' : fully_qualified,
        'E' : 3,
        'de': u':tretroller:',
        'es': u':patinete:',
        'fr': u':trottinette:',
        'pt': u':patinete:',
        'it': u':monopattino:'
    },
    u'\U0001F458': {
        'en' : u':kimono:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':kimono:',
        'es': u':kimono:',
        'fr': u':kimono:',
        'pt': u':quimono:',
        'it': u':kimono:'
    },
    u'\U0001F48F': {
        'en' : u':kiss:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':couplekiss:'],
        'de': u':sich_küssendes_paar:',
        'es': u':beso:',
        'fr': u':bisou:',
        'pt': u':beijo:',
        'it': u':bacio_tra_coppia:'
    },
    u'\U0001F48F\U0001F3FF': {
        'en' : u':kiss_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F48F\U0001F3FB': {
        'en' : u':kiss_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468': {
        'en' : u':kiss_man_man:',
        'status' : fully_qualified,
        'E' : 2,
        'de': u':sich_küssendes_paar_mann,_mann:',
        'es': u':beso_hombre_y_hombre:',
        'fr': u':bisou_homme_et_homme:',
        'pt': u':beijo_homem_e_homem:',
        'it': u':bacio_tra_coppia_uomo_e_uomo:'
    },
    u'\U0001F468\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468': {
        'en' : u':kiss_man_man:',
        'status' : minimally_qualified,
        'E' : 2
    },
    u'\U0001F468\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FF': {
        'en' : u':kiss_man_man_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FF': {
        'en' : u':kiss_man_man_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FB': {
        'en' : u':kiss_man_man_dark_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FB': {
        'en' : u':kiss_man_man_dark_skin_tone_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FE': {
        'en' : u':kiss_man_man_dark_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FE': {
        'en' : u':kiss_man_man_dark_skin_tone_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FC': {
        'en' : u':kiss_man_man_dark_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FC': {
        'en' : u':kiss_man_man_dark_skin_tone_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FD': {
        'en' : u':kiss_man_man_dark_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FD': {
        'en' : u':kiss_man_man_dark_skin_tone_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FB': {
        'en' : u':kiss_man_man_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FB': {
        'en' : u':kiss_man_man_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FF': {
        'en' : u':kiss_man_man_light_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FF': {
        'en' : u':kiss_man_man_light_skin_tone_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FE': {
        'en' : u':kiss_man_man_light_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FE': {
        'en' : u':kiss_man_man_light_skin_tone_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FC': {
        'en' : u':kiss_man_man_light_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FC': {
        'en' : u':kiss_man_man_light_skin_tone_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FD': {
        'en' : u':kiss_man_man_light_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FD': {
        'en' : u':kiss_man_man_light_skin_tone_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FE': {
        'en' : u':kiss_man_man_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FE': {
        'en' : u':kiss_man_man_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FF': {
        'en' : u':kiss_man_man_medium-dark_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FF': {
        'en' : u':kiss_man_man_medium-dark_skin_tone_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FB': {
        'en' : u':kiss_man_man_medium-dark_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FB': {
        'en' : u':kiss_man_man_medium-dark_skin_tone_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FC': {
        'en' : u':kiss_man_man_medium-dark_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FC': {
        'en' : u':kiss_man_man_medium-dark_skin_tone_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FD': {
        'en' : u':kiss_man_man_medium-dark_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FD': {
        'en' : u':kiss_man_man_medium-dark_skin_tone_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FC': {
        'en' : u':kiss_man_man_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FC': {
        'en' : u':kiss_man_man_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FF': {
        'en' : u':kiss_man_man_medium-light_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FF': {
        'en' : u':kiss_man_man_medium-light_skin_tone_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FB': {
        'en' : u':kiss_man_man_medium-light_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FB': {
        'en' : u':kiss_man_man_medium-light_skin_tone_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FE': {
        'en' : u':kiss_man_man_medium-light_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FE': {
        'en' : u':kiss_man_man_medium-light_skin_tone_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FD': {
        'en' : u':kiss_man_man_medium-light_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FD': {
        'en' : u':kiss_man_man_medium-light_skin_tone_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FD': {
        'en' : u':kiss_man_man_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FD': {
        'en' : u':kiss_man_man_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FF': {
        'en' : u':kiss_man_man_medium_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FF': {
        'en' : u':kiss_man_man_medium_skin_tone_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FB': {
        'en' : u':kiss_man_man_medium_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FB': {
        'en' : u':kiss_man_man_medium_skin_tone_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FE': {
        'en' : u':kiss_man_man_medium_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FE': {
        'en' : u':kiss_man_man_medium_skin_tone_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FC': {
        'en' : u':kiss_man_man_medium_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FC': {
        'en' : u':kiss_man_man_medium_skin_tone_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F48B': {
        'en' : u':kiss_mark:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':kiss:'],
        'de': u':kussabdruck:',
        'es': u':marca_de_beso:',
        'fr': u':trace_de_rouge_à_lèvres:',
        'pt': u':marca_de_beijo:',
        'it': u':impronta_della_bocca:'
    },
    u'\U0001F48F\U0001F3FE': {
        'en' : u':kiss_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F48F\U0001F3FC': {
        'en' : u':kiss_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F48F\U0001F3FD': {
        'en' : u':kiss_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FB': {
        'en' : u':kiss_person_person_dark_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FB': {
        'en' : u':kiss_person_person_dark_skin_tone_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FE': {
        'en' : u':kiss_person_person_dark_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FE': {
        'en' : u':kiss_person_person_dark_skin_tone_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FC': {
        'en' : u':kiss_person_person_dark_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FC': {
        'en' : u':kiss_person_person_dark_skin_tone_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FD': {
        'en' : u':kiss_person_person_dark_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FD': {
        'en' : u':kiss_person_person_dark_skin_tone_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FF': {
        'en' : u':kiss_person_person_light_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FF': {
        'en' : u':kiss_person_person_light_skin_tone_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FE': {
        'en' : u':kiss_person_person_light_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FE': {
        'en' : u':kiss_person_person_light_skin_tone_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FC': {
        'en' : u':kiss_person_person_light_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FC': {
        'en' : u':kiss_person_person_light_skin_tone_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FD': {
        'en' : u':kiss_person_person_light_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FD': {
        'en' : u':kiss_person_person_light_skin_tone_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FF': {
        'en' : u':kiss_person_person_medium-dark_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FF': {
        'en' : u':kiss_person_person_medium-dark_skin_tone_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FB': {
        'en' : u':kiss_person_person_medium-dark_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FB': {
        'en' : u':kiss_person_person_medium-dark_skin_tone_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FC': {
        'en' : u':kiss_person_person_medium-dark_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FC': {
        'en' : u':kiss_person_person_medium-dark_skin_tone_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FD': {
        'en' : u':kiss_person_person_medium-dark_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FD': {
        'en' : u':kiss_person_person_medium-dark_skin_tone_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FF': {
        'en' : u':kiss_person_person_medium-light_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FF': {
        'en' : u':kiss_person_person_medium-light_skin_tone_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FB': {
        'en' : u':kiss_person_person_medium-light_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FB': {
        'en' : u':kiss_person_person_medium-light_skin_tone_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FE': {
        'en' : u':kiss_person_person_medium-light_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FE': {
        'en' : u':kiss_person_person_medium-light_skin_tone_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FD': {
        'en' : u':kiss_person_person_medium-light_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FD': {
        'en' : u':kiss_person_person_medium-light_skin_tone_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FF': {
        'en' : u':kiss_person_person_medium_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FF': {
        'en' : u':kiss_person_person_medium_skin_tone_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FB': {
        'en' : u':kiss_person_person_medium_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FB': {
        'en' : u':kiss_person_person_medium_skin_tone_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FE': {
        'en' : u':kiss_person_person_medium_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FE': {
        'en' : u':kiss_person_person_medium_skin_tone_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FC': {
        'en' : u':kiss_person_person_medium_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FC': {
        'en' : u':kiss_person_person_medium_skin_tone_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468': {
        'en' : u':kiss_woman_man:',
        'status' : fully_qualified,
        'E' : 2,
        'de': u':sich_küssendes_paar_frau,_mann:',
        'es': u':beso_mujer_y_hombre:',
        'fr': u':bisou_femme_et_homme:',
        'pt': u':beijo_mulher_e_homem:',
        'it': u':bacio_tra_coppia_donna_e_uomo:'
    },
    u'\U0001F469\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468': {
        'en' : u':kiss_woman_man:',
        'status' : minimally_qualified,
        'E' : 2
    },
    u'\U0001F469\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FF': {
        'en' : u':kiss_woman_man_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FF': {
        'en' : u':kiss_woman_man_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FB': {
        'en' : u':kiss_woman_man_dark_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FB': {
        'en' : u':kiss_woman_man_dark_skin_tone_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FE': {
        'en' : u':kiss_woman_man_dark_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FE': {
        'en' : u':kiss_woman_man_dark_skin_tone_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FC': {
        'en' : u':kiss_woman_man_dark_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FC': {
        'en' : u':kiss_woman_man_dark_skin_tone_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FD': {
        'en' : u':kiss_woman_man_dark_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FD': {
        'en' : u':kiss_woman_man_dark_skin_tone_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FB': {
        'en' : u':kiss_woman_man_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FB': {
        'en' : u':kiss_woman_man_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FF': {
        'en' : u':kiss_woman_man_light_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FF': {
        'en' : u':kiss_woman_man_light_skin_tone_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FE': {
        'en' : u':kiss_woman_man_light_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FE': {
        'en' : u':kiss_woman_man_light_skin_tone_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FC': {
        'en' : u':kiss_woman_man_light_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FC': {
        'en' : u':kiss_woman_man_light_skin_tone_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FD': {
        'en' : u':kiss_woman_man_light_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FD': {
        'en' : u':kiss_woman_man_light_skin_tone_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FE': {
        'en' : u':kiss_woman_man_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FE': {
        'en' : u':kiss_woman_man_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FF': {
        'en' : u':kiss_woman_man_medium-dark_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FF': {
        'en' : u':kiss_woman_man_medium-dark_skin_tone_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FB': {
        'en' : u':kiss_woman_man_medium-dark_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FB': {
        'en' : u':kiss_woman_man_medium-dark_skin_tone_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FC': {
        'en' : u':kiss_woman_man_medium-dark_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FC': {
        'en' : u':kiss_woman_man_medium-dark_skin_tone_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FD': {
        'en' : u':kiss_woman_man_medium-dark_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FD': {
        'en' : u':kiss_woman_man_medium-dark_skin_tone_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FC': {
        'en' : u':kiss_woman_man_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FC': {
        'en' : u':kiss_woman_man_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FF': {
        'en' : u':kiss_woman_man_medium-light_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FF': {
        'en' : u':kiss_woman_man_medium-light_skin_tone_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FB': {
        'en' : u':kiss_woman_man_medium-light_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FB': {
        'en' : u':kiss_woman_man_medium-light_skin_tone_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FE': {
        'en' : u':kiss_woman_man_medium-light_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FE': {
        'en' : u':kiss_woman_man_medium-light_skin_tone_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FD': {
        'en' : u':kiss_woman_man_medium-light_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FD': {
        'en' : u':kiss_woman_man_medium-light_skin_tone_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FD': {
        'en' : u':kiss_woman_man_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FD': {
        'en' : u':kiss_woman_man_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FF': {
        'en' : u':kiss_woman_man_medium_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FF': {
        'en' : u':kiss_woman_man_medium_skin_tone_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FB': {
        'en' : u':kiss_woman_man_medium_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FB': {
        'en' : u':kiss_woman_man_medium_skin_tone_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FE': {
        'en' : u':kiss_woman_man_medium_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FE': {
        'en' : u':kiss_woman_man_medium_skin_tone_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FC': {
        'en' : u':kiss_woman_man_medium_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FC': {
        'en' : u':kiss_woman_man_medium_skin_tone_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F469': {
        'en' : u':kiss_woman_woman:',
        'status' : fully_qualified,
        'E' : 2,
        'de': u':sich_küssendes_paar_frau,_frau:',
        'es': u':beso_mujer_y_mujer:',
        'fr': u':bisou_femme_et_femme:',
        'pt': u':beijo_mulher_e_mulher:',
        'it': u':bacio_tra_coppia_donna_e_donna:'
    },
    u'\U0001F469\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F469': {
        'en' : u':kiss_woman_woman:',
        'status' : minimally_qualified,
        'E' : 2
    },
    u'\U0001F469\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FF': {
        'en' : u':kiss_woman_woman_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FF': {
        'en' : u':kiss_woman_woman_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FB': {
        'en' : u':kiss_woman_woman_dark_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FB': {
        'en' : u':kiss_woman_woman_dark_skin_tone_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FE': {
        'en' : u':kiss_woman_woman_dark_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FE': {
        'en' : u':kiss_woman_woman_dark_skin_tone_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FC': {
        'en' : u':kiss_woman_woman_dark_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FC': {
        'en' : u':kiss_woman_woman_dark_skin_tone_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FD': {
        'en' : u':kiss_woman_woman_dark_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FD': {
        'en' : u':kiss_woman_woman_dark_skin_tone_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FB': {
        'en' : u':kiss_woman_woman_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FB': {
        'en' : u':kiss_woman_woman_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FF': {
        'en' : u':kiss_woman_woman_light_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FF': {
        'en' : u':kiss_woman_woman_light_skin_tone_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FE': {
        'en' : u':kiss_woman_woman_light_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FE': {
        'en' : u':kiss_woman_woman_light_skin_tone_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FC': {
        'en' : u':kiss_woman_woman_light_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FC': {
        'en' : u':kiss_woman_woman_light_skin_tone_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FD': {
        'en' : u':kiss_woman_woman_light_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FD': {
        'en' : u':kiss_woman_woman_light_skin_tone_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FE': {
        'en' : u':kiss_woman_woman_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FE': {
        'en' : u':kiss_woman_woman_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FF': {
        'en' : u':kiss_woman_woman_medium-dark_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FF': {
        'en' : u':kiss_woman_woman_medium-dark_skin_tone_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FB': {
        'en' : u':kiss_woman_woman_medium-dark_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FB': {
        'en' : u':kiss_woman_woman_medium-dark_skin_tone_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FC': {
        'en' : u':kiss_woman_woman_medium-dark_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FC': {
        'en' : u':kiss_woman_woman_medium-dark_skin_tone_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FD': {
        'en' : u':kiss_woman_woman_medium-dark_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FD': {
        'en' : u':kiss_woman_woman_medium-dark_skin_tone_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FC': {
        'en' : u':kiss_woman_woman_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FC': {
        'en' : u':kiss_woman_woman_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FF': {
        'en' : u':kiss_woman_woman_medium-light_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FF': {
        'en' : u':kiss_woman_woman_medium-light_skin_tone_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FB': {
        'en' : u':kiss_woman_woman_medium-light_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FB': {
        'en' : u':kiss_woman_woman_medium-light_skin_tone_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FE': {
        'en' : u':kiss_woman_woman_medium-light_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FE': {
        'en' : u':kiss_woman_woman_medium-light_skin_tone_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FD': {
        'en' : u':kiss_woman_woman_medium-light_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FD': {
        'en' : u':kiss_woman_woman_medium-light_skin_tone_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FD': {
        'en' : u':kiss_woman_woman_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FD': {
        'en' : u':kiss_woman_woman_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FF': {
        'en' : u':kiss_woman_woman_medium_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FF': {
        'en' : u':kiss_woman_woman_medium_skin_tone_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FB': {
        'en' : u':kiss_woman_woman_medium_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FB': {
        'en' : u':kiss_woman_woman_medium_skin_tone_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FE': {
        'en' : u':kiss_woman_woman_medium_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FE': {
        'en' : u':kiss_woman_woman_medium_skin_tone_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FC': {
        'en' : u':kiss_woman_woman_medium_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FC': {
        'en' : u':kiss_woman_woman_medium_skin_tone_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F63D': {
        'en' : u':kissing_cat:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':küssende_katze:',
        'es': u':gato_besando:',
        'fr': u':chat_qui_fait_un_bisou:',
        'pt': u':rosto_de_gato_mandando_um_beijo:',
        'it': u':gatto_che_manda_baci:'
    },
    u'\U0001F617': {
        'en' : u':kissing_face:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [u':kissing:'],
        'de': u':küssendes_gesicht:',
        'es': u':cara_besando:',
        'fr': u':visage_faisant_un_bisou:',
        'pt': u':rosto_beijando:',
        'it': u':faccina_che_bacia:'
    },
    u'\U0001F61A': {
        'en' : u':kissing_face_with_closed_eyes:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':kissing_closed_eyes:'],
        'de': u':küssendes_gesicht_mit_geschlossenen_augen:',
        'es': u':cara_besando_con_los_ojos_cerrados:',
        'fr': u':visage_faisant_un_bisou_avec_les_yeux_fermés:',
        'pt': u':rosto_beijando_com_olhos_fechados:',
        'it': u':faccina_che_bacia_con_occhi_chiusi:'
    },
    u'\U0001F619': {
        'en' : u':kissing_face_with_smiling_eyes:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [u':kissing_smiling_eyes:'],
        'de': u':küssendes_gesicht_mit_lächelnden_augen:',
        'es': u':cara_besando_con_ojos_sonrientes:',
        'fr': u':visage_aux_yeux_rieurs_faisant_un_bisou:',
        'pt': u':rosto_beijando_com_olhos_sorridentes:',
        'it': u':faccina_che_bacia_con_occhi_sorridenti:'
    },
    u'\U0001F52A': {
        'en' : u':kitchen_knife:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':knife:', u':hocho:'],
        'de': u':küchenmesser:',
        'es': u':cuchillo_de_cocina:',
        'fr': u':couteau_de_cuisine:',
        'pt': u':faca_de_cozinha:',
        'it': u':coltello_da_cucina:'
    },
    u'\U0001FA81': {
        'en' : u':kite:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':drachen:',
        'es': u':cometa:',
        'fr': u':cerf-volant:',
        'pt': u':pipa:',
        'it': u':aquilone:'
    },
    u'\U0001F95D': {
        'en' : u':kiwi_fruit:',
        'status' : fully_qualified,
        'E' : 3,
        'de': u':kiwi:',
        'es': u':kiwi:',
        'fr': u':kiwi:',
        'pt': u':kiwi:',
        'it': u':kiwi:'
    },
    u'\U0001FAA2': {
        'en' : u':knot:',
        'status' : fully_qualified,
        'E' : 13,
        'de': u':knoten:',
        'es': u':nudo:',
        'fr': u':nœud:',
        'pt': u':nó:',
        'it': u':nodo:'
    },
    u'\U0001F428': {
        'en' : u':koala:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':koala:',
        'es': u':koala:',
        'fr': u':koala:',
        'pt': u':coala:',
        'it': u':koala:'
    },
    u'\U0001F97C': {
        'en' : u':lab_coat:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':laborkittel:',
        'es': u':bata_de_laboratorio:',
        'fr': u':blouse_blanche:',
        'pt': u':jaleco:',
        'it': u':camice:'
    },
    u'\U0001F3F7\U0000FE0F': {
        'en' : u':label:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': u':etikett:',
        'es': u':etiqueta:',
        'fr': u':étiquette:',
        'pt': u':etiqueta:',
        'it': u':etichetta:'
    },
    u'\U0001F3F7': {
        'en' : u':label:',
        'status' : unqualified,
        'E' : 0.7,
        'variant': True,
        'de': u':etikett:',
        'es': u':etiqueta:',
        'fr': u':étiquette:',
        'pt': u':etiqueta:',
        'it': u':etichetta:'
    },
    u'\U0001F94D': {
        'en' : u':lacrosse:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':lacrosse:',
        'es': u':lacrosse:',
        'fr': u':crosse:',
        'pt': u':lacrosse:',
        'it': u':lacrosse:'
    },
    u'\U0001FA9C': {
        'en' : u':ladder:',
        'status' : fully_qualified,
        'E' : 13,
        'de': u':leiter:',
        'es': u':escalera:',
        'fr': u':échelle:',
        'pt': u':escada:',
        'it': u':scala:'
    },
    u'\U0001F41E': {
        'en' : u':lady_beetle:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':beetle:'],
        'de': u':marienkäfer:',
        'es': u':mariquita:',
        'fr': u':coccinelle:',
        'pt': u':joaninha:',
        'it': u':coccinella:'
    },
    u'\U0001F4BB': {
        'en' : u':laptop:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':computer:'],
        'variant': True,
        'de': u':laptop:',
        'es': u':ordenador_portátil:',
        'fr': u':ordinateur_portable:',
        'pt': u':laptop:',
        'it': u':computer_portatile:'
    },
    u'\U0001F537': {
        'en' : u':large_blue_diamond:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':große_blaue_raute:',
        'es': u':rombo_azul_grande:',
        'fr': u':grand_losange_bleu:',
        'pt': u':losango_azul_grande:',
        'it': u':rombo_blu_grande:'
    },
    u'\U0001F536': {
        'en' : u':large_orange_diamond:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':große_orangefarbene_raute:',
        'es': u':rombo_naranja_grande:',
        'fr': u':grand_losange_orange:',
        'pt': u':losango_laranja_grande:',
        'it': u':rombo_arancione_grande:'
    },
    u'\U0001F317': {
        'en' : u':last_quarter_moon:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':abnehmender_halbmond:',
        'es': u':luna_en_cuarto_menguante:',
        'fr': u':dernier_quartier:',
        'pt': u':quarto_minguante:',
        'it': u':ultimo_quarto_di_luna:'
    },
    u'\U0001F31C': {
        'en' : u':last_quarter_moon_face:',
        'status' : fully_qualified,
        'E' : 0.7,
        'alias' : [u':last_quarter_moon_with_face:'],
        'variant': True,
        'de': u':mondsichel_mit_gesicht_rechts:',
        'es': u':luna_de_cuarto_menguante_con_cara:',
        'fr': u':dernier_quartier_de_lune_avec_visage:',
        'pt': u':rosto_da_lua_de_quarto_minguante:',
        'it': u':faccina_ultimo_quarto_di_luna:'
    },
    u'\U000023EE\U0000FE0F': {
        'en' : u':last_track_button:',
        'status' : fully_qualified,
        'E' : 0.7,
        'alias' : [u':previous_track_button:', u':black_left__pointing_double_triangle_with_vertical_bar:'],
        'variant': True,
        'de': u':vorheriger_titel:',
        'es': u':pista_anterior:',
        'fr': u':bouton_piste_précédente:',
        'pt': u':botão_de_faixa_anterior:',
        'it': u':pulsante_traccia_precedente:'
    },
    u'\U000023EE': {
        'en' : u':last_track_button:',
        'status' : unqualified,
        'E' : 0.7,
        'alias' : [u':previous_track_button:', u':black_left__pointing_double_triangle_with_vertical_bar:'],
        'variant': True,
        'de': u':vorheriger_titel:',
        'es': u':pista_anterior:',
        'fr': u':bouton_piste_précédente:',
        'pt': u':botão_de_faixa_anterior:',
        'it': u':pulsante_traccia_precedente:'
    },
    u'\U0000271D\U0000FE0F': {
        'en' : u':latin_cross:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': u':römisches_kreuz:',
        'es': u':cruz_latina:',
        'fr': u':croix_latine:',
        'pt': u':cruz_latina:',
        'it': u':croce_latina:'
    },
    u'\U0000271D': {
        'en' : u':latin_cross:',
        'status' : unqualified,
        'E' : 0.7,
        'variant': True,
        'de': u':römisches_kreuz:',
        'es': u':cruz_latina:',
        'fr': u':croix_latine:',
        'pt': u':cruz_latina:',
        'it': u':croce_latina:'
    },
    u'\U0001F343': {
        'en' : u':leaf_fluttering_in_wind:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':leaves:'],
        'de': u':blätter_im_wind:',
        'es': u':hojas_revoloteando_al_viento:',
        'fr': u':feuille_virevoltante:',
        'pt': u':folha_ao_vento:',
        'it': u':foglia_al_vento:'
    },
    u'\U0001F96C': {
        'en' : u':leafy_green:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':grünzeug:',
        'es': u':verdura_de_hoja_verde:',
        'fr': u':légume_à_feuilles_vertes:',
        'pt': u':verdura:',
        'it': u':verdure_a_foglia:'
    },
    u'\U0001F4D2': {
        'en' : u':ledger:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':spiralblock:',
        'es': u':libro_de_contabilidad:',
        'fr': u':carnet_de_compte:',
        'pt': u':livro_contábil:',
        'it': u':quaderno_ad_anelli:'
    },
    u'\U0001F91B': {
        'en' : u':left-facing_fist:',
        'status' : fully_qualified,
        'E' : 3,
        'alias' : [u':fist_left:'],
        'de': u':faust_nach_links:',
        'es': u':puño_hacia_la_izquierda:',
        'fr': u':poing_à_gauche:',
        'pt': u':punho_esquerdo:',
        'it': u':pugno_a_sinistra:'
    },
    u'\U0001F91B\U0001F3FF': {
        'en' : u':left-facing_fist_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 3,
        'de': u':faust_nach_links_dunkle_hautfarbe:',
        'es': u':puño_hacia_la_izquierda_tono_de_piel_oscuro:',
        'fr': u':poing_à_gauche_peau_foncée:',
        'pt': u':punho_esquerdo_pele_escura:',
        'it': u':pugno_a_sinistra_carnagione_scura:'
    },
    u'\U0001F91B\U0001F3FB': {
        'en' : u':left-facing_fist_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 3,
        'de': u':faust_nach_links_helle_hautfarbe:',
        'es': u':puño_hacia_la_izquierda_tono_de_piel_claro:',
        'fr': u':poing_à_gauche_peau_claire:',
        'pt': u':punho_esquerdo_pele_clara:',
        'it': u':pugno_a_sinistra_carnagione_chiara:'
    },
    u'\U0001F91B\U0001F3FE': {
        'en' : u':left-facing_fist_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 3,
        'de': u':faust_nach_links_mitteldunkle_hautfarbe:',
        'es': u':puño_hacia_la_izquierda_tono_de_piel_oscuro_medio:',
        'fr': u':poing_à_gauche_peau_mate:',
        'pt': u':punho_esquerdo_pele_morena_escura:',
        'it': u':pugno_a_sinistra_carnagione_abbastanza_scura:'
    },
    u'\U0001F91B\U0001F3FC': {
        'en' : u':left-facing_fist_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 3,
        'de': u':faust_nach_links_mittelhelle_hautfarbe:',
        'es': u':puño_hacia_la_izquierda_tono_de_piel_claro_medio:',
        'fr': u':poing_à_gauche_peau_moyennement_claire:',
        'pt': u':punho_esquerdo_pele_morena_clara:',
        'it': u':pugno_a_sinistra_carnagione_abbastanza_chiara:'
    },
    u'\U0001F91B\U0001F3FD': {
        'en' : u':left-facing_fist_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 3,
        'de': u':faust_nach_links_mittlere_hautfarbe:',
        'es': u':puño_hacia_la_izquierda_tono_de_piel_medio:',
        'fr': u':poing_à_gauche_peau_légèrement_mate:',
        'pt': u':punho_esquerdo_pele_morena:',
        'it': u':pugno_a_sinistra_carnagione_olivastra:'
    },
    u'\U00002194\U0000FE0F': {
        'en' : u':left-right_arrow:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':left_right_arrow:'],
        'variant': True,
        'de': u':pfeil_nach_links_und_rechts:',
        'es': u':flecha_izquierda_y_derecha:',
        'fr': u':flèche_gauche_droite:',
        'pt': u':seta_para_esquerda_e_direita:',
        'it': u':freccia_sinistra-destra:'
    },
    u'\U00002194': {
        'en' : u':left-right_arrow:',
        'status' : unqualified,
        'E' : 0.6,
        'alias' : [u':left_right_arrow:'],
        'variant': True,
        'de': u':pfeil_nach_links_und_rechts:',
        'es': u':flecha_izquierda_y_derecha:',
        'fr': u':flèche_gauche_droite:',
        'pt': u':seta_para_esquerda_e_direita:',
        'it': u':freccia_sinistra-destra:'
    },
    u'\U00002B05\U0000FE0F': {
        'en' : u':left_arrow:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':arrow_left:'],
        'variant': True,
        'de': u':pfeil_nach_links:',
        'es': u':flecha_hacia_la_izquierda:',
        'fr': u':flèche_gauche:',
        'pt': u':seta_para_a_esquerda:',
        'it': u':freccia_rivolta_a_sinistra:'
    },
    u'\U00002B05': {
        'en' : u':left_arrow:',
        'status' : unqualified,
        'E' : 0.6,
        'alias' : [u':arrow_left:'],
        'variant': True,
        'de': u':pfeil_nach_links:',
        'es': u':flecha_hacia_la_izquierda:',
        'fr': u':flèche_gauche:',
        'pt': u':seta_para_a_esquerda:',
        'it': u':freccia_rivolta_a_sinistra:'
    },
    u'\U000021AA\U0000FE0F': {
        'en' : u':left_arrow_curving_right:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':arrow_right_hook:'],
        'variant': True,
        'de': u':geschwungener_pfeil_nach_rechts:',
        'es': u':flecha_izquierda_curvándose_a_la_derecha:',
        'fr': u':flèche_courbe_droite:',
        'pt': u':seta_curva_da_esquerda_para_a_direita:',
        'it': u':freccia_curva_a_destra:'
    },
    u'\U000021AA': {
        'en' : u':left_arrow_curving_right:',
        'status' : unqualified,
        'E' : 0.6,
        'alias' : [u':arrow_right_hook:'],
        'variant': True,
        'de': u':geschwungener_pfeil_nach_rechts:',
        'es': u':flecha_izquierda_curvándose_a_la_derecha:',
        'fr': u':flèche_courbe_droite:',
        'pt': u':seta_curva_da_esquerda_para_a_direita:',
        'it': u':freccia_curva_a_destra:'
    },
    u'\U0001F6C5': {
        'en' : u':left_luggage:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':gepäckaufbewahrung:',
        'es': u':consigna:',
        'fr': u':consigne:',
        'pt': u':depósito_de_bagagem:',
        'it': u':simbolo_del_deposito_bagagli:'
    },
    u'\U0001F5E8\U0000FE0F': {
        'en' : u':left_speech_bubble:',
        'status' : fully_qualified,
        'E' : 2,
        'variant': True,
        'de': u':sprechblase_links:',
        'es': u':bocadillo_de_diálogo_por_la_izquierda:',
        'fr': u':bulle_de_parole_gauche:',
        'pt': u':balão_de_diálogo_à_esquerda:',
        'it': u':nuvoletta_nera:'
    },
    u'\U0001F5E8': {
        'en' : u':left_speech_bubble:',
        'status' : unqualified,
        'E' : 2,
        'variant': True,
        'de': u':sprechblase_links:',
        'es': u':bocadillo_de_diálogo_por_la_izquierda:',
        'fr': u':bulle_de_parole_gauche:',
        'pt': u':balão_de_diálogo_à_esquerda:',
        'it': u':nuvoletta_nera:'
    },
    u'\U0001FAF2': {
        'en' : u':leftwards_hand:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001FAF2\U0001F3FF': {
        'en' : u':leftwards_hand_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001FAF2\U0001F3FB': {
        'en' : u':leftwards_hand_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001FAF2\U0001F3FE': {
        'en' : u':leftwards_hand_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001FAF2\U0001F3FC': {
        'en' : u':leftwards_hand_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001FAF2\U0001F3FD': {
        'en' : u':leftwards_hand_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001F9B5': {
        'en' : u':leg:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':bein:',
        'es': u':pierna:',
        'fr': u':jambe:',
        'pt': u':perna:',
        'it': u':gamba:'
    },
    u'\U0001F9B5\U0001F3FF': {
        'en' : u':leg_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':bein_dunkle_hautfarbe:',
        'es': u':pierna_tono_de_piel_oscuro:',
        'fr': u':jambe_peau_foncée:',
        'pt': u':perna_pele_escura:',
        'it': u':gamba_carnagione_scura:'
    },
    u'\U0001F9B5\U0001F3FB': {
        'en' : u':leg_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':bein_helle_hautfarbe:',
        'es': u':pierna_tono_de_piel_claro:',
        'fr': u':jambe_peau_claire:',
        'pt': u':perna_pele_clara:',
        'it': u':gamba_carnagione_chiara:'
    },
    u'\U0001F9B5\U0001F3FE': {
        'en' : u':leg_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':bein_mitteldunkle_hautfarbe:',
        'es': u':pierna_tono_de_piel_oscuro_medio:',
        'fr': u':jambe_peau_mate:',
        'pt': u':perna_pele_morena_escura:',
        'it': u':gamba_carnagione_abbastanza_scura:'
    },
    u'\U0001F9B5\U0001F3FC': {
        'en' : u':leg_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':bein_mittelhelle_hautfarbe:',
        'es': u':pierna_tono_de_piel_claro_medio:',
        'fr': u':jambe_peau_moyennement_claire:',
        'pt': u':perna_pele_morena_clara:',
        'it': u':gamba_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9B5\U0001F3FD': {
        'en' : u':leg_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':bein_mittlere_hautfarbe:',
        'es': u':pierna_tono_de_piel_medio:',
        'fr': u':jambe_peau_légèrement_mate:',
        'pt': u':perna_pele_morena:',
        'it': u':gamba_carnagione_olivastra:'
    },
    u'\U0001F34B': {
        'en' : u':lemon:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':zitrone:',
        'es': u':limón:',
        'fr': u':citron:',
        'pt': u':limão:',
        'it': u':limone:'
    },
    u'\U0001F406': {
        'en' : u':leopard:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':leopard:',
        'es': u':leopardo:',
        'fr': u':léopard:',
        'pt': u':leopardo:',
        'it': u':leopardo:'
    },
    u'\U0001F39A\U0000FE0F': {
        'en' : u':level_slider:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': u':schieberegler:',
        'es': u':control_de_volumen:',
        'fr': u':curseur_de_niveau:',
        'pt': u':controle_de_volume:',
        'it': u':cursore_del_volume:'
    },
    u'\U0001F39A': {
        'en' : u':level_slider:',
        'status' : unqualified,
        'E' : 0.7,
        'variant': True,
        'de': u':schieberegler:',
        'es': u':control_de_volumen:',
        'fr': u':curseur_de_niveau:',
        'pt': u':controle_de_volume:',
        'it': u':cursore_del_volume:'
    },
    u'\U0001F4A1': {
        'en' : u':light_bulb:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':bulb:'],
        'de': u':glühbirne:',
        'es': u':bombilla:',
        'fr': u':ampoule:',
        'pt': u':lâmpada:',
        'it': u':lampadina:'
    },
    u'\U0001F688': {
        'en' : u':light_rail:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':s-bahn:',
        'es': u':tren_ligero:',
        'fr': u':métro_léger:',
        'pt': u':trem_urbano:',
        'it': u':metrotranvia:'
    },
    u'\U0001F3FB': {
        'en' : u':light_skin_tone:',
        'status' : component,
        'E' : 1,
        'alias' : [u':emoji_modifier_fitzpatrick_type__1__2:'],
        'de': u':helle_hautfarbe:',
        'es': u':tono_de_piel_claro:',
        'fr': u':peau_claire:',
        'pt': u':pele_clara:',
        'it': u':carnagione_chiara:'
    },
    u'\U0001F517': {
        'en' : u':link:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':verknüpfungssymbol:',
        'es': u':eslabón:',
        'fr': u':chaînons:',
        'pt': u':link:',
        'it': u':collegamento:'
    },
    u'\U0001F587\U0000FE0F': {
        'en' : u':linked_paperclips:',
        'status' : fully_qualified,
        'E' : 0.7,
        'alias' : [u':paperclips:'],
        'variant': True,
        'de': u':verhakte_büroklammern:',
        'es': u':clips_unidos:',
        'fr': u':trombones:',
        'pt': u':clipes_de_papel_conectados:',
        'it': u':graffette_attaccate:'
    },
    u'\U0001F587': {
        'en' : u':linked_paperclips:',
        'status' : unqualified,
        'E' : 0.7,
        'alias' : [u':paperclips:'],
        'variant': True,
        'de': u':verhakte_büroklammern:',
        'es': u':clips_unidos:',
        'fr': u':trombones:',
        'pt': u':clipes_de_papel_conectados:',
        'it': u':graffette_attaccate:'
    },
    u'\U0001F981': {
        'en' : u':lion:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [u':lion_face:'],
        'de': u':löwe:',
        'es': u':león:',
        'fr': u':tête_de_lion:',
        'pt': u':rosto_de_leão:',
        'it': u':leone:'
    },
    u'\U0001F484': {
        'en' : u':lipstick:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':lippenstift:',
        'es': u':pintalabios:',
        'fr': u':rouge_à_lèvres:',
        'pt': u':batom:',
        'it': u':rossetto:'
    },
    u'\U0001F6AE': {
        'en' : u':litter_in_bin_sign:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [u':put_litter_in_its_place:'],
        'de': u':symbol_papierkorb:',
        'es': u':señal_de_usar_papelera:',
        'fr': u':icône_poubelle:',
        'pt': u':símbolo_de_lixeira:',
        'it': u':simbolo_per_la_raccolta_dei_rifiuti:'
    },
    u'\U0001F98E': {
        'en' : u':lizard:',
        'status' : fully_qualified,
        'E' : 3,
        'de': u':eidechse:',
        'es': u':lagarto:',
        'fr': u':lézard:',
        'pt': u':lagartixa:',
        'it': u':lucertola:'
    },
    u'\U0001F999': {
        'en' : u':llama:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':lama:',
        'es': u':llama:',
        'fr': u':lama:',
        'pt': u':lhama:',
        'it': u':lama:'
    },
    u'\U0001F99E': {
        'en' : u':lobster:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':hummer:',
        'es': u':langosta:',
        'fr': u':homard:',
        'pt': u':lagosta:',
        'it': u':aragosta:'
    },
    u'\U0001F512': {
        'en' : u':locked:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':lock:'],
        'variant': True,
        'de': u':geschlossenes_schloss:',
        'es': u':candado_cerrado:',
        'fr': u':cadenas_fermé:',
        'pt': u':cadeado:',
        'it': u':lucchetto_chiuso:'
    },
    u'\U0001F510': {
        'en' : u':locked_with_key:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':closed_lock_with_key:'],
        'de': u':schloss_mit_schlüssel:',
        'es': u':candado_cerrado_y_llave:',
        'fr': u':cadenas_fermé_avec_clé:',
        'pt': u':cadeado_fechado_com_chave:',
        'it': u':lucchetto_chiuso_con_chiave:'
    },
    u'\U0001F50F': {
        'en' : u':locked_with_pen:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':lock_with_ink_pen:'],
        'de': u':schloss_mit_füller:',
        'es': u':candado_con_pluma_estilográfica:',
        'fr': u':cadenas_fermé_avec_stylo:',
        'pt': u':cadeado_com_caneta:',
        'it': u':lucchetto_con_penna_stilo:'
    },
    u'\U0001F682': {
        'en' : u':locomotive:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [u':steam_locomotive:'],
        'de': u':dampflokomotive:',
        'es': u':locomotora_de_vapor:',
        'fr': u':locomotive:',
        'pt': u':locomotiva:',
        'it': u':locomotiva:'
    },
    u'\U0001F36D': {
        'en' : u':lollipop:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':lutscher:',
        'es': u':piruleta:',
        'fr': u':sucette:',
        'pt': u':pirulito:',
        'it': u':lecca_lecca:'
    },
    u'\U0001FA98': {
        'en' : u':long_drum:',
        'status' : fully_qualified,
        'E' : 13,
        'de': u':afrikanische_trommel:',
        'es': u':tamboril:',
        'fr': u':djembé:',
        'pt': u':tambor_comprido:',
        'it': u':tamburo_lungo:'
    },
    u'\U0001F9F4': {
        'en' : u':lotion_bottle:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':creme:',
        'es': u':bote_de_crema:',
        'fr': u':bouteille_de_lotion:',
        'pt': u':frasco_de_loção:',
        'it': u':flacone:'
    },
    u'\U0001FAB7': {
        'en' : u':lotus:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001F62D': {
        'en' : u':loudly_crying_face:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':sob:'],
        'de': u':heulendes_gesicht:',
        'es': u':cara_llorando_fuerte:',
        'fr': u':visage_qui_pleure_à_chaudes_larmes:',
        'pt': u':rosto_chorando_aos_berros:',
        'it': u':faccina_disperata:'
    },
    u'\U0001F4E2': {
        'en' : u':loudspeaker:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':lautsprecher:',
        'es': u':altavoz_de_mano:',
        'fr': u':haut-parleur:',
        'pt': u':buzina:',
        'it': u':altoparlante:'
    },
    u'\U0001F91F': {
        'en' : u':love-you_gesture:',
        'status' : fully_qualified,
        'E' : 5,
        'alias' : [u':love_you_gesture:'],
        'de': u':ich-liebe-dich-geste:',
        'es': u':gesto_de_te_quiero:',
        'fr': u':signe_je_t’aime:',
        'pt': u':gesto_de_te_amo:',
        'it': u':gesto_ti_amo:'
    },
    u'\U0001F91F\U0001F3FF': {
        'en' : u':love-you_gesture_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':ich-liebe-dich-geste_dunkle_hautfarbe:',
        'es': u':gesto_de_te_quiero_tono_de_piel_oscuro:',
        'fr': u':signe_je_t’aime_peau_foncée:',
        'pt': u':gesto_de_te_amo_pele_escura:',
        'it': u':gesto_ti_amo_carnagione_scura:'
    },
    u'\U0001F91F\U0001F3FB': {
        'en' : u':love-you_gesture_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':ich-liebe-dich-geste_helle_hautfarbe:',
        'es': u':gesto_de_te_quiero_tono_de_piel_claro:',
        'fr': u':signe_je_t’aime_peau_claire:',
        'pt': u':gesto_de_te_amo_pele_clara:',
        'it': u':gesto_ti_amo_carnagione_chiara:'
    },
    u'\U0001F91F\U0001F3FE': {
        'en' : u':love-you_gesture_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':ich-liebe-dich-geste_mitteldunkle_hautfarbe:',
        'es': u':gesto_de_te_quiero_tono_de_piel_oscuro_medio:',
        'fr': u':signe_je_t’aime_peau_mate:',
        'pt': u':gesto_de_te_amo_pele_morena_escura:',
        'it': u':gesto_ti_amo_carnagione_abbastanza_scura:'
    },
    u'\U0001F91F\U0001F3FC': {
        'en' : u':love-you_gesture_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':ich-liebe-dich-geste_mittelhelle_hautfarbe:',
        'es': u':gesto_de_te_quiero_tono_de_piel_claro_medio:',
        'fr': u':signe_je_t’aime_peau_moyennement_claire:',
        'pt': u':gesto_de_te_amo_pele_morena_clara:',
        'it': u':gesto_ti_amo_carnagione_abbastanza_chiara:'
    },
    u'\U0001F91F\U0001F3FD': {
        'en' : u':love-you_gesture_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':ich-liebe-dich-geste_mittlere_hautfarbe:',
        'es': u':gesto_de_te_quiero_tono_de_piel_medio:',
        'fr': u':signe_je_t’aime_peau_légèrement_mate:',
        'pt': u':gesto_de_te_amo_pele_morena:',
        'it': u':gesto_ti_amo_carnagione_olivastra:'
    },
    u'\U0001F3E9': {
        'en' : u':love_hotel:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':stundenhotel:',
        'es': u':hotel_del_amor:',
        'fr': u':love_hotel:',
        'pt': u':motel:',
        'it': u':love_hotel:'
    },
    u'\U0001F48C': {
        'en' : u':love_letter:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':liebesbrief:',
        'es': u':carta_de_amor:',
        'fr': u':lettre_d’amour:',
        'pt': u':carta_de_amor:',
        'it': u':lettera_d’amore:'
    },
    u'\U0001FAAB': {
        'en' : u':low_battery:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001F9F3': {
        'en' : u':luggage:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':gepäck:',
        'es': u':equipaje:',
        'fr': u':bagage:',
        'pt': u':mala:',
        'it': u':valigia:'
    },
    u'\U0001FAC1': {
        'en' : u':lungs:',
        'status' : fully_qualified,
        'E' : 13,
        'de': u':lungen:',
        'es': u':pulmones:',
        'fr': u':poumons:',
        'pt': u':pulmões:',
        'it': u':polmoni:'
    },
    u'\U0001F925': {
        'en' : u':lying_face:',
        'status' : fully_qualified,
        'E' : 3,
        'de': u':lügendes_gesicht:',
        'es': u':cara_de_mentiroso:',
        'fr': u':visage_de_menteur:',
        'pt': u':rosto_de_mentiroso:',
        'it': u':faccina_bugiarda:'
    },
    u'\U0001F9D9': {
        'en' : u':mage:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':magier(in):',
        'es': u':persona_maga:',
        'fr': u':mage:',
        'pt': u':mago:',
        'it': u':mago:'
    },
    u'\U0001F9D9\U0001F3FF': {
        'en' : u':mage_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':magier(in)_dunkle_hautfarbe:',
        'es': u':persona_maga_tono_de_piel_oscuro:',
        'fr': u':mage_peau_foncée:',
        'pt': u':mago_pele_escura:',
        'it': u':mago_carnagione_scura:'
    },
    u'\U0001F9D9\U0001F3FB': {
        'en' : u':mage_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':magier(in)_helle_hautfarbe:',
        'es': u':persona_maga_tono_de_piel_claro:',
        'fr': u':mage_peau_claire:',
        'pt': u':mago_pele_clara:',
        'it': u':mago_carnagione_chiara:'
    },
    u'\U0001F9D9\U0001F3FE': {
        'en' : u':mage_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':magier(in)_mitteldunkle_hautfarbe:',
        'es': u':persona_maga_tono_de_piel_oscuro_medio:',
        'fr': u':mage_peau_mate:',
        'pt': u':mago_pele_morena_escura:',
        'it': u':mago_carnagione_abbastanza_scura:'
    },
    u'\U0001F9D9\U0001F3FC': {
        'en' : u':mage_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':magier(in)_mittelhelle_hautfarbe:',
        'es': u':persona_maga_tono_de_piel_claro_medio:',
        'fr': u':mage_peau_moyennement_claire:',
        'pt': u':mago_pele_morena_clara:',
        'it': u':mago_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9D9\U0001F3FD': {
        'en' : u':mage_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':magier(in)_mittlere_hautfarbe:',
        'es': u':persona_maga_tono_de_piel_medio:',
        'fr': u':mage_peau_légèrement_mate:',
        'pt': u':mago_pele_morena:',
        'it': u':mago_carnagione_olivastra:'
    },
    u'\U0001FA84': {
        'en' : u':magic_wand:',
        'status' : fully_qualified,
        'E' : 13,
        'de': u':zauberstab:',
        'es': u':varita_mágica:',
        'fr': u':baguette_magique:',
        'pt': u':varinha_mágica:',
        'it': u':bacchetta_magica:'
    },
    u'\U0001F9F2': {
        'en' : u':magnet:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':magnet:',
        'es': u':imán:',
        'fr': u':aimant:',
        'pt': u':ímã:',
        'it': u':calamita:'
    },
    u'\U0001F50D': {
        'en' : u':magnifying_glass_tilted_left:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':mag:'],
        'variant': True,
        'de': u':lupe_nach_links:',
        'es': u':lupa_orientada_hacia_la_izquierda:',
        'fr': u':loupe_orientée_à_gauche:',
        'pt': u':lupa_para_a_esquerda:',
        'it': u':lente_di_ingrandimento_rivolta_a_sinistra:'
    },
    u'\U0001F50E': {
        'en' : u':magnifying_glass_tilted_right:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':mag_right:'],
        'de': u':lupe_nach_rechts:',
        'es': u':lupa_orientada_hacia_la_derecha:',
        'fr': u':loupe_orientée_à_droite:',
        'pt': u':lupa_para_a_direita:',
        'it': u':lente_di_ingrandimento_rivolta_a_destra:'
    },
    u'\U0001F004': {
        'en' : u':mahjong_red_dragon:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':mahjong:'],
        'variant': True,
        'de': u':mahjong-stein:',
        'es': u':dragón_rojo_de_mahjong:',
        'fr': u':dragon_rouge_mahjong:',
        'pt': u':dragão_vermelho_de_mahjong:',
        'it': u':tessera_mahjong:'
    },
    u'\U00002642\U0000FE0F': {
        'en' : u':male_sign:',
        'status' : fully_qualified,
        'E' : 4,
        'variant': True,
        'de': u':männersymbol:',
        'es': u':signo_masculino:',
        'fr': u':symbole_de_l’homme:',
        'pt': u':símbolo_de_masculino:',
        'it': u':simbolo_genere_maschile:'
    },
    u'\U00002642': {
        'en' : u':male_sign:',
        'status' : unqualified,
        'E' : 4,
        'variant': True,
        'de': u':männersymbol:',
        'es': u':signo_masculino:',
        'fr': u':symbole_de_l’homme:',
        'pt': u':símbolo_de_masculino:',
        'it': u':simbolo_genere_maschile:'
    },
    u'\U0001F9A3': {
        'en' : u':mammoth:',
        'status' : fully_qualified,
        'E' : 13,
        'de': u':mammut:',
        'es': u':mamut:',
        'fr': u':mammouth:',
        'pt': u':mamute:',
        'it': u':mammut:'
    },
    u'\U0001F468': {
        'en' : u':man:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':mann:',
        'es': u':hombre:',
        'fr': u':homme:',
        'pt': u':homem:',
        'it': u':uomo:'
    },
    u'\U0001F468\U0000200D\U0001F3A8': {
        'en' : u':man_artist:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':künstler:',
        'es': u':artista_hombre:',
        'fr': u':artiste_homme:',
        'pt': u':artista_plástico:',
        'it': u':artista_uomo:'
    },
    u'\U0001F468\U0001F3FF\U0000200D\U0001F3A8': {
        'en' : u':man_artist_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':künstler_dunkle_hautfarbe:',
        'es': u':artista_hombre_tono_de_piel_oscuro:',
        'fr': u':artiste_homme_peau_foncée:',
        'pt': u':artista_plástico_pele_escura:',
        'it': u':artista_uomo_carnagione_scura:'
    },
    u'\U0001F468\U0001F3FB\U0000200D\U0001F3A8': {
        'en' : u':man_artist_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':künstler_helle_hautfarbe:',
        'es': u':artista_hombre_tono_de_piel_claro:',
        'fr': u':artiste_homme_peau_claire:',
        'pt': u':artista_plástico_pele_clara:',
        'it': u':artista_uomo_carnagione_chiara:'
    },
    u'\U0001F468\U0001F3FE\U0000200D\U0001F3A8': {
        'en' : u':man_artist_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':künstler_mitteldunkle_hautfarbe:',
        'es': u':artista_hombre_tono_de_piel_oscuro_medio:',
        'fr': u':artiste_homme_peau_mate:',
        'pt': u':artista_plástico_pele_morena_escura:',
        'it': u':artista_uomo_carnagione_abbastanza_scura:'
    },
    u'\U0001F468\U0001F3FC\U0000200D\U0001F3A8': {
        'en' : u':man_artist_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':künstler_mittelhelle_hautfarbe:',
        'es': u':artista_hombre_tono_de_piel_claro_medio:',
        'fr': u':artiste_homme_peau_moyennement_claire:',
        'pt': u':artista_plástico_pele_morena_clara:',
        'it': u':artista_uomo_carnagione_abbastanza_chiara:'
    },
    u'\U0001F468\U0001F3FD\U0000200D\U0001F3A8': {
        'en' : u':man_artist_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':künstler_mittlere_hautfarbe:',
        'es': u':artista_hombre_tono_de_piel_medio:',
        'fr': u':artiste_homme_peau_légèrement_mate:',
        'pt': u':artista_plástico_pele_morena:',
        'it': u':artista_uomo_carnagione_olivastra:'
    },
    u'\U0001F468\U0000200D\U0001F680': {
        'en' : u':man_astronaut:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':astronaut:',
        'es': u':astronauta_hombre:',
        'fr': u':astronaute_homme:',
        'pt': u':astronauta_homem:',
        'it': u':astronauta_uomo:'
    },
    u'\U0001F468\U0001F3FF\U0000200D\U0001F680': {
        'en' : u':man_astronaut_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':astronaut_dunkle_hautfarbe:',
        'es': u':astronauta_hombre_tono_de_piel_oscuro:',
        'fr': u':astronaute_homme_peau_foncée:',
        'pt': u':astronauta_homem_pele_escura:',
        'it': u':astronauta_uomo_carnagione_scura:'
    },
    u'\U0001F468\U0001F3FB\U0000200D\U0001F680': {
        'en' : u':man_astronaut_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':astronaut_helle_hautfarbe:',
        'es': u':astronauta_hombre_tono_de_piel_claro:',
        'fr': u':astronaute_homme_peau_claire:',
        'pt': u':astronauta_homem_pele_clara:',
        'it': u':astronauta_uomo_carnagione_chiara:'
    },
    u'\U0001F468\U0001F3FE\U0000200D\U0001F680': {
        'en' : u':man_astronaut_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':astronaut_mitteldunkle_hautfarbe:',
        'es': u':astronauta_hombre_tono_de_piel_oscuro_medio:',
        'fr': u':astronaute_homme_peau_mate:',
        'pt': u':astronauta_homem_pele_morena_escura:',
        'it': u':astronauta_uomo_carnagione_abbastanza_scura:'
    },
    u'\U0001F468\U0001F3FC\U0000200D\U0001F680': {
        'en' : u':man_astronaut_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':astronaut_mittelhelle_hautfarbe:',
        'es': u':astronauta_hombre_tono_de_piel_claro_medio:',
        'fr': u':astronaute_homme_peau_moyennement_claire:',
        'pt': u':astronauta_homem_pele_morena_clara:',
        'it': u':astronauta_uomo_carnagione_abbastanza_chiara:'
    },
    u'\U0001F468\U0001F3FD\U0000200D\U0001F680': {
        'en' : u':man_astronaut_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':astronaut_mittlere_hautfarbe:',
        'es': u':astronauta_hombre_tono_de_piel_medio:',
        'fr': u':astronaute_homme_peau_légèrement_mate:',
        'pt': u':astronauta_homem_pele_morena:',
        'it': u':astronauta_uomo_carnagione_olivastra:'
    },
    u'\U0001F468\U0000200D\U0001F9B2': {
        'en' : u':man_bald:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':mann_glatze:',
        'es': u':hombre_sin_pelo:',
        'fr': u':homme_chauve:',
        'pt': u':homem_careca:',
        'it': u':uomo_calvo:'
    },
    u'\U0001F9D4\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_beard:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D4\U0000200D\U00002642': {
        'en' : u':man_beard:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F6B4\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_biking:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':radfahrer:',
        'es': u':hombre_en_bicicleta:',
        'fr': u':cycliste_homme:',
        'pt': u':homem_ciclista:',
        'it': u':ciclista_uomo:'
    },
    u'\U0001F6B4\U0000200D\U00002642': {
        'en' : u':man_biking:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F6B4\U0001F3FF\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_biking_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':radfahrer_dunkle_hautfarbe:',
        'es': u':hombre_en_bicicleta_tono_de_piel_oscuro:',
        'fr': u':cycliste_homme_peau_foncée:',
        'pt': u':homem_ciclista_pele_escura:',
        'it': u':ciclista_uomo_carnagione_scura:'
    },
    u'\U0001F6B4\U0001F3FF\U0000200D\U00002642': {
        'en' : u':man_biking_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F6B4\U0001F3FB\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_biking_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':radfahrer_helle_hautfarbe:',
        'es': u':hombre_en_bicicleta_tono_de_piel_claro:',
        'fr': u':cycliste_homme_peau_claire:',
        'pt': u':homem_ciclista_pele_clara:',
        'it': u':ciclista_uomo_carnagione_chiara:'
    },
    u'\U0001F6B4\U0001F3FB\U0000200D\U00002642': {
        'en' : u':man_biking_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F6B4\U0001F3FE\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_biking_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':radfahrer_mitteldunkle_hautfarbe:',
        'es': u':hombre_en_bicicleta_tono_de_piel_oscuro_medio:',
        'fr': u':cycliste_homme_peau_mate:',
        'pt': u':homem_ciclista_pele_morena_escura:',
        'it': u':ciclista_uomo_carnagione_abbastanza_scura:'
    },
    u'\U0001F6B4\U0001F3FE\U0000200D\U00002642': {
        'en' : u':man_biking_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F6B4\U0001F3FC\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_biking_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':radfahrer_mittelhelle_hautfarbe:',
        'es': u':hombre_en_bicicleta_tono_de_piel_claro_medio:',
        'fr': u':cycliste_homme_peau_moyennement_claire:',
        'pt': u':homem_ciclista_pele_morena_clara:',
        'it': u':ciclista_uomo_carnagione_abbastanza_chiara:'
    },
    u'\U0001F6B4\U0001F3FC\U0000200D\U00002642': {
        'en' : u':man_biking_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F6B4\U0001F3FD\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_biking_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':radfahrer_mittlere_hautfarbe:',
        'es': u':hombre_en_bicicleta_tono_de_piel_medio:',
        'fr': u':cycliste_homme_peau_légèrement_mate:',
        'pt': u':homem_ciclista_pele_morena:',
        'it': u':ciclista_uomo_carnagione_olivastra:'
    },
    u'\U0001F6B4\U0001F3FD\U0000200D\U00002642': {
        'en' : u':man_biking_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F471\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_blond_hair:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':mann_blond:',
        'es': u':hombre_rubio:',
        'fr': u':homme_blond:',
        'pt': u':homem_cabelo_loiro:',
        'it': u':uomo_biondo:'
    },
    u'\U0001F471\U0000200D\U00002642': {
        'en' : u':man_blond_hair:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U000026F9\U0000FE0F\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_bouncing_ball:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':mann_mit_ball:',
        'es': u':hombre_botando_un_balón:',
        'fr': u':homme_avec_ballon:',
        'pt': u':homem_jogando_basquete:',
        'it': u':uomo_che_fa_rimbalzare_una_palla:'
    },
    u'\U000026F9\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_bouncing_ball:',
        'status' : unqualified,
        'E' : 4
    },
    u'\U000026F9\U0000FE0F\U0000200D\U00002642': {
        'en' : u':man_bouncing_ball:',
        'status' : unqualified,
        'E' : 4
    },
    u'\U000026F9\U0000200D\U00002642': {
        'en' : u':man_bouncing_ball:',
        'status' : unqualified,
        'E' : 4
    },
    u'\U000026F9\U0001F3FF\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_bouncing_ball_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':mann_mit_ball_dunkle_hautfarbe:',
        'es': u':hombre_botando_un_balón_tono_de_piel_oscuro:',
        'fr': u':homme_avec_ballon_peau_foncée:',
        'pt': u':homem_jogando_basquete_pele_escura:',
        'it': u':uomo_che_fa_rimbalzare_una_palla_carnagione_scura:'
    },
    u'\U000026F9\U0001F3FF\U0000200D\U00002642': {
        'en' : u':man_bouncing_ball_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U000026F9\U0001F3FB\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_bouncing_ball_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':mann_mit_ball_helle_hautfarbe:',
        'es': u':hombre_botando_un_balón_tono_de_piel_claro:',
        'fr': u':homme_avec_ballon_peau_claire:',
        'pt': u':homem_jogando_basquete_pele_clara:',
        'it': u':uomo_che_fa_rimbalzare_una_palla_carnagione_chiara:'
    },
    u'\U000026F9\U0001F3FB\U0000200D\U00002642': {
        'en' : u':man_bouncing_ball_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U000026F9\U0001F3FE\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_bouncing_ball_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':mann_mit_ball_mitteldunkle_hautfarbe:',
        'es': u':hombre_botando_un_balón_tono_de_piel_oscuro_medio:',
        'fr': u':homme_avec_ballon_peau_mate:',
        'pt': u':homem_jogando_basquete_pele_morena_escura:',
        'it': u':uomo_che_fa_rimbalzare_una_palla_carnagione_abbastanza_scura:'
    },
    u'\U000026F9\U0001F3FE\U0000200D\U00002642': {
        'en' : u':man_bouncing_ball_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U000026F9\U0001F3FC\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_bouncing_ball_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':mann_mit_ball_mittelhelle_hautfarbe:',
        'es': u':hombre_botando_un_balón_tono_de_piel_claro_medio:',
        'fr': u':homme_avec_ballon_peau_moyennement_claire:',
        'pt': u':homem_jogando_basquete_pele_morena_clara:',
        'it': u':uomo_che_fa_rimbalzare_una_palla_carnagione_abbastanza_chiara:'
    },
    u'\U000026F9\U0001F3FC\U0000200D\U00002642': {
        'en' : u':man_bouncing_ball_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U000026F9\U0001F3FD\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_bouncing_ball_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':mann_mit_ball_mittlere_hautfarbe:',
        'es': u':hombre_botando_un_balón_tono_de_piel_medio:',
        'fr': u':homme_avec_ballon_peau_légèrement_mate:',
        'pt': u':homem_jogando_basquete_pele_morena:',
        'it': u':uomo_che_fa_rimbalzare_una_palla_carnagione_olivastra:'
    },
    u'\U000026F9\U0001F3FD\U0000200D\U00002642': {
        'en' : u':man_bouncing_ball_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F647\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_bowing:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':sich_verbeugender_mann:',
        'es': u':hombre_haciendo_una_reverencia:',
        'fr': u':homme_qui_s’incline:',
        'pt': u':homem_fazendo_reverência:',
        'it': u':uomo_che_fa_inchino_profondo:'
    },
    u'\U0001F647\U0000200D\U00002642': {
        'en' : u':man_bowing:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F647\U0001F3FF\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_bowing_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':sich_verbeugender_mann_dunkle_hautfarbe:',
        'es': u':hombre_haciendo_una_reverencia_tono_de_piel_oscuro:',
        'fr': u':homme_qui_s’incline_peau_foncée:',
        'pt': u':homem_fazendo_reverência_pele_escura:',
        'it': u':uomo_che_fa_inchino_profondo_carnagione_scura:'
    },
    u'\U0001F647\U0001F3FF\U0000200D\U00002642': {
        'en' : u':man_bowing_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F647\U0001F3FB\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_bowing_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':sich_verbeugender_mann_helle_hautfarbe:',
        'es': u':hombre_haciendo_una_reverencia_tono_de_piel_claro:',
        'fr': u':homme_qui_s’incline_peau_claire:',
        'pt': u':homem_fazendo_reverência_pele_clara:',
        'it': u':uomo_che_fa_inchino_profondo_carnagione_chiara:'
    },
    u'\U0001F647\U0001F3FB\U0000200D\U00002642': {
        'en' : u':man_bowing_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F647\U0001F3FE\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_bowing_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':sich_verbeugender_mann_mitteldunkle_hautfarbe:',
        'es': u':hombre_haciendo_una_reverencia_tono_de_piel_oscuro_medio:',
        'fr': u':homme_qui_s’incline_peau_mate:',
        'pt': u':homem_fazendo_reverência_pele_morena_escura:',
        'it': u':uomo_che_fa_inchino_profondo_carnagione_abbastanza_scura:'
    },
    u'\U0001F647\U0001F3FE\U0000200D\U00002642': {
        'en' : u':man_bowing_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F647\U0001F3FC\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_bowing_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':sich_verbeugender_mann_mittelhelle_hautfarbe:',
        'es': u':hombre_haciendo_una_reverencia_tono_de_piel_claro_medio:',
        'fr': u':homme_qui_s’incline_peau_moyennement_claire:',
        'pt': u':homem_fazendo_reverência_pele_morena_clara:',
        'it': u':uomo_che_fa_inchino_profondo_carnagione_abbastanza_chiara:'
    },
    u'\U0001F647\U0001F3FC\U0000200D\U00002642': {
        'en' : u':man_bowing_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F647\U0001F3FD\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_bowing_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':sich_verbeugender_mann_mittlere_hautfarbe:',
        'es': u':hombre_haciendo_una_reverencia_tono_de_piel_medio:',
        'fr': u':homme_qui_s’incline_peau_légèrement_mate:',
        'pt': u':homem_fazendo_reverência_pele_morena:',
        'it': u':uomo_che_fa_inchino_profondo_carnagione_olivastra:'
    },
    u'\U0001F647\U0001F3FD\U0000200D\U00002642': {
        'en' : u':man_bowing_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F938\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_cartwheeling:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':rad_schlagender_mann:',
        'es': u':hombre_haciendo_una_voltereta_lateral:',
        'fr': u':homme_faisant_la_roue:',
        'pt': u':homem_fazendo_estrela:',
        'it': u':uomo_che_fa_la_ruota:'
    },
    u'\U0001F938\U0000200D\U00002642': {
        'en' : u':man_cartwheeling:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F938\U0001F3FF\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_cartwheeling_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':rad_schlagender_mann_dunkle_hautfarbe:',
        'es': u':hombre_haciendo_una_voltereta_lateral_tono_de_piel_oscuro:',
        'fr': u':homme_faisant_la_roue_peau_foncée:',
        'pt': u':homem_fazendo_estrela_pele_escura:',
        'it': u':uomo_che_fa_la_ruota_carnagione_scura:'
    },
    u'\U0001F938\U0001F3FF\U0000200D\U00002642': {
        'en' : u':man_cartwheeling_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F938\U0001F3FB\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_cartwheeling_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':rad_schlagender_mann_helle_hautfarbe:',
        'es': u':hombre_haciendo_una_voltereta_lateral_tono_de_piel_claro:',
        'fr': u':homme_faisant_la_roue_peau_claire:',
        'pt': u':homem_fazendo_estrela_pele_clara:',
        'it': u':uomo_che_fa_la_ruota_carnagione_chiara:'
    },
    u'\U0001F938\U0001F3FB\U0000200D\U00002642': {
        'en' : u':man_cartwheeling_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F938\U0001F3FE\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_cartwheeling_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':rad_schlagender_mann_mitteldunkle_hautfarbe:',
        'es': u':hombre_haciendo_una_voltereta_lateral_tono_de_piel_oscuro_medio:',
        'fr': u':homme_faisant_la_roue_peau_mate:',
        'pt': u':homem_fazendo_estrela_pele_morena_escura:',
        'it': u':uomo_che_fa_la_ruota_carnagione_abbastanza_scura:'
    },
    u'\U0001F938\U0001F3FE\U0000200D\U00002642': {
        'en' : u':man_cartwheeling_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F938\U0001F3FC\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_cartwheeling_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':rad_schlagender_mann_mittelhelle_hautfarbe:',
        'es': u':hombre_haciendo_una_voltereta_lateral_tono_de_piel_claro_medio:',
        'fr': u':homme_faisant_la_roue_peau_moyennement_claire:',
        'pt': u':homem_fazendo_estrela_pele_morena_clara:',
        'it': u':uomo_che_fa_la_ruota_carnagione_abbastanza_chiara:'
    },
    u'\U0001F938\U0001F3FC\U0000200D\U00002642': {
        'en' : u':man_cartwheeling_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F938\U0001F3FD\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_cartwheeling_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':rad_schlagender_mann_mittlere_hautfarbe:',
        'es': u':hombre_haciendo_una_voltereta_lateral_tono_de_piel_medio:',
        'fr': u':homme_faisant_la_roue_peau_légèrement_mate:',
        'pt': u':homem_fazendo_estrela_pele_morena:',
        'it': u':uomo_che_fa_la_ruota_carnagione_olivastra:'
    },
    u'\U0001F938\U0001F3FD\U0000200D\U00002642': {
        'en' : u':man_cartwheeling_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F9D7\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_climbing:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':bergsteiger:',
        'es': u':hombre_escalando:',
        'fr': u':homme_qui_grimpe:',
        'pt': u':homem_escalando:',
        'it': u':scalatore:'
    },
    u'\U0001F9D7\U0000200D\U00002642': {
        'en' : u':man_climbing:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F9D7\U0001F3FF\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_climbing_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':bergsteiger_dunkle_hautfarbe:',
        'es': u':hombre_escalando_tono_de_piel_oscuro:',
        'fr': u':homme_qui_grimpe_peau_foncée:',
        'pt': u':homem_escalando_pele_escura:',
        'it': u':scalatore_carnagione_scura:'
    },
    u'\U0001F9D7\U0001F3FF\U0000200D\U00002642': {
        'en' : u':man_climbing_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F9D7\U0001F3FB\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_climbing_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':bergsteiger_helle_hautfarbe:',
        'es': u':hombre_escalando_tono_de_piel_claro:',
        'fr': u':homme_qui_grimpe_peau_claire:',
        'pt': u':homem_escalando_pele_clara:',
        'it': u':scalatore_carnagione_chiara:'
    },
    u'\U0001F9D7\U0001F3FB\U0000200D\U00002642': {
        'en' : u':man_climbing_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F9D7\U0001F3FE\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_climbing_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':bergsteiger_mitteldunkle_hautfarbe:',
        'es': u':hombre_escalando_tono_de_piel_oscuro_medio:',
        'fr': u':homme_qui_grimpe_peau_mate:',
        'pt': u':homem_escalando_pele_morena_escura:',
        'it': u':scalatore_carnagione_abbastanza_scura:'
    },
    u'\U0001F9D7\U0001F3FE\U0000200D\U00002642': {
        'en' : u':man_climbing_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F9D7\U0001F3FC\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_climbing_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':bergsteiger_mittelhelle_hautfarbe:',
        'es': u':hombre_escalando_tono_de_piel_claro_medio:',
        'fr': u':homme_qui_grimpe_peau_moyennement_claire:',
        'pt': u':homem_escalando_pele_morena_clara:',
        'it': u':scalatore_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9D7\U0001F3FC\U0000200D\U00002642': {
        'en' : u':man_climbing_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F9D7\U0001F3FD\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_climbing_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':bergsteiger_mittlere_hautfarbe:',
        'es': u':hombre_escalando_tono_de_piel_medio:',
        'fr': u':homme_qui_grimpe_peau_légèrement_mate:',
        'pt': u':homem_escalando_pele_morena:',
        'it': u':scalatore_carnagione_olivastra:'
    },
    u'\U0001F9D7\U0001F3FD\U0000200D\U00002642': {
        'en' : u':man_climbing_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F477\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_construction_worker:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':bauarbeiter:',
        'es': u':profesional_de_la_construcción_hombre:',
        'fr': u':ouvrier_du_bâtiment:',
        'pt': u':pedreiro:',
        'it': u':operaio_edile_uomo:'
    },
    u'\U0001F477\U0000200D\U00002642': {
        'en' : u':man_construction_worker:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F477\U0001F3FF\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_construction_worker_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':bauarbeiter_dunkle_hautfarbe:',
        'es': u':profesional_de_la_construcción_hombre_tono_de_piel_oscuro:',
        'fr': u':ouvrier_du_bâtiment_peau_foncée:',
        'pt': u':pedreiro_pele_escura:',
        'it': u':operaio_edile_uomo_carnagione_scura:'
    },
    u'\U0001F477\U0001F3FF\U0000200D\U00002642': {
        'en' : u':man_construction_worker_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F477\U0001F3FB\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_construction_worker_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':bauarbeiter_helle_hautfarbe:',
        'es': u':profesional_de_la_construcción_hombre_tono_de_piel_claro:',
        'fr': u':ouvrier_du_bâtiment_peau_claire:',
        'pt': u':pedreiro_pele_clara:',
        'it': u':operaio_edile_uomo_carnagione_chiara:'
    },
    u'\U0001F477\U0001F3FB\U0000200D\U00002642': {
        'en' : u':man_construction_worker_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F477\U0001F3FE\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_construction_worker_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':bauarbeiter_mitteldunkle_hautfarbe:',
        'es': u':profesional_de_la_construcción_hombre_tono_de_piel_oscuro_medio:',
        'fr': u':ouvrier_du_bâtiment_peau_mate:',
        'pt': u':pedreiro_pele_morena_escura:',
        'it': u':operaio_edile_uomo_carnagione_abbastanza_scura:'
    },
    u'\U0001F477\U0001F3FE\U0000200D\U00002642': {
        'en' : u':man_construction_worker_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F477\U0001F3FC\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_construction_worker_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':bauarbeiter_mittelhelle_hautfarbe:',
        'es': u':profesional_de_la_construcción_hombre_tono_de_piel_claro_medio:',
        'fr': u':ouvrier_du_bâtiment_peau_moyennement_claire:',
        'pt': u':pedreiro_pele_morena_clara:',
        'it': u':operaio_edile_uomo_carnagione_abbastanza_chiara:'
    },
    u'\U0001F477\U0001F3FC\U0000200D\U00002642': {
        'en' : u':man_construction_worker_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F477\U0001F3FD\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_construction_worker_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':bauarbeiter_mittlere_hautfarbe:',
        'es': u':profesional_de_la_construcción_hombre_tono_de_piel_medio:',
        'fr': u':ouvrier_du_bâtiment_peau_légèrement_mate:',
        'pt': u':pedreiro_pele_morena:',
        'it': u':operaio_edile_uomo_carnagione_olivastra:'
    },
    u'\U0001F477\U0001F3FD\U0000200D\U00002642': {
        'en' : u':man_construction_worker_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F468\U0000200D\U0001F373': {
        'en' : u':man_cook:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':koch:',
        'es': u':chef_hombre:',
        'fr': u':cuisinier:',
        'pt': u':cozinheiro:',
        'it': u':cuoco:'
    },
    u'\U0001F468\U0001F3FF\U0000200D\U0001F373': {
        'en' : u':man_cook_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':koch_dunkle_hautfarbe:',
        'es': u':chef_hombre_tono_de_piel_oscuro:',
        'fr': u':cuisinier_peau_foncée:',
        'pt': u':cozinheiro_pele_escura:',
        'it': u':cuoco_carnagione_scura:'
    },
    u'\U0001F468\U0001F3FB\U0000200D\U0001F373': {
        'en' : u':man_cook_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':koch_helle_hautfarbe:',
        'es': u':chef_hombre_tono_de_piel_claro:',
        'fr': u':cuisinier_peau_claire:',
        'pt': u':cozinheiro_pele_clara:',
        'it': u':cuoco_carnagione_chiara:'
    },
    u'\U0001F468\U0001F3FE\U0000200D\U0001F373': {
        'en' : u':man_cook_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':koch_mitteldunkle_hautfarbe:',
        'es': u':chef_hombre_tono_de_piel_oscuro_medio:',
        'fr': u':cuisinier_peau_mate:',
        'pt': u':cozinheiro_pele_morena_escura:',
        'it': u':cuoco_carnagione_abbastanza_scura:'
    },
    u'\U0001F468\U0001F3FC\U0000200D\U0001F373': {
        'en' : u':man_cook_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':koch_mittelhelle_hautfarbe:',
        'es': u':chef_hombre_tono_de_piel_claro_medio:',
        'fr': u':cuisinier_peau_moyennement_claire:',
        'pt': u':cozinheiro_pele_morena_clara:',
        'it': u':cuoco_carnagione_abbastanza_chiara:'
    },
    u'\U0001F468\U0001F3FD\U0000200D\U0001F373': {
        'en' : u':man_cook_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':koch_mittlere_hautfarbe:',
        'es': u':chef_hombre_tono_de_piel_medio:',
        'fr': u':cuisinier_peau_légèrement_mate:',
        'pt': u':cozinheiro_pele_morena:',
        'it': u':cuoco_carnagione_olivastra:'
    },
    u'\U0001F468\U0000200D\U0001F9B1': {
        'en' : u':man_curly_hair:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':mann_lockiges_haar:',
        'es': u':hombre_pelo_rizado:',
        'fr': u':homme_cheveux_bouclés:',
        'pt': u':homem_cabelo_cacheado:',
        'it': u':uomo_capelli_ricci:'
    },
    u'\U0001F57A': {
        'en' : u':man_dancing:',
        'status' : fully_qualified,
        'E' : 3,
        'de': u':tanzender_mann:',
        'es': u':hombre_bailando:',
        'fr': u':danseur:',
        'pt': u':homem_dançando:',
        'it': u':uomo_che_balla:'
    },
    u'\U0001F57A\U0001F3FF': {
        'en' : u':man_dancing_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 3,
        'de': u':tanzender_mann_dunkle_hautfarbe:',
        'es': u':hombre_bailando_tono_de_piel_oscuro:',
        'fr': u':danseur_peau_foncée:',
        'pt': u':homem_dançando_pele_escura:',
        'it': u':uomo_che_balla_carnagione_scura:'
    },
    u'\U0001F57A\U0001F3FB': {
        'en' : u':man_dancing_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 3,
        'de': u':tanzender_mann_helle_hautfarbe:',
        'es': u':hombre_bailando_tono_de_piel_claro:',
        'fr': u':danseur_peau_claire:',
        'pt': u':homem_dançando_pele_clara:',
        'it': u':uomo_che_balla_carnagione_chiara:'
    },
    u'\U0001F57A\U0001F3FE': {
        'en' : u':man_dancing_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 3,
        'de': u':tanzender_mann_mitteldunkle_hautfarbe:',
        'es': u':hombre_bailando_tono_de_piel_oscuro_medio:',
        'fr': u':danseur_peau_mate:',
        'pt': u':homem_dançando_pele_morena_escura:',
        'it': u':uomo_che_balla_carnagione_abbastanza_scura:'
    },
    u'\U0001F57A\U0001F3FC': {
        'en' : u':man_dancing_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 3,
        'de': u':tanzender_mann_mittelhelle_hautfarbe:',
        'es': u':hombre_bailando_tono_de_piel_claro_medio:',
        'fr': u':danseur_peau_moyennement_claire:',
        'pt': u':homem_dançando_pele_morena_clara:',
        'it': u':uomo_che_balla_carnagione_abbastanza_chiara:'
    },
    u'\U0001F57A\U0001F3FD': {
        'en' : u':man_dancing_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 3,
        'de': u':tanzender_mann_mittlere_hautfarbe:',
        'es': u':hombre_bailando_tono_de_piel_medio:',
        'fr': u':danseur_peau_légèrement_mate:',
        'pt': u':homem_dançando_pele_morena:',
        'it': u':uomo_che_balla_carnagione_olivastra:'
    },
    u'\U0001F468\U0001F3FF': {
        'en' : u':man_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':mann_dunkle_hautfarbe:',
        'es': u':hombre_tono_de_piel_oscuro:',
        'fr': u':homme_peau_foncée:',
        'pt': u':homem_pele_escura:',
        'it': u':uomo_carnagione_scura:'
    },
    u'\U0001F468\U0001F3FF\U0000200D\U0001F9B2': {
        'en' : u':man_dark_skin_tone_bald:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':mann_dunkle_hautfarbe,_glatze:',
        'es': u':hombre_tono_de_piel_oscuro_y_sin_pelo:',
        'fr': u':homme_peau_foncée_et_chauve:',
        'pt': u':homem_pele_escura_e_careca:',
        'it': u':uomo_carnagione_scura_e_calvo:'
    },
    u'\U0001F9D4\U0001F3FF\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_dark_skin_tone_beard:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D4\U0001F3FF\U0000200D\U00002642': {
        'en' : u':man_dark_skin_tone_beard:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F471\U0001F3FF\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_dark_skin_tone_blond_hair:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':mann_dunkle_hautfarbe,_blond:',
        'es': u':hombre_rubio_tono_de_piel_oscuro:',
        'fr': u':homme_blond_peau_foncée:',
        'pt': u':homem_pele_escura_e_cabelo_loiro:',
        'it': u':uomo_biondo_carnagione_scura:'
    },
    u'\U0001F471\U0001F3FF\U0000200D\U00002642': {
        'en' : u':man_dark_skin_tone_blond_hair:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F468\U0001F3FF\U0000200D\U0001F9B1': {
        'en' : u':man_dark_skin_tone_curly_hair:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':mann_dunkle_hautfarbe,_lockiges_haar:',
        'es': u':hombre_tono_de_piel_oscuro_y_pelo_rizado:',
        'fr': u':homme_peau_foncée_et_cheveux_bouclés:',
        'pt': u':homem_pele_escura_e_cabelo_cacheado:',
        'it': u':uomo_carnagione_scura_e_capelli_ricci:'
    },
    u'\U0001F468\U0001F3FF\U0000200D\U0001F9B0': {
        'en' : u':man_dark_skin_tone_red_hair:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':mann_dunkle_hautfarbe,_rotes_haar:',
        'es': u':hombre_tono_de_piel_oscuro_y_pelo_pelirrojo:',
        'fr': u':homme_peau_foncée_et_cheveux_roux:',
        'pt': u':homem_pele_escura_e_cabelo_vermelho:',
        'it': u':uomo_carnagione_scura_e_capelli_rossi:'
    },
    u'\U0001F468\U0001F3FF\U0000200D\U0001F9B3': {
        'en' : u':man_dark_skin_tone_white_hair:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':mann_dunkle_hautfarbe,_weißes_haar:',
        'es': u':hombre_tono_de_piel_oscuro_y_pelo_blanco:',
        'fr': u':homme_peau_foncée_et_cheveux_blancs:',
        'pt': u':homem_pele_escura_e_cabelo_branco:',
        'it': u':uomo_carnagione_scura_e_capelli_bianchi:'
    },
    u'\U0001F575\U0000FE0F\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_detective:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':detektiv:',
        'es': u':detective_hombre:',
        'fr': u':détective_homme:',
        'pt': u':detetive_homem:',
        'it': u':investigatore:'
    },
    u'\U0001F575\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_detective:',
        'status' : unqualified,
        'E' : 4
    },
    u'\U0001F575\U0000FE0F\U0000200D\U00002642': {
        'en' : u':man_detective:',
        'status' : unqualified,
        'E' : 4
    },
    u'\U0001F575\U0000200D\U00002642': {
        'en' : u':man_detective:',
        'status' : unqualified,
        'E' : 4
    },
    u'\U0001F575\U0001F3FF\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_detective_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':detektiv_dunkle_hautfarbe:',
        'es': u':detective_hombre_tono_de_piel_oscuro:',
        'fr': u':détective_homme_peau_foncée:',
        'pt': u':detetive_homem_pele_escura:',
        'it': u':investigatore_carnagione_scura:'
    },
    u'\U0001F575\U0001F3FF\U0000200D\U00002642': {
        'en' : u':man_detective_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F575\U0001F3FB\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_detective_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':detektiv_helle_hautfarbe:',
        'es': u':detective_hombre_tono_de_piel_claro:',
        'fr': u':détective_homme_peau_claire:',
        'pt': u':detetive_homem_pele_clara:',
        'it': u':investigatore_carnagione_chiara:'
    },
    u'\U0001F575\U0001F3FB\U0000200D\U00002642': {
        'en' : u':man_detective_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F575\U0001F3FE\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_detective_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':detektiv_mitteldunkle_hautfarbe:',
        'es': u':detective_hombre_tono_de_piel_oscuro_medio:',
        'fr': u':détective_homme_peau_mate:',
        'pt': u':detetive_homem_pele_morena_escura:',
        'it': u':investigatore_carnagione_abbastanza_scura:'
    },
    u'\U0001F575\U0001F3FE\U0000200D\U00002642': {
        'en' : u':man_detective_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F575\U0001F3FC\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_detective_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':detektiv_mittelhelle_hautfarbe:',
        'es': u':detective_hombre_tono_de_piel_claro_medio:',
        'fr': u':détective_homme_peau_moyennement_claire:',
        'pt': u':detetive_homem_pele_morena_clara:',
        'it': u':investigatore_carnagione_abbastanza_chiara:'
    },
    u'\U0001F575\U0001F3FC\U0000200D\U00002642': {
        'en' : u':man_detective_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F575\U0001F3FD\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_detective_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':detektiv_mittlere_hautfarbe:',
        'es': u':detective_hombre_tono_de_piel_medio:',
        'fr': u':détective_homme_peau_légèrement_mate:',
        'pt': u':detetive_homem_pele_morena:',
        'it': u':investigatore_carnagione_olivastra:'
    },
    u'\U0001F575\U0001F3FD\U0000200D\U00002642': {
        'en' : u':man_detective_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F9DD\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_elf:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':elf:',
        'es': u':elfo_hombre:',
        'fr': u':elfe_homme:',
        'pt': u':elfo_homem:',
        'it': u':elfo_uomo:'
    },
    u'\U0001F9DD\U0000200D\U00002642': {
        'en' : u':man_elf:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F9DD\U0001F3FF\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_elf_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':elf_dunkle_hautfarbe:',
        'es': u':elfo_hombre_tono_de_piel_oscuro:',
        'fr': u':elfe_homme_peau_foncée:',
        'pt': u':elfo_homem_pele_escura:',
        'it': u':elfo_uomo_carnagione_scura:'
    },
    u'\U0001F9DD\U0001F3FF\U0000200D\U00002642': {
        'en' : u':man_elf_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F9DD\U0001F3FB\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_elf_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':elf_helle_hautfarbe:',
        'es': u':elfo_hombre_tono_de_piel_claro:',
        'fr': u':elfe_homme_peau_claire:',
        'pt': u':elfo_homem_pele_clara:',
        'it': u':elfo_uomo_carnagione_chiara:'
    },
    u'\U0001F9DD\U0001F3FB\U0000200D\U00002642': {
        'en' : u':man_elf_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F9DD\U0001F3FE\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_elf_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':elf_mitteldunkle_hautfarbe:',
        'es': u':elfo_hombre_tono_de_piel_oscuro_medio:',
        'fr': u':elfe_homme_peau_mate:',
        'pt': u':elfo_homem_pele_morena_escura:',
        'it': u':elfo_uomo_carnagione_abbastanza_scura:'
    },
    u'\U0001F9DD\U0001F3FE\U0000200D\U00002642': {
        'en' : u':man_elf_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F9DD\U0001F3FC\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_elf_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':elf_mittelhelle_hautfarbe:',
        'es': u':elfo_hombre_tono_de_piel_claro_medio:',
        'fr': u':elfe_homme_peau_moyennement_claire:',
        'pt': u':elfo_homem_pele_morena_clara:',
        'it': u':elfo_uomo_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9DD\U0001F3FC\U0000200D\U00002642': {
        'en' : u':man_elf_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F9DD\U0001F3FD\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_elf_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':elf_mittlere_hautfarbe:',
        'es': u':elfo_hombre_tono_de_piel_medio:',
        'fr': u':elfe_homme_peau_légèrement_mate:',
        'pt': u':elfo_homem_pele_morena:',
        'it': u':elfo_uomo_carnagione_olivastra:'
    },
    u'\U0001F9DD\U0001F3FD\U0000200D\U00002642': {
        'en' : u':man_elf_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F926\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_facepalming:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':sich_an_den_kopf_fassender_mann:',
        'es': u':hombre_con_la_mano_en_la_frente:',
        'fr': u':homme_avec_la_paume_sur_le_visage:',
        'pt': u':homem_decepcionado:',
        'it': u':uomo_esasperato:'
    },
    u'\U0001F926\U0000200D\U00002642': {
        'en' : u':man_facepalming:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F926\U0001F3FF\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_facepalming_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':sich_an_den_kopf_fassender_mann_dunkle_hautfarbe:',
        'es': u':hombre_con_la_mano_en_la_frente_tono_de_piel_oscuro:',
        'fr': u':homme_avec_la_paume_sur_le_visage_peau_foncée:',
        'pt': u':homem_decepcionado_pele_escura:',
        'it': u':uomo_esasperato_carnagione_scura:'
    },
    u'\U0001F926\U0001F3FF\U0000200D\U00002642': {
        'en' : u':man_facepalming_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F926\U0001F3FB\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_facepalming_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':sich_an_den_kopf_fassender_mann_helle_hautfarbe:',
        'es': u':hombre_con_la_mano_en_la_frente_tono_de_piel_claro:',
        'fr': u':homme_avec_la_paume_sur_le_visage_peau_claire:',
        'pt': u':homem_decepcionado_pele_clara:',
        'it': u':uomo_esasperato_carnagione_chiara:'
    },
    u'\U0001F926\U0001F3FB\U0000200D\U00002642': {
        'en' : u':man_facepalming_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F926\U0001F3FE\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_facepalming_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':sich_an_den_kopf_fassender_mann_mitteldunkle_hautfarbe:',
        'es': u':hombre_con_la_mano_en_la_frente_tono_de_piel_oscuro_medio:',
        'fr': u':homme_avec_la_paume_sur_le_visage_peau_mate:',
        'pt': u':homem_decepcionado_pele_morena_escura:',
        'it': u':uomo_esasperato_carnagione_abbastanza_scura:'
    },
    u'\U0001F926\U0001F3FE\U0000200D\U00002642': {
        'en' : u':man_facepalming_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F926\U0001F3FC\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_facepalming_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':sich_an_den_kopf_fassender_mann_mittelhelle_hautfarbe:',
        'es': u':hombre_con_la_mano_en_la_frente_tono_de_piel_claro_medio:',
        'fr': u':homme_avec_la_paume_sur_le_visage_peau_moyennement_claire:',
        'pt': u':homem_decepcionado_pele_morena_clara:',
        'it': u':uomo_esasperato_carnagione_abbastanza_chiara:'
    },
    u'\U0001F926\U0001F3FC\U0000200D\U00002642': {
        'en' : u':man_facepalming_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F926\U0001F3FD\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_facepalming_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':sich_an_den_kopf_fassender_mann_mittlere_hautfarbe:',
        'es': u':hombre_con_la_mano_en_la_frente_tono_de_piel_medio:',
        'fr': u':homme_avec_la_paume_sur_le_visage_peau_légèrement_mate:',
        'pt': u':homem_decepcionado_pele_morena:',
        'it': u':uomo_esasperato_carnagione_olivastra:'
    },
    u'\U0001F926\U0001F3FD\U0000200D\U00002642': {
        'en' : u':man_facepalming_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F468\U0000200D\U0001F3ED': {
        'en' : u':man_factory_worker:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':fabrikarbeiter:',
        'es': u':profesional_industrial_hombre:',
        'fr': u':ouvrier:',
        'pt': u':operário:',
        'it': u':operaio:'
    },
    u'\U0001F468\U0001F3FF\U0000200D\U0001F3ED': {
        'en' : u':man_factory_worker_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':fabrikarbeiter_dunkle_hautfarbe:',
        'es': u':profesional_industrial_hombre_tono_de_piel_oscuro:',
        'fr': u':ouvrier_peau_foncée:',
        'pt': u':operário_pele_escura:',
        'it': u':operaio_carnagione_scura:'
    },
    u'\U0001F468\U0001F3FB\U0000200D\U0001F3ED': {
        'en' : u':man_factory_worker_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':fabrikarbeiter_helle_hautfarbe:',
        'es': u':profesional_industrial_hombre_tono_de_piel_claro:',
        'fr': u':ouvrier_peau_claire:',
        'pt': u':operário_pele_clara:',
        'it': u':operaio_carnagione_chiara:'
    },
    u'\U0001F468\U0001F3FE\U0000200D\U0001F3ED': {
        'en' : u':man_factory_worker_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':fabrikarbeiter_mitteldunkle_hautfarbe:',
        'es': u':profesional_industrial_hombre_tono_de_piel_oscuro_medio:',
        'fr': u':ouvrier_peau_mate:',
        'pt': u':operário_pele_morena_escura:',
        'it': u':operaio_carnagione_abbastanza_scura:'
    },
    u'\U0001F468\U0001F3FC\U0000200D\U0001F3ED': {
        'en' : u':man_factory_worker_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':fabrikarbeiter_mittelhelle_hautfarbe:',
        'es': u':profesional_industrial_hombre_tono_de_piel_claro_medio:',
        'fr': u':ouvrier_peau_moyennement_claire:',
        'pt': u':operário_pele_morena_clara:',
        'it': u':operaio_carnagione_abbastanza_chiara:'
    },
    u'\U0001F468\U0001F3FD\U0000200D\U0001F3ED': {
        'en' : u':man_factory_worker_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':fabrikarbeiter_mittlere_hautfarbe:',
        'es': u':profesional_industrial_hombre_tono_de_piel_medio:',
        'fr': u':ouvrier_peau_légèrement_mate:',
        'pt': u':operário_pele_morena:',
        'it': u':operaio_carnagione_olivastra:'
    },
    u'\U0001F9DA\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_fairy:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':männliche_fee:',
        'es': u':hada_hombre:',
        'fr': u':féetaud:',
        'pt': u':homem_fada:',
        'it': u':folletto_alato:'
    },
    u'\U0001F9DA\U0000200D\U00002642': {
        'en' : u':man_fairy:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F9DA\U0001F3FF\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_fairy_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':männliche_fee_dunkle_hautfarbe:',
        'es': u':hada_hombre_tono_de_piel_oscuro:',
        'fr': u':féetaud_peau_foncée:',
        'pt': u':homem_fada_pele_escura:',
        'it': u':folletto_alato_carnagione_scura:'
    },
    u'\U0001F9DA\U0001F3FF\U0000200D\U00002642': {
        'en' : u':man_fairy_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F9DA\U0001F3FB\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_fairy_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':männliche_fee_helle_hautfarbe:',
        'es': u':hada_hombre_tono_de_piel_claro:',
        'fr': u':féetaud_peau_claire:',
        'pt': u':homem_fada_pele_clara:',
        'it': u':folletto_alato_carnagione_chiara:'
    },
    u'\U0001F9DA\U0001F3FB\U0000200D\U00002642': {
        'en' : u':man_fairy_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F9DA\U0001F3FE\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_fairy_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':männliche_fee_mitteldunkle_hautfarbe:',
        'es': u':hada_hombre_tono_de_piel_oscuro_medio:',
        'fr': u':féetaud_peau_mate:',
        'pt': u':homem_fada_pele_morena_escura:',
        'it': u':folletto_alato_carnagione_abbastanza_scura:'
    },
    u'\U0001F9DA\U0001F3FE\U0000200D\U00002642': {
        'en' : u':man_fairy_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F9DA\U0001F3FC\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_fairy_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':männliche_fee_mittelhelle_hautfarbe:',
        'es': u':hada_hombre_tono_de_piel_claro_medio:',
        'fr': u':féetaud_peau_moyennement_claire:',
        'pt': u':homem_fada_pele_morena_clara:',
        'it': u':folletto_alato_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9DA\U0001F3FC\U0000200D\U00002642': {
        'en' : u':man_fairy_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F9DA\U0001F3FD\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_fairy_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':männliche_fee_mittlere_hautfarbe:',
        'es': u':hada_hombre_tono_de_piel_medio:',
        'fr': u':féetaud_peau_légèrement_mate:',
        'pt': u':homem_fada_pele_morena:',
        'it': u':folletto_alato_carnagione_olivastra:'
    },
    u'\U0001F9DA\U0001F3FD\U0000200D\U00002642': {
        'en' : u':man_fairy_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F468\U0000200D\U0001F33E': {
        'en' : u':man_farmer:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':bauer:',
        'es': u':profesional_de_la_agricultura_hombre:',
        'fr': u':fermier:',
        'pt': u':fazendeiro:',
        'it': u':contadino:'
    },
    u'\U0001F468\U0001F3FF\U0000200D\U0001F33E': {
        'en' : u':man_farmer_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':bauer_dunkle_hautfarbe:',
        'es': u':profesional_de_la_agricultura_hombre_tono_de_piel_oscuro:',
        'fr': u':fermier_peau_foncée:',
        'pt': u':fazendeiro_pele_escura:',
        'it': u':contadino_carnagione_scura:'
    },
    u'\U0001F468\U0001F3FB\U0000200D\U0001F33E': {
        'en' : u':man_farmer_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':bauer_helle_hautfarbe:',
        'es': u':profesional_de_la_agricultura_hombre_tono_de_piel_claro:',
        'fr': u':fermier_peau_claire:',
        'pt': u':fazendeiro_pele_clara:',
        'it': u':contadino_carnagione_chiara:'
    },
    u'\U0001F468\U0001F3FE\U0000200D\U0001F33E': {
        'en' : u':man_farmer_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':bauer_mitteldunkle_hautfarbe:',
        'es': u':profesional_de_la_agricultura_hombre_tono_de_piel_oscuro_medio:',
        'fr': u':fermier_peau_mate:',
        'pt': u':fazendeiro_pele_morena_escura:',
        'it': u':contadino_carnagione_abbastanza_scura:'
    },
    u'\U0001F468\U0001F3FC\U0000200D\U0001F33E': {
        'en' : u':man_farmer_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':bauer_mittelhelle_hautfarbe:',
        'es': u':profesional_de_la_agricultura_hombre_tono_de_piel_claro_medio:',
        'fr': u':fermier_peau_moyennement_claire:',
        'pt': u':fazendeiro_pele_morena_clara:',
        'it': u':contadino_carnagione_abbastanza_chiara:'
    },
    u'\U0001F468\U0001F3FD\U0000200D\U0001F33E': {
        'en' : u':man_farmer_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':bauer_mittlere_hautfarbe:',
        'es': u':profesional_de_la_agricultura_hombre_tono_de_piel_medio:',
        'fr': u':fermier_peau_légèrement_mate:',
        'pt': u':fazendeiro_pele_morena:',
        'it': u':contadino_carnagione_olivastra:'
    },
    u'\U0001F468\U0000200D\U0001F37C': {
        'en' : u':man_feeding_baby:',
        'status' : fully_qualified,
        'E' : 13,
        'de': u':stillender_mann:',
        'es': u':hombre_alimentando_a_bebé:',
        'fr': u':homme_allaitant_un_bébé:',
        'pt': u':homem_alimentando_bebê:',
        'it': u':uomo_che_allatta:'
    },
    u'\U0001F468\U0001F3FF\U0000200D\U0001F37C': {
        'en' : u':man_feeding_baby_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13,
        'de': u':stillender_mann_dunkle_hautfarbe:',
        'es': u':hombre_alimentando_a_bebé_tono_de_piel_oscuro:',
        'fr': u':homme_allaitant_un_bébé_peau_foncée:',
        'pt': u':homem_alimentando_bebê_pele_escura:',
        'it': u':uomo_che_allatta_carnagione_scura:'
    },
    u'\U0001F468\U0001F3FB\U0000200D\U0001F37C': {
        'en' : u':man_feeding_baby_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13,
        'de': u':stillender_mann_helle_hautfarbe:',
        'es': u':hombre_alimentando_a_bebé_tono_de_piel_claro:',
        'fr': u':homme_allaitant_un_bébé_peau_claire:',
        'pt': u':homem_alimentando_bebê_pele_clara:',
        'it': u':uomo_che_allatta_carnagione_chiara:'
    },
    u'\U0001F468\U0001F3FE\U0000200D\U0001F37C': {
        'en' : u':man_feeding_baby_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13,
        'de': u':stillender_mann_mitteldunkle_hautfarbe:',
        'es': u':hombre_alimentando_a_bebé_tono_de_piel_oscuro_medio:',
        'fr': u':homme_allaitant_un_bébé_peau_mate:',
        'pt': u':homem_alimentando_bebê_pele_morena_escura:',
        'it': u':uomo_che_allatta_carnagione_abbastanza_scura:'
    },
    u'\U0001F468\U0001F3FC\U0000200D\U0001F37C': {
        'en' : u':man_feeding_baby_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13,
        'de': u':stillender_mann_mittelhelle_hautfarbe:',
        'es': u':hombre_alimentando_a_bebé_tono_de_piel_claro_medio:',
        'fr': u':homme_allaitant_un_bébé_peau_moyennement_claire:',
        'pt': u':homem_alimentando_bebê_pele_morena_clara:',
        'it': u':uomo_che_allatta_carnagione_abbastanza_chiara:'
    },
    u'\U0001F468\U0001F3FD\U0000200D\U0001F37C': {
        'en' : u':man_feeding_baby_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13,
        'de': u':stillender_mann_mittlere_hautfarbe:',
        'es': u':hombre_alimentando_a_bebé_tono_de_piel_medio:',
        'fr': u':homme_allaitant_un_bébé_peau_légèrement_mate:',
        'pt': u':homem_alimentando_bebê_pele_morena:',
        'it': u':uomo_che_allatta_carnagione_olivastra:'
    },
    u'\U0001F468\U0000200D\U0001F692': {
        'en' : u':man_firefighter:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':feuerwehrmann:',
        'es': u':bombero_hombre:',
        'fr': u':pompier_homme:',
        'pt': u':bombeiro_homem:',
        'it': u':pompiere_uomo:'
    },
    u'\U0001F468\U0001F3FF\U0000200D\U0001F692': {
        'en' : u':man_firefighter_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':feuerwehrmann_dunkle_hautfarbe:',
        'es': u':bombero_hombre_tono_de_piel_oscuro:',
        'fr': u':pompier_homme_peau_foncée:',
        'pt': u':bombeiro_homem_pele_escura:',
        'it': u':pompiere_uomo_carnagione_scura:'
    },
    u'\U0001F468\U0001F3FB\U0000200D\U0001F692': {
        'en' : u':man_firefighter_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':feuerwehrmann_helle_hautfarbe:',
        'es': u':bombero_hombre_tono_de_piel_claro:',
        'fr': u':pompier_homme_peau_claire:',
        'pt': u':bombeiro_homem_pele_clara:',
        'it': u':pompiere_uomo_carnagione_chiara:'
    },
    u'\U0001F468\U0001F3FE\U0000200D\U0001F692': {
        'en' : u':man_firefighter_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':feuerwehrmann_mitteldunkle_hautfarbe:',
        'es': u':bombero_hombre_tono_de_piel_oscuro_medio:',
        'fr': u':pompier_homme_peau_mate:',
        'pt': u':bombeiro_homem_pele_morena_escura:',
        'it': u':pompiere_uomo_carnagione_abbastanza_scura:'
    },
    u'\U0001F468\U0001F3FC\U0000200D\U0001F692': {
        'en' : u':man_firefighter_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':feuerwehrmann_mittelhelle_hautfarbe:',
        'es': u':bombero_hombre_tono_de_piel_claro_medio:',
        'fr': u':pompier_homme_peau_moyennement_claire:',
        'pt': u':bombeiro_homem_pele_morena_clara:',
        'it': u':pompiere_uomo_carnagione_abbastanza_chiara:'
    },
    u'\U0001F468\U0001F3FD\U0000200D\U0001F692': {
        'en' : u':man_firefighter_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':feuerwehrmann_mittlere_hautfarbe:',
        'es': u':bombero_hombre_tono_de_piel_medio:',
        'fr': u':pompier_homme_peau_légèrement_mate:',
        'pt': u':bombeiro_homem_pele_morena:',
        'it': u':pompiere_uomo_carnagione_olivastra:'
    },
    u'\U0001F64D\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_frowning:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':missmutiger_mann:',
        'es': u':hombre_frunciendo_el_ceño:',
        'fr': u':homme_fronçant_les_sourcils:',
        'pt': u':homem_franzindo_a_sobrancelha:',
        'it': u':uomo_corrucciato:'
    },
    u'\U0001F64D\U0000200D\U00002642': {
        'en' : u':man_frowning:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F64D\U0001F3FF\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_frowning_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':missmutiger_mann_dunkle_hautfarbe:',
        'es': u':hombre_frunciendo_el_ceño_tono_de_piel_oscuro:',
        'fr': u':homme_fronçant_les_sourcils_peau_foncée:',
        'pt': u':homem_franzindo_a_sobrancelha_pele_escura:',
        'it': u':uomo_corrucciato_carnagione_scura:'
    },
    u'\U0001F64D\U0001F3FF\U0000200D\U00002642': {
        'en' : u':man_frowning_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F64D\U0001F3FB\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_frowning_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':missmutiger_mann_helle_hautfarbe:',
        'es': u':hombre_frunciendo_el_ceño_tono_de_piel_claro:',
        'fr': u':homme_fronçant_les_sourcils_peau_claire:',
        'pt': u':homem_franzindo_a_sobrancelha_pele_clara:',
        'it': u':uomo_corrucciato_carnagione_chiara:'
    },
    u'\U0001F64D\U0001F3FB\U0000200D\U00002642': {
        'en' : u':man_frowning_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F64D\U0001F3FE\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_frowning_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':missmutiger_mann_mitteldunkle_hautfarbe:',
        'es': u':hombre_frunciendo_el_ceño_tono_de_piel_oscuro_medio:',
        'fr': u':homme_fronçant_les_sourcils_peau_mate:',
        'pt': u':homem_franzindo_a_sobrancelha_pele_morena_escura:',
        'it': u':uomo_corrucciato_carnagione_abbastanza_scura:'
    },
    u'\U0001F64D\U0001F3FE\U0000200D\U00002642': {
        'en' : u':man_frowning_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F64D\U0001F3FC\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_frowning_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':missmutiger_mann_mittelhelle_hautfarbe:',
        'es': u':hombre_frunciendo_el_ceño_tono_de_piel_claro_medio:',
        'fr': u':homme_fronçant_les_sourcils_peau_moyennement_claire:',
        'pt': u':homem_franzindo_a_sobrancelha_pele_morena_clara:',
        'it': u':uomo_corrucciato_carnagione_abbastanza_chiara:'
    },
    u'\U0001F64D\U0001F3FC\U0000200D\U00002642': {
        'en' : u':man_frowning_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F64D\U0001F3FD\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_frowning_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':missmutiger_mann_mittlere_hautfarbe:',
        'es': u':hombre_frunciendo_el_ceño_tono_de_piel_medio:',
        'fr': u':homme_fronçant_les_sourcils_peau_légèrement_mate:',
        'pt': u':homem_franzindo_a_sobrancelha_pele_morena:',
        'it': u':uomo_corrucciato_carnagione_olivastra:'
    },
    u'\U0001F64D\U0001F3FD\U0000200D\U00002642': {
        'en' : u':man_frowning_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F9DE\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_genie:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':männlicher_flaschengeist:',
        'es': u':genio_hombre:',
        'fr': u':génie_homme:',
        'pt': u':homem_gênio:',
        'it': u':genio_uomo:'
    },
    u'\U0001F9DE\U0000200D\U00002642': {
        'en' : u':man_genie:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F645\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_gesturing_NO:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':mann_mit_überkreuzten_armen:',
        'es': u':hombre_haciendo_el_gesto_de_no:',
        'fr': u':homme_faisant_un_geste_d’interdiction:',
        'pt': u':homem_fazendo_gesto_de_“não”:',
        'it': u':uomo_con_gesto_di_rifiuto:'
    },
    u'\U0001F645\U0000200D\U00002642': {
        'en' : u':man_gesturing_NO:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F645\U0001F3FF\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_gesturing_NO_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':mann_mit_überkreuzten_armen_dunkle_hautfarbe:',
        'es': u':hombre_haciendo_el_gesto_de_no_tono_de_piel_oscuro:',
        'fr': u':homme_faisant_un_geste_d’interdiction_peau_foncée:',
        'pt': u':homem_fazendo_gesto_de_“não”_pele_escura:',
        'it': u':uomo_con_gesto_di_rifiuto_carnagione_scura:'
    },
    u'\U0001F645\U0001F3FF\U0000200D\U00002642': {
        'en' : u':man_gesturing_NO_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F645\U0001F3FB\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_gesturing_NO_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':mann_mit_überkreuzten_armen_helle_hautfarbe:',
        'es': u':hombre_haciendo_el_gesto_de_no_tono_de_piel_claro:',
        'fr': u':homme_faisant_un_geste_d’interdiction_peau_claire:',
        'pt': u':homem_fazendo_gesto_de_“não”_pele_clara:',
        'it': u':uomo_con_gesto_di_rifiuto_carnagione_chiara:'
    },
    u'\U0001F645\U0001F3FB\U0000200D\U00002642': {
        'en' : u':man_gesturing_NO_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F645\U0001F3FE\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_gesturing_NO_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':mann_mit_überkreuzten_armen_mitteldunkle_hautfarbe:',
        'es': u':hombre_haciendo_el_gesto_de_no_tono_de_piel_oscuro_medio:',
        'fr': u':homme_faisant_un_geste_d’interdiction_peau_mate:',
        'pt': u':homem_fazendo_gesto_de_“não”_pele_morena_escura:',
        'it': u':uomo_con_gesto_di_rifiuto_carnagione_abbastanza_scura:'
    },
    u'\U0001F645\U0001F3FE\U0000200D\U00002642': {
        'en' : u':man_gesturing_NO_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F645\U0001F3FC\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_gesturing_NO_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':mann_mit_überkreuzten_armen_mittelhelle_hautfarbe:',
        'es': u':hombre_haciendo_el_gesto_de_no_tono_de_piel_claro_medio:',
        'fr': u':homme_faisant_un_geste_d’interdiction_peau_moyennement_claire:',
        'pt': u':homem_fazendo_gesto_de_“não”_pele_morena_clara:',
        'it': u':uomo_con_gesto_di_rifiuto_carnagione_abbastanza_chiara:'
    },
    u'\U0001F645\U0001F3FC\U0000200D\U00002642': {
        'en' : u':man_gesturing_NO_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F645\U0001F3FD\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_gesturing_NO_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':mann_mit_überkreuzten_armen_mittlere_hautfarbe:',
        'es': u':hombre_haciendo_el_gesto_de_no_tono_de_piel_medio:',
        'fr': u':homme_faisant_un_geste_d’interdiction_peau_légèrement_mate:',
        'pt': u':homem_fazendo_gesto_de_“não”_pele_morena:',
        'it': u':uomo_con_gesto_di_rifiuto_carnagione_olivastra:'
    },
    u'\U0001F645\U0001F3FD\U0000200D\U00002642': {
        'en' : u':man_gesturing_NO_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F646\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_gesturing_OK:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':mann_mit_händen_auf_dem_kopf:',
        'es': u':hombre_haciendo_el_gesto_de_de_acuerdo:',
        'fr': u':homme_faisant_un_geste_d’acceptation:',
        'pt': u':homem_fazendo_gesto_de_“ok”:',
        'it': u':uomo_con_gesto_ok:'
    },
    u'\U0001F646\U0000200D\U00002642': {
        'en' : u':man_gesturing_OK:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F646\U0001F3FF\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_gesturing_OK_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':mann_mit_händen_auf_dem_kopf_dunkle_hautfarbe:',
        'es': u':hombre_haciendo_el_gesto_de_de_acuerdo_tono_de_piel_oscuro:',
        'fr': u':homme_faisant_un_geste_d’acceptation_peau_foncée:',
        'pt': u':homem_fazendo_gesto_de_“ok”_pele_escura:',
        'it': u':uomo_con_gesto_ok_carnagione_scura:'
    },
    u'\U0001F646\U0001F3FF\U0000200D\U00002642': {
        'en' : u':man_gesturing_OK_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F646\U0001F3FB\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_gesturing_OK_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':mann_mit_händen_auf_dem_kopf_helle_hautfarbe:',
        'es': u':hombre_haciendo_el_gesto_de_de_acuerdo_tono_de_piel_claro:',
        'fr': u':homme_faisant_un_geste_d’acceptation_peau_claire:',
        'pt': u':homem_fazendo_gesto_de_“ok”_pele_clara:',
        'it': u':uomo_con_gesto_ok_carnagione_chiara:'
    },
    u'\U0001F646\U0001F3FB\U0000200D\U00002642': {
        'en' : u':man_gesturing_OK_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F646\U0001F3FE\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_gesturing_OK_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':mann_mit_händen_auf_dem_kopf_mitteldunkle_hautfarbe:',
        'es': u':hombre_haciendo_el_gesto_de_de_acuerdo_tono_de_piel_oscuro_medio:',
        'fr': u':homme_faisant_un_geste_d’acceptation_peau_mate:',
        'pt': u':homem_fazendo_gesto_de_“ok”_pele_morena_escura:',
        'it': u':uomo_con_gesto_ok_carnagione_abbastanza_scura:'
    },
    u'\U0001F646\U0001F3FE\U0000200D\U00002642': {
        'en' : u':man_gesturing_OK_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F646\U0001F3FC\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_gesturing_OK_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':mann_mit_händen_auf_dem_kopf_mittelhelle_hautfarbe:',
        'es': u':hombre_haciendo_el_gesto_de_de_acuerdo_tono_de_piel_claro_medio:',
        'fr': u':homme_faisant_un_geste_d’acceptation_peau_moyennement_claire:',
        'pt': u':homem_fazendo_gesto_de_“ok”_pele_morena_clara:',
        'it': u':uomo_con_gesto_ok_carnagione_abbastanza_chiara:'
    },
    u'\U0001F646\U0001F3FC\U0000200D\U00002642': {
        'en' : u':man_gesturing_OK_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F646\U0001F3FD\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_gesturing_OK_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':mann_mit_händen_auf_dem_kopf_mittlere_hautfarbe:',
        'es': u':hombre_haciendo_el_gesto_de_de_acuerdo_tono_de_piel_medio:',
        'fr': u':homme_faisant_un_geste_d’acceptation_peau_légèrement_mate:',
        'pt': u':homem_fazendo_gesto_de_“ok”_pele_morena:',
        'it': u':uomo_con_gesto_ok_carnagione_olivastra:'
    },
    u'\U0001F646\U0001F3FD\U0000200D\U00002642': {
        'en' : u':man_gesturing_OK_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F487\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_getting_haircut:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':mann_beim_haareschneiden:',
        'es': u':hombre_cortándose_el_pelo:',
        'fr': u':homme_qui_se_fait_couper_les_cheveux:',
        'pt': u':homem_cortando_o_cabelo:',
        'it': u':taglio_di_capelli_per_uomo:'
    },
    u'\U0001F487\U0000200D\U00002642': {
        'en' : u':man_getting_haircut:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F487\U0001F3FF\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_getting_haircut_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':mann_beim_haareschneiden_dunkle_hautfarbe:',
        'es': u':hombre_cortándose_el_pelo_tono_de_piel_oscuro:',
        'fr': u':homme_qui_se_fait_couper_les_cheveux_peau_foncée:',
        'pt': u':homem_cortando_o_cabelo_pele_escura:',
        'it': u':taglio_di_capelli_per_uomo_carnagione_scura:'
    },
    u'\U0001F487\U0001F3FF\U0000200D\U00002642': {
        'en' : u':man_getting_haircut_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F487\U0001F3FB\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_getting_haircut_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':mann_beim_haareschneiden_helle_hautfarbe:',
        'es': u':hombre_cortándose_el_pelo_tono_de_piel_claro:',
        'fr': u':homme_qui_se_fait_couper_les_cheveux_peau_claire:',
        'pt': u':homem_cortando_o_cabelo_pele_clara:',
        'it': u':taglio_di_capelli_per_uomo_carnagione_chiara:'
    },
    u'\U0001F487\U0001F3FB\U0000200D\U00002642': {
        'en' : u':man_getting_haircut_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F487\U0001F3FE\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_getting_haircut_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':mann_beim_haareschneiden_mitteldunkle_hautfarbe:',
        'es': u':hombre_cortándose_el_pelo_tono_de_piel_oscuro_medio:',
        'fr': u':homme_qui_se_fait_couper_les_cheveux_peau_mate:',
        'pt': u':homem_cortando_o_cabelo_pele_morena_escura:',
        'it': u':taglio_di_capelli_per_uomo_carnagione_abbastanza_scura:'
    },
    u'\U0001F487\U0001F3FE\U0000200D\U00002642': {
        'en' : u':man_getting_haircut_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F487\U0001F3FC\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_getting_haircut_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':mann_beim_haareschneiden_mittelhelle_hautfarbe:',
        'es': u':hombre_cortándose_el_pelo_tono_de_piel_claro_medio:',
        'fr': u':homme_qui_se_fait_couper_les_cheveux_peau_moyennement_claire:',
        'pt': u':homem_cortando_o_cabelo_pele_morena_clara:',
        'it': u':taglio_di_capelli_per_uomo_carnagione_abbastanza_chiara:'
    },
    u'\U0001F487\U0001F3FC\U0000200D\U00002642': {
        'en' : u':man_getting_haircut_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F487\U0001F3FD\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_getting_haircut_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':mann_beim_haareschneiden_mittlere_hautfarbe:',
        'es': u':hombre_cortándose_el_pelo_tono_de_piel_medio:',
        'fr': u':homme_qui_se_fait_couper_les_cheveux_peau_légèrement_mate:',
        'pt': u':homem_cortando_o_cabelo_pele_morena:',
        'it': u':taglio_di_capelli_per_uomo_carnagione_olivastra:'
    },
    u'\U0001F487\U0001F3FD\U0000200D\U00002642': {
        'en' : u':man_getting_haircut_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F486\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_getting_massage:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':mann,_der_eine_kopfmassage_bekommt:',
        'es': u':hombre_recibiendo_masaje:',
        'fr': u':homme_qui_se_fait_masser:',
        'pt': u':homem_recebendo_massagem_facial:',
        'it': u':uomo_che_riceve_un_massaggio:'
    },
    u'\U0001F486\U0000200D\U00002642': {
        'en' : u':man_getting_massage:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F486\U0001F3FF\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_getting_massage_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':mann,_der_eine_kopfmassage_bekommt_dunkle_hautfarbe:',
        'es': u':hombre_recibiendo_masaje_tono_de_piel_oscuro:',
        'fr': u':homme_qui_se_fait_masser_peau_foncée:',
        'pt': u':homem_recebendo_massagem_facial_pele_escura:',
        'it': u':uomo_che_riceve_un_massaggio_carnagione_scura:'
    },
    u'\U0001F486\U0001F3FF\U0000200D\U00002642': {
        'en' : u':man_getting_massage_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F486\U0001F3FB\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_getting_massage_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':mann,_der_eine_kopfmassage_bekommt_helle_hautfarbe:',
        'es': u':hombre_recibiendo_masaje_tono_de_piel_claro:',
        'fr': u':homme_qui_se_fait_masser_peau_claire:',
        'pt': u':homem_recebendo_massagem_facial_pele_clara:',
        'it': u':uomo_che_riceve_un_massaggio_carnagione_chiara:'
    },
    u'\U0001F486\U0001F3FB\U0000200D\U00002642': {
        'en' : u':man_getting_massage_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F486\U0001F3FE\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_getting_massage_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':mann,_der_eine_kopfmassage_bekommt_mitteldunkle_hautfarbe:',
        'es': u':hombre_recibiendo_masaje_tono_de_piel_oscuro_medio:',
        'fr': u':homme_qui_se_fait_masser_peau_mate:',
        'pt': u':homem_recebendo_massagem_facial_pele_morena_escura:',
        'it': u':uomo_che_riceve_un_massaggio_carnagione_abbastanza_scura:'
    },
    u'\U0001F486\U0001F3FE\U0000200D\U00002642': {
        'en' : u':man_getting_massage_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F486\U0001F3FC\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_getting_massage_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':mann,_der_eine_kopfmassage_bekommt_mittelhelle_hautfarbe:',
        'es': u':hombre_recibiendo_masaje_tono_de_piel_claro_medio:',
        'fr': u':homme_qui_se_fait_masser_peau_moyennement_claire:',
        'pt': u':homem_recebendo_massagem_facial_pele_morena_clara:',
        'it': u':uomo_che_riceve_un_massaggio_carnagione_abbastanza_chiara:'
    },
    u'\U0001F486\U0001F3FC\U0000200D\U00002642': {
        'en' : u':man_getting_massage_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F486\U0001F3FD\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_getting_massage_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':mann,_der_eine_kopfmassage_bekommt_mittlere_hautfarbe:',
        'es': u':hombre_recibiendo_masaje_tono_de_piel_medio:',
        'fr': u':homme_qui_se_fait_masser_peau_légèrement_mate:',
        'pt': u':homem_recebendo_massagem_facial_pele_morena:',
        'it': u':uomo_che_riceve_un_massaggio_carnagione_olivastra:'
    },
    u'\U0001F486\U0001F3FD\U0000200D\U00002642': {
        'en' : u':man_getting_massage_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F3CC\U0000FE0F\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_golfing:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':golfer:',
        'es': u':hombre_jugando_al_golf:',
        'fr': u':golfeur:',
        'pt': u':homem_golfista:',
        'it': u':golfista_uomo:'
    },
    u'\U0001F3CC\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_golfing:',
        'status' : unqualified,
        'E' : 4
    },
    u'\U0001F3CC\U0000FE0F\U0000200D\U00002642': {
        'en' : u':man_golfing:',
        'status' : unqualified,
        'E' : 4
    },
    u'\U0001F3CC\U0000200D\U00002642': {
        'en' : u':man_golfing:',
        'status' : unqualified,
        'E' : 4
    },
    u'\U0001F3CC\U0001F3FF\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_golfing_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':golfer_dunkle_hautfarbe:',
        'es': u':hombre_jugando_al_golf_tono_de_piel_oscuro:',
        'fr': u':golfeur_peau_foncée:',
        'pt': u':homem_golfista_pele_escura:',
        'it': u':golfista_uomo_carnagione_scura:'
    },
    u'\U0001F3CC\U0001F3FF\U0000200D\U00002642': {
        'en' : u':man_golfing_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F3CC\U0001F3FB\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_golfing_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':golfer_helle_hautfarbe:',
        'es': u':hombre_jugando_al_golf_tono_de_piel_claro:',
        'fr': u':golfeur_peau_claire:',
        'pt': u':homem_golfista_pele_clara:',
        'it': u':golfista_uomo_carnagione_chiara:'
    },
    u'\U0001F3CC\U0001F3FB\U0000200D\U00002642': {
        'en' : u':man_golfing_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F3CC\U0001F3FE\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_golfing_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':golfer_mitteldunkle_hautfarbe:',
        'es': u':hombre_jugando_al_golf_tono_de_piel_oscuro_medio:',
        'fr': u':golfeur_peau_mate:',
        'pt': u':homem_golfista_pele_morena_escura:',
        'it': u':golfista_uomo_carnagione_abbastanza_scura:'
    },
    u'\U0001F3CC\U0001F3FE\U0000200D\U00002642': {
        'en' : u':man_golfing_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F3CC\U0001F3FC\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_golfing_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':golfer_mittelhelle_hautfarbe:',
        'es': u':hombre_jugando_al_golf_tono_de_piel_claro_medio:',
        'fr': u':golfeur_peau_moyennement_claire:',
        'pt': u':homem_golfista_pele_morena_clara:',
        'it': u':golfista_uomo_carnagione_abbastanza_chiara:'
    },
    u'\U0001F3CC\U0001F3FC\U0000200D\U00002642': {
        'en' : u':man_golfing_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F3CC\U0001F3FD\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_golfing_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':golfer_mittlere_hautfarbe:',
        'es': u':hombre_jugando_al_golf_tono_de_piel_medio:',
        'fr': u':golfeur_peau_légèrement_mate:',
        'pt': u':homem_golfista_pele_morena:',
        'it': u':golfista_uomo_carnagione_olivastra:'
    },
    u'\U0001F3CC\U0001F3FD\U0000200D\U00002642': {
        'en' : u':man_golfing_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F482\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_guard:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':wachmann:',
        'es': u':guardia_hombre:',
        'fr': u':garde_homme:',
        'pt': u':guarda_homem:',
        'it': u':guardia_uomo:'
    },
    u'\U0001F482\U0000200D\U00002642': {
        'en' : u':man_guard:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F482\U0001F3FF\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_guard_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':wachmann_dunkle_hautfarbe:',
        'es': u':guardia_hombre_tono_de_piel_oscuro:',
        'fr': u':garde_homme_peau_foncée:',
        'pt': u':guarda_homem_pele_escura:',
        'it': u':guardia_uomo_carnagione_scura:'
    },
    u'\U0001F482\U0001F3FF\U0000200D\U00002642': {
        'en' : u':man_guard_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F482\U0001F3FB\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_guard_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':wachmann_helle_hautfarbe:',
        'es': u':guardia_hombre_tono_de_piel_claro:',
        'fr': u':garde_homme_peau_claire:',
        'pt': u':guarda_homem_pele_clara:',
        'it': u':guardia_uomo_carnagione_chiara:'
    },
    u'\U0001F482\U0001F3FB\U0000200D\U00002642': {
        'en' : u':man_guard_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F482\U0001F3FE\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_guard_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':wachmann_mitteldunkle_hautfarbe:',
        'es': u':guardia_hombre_tono_de_piel_oscuro_medio:',
        'fr': u':garde_homme_peau_mate:',
        'pt': u':guarda_homem_pele_morena_escura:',
        'it': u':guardia_uomo_carnagione_abbastanza_scura:'
    },
    u'\U0001F482\U0001F3FE\U0000200D\U00002642': {
        'en' : u':man_guard_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F482\U0001F3FC\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_guard_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':wachmann_mittelhelle_hautfarbe:',
        'es': u':guardia_hombre_tono_de_piel_claro_medio:',
        'fr': u':garde_homme_peau_moyennement_claire:',
        'pt': u':guarda_homem_pele_morena_clara:',
        'it': u':guardia_uomo_carnagione_abbastanza_chiara:'
    },
    u'\U0001F482\U0001F3FC\U0000200D\U00002642': {
        'en' : u':man_guard_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F482\U0001F3FD\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_guard_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':wachmann_mittlere_hautfarbe:',
        'es': u':guardia_hombre_tono_de_piel_medio:',
        'fr': u':garde_homme_peau_légèrement_mate:',
        'pt': u':guarda_homem_pele_morena:',
        'it': u':guardia_uomo_carnagione_olivastra:'
    },
    u'\U0001F482\U0001F3FD\U0000200D\U00002642': {
        'en' : u':man_guard_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F468\U0000200D\U00002695\U0000FE0F': {
        'en' : u':man_health_worker:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':arzt:',
        'es': u':profesional_sanitario_hombre:',
        'fr': u':professionnel_de_la_santé:',
        'pt': u':homem_profissional_da_saúde:',
        'it': u':operatore_sanitario:'
    },
    u'\U0001F468\U0000200D\U00002695': {
        'en' : u':man_health_worker:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F468\U0001F3FF\U0000200D\U00002695\U0000FE0F': {
        'en' : u':man_health_worker_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':arzt_dunkle_hautfarbe:',
        'es': u':profesional_sanitario_hombre_tono_de_piel_oscuro:',
        'fr': u':professionnel_de_la_santé_peau_foncée:',
        'pt': u':homem_profissional_da_saúde_pele_escura:',
        'it': u':operatore_sanitario_carnagione_scura:'
    },
    u'\U0001F468\U0001F3FF\U0000200D\U00002695': {
        'en' : u':man_health_worker_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F468\U0001F3FB\U0000200D\U00002695\U0000FE0F': {
        'en' : u':man_health_worker_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':arzt_helle_hautfarbe:',
        'es': u':profesional_sanitario_hombre_tono_de_piel_claro:',
        'fr': u':professionnel_de_la_santé_peau_claire:',
        'pt': u':homem_profissional_da_saúde_pele_clara:',
        'it': u':operatore_sanitario_carnagione_chiara:'
    },
    u'\U0001F468\U0001F3FB\U0000200D\U00002695': {
        'en' : u':man_health_worker_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F468\U0001F3FE\U0000200D\U00002695\U0000FE0F': {
        'en' : u':man_health_worker_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':arzt_mitteldunkle_hautfarbe:',
        'es': u':profesional_sanitario_hombre_tono_de_piel_oscuro_medio:',
        'fr': u':professionnel_de_la_santé_peau_mate:',
        'pt': u':homem_profissional_da_saúde_pele_morena_escura:',
        'it': u':operatore_sanitario_carnagione_abbastanza_scura:'
    },
    u'\U0001F468\U0001F3FE\U0000200D\U00002695': {
        'en' : u':man_health_worker_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F468\U0001F3FC\U0000200D\U00002695\U0000FE0F': {
        'en' : u':man_health_worker_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':arzt_mittelhelle_hautfarbe:',
        'es': u':profesional_sanitario_hombre_tono_de_piel_claro_medio:',
        'fr': u':professionnel_de_la_santé_peau_moyennement_claire:',
        'pt': u':homem_profissional_da_saúde_pele_morena_clara:',
        'it': u':operatore_sanitario_carnagione_abbastanza_chiara:'
    },
    u'\U0001F468\U0001F3FC\U0000200D\U00002695': {
        'en' : u':man_health_worker_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F468\U0001F3FD\U0000200D\U00002695\U0000FE0F': {
        'en' : u':man_health_worker_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':arzt_mittlere_hautfarbe:',
        'es': u':profesional_sanitario_hombre_tono_de_piel_medio:',
        'fr': u':professionnel_de_la_santé_peau_légèrement_mate:',
        'pt': u':homem_profissional_da_saúde_pele_morena:',
        'it': u':operatore_sanitario_carnagione_olivastra:'
    },
    u'\U0001F468\U0001F3FD\U0000200D\U00002695': {
        'en' : u':man_health_worker_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F9D8\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_in_lotus_position:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':mann_im_lotossitz:',
        'es': u':hombre_en_posición_de_loto:',
        'fr': u':homme_dans_la_posture_du_lotus:',
        'pt': u':homem_na_posição_de_lótus:',
        'it': u':uomo_in_posizione_del_loto:'
    },
    u'\U0001F9D8\U0000200D\U00002642': {
        'en' : u':man_in_lotus_position:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F9D8\U0001F3FF\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_in_lotus_position_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':mann_im_lotossitz_dunkle_hautfarbe:',
        'es': u':hombre_en_posición_de_loto_tono_de_piel_oscuro:',
        'fr': u':homme_dans_la_posture_du_lotus_peau_foncée:',
        'pt': u':homem_na_posição_de_lótus_pele_escura:',
        'it': u':uomo_in_posizione_del_loto_carnagione_scura:'
    },
    u'\U0001F9D8\U0001F3FF\U0000200D\U00002642': {
        'en' : u':man_in_lotus_position_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F9D8\U0001F3FB\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_in_lotus_position_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':mann_im_lotossitz_helle_hautfarbe:',
        'es': u':hombre_en_posición_de_loto_tono_de_piel_claro:',
        'fr': u':homme_dans_la_posture_du_lotus_peau_claire:',
        'pt': u':homem_na_posição_de_lótus_pele_clara:',
        'it': u':uomo_in_posizione_del_loto_carnagione_chiara:'
    },
    u'\U0001F9D8\U0001F3FB\U0000200D\U00002642': {
        'en' : u':man_in_lotus_position_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F9D8\U0001F3FE\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_in_lotus_position_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':mann_im_lotossitz_mitteldunkle_hautfarbe:',
        'es': u':hombre_en_posición_de_loto_tono_de_piel_oscuro_medio:',
        'fr': u':homme_dans_la_posture_du_lotus_peau_mate:',
        'pt': u':homem_na_posição_de_lótus_pele_morena_escura:',
        'it': u':uomo_in_posizione_del_loto_carnagione_abbastanza_scura:'
    },
    u'\U0001F9D8\U0001F3FE\U0000200D\U00002642': {
        'en' : u':man_in_lotus_position_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F9D8\U0001F3FC\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_in_lotus_position_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':mann_im_lotossitz_mittelhelle_hautfarbe:',
        'es': u':hombre_en_posición_de_loto_tono_de_piel_claro_medio:',
        'fr': u':homme_dans_la_posture_du_lotus_peau_moyennement_claire:',
        'pt': u':homem_na_posição_de_lótus_pele_morena_clara:',
        'it': u':uomo_in_posizione_del_loto_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9D8\U0001F3FC\U0000200D\U00002642': {
        'en' : u':man_in_lotus_position_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F9D8\U0001F3FD\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_in_lotus_position_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':mann_im_lotossitz_mittlere_hautfarbe:',
        'es': u':hombre_en_posición_de_loto_tono_de_piel_medio:',
        'fr': u':homme_dans_la_posture_du_lotus_peau_légèrement_mate:',
        'pt': u':homem_na_posição_de_lótus_pele_morena:',
        'it': u':uomo_in_posizione_del_loto_carnagione_olivastra:'
    },
    u'\U0001F9D8\U0001F3FD\U0000200D\U00002642': {
        'en' : u':man_in_lotus_position_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F468\U0000200D\U0001F9BD': {
        'en' : u':man_in_manual_wheelchair:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':mann_in_manuellem_rollstuhl:',
        'es': u':hombre_en_silla_de_ruedas_manual:',
        'fr': u':homme_en_fauteuil_roulant_manuel:',
        'pt': u':homem_em_cadeira_de_rodas_manual:',
        'it': u':uomo_su_sedia_a_rotelle_manuale:'
    },
    u'\U0001F468\U0001F3FF\U0000200D\U0001F9BD': {
        'en' : u':man_in_manual_wheelchair_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':mann_in_manuellem_rollstuhl_dunkle_hautfarbe:',
        'es': u':hombre_en_silla_de_ruedas_manual_tono_de_piel_oscuro:',
        'fr': u':homme_en_fauteuil_roulant_manuel_peau_foncée:',
        'pt': u':homem_em_cadeira_de_rodas_manual_pele_escura:',
        'it': u':uomo_su_sedia_a_rotelle_manuale_carnagione_scura:'
    },
    u'\U0001F468\U0001F3FB\U0000200D\U0001F9BD': {
        'en' : u':man_in_manual_wheelchair_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':mann_in_manuellem_rollstuhl_helle_hautfarbe:',
        'es': u':hombre_en_silla_de_ruedas_manual_tono_de_piel_claro:',
        'fr': u':homme_en_fauteuil_roulant_manuel_peau_claire:',
        'pt': u':homem_em_cadeira_de_rodas_manual_pele_clara:',
        'it': u':uomo_su_sedia_a_rotelle_manuale_carnagione_chiara:'
    },
    u'\U0001F468\U0001F3FE\U0000200D\U0001F9BD': {
        'en' : u':man_in_manual_wheelchair_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':mann_in_manuellem_rollstuhl_mitteldunkle_hautfarbe:',
        'es': u':hombre_en_silla_de_ruedas_manual_tono_de_piel_oscuro_medio:',
        'fr': u':homme_en_fauteuil_roulant_manuel_peau_mate:',
        'pt': u':homem_em_cadeira_de_rodas_manual_pele_morena_escura:',
        'it': u':uomo_su_sedia_a_rotelle_manuale_carnagione_abbastanza_scura:'
    },
    u'\U0001F468\U0001F3FC\U0000200D\U0001F9BD': {
        'en' : u':man_in_manual_wheelchair_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':mann_in_manuellem_rollstuhl_mittelhelle_hautfarbe:',
        'es': u':hombre_en_silla_de_ruedas_manual_tono_de_piel_claro_medio:',
        'fr': u':homme_en_fauteuil_roulant_manuel_peau_moyennement_claire:',
        'pt': u':homem_em_cadeira_de_rodas_manual_pele_morena_clara:',
        'it': u':uomo_su_sedia_a_rotelle_manuale_carnagione_abbastanza_chiara:'
    },
    u'\U0001F468\U0001F3FD\U0000200D\U0001F9BD': {
        'en' : u':man_in_manual_wheelchair_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':mann_in_manuellem_rollstuhl_mittlere_hautfarbe:',
        'es': u':hombre_en_silla_de_ruedas_manual_tono_de_piel_medio:',
        'fr': u':homme_en_fauteuil_roulant_manuel_peau_légèrement_mate:',
        'pt': u':homem_em_cadeira_de_rodas_manual_pele_morena:',
        'it': u':uomo_su_sedia_a_rotelle_manuale_carnagione_olivastra:'
    },
    u'\U0001F468\U0000200D\U0001F9BC': {
        'en' : u':man_in_motorized_wheelchair:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':mann_in_elektrischem_rollstuhl:',
        'es': u':hombre_en_silla_de_ruedas_eléctrica:',
        'fr': u':homme_en_fauteuil_motorisé:',
        'pt': u':homem_em_cadeira_de_rodas_motorizada:',
        'it': u':uomo_su_sedia_a_rotelle_motorizzata:'
    },
    u'\U0001F468\U0001F3FF\U0000200D\U0001F9BC': {
        'en' : u':man_in_motorized_wheelchair_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':mann_in_elektrischem_rollstuhl_dunkle_hautfarbe:',
        'es': u':hombre_en_silla_de_ruedas_eléctrica_tono_de_piel_oscuro:',
        'fr': u':homme_en_fauteuil_motorisé_peau_foncée:',
        'pt': u':homem_em_cadeira_de_rodas_motorizada_pele_escura:',
        'it': u':uomo_su_sedia_a_rotelle_motorizzata_carnagione_scura:'
    },
    u'\U0001F468\U0001F3FB\U0000200D\U0001F9BC': {
        'en' : u':man_in_motorized_wheelchair_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':mann_in_elektrischem_rollstuhl_helle_hautfarbe:',
        'es': u':hombre_en_silla_de_ruedas_eléctrica_tono_de_piel_claro:',
        'fr': u':homme_en_fauteuil_motorisé_peau_claire:',
        'pt': u':homem_em_cadeira_de_rodas_motorizada_pele_clara:',
        'it': u':uomo_su_sedia_a_rotelle_motorizzata_carnagione_chiara:'
    },
    u'\U0001F468\U0001F3FE\U0000200D\U0001F9BC': {
        'en' : u':man_in_motorized_wheelchair_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':mann_in_elektrischem_rollstuhl_mitteldunkle_hautfarbe:',
        'es': u':hombre_en_silla_de_ruedas_eléctrica_tono_de_piel_oscuro_medio:',
        'fr': u':homme_en_fauteuil_motorisé_peau_mate:',
        'pt': u':homem_em_cadeira_de_rodas_motorizada_pele_morena_escura:',
        'it': u':uomo_su_sedia_a_rotelle_motorizzata_carnagione_abbastanza_scura:'
    },
    u'\U0001F468\U0001F3FC\U0000200D\U0001F9BC': {
        'en' : u':man_in_motorized_wheelchair_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':mann_in_elektrischem_rollstuhl_mittelhelle_hautfarbe:',
        'es': u':hombre_en_silla_de_ruedas_eléctrica_tono_de_piel_claro_medio:',
        'fr': u':homme_en_fauteuil_motorisé_peau_moyennement_claire:',
        'pt': u':homem_em_cadeira_de_rodas_motorizada_pele_morena_clara:',
        'it': u':uomo_su_sedia_a_rotelle_motorizzata_carnagione_abbastanza_chiara:'
    },
    u'\U0001F468\U0001F3FD\U0000200D\U0001F9BC': {
        'en' : u':man_in_motorized_wheelchair_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':mann_in_elektrischem_rollstuhl_mittlere_hautfarbe:',
        'es': u':hombre_en_silla_de_ruedas_eléctrica_tono_de_piel_medio:',
        'fr': u':homme_en_fauteuil_motorisé_peau_légèrement_mate:',
        'pt': u':homem_em_cadeira_de_rodas_motorizada_pele_morena:',
        'it': u':uomo_su_sedia_a_rotelle_motorizzata_carnagione_olivastra:'
    },
    u'\U0001F9D6\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_in_steamy_room:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':mann_in_dampfsauna:',
        'es': u':hombre_en_una_sauna:',
        'fr': u':homme_au_hammam:',
        'pt': u':homem_na_sauna:',
        'it': u':uomo_in_sauna:'
    },
    u'\U0001F9D6\U0000200D\U00002642': {
        'en' : u':man_in_steamy_room:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F9D6\U0001F3FF\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_in_steamy_room_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':mann_in_dampfsauna_dunkle_hautfarbe:',
        'es': u':hombre_en_una_sauna_tono_de_piel_oscuro:',
        'fr': u':homme_au_hammam_peau_foncée:',
        'pt': u':homem_na_sauna_pele_escura:',
        'it': u':uomo_in_sauna_carnagione_scura:'
    },
    u'\U0001F9D6\U0001F3FF\U0000200D\U00002642': {
        'en' : u':man_in_steamy_room_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F9D6\U0001F3FB\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_in_steamy_room_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':mann_in_dampfsauna_helle_hautfarbe:',
        'es': u':hombre_en_una_sauna_tono_de_piel_claro:',
        'fr': u':homme_au_hammam_peau_claire:',
        'pt': u':homem_na_sauna_pele_clara:',
        'it': u':uomo_in_sauna_carnagione_chiara:'
    },
    u'\U0001F9D6\U0001F3FB\U0000200D\U00002642': {
        'en' : u':man_in_steamy_room_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F9D6\U0001F3FE\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_in_steamy_room_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':mann_in_dampfsauna_mitteldunkle_hautfarbe:',
        'es': u':hombre_en_una_sauna_tono_de_piel_oscuro_medio:',
        'fr': u':homme_au_hammam_peau_mate:',
        'pt': u':homem_na_sauna_pele_morena_escura:',
        'it': u':uomo_in_sauna_carnagione_abbastanza_scura:'
    },
    u'\U0001F9D6\U0001F3FE\U0000200D\U00002642': {
        'en' : u':man_in_steamy_room_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F9D6\U0001F3FC\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_in_steamy_room_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':mann_in_dampfsauna_mittelhelle_hautfarbe:',
        'es': u':hombre_en_una_sauna_tono_de_piel_claro_medio:',
        'fr': u':homme_au_hammam_peau_moyennement_claire:',
        'pt': u':homem_na_sauna_pele_morena_clara:',
        'it': u':uomo_in_sauna_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9D6\U0001F3FC\U0000200D\U00002642': {
        'en' : u':man_in_steamy_room_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F9D6\U0001F3FD\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_in_steamy_room_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':mann_in_dampfsauna_mittlere_hautfarbe:',
        'es': u':hombre_en_una_sauna_tono_de_piel_medio:',
        'fr': u':homme_au_hammam_peau_légèrement_mate:',
        'pt': u':homem_na_sauna_pele_morena:',
        'it': u':uomo_in_sauna_carnagione_olivastra:'
    },
    u'\U0001F9D6\U0001F3FD\U0000200D\U00002642': {
        'en' : u':man_in_steamy_room_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F935\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_in_tuxedo:',
        'status' : fully_qualified,
        'E' : 13,
        'de': u':mann_im_smoking:',
        'es': u':hombre_con_esmoquin:',
        'fr': u':homme_en_smoking:',
        'pt': u':homem_de_smoking:',
        'it': u':uomo_in_smoking:'
    },
    u'\U0001F935\U0000200D\U00002642': {
        'en' : u':man_in_tuxedo:',
        'status' : minimally_qualified,
        'E' : 13
    },
    u'\U0001F935\U0001F3FF\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_in_tuxedo_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13,
        'de': u':mann_im_smoking_dunkle_hautfarbe:',
        'es': u':hombre_con_esmoquin_tono_de_piel_oscuro:',
        'fr': u':homme_en_smoking_peau_foncée:',
        'pt': u':homem_de_smoking_pele_escura:',
        'it': u':uomo_in_smoking_carnagione_scura:'
    },
    u'\U0001F935\U0001F3FF\U0000200D\U00002642': {
        'en' : u':man_in_tuxedo_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13
    },
    u'\U0001F935\U0001F3FB\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_in_tuxedo_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13,
        'de': u':mann_im_smoking_helle_hautfarbe:',
        'es': u':hombre_con_esmoquin_tono_de_piel_claro:',
        'fr': u':homme_en_smoking_peau_claire:',
        'pt': u':homem_de_smoking_pele_clara:',
        'it': u':uomo_in_smoking_carnagione_chiara:'
    },
    u'\U0001F935\U0001F3FB\U0000200D\U00002642': {
        'en' : u':man_in_tuxedo_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13
    },
    u'\U0001F935\U0001F3FE\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_in_tuxedo_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13,
        'de': u':mann_im_smoking_mitteldunkle_hautfarbe:',
        'es': u':hombre_con_esmoquin_tono_de_piel_oscuro_medio:',
        'fr': u':homme_en_smoking_peau_mate:',
        'pt': u':homem_de_smoking_pele_morena_escura:',
        'it': u':uomo_in_smoking_carnagione_abbastanza_scura:'
    },
    u'\U0001F935\U0001F3FE\U0000200D\U00002642': {
        'en' : u':man_in_tuxedo_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13
    },
    u'\U0001F935\U0001F3FC\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_in_tuxedo_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13,
        'de': u':mann_im_smoking_mittelhelle_hautfarbe:',
        'es': u':hombre_con_esmoquin_tono_de_piel_claro_medio:',
        'fr': u':homme_en_smoking_peau_moyennement_claire:',
        'pt': u':homem_de_smoking_pele_morena_clara:',
        'it': u':uomo_in_smoking_carnagione_abbastanza_chiara:'
    },
    u'\U0001F935\U0001F3FC\U0000200D\U00002642': {
        'en' : u':man_in_tuxedo_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13
    },
    u'\U0001F935\U0001F3FD\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_in_tuxedo_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13,
        'de': u':mann_im_smoking_mittlere_hautfarbe:',
        'es': u':hombre_con_esmoquin_tono_de_piel_medio:',
        'fr': u':homme_en_smoking_peau_légèrement_mate:',
        'pt': u':homem_de_smoking_pele_morena:',
        'it': u':uomo_in_smoking_carnagione_olivastra:'
    },
    u'\U0001F935\U0001F3FD\U0000200D\U00002642': {
        'en' : u':man_in_tuxedo_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13
    },
    u'\U0001F468\U0000200D\U00002696\U0000FE0F': {
        'en' : u':man_judge:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':richter:',
        'es': u':fiscal_hombre:',
        'fr': u':juge_homme:',
        'pt': u':juiz:',
        'it': u':giudice_uomo:'
    },
    u'\U0001F468\U0000200D\U00002696': {
        'en' : u':man_judge:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F468\U0001F3FF\U0000200D\U00002696\U0000FE0F': {
        'en' : u':man_judge_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':richter_dunkle_hautfarbe:',
        'es': u':fiscal_hombre_tono_de_piel_oscuro:',
        'fr': u':juge_homme_peau_foncée:',
        'pt': u':juiz_pele_escura:',
        'it': u':giudice_uomo_carnagione_scura:'
    },
    u'\U0001F468\U0001F3FF\U0000200D\U00002696': {
        'en' : u':man_judge_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F468\U0001F3FB\U0000200D\U00002696\U0000FE0F': {
        'en' : u':man_judge_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':richter_helle_hautfarbe:',
        'es': u':fiscal_hombre_tono_de_piel_claro:',
        'fr': u':juge_homme_peau_claire:',
        'pt': u':juiz_pele_clara:',
        'it': u':giudice_uomo_carnagione_chiara:'
    },
    u'\U0001F468\U0001F3FB\U0000200D\U00002696': {
        'en' : u':man_judge_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F468\U0001F3FE\U0000200D\U00002696\U0000FE0F': {
        'en' : u':man_judge_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':richter_mitteldunkle_hautfarbe:',
        'es': u':fiscal_hombre_tono_de_piel_oscuro_medio:',
        'fr': u':juge_homme_peau_mate:',
        'pt': u':juiz_pele_morena_escura:',
        'it': u':giudice_uomo_carnagione_abbastanza_scura:'
    },
    u'\U0001F468\U0001F3FE\U0000200D\U00002696': {
        'en' : u':man_judge_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F468\U0001F3FC\U0000200D\U00002696\U0000FE0F': {
        'en' : u':man_judge_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':richter_mittelhelle_hautfarbe:',
        'es': u':fiscal_hombre_tono_de_piel_claro_medio:',
        'fr': u':juge_homme_peau_moyennement_claire:',
        'pt': u':juiz_pele_morena_clara:',
        'it': u':giudice_uomo_carnagione_abbastanza_chiara:'
    },
    u'\U0001F468\U0001F3FC\U0000200D\U00002696': {
        'en' : u':man_judge_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F468\U0001F3FD\U0000200D\U00002696\U0000FE0F': {
        'en' : u':man_judge_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':richter_mittlere_hautfarbe:',
        'es': u':fiscal_hombre_tono_de_piel_medio:',
        'fr': u':juge_homme_peau_légèrement_mate:',
        'pt': u':juiz_pele_morena:',
        'it': u':giudice_uomo_carnagione_olivastra:'
    },
    u'\U0001F468\U0001F3FD\U0000200D\U00002696': {
        'en' : u':man_judge_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F939\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_juggling:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':jongleur:',
        'es': u':hombre_haciendo_malabares:',
        'fr': u':jongleur:',
        'pt': u':homem_malabarista:',
        'it': u':giocoliere_uomo:'
    },
    u'\U0001F939\U0000200D\U00002642': {
        'en' : u':man_juggling:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F939\U0001F3FF\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_juggling_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':jongleur_dunkle_hautfarbe:',
        'es': u':hombre_haciendo_malabares_tono_de_piel_oscuro:',
        'fr': u':jongleur_peau_foncée:',
        'pt': u':homem_malabarista_pele_escura:',
        'it': u':giocoliere_uomo_carnagione_scura:'
    },
    u'\U0001F939\U0001F3FF\U0000200D\U00002642': {
        'en' : u':man_juggling_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F939\U0001F3FB\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_juggling_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':jongleur_helle_hautfarbe:',
        'es': u':hombre_haciendo_malabares_tono_de_piel_claro:',
        'fr': u':jongleur_peau_claire:',
        'pt': u':homem_malabarista_pele_clara:',
        'it': u':giocoliere_uomo_carnagione_chiara:'
    },
    u'\U0001F939\U0001F3FB\U0000200D\U00002642': {
        'en' : u':man_juggling_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F939\U0001F3FE\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_juggling_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':jongleur_mitteldunkle_hautfarbe:',
        'es': u':hombre_haciendo_malabares_tono_de_piel_oscuro_medio:',
        'fr': u':jongleur_peau_mate:',
        'pt': u':homem_malabarista_pele_morena_escura:',
        'it': u':giocoliere_uomo_carnagione_abbastanza_scura:'
    },
    u'\U0001F939\U0001F3FE\U0000200D\U00002642': {
        'en' : u':man_juggling_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F939\U0001F3FC\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_juggling_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':jongleur_mittelhelle_hautfarbe:',
        'es': u':hombre_haciendo_malabares_tono_de_piel_claro_medio:',
        'fr': u':jongleur_peau_moyennement_claire:',
        'pt': u':homem_malabarista_pele_morena_clara:',
        'it': u':giocoliere_uomo_carnagione_abbastanza_chiara:'
    },
    u'\U0001F939\U0001F3FC\U0000200D\U00002642': {
        'en' : u':man_juggling_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F939\U0001F3FD\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_juggling_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':jongleur_mittlere_hautfarbe:',
        'es': u':hombre_haciendo_malabares_tono_de_piel_medio:',
        'fr': u':jongleur_peau_légèrement_mate:',
        'pt': u':homem_malabarista_pele_morena:',
        'it': u':giocoliere_uomo_carnagione_olivastra:'
    },
    u'\U0001F939\U0001F3FD\U0000200D\U00002642': {
        'en' : u':man_juggling_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F9CE\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_kneeling:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':kniender_mann:',
        'es': u':hombre_de_rodillas:',
        'fr': u':homme_à_genoux:',
        'pt': u':homem_ajoelhando:',
        'it': u':uomo_inginocchiato:'
    },
    u'\U0001F9CE\U0000200D\U00002642': {
        'en' : u':man_kneeling:',
        'status' : minimally_qualified,
        'E' : 12
    },
    u'\U0001F9CE\U0001F3FF\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_kneeling_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':kniender_mann_dunkle_hautfarbe:',
        'es': u':hombre_de_rodillas_tono_de_piel_oscuro:',
        'fr': u':homme_à_genoux_peau_foncée:',
        'pt': u':homem_ajoelhando_pele_escura:',
        'it': u':uomo_inginocchiato_carnagione_scura:'
    },
    u'\U0001F9CE\U0001F3FF\U0000200D\U00002642': {
        'en' : u':man_kneeling_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 12
    },
    u'\U0001F9CE\U0001F3FB\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_kneeling_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':kniender_mann_helle_hautfarbe:',
        'es': u':hombre_de_rodillas_tono_de_piel_claro:',
        'fr': u':homme_à_genoux_peau_claire:',
        'pt': u':homem_ajoelhando_pele_clara:',
        'it': u':uomo_inginocchiato_carnagione_chiara:'
    },
    u'\U0001F9CE\U0001F3FB\U0000200D\U00002642': {
        'en' : u':man_kneeling_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 12
    },
    u'\U0001F9CE\U0001F3FE\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_kneeling_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':kniender_mann_mitteldunkle_hautfarbe:',
        'es': u':hombre_de_rodillas_tono_de_piel_oscuro_medio:',
        'fr': u':homme_à_genoux_peau_mate:',
        'pt': u':homem_ajoelhando_pele_morena_escura:',
        'it': u':uomo_inginocchiato_carnagione_abbastanza_scura:'
    },
    u'\U0001F9CE\U0001F3FE\U0000200D\U00002642': {
        'en' : u':man_kneeling_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 12
    },
    u'\U0001F9CE\U0001F3FC\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_kneeling_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':kniender_mann_mittelhelle_hautfarbe:',
        'es': u':hombre_de_rodillas_tono_de_piel_claro_medio:',
        'fr': u':homme_à_genoux_peau_moyennement_claire:',
        'pt': u':homem_ajoelhando_pele_morena_clara:',
        'it': u':uomo_inginocchiato_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9CE\U0001F3FC\U0000200D\U00002642': {
        'en' : u':man_kneeling_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 12
    },
    u'\U0001F9CE\U0001F3FD\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_kneeling_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':kniender_mann_mittlere_hautfarbe:',
        'es': u':hombre_de_rodillas_tono_de_piel_medio:',
        'fr': u':homme_à_genoux_peau_légèrement_mate:',
        'pt': u':homem_ajoelhando_pele_morena:',
        'it': u':uomo_inginocchiato_carnagione_olivastra:'
    },
    u'\U0001F9CE\U0001F3FD\U0000200D\U00002642': {
        'en' : u':man_kneeling_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 12
    },
    u'\U0001F3CB\U0000FE0F\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_lifting_weights:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':gewichtheber:',
        'es': u':hombre_levantando_pesas:',
        'fr': u':homme_haltérophile:',
        'pt': u':homem_levantando_peso:',
        'it': u':uomo_che_solleva_pesi:'
    },
    u'\U0001F3CB\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_lifting_weights:',
        'status' : unqualified,
        'E' : 4
    },
    u'\U0001F3CB\U0000FE0F\U0000200D\U00002642': {
        'en' : u':man_lifting_weights:',
        'status' : unqualified,
        'E' : 4
    },
    u'\U0001F3CB\U0000200D\U00002642': {
        'en' : u':man_lifting_weights:',
        'status' : unqualified,
        'E' : 4
    },
    u'\U0001F3CB\U0001F3FF\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_lifting_weights_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':gewichtheber_dunkle_hautfarbe:',
        'es': u':hombre_levantando_pesas_tono_de_piel_oscuro:',
        'fr': u':homme_haltérophile_peau_foncée:',
        'pt': u':homem_levantando_peso_pele_escura:',
        'it': u':uomo_che_solleva_pesi_carnagione_scura:'
    },
    u'\U0001F3CB\U0001F3FF\U0000200D\U00002642': {
        'en' : u':man_lifting_weights_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F3CB\U0001F3FB\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_lifting_weights_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':gewichtheber_helle_hautfarbe:',
        'es': u':hombre_levantando_pesas_tono_de_piel_claro:',
        'fr': u':homme_haltérophile_peau_claire:',
        'pt': u':homem_levantando_peso_pele_clara:',
        'it': u':uomo_che_solleva_pesi_carnagione_chiara:'
    },
    u'\U0001F3CB\U0001F3FB\U0000200D\U00002642': {
        'en' : u':man_lifting_weights_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F3CB\U0001F3FE\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_lifting_weights_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':gewichtheber_mitteldunkle_hautfarbe:',
        'es': u':hombre_levantando_pesas_tono_de_piel_oscuro_medio:',
        'fr': u':homme_haltérophile_peau_mate:',
        'pt': u':homem_levantando_peso_pele_morena_escura:',
        'it': u':uomo_che_solleva_pesi_carnagione_abbastanza_scura:'
    },
    u'\U0001F3CB\U0001F3FE\U0000200D\U00002642': {
        'en' : u':man_lifting_weights_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F3CB\U0001F3FC\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_lifting_weights_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':gewichtheber_mittelhelle_hautfarbe:',
        'es': u':hombre_levantando_pesas_tono_de_piel_claro_medio:',
        'fr': u':homme_haltérophile_peau_moyennement_claire:',
        'pt': u':homem_levantando_peso_pele_morena_clara:',
        'it': u':uomo_che_solleva_pesi_carnagione_abbastanza_chiara:'
    },
    u'\U0001F3CB\U0001F3FC\U0000200D\U00002642': {
        'en' : u':man_lifting_weights_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F3CB\U0001F3FD\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_lifting_weights_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':gewichtheber_mittlere_hautfarbe:',
        'es': u':hombre_levantando_pesas_tono_de_piel_medio:',
        'fr': u':homme_haltérophile_peau_légèrement_mate:',
        'pt': u':homem_levantando_peso_pele_morena:',
        'it': u':uomo_che_solleva_pesi_carnagione_olivastra:'
    },
    u'\U0001F3CB\U0001F3FD\U0000200D\U00002642': {
        'en' : u':man_lifting_weights_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F468\U0001F3FB': {
        'en' : u':man_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':mann_helle_hautfarbe:',
        'es': u':hombre_tono_de_piel_claro:',
        'fr': u':homme_peau_claire:',
        'pt': u':homem_pele_clara:',
        'it': u':uomo_carnagione_chiara:'
    },
    u'\U0001F468\U0001F3FB\U0000200D\U0001F9B2': {
        'en' : u':man_light_skin_tone_bald:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':mann_helle_hautfarbe,_glatze:',
        'es': u':hombre_tono_de_piel_claro_y_sin_pelo:',
        'fr': u':homme_peau_claire_et_chauve:',
        'pt': u':homem_pele_clara_e_careca:',
        'it': u':uomo_carnagione_chiara_e_calvo:'
    },
    u'\U0001F9D4\U0001F3FB\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_light_skin_tone_beard:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D4\U0001F3FB\U0000200D\U00002642': {
        'en' : u':man_light_skin_tone_beard:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F471\U0001F3FB\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_light_skin_tone_blond_hair:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':mann_helle_hautfarbe,_blond:',
        'es': u':hombre_rubio_tono_de_piel_claro:',
        'fr': u':homme_blond_peau_claire:',
        'pt': u':homem_pele_clara_e_cabelo_loiro:',
        'it': u':uomo_biondo_carnagione_chiara:'
    },
    u'\U0001F471\U0001F3FB\U0000200D\U00002642': {
        'en' : u':man_light_skin_tone_blond_hair:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F468\U0001F3FB\U0000200D\U0001F9B1': {
        'en' : u':man_light_skin_tone_curly_hair:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':mann_helle_hautfarbe,_lockiges_haar:',
        'es': u':hombre_tono_de_piel_claro_y_pelo_rizado:',
        'fr': u':homme_peau_claire_et_cheveux_bouclés:',
        'pt': u':homem_pele_clara_e_cabelo_cacheado:',
        'it': u':uomo_carnagione_chiara_e_capelli_ricci:'
    },
    u'\U0001F468\U0001F3FB\U0000200D\U0001F9B0': {
        'en' : u':man_light_skin_tone_red_hair:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':mann_helle_hautfarbe,_rotes_haar:',
        'es': u':hombre_tono_de_piel_claro_y_pelo_pelirrojo:',
        'fr': u':homme_peau_claire_et_cheveux_roux:',
        'pt': u':homem_pele_clara_e_cabelo_vermelho:',
        'it': u':uomo_carnagione_chiara_e_capelli_rossi:'
    },
    u'\U0001F468\U0001F3FB\U0000200D\U0001F9B3': {
        'en' : u':man_light_skin_tone_white_hair:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':mann_helle_hautfarbe,_weißes_haar:',
        'es': u':hombre_tono_de_piel_claro_y_pelo_blanco:',
        'fr': u':homme_peau_claire_et_cheveux_blancs:',
        'pt': u':homem_pele_clara_e_cabelo_branco:',
        'it': u':uomo_carnagione_chiara_e_capelli_bianchi:'
    },
    u'\U0001F9D9\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_mage:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':magier:',
        'es': u':mago:',
        'fr': u':mage_homme:',
        'pt': u':homem_mago:',
        'it': u':mago_uomo:'
    },
    u'\U0001F9D9\U0000200D\U00002642': {
        'en' : u':man_mage:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F9D9\U0001F3FF\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_mage_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':magier_dunkle_hautfarbe:',
        'es': u':mago_tono_de_piel_oscuro:',
        'fr': u':mage_homme_peau_foncée:',
        'pt': u':homem_mago_pele_escura:',
        'it': u':mago_uomo_carnagione_scura:'
    },
    u'\U0001F9D9\U0001F3FF\U0000200D\U00002642': {
        'en' : u':man_mage_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F9D9\U0001F3FB\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_mage_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':magier_helle_hautfarbe:',
        'es': u':mago_tono_de_piel_claro:',
        'fr': u':mage_homme_peau_claire:',
        'pt': u':homem_mago_pele_clara:',
        'it': u':mago_uomo_carnagione_chiara:'
    },
    u'\U0001F9D9\U0001F3FB\U0000200D\U00002642': {
        'en' : u':man_mage_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F9D9\U0001F3FE\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_mage_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':magier_mitteldunkle_hautfarbe:',
        'es': u':mago_tono_de_piel_oscuro_medio:',
        'fr': u':mage_homme_peau_mate:',
        'pt': u':homem_mago_pele_morena_escura:',
        'it': u':mago_uomo_carnagione_abbastanza_scura:'
    },
    u'\U0001F9D9\U0001F3FE\U0000200D\U00002642': {
        'en' : u':man_mage_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F9D9\U0001F3FC\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_mage_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':magier_mittelhelle_hautfarbe:',
        'es': u':mago_tono_de_piel_claro_medio:',
        'fr': u':mage_homme_peau_moyennement_claire:',
        'pt': u':homem_mago_pele_morena_clara:',
        'it': u':mago_uomo_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9D9\U0001F3FC\U0000200D\U00002642': {
        'en' : u':man_mage_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F9D9\U0001F3FD\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_mage_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':magier_mittlere_hautfarbe:',
        'es': u':mago_tono_de_piel_medio:',
        'fr': u':mage_homme_peau_légèrement_mate:',
        'pt': u':homem_mago_pele_morena:',
        'it': u':mago_uomo_carnagione_olivastra:'
    },
    u'\U0001F9D9\U0001F3FD\U0000200D\U00002642': {
        'en' : u':man_mage_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F468\U0000200D\U0001F527': {
        'en' : u':man_mechanic:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':mechaniker:',
        'es': u':profesional_de_la_mecánica_hombre:',
        'fr': u':mécanicien:',
        'pt': u':mecânico:',
        'it': u':meccanico_uomo:'
    },
    u'\U0001F468\U0001F3FF\U0000200D\U0001F527': {
        'en' : u':man_mechanic_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':mechaniker_dunkle_hautfarbe:',
        'es': u':profesional_de_la_mecánica_hombre_tono_de_piel_oscuro:',
        'fr': u':mécanicien_peau_foncée:',
        'pt': u':mecânico_pele_escura:',
        'it': u':meccanico_uomo_carnagione_scura:'
    },
    u'\U0001F468\U0001F3FB\U0000200D\U0001F527': {
        'en' : u':man_mechanic_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':mechaniker_helle_hautfarbe:',
        'es': u':profesional_de_la_mecánica_hombre_tono_de_piel_claro:',
        'fr': u':mécanicien_peau_claire:',
        'pt': u':mecânico_pele_clara:',
        'it': u':meccanico_uomo_carnagione_chiara:'
    },
    u'\U0001F468\U0001F3FE\U0000200D\U0001F527': {
        'en' : u':man_mechanic_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':mechaniker_mitteldunkle_hautfarbe:',
        'es': u':profesional_de_la_mecánica_hombre_tono_de_piel_oscuro_medio:',
        'fr': u':mécanicien_peau_mate:',
        'pt': u':mecânico_pele_morena_escura:',
        'it': u':meccanico_uomo_carnagione_abbastanza_scura:'
    },
    u'\U0001F468\U0001F3FC\U0000200D\U0001F527': {
        'en' : u':man_mechanic_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':mechaniker_mittelhelle_hautfarbe:',
        'es': u':profesional_de_la_mecánica_hombre_tono_de_piel_claro_medio:',
        'fr': u':mécanicien_peau_moyennement_claire:',
        'pt': u':mecânico_pele_morena_clara:',
        'it': u':meccanico_uomo_carnagione_abbastanza_chiara:'
    },
    u'\U0001F468\U0001F3FD\U0000200D\U0001F527': {
        'en' : u':man_mechanic_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':mechaniker_mittlere_hautfarbe:',
        'es': u':profesional_de_la_mecánica_hombre_tono_de_piel_medio:',
        'fr': u':mécanicien_peau_légèrement_mate:',
        'pt': u':mecânico_pele_morena:',
        'it': u':meccanico_uomo_carnagione_olivastra:'
    },
    u'\U0001F468\U0001F3FE': {
        'en' : u':man_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':mann_mitteldunkle_hautfarbe:',
        'es': u':hombre_tono_de_piel_oscuro_medio:',
        'fr': u':homme_peau_mate:',
        'pt': u':homem_pele_morena_escura:',
        'it': u':uomo_carnagione_abbastanza_scura:'
    },
    u'\U0001F468\U0001F3FE\U0000200D\U0001F9B2': {
        'en' : u':man_medium-dark_skin_tone_bald:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':mann_mitteldunkle_hautfarbe,_glatze:',
        'es': u':hombre_tono_de_piel_oscuro_medio_y_sin_pelo:',
        'fr': u':homme_peau_mate_et_chauve:',
        'pt': u':homem_pele_morena_escura_e_careca:',
        'it': u':uomo_carnagione_abbastanza_scura_e_calvo:'
    },
    u'\U0001F9D4\U0001F3FE\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_medium-dark_skin_tone_beard:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D4\U0001F3FE\U0000200D\U00002642': {
        'en' : u':man_medium-dark_skin_tone_beard:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F471\U0001F3FE\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_medium-dark_skin_tone_blond_hair:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':mann_mitteldunkle_hautfarbe,_blond:',
        'es': u':hombre_rubio_tono_de_piel_oscuro_medio:',
        'fr': u':homme_blond_peau_mate:',
        'pt': u':homem_pele_morena_escura_e_cabelo_loiro:',
        'it': u':uomo_biondo_carnagione_abbastanza_scura:'
    },
    u'\U0001F471\U0001F3FE\U0000200D\U00002642': {
        'en' : u':man_medium-dark_skin_tone_blond_hair:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F468\U0001F3FE\U0000200D\U0001F9B1': {
        'en' : u':man_medium-dark_skin_tone_curly_hair:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':mann_mitteldunkle_hautfarbe,_lockiges_haar:',
        'es': u':hombre_tono_de_piel_oscuro_medio_y_pelo_rizado:',
        'fr': u':homme_peau_mate_et_cheveux_bouclés:',
        'pt': u':homem_pele_morena_escura_e_cabelo_cacheado:',
        'it': u':uomo_carnagione_abbastanza_scura_e_capelli_ricci:'
    },
    u'\U0001F468\U0001F3FE\U0000200D\U0001F9B0': {
        'en' : u':man_medium-dark_skin_tone_red_hair:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':mann_mitteldunkle_hautfarbe,_rotes_haar:',
        'es': u':hombre_tono_de_piel_oscuro_medio_y_pelo_pelirrojo:',
        'fr': u':homme_peau_mate_et_cheveux_roux:',
        'pt': u':homem_pele_morena_escura_e_cabelo_vermelho:',
        'it': u':uomo_carnagione_abbastanza_scura_e_capelli_rossi:'
    },
    u'\U0001F468\U0001F3FE\U0000200D\U0001F9B3': {
        'en' : u':man_medium-dark_skin_tone_white_hair:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':mann_mitteldunkle_hautfarbe,_weißes_haar:',
        'es': u':hombre_tono_de_piel_oscuro_medio_y_pelo_blanco:',
        'fr': u':homme_peau_mate_et_cheveux_blancs:',
        'pt': u':homem_pele_morena_escura_e_cabelo_branco:',
        'it': u':uomo_carnagione_abbastanza_scura_e_capelli_bianchi:'
    },
    u'\U0001F468\U0001F3FC': {
        'en' : u':man_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':mann_mittelhelle_hautfarbe:',
        'es': u':hombre_tono_de_piel_claro_medio:',
        'fr': u':homme_peau_moyennement_claire:',
        'pt': u':homem_pele_morena_clara:',
        'it': u':uomo_carnagione_abbastanza_chiara:'
    },
    u'\U0001F468\U0001F3FC\U0000200D\U0001F9B2': {
        'en' : u':man_medium-light_skin_tone_bald:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':mann_mittelhelle_hautfarbe,_glatze:',
        'es': u':hombre_tono_de_piel_claro_medio_y_sin_pelo:',
        'fr': u':homme_peau_moyennement_claire_et_chauve:',
        'pt': u':homem_pele_morena_clara_e_careca:',
        'it': u':uomo_carnagione_abbastanza_chiara_e_calvo:'
    },
    u'\U0001F9D4\U0001F3FC\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_medium-light_skin_tone_beard:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D4\U0001F3FC\U0000200D\U00002642': {
        'en' : u':man_medium-light_skin_tone_beard:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F471\U0001F3FC\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_medium-light_skin_tone_blond_hair:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':mann_mittelhelle_hautfarbe,_blond:',
        'es': u':hombre_rubio_tono_de_piel_claro_medio:',
        'fr': u':homme_blond_peau_moyennement_claire:',
        'pt': u':homem_pele_morena_clara_e_cabelo_loiro:',
        'it': u':uomo_biondo_carnagione_abbastanza_chiara:'
    },
    u'\U0001F471\U0001F3FC\U0000200D\U00002642': {
        'en' : u':man_medium-light_skin_tone_blond_hair:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F468\U0001F3FC\U0000200D\U0001F9B1': {
        'en' : u':man_medium-light_skin_tone_curly_hair:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':mann_mittelhelle_hautfarbe,_lockiges_haar:',
        'es': u':hombre_tono_de_piel_claro_medio_y_pelo_rizado:',
        'fr': u':homme_peau_moyennement_claire_et_cheveux_bouclés:',
        'pt': u':homem_pele_morena_clara_e_cabelo_cacheado:',
        'it': u':uomo_carnagione_abbastanza_chiara_e_capelli_ricci:'
    },
    u'\U0001F468\U0001F3FC\U0000200D\U0001F9B0': {
        'en' : u':man_medium-light_skin_tone_red_hair:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':mann_mittelhelle_hautfarbe,_rotes_haar:',
        'es': u':hombre_tono_de_piel_claro_medio_y_pelo_pelirrojo:',
        'fr': u':homme_peau_moyennement_claire_et_cheveux_roux:',
        'pt': u':homem_pele_morena_clara_e_cabelo_vermelho:',
        'it': u':uomo_carnagione_abbastanza_chiara_e_capelli_rossi:'
    },
    u'\U0001F468\U0001F3FC\U0000200D\U0001F9B3': {
        'en' : u':man_medium-light_skin_tone_white_hair:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':mann_mittelhelle_hautfarbe,_weißes_haar:',
        'es': u':hombre_tono_de_piel_claro_medio_y_pelo_blanco:',
        'fr': u':homme_peau_moyennement_claire_et_cheveux_blancs:',
        'pt': u':homem_pele_morena_clara_e_cabelo_branco:',
        'it': u':uomo_carnagione_abbastanza_chiara_e_capelli_bianchi:'
    },
    u'\U0001F468\U0001F3FD': {
        'en' : u':man_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':mann_mittlere_hautfarbe:',
        'es': u':hombre_tono_de_piel_medio:',
        'fr': u':homme_peau_légèrement_mate:',
        'pt': u':homem_pele_morena:',
        'it': u':uomo_carnagione_olivastra:'
    },
    u'\U0001F468\U0001F3FD\U0000200D\U0001F9B2': {
        'en' : u':man_medium_skin_tone_bald:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':mann_mittlere_hautfarbe,_glatze:',
        'es': u':hombre_tono_de_piel_medio_y_sin_pelo:',
        'fr': u':homme_peau_légèrement_mate_et_chauve:',
        'pt': u':homem_pele_morena_e_careca:',
        'it': u':uomo_carnagione_olivastra_e_calvo:'
    },
    u'\U0001F9D4\U0001F3FD\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_medium_skin_tone_beard:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D4\U0001F3FD\U0000200D\U00002642': {
        'en' : u':man_medium_skin_tone_beard:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F471\U0001F3FD\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_medium_skin_tone_blond_hair:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':mann_mittlere_hautfarbe,_blond:',
        'es': u':hombre_rubio_tono_de_piel_medio:',
        'fr': u':homme_blond_peau_légèrement_mate:',
        'pt': u':homem_pele_morena_e_cabelo_loiro:',
        'it': u':uomo_biondo_carnagione_olivastra:'
    },
    u'\U0001F471\U0001F3FD\U0000200D\U00002642': {
        'en' : u':man_medium_skin_tone_blond_hair:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F468\U0001F3FD\U0000200D\U0001F9B1': {
        'en' : u':man_medium_skin_tone_curly_hair:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':mann_mittlere_hautfarbe,_lockiges_haar:',
        'es': u':hombre_tono_de_piel_medio_y_pelo_rizado:',
        'fr': u':homme_peau_légèrement_mate_et_cheveux_bouclés:',
        'pt': u':homem_pele_morena_e_cabelo_cacheado:',
        'it': u':uomo_carnagione_olivastra_e_capelli_ricci:'
    },
    u'\U0001F468\U0001F3FD\U0000200D\U0001F9B0': {
        'en' : u':man_medium_skin_tone_red_hair:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':mann_mittlere_hautfarbe,_rotes_haar:',
        'es': u':hombre_tono_de_piel_medio_y_pelo_pelirrojo:',
        'fr': u':homme_peau_légèrement_mate_et_cheveux_roux:',
        'pt': u':homem_pele_morena_e_cabelo_vermelho:',
        'it': u':uomo_carnagione_olivastra_e_capelli_rossi:'
    },
    u'\U0001F468\U0001F3FD\U0000200D\U0001F9B3': {
        'en' : u':man_medium_skin_tone_white_hair:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':mann_mittlere_hautfarbe,_weißes_haar:',
        'es': u':hombre_tono_de_piel_medio_y_pelo_blanco:',
        'fr': u':homme_peau_légèrement_mate_et_cheveux_blancs:',
        'pt': u':homem_pele_morena_e_cabelo_branco:',
        'it': u':uomo_carnagione_olivastra_e_capelli_bianchi:'
    },
    u'\U0001F6B5\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_mountain_biking:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':mountainbiker:',
        'es': u':hombre_en_bicicleta_de_montaña:',
        'fr': u':homme_en_vtt:',
        'pt': u':homem_fazendo_mountain_bike:',
        'it': u':ciclista_uomo_di_mountain_bike:'
    },
    u'\U0001F6B5\U0000200D\U00002642': {
        'en' : u':man_mountain_biking:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F6B5\U0001F3FF\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_mountain_biking_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':mountainbiker_dunkle_hautfarbe:',
        'es': u':hombre_en_bicicleta_de_montaña_tono_de_piel_oscuro:',
        'fr': u':homme_en_vtt_peau_foncée:',
        'pt': u':homem_fazendo_mountain_bike_pele_escura:',
        'it': u':ciclista_uomo_di_mountain_bike_carnagione_scura:'
    },
    u'\U0001F6B5\U0001F3FF\U0000200D\U00002642': {
        'en' : u':man_mountain_biking_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F6B5\U0001F3FB\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_mountain_biking_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':mountainbiker_helle_hautfarbe:',
        'es': u':hombre_en_bicicleta_de_montaña_tono_de_piel_claro:',
        'fr': u':homme_en_vtt_peau_claire:',
        'pt': u':homem_fazendo_mountain_bike_pele_clara:',
        'it': u':ciclista_uomo_di_mountain_bike_carnagione_chiara:'
    },
    u'\U0001F6B5\U0001F3FB\U0000200D\U00002642': {
        'en' : u':man_mountain_biking_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F6B5\U0001F3FE\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_mountain_biking_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':mountainbiker_mitteldunkle_hautfarbe:',
        'es': u':hombre_en_bicicleta_de_montaña_tono_de_piel_oscuro_medio:',
        'fr': u':homme_en_vtt_peau_mate:',
        'pt': u':homem_fazendo_mountain_bike_pele_morena_escura:',
        'it': u':ciclista_uomo_di_mountain_bike_carnagione_abbastanza_scura:'
    },
    u'\U0001F6B5\U0001F3FE\U0000200D\U00002642': {
        'en' : u':man_mountain_biking_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F6B5\U0001F3FC\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_mountain_biking_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':mountainbiker_mittelhelle_hautfarbe:',
        'es': u':hombre_en_bicicleta_de_montaña_tono_de_piel_claro_medio:',
        'fr': u':homme_en_vtt_peau_moyennement_claire:',
        'pt': u':homem_fazendo_mountain_bike_pele_morena_clara:',
        'it': u':ciclista_uomo_di_mountain_bike_carnagione_abbastanza_chiara:'
    },
    u'\U0001F6B5\U0001F3FC\U0000200D\U00002642': {
        'en' : u':man_mountain_biking_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F6B5\U0001F3FD\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_mountain_biking_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':mountainbiker_mittlere_hautfarbe:',
        'es': u':hombre_en_bicicleta_de_montaña_tono_de_piel_medio:',
        'fr': u':homme_en_vtt_peau_légèrement_mate:',
        'pt': u':homem_fazendo_mountain_bike_pele_morena:',
        'it': u':ciclista_uomo_di_mountain_bike_carnagione_olivastra:'
    },
    u'\U0001F6B5\U0001F3FD\U0000200D\U00002642': {
        'en' : u':man_mountain_biking_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F468\U0000200D\U0001F4BC': {
        'en' : u':man_office_worker:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':büroangestellter:',
        'es': u':oficinista_hombre:',
        'fr': u':employé_de_bureau:',
        'pt': u':funcionário_de_escritório:',
        'it': u':impiegato:'
    },
    u'\U0001F468\U0001F3FF\U0000200D\U0001F4BC': {
        'en' : u':man_office_worker_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':büroangestellter_dunkle_hautfarbe:',
        'es': u':oficinista_hombre_tono_de_piel_oscuro:',
        'fr': u':employé_de_bureau_peau_foncée:',
        'pt': u':funcionário_de_escritório_pele_escura:',
        'it': u':impiegato_carnagione_scura:'
    },
    u'\U0001F468\U0001F3FB\U0000200D\U0001F4BC': {
        'en' : u':man_office_worker_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':büroangestellter_helle_hautfarbe:',
        'es': u':oficinista_hombre_tono_de_piel_claro:',
        'fr': u':employé_de_bureau_peau_claire:',
        'pt': u':funcionário_de_escritório_pele_clara:',
        'it': u':impiegato_carnagione_chiara:'
    },
    u'\U0001F468\U0001F3FE\U0000200D\U0001F4BC': {
        'en' : u':man_office_worker_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':büroangestellter_mitteldunkle_hautfarbe:',
        'es': u':oficinista_hombre_tono_de_piel_oscuro_medio:',
        'fr': u':employé_de_bureau_peau_mate:',
        'pt': u':funcionário_de_escritório_pele_morena_escura:',
        'it': u':impiegato_carnagione_abbastanza_scura:'
    },
    u'\U0001F468\U0001F3FC\U0000200D\U0001F4BC': {
        'en' : u':man_office_worker_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':büroangestellter_mittelhelle_hautfarbe:',
        'es': u':oficinista_hombre_tono_de_piel_claro_medio:',
        'fr': u':employé_de_bureau_peau_moyennement_claire:',
        'pt': u':funcionário_de_escritório_pele_morena_clara:',
        'it': u':impiegato_carnagione_abbastanza_chiara:'
    },
    u'\U0001F468\U0001F3FD\U0000200D\U0001F4BC': {
        'en' : u':man_office_worker_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':büroangestellter_mittlere_hautfarbe:',
        'es': u':oficinista_hombre_tono_de_piel_medio:',
        'fr': u':employé_de_bureau_peau_légèrement_mate:',
        'pt': u':funcionário_de_escritório_pele_morena:',
        'it': u':impiegato_carnagione_olivastra:'
    },
    u'\U0001F468\U0000200D\U00002708\U0000FE0F': {
        'en' : u':man_pilot:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':pilot:',
        'es': u':piloto_hombre:',
        'fr': u':pilote_homme:',
        'pt': u':piloto_de_avião_homem:',
        'it': u':pilota_uomo:'
    },
    u'\U0001F468\U0000200D\U00002708': {
        'en' : u':man_pilot:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F468\U0001F3FF\U0000200D\U00002708\U0000FE0F': {
        'en' : u':man_pilot_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':pilot_dunkle_hautfarbe:',
        'es': u':piloto_hombre_tono_de_piel_oscuro:',
        'fr': u':pilote_homme_peau_foncée:',
        'pt': u':piloto_de_avião_homem_pele_escura:',
        'it': u':pilota_uomo_carnagione_scura:'
    },
    u'\U0001F468\U0001F3FF\U0000200D\U00002708': {
        'en' : u':man_pilot_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F468\U0001F3FB\U0000200D\U00002708\U0000FE0F': {
        'en' : u':man_pilot_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':pilot_helle_hautfarbe:',
        'es': u':piloto_hombre_tono_de_piel_claro:',
        'fr': u':pilote_homme_peau_claire:',
        'pt': u':piloto_de_avião_homem_pele_clara:',
        'it': u':pilota_uomo_carnagione_chiara:'
    },
    u'\U0001F468\U0001F3FB\U0000200D\U00002708': {
        'en' : u':man_pilot_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F468\U0001F3FE\U0000200D\U00002708\U0000FE0F': {
        'en' : u':man_pilot_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':pilot_mitteldunkle_hautfarbe:',
        'es': u':piloto_hombre_tono_de_piel_oscuro_medio:',
        'fr': u':pilote_homme_peau_mate:',
        'pt': u':piloto_de_avião_homem_pele_morena_escura:',
        'it': u':pilota_uomo_carnagione_abbastanza_scura:'
    },
    u'\U0001F468\U0001F3FE\U0000200D\U00002708': {
        'en' : u':man_pilot_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F468\U0001F3FC\U0000200D\U00002708\U0000FE0F': {
        'en' : u':man_pilot_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':pilot_mittelhelle_hautfarbe:',
        'es': u':piloto_hombre_tono_de_piel_claro_medio:',
        'fr': u':pilote_homme_peau_moyennement_claire:',
        'pt': u':piloto_de_avião_homem_pele_morena_clara:',
        'it': u':pilota_uomo_carnagione_abbastanza_chiara:'
    },
    u'\U0001F468\U0001F3FC\U0000200D\U00002708': {
        'en' : u':man_pilot_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F468\U0001F3FD\U0000200D\U00002708\U0000FE0F': {
        'en' : u':man_pilot_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':pilot_mittlere_hautfarbe:',
        'es': u':piloto_hombre_tono_de_piel_medio:',
        'fr': u':pilote_homme_peau_légèrement_mate:',
        'pt': u':piloto_de_avião_homem_pele_morena:',
        'it': u':pilota_uomo_carnagione_olivastra:'
    },
    u'\U0001F468\U0001F3FD\U0000200D\U00002708': {
        'en' : u':man_pilot_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F93E\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_playing_handball:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':handballspieler:',
        'es': u':hombre_jugando_al_balonmano:',
        'fr': u':handballeur:',
        'pt': u':jogador_de_handebol:',
        'it': u':pallamanista_uomo:'
    },
    u'\U0001F93E\U0000200D\U00002642': {
        'en' : u':man_playing_handball:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F93E\U0001F3FF\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_playing_handball_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':handballspieler_dunkle_hautfarbe:',
        'es': u':hombre_jugando_al_balonmano_tono_de_piel_oscuro:',
        'fr': u':handballeur_peau_foncée:',
        'pt': u':jogador_de_handebol_pele_escura:',
        'it': u':pallamanista_uomo_carnagione_scura:'
    },
    u'\U0001F93E\U0001F3FF\U0000200D\U00002642': {
        'en' : u':man_playing_handball_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F93E\U0001F3FB\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_playing_handball_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':handballspieler_helle_hautfarbe:',
        'es': u':hombre_jugando_al_balonmano_tono_de_piel_claro:',
        'fr': u':handballeur_peau_claire:',
        'pt': u':jogador_de_handebol_pele_clara:',
        'it': u':pallamanista_uomo_carnagione_chiara:'
    },
    u'\U0001F93E\U0001F3FB\U0000200D\U00002642': {
        'en' : u':man_playing_handball_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F93E\U0001F3FE\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_playing_handball_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':handballspieler_mitteldunkle_hautfarbe:',
        'es': u':hombre_jugando_al_balonmano_tono_de_piel_oscuro_medio:',
        'fr': u':handballeur_peau_mate:',
        'pt': u':jogador_de_handebol_pele_morena_escura:',
        'it': u':pallamanista_uomo_carnagione_abbastanza_scura:'
    },
    u'\U0001F93E\U0001F3FE\U0000200D\U00002642': {
        'en' : u':man_playing_handball_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F93E\U0001F3FC\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_playing_handball_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':handballspieler_mittelhelle_hautfarbe:',
        'es': u':hombre_jugando_al_balonmano_tono_de_piel_claro_medio:',
        'fr': u':handballeur_peau_moyennement_claire:',
        'pt': u':jogador_de_handebol_pele_morena_clara:',
        'it': u':pallamanista_uomo_carnagione_abbastanza_chiara:'
    },
    u'\U0001F93E\U0001F3FC\U0000200D\U00002642': {
        'en' : u':man_playing_handball_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F93E\U0001F3FD\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_playing_handball_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':handballspieler_mittlere_hautfarbe:',
        'es': u':hombre_jugando_al_balonmano_tono_de_piel_medio:',
        'fr': u':handballeur_peau_légèrement_mate:',
        'pt': u':jogador_de_handebol_pele_morena:',
        'it': u':pallamanista_uomo_carnagione_olivastra:'
    },
    u'\U0001F93E\U0001F3FD\U0000200D\U00002642': {
        'en' : u':man_playing_handball_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F93D\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_playing_water_polo:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':wasserballspieler:',
        'es': u':hombre_jugando_al_waterpolo:',
        'fr': u':joueur_de_water-polo:',
        'pt': u':homem_jogando_polo_aquático:',
        'it': u':pallanuotista_uomo:'
    },
    u'\U0001F93D\U0000200D\U00002642': {
        'en' : u':man_playing_water_polo:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F93D\U0001F3FF\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_playing_water_polo_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':wasserballspieler_dunkle_hautfarbe:',
        'es': u':hombre_jugando_al_waterpolo_tono_de_piel_oscuro:',
        'fr': u':joueur_de_water-polo_peau_foncée:',
        'pt': u':homem_jogando_polo_aquático_pele_escura:',
        'it': u':pallanuotista_uomo_carnagione_scura:'
    },
    u'\U0001F93D\U0001F3FF\U0000200D\U00002642': {
        'en' : u':man_playing_water_polo_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F93D\U0001F3FB\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_playing_water_polo_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':wasserballspieler_helle_hautfarbe:',
        'es': u':hombre_jugando_al_waterpolo_tono_de_piel_claro:',
        'fr': u':joueur_de_water-polo_peau_claire:',
        'pt': u':homem_jogando_polo_aquático_pele_clara:',
        'it': u':pallanuotista_uomo_carnagione_chiara:'
    },
    u'\U0001F93D\U0001F3FB\U0000200D\U00002642': {
        'en' : u':man_playing_water_polo_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F93D\U0001F3FE\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_playing_water_polo_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':wasserballspieler_mitteldunkle_hautfarbe:',
        'es': u':hombre_jugando_al_waterpolo_tono_de_piel_oscuro_medio:',
        'fr': u':joueur_de_water-polo_peau_mate:',
        'pt': u':homem_jogando_polo_aquático_pele_morena_escura:',
        'it': u':pallanuotista_uomo_carnagione_abbastanza_scura:'
    },
    u'\U0001F93D\U0001F3FE\U0000200D\U00002642': {
        'en' : u':man_playing_water_polo_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F93D\U0001F3FC\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_playing_water_polo_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':wasserballspieler_mittelhelle_hautfarbe:',
        'es': u':hombre_jugando_al_waterpolo_tono_de_piel_claro_medio:',
        'fr': u':joueur_de_water-polo_peau_moyennement_claire:',
        'pt': u':homem_jogando_polo_aquático_pele_morena_clara:',
        'it': u':pallanuotista_uomo_carnagione_abbastanza_chiara:'
    },
    u'\U0001F93D\U0001F3FC\U0000200D\U00002642': {
        'en' : u':man_playing_water_polo_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F93D\U0001F3FD\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_playing_water_polo_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':wasserballspieler_mittlere_hautfarbe:',
        'es': u':hombre_jugando_al_waterpolo_tono_de_piel_medio:',
        'fr': u':joueur_de_water-polo_peau_légèrement_mate:',
        'pt': u':homem_jogando_polo_aquático_pele_morena:',
        'it': u':pallanuotista_uomo_carnagione_olivastra:'
    },
    u'\U0001F93D\U0001F3FD\U0000200D\U00002642': {
        'en' : u':man_playing_water_polo_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F46E\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_police_officer:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':polizist:',
        'es': u':agente_de_policía_hombre:',
        'fr': u':policier:',
        'pt': u':policial_homem:',
        'it': u':poliziotto_uomo:'
    },
    u'\U0001F46E\U0000200D\U00002642': {
        'en' : u':man_police_officer:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F46E\U0001F3FF\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_police_officer_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':polizist_dunkle_hautfarbe:',
        'es': u':agente_de_policía_hombre_tono_de_piel_oscuro:',
        'fr': u':policier_peau_foncée:',
        'pt': u':policial_homem_pele_escura:',
        'it': u':poliziotto_uomo_carnagione_scura:'
    },
    u'\U0001F46E\U0001F3FF\U0000200D\U00002642': {
        'en' : u':man_police_officer_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F46E\U0001F3FB\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_police_officer_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':polizist_helle_hautfarbe:',
        'es': u':agente_de_policía_hombre_tono_de_piel_claro:',
        'fr': u':policier_peau_claire:',
        'pt': u':policial_homem_pele_clara:',
        'it': u':poliziotto_uomo_carnagione_chiara:'
    },
    u'\U0001F46E\U0001F3FB\U0000200D\U00002642': {
        'en' : u':man_police_officer_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F46E\U0001F3FE\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_police_officer_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':polizist_mitteldunkle_hautfarbe:',
        'es': u':agente_de_policía_hombre_tono_de_piel_oscuro_medio:',
        'fr': u':policier_peau_mate:',
        'pt': u':policial_homem_pele_morena_escura:',
        'it': u':poliziotto_uomo_carnagione_abbastanza_scura:'
    },
    u'\U0001F46E\U0001F3FE\U0000200D\U00002642': {
        'en' : u':man_police_officer_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F46E\U0001F3FC\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_police_officer_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':polizist_mittelhelle_hautfarbe:',
        'es': u':agente_de_policía_hombre_tono_de_piel_claro_medio:',
        'fr': u':policier_peau_moyennement_claire:',
        'pt': u':policial_homem_pele_morena_clara:',
        'it': u':poliziotto_uomo_carnagione_abbastanza_chiara:'
    },
    u'\U0001F46E\U0001F3FC\U0000200D\U00002642': {
        'en' : u':man_police_officer_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F46E\U0001F3FD\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_police_officer_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':polizist_mittlere_hautfarbe:',
        'es': u':agente_de_policía_hombre_tono_de_piel_medio:',
        'fr': u':policier_peau_légèrement_mate:',
        'pt': u':policial_homem_pele_morena:',
        'it': u':poliziotto_uomo_carnagione_olivastra:'
    },
    u'\U0001F46E\U0001F3FD\U0000200D\U00002642': {
        'en' : u':man_police_officer_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F64E\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_pouting:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':schmollender_mann:',
        'es': u':hombre_haciendo_pucheros:',
        'fr': u':homme_qui_boude:',
        'pt': u':homem_fazendo_bico:',
        'it': u':uomo_imbronciato:'
    },
    u'\U0001F64E\U0000200D\U00002642': {
        'en' : u':man_pouting:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F64E\U0001F3FF\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_pouting_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':schmollender_mann_dunkle_hautfarbe:',
        'es': u':hombre_haciendo_pucheros_tono_de_piel_oscuro:',
        'fr': u':homme_qui_boude_peau_foncée:',
        'pt': u':homem_fazendo_bico_pele_escura:',
        'it': u':uomo_imbronciato_carnagione_scura:'
    },
    u'\U0001F64E\U0001F3FF\U0000200D\U00002642': {
        'en' : u':man_pouting_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F64E\U0001F3FB\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_pouting_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':schmollender_mann_helle_hautfarbe:',
        'es': u':hombre_haciendo_pucheros_tono_de_piel_claro:',
        'fr': u':homme_qui_boude_peau_claire:',
        'pt': u':homem_fazendo_bico_pele_clara:',
        'it': u':uomo_imbronciato_carnagione_chiara:'
    },
    u'\U0001F64E\U0001F3FB\U0000200D\U00002642': {
        'en' : u':man_pouting_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F64E\U0001F3FE\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_pouting_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':schmollender_mann_mitteldunkle_hautfarbe:',
        'es': u':hombre_haciendo_pucheros_tono_de_piel_oscuro_medio:',
        'fr': u':homme_qui_boude_peau_mate:',
        'pt': u':homem_fazendo_bico_pele_morena_escura:',
        'it': u':uomo_imbronciato_carnagione_abbastanza_scura:'
    },
    u'\U0001F64E\U0001F3FE\U0000200D\U00002642': {
        'en' : u':man_pouting_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F64E\U0001F3FC\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_pouting_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':schmollender_mann_mittelhelle_hautfarbe:',
        'es': u':hombre_haciendo_pucheros_tono_de_piel_claro_medio:',
        'fr': u':homme_qui_boude_peau_moyennement_claire:',
        'pt': u':homem_fazendo_bico_pele_morena_clara:',
        'it': u':uomo_imbronciato_carnagione_abbastanza_chiara:'
    },
    u'\U0001F64E\U0001F3FC\U0000200D\U00002642': {
        'en' : u':man_pouting_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F64E\U0001F3FD\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_pouting_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':schmollender_mann_mittlere_hautfarbe:',
        'es': u':hombre_haciendo_pucheros_tono_de_piel_medio:',
        'fr': u':homme_qui_boude_peau_légèrement_mate:',
        'pt': u':homem_fazendo_bico_pele_morena:',
        'it': u':uomo_imbronciato_carnagione_olivastra:'
    },
    u'\U0001F64E\U0001F3FD\U0000200D\U00002642': {
        'en' : u':man_pouting_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F64B\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_raising_hand:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':mann_mit_erhobenem_arm:',
        'es': u':hombre_con_la_mano_levantada:',
        'fr': u':homme_qui_lève_la_main:',
        'pt': u':homem_levantando_a_mão:',
        'it': u':uomo_con_mano_alzata:'
    },
    u'\U0001F64B\U0000200D\U00002642': {
        'en' : u':man_raising_hand:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F64B\U0001F3FF\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_raising_hand_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':mann_mit_erhobenem_arm_dunkle_hautfarbe:',
        'es': u':hombre_con_la_mano_levantada_tono_de_piel_oscuro:',
        'fr': u':homme_qui_lève_la_main_peau_foncée:',
        'pt': u':homem_levantando_a_mão_pele_escura:',
        'it': u':uomo_con_mano_alzata_carnagione_scura:'
    },
    u'\U0001F64B\U0001F3FF\U0000200D\U00002642': {
        'en' : u':man_raising_hand_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F64B\U0001F3FB\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_raising_hand_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':mann_mit_erhobenem_arm_helle_hautfarbe:',
        'es': u':hombre_con_la_mano_levantada_tono_de_piel_claro:',
        'fr': u':homme_qui_lève_la_main_peau_claire:',
        'pt': u':homem_levantando_a_mão_pele_clara:',
        'it': u':uomo_con_mano_alzata_carnagione_chiara:'
    },
    u'\U0001F64B\U0001F3FB\U0000200D\U00002642': {
        'en' : u':man_raising_hand_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F64B\U0001F3FE\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_raising_hand_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':mann_mit_erhobenem_arm_mitteldunkle_hautfarbe:',
        'es': u':hombre_con_la_mano_levantada_tono_de_piel_oscuro_medio:',
        'fr': u':homme_qui_lève_la_main_peau_mate:',
        'pt': u':homem_levantando_a_mão_pele_morena_escura:',
        'it': u':uomo_con_mano_alzata_carnagione_abbastanza_scura:'
    },
    u'\U0001F64B\U0001F3FE\U0000200D\U00002642': {
        'en' : u':man_raising_hand_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F64B\U0001F3FC\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_raising_hand_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':mann_mit_erhobenem_arm_mittelhelle_hautfarbe:',
        'es': u':hombre_con_la_mano_levantada_tono_de_piel_claro_medio:',
        'fr': u':homme_qui_lève_la_main_peau_moyennement_claire:',
        'pt': u':homem_levantando_a_mão_pele_morena_clara:',
        'it': u':uomo_con_mano_alzata_carnagione_abbastanza_chiara:'
    },
    u'\U0001F64B\U0001F3FC\U0000200D\U00002642': {
        'en' : u':man_raising_hand_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F64B\U0001F3FD\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_raising_hand_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':mann_mit_erhobenem_arm_mittlere_hautfarbe:',
        'es': u':hombre_con_la_mano_levantada_tono_de_piel_medio:',
        'fr': u':homme_qui_lève_la_main_peau_légèrement_mate:',
        'pt': u':homem_levantando_a_mão_pele_morena:',
        'it': u':uomo_con_mano_alzata_carnagione_olivastra:'
    },
    u'\U0001F64B\U0001F3FD\U0000200D\U00002642': {
        'en' : u':man_raising_hand_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F468\U0000200D\U0001F9B0': {
        'en' : u':man_red_hair:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':mann_rotes_haar:',
        'es': u':hombre_pelo_pelirrojo:',
        'fr': u':homme_cheveux_roux:',
        'pt': u':homem_cabelo_vermelho:',
        'it': u':uomo_capelli_rossi:'
    },
    u'\U0001F6A3\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_rowing_boat:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':mann_im_ruderboot:',
        'es': u':hombre_remando_en_un_bote:',
        'fr': u':rameur_dans_une_barque:',
        'pt': u':homem_remando:',
        'it': u':uomo_in_barca_a_remi:'
    },
    u'\U0001F6A3\U0000200D\U00002642': {
        'en' : u':man_rowing_boat:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F6A3\U0001F3FF\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_rowing_boat_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':mann_im_ruderboot_dunkle_hautfarbe:',
        'es': u':hombre_remando_en_un_bote_tono_de_piel_oscuro:',
        'fr': u':rameur_dans_une_barque_peau_foncée:',
        'pt': u':homem_remando_pele_escura:',
        'it': u':uomo_in_barca_a_remi_carnagione_scura:'
    },
    u'\U0001F6A3\U0001F3FF\U0000200D\U00002642': {
        'en' : u':man_rowing_boat_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F6A3\U0001F3FB\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_rowing_boat_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':mann_im_ruderboot_helle_hautfarbe:',
        'es': u':hombre_remando_en_un_bote_tono_de_piel_claro:',
        'fr': u':rameur_dans_une_barque_peau_claire:',
        'pt': u':homem_remando_pele_clara:',
        'it': u':uomo_in_barca_a_remi_carnagione_chiara:'
    },
    u'\U0001F6A3\U0001F3FB\U0000200D\U00002642': {
        'en' : u':man_rowing_boat_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F6A3\U0001F3FE\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_rowing_boat_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':mann_im_ruderboot_mitteldunkle_hautfarbe:',
        'es': u':hombre_remando_en_un_bote_tono_de_piel_oscuro_medio:',
        'fr': u':rameur_dans_une_barque_peau_mate:',
        'pt': u':homem_remando_pele_morena_escura:',
        'it': u':uomo_in_barca_a_remi_carnagione_abbastanza_scura:'
    },
    u'\U0001F6A3\U0001F3FE\U0000200D\U00002642': {
        'en' : u':man_rowing_boat_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F6A3\U0001F3FC\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_rowing_boat_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':mann_im_ruderboot_mittelhelle_hautfarbe:',
        'es': u':hombre_remando_en_un_bote_tono_de_piel_claro_medio:',
        'fr': u':rameur_dans_une_barque_peau_moyennement_claire:',
        'pt': u':homem_remando_pele_morena_clara:',
        'it': u':uomo_in_barca_a_remi_carnagione_abbastanza_chiara:'
    },
    u'\U0001F6A3\U0001F3FC\U0000200D\U00002642': {
        'en' : u':man_rowing_boat_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F6A3\U0001F3FD\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_rowing_boat_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':mann_im_ruderboot_mittlere_hautfarbe:',
        'es': u':hombre_remando_en_un_bote_tono_de_piel_medio:',
        'fr': u':rameur_dans_une_barque_peau_légèrement_mate:',
        'pt': u':homem_remando_pele_morena:',
        'it': u':uomo_in_barca_a_remi_carnagione_olivastra:'
    },
    u'\U0001F6A3\U0001F3FD\U0000200D\U00002642': {
        'en' : u':man_rowing_boat_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F3C3\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_running:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':laufender_mann:',
        'es': u':hombre_corriendo:',
        'fr': u':homme_qui_court:',
        'pt': u':homem_correndo:',
        'it': u':uomo_che_corre:'
    },
    u'\U0001F3C3\U0000200D\U00002642': {
        'en' : u':man_running:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F3C3\U0001F3FF\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_running_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':laufender_mann_dunkle_hautfarbe:',
        'es': u':hombre_corriendo_tono_de_piel_oscuro:',
        'fr': u':homme_qui_court_peau_foncée:',
        'pt': u':homem_correndo_pele_escura:',
        'it': u':uomo_che_corre_carnagione_scura:'
    },
    u'\U0001F3C3\U0001F3FF\U0000200D\U00002642': {
        'en' : u':man_running_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F3C3\U0001F3FB\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_running_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':laufender_mann_helle_hautfarbe:',
        'es': u':hombre_corriendo_tono_de_piel_claro:',
        'fr': u':homme_qui_court_peau_claire:',
        'pt': u':homem_correndo_pele_clara:',
        'it': u':uomo_che_corre_carnagione_chiara:'
    },
    u'\U0001F3C3\U0001F3FB\U0000200D\U00002642': {
        'en' : u':man_running_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F3C3\U0001F3FE\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_running_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':laufender_mann_mitteldunkle_hautfarbe:',
        'es': u':hombre_corriendo_tono_de_piel_oscuro_medio:',
        'fr': u':homme_qui_court_peau_mate:',
        'pt': u':homem_correndo_pele_morena_escura:',
        'it': u':uomo_che_corre_carnagione_abbastanza_scura:'
    },
    u'\U0001F3C3\U0001F3FE\U0000200D\U00002642': {
        'en' : u':man_running_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F3C3\U0001F3FC\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_running_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':laufender_mann_mittelhelle_hautfarbe:',
        'es': u':hombre_corriendo_tono_de_piel_claro_medio:',
        'fr': u':homme_qui_court_peau_moyennement_claire:',
        'pt': u':homem_correndo_pele_morena_clara:',
        'it': u':uomo_che_corre_carnagione_abbastanza_chiara:'
    },
    u'\U0001F3C3\U0001F3FC\U0000200D\U00002642': {
        'en' : u':man_running_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F3C3\U0001F3FD\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_running_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':laufender_mann_mittlere_hautfarbe:',
        'es': u':hombre_corriendo_tono_de_piel_medio:',
        'fr': u':homme_qui_court_peau_légèrement_mate:',
        'pt': u':homem_correndo_pele_morena:',
        'it': u':uomo_che_corre_carnagione_olivastra:'
    },
    u'\U0001F3C3\U0001F3FD\U0000200D\U00002642': {
        'en' : u':man_running_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F468\U0000200D\U0001F52C': {
        'en' : u':man_scientist:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':wissenschaftler:',
        'es': u':profesional_de_la_ciencia_hombre:',
        'fr': u':scientifique_homme:',
        'pt': u':cientista_homem:',
        'it': u':scienziato:'
    },
    u'\U0001F468\U0001F3FF\U0000200D\U0001F52C': {
        'en' : u':man_scientist_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':wissenschaftler_dunkle_hautfarbe:',
        'es': u':profesional_de_la_ciencia_hombre_tono_de_piel_oscuro:',
        'fr': u':scientifique_homme_peau_foncée:',
        'pt': u':cientista_homem_pele_escura:',
        'it': u':scienziato_carnagione_scura:'
    },
    u'\U0001F468\U0001F3FB\U0000200D\U0001F52C': {
        'en' : u':man_scientist_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':wissenschaftler_helle_hautfarbe:',
        'es': u':profesional_de_la_ciencia_hombre_tono_de_piel_claro:',
        'fr': u':scientifique_homme_peau_claire:',
        'pt': u':cientista_homem_pele_clara:',
        'it': u':scienziato_carnagione_chiara:'
    },
    u'\U0001F468\U0001F3FE\U0000200D\U0001F52C': {
        'en' : u':man_scientist_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':wissenschaftler_mitteldunkle_hautfarbe:',
        'es': u':profesional_de_la_ciencia_hombre_tono_de_piel_oscuro_medio:',
        'fr': u':scientifique_homme_peau_mate:',
        'pt': u':cientista_homem_pele_morena_escura:',
        'it': u':scienziato_carnagione_abbastanza_scura:'
    },
    u'\U0001F468\U0001F3FC\U0000200D\U0001F52C': {
        'en' : u':man_scientist_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':wissenschaftler_mittelhelle_hautfarbe:',
        'es': u':profesional_de_la_ciencia_hombre_tono_de_piel_claro_medio:',
        'fr': u':scientifique_homme_peau_moyennement_claire:',
        'pt': u':cientista_homem_pele_morena_clara:',
        'it': u':scienziato_carnagione_abbastanza_chiara:'
    },
    u'\U0001F468\U0001F3FD\U0000200D\U0001F52C': {
        'en' : u':man_scientist_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':wissenschaftler_mittlere_hautfarbe:',
        'es': u':profesional_de_la_ciencia_hombre_tono_de_piel_medio:',
        'fr': u':scientifique_homme_peau_légèrement_mate:',
        'pt': u':cientista_homem_pele_morena:',
        'it': u':scienziato_carnagione_olivastra:'
    },
    u'\U0001F937\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_shrugging:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':schulterzuckender_mann:',
        'es': u':hombre_encogido_de_hombros:',
        'fr': u':homme_qui_hausse_les_épaules:',
        'pt': u':homem_dando_de_ombros:',
        'it': u':uomo_che_scrolla_le_spalle:'
    },
    u'\U0001F937\U0000200D\U00002642': {
        'en' : u':man_shrugging:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F937\U0001F3FF\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_shrugging_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':schulterzuckender_mann_dunkle_hautfarbe:',
        'es': u':hombre_encogido_de_hombros_tono_de_piel_oscuro:',
        'fr': u':homme_qui_hausse_les_épaules_peau_foncée:',
        'pt': u':homem_dando_de_ombros_pele_escura:',
        'it': u':uomo_che_scrolla_le_spalle_carnagione_scura:'
    },
    u'\U0001F937\U0001F3FF\U0000200D\U00002642': {
        'en' : u':man_shrugging_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F937\U0001F3FB\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_shrugging_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':schulterzuckender_mann_helle_hautfarbe:',
        'es': u':hombre_encogido_de_hombros_tono_de_piel_claro:',
        'fr': u':homme_qui_hausse_les_épaules_peau_claire:',
        'pt': u':homem_dando_de_ombros_pele_clara:',
        'it': u':uomo_che_scrolla_le_spalle_carnagione_chiara:'
    },
    u'\U0001F937\U0001F3FB\U0000200D\U00002642': {
        'en' : u':man_shrugging_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F937\U0001F3FE\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_shrugging_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':schulterzuckender_mann_mitteldunkle_hautfarbe:',
        'es': u':hombre_encogido_de_hombros_tono_de_piel_oscuro_medio:',
        'fr': u':homme_qui_hausse_les_épaules_peau_mate:',
        'pt': u':homem_dando_de_ombros_pele_morena_escura:',
        'it': u':uomo_che_scrolla_le_spalle_carnagione_abbastanza_scura:'
    },
    u'\U0001F937\U0001F3FE\U0000200D\U00002642': {
        'en' : u':man_shrugging_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F937\U0001F3FC\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_shrugging_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':schulterzuckender_mann_mittelhelle_hautfarbe:',
        'es': u':hombre_encogido_de_hombros_tono_de_piel_claro_medio:',
        'fr': u':homme_qui_hausse_les_épaules_peau_moyennement_claire:',
        'pt': u':homem_dando_de_ombros_pele_morena_clara:',
        'it': u':uomo_che_scrolla_le_spalle_carnagione_abbastanza_chiara:'
    },
    u'\U0001F937\U0001F3FC\U0000200D\U00002642': {
        'en' : u':man_shrugging_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F937\U0001F3FD\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_shrugging_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':schulterzuckender_mann_mittlere_hautfarbe:',
        'es': u':hombre_encogido_de_hombros_tono_de_piel_medio:',
        'fr': u':homme_qui_hausse_les_épaules_peau_légèrement_mate:',
        'pt': u':homem_dando_de_ombros_pele_morena:',
        'it': u':uomo_che_scrolla_le_spalle_carnagione_olivastra:'
    },
    u'\U0001F937\U0001F3FD\U0000200D\U00002642': {
        'en' : u':man_shrugging_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F468\U0000200D\U0001F3A4': {
        'en' : u':man_singer:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':sänger:',
        'es': u':cantante_hombre:',
        'fr': u':chanteur:',
        'pt': u':cantor_homem:',
        'it': u':cantante_uomo:'
    },
    u'\U0001F468\U0001F3FF\U0000200D\U0001F3A4': {
        'en' : u':man_singer_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':sänger_dunkle_hautfarbe:',
        'es': u':cantante_hombre_tono_de_piel_oscuro:',
        'fr': u':chanteur_peau_foncée:',
        'pt': u':cantor_homem_pele_escura:',
        'it': u':cantante_uomo_carnagione_scura:'
    },
    u'\U0001F468\U0001F3FB\U0000200D\U0001F3A4': {
        'en' : u':man_singer_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':sänger_helle_hautfarbe:',
        'es': u':cantante_hombre_tono_de_piel_claro:',
        'fr': u':chanteur_peau_claire:',
        'pt': u':cantor_homem_pele_clara:',
        'it': u':cantante_uomo_carnagione_chiara:'
    },
    u'\U0001F468\U0001F3FE\U0000200D\U0001F3A4': {
        'en' : u':man_singer_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':sänger_mitteldunkle_hautfarbe:',
        'es': u':cantante_hombre_tono_de_piel_oscuro_medio:',
        'fr': u':chanteur_peau_mate:',
        'pt': u':cantor_homem_pele_morena_escura:',
        'it': u':cantante_uomo_carnagione_abbastanza_scura:'
    },
    u'\U0001F468\U0001F3FC\U0000200D\U0001F3A4': {
        'en' : u':man_singer_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':sänger_mittelhelle_hautfarbe:',
        'es': u':cantante_hombre_tono_de_piel_claro_medio:',
        'fr': u':chanteur_peau_moyennement_claire:',
        'pt': u':cantor_homem_pele_morena_clara:',
        'it': u':cantante_uomo_carnagione_abbastanza_chiara:'
    },
    u'\U0001F468\U0001F3FD\U0000200D\U0001F3A4': {
        'en' : u':man_singer_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':sänger_mittlere_hautfarbe:',
        'es': u':cantante_hombre_tono_de_piel_medio:',
        'fr': u':chanteur_peau_légèrement_mate:',
        'pt': u':cantor_homem_pele_morena:',
        'it': u':cantante_uomo_carnagione_olivastra:'
    },
    u'\U0001F9CD\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_standing:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':stehender_mann:',
        'es': u':hombre_de_pie:',
        'fr': u':homme_debout:',
        'pt': u':homem_em_pé:',
        'it': u':uomo_in_piedi:'
    },
    u'\U0001F9CD\U0000200D\U00002642': {
        'en' : u':man_standing:',
        'status' : minimally_qualified,
        'E' : 12
    },
    u'\U0001F9CD\U0001F3FF\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_standing_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':stehender_mann_dunkle_hautfarbe:',
        'es': u':hombre_de_pie_tono_de_piel_oscuro:',
        'fr': u':homme_debout_peau_foncée:',
        'pt': u':homem_em_pé_pele_escura:',
        'it': u':uomo_in_piedi_carnagione_scura:'
    },
    u'\U0001F9CD\U0001F3FF\U0000200D\U00002642': {
        'en' : u':man_standing_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 12
    },
    u'\U0001F9CD\U0001F3FB\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_standing_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':stehender_mann_helle_hautfarbe:',
        'es': u':hombre_de_pie_tono_de_piel_claro:',
        'fr': u':homme_debout_peau_claire:',
        'pt': u':homem_em_pé_pele_clara:',
        'it': u':uomo_in_piedi_carnagione_chiara:'
    },
    u'\U0001F9CD\U0001F3FB\U0000200D\U00002642': {
        'en' : u':man_standing_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 12
    },
    u'\U0001F9CD\U0001F3FE\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_standing_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':stehender_mann_mitteldunkle_hautfarbe:',
        'es': u':hombre_de_pie_tono_de_piel_oscuro_medio:',
        'fr': u':homme_debout_peau_mate:',
        'pt': u':homem_em_pé_pele_morena_escura:',
        'it': u':uomo_in_piedi_carnagione_abbastanza_scura:'
    },
    u'\U0001F9CD\U0001F3FE\U0000200D\U00002642': {
        'en' : u':man_standing_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 12
    },
    u'\U0001F9CD\U0001F3FC\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_standing_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':stehender_mann_mittelhelle_hautfarbe:',
        'es': u':hombre_de_pie_tono_de_piel_claro_medio:',
        'fr': u':homme_debout_peau_moyennement_claire:',
        'pt': u':homem_em_pé_pele_morena_clara:',
        'it': u':uomo_in_piedi_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9CD\U0001F3FC\U0000200D\U00002642': {
        'en' : u':man_standing_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 12
    },
    u'\U0001F9CD\U0001F3FD\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_standing_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':stehender_mann_mittlere_hautfarbe:',
        'es': u':hombre_de_pie_tono_de_piel_medio:',
        'fr': u':homme_debout_peau_légèrement_mate:',
        'pt': u':homem_em_pé_pele_morena:',
        'it': u':uomo_in_piedi_carnagione_olivastra:'
    },
    u'\U0001F9CD\U0001F3FD\U0000200D\U00002642': {
        'en' : u':man_standing_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 12
    },
    u'\U0001F468\U0000200D\U0001F393': {
        'en' : u':man_student:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':student:',
        'es': u':estudiante_hombre:',
        'fr': u':étudiant:',
        'pt': u':estudante:',
        'it': u':studente_maschio:'
    },
    u'\U0001F468\U0001F3FF\U0000200D\U0001F393': {
        'en' : u':man_student_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':student_dunkle_hautfarbe:',
        'es': u':estudiante_hombre_tono_de_piel_oscuro:',
        'fr': u':étudiant_peau_foncée:',
        'pt': u':estudante_pele_escura:',
        'it': u':studente_maschio_carnagione_scura:'
    },
    u'\U0001F468\U0001F3FB\U0000200D\U0001F393': {
        'en' : u':man_student_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':student_helle_hautfarbe:',
        'es': u':estudiante_hombre_tono_de_piel_claro:',
        'fr': u':étudiant_peau_claire:',
        'pt': u':estudante_pele_clara:',
        'it': u':studente_maschio_carnagione_chiara:'
    },
    u'\U0001F468\U0001F3FE\U0000200D\U0001F393': {
        'en' : u':man_student_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':student_mitteldunkle_hautfarbe:',
        'es': u':estudiante_hombre_tono_de_piel_oscuro_medio:',
        'fr': u':étudiant_peau_mate:',
        'pt': u':estudante_pele_morena_escura:',
        'it': u':studente_maschio_carnagione_abbastanza_scura:'
    },
    u'\U0001F468\U0001F3FC\U0000200D\U0001F393': {
        'en' : u':man_student_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':student_mittelhelle_hautfarbe:',
        'es': u':estudiante_hombre_tono_de_piel_claro_medio:',
        'fr': u':étudiant_peau_moyennement_claire:',
        'pt': u':estudante_pele_morena_clara:',
        'it': u':studente_maschio_carnagione_abbastanza_chiara:'
    },
    u'\U0001F468\U0001F3FD\U0000200D\U0001F393': {
        'en' : u':man_student_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':student_mittlere_hautfarbe:',
        'es': u':estudiante_hombre_tono_de_piel_medio:',
        'fr': u':étudiant_peau_légèrement_mate:',
        'pt': u':estudante_pele_morena:',
        'it': u':studente_maschio_carnagione_olivastra:'
    },
    u'\U0001F9B8\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_superhero:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':superheld:',
        'es': u':superhéroe:',
        'fr': u':super-héros_homme:',
        'pt': u':homem_super-herói:',
        'it': u':supereroe_uomo:'
    },
    u'\U0001F9B8\U0000200D\U00002642': {
        'en' : u':man_superhero:',
        'status' : minimally_qualified,
        'E' : 11
    },
    u'\U0001F9B8\U0001F3FF\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_superhero_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':superheld_dunkle_hautfarbe:',
        'es': u':superhéroe_tono_de_piel_oscuro:',
        'fr': u':super-héros_homme_peau_foncée:',
        'pt': u':homem_super-herói_pele_escura:',
        'it': u':supereroe_uomo_carnagione_scura:'
    },
    u'\U0001F9B8\U0001F3FF\U0000200D\U00002642': {
        'en' : u':man_superhero_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 11
    },
    u'\U0001F9B8\U0001F3FB\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_superhero_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':superheld_helle_hautfarbe:',
        'es': u':superhéroe_tono_de_piel_claro:',
        'fr': u':super-héros_homme_peau_claire:',
        'pt': u':homem_super-herói_pele_clara:',
        'it': u':supereroe_uomo_carnagione_chiara:'
    },
    u'\U0001F9B8\U0001F3FB\U0000200D\U00002642': {
        'en' : u':man_superhero_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 11
    },
    u'\U0001F9B8\U0001F3FE\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_superhero_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':superheld_mitteldunkle_hautfarbe:',
        'es': u':superhéroe_tono_de_piel_oscuro_medio:',
        'fr': u':super-héros_homme_peau_mate:',
        'pt': u':homem_super-herói_pele_morena_escura:',
        'it': u':supereroe_uomo_carnagione_abbastanza_scura:'
    },
    u'\U0001F9B8\U0001F3FE\U0000200D\U00002642': {
        'en' : u':man_superhero_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 11
    },
    u'\U0001F9B8\U0001F3FC\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_superhero_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':superheld_mittelhelle_hautfarbe:',
        'es': u':superhéroe_tono_de_piel_claro_medio:',
        'fr': u':super-héros_homme_peau_moyennement_claire:',
        'pt': u':homem_super-herói_pele_morena_clara:',
        'it': u':supereroe_uomo_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9B8\U0001F3FC\U0000200D\U00002642': {
        'en' : u':man_superhero_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 11
    },
    u'\U0001F9B8\U0001F3FD\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_superhero_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':superheld_mittlere_hautfarbe:',
        'es': u':superhéroe_tono_de_piel_medio:',
        'fr': u':super-héros_homme_peau_légèrement_mate:',
        'pt': u':homem_super-herói_pele_morena:',
        'it': u':supereroe_uomo_carnagione_olivastra:'
    },
    u'\U0001F9B8\U0001F3FD\U0000200D\U00002642': {
        'en' : u':man_superhero_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 11
    },
    u'\U0001F9B9\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_supervillain:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':männlicher_bösewicht:',
        'es': u':supervillano:',
        'fr': u':super-vilain_homme:',
        'pt': u':homem_supervilão:',
        'it': u':supercattivo_uomo:'
    },
    u'\U0001F9B9\U0000200D\U00002642': {
        'en' : u':man_supervillain:',
        'status' : minimally_qualified,
        'E' : 11
    },
    u'\U0001F9B9\U0001F3FF\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_supervillain_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':männlicher_bösewicht_dunkle_hautfarbe:',
        'es': u':supervillano_tono_de_piel_oscuro:',
        'fr': u':super-vilain_homme_peau_foncée:',
        'pt': u':homem_supervilão_pele_escura:',
        'it': u':supercattivo_uomo_carnagione_scura:'
    },
    u'\U0001F9B9\U0001F3FF\U0000200D\U00002642': {
        'en' : u':man_supervillain_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 11
    },
    u'\U0001F9B9\U0001F3FB\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_supervillain_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':männlicher_bösewicht_helle_hautfarbe:',
        'es': u':supervillano_tono_de_piel_claro:',
        'fr': u':super-vilain_homme_peau_claire:',
        'pt': u':homem_supervilão_pele_clara:',
        'it': u':supercattivo_uomo_carnagione_chiara:'
    },
    u'\U0001F9B9\U0001F3FB\U0000200D\U00002642': {
        'en' : u':man_supervillain_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 11
    },
    u'\U0001F9B9\U0001F3FE\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_supervillain_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':männlicher_bösewicht_mitteldunkle_hautfarbe:',
        'es': u':supervillano_tono_de_piel_oscuro_medio:',
        'fr': u':super-vilain_homme_peau_mate:',
        'pt': u':homem_supervilão_pele_morena_escura:',
        'it': u':supercattivo_uomo_carnagione_abbastanza_scura:'
    },
    u'\U0001F9B9\U0001F3FE\U0000200D\U00002642': {
        'en' : u':man_supervillain_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 11
    },
    u'\U0001F9B9\U0001F3FC\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_supervillain_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':männlicher_bösewicht_mittelhelle_hautfarbe:',
        'es': u':supervillano_tono_de_piel_claro_medio:',
        'fr': u':super-vilain_homme_peau_moyennement_claire:',
        'pt': u':homem_supervilão_pele_morena_clara:',
        'it': u':supercattivo_uomo_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9B9\U0001F3FC\U0000200D\U00002642': {
        'en' : u':man_supervillain_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 11
    },
    u'\U0001F9B9\U0001F3FD\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_supervillain_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':männlicher_bösewicht_mittlere_hautfarbe:',
        'es': u':supervillano_tono_de_piel_medio:',
        'fr': u':super-vilain_homme_peau_légèrement_mate:',
        'pt': u':homem_supervilão_pele_morena:',
        'it': u':supercattivo_uomo_carnagione_olivastra:'
    },
    u'\U0001F9B9\U0001F3FD\U0000200D\U00002642': {
        'en' : u':man_supervillain_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 11
    },
    u'\U0001F3C4\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_surfing:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':surfer:',
        'es': u':hombre_haciendo_surf:',
        'fr': u':surfeur:',
        'pt': u':homem_surfista:',
        'it': u':surfista_uomo:'
    },
    u'\U0001F3C4\U0000200D\U00002642': {
        'en' : u':man_surfing:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F3C4\U0001F3FF\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_surfing_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':surfer_dunkle_hautfarbe:',
        'es': u':hombre_haciendo_surf_tono_de_piel_oscuro:',
        'fr': u':surfeur_peau_foncée:',
        'pt': u':homem_surfista_pele_escura:',
        'it': u':surfista_uomo_carnagione_scura:'
    },
    u'\U0001F3C4\U0001F3FF\U0000200D\U00002642': {
        'en' : u':man_surfing_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F3C4\U0001F3FB\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_surfing_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':surfer_helle_hautfarbe:',
        'es': u':hombre_haciendo_surf_tono_de_piel_claro:',
        'fr': u':surfeur_peau_claire:',
        'pt': u':homem_surfista_pele_clara:',
        'it': u':surfista_uomo_carnagione_chiara:'
    },
    u'\U0001F3C4\U0001F3FB\U0000200D\U00002642': {
        'en' : u':man_surfing_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F3C4\U0001F3FE\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_surfing_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':surfer_mitteldunkle_hautfarbe:',
        'es': u':hombre_haciendo_surf_tono_de_piel_oscuro_medio:',
        'fr': u':surfeur_peau_mate:',
        'pt': u':homem_surfista_pele_morena_escura:',
        'it': u':surfista_uomo_carnagione_abbastanza_scura:'
    },
    u'\U0001F3C4\U0001F3FE\U0000200D\U00002642': {
        'en' : u':man_surfing_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F3C4\U0001F3FC\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_surfing_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':surfer_mittelhelle_hautfarbe:',
        'es': u':hombre_haciendo_surf_tono_de_piel_claro_medio:',
        'fr': u':surfeur_peau_moyennement_claire:',
        'pt': u':homem_surfista_pele_morena_clara:',
        'it': u':surfista_uomo_carnagione_abbastanza_chiara:'
    },
    u'\U0001F3C4\U0001F3FC\U0000200D\U00002642': {
        'en' : u':man_surfing_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F3C4\U0001F3FD\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_surfing_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':surfer_mittlere_hautfarbe:',
        'es': u':hombre_haciendo_surf_tono_de_piel_medio:',
        'fr': u':surfeur_peau_légèrement_mate:',
        'pt': u':homem_surfista_pele_morena:',
        'it': u':surfista_uomo_carnagione_olivastra:'
    },
    u'\U0001F3C4\U0001F3FD\U0000200D\U00002642': {
        'en' : u':man_surfing_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F3CA\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_swimming:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':schwimmer:',
        'es': u':hombre_nadando:',
        'fr': u':nageur:',
        'pt': u':homem_nadando:',
        'it': u':nuotatore:'
    },
    u'\U0001F3CA\U0000200D\U00002642': {
        'en' : u':man_swimming:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F3CA\U0001F3FF\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_swimming_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':schwimmer_dunkle_hautfarbe:',
        'es': u':hombre_nadando_tono_de_piel_oscuro:',
        'fr': u':nageur_peau_foncée:',
        'pt': u':homem_nadando_pele_escura:',
        'it': u':nuotatore_carnagione_scura:'
    },
    u'\U0001F3CA\U0001F3FF\U0000200D\U00002642': {
        'en' : u':man_swimming_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F3CA\U0001F3FB\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_swimming_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':schwimmer_helle_hautfarbe:',
        'es': u':hombre_nadando_tono_de_piel_claro:',
        'fr': u':nageur_peau_claire:',
        'pt': u':homem_nadando_pele_clara:',
        'it': u':nuotatore_carnagione_chiara:'
    },
    u'\U0001F3CA\U0001F3FB\U0000200D\U00002642': {
        'en' : u':man_swimming_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F3CA\U0001F3FE\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_swimming_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':schwimmer_mitteldunkle_hautfarbe:',
        'es': u':hombre_nadando_tono_de_piel_oscuro_medio:',
        'fr': u':nageur_peau_mate:',
        'pt': u':homem_nadando_pele_morena_escura:',
        'it': u':nuotatore_carnagione_abbastanza_scura:'
    },
    u'\U0001F3CA\U0001F3FE\U0000200D\U00002642': {
        'en' : u':man_swimming_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F3CA\U0001F3FC\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_swimming_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':schwimmer_mittelhelle_hautfarbe:',
        'es': u':hombre_nadando_tono_de_piel_claro_medio:',
        'fr': u':nageur_peau_moyennement_claire:',
        'pt': u':homem_nadando_pele_morena_clara:',
        'it': u':nuotatore_carnagione_abbastanza_chiara:'
    },
    u'\U0001F3CA\U0001F3FC\U0000200D\U00002642': {
        'en' : u':man_swimming_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F3CA\U0001F3FD\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_swimming_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':schwimmer_mittlere_hautfarbe:',
        'es': u':hombre_nadando_tono_de_piel_medio:',
        'fr': u':nageur_peau_légèrement_mate:',
        'pt': u':homem_nadando_pele_morena:',
        'it': u':nuotatore_carnagione_olivastra:'
    },
    u'\U0001F3CA\U0001F3FD\U0000200D\U00002642': {
        'en' : u':man_swimming_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F468\U0000200D\U0001F3EB': {
        'en' : u':man_teacher:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':lehrer:',
        'es': u':docente_hombre:',
        'fr': u':enseignant:',
        'pt': u':professor:',
        'it': u':professore:'
    },
    u'\U0001F468\U0001F3FF\U0000200D\U0001F3EB': {
        'en' : u':man_teacher_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':lehrer_dunkle_hautfarbe:',
        'es': u':docente_hombre_tono_de_piel_oscuro:',
        'fr': u':enseignant_peau_foncée:',
        'pt': u':professor_pele_escura:',
        'it': u':professore_carnagione_scura:'
    },
    u'\U0001F468\U0001F3FB\U0000200D\U0001F3EB': {
        'en' : u':man_teacher_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':lehrer_helle_hautfarbe:',
        'es': u':docente_hombre_tono_de_piel_claro:',
        'fr': u':enseignant_peau_claire:',
        'pt': u':professor_pele_clara:',
        'it': u':professore_carnagione_chiara:'
    },
    u'\U0001F468\U0001F3FE\U0000200D\U0001F3EB': {
        'en' : u':man_teacher_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':lehrer_mitteldunkle_hautfarbe:',
        'es': u':docente_hombre_tono_de_piel_oscuro_medio:',
        'fr': u':enseignant_peau_mate:',
        'pt': u':professor_pele_morena_escura:',
        'it': u':professore_carnagione_abbastanza_scura:'
    },
    u'\U0001F468\U0001F3FC\U0000200D\U0001F3EB': {
        'en' : u':man_teacher_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':lehrer_mittelhelle_hautfarbe:',
        'es': u':docente_hombre_tono_de_piel_claro_medio:',
        'fr': u':enseignant_peau_moyennement_claire:',
        'pt': u':professor_pele_morena_clara:',
        'it': u':professore_carnagione_abbastanza_chiara:'
    },
    u'\U0001F468\U0001F3FD\U0000200D\U0001F3EB': {
        'en' : u':man_teacher_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':lehrer_mittlere_hautfarbe:',
        'es': u':docente_hombre_tono_de_piel_medio:',
        'fr': u':enseignant_peau_légèrement_mate:',
        'pt': u':professor_pele_morena:',
        'it': u':professore_carnagione_olivastra:'
    },
    u'\U0001F468\U0000200D\U0001F4BB': {
        'en' : u':man_technologist:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':it-experte:',
        'es': u':profesional_de_la_tecnología_hombre:',
        'fr': u':informaticien:',
        'pt': u':tecnólogo:',
        'it': u':tecnologo:'
    },
    u'\U0001F468\U0001F3FF\U0000200D\U0001F4BB': {
        'en' : u':man_technologist_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':it-experte_dunkle_hautfarbe:',
        'es': u':profesional_de_la_tecnología_hombre_tono_de_piel_oscuro:',
        'fr': u':informaticien_peau_foncée:',
        'pt': u':tecnólogo_pele_escura:',
        'it': u':tecnologo_carnagione_scura:'
    },
    u'\U0001F468\U0001F3FB\U0000200D\U0001F4BB': {
        'en' : u':man_technologist_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':it-experte_helle_hautfarbe:',
        'es': u':profesional_de_la_tecnología_hombre_tono_de_piel_claro:',
        'fr': u':informaticien_peau_claire:',
        'pt': u':tecnólogo_pele_clara:',
        'it': u':tecnologo_carnagione_chiara:'
    },
    u'\U0001F468\U0001F3FE\U0000200D\U0001F4BB': {
        'en' : u':man_technologist_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':it-experte_mitteldunkle_hautfarbe:',
        'es': u':profesional_de_la_tecnología_hombre_tono_de_piel_oscuro_medio:',
        'fr': u':informaticien_peau_mate:',
        'pt': u':tecnólogo_pele_morena_escura:',
        'it': u':tecnologo_carnagione_abbastanza_scura:'
    },
    u'\U0001F468\U0001F3FC\U0000200D\U0001F4BB': {
        'en' : u':man_technologist_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':it-experte_mittelhelle_hautfarbe:',
        'es': u':profesional_de_la_tecnología_hombre_tono_de_piel_claro_medio:',
        'fr': u':informaticien_peau_moyennement_claire:',
        'pt': u':tecnólogo_pele_morena_clara:',
        'it': u':tecnologo_carnagione_abbastanza_chiara:'
    },
    u'\U0001F468\U0001F3FD\U0000200D\U0001F4BB': {
        'en' : u':man_technologist_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':it-experte_mittlere_hautfarbe:',
        'es': u':profesional_de_la_tecnología_hombre_tono_de_piel_medio:',
        'fr': u':informaticien_peau_légèrement_mate:',
        'pt': u':tecnólogo_pele_morena:',
        'it': u':tecnologo_carnagione_olivastra:'
    },
    u'\U0001F481\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_tipping_hand:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':infoschalter-mitarbeiter:',
        'es': u':empleado_de_mostrador_de_información:',
        'fr': u':homme_paume_vers_le_haut:',
        'pt': u':homem_com_a_palma_virada_para_cima:',
        'it': u':uomo_con_suggerimento:'
    },
    u'\U0001F481\U0000200D\U00002642': {
        'en' : u':man_tipping_hand:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F481\U0001F3FF\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_tipping_hand_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':infoschalter-mitarbeiter_dunkle_hautfarbe:',
        'es': u':empleado_de_mostrador_de_información_tono_de_piel_oscuro:',
        'fr': u':homme_paume_vers_le_haut_peau_foncée:',
        'pt': u':homem_com_a_palma_virada_para_cima_pele_escura:',
        'it': u':uomo_con_suggerimento_carnagione_scura:'
    },
    u'\U0001F481\U0001F3FF\U0000200D\U00002642': {
        'en' : u':man_tipping_hand_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F481\U0001F3FB\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_tipping_hand_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':infoschalter-mitarbeiter_helle_hautfarbe:',
        'es': u':empleado_de_mostrador_de_información_tono_de_piel_claro:',
        'fr': u':homme_paume_vers_le_haut_peau_claire:',
        'pt': u':homem_com_a_palma_virada_para_cima_pele_clara:',
        'it': u':uomo_con_suggerimento_carnagione_chiara:'
    },
    u'\U0001F481\U0001F3FB\U0000200D\U00002642': {
        'en' : u':man_tipping_hand_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F481\U0001F3FE\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_tipping_hand_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':infoschalter-mitarbeiter_mitteldunkle_hautfarbe:',
        'es': u':empleado_de_mostrador_de_información_tono_de_piel_oscuro_medio:',
        'fr': u':homme_paume_vers_le_haut_peau_mate:',
        'pt': u':homem_com_a_palma_virada_para_cima_pele_morena_escura:',
        'it': u':uomo_con_suggerimento_carnagione_abbastanza_scura:'
    },
    u'\U0001F481\U0001F3FE\U0000200D\U00002642': {
        'en' : u':man_tipping_hand_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F481\U0001F3FC\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_tipping_hand_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':infoschalter-mitarbeiter_mittelhelle_hautfarbe:',
        'es': u':empleado_de_mostrador_de_información_tono_de_piel_claro_medio:',
        'fr': u':homme_paume_vers_le_haut_peau_moyennement_claire:',
        'pt': u':homem_com_a_palma_virada_para_cima_pele_morena_clara:',
        'it': u':uomo_con_suggerimento_carnagione_abbastanza_chiara:'
    },
    u'\U0001F481\U0001F3FC\U0000200D\U00002642': {
        'en' : u':man_tipping_hand_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F481\U0001F3FD\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_tipping_hand_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':infoschalter-mitarbeiter_mittlere_hautfarbe:',
        'es': u':empleado_de_mostrador_de_información_tono_de_piel_medio:',
        'fr': u':homme_paume_vers_le_haut_peau_légèrement_mate:',
        'pt': u':homem_com_a_palma_virada_para_cima_pele_morena:',
        'it': u':uomo_con_suggerimento_carnagione_olivastra:'
    },
    u'\U0001F481\U0001F3FD\U0000200D\U00002642': {
        'en' : u':man_tipping_hand_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F9DB\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_vampire:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':männlicher_vampir:',
        'es': u':vampiro_hombre:',
        'fr': u':vampire_homme:',
        'pt': u':homem_vampiro:',
        'it': u':vampiro_uomo:'
    },
    u'\U0001F9DB\U0000200D\U00002642': {
        'en' : u':man_vampire:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F9DB\U0001F3FF\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_vampire_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':männlicher_vampir_dunkle_hautfarbe:',
        'es': u':vampiro_hombre_tono_de_piel_oscuro:',
        'fr': u':vampire_homme_peau_foncée:',
        'pt': u':homem_vampiro_pele_escura:',
        'it': u':vampiro_uomo_carnagione_scura:'
    },
    u'\U0001F9DB\U0001F3FF\U0000200D\U00002642': {
        'en' : u':man_vampire_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F9DB\U0001F3FB\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_vampire_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':männlicher_vampir_helle_hautfarbe:',
        'es': u':vampiro_hombre_tono_de_piel_claro:',
        'fr': u':vampire_homme_peau_claire:',
        'pt': u':homem_vampiro_pele_clara:',
        'it': u':vampiro_uomo_carnagione_chiara:'
    },
    u'\U0001F9DB\U0001F3FB\U0000200D\U00002642': {
        'en' : u':man_vampire_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F9DB\U0001F3FE\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_vampire_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':männlicher_vampir_mitteldunkle_hautfarbe:',
        'es': u':vampiro_hombre_tono_de_piel_oscuro_medio:',
        'fr': u':vampire_homme_peau_mate:',
        'pt': u':homem_vampiro_pele_morena_escura:',
        'it': u':vampiro_uomo_carnagione_abbastanza_scura:'
    },
    u'\U0001F9DB\U0001F3FE\U0000200D\U00002642': {
        'en' : u':man_vampire_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F9DB\U0001F3FC\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_vampire_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':männlicher_vampir_mittelhelle_hautfarbe:',
        'es': u':vampiro_hombre_tono_de_piel_claro_medio:',
        'fr': u':vampire_homme_peau_moyennement_claire:',
        'pt': u':homem_vampiro_pele_morena_clara:',
        'it': u':vampiro_uomo_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9DB\U0001F3FC\U0000200D\U00002642': {
        'en' : u':man_vampire_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F9DB\U0001F3FD\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_vampire_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':männlicher_vampir_mittlere_hautfarbe:',
        'es': u':vampiro_hombre_tono_de_piel_medio:',
        'fr': u':vampire_homme_peau_légèrement_mate:',
        'pt': u':homem_vampiro_pele_morena:',
        'it': u':vampiro_uomo_carnagione_olivastra:'
    },
    u'\U0001F9DB\U0001F3FD\U0000200D\U00002642': {
        'en' : u':man_vampire_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F6B6\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_walking:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':fußgänger:',
        'es': u':hombre_caminando:',
        'fr': u':homme_qui_marche:',
        'pt': u':homem_andando:',
        'it': u':uomo_che_cammina:'
    },
    u'\U0001F6B6\U0000200D\U00002642': {
        'en' : u':man_walking:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F6B6\U0001F3FF\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_walking_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':fußgänger_dunkle_hautfarbe:',
        'es': u':hombre_caminando_tono_de_piel_oscuro:',
        'fr': u':homme_qui_marche_peau_foncée:',
        'pt': u':homem_andando_pele_escura:',
        'it': u':uomo_che_cammina_carnagione_scura:'
    },
    u'\U0001F6B6\U0001F3FF\U0000200D\U00002642': {
        'en' : u':man_walking_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F6B6\U0001F3FB\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_walking_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':fußgänger_helle_hautfarbe:',
        'es': u':hombre_caminando_tono_de_piel_claro:',
        'fr': u':homme_qui_marche_peau_claire:',
        'pt': u':homem_andando_pele_clara:',
        'it': u':uomo_che_cammina_carnagione_chiara:'
    },
    u'\U0001F6B6\U0001F3FB\U0000200D\U00002642': {
        'en' : u':man_walking_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F6B6\U0001F3FE\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_walking_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':fußgänger_mitteldunkle_hautfarbe:',
        'es': u':hombre_caminando_tono_de_piel_oscuro_medio:',
        'fr': u':homme_qui_marche_peau_mate:',
        'pt': u':homem_andando_pele_morena_escura:',
        'it': u':uomo_che_cammina_carnagione_abbastanza_scura:'
    },
    u'\U0001F6B6\U0001F3FE\U0000200D\U00002642': {
        'en' : u':man_walking_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F6B6\U0001F3FC\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_walking_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':fußgänger_mittelhelle_hautfarbe:',
        'es': u':hombre_caminando_tono_de_piel_claro_medio:',
        'fr': u':homme_qui_marche_peau_moyennement_claire:',
        'pt': u':homem_andando_pele_morena_clara:',
        'it': u':uomo_che_cammina_carnagione_abbastanza_chiara:'
    },
    u'\U0001F6B6\U0001F3FC\U0000200D\U00002642': {
        'en' : u':man_walking_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F6B6\U0001F3FD\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_walking_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':fußgänger_mittlere_hautfarbe:',
        'es': u':hombre_caminando_tono_de_piel_medio:',
        'fr': u':homme_qui_marche_peau_légèrement_mate:',
        'pt': u':homem_andando_pele_morena:',
        'it': u':uomo_che_cammina_carnagione_olivastra:'
    },
    u'\U0001F6B6\U0001F3FD\U0000200D\U00002642': {
        'en' : u':man_walking_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F473\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_wearing_turban:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':mann_mit_turban:',
        'es': u':hombre_con_turbante:',
        'fr': u':homme_en_turban:',
        'pt': u':homem_com_turbante:',
        'it': u':uomo_con_turbante:'
    },
    u'\U0001F473\U0000200D\U00002642': {
        'en' : u':man_wearing_turban:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F473\U0001F3FF\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_wearing_turban_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':mann_mit_turban_dunkle_hautfarbe:',
        'es': u':hombre_con_turbante_tono_de_piel_oscuro:',
        'fr': u':homme_en_turban_peau_foncée:',
        'pt': u':homem_com_turbante_pele_escura:',
        'it': u':uomo_con_turbante_carnagione_scura:'
    },
    u'\U0001F473\U0001F3FF\U0000200D\U00002642': {
        'en' : u':man_wearing_turban_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F473\U0001F3FB\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_wearing_turban_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':mann_mit_turban_helle_hautfarbe:',
        'es': u':hombre_con_turbante_tono_de_piel_claro:',
        'fr': u':homme_en_turban_peau_claire:',
        'pt': u':homem_com_turbante_pele_clara:',
        'it': u':uomo_con_turbante_carnagione_chiara:'
    },
    u'\U0001F473\U0001F3FB\U0000200D\U00002642': {
        'en' : u':man_wearing_turban_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F473\U0001F3FE\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_wearing_turban_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':mann_mit_turban_mitteldunkle_hautfarbe:',
        'es': u':hombre_con_turbante_tono_de_piel_oscuro_medio:',
        'fr': u':homme_en_turban_peau_mate:',
        'pt': u':homem_com_turbante_pele_morena_escura:',
        'it': u':uomo_con_turbante_carnagione_abbastanza_scura:'
    },
    u'\U0001F473\U0001F3FE\U0000200D\U00002642': {
        'en' : u':man_wearing_turban_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F473\U0001F3FC\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_wearing_turban_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':mann_mit_turban_mittelhelle_hautfarbe:',
        'es': u':hombre_con_turbante_tono_de_piel_claro_medio:',
        'fr': u':homme_en_turban_peau_moyennement_claire:',
        'pt': u':homem_com_turbante_pele_morena_clara:',
        'it': u':uomo_con_turbante_carnagione_abbastanza_chiara:'
    },
    u'\U0001F473\U0001F3FC\U0000200D\U00002642': {
        'en' : u':man_wearing_turban_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F473\U0001F3FD\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_wearing_turban_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':mann_mit_turban_mittlere_hautfarbe:',
        'es': u':hombre_con_turbante_tono_de_piel_medio:',
        'fr': u':homme_en_turban_peau_légèrement_mate:',
        'pt': u':homem_com_turbante_pele_morena:',
        'it': u':uomo_con_turbante_carnagione_olivastra:'
    },
    u'\U0001F473\U0001F3FD\U0000200D\U00002642': {
        'en' : u':man_wearing_turban_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F468\U0000200D\U0001F9B3': {
        'en' : u':man_white_hair:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':mann_weißes_haar:',
        'es': u':hombre_pelo_blanco:',
        'fr': u':homme_cheveux_blancs:',
        'pt': u':homem_cabelo_branco:',
        'it': u':uomo_capelli_bianchi:'
    },
    u'\U0001F470\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_with_veil:',
        'status' : fully_qualified,
        'E' : 13,
        'de': u':mann_mit_schleier:',
        'es': u':hombre_con_velo:',
        'fr': u':homme_avec_voile:',
        'pt': u':noivo_com_véu:',
        'it': u':sposo_con_velo:'
    },
    u'\U0001F470\U0000200D\U00002642': {
        'en' : u':man_with_veil:',
        'status' : minimally_qualified,
        'E' : 13
    },
    u'\U0001F470\U0001F3FF\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_with_veil_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13,
        'de': u':mann_mit_schleier_dunkle_hautfarbe:',
        'es': u':hombre_con_velo_tono_de_piel_oscuro:',
        'fr': u':homme_avec_voile_peau_foncée:',
        'pt': u':noivo_com_véu_pele_escura:',
        'it': u':sposo_con_velo_carnagione_scura:'
    },
    u'\U0001F470\U0001F3FF\U0000200D\U00002642': {
        'en' : u':man_with_veil_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13
    },
    u'\U0001F470\U0001F3FB\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_with_veil_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13,
        'de': u':mann_mit_schleier_helle_hautfarbe:',
        'es': u':hombre_con_velo_tono_de_piel_claro:',
        'fr': u':homme_avec_voile_peau_claire:',
        'pt': u':noivo_com_véu_pele_clara:',
        'it': u':sposo_con_velo_carnagione_chiara:'
    },
    u'\U0001F470\U0001F3FB\U0000200D\U00002642': {
        'en' : u':man_with_veil_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13
    },
    u'\U0001F470\U0001F3FE\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_with_veil_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13,
        'de': u':mann_mit_schleier_mitteldunkle_hautfarbe:',
        'es': u':hombre_con_velo_tono_de_piel_oscuro_medio:',
        'fr': u':homme_avec_voile_peau_mate:',
        'pt': u':noivo_com_véu_pele_morena_escura:',
        'it': u':sposo_con_velo_carnagione_abbastanza_scura:'
    },
    u'\U0001F470\U0001F3FE\U0000200D\U00002642': {
        'en' : u':man_with_veil_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13
    },
    u'\U0001F470\U0001F3FC\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_with_veil_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13,
        'de': u':mann_mit_schleier_mittelhelle_hautfarbe:',
        'es': u':hombre_con_velo_tono_de_piel_claro_medio:',
        'fr': u':homme_avec_voile_peau_moyennement_claire:',
        'pt': u':noivo_com_véu_pele_morena_clara:',
        'it': u':sposo_con_velo_carnagione_abbastanza_chiara:'
    },
    u'\U0001F470\U0001F3FC\U0000200D\U00002642': {
        'en' : u':man_with_veil_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13
    },
    u'\U0001F470\U0001F3FD\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_with_veil_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13,
        'de': u':mann_mit_schleier_mittlere_hautfarbe:',
        'es': u':hombre_con_velo_tono_de_piel_medio:',
        'fr': u':homme_avec_voile_peau_légèrement_mate:',
        'pt': u':noivo_com_véu_pele_morena:',
        'it': u':sposo_con_velo_carnagione_olivastra:'
    },
    u'\U0001F470\U0001F3FD\U0000200D\U00002642': {
        'en' : u':man_with_veil_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13
    },
    u'\U0001F468\U0000200D\U0001F9AF': {
        'en' : u':man_with_white_cane:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':mann_mit_gehstock:',
        'es': u':hombre_con_bastón:',
        'fr': u':homme_avec_canne_blanche:',
        'pt': u':homem_com_bengala_para_cego:',
        'it': u':uomo_con_bastone_bianco_di_orientamento:'
    },
    u'\U0001F468\U0001F3FF\U0000200D\U0001F9AF': {
        'en' : u':man_with_white_cane_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':mann_mit_gehstock_dunkle_hautfarbe:',
        'es': u':hombre_con_bastón_tono_de_piel_oscuro:',
        'fr': u':homme_avec_canne_blanche_peau_foncée:',
        'pt': u':homem_com_bengala_para_cego_pele_escura:',
        'it': u':uomo_con_bastone_bianco_di_orientamento_carnagione_scura:'
    },
    u'\U0001F468\U0001F3FB\U0000200D\U0001F9AF': {
        'en' : u':man_with_white_cane_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':mann_mit_gehstock_helle_hautfarbe:',
        'es': u':hombre_con_bastón_tono_de_piel_claro:',
        'fr': u':homme_avec_canne_blanche_peau_claire:',
        'pt': u':homem_com_bengala_para_cego_pele_clara:',
        'it': u':uomo_con_bastone_bianco_di_orientamento_carnagione_chiara:'
    },
    u'\U0001F468\U0001F3FE\U0000200D\U0001F9AF': {
        'en' : u':man_with_white_cane_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':mann_mit_gehstock_mitteldunkle_hautfarbe:',
        'es': u':hombre_con_bastón_tono_de_piel_oscuro_medio:',
        'fr': u':homme_avec_canne_blanche_peau_mate:',
        'pt': u':homem_com_bengala_para_cego_pele_morena_escura:',
        'it': u':uomo_con_bastone_bianco_di_orientamento_carnagione_abbastanza_scura:'
    },
    u'\U0001F468\U0001F3FC\U0000200D\U0001F9AF': {
        'en' : u':man_with_white_cane_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':mann_mit_gehstock_mittelhelle_hautfarbe:',
        'es': u':hombre_con_bastón_tono_de_piel_claro_medio:',
        'fr': u':homme_avec_canne_blanche_peau_moyennement_claire:',
        'pt': u':homem_com_bengala_para_cego_pele_morena_clara:',
        'it': u':uomo_con_bastone_bianco_di_orientamento_carnagione_abbastanza_chiara:'
    },
    u'\U0001F468\U0001F3FD\U0000200D\U0001F9AF': {
        'en' : u':man_with_white_cane_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':mann_mit_gehstock_mittlere_hautfarbe:',
        'es': u':hombre_con_bastón_tono_de_piel_medio:',
        'fr': u':homme_avec_canne_blanche_peau_légèrement_mate:',
        'pt': u':homem_com_bengala_para_cego_pele_morena:',
        'it': u':uomo_con_bastone_bianco_di_orientamento_carnagione_olivastra:'
    },
    u'\U0001F9DF\U0000200D\U00002642\U0000FE0F': {
        'en' : u':man_zombie:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':männlicher_zombie:',
        'es': u':zombi_hombre:',
        'fr': u':zombie_homme:',
        'pt': u':homem_zumbi:',
        'it': u':zombie_uomo:'
    },
    u'\U0001F9DF\U0000200D\U00002642': {
        'en' : u':man_zombie:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F96D': {
        'en' : u':mango:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':mango:',
        'es': u':mango:',
        'fr': u':mangue:',
        'pt': u':manga:',
        'it': u':mango:'
    },
    u'\U0001F570\U0000FE0F': {
        'en' : u':mantelpiece_clock:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': u':kaminuhr:',
        'es': u':reloj_de_sobremesa:',
        'fr': u':pendule:',
        'pt': u':relógio_de_mesa:',
        'it': u':orologio_da_mensola:'
    },
    u'\U0001F570': {
        'en' : u':mantelpiece_clock:',
        'status' : unqualified,
        'E' : 0.7,
        'variant': True,
        'de': u':kaminuhr:',
        'es': u':reloj_de_sobremesa:',
        'fr': u':pendule:',
        'pt': u':relógio_de_mesa:',
        'it': u':orologio_da_mensola:'
    },
    u'\U0001F9BD': {
        'en' : u':manual_wheelchair:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':manueller_rollstuhl:',
        'es': u':silla_de_ruedas_manual:',
        'fr': u':fauteuil_roulant_manuel:',
        'pt': u':cadeira_de_rodas_manual:',
        'it': u':sedia_a_rotelle_manuale:'
    },
    u'\U0001F45E': {
        'en' : u':man’s_shoe:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':mans_shoe:', u':shoe:'],
        'de': u':herrenschuh:',
        'es': u':zapato_de_hombre:',
        'fr': u':chaussure_d’homme:',
        'pt': u':sapato_masculino:',
        'it': u':scarpa_da_uomo:'
    },
    u'\U0001F5FE': {
        'en' : u':map_of_Japan:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':japan:'],
        'de': u':umriss_von_japan:',
        'es': u':mapa_de_japón:',
        'fr': u':carte_du_japon:',
        'pt': u':mapa_do_japão:',
        'it': u':mappa_del_giappone:'
    },
    u'\U0001F341': {
        'en' : u':maple_leaf:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':ahornblatt:',
        'es': u':hoja_de_arce:',
        'fr': u':feuille_d’érable:',
        'pt': u':folha_de_bordo:',
        'it': u':foglia_d’acero:'
    },
    u'\U0001F94B': {
        'en' : u':martial_arts_uniform:',
        'status' : fully_qualified,
        'E' : 3,
        'de': u':kampfsportanzug:',
        'es': u':uniforme_de_artes_marciales:',
        'fr': u':tenue_d’arts_martiaux:',
        'pt': u':quimono_de_artes_marciais:',
        'it': u':kimono_per_arti_marziali:'
    },
    u'\U0001F9C9': {
        'en' : u':mate:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':mate-tee:',
        'es': u':mate:',
        'fr': u':maté:',
        'pt': u':mate:',
        'it': u':mate:'
    },
    u'\U0001F356': {
        'en' : u':meat_on_bone:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':fleischhachse:',
        'es': u':carne_con_hueso:',
        'fr': u':viande_sur_un_os:',
        'pt': u':carne:',
        'it': u':carne:'
    },
    u'\U0001F9D1\U0000200D\U0001F527': {
        'en' : u':mechanic:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':mechaniker(in):',
        'es': u':profesional_de_la_mecánica:',
        'fr': u':mécanicien_(tous_genres):',
        'pt': u':mecânico_trabalhando:',
        'it': u':meccanico:'
    },
    u'\U0001F9D1\U0001F3FF\U0000200D\U0001F527': {
        'en' : u':mechanic_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':mechaniker(in)_dunkle_hautfarbe:',
        'es': u':profesional_de_la_mecánica_tono_de_piel_oscuro:',
        'fr': u':mécanicien_(tous_genres)_peau_foncée:',
        'pt': u':mecânico_trabalhando_pele_escura:',
        'it': u':meccanico_carnagione_scura:'
    },
    u'\U0001F9D1\U0001F3FB\U0000200D\U0001F527': {
        'en' : u':mechanic_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':mechaniker(in)_helle_hautfarbe:',
        'es': u':profesional_de_la_mecánica_tono_de_piel_claro:',
        'fr': u':mécanicien_(tous_genres)_peau_claire:',
        'pt': u':mecânico_trabalhando_pele_clara:',
        'it': u':meccanico_carnagione_chiara:'
    },
    u'\U0001F9D1\U0001F3FE\U0000200D\U0001F527': {
        'en' : u':mechanic_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':mechaniker(in)_mitteldunkle_hautfarbe:',
        'es': u':profesional_de_la_mecánica_tono_de_piel_oscuro_medio:',
        'fr': u':mécanicien_(tous_genres)_peau_mate:',
        'pt': u':mecânico_trabalhando_pele_morena_escura:',
        'it': u':meccanico_carnagione_abbastanza_scura:'
    },
    u'\U0001F9D1\U0001F3FC\U0000200D\U0001F527': {
        'en' : u':mechanic_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':mechaniker(in)_mittelhelle_hautfarbe:',
        'es': u':profesional_de_la_mecánica_tono_de_piel_claro_medio:',
        'fr': u':mécanicien_(tous_genres)_peau_moyennement_claire:',
        'pt': u':mecânico_trabalhando_pele_morena_clara:',
        'it': u':meccanico_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9D1\U0001F3FD\U0000200D\U0001F527': {
        'en' : u':mechanic_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':mechaniker(in)_mittlere_hautfarbe:',
        'es': u':profesional_de_la_mecánica_tono_de_piel_medio:',
        'fr': u':mécanicien_(tous_genres)_peau_légèrement_mate:',
        'pt': u':mecânico_trabalhando_pele_morena:',
        'it': u':meccanico_carnagione_olivastra:'
    },
    u'\U0001F9BE': {
        'en' : u':mechanical_arm:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':armprothese:',
        'es': u':brazo_mecánico:',
        'fr': u':bras_mécanique:',
        'pt': u':braço_mecânico:',
        'it': u':protesi_robotica_per_il_braccio:'
    },
    u'\U0001F9BF': {
        'en' : u':mechanical_leg:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':beinprothese:',
        'es': u':pierna_mecánica:',
        'fr': u':jambe_mécanique:',
        'pt': u':perna_mecânica:',
        'it': u':protesi_robotica_per_la_gamba:'
    },
    u'\U00002695\U0000FE0F': {
        'en' : u':medical_symbol:',
        'status' : fully_qualified,
        'E' : 4,
        'variant': True,
        'de': u':äskulapstab:',
        'es': u':símbolo_de_medicina:',
        'fr': u':caducée:',
        'pt': u':símbolo_da_medicina:',
        'it': u':simbolo_della_medicina:'
    },
    u'\U00002695': {
        'en' : u':medical_symbol:',
        'status' : unqualified,
        'E' : 4,
        'variant': True,
        'de': u':äskulapstab:',
        'es': u':símbolo_de_medicina:',
        'fr': u':caducée:',
        'pt': u':símbolo_da_medicina:',
        'it': u':simbolo_della_medicina:'
    },
    u'\U0001F3FE': {
        'en' : u':medium-dark_skin_tone:',
        'status' : component,
        'E' : 1,
        'alias' : [u':emoji_modifier_fitzpatrick_type__5:'],
        'de': u':mitteldunkle_hautfarbe:',
        'es': u':tono_de_piel_oscuro_medio:',
        'fr': u':peau_mate:',
        'pt': u':pele_morena_escura:',
        'it': u':carnagione_abbastanza_scura:'
    },
    u'\U0001F3FC': {
        'en' : u':medium-light_skin_tone:',
        'status' : component,
        'E' : 1,
        'alias' : [u':emoji_modifier_fitzpatrick_type__3:'],
        'de': u':mittelhelle_hautfarbe:',
        'es': u':tono_de_piel_claro_medio:',
        'fr': u':peau_moyennement_claire:',
        'pt': u':pele_morena_clara:',
        'it': u':carnagione_abbastanza_chiara:'
    },
    u'\U0001F3FD': {
        'en' : u':medium_skin_tone:',
        'status' : component,
        'E' : 1,
        'alias' : [u':emoji_modifier_fitzpatrick_type__4:'],
        'de': u':mittlere_hautfarbe:',
        'es': u':tono_de_piel_medio:',
        'fr': u':peau_légèrement_mate:',
        'pt': u':pele_morena:',
        'it': u':carnagione_olivastra:'
    },
    u'\U0001F4E3': {
        'en' : u':megaphone:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':mega:'],
        'de': u':megafon:',
        'es': u':megáfono:',
        'fr': u':porte-voix:',
        'pt': u':megafone:',
        'it': u':megafono:'
    },
    u'\U0001F348': {
        'en' : u':melon:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':honigmelone:',
        'es': u':melón:',
        'fr': u':melon:',
        'pt': u':melão:',
        'it': u':melone:'
    },
    u'\U0001FAE0': {
        'en' : u':melting_face:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001F4DD': {
        'en' : u':memo:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':pencil:'],
        'de': u':papier_und_bleistift:',
        'es': u':cuaderno_de_notas:',
        'fr': u':mémo:',
        'pt': u':memorando:',
        'it': u':memo:'
    },
    u'\U0001F46C': {
        'en' : u':men_holding_hands:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [u':two_men_holding_hands:'],
        'de': u':händchenhaltende_männer:',
        'es': u':hombres_de_la_mano:',
        'fr': u':deux_hommes_se_tenant_la_main:',
        'pt': u':dois_homens_de_mãos_dadas:',
        'it': u':due_uomini_che_si_tengono_per_mano:'
    },
    u'\U0001F46C\U0001F3FF': {
        'en' : u':men_holding_hands_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':händchenhaltende_männer_dunkle_hautfarbe:',
        'es': u':hombres_de_la_mano_tono_de_piel_oscuro:',
        'fr': u':deux_hommes_se_tenant_la_main_peau_foncée:',
        'pt': u':dois_homens_de_mãos_dadas_pele_escura:',
        'it': u':due_uomini_che_si_tengono_per_mano_carnagione_scura:'
    },
    u'\U0001F468\U0001F3FF\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FB': {
        'en' : u':men_holding_hands_dark_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':händchenhaltende_männer_dunkle_hautfarbe,_helle_hautfarbe:',
        'es': u':hombres_de_la_mano_tono_de_piel_oscuro_y_tono_de_piel_claro:',
        'fr': u':deux_hommes_se_tenant_la_main_peau_foncée_et_peau_claire:',
        'pt': u':dois_homens_de_mãos_dadas_pele_escura_e_pele_clara:',
        'it': u':due_uomini_che_si_tengono_per_mano_carnagione_scura_e_carnagione_chiara:'
    },
    u'\U0001F468\U0001F3FF\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FE': {
        'en' : u':men_holding_hands_dark_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':händchenhaltende_männer_dunkle_hautfarbe,_mitteldunkle_hautfarbe:',
        'es': u':hombres_de_la_mano_tono_de_piel_oscuro_y_tono_de_piel_oscuro_medio:',
        'fr': u':deux_hommes_se_tenant_la_main_peau_foncée_et_peau_mate:',
        'pt': u':dois_homens_de_mãos_dadas_pele_escura_e_pele_morena_escura:',
        'it': u':due_uomini_che_si_tengono_per_mano_carnagione_scura_e_carnagione_abbastanza_scura:'
    },
    u'\U0001F468\U0001F3FF\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FC': {
        'en' : u':men_holding_hands_dark_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':händchenhaltende_männer_dunkle_hautfarbe,_mittelhelle_hautfarbe:',
        'es': u':hombres_de_la_mano_tono_de_piel_oscuro_y_tono_de_piel_claro_medio:',
        'fr': u':deux_hommes_se_tenant_la_main_peau_foncée_et_peau_moyennement_claire:',
        'pt': u':dois_homens_de_mãos_dadas_pele_escura_e_pele_morena_clara:',
        'it': u':due_uomini_che_si_tengono_per_mano_carnagione_scura_e_carnagione_abbastanza_chiara:'
    },
    u'\U0001F468\U0001F3FF\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FD': {
        'en' : u':men_holding_hands_dark_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':händchenhaltende_männer_dunkle_hautfarbe,_mittlere_hautfarbe:',
        'es': u':hombres_de_la_mano_tono_de_piel_oscuro_y_tono_de_piel_medio:',
        'fr': u':deux_hommes_se_tenant_la_main_peau_foncée_et_peau_légèrement_mate:',
        'pt': u':dois_homens_de_mãos_dadas_pele_escura_e_pele_morena:',
        'it': u':due_uomini_che_si_tengono_per_mano_carnagione_scura_e_carnagione_olivastra:'
    },
    u'\U0001F46C\U0001F3FB': {
        'en' : u':men_holding_hands_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':händchenhaltende_männer_helle_hautfarbe:',
        'es': u':hombres_de_la_mano_tono_de_piel_claro:',
        'fr': u':deux_hommes_se_tenant_la_main_peau_claire:',
        'pt': u':dois_homens_de_mãos_dadas_pele_clara:',
        'it': u':due_uomini_che_si_tengono_per_mano_carnagione_chiara:'
    },
    u'\U0001F468\U0001F3FB\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FF': {
        'en' : u':men_holding_hands_light_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':händchenhaltende_männer_helle_hautfarbe,_dunkle_hautfarbe:',
        'es': u':hombres_de_la_mano_tono_de_piel_claro_y_tono_de_piel_oscuro:',
        'fr': u':deux_hommes_se_tenant_la_main_peau_claire_et_peau_foncée:',
        'pt': u':dois_homens_de_mãos_dadas_pele_clara_e_pele_escura:',
        'it': u':due_uomini_che_si_tengono_per_mano_carnagione_chiara_e_carnagione_scura:'
    },
    u'\U0001F468\U0001F3FB\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FE': {
        'en' : u':men_holding_hands_light_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':händchenhaltende_männer_helle_hautfarbe,_mitteldunkle_hautfarbe:',
        'es': u':hombres_de_la_mano_tono_de_piel_claro_y_tono_de_piel_oscuro_medio:',
        'fr': u':deux_hommes_se_tenant_la_main_peau_claire_et_peau_mate:',
        'pt': u':dois_homens_de_mãos_dadas_pele_clara_e_pele_morena_escura:',
        'it': u':due_uomini_che_si_tengono_per_mano_carnagione_chiara_e_carnagione_abbastanza_scura:'
    },
    u'\U0001F468\U0001F3FB\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FC': {
        'en' : u':men_holding_hands_light_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':händchenhaltende_männer_helle_hautfarbe,_mittelhelle_hautfarbe:',
        'es': u':hombres_de_la_mano_tono_de_piel_claro_y_tono_de_piel_claro_medio:',
        'fr': u':deux_hommes_se_tenant_la_main_peau_claire_et_peau_moyennement_claire:',
        'pt': u':dois_homens_de_mãos_dadas_pele_clara_e_pele_morena_clara:',
        'it': u':due_uomini_che_si_tengono_per_mano_carnagione_chiara_e_carnagione_abbastanza_chiara:'
    },
    u'\U0001F468\U0001F3FB\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FD': {
        'en' : u':men_holding_hands_light_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':händchenhaltende_männer_helle_hautfarbe,_mittlere_hautfarbe:',
        'es': u':hombres_de_la_mano_tono_de_piel_claro_y_tono_de_piel_medio:',
        'fr': u':deux_hommes_se_tenant_la_main_peau_claire_et_peau_légèrement_mate:',
        'pt': u':dois_homens_de_mãos_dadas_pele_clara_e_pele_morena:',
        'it': u':due_uomini_che_si_tengono_per_mano_carnagione_chiara_e_carnagione_olivastra:'
    },
    u'\U0001F46C\U0001F3FE': {
        'en' : u':men_holding_hands_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':händchenhaltende_männer_mitteldunkle_hautfarbe:',
        'es': u':hombres_de_la_mano_tono_de_piel_oscuro_medio:',
        'fr': u':deux_hommes_se_tenant_la_main_peau_mate:',
        'pt': u':dois_homens_de_mãos_dadas_pele_morena_escura:',
        'it': u':due_uomini_che_si_tengono_per_mano_carnagione_abbastanza_scura:'
    },
    u'\U0001F468\U0001F3FE\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FF': {
        'en' : u':men_holding_hands_medium-dark_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':händchenhaltende_männer_mitteldunkle_hautfarbe,_dunkle_hautfarbe:',
        'es': u':hombres_de_la_mano_tono_de_piel_oscuro_medio_y_tono_de_piel_oscuro:',
        'fr': u':deux_hommes_se_tenant_la_main_peau_mate_et_peau_foncée:',
        'pt': u':dois_homens_de_mãos_dadas_pele_morena_escura_e_pele_escura:',
        'it': u':due_uomini_che_si_tengono_per_mano_carnagione_abbastanza_scura_e_carnagione_scura:'
    },
    u'\U0001F468\U0001F3FE\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FB': {
        'en' : u':men_holding_hands_medium-dark_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':händchenhaltende_männer_mitteldunkle_hautfarbe,_helle_hautfarbe:',
        'es': u':hombres_de_la_mano_tono_de_piel_oscuro_medio_y_tono_de_piel_claro:',
        'fr': u':deux_hommes_se_tenant_la_main_peau_mate_et_peau_claire:',
        'pt': u':dois_homens_de_mãos_dadas_pele_morena_escura_e_pele_clara:',
        'it': u':due_uomini_che_si_tengono_per_mano_carnagione_abbastanza_scura_e_carnagione_chiara:'
    },
    u'\U0001F468\U0001F3FE\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FC': {
        'en' : u':men_holding_hands_medium-dark_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':händchenhaltende_männer_mitteldunkle_hautfarbe,_mittelhelle_hautfarbe:',
        'es': u':hombres_de_la_mano_tono_de_piel_oscuro_medio_y_tono_de_piel_claro_medio:',
        'fr': u':deux_hommes_se_tenant_la_main_peau_mate_et_peau_moyennement_claire:',
        'pt': u':dois_homens_de_mãos_dadas_pele_morena_escura_e_pele_morena_clara:',
        'it': u':due_uomini_che_si_tengono_per_mano_carnagione_abbastanza_scura_e_carnagione_abbastanza_chiara:'
    },
    u'\U0001F468\U0001F3FE\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FD': {
        'en' : u':men_holding_hands_medium-dark_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':händchenhaltende_männer_mitteldunkle_hautfarbe,_mittlere_hautfarbe:',
        'es': u':hombres_de_la_mano_tono_de_piel_oscuro_medio_y_tono_de_piel_medio:',
        'fr': u':deux_hommes_se_tenant_la_main_peau_mate_et_peau_légèrement_mate:',
        'pt': u':dois_homens_de_mãos_dadas_pele_morena_escura_e_pele_morena:',
        'it': u':due_uomini_che_si_tengono_per_mano_carnagione_abbastanza_scura_e_carnagione_olivastra:'
    },
    u'\U0001F46C\U0001F3FC': {
        'en' : u':men_holding_hands_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':händchenhaltende_männer_mittelhelle_hautfarbe:',
        'es': u':hombres_de_la_mano_tono_de_piel_claro_medio:',
        'fr': u':deux_hommes_se_tenant_la_main_peau_moyennement_claire:',
        'pt': u':dois_homens_de_mãos_dadas_pele_morena_clara:',
        'it': u':due_uomini_che_si_tengono_per_mano_carnagione_abbastanza_chiara:'
    },
    u'\U0001F468\U0001F3FC\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FF': {
        'en' : u':men_holding_hands_medium-light_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':händchenhaltende_männer_mittelhelle_hautfarbe,_dunkle_hautfarbe:',
        'es': u':hombres_de_la_mano_tono_de_piel_claro_medio_y_tono_de_piel_oscuro:',
        'fr': u':deux_hommes_se_tenant_la_main_peau_moyennement_claire_et_peau_foncée:',
        'pt': u':dois_homens_de_mãos_dadas_pele_morena_clara_e_pele_escura:',
        'it': u':due_uomini_che_si_tengono_per_mano_carnagione_abbastanza_chiara_e_carnagione_scura:'
    },
    u'\U0001F468\U0001F3FC\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FB': {
        'en' : u':men_holding_hands_medium-light_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':händchenhaltende_männer_mittelhelle_hautfarbe,_helle_hautfarbe:',
        'es': u':hombres_de_la_mano_tono_de_piel_claro_medio_y_tono_de_piel_claro:',
        'fr': u':deux_hommes_se_tenant_la_main_peau_moyennement_claire_et_peau_claire:',
        'pt': u':dois_homens_de_mãos_dadas_pele_morena_clara_e_pele_clara:',
        'it': u':due_uomini_che_si_tengono_per_mano_carnagione_abbastanza_chiara_e_carnagione_chiara:'
    },
    u'\U0001F468\U0001F3FC\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FE': {
        'en' : u':men_holding_hands_medium-light_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':händchenhaltende_männer_mittelhelle_hautfarbe,_mitteldunkle_hautfarbe:',
        'es': u':hombres_de_la_mano_tono_de_piel_claro_medio_y_tono_de_piel_oscuro_medio:',
        'fr': u':deux_hommes_se_tenant_la_main_peau_moyennement_claire_et_peau_mate:',
        'pt': u':dois_homens_de_mãos_dadas_pele_morena_clara_e_pele_morena_escura:',
        'it': u':due_uomini_che_si_tengono_per_mano_carnagione_abbastanza_chiara_e_carnagione_abbastanza_scura:'
    },
    u'\U0001F468\U0001F3FC\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FD': {
        'en' : u':men_holding_hands_medium-light_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':händchenhaltende_männer_mittelhelle_hautfarbe,_mittlere_hautfarbe:',
        'es': u':hombres_de_la_mano_tono_de_piel_claro_medio_y_tono_de_piel_medio:',
        'fr': u':deux_hommes_se_tenant_la_main_peau_moyennement_claire_et_peau_légèrement_mate:',
        'pt': u':dois_homens_de_mãos_dadas_pele_morena_clara_e_pele_morena:',
        'it': u':due_uomini_che_si_tengono_per_mano_carnagione_abbastanza_chiara_e_carnagione_olivastra:'
    },
    u'\U0001F46C\U0001F3FD': {
        'en' : u':men_holding_hands_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':händchenhaltende_männer_mittlere_hautfarbe:',
        'es': u':hombres_de_la_mano_tono_de_piel_medio:',
        'fr': u':deux_hommes_se_tenant_la_main_peau_légèrement_mate:',
        'pt': u':dois_homens_de_mãos_dadas_pele_morena:',
        'it': u':due_uomini_che_si_tengono_per_mano_carnagione_olivastra:'
    },
    u'\U0001F468\U0001F3FD\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FF': {
        'en' : u':men_holding_hands_medium_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':händchenhaltende_männer_mittlere_hautfarbe,_dunkle_hautfarbe:',
        'es': u':hombres_de_la_mano_tono_de_piel_medio_y_tono_de_piel_oscuro:',
        'fr': u':deux_hommes_se_tenant_la_main_peau_légèrement_mate_et_peau_foncée:',
        'pt': u':dois_homens_de_mãos_dadas_pele_morena_e_pele_escura:',
        'it': u':due_uomini_che_si_tengono_per_mano_carnagione_olivastra_e_carnagione_scura:'
    },
    u'\U0001F468\U0001F3FD\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FB': {
        'en' : u':men_holding_hands_medium_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':händchenhaltende_männer_mittlere_hautfarbe,_helle_hautfarbe:',
        'es': u':hombres_de_la_mano_tono_de_piel_medio_y_tono_de_piel_claro:',
        'fr': u':deux_hommes_se_tenant_la_main_peau_légèrement_mate_et_peau_claire:',
        'pt': u':dois_homens_de_mãos_dadas_pele_morena_e_pele_clara:',
        'it': u':due_uomini_che_si_tengono_per_mano_carnagione_olivastra_e_carnagione_chiara:'
    },
    u'\U0001F468\U0001F3FD\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FE': {
        'en' : u':men_holding_hands_medium_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':händchenhaltende_männer_mittlere_hautfarbe,_mitteldunkle_hautfarbe:',
        'es': u':hombres_de_la_mano_tono_de_piel_medio_y_tono_de_piel_oscuro_medio:',
        'fr': u':deux_hommes_se_tenant_la_main_peau_légèrement_mate_et_peau_mate:',
        'pt': u':dois_homens_de_mãos_dadas_pele_morena_e_pele_morena_escura:',
        'it': u':due_uomini_che_si_tengono_per_mano_carnagione_olivastra_e_carnagione_abbastanza_scura:'
    },
    u'\U0001F468\U0001F3FD\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FC': {
        'en' : u':men_holding_hands_medium_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':händchenhaltende_männer_mittlere_hautfarbe,_mittelhelle_hautfarbe:',
        'es': u':hombres_de_la_mano_tono_de_piel_medio_y_tono_de_piel_claro_medio:',
        'fr': u':deux_hommes_se_tenant_la_main_peau_légèrement_mate_et_peau_moyennement_claire:',
        'pt': u':dois_homens_de_mãos_dadas_pele_morena_e_pele_morena_clara:',
        'it': u':due_uomini_che_si_tengono_per_mano_carnagione_olivastra_e_carnagione_abbastanza_chiara:'
    },
    u'\U0001F46F\U0000200D\U00002642\U0000FE0F': {
        'en' : u':men_with_bunny_ears:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':männer_mit_hasenohren:',
        'es': u':hombres_con_orejas_de_conejo:',
        'fr': u':hommes_avec_des_oreilles_de_lapin:',
        'pt': u':homens_com_orelhas_de_coelho:',
        'it': u':uomini_con_orecchie_da_coniglio:'
    },
    u'\U0001F46F\U0000200D\U00002642': {
        'en' : u':men_with_bunny_ears:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F93C\U0000200D\U00002642\U0000FE0F': {
        'en' : u':men_wrestling:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':ringende_männer:',
        'es': u':hombres_luchando:',
        'fr': u':lutteurs:',
        'pt': u':homens_lutando:',
        'it': u':lottatori:'
    },
    u'\U0001F93C\U0000200D\U00002642': {
        'en' : u':men_wrestling:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U00002764\U0000FE0F\U0000200D\U0001FA79': {
        'en' : u':mending_heart:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U00002764\U0000200D\U0001FA79': {
        'en' : u':mending_heart:',
        'status' : unqualified,
        'E' : 13.1
    },
    u'\U0001F54E': {
        'en' : u':menorah:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [u':menorah_with_nine_branches:'],
        'de': u':menora:',
        'es': u':menorá:',
        'fr': u':chandelier_à_sept_branches:',
        'pt': u':menorá:',
        'it': u':menorah:'
    },
    u'\U0001F6B9': {
        'en' : u':men’s_room:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':mens:'],
        'variant': True,
        'de': u':herren:',
        'es': u':aseo_para_hombres:',
        'fr': u':symbole_toilettes_hommes:',
        'pt': u':banheiro_masculino:',
        'it': u':simbolo_del_bagno_degli_uomini:'
    },
    u'\U0001F9DC\U0000200D\U00002640\U0000FE0F': {
        'en' : u':mermaid:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':meerjungfrau:',
        'es': u':sirena:',
        'fr': u':sirène:',
        'pt': u':sereia:',
        'it': u':sirena_donna:'
    },
    u'\U0001F9DC\U0000200D\U00002640': {
        'en' : u':mermaid:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F9DC\U0001F3FF\U0000200D\U00002640\U0000FE0F': {
        'en' : u':mermaid_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':meerjungfrau_dunkle_hautfarbe:',
        'es': u':sirena_tono_de_piel_oscuro:',
        'fr': u':sirène_peau_foncée:',
        'pt': u':sereia_pele_escura:',
        'it': u':sirena_donna_carnagione_scura:'
    },
    u'\U0001F9DC\U0001F3FF\U0000200D\U00002640': {
        'en' : u':mermaid_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F9DC\U0001F3FB\U0000200D\U00002640\U0000FE0F': {
        'en' : u':mermaid_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':meerjungfrau_helle_hautfarbe:',
        'es': u':sirena_tono_de_piel_claro:',
        'fr': u':sirène_peau_claire:',
        'pt': u':sereia_pele_clara:',
        'it': u':sirena_donna_carnagione_chiara:'
    },
    u'\U0001F9DC\U0001F3FB\U0000200D\U00002640': {
        'en' : u':mermaid_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F9DC\U0001F3FE\U0000200D\U00002640\U0000FE0F': {
        'en' : u':mermaid_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':meerjungfrau_mitteldunkle_hautfarbe:',
        'es': u':sirena_tono_de_piel_oscuro_medio:',
        'fr': u':sirène_peau_mate:',
        'pt': u':sereia_pele_morena_escura:',
        'it': u':sirena_donna_carnagione_abbastanza_scura:'
    },
    u'\U0001F9DC\U0001F3FE\U0000200D\U00002640': {
        'en' : u':mermaid_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F9DC\U0001F3FC\U0000200D\U00002640\U0000FE0F': {
        'en' : u':mermaid_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':meerjungfrau_mittelhelle_hautfarbe:',
        'es': u':sirena_tono_de_piel_claro_medio:',
        'fr': u':sirène_peau_moyennement_claire:',
        'pt': u':sereia_pele_morena_clara:',
        'it': u':sirena_donna_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9DC\U0001F3FC\U0000200D\U00002640': {
        'en' : u':mermaid_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F9DC\U0001F3FD\U0000200D\U00002640\U0000FE0F': {
        'en' : u':mermaid_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':meerjungfrau_mittlere_hautfarbe:',
        'es': u':sirena_tono_de_piel_medio:',
        'fr': u':sirène_peau_légèrement_mate:',
        'pt': u':sereia_pele_morena:',
        'it': u':sirena_donna_carnagione_olivastra:'
    },
    u'\U0001F9DC\U0001F3FD\U0000200D\U00002640': {
        'en' : u':mermaid_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F9DC\U0000200D\U00002642\U0000FE0F': {
        'en' : u':merman:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':wassermann:',
        'es': u':sirena_hombre:',
        'fr': u':triton:',
        'pt': u':sereio:',
        'it': u':tritone:'
    },
    u'\U0001F9DC\U0000200D\U00002642': {
        'en' : u':merman:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F9DC\U0001F3FF\U0000200D\U00002642\U0000FE0F': {
        'en' : u':merman_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':wassermann_dunkle_hautfarbe:',
        'es': u':sirena_hombre_tono_de_piel_oscuro:',
        'fr': u':triton_peau_foncée:',
        'pt': u':sereio_pele_escura:',
        'it': u':tritone_carnagione_scura:'
    },
    u'\U0001F9DC\U0001F3FF\U0000200D\U00002642': {
        'en' : u':merman_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F9DC\U0001F3FB\U0000200D\U00002642\U0000FE0F': {
        'en' : u':merman_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':wassermann_helle_hautfarbe:',
        'es': u':sirena_hombre_tono_de_piel_claro:',
        'fr': u':triton_peau_claire:',
        'pt': u':sereio_pele_clara:',
        'it': u':tritone_carnagione_chiara:'
    },
    u'\U0001F9DC\U0001F3FB\U0000200D\U00002642': {
        'en' : u':merman_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F9DC\U0001F3FE\U0000200D\U00002642\U0000FE0F': {
        'en' : u':merman_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':wassermann_mitteldunkle_hautfarbe:',
        'es': u':sirena_hombre_tono_de_piel_oscuro_medio:',
        'fr': u':triton_peau_mate:',
        'pt': u':sereio_pele_morena_escura:',
        'it': u':tritone_carnagione_abbastanza_scura:'
    },
    u'\U0001F9DC\U0001F3FE\U0000200D\U00002642': {
        'en' : u':merman_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F9DC\U0001F3FC\U0000200D\U00002642\U0000FE0F': {
        'en' : u':merman_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':wassermann_mittelhelle_hautfarbe:',
        'es': u':sirena_hombre_tono_de_piel_claro_medio:',
        'fr': u':triton_peau_moyennement_claire:',
        'pt': u':sereio_pele_morena_clara:',
        'it': u':tritone_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9DC\U0001F3FC\U0000200D\U00002642': {
        'en' : u':merman_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F9DC\U0001F3FD\U0000200D\U00002642\U0000FE0F': {
        'en' : u':merman_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':wassermann_mittlere_hautfarbe:',
        'es': u':sirena_hombre_tono_de_piel_medio:',
        'fr': u':triton_peau_légèrement_mate:',
        'pt': u':sereio_pele_morena:',
        'it': u':tritone_carnagione_olivastra:'
    },
    u'\U0001F9DC\U0001F3FD\U0000200D\U00002642': {
        'en' : u':merman_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F9DC': {
        'en' : u':merperson:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':wassermensch:',
        'es': u':persona_sirena:',
        'fr': u':créature_aquatique:',
        'pt': u':pessoa_sereia:',
        'it': u':sirena:'
    },
    u'\U0001F9DC\U0001F3FF': {
        'en' : u':merperson_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':wassermensch_dunkle_hautfarbe:',
        'es': u':persona_sirena_tono_de_piel_oscuro:',
        'fr': u':créature_aquatique_peau_foncée:',
        'pt': u':pessoa_sereia_pele_escura:',
        'it': u':sirena_carnagione_scura:'
    },
    u'\U0001F9DC\U0001F3FB': {
        'en' : u':merperson_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':wassermensch_helle_hautfarbe:',
        'es': u':persona_sirena_tono_de_piel_claro:',
        'fr': u':créature_aquatique_peau_claire:',
        'pt': u':pessoa_sereia_pele_clara:',
        'it': u':sirena_carnagione_chiara:'
    },
    u'\U0001F9DC\U0001F3FE': {
        'en' : u':merperson_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':wassermensch_mitteldunkle_hautfarbe:',
        'es': u':persona_sirena_tono_de_piel_oscuro_medio:',
        'fr': u':créature_aquatique_peau_mate:',
        'pt': u':pessoa_sereia_pele_morena_escura:',
        'it': u':sirena_carnagione_abbastanza_scura:'
    },
    u'\U0001F9DC\U0001F3FC': {
        'en' : u':merperson_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':wassermensch_mittelhelle_hautfarbe:',
        'es': u':persona_sirena_tono_de_piel_claro_medio:',
        'fr': u':créature_aquatique_peau_moyennement_claire:',
        'pt': u':pessoa_sereia_pele_morena_clara:',
        'it': u':sirena_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9DC\U0001F3FD': {
        'en' : u':merperson_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':wassermensch_mittlere_hautfarbe:',
        'es': u':persona_sirena_tono_de_piel_medio:',
        'fr': u':créature_aquatique_peau_légèrement_mate:',
        'pt': u':pessoa_sereia_pele_morena:',
        'it': u':sirena_carnagione_olivastra:'
    },
    u'\U0001F687': {
        'en' : u':metro:',
        'status' : fully_qualified,
        'E' : 0.6,
        'variant': True,
        'de': u':u-bahn:',
        'es': u':metro:',
        'fr': u':métro:',
        'pt': u':metrô:',
        'it': u':metropolitana:'
    },
    u'\U0001F9A0': {
        'en' : u':microbe:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':mikrobe:',
        'es': u':microbio:',
        'fr': u':microbe:',
        'pt': u':micróbio:',
        'it': u':microbo:'
    },
    u'\U0001F3A4': {
        'en' : u':microphone:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':mikrofon:',
        'es': u':micrófono:',
        'fr': u':micro:',
        'pt': u':microfone:',
        'it': u':microfono:'
    },
    u'\U0001F52C': {
        'en' : u':microscope:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':mikroskop:',
        'es': u':microscopio:',
        'fr': u':microscope:',
        'pt': u':microscópio:',
        'it': u':microscopio:'
    },
    u'\U0001F595': {
        'en' : u':middle_finger:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [u':fu:', u':reversed_hand_with_middle_finger_extended:'],
        'de': u':mittelfinger:',
        'es': u':dedo_corazón_hacia_arriba:',
        'fr': u':doigt_d’honneur:',
        'pt': u':dedo_do_meio:',
        'it': u':dito_medio:'
    },
    u'\U0001F595\U0001F3FF': {
        'en' : u':middle_finger_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':mittelfinger_dunkle_hautfarbe:',
        'es': u':dedo_corazón_hacia_arriba_tono_de_piel_oscuro:',
        'fr': u':doigt_d’honneur_peau_foncée:',
        'pt': u':dedo_do_meio_pele_escura:',
        'it': u':dito_medio_carnagione_scura:'
    },
    u'\U0001F595\U0001F3FB': {
        'en' : u':middle_finger_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':mittelfinger_helle_hautfarbe:',
        'es': u':dedo_corazón_hacia_arriba_tono_de_piel_claro:',
        'fr': u':doigt_d’honneur_peau_claire:',
        'pt': u':dedo_do_meio_pele_clara:',
        'it': u':dito_medio_carnagione_chiara:'
    },
    u'\U0001F595\U0001F3FE': {
        'en' : u':middle_finger_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':mittelfinger_mitteldunkle_hautfarbe:',
        'es': u':dedo_corazón_hacia_arriba_tono_de_piel_oscuro_medio:',
        'fr': u':doigt_d’honneur_peau_mate:',
        'pt': u':dedo_do_meio_pele_morena_escura:',
        'it': u':dito_medio_carnagione_abbastanza_scura:'
    },
    u'\U0001F595\U0001F3FC': {
        'en' : u':middle_finger_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':mittelfinger_mittelhelle_hautfarbe:',
        'es': u':dedo_corazón_hacia_arriba_tono_de_piel_claro_medio:',
        'fr': u':doigt_d’honneur_peau_moyennement_claire:',
        'pt': u':dedo_do_meio_pele_morena_clara:',
        'it': u':dito_medio_carnagione_abbastanza_chiara:'
    },
    u'\U0001F595\U0001F3FD': {
        'en' : u':middle_finger_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':mittelfinger_mittlere_hautfarbe:',
        'es': u':dedo_corazón_hacia_arriba_tono_de_piel_medio:',
        'fr': u':doigt_d’honneur_peau_légèrement_mate:',
        'pt': u':dedo_do_meio_pele_morena:',
        'it': u':dito_medio_carnagione_olivastra:'
    },
    u'\U0001FA96': {
        'en' : u':military_helmet:',
        'status' : fully_qualified,
        'E' : 13,
        'de': u':militärhelm:',
        'es': u':casco_militar:',
        'fr': u':casque_militaire:',
        'pt': u':capacete_militar:',
        'it': u':elmetto_militare:'
    },
    u'\U0001F396\U0000FE0F': {
        'en' : u':military_medal:',
        'status' : fully_qualified,
        'E' : 0.7,
        'alias' : [u':medal_military:'],
        'variant': True,
        'de': u':militärorden:',
        'es': u':medalla_militar:',
        'fr': u':médaille_militaire:',
        'pt': u':medalha_militar:',
        'it': u':medaglia_militare:'
    },
    u'\U0001F396': {
        'en' : u':military_medal:',
        'status' : unqualified,
        'E' : 0.7,
        'alias' : [u':medal_military:'],
        'variant': True,
        'de': u':militärorden:',
        'es': u':medalla_militar:',
        'fr': u':médaille_militaire:',
        'pt': u':medalha_militar:',
        'it': u':medaglia_militare:'
    },
    u'\U0001F30C': {
        'en' : u':milky_way:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':milchstraße:',
        'es': u':vía_láctea:',
        'fr': u':voie_lactée:',
        'pt': u':via_láctea:',
        'it': u':via_lattea:'
    },
    u'\U0001F690': {
        'en' : u':minibus:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':kleinbus:',
        'es': u':minibús:',
        'fr': u':minibus:',
        'pt': u':van:',
        'it': u':pulmino:'
    },
    u'\U00002796': {
        'en' : u':minus:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':heavy_minus_sign:'],
        'de': u':minuszeichen:',
        'es': u':menos:',
        'fr': u':signe_moins:',
        'pt': u':símbolo_de_subtração:',
        'it': u':simbolo_della_sottrazione:'
    },
    u'\U0001FA9E': {
        'en' : u':mirror:',
        'status' : fully_qualified,
        'E' : 13,
        'de': u':spiegel:',
        'es': u':espejo:',
        'fr': u':miroir:',
        'pt': u':espelho:',
        'it': u':specchio:'
    },
    u'\U0001FAA9': {
        'en' : u':mirror_ball:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001F5FF': {
        'en' : u':moai:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':moyai:'],
        'de': u':statue:',
        'es': u':estatua_moái:',
        'fr': u':moai:',
        'pt': u':moai:',
        'it': u':moai:'
    },
    u'\U0001F4F1': {
        'en' : u':mobile_phone:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':iphone:'],
        'de': u':mobiltelefon:',
        'es': u':teléfono_móvil:',
        'fr': u':téléphone_portable:',
        'pt': u':telefone_celular:',
        'it': u':telefono_cellulare:'
    },
    u'\U0001F4F4': {
        'en' : u':mobile_phone_off:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':mobiltelefon_aus:',
        'es': u':teléfono_móvil_apagado:',
        'fr': u':téléphone_éteint:',
        'pt': u':telefone_celular_desligado:',
        'it': u':cellulare_spento:'
    },
    u'\U0001F4F2': {
        'en' : u':mobile_phone_with_arrow:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':calling:'],
        'de': u':mobiltelefon_mit_pfeil:',
        'es': u':móvil_con_una_flecha:',
        'fr': u':appel_entrant:',
        'pt': u':telefone_celular_com_seta:',
        'it': u':telefono_cellulare_con_freccia:'
    },
    u'\U0001F911': {
        'en' : u':money-mouth_face:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [u':money_mouth_face:', u':money__mouth_face:'],
        'de': u':gesicht_mit_dollarzeichen:',
        'es': u':cara_con_lengua_de_dinero:',
        'fr': u':argent_dans_les_yeux_et_la_bouche:',
        'pt': u':rosto_com_cifrões:',
        'it': u':faccina_avida_di_denaro:'
    },
    u'\U0001F4B0': {
        'en' : u':money_bag:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':moneybag:'],
        'variant': True,
        'de': u':geldsack:',
        'es': u':bolsa_de_dinero:',
        'fr': u':sac_plein_d’argent:',
        'pt': u':saco_de_dinheiro:',
        'it': u':sacco_di_soldi:'
    },
    u'\U0001F4B8': {
        'en' : u':money_with_wings:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':geldschein_mit_flügeln:',
        'es': u':billete_con_alas:',
        'fr': u':billet_avec_des_ailes:',
        'pt': u':dinheiro_voando:',
        'it': u':soldi_con_le_ali:'
    },
    u'\U0001F412': {
        'en' : u':monkey:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':affe:',
        'es': u':mono:',
        'fr': u':singe:',
        'pt': u':macaco:',
        'it': u':scimmia:'
    },
    u'\U0001F435': {
        'en' : u':monkey_face:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':affengesicht:',
        'es': u':cara_de_mono:',
        'fr': u':tête_de_singe:',
        'pt': u':rosto_de_macaco:',
        'it': u':muso_di_scimmia:'
    },
    u'\U0001F69D': {
        'en' : u':monorail:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':einschienenbahn:',
        'es': u':monorraíl:',
        'fr': u':monorail:',
        'pt': u':monotrilho:',
        'it': u':monorotaia:'
    },
    u'\U0001F96E': {
        'en' : u':moon_cake:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':mondkuchen:',
        'es': u':pastel_de_luna:',
        'fr': u':gâteau_de_lune:',
        'pt': u':bolo_lunar:',
        'it': u':torta_della_luna:'
    },
    u'\U0001F391': {
        'en' : u':moon_viewing_ceremony:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':rice_scene:'],
        'de': u':traditionelles_mondfest:',
        'es': u':ceremonia_de_contemplación_de_la_luna:',
        'fr': u':cérémonie_de_la_lune:',
        'pt': u':contemplação_da_lua:',
        'it': u':contemplazione_della_luna:'
    },
    u'\U0001F54C': {
        'en' : u':mosque:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':moschee:',
        'es': u':mezquita:',
        'fr': u':mosquée:',
        'pt': u':mesquita:',
        'it': u':moschea:'
    },
    u'\U0001F99F': {
        'en' : u':mosquito:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':moskito:',
        'es': u':mosquito:',
        'fr': u':moustique:',
        'pt': u':mosquito:',
        'it': u':zanzara:'
    },
    u'\U0001F6E5\U0000FE0F': {
        'en' : u':motor_boat:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': u':motorboot:',
        'es': u':barco_a_motor:',
        'fr': u':bateau_à_moteur:',
        'pt': u':barco:',
        'it': u':barca_a_motore:'
    },
    u'\U0001F6E5': {
        'en' : u':motor_boat:',
        'status' : unqualified,
        'E' : 0.7,
        'variant': True,
        'de': u':motorboot:',
        'es': u':barco_a_motor:',
        'fr': u':bateau_à_moteur:',
        'pt': u':barco:',
        'it': u':barca_a_motore:'
    },
    u'\U0001F6F5': {
        'en' : u':motor_scooter:',
        'status' : fully_qualified,
        'E' : 3,
        'de': u':motorroller:',
        'es': u':scooter:',
        'fr': u':scooter:',
        'pt': u':scooter:',
        'it': u':scooter:'
    },
    u'\U0001F3CD\U0000FE0F': {
        'en' : u':motorcycle:',
        'status' : fully_qualified,
        'E' : 0.7,
        'alias' : [u':racing_motorcycle:'],
        'variant': True,
        'de': u':motorrad:',
        'es': u':moto:',
        'fr': u':moto:',
        'pt': u':motocicleta:',
        'it': u':motocicletta:'
    },
    u'\U0001F3CD': {
        'en' : u':motorcycle:',
        'status' : unqualified,
        'E' : 0.7,
        'alias' : [u':racing_motorcycle:'],
        'variant': True,
        'de': u':motorrad:',
        'es': u':moto:',
        'fr': u':moto:',
        'pt': u':motocicleta:',
        'it': u':motocicletta:'
    },
    u'\U0001F9BC': {
        'en' : u':motorized_wheelchair:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':elektrischer_rollstuhl:',
        'es': u':silla_de_ruedas_eléctrica:',
        'fr': u':fauteuil_motorisé:',
        'pt': u':cadeira_de_rodas_motorizada:',
        'it': u':sedia_a_rotelle_motorizzata:'
    },
    u'\U0001F6E3\U0000FE0F': {
        'en' : u':motorway:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': u':autobahn:',
        'es': u':autopista:',
        'fr': u':autoroute:',
        'pt': u':estrada:',
        'it': u':autostrada:'
    },
    u'\U0001F6E3': {
        'en' : u':motorway:',
        'status' : unqualified,
        'E' : 0.7,
        'variant': True,
        'de': u':autobahn:',
        'es': u':autopista:',
        'fr': u':autoroute:',
        'pt': u':estrada:',
        'it': u':autostrada:'
    },
    u'\U0001F5FB': {
        'en' : u':mount_fuji:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':fuji:',
        'es': u':monte_fuji:',
        'fr': u':mont_fuji:',
        'pt': u':monte_fuji:',
        'it': u':monte_fuji:'
    },
    u'\U000026F0\U0000FE0F': {
        'en' : u':mountain:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': u':berg:',
        'es': u':montaña:',
        'fr': u':montagne:',
        'pt': u':montanha:',
        'it': u':montagna:'
    },
    u'\U000026F0': {
        'en' : u':mountain:',
        'status' : unqualified,
        'E' : 0.7,
        'variant': True,
        'de': u':berg:',
        'es': u':montaña:',
        'fr': u':montagne:',
        'pt': u':montanha:',
        'it': u':montagna:'
    },
    u'\U0001F6A0': {
        'en' : u':mountain_cableway:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':bergschwebebahn:',
        'es': u':teleférico_de_montaña:',
        'fr': u':téléphérique:',
        'pt': u':cabo_suspenso_usado_em_teleféricos_nas_montanhas:',
        'it': u':funicolare:'
    },
    u'\U0001F69E': {
        'en' : u':mountain_railway:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':bergbahn:',
        'es': u':ferrocarril_de_montaña:',
        'fr': u':train_de_montagne:',
        'pt': u':estrada_de_ferro_na_montanha:',
        'it': u':ferrovia_di_montagna:'
    },
    u'\U0001F401': {
        'en' : u':mouse:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [u':mouse2:'],
        'de': u':maus:',
        'es': u':ratón:',
        'fr': u':souris:',
        'pt': u':camundongo:',
        'it': u':topo:'
    },
    u'\U0001F42D': {
        'en' : u':mouse_face:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':mouse:'],
        'de': u':mäusegesicht:',
        'es': u':cara_de_ratón:',
        'fr': u':tête_de_souris:',
        'pt': u':rosto_de_camundongo:',
        'it': u':muso_di_topo:'
    },
    u'\U0001FAA4': {
        'en' : u':mouse_trap:',
        'status' : fully_qualified,
        'E' : 13,
        'de': u':mäusefalle:',
        'es': u':trampa_de_ratones:',
        'fr': u':tapette_à_souris:',
        'pt': u':ratoeira:',
        'it': u':trappola_per_topi:'
    },
    u'\U0001F444': {
        'en' : u':mouth:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':lips:'],
        'de': u':mund:',
        'es': u':boca:',
        'fr': u':bouche:',
        'pt': u':boca:',
        'it': u':bocca:'
    },
    u'\U0001F3A5': {
        'en' : u':movie_camera:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':filmkamera:',
        'es': u':cámara_de_cine:',
        'fr': u':caméra:',
        'pt': u':câmera_de_cinema:',
        'it': u':cinepresa:'
    },
    u'\U00002716\U0000FE0F': {
        'en' : u':multiply:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':heavy_multiplication_x:'],
        'variant': True,
        'de': u':multiplikationszeichen:',
        'es': u':multiplicación:',
        'fr': u':signe_multiplier:',
        'pt': u':sinal_de_multiplicação:',
        'it': u':segno_moltiplicazione:'
    },
    u'\U00002716': {
        'en' : u':multiply:',
        'status' : unqualified,
        'E' : 0.6,
        'alias' : [u':heavy_multiplication_x:'],
        'variant': True,
        'de': u':multiplikationszeichen:',
        'es': u':multiplicación:',
        'fr': u':signe_multiplier:',
        'pt': u':sinal_de_multiplicação:',
        'it': u':segno_moltiplicazione:'
    },
    u'\U0001F344': {
        'en' : u':mushroom:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':fliegenpilz:',
        'es': u':champiñón:',
        'fr': u':champignon:',
        'pt': u':cogumelo:',
        'it': u':fungo:'
    },
    u'\U0001F3B9': {
        'en' : u':musical_keyboard:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':klaviatur:',
        'es': u':teclado_musical:',
        'fr': u':piano:',
        'pt': u':teclado_musical:',
        'it': u':piano:'
    },
    u'\U0001F3B5': {
        'en' : u':musical_note:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':musiknote:',
        'es': u':nota_musical:',
        'fr': u':note_de_musique:',
        'pt': u':nota_musical:',
        'it': u':nota_musicale:'
    },
    u'\U0001F3B6': {
        'en' : u':musical_notes:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':notes:'],
        'de': u':musiknoten:',
        'es': u':notas_musicales:',
        'fr': u':notes_de_musique:',
        'pt': u':notas_musicais:',
        'it': u':note_musicali:'
    },
    u'\U0001F3BC': {
        'en' : u':musical_score:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':notenschlüssel:',
        'es': u':pentagrama:',
        'fr': u':partition:',
        'pt': u':partitura_musical:',
        'it': u':pentagramma:'
    },
    u'\U0001F507': {
        'en' : u':muted_speaker:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [u':mute:'],
        'de': u':durchgestrichener_lautsprecher:',
        'es': u':altavoz_silenciado:',
        'fr': u':muet:',
        'pt': u':alto-falante_silenciado:',
        'it': u':altoparlante_disattivato:'
    },
    u'\U0001F9D1\U0000200D\U0001F384': {
        'en' : u':mx_claus:',
        'status' : fully_qualified,
        'E' : 13,
        'de': u':weihnachtsperson:',
        'es': u':noel:',
        'fr': u':santa:',
        'pt': u':noel:',
        'it': u':santa_claus:'
    },
    u'\U0001F9D1\U0001F3FF\U0000200D\U0001F384': {
        'en' : u':mx_claus_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13,
        'de': u':weihnachtsperson_dunkle_hautfarbe:',
        'es': u':noel_tono_de_piel_oscuro:',
        'fr': u':santa_peau_foncée:',
        'pt': u':noel_pele_escura:',
        'it': u':santa_claus_carnagione_scura:'
    },
    u'\U0001F9D1\U0001F3FB\U0000200D\U0001F384': {
        'en' : u':mx_claus_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13,
        'de': u':weihnachtsperson_helle_hautfarbe:',
        'es': u':noel_tono_de_piel_claro:',
        'fr': u':santa_peau_claire:',
        'pt': u':noel_pele_clara:',
        'it': u':santa_claus_carnagione_chiara:'
    },
    u'\U0001F9D1\U0001F3FE\U0000200D\U0001F384': {
        'en' : u':mx_claus_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13,
        'de': u':weihnachtsperson_mitteldunkle_hautfarbe:',
        'es': u':noel_tono_de_piel_oscuro_medio:',
        'fr': u':santa_peau_mate:',
        'pt': u':noel_pele_morena_escura:',
        'it': u':santa_claus_carnagione_abbastanza_scura:'
    },
    u'\U0001F9D1\U0001F3FC\U0000200D\U0001F384': {
        'en' : u':mx_claus_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13,
        'de': u':weihnachtsperson_mittelhelle_hautfarbe:',
        'es': u':noel_tono_de_piel_claro_medio:',
        'fr': u':santa_peau_moyennement_claire:',
        'pt': u':noel_pele_morena_clara:',
        'it': u':santa_claus_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9D1\U0001F3FD\U0000200D\U0001F384': {
        'en' : u':mx_claus_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13,
        'de': u':weihnachtsperson_mittlere_hautfarbe:',
        'es': u':noel_tono_de_piel_medio:',
        'fr': u':santa_peau_légèrement_mate:',
        'pt': u':noel_pele_morena:',
        'it': u':santa_claus_carnagione_olivastra:'
    },
    u'\U0001F485': {
        'en' : u':nail_polish:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':nail_care:'],
        'de': u':nagellack:',
        'es': u':pintarse_las_uñas:',
        'fr': u':vernis_à_ongles:',
        'pt': u':esmalte_de_unha:',
        'it': u':smalto_per_unghie:'
    },
    u'\U0001F485\U0001F3FF': {
        'en' : u':nail_polish_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':nagellack_dunkle_hautfarbe:',
        'es': u':pintarse_las_uñas_tono_de_piel_oscuro:',
        'fr': u':vernis_à_ongles_peau_foncée:',
        'pt': u':esmalte_de_unha_pele_escura:',
        'it': u':smalto_per_unghie_carnagione_scura:'
    },
    u'\U0001F485\U0001F3FB': {
        'en' : u':nail_polish_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':nagellack_helle_hautfarbe:',
        'es': u':pintarse_las_uñas_tono_de_piel_claro:',
        'fr': u':vernis_à_ongles_peau_claire:',
        'pt': u':esmalte_de_unha_pele_clara:',
        'it': u':smalto_per_unghie_carnagione_chiara:'
    },
    u'\U0001F485\U0001F3FE': {
        'en' : u':nail_polish_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':nagellack_mitteldunkle_hautfarbe:',
        'es': u':pintarse_las_uñas_tono_de_piel_oscuro_medio:',
        'fr': u':vernis_à_ongles_peau_mate:',
        'pt': u':esmalte_de_unha_pele_morena_escura:',
        'it': u':smalto_per_unghie_carnagione_abbastanza_scura:'
    },
    u'\U0001F485\U0001F3FC': {
        'en' : u':nail_polish_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':nagellack_mittelhelle_hautfarbe:',
        'es': u':pintarse_las_uñas_tono_de_piel_claro_medio:',
        'fr': u':vernis_à_ongles_peau_moyennement_claire:',
        'pt': u':esmalte_de_unha_pele_morena_clara:',
        'it': u':smalto_per_unghie_carnagione_abbastanza_chiara:'
    },
    u'\U0001F485\U0001F3FD': {
        'en' : u':nail_polish_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':nagellack_mittlere_hautfarbe:',
        'es': u':pintarse_las_uñas_tono_de_piel_medio:',
        'fr': u':vernis_à_ongles_peau_légèrement_mate:',
        'pt': u':esmalte_de_unha_pele_morena:',
        'it': u':smalto_per_unghie_carnagione_olivastra:'
    },
    u'\U0001F4DB': {
        'en' : u':name_badge:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':namensschild:',
        'es': u':etiqueta_identificativa:',
        'fr': u':badge_nominatif:',
        'pt': u':crachá:',
        'it': u':tesserino_per_nome:'
    },
    u'\U0001F3DE\U0000FE0F': {
        'en' : u':national_park:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': u':nationalpark:',
        'es': u':parque_nacional:',
        'fr': u':parc_national:',
        'pt': u':parque_nacional:',
        'it': u':parco_nazionale:'
    },
    u'\U0001F3DE': {
        'en' : u':national_park:',
        'status' : unqualified,
        'E' : 0.7,
        'variant': True,
        'de': u':nationalpark:',
        'es': u':parque_nacional:',
        'fr': u':parc_national:',
        'pt': u':parque_nacional:',
        'it': u':parco_nazionale:'
    },
    u'\U0001F922': {
        'en' : u':nauseated_face:',
        'status' : fully_qualified,
        'E' : 3,
        'de': u':würgendes_gesicht:',
        'es': u':cara_de_náuseas:',
        'fr': u':visage_nauséeux:',
        'pt': u':rosto_nauseado:',
        'it': u':faccina_nauseata:'
    },
    u'\U0001F9FF': {
        'en' : u':nazar_amulet:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':nazar-amulett:',
        'es': u':ojo_turco:',
        'fr': u':mauvais_œil:',
        'pt': u':olho_turco:',
        'it': u':occhio_di_allah:'
    },
    u'\U0001F454': {
        'en' : u':necktie:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':hemd_mit_krawatte:',
        'es': u':corbata:',
        'fr': u':cravate:',
        'pt': u':gravata:',
        'it': u':cravatta:'
    },
    u'\U0001F913': {
        'en' : u':nerd_face:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':strebergesicht:',
        'es': u':cara_de_empollón:',
        'fr': u':visage_de_geek:',
        'pt': u':rosto_de_nerd:',
        'it': u':faccina_nerd:'
    },
    u'\U0001FABA': {
        'en' : u':nest_with_eggs:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001FA86': {
        'en' : u':nesting_dolls:',
        'status' : fully_qualified,
        'E' : 13,
        'de': u':matroschka:',
        'es': u':muñeca_rusa:',
        'fr': u':poupées_russes:',
        'pt': u':boneca_russa:',
        'it': u':matrioska:'
    },
    u'\U0001F610': {
        'en' : u':neutral_face:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': u':neutrales_gesicht:',
        'es': u':cara_neutral:',
        'fr': u':visage_neutre:',
        'pt': u':rosto_neutro:',
        'it': u':faccina_neutra:'
    },
    u'\U0001F311': {
        'en' : u':new_moon:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':neumond:',
        'es': u':luna_nueva:',
        'fr': u':nouvelle_lune:',
        'pt': u':lua_nova:',
        'it': u':luna_nuova:'
    },
    u'\U0001F31A': {
        'en' : u':new_moon_face:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [u':new_moon_with_face:'],
        'de': u':neumond_mit_gesicht:',
        'es': u':luna_nueva_con_cara:',
        'fr': u':nouvelle_lune_avec_visage:',
        'pt': u':rosto_da_lua_nova:',
        'it': u':faccina_luna_nuova:'
    },
    u'\U0001F4F0': {
        'en' : u':newspaper:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':zeitung:',
        'es': u':periódico:',
        'fr': u':journal:',
        'pt': u':jornal:',
        'it': u':quotidiano:'
    },
    u'\U000023ED\U0000FE0F': {
        'en' : u':next_track_button:',
        'status' : fully_qualified,
        'E' : 0.7,
        'alias' : [u':black_right__pointing_double_triangle_with_vertical_bar:'],
        'variant': True,
        'de': u':nächster_titel:',
        'es': u':pista_siguiente:',
        'fr': u':bouton_piste_suivante:',
        'pt': u':botão_de_próxima_faixa:',
        'it': u':traccia_successiva:'
    },
    u'\U000023ED': {
        'en' : u':next_track_button:',
        'status' : unqualified,
        'E' : 0.7,
        'alias' : [u':black_right__pointing_double_triangle_with_vertical_bar:'],
        'variant': True,
        'de': u':nächster_titel:',
        'es': u':pista_siguiente:',
        'fr': u':bouton_piste_suivante:',
        'pt': u':botão_de_próxima_faixa:',
        'it': u':traccia_successiva:'
    },
    u'\U0001F303': {
        'en' : u':night_with_stars:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':sternenhimmel:',
        'es': u':noche_estrellada:',
        'fr': u':nuit_étoilée:',
        'pt': u':noite_estrelada:',
        'it': u':notte_stellata:'
    },
    u'\U0001F564': {
        'en' : u':nine-thirty:',
        'status' : fully_qualified,
        'E' : 0.7,
        'alias' : [u':clock930:'],
        'variant': True,
        'de': u':9.30_uhr:',
        'es': u':nueve_y_media:',
        'fr': u':neuf_heures_et_demie:',
        'pt': u':nove_e_meia:',
        'it': u':ore_nove_e_mezza:'
    },
    u'\U0001F558': {
        'en' : u':nine_o’clock:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':clock9:'],
        'variant': True,
        'de': u':9.00_uhr:',
        'es': u':9_en_punto:',
        'fr': u':neuf_heures:',
        'pt': u':9_horas:',
        'it': u':ore_nove:'
    },
    u'\U0001F977': {
        'en' : u':ninja:',
        'status' : fully_qualified,
        'E' : 13,
        'de': u':ninja:',
        'es': u':ninja:',
        'fr': u':ninja:',
        'pt': u':ninja:',
        'it': u':ninja:'
    },
    u'\U0001F977\U0001F3FF': {
        'en' : u':ninja_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13,
        'de': u':ninja_dunkle_hautfarbe:',
        'es': u':ninja_tono_de_piel_oscuro:',
        'fr': u':ninja_peau_foncée:',
        'pt': u':ninja_pele_escura:',
        'it': u':ninja_carnagione_scura:'
    },
    u'\U0001F977\U0001F3FB': {
        'en' : u':ninja_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13,
        'de': u':ninja_helle_hautfarbe:',
        'es': u':ninja_tono_de_piel_claro:',
        'fr': u':ninja_peau_claire:',
        'pt': u':ninja_pele_clara:',
        'it': u':ninja_carnagione_chiara:'
    },
    u'\U0001F977\U0001F3FE': {
        'en' : u':ninja_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13,
        'de': u':ninja_mitteldunkle_hautfarbe:',
        'es': u':ninja_tono_de_piel_oscuro_medio:',
        'fr': u':ninja_peau_mate:',
        'pt': u':ninja_pele_morena_escura:',
        'it': u':ninja_carnagione_abbastanza_scura:'
    },
    u'\U0001F977\U0001F3FC': {
        'en' : u':ninja_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13,
        'de': u':ninja_mittelhelle_hautfarbe:',
        'es': u':ninja_tono_de_piel_claro_medio:',
        'fr': u':ninja_peau_moyennement_claire:',
        'pt': u':ninja_pele_morena_clara:',
        'it': u':ninja_carnagione_abbastanza_chiara:'
    },
    u'\U0001F977\U0001F3FD': {
        'en' : u':ninja_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13,
        'de': u':ninja_mittlere_hautfarbe:',
        'es': u':ninja_tono_de_piel_medio:',
        'fr': u':ninja_peau_légèrement_mate:',
        'pt': u':ninja_pele_morena:',
        'it': u':ninja_carnagione_olivastra:'
    },
    u'\U0001F6B3': {
        'en' : u':no_bicycles:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':fahrräder_verboten:',
        'es': u':bicicletas_prohibidas:',
        'fr': u':vélos_interdits:',
        'pt': u':proibido_andar_de_bicicleta:',
        'it': u':segnale_di_divieto_di_transito_delle_biciclette:'
    },
    u'\U000026D4': {
        'en' : u':no_entry:',
        'status' : fully_qualified,
        'E' : 0.6,
        'variant': True,
        'de': u':zutritt_verboten:',
        'es': u':dirección_prohibida:',
        'fr': u':sens_interdit:',
        'pt': u':entrada_proibida:',
        'it': u':segnale_di_divieto_di_accesso:'
    },
    u'\U0001F6AF': {
        'en' : u':no_littering:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [u':do_not_litter:'],
        'de': u':abfall_verboten:',
        'es': u':prohibido_tirar_basura:',
        'fr': u':dépôt_d’ordures_interdit:',
        'pt': u':proibido_jogar_lixo_no_chão:',
        'it': u':simbolo_del_divieto_di_gettare_rifiuti:'
    },
    u'\U0001F4F5': {
        'en' : u':no_mobile_phones:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':mobiltelefone_verboten:',
        'es': u':prohibido_el_uso_de_móviles:',
        'fr': u':téléphones_portables_interdits:',
        'pt': u':proibido_o_uso_de_telefone_celular:',
        'it': u':simbolo_che_vieta_l’utilizzo_dei_telefoni_cellulari:'
    },
    u'\U0001F51E': {
        'en' : u':no_one_under_eighteen:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':underage:'],
        'de': u':minderjährige_verboten:',
        'es': u':prohibido_para_menos_de_18_años:',
        'fr': u':18_ans_et_plus:',
        'pt': u':proibido_para_menores_de_18_anos:',
        'it': u':simbolo_di_divieto_ai_minorenni:'
    },
    u'\U0001F6B7': {
        'en' : u':no_pedestrians:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':fußgänger_verboten:',
        'es': u':prohibido_el_paso_de_peatones:',
        'fr': u':interdit_aux_piétons:',
        'pt': u':proibida_a_passagem_de_pedestres:',
        'it': u':simbolo_del_divieto_di_transito_pedoni:'
    },
    u'\U0001F6AD': {
        'en' : u':no_smoking:',
        'status' : fully_qualified,
        'E' : 0.6,
        'variant': True,
        'de': u':rauchverbot:',
        'es': u':prohibido_fumar:',
        'fr': u':interdiction_de_fumer:',
        'pt': u':proibido_fumar:',
        'it': u':segnale_di_vietato_fumare:'
    },
    u'\U0001F6B1': {
        'en' : u':non-potable_water:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [u':non__potable_water:'],
        'de': u':kein_trinkwasser:',
        'es': u':agua_no_potable:',
        'fr': u':eau_non_potable:',
        'pt': u':água_não_potável:',
        'it': u':simbolo_di_acqua_non_potabile:'
    },
    u'\U0001F443': {
        'en' : u':nose:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':nase:',
        'es': u':nariz:',
        'fr': u':nez:',
        'pt': u':nariz:',
        'it': u':naso:'
    },
    u'\U0001F443\U0001F3FF': {
        'en' : u':nose_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':nase_dunkle_hautfarbe:',
        'es': u':nariz_tono_de_piel_oscuro:',
        'fr': u':nez_peau_foncée:',
        'pt': u':nariz_pele_escura:',
        'it': u':naso_carnagione_scura:'
    },
    u'\U0001F443\U0001F3FB': {
        'en' : u':nose_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':nase_helle_hautfarbe:',
        'es': u':nariz_tono_de_piel_claro:',
        'fr': u':nez_peau_claire:',
        'pt': u':nariz_pele_clara:',
        'it': u':naso_carnagione_chiara:'
    },
    u'\U0001F443\U0001F3FE': {
        'en' : u':nose_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':nase_mitteldunkle_hautfarbe:',
        'es': u':nariz_tono_de_piel_oscuro_medio:',
        'fr': u':nez_peau_mate:',
        'pt': u':nariz_pele_morena_escura:',
        'it': u':naso_carnagione_abbastanza_scura:'
    },
    u'\U0001F443\U0001F3FC': {
        'en' : u':nose_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':nase_mittelhelle_hautfarbe:',
        'es': u':nariz_tono_de_piel_claro_medio:',
        'fr': u':nez_peau_moyennement_claire:',
        'pt': u':nariz_pele_morena_clara:',
        'it': u':naso_carnagione_abbastanza_chiara:'
    },
    u'\U0001F443\U0001F3FD': {
        'en' : u':nose_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':nase_mittlere_hautfarbe:',
        'es': u':nariz_tono_de_piel_medio:',
        'fr': u':nez_peau_légèrement_mate:',
        'pt': u':nariz_pele_morena:',
        'it': u':naso_carnagione_olivastra:'
    },
    u'\U0001F4D3': {
        'en' : u':notebook:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':notizbuch:',
        'es': u':cuaderno:',
        'fr': u':carnet:',
        'pt': u':caderno:',
        'it': u':quaderno:'
    },
    u'\U0001F4D4': {
        'en' : u':notebook_with_decorative_cover:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':notizbuch_mit_dekorativem_einband:',
        'es': u':cuaderno_con_tapa_decorativa:',
        'fr': u':carnet_avec_couverture:',
        'pt': u':caderno_decorado:',
        'it': u':quaderno_con_copertina_decorata:'
    },
    u'\U0001F529': {
        'en' : u':nut_and_bolt:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':mutter_und_schraube:',
        'es': u':tornillo_y_tuerca:',
        'fr': u':vis_et_écrou:',
        'pt': u':porca_e_parafuso:',
        'it': u':vite_e_bullone:'
    },
    u'\U0001F419': {
        'en' : u':octopus:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':oktopus:',
        'es': u':pulpo:',
        'fr': u':pieuvre:',
        'pt': u':polvo:',
        'it': u':polpo:'
    },
    u'\U0001F362': {
        'en' : u':oden:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':oden:',
        'es': u':brocheta:',
        'fr': u':brochette_de_poisson:',
        'pt': u':oden:',
        'it': u':oden_giapponese:'
    },
    u'\U0001F3E2': {
        'en' : u':office_building:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':office:'],
        'de': u':bürogebäude:',
        'es': u':edificio_de_oficinas:',
        'fr': u':immeuble_de_bureaux:',
        'pt': u':edifício_comercial:',
        'it': u':edificio:'
    },
    u'\U0001F9D1\U0000200D\U0001F4BC': {
        'en' : u':office_worker:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':büroangestellte(r):',
        'es': u':oficinista:',
        'fr': u':employé_de_bureau_(tous_genres):',
        'pt': u':trabalhador_de_escritório:',
        'it': u':persona_che_fa_un_lavoro_d’ufficio:'
    },
    u'\U0001F9D1\U0001F3FF\U0000200D\U0001F4BC': {
        'en' : u':office_worker_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':büroangestellte(r)_dunkle_hautfarbe:',
        'es': u':oficinista_tono_de_piel_oscuro:',
        'fr': u':employé_de_bureau_(tous_genres)_peau_foncée:',
        'pt': u':trabalhador_de_escritório_pele_escura:',
        'it': u':persona_che_fa_un_lavoro_d’ufficio_carnagione_scura:'
    },
    u'\U0001F9D1\U0001F3FB\U0000200D\U0001F4BC': {
        'en' : u':office_worker_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':büroangestellte(r)_helle_hautfarbe:',
        'es': u':oficinista_tono_de_piel_claro:',
        'fr': u':employé_de_bureau_(tous_genres)_peau_claire:',
        'pt': u':trabalhador_de_escritório_pele_clara:',
        'it': u':persona_che_fa_un_lavoro_d’ufficio_carnagione_chiara:'
    },
    u'\U0001F9D1\U0001F3FE\U0000200D\U0001F4BC': {
        'en' : u':office_worker_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':büroangestellte(r)_mitteldunkle_hautfarbe:',
        'es': u':oficinista_tono_de_piel_oscuro_medio:',
        'fr': u':employé_de_bureau_(tous_genres)_peau_mate:',
        'pt': u':trabalhador_de_escritório_pele_morena_escura:',
        'it': u':persona_che_fa_un_lavoro_d’ufficio_carnagione_abbastanza_scura:'
    },
    u'\U0001F9D1\U0001F3FC\U0000200D\U0001F4BC': {
        'en' : u':office_worker_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':büroangestellte(r)_mittelhelle_hautfarbe:',
        'es': u':oficinista_tono_de_piel_claro_medio:',
        'fr': u':employé_de_bureau_(tous_genres)_peau_moyennement_claire:',
        'pt': u':trabalhador_de_escritório_pele_morena_clara:',
        'it': u':persona_che_fa_un_lavoro_d’ufficio_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9D1\U0001F3FD\U0000200D\U0001F4BC': {
        'en' : u':office_worker_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':büroangestellte(r)_mittlere_hautfarbe:',
        'es': u':oficinista_tono_de_piel_medio:',
        'fr': u':employé_de_bureau_(tous_genres)_peau_légèrement_mate:',
        'pt': u':trabalhador_de_escritório_pele_morena:',
        'it': u':persona_che_fa_un_lavoro_d’ufficio_carnagione_olivastra:'
    },
    u'\U0001F479': {
        'en' : u':ogre:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':japanese_ogre:'],
        'de': u':ungeheuer:',
        'es': u':demonio_japonés_oni:',
        'fr': u':ogre:',
        'pt': u':ogro:',
        'it': u':orco:'
    },
    u'\U0001F6E2\U0000FE0F': {
        'en' : u':oil_drum:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': u':ölfass:',
        'es': u':barril_de_petróleo:',
        'fr': u':baril_de_pétrole:',
        'pt': u':barril_de_óleo:',
        'it': u':barile_di_petrolio:'
    },
    u'\U0001F6E2': {
        'en' : u':oil_drum:',
        'status' : unqualified,
        'E' : 0.7,
        'variant': True,
        'de': u':ölfass:',
        'es': u':barril_de_petróleo:',
        'fr': u':baril_de_pétrole:',
        'pt': u':barril_de_óleo:',
        'it': u':barile_di_petrolio:'
    },
    u'\U0001F5DD\U0000FE0F': {
        'en' : u':old_key:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': u':alter_schlüssel:',
        'es': u':llave_antigua:',
        'fr': u':clé_ancienne:',
        'pt': u':chave_antiga:',
        'it': u':chiave_antica:'
    },
    u'\U0001F5DD': {
        'en' : u':old_key:',
        'status' : unqualified,
        'E' : 0.7,
        'variant': True,
        'de': u':alter_schlüssel:',
        'es': u':llave_antigua:',
        'fr': u':clé_ancienne:',
        'pt': u':chave_antiga:',
        'it': u':chiave_antica:'
    },
    u'\U0001F474': {
        'en' : u':old_man:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':older_man:'],
        'de': u':älterer_mann:',
        'es': u':anciano:',
        'fr': u':homme_âgé:',
        'pt': u':homem_idoso:',
        'it': u':uomo_anziano:'
    },
    u'\U0001F474\U0001F3FF': {
        'en' : u':old_man_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':älterer_mann_dunkle_hautfarbe:',
        'es': u':anciano_tono_de_piel_oscuro:',
        'fr': u':homme_âgé_peau_foncée:',
        'pt': u':homem_idoso_pele_escura:',
        'it': u':uomo_anziano_carnagione_scura:'
    },
    u'\U0001F474\U0001F3FB': {
        'en' : u':old_man_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':älterer_mann_helle_hautfarbe:',
        'es': u':anciano_tono_de_piel_claro:',
        'fr': u':homme_âgé_peau_claire:',
        'pt': u':homem_idoso_pele_clara:',
        'it': u':uomo_anziano_carnagione_chiara:'
    },
    u'\U0001F474\U0001F3FE': {
        'en' : u':old_man_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':älterer_mann_mitteldunkle_hautfarbe:',
        'es': u':anciano_tono_de_piel_oscuro_medio:',
        'fr': u':homme_âgé_peau_mate:',
        'pt': u':homem_idoso_pele_morena_escura:',
        'it': u':uomo_anziano_carnagione_abbastanza_scura:'
    },
    u'\U0001F474\U0001F3FC': {
        'en' : u':old_man_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':älterer_mann_mittelhelle_hautfarbe:',
        'es': u':anciano_tono_de_piel_claro_medio:',
        'fr': u':homme_âgé_peau_moyennement_claire:',
        'pt': u':homem_idoso_pele_morena_clara:',
        'it': u':uomo_anziano_carnagione_abbastanza_chiara:'
    },
    u'\U0001F474\U0001F3FD': {
        'en' : u':old_man_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':älterer_mann_mittlere_hautfarbe:',
        'es': u':anciano_tono_de_piel_medio:',
        'fr': u':homme_âgé_peau_légèrement_mate:',
        'pt': u':homem_idoso_pele_morena:',
        'it': u':uomo_anziano_carnagione_olivastra:'
    },
    u'\U0001F475': {
        'en' : u':old_woman:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':older_woman:'],
        'de': u':ältere_frau:',
        'es': u':anciana:',
        'fr': u':femme_âgée:',
        'pt': u':idosa:',
        'it': u':donna_anziana:'
    },
    u'\U0001F475\U0001F3FF': {
        'en' : u':old_woman_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':ältere_frau_dunkle_hautfarbe:',
        'es': u':anciana_tono_de_piel_oscuro:',
        'fr': u':femme_âgée_peau_foncée:',
        'pt': u':idosa_pele_escura:',
        'it': u':donna_anziana_carnagione_scura:'
    },
    u'\U0001F475\U0001F3FB': {
        'en' : u':old_woman_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':ältere_frau_helle_hautfarbe:',
        'es': u':anciana_tono_de_piel_claro:',
        'fr': u':femme_âgée_peau_claire:',
        'pt': u':idosa_pele_clara:',
        'it': u':donna_anziana_carnagione_chiara:'
    },
    u'\U0001F475\U0001F3FE': {
        'en' : u':old_woman_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':ältere_frau_mitteldunkle_hautfarbe:',
        'es': u':anciana_tono_de_piel_oscuro_medio:',
        'fr': u':femme_âgée_peau_mate:',
        'pt': u':idosa_pele_morena_escura:',
        'it': u':donna_anziana_carnagione_abbastanza_scura:'
    },
    u'\U0001F475\U0001F3FC': {
        'en' : u':old_woman_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':ältere_frau_mittelhelle_hautfarbe:',
        'es': u':anciana_tono_de_piel_claro_medio:',
        'fr': u':femme_âgée_peau_moyennement_claire:',
        'pt': u':idosa_pele_morena_clara:',
        'it': u':donna_anziana_carnagione_abbastanza_chiara:'
    },
    u'\U0001F475\U0001F3FD': {
        'en' : u':old_woman_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':ältere_frau_mittlere_hautfarbe:',
        'es': u':anciana_tono_de_piel_medio:',
        'fr': u':femme_âgée_peau_légèrement_mate:',
        'pt': u':idosa_pele_morena:',
        'it': u':donna_anziana_carnagione_olivastra:'
    },
    u'\U0001F9D3': {
        'en' : u':older_person:',
        'status' : fully_qualified,
        'E' : 5,
        'alias' : [u':older_adult:'],
        'de': u':älterer_erwachsener:',
        'es': u':persona_adulta_madura:',
        'fr': u':personne_âgée:',
        'pt': u':idoso:',
        'it': u':persona_anziana:'
    },
    u'\U0001F9D3\U0001F3FF': {
        'en' : u':older_person_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':älterer_erwachsener_dunkle_hautfarbe:',
        'es': u':persona_adulta_madura_tono_de_piel_oscuro:',
        'fr': u':personne_âgée_peau_foncée:',
        'pt': u':idoso_pele_escura:',
        'it': u':persona_anziana_carnagione_scura:'
    },
    u'\U0001F9D3\U0001F3FB': {
        'en' : u':older_person_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':älterer_erwachsener_helle_hautfarbe:',
        'es': u':persona_adulta_madura_tono_de_piel_claro:',
        'fr': u':personne_âgée_peau_claire:',
        'pt': u':idoso_pele_clara:',
        'it': u':persona_anziana_carnagione_chiara:'
    },
    u'\U0001F9D3\U0001F3FE': {
        'en' : u':older_person_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':älterer_erwachsener_mitteldunkle_hautfarbe:',
        'es': u':persona_adulta_madura_tono_de_piel_oscuro_medio:',
        'fr': u':personne_âgée_peau_mate:',
        'pt': u':idoso_pele_morena_escura:',
        'it': u':persona_anziana_carnagione_abbastanza_scura:'
    },
    u'\U0001F9D3\U0001F3FC': {
        'en' : u':older_person_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':älterer_erwachsener_mittelhelle_hautfarbe:',
        'es': u':persona_adulta_madura_tono_de_piel_claro_medio:',
        'fr': u':personne_âgée_peau_moyennement_claire:',
        'pt': u':idoso_pele_morena_clara:',
        'it': u':persona_anziana_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9D3\U0001F3FD': {
        'en' : u':older_person_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':älterer_erwachsener_mittlere_hautfarbe:',
        'es': u':persona_adulta_madura_tono_de_piel_medio:',
        'fr': u':personne_âgée_peau_légèrement_mate:',
        'pt': u':idoso_pele_morena:',
        'it': u':persona_anziana_carnagione_olivastra:'
    },
    u'\U0001FAD2': {
        'en' : u':olive:',
        'status' : fully_qualified,
        'E' : 13,
        'de': u':olive:',
        'es': u':aceituna:',
        'fr': u':olive:',
        'pt': u':azeitona:',
        'it': u':oliva:'
    },
    u'\U0001F549\U0000FE0F': {
        'en' : u':om:',
        'status' : fully_qualified,
        'E' : 0.7,
        'alias' : [u':om_symbol:'],
        'variant': True,
        'de': u':om:',
        'es': u':om:',
        'fr': u':om:',
        'pt': u':om:',
        'it': u':om:'
    },
    u'\U0001F549': {
        'en' : u':om:',
        'status' : unqualified,
        'E' : 0.7,
        'alias' : [u':om_symbol:'],
        'variant': True,
        'de': u':om:',
        'es': u':om:',
        'fr': u':om:',
        'pt': u':om:',
        'it': u':om:'
    },
    u'\U0001F698': {
        'en' : u':oncoming_automobile:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': u':vorderansicht_auto:',
        'es': u':coche_próximo:',
        'fr': u':voiture_de_face:',
        'pt': u':carro_se_aproximando:',
        'it': u':auto_in_arrivo:'
    },
    u'\U0001F68D': {
        'en' : u':oncoming_bus:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': u':vorderansicht_bus:',
        'es': u':autobús_próximo:',
        'fr': u':bus_de_face:',
        'pt': u':ônibus_se_aproximando:',
        'it': u':bus_in_arrivo:'
    },
    u'\U0001F44A': {
        'en' : u':oncoming_fist:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':punch:', u':facepunch:', u':fist_oncoming:'],
        'de': u':geballte_faust:',
        'es': u':puño_cerrado:',
        'fr': u':poing_de_face:',
        'pt': u':soco:',
        'it': u':pugno_chiuso:'
    },
    u'\U0001F44A\U0001F3FF': {
        'en' : u':oncoming_fist_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':geballte_faust_dunkle_hautfarbe:',
        'es': u':puño_cerrado_tono_de_piel_oscuro:',
        'fr': u':poing_de_face_peau_foncée:',
        'pt': u':soco_pele_escura:',
        'it': u':pugno_chiuso_carnagione_scura:'
    },
    u'\U0001F44A\U0001F3FB': {
        'en' : u':oncoming_fist_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':geballte_faust_helle_hautfarbe:',
        'es': u':puño_cerrado_tono_de_piel_claro:',
        'fr': u':poing_de_face_peau_claire:',
        'pt': u':soco_pele_clara:',
        'it': u':pugno_chiuso_carnagione_chiara:'
    },
    u'\U0001F44A\U0001F3FE': {
        'en' : u':oncoming_fist_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':geballte_faust_mitteldunkle_hautfarbe:',
        'es': u':puño_cerrado_tono_de_piel_oscuro_medio:',
        'fr': u':poing_de_face_peau_mate:',
        'pt': u':soco_pele_morena_escura:',
        'it': u':pugno_chiuso_carnagione_abbastanza_scura:'
    },
    u'\U0001F44A\U0001F3FC': {
        'en' : u':oncoming_fist_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':geballte_faust_mittelhelle_hautfarbe:',
        'es': u':puño_cerrado_tono_de_piel_claro_medio:',
        'fr': u':poing_de_face_peau_moyennement_claire:',
        'pt': u':soco_pele_morena_clara:',
        'it': u':pugno_chiuso_carnagione_abbastanza_chiara:'
    },
    u'\U0001F44A\U0001F3FD': {
        'en' : u':oncoming_fist_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':geballte_faust_mittlere_hautfarbe:',
        'es': u':puño_cerrado_tono_de_piel_medio:',
        'fr': u':poing_de_face_peau_légèrement_mate:',
        'pt': u':soco_pele_morena:',
        'it': u':pugno_chiuso_carnagione_olivastra:'
    },
    u'\U0001F694': {
        'en' : u':oncoming_police_car:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': u':vorderansicht_polizeiwagen:',
        'es': u':coche_de_policía_próximo:',
        'fr': u':voiture_de_police_de_face:',
        'pt': u':viatura_policial_se_aproximando:',
        'it': u':macchina_della_polizia_in_arrivo:'
    },
    u'\U0001F696': {
        'en' : u':oncoming_taxi:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':vorderansicht_taxi:',
        'es': u':taxi_próximo:',
        'fr': u':taxi_de_face:',
        'pt': u':táxi_se_aproximando:',
        'it': u':taxi_in_arrivo:'
    },
    u'\U0001FA71': {
        'en' : u':one-piece_swimsuit:',
        'status' : fully_qualified,
        'E' : 12,
        'alias' : [u':one_piece_swimsuit:'],
        'de': u':einteiliger_badeanzug:',
        'es': u':traje_de_baño_de_una_pieza:',
        'fr': u':maillot_de_bain_une_pièce:',
        'pt': u':maiô:',
        'it': u':costume_da_bagno_intero:'
    },
    u'\U0001F55C': {
        'en' : u':one-thirty:',
        'status' : fully_qualified,
        'E' : 0.7,
        'alias' : [u':clock130:'],
        'variant': True,
        'de': u':1.30_uhr:',
        'es': u':una_y_media:',
        'fr': u':une_heure_et_demie:',
        'pt': u':uma_e_meia:',
        'it': u':ore_una_e_mezza:'
    },
    u'\U0001F550': {
        'en' : u':one_o’clock:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':clock1:'],
        'variant': True,
        'de': u':1.00_uhr:',
        'es': u':1_en_punto:',
        'fr': u':une_heure:',
        'pt': u':1_hora:',
        'it': u':ore_una:'
    },
    u'\U0001F9C5': {
        'en' : u':onion:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':zwiebel:',
        'es': u':cebolla:',
        'fr': u':oignon:',
        'pt': u':cebola:',
        'it': u':cipolla:'
    },
    u'\U0001F4D6': {
        'en' : u':open_book:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':book:'],
        'de': u':offenes_buch:',
        'es': u':libro_abierto:',
        'fr': u':livre_ouvert:',
        'pt': u':livro_aberto:',
        'it': u':libro_aperto:'
    },
    u'\U0001F4C2': {
        'en' : u':open_file_folder:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':geöffneter_ordner:',
        'es': u':carpeta_de_archivos_abierta:',
        'fr': u':dossier_ouvert:',
        'pt': u':pasta_de_arquivos_aberta:',
        'it': u':cartella_aperta:'
    },
    u'\U0001F450': {
        'en' : u':open_hands:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':offene_hände:',
        'es': u':manos_abiertas:',
        'fr': u':mains_ouvertes:',
        'pt': u':mãos_abertas:',
        'it': u':mani_aperte:'
    },
    u'\U0001F450\U0001F3FF': {
        'en' : u':open_hands_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':offene_hände_dunkle_hautfarbe:',
        'es': u':manos_abiertas_tono_de_piel_oscuro:',
        'fr': u':mains_ouvertes_peau_foncée:',
        'pt': u':mãos_abertas_pele_escura:',
        'it': u':mani_aperte_carnagione_scura:'
    },
    u'\U0001F450\U0001F3FB': {
        'en' : u':open_hands_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':offene_hände_helle_hautfarbe:',
        'es': u':manos_abiertas_tono_de_piel_claro:',
        'fr': u':mains_ouvertes_peau_claire:',
        'pt': u':mãos_abertas_pele_clara:',
        'it': u':mani_aperte_carnagione_chiara:'
    },
    u'\U0001F450\U0001F3FE': {
        'en' : u':open_hands_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':offene_hände_mitteldunkle_hautfarbe:',
        'es': u':manos_abiertas_tono_de_piel_oscuro_medio:',
        'fr': u':mains_ouvertes_peau_mate:',
        'pt': u':mãos_abertas_pele_morena_escura:',
        'it': u':mani_aperte_carnagione_abbastanza_scura:'
    },
    u'\U0001F450\U0001F3FC': {
        'en' : u':open_hands_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':offene_hände_mittelhelle_hautfarbe:',
        'es': u':manos_abiertas_tono_de_piel_claro_medio:',
        'fr': u':mains_ouvertes_peau_moyennement_claire:',
        'pt': u':mãos_abertas_pele_morena_clara:',
        'it': u':mani_aperte_carnagione_abbastanza_chiara:'
    },
    u'\U0001F450\U0001F3FD': {
        'en' : u':open_hands_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':offene_hände_mittlere_hautfarbe:',
        'es': u':manos_abiertas_tono_de_piel_medio:',
        'fr': u':mains_ouvertes_peau_légèrement_mate:',
        'pt': u':mãos_abertas_pele_morena:',
        'it': u':mani_aperte_carnagione_olivastra:'
    },
    u'\U0001F4ED': {
        'en' : u':open_mailbox_with_lowered_flag:',
        'status' : fully_qualified,
        'E' : 0.7,
        'alias' : [u':mailbox_with_no_mail:'],
        'variant': True,
        'de': u':offener_briefkasten_ohne_post:',
        'es': u':buzón_abierto_con_la_bandera_bajada:',
        'fr': u':boîte_aux_lettres_ouverte_drapeau_baissé:',
        'pt': u':caixa_de_correio_aberta_com_bandeira_abaixada:',
        'it': u':cassetta_postale_con_bandierina_abbassata:'
    },
    u'\U0001F4EC': {
        'en' : u':open_mailbox_with_raised_flag:',
        'status' : fully_qualified,
        'E' : 0.7,
        'alias' : [u':mailbox_with_mail:'],
        'variant': True,
        'de': u':offener_briefkasten_mit_post:',
        'es': u':buzón_abierto_con_la_bandera_levantada:',
        'fr': u':boîte_aux_lettres_ouverte_drapeau_levé:',
        'pt': u':caixa_de_correio_aberta_com_bandeira_levantada:',
        'it': u':cassetta_postale_aperta_bandierina_alzata:'
    },
    u'\U0001F4BF': {
        'en' : u':optical_disk:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':cd:'],
        'variant': True,
        'de': u':cd:',
        'es': u':disco_óptico:',
        'fr': u':cd:',
        'pt': u':cd:',
        'it': u':disco_ottico:'
    },
    u'\U0001F4D9': {
        'en' : u':orange_book:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':orangefarbenes_buch:',
        'es': u':libro_naranja:',
        'fr': u':livre_orange:',
        'pt': u':livro_laranja:',
        'it': u':libro_arancione:'
    },
    u'\U0001F7E0': {
        'en' : u':orange_circle:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':oranger_kreis:',
        'es': u':círculo_naranja:',
        'fr': u':disque_orange:',
        'pt': u':círculo_laranja:',
        'it': u':cerchio_arancione:'
    },
    u'\U0001F9E1': {
        'en' : u':orange_heart:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':oranges_herz:',
        'es': u':corazón_naranja:',
        'fr': u':cœur_orange:',
        'pt': u':coração_laranja:',
        'it': u':cuore_arancione:'
    },
    u'\U0001F7E7': {
        'en' : u':orange_square:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':oranges_viereck:',
        'es': u':cuadrado_naranja:',
        'fr': u':carré_orange:',
        'pt': u':quadrado_laranja:',
        'it': u':quadrato_arancione:'
    },
    u'\U0001F9A7': {
        'en' : u':orangutan:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':orang-utan:',
        'es': u':orangután:',
        'fr': u':orang-outan:',
        'pt': u':orangotango:',
        'it': u':orangotango:'
    },
    u'\U00002626\U0000FE0F': {
        'en' : u':orthodox_cross:',
        'status' : fully_qualified,
        'E' : 1,
        'variant': True,
        'de': u':orthodoxes_kreuz:',
        'es': u':cruz_ortodoxa:',
        'fr': u':croix_orthodoxe:',
        'pt': u':cruz_ortodoxa:',
        'it': u':croce_ortodossa:'
    },
    u'\U00002626': {
        'en' : u':orthodox_cross:',
        'status' : unqualified,
        'E' : 1,
        'variant': True,
        'de': u':orthodoxes_kreuz:',
        'es': u':cruz_ortodoxa:',
        'fr': u':croix_orthodoxe:',
        'pt': u':cruz_ortodoxa:',
        'it': u':croce_ortodossa:'
    },
    u'\U0001F9A6': {
        'en' : u':otter:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':otter:',
        'es': u':nutria:',
        'fr': u':loutre:',
        'pt': u':lontra:',
        'it': u':lontra:'
    },
    u'\U0001F4E4': {
        'en' : u':outbox_tray:',
        'status' : fully_qualified,
        'E' : 0.6,
        'variant': True,
        'de': u':postausgang:',
        'es': u':bandeja_de_salida:',
        'fr': u':boîte_d’envoi:',
        'pt': u':bandeja_de_saída:',
        'it': u':posta_inviata:'
    },
    u'\U0001F989': {
        'en' : u':owl:',
        'status' : fully_qualified,
        'E' : 3,
        'de': u':eule:',
        'es': u':búho:',
        'fr': u':chouette:',
        'pt': u':coruja:',
        'it': u':gufo:'
    },
    u'\U0001F402': {
        'en' : u':ox:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':ochse:',
        'es': u':buey:',
        'fr': u':bœuf:',
        'pt': u':boi:',
        'it': u':bue:'
    },
    u'\U0001F9AA': {
        'en' : u':oyster:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':auster:',
        'es': u':ostra:',
        'fr': u':huître:',
        'pt': u':ostra:',
        'it': u':ostrica:'
    },
    u'\U0001F4E6': {
        'en' : u':package:',
        'status' : fully_qualified,
        'E' : 0.6,
        'variant': True,
        'de': u':paket:',
        'es': u':paquete:',
        'fr': u':colis:',
        'pt': u':pacote:',
        'it': u':pacco:'
    },
    u'\U0001F4C4': {
        'en' : u':page_facing_up:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':vorderseite_eines_blattes:',
        'es': u':página_hacia_arriba:',
        'fr': u':page:',
        'pt': u':página_voltada_para_cima:',
        'it': u':pagina:'
    },
    u'\U0001F4C3': {
        'en' : u':page_with_curl:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':teilweise_eingerolltes_blatt:',
        'es': u':página_doblada:',
        'fr': u':page_enroulée:',
        'pt': u':página_dobrada:',
        'it': u':pagina_con_piega:'
    },
    u'\U0001F4DF': {
        'en' : u':pager:',
        'status' : fully_qualified,
        'E' : 0.6,
        'variant': True,
        'de': u':pager:',
        'es': u':busca:',
        'fr': u':bipeur:',
        'pt': u':pager:',
        'it': u':cercapersone:'
    },
    u'\U0001F58C\U0000FE0F': {
        'en' : u':paintbrush:',
        'status' : fully_qualified,
        'E' : 0.7,
        'alias' : [u':lower_left_paintbrush:'],
        'variant': True,
        'de': u':pinsel:',
        'es': u':pincel:',
        'fr': u':pinceau:',
        'pt': u':pincel:',
        'it': u':pennello:'
    },
    u'\U0001F58C': {
        'en' : u':paintbrush:',
        'status' : unqualified,
        'E' : 0.7,
        'alias' : [u':lower_left_paintbrush:'],
        'variant': True,
        'de': u':pinsel:',
        'es': u':pincel:',
        'fr': u':pinceau:',
        'pt': u':pincel:',
        'it': u':pennello:'
    },
    u'\U0001FAF3': {
        'en' : u':palm_down_hand:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001FAF3\U0001F3FF': {
        'en' : u':palm_down_hand_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001FAF3\U0001F3FB': {
        'en' : u':palm_down_hand_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001FAF3\U0001F3FE': {
        'en' : u':palm_down_hand_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001FAF3\U0001F3FC': {
        'en' : u':palm_down_hand_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001FAF3\U0001F3FD': {
        'en' : u':palm_down_hand_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001F334': {
        'en' : u':palm_tree:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':palme:',
        'es': u':palmera:',
        'fr': u':palmier:',
        'pt': u':palmeira:',
        'it': u':palma:'
    },
    u'\U0001FAF4': {
        'en' : u':palm_up_hand:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001FAF4\U0001F3FF': {
        'en' : u':palm_up_hand_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001FAF4\U0001F3FB': {
        'en' : u':palm_up_hand_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001FAF4\U0001F3FE': {
        'en' : u':palm_up_hand_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001FAF4\U0001F3FC': {
        'en' : u':palm_up_hand_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001FAF4\U0001F3FD': {
        'en' : u':palm_up_hand_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001F932': {
        'en' : u':palms_up_together:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':handflächen_nach_oben:',
        'es': u':palmas_hacia_arriba_juntas:',
        'fr': u':paume_contre_paume_doigts_vers_le_haut:',
        'pt': u':palmas_unidas_para_cima:',
        'it': u':mani_unite_in_alto:'
    },
    u'\U0001F932\U0001F3FF': {
        'en' : u':palms_up_together_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':handflächen_nach_oben_dunkle_hautfarbe:',
        'es': u':palmas_hacia_arriba_juntas_tono_de_piel_oscuro:',
        'fr': u':paume_contre_paume_doigts_vers_le_haut_peau_foncée:',
        'pt': u':palmas_unidas_para_cima_pele_escura:',
        'it': u':mani_unite_in_alto_carnagione_scura:'
    },
    u'\U0001F932\U0001F3FB': {
        'en' : u':palms_up_together_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':handflächen_nach_oben_helle_hautfarbe:',
        'es': u':palmas_hacia_arriba_juntas_tono_de_piel_claro:',
        'fr': u':paume_contre_paume_doigts_vers_le_haut_peau_claire:',
        'pt': u':palmas_unidas_para_cima_pele_clara:',
        'it': u':mani_unite_in_alto_carnagione_chiara:'
    },
    u'\U0001F932\U0001F3FE': {
        'en' : u':palms_up_together_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':handflächen_nach_oben_mitteldunkle_hautfarbe:',
        'es': u':palmas_hacia_arriba_juntas_tono_de_piel_oscuro_medio:',
        'fr': u':paume_contre_paume_doigts_vers_le_haut_peau_mate:',
        'pt': u':palmas_unidas_para_cima_pele_morena_escura:',
        'it': u':mani_unite_in_alto_carnagione_abbastanza_scura:'
    },
    u'\U0001F932\U0001F3FC': {
        'en' : u':palms_up_together_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':handflächen_nach_oben_mittelhelle_hautfarbe:',
        'es': u':palmas_hacia_arriba_juntas_tono_de_piel_claro_medio:',
        'fr': u':paume_contre_paume_doigts_vers_le_haut_peau_moyennement_claire:',
        'pt': u':palmas_unidas_para_cima_pele_morena_clara:',
        'it': u':mani_unite_in_alto_carnagione_abbastanza_chiara:'
    },
    u'\U0001F932\U0001F3FD': {
        'en' : u':palms_up_together_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':handflächen_nach_oben_mittlere_hautfarbe:',
        'es': u':palmas_hacia_arriba_juntas_tono_de_piel_medio:',
        'fr': u':paume_contre_paume_doigts_vers_le_haut_peau_légèrement_mate:',
        'pt': u':palmas_unidas_para_cima_pele_morena:',
        'it': u':mani_unite_in_alto_carnagione_olivastra:'
    },
    u'\U0001F95E': {
        'en' : u':pancakes:',
        'status' : fully_qualified,
        'E' : 3,
        'de': u':pfannkuchen:',
        'es': u':tortitas:',
        'fr': u':pancakes:',
        'pt': u':panquecas:',
        'it': u':pancake:'
    },
    u'\U0001F43C': {
        'en' : u':panda:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':panda_face:'],
        'de': u':panda:',
        'es': u':panda:',
        'fr': u':panda:',
        'pt': u':rosto_de_panda:',
        'it': u':panda:'
    },
    u'\U0001F4CE': {
        'en' : u':paperclip:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':büroklammer:',
        'es': u':clip:',
        'fr': u':trombone:',
        'pt': u':clipe_de_papel:',
        'it': u':graffetta:'
    },
    u'\U0001FA82': {
        'en' : u':parachute:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':fallschirm:',
        'es': u':paracaídas:',
        'fr': u':parachute:',
        'pt': u':paraquedas:',
        'it': u':paracadute:'
    },
    u'\U0001F99C': {
        'en' : u':parrot:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':papagei:',
        'es': u':loro:',
        'fr': u':perroquet:',
        'pt': u':papagaio:',
        'it': u':pappagallo:'
    },
    u'\U0000303D\U0000FE0F': {
        'en' : u':part_alternation_mark:',
        'status' : fully_qualified,
        'E' : 0.6,
        'variant': True,
        'de': u':teilalternationszeichen:',
        'es': u':marca_de_alternancia:',
        'fr': u':alternance:',
        'pt': u':sinal_japonês_indicando_início_de_música:',
        'it': u':simbolo_di_alternanza_delle_parti:'
    },
    u'\U0000303D': {
        'en' : u':part_alternation_mark:',
        'status' : unqualified,
        'E' : 0.6,
        'variant': True,
        'de': u':teilalternationszeichen:',
        'es': u':marca_de_alternancia:',
        'fr': u':alternance:',
        'pt': u':sinal_japonês_indicando_início_de_música:',
        'it': u':simbolo_di_alternanza_delle_parti:'
    },
    u'\U0001F389': {
        'en' : u':party_popper:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':tada:'],
        'de': u':konfettibombe:',
        'es': u':cañón_de_confeti:',
        'fr': u':cotillons:',
        'pt': u':cone_de_festa:',
        'it': u':spara_coriandoli:'
    },
    u'\U0001F973': {
        'en' : u':partying_face:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':partygesicht:',
        'es': u':cara_de_fiesta:',
        'fr': u':visage_festif:',
        'pt': u':rosto_festivo:',
        'it': u':faccina_che_festeggia:'
    },
    u'\U0001F6F3\U0000FE0F': {
        'en' : u':passenger_ship:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': u':passagierschiff:',
        'es': u':barco_de_pasajeros:',
        'fr': u':paquebot:',
        'pt': u':cruzeiro:',
        'it': u':nave_da_crociera:'
    },
    u'\U0001F6F3': {
        'en' : u':passenger_ship:',
        'status' : unqualified,
        'E' : 0.7,
        'variant': True,
        'de': u':passagierschiff:',
        'es': u':barco_de_pasajeros:',
        'fr': u':paquebot:',
        'pt': u':cruzeiro:',
        'it': u':nave_da_crociera:'
    },
    u'\U0001F6C2': {
        'en' : u':passport_control:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':passkontrolle:',
        'es': u':control_de_pasaportes:',
        'fr': u':contrôle_des_passeports:',
        'pt': u':controle_de_passaportes:',
        'it': u':simbolo_del_controllo_passaporti:'
    },
    u'\U000023F8\U0000FE0F': {
        'en' : u':pause_button:',
        'status' : fully_qualified,
        'E' : 0.7,
        'alias' : [u':double_vertical_bar:'],
        'variant': True,
        'de': u':pause:',
        'es': u':pausa:',
        'fr': u':bouton_pause:',
        'pt': u':botão_pausar:',
        'it': u':pulsante_pausa:'
    },
    u'\U000023F8': {
        'en' : u':pause_button:',
        'status' : unqualified,
        'E' : 0.7,
        'alias' : [u':double_vertical_bar:'],
        'variant': True,
        'de': u':pause:',
        'es': u':pausa:',
        'fr': u':bouton_pause:',
        'pt': u':botão_pausar:',
        'it': u':pulsante_pausa:'
    },
    u'\U0001F43E': {
        'en' : u':paw_prints:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':feet:'],
        'de': u':tatzenabdrücke:',
        'es': u':huellas_de_pezuñas:',
        'fr': u':empreintes_d’animaux:',
        'pt': u':patas:',
        'it': u':impronte_di_zampe:'
    },
    u'\U0000262E\U0000FE0F': {
        'en' : u':peace_symbol:',
        'status' : fully_qualified,
        'E' : 1,
        'variant': True,
        'de': u':friedenszeichen:',
        'es': u':símbolo_de_la_paz:',
        'fr': u':symbole_de_paix:',
        'pt': u':símbolo_da_paz:',
        'it': u':simbolo_della_pace:'
    },
    u'\U0000262E': {
        'en' : u':peace_symbol:',
        'status' : unqualified,
        'E' : 1,
        'variant': True,
        'de': u':friedenszeichen:',
        'es': u':símbolo_de_la_paz:',
        'fr': u':symbole_de_paix:',
        'pt': u':símbolo_da_paz:',
        'it': u':simbolo_della_pace:'
    },
    u'\U0001F351': {
        'en' : u':peach:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':pfirsich:',
        'es': u':melocotón:',
        'fr': u':pêche:',
        'pt': u':pêssego:',
        'it': u':pesca:'
    },
    u'\U0001F99A': {
        'en' : u':peacock:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':pfau:',
        'es': u':pavo_real:',
        'fr': u':paon:',
        'pt': u':pavão:',
        'it': u':pavone:'
    },
    u'\U0001F95C': {
        'en' : u':peanuts:',
        'status' : fully_qualified,
        'E' : 3,
        'de': u':erdnuss:',
        'es': u':cacahuetes:',
        'fr': u':cacahuètes:',
        'pt': u':amendoim:',
        'it': u':arachidi:'
    },
    u'\U0001F350': {
        'en' : u':pear:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':birne:',
        'es': u':pera:',
        'fr': u':poire:',
        'pt': u':pera:',
        'it': u':pera:'
    },
    u'\U0001F58A\U0000FE0F': {
        'en' : u':pen:',
        'status' : fully_qualified,
        'E' : 0.7,
        'alias' : [u':lower_left_ballpoint_pen:'],
        'variant': True,
        'de': u':kugelschreiber:',
        'es': u':bolígrafo:',
        'fr': u':stylo:',
        'pt': u':caneta:',
        'it': u':penna_a_sfera:'
    },
    u'\U0001F58A': {
        'en' : u':pen:',
        'status' : unqualified,
        'E' : 0.7,
        'alias' : [u':lower_left_ballpoint_pen:'],
        'variant': True,
        'de': u':kugelschreiber:',
        'es': u':bolígrafo:',
        'fr': u':stylo:',
        'pt': u':caneta:',
        'it': u':penna_a_sfera:'
    },
    u'\U0000270F\U0000FE0F': {
        'en' : u':pencil:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':pencil2:'],
        'variant': True,
        'de': u':bleistift:',
        'es': u':lápiz:',
        'fr': u':crayon:',
        'pt': u':lápis:',
        'it': u':matita:'
    },
    u'\U0000270F': {
        'en' : u':pencil:',
        'status' : unqualified,
        'E' : 0.6,
        'alias' : [u':pencil2:'],
        'variant': True,
        'de': u':bleistift:',
        'es': u':lápiz:',
        'fr': u':crayon:',
        'pt': u':lápis:',
        'it': u':matita:'
    },
    u'\U0001F427': {
        'en' : u':penguin:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':pinguin:',
        'es': u':pingüino:',
        'fr': u':pingouin:',
        'pt': u':pinguim:',
        'it': u':pinguino:'
    },
    u'\U0001F614': {
        'en' : u':pensive_face:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':pensive:'],
        'de': u':nachdenkliches_gesicht:',
        'es': u':cara_desanimada:',
        'fr': u':visage_pensif:',
        'pt': u':rosto_deprimido:',
        'it': u':faccina_pensierosa:'
    },
    u'\U0001F9D1\U0000200D\U0001F91D\U0000200D\U0001F9D1': {
        'en' : u':people_holding_hands:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':sich_an_den_händen_haltende_personen:',
        'es': u':dos_personas_dándose_la_mano:',
        'fr': u':deux_personnes_se_tenant_la_main:',
        'pt': u':pessoas_de_mãos_dadas:',
        'it': u':persone_che_si_tengono_per_mano:'
    },
    u'\U0001F9D1\U0001F3FF\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FF': {
        'en' : u':people_holding_hands_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':sich_an_den_händen_haltende_personen_dunkle_hautfarbe:',
        'es': u':dos_personas_dándose_la_mano_tono_de_piel_oscuro:',
        'fr': u':deux_personnes_se_tenant_la_main_peau_foncée:',
        'pt': u':pessoas_de_mãos_dadas_pele_escura:',
        'it': u':persone_che_si_tengono_per_mano_carnagione_scura:'
    },
    u'\U0001F9D1\U0001F3FF\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FB': {
        'en' : u':people_holding_hands_dark_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':sich_an_den_händen_haltende_personen_dunkle_hautfarbe,_helle_hautfarbe:',
        'es': u':dos_personas_dándose_la_mano_tono_de_piel_oscuro_y_tono_de_piel_claro:',
        'fr': u':deux_personnes_se_tenant_la_main_peau_foncée_et_peau_claire:',
        'pt': u':pessoas_de_mãos_dadas_pele_escura_e_pele_clara:',
        'it': u':persone_che_si_tengono_per_mano_carnagione_scura_e_carnagione_chiara:'
    },
    u'\U0001F9D1\U0001F3FF\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FE': {
        'en' : u':people_holding_hands_dark_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':sich_an_den_händen_haltende_personen_dunkle_hautfarbe,_mitteldunkle_hautfarbe:',
        'es': u':dos_personas_dándose_la_mano_tono_de_piel_oscuro_y_tono_de_piel_oscuro_medio:',
        'fr': u':deux_personnes_se_tenant_la_main_peau_foncée_et_peau_mate:',
        'pt': u':pessoas_de_mãos_dadas_pele_escura_e_pele_morena_escura:',
        'it': u':persone_che_si_tengono_per_mano_carnagione_scura_e_carnagione_abbastanza_scura:'
    },
    u'\U0001F9D1\U0001F3FF\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FC': {
        'en' : u':people_holding_hands_dark_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':sich_an_den_händen_haltende_personen_dunkle_hautfarbe,_mittelhelle_hautfarbe:',
        'es': u':dos_personas_dándose_la_mano_tono_de_piel_oscuro_y_tono_de_piel_claro_medio:',
        'fr': u':deux_personnes_se_tenant_la_main_peau_foncée_et_peau_moyennement_claire:',
        'pt': u':pessoas_de_mãos_dadas_pele_escura_e_pele_morena_clara:',
        'it': u':persone_che_si_tengono_per_mano_carnagione_scura_e_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9D1\U0001F3FF\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FD': {
        'en' : u':people_holding_hands_dark_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':sich_an_den_händen_haltende_personen_dunkle_hautfarbe,_mittlere_hautfarbe:',
        'es': u':dos_personas_dándose_la_mano_tono_de_piel_oscuro_y_tono_de_piel_medio:',
        'fr': u':deux_personnes_se_tenant_la_main_peau_foncée_et_peau_légèrement_mate:',
        'pt': u':pessoas_de_mãos_dadas_pele_escura_e_pele_morena:',
        'it': u':persone_che_si_tengono_per_mano_carnagione_scura_e_carnagione_olivastra:'
    },
    u'\U0001F9D1\U0001F3FB\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FB': {
        'en' : u':people_holding_hands_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':sich_an_den_händen_haltende_personen_helle_hautfarbe:',
        'es': u':dos_personas_dándose_la_mano_tono_de_piel_claro:',
        'fr': u':deux_personnes_se_tenant_la_main_peau_claire:',
        'pt': u':pessoas_de_mãos_dadas_pele_clara:',
        'it': u':persone_che_si_tengono_per_mano_carnagione_chiara:'
    },
    u'\U0001F9D1\U0001F3FB\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FF': {
        'en' : u':people_holding_hands_light_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':sich_an_den_händen_haltende_personen_helle_hautfarbe,_dunkle_hautfarbe:',
        'es': u':dos_personas_dándose_la_mano_tono_de_piel_claro_y_tono_de_piel_oscuro:',
        'fr': u':deux_personnes_se_tenant_la_main_peau_claire_et_peau_foncée:',
        'pt': u':pessoas_de_mãos_dadas_pele_clara_e_pele_escura:',
        'it': u':persone_che_si_tengono_per_mano_carnagione_chiara_e_carnagione_scura:'
    },
    u'\U0001F9D1\U0001F3FB\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FE': {
        'en' : u':people_holding_hands_light_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':sich_an_den_händen_haltende_personen_helle_hautfarbe,_mitteldunkle_hautfarbe:',
        'es': u':dos_personas_dándose_la_mano_tono_de_piel_claro_y_tono_de_piel_oscuro_medio:',
        'fr': u':deux_personnes_se_tenant_la_main_peau_claire_et_peau_mate:',
        'pt': u':pessoas_de_mãos_dadas_pele_clara_e_pele_morena_escura:',
        'it': u':persone_che_si_tengono_per_mano_carnagione_chiara_e_carnagione_abbastanza_scura:'
    },
    u'\U0001F9D1\U0001F3FB\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FC': {
        'en' : u':people_holding_hands_light_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':sich_an_den_händen_haltende_personen_helle_hautfarbe,_mittelhelle_hautfarbe:',
        'es': u':dos_personas_dándose_la_mano_tono_de_piel_claro_y_tono_de_piel_claro_medio:',
        'fr': u':deux_personnes_se_tenant_la_main_peau_claire_et_peau_moyennement_claire:',
        'pt': u':pessoas_de_mãos_dadas_pele_clara_e_pele_morena_clara:',
        'it': u':persone_che_si_tengono_per_mano_carnagione_chiara_e_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9D1\U0001F3FB\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FD': {
        'en' : u':people_holding_hands_light_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':sich_an_den_händen_haltende_personen_helle_hautfarbe,_mittlere_hautfarbe:',
        'es': u':dos_personas_dándose_la_mano_tono_de_piel_claro_y_tono_de_piel_medio:',
        'fr': u':deux_personnes_se_tenant_la_main_peau_claire_et_peau_légèrement_mate:',
        'pt': u':pessoas_de_mãos_dadas_pele_clara_e_pele_morena:',
        'it': u':persone_che_si_tengono_per_mano_carnagione_chiara_e_carnagione_olivastra:'
    },
    u'\U0001F9D1\U0001F3FE\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FE': {
        'en' : u':people_holding_hands_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':sich_an_den_händen_haltende_personen_mitteldunkle_hautfarbe:',
        'es': u':dos_personas_dándose_la_mano_tono_de_piel_oscuro_medio:',
        'fr': u':deux_personnes_se_tenant_la_main_peau_mate:',
        'pt': u':pessoas_de_mãos_dadas_pele_morena_escura:',
        'it': u':persone_che_si_tengono_per_mano_carnagione_abbastanza_scura:'
    },
    u'\U0001F9D1\U0001F3FE\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FF': {
        'en' : u':people_holding_hands_medium-dark_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':sich_an_den_händen_haltende_personen_mitteldunkle_hautfarbe,_dunkle_hautfarbe:',
        'es': u':dos_personas_dándose_la_mano_tono_de_piel_oscuro_medio_y_tono_de_piel_oscuro:',
        'fr': u':deux_personnes_se_tenant_la_main_peau_mate_et_peau_foncée:',
        'pt': u':pessoas_de_mãos_dadas_pele_morena_escura_e_pele_escura:',
        'it': u':persone_che_si_tengono_per_mano_carnagione_abbastanza_scura_e_carnagione_scura:'
    },
    u'\U0001F9D1\U0001F3FE\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FB': {
        'en' : u':people_holding_hands_medium-dark_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':sich_an_den_händen_haltende_personen_mitteldunkle_hautfarbe,_helle_hautfarbe:',
        'es': u':dos_personas_dándose_la_mano_tono_de_piel_oscuro_medio_y_tono_de_piel_claro:',
        'fr': u':deux_personnes_se_tenant_la_main_peau_mate_et_peau_claire:',
        'pt': u':pessoas_de_mãos_dadas_pele_morena_escura_e_pele_clara:',
        'it': u':persone_che_si_tengono_per_mano_carnagione_abbastanza_scura_e_carnagione_chiara:'
    },
    u'\U0001F9D1\U0001F3FE\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FC': {
        'en' : u':people_holding_hands_medium-dark_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':sich_an_den_händen_haltende_personen_mitteldunkle_hautfarbe,_mittelhelle_hautfarbe:',
        'es': u':dos_personas_dándose_la_mano_tono_de_piel_oscuro_medio_y_tono_de_piel_claro_medio:',
        'fr': u':deux_personnes_se_tenant_la_main_peau_mate_et_peau_moyennement_claire:',
        'pt': u':pessoas_de_mãos_dadas_pele_morena_escura_e_pele_morena_clara:',
        'it': u':persone_che_si_tengono_per_mano_carnagione_abbastanza_scura_e_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9D1\U0001F3FE\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FD': {
        'en' : u':people_holding_hands_medium-dark_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':sich_an_den_händen_haltende_personen_mitteldunkle_hautfarbe,_mittlere_hautfarbe:',
        'es': u':dos_personas_dándose_la_mano_tono_de_piel_oscuro_medio_y_tono_de_piel_medio:',
        'fr': u':deux_personnes_se_tenant_la_main_peau_mate_et_peau_légèrement_mate:',
        'pt': u':pessoas_de_mãos_dadas_pele_morena_escura_e_pele_morena:',
        'it': u':persone_che_si_tengono_per_mano_carnagione_abbastanza_scura_e_carnagione_olivastra:'
    },
    u'\U0001F9D1\U0001F3FC\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FC': {
        'en' : u':people_holding_hands_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':sich_an_den_händen_haltende_personen_mittelhelle_hautfarbe:',
        'es': u':dos_personas_dándose_la_mano_tono_de_piel_claro_medio:',
        'fr': u':deux_personnes_se_tenant_la_main_peau_moyennement_claire:',
        'pt': u':pessoas_de_mãos_dadas_pele_morena_clara:',
        'it': u':persone_che_si_tengono_per_mano_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9D1\U0001F3FC\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FF': {
        'en' : u':people_holding_hands_medium-light_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':sich_an_den_händen_haltende_personen_mittelhelle_hautfarbe,_dunkle_hautfarbe:',
        'es': u':dos_personas_dándose_la_mano_tono_de_piel_claro_medio_y_tono_de_piel_oscuro:',
        'fr': u':deux_personnes_se_tenant_la_main_peau_moyennement_claire_et_peau_foncée:',
        'pt': u':pessoas_de_mãos_dadas_pele_morena_clara_e_pele_escura:',
        'it': u':persone_che_si_tengono_per_mano_carnagione_abbastanza_chiara_e_carnagione_scura:'
    },
    u'\U0001F9D1\U0001F3FC\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FB': {
        'en' : u':people_holding_hands_medium-light_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':sich_an_den_händen_haltende_personen_mittelhelle_hautfarbe,_helle_hautfarbe:',
        'es': u':dos_personas_dándose_la_mano_tono_de_piel_claro_medio_y_tono_de_piel_claro:',
        'fr': u':deux_personnes_se_tenant_la_main_peau_moyennement_claire_et_peau_claire:',
        'pt': u':pessoas_de_mãos_dadas_pele_morena_clara_e_pele_clara:',
        'it': u':persone_che_si_tengono_per_mano_carnagione_abbastanza_chiara_e_carnagione_chiara:'
    },
    u'\U0001F9D1\U0001F3FC\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FE': {
        'en' : u':people_holding_hands_medium-light_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':sich_an_den_händen_haltende_personen_mittelhelle_hautfarbe,_mitteldunkle_hautfarbe:',
        'es': u':dos_personas_dándose_la_mano_tono_de_piel_claro_medio_y_tono_de_piel_oscuro_medio:',
        'fr': u':deux_personnes_se_tenant_la_main_peau_moyennement_claire_et_peau_mate:',
        'pt': u':pessoas_de_mãos_dadas_pele_morena_clara_e_pele_morena_escura:',
        'it': u':persone_che_si_tengono_per_mano_carnagione_abbastanza_chiara_e_carnagione_abbastanza_scura:'
    },
    u'\U0001F9D1\U0001F3FC\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FD': {
        'en' : u':people_holding_hands_medium-light_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':sich_an_den_händen_haltende_personen_mittelhelle_hautfarbe,_mittlere_hautfarbe:',
        'es': u':dos_personas_dándose_la_mano_tono_de_piel_claro_medio_y_tono_de_piel_medio:',
        'fr': u':deux_personnes_se_tenant_la_main_peau_moyennement_claire_et_peau_légèrement_mate:',
        'pt': u':pessoas_de_mãos_dadas_pele_morena_clara_e_pele_morena:',
        'it': u':persone_che_si_tengono_per_mano_carnagione_abbastanza_chiara_e_carnagione_olivastra:'
    },
    u'\U0001F9D1\U0001F3FD\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FD': {
        'en' : u':people_holding_hands_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':sich_an_den_händen_haltende_personen_mittlere_hautfarbe:',
        'es': u':dos_personas_dándose_la_mano_tono_de_piel_medio:',
        'fr': u':deux_personnes_se_tenant_la_main_peau_légèrement_mate:',
        'pt': u':pessoas_de_mãos_dadas_pele_morena:',
        'it': u':persone_che_si_tengono_per_mano_carnagione_olivastra:'
    },
    u'\U0001F9D1\U0001F3FD\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FF': {
        'en' : u':people_holding_hands_medium_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':sich_an_den_händen_haltende_personen_mittlere_hautfarbe,_dunkle_hautfarbe:',
        'es': u':dos_personas_dándose_la_mano_tono_de_piel_medio_y_tono_de_piel_oscuro:',
        'fr': u':deux_personnes_se_tenant_la_main_peau_légèrement_mate_et_peau_foncée:',
        'pt': u':pessoas_de_mãos_dadas_pele_morena_e_pele_escura:',
        'it': u':persone_che_si_tengono_per_mano_carnagione_olivastra_e_carnagione_scura:'
    },
    u'\U0001F9D1\U0001F3FD\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FB': {
        'en' : u':people_holding_hands_medium_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':sich_an_den_händen_haltende_personen_mittlere_hautfarbe,_helle_hautfarbe:',
        'es': u':dos_personas_dándose_la_mano_tono_de_piel_medio_y_tono_de_piel_claro:',
        'fr': u':deux_personnes_se_tenant_la_main_peau_légèrement_mate_et_peau_claire:',
        'pt': u':pessoas_de_mãos_dadas_pele_morena_e_pele_clara:',
        'it': u':persone_che_si_tengono_per_mano_carnagione_olivastra_e_carnagione_chiara:'
    },
    u'\U0001F9D1\U0001F3FD\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FE': {
        'en' : u':people_holding_hands_medium_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':sich_an_den_händen_haltende_personen_mittlere_hautfarbe,_mitteldunkle_hautfarbe:',
        'es': u':dos_personas_dándose_la_mano_tono_de_piel_medio_y_tono_de_piel_oscuro_medio:',
        'fr': u':deux_personnes_se_tenant_la_main_peau_légèrement_mate_et_peau_mate:',
        'pt': u':pessoas_de_mãos_dadas_pele_morena_e_pele_morena_escura:',
        'it': u':persone_che_si_tengono_per_mano_carnagione_olivastra_e_carnagione_abbastanza_scura:'
    },
    u'\U0001F9D1\U0001F3FD\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FC': {
        'en' : u':people_holding_hands_medium_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':sich_an_den_händen_haltende_personen_mittlere_hautfarbe,_mittelhelle_hautfarbe:',
        'es': u':dos_personas_dándose_la_mano_tono_de_piel_medio_y_tono_de_piel_claro_medio:',
        'fr': u':deux_personnes_se_tenant_la_main_peau_légèrement_mate_et_peau_moyennement_claire:',
        'pt': u':pessoas_de_mãos_dadas_pele_morena_e_pele_morena_clara:',
        'it': u':persone_che_si_tengono_per_mano_carnagione_olivastra_e_carnagione_abbastanza_chiara:'
    },
    u'\U0001FAC2': {
        'en' : u':people_hugging:',
        'status' : fully_qualified,
        'E' : 13,
        'de': u':sich_umarmende_personen:',
        'es': u':personas_abrazándose:',
        'fr': u':étreinte:',
        'pt': u':pessoas_se_abraçando:',
        'it': u':persone_che_si_abbracciano:'
    },
    u'\U0001F46F': {
        'en' : u':people_with_bunny_ears:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':dancers:'],
        'de': u':personen_mit_hasenohren:',
        'es': u':personas_con_orejas_de_conejo:',
        'fr': u':personnes_avec_des_oreilles_de_lapin:',
        'pt': u':pessoas_com_orelhas_de_coelho:',
        'it': u':persone_con_orecchie_da_coniglio:'
    },
    u'\U0001F93C': {
        'en' : u':people_wrestling:',
        'status' : fully_qualified,
        'E' : 3,
        'alias' : [u':wrestling:'],
        'de': u':ringer(in):',
        'es': u':personas_luchando:',
        'fr': u':personnes_faisant_de_la_lutte:',
        'pt': u':pessoas_lutando:',
        'it': u':persone_che_fanno_la_lotta:'
    },
    u'\U0001F3AD': {
        'en' : u':performing_arts:',
        'status' : fully_qualified,
        'E' : 0.6,
        'variant': True,
        'de': u':masken:',
        'es': u':máscaras_de_teatro:',
        'fr': u':spectacle_vivant:',
        'pt': u':máscara:',
        'it': u':maschere:'
    },
    u'\U0001F623': {
        'en' : u':persevering_face:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':persevere:'],
        'de': u':entschlossenes_gesicht:',
        'es': u':cara_desesperada:',
        'fr': u':visage_persévérant:',
        'pt': u':rosto_perseverante:',
        'it': u':faccina_perseverante:'
    },
    u'\U0001F9D1': {
        'en' : u':person:',
        'status' : fully_qualified,
        'E' : 5,
        'alias' : [u':adult:'],
        'de': u':erwachsener:',
        'es': u':persona_adulta:',
        'fr': u':adulte:',
        'pt': u':pessoa:',
        'it': u':persona:'
    },
    u'\U0001F9D1\U0000200D\U0001F9B2': {
        'en' : u':person_bald:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':erwachsener_glatze:',
        'es': u':persona_adulta_sin_pelo:',
        'fr': u':adulte_chauve:',
        'pt': u':pessoa_careca:',
        'it': u':persona_calvo:'
    },
    u'\U0001F9D4': {
        'en' : u':person_beard:',
        'status' : fully_qualified,
        'E' : 5,
        'alias' : [u':bearded_person:'],
        'de': u':mann_bart:',
        'es': u':hombre_con_barba:',
        'fr': u':homme_barbu:',
        'pt': u':homem_barba:',
        'it': u':uomo_con_la_barba:'
    },
    u'\U0001F6B4': {
        'en' : u':person_biking:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [u':bicyclist:'],
        'de': u':radfahrer(in):',
        'es': u':persona_en_bicicleta:',
        'fr': u':cycliste:',
        'pt': u':ciclista:',
        'it': u':ciclista:'
    },
    u'\U0001F6B4\U0001F3FF': {
        'en' : u':person_biking_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':radfahrer(in)_dunkle_hautfarbe:',
        'es': u':persona_en_bicicleta_tono_de_piel_oscuro:',
        'fr': u':cycliste_peau_foncée:',
        'pt': u':ciclista_pele_escura:',
        'it': u':ciclista_carnagione_scura:'
    },
    u'\U0001F6B4\U0001F3FB': {
        'en' : u':person_biking_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':radfahrer(in)_helle_hautfarbe:',
        'es': u':persona_en_bicicleta_tono_de_piel_claro:',
        'fr': u':cycliste_peau_claire:',
        'pt': u':ciclista_pele_clara:',
        'it': u':ciclista_carnagione_chiara:'
    },
    u'\U0001F6B4\U0001F3FE': {
        'en' : u':person_biking_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':radfahrer(in)_mitteldunkle_hautfarbe:',
        'es': u':persona_en_bicicleta_tono_de_piel_oscuro_medio:',
        'fr': u':cycliste_peau_mate:',
        'pt': u':ciclista_pele_morena_escura:',
        'it': u':ciclista_carnagione_abbastanza_scura:'
    },
    u'\U0001F6B4\U0001F3FC': {
        'en' : u':person_biking_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':radfahrer(in)_mittelhelle_hautfarbe:',
        'es': u':persona_en_bicicleta_tono_de_piel_claro_medio:',
        'fr': u':cycliste_peau_moyennement_claire:',
        'pt': u':ciclista_pele_morena_clara:',
        'it': u':ciclista_carnagione_abbastanza_chiara:'
    },
    u'\U0001F6B4\U0001F3FD': {
        'en' : u':person_biking_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':radfahrer(in)_mittlere_hautfarbe:',
        'es': u':persona_en_bicicleta_tono_de_piel_medio:',
        'fr': u':cycliste_peau_légèrement_mate:',
        'pt': u':ciclista_pele_morena:',
        'it': u':ciclista_carnagione_olivastra:'
    },
    u'\U0001F471': {
        'en' : u':person_blond_hair:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':person_with_blond_hair:', u':blond_haired_person:'],
        'de': u':person_blondes_haar:',
        'es': u':persona_adulta_rubia:',
        'fr': u':personne_blonde:',
        'pt': u':pessoa_cabelo_louro:',
        'it': u':persona_bionda:'
    },
    u'\U000026F9\U0000FE0F': {
        'en' : u':person_bouncing_ball:',
        'status' : fully_qualified,
        'E' : 0.7,
        'alias' : [u':person_with_ball:', u':bouncing_ball_person:'],
        'variant': True,
        'de': u':person_mit_ball:',
        'es': u':persona_botando_un_balón:',
        'fr': u':personne_avec_ballon:',
        'pt': u':pessoa_jogando_basquete:',
        'it': u':persona_che_fa_rimbalzare_una_palla:'
    },
    u'\U000026F9': {
        'en' : u':person_bouncing_ball:',
        'status' : unqualified,
        'E' : 0.7,
        'alias' : [u':person_with_ball:', u':bouncing_ball_person:'],
        'variant': True,
        'de': u':person_mit_ball:',
        'es': u':persona_botando_un_balón:',
        'fr': u':personne_avec_ballon:',
        'pt': u':pessoa_jogando_basquete:',
        'it': u':persona_che_fa_rimbalzare_una_palla:'
    },
    u'\U000026F9\U0001F3FF': {
        'en' : u':person_bouncing_ball_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 2,
        'de': u':person_mit_ball_dunkle_hautfarbe:',
        'es': u':persona_botando_un_balón_tono_de_piel_oscuro:',
        'fr': u':personne_avec_ballon_peau_foncée:',
        'pt': u':pessoa_jogando_basquete_pele_escura:',
        'it': u':persona_che_fa_rimbalzare_una_palla_carnagione_scura:'
    },
    u'\U000026F9\U0001F3FB': {
        'en' : u':person_bouncing_ball_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 2,
        'de': u':person_mit_ball_helle_hautfarbe:',
        'es': u':persona_botando_un_balón_tono_de_piel_claro:',
        'fr': u':personne_avec_ballon_peau_claire:',
        'pt': u':pessoa_jogando_basquete_pele_clara:',
        'it': u':persona_che_fa_rimbalzare_una_palla_carnagione_chiara:'
    },
    u'\U000026F9\U0001F3FE': {
        'en' : u':person_bouncing_ball_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 2,
        'de': u':person_mit_ball_mitteldunkle_hautfarbe:',
        'es': u':persona_botando_un_balón_tono_de_piel_oscuro_medio:',
        'fr': u':personne_avec_ballon_peau_mate:',
        'pt': u':pessoa_jogando_basquete_pele_morena_escura:',
        'it': u':persona_che_fa_rimbalzare_una_palla_carnagione_abbastanza_scura:'
    },
    u'\U000026F9\U0001F3FC': {
        'en' : u':person_bouncing_ball_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 2,
        'de': u':person_mit_ball_mittelhelle_hautfarbe:',
        'es': u':persona_botando_un_balón_tono_de_piel_claro_medio:',
        'fr': u':personne_avec_ballon_peau_moyennement_claire:',
        'pt': u':pessoa_jogando_basquete_pele_morena_clara:',
        'it': u':persona_che_fa_rimbalzare_una_palla_carnagione_abbastanza_chiara:'
    },
    u'\U000026F9\U0001F3FD': {
        'en' : u':person_bouncing_ball_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 2,
        'de': u':person_mit_ball_mittlere_hautfarbe:',
        'es': u':persona_botando_un_balón_tono_de_piel_medio:',
        'fr': u':personne_avec_ballon_peau_légèrement_mate:',
        'pt': u':pessoa_jogando_basquete_pele_morena:',
        'it': u':persona_che_fa_rimbalzare_una_palla_carnagione_olivastra:'
    },
    u'\U0001F647': {
        'en' : u':person_bowing:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':bow:'],
        'de': u':sich_verbeugende_person:',
        'es': u':persona_haciendo_una_reverencia:',
        'fr': u':personne_qui_s’incline:',
        'pt': u':pessoa_fazendo_reverência:',
        'it': u':persona_che_fa_un_inchino_profondo:'
    },
    u'\U0001F647\U0001F3FF': {
        'en' : u':person_bowing_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':sich_verbeugende_person_dunkle_hautfarbe:',
        'es': u':persona_haciendo_una_reverencia_tono_de_piel_oscuro:',
        'fr': u':personne_qui_s’incline_peau_foncée:',
        'pt': u':pessoa_fazendo_reverência_pele_escura:',
        'it': u':persona_che_fa_un_inchino_profondo_carnagione_scura:'
    },
    u'\U0001F647\U0001F3FB': {
        'en' : u':person_bowing_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':sich_verbeugende_person_helle_hautfarbe:',
        'es': u':persona_haciendo_una_reverencia_tono_de_piel_claro:',
        'fr': u':personne_qui_s’incline_peau_claire:',
        'pt': u':pessoa_fazendo_reverência_pele_clara:',
        'it': u':persona_che_fa_un_inchino_profondo_carnagione_chiara:'
    },
    u'\U0001F647\U0001F3FE': {
        'en' : u':person_bowing_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':sich_verbeugende_person_mitteldunkle_hautfarbe:',
        'es': u':persona_haciendo_una_reverencia_tono_de_piel_oscuro_medio:',
        'fr': u':personne_qui_s’incline_peau_mate:',
        'pt': u':pessoa_fazendo_reverência_pele_morena_escura:',
        'it': u':persona_che_fa_un_inchino_profondo_carnagione_abbastanza_scura:'
    },
    u'\U0001F647\U0001F3FC': {
        'en' : u':person_bowing_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':sich_verbeugende_person_mittelhelle_hautfarbe:',
        'es': u':persona_haciendo_una_reverencia_tono_de_piel_claro_medio:',
        'fr': u':personne_qui_s’incline_peau_moyennement_claire:',
        'pt': u':pessoa_fazendo_reverência_pele_morena_clara:',
        'it': u':persona_che_fa_un_inchino_profondo_carnagione_abbastanza_chiara:'
    },
    u'\U0001F647\U0001F3FD': {
        'en' : u':person_bowing_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':sich_verbeugende_person_mittlere_hautfarbe:',
        'es': u':persona_haciendo_una_reverencia_tono_de_piel_medio:',
        'fr': u':personne_qui_s’incline_peau_légèrement_mate:',
        'pt': u':pessoa_fazendo_reverência_pele_morena:',
        'it': u':persona_che_fa_un_inchino_profondo_carnagione_olivastra:'
    },
    u'\U0001F938': {
        'en' : u':person_cartwheeling:',
        'status' : fully_qualified,
        'E' : 3,
        'alias' : [u':cartwheeling:'],
        'de': u':rad_schlagende_person:',
        'es': u':persona_haciendo_voltereta_lateral:',
        'fr': u':personne_faisant_la_roue:',
        'pt': u':pessoa_fazendo_estrela:',
        'it': u':persona_che_fa_la_ruota:'
    },
    u'\U0001F938\U0001F3FF': {
        'en' : u':person_cartwheeling_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 3,
        'de': u':rad_schlagende_person_dunkle_hautfarbe:',
        'es': u':persona_haciendo_voltereta_lateral_tono_de_piel_oscuro:',
        'fr': u':personne_faisant_la_roue_peau_foncée:',
        'pt': u':pessoa_fazendo_estrela_pele_escura:',
        'it': u':persona_che_fa_la_ruota_carnagione_scura:'
    },
    u'\U0001F938\U0001F3FB': {
        'en' : u':person_cartwheeling_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 3,
        'de': u':rad_schlagende_person_helle_hautfarbe:',
        'es': u':persona_haciendo_voltereta_lateral_tono_de_piel_claro:',
        'fr': u':personne_faisant_la_roue_peau_claire:',
        'pt': u':pessoa_fazendo_estrela_pele_clara:',
        'it': u':persona_che_fa_la_ruota_carnagione_chiara:'
    },
    u'\U0001F938\U0001F3FE': {
        'en' : u':person_cartwheeling_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 3,
        'de': u':rad_schlagende_person_mitteldunkle_hautfarbe:',
        'es': u':persona_haciendo_voltereta_lateral_tono_de_piel_oscuro_medio:',
        'fr': u':personne_faisant_la_roue_peau_mate:',
        'pt': u':pessoa_fazendo_estrela_pele_morena_escura:',
        'it': u':persona_che_fa_la_ruota_carnagione_abbastanza_scura:'
    },
    u'\U0001F938\U0001F3FC': {
        'en' : u':person_cartwheeling_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 3,
        'de': u':rad_schlagende_person_mittelhelle_hautfarbe:',
        'es': u':persona_haciendo_voltereta_lateral_tono_de_piel_claro_medio:',
        'fr': u':personne_faisant_la_roue_peau_moyennement_claire:',
        'pt': u':pessoa_fazendo_estrela_pele_morena_clara:',
        'it': u':persona_che_fa_la_ruota_carnagione_abbastanza_chiara:'
    },
    u'\U0001F938\U0001F3FD': {
        'en' : u':person_cartwheeling_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 3,
        'de': u':rad_schlagende_person_mittlere_hautfarbe:',
        'es': u':persona_haciendo_voltereta_lateral_tono_de_piel_medio:',
        'fr': u':personne_faisant_la_roue_peau_légèrement_mate:',
        'pt': u':pessoa_fazendo_estrela_pele_morena:',
        'it': u':persona_che_fa_la_ruota_carnagione_olivastra:'
    },
    u'\U0001F9D7': {
        'en' : u':person_climbing:',
        'status' : fully_qualified,
        'E' : 5,
        'alias' : [u':climbing:'],
        'de': u':bergsteiger(in):',
        'es': u':persona_escalando:',
        'fr': u':personne_qui_grimpe:',
        'pt': u':pessoa_escalando:',
        'it': u':persona_che_scala:'
    },
    u'\U0001F9D7\U0001F3FF': {
        'en' : u':person_climbing_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':bergsteiger(in)_dunkle_hautfarbe:',
        'es': u':persona_escalando_tono_de_piel_oscuro:',
        'fr': u':personne_qui_grimpe_peau_foncée:',
        'pt': u':pessoa_escalando_pele_escura:',
        'it': u':persona_che_scala_carnagione_scura:'
    },
    u'\U0001F9D7\U0001F3FB': {
        'en' : u':person_climbing_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':bergsteiger(in)_helle_hautfarbe:',
        'es': u':persona_escalando_tono_de_piel_claro:',
        'fr': u':personne_qui_grimpe_peau_claire:',
        'pt': u':pessoa_escalando_pele_clara:',
        'it': u':persona_che_scala_carnagione_chiara:'
    },
    u'\U0001F9D7\U0001F3FE': {
        'en' : u':person_climbing_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':bergsteiger(in)_mitteldunkle_hautfarbe:',
        'es': u':persona_escalando_tono_de_piel_oscuro_medio:',
        'fr': u':personne_qui_grimpe_peau_mate:',
        'pt': u':pessoa_escalando_pele_morena_escura:',
        'it': u':persona_che_scala_carnagione_abbastanza_scura:'
    },
    u'\U0001F9D7\U0001F3FC': {
        'en' : u':person_climbing_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':bergsteiger(in)_mittelhelle_hautfarbe:',
        'es': u':persona_escalando_tono_de_piel_claro_medio:',
        'fr': u':personne_qui_grimpe_peau_moyennement_claire:',
        'pt': u':pessoa_escalando_pele_morena_clara:',
        'it': u':persona_che_scala_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9D7\U0001F3FD': {
        'en' : u':person_climbing_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':bergsteiger(in)_mittlere_hautfarbe:',
        'es': u':persona_escalando_tono_de_piel_medio:',
        'fr': u':personne_qui_grimpe_peau_légèrement_mate:',
        'pt': u':pessoa_escalando_pele_morena:',
        'it': u':persona_che_scala_carnagione_olivastra:'
    },
    u'\U0001F9D1\U0000200D\U0001F9B1': {
        'en' : u':person_curly_hair:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':erwachsener_lockiges_haar:',
        'es': u':persona_adulta_pelo_rizado:',
        'fr': u':adulte_cheveux_bouclés:',
        'pt': u':pessoa_cabelo_cacheado:',
        'it': u':persona_capelli_ricci:'
    },
    u'\U0001F9D1\U0001F3FF': {
        'en' : u':person_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':erwachsener_dunkle_hautfarbe:',
        'es': u':persona_adulta_tono_de_piel_oscuro:',
        'fr': u':adulte_peau_foncée:',
        'pt': u':pessoa_pele_escura:',
        'it': u':persona_carnagione_scura:'
    },
    u'\U0001F9D1\U0001F3FF\U0000200D\U0001F9B2': {
        'en' : u':person_dark_skin_tone_bald:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':erwachsener_dunkle_hautfarbe,_glatze:',
        'es': u':persona_adulta_tono_de_piel_oscuro_y_sin_pelo:',
        'fr': u':adulte_peau_foncée_et_chauve:',
        'pt': u':pessoa_pele_escura_e_careca:',
        'it': u':persona_carnagione_scura_e_calvo:'
    },
    u'\U0001F9D4\U0001F3FF': {
        'en' : u':person_dark_skin_tone_beard:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':mann_dunkle_hautfarbe,_bart:',
        'es': u':hombre_con_barba_tono_de_piel_oscuro:',
        'fr': u':homme_barbu_peau_foncée:',
        'pt': u':homem_pele_escura_e_barba:',
        'it': u':uomo_con_la_barba_carnagione_scura:'
    },
    u'\U0001F471\U0001F3FF': {
        'en' : u':person_dark_skin_tone_blond_hair:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':person_dunkle_hautfarbe,_blondes_haar:',
        'es': u':persona_adulta_rubia_tono_de_piel_oscuro:',
        'fr': u':personne_blonde_peau_foncée:',
        'pt': u':pessoa_pele_escura_e_cabelo_louro:',
        'it': u':persona_bionda_carnagione_scura:'
    },
    u'\U0001F9D1\U0001F3FF\U0000200D\U0001F9B1': {
        'en' : u':person_dark_skin_tone_curly_hair:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':erwachsener_dunkle_hautfarbe,_lockiges_haar:',
        'es': u':persona_adulta_tono_de_piel_oscuro_y_pelo_rizado:',
        'fr': u':adulte_peau_foncée_et_cheveux_bouclés:',
        'pt': u':pessoa_pele_escura_e_cabelo_cacheado:',
        'it': u':persona_carnagione_scura_e_capelli_ricci:'
    },
    u'\U0001F9D1\U0001F3FF\U0000200D\U0001F9B0': {
        'en' : u':person_dark_skin_tone_red_hair:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':erwachsener_dunkle_hautfarbe,_rotes_haar:',
        'es': u':persona_adulta_tono_de_piel_oscuro_y_pelo_pelirrojo:',
        'fr': u':adulte_peau_foncée_et_cheveux_roux:',
        'pt': u':pessoa_pele_escura_e_cabelo_vermelho:',
        'it': u':persona_carnagione_scura_e_capelli_rossi:'
    },
    u'\U0001F9D1\U0001F3FF\U0000200D\U0001F9B3': {
        'en' : u':person_dark_skin_tone_white_hair:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':erwachsener_dunkle_hautfarbe,_weißes_haar:',
        'es': u':persona_adulta_tono_de_piel_oscuro_y_pelo_blanco:',
        'fr': u':adulte_peau_foncée_et_cheveux_blancs:',
        'pt': u':pessoa_pele_escura_e_cabelo_branco:',
        'it': u':persona_carnagione_scura_e_capelli_bianchi:'
    },
    u'\U0001F926': {
        'en' : u':person_facepalming:',
        'status' : fully_qualified,
        'E' : 3,
        'alias' : [u':facepalm:'],
        'de': u':sich_an_den_kopf_fassende_person:',
        'es': u':persona_con_la_mano_en_la_frente:',
        'fr': u':personne_avec_la_paume_sur_le_visage:',
        'pt': u':pessoa_decepcionada:',
        'it': u':persona_esasperata:'
    },
    u'\U0001F926\U0001F3FF': {
        'en' : u':person_facepalming_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 3,
        'de': u':sich_an_den_kopf_fassende_person_dunkle_hautfarbe:',
        'es': u':persona_con_la_mano_en_la_frente_tono_de_piel_oscuro:',
        'fr': u':personne_avec_la_paume_sur_le_visage_peau_foncée:',
        'pt': u':pessoa_decepcionada_pele_escura:',
        'it': u':persona_esasperata_carnagione_scura:'
    },
    u'\U0001F926\U0001F3FB': {
        'en' : u':person_facepalming_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 3,
        'de': u':sich_an_den_kopf_fassende_person_helle_hautfarbe:',
        'es': u':persona_con_la_mano_en_la_frente_tono_de_piel_claro:',
        'fr': u':personne_avec_la_paume_sur_le_visage_peau_claire:',
        'pt': u':pessoa_decepcionada_pele_clara:',
        'it': u':persona_esasperata_carnagione_chiara:'
    },
    u'\U0001F926\U0001F3FE': {
        'en' : u':person_facepalming_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 3,
        'de': u':sich_an_den_kopf_fassende_person_mitteldunkle_hautfarbe:',
        'es': u':persona_con_la_mano_en_la_frente_tono_de_piel_oscuro_medio:',
        'fr': u':personne_avec_la_paume_sur_le_visage_peau_mate:',
        'pt': u':pessoa_decepcionada_pele_morena_escura:',
        'it': u':persona_esasperata_carnagione_abbastanza_scura:'
    },
    u'\U0001F926\U0001F3FC': {
        'en' : u':person_facepalming_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 3,
        'de': u':sich_an_den_kopf_fassende_person_mittelhelle_hautfarbe:',
        'es': u':persona_con_la_mano_en_la_frente_tono_de_piel_claro_medio:',
        'fr': u':personne_avec_la_paume_sur_le_visage_peau_moyennement_claire:',
        'pt': u':pessoa_decepcionada_pele_morena_clara:',
        'it': u':persona_esasperata_carnagione_abbastanza_chiara:'
    },
    u'\U0001F926\U0001F3FD': {
        'en' : u':person_facepalming_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 3,
        'de': u':sich_an_den_kopf_fassende_person_mittlere_hautfarbe:',
        'es': u':persona_con_la_mano_en_la_frente_tono_de_piel_medio:',
        'fr': u':personne_avec_la_paume_sur_le_visage_peau_légèrement_mate:',
        'pt': u':pessoa_decepcionada_pele_morena:',
        'it': u':persona_esasperata_carnagione_olivastra:'
    },
    u'\U0001F9D1\U0000200D\U0001F37C': {
        'en' : u':person_feeding_baby:',
        'status' : fully_qualified,
        'E' : 13,
        'de': u':stillende_person:',
        'es': u':persona_alimentando_a_bebé:',
        'fr': u':personne_allaitant_un_bébé:',
        'pt': u':pessoa_alimentando_bebê:',
        'it': u':persona_che_allatta:'
    },
    u'\U0001F9D1\U0001F3FF\U0000200D\U0001F37C': {
        'en' : u':person_feeding_baby_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13,
        'de': u':stillende_person_dunkle_hautfarbe:',
        'es': u':persona_alimentando_a_bebé_tono_de_piel_oscuro:',
        'fr': u':personne_allaitant_un_bébé_peau_foncée:',
        'pt': u':pessoa_alimentando_bebê_pele_escura:',
        'it': u':persona_che_allatta_carnagione_scura:'
    },
    u'\U0001F9D1\U0001F3FB\U0000200D\U0001F37C': {
        'en' : u':person_feeding_baby_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13,
        'de': u':stillende_person_helle_hautfarbe:',
        'es': u':persona_alimentando_a_bebé_tono_de_piel_claro:',
        'fr': u':personne_allaitant_un_bébé_peau_claire:',
        'pt': u':pessoa_alimentando_bebê_pele_clara:',
        'it': u':persona_che_allatta_carnagione_chiara:'
    },
    u'\U0001F9D1\U0001F3FE\U0000200D\U0001F37C': {
        'en' : u':person_feeding_baby_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13,
        'de': u':stillende_person_mitteldunkle_hautfarbe:',
        'es': u':persona_alimentando_a_bebé_tono_de_piel_oscuro_medio:',
        'fr': u':personne_allaitant_un_bébé_peau_mate:',
        'pt': u':pessoa_alimentando_bebê_pele_morena_escura:',
        'it': u':persona_che_allatta_carnagione_abbastanza_scura:'
    },
    u'\U0001F9D1\U0001F3FC\U0000200D\U0001F37C': {
        'en' : u':person_feeding_baby_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13,
        'de': u':stillende_person_mittelhelle_hautfarbe:',
        'es': u':persona_alimentando_a_bebé_tono_de_piel_claro_medio:',
        'fr': u':personne_allaitant_un_bébé_peau_moyennement_claire:',
        'pt': u':pessoa_alimentando_bebê_pele_morena_clara:',
        'it': u':persona_che_allatta_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9D1\U0001F3FD\U0000200D\U0001F37C': {
        'en' : u':person_feeding_baby_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13,
        'de': u':stillende_person_mittlere_hautfarbe:',
        'es': u':persona_alimentando_a_bebé_tono_de_piel_medio:',
        'fr': u':personne_allaitant_un_bébé_peau_légèrement_mate:',
        'pt': u':pessoa_alimentando_bebê_pele_morena:',
        'it': u':persona_che_allatta_carnagione_olivastra:'
    },
    u'\U0001F93A': {
        'en' : u':person_fencing:',
        'status' : fully_qualified,
        'E' : 3,
        'de': u':fechter(in):',
        'es': u':esgrima:',
        'fr': u':escrimeur:',
        'pt': u':esgrimista:',
        'it': u':schermidore:'
    },
    u'\U0001F64D': {
        'en' : u':person_frowning:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':frowning_person:'],
        'de': u':missmutige_person:',
        'es': u':persona_frunciendo_el_ceño:',
        'fr': u':personne_fronçant_les_sourcils:',
        'pt': u':franzindo_a_sobrancelha:',
        'it': u':persona_corrucciata:'
    },
    u'\U0001F64D\U0001F3FF': {
        'en' : u':person_frowning_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':missmutige_person_dunkle_hautfarbe:',
        'es': u':persona_frunciendo_el_ceño_tono_de_piel_oscuro:',
        'fr': u':personne_fronçant_les_sourcils_peau_foncée:',
        'pt': u':franzindo_a_sobrancelha_pele_escura:',
        'it': u':persona_corrucciata_carnagione_scura:'
    },
    u'\U0001F64D\U0001F3FB': {
        'en' : u':person_frowning_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':missmutige_person_helle_hautfarbe:',
        'es': u':persona_frunciendo_el_ceño_tono_de_piel_claro:',
        'fr': u':personne_fronçant_les_sourcils_peau_claire:',
        'pt': u':franzindo_a_sobrancelha_pele_clara:',
        'it': u':persona_corrucciata_carnagione_chiara:'
    },
    u'\U0001F64D\U0001F3FE': {
        'en' : u':person_frowning_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':missmutige_person_mitteldunkle_hautfarbe:',
        'es': u':persona_frunciendo_el_ceño_tono_de_piel_oscuro_medio:',
        'fr': u':personne_fronçant_les_sourcils_peau_mate:',
        'pt': u':franzindo_a_sobrancelha_pele_morena_escura:',
        'it': u':persona_corrucciata_carnagione_abbastanza_scura:'
    },
    u'\U0001F64D\U0001F3FC': {
        'en' : u':person_frowning_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':missmutige_person_mittelhelle_hautfarbe:',
        'es': u':persona_frunciendo_el_ceño_tono_de_piel_claro_medio:',
        'fr': u':personne_fronçant_les_sourcils_peau_moyennement_claire:',
        'pt': u':franzindo_a_sobrancelha_pele_morena_clara:',
        'it': u':persona_corrucciata_carnagione_abbastanza_chiara:'
    },
    u'\U0001F64D\U0001F3FD': {
        'en' : u':person_frowning_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':missmutige_person_mittlere_hautfarbe:',
        'es': u':persona_frunciendo_el_ceño_tono_de_piel_medio:',
        'fr': u':personne_fronçant_les_sourcils_peau_légèrement_mate:',
        'pt': u':franzindo_a_sobrancelha_pele_morena:',
        'it': u':persona_corrucciata_carnagione_olivastra:'
    },
    u'\U0001F645': {
        'en' : u':person_gesturing_NO:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':no_good:'],
        'de': u':person_mit_überkreuzten_armen:',
        'es': u':persona_haciendo_el_gesto_de_no:',
        'fr': u':personne_faisant_un_geste_d’interdiction:',
        'pt': u':pessoa_fazendo_gesto_de_“não”:',
        'it': u':persona_che_fa_segno_di_no:'
    },
    u'\U0001F645\U0001F3FF': {
        'en' : u':person_gesturing_NO_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':person_mit_überkreuzten_armen_dunkle_hautfarbe:',
        'es': u':persona_haciendo_el_gesto_de_no_tono_de_piel_oscuro:',
        'fr': u':personne_faisant_un_geste_d’interdiction_peau_foncée:',
        'pt': u':pessoa_fazendo_gesto_de_“não”_pele_escura:',
        'it': u':persona_che_fa_segno_di_no_carnagione_scura:'
    },
    u'\U0001F645\U0001F3FB': {
        'en' : u':person_gesturing_NO_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':person_mit_überkreuzten_armen_helle_hautfarbe:',
        'es': u':persona_haciendo_el_gesto_de_no_tono_de_piel_claro:',
        'fr': u':personne_faisant_un_geste_d’interdiction_peau_claire:',
        'pt': u':pessoa_fazendo_gesto_de_“não”_pele_clara:',
        'it': u':persona_che_fa_segno_di_no_carnagione_chiara:'
    },
    u'\U0001F645\U0001F3FE': {
        'en' : u':person_gesturing_NO_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':person_mit_überkreuzten_armen_mitteldunkle_hautfarbe:',
        'es': u':persona_haciendo_el_gesto_de_no_tono_de_piel_oscuro_medio:',
        'fr': u':personne_faisant_un_geste_d’interdiction_peau_mate:',
        'pt': u':pessoa_fazendo_gesto_de_“não”_pele_morena_escura:',
        'it': u':persona_che_fa_segno_di_no_carnagione_abbastanza_scura:'
    },
    u'\U0001F645\U0001F3FC': {
        'en' : u':person_gesturing_NO_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':person_mit_überkreuzten_armen_mittelhelle_hautfarbe:',
        'es': u':persona_haciendo_el_gesto_de_no_tono_de_piel_claro_medio:',
        'fr': u':personne_faisant_un_geste_d’interdiction_peau_moyennement_claire:',
        'pt': u':pessoa_fazendo_gesto_de_“não”_pele_morena_clara:',
        'it': u':persona_che_fa_segno_di_no_carnagione_abbastanza_chiara:'
    },
    u'\U0001F645\U0001F3FD': {
        'en' : u':person_gesturing_NO_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':person_mit_überkreuzten_armen_mittlere_hautfarbe:',
        'es': u':persona_haciendo_el_gesto_de_no_tono_de_piel_medio:',
        'fr': u':personne_faisant_un_geste_d’interdiction_peau_légèrement_mate:',
        'pt': u':pessoa_fazendo_gesto_de_“não”_pele_morena:',
        'it': u':persona_che_fa_segno_di_no_carnagione_olivastra:'
    },
    u'\U0001F646': {
        'en' : u':person_gesturing_OK:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':ok_woman:', u':ok_person:'],
        'de': u':person_mit_händen_auf_dem_kopf:',
        'es': u':persona_haciendo_el_gesto_de_de_acuerdo:',
        'fr': u':personne_faisant_un_geste_d’acceptation:',
        'pt': u':pessoa_fazendo_gesto_de_“ok”:',
        'it': u':persona_con_gesto_ok:'
    },
    u'\U0001F646\U0001F3FF': {
        'en' : u':person_gesturing_OK_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':person_mit_händen_auf_dem_kopf_dunkle_hautfarbe:',
        'es': u':persona_haciendo_el_gesto_de_de_acuerdo_tono_de_piel_oscuro:',
        'fr': u':personne_faisant_un_geste_d’acceptation_peau_foncée:',
        'pt': u':pessoa_fazendo_gesto_de_“ok”_pele_escura:',
        'it': u':persona_con_gesto_ok_carnagione_scura:'
    },
    u'\U0001F646\U0001F3FB': {
        'en' : u':person_gesturing_OK_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':person_mit_händen_auf_dem_kopf_helle_hautfarbe:',
        'es': u':persona_haciendo_el_gesto_de_de_acuerdo_tono_de_piel_claro:',
        'fr': u':personne_faisant_un_geste_d’acceptation_peau_claire:',
        'pt': u':pessoa_fazendo_gesto_de_“ok”_pele_clara:',
        'it': u':persona_con_gesto_ok_carnagione_chiara:'
    },
    u'\U0001F646\U0001F3FE': {
        'en' : u':person_gesturing_OK_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':person_mit_händen_auf_dem_kopf_mitteldunkle_hautfarbe:',
        'es': u':persona_haciendo_el_gesto_de_de_acuerdo_tono_de_piel_oscuro_medio:',
        'fr': u':personne_faisant_un_geste_d’acceptation_peau_mate:',
        'pt': u':pessoa_fazendo_gesto_de_“ok”_pele_morena_escura:',
        'it': u':persona_con_gesto_ok_carnagione_abbastanza_scura:'
    },
    u'\U0001F646\U0001F3FC': {
        'en' : u':person_gesturing_OK_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':person_mit_händen_auf_dem_kopf_mittelhelle_hautfarbe:',
        'es': u':persona_haciendo_el_gesto_de_de_acuerdo_tono_de_piel_claro_medio:',
        'fr': u':personne_faisant_un_geste_d’acceptation_peau_moyennement_claire:',
        'pt': u':pessoa_fazendo_gesto_de_“ok”_pele_morena_clara:',
        'it': u':persona_con_gesto_ok_carnagione_abbastanza_chiara:'
    },
    u'\U0001F646\U0001F3FD': {
        'en' : u':person_gesturing_OK_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':person_mit_händen_auf_dem_kopf_mittlere_hautfarbe:',
        'es': u':persona_haciendo_el_gesto_de_de_acuerdo_tono_de_piel_medio:',
        'fr': u':personne_faisant_un_geste_d’acceptation_peau_légèrement_mate:',
        'pt': u':pessoa_fazendo_gesto_de_“ok”_pele_morena:',
        'it': u':persona_con_gesto_ok_carnagione_olivastra:'
    },
    u'\U0001F487': {
        'en' : u':person_getting_haircut:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':haircut:'],
        'de': u':person_beim_haareschneiden:',
        'es': u':persona_cortándose_el_pelo:',
        'fr': u':personne_qui_se_fait_couper_les_cheveux:',
        'pt': u':pessoa_cortando_o_cabelo:',
        'it': u':taglio_di_capelli:'
    },
    u'\U0001F487\U0001F3FF': {
        'en' : u':person_getting_haircut_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':person_beim_haareschneiden_dunkle_hautfarbe:',
        'es': u':persona_cortándose_el_pelo_tono_de_piel_oscuro:',
        'fr': u':personne_qui_se_fait_couper_les_cheveux_peau_foncée:',
        'pt': u':pessoa_cortando_o_cabelo_pele_escura:',
        'it': u':taglio_di_capelli_carnagione_scura:'
    },
    u'\U0001F487\U0001F3FB': {
        'en' : u':person_getting_haircut_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':person_beim_haareschneiden_helle_hautfarbe:',
        'es': u':persona_cortándose_el_pelo_tono_de_piel_claro:',
        'fr': u':personne_qui_se_fait_couper_les_cheveux_peau_claire:',
        'pt': u':pessoa_cortando_o_cabelo_pele_clara:',
        'it': u':taglio_di_capelli_carnagione_chiara:'
    },
    u'\U0001F487\U0001F3FE': {
        'en' : u':person_getting_haircut_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':person_beim_haareschneiden_mitteldunkle_hautfarbe:',
        'es': u':persona_cortándose_el_pelo_tono_de_piel_oscuro_medio:',
        'fr': u':personne_qui_se_fait_couper_les_cheveux_peau_mate:',
        'pt': u':pessoa_cortando_o_cabelo_pele_morena_escura:',
        'it': u':taglio_di_capelli_carnagione_abbastanza_scura:'
    },
    u'\U0001F487\U0001F3FC': {
        'en' : u':person_getting_haircut_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':person_beim_haareschneiden_mittelhelle_hautfarbe:',
        'es': u':persona_cortándose_el_pelo_tono_de_piel_claro_medio:',
        'fr': u':personne_qui_se_fait_couper_les_cheveux_peau_moyennement_claire:',
        'pt': u':pessoa_cortando_o_cabelo_pele_morena_clara:',
        'it': u':taglio_di_capelli_carnagione_abbastanza_chiara:'
    },
    u'\U0001F487\U0001F3FD': {
        'en' : u':person_getting_haircut_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':person_beim_haareschneiden_mittlere_hautfarbe:',
        'es': u':persona_cortándose_el_pelo_tono_de_piel_medio:',
        'fr': u':personne_qui_se_fait_couper_les_cheveux_peau_légèrement_mate:',
        'pt': u':pessoa_cortando_o_cabelo_pele_morena:',
        'it': u':taglio_di_capelli_carnagione_olivastra:'
    },
    u'\U0001F486': {
        'en' : u':person_getting_massage:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [u':massage:'],
        'de': u':person,_die_eine_kopfmassage_bekommt:',
        'es': u':persona_recibiendo_masaje:',
        'fr': u':personne_qui_se_fait_masser:',
        'pt': u':pessoa_recebendo_massagem_facial:',
        'it': u':persona_che_riceve_un_massaggio:'
    },
    u'\U0001F486\U0001F3FF': {
        'en' : u':person_getting_massage_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':person,_die_eine_kopfmassage_bekommt_dunkle_hautfarbe:',
        'es': u':persona_recibiendo_masaje_tono_de_piel_oscuro:',
        'fr': u':personne_qui_se_fait_masser_peau_foncée:',
        'pt': u':pessoa_recebendo_massagem_facial_pele_escura:',
        'it': u':persona_che_riceve_un_massaggio_carnagione_scura:'
    },
    u'\U0001F486\U0001F3FB': {
        'en' : u':person_getting_massage_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':person,_die_eine_kopfmassage_bekommt_helle_hautfarbe:',
        'es': u':persona_recibiendo_masaje_tono_de_piel_claro:',
        'fr': u':personne_qui_se_fait_masser_peau_claire:',
        'pt': u':pessoa_recebendo_massagem_facial_pele_clara:',
        'it': u':persona_che_riceve_un_massaggio_carnagione_chiara:'
    },
    u'\U0001F486\U0001F3FE': {
        'en' : u':person_getting_massage_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':person,_die_eine_kopfmassage_bekommt_mitteldunkle_hautfarbe:',
        'es': u':persona_recibiendo_masaje_tono_de_piel_oscuro_medio:',
        'fr': u':personne_qui_se_fait_masser_peau_mate:',
        'pt': u':pessoa_recebendo_massagem_facial_pele_morena_escura:',
        'it': u':persona_che_riceve_un_massaggio_carnagione_abbastanza_scura:'
    },
    u'\U0001F486\U0001F3FC': {
        'en' : u':person_getting_massage_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':person,_die_eine_kopfmassage_bekommt_mittelhelle_hautfarbe:',
        'es': u':persona_recibiendo_masaje_tono_de_piel_claro_medio:',
        'fr': u':personne_qui_se_fait_masser_peau_moyennement_claire:',
        'pt': u':pessoa_recebendo_massagem_facial_pele_morena_clara:',
        'it': u':persona_che_riceve_un_massaggio_carnagione_abbastanza_chiara:'
    },
    u'\U0001F486\U0001F3FD': {
        'en' : u':person_getting_massage_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':person,_die_eine_kopfmassage_bekommt_mittlere_hautfarbe:',
        'es': u':persona_recibiendo_masaje_tono_de_piel_medio:',
        'fr': u':personne_qui_se_fait_masser_peau_légèrement_mate:',
        'pt': u':pessoa_recebendo_massagem_facial_pele_morena:',
        'it': u':persona_che_riceve_un_massaggio_carnagione_olivastra:'
    },
    u'\U0001F3CC\U0000FE0F': {
        'en' : u':person_golfing:',
        'status' : fully_qualified,
        'E' : 0.7,
        'alias' : [u':golfing:', u':golfer:'],
        'variant': True,
        'de': u':golfer(in):',
        'es': u':golfista:',
        'fr': u':joueur_de_golf:',
        'pt': u':golfista:',
        'it': u':persona_che_gioca_a_golf:'
    },
    u'\U0001F3CC': {
        'en' : u':person_golfing:',
        'status' : unqualified,
        'E' : 0.7,
        'alias' : [u':golfing:', u':golfer:'],
        'variant': True,
        'de': u':golfer(in):',
        'es': u':golfista:',
        'fr': u':joueur_de_golf:',
        'pt': u':golfista:',
        'it': u':persona_che_gioca_a_golf:'
    },
    u'\U0001F3CC\U0001F3FF': {
        'en' : u':person_golfing_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':golfer(in)_dunkle_hautfarbe:',
        'es': u':golfista_tono_de_piel_oscuro:',
        'fr': u':joueur_de_golf_peau_foncée:',
        'pt': u':golfista_pele_escura:',
        'it': u':persona_che_gioca_a_golf_carnagione_scura:'
    },
    u'\U0001F3CC\U0001F3FB': {
        'en' : u':person_golfing_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':golfer(in)_helle_hautfarbe:',
        'es': u':golfista_tono_de_piel_claro:',
        'fr': u':joueur_de_golf_peau_claire:',
        'pt': u':golfista_pele_clara:',
        'it': u':persona_che_gioca_a_golf_carnagione_chiara:'
    },
    u'\U0001F3CC\U0001F3FE': {
        'en' : u':person_golfing_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':golfer(in)_mitteldunkle_hautfarbe:',
        'es': u':golfista_tono_de_piel_oscuro_medio:',
        'fr': u':joueur_de_golf_peau_mate:',
        'pt': u':golfista_pele_morena_escura:',
        'it': u':persona_che_gioca_a_golf_carnagione_abbastanza_scura:'
    },
    u'\U0001F3CC\U0001F3FC': {
        'en' : u':person_golfing_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':golfer(in)_mittelhelle_hautfarbe:',
        'es': u':golfista_tono_de_piel_claro_medio:',
        'fr': u':joueur_de_golf_peau_moyennement_claire:',
        'pt': u':golfista_pele_morena_clara:',
        'it': u':persona_che_gioca_a_golf_carnagione_abbastanza_chiara:'
    },
    u'\U0001F3CC\U0001F3FD': {
        'en' : u':person_golfing_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':golfer(in)_mittlere_hautfarbe:',
        'es': u':golfista_tono_de_piel_medio:',
        'fr': u':joueur_de_golf_peau_légèrement_mate:',
        'pt': u':golfista_pele_morena:',
        'it': u':persona_che_gioca_a_golf_carnagione_olivastra:'
    },
    u'\U0001F6CC': {
        'en' : u':person_in_bed:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [u':sleeping_accommodation:', u':sleeping_bed:'],
        'de': u':im_bett_liegende_person:',
        'es': u':persona_en_la_cama:',
        'fr': u':personne_au_lit:',
        'pt': u':pessoa_deitada_na_cama:',
        'it': u':persona_a_letto:'
    },
    u'\U0001F6CC\U0001F3FF': {
        'en' : u':person_in_bed_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':im_bett_liegende_person_dunkle_hautfarbe:',
        'es': u':persona_en_la_cama_tono_de_piel_oscuro:',
        'fr': u':personne_au_lit_peau_foncée:',
        'pt': u':pessoa_deitada_na_cama_pele_escura:',
        'it': u':persona_a_letto_carnagione_scura:'
    },
    u'\U0001F6CC\U0001F3FB': {
        'en' : u':person_in_bed_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':im_bett_liegende_person_helle_hautfarbe:',
        'es': u':persona_en_la_cama_tono_de_piel_claro:',
        'fr': u':personne_au_lit_peau_claire:',
        'pt': u':pessoa_deitada_na_cama_pele_clara:',
        'it': u':persona_a_letto_carnagione_chiara:'
    },
    u'\U0001F6CC\U0001F3FE': {
        'en' : u':person_in_bed_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':im_bett_liegende_person_mitteldunkle_hautfarbe:',
        'es': u':persona_en_la_cama_tono_de_piel_oscuro_medio:',
        'fr': u':personne_au_lit_peau_mate:',
        'pt': u':pessoa_deitada_na_cama_pele_morena_escura:',
        'it': u':persona_a_letto_carnagione_abbastanza_scura:'
    },
    u'\U0001F6CC\U0001F3FC': {
        'en' : u':person_in_bed_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':im_bett_liegende_person_mittelhelle_hautfarbe:',
        'es': u':persona_en_la_cama_tono_de_piel_claro_medio:',
        'fr': u':personne_au_lit_peau_moyennement_claire:',
        'pt': u':pessoa_deitada_na_cama_pele_morena_clara:',
        'it': u':persona_a_letto_carnagione_abbastanza_chiara:'
    },
    u'\U0001F6CC\U0001F3FD': {
        'en' : u':person_in_bed_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':im_bett_liegende_person_mittlere_hautfarbe:',
        'es': u':persona_en_la_cama_tono_de_piel_medio:',
        'fr': u':personne_au_lit_peau_légèrement_mate:',
        'pt': u':pessoa_deitada_na_cama_pele_morena:',
        'it': u':persona_a_letto_carnagione_olivastra:'
    },
    u'\U0001F9D8': {
        'en' : u':person_in_lotus_position:',
        'status' : fully_qualified,
        'E' : 5,
        'alias' : [u':lotus_position:'],
        'de': u':person_im_lotossitz:',
        'es': u':persona_en_posición_de_loto:',
        'fr': u':personne_dans_la_posture_du_lotus:',
        'pt': u':pessoa_na_posição_de_lótus:',
        'it': u':persona_in_posizione_del_loto:'
    },
    u'\U0001F9D8\U0001F3FF': {
        'en' : u':person_in_lotus_position_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':person_im_lotossitz_dunkle_hautfarbe:',
        'es': u':persona_en_posición_de_loto_tono_de_piel_oscuro:',
        'fr': u':personne_dans_la_posture_du_lotus_peau_foncée:',
        'pt': u':pessoa_na_posição_de_lótus_pele_escura:',
        'it': u':persona_in_posizione_del_loto_carnagione_scura:'
    },
    u'\U0001F9D8\U0001F3FB': {
        'en' : u':person_in_lotus_position_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':person_im_lotossitz_helle_hautfarbe:',
        'es': u':persona_en_posición_de_loto_tono_de_piel_claro:',
        'fr': u':personne_dans_la_posture_du_lotus_peau_claire:',
        'pt': u':pessoa_na_posição_de_lótus_pele_clara:',
        'it': u':persona_in_posizione_del_loto_carnagione_chiara:'
    },
    u'\U0001F9D8\U0001F3FE': {
        'en' : u':person_in_lotus_position_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':person_im_lotossitz_mitteldunkle_hautfarbe:',
        'es': u':persona_en_posición_de_loto_tono_de_piel_oscuro_medio:',
        'fr': u':personne_dans_la_posture_du_lotus_peau_mate:',
        'pt': u':pessoa_na_posição_de_lótus_pele_morena_escura:',
        'it': u':persona_in_posizione_del_loto_carnagione_abbastanza_scura:'
    },
    u'\U0001F9D8\U0001F3FC': {
        'en' : u':person_in_lotus_position_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':person_im_lotossitz_mittelhelle_hautfarbe:',
        'es': u':persona_en_posición_de_loto_tono_de_piel_claro_medio:',
        'fr': u':personne_dans_la_posture_du_lotus_peau_moyennement_claire:',
        'pt': u':pessoa_na_posição_de_lótus_pele_morena_clara:',
        'it': u':persona_in_posizione_del_loto_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9D8\U0001F3FD': {
        'en' : u':person_in_lotus_position_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':person_im_lotossitz_mittlere_hautfarbe:',
        'es': u':persona_en_posición_de_loto_tono_de_piel_medio:',
        'fr': u':personne_dans_la_posture_du_lotus_peau_légèrement_mate:',
        'pt': u':pessoa_na_posição_de_lótus_pele_morena:',
        'it': u':persona_in_posizione_del_loto_carnagione_olivastra:'
    },
    u'\U0001F9D1\U0000200D\U0001F9BD': {
        'en' : u':person_in_manual_wheelchair:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':person_in_manuellem_rollstuhl:',
        'es': u':persona_en_silla_de_ruedas_manual:',
        'fr': u':personne_en_fauteuil_roulant_manuel:',
        'pt': u':pessoa_em_cadeira_de_rodas_manual:',
        'it': u':persona_su_sedia_a_rotelle_manuale:'
    },
    u'\U0001F9D1\U0001F3FF\U0000200D\U0001F9BD': {
        'en' : u':person_in_manual_wheelchair_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':person_in_manuellem_rollstuhl_dunkle_hautfarbe:',
        'es': u':persona_en_silla_de_ruedas_manual_tono_de_piel_oscuro:',
        'fr': u':personne_en_fauteuil_roulant_manuel_peau_foncée:',
        'pt': u':pessoa_em_cadeira_de_rodas_manual_pele_escura:',
        'it': u':persona_su_sedia_a_rotelle_manuale_carnagione_scura:'
    },
    u'\U0001F9D1\U0001F3FB\U0000200D\U0001F9BD': {
        'en' : u':person_in_manual_wheelchair_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':person_in_manuellem_rollstuhl_helle_hautfarbe:',
        'es': u':persona_en_silla_de_ruedas_manual_tono_de_piel_claro:',
        'fr': u':personne_en_fauteuil_roulant_manuel_peau_claire:',
        'pt': u':pessoa_em_cadeira_de_rodas_manual_pele_clara:',
        'it': u':persona_su_sedia_a_rotelle_manuale_carnagione_chiara:'
    },
    u'\U0001F9D1\U0001F3FE\U0000200D\U0001F9BD': {
        'en' : u':person_in_manual_wheelchair_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':person_in_manuellem_rollstuhl_mitteldunkle_hautfarbe:',
        'es': u':persona_en_silla_de_ruedas_manual_tono_de_piel_oscuro_medio:',
        'fr': u':personne_en_fauteuil_roulant_manuel_peau_mate:',
        'pt': u':pessoa_em_cadeira_de_rodas_manual_pele_morena_escura:',
        'it': u':persona_su_sedia_a_rotelle_manuale_carnagione_abbastanza_scura:'
    },
    u'\U0001F9D1\U0001F3FC\U0000200D\U0001F9BD': {
        'en' : u':person_in_manual_wheelchair_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':person_in_manuellem_rollstuhl_mittelhelle_hautfarbe:',
        'es': u':persona_en_silla_de_ruedas_manual_tono_de_piel_claro_medio:',
        'fr': u':personne_en_fauteuil_roulant_manuel_peau_moyennement_claire:',
        'pt': u':pessoa_em_cadeira_de_rodas_manual_pele_morena_clara:',
        'it': u':persona_su_sedia_a_rotelle_manuale_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9D1\U0001F3FD\U0000200D\U0001F9BD': {
        'en' : u':person_in_manual_wheelchair_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':person_in_manuellem_rollstuhl_mittlere_hautfarbe:',
        'es': u':persona_en_silla_de_ruedas_manual_tono_de_piel_medio:',
        'fr': u':personne_en_fauteuil_roulant_manuel_peau_légèrement_mate:',
        'pt': u':pessoa_em_cadeira_de_rodas_manual_pele_morena:',
        'it': u':persona_su_sedia_a_rotelle_manuale_carnagione_olivastra:'
    },
    u'\U0001F9D1\U0000200D\U0001F9BC': {
        'en' : u':person_in_motorized_wheelchair:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':person_in_motorisiertem_rollstuhl:',
        'es': u':persona_en_silla_de_ruedas_eléctrica:',
        'fr': u':personne_en_fauteuil_motorisé:',
        'pt': u':pessoa_em_cadeira_de_rodas_motorizada:',
        'it': u':persona_su_sedia_a_rotelle_motorizzata:'
    },
    u'\U0001F9D1\U0001F3FF\U0000200D\U0001F9BC': {
        'en' : u':person_in_motorized_wheelchair_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':person_in_motorisiertem_rollstuhl_dunkle_hautfarbe:',
        'es': u':persona_en_silla_de_ruedas_eléctrica_tono_de_piel_oscuro:',
        'fr': u':personne_en_fauteuil_motorisé_peau_foncée:',
        'pt': u':pessoa_em_cadeira_de_rodas_motorizada_pele_escura:',
        'it': u':persona_su_sedia_a_rotelle_motorizzata_carnagione_scura:'
    },
    u'\U0001F9D1\U0001F3FB\U0000200D\U0001F9BC': {
        'en' : u':person_in_motorized_wheelchair_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':person_in_motorisiertem_rollstuhl_helle_hautfarbe:',
        'es': u':persona_en_silla_de_ruedas_eléctrica_tono_de_piel_claro:',
        'fr': u':personne_en_fauteuil_motorisé_peau_claire:',
        'pt': u':pessoa_em_cadeira_de_rodas_motorizada_pele_clara:',
        'it': u':persona_su_sedia_a_rotelle_motorizzata_carnagione_chiara:'
    },
    u'\U0001F9D1\U0001F3FE\U0000200D\U0001F9BC': {
        'en' : u':person_in_motorized_wheelchair_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':person_in_motorisiertem_rollstuhl_mitteldunkle_hautfarbe:',
        'es': u':persona_en_silla_de_ruedas_eléctrica_tono_de_piel_oscuro_medio:',
        'fr': u':personne_en_fauteuil_motorisé_peau_mate:',
        'pt': u':pessoa_em_cadeira_de_rodas_motorizada_pele_morena_escura:',
        'it': u':persona_su_sedia_a_rotelle_motorizzata_carnagione_abbastanza_scura:'
    },
    u'\U0001F9D1\U0001F3FC\U0000200D\U0001F9BC': {
        'en' : u':person_in_motorized_wheelchair_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':person_in_motorisiertem_rollstuhl_mittelhelle_hautfarbe:',
        'es': u':persona_en_silla_de_ruedas_eléctrica_tono_de_piel_claro_medio:',
        'fr': u':personne_en_fauteuil_motorisé_peau_moyennement_claire:',
        'pt': u':pessoa_em_cadeira_de_rodas_motorizada_pele_morena_clara:',
        'it': u':persona_su_sedia_a_rotelle_motorizzata_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9D1\U0001F3FD\U0000200D\U0001F9BC': {
        'en' : u':person_in_motorized_wheelchair_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':person_in_motorisiertem_rollstuhl_mittlere_hautfarbe:',
        'es': u':persona_en_silla_de_ruedas_eléctrica_tono_de_piel_medio:',
        'fr': u':personne_en_fauteuil_motorisé_peau_légèrement_mate:',
        'pt': u':pessoa_em_cadeira_de_rodas_motorizada_pele_morena:',
        'it': u':persona_su_sedia_a_rotelle_motorizzata_carnagione_olivastra:'
    },
    u'\U0001F9D6': {
        'en' : u':person_in_steamy_room:',
        'status' : fully_qualified,
        'E' : 5,
        'alias' : [u':sauna_person:'],
        'de': u':person_in_dampfsauna:',
        'es': u':persona_en_una_sauna:',
        'fr': u':personne_au_hammam:',
        'pt': u':pessoa_na_sauna:',
        'it': u':persona_in_sauna:'
    },
    u'\U0001F9D6\U0001F3FF': {
        'en' : u':person_in_steamy_room_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':person_in_dampfsauna_dunkle_hautfarbe:',
        'es': u':persona_en_una_sauna_tono_de_piel_oscuro:',
        'fr': u':personne_au_hammam_peau_foncée:',
        'pt': u':pessoa_na_sauna_pele_escura:',
        'it': u':persona_in_sauna_carnagione_scura:'
    },
    u'\U0001F9D6\U0001F3FB': {
        'en' : u':person_in_steamy_room_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':person_in_dampfsauna_helle_hautfarbe:',
        'es': u':persona_en_una_sauna_tono_de_piel_claro:',
        'fr': u':personne_au_hammam_peau_claire:',
        'pt': u':pessoa_na_sauna_pele_clara:',
        'it': u':persona_in_sauna_carnagione_chiara:'
    },
    u'\U0001F9D6\U0001F3FE': {
        'en' : u':person_in_steamy_room_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':person_in_dampfsauna_mitteldunkle_hautfarbe:',
        'es': u':persona_en_una_sauna_tono_de_piel_oscuro_medio:',
        'fr': u':personne_au_hammam_peau_mate:',
        'pt': u':pessoa_na_sauna_pele_morena_escura:',
        'it': u':persona_in_sauna_carnagione_abbastanza_scura:'
    },
    u'\U0001F9D6\U0001F3FC': {
        'en' : u':person_in_steamy_room_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':person_in_dampfsauna_mittelhelle_hautfarbe:',
        'es': u':persona_en_una_sauna_tono_de_piel_claro_medio:',
        'fr': u':personne_au_hammam_peau_moyennement_claire:',
        'pt': u':pessoa_na_sauna_pele_morena_clara:',
        'it': u':persona_in_sauna_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9D6\U0001F3FD': {
        'en' : u':person_in_steamy_room_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':person_in_dampfsauna_mittlere_hautfarbe:',
        'es': u':persona_en_una_sauna_tono_de_piel_medio:',
        'fr': u':personne_au_hammam_peau_légèrement_mate:',
        'pt': u':pessoa_na_sauna_pele_morena:',
        'it': u':persona_in_sauna_carnagione_olivastra:'
    },
    u'\U0001F574\U0000FE0F': {
        'en' : u':person_in_suit_levitating:',
        'status' : fully_qualified,
        'E' : 0.7,
        'alias' : [u':man_in_business_suit_levitating:', u':business_suit_levitating:'],
        'variant': True,
        'de': u':schwebender_mann_im_anzug:',
        'es': u':hombre_trajeado_levitando:',
        'fr': u':homme_d’affaires_en_lévitation:',
        'pt': u':homem_de_terno_levitando:',
        'it': u':uomo_con_completo_che_levita:'
    },
    u'\U0001F574': {
        'en' : u':person_in_suit_levitating:',
        'status' : unqualified,
        'E' : 0.7,
        'alias' : [u':man_in_business_suit_levitating:', u':business_suit_levitating:'],
        'variant': True,
        'de': u':schwebender_mann_im_anzug:',
        'es': u':hombre_trajeado_levitando:',
        'fr': u':homme_d’affaires_en_lévitation:',
        'pt': u':homem_de_terno_levitando:',
        'it': u':uomo_con_completo_che_levita:'
    },
    u'\U0001F574\U0001F3FF': {
        'en' : u':person_in_suit_levitating_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':schwebender_mann_im_anzug_dunkle_hautfarbe:',
        'es': u':hombre_trajeado_levitando_tono_de_piel_oscuro:',
        'fr': u':homme_d’affaires_en_lévitation_peau_foncée:',
        'pt': u':homem_de_terno_levitando_pele_escura:',
        'it': u':uomo_con_completo_che_levita_carnagione_scura:'
    },
    u'\U0001F574\U0001F3FB': {
        'en' : u':person_in_suit_levitating_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':schwebender_mann_im_anzug_helle_hautfarbe:',
        'es': u':hombre_trajeado_levitando_tono_de_piel_claro:',
        'fr': u':homme_d’affaires_en_lévitation_peau_claire:',
        'pt': u':homem_de_terno_levitando_pele_clara:',
        'it': u':uomo_con_completo_che_levita_carnagione_chiara:'
    },
    u'\U0001F574\U0001F3FE': {
        'en' : u':person_in_suit_levitating_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':schwebender_mann_im_anzug_mitteldunkle_hautfarbe:',
        'es': u':hombre_trajeado_levitando_tono_de_piel_oscuro_medio:',
        'fr': u':homme_d’affaires_en_lévitation_peau_mate:',
        'pt': u':homem_de_terno_levitando_pele_morena_escura:',
        'it': u':uomo_con_completo_che_levita_carnagione_abbastanza_scura:'
    },
    u'\U0001F574\U0001F3FC': {
        'en' : u':person_in_suit_levitating_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':schwebender_mann_im_anzug_mittelhelle_hautfarbe:',
        'es': u':hombre_trajeado_levitando_tono_de_piel_claro_medio:',
        'fr': u':homme_d’affaires_en_lévitation_peau_moyennement_claire:',
        'pt': u':homem_de_terno_levitando_pele_morena_clara:',
        'it': u':uomo_con_completo_che_levita_carnagione_abbastanza_chiara:'
    },
    u'\U0001F574\U0001F3FD': {
        'en' : u':person_in_suit_levitating_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':schwebender_mann_im_anzug_mittlere_hautfarbe:',
        'es': u':hombre_trajeado_levitando_tono_de_piel_medio:',
        'fr': u':homme_d’affaires_en_lévitation_peau_légèrement_mate:',
        'pt': u':homem_de_terno_levitando_pele_morena:',
        'it': u':uomo_con_completo_che_levita_carnagione_olivastra:'
    },
    u'\U0001F935': {
        'en' : u':person_in_tuxedo:',
        'status' : fully_qualified,
        'E' : 3,
        'de': u':person_im_smoking:',
        'es': u':persona_con_esmoquin:',
        'fr': u':personne_en_smoking:',
        'pt': u':pessoa_de_smoking:',
        'it': u':persona_in_smoking:'
    },
    u'\U0001F935\U0001F3FF': {
        'en' : u':person_in_tuxedo_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 3,
        'de': u':person_im_smoking_dunkle_hautfarbe:',
        'es': u':persona_con_esmoquin_tono_de_piel_oscuro:',
        'fr': u':personne_en_smoking_peau_foncée:',
        'pt': u':pessoa_de_smoking_pele_escura:',
        'it': u':persona_in_smoking_carnagione_scura:'
    },
    u'\U0001F935\U0001F3FB': {
        'en' : u':person_in_tuxedo_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 3,
        'de': u':person_im_smoking_helle_hautfarbe:',
        'es': u':persona_con_esmoquin_tono_de_piel_claro:',
        'fr': u':personne_en_smoking_peau_claire:',
        'pt': u':pessoa_de_smoking_pele_clara:',
        'it': u':persona_in_smoking_carnagione_chiara:'
    },
    u'\U0001F935\U0001F3FE': {
        'en' : u':person_in_tuxedo_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 3,
        'de': u':person_im_smoking_mitteldunkle_hautfarbe:',
        'es': u':persona_con_esmoquin_tono_de_piel_oscuro_medio:',
        'fr': u':personne_en_smoking_peau_mate:',
        'pt': u':pessoa_de_smoking_pele_morena_escura:',
        'it': u':persona_in_smoking_carnagione_abbastanza_scura:'
    },
    u'\U0001F935\U0001F3FC': {
        'en' : u':person_in_tuxedo_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 3,
        'de': u':person_im_smoking_mittelhelle_hautfarbe:',
        'es': u':persona_con_esmoquin_tono_de_piel_claro_medio:',
        'fr': u':personne_en_smoking_peau_moyennement_claire:',
        'pt': u':pessoa_de_smoking_pele_morena_clara:',
        'it': u':persona_in_smoking_carnagione_abbastanza_chiara:'
    },
    u'\U0001F935\U0001F3FD': {
        'en' : u':person_in_tuxedo_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 3,
        'de': u':person_im_smoking_mittlere_hautfarbe:',
        'es': u':persona_con_esmoquin_tono_de_piel_medio:',
        'fr': u':personne_en_smoking_peau_légèrement_mate:',
        'pt': u':pessoa_de_smoking_pele_morena:',
        'it': u':persona_in_smoking_carnagione_olivastra:'
    },
    u'\U0001F939': {
        'en' : u':person_juggling:',
        'status' : fully_qualified,
        'E' : 3,
        'alias' : [u':juggling_person:'],
        'de': u':jongleur(in):',
        'es': u':persona_haciendo_malabares:',
        'fr': u':personne_qui_jongle:',
        'pt': u':malabarista:',
        'it': u':persona_che_fa_giocoleria:'
    },
    u'\U0001F939\U0001F3FF': {
        'en' : u':person_juggling_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 3,
        'de': u':jongleur(in)_dunkle_hautfarbe:',
        'es': u':persona_haciendo_malabares_tono_de_piel_oscuro:',
        'fr': u':personne_qui_jongle_peau_foncée:',
        'pt': u':malabarista_pele_escura:',
        'it': u':persona_che_fa_giocoleria_carnagione_scura:'
    },
    u'\U0001F939\U0001F3FB': {
        'en' : u':person_juggling_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 3,
        'de': u':jongleur(in)_helle_hautfarbe:',
        'es': u':persona_haciendo_malabares_tono_de_piel_claro:',
        'fr': u':personne_qui_jongle_peau_claire:',
        'pt': u':malabarista_pele_clara:',
        'it': u':persona_che_fa_giocoleria_carnagione_ch