"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.AuthenticationType = void 0;

var _opensearch_security_client = require("../../backend/opensearch_security_client");

var _tenant_resolver = require("../../multitenancy/tenant_resolver");

var _errors = require("../../errors");

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

class AuthenticationType {
  constructor(config, sessionStorageFactory, router, esClient, coreSetup, logger) {
    this.config = config;
    this.sessionStorageFactory = sessionStorageFactory;
    this.router = router;
    this.esClient = esClient;
    this.coreSetup = coreSetup;
    this.logger = logger;

    _defineProperty(this, "type", void 0);

    _defineProperty(this, "securityClient", void 0);

    _defineProperty(this, "authHandler", async (request, response, toolkit) => {
      var _this$config$multiten;

      // skip auth for APIs that do not require auth
      if (this.authNotRequired(request)) {
        return toolkit.authenticated();
      } // if browser request, auth logic is:
      //   1. check if request includes auth header or paramter(e.g. jwt in url params) is present, if so, authenticate with auth header.
      //   2. if auth header not present, check if auth cookie is present, if no cookie, send to authentication workflow
      //   3. verify whether auth cookie is valid, if not valid, send to authentication workflow
      //   4. if cookie is valid, pass to route handlers


      const authHeaders = {};
      let cookie;
      let authInfo; // if this is an REST API call, suppose the request includes necessary auth header
      // see https://www.elastic.co/guide/en/opensearch-dashboards/master/using-api.html

      if (this.requestIncludesAuthInfo(request)) {
        try {
          const additonalAuthHeader = this.getAdditionalAuthHeader(request);
          Object.assign(authHeaders, additonalAuthHeader);
          authInfo = await this.securityClient.authinfo(request, additonalAuthHeader);
          cookie = await this.getCookie(request, authInfo); // set tenant from cookie if exist

          const browserCookie = await this.sessionStorageFactory.asScoped(request).get();

          if (browserCookie && (0, _tenant_resolver.isValidTenant)(browserCookie.tenant)) {
            cookie.tenant = browserCookie.tenant;
          }

          this.sessionStorageFactory.asScoped(request).set(cookie);
        } catch (error) {
          return response.unauthorized({
            body: error.message
          });
        }
      } else {
        // no auth header in request, try cookie
        try {
          cookie = await this.sessionStorageFactory.asScoped(request).get();
        } catch (error) {
          this.logger.error(`Error parsing cookie: ${error.message}`);
          cookie = undefined;
        }

        if (!cookie || !(await this.isValidCookie(cookie))) {
          // clear cookie
          this.sessionStorageFactory.asScoped(request).clear(); // for assets, we can still pass it to resource handler as notHandled.
          // marking it as authenticated may result in login pop up when auth challenge
          // is enabled.

          if (request.url.pathname && request.url.pathname.startsWith('/bundles/')) {
            return toolkit.notHandled();
          } // send to auth workflow


          return this.handleUnauthedRequest(request, response, toolkit);
        } // extend session expiration time


        if (this.config.session.keepalive) {
          cookie.expiryTime = Date.now() + this.config.session.ttl;
          this.sessionStorageFactory.asScoped(request).set(cookie);
        } // cookie is valid
        // build auth header


        const authHeadersFromCookie = this.buildAuthHeaderFromCookie(cookie);
        Object.assign(authHeaders, authHeadersFromCookie);
        const additonalAuthHeader = this.getAdditionalAuthHeader(request);
        Object.assign(authHeaders, additonalAuthHeader);
      } // resolve tenant if necessary


      if ((_this$config$multiten = this.config.multitenancy) !== null && _this$config$multiten !== void 0 && _this$config$multiten.enabled && (0, _tenant_resolver.isMultitenantPath)(request)) {
        try {
          const tenant = await this.resolveTenant(request, cookie, authHeaders, authInfo); // return 401 if no tenant available

          if (!(0, _tenant_resolver.isValidTenant)(tenant)) {
            return response.badRequest({
              body: 'No available tenant for current user, please reach out to your system administrator'
            });
          } // set tenant in header


          Object.assign(authHeaders, {
            securitytenant: tenant
          }); // set tenant to cookie

          if (tenant !== cookie.tenant) {
            cookie.tenant = tenant;
            this.sessionStorageFactory.asScoped(request).set(cookie);
          }
        } catch (error) {
          this.logger.error(`Failed to resolve user tenant: ${error}`);

          if (error instanceof _errors.UnauthenticatedError) {
            if (request.url.pathname && request.url.pathname.startsWith('/bundles/')) {
              return toolkit.notHandled();
            }

            this.sessionStorageFactory.asScoped(request).clear();
            return this.handleUnauthedRequest(request, response, toolkit);
          }

          throw error;
        }
      }

      return toolkit.authenticated({
        requestHeaders: authHeaders
      });
    });

    this.securityClient = new _opensearch_security_client.SecurityClient(esClient);
    this.type = '';
  }

  authNotRequired(request) {
    const pathname = request.url.pathname;

    if (!pathname) {
      return false;
    } // allow requests to ignored routes


    if (AuthenticationType.ROUTES_TO_IGNORE.includes(pathname)) {
      return true;
    } // allow requests to routes that doesn't require authentication


    if (this.config.auth.unauthenticated_routes.indexOf(pathname) > -1) {
      // TODO: use opensearch-dashboards server user
      return true;
    }

    return false;
  }

  async resolveTenant(request, cookie, authHeader, authInfo) {
    if (!authInfo) {
      try {
        authInfo = await this.securityClient.authinfo(request, authHeader);
      } catch (error) {
        throw new _errors.UnauthenticatedError(error);
      }
    }

    const selectedTenant = (0, _tenant_resolver.resolveTenant)(request, authInfo.user_name, authInfo.roles, authInfo.tenants, this.config, cookie);
    return selectedTenant;
  }

  isPageRequest(request) {
    const path = request.url.pathname || '/';
    return path.startsWith('/app/') || path === '/' || path.startsWith('/goto/');
  } // abstract functions for concrete auth types to implement


}

exports.AuthenticationType = AuthenticationType;

_defineProperty(AuthenticationType, "ROUTES_TO_IGNORE", ['/api/core/capabilities', // FIXME: need to figureout how to bypass this API call
'/app/login']);

_defineProperty(AuthenticationType, "REST_API_CALL_HEADER", 'osd-xsrf');
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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