"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.callTokenEndpoint = callTokenEndpoint;
exports.composeLogoutUrl = composeLogoutUrl;
exports.getBaseRedirectUrl = getBaseRedirectUrl;
exports.getRootUrl = getRootUrl;
exports.parseTokenResponse = parseTokenResponse;

var _querystring = require("querystring");

/*
 *   Copyright OpenSearch Contributors
 *
 *   Licensed under the Apache License, Version 2.0 (the "License").
 *   You may not use this file except in compliance with the License.
 *   A copy of the License is located at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   or in the "license" file accompanying this file. This file is distributed
 *   on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 *   express or implied. See the License for the specific language governing
 *   permissions and limitations under the License.
 */
function parseTokenResponse(payload) {
  const payloadString = payload.toString();

  if (payloadString.trim()[0] === '{') {
    try {
      return JSON.parse(payloadString);
    } catch (error) {
      throw Error(`Invalid JSON payload: ${error}`);
    }
  }

  return (0, _querystring.parse)(payloadString);
}

function getRootUrl(config, core, request) {
  var _config$openid;

  const host = core.http.getServerInfo().hostname;
  const port = core.http.getServerInfo().port;
  let protocol = core.http.getServerInfo().protocol;
  let httpHost = `${host}:${port}`;

  if ((_config$openid = config.openid) !== null && _config$openid !== void 0 && _config$openid.trust_dynamic_headers) {
    const xForwardedHost = request.headers['x-forwarded-host'] || undefined;
    const xForwardedProto = request.headers['x-forwarded-proto'] || undefined;

    if (xForwardedHost) {
      httpHost = xForwardedHost;
    }

    if (xForwardedProto) {
      protocol = xForwardedProto;
    }
  }

  return `${protocol}://${httpHost}`;
}

function getBaseRedirectUrl(config, core, request) {
  var _config$openid2;

  if ((_config$openid2 = config.openid) !== null && _config$openid2 !== void 0 && _config$openid2.base_redirect_url) {
    const baseRedirectUrl = config.openid.base_redirect_url;
    return baseRedirectUrl.endsWith('/') ? baseRedirectUrl.slice(0, -1) : baseRedirectUrl;
  }

  const rootUrl = getRootUrl(config, core, request);

  if (core.http.basePath.serverBasePath) {
    return `${rootUrl}${core.http.basePath.serverBasePath}`;
  }

  return rootUrl;
}

async function callTokenEndpoint(tokenEndpoint, query, wreckClient) {
  var _tokenResponse$res;

  const tokenResponse = await wreckClient.post(tokenEndpoint, {
    payload: (0, _querystring.stringify)(query),
    headers: {
      'Content-Type': 'application/x-www-form-urlencoded'
    }
  });

  if (!((_tokenResponse$res = tokenResponse.res) !== null && _tokenResponse$res !== void 0 && _tokenResponse$res.statusCode) || tokenResponse.res.statusCode < 200 || tokenResponse.res.statusCode > 299) {
    throw new Error(`Failed calling token endpoint: ${tokenResponse.res.statusCode} ${tokenResponse.res.statusMessage}`);
  }

  const tokenPayload = parseTokenResponse(tokenResponse.payload);
  return {
    idToken: tokenPayload.id_token,
    accessToken: tokenPayload.access_token,
    refreshToken: tokenPayload.refresh_token,
    expiresIn: tokenPayload.expires_in
  };
}

function composeLogoutUrl(customLogoutUrl, idpEndsessionEndpoint, additionalQueryParams) {
  const logoutEndpont = customLogoutUrl || idpEndsessionEndpoint;
  const logoutUrl = new URL(logoutEndpont);
  Object.keys(additionalQueryParams).forEach(key => {
    logoutUrl.searchParams.append(key, additionalQueryParams[key]);
  });
  return logoutUrl.toString();
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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