"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.OpenIdAuthentication = void 0;

var fs = _interopRequireWildcard(require("fs"));

var _wreck = _interopRequireDefault(require("@hapi/wreck"));

var _http = _interopRequireDefault(require("http"));

var _https = _interopRequireDefault(require("https"));

var _routes = require("./routes");

var _authentication_type = require("../authentication_type");

var _helper = require("./helper");

var _next_url = require("../../../utils/next_url");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _getRequireWildcardCache(nodeInterop) { if (typeof WeakMap !== "function") return null; var cacheBabelInterop = new WeakMap(); var cacheNodeInterop = new WeakMap(); return (_getRequireWildcardCache = function (nodeInterop) { return nodeInterop ? cacheNodeInterop : cacheBabelInterop; })(nodeInterop); }

function _interopRequireWildcard(obj, nodeInterop) { if (!nodeInterop && obj && obj.__esModule) { return obj; } if (obj === null || typeof obj !== "object" && typeof obj !== "function") { return { default: obj }; } var cache = _getRequireWildcardCache(nodeInterop); if (cache && cache.has(obj)) { return cache.get(obj); } var newObj = {}; var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var key in obj) { if (key !== "default" && Object.prototype.hasOwnProperty.call(obj, key)) { var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null; if (desc && (desc.get || desc.set)) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } newObj.default = obj; if (cache) { cache.set(obj, newObj); } return newObj; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

class OpenIdAuthentication extends _authentication_type.AuthenticationType {
  constructor(config, sessionStorageFactory, router, esClient, core, logger) {
    var _this$config$openid, _this$config$openid2;

    super(config, sessionStorageFactory, router, esClient, core, logger);

    _defineProperty(this, "type", 'openid');

    _defineProperty(this, "openIdAuthConfig", void 0);

    _defineProperty(this, "authHeaderName", void 0);

    _defineProperty(this, "openIdConnectUrl", void 0);

    _defineProperty(this, "wreckClient", void 0);

    this.wreckClient = this.createWreckClient();
    this.openIdAuthConfig = {};
    this.authHeaderName = ((_this$config$openid = this.config.openid) === null || _this$config$openid === void 0 ? void 0 : _this$config$openid.header) || '';
    this.openIdAuthConfig.authHeaderName = this.authHeaderName;
    this.openIdConnectUrl = ((_this$config$openid2 = this.config.openid) === null || _this$config$openid2 === void 0 ? void 0 : _this$config$openid2.connect_url) || '';
    let scope = this.config.openid.scope;

    if (scope.indexOf('openid') < 0) {
      scope = `openid ${scope}`;
    }

    this.openIdAuthConfig.scope = scope;
    this.init();
  }

  async init() {
    try {
      const response = await this.wreckClient.get(this.openIdConnectUrl);
      const payload = JSON.parse(response.payload);
      this.openIdAuthConfig.authorizationEndpoint = payload.authorization_endpoint;
      this.openIdAuthConfig.tokenEndpoint = payload.token_endpoint;
      this.openIdAuthConfig.endSessionEndpoint = payload.end_session_endpoint || undefined;
      const routes = new _routes.OpenIdAuthRoutes(this.router, this.config, this.sessionStorageFactory, this.openIdAuthConfig, this.securityClient, this.coreSetup, this.wreckClient);
      routes.setupRoutes();
    } catch (error) {
      this.logger.error(error); // TODO: log more info

      throw new Error('Failed when trying to obtain the endpoints from your IdP');
    }
  }

  createWreckClient() {
    var _this$config$openid3, _this$config$openid4;

    const wreckHttpsOption = {};

    if ((_this$config$openid3 = this.config.openid) !== null && _this$config$openid3 !== void 0 && _this$config$openid3.root_ca) {
      wreckHttpsOption.ca = [fs.readFileSync(this.config.openid.root_ca)];
    }

    if (((_this$config$openid4 = this.config.openid) === null || _this$config$openid4 === void 0 ? void 0 : _this$config$openid4.verify_hostnames) === false) {
      this.logger.debug(`openId auth 'verify_hostnames' option is off.`);

      wreckHttpsOption.checkServerIdentity = (host, cert) => {
        return undefined;
      };
    }

    if (Object.keys(wreckHttpsOption).length > 0) {
      return _wreck.default.defaults({
        agents: {
          http: new _http.default.Agent(),
          https: new _https.default.Agent(wreckHttpsOption),
          httpsAllowUnauthorized: new _https.default.Agent({
            rejectUnauthorized: false
          })
        }
      });
    } else {
      return _wreck.default;
    }
  }

  requestIncludesAuthInfo(request) {
    return request.headers.authorization ? true : false;
  }

  getAdditionalAuthHeader(request) {
    return {};
  }

  getCookie(request, authInfo) {
    return {
      username: authInfo.user_name,
      credentials: {
        authHeaderValue: request.headers.authorization
      },
      authType: this.type,
      expiryTime: Date.now() + this.config.session.ttl
    };
  } // TODO: Add token expiration check here


  async isValidCookie(cookie) {
    var _cookie$credentials, _cookie$credentials2, _cookie$credentials3;

    if (cookie.authType !== this.type || !cookie.username || !cookie.expiryTime || !((_cookie$credentials = cookie.credentials) !== null && _cookie$credentials !== void 0 && _cookie$credentials.authHeaderValue) || !((_cookie$credentials2 = cookie.credentials) !== null && _cookie$credentials2 !== void 0 && _cookie$credentials2.expires_at)) {
      return false;
    }

    if (((_cookie$credentials3 = cookie.credentials) === null || _cookie$credentials3 === void 0 ? void 0 : _cookie$credentials3.expires_at) > Date.now()) {
      return true;
    } // need to renew id token


    if (cookie.credentials.refresh_token) {
      try {
        var _this$config$openid5, _this$config$openid6;

        const query = {
          grant_type: 'refresh_token',
          client_id: (_this$config$openid5 = this.config.openid) === null || _this$config$openid5 === void 0 ? void 0 : _this$config$openid5.client_id,
          client_secret: (_this$config$openid6 = this.config.openid) === null || _this$config$openid6 === void 0 ? void 0 : _this$config$openid6.client_secret,
          refresh_token: cookie.credentials.refresh_token
        };
        const refreshTokenResponse = await (0, _helper.callTokenEndpoint)(this.openIdAuthConfig.tokenEndpoint, query, this.wreckClient); // if no id_token from refresh token call, maybe the Idp doesn't allow refresh id_token

        if (refreshTokenResponse.idToken) {
          cookie.credentials = {
            authHeaderValue: `Bearer ${refreshTokenResponse.idToken}`,
            refresh_token: refreshTokenResponse.refreshToken,
            expires_at: Date.now() + refreshTokenResponse.expiresIn * 1000 // expiresIn is in second

          };
          return true;
        } else {
          return false;
        }
      } catch (error) {
        this.logger.error(error);
        return false;
      }
    } else {
      // no refresh token, and current token is expired
      return false;
    }
  }

  handleUnauthedRequest(request, response, toolkit) {
    if (this.isPageRequest(request)) {
      // nextUrl is a key value pair
      const nextUrl = (0, _next_url.composeNextUrlQueryParam)(request, this.coreSetup.http.basePath.serverBasePath);
      return response.redirected({
        headers: {
          location: `${this.coreSetup.http.basePath.serverBasePath}/auth/openid/login?${nextUrl}`
        }
      });
    } else {
      return response.unauthorized();
    }
  }

  buildAuthHeaderFromCookie(cookie) {
    var _cookie$credentials4;

    const header = {};
    const authHeaderValue = (_cookie$credentials4 = cookie.credentials) === null || _cookie$credentials4 === void 0 ? void 0 : _cookie$credentials4.authHeaderValue;

    if (authHeaderValue) {
      header.authorization = authHeaderValue;
    }

    return header;
  }

}

exports.OpenIdAuthentication = OpenIdAuthentication;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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