/*
 * Copyright (C) 2021 Apple Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1.  Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 * 2.  Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE INC. AND ITS CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL APPLE INC. OR ITS CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#include "CacheStorageEngineConnection.h"

#include "ArgumentCoders.h"
#include "CacheStorageEngineConnectionMessages.h"
#include "Decoder.h"
#include "HandleMessage.h"
#include "WebCoreArgumentCoders.h"
#include <WebCore/CacheQueryOptions.h>
#include <WebCore/ClientOrigin.h>
#include <WebCore/DOMCacheEngine.h>
#include <WebCore/ResourceRequest.h>
#include <WebCore/RetrieveRecordsOptions.h>
#include <optional>
#include <wtf/Vector.h>
#include <wtf/text/WTFString.h>

namespace Messages {

namespace CacheStorageEngineConnection {

void Open::callReply(IPC::Decoder& decoder, CompletionHandler<void(WebCore::DOMCacheEngine::CacheIdentifierOrError&&)>&& completionHandler)
{
    std::optional<WebCore::DOMCacheEngine::CacheIdentifierOrError> result;
    decoder >> result;
    if (!result) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*result));
}

void Open::cancelReply(CompletionHandler<void(WebCore::DOMCacheEngine::CacheIdentifierOrError&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<WebCore::DOMCacheEngine::CacheIdentifierOrError>::create());
}

void Open::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const WebCore::DOMCacheEngine::CacheIdentifierOrError& result)
{
    encoder.get() << result;
    connection.sendSyncReply(WTFMove(encoder));
}

void Remove::callReply(IPC::Decoder& decoder, CompletionHandler<void(WebCore::DOMCacheEngine::CacheIdentifierOrError&&)>&& completionHandler)
{
    std::optional<WebCore::DOMCacheEngine::CacheIdentifierOrError> result;
    decoder >> result;
    if (!result) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*result));
}

void Remove::cancelReply(CompletionHandler<void(WebCore::DOMCacheEngine::CacheIdentifierOrError&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<WebCore::DOMCacheEngine::CacheIdentifierOrError>::create());
}

void Remove::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const WebCore::DOMCacheEngine::CacheIdentifierOrError& result)
{
    encoder.get() << result;
    connection.sendSyncReply(WTFMove(encoder));
}

void Caches::callReply(IPC::Decoder& decoder, CompletionHandler<void(WebCore::DOMCacheEngine::CacheInfosOrError&&)>&& completionHandler)
{
    std::optional<WebCore::DOMCacheEngine::CacheInfosOrError> result;
    decoder >> result;
    if (!result) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*result));
}

void Caches::cancelReply(CompletionHandler<void(WebCore::DOMCacheEngine::CacheInfosOrError&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<WebCore::DOMCacheEngine::CacheInfosOrError>::create());
}

void Caches::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const WebCore::DOMCacheEngine::CacheInfosOrError& result)
{
    encoder.get() << result;
    connection.sendSyncReply(WTFMove(encoder));
}

void RetrieveRecords::callReply(IPC::Decoder& decoder, CompletionHandler<void(WebCore::DOMCacheEngine::RecordsOrError&&)>&& completionHandler)
{
    std::optional<WebCore::DOMCacheEngine::RecordsOrError> result;
    decoder >> result;
    if (!result) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*result));
}

void RetrieveRecords::cancelReply(CompletionHandler<void(WebCore::DOMCacheEngine::RecordsOrError&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<WebCore::DOMCacheEngine::RecordsOrError>::create());
}

void RetrieveRecords::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const WebCore::DOMCacheEngine::RecordsOrError& result)
{
    encoder.get() << result;
    connection.sendSyncReply(WTFMove(encoder));
}

void DeleteMatchingRecords::callReply(IPC::Decoder& decoder, CompletionHandler<void(WebCore::DOMCacheEngine::RecordIdentifiersOrError&&)>&& completionHandler)
{
    std::optional<WebCore::DOMCacheEngine::RecordIdentifiersOrError> result;
    decoder >> result;
    if (!result) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*result));
}

void DeleteMatchingRecords::cancelReply(CompletionHandler<void(WebCore::DOMCacheEngine::RecordIdentifiersOrError&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<WebCore::DOMCacheEngine::RecordIdentifiersOrError>::create());
}

void DeleteMatchingRecords::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const WebCore::DOMCacheEngine::RecordIdentifiersOrError& result)
{
    encoder.get() << result;
    connection.sendSyncReply(WTFMove(encoder));
}

void PutRecords::callReply(IPC::Decoder& decoder, CompletionHandler<void(WebCore::DOMCacheEngine::RecordIdentifiersOrError&&)>&& completionHandler)
{
    std::optional<WebCore::DOMCacheEngine::RecordIdentifiersOrError> result;
    decoder >> result;
    if (!result) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*result));
}

void PutRecords::cancelReply(CompletionHandler<void(WebCore::DOMCacheEngine::RecordIdentifiersOrError&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<WebCore::DOMCacheEngine::RecordIdentifiersOrError>::create());
}

void PutRecords::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const WebCore::DOMCacheEngine::RecordIdentifiersOrError& result)
{
    encoder.get() << result;
    connection.sendSyncReply(WTFMove(encoder));
}

void ClearMemoryRepresentation::callReply(IPC::Decoder& decoder, CompletionHandler<void(std::optional<WebCore::DOMCacheEngine::Error>&&)>&& completionHandler)
{
    std::optional<std::optional<WebCore::DOMCacheEngine::Error>> error;
    decoder >> error;
    if (!error) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*error));
}

void ClearMemoryRepresentation::cancelReply(CompletionHandler<void(std::optional<WebCore::DOMCacheEngine::Error>&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<std::optional<WebCore::DOMCacheEngine::Error>>::create());
}

void ClearMemoryRepresentation::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const std::optional<WebCore::DOMCacheEngine::Error>& error)
{
    encoder.get() << error;
    connection.sendSyncReply(WTFMove(encoder));
}

void EngineRepresentation::callReply(IPC::Decoder& decoder, CompletionHandler<void(String&&)>&& completionHandler)
{
    std::optional<String> representation;
    decoder >> representation;
    if (!representation) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*representation));
}

void EngineRepresentation::cancelReply(CompletionHandler<void(String&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<String>::create());
}

void EngineRepresentation::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const String& representation)
{
    encoder.get() << representation;
    connection.sendSyncReply(WTFMove(encoder));
}

} // namespace CacheStorageEngineConnection

} // namespace Messages

namespace WebKit {

void CacheStorageEngineConnection::didReceiveMessage(IPC::Connection& connection, IPC::Decoder& decoder)
{
    Ref protectedThis { *this };
    if (decoder.messageName() == Messages::CacheStorageEngineConnection::Reference::name())
        return IPC::handleMessage<Messages::CacheStorageEngineConnection::Reference>(connection, decoder, this, &CacheStorageEngineConnection::reference);
    if (decoder.messageName() == Messages::CacheStorageEngineConnection::Dereference::name())
        return IPC::handleMessage<Messages::CacheStorageEngineConnection::Dereference>(connection, decoder, this, &CacheStorageEngineConnection::dereference);
    if (decoder.messageName() == Messages::CacheStorageEngineConnection::Open::name())
        return IPC::handleMessageAsync<Messages::CacheStorageEngineConnection::Open>(connection, decoder, this, &CacheStorageEngineConnection::open);
    if (decoder.messageName() == Messages::CacheStorageEngineConnection::Remove::name())
        return IPC::handleMessageAsync<Messages::CacheStorageEngineConnection::Remove>(connection, decoder, this, &CacheStorageEngineConnection::remove);
    if (decoder.messageName() == Messages::CacheStorageEngineConnection::Caches::name())
        return IPC::handleMessageAsync<Messages::CacheStorageEngineConnection::Caches>(connection, decoder, this, &CacheStorageEngineConnection::caches);
    if (decoder.messageName() == Messages::CacheStorageEngineConnection::RetrieveRecords::name())
        return IPC::handleMessageAsync<Messages::CacheStorageEngineConnection::RetrieveRecords>(connection, decoder, this, &CacheStorageEngineConnection::retrieveRecords);
    if (decoder.messageName() == Messages::CacheStorageEngineConnection::DeleteMatchingRecords::name())
        return IPC::handleMessageAsync<Messages::CacheStorageEngineConnection::DeleteMatchingRecords>(connection, decoder, this, &CacheStorageEngineConnection::deleteMatchingRecords);
    if (decoder.messageName() == Messages::CacheStorageEngineConnection::PutRecords::name())
        return IPC::handleMessageAsync<Messages::CacheStorageEngineConnection::PutRecords>(connection, decoder, this, &CacheStorageEngineConnection::putRecords);
    if (decoder.messageName() == Messages::CacheStorageEngineConnection::ClearMemoryRepresentation::name())
        return IPC::handleMessageAsync<Messages::CacheStorageEngineConnection::ClearMemoryRepresentation>(connection, decoder, this, &CacheStorageEngineConnection::clearMemoryRepresentation);
    if (decoder.messageName() == Messages::CacheStorageEngineConnection::EngineRepresentation::name())
        return IPC::handleMessageAsync<Messages::CacheStorageEngineConnection::EngineRepresentation>(connection, decoder, this, &CacheStorageEngineConnection::engineRepresentation);
    UNUSED_PARAM(connection);
    UNUSED_PARAM(decoder);
#if ENABLE(IPC_TESTING_API)
    if (connection.ignoreInvalidMessageForTesting())
        return;
#endif // ENABLE(IPC_TESTING_API)
    ASSERT_NOT_REACHED_WITH_MESSAGE("Unhandled message %s to %" PRIu64, IPC::description(decoder.messageName()), decoder.destinationID());
}

} // namespace WebKit
