/*
 * Copyright (C) 2021 Apple Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1.  Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 * 2.  Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE INC. AND ITS CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL APPLE INC. OR ITS CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#include "NetworkStorageManager.h"

#include "ArgumentCoders.h"
#include "Decoder.h"
#include "FileSystemStorageError.h"
#include "HandleMessage.h"
#include "NetworkStorageManagerMessages.h"
#include "SharedFileHandle.h"
#include "StorageAreaIdentifier.h"
#include "StorageAreaImplIdentifier.h"
#include "StorageAreaMapIdentifier.h"
#include "StorageNamespaceIdentifier.h"
#include "WebCoreArgumentCoders.h"
#include <WebCore/ClientOrigin.h>
#include <WebCore/FileSystemHandleIdentifier.h>
#include <WebCore/FileSystemSyncAccessHandleIdentifier.h>
#include <WebCore/IDBCursorInfo.h>
#include <WebCore/IDBGetAllRecordsData.h>
#include <WebCore/IDBGetRecordData.h>
#include <WebCore/IDBIndexInfo.h>
#include <WebCore/IDBIterateCursorData.h>
#include <WebCore/IDBKeyData.h>
#include <WebCore/IDBKeyRangeData.h>
#include <WebCore/IDBObjectStoreInfo.h>
#include <WebCore/IDBRequestData.h>
#include <WebCore/IDBResourceIdentifier.h>
#include <WebCore/IDBTransactionInfo.h>
#include <WebCore/IDBValue.h>
#include <WebCore/IndexedDB.h>
#include <WebCore/StorageType.h>
#include <optional>
#include <utility>
#include <wtf/Expected.h>
#include <wtf/HashMap.h>
#include <wtf/Vector.h>
#include <wtf/text/WTFString.h>

namespace Messages {

namespace NetworkStorageManager {

void Persisted::callReply(IPC::Decoder& decoder, CompletionHandler<void(bool&&)>&& completionHandler)
{
    std::optional<bool> persisted;
    decoder >> persisted;
    if (!persisted) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*persisted));
}

void Persisted::cancelReply(CompletionHandler<void(bool&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<bool>::create());
}

void Persisted::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, bool persisted)
{
    encoder.get() << persisted;
    connection.sendSyncReply(WTFMove(encoder));
}

void Persist::callReply(IPC::Decoder& decoder, CompletionHandler<void(bool&&)>&& completionHandler)
{
    std::optional<bool> persisted;
    decoder >> persisted;
    if (!persisted) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*persisted));
}

void Persist::cancelReply(CompletionHandler<void(bool&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<bool>::create());
}

void Persist::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, bool persisted)
{
    encoder.get() << persisted;
    connection.sendSyncReply(WTFMove(encoder));
}

void FileSystemGetDirectory::callReply(IPC::Decoder& decoder, CompletionHandler<void(Expected<WebCore::FileSystemHandleIdentifier, WebKit::FileSystemStorageError>&&)>&& completionHandler)
{
    std::optional<Expected<WebCore::FileSystemHandleIdentifier, WebKit::FileSystemStorageError>> result;
    decoder >> result;
    if (!result) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*result));
}

void FileSystemGetDirectory::cancelReply(CompletionHandler<void(Expected<WebCore::FileSystemHandleIdentifier, WebKit::FileSystemStorageError>&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<Expected<WebCore::FileSystemHandleIdentifier, WebKit::FileSystemStorageError>>::create());
}

void FileSystemGetDirectory::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const Expected<WebCore::FileSystemHandleIdentifier, WebKit::FileSystemStorageError>& result)
{
    encoder.get() << result;
    connection.sendSyncReply(WTFMove(encoder));
}

void IsSameEntry::callReply(IPC::Decoder& decoder, CompletionHandler<void(bool&&)>&& completionHandler)
{
    std::optional<bool> result;
    decoder >> result;
    if (!result) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*result));
}

void IsSameEntry::cancelReply(CompletionHandler<void(bool&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<bool>::create());
}

void IsSameEntry::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, bool result)
{
    encoder.get() << result;
    connection.sendSyncReply(WTFMove(encoder));
}

void GetFileHandle::callReply(IPC::Decoder& decoder, CompletionHandler<void(Expected<WebCore::FileSystemHandleIdentifier, WebKit::FileSystemStorageError>&&)>&& completionHandler)
{
    std::optional<Expected<WebCore::FileSystemHandleIdentifier, WebKit::FileSystemStorageError>> result;
    decoder >> result;
    if (!result) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*result));
}

void GetFileHandle::cancelReply(CompletionHandler<void(Expected<WebCore::FileSystemHandleIdentifier, WebKit::FileSystemStorageError>&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<Expected<WebCore::FileSystemHandleIdentifier, WebKit::FileSystemStorageError>>::create());
}

void GetFileHandle::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const Expected<WebCore::FileSystemHandleIdentifier, WebKit::FileSystemStorageError>& result)
{
    encoder.get() << result;
    connection.sendSyncReply(WTFMove(encoder));
}

void GetDirectoryHandle::callReply(IPC::Decoder& decoder, CompletionHandler<void(Expected<WebCore::FileSystemHandleIdentifier, WebKit::FileSystemStorageError>&&)>&& completionHandler)
{
    std::optional<Expected<WebCore::FileSystemHandleIdentifier, WebKit::FileSystemStorageError>> result;
    decoder >> result;
    if (!result) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*result));
}

void GetDirectoryHandle::cancelReply(CompletionHandler<void(Expected<WebCore::FileSystemHandleIdentifier, WebKit::FileSystemStorageError>&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<Expected<WebCore::FileSystemHandleIdentifier, WebKit::FileSystemStorageError>>::create());
}

void GetDirectoryHandle::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const Expected<WebCore::FileSystemHandleIdentifier, WebKit::FileSystemStorageError>& result)
{
    encoder.get() << result;
    connection.sendSyncReply(WTFMove(encoder));
}

void RemoveEntry::callReply(IPC::Decoder& decoder, CompletionHandler<void(std::optional<WebKit::FileSystemStorageError>&&)>&& completionHandler)
{
    std::optional<std::optional<WebKit::FileSystemStorageError>> result;
    decoder >> result;
    if (!result) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*result));
}

void RemoveEntry::cancelReply(CompletionHandler<void(std::optional<WebKit::FileSystemStorageError>&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<std::optional<WebKit::FileSystemStorageError>>::create());
}

void RemoveEntry::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const std::optional<WebKit::FileSystemStorageError>& result)
{
    encoder.get() << result;
    connection.sendSyncReply(WTFMove(encoder));
}

void Resolve::callReply(IPC::Decoder& decoder, CompletionHandler<void(Expected<Vector<String>, WebKit::FileSystemStorageError>&&)>&& completionHandler)
{
    std::optional<Expected<Vector<String>, WebKit::FileSystemStorageError>> result;
    decoder >> result;
    if (!result) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*result));
}

void Resolve::cancelReply(CompletionHandler<void(Expected<Vector<String>, WebKit::FileSystemStorageError>&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<Expected<Vector<String>, WebKit::FileSystemStorageError>>::create());
}

void Resolve::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const Expected<Vector<String>, WebKit::FileSystemStorageError>& result)
{
    encoder.get() << result;
    connection.sendSyncReply(WTFMove(encoder));
}

void Move::callReply(IPC::Decoder& decoder, CompletionHandler<void(std::optional<WebKit::FileSystemStorageError>&&)>&& completionHandler)
{
    std::optional<std::optional<WebKit::FileSystemStorageError>> result;
    decoder >> result;
    if (!result) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*result));
}

void Move::cancelReply(CompletionHandler<void(std::optional<WebKit::FileSystemStorageError>&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<std::optional<WebKit::FileSystemStorageError>>::create());
}

void Move::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const std::optional<WebKit::FileSystemStorageError>& result)
{
    encoder.get() << result;
    connection.sendSyncReply(WTFMove(encoder));
}

void GetFile::callReply(IPC::Decoder& decoder, CompletionHandler<void(Expected<String, WebKit::FileSystemStorageError>&&)>&& completionHandler)
{
    std::optional<Expected<String, WebKit::FileSystemStorageError>> result;
    decoder >> result;
    if (!result) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*result));
}

void GetFile::cancelReply(CompletionHandler<void(Expected<String, WebKit::FileSystemStorageError>&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<Expected<String, WebKit::FileSystemStorageError>>::create());
}

void GetFile::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const Expected<String, WebKit::FileSystemStorageError>& result)
{
    encoder.get() << result;
    connection.sendSyncReply(WTFMove(encoder));
}

void CreateSyncAccessHandle::callReply(IPC::Decoder& decoder, CompletionHandler<void(Expected<std::pair<WebCore::FileSystemSyncAccessHandleIdentifier, IPC::SharedFileHandle>, WebKit::FileSystemStorageError>&&)>&& completionHandler)
{
    std::optional<Expected<std::pair<WebCore::FileSystemSyncAccessHandleIdentifier, IPC::SharedFileHandle>, WebKit::FileSystemStorageError>> result;
    decoder >> result;
    if (!result) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*result));
}

void CreateSyncAccessHandle::cancelReply(CompletionHandler<void(Expected<std::pair<WebCore::FileSystemSyncAccessHandleIdentifier, IPC::SharedFileHandle>, WebKit::FileSystemStorageError>&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<Expected<std::pair<WebCore::FileSystemSyncAccessHandleIdentifier, IPC::SharedFileHandle>, WebKit::FileSystemStorageError>>::create());
}

void CreateSyncAccessHandle::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const Expected<std::pair<WebCore::FileSystemSyncAccessHandleIdentifier, IPC::SharedFileHandle>, WebKit::FileSystemStorageError>& result)
{
    encoder.get() << result;
    connection.sendSyncReply(WTFMove(encoder));
}

void CloseSyncAccessHandle::callReply(IPC::Decoder& decoder, CompletionHandler<void(std::optional<WebKit::FileSystemStorageError>&&)>&& completionHandler)
{
    std::optional<std::optional<WebKit::FileSystemStorageError>> result;
    decoder >> result;
    if (!result) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*result));
}

void CloseSyncAccessHandle::cancelReply(CompletionHandler<void(std::optional<WebKit::FileSystemStorageError>&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<std::optional<WebKit::FileSystemStorageError>>::create());
}

void CloseSyncAccessHandle::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const std::optional<WebKit::FileSystemStorageError>& result)
{
    encoder.get() << result;
    connection.sendSyncReply(WTFMove(encoder));
}

void GetHandleNames::callReply(IPC::Decoder& decoder, CompletionHandler<void(Expected<Vector<String>, WebKit::FileSystemStorageError>&&)>&& completionHandler)
{
    std::optional<Expected<Vector<String>, WebKit::FileSystemStorageError>> result;
    decoder >> result;
    if (!result) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*result));
}

void GetHandleNames::cancelReply(CompletionHandler<void(Expected<Vector<String>, WebKit::FileSystemStorageError>&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<Expected<Vector<String>, WebKit::FileSystemStorageError>>::create());
}

void GetHandleNames::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const Expected<Vector<String>, WebKit::FileSystemStorageError>& result)
{
    encoder.get() << result;
    connection.sendSyncReply(WTFMove(encoder));
}

void GetHandle::callReply(IPC::Decoder& decoder, CompletionHandler<void(Expected<std::pair<WebCore::FileSystemHandleIdentifier, bool>, WebKit::FileSystemStorageError>&&)>&& completionHandler)
{
    std::optional<Expected<std::pair<WebCore::FileSystemHandleIdentifier, bool>, WebKit::FileSystemStorageError>> result;
    decoder >> result;
    if (!result) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*result));
}

void GetHandle::cancelReply(CompletionHandler<void(Expected<std::pair<WebCore::FileSystemHandleIdentifier, bool>, WebKit::FileSystemStorageError>&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<Expected<std::pair<WebCore::FileSystemHandleIdentifier, bool>, WebKit::FileSystemStorageError>>::create());
}

void GetHandle::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const Expected<std::pair<WebCore::FileSystemHandleIdentifier, bool>, WebKit::FileSystemStorageError>& result)
{
    encoder.get() << result;
    connection.sendSyncReply(WTFMove(encoder));
}

void ConnectToStorageArea::callReply(IPC::Decoder& decoder, CompletionHandler<void(WebKit::StorageAreaIdentifier&&, HashMap<String, String>&&, uint64_t&&)>&& completionHandler)
{
    std::optional<WebKit::StorageAreaIdentifier> identifier;
    decoder >> identifier;
    if (!identifier) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    std::optional<HashMap<String, String>> items;
    decoder >> items;
    if (!items) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    std::optional<uint64_t> messageIdentifier;
    decoder >> messageIdentifier;
    if (!messageIdentifier) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*identifier), WTFMove(*items), WTFMove(*messageIdentifier));
}

void ConnectToStorageArea::cancelReply(CompletionHandler<void(WebKit::StorageAreaIdentifier&&, HashMap<String, String>&&, uint64_t&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<WebKit::StorageAreaIdentifier>::create(), IPC::AsyncReplyError<HashMap<String, String>>::create(), IPC::AsyncReplyError<uint64_t>::create());
}

void ConnectToStorageArea::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const WebKit::StorageAreaIdentifier& identifier, const HashMap<String, String>& items, uint64_t messageIdentifier)
{
    encoder.get() << identifier;
    encoder.get() << items;
    encoder.get() << messageIdentifier;
    connection.sendSyncReply(WTFMove(encoder));
}

void ConnectToStorageAreaSync::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const WebKit::StorageAreaIdentifier& identifier, const HashMap<String, String>& items, uint64_t messageIdentifier)
{
    encoder.get() << identifier;
    encoder.get() << items;
    encoder.get() << messageIdentifier;
    connection.sendSyncReply(WTFMove(encoder));
}

void SetItem::callReply(IPC::Decoder& decoder, CompletionHandler<void(bool&&)>&& completionHandler)
{
    std::optional<bool> quotaException;
    decoder >> quotaException;
    if (!quotaException) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*quotaException));
}

void SetItem::cancelReply(CompletionHandler<void(bool&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<bool>::create());
}

void SetItem::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, bool quotaException)
{
    encoder.get() << quotaException;
    connection.sendSyncReply(WTFMove(encoder));
}

void RemoveItem::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void RemoveItem::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void RemoveItem::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

void Clear::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void Clear::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void Clear::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

} // namespace NetworkStorageManager

} // namespace Messages

namespace WebKit {

void NetworkStorageManager::didReceiveMessage(IPC::Connection& connection, IPC::Decoder& decoder)
{
    Ref protectedThis { *this };
    if (decoder.messageName() == Messages::NetworkStorageManager::Persisted::name())
        return IPC::handleMessageAsync<Messages::NetworkStorageManager::Persisted>(connection, decoder, this, &NetworkStorageManager::persisted);
    if (decoder.messageName() == Messages::NetworkStorageManager::Persist::name())
        return IPC::handleMessageAsync<Messages::NetworkStorageManager::Persist>(connection, decoder, this, &NetworkStorageManager::persist);
    if (decoder.messageName() == Messages::NetworkStorageManager::FileSystemGetDirectory::name())
        return IPC::handleMessageAsyncWantsConnection<Messages::NetworkStorageManager::FileSystemGetDirectory>(connection, decoder, this, &NetworkStorageManager::fileSystemGetDirectory);
    if (decoder.messageName() == Messages::NetworkStorageManager::CloseHandle::name())
        return IPC::handleMessage<Messages::NetworkStorageManager::CloseHandle>(connection, decoder, this, &NetworkStorageManager::closeHandle);
    if (decoder.messageName() == Messages::NetworkStorageManager::IsSameEntry::name())
        return IPC::handleMessageAsync<Messages::NetworkStorageManager::IsSameEntry>(connection, decoder, this, &NetworkStorageManager::isSameEntry);
    if (decoder.messageName() == Messages::NetworkStorageManager::GetFileHandle::name())
        return IPC::handleMessageAsyncWantsConnection<Messages::NetworkStorageManager::GetFileHandle>(connection, decoder, this, &NetworkStorageManager::getFileHandle);
    if (decoder.messageName() == Messages::NetworkStorageManager::GetDirectoryHandle::name())
        return IPC::handleMessageAsyncWantsConnection<Messages::NetworkStorageManager::GetDirectoryHandle>(connection, decoder, this, &NetworkStorageManager::getDirectoryHandle);
    if (decoder.messageName() == Messages::NetworkStorageManager::RemoveEntry::name())
        return IPC::handleMessageAsync<Messages::NetworkStorageManager::RemoveEntry>(connection, decoder, this, &NetworkStorageManager::removeEntry);
    if (decoder.messageName() == Messages::NetworkStorageManager::Resolve::name())
        return IPC::handleMessageAsync<Messages::NetworkStorageManager::Resolve>(connection, decoder, this, &NetworkStorageManager::resolve);
    if (decoder.messageName() == Messages::NetworkStorageManager::Move::name())
        return IPC::handleMessageAsync<Messages::NetworkStorageManager::Move>(connection, decoder, this, &NetworkStorageManager::move);
    if (decoder.messageName() == Messages::NetworkStorageManager::GetFile::name())
        return IPC::handleMessageAsync<Messages::NetworkStorageManager::GetFile>(connection, decoder, this, &NetworkStorageManager::getFile);
    if (decoder.messageName() == Messages::NetworkStorageManager::CreateSyncAccessHandle::name())
        return IPC::handleMessageAsync<Messages::NetworkStorageManager::CreateSyncAccessHandle>(connection, decoder, this, &NetworkStorageManager::createSyncAccessHandle);
    if (decoder.messageName() == Messages::NetworkStorageManager::CloseSyncAccessHandle::name())
        return IPC::handleMessageAsync<Messages::NetworkStorageManager::CloseSyncAccessHandle>(connection, decoder, this, &NetworkStorageManager::closeSyncAccessHandle);
    if (decoder.messageName() == Messages::NetworkStorageManager::GetHandleNames::name())
        return IPC::handleMessageAsync<Messages::NetworkStorageManager::GetHandleNames>(connection, decoder, this, &NetworkStorageManager::getHandleNames);
    if (decoder.messageName() == Messages::NetworkStorageManager::GetHandle::name())
        return IPC::handleMessageAsyncWantsConnection<Messages::NetworkStorageManager::GetHandle>(connection, decoder, this, &NetworkStorageManager::getHandle);
    if (decoder.messageName() == Messages::NetworkStorageManager::ConnectToStorageArea::name())
        return IPC::handleMessageAsyncWantsConnection<Messages::NetworkStorageManager::ConnectToStorageArea>(connection, decoder, this, &NetworkStorageManager::connectToStorageArea);
    if (decoder.messageName() == Messages::NetworkStorageManager::DisconnectFromStorageArea::name())
        return IPC::handleMessageWantsConnection<Messages::NetworkStorageManager::DisconnectFromStorageArea>(connection, decoder, this, &NetworkStorageManager::disconnectFromStorageArea);
    if (decoder.messageName() == Messages::NetworkStorageManager::CloneSessionStorageNamespace::name())
        return IPC::handleMessageWantsConnection<Messages::NetworkStorageManager::CloneSessionStorageNamespace>(connection, decoder, this, &NetworkStorageManager::cloneSessionStorageNamespace);
    if (decoder.messageName() == Messages::NetworkStorageManager::SetItem::name())
        return IPC::handleMessageAsyncWantsConnection<Messages::NetworkStorageManager::SetItem>(connection, decoder, this, &NetworkStorageManager::setItem);
    if (decoder.messageName() == Messages::NetworkStorageManager::RemoveItem::name())
        return IPC::handleMessageAsyncWantsConnection<Messages::NetworkStorageManager::RemoveItem>(connection, decoder, this, &NetworkStorageManager::removeItem);
    if (decoder.messageName() == Messages::NetworkStorageManager::Clear::name())
        return IPC::handleMessageAsyncWantsConnection<Messages::NetworkStorageManager::Clear>(connection, decoder, this, &NetworkStorageManager::clear);
    if (decoder.messageName() == Messages::NetworkStorageManager::OpenDatabase::name())
        return IPC::handleMessageWantsConnection<Messages::NetworkStorageManager::OpenDatabase>(connection, decoder, this, &NetworkStorageManager::openDatabase);
    if (decoder.messageName() == Messages::NetworkStorageManager::OpenDBRequestCancelled::name())
        return IPC::handleMessage<Messages::NetworkStorageManager::OpenDBRequestCancelled>(connection, decoder, this, &NetworkStorageManager::openDBRequestCancelled);
    if (decoder.messageName() == Messages::NetworkStorageManager::DeleteDatabase::name())
        return IPC::handleMessageWantsConnection<Messages::NetworkStorageManager::DeleteDatabase>(connection, decoder, this, &NetworkStorageManager::deleteDatabase);
    if (decoder.messageName() == Messages::NetworkStorageManager::EstablishTransaction::name())
        return IPC::handleMessage<Messages::NetworkStorageManager::EstablishTransaction>(connection, decoder, this, &NetworkStorageManager::establishTransaction);
    if (decoder.messageName() == Messages::NetworkStorageManager::DatabaseConnectionPendingClose::name())
        return IPC::handleMessage<Messages::NetworkStorageManager::DatabaseConnectionPendingClose>(connection, decoder, this, &NetworkStorageManager::databaseConnectionPendingClose);
    if (decoder.messageName() == Messages::NetworkStorageManager::DatabaseConnectionClosed::name())
        return IPC::handleMessage<Messages::NetworkStorageManager::DatabaseConnectionClosed>(connection, decoder, this, &NetworkStorageManager::databaseConnectionClosed);
    if (decoder.messageName() == Messages::NetworkStorageManager::AbortOpenAndUpgradeNeeded::name())
        return IPC::handleMessage<Messages::NetworkStorageManager::AbortOpenAndUpgradeNeeded>(connection, decoder, this, &NetworkStorageManager::abortOpenAndUpgradeNeeded);
    if (decoder.messageName() == Messages::NetworkStorageManager::DidFireVersionChangeEvent::name())
        return IPC::handleMessage<Messages::NetworkStorageManager::DidFireVersionChangeEvent>(connection, decoder, this, &NetworkStorageManager::didFireVersionChangeEvent);
    if (decoder.messageName() == Messages::NetworkStorageManager::DidFinishHandlingVersionChangeTransaction::name())
        return IPC::handleMessage<Messages::NetworkStorageManager::DidFinishHandlingVersionChangeTransaction>(connection, decoder, this, &NetworkStorageManager::didFinishHandlingVersionChangeTransaction);
    if (decoder.messageName() == Messages::NetworkStorageManager::AbortTransaction::name())
        return IPC::handleMessage<Messages::NetworkStorageManager::AbortTransaction>(connection, decoder, this, &NetworkStorageManager::abortTransaction);
    if (decoder.messageName() == Messages::NetworkStorageManager::CommitTransaction::name())
        return IPC::handleMessage<Messages::NetworkStorageManager::CommitTransaction>(connection, decoder, this, &NetworkStorageManager::commitTransaction);
    if (decoder.messageName() == Messages::NetworkStorageManager::CreateObjectStore::name())
        return IPC::handleMessage<Messages::NetworkStorageManager::CreateObjectStore>(connection, decoder, this, &NetworkStorageManager::createObjectStore);
    if (decoder.messageName() == Messages::NetworkStorageManager::DeleteObjectStore::name())
        return IPC::handleMessage<Messages::NetworkStorageManager::DeleteObjectStore>(connection, decoder, this, &NetworkStorageManager::deleteObjectStore);
    if (decoder.messageName() == Messages::NetworkStorageManager::RenameObjectStore::name())
        return IPC::handleMessage<Messages::NetworkStorageManager::RenameObjectStore>(connection, decoder, this, &NetworkStorageManager::renameObjectStore);
    if (decoder.messageName() == Messages::NetworkStorageManager::ClearObjectStore::name())
        return IPC::handleMessage<Messages::NetworkStorageManager::ClearObjectStore>(connection, decoder, this, &NetworkStorageManager::clearObjectStore);
    if (decoder.messageName() == Messages::NetworkStorageManager::CreateIndex::name())
        return IPC::handleMessage<Messages::NetworkStorageManager::CreateIndex>(connection, decoder, this, &NetworkStorageManager::createIndex);
    if (decoder.messageName() == Messages::NetworkStorageManager::DeleteIndex::name())
        return IPC::handleMessage<Messages::NetworkStorageManager::DeleteIndex>(connection, decoder, this, &NetworkStorageManager::deleteIndex);
    if (decoder.messageName() == Messages::NetworkStorageManager::RenameIndex::name())
        return IPC::handleMessage<Messages::NetworkStorageManager::RenameIndex>(connection, decoder, this, &NetworkStorageManager::renameIndex);
    if (decoder.messageName() == Messages::NetworkStorageManager::PutOrAdd::name())
        return IPC::handleMessageWantsConnection<Messages::NetworkStorageManager::PutOrAdd>(connection, decoder, this, &NetworkStorageManager::putOrAdd);
    if (decoder.messageName() == Messages::NetworkStorageManager::GetRecord::name())
        return IPC::handleMessage<Messages::NetworkStorageManager::GetRecord>(connection, decoder, this, &NetworkStorageManager::getRecord);
    if (decoder.messageName() == Messages::NetworkStorageManager::GetAllRecords::name())
        return IPC::handleMessage<Messages::NetworkStorageManager::GetAllRecords>(connection, decoder, this, &NetworkStorageManager::getAllRecords);
    if (decoder.messageName() == Messages::NetworkStorageManager::GetCount::name())
        return IPC::handleMessage<Messages::NetworkStorageManager::GetCount>(connection, decoder, this, &NetworkStorageManager::getCount);
    if (decoder.messageName() == Messages::NetworkStorageManager::DeleteRecord::name())
        return IPC::handleMessage<Messages::NetworkStorageManager::DeleteRecord>(connection, decoder, this, &NetworkStorageManager::deleteRecord);
    if (decoder.messageName() == Messages::NetworkStorageManager::OpenCursor::name())
        return IPC::handleMessage<Messages::NetworkStorageManager::OpenCursor>(connection, decoder, this, &NetworkStorageManager::openCursor);
    if (decoder.messageName() == Messages::NetworkStorageManager::IterateCursor::name())
        return IPC::handleMessage<Messages::NetworkStorageManager::IterateCursor>(connection, decoder, this, &NetworkStorageManager::iterateCursor);
    if (decoder.messageName() == Messages::NetworkStorageManager::GetAllDatabaseNamesAndVersions::name())
        return IPC::handleMessageWantsConnection<Messages::NetworkStorageManager::GetAllDatabaseNamesAndVersions>(connection, decoder, this, &NetworkStorageManager::getAllDatabaseNamesAndVersions);
    UNUSED_PARAM(connection);
    UNUSED_PARAM(decoder);
#if ENABLE(IPC_TESTING_API)
    if (connection.ignoreInvalidMessageForTesting())
        return;
#endif // ENABLE(IPC_TESTING_API)
    ASSERT_NOT_REACHED_WITH_MESSAGE("Unhandled message %s to %" PRIu64, IPC::description(decoder.messageName()), decoder.destinationID());
}

bool NetworkStorageManager::didReceiveSyncMessage(IPC::Connection& connection, IPC::Decoder& decoder, UniqueRef<IPC::Encoder>& replyEncoder)
{
    Ref protectedThis { *this };
    if (decoder.messageName() == Messages::NetworkStorageManager::ConnectToStorageAreaSync::name())
        return IPC::handleMessageSynchronousWantsConnection<Messages::NetworkStorageManager::ConnectToStorageAreaSync>(connection, decoder, replyEncoder, this, &NetworkStorageManager::connectToStorageAreaSync);
    UNUSED_PARAM(connection);
    UNUSED_PARAM(decoder);
    UNUSED_PARAM(replyEncoder);
#if ENABLE(IPC_TESTING_API)
    if (connection.ignoreInvalidMessageForTesting())
        return false;
#endif // ENABLE(IPC_TESTING_API)
    ASSERT_NOT_REACHED_WITH_MESSAGE("Unhandled synchronous message %s to %" PRIu64, description(decoder.messageName()), decoder.destinationID());
    return false;
}

} // namespace WebKit
