/*
 * Copyright (C) 2021 Apple Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1.  Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 * 2.  Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE INC. AND ITS CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL APPLE INC. OR ITS CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#include "WebPasteboardProxy.h"

#include "ArgumentCoders.h"
#include "Decoder.h"
#include "HandleMessage.h"
#if PLATFORM(COCOA)
#include "SandboxExtension.h"
#endif
#if PLATFORM(GTK)
#include "SharedBufferCopy.h"
#endif
#include "SharedMemory.h"
#include "WebCoreArgumentCoders.h"
#include "WebPasteboardProxyMessages.h"
#if PLATFORM(COCOA)
#include <WebCore/Color.h>
#endif
#include <WebCore/PageIdentifier.h>
#include <WebCore/Pasteboard.h>
#include <WebCore/PasteboardItemInfo.h>
#if PLATFORM(GTK)
#include <WebCore/SelectionData.h>
#endif
#include <optional>
#include <wtf/Vector.h>
#include <wtf/text/WTFString.h>

namespace Messages {

namespace WebPasteboardProxy {

void WriteCustomData::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, int64_t changeCount)
{
    encoder.get() << changeCount;
    connection.sendSyncReply(WTFMove(encoder));
}

void TypesSafeForDOMToReadAndWrite::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const Vector<String>& types)
{
    encoder.get() << types;
    connection.sendSyncReply(WTFMove(encoder));
}

void AllPasteboardItemInfo::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const std::optional<Vector<WebCore::PasteboardItemInfo>>& allInfo)
{
    encoder.get() << allInfo;
    connection.sendSyncReply(WTFMove(encoder));
}

void InformationForItemAtIndex::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const std::optional<WebCore::PasteboardItemInfo>& info)
{
    encoder.get() << info;
    connection.sendSyncReply(WTFMove(encoder));
}

void GetPasteboardItemsCount::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, uint64_t itemsCount)
{
    encoder.get() << itemsCount;
    connection.sendSyncReply(WTFMove(encoder));
}

void ReadStringFromPasteboard::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const String& string)
{
    encoder.get() << string;
    connection.sendSyncReply(WTFMove(encoder));
}

void ReadURLFromPasteboard::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const String& url, const String& title)
{
    encoder.get() << url;
    encoder.get() << title;
    connection.sendSyncReply(WTFMove(encoder));
}

void ReadBufferFromPasteboard::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const WebKit::SharedMemory::IPCHandle& handle)
{
    encoder.get() << handle;
    connection.sendSyncReply(WTFMove(encoder));
}

void ContainsStringSafeForDOMToReadForType::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, bool result)
{
    encoder.get() << result;
    connection.sendSyncReply(WTFMove(encoder));
}

#if PLATFORM(COCOA)

void GetNumberOfFiles::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, uint64_t numberOfFiles)
{
    encoder.get() << numberOfFiles;
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if PLATFORM(COCOA)

void GetPasteboardTypes::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const Vector<String>& types)
{
    encoder.get() << types;
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if PLATFORM(COCOA)

void GetPasteboardPathnamesForType::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const Vector<String>& pathnames, const Vector<WebKit::SandboxExtension::Handle>& sandboxExtensions)
{
    encoder.get() << pathnames;
    encoder.get() << sandboxExtensions;
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if PLATFORM(COCOA)

void GetPasteboardStringForType::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const String& string)
{
    encoder.get() << string;
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if PLATFORM(COCOA)

void GetPasteboardStringsForType::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const Vector<String>& strings)
{
    encoder.get() << strings;
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if PLATFORM(COCOA)

void GetPasteboardBufferForType::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const WebKit::SharedMemory::IPCHandle& handle)
{
    encoder.get() << handle;
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if PLATFORM(COCOA)

void GetPasteboardChangeCount::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, int64_t changeCount)
{
    encoder.get() << changeCount;
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if PLATFORM(COCOA)

void GetPasteboardColor::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const WebCore::Color& color)
{
    encoder.get() << color;
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if PLATFORM(COCOA)

void GetPasteboardURL::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const String& urlString)
{
    encoder.get() << urlString;
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if PLATFORM(COCOA)

void AddPasteboardTypes::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, int64_t changeCount)
{
    encoder.get() << changeCount;
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if PLATFORM(COCOA)

void SetPasteboardTypes::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, int64_t changeCount)
{
    encoder.get() << changeCount;
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if PLATFORM(COCOA)

void SetPasteboardURL::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, int64_t changeCount)
{
    encoder.get() << changeCount;
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if PLATFORM(COCOA)

void SetPasteboardColor::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, int64_t changeCount)
{
    encoder.get() << changeCount;
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if PLATFORM(COCOA)

void SetPasteboardStringForType::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, int64_t changeCount)
{
    encoder.get() << changeCount;
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if PLATFORM(COCOA)

void SetPasteboardBufferForType::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, int64_t changeCount)
{
    encoder.get() << changeCount;
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if PLATFORM(COCOA)

void ContainsURLStringSuitableForLoading::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, bool result)
{
    encoder.get() << result;
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if PLATFORM(COCOA)

void URLStringSuitableForLoading::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const String& url, const String& title)
{
    encoder.get() << url;
    encoder.get() << title;
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if PLATFORM(GTK)

void GetTypes::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const Vector<String>& types)
{
    encoder.get() << types;
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if PLATFORM(GTK)

void ReadText::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const String& text)
{
    encoder.get() << text;
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if PLATFORM(GTK)

void ReadFilePaths::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const Vector<String>& types)
{
    encoder.get() << types;
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if PLATFORM(GTK)

void ReadBuffer::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const IPC::SharedBufferCopy& data)
{
    encoder.get() << data;
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if USE(LIBWPE)

void GetPasteboardTypes::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const Vector<String>& types)
{
    encoder.get() << types;
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

} // namespace WebPasteboardProxy

} // namespace Messages

namespace WebKit {

void WebPasteboardProxy::didReceiveMessage(IPC::Connection& connection, IPC::Decoder& decoder)
{
#if PLATFORM(IOS_FAMILY)
    if (decoder.messageName() == Messages::WebPasteboardProxy::WriteURLToPasteboard::name())
        return IPC::handleMessageWantsConnection<Messages::WebPasteboardProxy::WriteURLToPasteboard>(connection, decoder, this, &WebPasteboardProxy::writeURLToPasteboard);
#endif
#if PLATFORM(IOS_FAMILY)
    if (decoder.messageName() == Messages::WebPasteboardProxy::WriteWebContentToPasteboard::name())
        return IPC::handleMessageWantsConnection<Messages::WebPasteboardProxy::WriteWebContentToPasteboard>(connection, decoder, this, &WebPasteboardProxy::writeWebContentToPasteboard);
#endif
#if PLATFORM(IOS_FAMILY)
    if (decoder.messageName() == Messages::WebPasteboardProxy::WriteImageToPasteboard::name())
        return IPC::handleMessageWantsConnection<Messages::WebPasteboardProxy::WriteImageToPasteboard>(connection, decoder, this, &WebPasteboardProxy::writeImageToPasteboard);
#endif
#if PLATFORM(IOS_FAMILY)
    if (decoder.messageName() == Messages::WebPasteboardProxy::WriteStringToPasteboard::name())
        return IPC::handleMessageWantsConnection<Messages::WebPasteboardProxy::WriteStringToPasteboard>(connection, decoder, this, &WebPasteboardProxy::writeStringToPasteboard);
#endif
#if PLATFORM(IOS_FAMILY)
    if (decoder.messageName() == Messages::WebPasteboardProxy::UpdateSupportedTypeIdentifiers::name())
        return IPC::handleMessage<Messages::WebPasteboardProxy::UpdateSupportedTypeIdentifiers>(connection, decoder, this, &WebPasteboardProxy::updateSupportedTypeIdentifiers);
#endif
#if PLATFORM(GTK)
    if (decoder.messageName() == Messages::WebPasteboardProxy::WriteToClipboard::name())
        return IPC::handleMessage<Messages::WebPasteboardProxy::WriteToClipboard>(connection, decoder, this, &WebPasteboardProxy::writeToClipboard);
#endif
#if PLATFORM(GTK)
    if (decoder.messageName() == Messages::WebPasteboardProxy::ClearClipboard::name())
        return IPC::handleMessage<Messages::WebPasteboardProxy::ClearClipboard>(connection, decoder, this, &WebPasteboardProxy::clearClipboard);
#endif
#if USE(LIBWPE)
    if (decoder.messageName() == Messages::WebPasteboardProxy::WriteWebContentToPasteboard::name())
        return IPC::handleMessage<Messages::WebPasteboardProxy::WriteWebContentToPasteboard>(connection, decoder, this, &WebPasteboardProxy::writeWebContentToPasteboard);
#endif
#if USE(LIBWPE)
    if (decoder.messageName() == Messages::WebPasteboardProxy::WriteStringToPasteboard::name())
        return IPC::handleMessage<Messages::WebPasteboardProxy::WriteStringToPasteboard>(connection, decoder, this, &WebPasteboardProxy::writeStringToPasteboard);
#endif
    UNUSED_PARAM(connection);
    UNUSED_PARAM(decoder);
#if ENABLE(IPC_TESTING_API)
    if (connection.ignoreInvalidMessageForTesting())
        return;
#endif // ENABLE(IPC_TESTING_API)
    ASSERT_NOT_REACHED_WITH_MESSAGE("Unhandled message %s to %" PRIu64, IPC::description(decoder.messageName()), decoder.destinationID());
}

bool WebPasteboardProxy::didReceiveSyncMessage(IPC::Connection& connection, IPC::Decoder& decoder, UniqueRef<IPC::Encoder>& replyEncoder)
{
    if (decoder.messageName() == Messages::WebPasteboardProxy::WriteCustomData::name())
        return IPC::handleMessageSynchronousWantsConnection<Messages::WebPasteboardProxy::WriteCustomData>(connection, decoder, replyEncoder, this, &WebPasteboardProxy::writeCustomData);
    if (decoder.messageName() == Messages::WebPasteboardProxy::TypesSafeForDOMToReadAndWrite::name())
        return IPC::handleMessageSynchronousWantsConnection<Messages::WebPasteboardProxy::TypesSafeForDOMToReadAndWrite>(connection, decoder, replyEncoder, this, &WebPasteboardProxy::typesSafeForDOMToReadAndWrite);
    if (decoder.messageName() == Messages::WebPasteboardProxy::AllPasteboardItemInfo::name())
        return IPC::handleMessageSynchronousWantsConnection<Messages::WebPasteboardProxy::AllPasteboardItemInfo>(connection, decoder, replyEncoder, this, &WebPasteboardProxy::allPasteboardItemInfo);
    if (decoder.messageName() == Messages::WebPasteboardProxy::InformationForItemAtIndex::name())
        return IPC::handleMessageSynchronousWantsConnection<Messages::WebPasteboardProxy::InformationForItemAtIndex>(connection, decoder, replyEncoder, this, &WebPasteboardProxy::informationForItemAtIndex);
    if (decoder.messageName() == Messages::WebPasteboardProxy::GetPasteboardItemsCount::name())
        return IPC::handleMessageSynchronousWantsConnection<Messages::WebPasteboardProxy::GetPasteboardItemsCount>(connection, decoder, replyEncoder, this, &WebPasteboardProxy::getPasteboardItemsCount);
    if (decoder.messageName() == Messages::WebPasteboardProxy::ReadStringFromPasteboard::name())
        return IPC::handleMessageSynchronousWantsConnection<Messages::WebPasteboardProxy::ReadStringFromPasteboard>(connection, decoder, replyEncoder, this, &WebPasteboardProxy::readStringFromPasteboard);
    if (decoder.messageName() == Messages::WebPasteboardProxy::ReadURLFromPasteboard::name())
        return IPC::handleMessageSynchronousWantsConnection<Messages::WebPasteboardProxy::ReadURLFromPasteboard>(connection, decoder, replyEncoder, this, &WebPasteboardProxy::readURLFromPasteboard);
    if (decoder.messageName() == Messages::WebPasteboardProxy::ReadBufferFromPasteboard::name())
        return IPC::handleMessageSynchronousWantsConnection<Messages::WebPasteboardProxy::ReadBufferFromPasteboard>(connection, decoder, replyEncoder, this, &WebPasteboardProxy::readBufferFromPasteboard);
    if (decoder.messageName() == Messages::WebPasteboardProxy::ContainsStringSafeForDOMToReadForType::name())
        return IPC::handleMessageSynchronousWantsConnection<Messages::WebPasteboardProxy::ContainsStringSafeForDOMToReadForType>(connection, decoder, replyEncoder, this, &WebPasteboardProxy::containsStringSafeForDOMToReadForType);
#if PLATFORM(COCOA)
    if (decoder.messageName() == Messages::WebPasteboardProxy::GetNumberOfFiles::name())
        return IPC::handleMessageSynchronousWantsConnection<Messages::WebPasteboardProxy::GetNumberOfFiles>(connection, decoder, replyEncoder, this, &WebPasteboardProxy::getNumberOfFiles);
#endif
#if PLATFORM(COCOA)
    if (decoder.messageName() == Messages::WebPasteboardProxy::GetPasteboardTypes::name())
        return IPC::handleMessageSynchronousWantsConnection<Messages::WebPasteboardProxy::GetPasteboardTypes>(connection, decoder, replyEncoder, this, &WebPasteboardProxy::getPasteboardTypes);
#endif
#if PLATFORM(COCOA)
    if (decoder.messageName() == Messages::WebPasteboardProxy::GetPasteboardPathnamesForType::name())
        return IPC::handleMessageSynchronousWantsConnection<Messages::WebPasteboardProxy::GetPasteboardPathnamesForType>(connection, decoder, replyEncoder, this, &WebPasteboardProxy::getPasteboardPathnamesForType);
#endif
#if PLATFORM(COCOA)
    if (decoder.messageName() == Messages::WebPasteboardProxy::GetPasteboardStringForType::name())
        return IPC::handleMessageSynchronousWantsConnection<Messages::WebPasteboardProxy::GetPasteboardStringForType>(connection, decoder, replyEncoder, this, &WebPasteboardProxy::getPasteboardStringForType);
#endif
#if PLATFORM(COCOA)
    if (decoder.messageName() == Messages::WebPasteboardProxy::GetPasteboardStringsForType::name())
        return IPC::handleMessageSynchronousWantsConnection<Messages::WebPasteboardProxy::GetPasteboardStringsForType>(connection, decoder, replyEncoder, this, &WebPasteboardProxy::getPasteboardStringsForType);
#endif
#if PLATFORM(COCOA)
    if (decoder.messageName() == Messages::WebPasteboardProxy::GetPasteboardBufferForType::name())
        return IPC::handleMessageSynchronousWantsConnection<Messages::WebPasteboardProxy::GetPasteboardBufferForType>(connection, decoder, replyEncoder, this, &WebPasteboardProxy::getPasteboardBufferForType);
#endif
#if PLATFORM(COCOA)
    if (decoder.messageName() == Messages::WebPasteboardProxy::GetPasteboardChangeCount::name())
        return IPC::handleMessageSynchronousWantsConnection<Messages::WebPasteboardProxy::GetPasteboardChangeCount>(connection, decoder, replyEncoder, this, &WebPasteboardProxy::getPasteboardChangeCount);
#endif
#if PLATFORM(COCOA)
    if (decoder.messageName() == Messages::WebPasteboardProxy::GetPasteboardColor::name())
        return IPC::handleMessageSynchronousWantsConnection<Messages::WebPasteboardProxy::GetPasteboardColor>(connection, decoder, replyEncoder, this, &WebPasteboardProxy::getPasteboardColor);
#endif
#if PLATFORM(COCOA)
    if (decoder.messageName() == Messages::WebPasteboardProxy::GetPasteboardURL::name())
        return IPC::handleMessageSynchronousWantsConnection<Messages::WebPasteboardProxy::GetPasteboardURL>(connection, decoder, replyEncoder, this, &WebPasteboardProxy::getPasteboardURL);
#endif
#if PLATFORM(COCOA)
    if (decoder.messageName() == Messages::WebPasteboardProxy::AddPasteboardTypes::name())
        return IPC::handleMessageSynchronousWantsConnection<Messages::WebPasteboardProxy::AddPasteboardTypes>(connection, decoder, replyEncoder, this, &WebPasteboardProxy::addPasteboardTypes);
#endif
#if PLATFORM(COCOA)
    if (decoder.messageName() == Messages::WebPasteboardProxy::SetPasteboardTypes::name())
        return IPC::handleMessageSynchronousWantsConnection<Messages::WebPasteboardProxy::SetPasteboardTypes>(connection, decoder, replyEncoder, this, &WebPasteboardProxy::setPasteboardTypes);
#endif
#if PLATFORM(COCOA)
    if (decoder.messageName() == Messages::WebPasteboardProxy::SetPasteboardURL::name())
        return IPC::handleMessageSynchronousWantsConnection<Messages::WebPasteboardProxy::SetPasteboardURL>(connection, decoder, replyEncoder, this, &WebPasteboardProxy::setPasteboardURL);
#endif
#if PLATFORM(COCOA)
    if (decoder.messageName() == Messages::WebPasteboardProxy::SetPasteboardColor::name())
        return IPC::handleMessageSynchronousWantsConnection<Messages::WebPasteboardProxy::SetPasteboardColor>(connection, decoder, replyEncoder, this, &WebPasteboardProxy::setPasteboardColor);
#endif
#if PLATFORM(COCOA)
    if (decoder.messageName() == Messages::WebPasteboardProxy::SetPasteboardStringForType::name())
        return IPC::handleMessageSynchronousWantsConnection<Messages::WebPasteboardProxy::SetPasteboardStringForType>(connection, decoder, replyEncoder, this, &WebPasteboardProxy::setPasteboardStringForType);
#endif
#if PLATFORM(COCOA)
    if (decoder.messageName() == Messages::WebPasteboardProxy::SetPasteboardBufferForType::name())
        return IPC::handleMessageSynchronousWantsConnection<Messages::WebPasteboardProxy::SetPasteboardBufferForType>(connection, decoder, replyEncoder, this, &WebPasteboardProxy::setPasteboardBufferForType);
#endif
#if PLATFORM(COCOA)
    if (decoder.messageName() == Messages::WebPasteboardProxy::ContainsURLStringSuitableForLoading::name())
        return IPC::handleMessageSynchronousWantsConnection<Messages::WebPasteboardProxy::ContainsURLStringSuitableForLoading>(connection, decoder, replyEncoder, this, &WebPasteboardProxy::containsURLStringSuitableForLoading);
#endif
#if PLATFORM(COCOA)
    if (decoder.messageName() == Messages::WebPasteboardProxy::URLStringSuitableForLoading::name())
        return IPC::handleMessageSynchronousWantsConnection<Messages::WebPasteboardProxy::URLStringSuitableForLoading>(connection, decoder, replyEncoder, this, &WebPasteboardProxy::urlStringSuitableForLoading);
#endif
#if PLATFORM(GTK)
    if (decoder.messageName() == Messages::WebPasteboardProxy::GetTypes::name())
        return IPC::handleMessageSynchronous<Messages::WebPasteboardProxy::GetTypes>(connection, decoder, replyEncoder, this, &WebPasteboardProxy::getTypes);
#endif
#if PLATFORM(GTK)
    if (decoder.messageName() == Messages::WebPasteboardProxy::ReadText::name())
        return IPC::handleMessageSynchronous<Messages::WebPasteboardProxy::ReadText>(connection, decoder, replyEncoder, this, &WebPasteboardProxy::readText);
#endif
#if PLATFORM(GTK)
    if (decoder.messageName() == Messages::WebPasteboardProxy::ReadFilePaths::name())
        return IPC::handleMessageSynchronous<Messages::WebPasteboardProxy::ReadFilePaths>(connection, decoder, replyEncoder, this, &WebPasteboardProxy::readFilePaths);
#endif
#if PLATFORM(GTK)
    if (decoder.messageName() == Messages::WebPasteboardProxy::ReadBuffer::name())
        return IPC::handleMessageSynchronous<Messages::WebPasteboardProxy::ReadBuffer>(connection, decoder, replyEncoder, this, &WebPasteboardProxy::readBuffer);
#endif
#if USE(LIBWPE)
    if (decoder.messageName() == Messages::WebPasteboardProxy::GetPasteboardTypes::name())
        return IPC::handleMessageSynchronous<Messages::WebPasteboardProxy::GetPasteboardTypes>(connection, decoder, replyEncoder, this, &WebPasteboardProxy::getPasteboardTypes);
#endif
    UNUSED_PARAM(connection);
    UNUSED_PARAM(decoder);
    UNUSED_PARAM(replyEncoder);
#if ENABLE(IPC_TESTING_API)
    if (connection.ignoreInvalidMessageForTesting())
        return false;
#endif // ENABLE(IPC_TESTING_API)
    ASSERT_NOT_REACHED_WITH_MESSAGE("Unhandled synchronous message %s to %" PRIu64, description(decoder.messageName()), decoder.destinationID());
    return false;
}

} // namespace WebKit
