"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const chalk_1 = __importDefault(require("chalk"));
const prompts_ncu_1 = __importDefault(require("prompts-ncu"));
const semver_1 = require("semver");
const logging_1 = require("../logging");
const keyValueBy_1 = __importDefault(require("../lib/keyValueBy"));
const version_util_1 = require("../version-util");
/**
 * @returns String safe for use in `new RegExp()`
 */
function escapeRegexp(s) {
    return s.replace(/[-/\\^$*+?.()|[\]{}]/g, '\\$&'); // Thanks Stack Overflow!
}
/**
 * Upgrade the dependency declarations in the package data.
 *
 * @param pkgData The package.json data, as utf8 text
 * @param oldDependencies Old dependencies {package: range}
 * @param newDependencies New dependencies {package: range}
 * @param newVersions New versions {package: version}
 * @param [options={}]
 * @returns The updated package data, as utf8 text
 * @description Side Effect: prompts
 */
async function upgradePackageData(pkgData, oldDependencies, newDependencies, newVersions, options = {}) {
    var _a;
    const chalk = options.color ? new chalk_1.default.Instance({ level: 1 }) : chalk_1.default;
    let newPkgData = pkgData;
    // interactive mode needs a newline before prompts
    if (options.interactive) {
        (0, logging_1.print)(options, '');
    }
    let newDependenciesFiltered = (0, keyValueBy_1.default)(newDependencies, (dep, version) => !options.minimal || !(0, semver_1.satisfies)(newVersions[dep], oldDependencies[dep]) ? { [dep]: version } : null);
    if (options.interactive) {
        // use toDependencyTable to create choices that are properly padded to align vertically
        const table = (0, logging_1.toDependencyTable)({
            from: oldDependencies,
            to: newDependencies,
            format: options.format,
        });
        const formattedLines = (0, keyValueBy_1.default)(table.toString().split('\n'), line => {
            const dep = line.trim().split(' ')[0];
            return {
                [dep]: line.trim(),
            };
        });
        let depsSelected = [];
        // do not prompt if there are no dependencies
        // prompts will crash if passed an empty list of choices
        if (Object.keys(newDependenciesFiltered).length > 0) {
            if ((_a = options.format) === null || _a === void 0 ? void 0 : _a.includes('group')) {
                const groups = (0, keyValueBy_1.default)(newDependenciesFiltered, (dep, to, accum) => {
                    const from = oldDependencies[dep];
                    const partUpgraded = (0, version_util_1.partChanged)(from, to);
                    return {
                        ...accum,
                        [partUpgraded]: {
                            ...accum[partUpgraded],
                            [dep]: to,
                        },
                    };
                });
                const choicesPatch = Object.keys(groups.patch || {}).map(dep => ({
                    title: formattedLines[dep],
                    value: dep,
                    selected: true,
                }));
                const choicesMinor = Object.keys(groups.minor || {}).map(dep => ({
                    title: formattedLines[dep],
                    value: dep,
                    selected: true,
                }));
                const choicesMajor = Object.keys(groups.major || {}).map(dep => ({
                    title: formattedLines[dep],
                    value: dep,
                    selected: false,
                }));
                const choicesNonsemver = Object.keys(groups.nonsemver || {}).map(dep => ({
                    title: formattedLines[dep],
                    value: dep,
                    selected: false,
                }));
                const response = await (0, prompts_ncu_1.default)({
                    choices: [
                        ...(choicesPatch.length > 0
                            ? [
                                {
                                    title: '\n' + chalk.green(chalk.bold('Patch') + '   Backwards-compatible bug fixes'),
                                    heading: true,
                                },
                            ]
                            : []),
                        ...choicesPatch,
                        ...(choicesMinor.length > 0
                            ? [{ title: '\n' + chalk.cyan(chalk.bold('Minor') + '   Backwards-compatible features'), heading: true }]
                            : []),
                        ...choicesMinor,
                        ...(choicesMajor.length > 0
                            ? [
                                {
                                    title: '\n' + chalk.red(chalk.bold('Major') + '   Potentially breaking API changes'),
                                    heading: true,
                                },
                            ]
                            : []),
                        ...choicesMajor,
                        ...(choicesNonsemver.length > 0
                            ? [
                                {
                                    title: '\n' + chalk.magenta(chalk.bold('Non-Semver') + '  Versions less than 1.0.0'),
                                    heading: true,
                                },
                            ]
                            : []),
                        ...choicesNonsemver,
                        { title: ' ', heading: true },
                    ],
                    hint: `
  ↑/↓: Select a package
  Space: Toggle selection
  a: Select all
  Enter: Upgrade`,
                    instructions: false,
                    message: 'Choose which packages to update',
                    name: 'value',
                    optionsPerPage: 50,
                    type: 'multiselect',
                    onState: (state) => {
                        if (state.aborted) {
                            process.nextTick(() => process.exit(1));
                        }
                    },
                });
                depsSelected = response.value;
            }
            else {
                const choices = Object.keys(newDependenciesFiltered).map(dep => ({
                    title: formattedLines[dep],
                    value: dep,
                    selected: true,
                }));
                const response = await (0, prompts_ncu_1.default)({
                    choices,
                    hint: 'Space to deselect. Enter to upgrade.',
                    instructions: false,
                    message: 'Choose which packages to update',
                    name: 'value',
                    optionsPerPage: 50,
                    type: 'multiselect',
                    onState: (state) => {
                        if (state.aborted) {
                            process.nextTick(() => process.exit(1));
                        }
                    },
                });
                depsSelected = response.value;
            }
        }
        newDependenciesFiltered = (0, keyValueBy_1.default)(depsSelected, (dep) => ({ [dep]: newDependencies[dep] }));
        // in interactive mode, do not group upgrades afterwards since the prompts are grouped
        (0, logging_1.printUpgrades)({ ...options, format: (options.format || []).filter(formatType => formatType !== 'group') }, {
            current: oldDependencies,
            upgraded: newDependenciesFiltered,
            total: Object.keys(newDependencies).length,
        });
    }
    // eslint-disable-next-line fp/no-loops
    for (const dependency in newDependenciesFiltered) {
        const expression = `"${dependency}"\\s*:\\s*"${escapeRegexp(`${oldDependencies[dependency]}"`)}`;
        const regExp = new RegExp(expression, 'g');
        newPkgData = newPkgData.replace(regExp, `"${dependency}": "${newDependencies[dependency]}"`);
    }
    return newPkgData;
}
exports.default = upgradePackageData;
//# sourceMappingURL=upgradePackageData.js.map