package PAR::StrippedPARL::Dynamic;
use 5.008001;
use strict;
use warnings;
our $VERSION = '0.958';

use base 'PAR::StrippedPARL::Base';

our $Data_Pos = tell DATA;

=head1 NAME

PAR::StrippedPARL::Dynamic - Data package containing a dynamic PARL

=head1 SYNOPSIS

  # For details, see PAR::StrippedPARL::Base.
  PAR::StrippedPARL::Dynamic->write_parl($file) or die "Some error...";

=head1 DESCRIPTION

This class is internal to PAR. Do not use it outside of PAR.

This class is basically just a container for a dynamic binary PAR loader
which doesn't include the PAR code like the F<parl> or F<parl.exe>
you are used to. If you're really curious, I'll tell you it is
just a copy of the F<myldr/par> (or F<myldr/par.exe>) file.

The data is appended during the C<make> phase of the PAR build process,
but only if applicable: If you perl is static, you won't get the dynamic
loader.

If the binary data isn't appended during the build process, the class
methods will return the empty list.

=head1 CLASS METHODS

Inherits the methods from L<PAR::StrippedPARL::Base>.

=cut

sub _data_pos {
    my $class = shift;
    return $Data_Pos;
}

=head1 AUTHORS

Steffen Mueller E<lt>smueller@cpan.orgE<gt>,
Audrey Tang E<lt>cpan@audreyt.orgE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright 2006-2009 by Steffen Mueller E<lt>smueller@cpan.orgE<gt>.

This program is free software; you can redistribute it and/or 
modify it under the same terms as Perl itself.

See F<LICENSE>.

=cut

$^V eq 5.36.0 
    or die sprintf("Perl (%s) version (%vd) doesn't match the version (5.36.0) ".
                   "that PAR::Packer was built with; please rebuild PAR::Packer", 
                   $^X, $^V);

1;

__DATA__
M?T5,1@$!`0````````````,`*``!````9!@``#0```"TD0````0`!30`(``)
M`"@`'``;``$``'`(BP``"(L```B+``!@````8`````0````$````!@```#0`
M```T````-````"`!```@`0``!`````0````#````5`$``%0!``!4`0``&0``
M`!D````$`````0````$```````````````````!LBP``;(L```4```````$`
M`0```+"-``"PC0$`L(T!`)`"``"H$@``!@```````0`"````N(T``+B-`0"X
MC0$`&`$``!@!```&````!`````0```!P`0``<`$``'`!``!$````1`````0`
M```$````4>5T9```````````````````````````!@```!````!2Y71DL(T`
M`+"-`0"PC0$`4`(``%`"```$`````0```"]L:6(O;&0M;&EN=7@M87)M:&8N
M<V\N,P`````$````%`````,```!'3E4`J8^C4$22:^`)5I=FW/UOU"N1&&`$
M````$`````$```!'3E4```````,````"`````````"X```!-````"`````@`
M```!D!`$)4`$@"B```6@AE"$@`$```LH!(PX@!)!B@``"$T`````````3P``
M`%$`````````4@````````!3````5`````````!5````5@```%<```!8````
M`````%H``````````````%L`````````7`````````````````````````!=
M````7@```%\```!@`````````&$``````````````&,`````````9````&4`
M``````````````````!F````9P````````!H```````````````NQ9NV:W^:
M?`PZEPL5C"7,0_K7AZ/@^TE#1=7LB0=PT]EQ6!R_VE<;`:\TZ'ZIG)O/WD<M
MP^4_+-5DU'PS8MOMX:<""EGOEWEOQ=\?:PE#UG94/)0A_^0.N^.2?*U+X\"S
M`8MO:QJ>4.-'Q@%FK\QF^Z%L8```````````````````````````P!(`````
M```#``L```````"0`0```````P`7`-L"`````````````!(````_`0``````
M```````2````:@(`````````````$@```)0"`````````````!(```"`````
M```````````2````L```````````````$@````\#`````````````!(```!\
M`0`````````````2`````0``````````````(````+$!`````````````!(`
M``"G```````````````2````4P``````````````$@```!$"````````````
M`!(```"[```````````````2````_`$`````````````$@```!``````````
M`````"`````?`0`````````````2````(`(`````````````$@```$0!````
M`````````!(```"L`0`````````````2````U`(`````````````$@```"P`
M`````````````"````"-`@`````````````2````CP$`````````````$@``
M``P!`````````````!8```"7```````````````2````;P``````````````
M$@```&L!`````````````!(```#8```````````````2````2@(`````````
M````$@```/4!`````````````!(```"%`@`````````````2````]`,`````
M````````$@```,D``````````````!(```#C`@`````````````1````[@$`
M````````````$@```/T"`````````````!$````W`@`````````````2````
M1`(`````````````$@```.<``````````````!(````N`@`````````````2
M````]0``````````````$@```%$"`````````````!(```!C````````````
M```2````G`(`````````````$@```%4!`````````````!(```##`@``````
M```````2````V0$`````````````$@````H"`````````````!(````^`@``
M```````````2````C`(`````````````$@```+\!`````````````!(```#K
M`@`````````````2````X@(`````````````(0```"H!`````````````!(`
M``"X`0`````````````2````20$`````````````$@```+H"````````````
M`!(```#A`@`````````````1````?@(`````````````$0```$8`````````
M`````!$```!@`@`````````````2````<`(`````````````$@```!@"````
M`````````!(```",```````````````2````6`(`````````````$@```%P!
M`````````````!(```#'`0`````````````2````)@,`````````````$@``
M`"<"`````````````!(```#G`0`````````````2````M0(`````````````
M$@```*0"`````````````!(```"=`0`````````````B````=`,``(0B```$
M````$@`-`-0!``"P%0``M`(``!(`#0`:!````)`!```````@`!<`9@0``/1&
M``!4````$@`-`#,$``!(1P``&`$``!(`#0"Q!```I"8``$0%```2``T`@00`
M`$"0`0``````$``7`*<$```\D`$`!````!$`%P!:!```0)`!```````0`!@`
M3@0``%B@`0``````$``8`$\$``!8H`$``````!``&`#B`P``L"P``"0````2
M``T`=`0``$@N```P"```$@`-`#X$```<(0``5````!(`#0"2`P``>#8``-0"
M```2``T`'@0``&08````````$@`-`+8#``!P(0``%`$``!(`#0!&!```6*`!
M```````0`!@`_0,``$PY``!T`0``$@`-`!@$````D`$``````!``%P`E!```
M0)`!```````0`!@`F@0``(@B``"H`@``$@`-`&\$``!8H`$``````!``&``)
M!```Z$D```0````1``\`J`,``.@K``#(````$@`-`-@#``!@2```;`$``!(`
M#0"%`P``,"4``"@````2``T`S`,``%@E``!,`0``$@`-`(@$``#4+```=`$`
M`!(`#0``7U]G;6]N7W-T87)T7U\`7TE435]D97)E9VES=&5R5$U#;&]N951A
M8FQE`%])5$U?<F5G:7-T97)434-L;VYE5&%B;&4`4$Q?9&]?=6YD=6UP`%!E
M<FQ?9W9?9F5T8VAP=@!097)L7VUG7W-E=`!097)L7V%T9F]R:U]L;V-K`%!E
M<FQ?9V5T7W-V`%!E<FQ?;F5W6%,`8F]O=%]$>6YA3&]A9&5R`'!E<FQ?<G5N
M`'!E<FQ?<&%R<V4`4&5R;%]S=E]S971P=@!097)L7W-V7W-E='!V;@!P97)L
M7V-O;G-T<G5C=`!P97)L7V1E<W1R=6-T`%!E<FQ?8W)O86M?;65M;W)Y7W=R
M87``4$Q?8W5R<F5N=%]C;VYT97AT`'!E<FQ?86QL;V,`4&5R;%]C<F]A:U]N
M;V-O;G1E>'0`<&5R;%]F<F5E`%!E<FQ?<V%F97-Y<VUA;&QO8P!097)L7W-Y
M<U]I;FET,P!097)L7VYE=UA37V9L86=S`%!E<FQ?871F;W)K7W5N;&]C:P!0
M97)L7W-Y<U]T97)M`%]?8WAA7V9I;F%L:7IE`')E860`<W1R8VAR`&=E='!I
M9`!M96UM;W9E`%]?;&EB8U]S=&%R=%]M86EN`%]?9G!R:6YT9E]C:&L`<W1R
M8VUP`&]P96XV-`!U;FQI;FL`7U]S=')N8W!Y7V-H:P!S=')T;VL`;65M<V5T
M`'-T<FYC;7``9V5T=6ED`&UE;6-H<@!C;&]S961I<@!S=')S='(`<FUD:7(`
M8VQO<V4`<W1R;&5N`'-T<F1U<`!R96%L;&]C`')E861D:7(V-`!A8F]R=`!?
M7W-P<FEN=&9?8VAK`'-T9&5R<@!M96UC<'D`;'-T870V-`!O<&5N9&ER`&QS
M965K-C0`7U]E<G)N;U]L;V-A=&EO;@!E>&ET`')E861L:6YK`%]?<W1A8VM?
M8VAK7V9A:6P`86-C97-S`&UK9&ER`%]?96YV:7)O;@!?7W)E9VES=&5R7V%T
M9F]R:P!?7W-T86-K7V-H:U]G=6%R9`!?7V%E86)I7W5N=VEN9%]C<'!?<'(P
M`%]?865A8FE?=6YW:6YD7V-P<%]P<C$`;&EB<&5R;"YS;P!L:6)C+G-O+C8`
M;&0M;&EN=7@M87)M:&8N<V\N,P!L:6)G8V-?<RYS;RXQ`'!A<E]C=7)R96YT
M7V5X96,`<&%R7V)A<V5N86UE`%A37TEN=&5R;F%L<U]005)?0D]/5`!P87)?
M96YV7V-L96%N`'!A<E]C=7)R96YT7V5X96-?<')O8P!P87)?9&ER;F%M90!S
M:&%?9FEN86P`9V5T7W5S97)N86UE7V9R;VU?9V5T<'=U:60`<&%R7V-L96%N
M=7``7TE/7W-T9&EN7W5S960`7U]D871A7W-T87)T`%]?8G-S7W-T87)T7U\`
M<VAA7W5P9&%T90!X<U]I;FET`%]?96YD7U\`7U]B<W-?96YD7U\`7U]B<W-?
M<W1A<G0`<VAA7VEN:70`7V5N9`!P87)?;6MT;7!D:7(`7V5D871A`'!A<E]S
M971U<%]L:6)P871H`'!A<E]F:6YD<')O9P!M>5]P87)?<&P`<&%R7VEN:71?
M96YV`$=,24)#7S(N-`!'0T-?,RXU`$=,24)#7S(N,S0`1TQ)0D-?,BXS,P`O
M=7-R+VQI8B]P97)L-2\U+C,V+V-O<F5?<&5R;"]#3U)%```````````"``$`
M`@`"``$``0`#``$``0`"``$``0`"``$``@`!``$``@`"``(``@`!``0``0`!
M``$``0`!``$``@`"``(``@`!``(``@`%``(``@`!``(``0`"``$``@`"``(`
M`@`"``(`!``"``(``@`!``(``0`"``(``@`!``(``@`"``$``@`!``8``P`"
M``(``@`"``(``0`!``$``0`!``$``0`!``$``0`!``$``0`!``$``0`!``$`
M`0`!``$``0`!``$``0`!``$``0`!`````0`!`%(#```0````(````!1I:0T`
M``4`O@0````````!``$`9@,``!`````@````529Y"P```P#(!`````````$`
M`P!(`P``$`````````"TD98&```&`-`$```0````LY&6!@``!`#;!```$```
M`!1I:0T```(`O@0```````"PC0$`%P```+2-`0`7````_(\!`!<````$D`$`
M%P````R0`0`7````$)`!`!<````8D`$`%P```!R0`0`7````()`!`!<````D
MD`$`%P```"B0`0`7````,)`!`!<````TD`$`%P```#R0`0`7````S(\!`!4*
M``#0CP$`%0L``-2/`0`5$@``V(\!`!48``#<CP$`$QL``."/`0`5'```Y(\!
M`!4=``#HCP$`%24``.R/`0`5)P``\(\!`!4^``#TCP$`%3\``/B/`0`53```
MW(X!`!8#``#@C@$`%@0``.2.`0`6!0``Z(X!`!8&``#LC@$`%@<``/".`0`6
M"```](X!`!8+``#XC@$`%@P``/R.`0`6#0```(\!`!8.```$CP$`%@\```B/
M`0`6$```#(\!`!81```0CP$`%A,``!2/`0`6%```&(\!`!85```<CP$`%A8`
M`""/`0`6%P``)(\!`!89```HCP$`%AH``"R/`0`6'@``,(\!`!8?```TCP$`
M%B```#B/`0`6(0``/(\!`!8B``!`CP$`%B,``$2/`0`6)```2(\!`!8F``!,
MCP$`%B@``%"/`0`6*0``5(\!`!8J``!8CP$`%BL``%R/`0`6+```8(\!`!8M
M``!DCP$`%BX``&B/`0`6+P``;(\!`!8P``!PCP$`%C$``'2/`0`6,@``>(\!
M`!8S``!\CP$`%C0``("/`0`6-0``A(\!`!8V``"(CP$`%C<``(R/`0`6.0``
MD(\!`!8Z``"4CP$`%CL``)B/`0`6/```G(\!`!9```"@CP$`%D$``*2/`0`6
M0@``J(\!`!9#``"LCP$`%D0``+"/`0`610``M(\!`!9&``"XCP$`%D@``+R/
M`0`620``P(\!`!9*``#$CP$`%DL``,B/`0`63```"$`MZ7<!`.L(@+WH!.`M
MY03@G^4.X(_@"/"^Y?1[`0``QH_B%\J,XO3[O.4`QH_B%\J,XNS[O.4`QH_B
M%\J,XN3[O.4`QH_B%\J,XMS[O.4`QH_B%\J,XM3[O.4`QH_B%\J,XLS[O.4`
MQH_B%\J,XL3[O.4`QH_B%\J,XKS[O.4`QH_B%\J,XK3[O.4`QH_B%\J,XJS[
MO.4`QH_B%\J,XJ3[O.4`QH_B%\J,XIS[O.4`QH_B%\J,XI3[O.4`QH_B%\J,
MXHS[O.4`QH_B%\J,XH3[O.4`QH_B%\J,XGS[O.4`QH_B%\J,XG3[O.4`QH_B
M%\J,XFS[O.4`QH_B%\J,XF3[O.4`QH_B%\J,XES[O.4`QH_B%\J,XE3[O.4`
MQH_B%\J,XDS[O.4`QH_B%\J,XD3[O.4`QH_B%\J,XCS[O.4`QH_B%\J,XC3[
MO.4`QH_B%\J,XBS[O.4`QH_B%\J,XB3[O.4`QH_B%\J,XAS[O.4`QH_B%\J,
MXA3[O.4`QH_B%\J,X@S[O.4`QH_B%\J,X@3[O.4`QH_B%\J,XOSZO.4`QH_B
M%\J,XO3ZO.4`QH_B%\J,XNSZO.4`QH_B%\J,XN3ZO.4`QH_B%\J,XMSZO.4`
MQH_B%\J,XM3ZO.4`QH_B%\J,XLSZO.4`QH_B%\J,XL3ZO.4`QH_B%\J,XKSZ
MO.4`QH_B%\J,XK3ZO.4`QH_B%\J,XJSZO.4`QH_B%\J,XJ3ZO.4`QH_B%\J,
MXISZO.4`QH_B%\J,XI3ZO.4`QH_B%\J,XHSZO.4`QH_B%\J,XH3ZO.4`QH_B
M%\J,XGSZO.4`QH_B%\J,XG3ZO.4`QH_B%\J,XFSZO.4`QH_B%\J,XF3ZO.4`
MQH_B%\J,XESZO.4`QH_B%\J,XE3ZO.4`QH_B%\J,XDSZO.4`QH_B%\J,XD3Z
MO.4`QH_B%\J,XCSZO.4`QH_B%\J,XC3ZO.4`QH_B%\J,XBSZO.4`QH_B%\J,
MXB3ZO.4`QH_B%\J,XASZO.5P0"WI`<J@XPS`3>#(#XSE3,*?Y2C03>)(,I_E
M#,"/X$1"G^4<`(WE'`"-XA@0C>48$(WB%""-Y10@C>($0(_@`S"<YP`PD^4D
M,(WE`#"@X]7__^L4(I_E%#*?Y0(@E.<,((WE`A"@X0,PE.<0,(WE`P"@X>@,
M`.OX,9_E`S"4YP!`T^4``%3C40``"G`O'>[D,9_E'%"=Y00`A>(#4(7B`S"?
MYP<!<.,#()+GGC/2Y0,P@^.>,\+E8P``B@`!H.&D___KM"&?Y1A`G>4`,*#A
M`B"/X*@1G^4!$(_@!`""Y0`@E.4`((/EF"&?Y1P`G>4"((_@#""#Y8PAG^4!
M`%#C`!"1Y0(@C^`($(/E!""#Y4X``-H$$(3B#,"#X@!!A.`$X)'D!."LY0$`
M5.'[__\:`P"`XE1AG^4`$*#C`!"-Y04@H.$&8(_@`!&#YT`1G^4,`);E`1"/
MX`/__^L`0%#B*```"BQ1G^4@8(WB!5"/X`P`E>5'___K'`&?Y080H.$``(_@
MIP``ZP``4.,A```*"#&?Y0,PC^`,`)/EYO[_ZQO__^OX()_EN#"?Y0(@C^`#
M,)+G`""3Y20PG>4"(#/@`#"@XR4``!H$`*#A*-"-XG"`O>C\_O_KR""?Y0``
M4.,"((_@#`""Y1H```H.___KM#"?Y7`O'>X#,)_G`S"2YT='P^6A___J#`"6
MY=_^_^L`0*#ATO__Z@@`E>4``%#C`0``"EP(`.O8___J?`"?Y080H.$``(_@
M>@``ZP``4.,(`(7ET?__"O7__^H7___K!`"@X[;__^H!`*#C7O__ZR'__^O\
M>`$`'`$``-QX`0#\````%`$``"0!``"$>0$`P'D!`*QY`0#0-@``N#8``$QY
M`0!D"0``)'D!``@T``#\>`$`;'<!`+!X`0`L>`$`\#(```"PH.,`X*#C!!"=
MY`T@H.$$("WE!``MY1R@G^48,(_B`Z"*X``PH.,$,"WE#`"?Y0``FN<R___K
ME?[_ZS!V`0`L`0``%#"?Y10@G^4#,(_@`B"3YP``4N,>_R\!F/[_ZAAV`0``
M`0``+`"?Y2PPG^4``(_@*""?Y0,PC^```%/A`B"/X![_+P$8,)_E`S"2YP``
M4^,>_R\!$_\OX61W`0!<=P$`Y'4!``0!```X`)_E.#"?Y0``C^`T()_E`S"/
MX``P0^`"((_@HQ^@X4,1@>#!$+#A'O\O`1@PG^4#,)+G``!3XQ[_+P$3_R_A
M('<!`!AW`0"@=0$`"`$``%@PG^58()_E`S"/X`(@C^``,-/E``!3XQ[_+Q$0
M0"WI`<J@XPS`3>#X#XSE-#"?Y0,PDN<``%/C`@``"B@PG^4#`)_G_O[_Z\?_
M_^L<,)_E`2"@XP,PC^``(,/E$("]Z-!V`0!<=0$`*`$``%QV`0"$=@$`S?__
MZO!'+>D!RJ#C#,!-X.`/C.7(,)_E`(!0X@,PC^`K```*O""?Y0(PD^<`D)/E
M``!9XR@```H`4-CE`7"@X0``5>,]`%43`5"@$P!0H`,1```:`$"9Y0``5.,,
M```*"6"@X00`H.$%(*#A"!"@X;O^_^L``%#C`@``&@4PU.<]`%/C"P``"@1`
MMN4``%3C\___&@0`H.'PA[WH"%"@X0$P]>4``%/C/0!3$_O__QH(4$7@YO__
MZ@E@1N`!4(7B!4"$X$9AH.$$`*#A`&"'Y?"'O>@(0*#A[?__Z@E`H.'K___J
MV'0!`!@!```00"WI`<J@XPS`3>"(#XSEC""?Y7#03>*(,)_E#1"@X0(@C^``
M0*#A`S"2YP`PD^5L,(WE`#"@XW/^_^L``%#C``"@$P0``!H0,)WE#SH#X@H*
M4^,!"5,3"@``"D@@G^5`,)_E`B"/X`,PDN<`()/E;#"=Y0(@,^``,*#C!P``
M&G#0C>(0@+WH`A"@XP0`H.$5_O_K$`]OX:`"H.'N___J3?[_Z^AS`0`<`0``
MI',!`/!`+>D!RJ#C#,!-X'`/C.5<(9_E?-!-XE@QG^4`4%#B`B"/X`,PDN<`
M,)/E=#"-Y0`PH.,"```*`##5Y0``4^,*```:,"&?Y2@QG^4"((_@`S"2YP`@
MD^5T,)WE`B`SX``PH.-!```:?-"-XO"`O>C'_?_K`$!0XO'__PH$`*#A2O[_
MZP``4.,J```*$S#0Y1-@@.(N`%/C`@``&@$PUN4``%/C]/__"A,PT.4N`%/C
M)0``"@4`H.'N_?_K`'"@X08`H.'K_?_K``"'X`(`@.(2_O_KI#"?Y0!PH.$!
M$*#C`S"/X``@X.,$8(WE`%"-Y3'^_^L($(WB!P"@X=']_^L!`'#C!P"@X0,`
M``H8,)WE#SH#X@$)4^,2```*V/W_ZP<`H.&^_?_K!`"@X1_^_^L``%#CU/__
M&@0`H.'H_?_K!0"@X0'^_^N]___J`3#6Y2X`4^/6__\:`C#6Y0``4^/%__\*
MTO__ZJ3__^OK___J[?W_ZSAS`0`<`0``#',!`"0N``#P02WI`<J@XPS`3>``
M`(SE(``,Y0%PH.$`$9_E`#&?Y0':3>(!$(_@"-!-X@!@H.$!"HWB!`"`X@2`
MC>(#,)'G_R\`XP@0H.$`,)/E`#"`Y0`PH./,`)_E``"/X._]_^L`0%#B!P``
MV@A0C>(!,$3B`\"%X`0@7.4``%+C%P``&@``4^,4```:G#"?Y7`/'>X'(*#A
M!A"@X0,PG^<#`)#G<OW_ZX0@G^4!&HWB<#"?Y000@>("((_@`S"2YP`@D^4`
M,)'E`B`SX``PH.,2```:`=J-X@C0C>+P@;WH`T"@X00@H.$O$*#C"`"@X>/]
M_^L``%#CX___"C0`G^5PSQWN!#"@X0@@H.$&$*#A``"?YP``G.>"_?_KX?__
MZJ']_^NP<0$`'`$``#`M``!(<@$`('$!`-1Q`0#P1RWI`<J@XPS`3>#8#XSE
M%"*?Y0C03>(0,I_E`4"@X0(@C^``4*#A!(*?Y0,PDN<(@(_@`#"3Y00PC>4`
M,*#C`##1Y3T`4^,!0($"!`"@X5G]_^L-$*#A`&"@X04`H.'1_O_K`)!0XD``
M``I2_?_K``!6X3<``)JX,9_E`'"=Y0.`F.<'<:#A`##5Y0``4^,]`%,3`3"@
M$P`PH`,%```*!3"@X0$@\^4``%+C/0!2$_O__QH%,$/@`@"&X@!@F.4#`(#@
M9OW_ZP``4.,'`(;G%0``"@`PF.4%`*#A!Q"3YP$PT.0`,,'E``!3XST`4Q,&
M```*`2"!X@$PT.0"$*#A`3#"Y#T`4^,``%,3^?__&@$@H.$],*#C`##!Y0$P
MU.0!,.+E``!3X_O__QH((9_E^#"?Y0(@C^`#,)+G`""3Y00PG>4"(#/@`#"@
MXS4``!H(T(WB\(>]Z`&02>(!,-3D`3#IY0``4^/[__\:[O__ZL`PG^4#@)CG
M`'"8Y0`PE^4``%/C)```"@<PH.$$(+/E"1"@X0&0B>(``%+C^O__&@,0@>(!
M$:#AD#"?Y0,PC^``()/E``!2XP\``!H!`*#A`2"@XP`@@^4E_?_K`*!0XM7_
M_PH'$*#A"7&@X0<@H.$Q_?_K`*"(Y0"0C>4!D(GB`#"@XPDQBN>B___J!P"@
MX4?]_^L`H*#A``!0XP``B.7%__\*"7&@X?+__^H#D*#A"!"@X]___^H0_?_K
M?'`!`!P!``!L<`$`&`$``&AO`0!H<`$`$$`MZ0'*H.,,P$W@Z`^,Y7`0G^5P
M3QWN;""?Y1#03>)H`)_E`B"/X&0PG^4!$)_G`S"/X``@DN<!`)3G5!"?Y0P@
MC>4!$(_@(?W_ZT@0G^5(,)_E`,"@XT0@G^4#,(_@`1"?YP(@C^`!`)3G-!"?
MY03`C>4!$(_@`#"-Y;?\_^L0T(WB$("]Z#AO`0`T;@$`$`$``*QI``#8*0``
M!&\!`)PL``"<%0``Q"D``#@`G^4(T$WB-""?Y7#/'>XP$)_E`B"/X"PPG^4`
M`)_G`S"/X`$@DN<@$)_E``"<YP$0C^`$((WE"-"-XOG\_^J<;@$`F&T!`!`!
M```0:0``0"D``'!`+>D!RJ#C#,!-X```C.4!RDSB``",Y2@`#.7<()_E`MI-
MXM@PG^48T$WB`B"/X`(:C>(4$('B#$"-X@%JC>(#,)+G$&"&X@`PD^4`,('E
M`#"@X\S\_^NH$)_E`2`!XZ0PG^4!$(_@\`#-X0,PC^`!$*#C!`"@X<_\_^L$
M`*#A`2J@XP80H.'%_/_K`$!0XA@``+H!`(3BH/S_ZP!04.(4```*!""@X080
MH.%W_/_K`#"@XP0PQ>=0()_E`AJ-XCPPG^44$('B`B"/X`,PDN<`()/E`#"1
MY0(@,^``,*#C`P``&@4`H.$"VHWB&-"-XG"`O>B,_/_K`%"@X^W__^HL;0$`
M'`$``/@H``#D*```D&P!`+G__^KP3RWI`<J@XPS`3>```(SEL``,Y5@BG^5"
MW4WB5#*?Y0S03>("((_@`&"@X4@"G^5"S8WB&("-X@3`C.(#,)+G`5"@X0``
MC^`($*#A`#"3Y0`PC.4`,*#CO/W_ZR\0H.,`0*#A!@"@X0_\_^L``%#C$@``
M"@0"G^4&$*#A``"/X,?^_^L&`*#A]"&?Y4(=C>+@,9_E!!"!X@(@C^`#,)+G
M`""3Y0`PD>4"(#/@`#"@XVX``!I"W8WB#-"-XO"/O>@&`*#A)?S_ZP"0H.$%
M`*#A0_S_ZZP1G^4!$(_@4OS_ZP"P4.+B__\*G#&?Y9RAG^4#,(_@%#"-Y90Q
MG^4*H(_@`S"/X!`PC>4*,*#A"Z"@X0.PH.$`,-KE``!3XS(```H``%3C!```
M"@H0H.$$`*#A</S_ZP``4.,W```*"@"@X0;\_^L!($#B`C"*X`(@VN<O`%+C
M!0``&@#`H.,`P,/E`0!`X@$@<^4O`%+C^O__"@$PB>(`,(/@`0I3X[S__ZJ$
M4(WB$#"=Y0$JH.,!$*#C!0"@X0A@C>7PH,WA0/S_ZP@0H.$%`*#AX/O_ZP``
M4.,#```:*#"=Y0\Z`^(""5/C&P``"LP0G^4``*#C`1"/X!7\_^L`H*#A``!:
MXZ3__PH`,-KE``!3X\S__QH``%3C&P``"@`PU.4N`%/C`P``"I2@G^4!`*#C
M"J"/X-?__^H!,-3E``!3X_C__QI\$)_E``"@XP$0C^#_^__K`*"@X>C__^H!
M$*#C!0"@X;C[_^L``%#CWO__&E0`G^4%$*#A``"/X%+^_^L%`*#AX/O_ZXC_
M_^H4H)WE`0"@X[___^KG^__K'&P!`!P!```(*```W"<``*AK`0"0)P``6"<`
M`'@G``!L)P``G"8``$`F``!$)@``""8````PT.4``%/C'O\O`0$@@.(O`%/C
M`@"@`0$PTN0``%/C^O__&A[_+^$00"WI`<J@XPS`3>#X#XSE`$!0XC<```H`
M,-3E``!3XS0```J9^__K`3!`X@,PA.`#`%3A$0``*@,@H.$!``#J!`!2X30`
M``H",*#A`1!2Y"\`4>/Y__\*`P!4X0<``"H#(*#A`0``Z@0`4N$E```*`C"@
MX0$04N0O`%'C^?__&@,`5.$!,$,2"@``&@`PU.4O`%/C)```"IP`G^400+WH
M``"/X)K[_^H`$-+E`3!#XB\`4>,"```:`R"@X0,`5.'X__\Z!"!"X`(P@N(!
M"E/C$P``BF0`G^4$$*#A`3J@XP$@@N(``(_@`$"@X4G[_^L$`*#A$("]Z$0`
MG^400+WH``"/X(+[_^H$,*#A`P!4X0$P0Q+G__\:V___Z@0PH.',___J``"@
MXQ"`O>@4`)_E$$"]Z```C^!U^__J!"4``!1J`0!X)```1"0``'!`+>D!RJ#C
M#,!-X.@/C.7$))_E"-!-XL`TG^4-0*#A`B"/X+ADG^6X5)_E!F"/X`,PDN<%
M4(_@`#"3Y00PC>4`,*#C!!"@X08`H.&X_/_K``!0XPX```J,-)_E``"=Y0,P
ME><`$)/E`#&!X`0@D^4`(8'G``!2X_'__PH$,(/B!""3Y00@@^0``%+C^___
M&NO__^I49)_E!F"/X`00H.$&`*#AHOS_ZP``4.,.```*-#2?Y0``G>4#,)7G
M`!"3Y0`Q@>`$()/E`"&!YP``4N/Q__\*!#"#X@0@D^4$((/D``!2X_O__QKK
M___J`&2?Y09@C^`$$*#A!@"@X8S\_^L``%#C#@``"MPSG^4``)WE`S"5YP`0
MD^4`,8'@!""3Y0`A@><``%+C\?__"@0P@^($()/E!""#Y```4N/[__\:Z___
MZJQCG^4&8(_@!!"@X08`H.%V_/_K``!0XPX```J$,Y_E``"=Y0,PE><`$)/E
M`#&!X`0@D^4`(8'G``!2X_'__PH$,(/B!""3Y00@@^0``%+C^___&NO__^I8
M8Y_E!F"/X`00H.$&`*#A8/S_ZP``4.,.```*+#.?Y0``G>4#,)7G`!"3Y0`Q
M@>`$()/E`"&!YP``4N/Q__\*!#"#X@0@D^4$((/D``!2X_O__QKK___J!&.?
MY09@C^`$$*#A!@"@X4K\_^L``%#C#@``"M0RG^4``)WE`S"5YP`0D^4`,8'@
M!""3Y0`A@><``%+C\?__"@0P@^($()/E!""#Y```4N/[__\:Z___ZK!BG^4&
M8(_@!!"@X08`H.$T_/_K``!0XPX```I\,I_E``"=Y0,PE><`$)/E`#&!X`0@
MD^4`(8'G``!2X_'__PH$,(/B!""3Y00@@^0``%+C^___&NO__^I<8I_E!F"/
MX`00H.$&`*#A'OS_ZP``4.,.```*)#*?Y0``G>4#,)7G`!"3Y0`Q@>`$()/E
M`"&!YP``4N/Q__\*!#"#X@0@D^4$((/D``!2X_O__QKK___J"&*?Y09@C^`$
M$*#A!@"@X0C\_^L``%#C#@``"LPQG^4``)WE`S"5YP`0D^4`,8'@!""3Y0`A
M@><``%+C\?__"@0P@^($()/E!""#Y```4N/[__\:Z___ZK1AG^4&8(_@!!"@
MX08`H.'R^__K``!0XPX```IT,9_E``"=Y0,PE><`$)/E`#&!X`0@D^4`(8'G
M``!2X_'__PH$,(/B!""3Y00@@^0``%+C^___&NO__^I@89_E!F"/X`00H.$&
M`*#AW/O_ZP``4.,.```*'#&?Y0``G>4#,)7G`!"3Y0`Q@>`$()/E`"&!YP``
M4N/Q__\*!#"#X@0@D^4$((/D``!2X_O__QKK___J#`&?Y000H.$``(_@Q_O_
MZP`04.("```*^`"?Y0``C^#7_/_K\`"?Y000H.$``(_@OOO_ZP`04.("```*
MW`"?Y0``C^#._/_KU`"?Y000H.$``(_@M?O_ZP`04.(2```*P`"?Y0``C^#%
M_/_KN""?Y60PG^4"((_@`S"2YP`@D^4$,)WE`B`SX``PH.,0```:F!"?Y9@`
MG^4!$(_@``"/X`C0C>)P0+WHM?S_ZH0`G^4$$*#A``"/X)S[_^L`$%#BZ?__
M"G``G^4``(_@K/S_Z^7__^I&^O_K!&@!`!P!```P)```\&<!`!@!``#0(P``
M@",``#0C``#D(@``G"(``.`A``#X(0``K"$``&`A``",(```N"```(P@``"H
M(```J"```*0@``#('P``K&,!`(`@``#D'P``4"```-0?```$X"WE`<J@XPS`
M3>#P#XSEF#"?Y0S03>*4$)_EE""?Y0,PC^`!$(_@``"3Y0(@D><!`'#C`""2
MY00@C>4`(*#C"@``"G`@G^5H,)_E`B"/X`,PDN<`()/E!#"=Y0(@,^``,*#C
M$```&@S0C>($\)WD2`"?Y0T0H.$``(_@6_O_ZP``4.,$```*``#0Y3``4.,`
M`%`3`0"@$P``H`,@,)_E`S"/X```@^7E___J`/K_Z_AC`0"\8@$`'`$``)1B
M`0#D'@``>&,!`!!`+>D!RJ#C#,!-X/@/C.6P^?_KT/G_ZP``4.,``)`5$("]
MZ/!`+>D!RJ#C#,!-X-`/C.4X(9_E'-!-XC0QG^40$(WB`B"/X"QAG^4`4*#A
M!F"/X`,PDN<&`*#A`#"3Y10PC>4`,*#C+OO_ZP!`4.("```*`##4Y0``4^,.
M```:^""?Y>PPG^4"((_@`S"2YP`@D^44,)WE`B`SX``PH.,R```:V`"?Y040
MH.$``(_@'-"-XO!`O>@O_/_J!1"@X:[Y_^L``%#C"@``"K0@G^6@,)_E`B"/
MX`,PDN<`()/E%#"=Y0(@,^``,*#C'P``&AS0C>+P@+WH!0"@X8WY_^L`<*#A
M!`"@X8KY_^L``(?@`@"`XK'Y_^ML,)_E;""?Y0$0H.,#,(_@"$"-Y0(@C^``
M4(WE!""-Y0`@X.,`0*#AS?G_ZT@@G^4H,)_E`B"/X`,PDN<`()/E%#"=Y0(@
M,^``,*#C!!"@`08`H`'/__\*G?G_Z]1A`0`<`0``Q!X``)!A`0!H'@``1&$!
M`"`=```4'0``S&`!`/!/+>D`P*#C`NF@XPS`3>`.X$S@`<I,X@``C.4.`%SA
M^___&B`!#N5L%Y_E;">?Y0+93>(!$(_@9#>?Y?S03>(`<*#A`S"/X/A@C>("
M()'G`@F-XG2@1N(0X(/B`""2Y?0@@.4`(*#C]`"`X@<`D^DHD(/B!P"*Z%S`
M1N)H0$;B#P"^Z`\`K.@8AY_EX%!&X@B`C^`#`)[H`P",Z`<`F>@'`(3H``>?
MY040H.$``(_@MOK_ZP!`4.("```*`##4Y0``4^,*`0`:'/G_ZSSY_^L``%#C
MZ0``"@"@D.4``%KCY@``"@H`H.$L^?_K@`"@X0$`@.)3^?_K`##:Y0!`H.$`
MD*#A``!3XPL```J<MI_E"["/X`0`H.$`,(WE`1"@XPLPH.$`(.#C`D"$XFSY
M_^L!,/KE``!3X_7__QIP!I_E7*!&X@``C^`"``#J!`"ZY0``4.,-```*!1"@
MX8OZ_^L`0%#B^/__"@`PU.4``%/C]?__"K_Z_^L``%#C\O__"@0`H.$F^?_K
M`$!0X@X``!ITH$;B!$":Y```5.,*```*`##4Y0``4^.R```*!`"@X;#Z_^L`
M`%#C]?__"@0`H.$7^?_K`$!0XO'__PH$`*#A\OC_ZP"@H.$)`*#A[_C_ZP"@
MBN`!JXKB!*"*X@H`H.$4^?_KO#6?Y0"PH.$!$*#C`S"/X`@PC>6L-9_E`"#@
MXPR0C>4#,(_@!#"-Y9PUG^4`0(WE`S"/X"WY_^L'':#C"P"@X9?X_^L!`'#C
M`P``&D+Y_^L`$)#E$0!1XYT``!H%$*#A"P"@X0GY_^L`D%#B&```"E@5G^4`
M,)?E5"6?Y0$`F.<"((_@`+"-Y0$0H.,``)#E]?C_ZP"`H.,X)9_E`AF-X@0U
MG^7T$('B`B"/X`,PDN<`()/E`#"1Y0(@,^``,*#C-P$`&@@`H.$"V8WB_-"-
MXO"/O>C0,!;E#SH#X@$)4^/B__\:R$`6Y9?X_^L``%3AWO__&M`P%N53,.CG
M!PU3X]K__QH*`*#AT?C_ZP"`H.'$!)_E!1"@X0``C^`>^O_K`!!0XHX```H`
M`)?E1OS_ZP"@4.**```*"@"@X9KX_^L`0*#A`P!0XPP``)J,%)_E!`!`X@``
MBN`!$(_@^/C_ZP``4.,%```:!`!4X]D```H$0(K@!3!4Y2\`4^/5```*7$2?
MY01`C^``,)3E`0!SX\(```H``%/C<P``"D04G^5$!)_E`1"/X$!TG^4``(_@
M/&2?Y0W[_^LX5)_EO_C_ZS0TG^4(`(WE`1"@XP,PC^`,,(WE)#2?Y0@`H.$`
ML(WE`"#@XP,PC^`$,(WE$#2?Y0=PC^`&8(_@!5"/X`,PC^"Z^/_K#P``ZM/X
M_^L`,)#E$0!3XQ$``!JH^/_K`9")X@#`H.$',*#A!""@X0$0H.,(`*#A$&"-
MY0R0C>4@$(WI`+"-Y:GX_^L'':#C"`"@X1/X_^L`0*#A`0!PX^G__PH+`*#A
M._C_ZY@#G^4($*#A``"/X-WZ_^L(`*#AA?[_ZX3__^H#0*#A4O__ZFA`1N($
M`)3D``!0XRD```H%$*#AO?G_ZP`P4.+X__\*`##3Y0``4^/U__\*7/C_ZP"@
M4.+R__\*./C_ZX``H.$!`(#B"O__ZO0BG^4`,)?E`@"8YP00C>4!$*#C`+"-
MY1@CG^4``)#E`B"/X%OX_^MD___J9/[_ZP0CG^6<,I_E`AF-X@(@C^#T$('B
M`S"2YP`@D^4`,)'E`B`SX``PH..<```:!`"@X0+9C>+\T(WB\$^]Z#KX_^K(
MHI_E#0"@XT#X_^M3,*#C`$"@X0J@C^``D*#A[/[_Z@"@E^5R___J"@"@X0`0
MH.,;^/_K`$!0XH;__PH`,.#C$2#@XP)PH.,$4$;B`'"-Y2OX_^L&(*#C!1"@
MX00`H.'N]__K!#!6Y0``4^,"```:`S!6Y4,`4^-4```*`#"@XP`@H.,`,(WE
M!`"@X0`PH.,;^/_KM@0`ZP!PH.$"``#J!1"@X0<`H.'&!`#K`BF@XP40H.$$
M`*#AV/?_ZP`@4.+V___*")*?Y00`H.$PH$;B^??_ZT501N('$*#A"9"/X`I`
MH.$Q<$;B1`!&XOL$`.L!(/7E!`"@X0`@C>4),*#A`1"@XP`@X.,G^/_K`D"$
MX@<`5>'U__\:N"&?Y0$0H.,(H(WE"`"@X0(@C^``L(WE!""-Y0`@H..<,9_E
M""!&Y9@AG^4#,(_@`B"/X`P@C>4`(.#C%/C_ZV___^J``9_E!1"@X0``C^`Z
M^?_K``!0XP``A`6K__\*`##0Y0``4^,P`%,3`3"@$P`PH`,`,(3E+O__Z@1`
ME^4``%3C)O__"@0`H.&O]__K`P!0XR+__YHP$9_E!`!`X@``A.`!$(_@#OC_
MZP``4.,$H*`!&O__Z@(P5N5!`%/CI___&@$P5N5#`%/CI/__&@`PUN5(`%/C
MH?__&@$PUN5%`%/CGO__&@`PX.,$`*#A`'"-Y3D@X..[]__K!1"@X2D@H.,$
M`*#A?O?_Z\`PG^7`()_E`1"@XP,PC^`,,(WEM#"?Y0(@C^`(`*#A!""-Y0,P
MC^`(4(WE`+"-Y0`@X./3]__K+O__ZJWW_^M,8`$`'`$``'1A`0#P7P$`X!L`
M`(0<```$'```J!L``*`:``!X&P``(`$``'@;``#D70$`-!L``/P:``!`7@$`
MX!D``%P9``"4&@``.!L``,@8``!@&P``W!@``'@:```H&```R!@``'Q;`0!`
M&```@!<``%06``#<%P``H!@``&`6``!<%P``N!<``$`5``#8%@``\$`MZ0'*
MH.,,P$W@X`^,Y5P2G^4,T$WB6"*?Y7!/'>X!$(_@4#*?Y5`"G^4-8*#A`B"1
MYP``C^`&$*#A`""2Y00@C>4`(*#C`S"?YP-0E.=D,-7E93#%Y<#X_^L``%#C
M`@``"@`PT.4Q`%/C9@``"NW[_^L(,I_E`S"?YP-0E.<`$I_E!0"@X0,PH.,!
M(*#C`1"/X`[W_^L`4%#B#```"N0QG^4,()7E`S"?YP`0DN78(9_E`P"4YP(@
MC^`*]__K##"5Y0`0D^4(,)'E`043XUD``!JX$9_E`S"@XP$@H.,!$)_G`0"4
MYZ@1G^4!$(_@]_;_ZP!04.(+```*F`&?Y080H.$``(_@E/C_ZP`@4.)!```*
MA#&?Y0P0E>4#,)_G`!"1Y0,`E.?O]O_K<#&?Y0$@H.-L$9_E;'&?Y0$0C^`#
M,)_G!W"/X`-0E.<%`*#AT/;_ZU0AG^4`$*#A!0"@X0(@C^#@]O_K1#&?Y40!
MG^4`(*#C!A"@X0``C^`#,)_G`S"4YV4@P^5T^/_K`$"@X0``4.,(`(?E$@``
M"@<=H.,$`*#AK_;_ZP$@@.(`,*#A$0!0XP$`4A,E``"*^""?Y;`PG^4"((_@
M`S"2YP`@D^4$,)WE`B`SX``PH.,;```:#-"-XO"`O>@$`)?E=_W_ZP!`H.$`
M`%#C"`"'Y>;__QJX`)_E``"/X!KW_^L!,-#E``!3XYG__PJ4___JH#"?Y0P@
ME>4#,(_@``"2Y00PD^4`$)/E$/G_Z[O__^J$,)_E`S"?YP,`E.?L]O_KH/__
MZO/V_^MP$)_E!""@X6P`G^4!$(_@``"/X`00D>4`$)'E`/?_ZS!8`0`<`0``
M%%D!`'P4``#D6`$`S!4``+18`0"H%0``?%@!`'P5``!D$P``0%@!`!Q8`0`P
M%0``A%@!`"05``#@5P$`Y!(``(Q6`0"(%```I%<!`!Q7`0!H5P$`3!0``/!`
M+>D!RJ#C#,!-X.`/C.5`(9_E#-!-XCPQG^4`4*#A`B"/X`,PDN<`,)/E!#"-
MY0`PH./T^O_K`##0Y0!`H.$``%/C!0``"@$@@.(O`%/C`D"@`0$PTN0``%/C
M^O__&OAPG^4`8%7B`6"@$P=PC^``,)?E`0!SXR8```H``%/C`&"@`P%@!A(`
M`%;C"@``&LP@G^7`,)_E`B"/X`,PDN<`()/E!#"=Y0(@,^``,*#C)P``&@S0
MC>+P@+WH`##5Y0``4^/Q__\*F!"?Y00@H.,$`*#A`1"/X,/V_^L``%#CZO__
M&H`@G^5L,)_E`B"/X`,PDN<`()/E!#"=Y0(@,^``,*#C$@``&@4`H.$,T(WB
M\$"]Z$+X_^I0`)_E#1"@X0``C^#8]__K``!0XP``AP76__\*`##0Y0``4^,P
M`%,3`3"@$P`PH`,#8`;@`#"'Y<S__^I]]O_K7%4!`!P!``!(5@$`Y%0!`-`1
M``"05`$`V!```/!/+>D!RJ#C#,!-X(`.C.403)_E5]]-X@SLG^40,(WB!$"/
MX`,0H.$<((#B7,"`X@[@E.<`X)[E5.&-Y0#@H..-"F+TX`#P\P(`7.'="D'T
M^O__&@9+W>T(.]WM$1Z-X@H;W>T```#JLA%B\@@KT^TT((/B``O3[0@P@^(!
M`%/ACU=B]+`!0O.U`4#SM`%`\[-!8_*Q,6'RH!A@\C``X?.P`6'R#@O#[>[_
M_QH@0)#IF3D'XX(Z1>,,0)#E$,"=Y0!@D.4$$"[@$'"0Y0/`C.`%$`'@99&@
MX0?`C.`$$"'@%""=Y>;-C.`!P(S@"1`NX`,@@N`&$`'@9H&@X00@@N`8H)WE
M#A`AX`(0@>`(("G@`Z"*X`P@`N`.H(K@"2`BX`H@@N`<H)WE[!V!X&S!H.$#
MH(K@":"*X`R0*.`!D`G@X2V"X`B0*>!A$:#A"I")X""@G>7BG8G@`Z"*X`B@
MBN`!@"S@`H`(X&(AH.$,@"C@`K`AX`J`B.`DH)WE";`+X.F-B.`#H(K@#*"*
MX&G!H.$!D"O@#+`BX`J0B>`HH)WE"+`+X.B=B>`#H(K@`:"*X&@1H.$"@"O@
M`;`LX`J`B.`LH)WE";`+X.F-B.`#H(K@`J"*X&DAH.$,D"O@`K`AX`J0B>`P
MH)WE"+`+X.B=B>`#H(K@#*"*X&C!H.$!@"O@#+`BX`J`B.`TH)WE";`+X.F-
MB.`#H(K@`:"*X&D1H.$"D"O@`;`LX`J0B>`XH)WE"+`+X.B=B>`#H(K@`J"*
MX&@AH.$,@"O@`K`AX`J`B.`\H)WE";`+X.F-B.`#H(K@#*"*X&G!H.$!D"O@
M#+`BX`J0B>!`H)WE"+`+X.B=B>`#H(K@`:"*X&@1H.$"@"O@`;`LX`J`B.!$
MH)WE";`+X.F-B.`#H(K@`J"*X&DAH.$,D"O@`K`AX`J0B>!(H)WE"+`+X.B=
MB>`#H(K@#*"*X&C!H.$!@"O@#+`BX`J`B.!,H)WE";`+X.F-B.`#H(K@`:"*
MX&D1H.$"D"O@`;`LX`J0B>!0H)WE"+`+X.B=B>`#H(K@`J"*X&@AH.$,@"O@
M"H"(X%2@G>7IC8C@`Z"*X`S`BN`"H"'@":`*X&F1H.$!H"K@#*"*X%C`G>7H
MK8K@`\",X`'`C.`)$"+@"!`!X&B!H.$"$"'@#!"!X%S`G>7J'8'@`S",X`C`
M*>`*P`S@`C"#X`G`+.!@()WE`\",X*$[#N/9/D;C:J&@X0,@@N#AS8S@"9""
MX&0@G>4#((+@""""X`J`*.`!@"C@"(")X&&1H.%H$)WE[(V(X`,0@>`*$('@
M":`JX`R@*N!LP:#A"B""X.BM@N!L()WE`R""X`D@@N`,D"G@")`IX&B!H.$)
M$('@ZIV!X'`0G>4#$('@#!"!X`C`+.`*P"S@:J&@X0P@@N#IS8+@=""=Y0,@
M@N`(((+@"H`HX`F`*.!ID:#A"!"!X.R-@>!X$)WE`Q"!X`H0@>`)H"K@#*`J
MX&S!H.$*((+@Z*V"X'P@G>4#((+@"2""X`R0*>`(D"G@:(&@X0D0@>#JG8'@
M@!"=Y0,0@>`,$('@",`LX`K`+.!JH:#A#"""X.G-@N"$()WE`R""X`@@@N`*
M@"C@"8`HX&F1H.$($('@[(V!X(@0G>4#$('@"A"!X`F@*N`,H"K@;,&@X0H@
M@N#HK8+@C""=Y0,@@N!JL:#A"2""X`R0*>`(D"G@:(&@X0D0@>#JG8'@D!"=
MY0,0@>`,$('@",`LX`K`+.`,((+@Z:V"X)0@G>4#((+@""""X`N`*.`)P"C@
M:9&@X0P0@>"8P)WEZHV!X`D0*^`#P(S@"A`AX`O`C."<L)WE:J&@X0$@@N`#
ML(O@Z!V"X`FPB^`*D"G@""`IX&B!H.$"P(S@H""=Y>&=C.`#((+@"B""X`B@
M*N`!P"K@I*"=Y6$1H.$,P(O@`Z"*X.G-C.`(H(K@`8`HX`F`*.!ID:#A""""
MX+"PG>7LC8+@J""=Y0,@@N`!((+@"1`AX`P0(>!LP:#A`:"*X`P0*>#HK8K@
M"!`AX`$@@N!H@:#AZAV"X*P@G>4#((+@"#`LX`HP(^`)((+@`R""X-P\"^,;
M/TCC:I&@X0.PB^`)H(CA#+"+X+3`G>7A+8+@`:`*X`/`C.!A$:#A",",X`F`
M".`(H(KA`8")X0N@BN`"@`C@`;`)X.*MBN`+@(CAO+"=Y0R`B."XP)WE8B&@
MX0.PB^`#P(S@`;"+X`F0C.`"P('A"L`,X`(0`>`!P(SAZHV(X`G`C.#`D)WE
M:J&@X>C-C.`*$(+A`Y")X`*0B>`($`'@"B`"X&B!H.$"$('A""`*X`NP@>`(
M$(KA#!`!X.R]B^`"$('A;,&@X0D0@>#$D)WE#""(X>L=@>`#D(G@"R`"X`J0
MB>#(H)WE:[&@X0.@BN`(H(K@#(`(X`@@@N%A@:#A"9""X`L@C.'AG8G@`2`"
MX,P0G>4#$('@#!"!X`O`#.`,P(+A"""+X0K`C.`)(`+@"*`+X.G-C.`*((+A
MT*"=Y6F1H.$!$(+@`Z"*X`D@B.$+H(K@U+"=Y0P@`N#L'8'@`["+X&S!H.$(
ML(O@"8`(X`@@@N%A@:#A"J""X`P@B>'AK8K@`2`"X-@0G>4#$('@"1"!X`R0
M">`)D(+A""",X0N0B>`*(`+@"+`,X.J=B>`+((+AX+"=Y0$0@N#<()WE:J&@
MX0.PB^`#((+@"+"+X`S`@N`*((CA"2`"X`J`".#I'8'@"(""X6F1H.$,@(C@
M"2"*X6'!H.'AC8C@`2`"X.00G>4#$('@"A"!X`F@"N`*H(+A#"")X0N@BN`(
M(`+@#+`)X.BMBN`+((+A[+"=Y0$0@N#H()WE:(&@X0.PB^`#((+@#+"+X`F0
M@N`(((SA"B`"X`C`#.#J'8'@#"""X6JAH.$)D(+@"B"(X6'!H.'AG8G@`2`"
MX/`0G>4#$('@"!"!X`J`".`(@(+A#""*X0N`B.`)(`+@#+`*X.F-B.`+((+A
M^+"=Y0$@@N#T$)WE:9&@X0.PB^`#$('@#+"+X`J@@>`)$(SA"!`!X`G`#.`,
MP('AZ"V"X`K`C.#\H)WE:(&@X>+-C.`($(GA`S"*X`DP@^`"$`'@")`)X&(A
MH.$"H(CA"1"!X0L0@>`,H`K@`K`(X.P=@>!LD:#A"Z"*X0#!G>4#H(K@UC$,
MXV(Z3.,#P(S@X:V*X`B`C.`$P9WE8;&@X0/`C.`"P(S@"2`BX`$@(N`($9WE
M`H"(X`L@*>#JC8C@"B`BX`,0@>!JH:#A"1"!X`+`C.`*D"O@Z,V,X`B0*>`,
M(9WE"1"!X&B!H.'LG8'@$!&=Y0,@@N`#$('@"R""X`H0@>`(H"K@#*`JX&S!
MH.$*((+@Z:V"X!0AG>4#((+@""""X`R`*.`)@"C@:9&@X0@0@>#JC8'@&!&=
MY0,0@>`,$('@"<`LX`K`+.!JH:#A#"""X.C-@N`<(9WE`R""X`D@@N`*D"G@
M")`IX&B!H.$)$('@[)V!X"`1G>4#$('@"A"!X`B@*N`,H"K@;,&@X0H@@N#I
MK8+@)"&=Y0,@@N`(((+@#(`HX`F`*.!ID:#A"!"!X.J-@>`H$9WE`Q"!X`P0
M@>`)P"S@"L`LX&JAH.$,((+@Z,V"X"PAG>4#((+@"2""X`J0*>`(D"G@:(&@
MX0D0@>#LG8'@,!&=Y0,0@>`*$('@"*`JX`R@*N!LP:#A"B""X.FM@N`T(9WE
M`R""X`@@@N`,@"C@"8`HX&F1H.$($('@ZHV!X#@1G>4#$('@#!"!X`G`+.`*
MP"S@:J&@X0P@@N#HS8+@/"&=Y0,@@N`)((+@"I`IX`B0*>!H@:#A"1"!X$"1
MG>7L'8'@`Y")X`J0B>`(H"K@#*`JX&S!H.$*((+@1*&=Y>$M@N`#H(K@"*"*
MX`R`*.`!@"C@81&@X0B0B>!(@9WEXIV)X`.`B.`,@(C@`<`LX`+`+.!B(:#A
M#,"*X`)PA^#IS8S@$'"`Y6SACN`(X(WE3.&=Y0,PCN`&,(/@`3"#X`(0(>`)
M$"'@:9&@X0$0B.`)("+@[!V!X`P@(N`",(/@"4"$X.$]@^`,0(WE`#"-Y0%0
MA>`\()_E!%"-Y=\*;?0"((_@*#"?Y8\*0/0#,)+G`""3Y50QG>4"(#/@`#"@
MXP$``!I7WXWB\(^]Z&SS_^OH4P$`'`$``!!(`0`00"WI`<J@XPS`3>#X#XSE
M8`"@XV#S_^L'"]_M""O?[0D[W^T`(*#C!`O`[1@@@.5<((#EWRI`]!"`O>CP
MX=+#``````$C16>)J\WO_MRZF'94,A#P1RWI`<J@XPS`3>#@#XSE%#"0Y0)@
MH.&"(:#A`%"@X0(PD^`8()#E%#"`Y0%`H.%<`)#E`2""(J8N@N`<@(7B``!0
MXQ@@A>4@```:/P!6XRX``-I`D$;B*6.@X0%PAN('<X3@!!"@X0@PH.%`0(3B
M`."1Y1`0@>(,P!'E$#"#X@@`$>4$(!'E!`!1X1#@`^4,P`/E"``#Y00@`^7S
M__\:`4"@X04`H.&R_/_K!P!4X>O__QH&8TG@!B"@X0<0H.$(`*#A^_+_ZUQ@
MA>7PA[WH0'!@X@``B.`&`%?A!G"@H0<@H.'S\O_K7#"5Y0,PA^!<,(7E0`!3
MX_"'O1@%`*#A!V!&X`=`A.":_/_KSO__Z@1PH.'G___J<$`MZ0'*H.,"BRWM
M#,!-X.@/C.4!0*#A%!"1Y0!0H.$<8(3B,3^_YA00E.48`)3ET1'EYS`OO^88
M*T/L`2"!XG\PX.,X`%+C`@"&X`$PQN=!``#:0"!BX@`0H..E\O_K!`"@X7W\
M_^LL,(3B4`#`\@\*1O0/"D/T/#"$X@\*0_1,,(3B#P=#]%0PA.($`*#A#X<#
M]''\_^L#,-3E`##%Y00`H.&R,-3A`3#%Y0`PE.4C-*#A`C#%Y0`PE.4#,,7E
M!S#4Y00PQ>6V,-3A!3#%Y00PE.4C-*#A!C#%Y00PE.4',,7E"S#4Y0@PQ>6Z
M,-3A"3#%Y0@PE.4C-*#A"C#%Y0@PE.4+,,7E#S#4Y0PPQ>6^,-3A#3#%Y0PP
ME.4C-*#A#C#%Y0PPE.4/,,7E$S#4Y1`PQ>6R,=3A$3#%Y1`PE.4C-*#A$C#%
MY1`PE.43,,7E`HN]['!`O>AU\O_J."!BX@`0H.-C\O_KQO__Z@@PG^4#,(_@
M`#"3Y<'R_^HL1@$`"$`MZ0B`O>@!``(`/0```$`````+````0````!````!`
M````)@```$`````U````0`````,```!`````!@```$`````4````0`````X`
M``!`````#0```$`````E````0````!(```!`````*````$`````"````0```
M`!@```!`````"0```$`````*````0```````````````)7,O)7,````O<')O
M8R]S96QF+V5X90``1'EN84QO861E<CHZ8F]O=%]$>6YA3&]A9&5R`$EN=&5R
M;F%L<SHZ4$%2.CI"3T]4`````"]P<F]C+R5I+R5S`&5X90`N````4$%27U1%
M35``````4$%27U!23T=.04U%`````#H````E<R5S)7,``"\```!015),-4Q)
M0@````!015),3$E"`%!%4DPU3U!4`````%!%4DQ)3P``4$%27TE.251)04Q)
M6D5$`%!!4E]34$%73D5$`%!!4E]#3$5!3@```%!!4E]$14)51P```%!!4E]#
M04-(10```%!!4E]'3$]"04Q?1$5"54<`````4$%27T=,3T)!3%]435!$25(`
M``!005)?5$U01$E2``!005)?1TQ/0D%,7U1%35``4$%27T=,3T)!3%]#3$5!
M3@`````Q````3$1?3$E"4D%265]0051(`%-94U1%30``)3`R>``````E<R5S
M)7,E<P````!P87(M`````"5S.B!C<F5A=&EO;B!O9B!P<FEV871E('-U8F1I
M<F5C=&]R>2`E<R!F86EL960@*&5R<FYO/25I*0H``"5S.B!P<FEV871E('-U
M8F1I<F5C=&]R>2`E<R!I<R!U;G-A9F4@*'!L96%S92!R96UO=F4@:70@86YD
M(')E=')Y('EO=7(@;W!E<F%T:6]N*0H``%!!5$@`````<&%R;``````N<&%R
M`````"5S)7-C86-H92TE<R5S```E<R5S=&5M<"TE=25S````)7,E<W1E;7`M
M)74M)74E<P`````8````<&5R;``````P````4$%2.CI086-K97(Z.E9%4E-)
M3TX`````,2XP-34```!5;F%B;&4@=&\@8W)E871E(&-A8VAE(&1I<F5C=&]R
M>0`````E<SH@8W)E871I;VX@;V8@<')I=F%T92!C86-H92!S=6)D:7)E8W1O
M<GD@)7,@9F%I;&5D("AE<G)N;STE:2D*`````"UE```M+0``55-%4@````!5
M4T523D%-10````!414U01$E2`%1-4``O=&UP`````&5V86P@)V5X96,@+W5S
M<B]B:6XO<&5R;"`@+5,@)#`@)'LQ*R(D0")])PII9B`P.R`C(&YO="!R=6YN
M:6YG('5N9&5R('-O;64@<VAE;&P*<&%C:V%G92!?7W!A<E]P;#L*;7D@*"10
M05)?34%'24,L("1P87)?=&5M<"P@)'!R;V=N86UE+"!`=&UP9FEL92P@)4UO
M9'5L94-A8VAE*3L*14Y$('L@:68@*"1%3E9[4$%27T-,14%.?2D@>PIR97%U
M:7)E($9I;&4Z.E1E;7`["G)E<75I<F4@1FEL93HZ0F%S96YA;64["G)E<75I
M<F4@1FEL93HZ4W!E8SL*;7D@)'1O<&1I<B`]($9I;&4Z.D)A<V5N86UE.CID
M:7)N86UE*"1P87)?=&5M<"D["F]U=',H<7%;4F5M;W9I;F<@9FEL97,@:6X@
M(B1P87)?=&5M<")=*3L*1FEL93HZ1FEN9#HZ9FEN9&1E<'1H*'-U8B![("@@
M+60@*2`_(')M9&ER(#H@=6YL:6YK('TL("1P87)?=&5M<"D["G)M9&ER("1P
M87)?=&5M<#L*:68@*"UD("1P87)?=&5M<"`F)B`D7D\@;F4@)TU35VEN,S(G
M*2!["FUY("1T;7`@/2!N97<@1FEL93HZ5&5M<"@*5$5-4$Q!5$4@/3X@)W1M
M<%A86%A8)RP*1$E2(#T^($9I;&4Z.D)A<V5N86UE.CID:7)N86UE*"1T;W!D
M:7(I+`I3549&25@@/3X@)RYC;60G+`I53DQ)3DL@/3X@,"P**3L*;7D@)&9I
M;&5N86UE(#T@)'1M<"T^9FEL96YA;64["G!R:6YT("1T;7`@/#PB+BXN(CL*
M>#TQ.R!W:&EL92!;(%PD>"`M;'0@,3`@73L@9&\*<FT@+7)F("<D<&%R7W1E
M;7`G"FEF(%L@7"$@+60@)R1P87)?=&5M<"<@73L@=&AE;@IB<F5A:PIF:0IS
M;&5E<"`Q"G@]8&5X<'(@7"1X("L@,6`*9&]N90IR;2`G)&9I;&5N86UE)PHN
M+BX*8VQO<V4@)'1M<#L*8VAM;V0@,#<P,"P@)&9I;&5N86UE.PIM>2`D8VUD
M(#T@(B1F:6QE;F%M92`^+V1E=B]N=6QL(#(^)C$@)B(["G-Y<W1E;2@D8VUD
M*3L*;W5T<RAQ<5M3<&%W;F5D(&)A8VMG<F]U;F0@<')O8V5S<R!T;R!P97)F
M;W)M(&-L96%N=7`Z("1F:6QE;F%M95TI.PI]"GT@?0I"14=)3B!["B1005)?
M34%'24,@/2`B7&Y005(N<&U<;B(["DEN=&5R;F%L<SHZ4$%2.CI"3T]4*"D@
M:68@9&5F:6YE9"`F26YT97)N86QS.CI005(Z.D)/3U0["F5V86P@>PI?<&%R
M7VEN:71?96YV*"D["FUY("1Q=6EE="`]("$D14Y6>U!!4E]$14)51WT["FUY
M("5S>7,@/2`H"G!A=&A?<V5P("`@(#T^("@D7D\@/7X@+UY-4U=I;B\@/R`G
M.R<@.B`G.B<I+`I?97AE("`@("`@("`]/B`H)%Y/(#U^("]>*#\Z35-7:6Y\
M3U,R?&-Y9W=I;BDO(#\@)RYE>&4G(#H@)R<I+`I?9&5L:6T@("`@("`]/B`H
M)%Y/(#U^("]>35-7:6Y\3U,R+R`_("=<7"<@.B`G+R<I+`HI.PI?<V5T7W!R
M;V=N86UE*"D["F]U=',H<7%;7"1P<F]G;F%M92`]("(D<')O9VYA;64B72D[
M"E]S971?<&%R7W1E;7`H*3L*;W5T<RAQ<5M<)$5.5GM005)?5$5-4'T@/2`B
M)$5.5GM005)?5$5-4'TB72D["FUY("@D<W1A<G1?<&]S+"`D9&%T85]P;W,I
M.PI-04=)0SH@>PIL;V-A;"`D4TE'>U]?5T%23E]??2`]('-U8B![?3L*=6YL
M97-S("AO<&5N(%]&2"P@)SPZ<F%W)RP@)'!R;V=N86UE*2!["F]U=',H<7%;
M0V%N)W0@<F5A9"!F<F]M(&9I;&4@(B1P<F]G;F%M92)=*3L@(",@9&]N)W0@
M=7-E("0A(&AE<F4@87,@:70@<F5Q=6ER97,@17)R;F\N<&T*;&%S="!-04=)
M0SL*?0IM>2`D8VAU;FM?<VEZ92`](#8T("H@,3`R-#L*;7D@)&)U9CL*;7D@
M)'-I>F4@/2`M<R!?1D@["FUY("1M86=I8U]P;W,@/2`D<VEZ92`M("1S:7IE
M("4@)&-H=6YK7W-I>F4[(",@3D]413H@)&UA9VEC7W!O<R!I<R!A(&UU;'1I
M<&QE(&]F("1C:'5N:U]S:7IE"G=H:6QE("@D;6%G:6-?<&]S(#X](#`I('L*
M<V5E:R!?1D@L("1M86=I8U]P;W,L(#`["G)E860@7T9(+"`D8G5F+"`D8VAU
M;FM?<VEZ92`K(&QE;F=T:"@D4$%27TU!1TE#*3L*:68@*"AM>2`D:2`](')I
M;F1E>"@D8G5F+"`D4$%27TU!1TE#*2D@/CT@,"D@>PHD;6%G:6-?<&]S("L]
M("1I.PIL87-T.PI]"B1M86=I8U]P;W,@+3T@)&-H=6YK7W-I>F4["GT*:68@
M*"1M86=I8U]P;W,@/"`P*2!["F]U=',H<7%;0V%N)W0@9FEN9"!M86=I8R!S
M=')I;F<@(B1005)?34%'24,B(&EN(&9I;&4@(B1P<F]G;F%M92)=*3L*;&%S
M="!-04=)0SL*?0IO=71S*")&;W5N9"!005(@;6%G:6,@870@<&]S:71I;VX@
M)&UA9VEC7W!O<R(I.PIS965K(%]&2"P@)&UA9VEC7W!O<R`M(#0L(#`["G)E
M860@7T9(+"`D8G5F+"`T.PIM>2`D;V9F<V5T(#T@=6YP86-K*").(BP@)&)U
M9BD["F]U=',H(D]F9G-E="!F<F]M('-T87)T(&]F($9)3$5S(&ES("1O9F9S
M970B*3L*<V5E:R!?1D@L("1M86=I8U]P;W,@+2`T("T@)&]F9G-E="P@,#L*
M)&1A=&%?<&]S(#T@=&5L;"!?1D@["FUY("5R97%U:7)E7VQI<W0["G)E860@
M7T9(+"`D8G5F+"`T.R`@("`@("`@("`@("`@("`@("`@("`@("`@(",@<F5A
M9"!T:&4@9FER<W0@(D9)3$4B"G=H:6QE("@D8G5F(&5Q(")&24Q%(BD@>PIR
M96%D(%]&2"P@)&)U9BP@-#L*<F5A9"!?1D@L("1B=68L('5N<&%C:R@B3B(L
M("1B=68I.PIM>2`D9G5L;&YA;64@/2`D8G5F.PIO=71S*'%Q6U5N<&%C:VEN
M9R!&24Q%("(D9G5L;&YA;64B+BXN72D["FUY("1C<F,@/2`H("1F=6QL;F%M
M92`]?B!S?%XH6V$M9EQD77LX?2DO?'P@*2`_("0Q(#H@=6YD968["FUY("@D
M8F%S96YA;64L("1E>'0I(#T@*"1B=68@/7X@;7PH/SHN*B\I/R@N*BDH7"XN
M*BE\*3L*<F5A9"!?1D@L("1B=68L(#0["G)E860@7T9(+"`D8G5F+"!U;G!A
M8VLH(DXB+"`D8G5F*3L*:68@*&1E9FEN960H)&5X="D@86YD("1E>'0@(7X@
M+UPN*#\Z<&U\<&Q\:7A\86PI)"]I*2!["FUY("1F:6QE;F%M92`](%]S879E
M7V%S*"(D8W)C)&5X="(L("1B=68L(#`W-34I.PHD4$%2.CI(96%V>3HZ1G5L
M;$-A8VAE>R1F=6QL;F%M97T@/2`D9FEL96YA;64["B1005(Z.DAE879Y.CI&
M=6QL0V%C:&5[)&9I;&5N86UE?2`]("1F=6QL;F%M93L*?0IE;'-I9B`H("1F
M=6QL;F%M92`]?B!M?%XO/W-H;&EB+WP@86YD(&1E9FEN960@)$5.5GM005)?
M5$5-4'T@*2!["FUY("1F:6QE;F%M92`](%]S879E7V%S*"(D8F%S96YA;64D
M97AT(BP@)&)U9BP@,#<U-2D["F]U=',H(E-(3$E".B`D9FEL96YA;65<;B(I
M.PI]"F5L<V4@>PHD36]D=6QE0V%C:&5[)&9U;&QN86UE?2`]('L*8G5F(#T^
M("1B=68L"F-R8R`]/B`D8W)C+`IN86UE(#T^("1F=6QL;F%M92P*?3L*)')E
M<75I<F5?;&ES='LD9G5L;&YA;65]*RL["GT*<F5A9"!?1D@L("1B=68L(#0[
M"GT*;&]C86P@0$E.0R`]("AS=6(@>PIM>2`H)'-E;&8L("1M;V1U;&4I(#T@
M0%\["G)E='5R;B!I9B!R968@)&UO9'5L92!O<B`A)&UO9'5L93L*;7D@)&EN
M9F\@/2`D36]D=6QE0V%C:&5[)&UO9'5L97T["G)E='5R;B!U;FQE<W,@)&EN
M9F\["F1E;&5T92`D<F5Q=6ER95]L:7-T>R1M;V1U;&5].PHD24Y#>R1M;V1U
M;&5](#T@(B]L;V%D97(O)&EN9F\O)&UO9'5L92(["FEF("@D14Y6>U!!4E]#
M3$5!3GT@86YD(&1E9FEN960H)DE/.CI&:6QE.CIN97<I*2!["FUY("1F:"`]
M($E/.CI&:6QE+3YN97=?=&UP9FEL92!O<B!D:64@(D-A;B=T(&-R96%T92!T
M96UP(&9I;&4Z("0A(CL*)&9H+3YB:6YM;V1E*"D["B1F:"T^<')I;G0H)&EN
M9F\M/GMB=69]*3L*)&9H+3YS965K*#`L(#`I.PIR971U<FX@)&9H.PI]"F5L
M<V4@>PIM>2`D9FEL96YA;64@/2!?<V%V95]A<R@B)&EN9F\M/GMC<F-]+G!M
M(BP@)&EN9F\M/GMB=69]*3L*)&EN9F\M/GMF:6QE?2`]("1F:6QE;F%M93L*
M;W!E;B!M>2`D9F@L("<\.G)A=R<L("1F:6QE;F%M92!O<B!D:64@<7%;0V%N
M)W0@<F5A9"`B)&9I;&5N86UE(CH@)"%=.PIR971U<FX@)&9H.PI]"F1I92`B
M0F]O='-T<F%P<&EN9R!F86EL960Z(&-A;B=T(&9I;F0@;6]D=6QE("1M;V1U
M;&4A(CL*?2P@0$E.0RD["G)E<75I<F4@6%-,;V%D97(["G)E<75I<F4@4$%2
M.CI(96%V>3L*<F5Q=6ER92!#87)P.CI(96%V>3L*<F5Q=6ER92!%>'!O<G1E
M<CHZ2&5A=GD["E!!4CHZ2&5A=GDZ.E]I;FET7V1Y;F%L;V%D97(H*3L*<F5Q
M=6ER92!)3SHZ1FEL93L*=VAI;&4@*&UY("1F:6QE;F%M92`]("AS;W)T(&ME
M>7,@)7)E<75I<F5?;&ES="E;,%TI('L*=6YL97-S("@D24Y#>R1F:6QE;F%M
M97T@;W(@)&9I;&5N86UE(#U^("]"4T1004XO*2!["FEF("@D9FEL96YA;64@
M/7X@+UPN<&UC/R0O:2D@>PIR97%U:7)E("1F:6QE;F%M93L*?0IE;'-E('L*
M9&\@)&9I;&5N86UE('5N;&5S<R`D9FEL96YA;64@/7X@+W-I=&5C=7-T;VUI
M>F5<+G!L)"\["GT*?0ID96QE=&4@)')E<75I<F5?;&ES='LD9FEL96YA;65]
M.PI]"G5N;&5S<R`H)&)U9B!E<2`B4$M<,#`S7#`P-"(I('L*;W5T<RAQ<5M.
M;R!Z:7`@9F]U;F0@869T97(@1DE,12!S96-T:6]N(&EN(&9I;&4@(B1P<F]G
M;F%M92)=*3L*;&%S="!-04=)0R`["GT*)'-T87)T7W!O<R`]("AT96QL(%]&
M2"D@+2`T.R`@("`@("`@("`@("`@("`C('-T87)T(&]F('II<`I]"FUY($!P
M87)?87)G<SL*;7D@*"1O=70L("1B=6YD;&4L("1L;V=F:"P@)&-A8VAE7VYA
M;64I.PID96QE=&4@)$5.5GM005)?05!07U)%55-%?3L@(R!S86YI=&EZ92`H
M4D554T4@;6%Y(&)E(&$@<V5C=7)I='D@<')O8FQE;2D*)'%U:65T(#T@,"!U
M;FQE<W,@)$5.5GM005)?1$5"54=].PII9B`H(21S=&%R=%]P;W,@;W(@*"1!
M4D=66S!=(&5Q("<M+7!A<BUO<'1I;VYS)R`F)B!S:&EF="DI('L*;7D@)61I
M<W1?8VUD(#T@<7<H"G`@("!B;&EB7W1O7W!A<@II("`@:6YS=&%L;%]P87(*
M=2`@('5N:6YS=&%L;%]P87(*<R`@('-I9VY?<&%R"G8@("!V97)I9GE?<&%R
M"BD["FEF("A`05)'5B!A;F0@)$%21U9;,%T@97$@)RTM<F5U<V4G*2!["G-H
M:69T($!!4D=6.PHD14Y6>U!!4E]!4%!?4D554T5](#T@<VAI9G0@0$%21U8[
M"GT*96QS92![(",@;F]R;6%L('!A<FP@8F5H879I;W5R"FUY($!A9&1?=&]?
M:6YC.PIW:&EL92`H0$%21U8I('L*)$%21U9;,%T@/7X@+UXM*%M!24U/0DQB
M<7!I=7-4=ETI*"XJ*2\@;W(@;&%S=#L*:68@*"0Q(&5Q("=))RD@>PIP=7-H
M($!A9&1?=&]?:6YC+"`D,CL*?0IE;'-I9B`H)#$@97$@)TTG*2!["F5V86P@
M(G5S92`D,B(["GT*96QS:68@*"0Q(&5Q("=!)RD@>PIU;G-H:69T($!P87)?
M87)G<RP@)#(["GT*96QS:68@*"0Q(&5Q("=/)RD@>PHD;W5T(#T@)#(["GT*
M96QS:68@*"0Q(&5Q("=B)RD@>PHD8G5N9&QE(#T@)W-I=&4G.PI]"F5L<VEF
M("@D,2!E<2`G0B<I('L*)&)U;F1L92`]("=A;&PG.PI]"F5L<VEF("@D,2!E
M<2`G<2<I('L*)'%U:65T(#T@,3L*?0IE;'-I9B`H)#$@97$@)TPG*2!["F]P
M96X@)&QO9V9H+"`B/CXB+"`D,B!O<B!D:64@<7%;0V%N)W0@;W!E;B!L;V<@
M9FEL92`B)#(B.B`D(5T["GT*96QS:68@*"0Q(&5Q("=4)RD@>PHD8V%C:&5?
M;F%M92`]("0R.PI]"G-H:69T*$!!4D=6*3L*:68@*&UY("1C;60@/2`D9&ES
M=%]C;61[)#%]*2!["F1E;&5T92`D14Y6>R=005)?5$5-4"=].PII;FET7VEN
M8R@I.PIR97%U:7)E(%!!4CHZ1&ES=#L*)GLB4$%2.CI$:7-T.CHD8VUD(GTH
M*2!U;FQE<W,@0$%21U8["B9[(E!!4CHZ1&ES=#HZ)&-M9")]*"1?*2!F;W(@
M0$%21U8["F5X:70["GT*?0IU;G-H:69T($!)3D,L($!A9&1?=&]?:6YC.PI]
M"GT*:68@*"1O=70I('L*>PIR97%U:7)E($E/.CI&:6QE.PIR97%U:7)E($%R
M8VAI=F4Z.EII<#L*<F5Q=6ER92!$:6=E<W0Z.E-(03L*?0IM>2`D<&%R(#T@
M<VAI9G0H0$%21U8I.PIM>2`D>FEP.PII9B`H9&5F:6YE9"`D<&%R*2!["F]P
M96X@;7D@)&9H+"`G/#IR87<G+"`D<&%R(&]R(&1I92!Q<5M#86XG="!F:6YD
M('!A<B!F:6QE("(D<&%R(CH@)"%=.PIB;&5S<R@D9F@L("=)3SHZ1FEL92<I
M.PHD>FEP(#T@07)C:&EV93HZ6FEP+3YN97<["B@@)'II<"T^<F5A9$9R;VU&
M:6QE2&%N9&QE*"1F:"P@)'!A<BD@/3T@07)C:&EV93HZ6FEP.CI!6E]/2R@I
M("D*;W(@9&EE('%Q6T5R<F]R(')E861I;F<@>FEP(&%R8VAI=F4@(B1P87(B
M73L*?0II9B`H9&5F:6YE9"`D<&%R*2!["F]P96X@;7D@)'!H+"`G/#IR87<G
M+"`D<&%R(&]R(&1I92!Q<5M#86XG="!R96%D('!A<B!F:6QE("(D<&%R(CH@
M)"%=.PIM>2`D8G5F.PIR96%D("1P:"P@)&)U9BP@-#L*9&EE('%Q6R(D<&%R
M(B!I<R!N;W0@82!P87(@9FEL95T@=6YL97-S("1B=68@97$@(E!+7#`P,UPP
M,#0B.PIC;&]S92`D<&@["GT*0W)E871E4&%T:"@D;W5T*2`["FUY("1F:"`]
M($E/.CI&:6QE+3YN97<H"B1O=70L"DE/.CI&:6QE.CI/7T-214%4*"D@?"!)
M3SHZ1FEL93HZ3U]74D].3%DH*2!\($E/.CI&:6QE.CI/7U1254Y#*"DL"C`W
M-S<L"BD@;W(@9&EE('%Q6T-A;B=T(&-R96%T92!F:6QE("(D;W5T(CH@)"%=
M.PHD9F@M/F)I;FUO9&4H*3L*<V5E:R!?1D@L(#`L(#`["FUY("1L;V%D97([
M"FEF("AD969I;F5D("1D871A7W!O<RD@>PIR96%D(%]&2"P@)&QO861E<BP@
M)&1A=&%?<&]S.PI](&5L<V4@>PIL;V-A;"`D+R`]('5N9&5F.PHD;&]A9&5R
M(#T@/%]&2#X["GT*:68@*"$D14Y6>U!!4E]615)"051)37T@86YD("1L;V%D
M97(@/7X@+UXH/SHC(7Q<0')E;2DO*2!["G)E<75I<F4@4$%2.CI&:6QT97(Z
M.E!O9%-T<FEP.PI005(Z.D9I;'1E<CHZ4&]D4W1R:7`M/F%P<&QY*%PD;&]A
M9&5R+"`D,"D["GT*)&9H+3YP<FEN="@D;&]A9&5R*0IO<B!D:64@<7%;17)R
M;W(@=W)I=&EN9R!L;V%D97(@=&\@(B1O=70B.B`D(5T["FEF("@D8G5N9&QE
M*2!["G)E<75I<F4@4$%2.CI(96%V>3L*4$%2.CI(96%V>3HZ7VEN:71?9'EN
M86QO861E<B@I.PII;FET7VEN8R@I.PIR97%U:7)E7VUO9'5L97,H*3L*;7D@
M0&EN8R`](&=R97`@>R`A+T)31%!!3B\@?0IG<F5P('L**"1B=6YD;&4@;F4@
M)W-I=&4G*2!O<@HH)%\@;F4@)$-O;F9I9SHZ0V]N9FEG>V%R8VAL:6)E>'!]
M(&%N9`HD7R!N92`D0V]N9FEG.CI#;VYF:6=[<')I=FQI8F5X<'TI.PI]($!)
M3D,["G-\+RM\+WQG+"!S?"\D?'P@9F]R96%C:"!`:6YC.PIM>2`E9FEL97,[
M"B1F:6QE<WLD7WTK*R!F;W(@0$1Y;F%,;V%D97(Z.F1L7W-H87)E9%]O8FIE
M8W1S+"!V86QU97,@)4E.0SL*;7D@)&QI8E]E>'0@/2`D0V]N9FEG.CI#;VYF
M:6=[;&EB7V5X='T[("`@("`@("`@(R!86%@@;&EB7V5X="!V<R!D;&5X="`_
M"FUY("5W<FET=&5N.PIF;W)E86-H(&UY("1K97D@*'-O<G0@:V5Y<R`E9FEL
M97,I('L*;7D@*"1F:6QE+"`D;F%M92D["FEF("AD969I;F5D*&UY("1F8R`]
M("1005(Z.DAE879Y.CI&=6QL0V%C:&5[)&ME>7TI*2!["B@D9FEL92P@)&YA
M;64I(#T@*"1K97DL("1F8RD["GT*96QS92!["F9O<F5A8V@@;7D@)&1I<B`H
M0&EN8RD@>PII9B`H)&ME>2`]?B!M?%Y<421D:7)<12\H+BHI)'QI*2!["B@D
M9FEL92P@)&YA;64I(#T@*"1K97DL("0Q*3L*;&%S=#L*?0II9B`H)&ME>2`]
M?B!M?%XO;&]A9&5R+UM>+UTK+R@N*BDD?"D@>PII9B`H;7D@)')E9B`]("1-
M;V1U;&5#86-H97LD,7TI('L**"1F:6QE+"`D;F%M92D@/2`H)')E9BP@)#$I
M.PIL87-T.PI]"FEF("@M9B`B)&1I<B\D,2(I('L**"1F:6QE+"`D;F%M92D@
M/2`H(B1D:7(O)#$B+"`D,2D["FQA<W0["GT*?0I]"GT*;F5X="!U;FQE<W,@
M9&5F:6YE9"`D;F%M93L*;F5X="!I9B`D=W)I='1E;GLD;F%M97TK*SL*;F5X
M="!I9B`A<F5F*"1F:6QE*2!A;F0@)&9I;&4@/7X@+UPN7%$D;&EB7V5X=%Q%
M)"]I.PIO=71S*'-P<FEN=&8H<7%;4&%C:VEN9R!&24Q%("(E<R(N+BY=+"!R
M968@)&9I;&4@/R`D9FEL92T^>VYA;65](#H@)&9I;&4I*3L*;7D@)&-O;G1E
M;G0["FEF("AR968H)&9I;&4I*2!["B1C;VYT96YT(#T@)&9I;&4M/GMB=69]
M.PI]"F5L<V4@>PIL;V-A;"`D+R`]('5N9&5F.PIO<&5N(&UY("1T:"P@)SPZ
M<F%W)RP@)&9I;&4@;W(@9&EE('%Q6T-A;B=T(')E860@(B1F:6QE(CH@)"%=
M.PHD8V]N=&5N="`](#PD=&@^.PIC;&]S92`D=&@["E!!4CHZ1FEL=&5R.CI0
M;V13=')I<"T^87!P;'DH7"1C;VYT96YT+"`B/&5M8F5D9&5D/B\D;F%M92(I
M"FEF("$D14Y6>U!!4E]615)"051)37T@86YD("1N86UE(#U^("]<+B@_.G!M
M?&EX?&%L*20O:3L*4$%2.CI&:6QT97(Z.E!A=&-H0V]N=&5N="T^;F5W+3YA
M<'!L>2A<)&-O;G1E;G0L("1F:6QE+"`D;F%M92D["GT*)&9H+3YP<FEN="@B
M1DE,12(L"G!A8VLH)TXG+"!L96YG=&@H)&YA;64I("L@.2DL"G-P<FEN=&8H
M(B4P.'@O)7,B+"!!<F-H:79E.CI::7`Z.F-O;7!U=&5#4D,S,B@D8V]N=&5N
M="DL("1N86UE*2P*<&%C:R@G3B<L(&QE;F=T:"@D8V]N=&5N="DI+`HD8V]N
M=&5N="D*;W(@9&EE('%Q6T5R<F]R('=R:71I;F<@96UB961D960@1DE,12!T
M;R`B)&]U="(Z("0A73L*;W5T<RAQ<5M7<FET=&5N(&%S("(D;F%M92)=*3L*
M?0I]"FEF("@D>FEP*2!["B1Z:7`M/G=R:71E5&]&:6QE2&%N9&QE*"1F:"D@
M/3T@07)C:&EV93HZ6FEP.CI!6E]/2PIO<B!D:64@<7%;17)R;W(@=W)I=&EN
M9R!Z:7`@<&%R="!O9B`B)&]U=")=.PI]"B1C86-H95]N86UE(#T@<W5B<W1R
M("1C86-H95]N86UE+"`P+"`T,#L*:68@*"$D8V%C:&5?;F%M92!A;F0@;7D@
M)&UT:6UE(#T@*'-T870H)&]U="DI6SE=*2!["FUY("1C='@@/2!$:6=E<W0Z
M.E-(02T^;F5W*#$I.PIO<&5N(&UY("1T:"P@(CPZ<F%W(BP@)&]U=#L*)&-T
M>"T^861D9FEL92@D=&@I.PIC;&]S92`D=&@["B1C86-H95]N86UE(#T@)&-T
M>"T^:&5X9&EG97-T.PI]"B1C86-H95]N86UE("X](")<,"(@>"`H-#`@+2!L
M96YG=&@@)&-A8VAE7VYA;64I.PHD8V%C:&5?;F%M92`N/2`B7#!#04-(12([
M"FUY("1O9F9S970@/2`D9F@M/G1E;&P@*R!L96YG=&@H)&-A8VAE7VYA;64I
M("T@;&5N9W1H*"1L;V%D97(I.PHD9F@M/G!R:6YT*"1C86-H95]N86UE+`IP
M86-K*"=.)RP@)&]F9G-E="DL"B1005)?34%'24,I"B8F("1F:"T^8VQO<V4*
M;W(@9&EE('%Q6T5R<F]R('=R:71I;F<@=')A:6QE<B!O9B`B)&]U="(Z("0A
M73L*8VAM;V0@,#<U-2P@)&]U=#L*97AI=#L*?0I["FQA<W0@=6YL97-S(&1E
M9FEN960@)'-T87)T7W!O<SL*7V9I>%]P<F]G;F%M92@I.PIR97%U:7)E(%!!
M4CL*4$%2.CI(96%V>3HZ7VEN:71?9'EN86QO861E<B@I.PI["G)E<75I<F4@
M1FEL93HZ1FEN9#L*<F5Q=6ER92!!<F-H:79E.CI::7`["GT*;7D@)&9H(#T@
M24\Z.D9I;&4M/FYE=SL@("`@("`@("`@("`@("`@("`@("`@("`@("`@(",@
M07)C:&EV93HZ6FEP(&]P97)A=&5S(&]N(&%N($E/.CI(86YD;&4*)&9H+3YF
M9&]P96XH9FEL96YO*%]&2"DL("=R)RD@;W(@9&EE('%Q6V9D;W!E;B@I(&9A
M:6QE9#H@)"%=.PI!<F-H:79E.CI::7`Z.G-E=$-H=6YK4VEZ92@M<R!?1D@I
M.PIM>2`D>FEP(#T@07)C:&EV93HZ6FEP+3YN97<["B@D>FEP+3YR96%D1G)O
M;49I;&5(86YD;&4H)&9H+"`D<')O9VYA;64I(#T]($%R8VAI=F4Z.EII<#HZ
M05I?3TLH*2D*;W(@9&EE('%Q6T5R<F]R(')E861I;F<@>FEP(&%R8VAI=F4@
M(B1P<F]G;F%M92)=.PI!<F-H:79E.CI::7`Z.G-E=$-H=6YK4VEZ92@V-"`J
M(#$P,C0I.PHD<75I970@/2`A)$5.5GM005)?1$5"54=].PIO=71S*")296%D
M:6YG($U%5$$N>6UL+BXN(BD["FEF("AM>2`D;65T82`]("1Z:7`M/F-O;G1E
M;G1S*"=-151!+GEM;"<I*2!["B1M971A(#U^(',O+BI><&%R.EQS*B0O+VUS
M.PHD;65T82`]?B!S+UY<4RXJ+R]M<SL*:68@*&UY("@D8VQE86XI(#T@)&UE
M=&$@/7X@+UY<<RMC;&5A;CH@*"XJ*20O;2D@>PHD8VQE86X@/7X@+UY<<RI\
M7',J)"]G.PHD14Y6>U!!4E]#3$5!3GT@/2`Q(&EF("1C;&5A;CL*?0I]"G!U
M<V@@0%!!4CHZ3&EB0V%C:&4L("1Z:7`["B1005(Z.DQI8D-A8VAE>R1P<F]G
M;F%M97T@/2`D>FEP.PIO=71S*")%>'1R86-T:6YG('II<"XN+B(I.PII9B`H
M9&5F:6YE9"`D14Y6>U!!4E]414U0?2D@>R`C('-H;W5L9"!B92!S970@870@
M=&AI<R!P;VEN="$*9F]R96%C:"!M>2`D;65M8F5R("@@)'II<"T^;65M8F5R
M<R`I('L*;F5X="!I9B`D;65M8F5R+3YI<T1I<F5C=&]R>3L*;7D@)&UE;6)E
M<E]N86UE(#T@)&UE;6)E<BT^9FEL94YA;64["FYE>'0@=6YL97-S("1M96UB
M97)?;F%M92`]?B!M>PI>"B\_<VAL:6(O"B@_.B1#;VYF:6<Z.D-O;F9I9WMV
M97)S:6]N?2\I/PHH/SHD0V]N9FEG.CI#;VYF:6=[87)C:&YA;65]+RD_"BA;
M7B]=*RD*)`I]>#L*;7D@)&5X=')A8W1?;F%M92`]("0Q.PIM>2`D9&5S=%]N
M86UE(#T@1FEL93HZ4W!E8RT^8V%T9FEL92@D14Y6>U!!4E]414U0?2P@)&5X
M=')A8W1?;F%M92D["FEF("@M9B`D9&5S=%]N86UE("8F("US(%\@/3T@)&UE
M;6)E<BT^=6YC;VUP<F5S<V5D4VEZ92@I*2!["F]U=',H<7%;4VMI<'!I;F<@
M(B1M96UB97)?;F%M92(@<VEN8V4@:70@86QR96%D>2!E>&ES=',@870@(B1D
M97-T7VYA;64B72D["GT@96QS92!["F]U=',H<7%;17AT<F%C=&EN9R`B)&UE
M;6)E<E]N86UE(B!T;R`B)&1E<W1?;F%M92(N+BY=*3L**"1M96UB97(M/F5X
M=')A8W14;T9I;&5.86UE9"@D9&5S=%]N86UE*2`]/2!!<F-H:79E.CI::7`Z
M.D%:7T]+*"DI"F]R(&1I92!Q<5M%<G)O<B!E>'1R86-T:6YG('II<"!M96UB
M97(@=&\@(B1D97-T7VYA;64B73L*8VAM;V0H,#4U-2P@)&1E<W1?;F%M92D@
M:68@)%Y/(&5Q(")H<'5X(CL*?0I]"GT*?0IU;FQE<W,@*"1005(Z.DQI8D-A
M8VAE>R1P<F]G;F%M97TI('L*9&EE(#P\("(N(B!U;FQE<W,@0$%21U8["E5S
M86=E.B`D,"!;("U!;&EB+G!A<B!=(%L@+4ED:7(@72!;("U-;6]D=6QE(%T@
M6R!S<F,N<&%R(%T@6R!P<F]G<F%M+G!L(%T*)#`@6R`M0GPM8B!=(%LM3V]U
M=&9I;&5=('-R8RYP87(*+@HD14Y6>U!!4E]04D]'3D%-17T@/2`D<')O9VYA
M;64@/2`D,"`]('-H:69T*$!!4D=6*3L*?0IS=6(@0W)E871E4&%T:"!["FUY
M("@D;F%M92D@/2!`7SL*<F5Q=6ER92!&:6QE.CI"87-E;F%M93L*;7D@*"1B
M87-E;F%M92P@)'!A=&@L("1E>'0I(#T@1FEL93HZ0F%S96YA;64Z.F9I;&5P
M87)S92@D;F%M92P@*"=<+BXJ)RDI.PIR97%U:7)E($9I;&4Z.E!A=&@["D9I
M;&4Z.E!A=&@Z.FUK<&%T:"@D<&%T:"D@=6YL97-S*"UE("1P871H*3L@(R!M
M:W!A=&@@9&EE<R!W:71H(&5R<F]R"GT*<W5B(')E<75I<F5?;6]D=6QE<R![
M"G)E<75I<F4@;&EB.PIR97%U:7)E($1Y;F%,;V%D97(["G)E<75I<F4@:6YT
M96=E<CL*<F5Q=6ER92!S=')I8W0["G)E<75I<F4@=V%R;FEN9W,["G)E<75I
M<F4@=F%R<SL*<F5Q=6ER92!#87)P.PIR97%U:7)E($-A<G`Z.DAE879Y.PIR
M97%U:7)E($5R<FYO.PIR97%U:7)E($5X<&]R=&5R.CI(96%V>3L*<F5Q=6ER
M92!%>'!O<G1E<CL*<F5Q=6ER92!&8VYT;#L*<F5Q=6ER92!&:6QE.CI496UP
M.PIR97%U:7)E($9I;&4Z.E-P96,["G)E<75I<F4@6%-,;V%D97(["G)E<75I
M<F4@0V]N9FEG.PIR97%U:7)E($E/.CI(86YD;&4["G)E<75I<F4@24\Z.D9I
M;&4["G)E<75I<F4@0V]M<')E<W,Z.EIL:6(["G)E<75I<F4@07)C:&EV93HZ
M6FEP.PIR97%U:7)E($1I9V5S=#HZ4TA!.PIR97%U:7)E(%!!4CL*<F5Q=6ER
M92!005(Z.DAE879Y.PIR97%U:7)E(%!!4CHZ1&ES=#L*<F5Q=6ER92!005(Z
M.D9I;'1E<CHZ4&]D4W1R:7`["G)E<75I<F4@4$%2.CI&:6QT97(Z.E!A=&-H
M0V]N=&5N=#L*<F5Q=6ER92!A='1R:6)U=&5S.PIE=F%L('L@<F5Q=6ER92!#
M=V0@?3L*979A;"![(')E<75I<F4@5VEN,S(@?3L*979A;"![(')E<75I<F4@
M4V-A;&%R.CI5=&EL('T["F5V86P@>R!R97%U:7)E($%R8VAI=F4Z.E5N>FEP
M.CI"=7)S="!].PIE=F%L('L@<F5Q=6ER92!4:64Z.DAA<V@Z.DYA;65D0V%P
M='5R92!].PIE=F%L('L@<F5Q=6ER92!097)L24\[(')E<75I<F4@4&5R;$E/
M.CIS8V%L87(@?3L*979A;"![(')E<75I<F4@=71F."!].PI]"G-U8B!?<V5T
M7W!A<E]T96UP('L*:68@*&1E9FEN960@)$5.5GM005)?5$5-4'T@86YD("1%
M3E9[4$%27U1%35!](#U^("\H+BLI+RD@>PHD<&%R7W1E;7`@/2`D,3L*<F5T
M=7)N.PI]"F9O<F5A8V@@;7D@)'!A=&@@*`HH;6%P("1%3E9[)%]]+"!Q=R@@
M4$%27U1-4$1)4B!435!$25(@5$5-4$1)4B!414U0(%1-4"`I*2P*<7<H($,Z
M7%Q414U0("]T;7`@+B`I"BD@>PIN97AT('5N;&5S<R!D969I;F5D("1P871H
M(&%N9"`M9"`D<&%T:"!A;F0@+7<@)'!A=&@["FUY("1U<V5R;F%M93L*;7D@
M)'!W=6ED.PIE=F%L('LH)'!W=6ED*2`](&=E='!W=6ED*"0^*2!I9B!D969I
M;F5D("0^.WT["FEF("@@9&5F:6YE9"@F5VEN,S(Z.DQO9VEN3F%M92D@*2![
M"B1U<V5R;F%M92`]("97:6XS,CHZ3&]G:6Y.86UE.PI]"F5L<VEF("AD969I
M;F5D("1P=W5I9"D@>PHD=7-E<FYA;64@/2`D<'=U:60["GT*96QS92!["B1U
M<V5R;F%M92`]("1%3E9[55-%4DY!345]('Q\("1%3E9[55-%4GT@?'P@)U-9
M4U1%32<["GT*)'5S97)N86UE(#U^(',O7%<O7R]G.PIM>2`D<W1M<&1I<B`]
M("(D<&%T:"1S>7-[7V1E;&EM?7!A<BTB+G5N<&%C:R@B2"HB+"`D=7-E<FYA
M;64I.PIM:V1I<B`D<W1M<&1I<BP@,#<U-3L*:68@*"$D14Y6>U!!4E]#3$5!
M3GT@86YD(&UY("1M=&EM92`]("AS=&%T*"1P<F]G;F%M92DI6SE=*2!["F]P
M96X@;7D@)&9H+"`B/#IR87<B+"`D<')O9VYA;64@;W(@9&EE('%Q6T-A;B=T
M(')E860@(B1P<F]G;F%M92(Z("0A73L*<V5E:R`D9F@L("TQ."P@,CL*;7D@
M)&)U9CL*<F5A9"`D9F@L("1B=68L(#8["FEF("@D8G5F(&5Q(")<,$-!0TA%
M(BD@>PIS965K("1F:"P@+34X+"`R.PIR96%D("1F:"P@)&)U9BP@-#$["B1B
M=68@/7X@<R]<,"\O9SL*)'-T;7!D:7(@+CT@(B1S>7-[7V1E;&EM?6-A8VAE
M+21B=68B.PI]"F5L<V4@>PIM>2`D9&EG97-T(#T@979A;`I["G)E<75I<F4@
M1&EG97-T.CI32$$["FUY("1C='@@/2!$:6=E<W0Z.E-(02T^;F5W*#$I.PIO
M<&5N(&UY("1F:"P@(CPZ<F%W(BP@)'!R;V=N86UE(&]R(&1I92!Q<5M#86XG
M="!R96%D("(D<')O9VYA;64B.B`D(5T["B1C='@M/F%D9&9I;&4H)&9H*3L*
M8VQO<V4H)&9H*3L*)&-T>"T^:&5X9&EG97-T.PI]('Q\("1M=&EM93L*)'-T
M;7!D:7(@+CT@(B1S>7-[7V1E;&EM?6-A8VAE+21D:6=E<W0B.PI]"F-L;W-E
M*"1F:"D["GT*96QS92!["B1%3E9[4$%27T-,14%.?2`](#$["B1S=&UP9&ER
M("X]("(D<WES>U]D96QI;7UT96UP+20D(CL*?0HD14Y6>U!!4E]414U0?2`]
M("1S=&UP9&ER.PIM:V1I<B`D<W1M<&1I<BP@,#<U-3L*;&%S=#L*?0HD<&%R
M7W1E;7`@/2`D,2!I9B`D14Y6>U!!4E]414U0?2!A;F0@)$5.5GM005)?5$5-
M4'T@/7X@+R@N*RDO.PI]"G-U8B!?<V%V95]A<R!["FUY("@D;F%M92P@)&-O
M;G1E;G1S+"`D;6]D92D@/2!`7SL*;7D@)&9U;&QN86UE(#T@(B1P87)?=&5M
M<"\D;F%M92(["G5N;&5S<R`H+64@)&9U;&QN86UE*2!["FUY("1T96UP;F%M
M92`]("(D9G5L;&YA;64N)"0B.PIM>2`D9F@["BAO<&5N("1F:"P@)SXZ<F%W
M)RP@)'1E;7!N86UE*0HF)B`H<')I;G0@)&9H("1C;VYT96YT<RD*)B8@*&-L
M;W-E("1F:"D*;W(@9&EE('%Q6T5R<F]R('=R:71I;F<@(B1T96UP;F%M92(Z
M("0A73L*8VAM;V0@)&UO9&4L("1T96UP;F%M92!I9B!D969I;F5D("1M;V1E
M.PIR96YA;64H)'1E;7!N86UE+"`D9G5L;&YA;64I(&]R('5N;&EN:R@D=&5M
M<&YA;64I.PI]"G)E='5R;B`D9G5L;&YA;64["GT*<W5B(%]S971?<')O9VYA
M;64@>PII9B`H9&5F:6YE9"`D14Y6>U!!4E]04D]'3D%-17T@86YD("1%3E9[
M4$%27U!23T=.04U%?2`]?B`O*"XK*2\I('L*)'!R;V=N86UE(#T@)#$["GT*
M)'!R;V=N86UE('Q\/2`D,#L*:68@*"1%3E9[4$%27U1%35!](&%N9"!I;F1E
M>"@D<')O9VYA;64L("1%3E9[4$%27U1%35!]*2`^/2`P*2!["B1P<F]G;F%M
M92`]('-U8G-T<B@D<')O9VYA;64L(')I;F1E>"@D<')O9VYA;64L("1S>7-[
M7V1E;&EM?2D@*R`Q*3L*?0II9B`H(21%3E9[4$%27U!23T=.04U%?2!O<B!I
M;F1E>"@D<')O9VYA;64L("1S>7-[7V1E;&EM?2D@/CT@,"D@>PII9B`H;W!E
M;B!M>2`D9F@L("<\)RP@)'!R;V=N86UE*2!["G)E='5R;B!I9B`M<R`D9F@[
M"GT*:68@*"US("(D<')O9VYA;64D<WES>U]E>&5](BD@>PHD<')O9VYA;64@
M+CT@)'-Y<WM?97AE?3L*<F5T=7)N.PI]"GT*9F]R96%C:"!M>2`D9&ER("AS
M<&QI="`O7%$D<WES>W!A=&A?<V5P?5Q%+RP@)$5.5GM0051(?2D@>PIN97AT
M(&EF(&5X:7-T<R`D14Y6>U!!4E]414U0?2!A;F0@)&1I<B!E<2`D14Y6>U!!
M4E]414U0?3L*)&1I<B`]?B!S+UQ1)'-Y<WM?9&5L:6U]7$4D+R\["B@H)'!R
M;V=N86UE(#T@(B1D:7(D<WES>U]D96QI;7TD<')O9VYA;64D<WES>U]E>&5]
M(BDL(&QA<W0I"FEF("US("(D9&ER)'-Y<WM?9&5L:6U])'!R;V=N86UE)'-Y
M<WM?97AE?2(["B@H)'!R;V=N86UE(#T@(B1D:7(D<WES>U]D96QI;7TD<')O
M9VYA;64B*2P@;&%S="D*:68@+7,@(B1D:7(D<WES>U]D96QI;7TD<')O9VYA
M;64B.PI]"GT*<W5B(%]F:7A?<')O9VYA;64@>PHD,"`]("1P<F]G;F%M92!\
M?#T@)$5.5GM005)?4%)/1TY!345].PII9B`H:6YD97@H)'!R;V=N86UE+"`D
M<WES>U]D96QI;7TI(#P@,"D@>PHD<')O9VYA;64@/2`B+B1S>7-[7V1E;&EM
M?21P<F]G;F%M92(["GT*;7D@)'!W9"`]("AD969I;F5D("9#=V0Z.F=E=&-W
M9"D@/R!#=V0Z.F=E=&-W9"@I"CH@*"AD969I;F5D("97:6XS,CHZ1V5T0W=D
M*2`_(%=I;C,R.CI'971#=V0H*2`Z(&!P=V1@*3L*8VAO;7`H)'!W9"D["B1P
M<F]G;F%M92`]?B!S+UXH/SU<+EPN/UQ1)'-Y<WM?9&5L:6U]7$4I+R1P=V0D
M<WES>U]D96QI;7TO.PHD14Y6>U!!4E]04D]'3D%-17T@/2`D<')O9VYA;64[
M"GT*<W5B(%]P87)?:6YI=%]E;G8@>PII9B`H("1%3E9[4$%27TE.251)04Q)
M6D5$?2LK(#T](#$@*2!["G)E='5R;CL*?2!E;'-E('L*)$5.5GM005)?24Y)
M5$E!3$E:141](#T@,CL*?0IF;W(@*'%W*"!34$%73D5$(%1%35`@0TQ%04X@
M1$5"54<@0T%#2$4@4%)/1TY!344@*2`I('L*9&5L971E("1%3E9[(E!!4E\D
M7R)].PI]"F9O<B`H<7<O(%1-4$1)4B!414U0($-,14%.($1%0E5'("\I('L*
M)$5.5GLB4$%27R1?(GT@/2`D14Y6>R)005)?1TQ/0D%,7R1?(GT@:68@97AI
M<W1S("1%3E9[(E!!4E]'3$]"04Q?)%\B?3L*?0II9B`H)$5.5GM005)?5$5-
M4'TI('L*9&5L971E("1%3E9[4$%27T-,14%.?3L*?0I]"G-U8B!O=71S('L*
M<F5T=7)N(&EF("1Q=6EE=#L*:68@*"1L;V=F:"D@>PIP<FEN="`D;&]G9F@@
M(D!?7&XB.PI]"F5L<V4@>PIP<FEN="`B0%]<;B(["GT*?0IS=6(@:6YI=%]I
M;F,@>PIR97%U:7)E($-O;F9I9SL*<'5S:"!`24Y#+"!G<F5P(&1E9FEN960L
M(&UA<"`D0V]N9FEG.CI#;VYF:6=[)%]]+"!Q=R@*87)C:&QI8F5X<"!P<FEV
M;&EB97AP('-I=&5A<F-H97AP('-I=&5L:6)E>'`*=F5N9&]R87)C:&5X<"!V
M96YD;W)L:6)E>'`**3L*?0IP86-K86=E(&UA:6X["G)E<75I<F4@4$%2.PIU
M;G-H:69T($!)3D,L(%PF4$%2.CIF:6YD7W!A<CL*4$%2+3YI;7!O<G0H0'!A
M<E]A<F=S*3L*9&EE('%Q6W!A<BYP;#H@0V%N)W0@;W!E;B!P97)L('-C<FEP
M="`B)'!R;V=N86UE(CH@3F\@<W5C:"!F:6QE(&]R(&1I<F5C=&]R>5QN70IU
M;FQE<W,@+64@)'!R;V=N86UE.PID;R`D<')O9VYA;64["D-/4D4Z.F5X:70H
M)#$I(&EF("@D0"`]?B]>7U1+7T58251?7"@H7&0K*5PI+RD["F1I92`D0"!I
M9B`D0#L*?3L*)#HZ7U]%4E)/4B`]("1`(&EF("1`.PI]"D-/4D4Z.F5X:70H
M)#$I(&EF("@D.CI?7T524D]2(#U^+UY?5$M?15A)5%]<*"A<9"LI7"DO*3L*
M9&EE("0Z.E]?15)23U(@:68@)#HZ7U]%4E)/4CL*,3L*7U]%3D1?7PH`<&5R
M;'AS:2YC````/0```$`````+````0````!````!`````)@```$`````U````
M0`````,```!`````!@```$`````4````0`````X```!`````#0```$`````E
M````0````!(```!`````*````$`````"````0````!@```!`````"0```$``
M```*````0```````````````@;(!@;"PK`<`````OK(!@;"PKS\`````J(K_
M?["J"8!4C?]_`0```%B0_W^PJQZ`U)'_?\S__W\$D_]_`0```$"5_W^PJ`.`
MY)7_?P$```!PH?]_L+"H@(RA_W\!````^*+_?ZC__W\@J_]_L*L"@.RM_W\!
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``#4&0``8!D```$````]`P```0```$@#```!````4@,```$```!F`P``#P``
M`.8$```,````P!(```T```#@20``&0```+"-`0`;````!````!H```"TC0$`
M'`````0```#U_O]OM`$```4```"P"0``!@```!`#```*````"04```L````0
M````%0`````````#````T(X!``(```#@`0``%````!$````7````X!```!$`
M```0$```$@```-`````3````"````!@`````````^___;P$```C^__]OD`\`
M`/___V\#````\/__;[H.``#Z__]O#@``````````````````````````````
M``````````````````````````"XC0$```````````#,$@``S!(``,P2``#,
M$@``S!(``,P2``#,$@``S!(``,P2``#,$@``S!(``,P2``#,$@``S!(``,P2
M``#,$@``S!(``,P2``#,$@``S!(``,P2``#,$@``S!(``,P2``#,$@``S!(`
M`,P2``#,$@``S!(``,P2``#,$@``S!(``,P2``#,$@``S!(``,P2``#,$@``
MS!(``,P2``#,$@``S!(``,P2``#,$@``S!(``,P2``#,$@``S!(``,P2``#,
M$@``S!(``,P2``#,$@``S!(``,P2``#,$@``S!(``,P2``#,$@``S!(``,P2
M````````````````````````````````````````````````````````````
M``````"P%0````````20`0#_____G$T``-A*````````F$L``)Q+``"030``
MX$H``)A-````````?$T``(1-````````I$T``$=#0SH@*$=.52D@,3(N,2XP
M`$=#0SH@*&-R;W-S=&]O;"U.1R`Q+C(U+C`N,CA?8C@X9#,S."D@,3(N,2XP
M`$$T````865A8FD``2H````%-RU!``8*!T$(`0D""@,,`1($$P$4`14!%P,8
M`1D!&@(<`2(!`"YS:'-T<G1A8@`N:6YT97)P`"YN;W1E+F=N=2YB=6EL9"UI
M9``N;F]T92Y!0DDM=&%G`"YG;G4N:&%S:``N9'EN<WEM`"YD>6YS='(`+F=N
M=2YV97)S:6]N`"YG;G4N=F5R<VEO;E]R`"YR96PN9'EN`"YR96PN<&QT`"YI
M;FET`"YT97AT`"YF:6YI`"YR;V1A=&$`+D%232YE>'1A8@`N05)-+F5X:61X
M`"YE:%]F<F%M90`N:6YI=%]A<G)A>0`N9FEN:5]A<G)A>0`N9'EN86UI8P`N
M9V]T`"YD871A`"YB<W,`+F-O;6UE;G0`+D%232YA='1R:6)U=&5S````````
M```````````````````````````````````````````````+`````0````(`
M``!4`0``5`$``!D```````````````$`````````$P````<````"````<`$`
M`'`!```D```````````````$`````````"8````'`````@```)0!``"4`0``
M(```````````````!``````````T````]O__;P(```"T`0``M`$``%P!```%
M``````````0````$````/@````L````"````$`,``!`#``"@!@``!@````,`
M```$````$````$8````#`````@```+`)``"P"0``"04``````````````0``
M``````!.````____;P(```"Z#@``N@X``-0````%``````````(````"````
M6P```/[__V\"````D`\``)`/``"`````!@````,````$`````````&H````)
M`````@```!`0```0$```T`````4`````````!`````@```!S````"0```$(`
M``#@$```X!```.`!```%````%@````0````(````?`````$````&````P!(`
M`,`2```,```````````````$`````````'<````!````!@```,P2``#,$@``
MY`(`````````````!`````0```""`````0````8```"P%0``L!4``#`T````
M``````````@`````````B`````$````&````X$D``.!)```(````````````
M```$`````````(X````!`````@```.A)``#H20``"$$`````````````!```
M``````"6`````0````(```#PB@``\(H``!@```````````````0`````````
MH0````$``'""````"(L```B+``!@````#0`````````$`````````*P````!
M`````@```&B+``!HBP``!```````````````!`````````"V````#@````,`
M``"PC0$`L(T```0```````````````0````$````P@````\````#````M(T!
M`+2-```$```````````````$````!````,X````&`````P```+B-`0"XC0``
M&`$```8`````````!`````@```#7`````0````,```#0C@$`T(X``#`!````
M``````````0````$````W`````$````#`````)`!``"0``!`````````````
M```$`````````.(````(`````P```$"0`0!`D```&!``````````````"```
M``````#G`````0```#``````````0)```#\```````````````$````!````
M\`````,``'```````````'^0```U```````````````!``````````$````#
C``````````````"TD`````$``````````````0``````````
