/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8; coding: utf-8 -*-
 * gtksourceversion.h
 * This file is part of GtkSourceView
 *
 * Copyright (C) 2015 - Paolo Borelli <pborelli@gnome.org>
 *
 * GtkSourceView is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * GtkSourceView is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, see <http://www.gnu.org/licenses/>.
 */

#pragma once

#include <glib.h>

G_BEGIN_DECLS

/**
 * GTK_SOURCE_MAJOR_VERSION:
 *
 * Like gtk_source_get_major_version(), but from the headers used at
 * application compile time, rather than from the library linked
 * against at application run time.
 */
#define GTK_SOURCE_MAJOR_VERSION (5)

/**
 * GTK_SOURCE_MINOR_VERSION:
 *
 * Like gtk_source_get_minor_version(), but from the headers used at
 * application compile time, rather than from the library linked
 * against at application run time.
 */
#define GTK_SOURCE_MINOR_VERSION (4)

/**
 * GTK_SOURCE_MICRO_VERSION:
 *
 * Like gtk_source_get_micro_version(), but from the headers used at
 * application compile time, rather than from the library linked
 * against at application run time.
 */
#define GTK_SOURCE_MICRO_VERSION (2)


/**
 * GTK_SOURCE_CHECK_VERSION:
 * @major: major version (e.g. 3 for version 3.20.0)
 * @minor: minor version (e.g. 20 for version 3.20.0)
 * @micro: micro version (e.g. 0 for version 3.20.0)
 *
 * Macro to test the version of GtkSourceView being compiled against.
 *
 * Returns: %TRUE if the version of the GtkSourceView header files
 * is the same as or newer than the passed-in version.
 */
#define GTK_SOURCE_CHECK_VERSION(major, minor, micro) \
	(GTK_SOURCE_MAJOR_VERSION > (major) || \
	(GTK_SOURCE_MAJOR_VERSION == (major) && GTK_SOURCE_MINOR_VERSION > (minor)) || \
	(GTK_SOURCE_MAJOR_VERSION == (major) && GTK_SOURCE_MINOR_VERSION == (minor) && \
	 GTK_SOURCE_MICRO_VERSION >= (micro)))

#ifndef _GTK_SOURCE_EXTERN
#define _GTK_SOURCE_EXTERN extern
#endif

#ifndef __GTK_DOC_IGNORE__
#define GTK_SOURCE_ENUM_EXTERN _GTK_SOURCE_EXTERN
#endif

/**
 * GTK_SOURCE_VERSION_5_0:
 *
 * A macro that evaluates to the 5.0 version of GtkSourceView,
 * in a format that can be used by the C pre-processor.
 *
 * Since: 5.0
 */
#define GTK_SOURCE_VERSION_5_0 (G_ENCODE_VERSION (5, 0))

/**
 * GTK_SOURCE_VERSION_5_2:
 *
 * A macro that evaluates to the 5.2 version of GtkSourceView,
 * in a format that can be used by the C pre-processor.
 *
 * Since: 5.2
 */
#define GTK_SOURCE_VERSION_5_2 (G_ENCODE_VERSION (5, 2))

/**
 * GTK_SOURCE_VERSION_5_4:
 *
 * A macro that evaluates to the 5.4 version of GtkSourceView,
 * in a format that can be used by the C pre-processor.
 *
 * Since: 5.4
 */
#define GTK_SOURCE_VERSION_5_4 (G_ENCODE_VERSION (5, 4))

/* Define GTK_SOURCE_VERSION_CUR_STABLE */
#ifndef __GTK_DOC_IGNORE__
#  if (GTK_SOURCE_MINOR_VERSION % 2)
#    define GTK_SOURCE_VERSION_CUR_STABLE (G_ENCODE_VERSION (GTK_SOURCE_MAJOR_VERSION, GTK_SOURCE_MINOR_VERSION + 1))
#  else
#    define GTK_SOURCE_VERSION_CUR_STABLE (G_ENCODE_VERSION (GTK_SOURCE_MAJOR_VERSION, GTK_SOURCE_MINOR_VERSION))
#  endif
#endif /* __GTK_DOC_IGNORE__ */

/* Define GTK_SOURCE_VERSION_PREV_STABLE */
#ifndef __GTK_DOC_IGNORE__
#if (GTK_SOURCE_MINOR_VERSION % 2)
#define GTK_SOURCE_VERSION_PREV_STABLE (G_ENCODE_VERSION (GTK_SOURCE_MAJOR_VERSION, GTK_SOURCE_MINOR_VERSION - 1))
#else
#define GTK_SOURCE_VERSION_PREV_STABLE (G_ENCODE_VERSION (GTK_SOURCE_MAJOR_VERSION, GTK_SOURCE_MINOR_VERSION - 2))
#endif
#endif /* __GTK_DOC_IGNORE__ */

/**
 * GTK_SOURCE_VERSION_MIN_REQUIRED:
 *
 * A macro that should be defined by the user prior to including
 * the gtksource.h header.
 * The definition should be one of the predefined version
 * macros: %GTK_SOURCE_VERSION_3_0, %GTK_SOURCE_VERSION_3_2, ...
 *
 * This macro defines the lower bound for the GtkSourceView API to use.
 *
 * If a function has been deprecated in a newer version of GtkSourceView,
 * defining this symbol hides the compiler warnings for those functions
 * without disabling warnings for the other deprecated functions.
 *
 * <warning>
 * Warning: if you define this macro, do not forget to update it! Especially
 * when writing new code. Otherwise you can miss the new deprecations.
 * </warning>
 */
#ifndef GTK_SOURCE_VERSION_MIN_REQUIRED
#define GTK_SOURCE_VERSION_MIN_REQUIRED (GTK_SOURCE_VERSION_CUR_STABLE)
#endif

/**
 * GTK_SOURCE_VERSION_MAX_ALLOWED:
 *
 * A macro that should be defined by the user prior to including
 * the gtksource.h header.
 * The definition should be one of the predefined version
 * macros: %GTK_SOURCE_VERSION_3_0, %GTK_SOURCE_VERSION_3_2, ...
 *
 * This macro defines the upper bound for the GtkSourceView API to use.
 *
 * If a function has been introduced in a newer version of GtkSourceView,
 * it is possible to use this symbol to get compiler warnings when
 * trying to use that function.
 */
#ifndef GTK_SOURCE_VERSION_MAX_ALLOWED
#if GTK_SOURCE_VERSION_MIN_REQUIRED > GTK_SOURCE_VERSION_PREV_STABLE
#define GTK_SOURCE_VERSION_MAX_ALLOWED  GTK_SOURCE_VERSION_MIN_REQUIRED
#else
#define GTK_SOURCE_VERSION_MAX_ALLOWED GTK_SOURCE_VERSION_CUR_STABLE
#endif
#endif

/* sanity checks */
#if GTK_SOURCE_VERSION_MAX_ALLOWED < GTK_SOURCE_VERSION_MIN_REQUIRED
#error "GTK_SOURCE_VERSION_MAX_ALLOWED must be >= GTK_SOURCE_VERSION_MIN_REQUIRED"
#endif
#if GTK_SOURCE_VERSION_MIN_REQUIRED < GTK_SOURCE_VERSION_5_0
#error "GTK_SOURCE_VERSION_MIN_REQUIRED must be >= GTK_SOURCE_VERSION_5_0"
#endif

#ifndef __GTK_DOC_IGNORE__
#define GTK_SOURCE_AVAILABLE_IN_ALL _GTK_SOURCE_EXTERN
#endif

/* Every new stable minor release should add a set of macros here */

#ifndef __GTK_DOC_IGNORE__
#if GTK_SOURCE_VERSION_MAX_ALLOWED < GTK_SOURCE_VERSION_5_2
#define GTK_SOURCE_AVAILABLE_IN_5_2 G_UNAVAILABLE(5, 2) _GTK_SOURCE_EXTERN
#else
#define GTK_SOURCE_AVAILABLE_IN_5_2 _GTK_SOURCE_EXTERN
#endif
#endif /* __GTK_DOC_IGNORE__ */

#ifndef __GTK_DOC_IGNORE__
#if GTK_SOURCE_VERSION_MAX_ALLOWED < GTK_SOURCE_VERSION_5_4
#define GTK_SOURCE_AVAILABLE_IN_5_4 G_UNAVAILABLE(5, 4) _GTK_SOURCE_EXTERN
#else
#define GTK_SOURCE_AVAILABLE_IN_5_4 _GTK_SOURCE_EXTERN
#endif
#endif /* __GTK_DOC_IGNORE__ */

GTK_SOURCE_AVAILABLE_IN_ALL
guint    gtk_source_get_major_version (void);
GTK_SOURCE_AVAILABLE_IN_ALL
guint    gtk_source_get_minor_version (void);
GTK_SOURCE_AVAILABLE_IN_ALL
guint    gtk_source_get_micro_version (void);
GTK_SOURCE_AVAILABLE_IN_ALL
gboolean gtk_source_check_version     (guint major,
                                       guint minor,
                                       guint micro);

G_END_DECLS
