/*
 * Copyright (C) 2021 Apple Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1.  Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 * 2.  Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE INC. AND ITS CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL APPLE INC. OR ITS CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#include "WebUserContentControllerProxy.h"

#include "ArgumentCoders.h"
#include "DataReference.h"
#include "Decoder.h"
#include "FrameInfoData.h"
#include "HandleMessage.h"
#include "WebPageProxyIdentifier.h"
#include "WebUserContentControllerProxyMessages.h"
#include <wtf/text/WTFString.h>

namespace Messages {

namespace WebUserContentControllerProxy {

void DidPostMessage::callReply(IPC::Decoder& decoder, CompletionHandler<void(IPC::DataReference&&, String&&)>&& completionHandler)
{
    std::optional<IPC::DataReference> resultValue;
    decoder >> resultValue;
    if (!resultValue) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    std::optional<String> errorMessage;
    decoder >> errorMessage;
    if (!errorMessage) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*resultValue), WTFMove(*errorMessage));
}

void DidPostMessage::cancelReply(CompletionHandler<void(IPC::DataReference&&, String&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<IPC::DataReference>::create(), IPC::AsyncReplyError<String>::create());
}

void DidPostMessage::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const IPC::DataReference& resultValue, const String& errorMessage)
{
    encoder.get() << resultValue;
    encoder.get() << errorMessage;
    connection.sendSyncReply(WTFMove(encoder));
}

} // namespace WebUserContentControllerProxy

} // namespace Messages

namespace WebKit {

void WebUserContentControllerProxy::didReceiveMessage(IPC::Connection& connection, IPC::Decoder& decoder)
{
    Ref protectedThis { *this };
    if (decoder.messageName() == Messages::WebUserContentControllerProxy::DidPostMessage::name())
        return IPC::handleMessageAsync<Messages::WebUserContentControllerProxy::DidPostMessage>(connection, decoder, this, &WebUserContentControllerProxy::didPostMessage);
    UNUSED_PARAM(connection);
    UNUSED_PARAM(decoder);
#if ENABLE(IPC_TESTING_API)
    if (connection.ignoreInvalidMessageForTesting())
        return;
#endif // ENABLE(IPC_TESTING_API)
    ASSERT_NOT_REACHED_WITH_MESSAGE("Unhandled message %s to %" PRIu64, IPC::description(decoder.messageName()), decoder.destinationID());
}

} // namespace WebKit
