"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createSavedSearchReport = createSavedSearchReport;

var _dataReportHelpers = require("./dataReportHelpers");

var _helpers = require("./helpers");

/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

/**
 * Specify how long scroll context should be maintained for scrolled search
 */
const scrollTimeout = '1m';

async function createSavedSearchReport(report, client, dateFormat, csvSeparator, isScheduledTask = true, logger) {
  const params = report.report_definition.report_params;
  const reportFormat = params.core_params.report_format;
  const reportName = params.report_name;
  await populateMetaData(client, report, isScheduledTask, logger);
  const data = await generateReportData(client, params.core_params, dateFormat, csvSeparator, isScheduledTask, logger);
  const curTime = new Date();
  const timeCreated = curTime.valueOf();
  const fileName = (0, _helpers.getFileName)(reportName, curTime) + '.' + reportFormat;
  return {
    timeCreated,
    dataUrl: data,
    fileName
  };
}
/**
 * Populate parameters and saved search info related to meta data object.
 * @param client  OpenSearch client
 * @param report  Report input
 */


async function populateMetaData(client, report, isScheduledTask, logger) {
  _dataReportHelpers.metaData.saved_search_id = report.report_definition.report_params.core_params.saved_search_id;
  _dataReportHelpers.metaData.report_format = report.report_definition.report_params.core_params.report_format;
  _dataReportHelpers.metaData.start = report.time_from;
  _dataReportHelpers.metaData.end = report.time_to; // Get saved search info

  let resIndexPattern = {};
  const ssParams = {
    index: '.kibana',
    id: 'search:' + _dataReportHelpers.metaData.saved_search_id
  };
  const ssInfos = await (0, _helpers.callCluster)(client, 'get', ssParams, isScheduledTask);
  _dataReportHelpers.metaData.sorting = ssInfos._source.search.sort;
  _dataReportHelpers.metaData.type = ssInfos._source.type;
  _dataReportHelpers.metaData.searchSourceJSON = ssInfos._source.search.kibanaSavedObjectMeta.searchSourceJSON; // Get the list of selected columns in the saved search.Otherwise select all the fields under the _source

  await (0, _dataReportHelpers.getSelectedFields)(ssInfos._source.search.columns); // Get index name

  for (const item of ssInfos._source.references) {
    if (item.name === JSON.parse(_dataReportHelpers.metaData.searchSourceJSON).indexRefName) {
      // Get index-pattern information
      const indexPattern = await (0, _helpers.callCluster)(client, 'get', {
        index: '.kibana',
        id: 'index-pattern:' + item.id
      }, isScheduledTask);
      resIndexPattern = indexPattern._source['index-pattern'];
      _dataReportHelpers.metaData.paternName = resIndexPattern.title;
      _dataReportHelpers.metaData.timeFieldName = resIndexPattern.timeFieldName, _dataReportHelpers.metaData.fields = resIndexPattern.fields; // Get all fields
      // Getting fields of type Date

      const dateFields = [];

      for (const item of JSON.parse(_dataReportHelpers.metaData.fields)) {
        if (item.type === 'date') {
          dateFields.push(item.name);
        }
      }

      _dataReportHelpers.metaData.dateFields = dateFields;
    }
  }
}
/**
 * Generate CSV data by query and convert OpenSearch data set.
 * @param client  OpenSearch client
 * @param limit   limit size of result data set
 */


async function generateReportData(client, params, dateFormat, csvSeparator, isScheduledTask, logger) {
  let opensearchData = {};
  const arrayHits = [];
  const report = {
    _source: _dataReportHelpers.metaData
  };
  const indexPattern = report._source.paternName;
  const maxResultSize = await getMaxResultSize();
  const opensearchCount = await getOpenSearchDataSize();
  const total = Math.min(opensearchCount.count, params.limit);

  if (total === 0) {
    return '';
  }

  const reqBody = (0, _dataReportHelpers.buildRequestBody)(report, 0);
  logger.info(`[Reporting csv module] DSL request body: ${JSON.stringify(reqBody)}`);

  if (total > maxResultSize) {
    await getOpenSearchDataByScroll();
  } else {
    await getOpenSearchDataBySearch();
  }

  return convertOpenSearchDataToCsv(); // Fetch OpenSearch query max size windows to decide search or scroll

  async function getMaxResultSize() {
    const settings = await (0, _helpers.callCluster)(client, 'indices.getSettings', {
      index: indexPattern,
      includeDefaults: true
    }, isScheduledTask);
    let maxResultSize = Number.MAX_SAFE_INTEGER;

    for (let indexName in settings) {
      // The location of max result window differs if default overridden.
      maxResultSize = Math.min(maxResultSize, settings[indexName].settings.index.max_result_window || settings[indexName].defaults.index.max_result_window);
    }

    return maxResultSize;
  } // Build the OpenSearch Count query to count the size of result


  async function getOpenSearchDataSize() {
    const countReq = (0, _dataReportHelpers.buildRequestBody)(report, 1);
    return await (0, _helpers.callCluster)(client, 'count', {
      index: indexPattern,
      body: countReq
    }, isScheduledTask);
  }

  async function getOpenSearchDataByScroll() {
    const searchParams = {
      index: report._source.paternName,
      scroll: scrollTimeout,
      body: reqBody,
      size: maxResultSize
    }; // Open scroll context by fetching first batch

    opensearchData = await (0, _helpers.callCluster)(client, 'search', searchParams, isScheduledTask);
    arrayHits.push(opensearchData.hits); // Start scrolling till the end

    const nbScroll = Math.floor(total / maxResultSize);

    for (let i = 0; i < nbScroll; i++) {
      const resScroll = await (0, _helpers.callCluster)(client, 'scroll', {
        scrollId: opensearchData._scroll_id,
        scroll: scrollTimeout
      }, isScheduledTask);

      if (Object.keys(resScroll.hits.hits).length > 0) {
        arrayHits.push(resScroll.hits);
      }
    } // Clear scroll context


    await (0, _helpers.callCluster)(client, 'clearScroll', {
      scrollId: opensearchData._scroll_id
    }, isScheduledTask);
  }

  async function getOpenSearchDataBySearch() {
    const searchParams = {
      index: report._source.paternName,
      body: reqBody,
      size: total
    };
    opensearchData = await (0, _helpers.callCluster)(client, 'search', searchParams, isScheduledTask);
    arrayHits.push(opensearchData.hits);
  } // Parse OpenSearch data and convert to CSV


  async function convertOpenSearchDataToCsv() {
    const dataset = [];
    dataset.push((0, _dataReportHelpers.getOpenSearchData)(arrayHits, report, params, dateFormat));
    return await (0, _dataReportHelpers.convertToCSV)(dataset, csvSeparator);
  }
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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