"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.PRIVATE_TENANTS = exports.GLOBAL_TENANTS = void 0;
exports.isMultitenantPath = isMultitenantPath;
exports.isValidTenant = isValidTenant;
exports.resolveTenant = resolveTenant;

var _lodash = require("lodash");

/*
 *   Copyright OpenSearch Contributors
 *
 *   Licensed under the Apache License, Version 2.0 (the "License").
 *   You may not use this file except in compliance with the License.
 *   A copy of the License is located at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   or in the "license" file accompanying this file. This file is distributed
 *   on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 *   express or implied. See the License for the specific language governing
 *   permissions and limitations under the License.
 */
const PRIVATE_TENANT_SYMBOL = '__user__';
const GLOBAL_TENANT_SYMBOL = '';
const PRIVATE_TENANTS = [PRIVATE_TENANT_SYMBOL, 'private'];
exports.PRIVATE_TENANTS = PRIVATE_TENANTS;
const GLOBAL_TENANTS = ['global', GLOBAL_TENANT_SYMBOL];
/**
 * Resovles the tenant the user is using.
 *
 * @param request OpenSearchDashboards request.
 * @param config security plugin config.
 * @param cookie cookie extracted from the request. The cookie should have been parsed by AuthenticationHandler.
 * pass it as parameter instead of extracting again.
 * @param authInfo authentication info, the Elasticsearch authinfo API response.
 *
 * @returns user preferred tenant of the request.
 */

exports.GLOBAL_TENANTS = GLOBAL_TENANTS;

function resolveTenant(request, username, roles, availabeTenants, config, cookie) {
  var _request$url, _request$url$searchPa, _request$url2, _request$url2$searchP, _config$multitenancy, _config$multitenancy2, _config$multitenancy3;

  const DEFAULT_READONLY_ROLES = ['kibana_read_only'];
  let selectedTenant;
  const securityTenant_ = request === null || request === void 0 ? void 0 : (_request$url = request.url) === null || _request$url === void 0 ? void 0 : (_request$url$searchPa = _request$url.searchParams) === null || _request$url$searchPa === void 0 ? void 0 : _request$url$searchPa.get('securityTenant_');
  const securitytenant = request === null || request === void 0 ? void 0 : (_request$url2 = request.url) === null || _request$url2 === void 0 ? void 0 : (_request$url2$searchP = _request$url2.searchParams) === null || _request$url2$searchP === void 0 ? void 0 : _request$url2$searchP.get('securitytenant');

  if (securityTenant_) {
    selectedTenant = securityTenant_;
  } else if (securitytenant) {
    selectedTenant = securitytenant;
  } else if (request.headers.securitytenant || request.headers.securityTenant_) {
    selectedTenant = request.headers.securitytenant ? request.headers.securitytenant : request.headers.securityTenant_;
  } else if (isValidTenant(cookie.tenant)) {
    selectedTenant = cookie.tenant;
  } else {
    selectedTenant = undefined;
  }

  const isReadonly = roles === null || roles === void 0 ? void 0 : roles.some(role => {
    var _config$readonly_mode;

    return ((_config$readonly_mode = config.readonly_mode) === null || _config$readonly_mode === void 0 ? void 0 : _config$readonly_mode.roles.includes(role)) || DEFAULT_READONLY_ROLES.includes(role);
  });
  const preferredTenants = (_config$multitenancy = config.multitenancy) === null || _config$multitenancy === void 0 ? void 0 : _config$multitenancy.tenants.preferred;
  const globalTenantEnabled = (_config$multitenancy2 = config.multitenancy) === null || _config$multitenancy2 === void 0 ? void 0 : _config$multitenancy2.tenants.enable_global;
  const privateTenantEnabled = ((_config$multitenancy3 = config.multitenancy) === null || _config$multitenancy3 === void 0 ? void 0 : _config$multitenancy3.tenants.enable_private) && !isReadonly;
  return resolve(username, selectedTenant, preferredTenants, availabeTenants, globalTenantEnabled, privateTenantEnabled);
}
/**
 * Determines whether the request requires tenant info.
 * @param request opensearch-dashboards request.
 *
 * @returns true if the request requires tenant info, otherwise false.
 */


function isMultitenantPath(request) {
  var _request$url$pathname, _request$url$pathname2, _request$url$pathname3, _request$url$pathname4, _request$url$pathname5;

  return ((_request$url$pathname = request.url.pathname) === null || _request$url$pathname === void 0 ? void 0 : _request$url$pathname.startsWith('/opensearch')) || ((_request$url$pathname2 = request.url.pathname) === null || _request$url$pathname2 === void 0 ? void 0 : _request$url$pathname2.startsWith('/api')) || ((_request$url$pathname3 = request.url.pathname) === null || _request$url$pathname3 === void 0 ? void 0 : _request$url$pathname3.startsWith('/app')) || ( // short url path
  (_request$url$pathname4 = request.url.pathname) === null || _request$url$pathname4 === void 0 ? void 0 : _request$url$pathname4.startsWith('/goto')) || // bootstrap.js depends on tenant info to fetch opensearch-dashboards configs in tenant index
  (((_request$url$pathname5 = request.url.pathname) === null || _request$url$pathname5 === void 0 ? void 0 : _request$url$pathname5.indexOf('bootstrap.js')) || -1) > -1 || request.url.pathname === '/';
}

function resolve(username, requestedTenant, preferredTenants, availableTenants, // is an object like { tenant_name_1: true, tenant_name_2: false, ... }
globalTenantEnabled, privateTenantEnabled) {
  const availableTenantsClone = (0, _lodash.cloneDeep)(availableTenants);
  delete availableTenantsClone[username];

  if (!globalTenantEnabled && !privateTenantEnabled && (0, _lodash.isEmpty)(availableTenantsClone)) {
    return undefined;
  }

  if (isValidTenant(requestedTenant)) {
    requestedTenant = requestedTenant;

    if (requestedTenant in availableTenants) {
      return requestedTenant;
    }

    if (privateTenantEnabled && username in availableTenants && PRIVATE_TENANTS.indexOf(requestedTenant) > -1) {
      return PRIVATE_TENANT_SYMBOL;
    }

    if (globalTenantEnabled && GLOBAL_TENANTS.indexOf(requestedTenant) > -1) {
      return GLOBAL_TENANT_SYMBOL;
    }
  }

  if (preferredTenants && !(0, _lodash.isEmpty)(preferredTenants)) {
    for (const element of preferredTenants) {
      const tenant = element.toLowerCase();

      if (globalTenantEnabled && GLOBAL_TENANTS.indexOf(tenant) > -1) {
        return GLOBAL_TENANT_SYMBOL;
      }

      if (privateTenantEnabled && PRIVATE_TENANTS.indexOf(tenant) > -1 && username in availableTenants) {
        return PRIVATE_TENANT_SYMBOL;
      }

      if (tenant in availableTenants) {
        return tenant;
      }
    }
  }

  if (globalTenantEnabled) {
    return GLOBAL_TENANT_SYMBOL;
  }

  if (privateTenantEnabled) {
    return PRIVATE_TENANT_SYMBOL;
  } // fall back to the first tenant in the available tenants


  return (0, _lodash.findKey)(availableTenantsClone, () => true);
}
/**
 * Return true if tenant parameter is a valid tenent.
 *
 * Note: empty string '' is valid, which means global tenant.
 *
 * @param tenant
 */


function isValidTenant(tenant) {
  return tenant !== undefined && tenant !== null;
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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