/*
 * Copyright (C) 2021 Apple Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1.  Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 * 2.  Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE INC. AND ITS CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL APPLE INC. OR ITS CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#if ENABLE(SERVICE_WORKER)
#include "WebSWClientConnection.h"

#include "ArgumentCoders.h"
#include "Decoder.h"
#include "HandleMessage.h"
#include "SharedMemory.h"
#include "WebCoreArgumentCoders.h"
#include "WebSWClientConnectionMessages.h"
#include <WebCore/ExceptionData.h>
#include <WebCore/FetchOptions.h>
#include <WebCore/MessageWithMessagePorts.h>
#include <WebCore/ProcessQualified.h>
#include <WebCore/ScriptExecutionContextIdentifier.h>
#include <WebCore/ServiceWorkerData.h>
#include <WebCore/ServiceWorkerIdentifier.h>
#include <WebCore/ServiceWorkerRegistrationData.h>
#include <WebCore/ServiceWorkerRegistrationKey.h>
#include <WebCore/ServiceWorkerTypes.h>
#include <WebCore/ServiceWorkerUpdateViaCache.h>
#include <optional>
#include <wtf/HashSet.h>
#include <wtf/ObjectIdentifier.h>
#include <wtf/WallTime.h>
#include <wtf/text/WTFString.h>

namespace Messages {

namespace WebSWClientConnection {

void SetDocumentIsControlled::callReply(IPC::Decoder& decoder, CompletionHandler<void(bool&&)>&& completionHandler)
{
    std::optional<bool> isSuccess;
    decoder >> isSuccess;
    if (!isSuccess) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*isSuccess));
}

void SetDocumentIsControlled::cancelReply(CompletionHandler<void(bool&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<bool>::create());
}

void SetDocumentIsControlled::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, bool isSuccess)
{
    encoder.get() << isSuccess;
    connection.sendSyncReply(WTFMove(encoder));
}

} // namespace WebSWClientConnection

} // namespace Messages

namespace WebKit {

void WebSWClientConnection::didReceiveMessage(IPC::Connection& connection, IPC::Decoder& decoder)
{
    Ref protectedThis { *this };
    if (decoder.messageName() == Messages::WebSWClientConnection::JobRejectedInServer::name())
        return IPC::handleMessage<Messages::WebSWClientConnection::JobRejectedInServer>(connection, decoder, this, &WebSWClientConnection::jobRejectedInServer);
    if (decoder.messageName() == Messages::WebSWClientConnection::RegistrationJobResolvedInServer::name())
        return IPC::handleMessage<Messages::WebSWClientConnection::RegistrationJobResolvedInServer>(connection, decoder, this, &WebSWClientConnection::registrationJobResolvedInServer);
    if (decoder.messageName() == Messages::WebSWClientConnection::StartScriptFetchForServer::name())
        return IPC::handleMessage<Messages::WebSWClientConnection::StartScriptFetchForServer>(connection, decoder, this, &WebSWClientConnection::startScriptFetchForServer);
    if (decoder.messageName() == Messages::WebSWClientConnection::UpdateRegistrationState::name())
        return IPC::handleMessage<Messages::WebSWClientConnection::UpdateRegistrationState>(connection, decoder, this, &WebSWClientConnection::updateRegistrationState);
    if (decoder.messageName() == Messages::WebSWClientConnection::UpdateWorkerState::name())
        return IPC::handleMessage<Messages::WebSWClientConnection::UpdateWorkerState>(connection, decoder, this, &WebSWClientConnection::updateWorkerState);
    if (decoder.messageName() == Messages::WebSWClientConnection::FireUpdateFoundEvent::name())
        return IPC::handleMessage<Messages::WebSWClientConnection::FireUpdateFoundEvent>(connection, decoder, this, &WebSWClientConnection::fireUpdateFoundEvent);
    if (decoder.messageName() == Messages::WebSWClientConnection::SetRegistrationLastUpdateTime::name())
        return IPC::handleMessage<Messages::WebSWClientConnection::SetRegistrationLastUpdateTime>(connection, decoder, this, &WebSWClientConnection::setRegistrationLastUpdateTime);
    if (decoder.messageName() == Messages::WebSWClientConnection::SetRegistrationUpdateViaCache::name())
        return IPC::handleMessage<Messages::WebSWClientConnection::SetRegistrationUpdateViaCache>(connection, decoder, this, &WebSWClientConnection::setRegistrationUpdateViaCache);
    if (decoder.messageName() == Messages::WebSWClientConnection::NotifyClientsOfControllerChange::name())
        return IPC::handleMessage<Messages::WebSWClientConnection::NotifyClientsOfControllerChange>(connection, decoder, this, &WebSWClientConnection::notifyClientsOfControllerChange);
    if (decoder.messageName() == Messages::WebSWClientConnection::SetSWOriginTableIsImported::name())
        return IPC::handleMessage<Messages::WebSWClientConnection::SetSWOriginTableIsImported>(connection, decoder, this, &WebSWClientConnection::setSWOriginTableIsImported);
    if (decoder.messageName() == Messages::WebSWClientConnection::SetSWOriginTableSharedMemory::name())
        return IPC::handleMessage<Messages::WebSWClientConnection::SetSWOriginTableSharedMemory>(connection, decoder, this, &WebSWClientConnection::setSWOriginTableSharedMemory);
    if (decoder.messageName() == Messages::WebSWClientConnection::PostMessageToServiceWorkerClient::name())
        return IPC::handleMessage<Messages::WebSWClientConnection::PostMessageToServiceWorkerClient>(connection, decoder, this, &WebSWClientConnection::postMessageToServiceWorkerClient);
    if (decoder.messageName() == Messages::WebSWClientConnection::SetDocumentIsControlled::name())
        return IPC::handleMessageAsync<Messages::WebSWClientConnection::SetDocumentIsControlled>(connection, decoder, this, &WebSWClientConnection::setDocumentIsControlled);
    UNUSED_PARAM(connection);
    UNUSED_PARAM(decoder);
#if ENABLE(IPC_TESTING_API)
    if (connection.ignoreInvalidMessageForTesting())
        return;
#endif // ENABLE(IPC_TESTING_API)
    ASSERT_NOT_REACHED_WITH_MESSAGE("Unhandled message %s to %" PRIu64, IPC::description(decoder.messageName()), decoder.destinationID());
}

} // namespace WebKit

#endif // ENABLE(SERVICE_WORKER)
