"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createVisualReport = void 0;

var _puppeteerCore = _interopRequireDefault(require("puppeteer-core"));

var _dompurify = _interopRequireDefault(require("dompurify"));

var _jsdom = require("jsdom");

var _constants = require("../constants");

var _helpers = require("../helpers");

var _fs = _interopRequireDefault(require("fs"));

var _lodash = _interopRequireDefault(require("lodash"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */
const createVisualReport = async (reportParams, queryUrl, logger, extraHeaders, timezone) => {
  const {
    core_params,
    report_name: reportName,
    report_source: reportSource
  } = reportParams;
  const coreParams = core_params;
  const {
    header,
    footer,
    window_height: windowHeight,
    window_width: windowWidth,
    report_format: reportFormat
  } = coreParams;
  const window = new _jsdom.JSDOM('').window;
  const DOMPurify = (0, _dompurify.default)(window);
  let keywordFilteredHeader = header ? _constants.converter.makeHtml(header) : _constants.DEFAULT_REPORT_HEADER;
  let keywordFilteredFooter = footer ? _constants.converter.makeHtml(footer) : '';
  keywordFilteredHeader = DOMPurify.sanitize(keywordFilteredHeader);
  keywordFilteredFooter = DOMPurify.sanitize(keywordFilteredFooter); // filter blocked keywords in header and footer

  if (keywordFilteredHeader !== '') {
    keywordFilteredHeader = (0, _constants.replaceBlockedKeywords)(keywordFilteredHeader);
  }

  if (keywordFilteredFooter !== '') {
    keywordFilteredFooter = (0, _constants.replaceBlockedKeywords)(keywordFilteredFooter);
  } // set up puppeteer


  const browser = await _puppeteerCore.default.launch({
    headless: true,

    /**
     * TODO: temp fix to disable sandbox when launching chromium on Linux instance
     * https://github.com/puppeteer/puppeteer/blob/main/docs/troubleshooting.md#setting-up-chrome-linux-sandbox
     */
    args: ['--no-sandbox', '--disable-setuid-sandbox', '--disable-gpu', '--no-zygote', '--single-process', '--font-render-hinting=none'],
    executablePath: _constants.CHROMIUM_PATH,
    ignoreHTTPSErrors: true,
    env: {
      TZ: timezone || 'UTC'
    }
  });
  const page = await browser.newPage();
  page.setDefaultNavigationTimeout(0);
  page.setDefaultTimeout(100000); // use 100s timeout instead of default 30s
  // Set extra headers that are needed

  if (!_lodash.default.isEmpty(extraHeaders)) {
    await page.setExtraHTTPHeaders(extraHeaders);
  }

  logger.info(`original queryUrl ${queryUrl}`);
  await page.goto(queryUrl, {
    waitUntil: 'networkidle0'
  }); // should add to local storage after page.goto, then access the page again - browser must have an url to register local storage item on it

  await page.evaluate(
  /* istanbul ignore next */
  key => {
    localStorage.setItem(key, 'false');
  }, _constants.SECURITY_CONSTANTS.TENANT_LOCAL_STORAGE_KEY);
  await page.goto(queryUrl, {
    waitUntil: 'networkidle0'
  });
  logger.info(`page url ${page.url()}`);
  await page.setViewport({
    width: windowWidth,
    height: windowHeight
  });
  let buffer; // remove unwanted elements

  await page.evaluate(
  /* istanbul ignore next */
  (reportSource, REPORT_TYPE) => {
    // remove buttons
    document.querySelectorAll("[class^='euiButton']").forEach(e => e.remove()); // remove top navBar

    document.querySelectorAll("[class^='euiHeader']").forEach(e => e.remove()); // remove visualization editor

    if (reportSource === REPORT_TYPE.visualization) {
      var _document$querySelect, _document$querySelect2;

      (_document$querySelect = document.querySelector('[data-test-subj="splitPanelResizer"]')) === null || _document$querySelect === void 0 ? void 0 : _document$querySelect.remove();
      (_document$querySelect2 = document.querySelector('.visEditor__collapsibleSidebar')) === null || _document$querySelect2 === void 0 ? void 0 : _document$querySelect2.remove();
    }

    document.body.style.paddingTop = '0px';
  }, reportSource, _constants.REPORT_TYPE); // force wait for any resize to load after the above DOM modification

  await page.waitFor(1000); // crop content

  switch (reportSource) {
    case _constants.REPORT_TYPE.dashboard:
      await page.waitForSelector(_constants.SELECTOR.dashboard, {
        visible: true
      });
      break;

    case _constants.REPORT_TYPE.visualization:
      await page.waitForSelector(_constants.SELECTOR.visualization, {
        visible: true
      });
      break;

    case _constants.REPORT_TYPE.notebook:
      await page.waitForSelector(_constants.SELECTOR.notebook, {
        visible: true
      });
      break;

    default:
      throw Error(`report source can only be one of [Dashboard, Visualization]`);
  } // wait for dynamic page content to render


  await waitForDynamicContent(page);
  await addReportStyle(page);
  await addReportHeader(page, keywordFilteredHeader);
  await addReportFooter(page, keywordFilteredFooter); // create pdf or png accordingly

  if (reportFormat === _constants.FORMAT.pdf) {
    const scrollHeight = await page.evaluate(
    /* istanbul ignore next */
    () => document.documentElement.scrollHeight);
    buffer = await page.pdf({
      margin: undefined,
      width: windowWidth,
      height: scrollHeight + 'px',
      printBackground: true,
      pageRanges: '1'
    });
  } else if (reportFormat === _constants.FORMAT.png) {
    buffer = await page.screenshot({
      fullPage: true
    });
  }

  const curTime = new Date();
  const timeCreated = curTime.valueOf();
  const fileName = `${(0, _helpers.getFileName)(reportName, curTime)}.${reportFormat}`;
  await browser.close();
  return {
    timeCreated,
    dataUrl: buffer.toString('base64'),
    fileName
  };
};

exports.createVisualReport = createVisualReport;

const addReportStyle = async page => {
  const css = _fs.default.readFileSync(`${__dirname}/style.css`).toString();

  await page.evaluate(
  /* istanbul ignore next */
  style => {
    const styleElement = document.createElement('style');
    styleElement.innerHTML = style;
    document.getElementsByTagName('head')[0].appendChild(styleElement);
  }, css);
};

const addReportHeader = async (page, header) => {
  const headerHtml = _fs.default.readFileSync(`${__dirname}/header_template.html`).toString().replace('<!--CONTENT-->', header);

  await page.evaluate(
  /* istanbul ignore next */
  headerHtml => {
    var _content$parentNode;

    const content = document.body.firstChild;
    const headerContainer = document.createElement('div');
    headerContainer.className = 'reportWrapper';
    headerContainer.innerHTML = headerHtml;
    content === null || content === void 0 ? void 0 : (_content$parentNode = content.parentNode) === null || _content$parentNode === void 0 ? void 0 : _content$parentNode.insertBefore(headerContainer, content);
  }, headerHtml);
};

const addReportFooter = async (page, footer) => {
  const headerHtml = _fs.default.readFileSync(`${__dirname}/footer_template.html`).toString().replace('<!--CONTENT-->', footer);

  await page.evaluate(
  /* istanbul ignore next */
  headerHtml => {
    var _content$parentNode2;

    const content = document.body.firstChild;
    const headerContainer = document.createElement('div');
    headerContainer.className = 'reportWrapper';
    headerContainer.innerHTML = headerHtml;
    content === null || content === void 0 ? void 0 : (_content$parentNode2 = content.parentNode) === null || _content$parentNode2 === void 0 ? void 0 : _content$parentNode2.insertBefore(headerContainer, null);
  }, headerHtml);
}; // add waitForDynamicContent function


const waitForDynamicContent = async (page, timeout = 30000, interval = 1000, checks = 5) => {
  const maxChecks = timeout / interval;
  let passedChecks = 0;
  let previousLength = 0;
  let i = 0;

  while (i++ <= maxChecks) {
    let pageContent = await page.content();
    let currentLength = pageContent.length;
    previousLength === 0 || previousLength != currentLength ? passedChecks = 0 : passedChecks++;

    if (passedChecks >= checks) {
      break;
    }

    previousLength = currentLength;
    await page.waitFor(interval);
  }
};
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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