"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.OpenIdAuthRoutes = void 0;

var _configSchema = require("@osd/config-schema");

var _cryptiles = require("@hapi/cryptiles");

var _querystring = require("querystring");

var _helper = require("./helper");

var _next_url = require("../../../utils/next_url");

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

class OpenIdAuthRoutes {
  constructor(router, config, sessionStorageFactory, openIdAuthConfig, securityClient, core, wreckClient) {
    this.router = router;
    this.config = config;
    this.sessionStorageFactory = sessionStorageFactory;
    this.openIdAuthConfig = openIdAuthConfig;
    this.securityClient = securityClient;
    this.core = core;
    this.wreckClient = wreckClient;
  }

  redirectToLogin(request, response) {
    this.sessionStorageFactory.asScoped(request).clear();
    return response.redirected({
      headers: {
        location: `${this.core.http.basePath.serverBasePath}/auth/openid/login`
      }
    });
  }

  setupRoutes() {
    this.router.get({
      path: `/auth/openid/login`,
      validate: {
        query: _configSchema.schema.object({
          code: _configSchema.schema.maybe(_configSchema.schema.string()),
          nextUrl: _configSchema.schema.maybe(_configSchema.schema.string({
            validate: _next_url.validateNextUrl
          })),
          state: _configSchema.schema.maybe(_configSchema.schema.string()),
          refresh: _configSchema.schema.maybe(_configSchema.schema.string())
        }, {
          unknowns: 'allow'
        })
      },
      options: {
        authRequired: false
      }
    }, async (context, request, response) => {
      var _this$config$openid2, _this$config$openid3;

      // implementation refers to https://github.com/hapijs/bell/blob/master/lib/oauth.js
      // Sign-in initialization
      if (!request.query.code) {
        var _this$config$openid;

        const nonce = (0, _cryptiles.randomString)(OpenIdAuthRoutes.NONCE_LENGTH);
        const query = {
          client_id: (_this$config$openid = this.config.openid) === null || _this$config$openid === void 0 ? void 0 : _this$config$openid.client_id,
          response_type: 'code',
          redirect_uri: `${(0, _helper.getBaseRedirectUrl)(this.config, this.core, request)}/auth/openid/login`,
          state: nonce,
          scope: this.openIdAuthConfig.scope
        };
        const queryString = (0, _querystring.stringify)(query);
        const location = `${this.openIdAuthConfig.authorizationEndpoint}?${queryString}`;
        const cookie = {
          oidc: {
            state: nonce,
            nextUrl: request.query.nextUrl || '/'
          }
        };
        this.sessionStorageFactory.asScoped(request).set(cookie);
        return response.redirected({
          headers: {
            location
          }
        });
      } // Authentication callback
      // validate state first


      let cookie;

      try {
        var _cookie$oidc;

        cookie = await this.sessionStorageFactory.asScoped(request).get();

        if (!cookie || !((_cookie$oidc = cookie.oidc) !== null && _cookie$oidc !== void 0 && _cookie$oidc.state) || cookie.oidc.state !== request.query.state) {
          return this.redirectToLogin(request, response);
        }
      } catch (error) {
        return this.redirectToLogin(request, response);
      }

      const nextUrl = cookie.oidc.nextUrl;
      const clientId = (_this$config$openid2 = this.config.openid) === null || _this$config$openid2 === void 0 ? void 0 : _this$config$openid2.client_id;
      const clientSecret = (_this$config$openid3 = this.config.openid) === null || _this$config$openid3 === void 0 ? void 0 : _this$config$openid3.client_secret;
      const query = {
        grant_type: 'authorization_code',
        code: request.query.code,
        redirect_uri: `${(0, _helper.getBaseRedirectUrl)(this.config, this.core, request)}/auth/openid/login`,
        client_id: clientId,
        client_secret: clientSecret
      };

      try {
        var _this$config$openid4;

        const tokenResponse = await (0, _helper.callTokenEndpoint)(this.openIdAuthConfig.tokenEndpoint, query, this.wreckClient);
        const user = await this.securityClient.authenticateWithHeader(request, this.openIdAuthConfig.authHeaderName, `Bearer ${tokenResponse.idToken}`); // set to cookie

        const sessionStorage = {
          username: user.username,
          credentials: {
            authHeaderValue: `Bearer ${tokenResponse.idToken}`,
            expires_at: Date.now() + tokenResponse.expiresIn * 1000 // expiresIn is in second

          },
          authType: 'openid',
          expiryTime: Date.now() + this.config.session.ttl
        };

        if ((_this$config$openid4 = this.config.openid) !== null && _this$config$openid4 !== void 0 && _this$config$openid4.refresh_tokens && tokenResponse.refreshToken) {
          Object.assign(sessionStorage.credentials, {
            refresh_token: tokenResponse.refreshToken
          });
        }

        this.sessionStorageFactory.asScoped(request).set(sessionStorage);
        return response.redirected({
          headers: {
            location: nextUrl
          }
        });
      } catch (error) {
        context.security_plugin.logger.error(`OpenId authentication failed: ${error}`);

        if (error.toString().toLowerCase().includes('authentication exception')) {
          return response.unauthorized();
        } else {
          return this.redirectToLogin(request, response);
        }
      }
    });
    this.router.get({
      path: `/auth/logout`,
      validate: false
    }, async (context, request, response) => {
      var _this$config$openid5;

      const cookie = await this.sessionStorageFactory.asScoped(request).get();
      this.sessionStorageFactory.asScoped(request).clear(); // authHeaderValue is the bearer header, e.g. "Bearer <auth_token>"

      const token = cookie === null || cookie === void 0 ? void 0 : cookie.credentials.authHeaderValue.split(' ')[1]; // get auth token

      const logoutQueryParams = {
        post_logout_redirect_uri: (0, _helper.getBaseRedirectUrl)(this.config, this.core, request),
        id_token_hint: token
      };
      const endSessionUrl = (0, _helper.composeLogoutUrl)((_this$config$openid5 = this.config.openid) === null || _this$config$openid5 === void 0 ? void 0 : _this$config$openid5.logout_url, this.openIdAuthConfig.endSessionEndpoint, logoutQueryParams);
      return response.redirected({
        headers: {
          location: endSessionUrl
        }
      });
    });
  }

}

exports.OpenIdAuthRoutes = OpenIdAuthRoutes;

_defineProperty(OpenIdAuthRoutes, "NONCE_LENGTH", 22);
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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