"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SamlAuthRoutes = void 0;

var _configSchema = require("@osd/config-schema");

var _common = require("../../../../common");

var _next_url = require("../../../utils/next_url");

var _index = require("../../../../common/index");

/*
 *   Copyright OpenSearch Contributors
 *
 *   Licensed under the Apache License, Version 2.0 (the "License").
 *   You may not use this file except in compliance with the License.
 *   A copy of the License is located at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   or in the "license" file accompanying this file. This file is distributed
 *   on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 *   express or implied. See the License for the specific language governing
 *   permissions and limitations under the License.
 */
class SamlAuthRoutes {
  constructor(router, config, sessionStorageFactory, securityClient, coreSetup) {
    this.router = router;
    this.config = config;
    this.sessionStorageFactory = sessionStorageFactory;
    this.securityClient = securityClient;
    this.coreSetup = coreSetup;
  }

  setupRoutes() {
    this.router.get({
      path: '/auth/saml/login',
      validate: {
        query: _configSchema.schema.object({
          nextUrl: _configSchema.schema.maybe(_configSchema.schema.string({
            validate: _next_url.validateNextUrl
          }))
        })
      },
      options: {
        authRequired: false
      }
    }, async (context, request, response) => {
      if (request.auth.isAuthenticated) {
        return response.redirected({
          headers: {
            location: `${this.coreSetup.http.basePath.serverBasePath}/app/opensearch-dashboards`
          }
        });
      }

      try {
        const samlHeader = await this.securityClient.getSamlHeader(request);
        const cookie = {
          saml: {
            nextUrl: request.query.nextUrl,
            requestId: samlHeader.requestId
          }
        };
        this.sessionStorageFactory.asScoped(request).set(cookie);
        return response.redirected({
          headers: {
            location: samlHeader.location
          }
        });
      } catch (error) {
        context.security_plugin.logger.error(`Failed to get saml header: ${error}`);
        return response.internalError(); // TODO: redirect to error page?
      }
    });
    this.router.post({
      path: '/_plugins/_security/saml/acs',
      validate: {
        body: _configSchema.schema.any()
      },
      options: {
        authRequired: false
      }
    }, async (context, request, response) => {
      let requestId = '';
      let nextUrl = '/';

      try {
        const cookie = await this.sessionStorageFactory.asScoped(request).get();

        if (cookie) {
          var _cookie$saml, _cookie$saml2;

          requestId = ((_cookie$saml = cookie.saml) === null || _cookie$saml === void 0 ? void 0 : _cookie$saml.requestId) || '';
          nextUrl = ((_cookie$saml2 = cookie.saml) === null || _cookie$saml2 === void 0 ? void 0 : _cookie$saml2.nextUrl) || `${this.coreSetup.http.basePath.serverBasePath}/app/opensearch-dashboards`;
        }

        if (!requestId) {
          return response.badRequest({
            body: 'Invalid requestId'
          });
        }
      } catch (error) {
        context.security_plugin.logger.error(`Failed to parse cookie: ${error}`);
        return response.badRequest();
      }

      try {
        const credentials = await this.securityClient.authToken(requestId, request.body.SAMLResponse, undefined);
        const user = await this.securityClient.authenticateWithHeader(request, 'authorization', credentials.authorization);
        let expiryTime = Date.now() + this.config.session.ttl;
        const [headerEncoded, payloadEncoded, signature] = credentials.authorization.split('.');

        if (!payloadEncoded) {
          context.security_plugin.logger.error('JWT token payload not found');
        }

        const tokenPayload = JSON.parse(Buffer.from(payloadEncoded, 'base64').toString());

        if (tokenPayload.exp) {
          expiryTime = parseInt(tokenPayload.exp, 10) * 1000;
        }

        const cookie = {
          username: user.username,
          credentials: {
            authHeaderValue: credentials.authorization
          },
          authType: _index.AuthType.SAML,
          expiryTime
        };
        this.sessionStorageFactory.asScoped(request).set(cookie);
        return response.redirected({
          headers: {
            location: nextUrl
          }
        });
      } catch (error) {
        context.security_plugin.logger.error(`SAML SP initiated authentication workflow failed: ${error}`);
      }

      return response.internalError();
    });
    this.router.post({
      path: '/_plugins/_security/saml/acs/idpinitiated',
      validate: {
        body: _configSchema.schema.any()
      },
      options: {
        authRequired: false
      }
    }, async (context, request, response) => {
      const acsEndpoint = `${this.coreSetup.http.basePath.serverBasePath}/_plugins/_security/saml/acs/idpinitiated`;

      try {
        const credentials = await this.securityClient.authToken(undefined, request.body.SAMLResponse, acsEndpoint);
        const user = await this.securityClient.authenticateWithHeader(request, 'authorization', credentials.authorization);
        let expiryTime = Date.now() + this.config.session.ttl;
        const [headerEncoded, payloadEncoded, signature] = credentials.authorization.split('.');

        if (!payloadEncoded) {
          context.security_plugin.logger.error('JWT token payload not found');
        }

        const tokenPayload = JSON.parse(Buffer.from(payloadEncoded, 'base64').toString());

        if (tokenPayload.exp) {
          expiryTime = parseInt(tokenPayload.exp, 10) * 1000;
        }

        const cookie = {
          username: user.username,
          credentials: {
            authHeaderValue: credentials.authorization
          },
          authType: _index.AuthType.SAML,
          expiryTime
        };
        this.sessionStorageFactory.asScoped(request).set(cookie);
        return response.redirected({
          headers: {
            location: `${this.coreSetup.http.basePath.serverBasePath}/app/opensearch-dashboards`
          }
        });
      } catch (error) {
        context.security_plugin.logger.error(`SAML IDP initiated authentication workflow failed: ${error}`);
      }

      return response.internalError();
    });
    this.router.get({
      path: _common.API_AUTH_LOGOUT,
      validate: false
    }, async (context, request, response) => {
      try {
        const authInfo = await this.securityClient.authinfo(request);
        this.sessionStorageFactory.asScoped(request).clear(); // TODO: need a default logout page

        const redirectUrl = authInfo.sso_logout_url || this.coreSetup.http.basePath.serverBasePath || '/';
        return response.redirected({
          headers: {
            location: redirectUrl
          }
        });
      } catch (error) {
        context.security_plugin.logger.error(`SAML logout failed: ${error}`);
        return response.badRequest();
      }
    });
  }

}

exports.SamlAuthRoutes = SamlAuthRoutes;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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