"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;
exports.registerADRoutes = registerADRoutes;

var _lodash = require("lodash");

var _constants = require("../utils/constants");

var _helpers = require("../utils/helpers");

var _adHelpers = require("./utils/adHelpers");

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

function registerADRoutes(apiRouter, adService) {
  apiRouter.post('/detectors', adService.putDetector);
  apiRouter.put('/detectors/{detectorId}', adService.putDetector);
  apiRouter.post('/detectors/_search', adService.searchDetector);
  apiRouter.post('/detectors/results/_search/', adService.searchResults);
  apiRouter.post('/detectors/results/_search', adService.searchResults);
  apiRouter.post('/detectors/results/_search/{resultIndex}/{onlyQueryCustomResultIndex}', adService.searchResults);
  apiRouter.get('/detectors/{detectorId}', adService.getDetector);
  apiRouter.get('/detectors', adService.getDetectors);
  apiRouter.post('/detectors/preview', adService.previewDetector);
  apiRouter.get('/detectors/{id}/results/{isHistorical}/{resultIndex}/{onlyQueryCustomResultIndex}', adService.getAnomalyResults);
  apiRouter.get('/detectors/{id}/results/{isHistorical}', adService.getAnomalyResults);
  apiRouter.delete('/detectors/{detectorId}', adService.deleteDetector);
  apiRouter.post('/detectors/{detectorId}/start', adService.startDetector);
  apiRouter.post('/detectors/{detectorId}/stop/{isHistorical}', adService.stopDetector);
  apiRouter.get('/detectors/{detectorId}/_profile', adService.getDetectorProfile);
  apiRouter.get('/detectors/{detectorName}/_match', adService.matchDetector);
  apiRouter.get('/detectors/_count', adService.getDetectorCount);
  apiRouter.post('/detectors/{detectorId}/_topAnomalies/{isHistorical}', adService.getTopAnomalyResults);
  apiRouter.post('/detectors/_validate/{validationType}', adService.validateDetector);
}

class AdService {
  constructor(client) {
    _defineProperty(this, "client", void 0);

    _defineProperty(this, "deleteDetector", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          detectorId
        } = request.params;
        const response = await this.client.asScoped(request).callAsCurrentUser('ad.deleteDetector', {
          detectorId
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: response
          }
        });
      } catch (err) {
        console.log('Anomaly detector - deleteDetector', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });

    _defineProperty(this, "previewDetector", async (context, request, opensearchDashboardsResponse) => {
      try {
        const requestBody = JSON.stringify((0, _adHelpers.convertPreviewInputKeysToSnakeCase)(request.body));
        const response = await this.client.asScoped(request).callAsCurrentUser('ad.previewDetector', {
          body: requestBody
        });
        const transformedKeys = (0, _helpers.mapKeysDeep)(response, _helpers.toCamel);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            //@ts-ignore
            response: (0, _adHelpers.anomalyResultMapper)(transformedKeys.anomalyResult)
          }
        });
      } catch (err) {
        console.log('Anomaly detector - previewDetector', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });

    _defineProperty(this, "putDetector", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          detectorId
        } = request.params; //@ts-ignore

        const ifSeqNo = request.body.seqNo; //@ts-ignore

        const ifPrimaryTerm = request.body.primaryTerm;
        const requestBody = JSON.stringify((0, _adHelpers.convertDetectorKeysToSnakeCase)(request.body));
        let params = {
          detectorId: detectorId,
          ifSeqNo: ifSeqNo,
          ifPrimaryTerm: ifPrimaryTerm,
          body: requestBody
        };
        let response;

        if ((0, _lodash.isNumber)(ifSeqNo) && (0, _lodash.isNumber)(ifPrimaryTerm)) {
          response = await this.client.asScoped(request).callAsCurrentUser('ad.updateDetector', params);
        } else {
          response = await this.client.asScoped(request).callAsCurrentUser('ad.createDetector', {
            body: params.body
          });
        }

        const resp = { ...response.anomaly_detector,
          id: response._id,
          primaryTerm: response._primary_term,
          seqNo: response._seq_no
        };
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: (0, _adHelpers.convertDetectorKeysToCamelCase)(resp)
          }
        });
      } catch (err) {
        console.log('Anomaly detector - PutDetector', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });

    _defineProperty(this, "validateDetector", async (context, request, opensearchDashboardsResponse) => {
      try {
        let {
          validationType
        } = request.params;
        const requestBody = JSON.stringify((0, _adHelpers.convertPreviewInputKeysToSnakeCase)(request.body));
        const response = await this.client.asScoped(request).callAsCurrentUser('ad.validateDetector', {
          body: requestBody,
          validationType: validationType
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: response
          }
        });
      } catch (err) {
        console.log('Anomaly detector - validateDetector', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });

    _defineProperty(this, "getDetector", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          detectorId
        } = request.params;
        const detectorResponse = await this.client.asScoped(request).callAsCurrentUser('ad.getDetector', {
          detectorId
        }); // Populating static detector fields

        const staticFields = {
          id: detectorResponse._id,
          primaryTerm: detectorResponse._primary_term,
          seqNo: detectorResponse._seq_no,
          ...(0, _adHelpers.convertStaticFieldsToCamelCase)(detectorResponse.anomaly_detector)
        }; // Get real-time and historical task info to populate the
        // task and job-related fields
        // We wrap these calls in a try/catch, and suppress any index_not_found_exceptions
        // which can occur if no detector jobs have been ran on a new cluster.

        let realtimeTasksResponse = {};
        let historicalTasksResponse = {};

        try {
          realtimeTasksResponse = await this.client.asScoped(request).callAsCurrentUser('ad.searchTasks', {
            body: (0, _adHelpers.getLatestTaskForDetectorQuery)(detectorId, true)
          });
          historicalTasksResponse = await this.client.asScoped(request).callAsCurrentUser('ad.searchTasks', {
            body: (0, _adHelpers.getLatestTaskForDetectorQuery)(detectorId, false)
          });
        } catch (err) {
          if (!(0, _adHelpers.isIndexNotFoundError)(err)) {
            throw err;
          }
        }

        const realtimeTask = (0, _lodash.get)((0, _lodash.get)(realtimeTasksResponse, 'hits.hits', []).map(taskResponse => {
          return {
            id: (0, _lodash.get)(taskResponse, '_id'),
            ...(0, _lodash.get)(taskResponse, '_source')
          };
        }), 0);
        const historicalTask = (0, _lodash.get)((0, _lodash.get)(historicalTasksResponse, 'hits.hits', []).map(taskResponse => {
          return {
            id: (0, _lodash.get)(taskResponse, '_id'),
            ...(0, _lodash.get)(taskResponse, '_source')
          };
        }), 0);
        const taskAndJobFields = (0, _adHelpers.convertTaskAndJobFieldsToCamelCase)(realtimeTask, historicalTask, detectorResponse.anomaly_detector_job); // Combine the static and task-and-job-related fields into
        // a final response

        const finalResponse = { ...staticFields,
          ...taskAndJobFields
        };
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: finalResponse
          }
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to get detector', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });

    _defineProperty(this, "startDetector", async (context, request, opensearchDashboardsResponse) => {
      try {
        var _request$body, _request$body2;

        const {
          detectorId
        } = request.params; //@ts-ignore

        const startTime = (_request$body = request.body) === null || _request$body === void 0 ? void 0 : _request$body.startTime; //@ts-ignore

        const endTime = (_request$body2 = request.body) === null || _request$body2 === void 0 ? void 0 : _request$body2.endTime;
        let requestParams = {
          detectorId: detectorId
        };
        let requestPath = 'ad.startDetector'; // If a start and end time are passed: we want to start a historical analysis

        if ((0, _lodash.isNumber)(startTime) && (0, _lodash.isNumber)(endTime)) {
          requestParams = { ...requestParams,
            body: {
              start_time: startTime,
              end_time: endTime
            }
          };
          requestPath = 'ad.startHistoricalDetector';
        }

        const response = await this.client.asScoped(request).callAsCurrentUser(requestPath, requestParams);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: response
          }
        });
      } catch (err) {
        console.log('Anomaly detector - startDetector', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });

    _defineProperty(this, "stopDetector", async (context, request, opensearchDashboardsResponse) => {
      try {
        let {
          detectorId,
          isHistorical
        } = request.params;
        isHistorical = JSON.parse(isHistorical);
        const requestPath = isHistorical ? 'ad.stopHistoricalDetector' : 'ad.stopDetector';
        const response = await this.client.asScoped(request).callAsCurrentUser(requestPath, {
          detectorId
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: response
          }
        });
      } catch (err) {
        console.log('Anomaly detector - stopDetector', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });

    _defineProperty(this, "getDetectorProfile", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          detectorId
        } = request.params;
        const response = await this.client.asScoped(request).callAsCurrentUser('ad.detectorProfile', {
          detectorId
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response
          }
        });
      } catch (err) {
        console.log('Anomaly detector - detectorProfile', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });

    _defineProperty(this, "searchDetector", async (context, request, opensearchDashboardsResponse) => {
      try {
        const requestBody = JSON.stringify(request.body);
        const response = await this.client.asScoped(request).callAsCurrentUser('ad.searchDetector', {
          body: requestBody
        });
        const totalDetectors = (0, _lodash.get)(response, 'hits.total.value', 0);
        const detectors = (0, _lodash.get)(response, 'hits.hits', []).map(detector => ({ ...(0, _adHelpers.convertDetectorKeysToCamelCase)(detector._source),
          id: detector._id,
          seqNo: detector._seq_no,
          primaryTerm: detector._primary_term
        }));
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: {
              totalDetectors,
              detectors
            }
          }
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to search detectors', err);

        if ((0, _adHelpers.isIndexNotFoundError)(err)) {
          return opensearchDashboardsResponse.ok({
            body: {
              ok: true,
              response: {
                totalDetectors: 0,
                detectors: []
              }
            }
          });
        }

        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });

    _defineProperty(this, "searchResults", async (context, request, opensearchDashboardsResponse) => {
      try {
        var {
          resultIndex,
          onlyQueryCustomResultIndex
        } = request.params;

        if (!resultIndex || !resultIndex.startsWith(_constants.CUSTOM_AD_RESULT_INDEX_PREFIX)) {
          // Set resultIndex as '' means no custom result index specified, will only search anomaly result from default index.
          resultIndex = '';
        }

        let requestParams = {
          resultIndex: resultIndex,
          onlyQueryCustomResultIndex: onlyQueryCustomResultIndex
        };
        const requestBody = JSON.stringify(request.body);
        const response = !resultIndex ? await this.client.asScoped(request).callAsCurrentUser('ad.searchResults', {
          body: requestBody
        }) : await this.client.asScoped(request).callAsCurrentUser('ad.searchResultsFromCustomResultIndex', { ...requestParams,
          body: requestBody
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response
          }
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to search anomaly result', err);

        if ((0, _adHelpers.isIndexNotFoundError)(err)) {
          return opensearchDashboardsResponse.ok({
            body: {
              ok: true,
              response: {
                totalDetectors: 0,
                detectors: []
              }
            }
          });
        }

        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });

    _defineProperty(this, "getDetectors", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          from = 0,
          size = 20,
          search = '',
          indices = '',
          sortDirection = _constants.SORT_DIRECTION.DESC,
          sortField = 'name'
        } = request.query;
        const mustQueries = [];

        if (search.trim()) {
          mustQueries.push({
            query_string: {
              fields: ['name', 'description'],
              default_operator: 'AND',
              query: `*${search.trim().split('-').join('* *')}*`
            }
          });
        }

        if (indices.trim()) {
          mustQueries.push({
            query_string: {
              fields: ['indices'],
              default_operator: 'AND',
              query: `*${indices.trim().split('-').join('* *')}*`
            }
          });
        } //Allowed sorting columns


        const sortQueryMap = {
          name: {
            'name.keyword': sortDirection
          },
          indices: {
            'indices.keyword': sortDirection
          },
          lastUpdateTime: {
            last_update_time: sortDirection
          }
        };
        let sort = {};
        const sortQuery = sortQueryMap[sortField];

        if (sortQuery) {
          sort = sortQuery;
        } //Preparing search request


        const requestBody = {
          sort,
          size,
          from,
          query: {
            bool: {
              must: mustQueries
            }
          }
        };
        const response = await this.client.asScoped(request).callAsCurrentUser('ad.searchDetector', {
          body: requestBody
        });
        const totalDetectors = (0, _lodash.get)(response, 'hits.total.value', 0); //Get all detectors from search detector API

        const allDetectors = (0, _lodash.get)(response, 'hits.hits', []).reduce((acc, detectorResponse) => ({ ...acc,
          [detectorResponse._id]: {
            id: detectorResponse._id,
            primaryTerm: detectorResponse._primary_term,
            seqNo: detectorResponse._seq_no,
            ...(0, _adHelpers.convertStaticFieldsToCamelCase)(detectorResponse._source)
          }
        }), {});
        const allDetectorsMap = Object.values(allDetectors).reduce((acc, detector) => ({ ...acc,
          [detector.id]: detector
        }), {}); //Given each detector from previous result, get aggregation to power list

        const allDetectorIds = Object.keys(allDetectorsMap);
        let requestParams = {
          // If specifying result index, will query anomaly result from both default and custom result indices.
          // If no valid result index specified, just query anomaly result from default result index.
          // Here we specify custom AD result index prefix pattern to query all custom result indices.
          resultIndex: _constants.CUSTOM_AD_RESULT_INDEX_PREFIX + '*',
          onlyQueryCustomResultIndex: 'false'
        };
        const aggregationResult = await this.client.asScoped(request).callAsCurrentUser('ad.searchResultsFromCustomResultIndex', { ...requestParams,
          body: (0, _adHelpers.getResultAggregationQuery)(allDetectorIds, {
            from,
            size,
            sortField,
            sortDirection,
            search,
            indices
          })
        });
        const aggsDetectors = (0, _lodash.get)(aggregationResult, 'aggregations.unique_detectors.buckets', []).reduce((acc, agg) => {
          return { ...acc,
            [agg.key]: { ...allDetectorsMap[agg.key],
              totalAnomalies: agg.total_anomalies_in_24hr.doc_count,
              lastActiveAnomaly: agg.latest_anomaly_time.value
            }
          };
        }, {}); // Aggregation will not return values where anomalies for detectors are not generated, loop through it and fill values with 0

        const unUsedDetectors = (0, _lodash.pullAll)(allDetectorIds, Object.keys(aggsDetectors)).reduce((acc, unusedDetector) => {
          return { ...acc,
            [unusedDetector]: { ...allDetectorsMap[unusedDetector],
              totalAnomalies: 0,
              lastActiveAnomaly: 0
            }
          };
        }, {}); // If sorting criteria is from the aggregation manage pagination in memory.

        let finalDetectors = (0, _lodash.orderBy)({ ...aggsDetectors,
          ...unUsedDetectors
        }, [sortField], [sortDirection]);

        if (!sortQueryMap[sortField]) {
          finalDetectors = Object.values(finalDetectors).slice(from, from + size).reduce((acc, detector) => ({ ...acc,
            [detector.id]: detector
          }), {});
        } // Fetch the latest realtime and historical tasks for all detectors
        // using terms aggregations
        // We wrap these calls in a try/catch, and suppress any index_not_found_exceptions
        // which can occur if no detector jobs have been ran on a new cluster.


        let realtimeTasksResponse = {};
        let historicalTasksResponse = {};

        try {
          realtimeTasksResponse = await this.client.asScoped(request).callAsCurrentUser('ad.searchTasks', {
            body: (0, _adHelpers.getLatestDetectorTasksQuery)(true)
          });
          historicalTasksResponse = await this.client.asScoped(request).callAsCurrentUser('ad.searchTasks', {
            body: (0, _adHelpers.getLatestDetectorTasksQuery)(false)
          });
        } catch (err) {
          if (!(0, _adHelpers.isIndexNotFoundError)(err)) {
            throw err;
          }
        }

        const realtimeTasks = (0, _lodash.get)(realtimeTasksResponse, 'aggregations.detectors.buckets', []).reduce((acc, bucket) => {
          return { ...acc,
            [bucket.key]: {
              realtimeTask: (0, _lodash.get)(bucket, 'latest_tasks.hits.hits.0', undefined)
            }
          };
        }, {});
        const historicalTasks = (0, _lodash.get)(historicalTasksResponse, 'aggregations.detectors.buckets', []).reduce((acc, bucket) => {
          return { ...acc,
            [bucket.key]: {
              historicalTask: (0, _lodash.get)(bucket, 'latest_tasks.hits.hits.0', undefined)
            }
          };
        }, {}); // Get real-time and historical task info by looping through each detector & retrieving
        //    - curState by getting real-time task state
        //    - enabledTime by getting real-time task's execution_start time
        //    - taskId by getting historical task's _id

        finalDetectors.forEach(detector => {
          const realtimeTask = (0, _lodash.get)(realtimeTasks[detector.id], 'realtimeTask._source');
          detector.curState = (0, _adHelpers.getTaskState)(realtimeTask);
          detector.enabledTime = (0, _lodash.get)(realtimeTask, 'execution_start_time');
          detector.taskId = (0, _lodash.get)(historicalTasks[detector.id], 'historicalTask._id');
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: {
              totalDetectors,
              detectorList: Object.values(finalDetectors)
            }
          }
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to search detectors', err);

        if ((0, _adHelpers.isIndexNotFoundError)(err)) {
          return opensearchDashboardsResponse.ok({
            body: {
              ok: true,
              response: {
                totalDetectors: 0,
                detectorList: []
              }
            }
          });
        }

        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });

    _defineProperty(this, "getAnomalyResults", async (context, request, opensearchDashboardsResponse) => {
      let {
        id,
        isHistorical,
        resultIndex,
        onlyQueryCustomResultIndex
      } = request.params;

      if (!resultIndex || !resultIndex.startsWith(_constants.CUSTOM_AD_RESULT_INDEX_PREFIX)) {
        // Set resultIndex as '' means no custom result index specified, will only search anomaly result from default index.
        resultIndex = '';
      }

      isHistorical = JSON.parse(isHistorical); // Search by task id if historical, or by detector id if realtime

      const searchTerm = isHistorical ? {
        task_id: id
      } : {
        detector_id: id
      };

      try {
        const {
          from = 0,
          size = 20,
          sortDirection = _constants.SORT_DIRECTION.DESC,
          sortField = _constants.AD_DOC_FIELDS.DATA_START_TIME,
          startTime = 0,
          endTime = 0,
          fieldName = '',
          anomalyThreshold = -1,
          entityList = ''
        } = request.query;
        const entityListAsObj = entityList.length === 0 ? {} : JSON.parse(entityList);
        const entityFilters = (0, _lodash.isEmpty)(entityListAsObj) ? [] : (0, _adHelpers.getFiltersFromEntityList)(entityListAsObj); //Allowed sorting columns

        const sortQueryMap = {
          anomalyGrade: {
            anomaly_grade: sortDirection
          },
          confidence: {
            confidence: sortDirection
          },
          [_constants.AD_DOC_FIELDS.DATA_START_TIME]: {
            [_constants.AD_DOC_FIELDS.DATA_START_TIME]: sortDirection
          },
          [_constants.AD_DOC_FIELDS.DATA_END_TIME]: {
            [_constants.AD_DOC_FIELDS.DATA_END_TIME]: sortDirection
          }
        };
        let sort = {};
        const sortQuery = sortQueryMap[sortField];

        if (sortQuery) {
          sort = sortQuery;
        } //Preparing search request


        const requestBody = {
          sort,
          size,
          from,
          query: {
            bool: {
              filter: [{
                term: searchTerm
              }, {
                range: {
                  anomaly_grade: {
                    gt: anomalyThreshold
                  }
                }
              }, ...entityFilters]
            }
          }
        }; // If querying RT results: remove any results that include a task_id, as this indicates
        // a historical result from a historical task.

        if (!isHistorical) {
          requestBody.query.bool = { ...requestBody.query.bool,
            ...{
              must_not: {
                exists: {
                  field: 'task_id'
                }
              }
            }
          };
        }

        try {
          const filterSize = requestBody.query.bool.filter.length;

          if (fieldName) {
            (startTime || endTime) && (0, _lodash.set)(requestBody.query.bool.filter, `${filterSize}.range.${fieldName}.format`, 'epoch_millis');
            startTime && (0, _lodash.set)(requestBody.query.bool.filter, `${filterSize}.range.${fieldName}.gte`, startTime);
            endTime && (0, _lodash.set)(requestBody.query.bool.filter, `${filterSize}.range.${fieldName}.lte`, endTime);
          }
        } catch (error) {
          console.log('wrong date range filter', error);
        }

        let requestParams = {
          resultIndex: resultIndex,
          onlyQueryCustomResultIndex: onlyQueryCustomResultIndex
        };
        const response = !resultIndex ? await this.client.asScoped(request).callAsCurrentUser('ad.searchResults', {
          body: requestBody
        }) : await this.client.asScoped(request).callAsCurrentUser('ad.searchResultsFromCustomResultIndex', { ...requestParams,
          body: requestBody
        });
        const totalResults = (0, _lodash.get)(response, 'hits.total.value', 0);
        const detectorResult = [];
        const featureResult = {};
        (0, _lodash.get)(response, 'hits.hits', []).forEach(result => {
          detectorResult.push({
            startTime: result._source.data_start_time,
            endTime: result._source.data_end_time,
            plotTime: result._source.data_end_time,
            confidence: result._source.confidence != null && result._source.confidence !== 'NaN' && result._source.confidence > 0 ? (0, _helpers.toFixedNumberForAnomaly)(Number.parseFloat(result._source.confidence)) : 0,
            anomalyGrade: result._source.anomaly_grade != null && result._source.anomaly_grade !== 'NaN' && result._source.anomaly_grade > 0 ? (0, _helpers.toFixedNumberForAnomaly)(Number.parseFloat(result._source.anomaly_grade)) : 0,
            ...(result._source.entity != null ? {
              entity: result._source.entity
            } : {}),
            // TODO: we should refactor other places to read feature data from
            // AnomalyResult, instead of having separate FeatureData which is hard
            // to know feature data belongs to which anomaly result
            features: this.getFeatureData(result)
          });

          result._source.feature_data.forEach(featureData => {
            if (!featureResult[featureData.feature_id]) {
              featureResult[featureData.feature_id] = [];
            }

            featureResult[featureData.feature_id].push({
              startTime: result._source.data_start_time,
              endTime: result._source.data_end_time,
              plotTime: result._source.data_end_time,
              data: featureData.data != null && featureData.data !== 'NaN' ? (0, _helpers.toFixedNumberForAnomaly)(Number.parseFloat(featureData.data)) : 0
            });
          });
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: {
              totalAnomalies: totalResults,
              results: detectorResult,
              featureResults: featureResult
            }
          }
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to get results', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });

    _defineProperty(this, "getTopAnomalyResults", async (context, request, opensearchDashboardsResponse) => {
      try {
        let {
          detectorId,
          isHistorical
        } = request.params;
        isHistorical = JSON.parse(isHistorical);
        const requestPath = isHistorical ? 'ad.topHistoricalAnomalyResults' : 'ad.topAnomalyResults';
        const response = await this.client.asScoped(request).callAsCurrentUser(requestPath, {
          detectorId: detectorId,
          body: request.body
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: response
          }
        });
      } catch (err) {
        console.log('Anomaly detector - getTopAnomalyResults', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });

    _defineProperty(this, "matchDetector", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          detectorName
        } = request.params;
        const response = await this.client.asScoped(request).callAsCurrentUser('ad.matchDetector', {
          detectorName
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: response
          }
        });
      } catch (err) {
        console.log('Anomaly detector - matchDetector', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });

    _defineProperty(this, "getDetectorCount", async (context, request, opensearchDashboardsResponse) => {
      try {
        const response = await this.client.asScoped(request).callAsCurrentUser('ad.detectorCount');
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: response
          }
        });
      } catch (err) {
        console.log('Anomaly detector - getDetectorCount', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });

    _defineProperty(this, "getFeatureData", rawResult => {
      const featureResult = {};

      rawResult._source.feature_data.forEach(featureData => {
        featureResult[featureData.feature_id] = {
          startTime: rawResult._source.data_start_time,
          endTime: rawResult._source.data_end_time,
          plotTime: rawResult._source.data_end_time,
          data: featureData.data != null && featureData.data !== 'NaN' ? (0, _helpers.toFixedNumberForAnomaly)(Number.parseFloat(featureData.data)) : 0
        };
      });

      return featureResult;
    });

    this.client = client;
  }

}

exports.default = AdService;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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