/*
 * Copyright (C) 2017 Purism SPC
 *
 * SPDX-License-Identifier: LGPL-2.1-or-later
 */

#pragma once

#if !defined(_HANDY_INSIDE) && !defined(HANDY_COMPILATION)
#error "Only <handy.h> can be included directly."
#endif

/**
 * HDY_MAJOR_VERSION:
 *
 * Handy major version component (e.g. 1 if the version is 1.2.3).
 */
#define HDY_MAJOR_VERSION              (1)

/**
 * HDY_MINOR_VERSION:
 *
 * Handy minor version component (e.g. 2 if the version is 1.2.3).
 */
#define HDY_MINOR_VERSION              (6)

/**
 * HDY_MICRO_VERSION:
 *
 * Handy micro version component (e.g. 3 if the version is 1.2.3).
 */
#define HDY_MICRO_VERSION              (3)

/**
 * HDY_VERSION:
 *
 * Handy version (e.g. 1.2.3).
 */
#define HDY_VERSION                    (1.6.3)

/**
 * HDY_VERSION_S:
 *
 * Handy version, encoded as a string, useful for printing and
 * concatenation.
 */
#define HDY_VERSION_S                  "1.6.3"

#define HDY_ENCODE_VERSION(major,minor,micro) \
        ((major) << 24 | (minor) << 16 | (micro) << 8)

/**
 * HDY_VERSION_HEX:
 *
 * Handy version, encoded as an hexadecimal number, useful for
 * integer comparisons.
 */
#define HDY_VERSION_HEX \
        (HDY_ENCODE_VERSION (HDY_MAJOR_VERSION, HDY_MINOR_VERSION, HDY_MICRO_VERSION))

/**
 * HDY_CHECK_VERSION:
 * @major: required major version
 * @minor: required minor version
 * @micro: required micro version
 *
 * Compile-time version checking. Evaluates to `TRUE` if the version
 * of handy is greater than the required one.
 */
#define HDY_CHECK_VERSION(major,minor,micro)   \
        (HDY_MAJOR_VERSION > (major) || \
         (HDY_MAJOR_VERSION == (major) && HDY_MINOR_VERSION > (minor)) || \
         (HDY_MAJOR_VERSION == (major) && HDY_MINOR_VERSION == (minor) && \
          HDY_MICRO_VERSION >= (micro)))

/**
 * HDY_VERSION_1_2:
 *
 * A macro that evaluates to the 1.2 version of Handy, in a format
 * that can be used by the C pre-processor.
 *
 * Since: 1.2
 */
#define HDY_VERSION_1_2 (HDY_ENCODE_VERSION (1, 2, 0))

/**
 * HDY_VERSION_1_4:
 *
 * A macro that evaluates to the 1.4 version of Handy, in a format
 * that can be used by the C pre-processor.
 *
 * Since: 1.4
 */
#define HDY_VERSION_1_4 (HDY_ENCODE_VERSION (1, 4, 0))

/**
 * HDY_VERSION_1_6:
 *
 * A macro that evaluates to the 1.6 version of Handy, in a format
 * that can be used by the C pre-processor.
 *
 * Since: 1.6
 */
#define HDY_VERSION_1_6 (HDY_ENCODE_VERSION (1, 6, 0))

#ifndef _HDY_EXTERN
#define _HDY_EXTERN extern
#endif

#if defined(HDY_DISABLE_DEPRECATION_WARNINGS) || defined(HANDY_COMPILATION)
#  define _HDY_DEPRECATED             _HDY_EXTERN
#  define _HDY_DEPRECATED_FOR(f)      _HDY_EXTERN
#  define _HDY_DEPRECATED_TYPE
#  define _HDY_DEPRECATED_TYPE_FOR(f)
#else
#  define _HDY_DEPRECATED             G_DEPRECATED        _HDY_EXTERN
#  define _HDY_DEPRECATED_FOR(f)      G_DEPRECATED_FOR(f) _HDY_EXTERN
#  define _HDY_DEPRECATED_TYPE        G_DEPRECATED
#  define _HDY_DEPRECATED_TYPE_FOR(f) G_DEPRECATED_FOR(f)
#endif

#ifndef HDY_VERSION_MAX_ALLOWED
# define HDY_VERSION_MAX_ALLOWED HDY_VERSION_1_6
#endif

#ifndef HDY_VERSION_MIN_REQUIRED
# define HDY_VERSION_MIN_REQUIRED HDY_VERSION_1_6
#endif

#define HDY_UNAVAILABLE(major, minor) G_UNAVAILABLE(major, minor) _HDY_EXTERN

#define HDY_AVAILABLE_IN_ALL _HDY_EXTERN

#if HDY_VERSION_MAX_ALLOWED < HDY_VERSION_1_2
# define HDY_AVAILABLE_IN_1_2 HDY_UNAVAILABLE(1, 2)
#else
# define HDY_AVAILABLE_IN_1_2 _HDY_EXTERN
#endif

#if HDY_VERSION_MIN_REQUIRED >= HDY_VERSION_1_2
# define HDY_DEPRECATED_IN_1_2             _HDY_DEPRECATED
# define HDY_DEPRECATED_IN_1_2_FOR(f)      _HDY_DEPRECATED_FOR(f)
# define HDY_DEPRECATED_TYPE_IN_1_2        _HDY_DEPRECATED_TYPE
# define HDY_DEPRECATED_TYPE_IN_1_2_FOR(f) _HDY_DEPRECATED_TYPE_FOR(f)
#else
# define HDY_DEPRECATED_IN_1_2             _HDY_EXTERN
# define HDY_DEPRECATED_IN_1_2_FOR(f)      _HDY_EXTERN
# define HDY_DEPRECATED_TYPE_IN_1_2
# define HDY_DEPRECATED_TYPE_IN_1_2_FOR(f)
#endif

#if HDY_VERSION_MAX_ALLOWED < HDY_VERSION_1_4
# define HDY_AVAILABLE_IN_1_4 HDY_UNAVAILABLE(1, 4)
#else
# define HDY_AVAILABLE_IN_1_4 _HDY_EXTERN
#endif

#if HDY_VERSION_MIN_REQUIRED >= HDY_VERSION_1_4
# define HDY_DEPRECATED_IN_1_4             _HDY_DEPRECATED
# define HDY_DEPRECATED_IN_1_4_FOR(f)      _HDY_DEPRECATED_FOR(f)
# define HDY_DEPRECATED_TYPE_IN_1_4        _HDY_DEPRECATED_TYPE
# define HDY_DEPRECATED_TYPE_IN_1_4_FOR(f) _HDY_DEPRECATED_TYPE_FOR(f)
#else
# define HDY_DEPRECATED_IN_1_4             _HDY_EXTERN
# define HDY_DEPRECATED_IN_1_4_FOR(f)      _HDY_EXTERN
# define HDY_DEPRECATED_TYPE_IN_1_4
# define HDY_DEPRECATED_TYPE_IN_1_4_FOR(f)
#endif

#if HDY_VERSION_MAX_ALLOWED < HDY_VERSION_1_6
# define HDY_AVAILABLE_IN_1_6 HDY_UNAVAILABLE(1, 6)
#else
# define HDY_AVAILABLE_IN_1_6 _HDY_EXTERN
#endif

#if HDY_VERSION_MIN_REQUIRED >= HDY_VERSION_1_6
# define HDY_DEPRECATED_IN_1_6             _HDY_DEPRECATED
# define HDY_DEPRECATED_IN_1_6_FOR(f)      _HDY_DEPRECATED_FOR(f)
# define HDY_DEPRECATED_TYPE_IN_1_6        _HDY_DEPRECATED_TYPE
# define HDY_DEPRECATED_TYPE_IN_1_6_FOR(f) _HDY_DEPRECATED_TYPE_FOR(f)
#else
# define HDY_DEPRECATED_IN_1_6             _HDY_EXTERN
# define HDY_DEPRECATED_IN_1_6_FOR(f)      _HDY_EXTERN
# define HDY_DEPRECATED_TYPE_IN_1_6
# define HDY_DEPRECATED_TYPE_IN_1_6_FOR(f)
#endif
