/*
 * Copyright (C) 2021 Apple Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1.  Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 * 2.  Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE INC. AND ITS CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL APPLE INC. OR ITS CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#include "NotificationManagerMessageHandler.h"

#include "ArgumentCoders.h"
#include "Decoder.h"
#include "HandleMessage.h"
#include "NotificationManagerMessageHandlerMessages.h"
#include "WebCoreArgumentCoders.h"
#include <WebCore/NotificationData.h>
#include <wtf/UUID.h>
#include <wtf/Vector.h>
#include <wtf/text/WTFString.h>

namespace Messages {

namespace NotificationManagerMessageHandler {

void RequestSystemNotificationPermission::callReply(IPC::Decoder& decoder, CompletionHandler<void(bool&&)>&& completionHandler)
{
    std::optional<bool> allowed;
    decoder >> allowed;
    if (!allowed) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*allowed));
}

void RequestSystemNotificationPermission::cancelReply(CompletionHandler<void(bool&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<bool>::create());
}

void RequestSystemNotificationPermission::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, bool allowed)
{
    encoder.get() << allowed;
    connection.sendSyncReply(WTFMove(encoder));
}

} // namespace NotificationManagerMessageHandler

} // namespace Messages

namespace WebKit {

void NotificationManagerMessageHandler::didReceiveMessage(IPC::Connection& connection, IPC::Decoder& decoder)
{
    if (decoder.messageName() == Messages::NotificationManagerMessageHandler::RequestSystemNotificationPermission::name())
        return IPC::handleMessageAsync<Messages::NotificationManagerMessageHandler::RequestSystemNotificationPermission>(connection, decoder, this, &NotificationManagerMessageHandler::requestSystemNotificationPermission);
    if (decoder.messageName() == Messages::NotificationManagerMessageHandler::ShowNotification::name())
        return IPC::handleMessageWantsConnection<Messages::NotificationManagerMessageHandler::ShowNotification>(connection, decoder, this, &NotificationManagerMessageHandler::showNotification);
    if (decoder.messageName() == Messages::NotificationManagerMessageHandler::CancelNotification::name())
        return IPC::handleMessage<Messages::NotificationManagerMessageHandler::CancelNotification>(connection, decoder, this, &NotificationManagerMessageHandler::cancelNotification);
    if (decoder.messageName() == Messages::NotificationManagerMessageHandler::ClearNotifications::name())
        return IPC::handleMessage<Messages::NotificationManagerMessageHandler::ClearNotifications>(connection, decoder, this, &NotificationManagerMessageHandler::clearNotifications);
    if (decoder.messageName() == Messages::NotificationManagerMessageHandler::DidDestroyNotification::name())
        return IPC::handleMessage<Messages::NotificationManagerMessageHandler::DidDestroyNotification>(connection, decoder, this, &NotificationManagerMessageHandler::didDestroyNotification);
    UNUSED_PARAM(connection);
    UNUSED_PARAM(decoder);
#if ENABLE(IPC_TESTING_API)
    if (connection.ignoreInvalidMessageForTesting())
        return;
#endif // ENABLE(IPC_TESTING_API)
    ASSERT_NOT_REACHED_WITH_MESSAGE("Unhandled message %s to %" PRIu64, IPC::description(decoder.messageName()), decoder.destinationID());
}

} // namespace WebKit
