"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.BotCommandHandler = exports.BotCommandError = exports.BotCommand = void 0;
require("reflect-metadata");
const markdown_it_1 = __importDefault(require("markdown-it"));
const string_argv_1 = __importDefault(require("string-argv"));
const md = new markdown_it_1.default();
const botCommandSymbol = Symbol("botCommandMetadata");
/**
 * Expose a function as a command. The arugments of the function *must* take a single
 * `CommandArguments` parameter.
 * @param options Metadata about the command.
 */
function BotCommand(options) {
    Reflect.metadata(botCommandSymbol, options);
}
exports.BotCommand = BotCommand;
/**
 * Error to be thrown by commands that could not complete a request.
 */
class BotCommandError extends Error {
    /**
     * Construct a `BotCommandError` instance.
     * @param error The inner error
     * @param humanText The error to be shown to the user.
     */
    constructor(error, humanText) {
        super(typeof error === "string" ? error : error.message);
        this.humanText = humanText;
        if (typeof error !== "string") {
            this.stack = error.stack;
        }
    }
}
exports.BotCommandError = BotCommandError;
class BotCommandHandler {
    /**
     * Construct a new command helper.
     * @param prototype The prototype of the class to bind to for bot commands.
     *                  It should contain at least one `BotCommand`.
     * @param instance The instance of the above prototype to bind to for function calls.
     * @param prefix A prefix to be stripped from commands (useful if using multiple handlers). The prefix
     * should **include** any whitspace E.g. `!irc `.
     */
    constructor(prototype, instance, prefix) {
        this.prefix = prefix;
        let content = "Commands:\n";
        const botCommands = {};
        Object.getOwnPropertyNames(prototype).forEach(propetyKey => {
            var _a;
            const b = Reflect.getMetadata(botCommandSymbol, prototype, propetyKey);
            if (!b) {
                // Not a bot command function.
                return;
            }
            const requiredArgs = b.requiredArgs.join(" ");
            const optionalArgs = ((_a = b.optionalArgs) === null || _a === void 0 ? void 0 : _a.map((arg) => `[${arg}]`).join(" ")) || "";
            content += ` - \`${this.prefix || ""}${b.name}\` ${requiredArgs} ${optionalArgs} - ${b.help}\n`;
            // We know that this is safe.
            botCommands[b.name] = {
                fn: prototype[propetyKey].bind(instance),
                requiredArgs: b.requiredArgs,
                optionalArgs: b.optionalArgs,
            };
        });
        if (Object.keys(botCommands).length === 0) {
            throw Error('Prototype did not have any bot commands bound');
        }
        this.helpMessage = {
            msgtype: "m.notice",
            body: content,
            formatted_body: md.render(content),
            format: "org.matrix.custom.html"
        };
        this.botCommands = botCommands;
    }
    /**
     * Process a command given by a user.
     * @param userCommand The command string given by the user in it's entireity. Should be plain text.
     * @throws With a `BotCommandError` if the command didn't contain enough arugments. Any errors thrown
     *         from the handler function will be passed through.
     * @returns `true` if the command was handled by this handler instance.
     */
    async handleCommand(userCommand, request) {
        // The processor may require a prefix (like `!github `). Check for it
        // and strip away if found.
        if (this.prefix) {
            if (!userCommand.startsWith(this.prefix)) {
                return false;
            }
            userCommand = userCommand.substring(this.prefix.length);
        }
        const parts = string_argv_1.default(userCommand);
        // This loop is a little complex:
        // We want to find the most likely candiate for handling this command
        // which we do so by joining together the whole command string and
        // matching against any commands with the same name.
        // If we can't find any, we strip away the last arg and try again.
        // E.g. In the case of `add one + two`, we would search for:
        // - `add one + two`
        // - `add one +`
        // - `add one`
        // - `add`
        // We iterate backwards so that command trees can be respected.
        for (let i = parts.length; i > 0; i--) {
            const cmdPrefix = parts.slice(0, i).join(" ").toLowerCase();
            const command = this.botCommands[cmdPrefix];
            if (!command) {
                continue;
            }
            // We have a match!
            if (command.requiredArgs.length > parts.length - i) {
                throw new BotCommandError("Missing arguments", "Missing required arguments for this command");
            }
            await command.fn({
                request,
                args: parts.slice(i),
            });
            return true;
        }
        return false;
    }
}
exports.BotCommandHandler = BotCommandHandler;
//# sourceMappingURL=bot-commands.js.map