"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    Object.defineProperty(o, k2, { enumerable: true, get: function() { return m[k]; } });
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (k !== "default" && Object.prototype.hasOwnProperty.call(mod, k)) __createBinding(result, mod, k);
    __setModuleDefault(result, mod);
    return result;
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.ProvisioningApi = void 0;
const express_1 = __importStar(require("express"));
const uuid_1 = require("uuid");
const axios_1 = __importDefault(require("axios"));
const cors_1 = __importDefault(require("cors"));
const logging_1 = __importDefault(require("../components/logging"));
const request_1 = __importDefault(require("./request"));
const log = logging_1.default.get("ProvisioningApi");
const DEFAULT_WIDGET_TOKEN_PREFIX = "br-sdk-utoken-";
const DEFAULT_WIDGET_TOKEN_LIFETIME_MS = 24 * 60 * 60 * 1000; // One day
/**
 * The provisioning API serves two classes of clients:
 *  - Integration managers which provide a unique secret token, and a userId
 *  - Widget users which provide a openId token.
 */
class ProvisioningApi {
    constructor(store, opts) {
        this.store = store;
        this.opts = opts;
        this.app = express_1.default();
        this.app.use((req, _res, next) => {
            log.info(`${req.method} ${req.path} ${req.ip || ''} ${req.headers["user-agent"] || ''}`);
            next();
        });
        this.widgetTokenPrefix = opts.widgetTokenPrefix || DEFAULT_WIDGET_TOKEN_PREFIX;
        this.widgetTokenLifetimeMs = opts.widgetTokenLifetimeMs || DEFAULT_WIDGET_TOKEN_LIFETIME_MS;
        this.opts.apiPrefix = opts.apiPrefix || "/api";
        this.app.get('/health', this.getHealth.bind(this));
        this.app.use(cors_1.default());
        if (opts.widgetFrontendLocation) {
            this.app.use('/', express_1.default.static(opts.widgetFrontendLocation));
        }
        this.baseRoute = express_1.Router();
        this.baseRoute.use(express_1.default.json());
        // Unsecured requests
        this.baseRoute.post(`/v1/exchange_openid`, (req, res) => this.postExchangeOpenId(req, res));
        // Secure requests
        // addRoute ensures all successful requests are of type ProvisioningRequest
        this.baseRoute.use("/", this.authenticateRequest.bind(this));
        this.addRoute("get", "/v1/session", this.getSession.bind(this));
        this.addRoute("delete", "/v1/session", this.getSession.bind(this));
        this.addRoute("delete", "/v1/session/all", this.getSession.bind(this));
        this.app.use(this.opts.apiPrefix, this.baseRoute);
    }
    start(port, hostname = "0.0.0.0", backlog = 10) {
        if (this.opts.expressApp) {
            log.warn(`Ignoring call to start(), api configured to use parent express app`);
            return;
        }
        log.info(`Widget API listening on port ${port}`);
        this.server = this.app.listen(port, hostname, backlog);
    }
    close() {
        var _a;
        (_a = this.server) === null || _a === void 0 ? void 0 : _a.close();
    }
    addRoute(method, path, handler, fnName) {
        this.baseRoute[method](path, (req, res, next) => {
            const expRequest = req;
            const provisioningRequest = new request_1.default(expRequest, expRequest.matrixUserId, expRequest.matrixWidgetToken ? "widget" : "provisioner", expRequest.matrixWidgetToken, fnName);
            handler(provisioningRequest, res, next);
        });
    }
    authenticateRequest(req, res, next) {
        var _a, _b;
        const authHeader = (_a = req.header("Authorization")) === null || _a === void 0 ? void 0 : _a.toLowerCase();
        if (!authHeader) {
            res.status(400).send({
                error: 'No Authorization header'
            });
            return;
        }
        const token = authHeader.startsWith("bearer ") && authHeader.substr("bearer ".length);
        if (!token) {
            return;
        }
        const requestProv = req;
        if (token === this.opts.provisioningToken) {
            // Integration managers splice in the user_id in the body.
            const userId = (_b = req.body) === null || _b === void 0 ? void 0 : _b.user_id;
            if (!userId) {
                res.status(400).send({
                    error: 'No user_id in body'
                });
                return;
            }
            requestProv.matrixUserId = userId;
            requestProv.matrixWidgetToken = undefined;
            next();
            return;
        }
        this.store.getSessionForToken(token).then(session => {
            if (session.expiresTs < Date.now()) {
                this.store.deleteSession(token);
                throw Error('Token expired');
            }
            requestProv.matrixUserId = session.userId;
            requestProv.matrixWidgetToken = token;
            next();
        }).catch(() => {
            res.status(401).send({
                error: 'Could not authenticate with token'
            });
        });
    }
    getHealth(req, res) {
        res.send({ ok: true });
    }
    getSession(req, res) {
        res.send({
            userId: req.userId,
            type: req.requestSource,
        });
    }
    async deleteSession(req, res) {
        if (!req.widgetToken) {
            req.log.debug("tried to delete session");
            res.status(400).send({
                error: "Session cannot be deleted",
            });
            return;
        }
        try {
            await this.store.deleteSession(req.widgetToken);
        }
        catch (ex) {
            req.log.error("Failed to delete session", ex);
            res.status(500).send({
                error: "Session could be deleted",
            });
        }
    }
    async deleteAllSessions(req, res) {
        if (!req.widgetToken) {
            req.log.debug("tried to delete session");
            res.status(400).send({
                error: "Session cannot be deleted",
            });
            return;
        }
        try {
            await this.store.deleteAllSessions(req.userId);
        }
        catch (ex) {
            req.log.error("Failed to delete all sessions", ex);
            res.status(500).send({
                error: "Sessions could be deleted",
            });
        }
    }
    async postExchangeOpenId(req, res) {
        var _a, _b;
        const server = (_a = req.body) === null || _a === void 0 ? void 0 : _a.matrixServer;
        const openIdToken = (_b = req.body) === null || _b === void 0 ? void 0 : _b.openIdToken;
        let url;
        // TODO: Need a MUCH better impl:
        try {
            const wellKnown = await axios_1.default.get(`https://${server}/.well-known/matrix/server`, {
                validateStatus: null
            });
            if (wellKnown.status === 200) {
                url = `https://${wellKnown.data["m.server"]}`;
            }
            else {
                url = `https://${server}:8448`;
            }
        }
        catch (ex) {
            log.warn(`Failed to fetch the server URL for ${server}`, ex);
            // TODO: need better fail logic
            res.status(500).send({
                error: "Failed to fetch server url",
            });
            return;
        }
        // Now do the token exchange
        try {
            const response = await axios_1.default.get(`${url}/_matrix/federation/v1/openid/userinfo`, {
                params: {
                    access_token: openIdToken,
                },
            });
            if (!response.data.sub) {
                log.warn(`Server responded with invalid sub information for ${server}`, response.data);
                res.status(500).send({
                    error: "Server did not respond with the correct sub information",
                });
                return;
            }
            const userId = response.data.sub;
            const token = this.widgetTokenPrefix + uuid_1.v4().replace(/-/g, "");
            const expiresTs = Date.now() + this.widgetTokenLifetimeMs;
            await this.store.createSession({
                userId,
                token,
                expiresTs,
            });
            res.send({ token, userId });
        }
        catch (ex) {
            log.warn(`Failed to exhcnage the token for ${server}`, ex);
            res.status(500).send({
                error: "Failed to exchange token",
            });
            return;
        }
    }
}
exports.ProvisioningApi = ProvisioningApi;
//# sourceMappingURL=api.js.map