import { MatrixClient } from "./MatrixClient";
import { MSC2380MediaInfo } from "./models/unstable/MediaInfo";
/**
 * Represents a profile for a group
 * @category Unstable APIs
 */
export interface GroupProfile {
    /**
     * The name of the group
     */
    name: string;
    /**
     * The avatar for the group. Must be a MSC URI.
     */
    avatar_url: string;
    /**
     * The short description for the group. Equivalent to a room's topic.
     */
    short_description: string;
    /**
     * The long description for the group. Most clients will support HTML
     * in this.
     */
    long_description: string;
}
/**
 * Unstable APIs that shouldn't be used in most circumstances.
 * @category Unstable APIs
 */
export declare class UnstableApis {
    private client;
    constructor(client: MatrixClient);
    /**
     * Gets the local room aliases that are published for a given room.
     * @param {string} roomId The room ID to get local aliases for.
     * @returns {Promise<string[]>} Resolves to the aliases on the room, or an empty array.
     * @deprecated Relies on MSC2432 endpoint.
     */
    getRoomAliases(roomId: string): Promise<string[]>;
    /**
     * Creates a group.
     * @param {string} localpart The localpart for the group
     * @return {Promise<string>} Resolves to the created group ID.
     */
    createGroup(localpart: string): Promise<string>;
    /**
     * Invites a user to the group.
     * @param {string} groupId The group ID to invite the user to.
     * @param {string} userId The user ID to invite to the group.
     * @return {Promise<"join" | "invite" | "reject">} Resolves to the invite state for
     *  the user. This is normally "invite", but may be "join" or "reject" if the user's
     *  homeserver accepted/rejected the invite right away.
     */
    inviteUserToGroup(groupId: string, userId: string): Promise<"join" | "invite" | "reject">;
    /**
     * Kicks a user from a group.
     * @param {string} groupId The group ID to kick the user from.
     * @param {string} userId The user ID to kick from the group.
     * @return {Promise<any>} Resolves when completed.
     */
    kickUserFromGroup(groupId: string, userId: string): Promise<any>;
    /**
     * Updates a group's profile
     * @param {string} groupId The group ID to update.
     * @param {GroupProfile} profile The profile to update the group with.
     * @return {Promise<any>} Resolves when completed.
     */
    setGroupProfile(groupId: string, profile: GroupProfile): Promise<any>;
    /**
     * Sets a group's join policy to either be publicly joinable (open) or
     * require an invite (invite).
     * @param {string} groupId The group ID to set the policy for.
     * @param {"open" | "invite"} policy The policy to set.
     * @return {Promise<any>} Resolves when completed.
     */
    setGroupJoinPolicy(groupId: string, policy: "open" | "invite"): Promise<any>;
    /**
     * Adds a room to a group.
     * @param {string} groupId The group ID to add the room to.
     * @param {string} roomId The room ID to add to the group.
     * @param {boolean} isPublic Whether this group-room association is visible to non-members. Optional. Defaults to true.
     * @return {Promise<any>} Resolves when completed.
     */
    addRoomToGroup(groupId: string, roomId: string, isPublic?: boolean): Promise<any>;
    /**
     * Updates the visibility of a room in a group.
     * @param {string} groupId The group ID of the room to update.
     * @param {string} roomId The room ID of the room to update.
     * @param {boolean} isPublic Whether this group-room association is visible to non-members.
     * @return {Promise<any>} Resolves when completed.
     */
    updateGroupRoomVisibility(groupId: string, roomId: string, isPublic: boolean): Promise<any>;
    /**
     * Removes a room from a group.
     * @param {string} groupId The group ID to remove the room from.
     * @param {string} roomId The room ID to remove from the group.
     * @return {Promise<any>} Resolves when completed.
     */
    removeRoomFromGroup(groupId: string, roomId: string): Promise<any>;
    /**
     * Gets a group's profile.
     * @param {string} groupId The group ID to fetch the profile of.
     * @return {Promise<GroupProfile>} Resolves to the profile of the group.
     */
    getGroupProfile(groupId: string): Promise<GroupProfile>;
    /**
     * Gets the users in a group.
     * @param {string} groupId The group ID of which to get the users.
     * @return {Promise<any[]>} Resolves to an array of all the users in the group.
     */
    getGroupUsers(groupId: string): Promise<any[]>;
    /**
     * Gets the invited users of a group.
     * @param {string} groupId The group ID of which to get the invited users.
     * @return {Promise<any[]>} Resolves to an array of all the users invited to the group.
     */
    getGroupInvitedUsers(groupId: string): Promise<any[]>;
    /**
     * Gets the rooms of a group.
     * @param {string} groupId The group ID of which to get all the rooms.
     * @return {Promise<any[]>} Resolves to an array of all the rooms of the group.
     */
    getGroupRooms(groupId: string): Promise<any[]>;
    /**
     * Accepts an invite to a group.
     * @param {string} groupId The group ID of which to accept the invite of.
     * @return {Promise<any>} Resolves when completed.
     */
    acceptGroupInvite(groupId: string): Promise<any>;
    /**
     * Joins a group.
     * @param {string} groupId The group ID to join.
     * @return {Promise<any>} Resolves when completed.
     */
    joinGroup(groupId: string): Promise<any>;
    /**
     * Leaves a group.
     * @param {string} groupId The group ID of the group to leave.
     * @return {Promise<any>} Resolves when completed.
     */
    leaveGroup(groupId: string): Promise<any>;
    /**
     * Sets the publicity of a group.
     * @param {string} groupId The group ID to set the publicity of.
     * @param {boolean} publicise If the group should be publicised.
     * @return {Promise<any>} Resolves when completed.
     */
    setGroupPublicity(groupId: string, publicise: boolean): Promise<any>;
    /**
     * Gets all group IDs joined.
     * @return {Promise<string[]>} Resolves to the group IDs of the joined groups.
     */
    getJoinedGroups(): Promise<string[]>;
    /**
     * Gets the group IDs that the specified user has publicised.
     * @param {string} userId The user ID to fetch the publicised groups of.
     * @return {Promise<string[]>} Resolves to the publicised group IDs of that user.
     */
    getPublicisedGroups(userId: string): Promise<string[]>;
    /**
     * Adds a reaction to an event. The contract for this function may change in the future.
     * @param {string} roomId The room ID to react in
     * @param {string} eventId The event ID to react against, in the given room
     * @param {string} emoji The emoji to react with
     * @returns {Promise<string>} Resolves to the event ID of the reaction
     */
    addReactionToEvent(roomId: string, eventId: string, emoji: string): Promise<string>;
    /**
     * Get relations for a given event.
     * @param {string} roomId The room ID to for the given event.
     * @param {string} eventId The event ID to list reacations for.
     * @param {string?} relationType The type of reaction (e.g. `m.room.member`) to filter for. Optional.
     * @param {string?} eventType The type of event to look for (e.g. `m.room.member`). Optional.
     * @returns {Promise<{original_event: any, chunk: any[]}>} Resolves a object containing the original event, and a chunk of relations
     */
    getRelationsForEvent(roomId: string, eventId: string, relationType?: string, eventType?: string): Promise<{
        original_event: any;
        chunk: any[];
    }>;
    /**
     * Get information about a media item. Implements MSC2380
     * @param {string} mxc The MXC to get information about.
     * @returns {Promise<MSC2380MediaInfo>} Resolves a object containing the media information.
     */
    getMediaInfo(mxcUrl: string): Promise<MSC2380MediaInfo>;
}
