"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SamlAuthRoutes = void 0;

var _configSchema = require("@osd/config-schema");

var _next_url = require("../../../utils/next_url");

/*
 *   Copyright OpenSearch Contributors
 *
 *   Licensed under the Apache License, Version 2.0 (the "License").
 *   You may not use this file except in compliance with the License.
 *   A copy of the License is located at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   or in the "license" file accompanying this file. This file is distributed
 *   on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 *   express or implied. See the License for the specific language governing
 *   permissions and limitations under the License.
 */
class SamlAuthRoutes {
  constructor(router, // @ts-ignore: unused variable
  config, sessionStorageFactory, securityClient, coreSetup) {
    this.router = router;
    this.config = config;
    this.sessionStorageFactory = sessionStorageFactory;
    this.securityClient = securityClient;
    this.coreSetup = coreSetup;
  }

  setupRoutes() {
    this.router.get({
      path: `/auth/saml/login`,
      validate: {
        query: _configSchema.schema.object({
          nextUrl: _configSchema.schema.maybe(_configSchema.schema.string({
            validate: _next_url.validateNextUrl
          })),
          redirectHash: _configSchema.schema.string()
        })
      },
      options: {
        authRequired: false
      }
    }, async (context, request, response) => {
      if (request.auth.isAuthenticated) {
        return response.redirected({
          headers: {
            location: `${this.coreSetup.http.basePath.serverBasePath}/app/opensearch-dashboards`
          }
        });
      }

      try {
        const samlHeader = await this.securityClient.getSamlHeader(request);
        const cookie = {
          saml: {
            nextUrl: request.query.nextUrl,
            requestId: samlHeader.requestId,
            redirectHash: request.query.redirectHash === 'true'
          }
        };
        this.sessionStorageFactory.asScoped(request).set(cookie);
        return response.redirected({
          headers: {
            location: samlHeader.location
          }
        });
      } catch (error) {
        context.security_plugin.logger.error(`Failed to get saml header: ${error}`);
        return response.internalError(); // TODO: redirect to error page?
      }
    });
    this.router.post({
      path: `/_opendistro/_security/saml/acs`,
      validate: {
        body: _configSchema.schema.any()
      },
      options: {
        authRequired: false
      }
    }, async (context, request, response) => {
      let requestId = '';
      let nextUrl = '/';
      let redirectHash = false;

      try {
        const cookie = await this.sessionStorageFactory.asScoped(request).get();

        if (cookie) {
          var _cookie$saml, _cookie$saml2, _cookie$saml3;

          requestId = ((_cookie$saml = cookie.saml) === null || _cookie$saml === void 0 ? void 0 : _cookie$saml.requestId) || '';
          nextUrl = ((_cookie$saml2 = cookie.saml) === null || _cookie$saml2 === void 0 ? void 0 : _cookie$saml2.nextUrl) || `${this.coreSetup.http.basePath.serverBasePath}/app/opensearch-dashboards`;
          redirectHash = ((_cookie$saml3 = cookie.saml) === null || _cookie$saml3 === void 0 ? void 0 : _cookie$saml3.redirectHash) || false;
        }

        if (!requestId) {
          return response.badRequest({
            body: 'Invalid requestId'
          });
        }
      } catch (error) {
        context.security_plugin.logger.error(`Failed to parse cookie: ${error}`);
        return response.badRequest();
      }

      try {
        const credentials = await this.securityClient.authToken(requestId, request.body.SAMLResponse, undefined);
        const user = await this.securityClient.authenticateWithHeader(request, 'authorization', credentials.authorization);
        let expiryTime = Date.now() + this.config.session.ttl;
        const [headerEncoded, payloadEncoded, signature] = credentials.authorization.split('.');

        if (!payloadEncoded) {
          context.security_plugin.logger.error('JWT token payload not found');
        }

        const tokenPayload = JSON.parse(Buffer.from(payloadEncoded, 'base64').toString());

        if (tokenPayload.exp) {
          expiryTime = parseInt(tokenPayload.exp, 10) * 1000;
        }

        const cookie = {
          username: user.username,
          credentials: {
            authHeaderValue: credentials.authorization
          },
          authType: 'saml',
          // TODO: create constant
          expiryTime
        };
        this.sessionStorageFactory.asScoped(request).set(cookie);

        if (redirectHash) {
          return response.redirected({
            headers: {
              location: `${this.coreSetup.http.basePath.serverBasePath}/auth/saml/redirectUrlFragment?nextUrl=${escape(nextUrl)}`
            }
          });
        } else {
          return response.redirected({
            headers: {
              location: nextUrl
            }
          });
        }
      } catch (error) {
        context.security_plugin.logger.error(`SAML SP initiated authentication workflow failed: ${error}`);
      }

      return response.internalError();
    });
    this.router.post({
      path: `/_opendistro/_security/saml/acs/idpinitiated`,
      validate: {
        body: _configSchema.schema.any()
      },
      options: {
        authRequired: false
      }
    }, async (context, request, response) => {
      const acsEndpoint = `${this.coreSetup.http.basePath.serverBasePath}/_opendistro/_security/saml/acs/idpinitiated`;

      try {
        const credentials = await this.securityClient.authToken(undefined, request.body.SAMLResponse, acsEndpoint);
        const user = await this.securityClient.authenticateWithHeader(request, 'authorization', credentials.authorization);
        let expiryTime = Date.now() + this.config.session.ttl;
        const [headerEncoded, payloadEncoded, signature] = credentials.authorization.split('.');

        if (!payloadEncoded) {
          context.security_plugin.logger.error('JWT token payload not found');
        }

        const tokenPayload = JSON.parse(Buffer.from(payloadEncoded, 'base64').toString());

        if (tokenPayload.exp) {
          expiryTime = parseInt(tokenPayload.exp, 10) * 1000;
        }

        const cookie = {
          username: user.username,
          credentials: {
            authHeaderValue: credentials.authorization
          },
          authType: 'saml',
          // TODO: create constant
          expiryTime
        };
        this.sessionStorageFactory.asScoped(request).set(cookie);
        return response.redirected({
          headers: {
            location: `${this.coreSetup.http.basePath.serverBasePath}/app/opensearch-dashboards`
          }
        });
      } catch (error) {
        context.security_plugin.logger.error(`SAML IDP initiated authentication workflow failed: ${error}`);
      }

      return response.internalError();
    }); // captureUrlFragment is the first route that will be invoked in the SP initiated login.
    // This route will execute the captureUrlFragment.js script.

    this.coreSetup.http.resources.register({
      path: '/auth/saml/captureUrlFragment',
      validate: {
        query: _configSchema.schema.object({
          nextUrl: _configSchema.schema.maybe(_configSchema.schema.string({
            validate: _next_url.validateNextUrl
          }))
        })
      },
      options: {
        authRequired: false
      }
    }, async (context, request, response) => {
      this.sessionStorageFactory.asScoped(request).clear();
      const serverBasePath = this.coreSetup.http.basePath.serverBasePath;
      return response.renderHtml({
        body: `
            <!DOCTYPE html>
            <title>OSD SAML Capture</title>
            <link rel="icon" href="data:,">
            <script src="${serverBasePath}/auth/saml/captureUrlFragment.js"></script>
          `
      });
    }); // This script will store the URL Hash in browser's local storage.

    this.coreSetup.http.resources.register({
      path: '/auth/saml/captureUrlFragment.js',
      validate: false,
      options: {
        authRequired: false
      }
    }, async (context, request, response) => {
      this.sessionStorageFactory.asScoped(request).clear();
      return response.renderJs({
        body: `let samlHash=window.location.hash.toString();
                 let redirectHash = false;
                 if (samlHash !== "") {
                    window.localStorage.removeItem('samlHash');
                    window.localStorage.setItem('samlHash', samlHash);
                     redirectHash = true;
                  }
                 let params = new URLSearchParams(window.location.search);
                 let nextUrl = params.get("nextUrl");
                 finalUrl = "login?nextUrl=" + encodeURIComponent(nextUrl);
                 finalUrl += "&redirectHash=" + encodeURIComponent(redirectHash);
                 window.location.replace(finalUrl);

                `
      });
    }); //  Once the User is authenticated via the '_opendistro/_security/saml/acs' route,
    //  the browser will be redirected to '/auth/saml/redirectUrlFragment' route,
    //  which will execute the redirectUrlFragment.js.

    this.coreSetup.http.resources.register({
      path: '/auth/saml/redirectUrlFragment',
      validate: {
        query: _configSchema.schema.object({
          nextUrl: _configSchema.schema.any()
        })
      },
      options: {
        authRequired: true
      }
    }, async (context, request, response) => {
      const serverBasePath = this.coreSetup.http.basePath.serverBasePath;
      return response.renderHtml({
        body: `
            <!DOCTYPE html>
            <title>OSD SAML Success</title>
            <link rel="icon" href="data:,">
            <script src="${serverBasePath}/auth/saml/redirectUrlFragment.js"></script>
          `
      });
    }); // This script will pop the Hash from local storage if it exists.
    // And forward the browser to the next url.

    this.coreSetup.http.resources.register({
      path: '/auth/saml/redirectUrlFragment.js',
      validate: false,
      options: {
        authRequired: true
      }
    }, async (context, request, response) => {
      return response.renderJs({
        body: `let samlHash=window.localStorage.getItem('samlHash');
                 window.localStorage.removeItem('samlHash');
                 let params = new URLSearchParams(window.location.search);
                 let nextUrl = params.get("nextUrl");
                 finalUrl = nextUrl + samlHash;
                 window.location.replace(finalUrl);
                `
      });
    });
    this.router.get({
      path: `/auth/logout`,
      validate: false
    }, async (context, request, response) => {
      try {
        const authInfo = await this.securityClient.authinfo(request);
        this.sessionStorageFactory.asScoped(request).clear(); // TODO: need a default logout page

        const redirectUrl = authInfo.sso_logout_url || this.coreSetup.http.basePath.serverBasePath || '/';
        return response.redirected({
          headers: {
            location: redirectUrl
          }
        });
      } catch (error) {
        context.security_plugin.logger.error(`SAML logout failed: ${error}`);
        return response.badRequest();
      }
    });
  }

}

exports.SamlAuthRoutes = SamlAuthRoutes;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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