"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SamlAuthentication = void 0;

var _querystring = require("querystring");

var _security_cookie = require("../../../session/security_cookie");

var _routes = require("./routes");

var _authentication_type = require("../authentication_type");

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

class SamlAuthentication extends _authentication_type.AuthenticationType {
  constructor(config, sessionStorageFactory, router, esClient, coreSetup, logger) {
    super(config, sessionStorageFactory, router, esClient, coreSetup, logger);

    _defineProperty(this, "type", 'saml');

    _defineProperty(this, "redirectSAMlCapture", (request, toolkit) => {
      const nextUrl = this.generateNextUrl(request);
      const clearOldVersionCookie = (0, _security_cookie.clearOldVersionCookieValue)(this.config);
      return toolkit.redirected({
        location: `${this.coreSetup.http.basePath.serverBasePath}/auth/saml/captureUrlFragment?nextUrl=${nextUrl}`,
        'set-cookie': clearOldVersionCookie
      });
    });

    this.setupRoutes();
  }

  generateNextUrl(request) {
    const path = this.coreSetup.http.basePath.serverBasePath + (request.url.pathname || '/app/opensearch-dashboards');
    return (0, _querystring.escape)(path);
  }

  setupRoutes() {
    const samlAuthRoutes = new _routes.SamlAuthRoutes(this.router, this.config, this.sessionStorageFactory, this.securityClient, this.coreSetup);
    samlAuthRoutes.setupRoutes();
  }

  requestIncludesAuthInfo(request) {
    return request.headers[SamlAuthentication.AUTH_HEADER_NAME] ? true : false;
  }

  getAdditionalAuthHeader(request) {
    return {};
  }

  getCookie(request, authInfo) {
    return {
      username: authInfo.user_name,
      credentials: {
        authHeaderValue: request.headers[SamlAuthentication.AUTH_HEADER_NAME]
      },
      authType: this.type,
      expiryTime: Date.now() + this.config.session.ttl
    };
  }

  async isValidCookie(cookie) {
    var _cookie$credentials;

    return cookie.authType === this.type && cookie.username && cookie.expiryTime && ((_cookie$credentials = cookie.credentials) === null || _cookie$credentials === void 0 ? void 0 : _cookie$credentials.authHeaderValue);
  }

  handleUnauthedRequest(request, response, toolkit) {
    if (this.isPageRequest(request)) {
      return this.redirectSAMlCapture(request, toolkit);
    } else {
      return response.unauthorized();
    }
  }

  buildAuthHeaderFromCookie(cookie) {
    var _cookie$credentials2;

    const headers = {};
    headers[SamlAuthentication.AUTH_HEADER_NAME] = (_cookie$credentials2 = cookie.credentials) === null || _cookie$credentials2 === void 0 ? void 0 : _cookie$credentials2.authHeaderValue;
    return headers;
  }

}

exports.SamlAuthentication = SamlAuthentication;

_defineProperty(SamlAuthentication, "AUTH_HEADER_NAME", 'authorization');
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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