"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SamlAuthRoutes = void 0;

var _configSchema = require("@osd/config-schema");

var _next_url = require("../../../utils/next_url");

/*
 *   Copyright OpenSearch Contributors
 *
 *   Licensed under the Apache License, Version 2.0 (the "License").
 *   You may not use this file except in compliance with the License.
 *   A copy of the License is located at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   or in the "license" file accompanying this file. This file is distributed
 *   on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 *   express or implied. See the License for the specific language governing
 *   permissions and limitations under the License.
 */
class SamlAuthRoutes {
  constructor(router, // @ts-ignore: unused variable
  config, sessionStorageFactory, securityClient, coreSetup) {
    this.router = router;
    this.config = config;
    this.sessionStorageFactory = sessionStorageFactory;
    this.securityClient = securityClient;
    this.coreSetup = coreSetup;
  }

  setupRoutes() {
    this.router.get({
      path: `/auth/saml/login`,
      validate: {
        query: _configSchema.schema.object({
          nextUrl: _configSchema.schema.maybe(_configSchema.schema.string({
            validate: _next_url.validateNextUrl
          }))
        })
      },
      options: {
        authRequired: false
      }
    }, async (context, request, response) => {
      if (request.auth.isAuthenticated) {
        return response.redirected({
          headers: {
            location: `${this.coreSetup.http.basePath.serverBasePath}/app/opensearch-dashboards`
          }
        });
      }

      try {
        const samlHeader = await this.securityClient.getSamlHeader(request);
        const cookie = {
          saml: {
            nextUrl: request.query.nextUrl,
            requestId: samlHeader.requestId
          }
        };
        this.sessionStorageFactory.asScoped(request).set(cookie);
        return response.redirected({
          headers: {
            location: samlHeader.location
          }
        });
      } catch (error) {
        context.security_plugin.logger.error(`Failed to get saml header: ${error}`);
        return response.internalError(); // TODO: redirect to error page?
      }
    });
    this.router.post({
      path: `/_opendistro/_security/saml/acs`,
      validate: {
        body: _configSchema.schema.any()
      },
      options: {
        authRequired: false
      }
    }, async (context, request, response) => {
      let requestId = '';
      let nextUrl = '/';

      try {
        const cookie = await this.sessionStorageFactory.asScoped(request).get();

        if (cookie) {
          var _cookie$saml, _cookie$saml2;

          requestId = ((_cookie$saml = cookie.saml) === null || _cookie$saml === void 0 ? void 0 : _cookie$saml.requestId) || '';
          nextUrl = ((_cookie$saml2 = cookie.saml) === null || _cookie$saml2 === void 0 ? void 0 : _cookie$saml2.nextUrl) || `${this.coreSetup.http.basePath.serverBasePath}/app/opensearch-dashboards`;
        }

        if (!requestId) {
          return response.badRequest({
            body: 'Invalid requestId'
          });
        }
      } catch (error) {
        context.security_plugin.logger.error(`Failed to parse cookie: ${error}`);
        return response.badRequest();
      }

      try {
        const credentials = await this.securityClient.authToken(requestId, request.body.SAMLResponse, undefined);
        const user = await this.securityClient.authenticateWithHeader(request, 'authorization', credentials.authorization);
        let expiryTime = Date.now() + this.config.session.ttl;
        const [headerEncoded, payloadEncoded, signature] = credentials.authorization.split('.');

        if (!payloadEncoded) {
          context.security_plugin.logger.error('JWT token payload not found');
        }

        const tokenPayload = JSON.parse(Buffer.from(payloadEncoded, 'base64').toString());

        if (tokenPayload.exp) {
          expiryTime = parseInt(tokenPayload.exp, 10) * 1000;
        }

        const cookie = {
          username: user.username,
          credentials: {
            authHeaderValue: credentials.authorization
          },
          authType: 'saml',
          // TODO: create constant
          expiryTime
        };
        this.sessionStorageFactory.asScoped(request).set(cookie);
        return response.redirected({
          headers: {
            location: nextUrl
          }
        });
      } catch (error) {
        context.security_plugin.logger.error(`SAML SP initiated authentication workflow failed: ${error}`);
      }

      return response.internalError();
    });
    this.router.post({
      path: `/_opendistro/_security/saml/acs/idpinitiated`,
      validate: {
        body: _configSchema.schema.any()
      },
      options: {
        authRequired: false
      }
    }, async (context, request, response) => {
      const acsEndpoint = `${this.coreSetup.http.basePath.serverBasePath}/_opendistro/_security/saml/acs/idpinitiated`;

      try {
        const credentials = await this.securityClient.authToken(undefined, request.body.SAMLResponse, acsEndpoint);
        const user = await this.securityClient.authenticateWithHeader(request, 'authorization', credentials.authorization);
        let expiryTime = Date.now() + this.config.session.ttl;
        const [headerEncoded, payloadEncoded, signature] = credentials.authorization.split('.');

        if (!payloadEncoded) {
          context.security_plugin.logger.error('JWT token payload not found');
        }

        const tokenPayload = JSON.parse(Buffer.from(payloadEncoded, 'base64').toString());

        if (tokenPayload.exp) {
          expiryTime = parseInt(tokenPayload.exp, 10) * 1000;
        }

        const cookie = {
          username: user.username,
          credentials: {
            authHeaderValue: credentials.authorization
          },
          authType: 'saml',
          // TODO: create constant
          expiryTime
        };
        this.sessionStorageFactory.asScoped(request).set(cookie);
        return response.redirected({
          headers: {
            location: `${this.coreSetup.http.basePath.serverBasePath}/app/opensearch-dashboards`
          }
        });
      } catch (error) {
        context.security_plugin.logger.error(`SAML IDP initiated authentication workflow failed: ${error}`);
      }

      return response.internalError();
    });
    this.router.get({
      path: `/auth/logout`,
      validate: false
    }, async (context, request, response) => {
      try {
        const authInfo = await this.securityClient.authinfo(request);
        this.sessionStorageFactory.asScoped(request).clear(); // TODO: need a default logout page

        const redirectUrl = authInfo.sso_logout_url || this.coreSetup.http.basePath.serverBasePath || '/';
        return response.redirected({
          headers: {
            location: redirectUrl
          }
        });
      } catch (error) {
        context.security_plugin.logger.error(`SAML logout failed: ${error}`);
        return response.badRequest();
      }
    });
  }

}

exports.SamlAuthRoutes = SamlAuthRoutes;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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