/*
 * Copyright (C) 2017, 2018 TypeFox and others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at http://www.apache.org/licenses/LICENSE-2.0
 */
import * as lsp from 'vscode-languageserver';
export class LspClientLogger {
    constructor(client, level) {
        this.client = client;
        this.level = level;
    }
    sendMessage(severity, messageObjects) {
        if (this.level >= severity) {
            const message = messageObjects.map(p => {
                if (typeof p === 'object') {
                    return JSON.stringify(p, null, 2);
                }
                else {
                    return p;
                }
            }).join(' ');
            this.client.logMessage({
                type: severity,
                message: message,
            });
        }
    }
    error(...arg) {
        this.sendMessage(lsp.MessageType.Error, arg);
    }
    warn(...arg) {
        this.sendMessage(lsp.MessageType.Warning, arg);
    }
    info(...arg) {
        this.sendMessage(lsp.MessageType.Info, arg);
    }
    log(...arg) {
        this.sendMessage(lsp.MessageType.Log, arg);
    }
}
export var ConsoleLogLevel;
(function (ConsoleLogLevel) {
    ConsoleLogLevel["error"] = "error";
    ConsoleLogLevel["warn"] = "warn";
    ConsoleLogLevel["info"] = "warn";
    ConsoleLogLevel["verbose"] = "verbose";
})(ConsoleLogLevel = ConsoleLogLevel || (ConsoleLogLevel = {}));
export class ConsoleLogger {
    constructor(level = lsp.MessageType.Info) {
        this.level = level;
    }
    static toMessageTypeLevel(type) {
        switch (type) {
            case 'error':
                return lsp.MessageType.Error;
            case 'warn':
                return lsp.MessageType.Warning;
            case 'log':
                return lsp.MessageType.Log;
            case 'info':
            default:
                return lsp.MessageType.Info;
        }
    }
    print(type, level, ...arg) {
        if (this.level >= level) {
            // eslint-disable-next-line no-console
            console[type](...this.toStrings(arg));
        }
    }
    toStrings(...arg) {
        return arg.map(a => JSON.stringify(a, null, 2));
    }
    error(...arg) {
        this.print('error', lsp.MessageType.Error, arg);
    }
    warn(...arg) {
        this.print('error', lsp.MessageType.Warning, arg);
    }
    info(...arg) {
        this.print('error', lsp.MessageType.Info, arg);
    }
    log(...arg) {
        this.print('error', lsp.MessageType.Log, arg);
    }
}
export class PrefixingLogger {
    constructor(logger, prefix) {
        this.logger = logger;
        this.prefix = prefix;
    }
    error(...arg) {
        this.logger.error(this.prefix, ...arg);
    }
    warn(...arg) {
        this.logger.warn(this.prefix, ...arg);
    }
    info(...arg) {
        this.logger.info(this.prefix, ...arg);
    }
    log(...arg) {
        this.logger.log(this.prefix, ...arg);
    }
}
//# sourceMappingURL=logger.js.map