"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.validateReportDefinition = exports.validateReport = exports.regexReportName = exports.regexRelativeUrl = exports.regexEmailAddress = exports.regexDuration = exports.isValidRelativeUrl = void 0;

var _path = _interopRequireDefault(require("path"));

var _model = require("../../server/model");

var _constants = require("../../server/routes/utils/constants");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */
const isValidRelativeUrl = relativeUrl => {
  let normalizedRelativeUrl = relativeUrl;

  if (!relativeUrl.includes('observability#/notebooks') && !relativeUrl.includes('notebooks-dashboards')) {
    normalizedRelativeUrl = _path.default.posix.normalize(relativeUrl);
  } // check pattern
  // ODFE pattern: /app/dashboards#/view/7adfa750-4c81-11e8-b3d7-01146121b73d?_g
  // AES pattern: /_plugin/kibana/app/dashboards#/view/7adfa750-4c81-11e8-b3d7-01146121b73d?_g


  const isValid = regexRelativeUrl.test(normalizedRelativeUrl);
  return isValid;
};
/**
 * moment.js isValid() API fails to validate time duration, so use regex
 * https://github.com/moment/moment/issues/1805
 **/


exports.isValidRelativeUrl = isValidRelativeUrl;
const regexDuration = /^(-?)P(?=\d|T\d)(?:(\d+)Y)?(?:(\d+)M)?(?:(\d+)([DW]))?(?:T(?:(\d+)H)?(?:(\d+)M)?(?:(\d+(?:\.\d+)?)S)?)?$/;
exports.regexDuration = regexDuration;
const regexEmailAddress = /\S+@\S+\.\S+/;
exports.regexEmailAddress = regexEmailAddress;
const regexReportName = /^[\w\-\s\(\)\[\]\,\_\-+]+$/;
exports.regexReportName = regexReportName;
const regexRelativeUrl = /^\/(_plugin\/kibana\/|_dashboards\/)?app\/(dashboards|visualize|discover|observability-dashboards|notebooks-dashboards\?view=output_only(&security_tenant=.+)?)(\?security_tenant=.+)?#\/(notebooks\/|view\/|edit\/)?[^\/]+$/;
exports.regexRelativeUrl = regexRelativeUrl;

const validateReport = async (client, report, basePath) => {
  report.query_url = report.query_url.replace(basePath, '');
  report.report_definition.report_params.core_params.base_url = report.report_definition.report_params.core_params.base_url.replace(basePath, ''); // validate basic schema

  report = _model.reportSchema.validate(report); // parse to retrieve data

  const {
    query_url: queryUrl,
    report_definition: {
      report_params: {
        report_source: reportSource
      }
    }
  } = report; // Check if saved object actually exists

  await validateSavedObject(client, queryUrl, reportSource);
  return report;
};

exports.validateReport = validateReport;

const validateReportDefinition = async (client, reportDefinition, basePath) => {
  reportDefinition.report_params.core_params.base_url = reportDefinition.report_params.core_params.base_url.replace(basePath, ''); // validate basic schema

  reportDefinition = _model.reportDefinitionSchema.validate(reportDefinition); // parse to retrieve data

  const {
    report_params: {
      report_source: reportSource,
      core_params: {
        base_url: baseUrl
      }
    }
  } = reportDefinition; // Check if saved object actually exists

  await validateSavedObject(client, baseUrl, reportSource);
  return reportDefinition;
};

exports.validateReportDefinition = validateReportDefinition;

const validateSavedObject = async (client, url, source) => {
  const getId = url => {
    var _url$split$pop;

    return (_url$split$pop = url.split('/').pop()) === null || _url$split$pop === void 0 ? void 0 : _url$split$pop.replace(/\?\S+$/, '');
  };

  const getType = source => {
    switch (source) {
      case _constants.REPORT_TYPE.dashboard:
        return 'dashboard';

      case _constants.REPORT_TYPE.savedSearch:
        return 'search';

      case _constants.REPORT_TYPE.visualization:
        return 'visualization';

      case _constants.REPORT_TYPE.notebook:
        return 'notebook';
    }
  };

  let exist = false;
  let savedObjectId = '';

  if (getType(source) === 'notebook') {
    // no backend check for notebooks because we would just be checking against the notebooks api again
    exist = true;
  } else {
    savedObjectId = `${getType(source)}:${getId(url)}`;
    const params = {
      index: '.kibana',
      id: savedObjectId
    };
    exist = await client.callAsCurrentUser('exists', params);
  }

  if (!exist) {
    throw Error(`saved object with id ${savedObjectId} does not exist`);
  }
};
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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