"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
Object.defineProperty(exports, "SecurityPluginSetup", {
  enumerable: true,
  get: function () {
    return _types.SecurityPluginSetup;
  }
});
Object.defineProperty(exports, "SecurityPluginStart", {
  enumerable: true,
  get: function () {
    return _types.SecurityPluginStart;
  }
});
exports.configSchema = exports.config = void 0;
exports.plugin = plugin;

var _configSchema = require("@osd/config-schema");

var _plugin = require("./plugin");

var _types = require("./types");

/*
 *   Copyright OpenSearch Contributors
 *
 *   Licensed under the Apache License, Version 2.0 (the "License").
 *   You may not use this file except in compliance with the License.
 *   A copy of the License is located at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   or in the "license" file accompanying this file. This file is distributed
 *   on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 *   express or implied. See the License for the specific language governing
 *   permissions and limitations under the License.
 */
const configSchema = _configSchema.schema.object({
  enabled: _configSchema.schema.boolean({
    defaultValue: true
  }),
  allow_client_certificates: _configSchema.schema.boolean({
    defaultValue: false
  }),
  readonly_mode: _configSchema.schema.object({
    roles: _configSchema.schema.arrayOf(_configSchema.schema.string(), {
      defaultValue: []
    })
  }),
  clusterPermissions: _configSchema.schema.object({
    include: _configSchema.schema.arrayOf(_configSchema.schema.string(), {
      defaultValue: []
    })
  }),
  indexPermissions: _configSchema.schema.object({
    include: _configSchema.schema.arrayOf(_configSchema.schema.string(), {
      defaultValue: []
    })
  }),
  disabledTransportCategories: _configSchema.schema.object({
    exclude: _configSchema.schema.arrayOf(_configSchema.schema.string(), {
      defaultValue: []
    })
  }),
  disabledRestCategories: _configSchema.schema.object({
    exclude: _configSchema.schema.arrayOf(_configSchema.schema.string(), {
      defaultValue: []
    })
  }),
  cookie: _configSchema.schema.object({
    secure: _configSchema.schema.boolean({
      defaultValue: false
    }),
    name: _configSchema.schema.string({
      defaultValue: 'security_authentication'
    }),
    password: _configSchema.schema.string({
      defaultValue: 'security_cookie_default_password',
      minLength: 32
    }),
    ttl: _configSchema.schema.number({
      defaultValue: 60 * 60 * 1000
    }),
    domain: _configSchema.schema.nullable(_configSchema.schema.string()),
    isSameSite: _configSchema.schema.oneOf([_configSchema.schema.literal('Strict'), _configSchema.schema.literal('Lax'), _configSchema.schema.literal('None'), _configSchema.schema.literal(false)], {
      defaultValue: false
    })
  }),
  session: _configSchema.schema.object({
    ttl: _configSchema.schema.number({
      defaultValue: 60 * 60 * 1000
    }),
    keepalive: _configSchema.schema.boolean({
      defaultValue: true
    })
  }),
  auth: _configSchema.schema.object({
    type: _configSchema.schema.string({
      defaultValue: '',

      validate(value) {
        if (!['', 'basicauth', 'jwt', 'openid', 'saml', 'proxy', 'kerberos', 'proxycache'].includes(value)) {
          return `allowed auth.type are ['', 'basicauth', 'jwt', 'openid', 'saml', 'proxy', 'kerberos', 'proxycache']`;
        }
      }

    }),
    anonymous_auth_enabled: _configSchema.schema.boolean({
      defaultValue: false
    }),
    unauthenticated_routes: _configSchema.schema.arrayOf(_configSchema.schema.string(), {
      defaultValue: ['/api/reporting/stats']
    }),
    forbidden_usernames: _configSchema.schema.arrayOf(_configSchema.schema.string(), {
      defaultValue: []
    }),
    logout_url: _configSchema.schema.string({
      defaultValue: ''
    })
  }),
  basicauth: _configSchema.schema.object({
    enabled: _configSchema.schema.boolean({
      defaultValue: true
    }),
    unauthenticated_routes: _configSchema.schema.arrayOf(_configSchema.schema.string(), {
      defaultValue: []
    }),
    forbidden_usernames: _configSchema.schema.arrayOf(_configSchema.schema.string(), {
      defaultValue: []
    }),
    header_trumps_session: _configSchema.schema.boolean({
      defaultValue: false
    }),
    alternative_login: _configSchema.schema.object({
      headers: _configSchema.schema.arrayOf(_configSchema.schema.string(), {
        defaultValue: []
      }),
      show_for_parameter: _configSchema.schema.string({
        defaultValue: ''
      }),
      valid_redirects: _configSchema.schema.arrayOf(_configSchema.schema.string(), {
        defaultValue: []
      }),
      button_text: _configSchema.schema.string({
        defaultValue: 'Login with provider'
      }),
      buttonstyle: _configSchema.schema.string({
        defaultValue: ''
      })
    }),
    loadbalancer_url: _configSchema.schema.maybe(_configSchema.schema.string()),
    login: _configSchema.schema.object({
      title: _configSchema.schema.string({
        defaultValue: 'Please login to OpenSearch Dashboards'
      }),
      subtitle: _configSchema.schema.string({
        defaultValue: 'If you have forgotten your username or password, please ask your system administrator'
      }),
      showbrandimage: _configSchema.schema.boolean({
        defaultValue: true
      }),
      brandimage: _configSchema.schema.string({
        defaultValue: ''
      }),
      // TODO: update brand image
      buttonstyle: _configSchema.schema.string({
        defaultValue: ''
      })
    })
  }),
  multitenancy: _configSchema.schema.object({
    enabled: _configSchema.schema.boolean({
      defaultValue: false
    }),
    show_roles: _configSchema.schema.boolean({
      defaultValue: false
    }),
    enable_filter: _configSchema.schema.boolean({
      defaultValue: false
    }),
    debug: _configSchema.schema.boolean({
      defaultValue: false
    }),
    tenants: _configSchema.schema.object({
      enable_private: _configSchema.schema.boolean({
        defaultValue: true
      }),
      enable_global: _configSchema.schema.boolean({
        defaultValue: true
      }),
      preferred: _configSchema.schema.arrayOf(_configSchema.schema.string(), {
        defaultValue: []
      })
    })
  }),
  configuration: _configSchema.schema.object({
    enabled: _configSchema.schema.boolean({
      defaultValue: true
    })
  }),
  accountinfo: _configSchema.schema.object({
    enabled: _configSchema.schema.boolean({
      defaultValue: false
    })
  }),
  openid: _configSchema.schema.maybe(_configSchema.schema.object({
    connect_url: _configSchema.schema.maybe(_configSchema.schema.string()),
    header: _configSchema.schema.string({
      defaultValue: 'Authorization'
    }),
    // TODO: test if siblingRef() works here
    // client_id is required when auth.type is openid
    client_id: _configSchema.schema.conditional(_configSchema.schema.siblingRef('auth.type'), 'openid', _configSchema.schema.string(), _configSchema.schema.maybe(_configSchema.schema.string())),
    client_secret: _configSchema.schema.string({
      defaultValue: ''
    }),
    scope: _configSchema.schema.string({
      defaultValue: 'openid profile email address phone'
    }),
    base_redirect_url: _configSchema.schema.string({
      defaultValue: ''
    }),
    logout_url: _configSchema.schema.string({
      defaultValue: ''
    }),
    root_ca: _configSchema.schema.string({
      defaultValue: ''
    }),
    verify_hostnames: _configSchema.schema.boolean({
      defaultValue: true
    }),
    refresh_tokens: _configSchema.schema.boolean({
      defaultValue: true
    }),
    trust_dynamic_headers: _configSchema.schema.boolean({
      defaultValue: false
    })
  })),
  proxycache: _configSchema.schema.maybe(_configSchema.schema.object({
    // when auth.type is proxycache, user_header, roles_header and proxy_header_ip are required
    user_header: _configSchema.schema.conditional(_configSchema.schema.siblingRef('auth.type'), 'proxycache', _configSchema.schema.string(), _configSchema.schema.maybe(_configSchema.schema.string())),
    roles_header: _configSchema.schema.conditional(_configSchema.schema.siblingRef('auth.type'), 'proxycache', _configSchema.schema.string(), _configSchema.schema.maybe(_configSchema.schema.string())),
    proxy_header: _configSchema.schema.maybe(_configSchema.schema.string({
      defaultValue: 'x-forwarded-for'
    })),
    proxy_header_ip: _configSchema.schema.conditional(_configSchema.schema.siblingRef('auth.type'), 'proxycache', _configSchema.schema.string(), _configSchema.schema.maybe(_configSchema.schema.string())),
    login_endpoint: _configSchema.schema.maybe(_configSchema.schema.string({
      defaultValue: ''
    }))
  })),
  jwt: _configSchema.schema.maybe(_configSchema.schema.object({
    enabled: _configSchema.schema.boolean({
      defaultValue: false
    }),
    login_endpoint: _configSchema.schema.maybe(_configSchema.schema.string()),
    url_param: _configSchema.schema.string({
      defaultValue: 'authorization'
    }),
    header: _configSchema.schema.string({
      defaultValue: 'Authorization'
    })
  })),
  ui: _configSchema.schema.object({
    basicauth: _configSchema.schema.object({
      // the login config here is the same as old config `_security.basicauth.login`
      // Since we are now rendering login page to browser app, so move these config to browser side.
      login: _configSchema.schema.object({
        title: _configSchema.schema.string({
          defaultValue: 'Please login to OpenSearch Dashboards'
        }),
        subtitle: _configSchema.schema.string({
          defaultValue: 'If you have forgotten your username or password, please ask your system administrator'
        }),
        showbrandimage: _configSchema.schema.boolean({
          defaultValue: true
        }),
        brandimage: _configSchema.schema.string({
          defaultValue: ''
        }),
        buttonstyle: _configSchema.schema.string({
          defaultValue: ''
        })
      })
    }),
    autologout: _configSchema.schema.boolean({
      defaultValue: true
    }),
    backend_configurable: _configSchema.schema.boolean({
      defaultValue: true
    })
  })
});

exports.configSchema = configSchema;
const config = {
  exposeToBrowser: {
    enabled: true,
    auth: true,
    ui: true,
    multitenancy: true,
    readonly_mode: true,
    clusterPermissions: true,
    indexPermissions: true,
    disabledTransportCategories: true,
    disabledRestCategories: true
  },
  schema: configSchema,
  deprecations: ({
    rename,
    unused
  }) => [rename('basicauth.login.title', 'ui.basicauth.login.title'), rename('basicauth.login.subtitle', 'ui.basicauth.login.subtitle'), rename('basicauth.login.showbrandimage', 'ui.basicauth.login.showbrandimage'), rename('basicauth.login.brandimage', 'ui.basicauth.login.brandimage'), rename('basicauth.login.buttonstyle', 'ui.basicauth.login.buttonstyle')]
}; //  This exports static code and TypeScript types,
//  as well as, OpenSearchDashboards Platform `plugin()` initializer.

exports.config = config;

function plugin(initializerContext) {
  return new _plugin.SecurityPlugin(initializerContext);
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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